within SolarHouse;
package BuildingParts "Heat transfer in the solids"
  model SolidLayer "Model heat conduction through single wall layer"
    extends Iface.BlockIcon;
    parameter SI.ThermalConductivity lambda "Thermal Conductivity";
    parameter SI.Density rho "Density of material";
    parameter SI.SpecificHeatCapacity c "Specific heat capacity";
    parameter SI.Thickness L "Layer thickness";
    parameter SI.Area A "Surface of layer";
    Modelica.Thermal.HeatTransfer.HeatCapacitor LayerCapacity(
                                   C=rho*c*L*A) 
      annotation (Placement(transformation(extent={{-12,0},{12,24}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.ThermalConductor InnerSide(
                                  G=2*lambda*A/L) 
      annotation (Placement(transformation(extent={{-60,-10},{-40,10}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.ThermalConductor OuterSide(
                                  G=2*lambda*A/L) 
      annotation (Placement(transformation(extent={{40,-10},{60,10}}, rotation=
              0)));
    HT.Interfaces.HeatPort_a inside annotation (Placement(transformation(extent
            ={{-120,-10},{-100,10}}, rotation=0)));
    HT.Interfaces.HeatPort_b outside annotation (Placement(transformation(
            extent={{100,-10},{120,10}}, rotation=0)));
    annotation (Diagram(graphics),
                         Icon(graphics));
  equation
    connect(InnerSide.port_a, inside) annotation (Line(points={{-60,6.10623e-16},
            {-46,0},{-46,5.55112e-16},{-110,5.55112e-16}}, color={191,0,0}));
    connect(LayerCapacity.port, OuterSide.port_a) annotation (Line(points={{
            -3.77476e-16,4.44089e-16},{0,4.44089e-16},{0,6.10623e-16},{40,
            6.10623e-16}}, color={191,0,0}));
    connect(InnerSide.port_b, LayerCapacity.port) annotation (Line(points={{-40,
            6.10623e-16},{-12,0},{-12,4.44089e-16},{-3.77476e-16,4.44089e-16}}, 
          color={191,0,0}));
    connect(OuterSide.port_b, outside) annotation (Line(points={{60,6.10623e-16},
            {56,0},{56,5.55112e-16},{110,5.55112e-16}}, color={191,0,0}));
  end SolidLayer;

  partial model Wall "Base model of a wall"
    extends Iface.BlockIcon;
    parameter SI.Area Area "Area of the wall's indoor surface";
    parameter SI.SpectralAbsorptionFactor absorpt
      "Absorption coefficient of the inner wall surface";
    parameter SI.Temperature Tout_start(start=from_degC(20))
      "Initial temperature at indoor surface"                                                        annotation(Dialog(tab="Initialization"));
    parameter SI.Temperature Tin_start(start=from_degC(20))
      "Initial temperature at outdoor surface"                                                       annotation(Dialog(tab="Initialization"));
    parameter Boolean steadyStateStart=false
      "Steady-state initial temperatures of the layers"                                        annotation(Dialog(tab="Initialization"));
    parameter SI.Length wallWidth "Thickness of the wall" annotation(Hide=true);
    HT.Interfaces.HeatPort_a InnerSide 
      annotation (Placement(transformation(extent={{-120,-10},{-100,10}},
            rotation=0)));
    HT.Interfaces.HeatPort_b OuterSide annotation (Placement(transformation(
            extent={{100,-10},{120,10}}, rotation=0)));

    annotation (Diagram(graphics),
                         Icon(graphics={
          Rectangle(
            extent={{-40,40},{-20,-80}},
            lineColor={0,0,0},
            fillColor={225,93,17},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-20,40},{40,80},{40,-40},{-20,-80},{-20,40}},
            lineColor={0,0,0},
            fillPattern=FillPattern.VerticalCylinder,
            fillColor={225,93,17}),
          Polygon(
            points={{-40,40},{20,80},{40,80},{-20,40},{-40,40}},
            lineColor={0,0,0},
            fillColor={225,93,17},
            fillPattern=FillPattern.Backward)}));
  end Wall;

  model Wall1 "Model of a wall with one layer"
    extends Wall(final wallWidth=Layer1.L);
    SolidLayer Layer1(
      A=Area,
      rho=900,
      lambda=0.640,
      c=1.6*3600,
      L=0.045,
      LayerCapacity(T(start=(Tout_start + Tin_start)/2), steadyStateStart=
            steadyStateStart)) annotation (Placement(transformation(extent={{
              -10,-10},{10,10}}, rotation=0)));
    annotation (Diagram(graphics));
  equation
    connect(InnerSide, Layer1.inside) annotation (Line(points={{-110,
            5.55112e-16},{-62,5.55112e-16},{-62,6.10623e-16},{-11,6.10623e-16}}, 
          color={191,0,0}));
    connect(Layer1.outside, OuterSide) annotation (Line(points={{11,6.10623e-16},
            {59.5,6.10623e-16},{59.5,5.55112e-16},{110,5.55112e-16}}, color={
            191,0,0}));
  end Wall1;

  model Wall3 "Model of a wall with 3 layers"
    extends Wall(final wallWidth=Layer1.L + Layer2.L + Layer3.L);
    SolidLayer Layer1(
      lambda=0.210,
      L=0.012,
      A=Area,
      rho=900,
      c=0.21*3600,
      LayerCapacity(T(start=Tout_start + Layer1.L/2/(Layer1.L + Layer2.L +
              Layer3.L)*(Tin_start - Tout_start)), steadyStateStart=
            steadyStateStart)) "plaster layer" 
      annotation (Placement(transformation(extent={{-60,-10},{-40,10}},
            rotation=0)));
    SolidLayer Layer2(
      lambda=0.041,
      A=Area,
      L=0.040,
      rho=30,
      c=0.35*3600,
      LayerCapacity(T(start=Tout_start + (Layer1.L + Layer2.L/2)/(Layer1.L +
              Layer2.L + Layer3.L)*(Tin_start - Tout_start)),
          steadyStateStart=steadyStateStart)) "polistyren" 
      annotation (Placement(transformation(extent={{-10,-10},{10,10}}, rotation
            =0)));
    SolidLayer Layer3(
      A=Area,
      lambda=0.640,
      L=0.190,
      rho=900,
      c=1.6*3600,
      LayerCapacity(T(start=Tin_start + Layer3.L/2/(Layer1.L + Layer2.L +
              Layer3.L)*(Tout_start - Tin_start)), steadyStateStart=
            steadyStateStart)) "brick" annotation (Placement(transformation(
            extent={{40,-10},{60,10}}, rotation=0)));
  equation
    connect(Layer2.outside, Layer3.inside) annotation (Line(points={{11,
            6.10623e-16},{32,6.10623e-16},{32,0},{36,0},{36,6.10623e-16},{39,
            6.10623e-16}}, color={191,0,0}));
    annotation (Diagram(graphics));
    connect(InnerSide, Layer1.inside) annotation (Line(points={{-110,
            5.55112e-16},{-86,5.55112e-16},{-86,6.10623e-16},{-61,6.10623e-16}}, 
          color={191,0,0}));
    connect(Layer1.outside, Layer2.inside) annotation (Line(points={{-39,
            6.10623e-16},{-32,6.10623e-16},{-32,6.10623e-16},{-25,6.10623e-16},
            {-25,6.10623e-16},{-11,6.10623e-16}}, color={191,0,0}));
    connect(Layer3.outside, OuterSide) annotation (Line(points={{61,6.10623e-16},
            {85.5,6.10623e-16},{85.5,5.55112e-16},{110,5.55112e-16}}, color={
            191,0,0}));
  end Wall3;

  model Wall5 "Model of a layered wall"
    extends Wall(final wallWidth=Layer1.L + Layer2.L + Layer3.L +
        Layer4.L + Layer5.L);

    SolidLayer Layer1(
      lambda=0.210,
      L=0.012,
      A=Area,
      rho=900,
      c=.21*3600,
      LayerCapacity(T(start=Tout_start + Layer1.L/2/wallWidth*(Tin_start -
              Tout_start)), steadyStateStart=steadyStateStart)) 
      annotation (Placement(transformation(extent={{-84,-10},{-64,10}},
            rotation=0)));
    SolidLayer Layer2(
      lambda=0.041,
      L=0.020,
      A=Area,
      rho=30,
      c=.35*3600,
      LayerCapacity(T(start=Tout_start + (Layer1.L + Layer2.L/2)/wallWidth*(
              Tin_start - Tout_start)), steadyStateStart=steadyStateStart)) 
      annotation (Placement(transformation(extent={{-44,-10},{-24,10}},
            rotation=0)));
    SolidLayer Layer3(
      lambda=0.041,
      L=0.020,
      A=Area,
      rho=30,
      c=.35*3600,
      LayerCapacity(T(start=Tout_start + (Layer1.L + Layer2.L + Layer3.L/2)/
              wallWidth*(Tin_start - Tout_start)), steadyStateStart=
            steadyStateStart)) annotation (Placement(transformation(extent={{
              -10,-10},{10,10}}, rotation=0)));
    SolidLayer Layer4(
      lambda=0.640,
      L=0.095,
      A=Area,
      rho=900,
      c=1.6*3600,
      LayerCapacity(T(start=Tin_start + (Layer5.L + Layer4.L/2)/wallWidth*(
              Tout_start - Tin_start)), steadyStateStart=steadyStateStart)) 
      annotation (Placement(transformation(extent={{28,-10},{48,10}}, rotation=
              0)));
    SolidLayer Layer5(
      lambda=0.640,
      L=0.095,
      A=Area,
      rho=900,
      c=1.6*3600,
      LayerCapacity(T(start=Tin_start + Layer5.L/2/wallWidth*(Tout_start -
              Tin_start)), steadyStateStart=steadyStateStart)) 
      annotation (Placement(transformation(extent={{64,-10},{84,10}}, rotation=
              0)));
    annotation (Diagram(graphics));
  equation
    connect(InnerSide, Layer1.inside) annotation (Line(points={{-110,
            5.55112e-16},{-98,5.55112e-16},{-98,6.10623e-16},{-85,6.10623e-16}}, 
          color={191,0,0}));
    connect(Layer1.outside, Layer2.inside) annotation (Line(points={{-63,
            6.10623e-16},{-58.5,6.10623e-16},{-58.5,6.10623e-16},{-54,
            6.10623e-16},{-54,6.10623e-16},{-45,6.10623e-16}}, color={191,0,0}));
    connect(Layer2.outside, Layer3.inside) annotation (Line(points={{-23,
            6.10623e-16},{-20,6.10623e-16},{-20,6.10623e-16},{-17,6.10623e-16},
            {-17,6.10623e-16},{-11,6.10623e-16}}, color={191,0,0}));
    connect(Layer3.outside, Layer4.inside) annotation (Line(points={{11,
            6.10623e-16},{15,6.10623e-16},{15,6.10623e-16},{19,6.10623e-16},{19,
            6.10623e-16},{27,6.10623e-16}}, color={191,0,0}));
    connect(Layer4.outside, Layer5.inside) annotation (Line(points={{49,
            6.10623e-16},{52.5,6.10623e-16},{52.5,6.10623e-16},{56,6.10623e-16},
            {56,6.10623e-16},{63,6.10623e-16}}, color={191,0,0}));
    connect(Layer5.outside, OuterSide) annotation (Line(points={{85,6.10623e-16},
            {98.5,6.10623e-16},{98.5,5.55112e-16},{110,5.55112e-16}}, color={
            191,0,0}));
  end Wall5;

  model OverNightHeatInsolation
    "Lumped thermal element transporting heat without storing it"
    extends HT.Interfaces.Element1D;
    parameter SI.ThermalConductance G
      "Constant thermal conductance of material";
    annotation (
      Icon(graphics={
          Rectangle(
            extent={{-100,6},{99,-70}},
            lineColor={255,255,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-100,70},{100,6}},
            lineColor={0,0,0},
            pattern=LinePattern.None,
            fillColor={192,192,192},
            fillPattern=FillPattern.Backward),
          Line(
            points={{-100,70},{-100,-70}},
            color={0,0,0},
            thickness=0.5),
          Line(
            points={{100,70},{100,-70}},
            color={0,0,0},
            thickness=0.5),
          Text(extent={{-139,-72},{141,-132}}, textString=
                                                   "%name"),
          Line(points={{-40,6},{-40,-70}}, color={0,127,255}),
          Line(points={{-40,-70},{-30,-50}}, color={0,127,255}),
          Line(points={{-40,-70},{-50,-50}}, color={0,127,255}),
          Line(points={{-86,6},{-86,-70}}, color={0,127,255}),
          Line(points={{-86,-70},{-76,-50}}, color={0,127,255}),
          Line(points={{-86,-70},{-96,-50}}, color={0,127,255}),
          Line(points={{0,6},{0,-70}}, color={0,127,255}),
          Line(points={{0,-70},{10,-50}}, color={0,127,255}),
          Line(points={{0,-70},{-10,-50}}, color={0,127,255}),
          Line(points={{34,6},{34,-70}}, color={0,127,255}),
          Line(points={{34,-70},{44,-50}}, color={0,127,255}),
          Line(points={{34,-70},{24,-50}}, color={0,127,255}),
          Line(points={{70,6},{70,-70}}, color={0,127,255}),
          Line(points={{70,-70},{80,-50}}, color={0,127,255}),
          Line(points={{70,-70},{60,-50}}, color={0,127,255}),
          Line(points={{-94,-34},{100,-34}}, color={191,0,0}),
          Line(points={{80,-24},{100,-34}}, color={191,0,0}),
          Line(points={{80,-44},{100,-34}}, color={191,0,0})}),
      Diagram(graphics={Line(
            points={{-80,0},{80,0}},
            color={255,0,0},
            thickness=0.5,
            arrow={Arrow.None,Arrow.Filled})}),
      Documentation(info="<HTML>
<p>
This is a model for transport of heat without storing it.
It may be used for complicated geometries where
the thermal conductance G (= inverse of thermal resistance)
is determined by measurements and is assumed to be constant
over the range of operations. If the component consists mainly of
one type of material and a regular geometry, it may be calculated,
e.g., with one of the following equations:
</p>
<ul>
<li><p>
    Conductance for a <b>box</b> geometry under the assumption
    that heat flows along the box length:</p>
    <pre>
    G = k*A/L
    k: Thermal conductivity (material constant)
    A: Area of box
    L: Length of box
    </pre>
    </li>
<li><p>
    Conductance for a <b>cylindrical</b> geometry under the assumption
    that heat flows from the inside to the outside radius
    of the cylinder:</p>
    <pre>
    G = 2*pi*k*L/log(r_out/r_in)
    pi   : Modelica.Constants.pi
    k    : Thermal conductivity (material constant)
    L    : Length of cylinder
    log  : Modelica.Math.log;
    r_out: Outer radius of cylinder
    r_in : Inner radius of cylinder
    </pre>
    </li>
</li>
</ul>
<pre>
    Typical values for k at 20 degC in W/(m.K):
      aluminium   220
      concrete      1
      copper      384
      iron         74
      silver      407
      steel        45 .. 15 (V2A)
      wood         0.1 ... 0.2
</pre>
</HTML>
"));
    Iface.RealInput Position "Position of roller" 
                           annotation (Placement(transformation(
          origin={0,80},
          extent={{-10,-10},{10,10}},
          rotation=270)));
  equation
    //  if noEvent(Position < 1e-10) then
    //    dT = 0;
    //  else
    Position*Q_flow = G*dT;
    //  end if;
  end OverNightHeatInsolation;

  model FurnitureModel "Model of the furniture in a room"
    extends Iface.BlockIcon;
    parameter SI.Area FlArea "Area of the floor";
    parameter SI.Area WallsArea[:] "Areas of all the walls";
    parameter Real FloorPortion;
    parameter Real WallsPortion;
    parameter SI.Thickness d;
    parameter SI.SpecificHeatCapacity c;
    parameter SI.Density rho;
    parameter Boolean steadyStateStart=false;
    final parameter SI.Area A_furnit=FloorPortion*2*FlArea + WallsPortion*sum(
        WallsArea);
  public
    Modelica.Thermal.HeatTransfer.HeatCapacitor FurnitureCapacity(
                                       C=A_furnit*d*c*rho,final
        steadyStateStart=steadyStateStart);
    annotation (Diagram(graphics),
                         Icon(graphics={
          Polygon(
            points={{0,-24},{0,-82},{4,-80},{4,-22},{0,-24}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{0,-82},{-4,-80},{-4,-22},{0,-24},{0,-82}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-56,-4},{-56,-62},{-52,-60},{-52,-2},{-56,-4}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-56,-62},{-60,-60},{-60,-2},{-56,-4},{-56,-62}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{56,-4},{56,-64},{60,-62},{60,-4},{56,-4}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{56,-64},{52,-62},{52,-4},{56,-4},{56,-64}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-60,0},{0,20},{60,0},{0,-20},{-60,0}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={173,87,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-60,0},{-60,-4},{0,-24},{0,-20},{-60,0}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Forward),
          Polygon(
            points={{0,84},{0,16},{4,18},{4,86},{0,84}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{0,16},{-4,18},{-4,86},{0,84},{0,16}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{56,66},{56,-2},{60,0},{60,68},{56,66}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{56,-2},{52,0},{52,68},{56,66},{56,-2}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{0,-24},{60,-4},{60,0},{0,-20},{0,-24}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{56,46},{-4,66},{-4,86},{56,66},{56,46}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-4,86},{0,88},{60,68},{56,66},{-4,86}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={173,87,0},
            fillPattern=FillPattern.Solid)}));
    HT.Interfaces.HeatPort_a port 
      annotation (Placement(transformation(
          origin={0,-110},
          extent={{-10,-10},{10,10}},
          rotation=270)));
  equation
    connect(FurnitureCapacity.port, port);
  end FurnitureModel;

  model InteriorModel "Model of air mass and furniture of the interior"
    parameter Integer NV=4 "Number of vertical surfaces";
    parameter Integer NH=2 "Number of horizontal surfaces";
    parameter SI.Area VArea[NV] "Array of vertical surface areas";
    parameter SI.Area HArea[NH] "Array of horizontal surface areas";
    parameter SI.Height VHeights[NV] "Height of walls";
    parameter SI.Volume AirVolume "Volume of the air";
    parameter Real c_air(final quantity="SpecificVolumeHeatCapacity", final unit
        =    "J/K.m3") "Specific heat capacity of the air";
    parameter Real FloorPortion "Poriton of floor area in furniture model area";
    parameter Real WallsPortion "Portion of walls area in furniture model area";
    parameter SI.Thickness d_furn "Thickness of furniture model";
    parameter SI.SpecificHeatCapacity c_furn "Specific heat of furniture";
    parameter SI.Density rho_furn "Density of furniture material";
    parameter SI.Temp_C T_init=18 "Initial temperature of interior";
    parameter Boolean steadyStateStart=false;
    Modelica.Thermal.HeatTransfer.HeatCapacitor AirHeatCapacity(
      final C=c_air*AirVolume,
      T(final start=SI.Conversions.from_degC(T_init)),
      steadyStateStart=steadyStateStart) "Heat Capacity of the air inside" 
      annotation (Placement(transformation(extent={{-10,-8},{10,12}}, rotation=
              0)));
    FurnitureModel Furniture(
      final FlArea=sum(HArea),
      final WallsArea=VArea,
      final FloorPortion=FloorPortion,
      final WallsPortion=WallsPortion,
      final d=d_furn,
      final c=c_furn,
      final rho=rho_furn,
      FurnitureCapacity(T(final start=SI.Conversions.from_degC(T_init))),
      steadyStateStart=steadyStateStart)
      "Model of the furniture inside the room" 
      annotation (Placement(transformation(extent={{38,6},{58,26}}, rotation=0)));
    ThermalInteractions.ConvectionWallAir WallsAirConv[NV](area=VArea, height=
         VHeights) "Convection of wall air" 
      annotation (Placement(transformation(extent={{-52,-18},{-32,2}}, rotation
            =0)));
    ThermalInteractions.ConvectionFloorAir HorizSurfAirConv[NH](area=HArea)
      "Convection of horizontal surfaces" 
      annotation (Placement(transformation(extent={{-62,-70},{-42,-50}},
            rotation=0)));
    annotation (
      Diagram(graphics),
      Icon(graphics={
          Polygon(
            points={{0,80},{100,40},{100,-60},{0,-20},{0,80}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={85,170,255},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-100,-60},{-100,40},{0,80},{0,-20},{-100,-60}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={170,213,255},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{0,-100},{100,-60},{0,-20},{-100,-60},{0,-100}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={0,128,255},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{26,-28},{26,-68},{30,-66},{30,-26},{26,-28}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{26,-68},{22,-66},{22,-26},{26,-28},{26,-68}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-12,-18},{-12,-56},{-8,-54},{-8,-16},{-12,-18}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-12,-56},{-16,-54},{-16,-16},{-12,-18},{-12,-56}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{68,-14},{68,-54},{72,-52},{72,-12},{68,-14}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{68,-54},{64,-52},{64,-14},{68,-14},{68,-54}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-16,-14},{28,2},{72,-12},{26,-28},{-16,-14}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={173,87,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-16,-14},{-16,-18},{26,-32},{26,-28},{-16,-14}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Forward),
          Polygon(
            points={{28,48},{28,-2},{32,0},{32,50},{28,48}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{28,-2},{24,0},{24,50},{28,48},{28,-2}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{68,36},{68,-14},{72,-12},{72,38},{68,36}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{68,-14},{64,-12},{64,40},{68,38},{68,-14}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{26,-32},{72,-16},{72,-12},{26,-28},{26,-32}},
            lineColor={0,0,0},
            fillColor={76,38,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{68,16},{24,32},{24,50},{68,36},{68,16}},
            lineColor={0,0,0},
            fillColor={130,65,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{24,50},{28,52},{72,38},{68,36},{24,50}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={173,87,0},
            fillPattern=FillPattern.Solid),
          Line(
            points={{-100,38},{-56,38}},
            color={255,0,0},
            arrow={Arrow.None,Arrow.Filled}),
          Line(
            points={{-100,6},{-56,6}},
            color={255,0,0},
            arrow={Arrow.None,Arrow.Filled}),
          Line(
            points={{-100,-24},{-56,-24}},
            color={255,0,0},
            arrow={Arrow.None,Arrow.Filled}),
          Line(
            points={{-100,-56},{-56,-56}},
            color={255,0,0},
            arrow={Arrow.None,Arrow.Filled})}),
      Documentation(info="<html>
Model of a interior of the room. It consists of two parts: model of the inside air and model of the furniture. The only interaction of wall surrouding this interior with air inside is through convection.
</html>"));
    HT.Interfaces.HeatPort_a Walls[size(WallsAirConv, 1)] "Vertical surfaces" 
      annotation (Placement(transformation(extent={{-120,30},{-100,50}},
            rotation=0)));
    HT.Interfaces.HeatPort_a FloorCeil[size(HorizSurfAirConv, 1)]
      "Horizontal surfaces" annotation (Placement(transformation(extent={{-120,
              -70},{-100,-50}}, rotation=0)));
    ThermalInteractions.ConvectionWallAir FurnitureAirConv(area=Furniture.
          A_furnit, height=WallsPortion*sum(VHeights)/size(VHeights, 1)) 
      annotation (Placement(transformation(extent={{34,-18},{14,2}}, rotation=0)));
    HT.Interfaces.HeatPort_b AirPort annotation (Placement(transformation(
            extent={{100,-70},{120,-50}}, rotation=0)));
    HT.Interfaces.HeatPort_b FurniturePort 
      annotation (Placement(transformation(extent={{100,28},{122,48}}, rotation
            =0)));
  equation
    connect(WallsAirConv.Solid, Walls) annotation (Line(points={{-53,-8},{-70,
            -8},{-70,40},{-110,40}}, color={191,0,0}));
    connect(HorizSurfAirConv.Solid, FloorCeil) annotation (Line(points={{-63,
            -60},{-110,-60}}, color={191,0,0}));
    for i in 1:NV loop
      connect(WallsAirConv[i].Air, AirHeatCapacity.port) annotation (Line(
            points={{-31,-8},{-15.5,-8},{-15.5,-8},{6.10623e-16,-8}}, color={
              191,0,0}));
    end for;
    for i in 1:NH loop
      connect(HorizSurfAirConv[i].Air, AirHeatCapacity.port) annotation (Line(
            points={{-41,-60},{6.10623e-16,-60},{6.10623e-16,-8}}, color={191,0,
              0}));
    end for;
    connect(FurnitureAirConv.Air, AirHeatCapacity.port) annotation (Line(points
          ={{13,-8},{6.5,-8},{6.5,-8},{6.10623e-16,-8}}, color={191,0,0}));
    connect(FurnitureAirConv.Solid, Furniture.port) annotation (Line(points={{
            35,-8},{48,-8},{48,5}}, color={191,0,0}));
    connect(AirHeatCapacity.port, AirPort) annotation (Line(points={{
            6.10623e-16,-8},{6.10623e-16,-60},{110,-60}}, color={191,0,0}));
    connect(Furniture.port, FurniturePort) annotation (Line(points={{48,5},{48,
            -8},{80,-8},{80,38},{111,38}}, color={191,0,0}));
  end InteriorModel;

  model Window "Model of double-glazed window"
    //extends Iface.BlockIcon;
    import SolarHouse.RadiationTransfer.*;
    parameter SI.Area A "Surface of layer";
    parameter SolarHouse.Interfaces.SpecificPlaneHeatCapacity C=
                                          1260 "Capacity of a single pane";
    parameter Real U(
      final quantity="SpecificThermalConductivity",
      final unit="W/(m2.K)") = 2.0 "Thermal conductivity per square meter";
    parameter SI.Position ShapePoly[3, :] "Geometry of the window";
    parameter SI.Position WallOnLeft[3, :] "Wall on left of the window";
    parameter SI.Position WallOpposite[3, :] "Wall opposite to the window";
    parameter SI.Position WallOnRight[3, :] "Wall on right of the window";
    parameter SI.Position WallBeneath[3, :] "Wall beneath eg. floor";
    final parameter Real RefCoeff=1 - (Glazing.TR_DIF + Glazing.AB_DIF_O +
        Glazing.AB_DIF_I) "Reflection coefficient";
    parameter Integer n=5 "Number of reflections considered";
    /*
  parameter Integer NWalls = 6 "Number of walls in a room";
  parameter Integer NWins = 1 
    "Number of windows in a room (including this one)";
  parameter SI.Area WallAreas[NWalls] "Area of walls in the room";
  parameter SI.Area WinAreas[NWins] "Area of windows in the room";
  parameter Integer SRC(final min = 0, final max = NWins) 
    "Index of this window in upper table of areas";
  parameter Integer WinOnWall[NWins] 
    "Lookup-table of indices which windows are on what walls";
*/
    parameter Real sfactor[Reflector.NWalls, Reflector.NWalls]
      "Matrix of shape factors";
    parameter SI.Temperature Tout_start(start=from_degC(20));
    parameter SI.Temperature Tin_start(start=from_degC(20));
    parameter Boolean steadyStateStart=false;
    HT.Interfaces.HeatPort_a Outside annotation (Placement(transformation(
            extent={{-120,30},{-100,50}}, rotation=0)));
    annotation (
      Diagram(graphics),
      DymolaStoredErrors,
      Icon(graphics={
          Rectangle(
            extent={{-100,100},{100,-100}},
            lineColor={0,0,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-68,80},{68,-80}},
            lineColor={0,0,0},
            fillColor={170,113,57},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-58,70},{-4,2}},
            lineColor={0,0,0},
            fillColor={85,170,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{4,70},{58,2}},
            lineColor={0,0,0},
            fillColor={85,170,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-58,-4},{-4,-70}},
            lineColor={0,0,0},
            fillColor={85,170,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{4,-4},{58,-70}},
            lineColor={0,0,0},
            fillColor={85,170,255},
            fillPattern=FillPattern.Solid),
          Text(
            extent={{-164,-118},{164,-148}},
            lineColor={0,0,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid,
            textString=
                 "%name")}));
    HT.Interfaces.HeatPort_b Inside annotation (Placement(transformation(extent
            ={{100,30},{120,50}}, rotation=0)));
    SolarHouse.RadiationTransfer.LightTransmission Glazing(
      area=A,
      n={1.52,1.52},
      PP={0.8,0.8},
      normal=Geometry.Normal(ShapePoly)) 
      annotation (Placement(transformation(extent={{-60,-44},{-38,-22}},
            rotation=0)));
    Interfaces.DirLightIn Sunlight annotation (Placement(transformation(extent=
              {{-120,-70},{-100,-50}}, rotation=0)));
    LightDirection SunRayDirection annotation (Placement(transformation(extent=
              {{-60,-98},{-40,-78}}, rotation=0)));
    HT.Interfaces.HeatPort_b ToWalls[Reflector.nout] 
      annotation (Placement(transformation(extent={{100,-70},{120,-50}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.HeatCapacitor OuterPane(
      final C=C*A,
      T(start=Tout_start),
      steadyStateStart=steadyStateStart) 
      annotation (Placement(transformation(extent={{-60,30},{-40,50}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.HeatCapacitor InnerPane(
      final C=C*A,
      T(start=Tin_start),
      steadyStateStart=steadyStateStart) annotation (Placement(transformation(
            extent={{60,30},{80,50}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.ThermalConductor AirInside(
                                  G=U*A) annotation (Placement(transformation(
            extent={{0,42},{20,62}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation PaneRadiation(
                                   Gr=A) annotation (Placement(transformation(
            extent={{-2,10},{22,30}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.Sources.PrescribedHeatFlow OPAbsorptedLight 
      annotation (Placement(transformation(extent={{-16,-14},{-42,8}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.Sources.PrescribedHeatFlow IPAbsorptedLight 
      annotation (Placement(transformation(extent={{26,-14},{52,8}}, rotation=0)));
    OverNightHeatInsolation NightIsolation(G=1/1.22) 
      annotation (Placement(transformation(extent={{-86,20},{-66,40}}, rotation
            =0)));
    replaceable SunlightBoxDistrib DirectIlluminated(
      Window=ShapePoly,
      Left=WallOnLeft,
      Opposite=WallOpposite,
      Right=WallOnRight,
      Beneath=WallBeneath) annotation (Placement(transformation(extent={{-20,
              -96},{2,-74}}, rotation=0)));
    SunlightReflections Reflector(nillum=size(DirectIlluminated.Distribution,
          1), sfactor=sfactor) "Calculates reflections inside the room" 
      annotation (Placement(transformation(extent={{30,-58},{52,-36}}, rotation
            =0)));
    output SI.HeatFlux DirTrans;
    Modelica.Thermal.HeatTransfer.Sources.PrescribedHeatFlow HeatToWalls[
                                      Reflector.nout] 
      annotation (Placement(transformation(extent={{66,-64},{86,-44}}, rotation
            =0)));
    Iface.RealInput RollerPosition 
      annotation (Placement(transformation(
          origin={0,110},
          extent={{-10,-10},{10,10}},
          rotation=270)));
  equation
    DirTrans*A = Glazing.Transmitted.q_dir;
    connect(OuterPane.port, AirInside.port_a) annotation (Line(points={{-50,30},
            {-16,30},{-16,52},{-5.55112e-16,52}}, color={191,0,0}));
    connect(AirInside.port_b, InnerPane.port) annotation (Line(points={{20,52},
            {36,52},{36,30},{70,30}}, color={191,0,0}));
    connect(PaneRadiation.port_a, OuterPane.port) annotation (Line(points={{-2,
            20},{-16,20},{-16,30},{-50,30}}, color={191,0,0}));
    connect(PaneRadiation.port_b, InnerPane.port) annotation (Line(points={{22,
            20},{36,20},{36,30},{70,30}}, color={191,0,0}));
    connect(InnerPane.port, Inside) annotation (Line(points={{70,30},{90,30},{
            90,40},{110,40}}, color={191,0,0}));
    connect(Glazing.Absorpted1, OPAbsorptedLight.Q_flow) annotation (Line(
          points={{-36.9,-26.4},{-6,-26.4},{-6,-3},{-16,-3}}, color={0,0,127}));
    connect(Glazing.Absorpted2, IPAbsorptedLight.Q_flow) annotation (Line(
          points={{-36.9,-33},{12,-33},{12,-3},{26,-3}}, color={0,0,127}));
    connect(OPAbsorptedLight.port, OuterPane.port) annotation (Line(points={{
            -42,-3},{-50,-3},{-50,30}}, color={191,0,0}));
    connect(IPAbsorptedLight.port, InnerPane.port) annotation (Line(points={{52,
            -3},{70,-3},{70,30}}, color={191,0,0}));
    connect(Outside, NightIsolation.port_a) annotation (Line(points={{-110,40},
            {-94,40},{-94,30},{-86,30}}, color={191,0,0}));
    connect(NightIsolation.port_b, OuterPane.port) annotation (Line(points={{
            -66,30},{-50,30}}, color={191,0,0}));
    connect(SunRayDirection.dir, DirectIlluminated.IncidenceRay) annotation (Line(
          points={{-39,-88},{-30,-88},{-30,-89.4},{-21.1,-89.4}}, color={0,0,
            127}));
    connect(RollerPosition, NightIsolation.Position) annotation (Line(points={{
            5.55112e-16,110},{5.55112e-16,70},{-76,70},{-76,38}}, color={0,0,
            127}));
    connect(RollerPosition, Glazing.RollerPos) annotation (Line(points={{
            5.55112e-16,110},{5.55112e-16,70},{-76,70},{-76,-26.4},{-61.1,-26.4}}, 
          color={0,0,127}));
    connect(RollerPosition, DirectIlluminated.RollerPos) annotation (Line(
          points={{5.55112e-16,110},{5.55112e-16,70},{-76,70},{-76,-74},{-30,
            -74},{-30,-80.6},{-21.1,-80.6}}, color={0,0,127}));
    connect(DirectIlluminated.Distribution, Reflector.dist) annotation (Line(
          points={{3.1,-85},{12,-85},{12,-51.4},{28.9,-51.4}}, color={0,0,127}));
    connect(Reflector.Reflected, Glazing.Inside) annotation (Line(points={{41,
            -59.1},{41,-64},{-66,-64},{-66,-39.6},{-61.1,-39.6}}, color={0,0,
            127}));
    connect(Sunlight, Glazing.Outside) annotation (Line(points={{-110,-60},{-86,
            -60},{-86,-33},{-61.1,-33}}, color={161,161,54}));
    connect(Sunlight, SunRayDirection.LightConn) annotation (Line(points={{-110,
            -60},{-86,-60},{-86,-88},{-61,-88}}, color={161,161,54}));
    connect(Glazing.Transmitted, Reflector.Light) annotation (Line(points={{
            -36.9,-39.6},{12,-39.6},{12,-42.6},{28.9,-42.6}}, color={255,170,85}));
    connect(Reflector.HeatDistrib, HeatToWalls.Q_flow) annotation (Line(points=
            {{53.1,-47},{59.55,-47},{59.55,-54},{66,-54}}, color={0,0,127}));
    connect(HeatToWalls.port, ToWalls) annotation (Line(points={{86,-54},{96,
            -54},{96,-60},{110,-60}}, color={191,0,0}));
  end Window;

  partial model OneWindowRoom "Generic box-shaped room model"
    extends Icons.RoomIcon;
    parameter SI.Temperature T_indoor_start(start=18);
    parameter SI.Temperature T_outdoor_start(start=18);
    parameter Boolean steadyStateStart=false;
    replaceable BuildingParts.Wall5 Ceiling(
      absorpt=0.70,
      Tout_start=T_outdoor_start,
      Tin_start=T_indoor_start,
      steadyStateStart=steadyStateStart) 
      annotation (Placement(transformation(
          origin={0,70},
          extent={{10,-10},{-10,10}},
          rotation=270)));
    replaceable BuildingParts.Wall5 WinWall(
      absorpt=0.70,
      Layer2(
        lambda=0.210,
        rho=900,
        c=.210*3600,
        L=0.012),
      Layer3(
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer4(
        lambda=0.041,
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer5(
        lambda=0.210,
        rho=900,
        c=0.210*3600,
        L=0.012),
      Tout_start=T_outdoor_start,
      Tin_start=T_indoor_start,
      steadyStateStart=steadyStateStart) 
      annotation (Placement(transformation(extent={{-60,10},{-80,30}}, rotation
            =0)));
    replaceable BuildingParts.Wall5 WallOnRight(
      absorpt=0.70,
      Layer2(
        lambda=0.210,
        rho=900,
        c=.210*3600,
        L=0.012),
      Layer3(
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer4(
        lambda=0.041,
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer5(
        lambda=0.210,
        rho=900,
        c=0.210*3600,
        L=0.012),
      Tout_start=T_outdoor_start,
      Tin_start=T_indoor_start,
      steadyStateStart=steadyStateStart) 
      annotation (Placement(transformation(extent={{-60,-20},{-80,0}}, rotation
            =0)));
    replaceable BuildingParts.Wall5 OppositeWall(
      absorpt=0.70,
      Tout_start=T_outdoor_start,
      Tin_start=T_indoor_start,
      steadyStateStart=steadyStateStart) annotation (Placement(transformation(
            extent={{60,40},{80,60}}, rotation=0)));
    replaceable BuildingParts.Wall5 WallOnLeft(
      absorpt=0.70,
      Layer2(
        lambda=0.210,
        rho=900,
        c=.210*3600,
        L=0.012),
      Layer3(
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer4(
        lambda=0.041,
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer5(
        lambda=0.210,
        rho=900,
        c=0.210*3600,
        L=0.012),
      Tout_start=T_outdoor_start,
      Tin_start=T_indoor_start,
      steadyStateStart=steadyStateStart) annotation (Placement(transformation(
            extent={{60,-20},{80,0}}, rotation=0)));
    replaceable BuildingParts.Wall5 Floor(
      absorpt=0.70,
      Layer2(
        lambda=0.210,
        rho=900,
        c=.210*3600,
        L=0.012),
      Layer3(
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer4(
        lambda=0.041,
        rho=40,
        c=0.230*3600,
        L=0.050),
      Layer5(
        rho=900,
        lambda=0.410,
        c=0.670*3600,
        L=0.025),
      Tout_start=T_outdoor_start,
      Tin_start=T_indoor_start,
      steadyStateStart=steadyStateStart) 
      annotation (Placement(transformation(
          origin={0,-26},
          extent={{10,-10},{-10,10}},
          rotation=90)));
    replaceable BuildingParts.Window Window(
      U=7.7,
      Reflector(
        SRC=1,
        WinOnWall={1},
        Illumin={2,3,4,6},
        NWalls=6,
        NWins=1,
        RWall={1 - WinWall.absorpt,1 - WallOnRight.absorpt,1 - OppositeWall.
            absorpt,1 - WallOnLeft.absorpt,1 - Floor.absorpt,1 - Ceiling.
            absorpt},
        Walls={WinWall.Area,WallOnRight.Area,OppositeWall.Area,WallOnLeft.
            Area,Floor.Area,Ceiling.Area},
        Windows={Window.A},
        RWin={Window.RefCoeff},
        nillum=4),
      Tout_start=T_outdoor_start,
      Tin_start=T_indoor_start,
      steadyStateStart=steadyStateStart) 
      annotation (Placement(transformation(extent={{-80,40},{-60,60}}, rotation
            =0)));
    SolarHouse.ThermalInteractions.NaturalVentilation Ventilator(
                                                  n=1, V=Interior.AirVolume) 
      annotation (Placement(transformation(extent={{50,14},{70,34}}, rotation=0)));
    ThermalInteractions.RadiationExchange6win1 RadiationExchange(
      WallA={WinWall.Area,WallOnRight.Area,OppositeWall.Area,WallOnLeft.Area,
          Floor.Area,Ceiling.Area},
      WinA=Window.A,
      WinEm=Window.RefCoeff,
      WinIdx=1,
      WallEm={WinWall.absorpt,WallOnRight.absorpt,OppositeWall.absorpt,
          WallOnLeft.absorpt,Floor.absorpt,Ceiling.absorpt}) 
      annotation (Placement(transformation(extent={{-30,26},{-10,46}}, rotation
            =0)));
    BuildingParts.InteriorModel Interior(
      c_air=1410,
      FloorPortion=0.1,
      WallsPortion=0.1,
      d_furn=0.001,
      c_furn=0.13*3600,
      rho_furn=7800,
      HArea={Floor.Area,Ceiling.Area},
      NV=5,
      VArea={WinWall.Area,WallOnRight.Area,OppositeWall.Area,WallOnLeft.Area,
          Window.A},
      VHeights=ones(5),
      T_init=to_degC(T_indoor_start),
      steadyStateStart=steadyStateStart) "Model of the interior of the room" 
      annotation (Placement(transformation(extent={{6,6},{26,26}}, rotation=0)));
    annotation (Diagram(graphics),
                         Icon(graphics));
    HT.Interfaces.HeatPort_b WinWallPort "Wall with the window in it" 
      annotation (Placement(transformation(extent={{-120,-50},{-100,-30}},
            rotation=0)));
    HT.Interfaces.HeatPort_b WallOnRightPort "Wall on the right of the window" 
      annotation (Placement(transformation(extent={{-70,-90},{-90,-70}},
            rotation=0)));
    HT.Interfaces.HeatPort_b WallOppositePort "Wall opposite to the window" 
      annotation (Placement(transformation(extent={{100,30},{120,50}}, rotation
            =0)));
    HT.Interfaces.HeatPort_b WallOnLeftPort "Wall on the left of the window" 
      annotation (Placement(transformation(extent={{100,-50},{120,-30}},
            rotation=0)));
    HT.Interfaces.HeatPort_b FloorPort 
      annotation (Placement(transformation(
          origin={0,-110},
          extent={{-10,-10},{10,10}},
          rotation=90)));
    HT.Interfaces.HeatPort_b CeilingPort 
      annotation (Placement(transformation(
          origin={0,110},
          extent={{-10,-10},{10,10}},
          rotation=270)));
    HT.Interfaces.HeatPort_a WinPort "Window" 
      annotation (Placement(transformation(extent={{-120,-10},{-100,10}},
            rotation=0)));
    Interfaces.DirLightIn Sunlight 
      annotation (Placement(transformation(
          origin={-40,110},
          extent={{-10,-10},{10,10}},
          rotation=270)));
    Iface.RealInput RollerPosition annotation (Placement(transformation(extent=
              {{-120,30},{-100,50}}, rotation=0)));
    input HT.Interfaces.HeatPort_a OutsideAir 
      annotation (Placement(transformation(extent={{70,70},{90,90}}, rotation=0)));
  equation
    connect(WinWall.InnerSide, Interior.Walls[1]) annotation (Line(points={{-59,
            20},{5,20}}, color={191,0,0}));
    connect(WallOnRight.InnerSide, Interior.Walls[2]) annotation (Line(points={
            {-59,-10},{-48,-10},{-48,20},{5,20}}, color={191,0,0}));
    connect(OppositeWall.InnerSide, Interior.Walls[3]) annotation (Line(points=
            {{59,50},{-48,50},{-48,20},{5,20}}, color={191,0,0}));
    connect(WallOnLeft.InnerSide, Interior.Walls[4]) annotation (Line(points={{
            59,-10},{-48,-10},{-48,20},{5,20}}, color={191,0,0}));
    connect(Window.Inside, Interior.Walls[5]) annotation (Line(points={{-59,54},
            {-48,54},{-48,20},{5,20}}, color={191,0,0}));
    connect(Interior.FloorCeil[1], Floor.InnerSide) annotation (Line(points={{5,
            10},{-1.73933e-16,10},{-1.73933e-16,-15}}, color={191,0,0}));
    connect(Interior.FloorCeil[2], Ceiling.InnerSide) annotation (Line(points={
            {5,10},{6.29108e-17,10},{6.29108e-17,59}}, color={191,0,0}));
    connect(WinWall.InnerSide, RadiationExchange.wall1) annotation (Line(points
          ={{-59,20},{-48,20},{-48,41},{-29,41}}, color={191,0,0}));
    connect(WallOnRight.InnerSide, RadiationExchange.wall2) annotation (Line(
          points={{-59,-10},{-48,-10},{-48,31},{-29,31}}, color={191,0,0}));
    connect(OppositeWall.InnerSide, RadiationExchange.wall5) annotation (Line(
          points={{59,50},{0,50},{0,41},{-11,41}}, color={191,0,0}));
    connect(WallOnLeft.InnerSide, RadiationExchange.wall4) annotation (Line(
          points={{59,-10},{0,-10},{0,31},{-11,31}}, color={191,0,0}));
    connect(Floor.InnerSide, RadiationExchange.wall3) annotation (Line(points={
            {-1.73933e-16,-15},{-1.73933e-16,-10},{-20,-10},{-20,27}}, color={
            191,0,0}));
    connect(Ceiling.InnerSide, RadiationExchange.wall6) annotation (Line(points
          ={{6.29108e-17,59},{6.29108e-17,50},{-20,50},{-20,45}}, color={191,0,
            0}));
    connect(Window.ToWalls[1], WinWall.InnerSide) annotation (Line(points={{-59,
            44},{-48,44},{-48,20},{-59,20}}, color={191,0,0}));
    connect(Window.ToWalls[2], WallOnRight.InnerSide) annotation (Line(points={
            {-59,44},{-48,44},{-48,-10},{-59,-10}}, color={191,0,0}));
    connect(Window.ToWalls[3], OppositeWall.InnerSide) annotation (Line(points=
            {{-59,44},{-48,44},{-48,50},{59,50}}, color={191,0,0}));
    connect(Window.ToWalls[4], WallOnLeft.InnerSide) annotation (Line(points={{
            -59,44},{-48,44},{-48,-10},{59,-10}}, color={191,0,0}));
    connect(Window.ToWalls[5], Floor.InnerSide) annotation (Line(points={{-59,
            44},{-48,44},{-48,-10},{-1.73933e-16,-10},{-1.73933e-16,-15}},
          color={191,0,0}));
    connect(Window.ToWalls[6], Ceiling.InnerSide) annotation (Line(points={{-59,
            44},{-48,44},{-48,50},{6.29108e-17,50},{6.29108e-17,59}}, color={
            191,0,0}));
    connect(RadiationExchange.win, Window.Inside) annotation (Line(points={{-29,
            36},{-48,36},{-48,54},{-59,54}}, color={191,0,0}));
    connect(WinWall.OuterSide, WinWallPort) annotation (Line(points={{-81,20},{
            -95.5,20},{-95.5,-40},{-110,-40}}, color={191,0,0}));
    connect(WallOnRight.OuterSide, WallOnRightPort) annotation (Line(points={{
            -81,-10},{-88,-10},{-88,-80},{-80,-80}}, color={191,0,0}));
    connect(OppositeWall.OuterSide, WallOppositePort) annotation (Line(points={
            {81,50},{95.5,50},{95.5,40},{110,40}}, color={191,0,0}));
    connect(WallOnLeft.OuterSide, WallOnLeftPort) annotation (Line(points={{81,
            -10},{95.5,-10},{95.5,-40},{110,-40}}, color={191,0,0}));
    connect(Floor.OuterSide, FloorPort) annotation (Line(points={{1.17313e-15,
            -37},{1.17313e-15,-110},{5.55112e-16,-110}}, color={191,0,0}));
    connect(Ceiling.OuterSide, CeilingPort) annotation (Line(points={{
            -1.28416e-15,81},{-1.28416e-15,80.5},{5.55112e-16,80.5},{
            5.55112e-16,110}}, color={191,0,0}));
    connect(Window.Outside, WinPort) annotation (Line(points={{-81,54},{-90,54},
            {-90,5.55112e-16},{-110,5.55112e-16}}, color={191,0,0}));
    connect(Window.Sunlight, Sunlight) annotation (Line(points={{-81,44},{-86,
            44},{-86,68},{-40,68},{-40,110}}, color={161,161,54}));
    connect(Window.RollerPosition, RollerPosition) annotation (Line(points={{
            -70,61},{-70,64},{-94,64},{-94,40},{-110,40}}, color={0,0,127}));
    connect(Interior.AirPort, Ventilator.InsideAir) annotation (Line(points={{
            27,10},{38,10},{38,24},{49,24}}, color={191,0,0}));
    connect(Ventilator.OutsideAir, OutsideAir) annotation (Line(points={{71,24},
            {90,24},{90,80},{80,80}}, color={191,0,0}));
  end OneWindowRoom;
end BuildingParts;
