within SolarHouse;
package Examples
  import Modelica.SIunits.Conversions.*;
  
  package TestPackage
    model TestModel
      Modelica.Thermal.HeatTransfer.FixedTemperature outside(T=300);
      Modelica.Thermal.HeatTransfer.FixedHeatFlow inside(Q_flow=200);
      BuildingParts.Wall3 stena(Area=2,absorpt=1,Layer1(L=0.1),Layer2(L=0.2),Layer3(L=0.05));
      BuildingParts.Wall3 stena2(Area=2,absorpt=1,Layer1(L=0.1),Layer2(L=0.2),Layer3(L=0.05));
    equation
      connect(inside.port,stena.InnerSide);
      connect(stena.OuterSide,stena2.InnerSide);
      connect(stena2.OuterSide,outside.port);
    end TestModel;
  end TestPackage;
  
  package Kamra "Model testne celice"
    package Components
      record HouseGeometry "Geometry of the house walls - x points to north"
        extends Modelica.Icons.Record;
      protected
        type Angle_deg = SI.Conversions.NonSIunits.Angle_deg;
      public
        parameter SI.Breadth ww=3.0 "wall width";
        parameter SI.Length wl=3.0 "wall length";
        parameter SI.Height wh=3.0 "wall height";
        parameter SI.Position winx=0.25 "window x position";
        parameter SI.Position winy=0.25 "window y position";
        parameter SI.Breadth winw=2.5 "window width";
        parameter SI.Height winh=2.0 "window height";
        // rotations
        parameter Angle_deg phi=0 "rotation around z-axis";
        parameter Angle_deg theta=0 "rotation around x-axis";
        parameter Angle_deg psi=0 "rotation around 'new' z-axis";
        //
      protected
        parameter Real T[4, 4]=Geometry.EulerAngles(
                  SI.Conversions.from_deg(phi),
                  SI.Conversions.from_deg(theta),
                  SI.Conversions.from_deg(psi));
        //
      public
        parameter SI.PathLength SouthW[3, 4]=Geometry.PolyToRelCoord(T, [0, 0,
            0, 0; 0, ww, ww, 0; 0, 0, wh, wh]) "south wall";
        final parameter SI.Area SWArea=ww*wh "south wall area";
        parameter SI.PathLength WestW[3, 4]=Geometry.PolyToRelCoord(T, [0, wl,
            wl, 0; ww, ww, ww, ww; 0, 0, wh, wh]) "west wall";
        final parameter SI.Area WWArea=wl*wh "west wall area";
        parameter SI.PathLength NorthW[3, 4]=Geometry.PolyToRelCoord(T, [wl, wl,
            wl, wl; 0, 0, ww, ww; 0, wh, wh, 0]) "north wall";
        final parameter SI.Area NWArea=ww*wh "north wall area";
        parameter SI.PathLength EastW[3, 4]=Geometry.PolyToRelCoord(T, [0, 0,
            wl, wl; 0, 0, 0, 0; 0, wh, wh, 0]) "east wall";
        final parameter SI.Area EWArea=wl*wh "east wall area";
        parameter SI.PathLength Floor[3, 4]=Geometry.PolyToRelCoord(T, [0, wl,
            wl, 0; 0, 0, ww, ww; 0, 0, 0, 0]) "floor";
        final parameter SI.Area FlArea=wl*ww "floor area";
        parameter SI.PathLength Ceil[3, 4]=Geometry.PolyToRelCoord(T, [0, 0, wl,
            wl; 0, ww, ww, 0; wh, wh, wh, wh]) "ceiling";
        final parameter SI.Area CeArea=wl*ww "ceiling area";
        // window in south wall
        parameter SI.PathLength SWin[3, 4]=Geometry.PolyToRelCoord(T, [0, 0, 0,
            0; ww - winx - winw, ww - winx, ww - winx, ww - winx - winw; winy,
            winy, winy + winh, winy + winh]) "south window";
        parameter SI.Area SWinA=winw*winh "south window area";
        // room volume
        final parameter SI.Volume RoomVol=wl*ww*wh;
        // radiation shape factors
        constant Integer SW=1 "south wall";
        constant Integer EW=2 "east wall";
        constant Integer NW=3 "north wall";
        constant Integer WW=4 "west wall";
        constant Integer FL=5 "floor";
        constant Integer CE=6 "ceiling";
        parameter Real sfactor[6, 6]=ShapeFactorBox(
                  ww,
                  wl,
                  wh);
      protected
        function ShapeFactorBox
          input Real a;
          input Real b;
          input Real c;
          output Real f[6, 6];
        algorithm
          // initialize table
          f := zeros(6, 6);
          // south wall is source
          f[SW, EW] := Geometry.SFactorPerpendic(
                    b,
                    c,
                    a);
          f[SW, NW] := Geometry.SFactorParallel(
                    a,
                    c,
                    b);
          f[SW, WW] := f[SW, EW];
          f[SW, CE] := Geometry.SFactorPerpendic(
                    b,
                    a,
                    c);
          f[SW, FL] := f[SW, CE];
          // east wall is source
          f[EW, SW] := f[SW, EW]*b/a;
          f[EW, NW] := f[EW, SW];
          f[EW, WW] := Geometry.SFactorParallel(
                    b,
                    c,
                    a);
          f[EW, CE] := Geometry.SFactorPerpendic(
                    c,
                    b,
                    a);
          f[EW, FL] := f[EW, CE];
          // north wall
          f[NW, SW] := f[SW, NW];
          f[NW, EW] := f[SW, WW];
          f[NW, WW] := f[SW, EW];
          f[NW, CE] := f[SW, CE];
          f[NW, FL] := f[SW, FL];
          // west wall
          f[WW, SW] := f[EW, NW];
          f[WW, EW] := f[EW, WW];
          f[WW, NW] := f[EW, SW];
          f[WW, CE] := f[EW, CE];
          f[WW, FL] := f[EW, FL];
          // ceiling
          f[CE, SW] := f[SW, CE]*b/c;
          f[CE, EW] := f[EW, CE]*c/a;
          f[CE, NW] := f[CE, SW];
          f[CE, WW] := f[CE, EW];
          f[CE, FL] := Geometry.SFactorParallel(
                    a,
                    b,
                    c);
          // floor
          f[FL, SW] := f[CE, SW];
          f[FL, EW] := f[CE, EW];
          f[FL, NW] := f[CE, NW];
          f[FL, WW] := f[CE, WW];
          f[FL, CE] := f[CE, FL];
        end ShapeFactorBox;
      end HouseGeometry;

      model KamraBuilding "Model of a building"
        extends BuildingParts.OneWindowRoom(
          WinWall(Area=GeoData.SWArea),
          WallOnRight(Area=GeoData.EWArea),
          Floor(Area=GeoData.FlArea),
          WallOnLeft(Area=GeoData.WWArea),
          OppositeWall(
            Area=GeoData.NWArea,
            Layer2(
              lambda=0.210,
              rho=900,
              c=.210*3600,
              L=0.012),
            Layer3(
              rho=40,
              c=0.230*3600,
              L=0.050),
            Layer4(
              lambda=0.041,
              rho=40,
              c=0.230*3600,
              L=0.050),
            Layer5(
              lambda=0.210,
              rho=900,
              c=0.210*3600,
              L=0.012)),
          Ceiling(
            Area=GeoData.CeArea,
            Layer2(
              lambda=0.210,
              rho=900,
              c=0.210*3600,
              L=0.012),
            Layer3(
              lambda=0.410,
              rho=900,
              c=0.670*3600,
              L=0.025),
            Layer4(
              lambda=0.041,
              rho=40,
              c=0.230*3600,
              L=0.010),
            Layer5(
              lambda=0.041,
              rho=40,
              c=0.230*3600,
              L=0.010)),
          Interior(VHeights={GeoData.wh,GeoData.wh,GeoData.wh,GeoData.wh,
                GeoData.winh}, AirVolume=GeoData.RoomVol),
          RadiationExchange(SFactor=GeoData.sfactor),
          Window(
            A=GeoData.SWinA,
            ShapePoly=GeoData.SWin,
            WallOnLeft=GeoData.EastW,
            WallOpposite=GeoData.NorthW,
            WallOnRight=GeoData.EastW,
            WallBeneath=GeoData.Floor,
            sfactor=GeoData.sfactor));
        SolarHouse.Examples.Kamra.Components.HouseGeometry GeoData
          "Geometrical data of the building" 
          annotation (Placement(transformation(extent={{40,-100},{60,-80}},
                rotation=0)));
      end KamraBuilding;
    end Components;

    model KamraJeseni
      extends Icons.HouseIcon;
      import Modelica.SIunits.Conversions.*;
      annotation (
        Diagram(coordinateSystem(
            preserveAspectRatio=false,
            extent={{-100,-100},{100,100}},
            grid={2,2}), graphics),
        experiment(StopTime=432000, Algorithm="Dassl"),
        experimentSetupOutput);
      HT.Celsius.PrescribedTemperature AirOutside 
        annotation (Placement(transformation(
            origin={10,70},
            extent={{10,-10},{-10,10}},
            rotation=180)));
      Weather.SunRadiation Sun(start_time(
          hour=0,
          day=9,
          month=10)) annotation (Placement(transformation(extent={{-66,4},{-46,
                24}}, rotation=0)));
      ThermalInteractions.ConvectionWallOutside EConvOut(area=Room.
            WallOnRight.Area) "EastWall.Area" 
        annotation (Placement(transformation(extent={{-30,-86},{-50,-66}},
              rotation=0)));
      ThermalInteractions.ConvectionWallOutside NConvOut(area=Room.
            OppositeWall.Area) annotation (Placement(transformation(extent={{48,
                -6},{68,14}}, rotation=0)));
      ThermalInteractions.ConvectionWallOutside WConvOut(area=Room.WallOnLeft.
             Area) annotation (Placement(transformation(extent={{48,-44},{68,
                -24}}, rotation=0)));
      ThermalInteractions.ConvectionWallOutside CeilConvOut(area=Room.Ceiling.
             Area) annotation (Placement(transformation(extent={{48,20},{68,40}}, 
              rotation=0)));
      Modelica.Blocks.Sources.Constant Cloudiness(k=0) 
        annotation (Placement(transformation(extent={{-100,-20},{-80,0}},
              rotation=0)));
      Modelica.Blocks.Sources.Constant UpWind(k(unit="m/s") = 1) 
        annotation (Placement(transformation(extent={{100,40},{80,60}},
              rotation=0)));
      Modelica.Blocks.Sources.CombiTimeTable Measurements(
        tableOnFile=true,
        tableName="Jesen19971009",
        fileName="jesen.mat",
        extrapolation=Modelica.Blocks.Types.Extrapolation.HoldLastPoint,
        columns={2,4,5,3,6}) annotation (Placement(transformation(extent={{-100,
                40},{-80,60}}, rotation=0)));
      HT.Celsius.FixedTemperature EarthTemp(T=9.5) 
        annotation (Placement(transformation(extent={{60,-80},{40,-60}},
              rotation=0)));
      SolarHouse.Examples.Kamra.Components.KamraBuilding Room(
        GeoData(
          ww=1,
          wl=1,
          wh=1,
          winx=0.5,
          winy=0.5,
          winw=0.9,
          winh=0.9),
        WinWall(absorpt=0.45),
        WallOnRight(absorpt=0.45),
        Floor(absorpt=0.45),
        WallOnLeft(absorpt=0.45),
        OppositeWall(absorpt=0.45),
        Ceiling(absorpt=0.45),
        T_indoor_start=from_degC(15),
        T_outdoor_start=from_degC(10),
        steadyStateStart=true) annotation (Placement(transformation(extent={{-8,
                -40},{32,0}}, rotation=0)));
      ThermalInteractions.ConvectionWallOutside WinConv(area=Room.Window.A) 
        annotation (Placement(transformation(extent={{-30,-32},{-50,-12}},
              rotation=0)));
      ThermalInteractions.ConvectionWallOutside SConvOut(area=Room.WinWall.
            Area) annotation (Placement(transformation(extent={{-30,-60},{-50,
                -40}}, rotation=0)));
    protected
      HT.Interfaces.HeatPort_b port1 
        annotation (Placement(transformation(extent={{-61.5,-77.5},{-58.5,-74.5}}, 
              rotation=0)));
    equation
      connect(Measurements.y[1], Sun.q_dir) annotation (Line(points={{-79,50},{
              -70,50},{-70,19},{-67,19}}, color={0,0,127}));
      connect(UpWind.y, EConvOut.Wind) annotation (Line(points={{79,50},{-40,50},
              {-40,-67}}, color={0,0,127}));
      connect(UpWind.y, CeilConvOut.Wind) annotation (Line(points={{79,50},{58,
              50},{58,39}}, color={0,0,127}));
      connect(UpWind.y, NConvOut.Wind) annotation (Line(points={{79,50},{58,50},
              {58,13}}, color={0,0,127}));
      connect(UpWind.y, WConvOut.Wind) annotation (Line(points={{79,50},{58,50},
              {58,-25}}, color={0,0,127}));
      connect(Measurements.y[2], AirOutside.T) annotation (Line(points={{-79,50},
              {-56,50},{-56,70},{-2,70}}, color={0,0,127}));
      connect(Measurements.y[4], Sun.q_dif) annotation (Line(points={{-79,50},{
              -74,50},{-74,14},{-67,14}}, color={0,0,127}));
      connect(Cloudiness.y, Sun.OBL) annotation (Line(points={{-79,-10},{-74,
              -10},{-74,9},{-67,9}}, color={0,0,127}));
      connect(UpWind.y, WinConv.Wind) annotation (Line(points={{79,50},{-40,50},
              {-40,-13}}, color={0,0,127}));
      connect(UpWind.y, SConvOut.Wind) annotation (Line(points={{79,50},{-40,50},
              {-40,-41}}, color={0,0,127}));
      connect(CeilConvOut.Solid, Room.CeilingPort) annotation (Line(points={{47,
              30},{12,30},{12,2}}, color={191,0,0}));
      connect(EarthTemp.port, Room.FloorPort) annotation (Line(points={{40,-70},
              {12,-70},{12,-42}}, color={191,0,0}));
      connect(Sun.Ray, Room.Sunlight) annotation (Line(points={{-45,14},{4,14},
              {4,2}}, color={161,161,54}));
      connect(Measurements.y[5], Room.RollerPosition) annotation (Line(points={
              {-79,50},{-56,50},{-56,38},{-20,38},{-20,-12},{-10,-12}}, color={
              0,0,127}));
      connect(AirOutside.port, WConvOut.Air) annotation (Line(points={{20,70},{
              74,70},{74,-34},{69,-34}}, color={191,0,0}));
      connect(AirOutside.port, NConvOut.Air) annotation (Line(points={{20,70},{
              74,70},{74,4},{69,4}}, color={191,0,0}));
      connect(AirOutside.port, CeilConvOut.Air) annotation (Line(points={{20,70},
              {74,70},{74,30},{69,30}}, color={191,0,0}));
      connect(AirOutside.port, port1) annotation (Line(points={{20,70},{74,70},
              {74,-94},{-60,-94},{-60,-76}}, color={191,0,0}));
      connect(port1, EConvOut.Air) annotation (Line(points={{-60,-76},{-51,-76}}, 
            color={191,0,0}));
      connect(port1, SConvOut.Air) annotation (Line(points={{-60,-76},{-60,-50},
              {-51,-50}}, color={191,0,0}));
      connect(port1, WinConv.Air) annotation (Line(points={{-60,-76},{-60,-22},
              {-51,-22}}, color={191,0,0}));
      connect(Room.OutsideAir, AirOutside.port) annotation (Line(points={{28,-4},
              {28,70},{20,70}}, color={191,0,0}));
      connect(WinConv.Solid, Room.WinPort) annotation (Line(points={{-29,-22},{
              -20,-22},{-20,-20},{-10,-20}}, color={191,0,0}));
      connect(SConvOut.Solid, Room.WinWallPort) annotation (Line(points={{-29,
              -50},{-20,-50},{-20,-28},{-10,-28}}, color={191,0,0}));
      connect(EConvOut.Solid, Room.WallOnRightPort) annotation (Line(points={{
              -29,-76},{-14,-76},{-14,-36},{-4,-36}}, color={191,0,0}));
      connect(Room.WallOnLeftPort, WConvOut.Solid) annotation (Line(points={{34,
              -28},{40,-28},{40,-34},{47,-34}}, color={191,0,0}));
      connect(Room.WallOppositePort, NConvOut.Solid) annotation (Line(points={{
              34,-12},{40,-12},{40,4},{47,4}}, color={191,0,0}));
    end KamraJeseni;

  end Kamra;

end Examples;
