within SolarHouse;
package RadiationTransfer "Utilities for optical coefficient calculation"
  import Modelica.Math.*;
  package HelperFunctions "Some helper functions"
    function simpson "Simpson's numerical integration formula"
      extends Modelica.Icons.Function;
      input Real vals[:] "function values";
      input Real a "Lower limit";
      input Real b "Upper limit";
      output Real res;
    algorithm
      //assert(size(vals,1) >= 3,"Not enough points for Simpson integration!");
      if size(vals, 1) >= 3 then
        res := (b - a)/(size(vals, 1) - 1)*(vals[1] + 2*sum(vals[3:2:end - 1])
           + 4*sum(vals[2:2:end - 1]) + vals[end])/3;
      else
        res := (b - a)*sum(vals)/2;
      end if;
    end simpson;

    function ReflectionCoeff "Calculation of reflection coeffient A(theta)"
      input SI.Angle theta "Angle of incidence";
      input Real n "Refractive index of a medium";
      output Real A;
    protected
      SI.Angle theta1 "Angle of refraction";
    algorithm
      if theta == 0 then
        A := (3*n^2 - 6*n + 3)/(2*n^2 + 4*n + 2);
      else
        theta1 := asin(sin(theta)/n);
        A := (1/2)*((sin(theta - theta1)/sin(theta + theta1))^2 + (tan(theta -
          theta1)/tan(theta + theta1))^2);
      end if;
      if A > 1 then
        A := 1;
      end if;
    end ReflectionCoeff;

    function TR "Calculate Transmission coefficient"
      input Real A[2] "Reflection Coeff";
      output Real coef;
    protected
      Real denom;
    algorithm
      denom := ((1 + A[1])*(1 + A[2]) - 4*A[1]*A[2]);
      coef := if denom > 0 then (1 - A[1])*(1 - A[2])/denom else 0;
    end TR;

    function TR1 "Calculate transmission coefficient of first pane"
      input Real A[2] "Reflection coefficients for both panes";
      output Real coef "Returned result";
    protected
      Real denom;
    algorithm
      denom := ((1 + A[1])*(1 + A[2]) - 4*A[1]*A[2]);
      coef := if denom > 0 then (1 - A[1])*(1 + A[2])/denom else .5;
    end TR1;

    function TRB "Calculate transmission coefficient of first pane"
      input Real A[2] "Reflection coefficients for both panes";
      output Real coef "Returned result";
    protected
      Real denom;
    algorithm
      denom := ((1 + A[1])*(1 + A[2]) - 4*A[1]*A[2]);
      coef := if denom > 0 then 2*A[2]*(1 - A[1])/denom else 0;
    end TRB;

    function tr_exp
      "First exponent-expression calculated needed for TR and AB coefficients"
      input SI.Angle theta "Angle of incidence";
      input Real PP "Transmission coefficient at rectangular incidence angle";
      input Real n "Refractive indices";
      output Real res;
    algorithm
      res := exp(-log(2*n/(PP*(n^2 + 1)))/sqrt(1 - (sin(theta)/n)^2));
    end tr_exp;

    function TR_DIR "Calculate direct irradiation transmision coefficient"
      input Real TR "Transmission Coefficient TR(theta)";
      input Real exp1;
      input Real exp2;
      output Real coef;
    algorithm
      coef := TR*exp1*exp2;
    end TR_DIR;

    function AB_DIR_O
      "Calcualte direct irradiation absorbtion coefficient of inner pane"
      input Real A[2];
      input Real exp1;
      output Real coef;
    protected
      Real TRB;
    algorithm
      //coef := TR1 *(1-exp1);
      if A[1] < 1 then
        TRB := 2*A[2]*(1 - A[1])/((1 + A[1])*(1 + A[2]) - 4*A[1]*A[2]);
        coef := (1 - A[1])/(1 + A[1])*(1 - exp1)*(1 + TRB*exp1);
      else
        coef := 0;
      end if;
    end AB_DIR_O;

    function AB_DIR_I
      "Calculate direct irradiation absorbtion coefficient of outer pane"
      input Real TR "Transmission Coefficient TR(theta)";
      input Real exp1;
      input Real exp2;
      output Real coef;
    algorithm
      coef := TR*exp1*(1 - exp2);
    end AB_DIR_I;

    function TR_DIR_sin
      input SI.Angle theta "Incident angle";
      input Real n[2] "Refraction indices";
      input Real PP[2] "Perpendicular incidence transmission coefficient";
      output Real coef;
    protected
      Real A[2];
    algorithm
      A := {SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(theta,
        n[1]),SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(
        theta, n[2])};
      coef := sin(theta)*SolarHouse.RadiationTransfer.HelperFunctions.TR_DIR(
            SolarHouse.RadiationTransfer.HelperFunctions.TR(A),
            SolarHouse.RadiationTransfer.HelperFunctions.tr_exp(
              theta,
              PP[1],
              n[1]),
            SolarHouse.RadiationTransfer.HelperFunctions.tr_exp(
              theta,
              PP[2],
              n[2]));
    end TR_DIR_sin;

    function AB_DIR_I_sin
      input SI.Angle theta "Incident angle";
      input Real n[2] "Refraction indices";
      input Real PP[2] "Perpendicular incidence transmission coefficient";
      output Real coef;
    protected
      Real A[2];
    algorithm
      A := {SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(theta,
        n[1]),SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(
        theta, n[2])};
      coef := sin(theta)*SolarHouse.RadiationTransfer.HelperFunctions.AB_DIR_I(
            SolarHouse.RadiationTransfer.HelperFunctions.TR(A),
            SolarHouse.RadiationTransfer.HelperFunctions.tr_exp(
              theta,
              PP[1],
              n[1]),
            SolarHouse.RadiationTransfer.HelperFunctions.tr_exp(
              theta,
              PP[2],
              n[2]));
    end AB_DIR_I_sin;

    function AB_DIR_O_sin
      "Calcualte direct irradiation absorbtion coefficient of inner pane"
      input SI.Angle theta;
      input Real n[2];
      input Real PP[2];
      output Real coef;
    protected
      Real A[2];
      Real exp1;
      Real TRB;
    algorithm
      A := {SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(theta,
        n[1]),SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(
        theta, n[2])};
      if A[1] < 1 then
        exp1 := SolarHouse.RadiationTransfer.HelperFunctions.tr_exp(
              theta,
              PP[1],
              n[1]);
        TRB := 2*A[2]*(1 - A[1])/((1 + A[1])*(1 + A[2]) - 4*A[1]*A[2]);
        coef := (1 - A[1])/(1 + A[1])*(1 - exp1)*(1 + TRB*exp1)*sin(theta);
      else
        coef := 0;
      end if;
    end AB_DIR_O_sin;

    function PermeabilityCoeffs
      "Calculate transmission and absorption coefficients"
      extends Modelica.Icons.Function;
      input SI.Angle theta;
      input Real n[2] "Refraction indices";
      input Real PP[2] "Perpendicular incidence transmission coefficient";
      output SI.SpectralTransmissionFactor TRDIR
        "Transmission coefficient of a window";
      output SI.SpectralAbsorptionFactor ABDIROUT
        "Outer pane absorption coefficient";
      output SI.SpectralAbsorptionFactor ABDIRIN
        "Inner pane absorption coefficient";
    protected
      Real A[2] "Reflection coefficients";
      Real TRA "Transmitted flow without absorption for both panes";
      Real exp1;
      Real exp2;
    algorithm
      if theta < Const.pi/2 then
        A := {SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(
          theta, n[1]),
          SolarHouse.RadiationTransfer.HelperFunctions.ReflectionCoeff(theta, n[
          2])};
        TRA := SolarHouse.RadiationTransfer.HelperFunctions.TR(A);
        exp1 := SolarHouse.RadiationTransfer.HelperFunctions.tr_exp(
              theta,
              PP[1],
              n[1]);
        exp2 := SolarHouse.RadiationTransfer.HelperFunctions.tr_exp(
              theta,
              PP[2],
              n[2]);
        TRDIR := SolarHouse.RadiationTransfer.HelperFunctions.TR_DIR(
              TRA,
              exp1,
              exp2);
        ABDIROUT := SolarHouse.RadiationTransfer.HelperFunctions.AB_DIR_O(A,
          exp1);
        ABDIRIN := SolarHouse.RadiationTransfer.HelperFunctions.AB_DIR_I(
              TRA,
              exp1,
              exp2);
      else
        TRDIR := 0;
        ABDIROUT := 0;
        ABDIRIN := 0;
      end if;
    end PermeabilityCoeffs;
  end HelperFunctions;

  // =============================

  // =============================
  //  direct irradiation coefficient calculation functions
  //

  // ============================
  //  diffuse irradiation coefficient calculation functions
  //  (inefficient, but calculated only once)
  //

  model LightTransmission
    "Transmission of solar thermal radiation through the window"
    extends Modelica.Blocks.Interfaces.BlockIcon;
    annotation (Diagram(graphics),
                         Icon(graphics={
          Rectangle(
            extent={{-36,74},{-24,-74}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={170,213,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{16,74},{28,-74}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={170,213,255},
            fillPattern=FillPattern.Solid),
          Line(
            points={{-82,54},{-36,18}},
            color={255,255,0},
            thickness=1),
          Line(
            points={{-24,16},{16,0}},
            color={255,255,0},
            thickness=1),
          Line(
            points={{28,-4},{74,-40}},
            color={255,255,0},
            thickness=1),
          Line(points={{-36,18},{-62,8}}, color={255,255,0}),
          Line(points={{16,0},{-2,-8},{-6,-10}}, color={255,255,0})}),
      Documentation(info="<html>
Calculates amout of solar thermal radiation that is transmitted through the double-glazed window and amounts absorpted in the first and second pane respecitvely.
<img src=\"../images/RadiationThroughWindow.png\" />
</html>"));
    parameter Real normal[3] "Normal vector to the panes";
    parameter SI.Area area "Area of a pane";
    parameter Integer steps=100;
    parameter Real n[2] "Refraction indices";
    parameter Real PP[2] "Perpendicular incidence transmission coefficients";
    parameter SI.SpectralTransmissionFactor TR_DIF=SolarHouse.RadiationTransfer.HelperFunctions.simpson(
            {SolarHouse.RadiationTransfer.HelperFunctions.TR_DIR_sin(
              th,
              n,
              PP) for th in linspace(
              0,
              Const.pi/2,
              steps)},
            0,
            Const.pi/2);
    parameter SI.SpectralAbsorptionFactor AB_DIF_O=SolarHouse.RadiationTransfer.HelperFunctions.simpson(
            {SolarHouse.RadiationTransfer.HelperFunctions.AB_DIR_O_sin(
              th,
              n,
              PP) for th in linspace(
              0,
              Const.pi/2,
              steps)},
            0,
            Const.pi/2);
    parameter SI.SpectralAbsorptionFactor AB_DIF_I=SolarHouse.RadiationTransfer.HelperFunctions.simpson(
            {SolarHouse.RadiationTransfer.HelperFunctions.AB_DIR_I_sin(
              th,
              n,
              PP) for th in linspace(
              0,
              Const.pi/2,
              steps)},
            0,
            Const.pi/2);
    Interfaces.DirLightIn Outside annotation (Placement(transformation(extent={
              {-120,-10},{-100,10}}, rotation=0)));
    Iface.RealOutput Absorpted1 "Apsorpted heat in first pane" 
                                     annotation (Placement(transformation(
            extent={{100,50},{120,70}}, rotation=0)));
    Iface.RealOutput Absorpted2 "Absorpted in inner pane" 
                                annotation (Placement(transformation(extent={{
              100,-10},{120,10}}, rotation=0)));
    Interfaces.LightOut Transmitted annotation (Placement(transformation(extent
            ={{100,-70},{120,-50}}, rotation=0)));
    //protected
    Real cos_theta "Cosinus of ray incidence angle";
    SI.Angle theta;
    SI.SpectralTransmissionFactor TR_DIR;
    SI.SpectralAbsorptionFactor AB_DIR_O;
    SI.SpectralAbsorptionFactor AB_DIR_I;
    SI.Area uncovered(start=area)
      "Portion of window area that is uncovered by roller";
    Iface.RealInput Inside 
      annotation (Placement(transformation(extent={{-120,-70},{-100,-50}},
            rotation=0)));
    Iface.RealInput RollerPos 
      annotation (Placement(transformation(extent={{-120,50},{-100,70}},
            rotation=0)));
  equation
    uncovered = area*(1 - RollerPos);
    cos_theta = Outside.dir*normal;
    theta = acos(cos_theta);
    (TR_DIR,AB_DIR_O,AB_DIR_I) = SolarHouse.RadiationTransfer.HelperFunctions.PermeabilityCoeffs(
          acos(cos_theta),
          n,
          PP);
    // absorption of outside pane
    Absorpted1 = uncovered*(AB_DIR_O*Outside.q_dir*cos_theta + AB_DIF_O*
      Outside.q_dif) + Inside*AB_DIF_O*(1 - AB_DIF_I);
    // absorption of inside pane
    Absorpted2 = uncovered*(AB_DIR_I*Outside.q_dir*cos_theta + AB_DIF_I*
      Outside.q_dif) + Inside*AB_DIF_I;
    // transmited light
    Transmitted.q_dir = uncovered*TR_DIR*Outside.q_dir*cos_theta;
    Transmitted.q_dif = uncovered*TR_DIF*Outside.q_dif;
  end LightTransmission;

  block RayPlaneAngle "Incidence angle of the sun's rays"
    extends Iface.BlockIcon;
    parameter Real normal[3] "Normal of the plane (should point outward)";
    Iface.RealOutput cos_theta "Incidence angle" 
                               annotation (Placement(transformation(extent={{
              100,-10},{120,10}}, rotation=0)));
    Iface.RealInput Ray[3] "Direction of the ray (should be normalized)" 
                           annotation (Placement(transformation(extent={{-120,
              -10},{-100,10}}, rotation=0)));
  equation
    cos_theta = Ray*normal;
  end RayPlaneAngle;

  partial block SunlightDistrib
    "Base class for calculation of the distribution of the direct-solar component"
    extends Iface.BlockIcon;
    Iface.RealInput IncidenceRay[3] "Direction vector of incidence" 
      annotation (Placement(transformation(extent={{-120,-50},{-100,-30}},
            rotation=0)));
    Iface.RealOutput Distribution[:] annotation (Placement(transformation(
            extent={{100,-10},{120,10}}, rotation=0)));
    annotation (Diagram(graphics),
                         Icon(graphics={
          Polygon(
            points={{-88,-28},{30,-28},{90,-88},{-28,-88},{-88,-28}},
            lineColor={0,0,0},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{30,70},{30,-28},{90,-88},{90,10},{30,70}},
            lineColor={0,0,0},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{30,-28},{-88,70}},
            lineColor={0,0,0},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{10,-28},{-20,-36},{16,-72},{58,-56},{30,-28},{10,-28}},
            lineColor={0,0,0},
            fillColor={95,95,95},
            fillPattern=FillPattern.Backward),
          Polygon(
            points={{30,-28},{58,-56},{58,-24},{30,4},{30,-28}},
            lineColor={0,0,0},
            fillColor={95,95,95},
            fillPattern=FillPattern.Forward),
          Polygon(
            points={{8,-28},{30,-28},{30,4},{8,8},{8,-28}},
            lineColor={0,0,0},
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid),
          Polygon(points={{-88,70},{-28,10},{-28,-88},{-88,-28},{-88,70}},
              lineColor={0,0,0}),
          Rectangle(extent={{-28,-88},{90,10}}, lineColor={0,0,0}),
          Line(points={{-76,36},{8,8}}, color={0,0,0}),
          Line(points={{-46,6},{58,-24}}, color={0,0,0}),
          Line(points={{-76,-20},{-20,-36}}, color={0,0,0}),
          Line(points={{-46,-50},{16,-72}}, color={0,0,0}),
          Polygon(
            points={{-76,-20},{-46,-50},{-46,6},{-76,36},{-76,-20}},
            lineColor={0,0,0},
            fillColor={85,170,255},
            fillPattern=FillPattern.Solid)}),
      Documentation(info="<html>
Base clas with input and output connectors definitions for calculation of distribution of solar radiation component across the walls. The distribution depends on the geometry of the room.
</html>"));

    Iface.RealInput RollerPos "Position of the Roller" 
                               annotation (Placement(transformation(extent={{
              -120,30},{-100,50}}, rotation=0)));
  end SunlightDistrib;

  model SunlightBoxDistrib
    "Distribution of direct solar radiation in box-shaped room"
    extends SunlightDistrib(redeclare Iface.RealOutput Distribution[4]);
    parameter SI.Position Window[3, 4];
    parameter SI.Position Left[3, :] "Wall on the left";
    parameter SI.Position Opposite[3, :] "Wall opposite of window";
    parameter SI.Position Right[3, :] "Wall on the right";
    parameter SI.Position Beneath[3, :] "Wall beneath";
  public
    parameter Real TransMatrix[4, 4]=Geometry.RelTransMatrix(
            Window[:, 1],
            Window[:, 2],
            Window[:, 3]);
  protected
    parameter SI.Position RWin[2, size(Window, 2)]=Geometry.ToPlane(
        Geometry.PolyToRelCoord(TransMatrix, Window), {0,0,1});
    parameter SI.Area WinArea=Geometry.PolygonArea(RWin);
    parameter SI.Position RLeft[3, size(Left, 2)]=Geometry.PolyToRelCoord(
        TransMatrix, Left);
    parameter SI.Position ROpposite[3, size(Opposite, 2)]=
        Geometry.PolyToRelCoord(TransMatrix, Opposite);
    parameter SI.Position RRight[3, size(Right, 2)]=Geometry.PolyToRelCoord(
        TransMatrix, Right);
    parameter SI.Position RBeneath[3, size(Beneath, 2)]=
        Geometry.PolyToRelCoord(TransMatrix, Beneath);
  public
    constant Integer L=1 "left of";
    constant Integer O=2 "opposite of";
    constant Integer R=3 "right of";
    constant Integer B=4 "beneath";
  public
    SI.Position Uncovered[2, size(RWin, 2)];
    Real RRay[3] "relative sun ray";
  equation
    RRay = TransMatrix[1:3, 1:3]*IncidenceRay;
    Uncovered = Geometry.roller(
          RWin,
          {3,4},
          {2,1},
          RollerPos);
    if noEvent(RRay[3] > 0 and RollerPos < 1) then
      WinArea*(1 - RollerPos)*Distribution[L] = Geometry.IntersectionArea(
        Uncovered, Geometry.ToPlane(RLeft, RRay));
      WinArea*(1 - RollerPos)*Distribution[O] = Geometry.IntersectionArea(
        Uncovered, Geometry.ToPlane(ROpposite, RRay));
      WinArea*(1 - RollerPos)*Distribution[R] = Geometry.IntersectionArea(
        Uncovered, Geometry.ToPlane(RRight, RRay));
      WinArea*(1 - RollerPos)*Distribution[B] = Geometry.IntersectionArea(
        Uncovered, Geometry.ToPlane(RBeneath, RRay));
    else
      Distribution = zeros(size(Distribution, 1));
    end if;
    annotation (Diagram(graphics),
                         Documentation(info="<html>
Caluclates distribution od direct solar radiation in a box-shaped room regarding the direction of <tt>IncidenceRay</tt>.<br />
The output <tt>Distribution</tt> are portions of total radiation penetrating through the windows portitions for 
wall left, right and opposite of the window and floor.
</html>"));
  end SunlightBoxDistrib;

  model SunlightReflections
    "Calculates heat flow rates into each wall due to solar radiation"
    extends Iface.BlockIcon;
    Interfaces.LightIn Light "Intensity of light comming through window" 
      annotation (Placement(transformation(extent={{-120,30},{-100,50}},
            rotation=0)));
    Iface.RealInput dist[nillum]
      "Distribution of direct solar radiation on walls" 
      annotation (Placement(transformation(extent={{-120,-50},{-100,-30}},
            rotation=0)));
    Iface.RealOutput Reflected
      "Reflected radiation which returns to the source window" 
      annotation (Placement(transformation(
          origin={0,-110},
          extent={{-10,-10},{10,10}},
          rotation=270)));
    Iface.RealOutput HeatDistrib[nout]
      "Distributed solar radiation across the walls" 
      annotation (Placement(transformation(extent={{100,-10},{120,10}},
            rotation=0)));
  public
    parameter Integer n=5 "Number of reflection";
    parameter Integer NWalls=1 "Number of Walls";
    parameter Integer NWins=1 "Number of windows";
    parameter Integer SRC(max=NWins) "Index of the source window";
    parameter SI.Area Walls[NWalls] "Area of all walls";
    parameter SI.Area Windows[NWins] "Area of windows";
    parameter Integer WinOnWall[NWins]
      "Lookup-table of indices which windows are on what walls";
    parameter Real sfactor[NWalls, NWalls] "Matrix of shape factors";
    // reflection factors
    parameter SI.SpectralReflectionFactor RWall[size(Walls, 1)]
      "Reflection factors of the wall";
    parameter SI.SpectralReflectionFactor RWin[size(Windows, 1)]
      "Reflection factors of the windows";
    // portions
    parameter Integer nillum "Number of illuminated walls by direct light";
    parameter Integer Illumin[nillum]
      "Look-up table for direct light illuminated walls";
    final parameter Integer nout=NWalls + NWins - 1 "Number of outputs";
  protected
    final parameter Real WallP[NWalls]=InitWallP(
            Walls,
            Windows,
            WinOnWall)
      "Portion of solid wall area (without windows) to whole area";
    final parameter Real WinP[NWins]=InitWinP(
            Walls,
            Windows,
            WinOnWall) "Portion of window area to the whole wall";
    SI.HeatFlowRate QWall[NWalls] "Current reflection light flow to walls";
    SI.HeatFlowRate QWin[NWins] "Current reflection light flow to windows";
    SI.HeatFlowRate QnWall[NWalls] "Next step reflection";
    SI.HeatFlowRate QnWin[NWins] "Next step reflection";
    SI.HeatFlowRate SumQWall[NWalls]
      "Sum of heat flow of all reflections for walls";
    SI.HeatFlowRate SumQWin[NWins]
      "Sum of heat flow of all reflection for windows";
  protected
    function InitWallP "Work-around for non-working initial algorithm"
      input SI.Area Walls[:];
      input SI.Area Wins[:];
      input Integer Lookup[size(Wins, 1)];
      output Real Portion[size(Walls, 1)];
    algorithm
      Portion := Walls;
      for i in 1:size(Wins, 1) loop
        Portion[Lookup[i]] := Portion[Lookup[i]] - Wins[i];
      end for;
      for i in 1:size(Portion, 1) loop
        Portion[i] := Portion[i]/Walls[i];
      end for;
    end InitWallP;

    function InitWinP "Work around for non-working initial algorithm"
      input SI.Area Walls[:];
      input SI.Area Wins[:];
      input Integer Lookup[size(Wins, 1)];
      output Real Portion[size(Wins, 1)];
    algorithm
      for i in 1:size(Portion, 1) loop
        Portion[i] := Wins[i]/Walls[Lookup[i]];
      end for;
    end InitWinP;
  algorithm
    for i in 1:NWalls loop
      QWall[i] := WallP[i]*sfactor[SRC, i]*Light.q_dif;
    end for;
    //QWall := {WallP[i]*sfactor[SRC,i]*Light.q_dif for i in 1:NWalls};
    for i in 1:nillum loop
      QWall[Illumin[i]] := QWall[Illumin[i]] + dist[i]*Light.q_dir;
    end for;
    for i in 1:NWins loop
      QWin[i] := WinP[i]*sfactor[SRC, WinOnWall[i]]*Light.q_dif;
    end for;
    //QWin := {WinP[i]*sfactor[SRC,WinOnWall[i]]*Light.q_dif for i in 1:NWins};
    // add to sum
    SumQWall := QWall;
    SumQWin := QWin;
    // calculate n reflections
    for k in 1:n loop
      // calculate recieved light
      for i in 1:NWalls loop
        QnWall[i] := 0;
        for j in 1:NWalls loop
          QnWall[i] := QnWall[i] + sfactor[j, i]*RWall[j]*QWall[j];
        end for;
        for j in 1:NWins loop
          QnWall[i] := QnWall[i] + sfactor[WinOnWall[j], i]*RWin[j]*QWin[j];
        end for;
        QnWall[i] := WallP[i]*QnWall[i];
        //      QnWall[i] := WallP[i]*(sum(sfactor[j,i]*RWall[j]*QWall[j] for j in 1:NWalls)
        //                    + sum(sfactor[WinOnWall[j],i]*RWin[j]*QWin[j] for j in 1:NWins));
      end for;
      // add to sum
      SumQWall := SumQWall + QnWall;
      // previous step is now new step
      QWall := QnWall;
      // calculate same for windows
      for i in 1:NWins loop
        QnWin[i] := 0;
        for j in 1:NWalls loop
          QnWin[i] := QnWin[i] + sfactor[j, WinOnWall[i]]*RWall[j]*QWall[j];
        end for;
        for j in 1:NWins loop
          QnWin[i] := QnWin[i] + sfactor[WinOnWall[j], WinOnWall[i]]*RWin[j]*
            QWin[j];
        end for;
        QnWin[i] := WinP[i]*QnWin[i];
        //    QnWin[i] := WinP[i]*(sum(sfactor[j,WinOnWall[i]]*RWall[j]*QWall[j] for j in 1:NWalls)
        //                 + sum(sfactor[WinOnWall[j],WinOnWall[i]]*RWin[j]*QWin[j] for j in 1:NWins));
      end for;
      SumQWin := SumQWin + QnWin;
      QWin := QnWin;
    end for;
    // calculate absorbted light from all reflections
    for i in 1:NWalls loop
      SumQWall[i] := (1 - RWall[i])*SumQWall[i];
    end for;
    for i in 1:NWins loop
      SumQWin[i] := (1 - RWin[i])*SumQWin[i];
    end for;
    //SumQWall := {(1-RWall[i])*SumQWall[i] for i in 1:NWalls};
    //SumQWin := {(1-RWin[i])*SumQWin[i] for i in 1:NWins};
    HeatDistrib := cat(
          1,
          SumQWall,
          SumQWin[cat(
            1,
            1:SRC - 1,
            SRC + 1:NWins)]);
    Reflected := SumQWin[SRC];
    annotation (Diagram(graphics),
                         Icon(graphics={
          Polygon(
            points={{28,74},{28,-24},{88,-84},{88,14},{28,74}},
            lineColor={0,0,0},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{28,-24},{-90,74}},
            lineColor={0,0,0},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-90,-24},{28,-24},{88,-84},{-30,-84},{-90,-24}},
            lineColor={0,0,0},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid),
          Line(
            points={{-100,40},{16,-66},{60,-12},{-12,24},{6,-42},{82,42}},
            color={255,0,0},
            thickness=0.5,
            arrow={Arrow.None,Arrow.Filled})}),
      Documentation(info=
            "<html>Caluclates the final distribution of the heat across the interior due to (all) the solar radiation penetrating through window.</html>"));
  end SunlightReflections;

  block LightDirection "Extract direction vector from DirLight connector"
    //extends Iface.BlockIcon;
    Interfaces.DirLightIn LightConn annotation (Placement(transformation(extent
            ={{-120,-10},{-100,10}}, rotation=0)));
    Iface.RealOutput dir[3] "Direction vector" 
      annotation (Placement(transformation(extent={{100,-10},{120,10}},
            rotation=0)));
    annotation (Diagram(graphics),
                         Icon(graphics={
          Rectangle(extent={{-80,80},{80,-80}}, lineColor={0,0,0}),
          Line(points={{80,0},{100,0}}, color={0,0,0}),
          Text(
            extent={{-80,74},{26,2}},
            lineColor={161,161,54},
            textString=
                 "Beam"),
          Text(
            extent={{-10,-14},{80,-76}},
            lineColor={0,0,0},
            pattern=LinePattern.None,
            textString=
                 "dir"),
          Line(points={{-80,-80},{80,80},{-80,80},{-80,-80}}, color={161,161,54}), 

          Line(points={{-100,0},{-80,0}}, color={161,161,54})}));

  equation
    dir = LightConn.dir;
  end LightDirection;
  annotation (Documentation(info="<html>
Components which models transmission and distribution of solar thermal radiation.
</html>"));
end RadiationTransfer;
