within SolarHouse;
package ThermalInteractions
  "Custom components for some thermal-interactions models"
  model Flux2FlowRate "Convert HeatFluxPort to HeatPort connector"
    parameter SI.Area Area "Area of the surface";
    Interfaces.HeatFluxPort_a port_a annotation (Placement(transformation(
            extent={{-120,-10},{-100,10}}, rotation=0)));
    HT.Interfaces.HeatPort_b port_b annotation (Placement(transformation(extent
            ={{100,-10},{120,10}}, rotation=0)));
  equation
    port_a.T = port_b.T;
    port_a.Q_flux*Area + port_b.Q_flow = 0;
    annotation (Diagram(graphics={Line(points={{-80,0},{80,0}}, color={127,0,0}), 
            Line(points={{60,8},{80,0},{60,-8}}, color={127,0,0})}),
                              Icon(graphics={
          Line(points={{-100,0},{-80,0},{-80,80},{80,80},{-80,-80},{-80,0}},
              color={255,0,0}),
          Line(points={{80,78},{80,-80},{-80,-80}}, color={127,0,0}),
          Line(points={{80,0},{100,0}}, color={127,0,0}),
          Text(
            extent={{-24,-20},{80,-80}},
            lineColor={127,0,0},
            fillColor={255,85,85},
            fillPattern=FillPattern.Solid,
            textString=
                 "FlowRate"),
          Text(
            extent={{-58,46},{-12,16}},
            lineColor={255,0,0},
            fillColor={255,85,85},
            fillPattern=FillPattern.Solid,
            textString=
                 "Flux")}));
  end Flux2FlowRate;

  model FlowRate2Flux "Convert HeatPort to HeatFluxPort to connector"
    parameter SI.Area Area "Area of the surface";
    HT.Interfaces.HeatPort_a port_a annotation (Placement(transformation(extent
            ={{-120,-10},{-100,10}}, rotation=0)));
    Interfaces.HeatFluxPort_b port_b annotation (Placement(transformation(
            extent={{100,-10},{120,10}}, rotation=0)));
  equation
    port_a.T = port_b.T;
    port_a.Q_flow + port_b.Q_flux*Area = 0;
    annotation (Diagram(graphics={Line(points={{-80,0},{80,0}}, color={127,0,0}), 
            Line(points={{60,8},{80,0},{60,-8}}, color={127,0,0})}),
                              Icon(graphics={
          Line(points={{-100,0},{-80,0},{-80,80},{80,80},{-80,-80},{-80,0}},
              color={127,0,0}),
          Line(points={{80,80},{80,-80},{-80,-80}}, color={255,0,0}),
          Line(points={{80,0},{100,0}}, color={255,0,0}),
          Text(
            extent={{-80,80},{24,20}},
            lineColor={127,0,0},
            fillColor={255,85,85},
            fillPattern=FillPattern.Solid,
            textString=
                 "FlowRate"),
          Text(
            extent={{14,-30},{60,-60}},
            lineColor={255,0,0},
            fillColor={255,85,85},
            fillPattern=FillPattern.Solid,
            textString=
                 "Flux")}));
  end FlowRate2Flux;

  partial model Convection "Base class for convection components"
    annotation (Icon(graphics={
          Rectangle(
            extent={{-70,80},{99,-80}},
            lineColor={255,255,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-100,80},{-70,-80}},
            lineColor={0,0,0},
            fillColor={192,192,192},
            fillPattern=FillPattern.Backward),
          Text(extent={{-164,-98},{164,-148}}, textString=
                                                   "%name"),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{80,30},{100,20}}, color={191,0,0}),
          Line(points={{80,10},{100,20}}, color={191,0,0}),
          Line(points={{-59,20},{100,20}}, color={191,0,0}),
          Line(points={{-60,-20},{100,-20}}, color={191,0,0}),
          Line(points={{80,-10},{100,-20}}, color={191,0,0}),
          Line(points={{80,-30},{100,-20}}, color={191,0,0})}),
        Diagram(graphics={
          Rectangle(
            extent={{-100,80},{-70,-80}},
            lineColor={0,0,0},
            fillColor={192,192,192},
            fillPattern=FillPattern.Backward),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{-59,20},{100,20}}, color={191,0,0}),
          Line(points={{80,30},{100,20}}, color={191,0,0}),
          Line(points={{80,10},{100,20}}, color={191,0,0}),
          Line(points={{-60,-20},{100,-20}}, color={191,0,0}),
          Line(points={{80,-10},{100,-20}}, color={191,0,0}),
          Line(points={{80,-30},{100,-20}}, color={191,0,0}),
          Text(
            extent={{-35,42},{-5,20}},
            lineColor={255,0,0},
            textString=
                 "Q_flow")}));
    HT.Interfaces.HeatPort_a Solid annotation (Placement(transformation(extent=
              {{-120,-10},{-100,10}}, rotation=0)));
    HT.Interfaces.HeatPort_b Air annotation (Placement(transformation(extent={{
              100,-10},{120,10}}, rotation=0)));
  end Convection;

  model ConvectionWallAir
    "Convection at the horizontal indoor surfaces (empirical expression)"
    extends Convection;
    parameter SI.Height height "Wall height";
    parameter SI.Area area "Wall area";
    SI.ThermalConductance Gr;
  equation
    Gr = area*1.2*noEvent(abs(Solid.T - Air.T)^(0.29 + 0.14*height));
    Gr*(Solid.T - Air.T) = Solid.Q_flow;
    Air.Q_flow = -Solid.Q_flow;
  end ConvectionWallAir;

  model ConvectionFloorAir "Convection at the floor (empirical expression)"
    extends Convection;
    parameter SI.Area area "Wall area";
    SI.ThermalConductance Gr;
  equation
    if noEvent(Solid.T > Air.T) then
      Gr = 1.0*area*(Solid.T - Air.T)^0.33;
    else
      Gr = 1.9*area*(Air.T - Solid.T)^0.33;
    end if;
    Gr*(Solid.T - Air.T) = Solid.Q_flow;
    Air.Q_flow = -Solid.Q_flow;
  end ConvectionFloorAir;

  model ConvectionWallOutside
    "Convection at the walls outside (empirical expression)"
    annotation (Icon(graphics={
          Rectangle(
            extent={{-70,80},{99,-80}},
            lineColor={255,255,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-100,80},{-70,-80}},
            lineColor={0,0,0},
            fillColor={192,192,192},
            fillPattern=FillPattern.Backward),
          Text(extent={{-164,-98},{164,-148}}, textString=
                                                   "%name"),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{80,30},{100,20}}, color={191,0,0}),
          Line(points={{80,10},{100,20}}, color={191,0,0}),
          Line(points={{-59,20},{100,20}}, color={191,0,0}),
          Line(points={{-60,-20},{100,-20}}, color={191,0,0}),
          Line(points={{80,-10},{100,-20}}, color={191,0,0}),
          Line(points={{80,-30},{100,-20}}, color={191,0,0})}),
        Diagram(graphics={
          Rectangle(
            extent={{-100,80},{-70,-80}},
            lineColor={0,0,0},
            fillColor={192,192,192},
            fillPattern=FillPattern.Backward),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{-40,80},{-40,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{0,80},{0,-80}}, color={0,127,255}),
          Line(points={{34,80},{34,-80}}, color={0,127,255}),
          Line(points={{34,-80},{44,-60}}, color={0,127,255}),
          Line(points={{34,-80},{24,-60}}, color={0,127,255}),
          Line(points={{0,-80},{10,-60}}, color={0,127,255}),
          Line(points={{0,-80},{-10,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-30,-60}}, color={0,127,255}),
          Line(points={{-40,-80},{-50,-60}}, color={0,127,255}),
          Line(points={{70,-80},{60,-60}}, color={0,127,255}),
          Line(points={{70,-80},{80,-60}}, color={0,127,255}),
          Line(points={{70,80},{70,-80}}, color={0,127,255}),
          Line(points={{100,0},{100,0}}, color={0,127,255}),
          Line(points={{100,80},{100,-80}}, color={0,127,255}),
          Line(points={{100,-80},{90,-60}}, color={0,127,255}),
          Line(points={{100,-80},{110,-60}}, color={0,127,255}),
          Line(points={{-59,20},{100,20}}, color={191,0,0}),
          Line(points={{80,30},{100,20}}, color={191,0,0}),
          Line(points={{80,10},{100,20}}, color={191,0,0}),
          Line(points={{-60,-20},{100,-20}}, color={191,0,0}),
          Line(points={{80,-10},{100,-20}}, color={191,0,0}),
          Line(points={{80,-30},{100,-20}}, color={191,0,0}),
          Text(
            extent={{-35,42},{-5,20}},
            lineColor={255,0,0},
            textString=
                 "Q_flow")}));
    parameter SI.Area area "Wall area";
    HT.Interfaces.HeatPort_a Solid annotation (Placement(transformation(extent=
              {{-120,-10},{-100,10}}, rotation=0)));
    HT.Interfaces.HeatPort_b Air annotation (Placement(transformation(extent={{
              100,-10},{120,10}}, rotation=0)));
    Iface.RealInput Wind "Velocity of the wind blowing upward the wall" 
      annotation (Placement(transformation(
          origin={0,90},
          extent={{10,-10},{-10,10}},
          rotation=90)));
    SI.ThermalConductance Gr;
  equation
    Gr = area*(4.1*Wind + 5.7);
    Gr*(Solid.T - Air.T) = Solid.Q_flow;
    Air.Q_flow = -Solid.Q_flow;
  end ConvectionWallOutside;

  model RadiationExchange6win1
    "Thermal radiation exchange in a box-shaped room with one window"
    parameter SI.Area WallA[6] "Area of the walls";
    parameter SI.Area WinA "Area of the window";
    parameter SI.Emissivity WallEm[6] "Emissivities of the walls";
    parameter SI.Emissivity WinEm "Emissivity of the window";
    parameter Real SFactor[6, 6] "Matrix of shape factors";
    parameter Integer WinIdx "Index of the wall in which window is";
    HT.Interfaces.HeatPort_a wall1 annotation (Placement(transformation(extent=
              {{-100,40},{-80,60}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall2 annotation (Placement(transformation(extent=
              {{-100,-60},{-80,-40}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall3 
      annotation (Placement(transformation(
          origin={0,-90},
          extent={{-10,10},{10,-10}},
          rotation=90)));
    HT.Interfaces.HeatPort_a wall4 annotation (Placement(transformation(extent=
              {{100,-60},{80,-40}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall5 annotation (Placement(transformation(extent=
              {{100,40},{80,60}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall6 
      annotation (Placement(transformation(
          origin={0,90},
          extent={{-10,-10},{10,10}},
          rotation=270)));
    HT.Interfaces.HeatPort_b win annotation (Placement(transformation(extent={{
              -80,-10},{-100,10}}, rotation=0)));
    annotation (
      Icon(graphics={
          Polygon(
            points={{-80,60},{-80,-60},{-10,-80},{10,-80},{80,-60},{80,60},{10,
                80},{-10,80},{-80,60}},
            lineColor={0,0,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Line(points={{-62,46},{-62,-50},{52,-50}}, color={0,0,255}),
          Line(points={{-62,-50},{48,0}}, color={0,0,255}),
          Line(points={{-62,-50},{2,42}}, color={0,0,255}),
          Line(points={{54,50},{-60,50},{-60,-40}}, color={255,0,0}),
          Line(points={{-60,50},{-22,-40}}, color={255,0,0}),
          Line(points={{-60,50},{38,-12}}, color={255,0,0})}),
      Diagram(graphics),
      Documentation(info="<html>
This model models longwave radiation exchange among 6 walls and one window.
</html>"));
    Modelica.Thermal.HeatTransfer.BodyRadiation w12(
                         Gr=WallEm[1]*WallEm[2]*WallA[1]*SFactor[1, 2]) 
      annotation (Placement(transformation(
          origin={-38,26},
          extent={{-6,-6},{6,6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w23(
                         Gr=WallEm[2]*WallEm[3]*WallA[2]*SFactor[2, 3]) 
      annotation (Placement(transformation(extent={{-28,-78},{-14,-66}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w34(
                         Gr=WallEm[3]*WallEm[4]*WallA[3]*SFactor[3, 4]) 
      annotation (Placement(transformation(extent={{46,-78},{60,-66}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w45(
                         Gr=WallEm[4]*WallEm[5]*WallA[4]*SFactor[4, 5]) 
      annotation (Placement(transformation(
          origin={70,-26},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w56(
                         Gr=WallEm[5]*WallEm[6]*WallA[5]*SFactor[5, 6]) 
      annotation (Placement(transformation(extent={{60,64},{46,76}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w61(
                         Gr=WallEm[6]*WallEm[1]*WallA[6]*SFactor[6, 1]) 
      annotation (Placement(transformation(extent={{-8,64},{-22,76}}, rotation=
              0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w13(
                         Gr=WallEm[1]*WallEm[3]*WallA[1]*SFactor[1, 3]) 
      annotation (Placement(transformation(
          origin={-26,-14},
          extent={{-6,-6},{6,6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w24(
                         Gr=WallEm[2]*WallEm[4]*WallA[2]*SFactor[2, 4]) 
      annotation (Placement(transformation(extent={{-6,-56},{6,-44}}, rotation=
              0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w35(
                         Gr=WallEm[3]*WallEm[5]*WallA[3]*SFactor[3, 5]) 
      annotation (Placement(transformation(
          origin={58,26},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w46(
                         Gr=WallEm[4]*WallEm[6]*WallA[4]*SFactor[4, 6]) 
      annotation (Placement(transformation(
          origin={46,-14},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w51(
                         Gr=WallEm[5]*WallEm[1]*WallA[5]*SFactor[5, 1]) 
      annotation (Placement(transformation(extent={{36,44},{22,56}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w62(
                         Gr=WallEm[6]*WallEm[2]*WallA[6]*SFactor[6, 2]) 
      annotation (Placement(transformation(
          origin={-14,20},
          extent={{-6,6},{6,-6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w14(
                         Gr=WallEm[1]*WallEm[4]*WallA[1]*SFactor[1, 4]) 
      annotation (Placement(transformation(extent={{22,-44},{36,-32}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w25(
                         Gr=WallEm[2]*WallEm[5]*WallA[2]*SFactor[2, 5]) 
      annotation (Placement(transformation(extent={{22,24},{36,36}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w36(
                         Gr=WallEm[3]*WallEm[6]*WallA[3]*SFactor[3, 6]) 
      annotation (Placement(transformation(
          origin={16,-14},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win1(
                          Gr=WinEm*WallEm[1]*WinA*SFactor[WinIdx, 1]) 
      annotation (Placement(transformation(
          origin={-80,26},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win2(
                          Gr=WinEm*WallEm[2]*WinA*SFactor[WinIdx, 2]) 
      annotation (Placement(transformation(
          origin={-80,-26},
          extent={{-6,6},{6,-6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win3(
                          Gr=WinEm*WallEm[3]*WinA*SFactor[WinIdx, 3]) 
      annotation (Placement(transformation(extent={{-66,-36},{-52,-24}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win4(
                          Gr=WinEm*WallEm[4]*WinA*SFactor[WinIdx, 4]) 
      annotation (Placement(transformation(extent={{-66,-18},{-52,-6}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win5(
                          Gr=WinEm*WallEm[5]*WinA*SFactor[WinIdx, 5]) 
      annotation (Placement(transformation(extent={{-66,0},{-52,12}}, rotation=
              0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win6(
                          Gr=WinEm*WallEm[6]*WinA*SFactor[WinIdx, 6]) 
      annotation (Placement(transformation(extent={{-66,18},{-52,30}}, rotation
            =0)));
  equation
    connect(wall1, w12.port_a) annotation (Line(points={{-90,50},{-38,50},{-38,
            32}}, color={191,0,0}));
    connect(w12.port_b, wall2) annotation (Line(points={{-38,20},{-38,-50},{-90,
            -50}}, color={191,0,0}));
    connect(wall2, w23.port_a) annotation (Line(points={{-90,-50},{-38,-50},{
            -38,-72},{-28,-72}}, color={191,0,0}));
    connect(wall3, w23.port_b) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-72},{-14,-72}}, color={191,0,0}));
    connect(wall3, w34.port_a) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-72},{46,-72}}, color={191,0,0}));
    connect(w34.port_b, wall4) annotation (Line(points={{60,-72},{70,-72},{70,
            -50},{90,-50}}, color={191,0,0}));
    connect(wall4, w45.port_a) annotation (Line(points={{90,-50},{70,-50},{70,
            -32}}, color={191,0,0}));
    connect(w45.port_b, wall5) annotation (Line(points={{70,-20},{70,50},{90,50}}, 
          color={191,0,0}));
    connect(w56.port_a, wall5) annotation (Line(points={{60,70},{70,70},{70,50},
            {90,50}}, color={191,0,0}));
    connect(w56.port_b, wall6) annotation (Line(points={{46,70},{5.55112e-16,70},
            {5.55112e-16,90}}, color={191,0,0}));
    connect(wall6, w61.port_a) annotation (Line(points={{5.55112e-16,90},{
            5.55112e-16,70},{-8,70}}, color={191,0,0}));
    connect(w61.port_b, wall1) annotation (Line(points={{-22,70},{-38,70},{-38,
            50},{-90,50}}, color={191,0,0}));
    connect(wall1, w13.port_a) annotation (Line(points={{-90,50},{-26,50},{-26,
            -8}}, color={191,0,0}));
    connect(w13.port_b, wall3) annotation (Line(points={{-26,-20},{-26,-60},{
            -5.55112e-16,-60},{-5.55112e-16,-90}}, color={191,0,0}));
    connect(wall2, w24.port_a) annotation (Line(points={{-90,-50},{-6,-50}},
          color={191,0,0}));
    connect(w24.port_b, wall4) annotation (Line(points={{6,-50},{90,-50}},
          color={191,0,0}));
    connect(wall3, w35.port_a) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-60},{58,-60},{58,20}}, color={191,0,0}));
    connect(w35.port_b, wall5) annotation (Line(points={{58,32},{58,50},{90,50}}, 
          color={191,0,0}));
    connect(w46.port_a, wall4) annotation (Line(points={{46,-20},{46,-50},{90,
            -50}}, color={191,0,0}));
    connect(w46.port_b, wall6) annotation (Line(points={{46,-8},{46,60},{
            5.55112e-16,60},{5.55112e-16,90}}, color={191,0,0}));
    connect(w51.port_a, wall5) annotation (Line(points={{36,50},{90,50}}, color
          ={191,0,0}));
    connect(w51.port_b, wall1) annotation (Line(points={{22,50},{-90,50}},
          color={191,0,0}));
    connect(wall6, w62.port_a) annotation (Line(points={{5.55112e-16,90},{
            5.55112e-16,40},{-14,40},{-14,26}}, color={191,0,0}));
    connect(w62.port_b, wall2) annotation (Line(points={{-14,14},{-14,-50},{-90,
            -50}}, color={191,0,0}));
    connect(wall1, w14.port_a) annotation (Line(points={{-90,50},{0,50},{0,-38},
            {22,-38}}, color={191,0,0}));
    connect(w14.port_b, wall4) annotation (Line(points={{36,-38},{40,-38},{40,
            -50},{90,-50}}, color={191,0,0}));
    connect(wall2, w25.port_a) annotation (Line(points={{-90,-50},{-14,-50},{
            -14,-38},{0,-38},{0,30},{22,30}}, color={191,0,0}));
    connect(w25.port_b, wall5) annotation (Line(points={{36,30},{40,30},{40,50},
            {90,50}}, color={191,0,0}));
    connect(wall3, w36.port_a) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-72},{16,-72},{16,-20}}, color={191,0,0}));
    connect(w36.port_b, wall6) annotation (Line(points={{16,-8},{16,40},{
            5.55112e-16,40},{5.55112e-16,90}}, color={191,0,0}));
    connect(win1.port_a, win) annotation (Line(points={{-80,20},{-80,20},{-80,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win1.port_b, wall1) annotation (Line(points={{-80,32},{-80,32},{-80,
            50},{-90,50}}, color={191,0,0}));
    connect(win2.port_a, win) annotation (Line(points={{-80,-20},{-80,-20},{-80,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win2.port_b, wall2) annotation (Line(points={{-80,-32},{-80,-32},{
            -80,-50},{-90,-50}}, color={191,0,0}));
    connect(win3.port_a, win) annotation (Line(points={{-66,-30},{-70,-30},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win3.port_b, wall3) annotation (Line(points={{-52,-30},{-38,-30},{
            -38,-50},{-26,-50},{-26,-60},{-5.55112e-16,-60},{-5.55112e-16,-90}}, 
          color={191,0,0}));
    connect(win4.port_a, win) annotation (Line(points={{-66,-12},{-70,-12},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win4.port_b, wall4) annotation (Line(points={{-52,-12},{-38,-12},{
            -38,-50},{-26,-50},{-26,-60},{58,-60},{58,-50},{90,-50}}, color={
            191,0,0}));
    connect(win5.port_a, win) annotation (Line(points={{-66,6},{-70,6},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win6.port_a, win) annotation (Line(points={{-66,24},{-70,24},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win5.port_b, wall5) annotation (Line(points={{-52,6},{-26,6},{-26,
            50},{0,50},{0,60},{46,60},{46,50},{90,50}}, color={191,0,0}));
    connect(win6.port_b, wall6) annotation (Line(points={{-52,24},{-48,24},{-48,
            50},{5.55112e-16,50},{5.55112e-16,90}}, color={191,0,0}));
  end RadiationExchange6win1;

  model RadiationExchange7win1
    "Thermal radiation exchange among 7 indoor surfaces and window"
    parameter SI.Area WallA[7] "Area of the walls";
    parameter SI.Area WinA "Area of the window";
    parameter SI.Emissivity WallEm[7] "Emissivities of the walls";
    parameter SI.Emissivity WinEm "Emissivity of the window";
    parameter Real SFactor[7, 7] "Matrix of shape factors";
    parameter Integer WinIdx=1 "Index of the wall in which window is";
    HT.Interfaces.HeatPort_a wall1 annotation (Placement(transformation(extent=
              {{-100,40},{-80,60}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall2 annotation (Placement(transformation(extent=
              {{-100,-60},{-80,-40}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall3 
      annotation (Placement(transformation(
          origin={0,-90},
          extent={{-10,10},{10,-10}},
          rotation=90)));
    HT.Interfaces.HeatPort_a wall4 annotation (Placement(transformation(extent=
              {{100,-60},{80,-40}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall6 annotation (Placement(transformation(extent=
              {{100,40},{80,60}}, rotation=0)));
    HT.Interfaces.HeatPort_a wall7 
      annotation (Placement(transformation(
          origin={0,90},
          extent={{-10,-10},{10,10}},
          rotation=270)));
    HT.Interfaces.HeatPort_b win annotation (Placement(transformation(extent={{
              -80,-10},{-100,10}}, rotation=0)));
    annotation (
      Icon(graphics={
          Polygon(
            points={{-80,60},{-80,-60},{-10,-80},{10,-80},{80,-60},{80,60},{10,
                80},{-10,80},{-80,60}},
            lineColor={0,0,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Line(points={{-62,46},{-62,-50},{52,-50}}, color={0,0,255}),
          Line(points={{-62,-50},{48,0}}, color={0,0,255}),
          Line(points={{-62,-50},{2,42}}, color={0,0,255}),
          Line(points={{54,50},{-60,50},{-60,-40}}, color={255,0,0}),
          Line(points={{-60,50},{-22,-40}}, color={255,0,0}),
          Line(points={{-60,50},{38,-12}}, color={255,0,0})}),
      Diagram(graphics),
      Documentation(info="<html>
This model models longwave radiation exchange among 6 walls and one window.
</html>"));
    Modelica.Thermal.HeatTransfer.BodyRadiation w12(
                         Gr=WallEm[1]*WallEm[2]*WallA[1]*SFactor[1, 2]) 
      annotation (Placement(transformation(
          origin={-38,26},
          extent={{-6,-6},{6,6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w23(
                         Gr=WallEm[2]*WallEm[3]*WallA[2]*SFactor[2, 3]) 
      annotation (Placement(transformation(extent={{-28,-78},{-14,-66}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w34(
                         Gr=WallEm[3]*WallEm[4]*WallA[3]*SFactor[3, 4]) 
      annotation (Placement(transformation(extent={{20,-78},{34,-66}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w46(
                         Gr=WallEm[4]*WallEm[6]*WallA[4]*SFactor[4, 6]) 
      annotation (Placement(transformation(
          origin={48,-26},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w67(
                         Gr=WallEm[6]*WallEm[7]*WallA[6]*SFactor[6, 7]) 
      annotation (Placement(transformation(extent={{24,64},{10,76}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w71(
                         Gr=WallEm[7]*WallEm[1]*WallA[7]*SFactor[7, 1]) 
      annotation (Placement(transformation(extent={{-18,64},{-32,76}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w13(
                         Gr=WallEm[1]*WallEm[3]*WallA[1]*SFactor[1, 3]) 
      annotation (Placement(transformation(
          origin={-26,-14},
          extent={{-6,-6},{6,6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w24(
                         Gr=WallEm[2]*WallEm[4]*WallA[2]*SFactor[2, 4]) 
      annotation (Placement(transformation(extent={{-6,-56},{6,-44}}, rotation=
              0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w36(
                         Gr=WallEm[3]*WallEm[6]*WallA[3]*SFactor[3, 6]) 
      annotation (Placement(transformation(
          origin={38,16},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w47(
                         Gr=WallEm[4]*WallEm[7]*WallA[4]*SFactor[4, 7]) 
      annotation (Placement(transformation(
          origin={22,-4},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w61(
                         Gr=WallEm[6]*WallEm[1]*WallA[6]*SFactor[6, 1]) 
      annotation (Placement(transformation(extent={{18,44},{4,56}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w72(
                         Gr=WallEm[7]*WallEm[2]*WallA[7]*SFactor[7, 2]) 
      annotation (Placement(transformation(
          origin={-16,20},
          extent={{-6,6},{6,-6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w14(
                         Gr=WallEm[1]*WallEm[4]*WallA[1]*SFactor[1, 4]) 
      annotation (Placement(transformation(extent={{20,-44},{34,-32}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w26(
                         Gr=WallEm[2]*WallEm[6]*WallA[2]*SFactor[2, 6]) 
      annotation (Placement(transformation(extent={{-2,24},{12,36}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w37(
                         Gr=WallEm[3]*WallEm[7]*WallA[3]*SFactor[3, 7]) 
      annotation (Placement(transformation(
          origin={2,-8},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win1(
                          Gr=WinEm*WallEm[1]*WinA*SFactor[WinIdx, 1]) 
      annotation (Placement(transformation(
          origin={-80,26},
          extent={{-6,-6},{6,6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win2(
                          Gr=WinEm*WallEm[2]*WinA*SFactor[WinIdx, 2]) 
      annotation (Placement(transformation(
          origin={-80,-26},
          extent={{-6,6},{6,-6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win3(
                          Gr=WinEm*WallEm[3]*WinA*SFactor[WinIdx, 3]) 
      annotation (Placement(transformation(extent={{-66,-46},{-52,-34}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win4(
                          Gr=WinEm*WallEm[4]*WinA*SFactor[WinIdx, 4]) 
      annotation (Placement(transformation(extent={{-66,-28},{-52,-16}},
            rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win6(
                          Gr=WinEm*WallEm[6]*WinA*SFactor[WinIdx, 6]) 
      annotation (Placement(transformation(extent={{-66,8},{-52,20}}, rotation=
              0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win7(
                          Gr=WinEm*WallEm[7]*WinA*SFactor[WinIdx, 7]) 
      annotation (Placement(transformation(extent={{-66,26},{-52,38}}, rotation
            =0)));
    HT.Interfaces.HeatPort_a wall5 
      annotation (Placement(transformation(extent={{100,-10},{80,10}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w56(
                         Gr=WallEm[5]*WallEm[6]*WallA[5]*SFactor[5, 6]) 
      annotation (Placement(transformation(
          origin={88,28},
          extent={{-6,6},{6,-6}},
          rotation=90)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w57(
                         Gr=WallEm[5]*WallEm[7]*WallA[5]*SFactor[5, 7]) 
      annotation (Placement(transformation(extent={{60,20},{74,32}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w51(
                         Gr=WallEm[5]*WallEm[1]*WallA[5]*SFactor[5, 1]) 
      annotation (Placement(transformation(extent={{60,2},{74,14}}, rotation=0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w52(
                         Gr=WallEm[5]*WallEm[2]*WallA[5]*SFactor[5, 2]) 
      annotation (Placement(transformation(extent={{60,-16},{74,-4}}, rotation=
              0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w53(
                         Gr=WallEm[5]*WallEm[3]*WallA[5]*SFactor[5, 3]) 
      annotation (Placement(transformation(extent={{60,-34},{74,-22}}, rotation
            =0)));
    Modelica.Thermal.HeatTransfer.BodyRadiation w54(
                         Gr=WallEm[5]*WallEm[4]*WallA[5]*SFactor[5, 4]) 
      annotation (Placement(transformation(
          origin={88,-22},
          extent={{-6,-6},{6,6}},
          rotation=270)));
    Modelica.Thermal.HeatTransfer.BodyRadiation win5(
                          Gr=WinEm*WallEm[5]*WinA*SFactor[WinIdx, 5]) 
      annotation (Placement(transformation(extent={{-66,-10},{-52,2}}, rotation
            =0)));
  equation
    connect(wall1, w12.port_a) annotation (Line(points={{-90,50},{-38,50},{-38,
            32}}, color={191,0,0}));
    connect(w12.port_b, wall2) annotation (Line(points={{-38,20},{-38,-50},{-90,
            -50}}, color={191,0,0}));
    connect(wall2, w23.port_a) annotation (Line(points={{-90,-50},{-38,-50},{
            -38,-72},{-28,-72}}, color={191,0,0}));
    connect(wall3, w23.port_b) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-72},{-14,-72}}, color={191,0,0}));
    connect(wall3, w34.port_a) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-72},{20,-72}}, color={191,0,0}));
    connect(w34.port_b, wall4) annotation (Line(points={{34,-72},{38,-72},{38,
            -50},{90,-50}}, color={191,0,0}));
    connect(wall4, w46.port_a) annotation (Line(points={{90,-50},{48,-50},{48,
            -32}}, color={191,0,0}));
    connect(w46.port_b, wall6) annotation (Line(points={{48,-20},{48,50},{90,50}}, 
          color={191,0,0}));
    connect(w67.port_a, wall6) annotation (Line(points={{24,70},{70,70},{70,50},
            {90,50}}, color={191,0,0}));
    connect(w67.port_b, wall7) annotation (Line(points={{10,70},{5.55112e-16,70},
            {5.55112e-16,90}}, color={191,0,0}));
    connect(wall7, w71.port_a) annotation (Line(points={{5.55112e-16,90},{
            5.55112e-16,70},{-18,70}}, color={191,0,0}));
    connect(w71.port_b, wall1) annotation (Line(points={{-32,70},{-38,70},{-38,
            50},{-90,50}}, color={191,0,0}));
    connect(wall1, w13.port_a) annotation (Line(points={{-90,50},{-26,50},{-26,
            -8}}, color={191,0,0}));
    connect(w13.port_b, wall3) annotation (Line(points={{-26,-20},{-26,-60},{
            -5.55112e-16,-60},{-5.55112e-16,-90}}, color={191,0,0}));
    connect(wall2, w24.port_a) annotation (Line(points={{-90,-50},{-6,-50}},
          color={191,0,0}));
    connect(w24.port_b, wall4) annotation (Line(points={{6,-50},{90,-50}},
          color={191,0,0}));
    connect(wall3, w36.port_a) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-60},{38,-60},{38,10}}, color={191,0,0}));
    connect(w36.port_b, wall6) annotation (Line(points={{38,22},{38,50},{90,50}}, 
          color={191,0,0}));
    connect(w47.port_a, wall4) annotation (Line(points={{22,-10},{22,-26},{38,
            -26},{38,-50},{90,-50}}, color={191,0,0}));
    connect(w47.port_b, wall7) annotation (Line(points={{22,2},{22,60},{
            5.55112e-16,60},{5.55112e-16,90}}, color={191,0,0}));
    connect(w61.port_a, wall6) annotation (Line(points={{18,50},{90,50}}, color
          ={191,0,0}));
    connect(w61.port_b, wall1) annotation (Line(points={{4,50},{-90,50}}, color
          ={191,0,0}));
    connect(wall7, w72.port_a) annotation (Line(points={{5.55112e-16,90},{0,70},
            {0,50},{-16,50},{-16,26}}, color={191,0,0}));
    connect(w72.port_b, wall2) annotation (Line(points={{-16,14},{-16,-50},{-90,
            -50}}, color={191,0,0}));
    connect(wall1, w14.port_a) annotation (Line(points={{-90,50},{-8,50},{-8,
            -38},{20,-38}}, color={191,0,0}));
    connect(w14.port_b, wall4) annotation (Line(points={{34,-38},{38,-38},{38,
            -50},{90,-50}}, color={191,0,0}));
    connect(wall2, w26.port_a) annotation (Line(points={{-90,-50},{-16,-50},{
            -16,-38},{-8,-38},{-8,30},{-2,30}}, color={191,0,0}));
    connect(w26.port_b, wall6) annotation (Line(points={{12,30},{38,30},{38,50},
            {90,50}}, color={191,0,0}));
    connect(wall3, w37.port_a) annotation (Line(points={{-5.55112e-16,-90},{
            -5.55112e-16,-72},{14,-72},{14,-38},{2,-38},{2,-14}}, color={191,0,
            0}));
    connect(w37.port_b, wall7) annotation (Line(points={{2,-2},{2,16},{-8,16},{
            -8,50},{0,50},{0,90},{5.55112e-16,90}}, color={191,0,0}));
    connect(win1.port_a, win) annotation (Line(points={{-80,20},{-80,20},{-80,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win1.port_b, wall1) annotation (Line(points={{-80,32},{-80,32},{-80,
            50},{-90,50}}, color={191,0,0}));
    connect(win2.port_a, win) annotation (Line(points={{-80,-20},{-80,-20},{-80,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win2.port_b, wall2) annotation (Line(points={{-80,-32},{-80,-32},{
            -80,-50},{-90,-50}}, color={191,0,0}));
    connect(win3.port_a, win) annotation (Line(points={{-66,-40},{-70,-40},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win3.port_b, wall3) annotation (Line(points={{-52,-40},{-38,-40},{
            -38,-50},{-26,-50},{-26,-60},{-5.55112e-16,-60},{-5.55112e-16,-90}}, 
          color={191,0,0}));
    connect(win4.port_a, win) annotation (Line(points={{-66,-22},{-70,-22},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win4.port_b, wall4) annotation (Line(points={{-52,-22},{-38,-22},{
            -38,-50},{-26,-50},{-26,-60},{38,-60},{38,-50},{90,-50}}, color={
            191,0,0}));
    connect(win6.port_a, win) annotation (Line(points={{-66,14},{-70,14},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win7.port_a, win) annotation (Line(points={{-66,32},{-70,32},{-70,
            5.55112e-16},{-90,5.55112e-16}}, color={191,0,0}));
    connect(win6.port_b, wall6) annotation (Line(points={{-52,14},{-26,14},{-26,
            50},{0,50},{0,60},{48,60},{48,50},{90,50}}, color={191,0,0}));
    connect(win7.port_b, wall7) annotation (Line(points={{-52,32},{-48,32},{-48,
            50},{5.55112e-16,50},{5.55112e-16,90}}, color={191,0,0}));
    connect(wall5, w56.port_a) annotation (Line(
        points={{90,5.55112e-16},{90,0},{88,0},{88,22}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w54.port_a, wall5) annotation (Line(
        points={{88,-16},{88,5.55112e-16},{90,5.55112e-16}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w57.port_b, wall5) annotation (Line(
        points={{74,26},{78,26},{78,5.55112e-16},{90,5.55112e-16}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w51.port_b, wall5) annotation (Line(
        points={{74,8},{78,8},{78,5.55112e-16},{90,5.55112e-16}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w52.port_b, wall5) annotation (Line(
        points={{74,-10},{78,-10},{78,5.55112e-16},{90,5.55112e-16}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w53.port_b, wall5) annotation (Line(
        points={{74,-28},{78,-28},{78,5.55112e-16},{90,5.55112e-16}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w54.port_b, wall4) annotation (Line(
        points={{88,-28},{88,-28},{88,-50},{90,-50}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w53.port_a, wall3) annotation (Line(
        points={{60,-28},{56,-28},{56,-50},{14,-50},{14,-72},{-5.55112e-16,-72},
            {-5.55112e-16,-90}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w52.port_a, wall2) annotation (Line(
        points={{60,-10},{56,-10},{56,-50},{14,-50},{14,-60},{-26,-60},{-26,-50},
            {-90,-50}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w51.port_a, wall1) annotation (Line(
        points={{60,8},{56,8},{56,50},{48,50},{48,60},{0,60},{0,50},{-90,50}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(w57.port_a, wall7) annotation (Line(
        points={{60,26},{56,26},{56,50},{48,50},{48,60},{5.55112e-16,60},{
            5.55112e-16,90}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(win5.port_a, win) annotation (Line(
        points={{-66,-4},{-70,-4},{-70,5.55112e-16},{-90,5.55112e-16}},
        color={191,0,0},
        pattern=LinePattern.None));
    connect(win5.port_b, wall5) annotation (Line(
        points={{-52,-4},{-38,-4},{-38,-50},{-16,-50},{-16,-38},{14,-38},{14,
            -50},{38,-50},{38,5.55112e-16},{90,5.55112e-16}},
        color={191,0,0},
        pattern=LinePattern.None));
  end RadiationExchange7win1;

  model NaturalVentilation
    "Model of the natural ventilation (empirical expression)"
    extends Iface.BlockIcon;
    annotation (Icon(graphics={
          Ellipse(
            extent={{-10,10},{10,-10}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={0,0,0},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-30,14},{-30,-8},{-80,0},{-80,40},{-30,14}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-10,30},{14,30},{40,80},{0,80},{-10,30}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{30,10},{30,-14},{80,-40},{80,0},{30,10}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-14,-30},{10,-30},{0,-80},{-40,-80},{-14,-30}},
            lineColor={0,0,255},
            pattern=LinePattern.None,
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid)}),
                                        Diagram(graphics));
    parameter SI.Volume V "volume of the room";
    parameter Real n(final unit="1/h") "number of air exchanges per hour";
    HT.Interfaces.HeatPort_a InsideAir 
      annotation (Placement(transformation(extent={{-120,-10},{-100,10}},
            rotation=0)));
    HT.Interfaces.HeatPort_b OutsideAir 
      annotation (Placement(transformation(extent={{100,-10},{120,10}},
            rotation=0)));
  equation
    InsideAir.Q_flow = -0.394*(1 - InsideAir.T/273.15)*V*(InsideAir.T -
      OutsideAir.T)*n;
    InsideAir.Q_flow + OutsideAir.Q_flow = 0;
  end NaturalVentilation;
end ThermalInteractions;
