within SolarHouse;
package Geometry "Helpful geometrical rutines"
  import Modelica.Math.*;
  function norm "Norm of the vector"
    extends Modelica.Icons.Function;
    input Real vec[:];
    output Real res;
  algorithm
    res := sqrt(sum(vec[i]^2 for i in 1:size(vec, 1)));
  end norm;

  function Normal "Calculate normal vector from frist two edges of polynom"
    extends Modelica.Icons.Function;
    input Real poly[3, :];
    output Real normvec[3];
  algorithm
    normvec := cross(poly[:, 2] - poly[:, 1], poly[:, 3] - poly[:, 2]);
    if (normvec*normvec <> 0) then
      normvec := normvec/norm(normvec);
    end if;
  end Normal;

  function RelTransMatrix "Computes transformation matrix"
    extends Modelica.Icons.Function;
    input Real r1[3];
    input Real r2[3];
    input Real r3[3];
    output Real A[4, 4];
  protected
    Real e1[3] "New x-axis";
    Real e2[3] "New y-axis";
    Real e3[3] "New z-axis";
    Real R[4, 4] "Rotational matrix";
    Real T[4, 4] "Translation matrix";
  algorithm
    e1 := r2 - r1;
    e2 := r3 - r2;
    e3 := cross(e1, e2);
    e3 := e3/norm(e3);
    e1 := e1/norm(e1);
    e2 := cross(e3, e1);
    R := diagonal(vector(ones(1, 4)));
    R[1, 1:3] := e1;
    R[2, 1:3] := e2;
    R[3, 1:3] := e3;
    //  R[1:3,1:3] := [array(e1); array(e2); array(e3)];
    T := diagonal(vector(ones(1, 4)));
    T[1:3, 4] := -r1;
    A := R*T;
  end RelTransMatrix;

  function ToPlane "Project on xy plane"
    extends Modelica.Icons.Function;
    input Real r[3, :] "Polygon to be projected";
    input Real e[3] "Direction of projection (projection ray)";
    output Real p[2, size(r, 2)] "Projected polygon";
  algorithm
    for i in 1:size(r, 2) loop
      //p := cat(2,p,[r[1,i]-r[3,i] * e[1]/e[3];r[2,i]-r[3,i] * e[2]/e[3]]);
      p[:, i] := {r[1, i] - r[3, i]*e[1]/e[3],r[2, i] - r[3, i]*e[2]/e[3]};
    end for;
    // check for anticlockwise orientation
    if not onLeft(
            p[:, 1],
            p[:, 2],
            p[:, 3]) then
      p := p[:, end:-1:1];
    end if;
  end ToPlane;

  function ToRelCoord "Transforms vector to relative coordinate system"
    extends Modelica.Icons.Function;
    input Real T[4, 4] "Transformation matrix";
    input Real v[3] "Input vector";
    output Real res[3] "Result";
  protected
    Real p[4] "auxillary vector variable";
  algorithm
    p := vector(T*[v; 1]);
    res := p[1:3];
  end ToRelCoord;

  function PolyToRelCoord "Transforms polygon to relative coordinate system"
    extends Modelica.Icons.Function;
    input Real T[4, 4] "Transformation matrix";
    input SI.Position s[3, :] "Polygon to be transformed";
    output SI.Position r[3, size(s, 2)] "Transformed polygon";
  protected
    Real p[4] "auxillary vector variable";
  algorithm
    for i in 1:size(s, 2) loop
      p := vector(T*[s[:, i]; 1]);
      r[:, i] := p[1:3];
      //r := cat(2,r,[p[1];p[2];p[3]]);
    end for;
  end PolyToRelCoord;

  function PolygonIntersection "Calculates intersection polygon"
    extends Modelica.Icons.Function;
    input Real poly1[2, :];
    input Real poly2[2, :];
    output Real intersec[2, :];
    //. auxillary variables
  protected
    Real calliper[2] "Direction vector of callipers";
    Integer pc1 "step counter for first polygon";
    Integer pc2 "step counter for second polygon";
    Integer v1 "current calliper1 position vertex index";
    Integer v2 "current calliper2 position vertex index";
    Integer v1n "next vertex index of polygon 1";
    Integer v2n "next vertex index of polygon 2";
    Integer sp1 "number of vertices in first polygon";
    Integer sp2 "number of vertices in second polygon";
    Real edge1[2] "Current edge of poly1 direction vector";
    Real edge2[2] "Current edge of poly2 direction vector";
    Real R1[2, 2] "rotational matrix of first polygon";
    Real R2[2, 2] "rotational matrix of second polygon";
    Boolean left "True if second calliper is on left side of the first";
    Integer pockets[2, :] "List of all the pocket lids found so far";
    Boolean pockdir[:] "Direction of pocket lids: true => p1 -> p2";
    Boolean status;
  algorithm
    sp1 := size(poly1, 2);
    sp2 := size(poly2, 2);
    //
    //  1. part: Computing the Convex Hull
    //.
    // initializing first calliper - finding leftmost vertex
    v1 := 1;
    for i in 2:sp1 loop
      v1 := if (poly1[1, i] < poly1[1, v1]) then i else v1;
    end for;
    // initializing second calliper
    v2 := 1;
    for i in 2:sp2 loop
      v2 := if (poly2[1, i] < poly2[1, v2]) then i else v2;
    end for;
    calliper := {0,1};
    // is second calliper on left side of the first?
    left := poly2[1, v2] < poly1[1, v1];
    // initalize number of steps left
    pc1 := sp1;
    pc2 := sp2;
    // loop until both callipers return to their start position
    while (pc1 >= 0) and (pc2 >= 0) loop
      // calculate direction vectors of both edges
      edge1 := poly1[:, prev(v1, sp1)] - poly1[:, v1];
      edge1 := edge1/norm(edge1);
      edge2 := poly2[:, prev(v2, sp2)] - poly2[:, v2];
      edge2 := edge2/norm(edge2);
      // calculate rotational matrices
      R1 := Rotational2D(calliper, edge1);
      R2 := Rotational2D(calliper, edge2);
      // determine smallest angle of rotation (cos(th1) > cos(th2))
      // and rotate callipers
      if (R1[1, 1] >= R2[1, 1]) and (pc1 >= 0) then
        calliper := R1*calliper;
        // pocket found? (side has changed)
        if left <> onLeft(
                poly1[:, v1],
                poly1[:, v1] + calliper,
                poly2[:, v2]) then
          // add to the list of found pockets
          // left -> right => [v1,v2]; right -> left => [v2,v1]
          pockets := cat(
                2,
                pockets,
                [v1; v2]);
          pockdir := cat(
                1,
                pockdir,
                {left});
          left := not left;
        end if;
        // update indices
        v1 := prev(v1, sp1);
        pc1 := pc1 - 1;
      else
        calliper := R2*calliper;
        // pocket found? (side has changed)
        if left <> onLeft(
                poly1[:, v1],
                poly1[:, v1] + calliper,
                poly2[:, v2]) then
          // add to the list of found pockets
          // left -> right => [v1,v2]; right -> left => [v2,v1]
          pockets := cat(
                2,
                pockets,
                [v1; v2]);
          pockdir := cat(
                1,
                pockdir,
                {left});
          left := not left;
        end if;
        // update indices
        v2 := prev(v2, sp2);
        pc2 := pc2 - 1;
      end if;
    end while;
    //
    // 2. part: Finding the intersections in the pockets
    //
    if size(pockets, 2) > 0 then
      status := true;
      for i in 1:size(pockets, 2) loop
        if status then
          // break statment work-around
          if pockdir[i] then
            // find intersecting edges
            (v2,v1,status) := FindBottom(cat(
                  2,
                  poly2[:, pockets[2, i]:-1:1],
                  poly2[:, end:-1:pockets[2, i] + 1]), cat(
                  2,
                  poly1[:, pockets[1, i]:end],
                  poly1[:, 1:pockets[1, i] - 1]));
            // recalculate indices
            v1 := mod(v1 + pockets[1, i] - 2, size(poly1, 2)) + 1;
            v2 := mod(size(poly2, 2) + pockets[2, i] - v2 - 1, size(poly2, 2))
               + 1;
          else
            // find intersecting edges
            (v1,v2,status) := FindBottom(cat(
                  2,
                  poly1[:, pockets[1, i]:-1:1],
                  poly1[:, end:-1:pockets[1, i] + 1]), cat(
                  2,
                  poly2[:, pockets[2, i]:end],
                  poly2[:, 1:pockets[2, i] - 1]));
            // recalculate indices
            v1 := mod(size(poly1, 2) + pockets[1, i] - v1 - 1, size(poly1, 2))
               + 1;
            v2 := mod(v2 + pockets[2, i] - 2, size(poly2, 2)) + 1;
          end if;
          // save back to same structure
          pockets[:, i] := {v1,v2};
        end if;
      end for;
      //
      // 3. part: Merging
      //
      if status then
        // status == intersections found
        for i in size(pockets, 2):-1:1 loop
          v1 := pockets[1, i];
          v2 := pockets[2, i];
          v1n := next(v1, size(poly1, 2));
          v2n := next(v2, size(poly2, 2));
          // calculate intersection
          edge2 := LineIntersection(
                poly1[:, v1],
                poly1[:, v1n],
                poly2[:, v2],
                poly2[:, v2n]);
          intersec := cat(
                2,
                intersec,
                [edge2[1]; edge2[2]]);
          if (pockdir[i]) then
            while (v1 <> pockets[1, prev(i, size(pockets, 2))]) loop
              intersec := cat(
                    2,
                    intersec,
                    [poly1[1, v1n]; poly1[2, v1n]]);
              v1 := v1n;
              v1n := next(v1n, size(poly1, 2));
            end while;
          else
            while (v2 <> pockets[2, prev(i, size(pockets, 2))]) loop
              intersec := cat(
                    2,
                    intersec,
                    [poly2[1, v2n]; poly2[2, v2n]]);
              v2 := v2n;
              v2n := next(v2n, size(poly2, 2));
            end while;
          end if;
        end for;
      else
        // polygons do not intersect
        intersec := fill(
              0,
              2,
              0);
      end if;
    else
      // there is one polygon inside other
      intersec := if left then poly1 else poly2;
    end if;
  end PolygonIntersection;

public
  function PolygonArea "Calculate area of a polygon"
    extends Modelica.Icons.Function;
    input Real poly[2, :];
    output Real area;
  algorithm
    area := 0;
    for i in 1:size(poly, 2) loop
      area := area + poly[1, i]*(poly[2, next(i, size(poly, 2))] - poly[2,
        prev(i, size(poly, 2))]);
    end for;
    area := area/2;
  end PolygonArea;

public
  function Polygon3DArea "Calculate area of a polygon"
    extends Modelica.Icons.Function;
    input Real poly[3, :];
    input Real normal[3] "plane normalized normal vector";
    output Real area;
  protected
    Real n[3] "normal with absolute components";
  algorithm
    n := abs(normal);
    if (n[1] > n[2]) and (n[1] > n[3]) then
      area := PolygonArea(poly[2:3, :]);
      area := abs(area/n[1]);
    elseif (n[2] > n[1]) and (n[2] > n[3]) then
      area := PolygonArea(poly[{1,3}, :]);
      area := abs(area/n[2]);
    else
      area := PolygonArea(poly[1:2, :]);
      area := abs(area/n[3]);
    end if;
  end Polygon3DArea;

  function LineIntersection "Calculates intersection of two line segments"
    extends Modelica.Icons.Function;
    input Real p1[2];
    input Real p2[2];
    input Real r1[2];
    input Real r2[2];
    output Real inter[2];
  protected
    Real t;
  algorithm
    t := ((r2[1] - r1[1])*(p1[2] - r1[2]) - (r2[2] - r1[2])*(p1[1] - r1[1]))/
      ((r2[2] - r1[2])*(p2[1] - p1[1]) - (r2[1] - r1[1])*(p2[2] - p1[2]));
    inter := p1 + t*(p2 - p1);
  end LineIntersection;

protected
  function FindBottom "Toussaint's algorithm; get instersection point"
    input Real P[2, :];
    input Real Q[2, :];
    output Integer i;
    output Integer j;
    output Boolean status;
  protected
    Boolean finished;
  algorithm
    finished := false;
    i := 1;
    j := 1;
    status := true;
    while not finished loop
      finished := true;
      while (if status then onLeft(
              P[:, i],
              P[:, i + 1],
              Q[:, j + 1]) else false) loop
        j := j + 1;
        finished := false;
        if j == size(Q, 2) then
          status := false;
          finished := true;
        end if;
      end while;
      while (if status then not onLeft(
              Q[:, j],
              Q[:, j + 1],
              P[:, i + 1]) else false) loop
        i := i + 1;
        finished := false;
        if i == size(P, 2) then
          status := false;
          finished := true;
        end if;
      end while;
    end while;
  end FindBottom;
  annotation (Icon(graphics));
public
  function IntersectionArea "Calculates area of intersection polygon"
    extends Modelica.Icons.Function;
    input Real poly1[2, :];
    input Real poly2[2, :];
    output Real area;
    //. auxillary variables
  protected
    Real calliper[2] "Direction vector of callipers";
    Integer pc1 "step counter for first polygon";
    Integer pc2 "step counter for second polygon";
    Integer v1 "current calliper1 position vertex index";
    Integer v2 "current calliper2 position vertex index";
    Integer v1n "next vertex index of polygon 1";
    Integer v2n "next vertex index of polygon 2";
    Integer sp1 "number of vertices in first polygon";
    Integer sp2 "number of vertices in second polygon";
    Real edge1[2] "Current edge of poly1 direction vector";
    Real edge2[2] "Current edge of poly2 direction vector";
    Real R1[2, 2] "rotational matrix of first polygon";
    Real R2[2, 2] "rotational matrix of second polygon";
    Boolean left "True if second calliper is on left side of the first";
    Integer pockets[2, size(poly1, 2) + size(poly2, 2) + 10]
      "List of all the pocket lids found so far";
    Integer pocksz;
    Boolean pockdir[size(poly1, 2) + size(poly2, 2) + 10]
      "Direction of pocket lids: true => p1 -> p2";
    Boolean status;
    // intersection
    Real intersec[2, size(poly1, 2) + size(poly2, 2) + 10];
    Integer intersecsz;
  algorithm
    // array "sizes" intialization
    intersecsz := 1;
    pocksz := 1;
    //
    sp1 := size(poly1, 2);
    sp2 := size(poly2, 2);
    //
    //  1. part: Computing the Convex Hull
    //.
    // initializing first calliper - finding leftmost vertex
    v1 := 1;
    for i in 2:sp1 loop
      v1 := if (poly1[1, i] < poly1[1, v1]) then i else v1;
    end for;
    // initializing second calliper
    v2 := 1;
    for i in 2:sp2 loop
      v2 := if (poly2[1, i] < poly2[1, v2]) then i else v2;
    end for;
    calliper := {0,1};
    // is second calliper on left side of the first?
    left := poly2[1, v2] < poly1[1, v1];
    // initalize number of steps left
    pc1 := sp1;
    pc2 := sp2;
    // loop until both callipers return to their start position
    while (pc1 >= 0) and (pc2 >= 0) loop
      // calculate direction vectors of both edges
      edge1 := poly1[:, prev(v1, sp1)] - poly1[:, v1];
      assert(sum(abs(edge1)) > 0, "Edge1 has zero-length");
      edge1 := edge1/norm(edge1);
      edge2 := poly2[:, prev(v2, sp2)] - poly2[:, v2];
      assert(sum(abs(edge2)) > 0, "Edge2 has zero-length");
      edge2 := edge2/norm(edge2);
      // calculate rotational matrices
      R1 := Rotational2D(calliper, edge1);
      R2 := Rotational2D(calliper, edge2);
      // determine smallest angle of rotation (cos(th1) > cos(th2))
      // and rotate callipers
      if (R1[1, 1] >= R2[1, 1]) and (pc1 >= 0) then
        calliper := R1*calliper;
        // pocket found? (side has changed)
        if left <> onLeft(
                poly1[:, v1],
                poly1[:, v1] + calliper,
                poly2[:, v2]) then
          // add to the list of found pockets
          // left -> right => [v1,v2]; right -> left => [v2,v1]
          pockets[:, pocksz] := {v1,v2};
          pockdir[pocksz] := left;
          pocksz := pocksz + 1;
          left := not left;
        end if;
        // update indices
        v1 := prev(v1, sp1);
        pc1 := pc1 - 1;
      else
        calliper := R2*calliper;
        // pocket found? (side has changed)
        if left <> onLeft(
                poly1[:, v1],
                poly1[:, v1] + calliper,
                poly2[:, v2]) then
          // add to the list of found pockets
          // left -> right => [v1,v2]; right -> left => [v2,v1]
          pockets[:, pocksz] := {v1,v2};
          pockdir[pocksz] := left;
          pocksz := pocksz + 1;
          left := not left;
        end if;
        // update indices
        v2 := prev(v2, sp2);
        pc2 := pc2 - 1;
      end if;
    end while;
    /*
  if mod(pocksz,2) <> then
    pocksz := pocksz -1;
  end if;
*/
    //
    // 2. part: Finding the intersections in the pockets
    //
    if pocksz > 1 then
      status := true;
      for i in 1:pocksz - 1 loop
        if status then
          // break statment work-around
          if pockdir[i] then
            // find intersecting edges
            (v2,v1,status) := FindBottom(cat(
                  2,
                  poly2[:, pockets[2, i]:-1:1],
                  poly2[:, end:-1:pockets[2, i] + 1]), cat(
                  2,
                  poly1[:, pockets[1, i]:end],
                  poly1[:, 1:pockets[1, i] - 1]));
            // recalculate indices
            v1 := mod(v1 + pockets[1, i] - 2, size(poly1, 2)) + 1;
            v2 := mod(size(poly2, 2) + pockets[2, i] - v2 - 1, size(poly2, 2))
               + 1;
          else
            // find intersecting edges
            (v1,v2,status) := FindBottom(cat(
                  2,
                  poly1[:, pockets[1, i]:-1:1],
                  poly1[:, end:-1:pockets[1, i] + 1]), cat(
                  2,
                  poly2[:, pockets[2, i]:end],
                  poly2[:, 1:pockets[2, i] - 1]));
            // recalculate indices
            v1 := mod(size(poly1, 2) + pockets[1, i] - v1 - 1, size(poly1, 2))
               + 1;
            v2 := mod(v2 + pockets[2, i] - 2, size(poly2, 2)) + 1;
          end if;
          // save back to same structure
          pockets[:, i] := {v1,v2};
        end if;
      end for;
      //
      // 3. part: Merging
      //
      if status then
        // status == intersections found
        for i in (pocksz - 1):-1:1 loop
          v1 := pockets[1, i];
          v2 := pockets[2, i];
          v1n := next(v1, size(poly1, 2));
          v2n := next(v2, size(poly2, 2));
          // calculate intersection
          edge2 := LineIntersection(
                poly1[:, v1],
                poly1[:, v1n],
                poly2[:, v2],
                poly2[:, v2n]);
          // intersec := cat(2,intersec,[edge2[1]; edge2[2]]);
          intersec[:, intersecsz] := edge2;
          intersecsz := intersecsz + 1;
          if (pockdir[i]) then
            while (v1 <> pockets[1, prev(i, pocksz - 1)]) loop
              //intersec := cat(2,intersec,[poly1[1,v1n];poly1[2,v1n]]);
              intersec[:, intersecsz] := poly1[:, v1n];
              intersecsz := intersecsz + 1;
              v1 := v1n;
              v1n := next(v1n, size(poly1, 2));
            end while;
          else
            while (v2 <> pockets[2, prev(i, pocksz - 1)]) loop
              // intersec := cat(2,intersec,[poly2[1,v2n];poly2[2,v2n]]);
              intersec[:, intersecsz] := poly2[:, v2n];
              intersecsz := intersecsz + 1;
              v2 := v2n;
              v2n := next(v2n, size(poly2, 2));
            end while;
          end if;
        end for;
      else
        // polygons do not intersect
        //intersec := fill(0,2,0);
        intersecsz := 1;
      end if;
    else
      // there is one polygon inside other
      // intersec := if left then poly1 else poly2;
      if left then
        intersec[:, 1:sp1] := poly1;
        intersecsz := sp1 + 1;
      else
        intersec[:, 1:sp2] := poly2;
        intersecsz := sp2 + 1;
      end if;
    end if;
    // calulate area
    area := PolygonArea(intersec[:, 1:intersecsz - 1]);
  end IntersectionArea;

  function EulerAngles "Computes transformation matrix for given Euler angles"
    extends Modelica.Icons.Function;
    input SI.Angle phi;
    input SI.Angle theta;
    input SI.Angle psi;
    output Real A[4, 4];
  algorithm
    A := diagonal(ones(4));
    A[1, 1] := cos(psi)*cos(phi) - cos(theta)*sin(phi)*sin(psi);
    A[1, 2] := cos(psi)*sin(phi) + cos(theta)*cos(phi)*sin(psi);
    A[1, 3] := sin(psi)*sin(theta);
    A[2, 1] := -sin(psi)*cos(phi) - cos(theta)*sin(phi)*cos(psi);
    A[2, 2] := -sin(psi)*sin(phi) + cos(theta)*cos(phi)*cos(psi);
    A[2, 3] := cos(psi)*sin(theta);
    A[3, 1] := sin(theta)*sin(phi);
    A[3, 2] := -sin(theta)*cos(phi);
    A[3, 3] := cos(theta);
  end EulerAngles;

protected
  record Line
    Real N[2];
    Real c;
  end Line;

  function computeLine
    input Real P1[2];
    input Real P2[2];
    output Line params;
  protected
    Real n[2];
    Real d;
  algorithm
    n := P2 - P1;
    params.N := {n[2],-n[1]}/norm(n);
    params.c := P1*params.N;
  end computeLine;

  function Rotational2D
    "Compute 2D rotational matrix which rotates vec1 to vec2"
    input Real vec1[2];
    input Real vec2[2];
    output Real rot[2, 2];
  algorithm
    rot[1, 1] := vec1*vec2;
    rot[1, 2] := -(vec1[1]*vec2[2] - vec2[1]*vec1[2]);
    rot[2, 1] := -rot[1, 2];
    rot[2, 2] := rot[1, 1];
  end Rotational2D;

  function onLeft
    "true if point is on left side of the line determined by p1 and p2"
    input Real p1[2] "First point on line";
    input Real p2[2] "Second point on line";
    input Real p[2];
    output Boolean left;
  algorithm
    left := (p1[1]*(p2[2] - p[2]) + p2[1]*(p[2] - p1[2]) + p[1]*(p1[2] - p2[2]))
       > 0;
  end onLeft;

  function next "Calculate rotating index"
    input Integer i;
    input Integer n;
    output Integer index;
  algorithm
    index := if i == n then 1 else i + 1;
  end next;

  function prev "Calculate rotating index"
    input Integer i;
    input Integer n;
    output Integer index;
  algorithm
    index := if i == 1 then n else i - 1;
  end prev;

public
  function Translation "Computes transformation matrix for given Euler angles"
    extends Modelica.Icons.Function;
    input SI.Position x;
    input SI.Position y;
    input SI.Position z;
    output Real A[4, 4];
  algorithm
    A := diagonal(ones(4));
    A[1, 4] := x;
    A[2, 4] := y;
    A[3, 4] := z;
  end Translation;

public
  block RotationFunc "Transforms vector"
    extends Modelica.Icons.Function;
    input Real u[3];
    output Real y[3];
    parameter Real TransMatrix[4, 4];
  equation
    y = TransMatrix[1:3, 1]*u[1] + TransMatrix[1:3, 2]*u[2] + TransMatrix[1:3,
      3]*u[3];
  end RotationFunc;

  function SFactorParallel "Shape factor for parallel planes"
    extends Modelica.Icons.Function;
    input SI.Length a;
    input SI.Length b;
    input SI.Length c;
    output Real sfactor;
  protected
    Real X;
    Real Y;
  algorithm
    X := a/b;
    Y := c/b;
    sfactor := (1/2*Math.log((1 + X^2)*(1 + Y^2)/(1 + X^2 + Y^2)) + X*sqrt(1
       + Y^2)*Math.atan(X/sqrt(1 + X^2)) + Y*sqrt(1 + X^2)*Math.atan(Y/sqrt(1
       + X^2)) - X*Math.atan(X) - Y*Math.atan(Y))*2/(Const.pi*X*Y);
  end SFactorParallel;

  function SFactorPerpendic "Shape factor for perpendicular planes"
    extends Modelica.Icons.Function;
    input SI.Length a;
    input SI.Length b;
    input SI.Length c;
    output Real sfactor;
  protected
    Real X;
    Real Y;
  algorithm
    X := a/b;
    Y := c/b;
    sfactor := (Y*Math.atan(1/Y) + X*Math.atan(1/X) - sqrt(X^2 + Y^2)*
      Math.atan(1/sqrt(X^2 + Y^2)) +.25 *Math.log((1 + X^2)*(1 + Y^2)/(1 + X^
      2 + Y^2)) + (Y^2/4)*Math.log(Y^2*(1 + X^2 + Y^2)/(1 + X^2)/(1 + Y^2))
       + (X^2/4)*Math.log(X^2*(1 + X^2 + Y^2)/(1 + X^2)/(1 + Y^2)))/(Const.pi
      *Y);
  end SFactorPerpendic;

protected
  function SFactorNumeric "Shape factor for perpendicular planes"
    extends Modelica.Icons.Function;
    input SI.Position S1[3, :];
    input SI.Position S2[3, :];
    input Integer N;
    output Real sfactor;
  protected
    function f "integrated function"
      input Real n1[3] "normal of the first surface";
      input Real n2[3] "normal of the second surface";
      input Real r[3] "vector between points";
      output Real res;
    protected
      Real denum "denominator";
    algorithm
      denum := r*r;
      res := if denum > 0 then -(n1*r)*(n2*r)/denum^2 else 1;
    end f;

    function Received "Radiation recived"
      input SI.Position p[3] "recieving point";
      input SI.Position S[3, :] "emitting surface";
      input Integer N;
      input Real n1[3] "normale to the first surface";
      input Real n2[3] "normale to the second surface";
      output Real res;
    protected
      SI.Position g[3] "lower bound";
      SI.Position h[3] "upper bound";
      Real series[N - 1];
    algorithm
      res := 0;
      for v in 2:(size(S, 2) - 1) loop
        series := zeros(N - 1);
        for i in 0:(N - 2) loop
          g := S[:, 1] + (i/(N - 1))*(S[:, v + 1] - S[:, 1]);
          h := S[:, v] + (i/(N - 1))*(S[:, v + 1] - S[:, v]);
          series[i + 1] := simpson(
                  {f(
                    n1,
                    n2,
                    p - g - t*(h - g)) for t in linspace(
                    0,
                    1,
                    N - 1)},
                  0,
                  1);
        end for;
        res := res + simpson(
                series,
                0,
                (N - 2)/(N - 1));
      end for;
    end Received;
    Real n1[3];
    Real n2[3];
    SI.Position g[3] "lower bound";
    SI.Position h[3] "upper bound";
    Real series[N - 1];
  algorithm
    n1 := -Geometry.Normal(S1);
    n2 := -Geometry.Normal(S2);
    sfactor := 0;
    for v in 2:(size(S1, 2) - 1) loop
      series := zeros(N - 1);
      for i in 0:(N - 2) loop
        g := S1[:, 1] + (i/(N - 2))*(S1[:, v + 1] - S1[:, 1]);
        h := S1[:, v] + (i/(N - 2))*(S1[:, v + 1] - S1[:, v]);
        series[i + 1] := simpson(
              {Received(
                g + t*(h - g),
                S2,
                N,
                n1,
                n2) for t in linspace(
                0,
                1,
                N - 1)},
              0,
              1);
      end for;
      sfactor := sfactor + simpson(
            series,
            0,
            1);
    end for;
  end SFactorNumeric;

public
  function SFactorNumerical "Shape factor calculated with Gauss-Kronrod rules"
    extends Modelica.Icons.Function;
    annotation (Include="#include \"sfactor.c\"",Library={"gsl"});
    input Real[3, :] radiated "Polygon shaped radiation surface";
    input Real[3, :] irradiated "Polygon shaped irradiated surface";
    output Real factor;
  protected
    Integer status "Status code, 0 if successful, it is ignored";
  external "C" status=  sfactor(
          radiated,
          size(radiated, 1),
          size(radiated, 2),
          irradiated,
          size(irradiated, 1),
          size(irradiated, 2),
          factor);
  end SFactorNumerical;

public
  function roller "Callculates uncurtained window"
    extends Modelica.Icons.Function;
    input SI.Length window[:, :];
    input Integer top[:];
    input Integer bottom[size(top, 1)];
    input Real rollerpos;
    output Real curtained[size(window, 1), size(window, 2)];
  algorithm
    curtained := window;
    for i in 1:size(top, 1) loop
      curtained[:, top[i]] := window[:, top[i]] + rollerpos*(window[:, bottom[
        i]] - window[:, top[i]]);
    end for;
  end roller;
end Geometry;
