within SolarHouse;
package Weather "Components modelling surroundings of the building"
  package HelperFunctions
    record tm
      Real hour;
      Integer day;
      Integer month;
    end tm;

    function mktime
      input SolarHouse.Weather.HelperFunctions.tm trec;
      output SI.Time retval;
      parameter Integer mon_day[12]={0,31,59,90,120,151,181,212,243,273,304,334};
    algorithm
      retval := ((mon_day[trec.month] + trec.day - 1)*24 + trec.hour)*3600;
    end mktime;

    function time_indices
      input SI.Time time_t;
      output Real hour;
      output Integer yday;
    algorithm
      yday := 1 + integer(mod(time_t/(24*3600), 365));
      hour := mod(time_t, 24*3600)/3600;
    end time_indices;
  end HelperFunctions;

  model SunRadiation
    "Position of the sun on the sky and intensity of the solar radiation"
    extends Iface.BlockIcon;
    parameter SolarHouse.Weather.HelperFunctions.tm start_time(
      hour=0,
      day=15,
      month=1);
    parameter SI.Angle beta=SI.Conversions.from_deg(46.05)
      "Geographical latitude";
    //SI.HeatFlowRate q_dif "Diffuse radiation";
    Modelica.Blocks.Interfaces.RealInput q_dir
      "Radiation comming directly from sun" 
      annotation (Placement(transformation(extent={{-120,40},{-100,60}},
            rotation=0)));
    Modelica.Blocks.Interfaces.RealInput q_dif "Portion of direct radiation" 
      annotation (Placement(transformation(extent={{-120,-10},{-100,10}},
            rotation=0)));
    Modelica.Blocks.Interfaces.RealInput OBL "Degree of cloudiness" 
      annotation (Placement(transformation(extent={{-120,-60},{-100,-40}},
            rotation=0)));
    Interfaces.DirLightOut Ray annotation (Placement(transformation(extent={{
              100,-10},{120,10}}, rotation=0)));
  protected
    parameter SI.Time stime=SolarHouse.Weather.HelperFunctions.mktime(
                                   start_time);
    Integer yday;
    Real hour;
    SI.Time Eqt "equation of time";
    SI.Angle psi "Earth rotation (solar azimuth)";
    Real cos_psi;
    Real sin_psi;
    SI.Angle delta "Earth revolution (solar latitude)";
    // direction calculation
    Real sin_a "Sinus of solar incidence angle";
    Real cos_a;
    Real sin_phi;
    Real cos_phi;
    //  SI.Angle alpha;
    //SI.Angle phi;
    Real cos_phi_sign;
  equation
    // calculate time indices
    (hour,yday) = SolarHouse.Weather.HelperFunctions.time_indices(
                               stime + time);
    if yday <= 106 then
      Eqt = -14.2/60*sin(Const.pi*(yday + 7)/111);
    elseif yday <= 166 then
      Eqt = 4.0/60*sin(Const.pi*(yday - 106)/59);
    elseif yday <= 246 then
      Eqt = -6.5/60*sin(Const.pi*(yday - 166)/80);
    else
      Eqt = 16.4/60*sin(Const.pi*(yday - 247)/113);
    end if;
    // calculate angles
    psi = Const.pi*(1 - (hour + Eqt)/12);
    // DEBUG
    cos_psi = cos(psi);
    sin_psi = sin(psi);
    delta = Const.pi/180*23.45*sin(2*Const.pi*(284 + yday)/365);
    // calclating direction vector
    sin_a = sin(beta)*sin(delta) + cos(beta)*cos(delta)*cos(psi);
    // check for daylight
    cos_a = sqrt(1 - (sin_a)^2);
    sin_phi = sin(psi)*cos(delta)/cos_a;
    cos_phi_sign = noEvent(sign(cos(psi) - sin(psi)^2*cos(delta)/((cos_a)^2)*
      (sin_a)*cos(beta)));
    cos_phi = cos_phi_sign*sqrt(1 - (sin_phi)^2);
    if sin_a > 0 then
      Ray.dir = {cos_a*cos_phi,cos_a*sin_phi,-sin_a};
      // direct sun radiation on plane
      Ray.q_dir = (1 - OBL)*q_dir;
      // diffuse sun radiation on plane
      Ray.q_dif = q_dif;
      // DEBUG
      //  alpha := Math.asin(sin_a);
      // phi := Math.asin(sin_phi);
    else
      Ray.dir = {0,0,0};
      Ray.q_dir = 0;
      Ray.q_dif = 0;
    end if;
    annotation (Diagram(graphics),
                         Icon(graphics={Polygon(
            points={{-10,30},{0,88},{10,30},{62,62},{30,10},{86,0},{30,-10},{60,
                -60},{10,-30},{0,-86},{-12,-30},{-60,-60},{-30,-10},{-86,0},{
                -30,10},{-62,62},{-10,30}},
            lineColor={255,128,0},
            fillColor={255,128,0},
            fillPattern=FillPattern.Solid), Ellipse(
            extent={{-28,28},{28,-28}},
            lineColor={255,255,85},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid)}));
  end SunRadiation;

  model SunRadiationEstimatedDiffuseRadiation
    "Position of the sun on the sky and intensity of the solar radiation"
    extends Iface.BlockIcon;
    parameter SolarHouse.Weather.HelperFunctions.tm start_time(
      hour=0,
      day=15,
      month=1);
    parameter SI.Angle beta=SI.Conversions.from_deg(46.05)
      "Geographical latitude";
    //SI.HeatFlowRate q_dif "Diffuse radiation";
    Modelica.Blocks.Interfaces.RealInput q_sol
      "Radiation comming directly from the sun" 
      annotation (Placement(transformation(extent={{-120,40},{-100,60}},
            rotation=0)));
    Modelica.Blocks.Interfaces.RealInput DIG "Portion of direct radiation" 
      annotation (Placement(transformation(extent={{-120,-10},{-100,10}},
            rotation=0)));
    Modelica.Blocks.Interfaces.RealInput OBL "Degree of cloudiness" 
      annotation (Placement(transformation(extent={{-120,-60},{-100,-40}},
            rotation=0)));
    Interfaces.DirLightOut Ray annotation (Placement(transformation(extent={{
              100,-10},{120,10}}, rotation=0)));
  protected
    parameter SI.Time stime=SolarHouse.Weather.HelperFunctions.mktime(
                                   start_time);
    Integer yday;
    Real hour;
    SI.Time Eqt "equation of time";
    SI.Angle psi "Earth rotation (solar azimuth)";
    Real cos_psi;
    Real sin_psi;
    SI.Angle delta "Earth revolution (solar latitude)";
    // direction calculation
    Real sin_a "Sinus of solar incidence angle";
    Real cos_a;
    Real sin_phi;
    Real cos_phi;
    Real cos_phi_sign;
  equation
    // calculate time indices
    (hour,yday) = SolarHouse.Weather.HelperFunctions.time_indices(
                               stime + time);
    if yday <= 106 then
      Eqt = -14.2/60*sin(Const.pi*(yday + 7)/111);
    elseif yday <= 166 then
      Eqt = 4.0/60*sin(Const.pi*(yday - 106)/59);
    elseif yday <= 246 then
      Eqt = -6.5/60*sin(Const.pi*(yday - 166)/80);
    else
      Eqt = 16.4/60*sin(Const.pi*(yday - 247)/113);
    end if;
    // calculate angles
    psi = Const.pi*(1 - (hour + Eqt)/12);
    // DEBUG
    cos_psi = cos(psi);
    sin_psi = sin(psi);
    delta = Const.pi/180*23.45*sin(2*Const.pi*(284 + yday)/365);
    // calclating direction vector
    sin_a = sin(beta)*sin(delta) + cos(beta)*cos(delta)*cos(psi);
    // check for daylight
    cos_a = sqrt(1 - (sin_a)^2);
    sin_phi = sin(psi)*cos(delta)/cos_a;
    cos_phi_sign = noEvent(sign(cos(psi) - sin(psi)^2*cos(delta)/((cos_a)^2)*
      (sin_a)*cos(beta)));
    cos_phi = cos_phi_sign*sqrt(1 - (sin_phi)^2);
    if sin_a > 0 then
      Ray.dir = {cos_a*cos_phi,cos_a*sin_phi,-sin_a};
      // direct sun radiation on plane
      Ray.q_dir = DIG*(1 - OBL)*q_sol;
      // diffuse sun radiation on plane
      Ray.q_dif = q_sol - Ray.q_dir;
      // DEBUG
      //  alpha := Math.asin(sin_a);
      // phi := Math.asin(sin_phi);
    else
      Ray.dir = {0,0,0};
      Ray.q_dir = 0;
      Ray.q_dif = 0;
    end if;
    annotation (Diagram(graphics),
                         Icon(graphics={Polygon(
            points={{-10,30},{0,88},{10,30},{62,62},{30,10},{86,0},{30,-10},{60,
                -60},{10,-30},{0,-86},{-12,-30},{-60,-60},{-30,-10},{-86,0},{
                -30,10},{-62,62},{-10,30}},
            lineColor={255,128,0},
            fillColor={255,128,0},
            fillPattern=FillPattern.Solid), Ellipse(
            extent={{-28,28},{28,-28}},
            lineColor={255,255,85},
            fillColor={255,255,0},
            fillPattern=FillPattern.Solid)}));
  end SunRadiationEstimatedDiffuseRadiation;

end Weather;
