package WasteWater 
  extends Modelica.Icons.Library;

  package ASM1 
    extends Modelica.Icons.Library;

    model deni 
      extends WasteWater.Icons.deni;
      extends WasteWater.ASM1.Interfaces.ASM1base;
      parameter Modelica.SIunits.Volume V = 1000 "Volume of denitrification tank";
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-110, -10}, {-90, 10}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out Out annotation(Placement(transformation(extent = {{90, -10}, {110, 10}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out MeasurePort annotation(Placement(transformation(extent = {{50, 40}, {60, 50}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput T annotation(Placement(transformation(extent = {{-110, 30}, {-90, 50}}, rotation = 0)));
       annotation(Window(x = 0.15, y = 0.05, width = 0.35, height = 0.49), Documentation(info = "This component models the ASM1 processes and reactions taking place in an unaerated (denitrification) tank
        of a wastewater treatment plant.

        The InPort signal gives the tank temperature to the model.

        Parameters:

            - all stoichiometric and kinetic parameters of the activated sludge model No.1 (ASM1)
          V - volume of the tank [m3]
        "));
    equation
      aeration = 0;
      inputSi = ((In.Si - Si) * In.Q) / V;
      inputSs = ((In.Ss - Ss) * In.Q) / V;
      inputXi = ((In.Xi - Xi) * In.Q) / V;
      inputXs = ((In.Xs - Xs) * In.Q) / V;
      inputXbh = ((In.Xbh - Xbh) * In.Q) / V;
      inputXba = ((In.Xba - Xba) * In.Q) / V;
      inputXp = ((In.Xp - Xp) * In.Q) / V;
      inputSo = ((In.So - So) * In.Q) / V;
      inputSno = ((In.Sno - Sno) * In.Q) / V;
      inputSnh = ((In.Snh - Snh) * In.Q) / V;
      inputSnd = ((In.Snd - Snd) * In.Q) / V;
      inputXnd = ((In.Xnd - Xnd) * In.Q) / V;
      inputSalk = ((In.Salk - Salk) * In.Q) / V;
    end deni;

    model nitri 
      extends WasteWater.Icons.nitri;
      extends WasteWater.ASM1.Interfaces.ASM1base;
      parameter Modelica.SIunits.Volume V = 1000 "Volume of nitrification tank";
      parameter Real alpha = 0.7 "Oxygen transfer factor";
      parameter Modelica.SIunits.Length de = 4.5 "depth of aeration";
      parameter Real R_air = 23.5 "specific oxygen feed factor [gO2/(m^3*m)]";
      .WasteWater.WasteWaterUnits.MassConcentration So_sat "Dissolved oxygen saturation";
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-110, -10}, {-90, 10}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out Out annotation(Placement(transformation(extent = {{90, -10}, {110, 10}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out MeasurePort annotation(Placement(transformation(extent = {{50, 40}, {60, 50}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput T annotation(Placement(transformation(extent = {{-110, 30}, {-90, 50}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.AirFlow AirIn annotation(Placement(transformation(extent = {{-5, -103}, {5, -93}}, rotation = 0)));
       annotation(Window(x = 0.34, y = 0.1, width = 0.35, height = 0.47), Documentation(info = "This component models the ASM1 processes and reactions taking place in an aerated (nitrification) tank
        of a wastewater treatment plant.

        The InPort signal gives the tank temperature to the model.

        Parameters:

                - all soichiometric and kinetic parameters of the activated sludge model No.1 (ASM1)
          V     - volume of the tank [m3]
          alpha - oxygen transfer factor
          de    - depth of the aeration system [m]
          R_air - specific oxygen feed factor [g O2/(m3*m)]
        "));
    equation
      So_sat = 13.89 + (-0.3825 + (0.007311 - 6.588e-05 * T) * T) * T;
      aeration = ((alpha * (So_sat - So)) / So_sat * AirIn.Q_air * R_air * de) / V;
      inputSi = ((In.Si - Si) * In.Q) / V;
      inputSs = ((In.Ss - Ss) * In.Q) / V;
      inputXi = ((In.Xi - Xi) * In.Q) / V;
      inputXs = ((In.Xs - Xs) * In.Q) / V;
      inputXbh = ((In.Xbh - Xbh) * In.Q) / V;
      inputXba = ((In.Xba - Xba) * In.Q) / V;
      inputXp = ((In.Xp - Xp) * In.Q) / V;
      inputSo = ((In.So - So) * In.Q) / V;
      inputSno = ((In.Sno - Sno) * In.Q) / V;
      inputSnh = ((In.Snh - Snh) * In.Q) / V;
      inputSnd = ((In.Snd - Snd) * In.Q) / V;
      inputXnd = ((In.Xnd - Xnd) * In.Q) / V;
      inputSalk = ((In.Salk - Salk) * In.Q) / V;
    end nitri;

    model blower 
      extends WasteWater.Icons.blower;
      parameter .WasteWater.WasteWaterUnits.VolumeFlowRate Q_max = 20000 "maximum blower capacity";
      parameter .WasteWater.WasteWaterUnits.VolumeFlowRate Q_min = 0.0 "minimum blower capacity";
      Real H;
      WasteWater.ASM1.Interfaces.AirFlow AirOut annotation(Placement(transformation(extent = {{-20, 90}, {0, 110}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u annotation(Placement(transformation(origin = {98, -30}, extent = {{-10, -10}, {10, 10}}, rotation = 180)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component models a blower of a wastewater treatment plant which generates an airflow that is needed
        for the nitrification.
        The blower is connected to the nitrification tank.
        The airflow is controlled by a signal u (-1 <= u <= 1).

        Parameter:

          Qmax - maximum blower capacity [m3 Air/d], this is produced when the control signal u is 1 or greater.
          Qmin - minimum blower capacity [m3 Air/d], this is produced when the control signal u is -1 or below.

        "));
    equation
      H = 0.5 * (-Q_min + Q_max) + u * 0.5 * (-Q_min + Q_max) + Q_min;
      AirOut.Q_air = -(if H > Q_max then Q_max else if H < Q_min then Q_min else H);
    end blower;

    model pump 
      extends WasteWater.Icons.pump;
      parameter .WasteWater.WasteWaterUnits.VolumeFlowRate Q_min = 0.0 "minimum pump capacity";
      parameter .WasteWater.WasteWaterUnits.VolumeFlowRate Q_max = 20000 "maximum pump capacity";
      Real H;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-110, -43}, {-90, -23}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out Out annotation(Placement(transformation(extent = {{90, 18}, {110, 38}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u annotation(Placement(transformation(extent = {{-99, 15}, {-79, 35}}, rotation = 0)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1})), Documentation(info = "This component models an ASM1 wastewater pump. It generates a wastewater flow
        that is controlled by the signal u (-1 <= u <=1).

        Parameter:

          Qmax - maximum pump capacity [m3/d], this is produced when the control signal u is 1 or greater.
          Qmin - minimum pump capacity [m3/d], this is produced when the control signal u is -1 or below.

        "));
    equation
      H = 0.5 * (-Q_min + Q_max) + u * 0.5 * (-Q_min + Q_max) + Q_min;
      Out.Q = -(if H > Q_max then Q_max else if H < Q_min then Q_min else H);
      Out.Q + In.Q = 0;
      Out.Si = In.Si;
      Out.Ss = In.Ss;
      Out.Xi = In.Xi;
      Out.Xs = In.Xs;
      Out.Xbh = In.Xbh;
      Out.Xba = In.Xba;
      Out.Xp = In.Xp;
      Out.So = In.So;
      Out.Sno = In.Sno;
      Out.Snh = In.Snh;
      Out.Snd = In.Snd;
      Out.Xnd = In.Xnd;
      Out.Salk = In.Salk;
    end pump;

    model WWSource 
      extends WasteWater.Icons.WWSource;
      WasteWater.ASM1.Interfaces.WWFlowAsm1out Out annotation(Placement(transformation(extent = {{88, -80}, {108, -60}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput[14] data annotation(Placement(transformation(extent = {{-100, -10}, {-80, 10}}, rotation = 0)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component provides all ASM1 data at the influent of a wastewater treatment plant.
        The dimension of InPort is 14.

          1 volumeflowrate Q of incoming wastewater [m3/d]
          2 Si  [g COD/m3]
          3 Ss  [g COD/m3]
          4 Xi  [g COD/m3]
          5 Xs  [g COD/m3]
          6 Xbh [g COD/m3]
          7 Xba [g COD/m3]
          8 Xp  [g COD/m3]
          9 So  [g O2/m3]
         10 Sno [g N/m3]
         11 Snh [g N/m3]
         12 Snd [g N/m3]
         13 Xnd [g N/m3]
         14 Salk[mmol/l]
        "));
    equation
      Out.Q = -data[1];
      Out.Si = data[2];
      Out.Ss = data[3];
      Out.Xi = data[4];
      Out.Xs = data[5];
      Out.Xbh = data[6];
      Out.Xba = data[7];
      Out.Xp = data[8];
      Out.So = data[9];
      Out.Sno = data[10];
      Out.Snh = data[11];
      Out.Snd = data[12];
      Out.Xnd = data[13];
      Out.Salk = data[14];
    end WWSource;

    model EffluentSink 
      extends WasteWater.Icons.EffluentSink;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-110, 10}, {-90, 30}}, rotation = 0)));
       annotation(Window(x = 0.03, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component terminates an ASM1 wastewater treatment plant model e.g. the wastewater flow to the receiving water.
        "));
    end EffluentSink;

    model SludgeSink 
      extends WasteWater.Icons.SludgeSink;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-110, -22}, {-90, -2}}, rotation = 0)));
       annotation(Window(x = 0.06, y = 0.03, width = 0.35, height = 0.49), Documentation(info = "This component terminates the waste sludge stream of an ASM1 wastewater treatment plant model.
        Storage or further sludge treatment is not jet considered."));
    end SludgeSink;

    model divider2 
      extends WasteWater.Icons.divider2;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-110, -7}, {-90, 13}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out Out1 annotation(Placement(transformation(extent = {{90, 16}, {110, 36}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out Out2 annotation(Placement(transformation(extent = {{90, -25}, {110, -5}}, rotation = 0)));
       annotation(Window(x = 0.21, y = 0.03, width = 0.35, height = 0.49), Documentation(info = "This component divides one ASM1 wastewater flow into two ASM1 wastewater flows."));
    equation
      In.Q + Out1.Q + Out2.Q = 0;
      Out1.Si = In.Si;
      Out1.Ss = In.Ss;
      Out1.Xi = In.Xi;
      Out1.Xs = In.Xs;
      Out1.Xbh = In.Xbh;
      Out1.Xba = In.Xba;
      Out1.Xp = In.Xp;
      Out1.So = In.So;
      Out1.Sno = In.Sno;
      Out1.Snh = In.Snh;
      Out1.Snd = In.Snd;
      Out1.Xnd = In.Xnd;
      Out1.Salk = In.Salk;
      Out2.Si = In.Si;
      Out2.Ss = In.Ss;
      Out2.Xi = In.Xi;
      Out2.Xs = In.Xs;
      Out2.Xbh = In.Xbh;
      Out2.Xba = In.Xba;
      Out2.Xp = In.Xp;
      Out2.So = In.So;
      Out2.Sno = In.Sno;
      Out2.Snh = In.Snh;
      Out2.Snd = In.Snd;
      Out2.Xnd = In.Xnd;
      Out2.Salk = In.Salk;
    end divider2;

    model mixer3 
      extends WasteWater.Icons.mixer3;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In1 annotation(Placement(transformation(extent = {{-110, 25}, {-90, 45}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In2 annotation(Placement(transformation(extent = {{-110, -15}, {-90, 5}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In3 annotation(Placement(transformation(extent = {{-110, -55}, {-90, -35}}, rotation = 0)));
      WasteWater.ASM1.Interfaces.WWFlowAsm1out Out annotation(Placement(transformation(extent = {{90, -14}, {110, 6}}, rotation = 0)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component mixes 3 flows of wastewater (ASM1) of different concentration and different amount."));
    equation
      In1.Q + In2.Q + In3.Q + Out.Q = 0;
      Out.Si = (In1.Si * In1.Q + In2.Si * In2.Q + In3.Si * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Ss = (In1.Ss * In1.Q + In2.Ss * In2.Q + In3.Ss * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Xi = (In1.Xi * In1.Q + In2.Xi * In2.Q + In3.Xi * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Xs = (In1.Xs * In1.Q + In2.Xs * In2.Q + In3.Xs * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Xbh = (In1.Xbh * In1.Q + In2.Xbh * In2.Q + In3.Xbh * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Xba = (In1.Xba * In1.Q + In2.Xba * In2.Q + In3.Xba * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Xp = (In1.Xp * In1.Q + In2.Xp * In2.Q + In3.Xp * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.So = (In1.So * In1.Q + In2.So * In2.Q + In3.So * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Sno = (In1.Sno * In1.Q + In2.Sno * In2.Q + In3.Sno * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Snh = (In1.Snh * In1.Q + In2.Snh * In2.Q + In3.Snh * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Snd = (In1.Snd * In1.Q + In2.Snd * In2.Q + In3.Snd * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Xnd = (In1.Xnd * In1.Q + In2.Xnd * In2.Q + In3.Xnd * In3.Q) / (In1.Q + In2.Q + In3.Q);
      Out.Salk = (In1.Salk * In1.Q + In2.Salk * In2.Q + In3.Salk * In3.Q) / (In1.Q + In2.Q + In3.Q);
    end mixer3;

    model sensor_COD 
      extends WasteWater.Icons.sensor_COD;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-10, -110}, {10, -90}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput COD annotation(Placement(transformation(extent = {{88, -10}, {108, 10}}, rotation = 0)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component measures the chemical oxygen demand (COD) concentration [g/m3]
        of ASM1 wastewater and provides the result as output signal (to be
        further processed with blocks of the Modelica.Blocks library).
        "));
    equation
      In.Q = 0.0;
      COD = In.Si + In.Ss + In.Xi + In.Xs + In.Xbh + In.Xba + In.Xp;
    end sensor_COD;

    model sensor_NH 
      extends WasteWater.Icons.sensor_NH;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-10, -110}, {10, -90}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput Snh annotation(Placement(transformation(extent = {{88, -10}, {108, 10}}, rotation = 0)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component measures the ammonium nitrogen concentration [g/m3]
        of ASM1 wastewater and provides the result as output signal (to be
        further processed with blocks of the Modelica.Blocks library).
        "));
    equation
      In.Q = 0;
      Snh = In.Snh;
    end sensor_NH;

    model sensor_NO 
      extends WasteWater.Icons.sensor_NO;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-10, -110}, {10, -90}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput Sno annotation(Placement(transformation(extent = {{88, -10}, {108, 10}}, rotation = 0)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component measures the nitrate nitrogen concentration [g/m3]
        of ASM1 wastewater and provides the result as output signal (to be
        further processed with blocks of the Modelica.Blocks library).
        "));
    equation
      In.Q = 0;
      Sno = In.Sno;
    end sensor_NO;

    model sensor_O2 
      extends WasteWater.Icons.sensor_O2;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-10, -110}, {10, -90}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput So annotation(Placement(transformation(extent = {{88, -10}, {108, 10}}, rotation = 0)));
       annotation(Window(x = 0.13, y = 0.14, width = 0.35, height = 0.49), Documentation(info = "This component measures the dissolved oxygen concentration [g/m3]
        of ASM1 wastewater and provides the result as output signal (to be
        further processed with blocks of the Modelica.Blocks library).
        "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "O2"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0}), Text(extent = {{-80, 100}, {80, 60}}, textString = "%name")}));
    equation
      In.Q = 0;
      So = In.So;
    end sensor_O2;

    model sensor_TKN 
      extends WasteWater.Icons.sensor_TKN;
      extends WasteWater.ASM1.Interfaces.stoichiometry;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-10, -110}, {10, -90}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput[2] TKN annotation(Placement(transformation(extent = {{88, -10}, {108, 10}}, rotation = 0)));
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component measures the Total Kjeldal Nitrogen (TKN) and the
        total nitrogen (N_total) concentration [g/m3] of ASM1 wastewater
        and provides the result as output signal (to be further processed
        with blocks of the Modelica.Blocks library).

        signal[1] - TKN
        signal[2] - N_total
        "));
    equation
      In.Q = 0.0;
      TKN[1] = In.Snh + In.Snd + In.Xnd + i_xb * (In.Xbh + In.Xba) + i_xp * (In.Xp + In.Xi);
      TKN[2] = TKN[1] + In.Sno;
    end sensor_TKN;

    model sensor_TSS 
      extends WasteWater.Icons.sensor_TSS;
      WasteWater.ASM1.Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-10, -110}, {10, -90}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput TSS annotation(Placement(transformation(extent = {{88, -10}, {108, 10}}, rotation = 0)));
       annotation(Window(x = 0.03, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component measures the total suspended solids concentration [g/m3]
        of ASM1 wastewater and provides the result as output signal (to be
        further processed with blocks of the Modelica.Blocks library).
        "));
    equation
      In.Q = 0;
      TSS = 0.75 * (In.Xs + In.Xbh + In.Xba + In.Xp + In.Xi);
    end sensor_TSS;

    package Examples 
      extends Modelica.Icons.Library;

      class SmallPlant 
        extends Modelica.Icons.Example;
        .WasteWater.ASM1.EffluentSink Effluent annotation(Placement(transformation(extent = {{88, -28}, {108, -8}}, rotation = 0)));
        .WasteWater.ASM1.SludgeSink WasteSludge annotation(Placement(transformation(extent = {{87, -51}, {107, -31}}, rotation = 0)));
        .WasteWater.ASM1.divider2 divider annotation(Placement(transformation(extent = {{20, -6}, {40, 14}}, rotation = 0)));
        .WasteWater.ASM1.nitri tank3(V = 1333) annotation(Placement(transformation(extent = {{-6, -6}, {14, 14}}, rotation = 0)));
        .WasteWater.ASM1.nitri tank2(V = 1333) annotation(Placement(transformation(extent = {{-34, -6}, {-14, 14}}, rotation = 0)));
        .WasteWater.ASM1.deni tank1(V = 3000) annotation(Placement(transformation(extent = {{-65, -6}, {-45, 14}}, rotation = 0)));
        .WasteWater.ASM1.mixer3 mixer annotation(Placement(transformation(extent = {{-104, 22}, {-84, 42}}, rotation = 0)));
        Modelica.Blocks.Tables.CombiTable1Ds CombiTableTime(tableOnFile = true, tableName = "Inf_dry", fileName = "Inf_dry.txt", columns = {16, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15}) annotation(Placement(transformation(extent = {{-114, 78}, {-94, 98}}, rotation = 0)));
        .WasteWater.ASM1.WWSource WWSource1 annotation(Placement(transformation(extent = {{-88, 78}, {-68, 98}}, rotation = 0)));
        .WasteWater.ASM1.blower blower1(Q_max = 34574.2654508612) annotation(Placement(transformation(extent = {{-33, -62}, {-13, -42}}, rotation = 0)));
        .WasteWater.ASM1.blower blower2(Q_max = 34574.2654508612) annotation(Placement(transformation(extent = {{-6, -62}, {14, -42}}, rotation = 0)));
        .WasteWater.ASM1.sensor_O2 sensorO2 annotation(Placement(transformation(extent = {{0, 24}, {20, 44}}, rotation = 0)));
        Modelica.Blocks.Math.Feedback Feedback annotation(Placement(transformation(extent = {{62, 40}, {82, 60}}, rotation = 0)));
        Modelica.Blocks.Continuous.PI PI1(k = 500, T = 0.001) annotation(Placement(transformation(extent = {{88, 40}, {108, 60}}, rotation = 0)));
        Modelica.Blocks.Sources.Constant Constant1 annotation(Placement(transformation(extent = {{-67, -87}, {-47, -67}}, rotation = 0)));
        .WasteWater.ASM1.pump RecyclePump(Q_max = 46115) annotation(Placement(transformation(origin = {-84, -12}, extent = {{-10, -10}, {10, 10}}, rotation = 180)));
        .WasteWater.ASM1.pump ReturnPump(Q_max = 9223) annotation(Placement(transformation(origin = {26, -26}, extent = {{-10, -10}, {10, 10}}, rotation = 180)));
        .WasteWater.ASM1.pump WastePump(Q_max = 193) annotation(Placement(transformation(extent = {{59, -55}, {79, -35}}, rotation = 0)));
        Modelica.Blocks.Sources.Constant Constant2 annotation(Placement(transformation(extent = {{22, -68}, {42, -48}}, rotation = 0)));
        Modelica.Blocks.Sources.Constant Temperature(k = 15) annotation(Placement(transformation(extent = {{-94, 50}, {-82, 62}}, rotation = 0)));
        sensor_NH sensor_NH1 annotation(Placement(transformation(extent = {{64, 15}, {80, 31}}, rotation = 0)));
        .WasteWater.ASM1.sensor_NO sensor_NO1 annotation(Placement(transformation(extent = {{81, 15}, {97, 31}}, rotation = 0)));
        .WasteWater.ASM1.sensor_TKN sensor_TKN1 annotation(Placement(transformation(extent = {{97, 14}, {113, 30}}, rotation = 0)));
        .WasteWater.ASM1.sensor_COD sensor_COD1 annotation(Placement(transformation(extent = {{97, -5}, {113, 11}}, rotation = 0)));
        Modelica.Blocks.Sources.Step OxygenSetpoint(height = 1.5) annotation(Placement(transformation(extent = {{37, 40}, {57, 60}}, rotation = 0)));
        .WasteWater.ASM1.SecClar.Krebs.SecClarModKrebs Settler annotation(Placement(transformation(extent = {{48, -4}, {68, 16}}, rotation = 0)));
        .WasteWater.ASM1.sensor_TSS sensor_TSS1 annotation(Placement(transformation(extent = {{32, 14}, {49, 30}}, rotation = 0)));
        .WasteWater.Blocks.SimulationTime simulationTime annotation(Placement(transformation(extent = {{-149, 78}, {-129, 98}}, rotation = 0)));
         annotation(points = [-93, 88; -88, 88], Window(x = 0.33, y = 0.04, width = 0.85, height = 0.87), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-120, -100}, {120, 105}}, grid = {1, 1}), graphics = {Line(points = {{-22, 58}, {-22, 58}}, color = {0, 0, 255})}), Documentation(info = "This fictitious plant provides an ASM1 example model with a small number of equations.
          It consists of one denitrification and 2 nitrification tanks and a settler.

          Change into the directory ../ASM1 and translate the model.
          Before simulating the model load initial values from the script file small_asm1.mos
          that is provided besides the model.
          A 14 days dynamic influent data file is provided. So you may simulate up to 14 days.
          But start with 1 day as it may take some time for simulation.
          After simulation you may have a look at internal concentrations but most interesting
          are the relevant concentrations at the effluent of a plant which can be viewed via the 
          sensors at the effluent of the secondary clarifier.

          Main Author:
             Gerald Reichl
             Technische Universitaet Ilmenau
             Faculty of Informatics and Automation
             Department Dynamics and Simulation of ecological Systems
             P.O. Box 10 05 65
             98684 Ilmenau
             Germany
             email: gerald.reichl@tu-ilmenau.de
          "), Line(points = {{-93, 88}, {-88, 88}}));
      equation
        connect(tank3.Out, divider.In) annotation(Line(points = {{14, 4}, {17, 4}, {17, 4.3}, {20, 4.3}}));
        connect(mixer.Out, tank1.In) annotation(Line(points = {{-84, 31.6}, {-77, 31.6}, {-77, 4}, {-65, 4}}));
        connect(mixer.In1, WWSource1.Out) annotation(Line(points = {{-104, 35.5}, {-104, 74}, {-68, 74}, {-68, 81}, {-68.2, 81}}));
        connect(CombiTableTime.y, WWSource1.data) annotation(Line(points = {{-93, 88}, {-87, 88}}));
        connect(blower2.AirOut, tank3.AirIn) annotation(Line(points = {{3, -42}, {3, -23.9}, {4, -23.9}, {4, -5.8}}));
        connect(Feedback.y, PI1.u) annotation(Line(points = {{81, 50}, {86, 50}}));
        connect(PI1.y, blower2.u) annotation(Line(points = {{109, 50}, {114, 50}, {114, -84}, {18, -84}, {18, -55}, {13.8, -55}, {13.8, -55}}));
        connect(divider.Out2, RecyclePump.In) annotation(Line(points = {{40, 2.5}, {44, 2.5}, {44, -8.7}, {-74, -8.7}}));
        connect(RecyclePump.Out, mixer.In3) annotation(Line(points = {{-94, -14.8}, {-104, -14.8}, {-104, 27.5}}));
        connect(ReturnPump.Out, mixer.In2) annotation(Line(points = {{16, -28.8}, {15.5, -28.8}, {15.5, -30}, {-112, -30}, {-112, 31.5}, {-104, 31.5}}));
        connect(sensorO2.So, Feedback.u2) annotation(Line(points = {{19.8, 34}, {72, 34}, {72, 42}}));
        connect(Temperature.y, tank1.T) annotation(Line(points = {{-81.4, 56}, {-71, 56}, {-71, 8}, {-65, 8}}, color = {0, 0, 255}));
        connect(Temperature.y, tank2.T) annotation(Line(points = {{-81.4, 56}, {-39, 56}, {-39, 8}, {-34, 8}}, color = {0, 0, 255}));
        connect(Temperature.y, tank3.T) annotation(Line(points = {{-81.4, 56}, {-39, 56}, {-39, 14}, {-5.9, 14}, {-5.9, 8}, {-6, 8}}, color = {0, 0, 255}));
        connect(OxygenSetpoint.y, Feedback.u1) annotation(Line(points = {{58, 50}, {64, 50}}, color = {0, 0, 255}));
        connect(Constant1.y, blower1.u) annotation(Line(points = {{-46, -77}, {-7.2, -77}, {-7.2, -55}, {-13.2, -55}}, color = {0, 0, 255}));
        connect(WastePump.Out, WasteSludge.In) annotation(Line(points = {{79, -42.2}, {81, -42.2}, {81, -42.4}, {83, -42.4}, {83, -42.2}, {87, -42.2}}));
        connect(WastePump.u, Constant2.y) annotation(Line(points = {{60.1, -42.5}, {46, -42.5}, {46, -58}, {43, -58}}, color = {0, 0, 255}));
        connect(tank2.Out, tank3.In) annotation(Line(points = {{-14, 4}, {-6, 4}}));
        connect(tank1.Out, tank2.In) annotation(Line(points = {{-45, 4}, {-34, 4}}));
        connect(blower1.AirOut, tank2.AirIn) annotation(Line(points = {{-24, -42}, {-24, -5.8}}));
        connect(Constant1.y, RecyclePump.u) annotation(Line(points = {{-46, -77}, {-39, -77}, {-39, -14.5}, {-75.1, -14.5}}, color = {0, 0, 255}));
        connect(Settler.Effluent, Effluent.In) annotation(Line(points = {{68.2, 11.7}, {78, 11.7}, {78, -16}, {88, -16}}));
        connect(Settler.Return, ReturnPump.In) annotation(Line(points = {{55, -3.6}, {55, -22.7}, {36, -22.7}}));
        connect(WastePump.In, Settler.Waste) annotation(Line(points = {{59, -48.3}, {52, -48.3}, {52, -31}, {61, -31}, {61, -3.6}}));
        connect(sensor_NH1.In, Settler.Effluent) annotation(Line(points = {{72, 15}, {72, 11.7}, {68.2, 11.7}}));
        connect(sensor_NO1.In, Settler.Effluent) annotation(Line(points = {{89, 15}, {89, 11.7}, {68.2, 11.7}}));
        connect(sensor_TKN1.In, Settler.Effluent) annotation(Line(points = {{105, 14}, {105, 11.7}, {68.2, 11.7}}));
        connect(sensor_COD1.In, Settler.Effluent) annotation(Line(points = {{105, -5}, {105, 11.7}, {68.2, 11.7}}));
        connect(Constant2.y, ReturnPump.u) annotation(Line(points = {{43, -58}, {48, -58}, {48, -28.5}, {34.9, -28.5}}, color = {0, 0, 255}));
        connect(divider.Out1, Settler.Feed) annotation(Line(points = {{40, 6.6}, {44.5, 6.6}, {44.5, 7.4}, {48, 7.4}}));
        connect(tank3.MeasurePort, sensorO2.In) annotation(Line(points = {{9.5, 8.5}, {9.5, 15.75}, {10, 15.75}, {10, 24}}));
        connect(sensor_TSS1.In, divider.Out1) annotation(Line(points = {{40.5, 14}, {40.5, 10.6}, {40, 10.6}, {40, 6.6}}));
        connect(simulationTime.y, CombiTableTime.u) annotation(Line(points = {{-128, 88}, {-116, 88}}, color = {0, 0, 127}, smooth = Smooth.None));
      end SmallPlant;
       annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This package contains example ASM1 wastewater treatment plant models to demonstrate the usage of 
        the WasteWater.ASM1 library.
        Open the models and simulate them according to the description provided in the models.

        The following demo models are present:

         - SmallPlant
         - BenchPlant
         - ComplexPlant

        Main Author:
           Gerald Reichl
           Technische Universitaet Ilmenau
           Faculty of Informatics and Automation
           Department Dynamics and Simulation of ecological Systems
           P.O. Box 10 05 65
           98684 Ilmenau
           Germany
           email: gerald.reichl@tu-ilmenau.de

        This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
        disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

        Copyright (C) 2003, Gerald Reichl
        "));
    end Examples;

    package Interfaces 
      extends Modelica.Icons.Library;

      connector WWFlowAsm1in 
        flow .WasteWater.WasteWaterUnits.VolumeFlowRate Q;
        .WasteWater.WasteWaterUnits.MassConcentration Si;
        .WasteWater.WasteWaterUnits.MassConcentration Ss;
        .WasteWater.WasteWaterUnits.MassConcentration Xi;
        .WasteWater.WasteWaterUnits.MassConcentration Xs;
        .WasteWater.WasteWaterUnits.MassConcentration Xbh;
        .WasteWater.WasteWaterUnits.MassConcentration Xba;
        .WasteWater.WasteWaterUnits.MassConcentration Xp;
        .WasteWater.WasteWaterUnits.MassConcentration So;
        .WasteWater.WasteWaterUnits.MassConcentration Sno;
        .WasteWater.WasteWaterUnits.MassConcentration Snh;
        .WasteWater.WasteWaterUnits.MassConcentration Snd;
        .WasteWater.WasteWaterUnits.MassConcentration Xnd;
        .WasteWater.WasteWaterUnits.Alkalinity Salk;
         annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}, lineColor = {0, 0, 255}, fillColor = {0, 0, 191}, fillPattern = FillPattern.Solid), Text(extent = {{-88, 92}, {88, -94}}, lineColor = {255, 255, 255}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid, textString = "%name")}), Documentation(info = "Connectors WWFlowAsm1in and WWFlowAsm1out are nearly identical.
          The difference is in the icons to more easily identify the inflow and outflow
          side of a component.
          The connector consists of one flow variable and 13 potential variables (ASM1 concentrations).
          "));
      end WWFlowAsm1in;

      connector WWFlowAsm1out 
        flow .WasteWater.WasteWaterUnits.VolumeFlowRate Q;
        .WasteWater.WasteWaterUnits.MassConcentration Si;
        .WasteWater.WasteWaterUnits.MassConcentration Ss;
        .WasteWater.WasteWaterUnits.MassConcentration Xi;
        .WasteWater.WasteWaterUnits.MassConcentration Xs;
        .WasteWater.WasteWaterUnits.MassConcentration Xbh;
        .WasteWater.WasteWaterUnits.MassConcentration Xba;
        .WasteWater.WasteWaterUnits.MassConcentration Xp;
        .WasteWater.WasteWaterUnits.MassConcentration So;
        .WasteWater.WasteWaterUnits.MassConcentration Sno;
        .WasteWater.WasteWaterUnits.MassConcentration Snh;
        .WasteWater.WasteWaterUnits.MassConcentration Snd;
        .WasteWater.WasteWaterUnits.MassConcentration Xnd;
        .WasteWater.WasteWaterUnits.Alkalinity Salk;
         annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}), Text(extent = {{-88, 92}, {94, -92}}, textString = "%name")}), Documentation(info = "Connectors WWFlowAsm1in and WWFlowAsm1out are nearly identical.
          The difference is in the icons to more easily identify the inflow and outflow
          side of a component.
          The connector consists of one flow variable and 13 potential variables (ASM1 concentrations).
          "));
      end WWFlowAsm1out;

      connector AirFlow 
        flow .WasteWater.WasteWaterUnits.VolumeFlowRate Q_air;
         annotation(Documentation(info = "The Airflow connector consits of a flow variable describing the exchange of
          air between blower and nitrification tank."), Window(x = 0.4, y = 0.35, width = 0.6, height = 0.6));
      end AirFlow;

      partial model stoichiometry 
        parameter Real Y_h = 0.67 "Heterotrophic Yield [g Xbh COD formed/(g COD utilised)]";
        parameter Real Y_a = 0.24 "Autotrophic Yield [g Xba COD formed/(g N utilised)]";
        parameter Real f_p = 0.08 "Fraction of biomass to particulate products [-]";
        parameter Real i_xb = 0.086 "Fraction nitrogen in biomass [g N/(g COD)]";
        parameter Real i_xp = 0.06 "Fraction nitrogen in particulate products [g N/(g COD)]";
         annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This is a partial model providing the stoichiometric coefficients of the ASM1 model."));
      end stoichiometry;

      partial model ASM1base 
        extends Interfaces.stoichiometry;
        Real mu_h "Maximum heterotrophic growth rate f(T) [day^-1]";
        Real b_h "Heterotrophic decay rate f(T) [day^-1]";
        Real mu_a "Maximum autotrophic growth rate f(T) [day^-1]";
        Real b_a "Autotrophic decay rate f(T) [day^-1]";
        Real k_a "Ammonification rate f(T) [m3/(g COD day)]";
        Real k_h "Maximum specific hydrolysis rate f(T) [g Xs/(g Xbh COD day)]";
        Real K_x "Half-saturation (hydrolysis) f(T) [g Xs/(g Xbh COD)]";
        parameter Real mu_h_T = 4.0 "Maximum heterotrophic growth rate at T=15 deg C [day^-1]";
        parameter Real b_h_T = 0.28 "Heterotrophic decay rate at T=15 deg C [day^-1]";
        parameter Real mu_a_T = 0.5 "Maximum autotrophic growth rate at T=15 deg C[day^-1]";
        parameter Real b_a_T = 0.1 "Autotrophic decay rate at T=15 deg C [day^-1]";
        parameter Real k_a_T = 0.06 "Ammonification rate at T=15 deg C [m3/(g COD day)]";
        parameter Real k_h_T = 1.75 "Maximum specific hydrolysis rate at T=15 deg C [g Xs/(g Xbh COD day)]";
        parameter Real K_x_T = 0.0175 "Half-saturation (hydrolysis) at T=15 deg C [g Xs/(g Xbh COD)]";
        parameter Real K_nh = 1.0 "Half-saturation (auto. growth) [g NH-N/m3]";
        parameter Real K_s = 20.0 "Half-saturation (hetero. growth) [g COD/m3]";
        parameter Real K_oh = 0.2 "Half-saturation (hetero. oxygen) [g O/m3]";
        parameter Real K_no = 0.5 "Half-saturation (nitrate) [g NO-N/m3]";
        parameter Real K_oa = 0.4 "Half-saturation (auto. oxygen) [g O/m3]";
        parameter Real ny_g = 0.8 "Anoxic growth rate correction factor [-]";
        parameter Real ny_h = 0.4 "Anoxic hydrolysis rate correction factor [-]";
        .WasteWater.WasteWaterUnits.MassConcentration Si "Soluble inert organic matter";
        .WasteWater.WasteWaterUnits.MassConcentration Ss "Readily biodegradable substrate";
        .WasteWater.WasteWaterUnits.MassConcentration Xi "Particulate inert organic matter";
        .WasteWater.WasteWaterUnits.MassConcentration Xs "Slowly biodegradable substrate";
        .WasteWater.WasteWaterUnits.MassConcentration Xbh "Active heterotrophic biomass";
        .WasteWater.WasteWaterUnits.MassConcentration Xba "Active autotrophic biomass";
        .WasteWater.WasteWaterUnits.MassConcentration Xp "Particulate products from biomass decay";
        .WasteWater.WasteWaterUnits.MassConcentration So "Dissolved oxygen";
        .WasteWater.WasteWaterUnits.MassConcentration Sno "Nitrate and nitrite nitrogen";
        .WasteWater.WasteWaterUnits.MassConcentration Snh "Ammonium nitrogen";
        .WasteWater.WasteWaterUnits.MassConcentration Snd "Soluble biodegradable organic nitrogen";
        .WasteWater.WasteWaterUnits.MassConcentration Xnd "Particulate biodegradable organic nitrogen";
        .WasteWater.WasteWaterUnits.Alkalinity Salk "Alkalinity";
        Real p1;
        Real p2;
        Real p3;
        Real p4;
        Real p5;
        Real p6;
        Real p7;
        Real p8;
        Real r1;
        Real r2;
        Real r3;
        Real r4;
        Real r5;
        Real r6;
        Real r7;
        Real r8;
        Real r9;
        Real r10;
        Real r11;
        Real r12;
        Real r13;
        Real inputSi;
        Real inputSs;
        Real inputXi;
        Real inputXs;
        Real inputXbh;
        Real inputXba;
        Real inputXp;
        Real inputSo;
        Real inputSno;
        Real inputSnh;
        Real inputSnd;
        Real inputXnd;
        Real inputSalk;
        Real aeration;
        Interfaces.WWFlowAsm1in In annotation(Placement(transformation(extent = {{-110, -10}, {-90, 10}}, rotation = 0)));
        Interfaces.WWFlowAsm1out Out annotation(Placement(transformation(extent = {{90, -10}, {110, 10}}, rotation = 0)));
        Interfaces.WWFlowAsm1out MeasurePort annotation(Placement(transformation(extent = {{50, 40}, {60, 50}}, rotation = 0)));
        Modelica.Blocks.Interfaces.RealInput T annotation(Placement(transformation(extent = {{-110, 30}, {-90, 50}}, rotation = 0)));
         annotation(Window(x = 0.03, y = 0.03, width = 0.35, height = 0.49), Documentation(info = "This partial model provides connectors and equations that are needed in the biological 
          components (nitrification and denitrification tank) for ASM1 wastewater treatment plant models.
          Parameters are coded according the ASM1 [1,2] standard distribution.
          Changes to this parameters are subject to the modeller.

          Main Author:
             Gerald Reichl
             Technische Universitaet Ilmenau
             Faculty of Informatics and Automation
             Department Dynamics and Simulation of ecological Systems
             P.O. Box 10 05 65
             98684 Ilmenau
             Germany
             email: gerald.reichl@tu-ilmenau.de


          References:

          [1] M. Henze and C.P.L. Grady Jr and W. Gujer and G.v.R. Marais and T. Matsuo:
              Activated Sludge Model No.1. IAWQ, 1987.
          [2] M. Henze and W.Gujer and T. Mino and. M.v. Loosdrecht: Activated Sludge
              Models ASM1, ASM2, ASM2d, and ASM3. IWA Task Group on Mathematical Modelling
              for Design and Operation of Biological Wastewater Treatment, 2000.


          This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
          disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

          Copyright (C) 2000 - 2002, Gerald Reichl
          "));
      equation
        mu_h = mu_h_T * exp(0.069 * (T - 15));
        b_h = b_h_T * exp(0.069 * (T - 15));
        mu_a = mu_a_T * exp(0.098 * (T - 15));
        b_a = b_a_T * exp(0.08 * (T - 15));
        k_a = k_a_T * exp(0.069 * (T - 15));
        k_h = k_h_T * exp(0.11 * (T - 15));
        K_x = K_x_T * exp(0.11 * (T - 15));
        p1 = mu_h * Ss / (K_s + Ss) * So / (K_oh + So) * Xbh;
        p2 = mu_h * Ss / (K_s + Ss) * K_oh / (K_oh + So) * Sno / (K_no + Sno) * ny_g * Xbh;
        p3 = mu_a * Snh / (K_nh + Snh) * So / (K_oa + So) * Xba;
        p4 = b_h * Xbh;
        p5 = b_a * Xba;
        p6 = k_a * Snd * Xbh;
        p7 = k_h * Xs / Xbh / (K_x + Xs / Xbh) * (So / (K_oh + So) + ny_h * K_oh / (K_oh + So) * Sno / (K_no + Sno)) * Xbh;
        p8 = (p7 * Xnd) / Xs;
        r1 = 0;
        r2 = (-p1 - p2) / Y_h + p7;
        r3 = 0;
        r4 = (1 - f_p) * (p4 + p5) - p7;
        r5 = p1 + p2 - p4;
        r6 = p3 - p5;
        r7 = f_p * (p4 + p5);
        r8 = -(1 - Y_h) / Y_h * p1 - (4.57 - Y_a) / Y_a * p3;
        r9 = -(1 - Y_h) / (2.86 * Y_h) * p2 + p3 / Y_a;
        r10 = -i_xb * (p1 + p2) - (i_xb + 1 / Y_a) * p3 + p6;
        r11 = -p6 + p8;
        r12 = (i_xb - f_p * i_xp) * (p4 + p5) - p8;
        r13 = -i_xb / 14 * p1 + ((1 - Y_h) / (14 * 2.86 * Y_h) - i_xb / 14) * p2 - (i_xb / 14 + 1 / (7 * Y_a)) * p3 + p6 / 14;
        der(Si) = inputSi + r1;
        der(Ss) = inputSs + r2;
        der(Xi) = inputXi + r3;
        der(Xs) = inputXs + r4;
        der(Xbh) = inputXbh + r5;
        der(Xba) = inputXba + r6;
        der(Xp) = inputXp + r7;
        der(So) = inputSo + r8 + aeration;
        der(Sno) = inputSno + r9;
        der(Snh) = inputSnh + r10;
        der(Snd) = inputSnd + r11;
        der(Xnd) = inputXnd + r12;
        der(Salk) = inputSalk + r13;
        Out.Q + In.Q = 0;
        Out.Si = Si;
        Out.Ss = Ss;
        Out.Xi = Xi;
        Out.Xs = Xs;
        Out.Xbh = Xbh;
        Out.Xba = Xba;
        Out.Xp = Xp;
        Out.So = So;
        Out.Sno = Sno;
        Out.Snh = Snh;
        Out.Snd = Snd;
        Out.Xnd = Xnd;
        Out.Salk = Salk;
        MeasurePort.Si = Si;
        MeasurePort.Ss = Ss;
        MeasurePort.Xi = Xi;
        MeasurePort.Xs = Xs;
        MeasurePort.Xbh = Xbh;
        MeasurePort.Xba = Xba;
        MeasurePort.Xp = Xp;
        MeasurePort.So = So;
        MeasurePort.Sno = Sno;
        MeasurePort.Snh = Snh;
        MeasurePort.Snd = Snd;
        MeasurePort.Xnd = Xnd;
        MeasurePort.Salk = Salk;
      end ASM1base;
       annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This package contains connectors and interfaces (partial models) for
        wastewater treatment components based on the Acticated Sludge Model No.1 (ASM1).

        Main Author:
           Gerald Reichl
           Technische Universitaet Ilmenau
           Faculty of Informatics and Automation
           Department Dynamics and Simulation of ecological Systems
           P.O. Box 10 05 65
           98684 Ilmenau
           Germany

        This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
        disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

        Copyright (C) 2000 - 2001, Gerald Reichl
        "));
    end Interfaces;

    package SecClar 
      extends Modelica.Icons.Library;

      package Krebs 
        extends Modelica.Icons.Library;

        package Interfaces 
          extends Modelica.Icons.Library;

          partial model SCVar 
            .WasteWater.WasteWaterUnits.MassConcentration Xf "total sludge concentration";
            .WasteWater.WasteWaterUnits.MassConcentration XB "sludge concentration in sludge layer";
            .WasteWater.WasteWaterUnits.MassConcentration XR "sludge concentration of return";
            .WasteWater.WasteWaterUnits.MassConcentration Si1 "Soluble inert organic matter in first stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Ss1 "Readily biodegradable substrate in first stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration So1 "Dissolved oxygen in first stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Sno1 "Nitrate and nitrite nitrogen in first stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Snh1 "Ammonium nitrogen in first stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Snd1 "Soluble biodegradable organic nitrogen in first stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.Alkalinity Salk1 "Alkalinity in first stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Si2 "Soluble inert organic matter in second stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Ss2 "Readily biodegradable substrate in second stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration So2 "Dissolved oxygen in second stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Sno2 "Nitrate and nitrite nitrogen in second stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Snh2 "Ammonium nitrogen in second stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.MassConcentration Snd2 "Soluble biodegradable organic nitrogen in second stirrer tank of the excess layer";
            .WasteWater.WasteWaterUnits.Alkalinity Salk2 "Alkalinity in second stirrer tank of the excess layer";
             annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "partial models providing ASM1 variables"));
          end SCVar;

          partial model ratios 
            Real rXi;
            Real rXs;
            Real rXbh;
            Real rXba;
            Real rXp;
            Real rXnd;
             annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "partial model for ASM1 ratios of solid components"));
          end ratios;
           annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This package contains partial models for ASM1 secondary clarifier models.

            Main Author:
               Gerald Reichl
               Technische Universitaet Ilmenau
               Faculty of Informatics and Automation
               Department Dynamics and Simulation of ecological Systems
               P.O. Box 10 05 65
               98684 Ilmenau
               Germany

            This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
            disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

            Copyright (C) 2003, Gerald Reichl
            "));
        end Interfaces;

        model SecClarModKrebs 
          extends WasteWater.Icons.SecClarKrebs;
          extends .WasteWater.ASM1.SecClar.Krebs.Interfaces.SCVar;
          extends .WasteWater.ASM1.SecClar.Krebs.Interfaces.ratios;
          parameter .Modelica.SIunits.Length hsc = 4.0 "height of secondary clarifier";
          parameter .Modelica.SIunits.Area Asc = 1500.0 "area of secondary clarifier";
          parameter .WasteWater.WasteWaterUnits.SludgeVolumeIndex ISV = 130 "Sludge Volume Index";
          Real te "thickening time in sludge layer in [d]";
          .Modelica.SIunits.Length hs "height of sludge layer";
          .Modelica.SIunits.Length he "height of excess layer";
          .WasteWater.ASM1.Interfaces.WWFlowAsm1in Feed annotation(Placement(transformation(extent = {{-110, 4}, {-90, 24}}, rotation = 0)));
          .WasteWater.ASM1.Interfaces.WWFlowAsm1out Effluent annotation(Placement(transformation(extent = {{92, 47}, {112, 67}}, rotation = 0)));
          .WasteWater.ASM1.Interfaces.WWFlowAsm1out Return annotation(Placement(transformation(extent = {{-40, -106}, {-20, -86}}, rotation = 0)));
          .WasteWater.ASM1.Interfaces.WWFlowAsm1out Waste annotation(Placement(transformation(extent = {{20, -106}, {40, -86}}, rotation = 0)));
           annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Documentation(info = "This component models an ASM1 secondary clarifier based on Krebs conceptional model.
            It consists of two compartments: a \"sludge-bed\" and a clear water zone above.

            Parameters:
              hsc -  height of clarifier [m]
              Asc -  surface area of secondary clarifier [m2]
              ISV -  Sludge Volume Index [ml/g]
            "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-90, 80}, {92, 14}}, lineColor = {0, 0, 255}, lineThickness = 0.5), Rectangle(extent = {{-90, 14}, {92, -86}}, lineColor = {0, 0, 255}, lineThickness = 0.5), Polygon(points = {{-8, -20}, {-8, -38}, {-16, -38}, {0, -48}, {16, -38}, {8, -38}, {8, -20}, {-8, -20}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{-8, 34}, {-8, 54}, {-16, 54}, {0, 64}, {16, 54}, {8, 54}, {8, 34}, {-8, 34}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Text(extent = {{-90, 78}, {-34, 66}}, textString = "top_layer"), Text(extent = {{-90, 20}, {-30, -16}}, textString = "bottom_layer"), Line(points = {{-90, 48}, {92, 48}}, color = {0, 0, 255}, pattern = LinePattern.Dash)}), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2})));
        equation
          Xf = 0.75 * (Feed.Xs + Feed.Xbh + Feed.Xba + Feed.Xp + Feed.Xi);
          rXs = Feed.Xs / Xf;
          rXbh = Feed.Xbh / Xf;
          rXba = Feed.Xba / Xf;
          rXp = Feed.Xp / Xf;
          rXi = Feed.Xi / Xf;
          rXnd = Feed.Xnd / Xf;
          if initial() then
            XB = Feed.Q / (0.7 * (-(Return.Q + Waste.Q))) * Xf;
          end if;
          te = (5 / 7 * Asc * hs) / (-(Return.Q + Waste.Q));
          XB = 1000 / ISV * (te * 24) ^ (1 / 3) * 1000;
          XR = 0.7 * XB;
          der(hs) = (Feed.Q * Xf - (-(Return.Q + Waste.Q)) * XR) / (Asc / 2 * XB);
          he = hsc - hs;
          der(Si1) = (Feed.Q * Feed.Si - (-Effluent.Q) * Si1 - (-(Waste.Q + Return.Q)) * Si1) / (Asc * he) / 2;
          der(Ss1) = (Feed.Q * Feed.Ss - (-Effluent.Q) * Ss1 - (-(Waste.Q + Return.Q)) * Ss1) / (Asc * he) / 2;
          der(So1) = (Feed.Q * Feed.So - (-Effluent.Q) * So1 - (-(Waste.Q + Return.Q)) * So1) / (Asc * he) / 2;
          der(Sno1) = (Feed.Q * Feed.Sno - (-Effluent.Q) * Sno1 - (-(Waste.Q + Return.Q)) * Sno1) / (Asc * he) / 2;
          der(Snh1) = (Feed.Q * Feed.Snh - (-Effluent.Q) * Snh1 - (-(Waste.Q + Return.Q)) * Snh1) / (Asc * he) / 2;
          der(Snd1) = (Feed.Q * Feed.Snd - (-Effluent.Q) * Snd1 - (-(Waste.Q + Return.Q)) * Snd1) / (Asc * he) / 2;
          der(Salk1) = (Feed.Q * Feed.Salk - (-Effluent.Q) * Salk1 - (-(Waste.Q + Return.Q)) * Salk1) / (Asc * he) / 2;
          der(Si2) = ((-Effluent.Q) * Si1 - (-Effluent.Q) * Si2) / (Asc * he) / 2;
          der(Ss2) = ((-Effluent.Q) * Ss1 - (-Effluent.Q) * Ss2) / (Asc * he) / 2;
          der(So2) = ((-Effluent.Q) * So1 - (-Effluent.Q) * So2) / (Asc * he) / 2;
          der(Sno2) = ((-Effluent.Q) * Sno1 - (-Effluent.Q) * Sno2) / (Asc * he) / 2;
          der(Snh2) = ((-Effluent.Q) * Snh1 - (-Effluent.Q) * Snh2) / (Asc * he) / 2;
          der(Snd2) = ((-Effluent.Q) * Snd1 - (-Effluent.Q) * Snd2) / (Asc * he) / 2;
          der(Salk2) = ((-Effluent.Q) * Salk1 - (-Effluent.Q) * Salk2) / (Asc * he) / 2;
          Feed.Q + Effluent.Q + Return.Q + Waste.Q = 0;
          Effluent.Si = Si2;
          Effluent.Ss = Ss2;
          Effluent.So = So2;
          Effluent.Sno = Sno2;
          Effluent.Snh = Snh2;
          Effluent.Snd = Snd2;
          Effluent.Salk = Salk2;
          Effluent.Xi = 0.0 * XR;
          Effluent.Xs = 0.0 * XR;
          Effluent.Xbh = 0.0 * XR;
          Effluent.Xba = 0.0 * XR;
          Effluent.Xp = 0.0 * XR;
          Effluent.Xnd = 0.0 * XR;
          Return.Si = Si1;
          Return.Ss = Ss1;
          Return.So = So1;
          Return.Sno = Sno1;
          Return.Snh = Snh1;
          Return.Snd = Snd1;
          Return.Salk = Salk1;
          Return.Xi = rXi * XR;
          Return.Xs = rXs * XR;
          Return.Xbh = rXbh * XR;
          Return.Xba = rXba * XR;
          Return.Xp = rXp * XR;
          Return.Xnd = rXnd * XR;
          Waste.Si = Si1;
          Waste.Ss = Ss1;
          Waste.So = So1;
          Waste.Sno = Sno1;
          Waste.Snh = Snh1;
          Waste.Snd = Snd1;
          Waste.Salk = Salk1;
          Waste.Xi = rXi * XR;
          Waste.Xs = rXs * XR;
          Waste.Xbh = rXbh * XR;
          Waste.Xba = rXba * XR;
          Waste.Xp = rXp * XR;
          Waste.Xnd = rXnd * XR;
        end SecClarModKrebs;
         annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This package contains an ASM1 secondary clarifier model and an Interfaces sub-library
          based on Krebs conceptional model [1].
          The settler model consists of two compartments, a \"sludge-bed\" and a clear water zone above.

          Main Author:
             Gerald Reichl
             Technische Universitaet Ilmenau
             Faculty of Informatics and Automation
             Department Dynamics and Simulation of ecological Systems
             P.O. Box 10 05 65
             98684 Ilmenau
             Germany
             email: gerald.reichl@tu-ilmenau.de

          References:

          [1] P. Krebs and M. Armbruster and W. Rodi: Numerische Nachklaerbeckenmodelle. Korrespondenz Abwasser. 47 (7)
              2000. pp 985-999.

          This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
          disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

          Copyright (C) 2003, Gerald Reichl
          "));
      end Krebs;
       annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This library provides a collection of ASM1 secondary clarifier models based on
        several theories.

        The library currently is structured in following sub-libraries.

         - Takacs: secondary clarifier model by Takacs et al [1]
         - Haertel: secondary clarifier model by Haertel [2]
         - Otterpohl: secondary clarifier model by Otterpohl [3]
         - Krebs: secondary clarifier model by Krebs [4]
         - Simple: very basic secondary clarifier model


        Main Author:
           Gerald Reichl
           Technische Universitaet Ilmenau
           Faculty of Informatics and Automation
           Department Dynamics and Simulation of ecological Systems
           P.O. Box 10 05 65
           98684 Ilmenau
           Germany
           email: gerald.reichl@tu-ilmenau.de

        References:

        [1] I. Takacs and G.G. Patry and D. Nolasco: A dynamic model of the clarification-thickening
            process. Water Research. 25 (1991) 10, pp 1263-1271.

        [2] L. Haertel: Modellansaetze zur dynamischen Simulation des Belebtschlammverfahrens.
            TH Darmstadt, Dissertation, 1990.

        [3] R. Otterpohl and M. Freund: Dynamic models for clarifiers of activated sludge plants
            with dry and wet weather flows. Water Science and Technology. 26 (1992), pp 1391-1400.

        [4] P. Krebs and M. Armbruster and W. Rodi: Numerische Nachklaerbeckenmodelle. Korrespondenz Abwasser. 47 (7)
            2000. pp 985-999.


        This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
        disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

        Copyright (C) 2000 - 2003, Gerald Reichl
        "));
    end SecClar;
     annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This library contains components to build models of biological municipal
      wastewater treatment plants based on the Activated Sludge Model No.1 (ASM1) by the
      International Association on Water Quality (IAWQ) [1,2].


      The library currently is structured in following sub-libraries.

       - Interfaces (partial ASM1 models and connectors)
       - PreClar (primary clarifier models)
       - SecClar (several secondary settling tank models)
       - Examples (wastewater treatment plant models)

      Main Author:
         Gerald Reichl
         Technische Universitaet Ilmenau
         Faculty of Informatics and Automation
         Department Dynamics and Simulation of ecological Systems
         P.O. Box 10 05 65
         98684 Ilmenau
         Germany
         email: gerald.reichl@tu-ilmenau.de


      References:

      [1] M. Henze and C.P.L. Grady Jr and W. Gujer and G.v.R. Marais and T. Matsuo:
          Activated Sludge Model No.1. IAWQ, 1987.
      [2] M. Henze and W.Gujer and T. Mino and. M.v. Loosdrecht: Activated Sludge
          Models ASM1, ASM2, ASM2d, and ASM3. IWA Task Group on Mathematical Modelling
          for Design and Operation of Biological Wastewater Treatment, 2000.


      This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
      disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

      Copyright (C) 2000 - 2002, Gerald Reichl
      "));
  end ASM1;

  package Icons 
    extends Modelica.Icons.Library;

    partial block deni 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-80, 20}, {80, -80}}, lineColor = {255, 255, 255}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-80, -6}, {-80, -80}, {80, -80}, {80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-80, 6}, {-80, 20}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {80, 20}}, color = {0, 0, 0}, thickness = 0.5), Rectangle(extent = {{-90, 6}, {-80, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{80, 6}, {90, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-90, -6}, {-80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-90, 6}, {-80, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {90, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, -6}, {90, -6}}, color = {0, 0, 0}, thickness = 0.5), Ellipse(extent = {{-2, -34}, {2, -38}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Line(points = {{0, -34}, {0, 40}, {0, 44}}, color = {0, 0, 0}, thickness = 1), Text(extent = {{-98, 100}, {100, 60}}, textString = "%name"), Line(points = {{-2, -36}, {-10, -36}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{10, -36}, {2, -36}}, color = {0, 0, 0}, thickness = 0.5), Ellipse(extent = {{-62, -32}, {-10, -40}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {128, 128, 128}, fillPattern = FillPattern.Solid), Ellipse(extent = {{10, -32}, {62, -40}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {128, 128, 128}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-72, 0}, {-70, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-54, -2}, {-52, -4}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-38, 0}, {-36, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-68, -16}, {-66, -18}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-44, -20}, {-42, -22}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-14, -4}, {-12, -6}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-28, -16}, {-26, -18}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{10, 0}, {12, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{28, -2}, {30, -4}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{44, 0}, {46, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{14, -16}, {16, -18}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{38, -20}, {40, -22}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{68, -4}, {70, -6}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{61, -24}, {63, -26}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-68, -48}, {-66, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-50, -50}, {-48, -52}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-34, -48}, {-32, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-64, -64}, {-62, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-40, -68}, {-38, -70}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-10, -52}, {-8, -54}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-24, -64}, {-22, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{8, -48}, {10, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{26, -50}, {28, -52}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{42, -48}, {44, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{12, -64}, {14, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{36, -68}, {38, -70}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{66, -52}, {68, -54}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{52, -64}, {54, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-61, -6}, {-51, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-58, -6}, {-55, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-90, 40}, {-55, 40}}, color = {0, 0, 0}), Rectangle(extent = {{54, -6}, {57, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{51, -6}, {61, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid)}), Documentation(info = "Icon for a denitrification tank"), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-80, 20}, {80, -80}}, lineColor = {255, 255, 255}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-80, -6}, {-80, -80}, {80, -80}, {80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-80, 6}, {-80, 20}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {80, 20}}, color = {0, 0, 0}, thickness = 0.5), Rectangle(extent = {{-90, 6}, {-80, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{80, 6}, {90, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-90, -6}, {-80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-90, 6}, {-80, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {90, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, -6}, {90, -6}}, color = {0, 0, 0}, thickness = 0.5), Ellipse(extent = {{-2, -34}, {2, -38}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Line(points = {{0, -34}, {0, 40}, {0, 44}}, color = {0, 0, 0}, thickness = 1), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{-2, -36}, {-10, -36}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{10, -36}, {2, -36}}, color = {0, 0, 0}, thickness = 0.5), Ellipse(extent = {{-62, -32}, {-10, -40}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {128, 128, 128}, fillPattern = FillPattern.Solid), Ellipse(extent = {{10, -32}, {62, -40}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {128, 128, 128}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-72, 0}, {-70, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-54, -2}, {-52, -4}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-38, 0}, {-36, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-68, -16}, {-66, -18}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-44, -20}, {-42, -22}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-14, -4}, {-12, -6}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-28, -16}, {-26, -18}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{10, 0}, {12, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{28, -2}, {30, -4}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{44, 0}, {46, -2}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{14, -16}, {16, -18}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{38, -20}, {40, -22}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{68, -4}, {70, -6}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{61, -24}, {63, -26}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-68, -48}, {-66, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-50, -50}, {-48, -52}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-34, -48}, {-32, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-64, -64}, {-62, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-40, -68}, {-38, -70}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-10, -52}, {-8, -54}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-24, -64}, {-22, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{8, -48}, {10, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{26, -50}, {28, -52}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{42, -48}, {44, -50}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{12, -64}, {14, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{36, -68}, {38, -70}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{66, -52}, {68, -54}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{52, -64}, {54, -66}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-61, -6}, {-51, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-58, -6}, {-55, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-90, 40}, {-55, 40}}, color = {0, 0, 0}), Rectangle(extent = {{54, -6}, {57, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{51, -6}, {61, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid)}));
    end deni;

    partial model nitri 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-80, 20}, {80, -80}}, lineColor = {255, 255, 255}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-80, -6}, {-80, -80}, {80, -80}, {80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-80, 6}, {-80, 20}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {80, 20}}, color = {0, 0, 0}, thickness = 0.5), Rectangle(extent = {{-90, 6}, {-80, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{80, 6}, {90, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-90, -6}, {-80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-90, 6}, {-80, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {90, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, -6}, {90, -6}}, color = {0, 0, 0}, thickness = 0.5), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{-70, -74}, {70, -74}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, -80}, {0, -74}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, -80}, {0, -94}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{-70, -70}, {-68, -74}, {-62, -74}, {-60, -70}, {-70, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{60, -70}, {62, -74}, {68, -74}, {70, -70}, {60, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{4, -70}, {6, -74}, {12, -74}, {14, -70}, {4, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-14, -70}, {-12, -74}, {-6, -74}, {-4, -70}, {-14, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-52, -70}, {-50, -74}, {-44, -74}, {-42, -70}, {-52, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-32, -70}, {-30, -74}, {-24, -74}, {-22, -70}, {-32, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{42, -70}, {44, -74}, {50, -74}, {52, -70}, {42, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{22, -70}, {24, -74}, {30, -74}, {32, -70}, {22, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -56}, {-62, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, -66}, {-60, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -66}, {-66, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-70, -62}, {-68, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -62}, {-62, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -56}, {-66, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -50}, {-66, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -42}, {-66, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -36}, {-66, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -32}, {-66, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -24}, {-62, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -18}, {-66, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -8}, {-62, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, 0}, {-66, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, 12}, {-60, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-70, -70}, {-68, -74}, {-62, -74}, {-60, -70}, {-70, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -56}, {-62, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, -66}, {-60, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -66}, {-66, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-70, -62}, {-68, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -62}, {-62, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -56}, {-66, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -50}, {-66, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -42}, {-66, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -36}, {-66, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -32}, {-66, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -24}, {-62, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -18}, {-66, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -8}, {-62, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, 0}, {-66, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, 12}, {-60, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-52, -70}, {-50, -74}, {-44, -74}, {-42, -70}, {-52, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -56}, {-44, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-44, -66}, {-42, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -66}, {-48, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-52, -62}, {-50, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -62}, {-44, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -56}, {-48, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -50}, {-48, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -42}, {-48, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -36}, {-48, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -32}, {-48, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -24}, {-44, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -18}, {-48, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -8}, {-44, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, 0}, {-48, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-44, 12}, {-42, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-32, -70}, {-30, -74}, {-24, -74}, {-22, -70}, {-32, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -56}, {-24, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-24, -66}, {-22, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -66}, {-28, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-32, -62}, {-30, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -62}, {-24, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -56}, {-28, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -50}, {-28, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -42}, {-28, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -36}, {-28, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -32}, {-28, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -24}, {-24, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -18}, {-28, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -8}, {-24, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, 0}, {-28, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-24, 12}, {-22, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-14, -70}, {-12, -74}, {-6, -74}, {-4, -70}, {-14, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -56}, {-6, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-6, -66}, {-4, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -66}, {-10, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-14, -62}, {-12, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -62}, {-6, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -56}, {-10, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -50}, {-10, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -42}, {-10, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -36}, {-10, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -32}, {-10, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -24}, {-6, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -18}, {-10, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -8}, {-6, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, 0}, {-10, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-6, 12}, {-4, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{4, -70}, {6, -74}, {12, -74}, {14, -70}, {4, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -56}, {12, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{12, -66}, {14, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -66}, {8, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{4, -62}, {6, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -62}, {12, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -56}, {8, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -50}, {8, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -42}, {8, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -36}, {8, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -32}, {8, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -24}, {12, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -18}, {8, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -8}, {12, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, 0}, {8, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{12, 12}, {14, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{22, -70}, {24, -74}, {30, -74}, {32, -70}, {22, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -56}, {30, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{30, -66}, {32, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -66}, {26, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{22, -62}, {24, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -62}, {30, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -56}, {26, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -50}, {26, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -42}, {26, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -36}, {26, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -32}, {26, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -24}, {30, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -18}, {26, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -8}, {30, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, 0}, {26, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{30, 12}, {32, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{42, -70}, {44, -74}, {50, -74}, {52, -70}, {42, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -56}, {50, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{50, -66}, {52, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -66}, {46, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{42, -62}, {44, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -62}, {50, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -56}, {46, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -50}, {46, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -42}, {46, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -36}, {46, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -32}, {46, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -24}, {50, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -18}, {46, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{40, -8}, {38, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, 0}, {46, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{50, 12}, {52, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{60, -70}, {62, -74}, {68, -74}, {70, -70}, {60, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -56}, {68, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{68, -66}, {70, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -66}, {64, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{60, -62}, {62, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -62}, {68, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -56}, {64, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -50}, {64, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -42}, {64, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -36}, {64, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -32}, {64, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -24}, {68, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -18}, {64, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -8}, {68, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, 0}, {64, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{68, 12}, {70, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Documentation, Rectangle(extent = {{54, -6}, {57, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{50, -6}, {60, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-56, -6}, {-53, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-60, -6}, {-50, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-90, 40}, {-53, 40}}, color = {0, 0, 0})}, Documentation(info = "Nitrifikationsbecken")), Documentation(info = "Icon for a nitrification tank "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-80, 20}, {80, -80}}, lineColor = {255, 255, 255}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-80, -6}, {-80, -80}, {80, -80}, {80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-80, 6}, {-80, 20}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {80, 20}}, color = {0, 0, 0}, thickness = 0.5), Rectangle(extent = {{-90, 6}, {-80, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{80, 6}, {90, -6}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-90, -6}, {-80, -6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-90, 6}, {-80, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, 6}, {90, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, -6}, {90, -6}}, color = {0, 0, 0}, thickness = 0.5), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{-70, -74}, {70, -74}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, -80}, {0, -74}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, -80}, {0, -94}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{-70, -70}, {-68, -74}, {-62, -74}, {-60, -70}, {-70, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{60, -70}, {62, -74}, {68, -74}, {70, -70}, {60, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{4, -70}, {6, -74}, {12, -74}, {14, -70}, {4, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-14, -70}, {-12, -74}, {-6, -74}, {-4, -70}, {-14, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-52, -70}, {-50, -74}, {-44, -74}, {-42, -70}, {-52, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-32, -70}, {-30, -74}, {-24, -74}, {-22, -70}, {-32, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{42, -70}, {44, -74}, {50, -74}, {52, -70}, {42, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{22, -70}, {24, -74}, {30, -74}, {32, -70}, {22, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -56}, {-62, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, -66}, {-60, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -66}, {-66, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-70, -62}, {-68, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -62}, {-62, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -56}, {-66, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -50}, {-66, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -42}, {-66, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -36}, {-66, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -32}, {-66, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -24}, {-62, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -18}, {-66, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -8}, {-62, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, 0}, {-66, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, 12}, {-60, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-70, -70}, {-68, -74}, {-62, -74}, {-60, -70}, {-70, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -56}, {-62, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, -66}, {-60, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -66}, {-66, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-70, -62}, {-68, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -62}, {-62, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -56}, {-66, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -50}, {-66, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -42}, {-66, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -36}, {-66, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -32}, {-66, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -24}, {-62, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, -18}, {-66, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -8}, {-62, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-68, 0}, {-66, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-62, 12}, {-60, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-52, -70}, {-50, -74}, {-44, -74}, {-42, -70}, {-52, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -56}, {-44, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-44, -66}, {-42, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -66}, {-48, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-52, -62}, {-50, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -62}, {-44, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -56}, {-48, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -50}, {-48, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -42}, {-48, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -36}, {-48, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -32}, {-48, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -24}, {-44, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, -18}, {-48, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-46, -8}, {-44, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-50, 0}, {-48, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-44, 12}, {-42, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-32, -70}, {-30, -74}, {-24, -74}, {-22, -70}, {-32, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -56}, {-24, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-24, -66}, {-22, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -66}, {-28, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-32, -62}, {-30, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -62}, {-24, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -56}, {-28, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -50}, {-28, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -42}, {-28, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -36}, {-28, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -32}, {-28, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -24}, {-24, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, -18}, {-28, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-26, -8}, {-24, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-30, 0}, {-28, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-24, 12}, {-22, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-14, -70}, {-12, -74}, {-6, -74}, {-4, -70}, {-14, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -56}, {-6, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-6, -66}, {-4, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -66}, {-10, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-14, -62}, {-12, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -62}, {-6, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -56}, {-10, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -50}, {-10, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -42}, {-10, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -36}, {-10, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -32}, {-10, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -24}, {-6, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, -18}, {-10, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-8, -8}, {-6, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-12, 0}, {-10, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-6, 12}, {-4, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{4, -70}, {6, -74}, {12, -74}, {14, -70}, {4, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -56}, {12, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{12, -66}, {14, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -66}, {8, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{4, -62}, {6, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -62}, {12, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -56}, {8, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -50}, {8, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -42}, {8, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -36}, {8, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -32}, {8, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -24}, {12, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, -18}, {8, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{10, -8}, {12, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{6, 0}, {8, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{12, 12}, {14, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{22, -70}, {24, -74}, {30, -74}, {32, -70}, {22, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -56}, {30, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{30, -66}, {32, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -66}, {26, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{22, -62}, {24, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -62}, {30, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -56}, {26, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -50}, {26, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -42}, {26, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -36}, {26, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -32}, {26, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -24}, {30, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, -18}, {26, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{28, -8}, {30, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{24, 0}, {26, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{30, 12}, {32, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{42, -70}, {44, -74}, {50, -74}, {52, -70}, {42, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -56}, {50, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{50, -66}, {52, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -66}, {46, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{42, -62}, {44, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -62}, {50, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -56}, {46, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -50}, {46, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -42}, {46, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -36}, {46, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -32}, {46, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{48, -24}, {50, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, -18}, {46, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{40, -8}, {38, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{44, 0}, {46, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{50, 12}, {52, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{60, -70}, {62, -74}, {68, -74}, {70, -70}, {60, -70}}, lineColor = {0, 0, 255}, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -56}, {68, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{68, -66}, {70, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -66}, {64, -68}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{60, -62}, {62, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -62}, {68, -64}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -56}, {64, -58}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -50}, {64, -48}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -42}, {64, -44}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -36}, {64, -38}}, lineColor = {0, 127, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -32}, {64, -34}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -24}, {68, -26}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, -18}, {64, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{66, -8}, {68, -10}}, lineColor = {0, 127, 255}, fillColor = {0, 95, 191}, fillPattern = FillPattern.Solid), Rectangle(extent = {{62, 0}, {64, -2}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{68, 12}, {70, 10}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{54, -6}, {57, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{50, -6}, {60, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-56, -6}, {-53, 40}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-60, -6}, {-50, -16}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-90, 40}, {-53, 40}}, color = {0, 0, 0})}));
    end nitri;

    partial model SecClarKrebs 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Polygon(points = {{-20, -70}, {20, -70}, {4, -84}, {-4, -84}, {-20, -70}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-4, -84}, {4, -92}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{-80, -48}, {-36, -64}, {38, -64}, {80, -48}, {-80, -48}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-80, 62}, {80, 16}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-80, 16}, {80, -40}}, lineColor = {223, 191, 159}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Polygon(points = {{-36, -64}, {38, -64}, {20, -70}, {-20, -70}, {-36, -64}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{4, -92}, {4, -84}, {20, -70}, {80, -48}}, color = {0, 0, 0}, thickness = 0.5), Rectangle(extent = {{-80, -40}, {80, -48}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{80, 62}, {92, 54}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Line(points = {{80, 54}, {92, 54}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-4, -92}, {-4, -84}, {-20, -70}, {-80, -48}, {-80, 10}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-80, 62}, {-80, 16}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-80, 10}, {-90, 10}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-80, 16}, {-90, 16}}, color = {0, 0, 0}, thickness = 0.5), Rectangle(extent = {{-20, -92}, {20, -98}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Line(points = {{-20, -92}, {-4, -92}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-20, -98}, {20, -98}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{20, -92}, {4, -92}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{80, -48}, {80, 54}}, color = {0, 0, 0}, thickness = 0.5), Text(extent = {{-100, -60}, {-40, -80}}, textString = "return"), Text(extent = {{40, -60}, {100, -80}}, textString = "waste"), Polygon(points = {{-30, -10}, {-26, -10}, {-26, -26}, {-20, -24}, {-28, -38}, {-36, -24}, {-30, -26}, {-30, -10}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{18, -28}, {22, -28}, {22, -44}, {28, -42}, {20, -56}, {12, -42}, {18, -44}, {18, -28}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, 16}, {-80, 10}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid)}), Documentation(info = "Icon for a secondary clarifier"));
    end SecClarKrebs;

    partial model blower 
       annotation(Documentation(info = "Icon for an air blower"), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-101, -70}, {70, -99}}, textString = "%name"), Rectangle(extent = {{-90, 10}, {-10, -68}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-87, 7}, {-13, -65}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Rectangle(extent = {{-80, -74}, {-60, -68}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-84, 5}, {-16, -63}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-51, -24}, {-24, -9}, {-47, -25}, {-51, -24}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-53, -35}, {-73, -53}, {-49, -35}, {-53, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-48, -29}, {-28, -51}, {-48, -33}, {-48, -29}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-56, -32}, {-72, -6}, {-56, -27}, {-56, -32}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-46, -26}, {-18, -33}, {-46, -29}, {-46, -26}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Polygon(points = {{-46, -34}, {-50, -60}, {-49, -34}, {-46, -34}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-54, -35}, {-82, -27}, {-54, -32}, {-54, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-56, -26}, {-48, 3}, {-52, -26}, {-56, -26}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-58, -23}, {-44, -36}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-86, -53}, {-12, -55}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-87, -19}, {-12, -21}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-87, -2}, {-12, -4}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{12, 32}, {48, 10}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{12, 52}, {48, 17}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-66, 34}, {-34, 16}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-49, 40}, {-52, 35}, {-50, 35}, {-50, 22}, {-48, 22}, {-48, 35}, {-46, 35}, {-49, 40}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-87, -38}, {-12, -40}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-68, 33}, {-32, 10}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-68, 52}, {-32, 18}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-65, 50}, {-34, 19}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-53, 52}, {31, 29}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-54, 49}, {29, 32}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-65, 35}, {-35, 16}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{15, 34}, {45, 16}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-27, 90}, {7, 52}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-24, 83}, {4, 48}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Ellipse(extent = {{16, 49}, {45, 19}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-10, 10}, {70, -68}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-7, 7}, {67, -65}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Rectangle(extent = {{40, -74}, {60, -68}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-4, 5}, {64, -63}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{29, -24}, {56, -9}, {33, -25}, {29, -24}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{27, -35}, {7, -53}, {31, -35}, {27, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{32, -29}, {52, -51}, {32, -33}, {32, -29}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{24, -32}, {8, -6}, {24, -27}, {24, -32}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{34, -26}, {62, -33}, {34, -29}, {34, -26}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Polygon(points = {{34, -34}, {30, -60}, {31, -34}, {34, -34}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{26, -35}, {-2, -27}, {26, -32}, {26, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{24, -26}, {32, 3}, {28, -26}, {24, -26}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{22, -23}, {36, -36}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -53}, {66, -55}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -19}, {66, -21}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -2}, {66, -4}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -38}, {66, -40}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{70, -13}, {80, -47}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{80, -27}, {87, -33}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {160, 160, 164}), Rectangle(extent = {{85, -21}, {88, -39}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {128, 128, 128}), Polygon(points = {{-49, 41}, {-55, 35}, {-51, 35}, {-51, 22}, {-48, 22}, {-48, 35}, {-44, 35}, {-49, 41}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid), Polygon(points = {{29, 41}, {23, 35}, {27, 35}, {27, 22}, {30, 22}, {30, 35}, {34, 35}, {29, 41}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid), Polygon(points = {{-11, 74}, {-16, 68}, {-12, 68}, {-12, 55}, {-9, 55}, {-9, 68}, {-5, 68}, {-11, 74}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-98, -70}, {70, -100}}, textString = "%name"), Rectangle(extent = {{-90, 10}, {-10, -68}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-87, 7}, {-13, -65}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Rectangle(extent = {{-80, -74}, {-60, -68}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-84, 5}, {-16, -63}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-51, -24}, {-24, -9}, {-47, -25}, {-51, -24}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-53, -35}, {-73, -53}, {-49, -35}, {-53, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-48, -29}, {-28, -51}, {-48, -33}, {-48, -29}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-56, -32}, {-72, -6}, {-56, -27}, {-56, -32}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-46, -26}, {-18, -33}, {-46, -29}, {-46, -26}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Polygon(points = {{-46, -34}, {-50, -60}, {-49, -34}, {-46, -34}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-54, -35}, {-82, -27}, {-54, -32}, {-54, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-56, -26}, {-48, 3}, {-52, -26}, {-56, -26}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-58, -23}, {-44, -36}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-86, -53}, {-12, -55}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-87, -19}, {-12, -21}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-87, -2}, {-12, -4}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{12, 32}, {48, 10}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{12, 52}, {48, 17}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-66, 34}, {-34, 16}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{-49, 40}, {-52, 35}, {-50, 35}, {-50, 22}, {-48, 22}, {-48, 35}, {-46, 35}, {-49, 40}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-87, -38}, {-12, -40}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-68, 33}, {-32, 10}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-68, 52}, {-32, 18}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-65, 50}, {-34, 19}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-53, 52}, {31, 29}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-54, 49}, {29, 32}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-65, 35}, {-35, 16}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{15, 34}, {45, 16}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-27, 90}, {7, 52}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-24, 83}, {4, 48}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Ellipse(extent = {{16, 49}, {45, 19}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-10, 10}, {70, -68}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-7, 7}, {67, -65}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Rectangle(extent = {{40, -74}, {60, -68}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-4, 5}, {64, -63}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {127, 191, 255}, fillPattern = FillPattern.Solid), Polygon(points = {{29, -24}, {56, -9}, {33, -25}, {29, -24}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{27, -35}, {7, -53}, {31, -35}, {27, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{32, -29}, {52, -51}, {32, -33}, {32, -29}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{24, -32}, {8, -6}, {24, -27}, {24, -32}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{34, -26}, {62, -33}, {34, -29}, {34, -26}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Polygon(points = {{34, -34}, {30, -60}, {31, -34}, {34, -34}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{26, -35}, {-2, -27}, {26, -32}, {26, -35}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{24, -26}, {32, 3}, {28, -26}, {24, -26}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{22, -23}, {36, -36}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -53}, {66, -55}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -19}, {66, -21}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -2}, {66, -4}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-8, -38}, {66, -40}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{70, -13}, {80, -47}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{80, -27}, {87, -33}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {160, 160, 164}), Rectangle(extent = {{85, -21}, {88, -39}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {128, 128, 128}), Polygon(points = {{-49, 41}, {-55, 35}, {-51, 35}, {-51, 22}, {-48, 22}, {-48, 35}, {-44, 35}, {-49, 41}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid), Polygon(points = {{29, 41}, {23, 35}, {27, 35}, {27, 22}, {30, 22}, {30, 35}, {34, 35}, {29, 41}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid), Polygon(points = {{-11, 74}, {-16, 68}, {-12, 68}, {-12, 55}, {-9, 55}, {-9, 68}, {-5, 68}, {-11, 74}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {159, 223, 223}, fillPattern = FillPattern.Solid)}), Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49));
    end blower;

    partial model pump 
       annotation(Documentation(info = "Icon for a wastewater pump "), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-90, 99}, {90, 59}}, textString = "%name"), Rectangle(extent = {{-60, 55}, {60, -55}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {160, 160, 164}), Rectangle(extent = {{-90, -50}, {0, -15}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {0, 0, 0}), Rectangle(extent = {{0, 46}, {90, 10}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {0, 0, 0}), Ellipse(extent = {{-51, 50}, {51, -51}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Ellipse(extent = {{-48, 47}, {48, -48}}, lineColor = {0, 0, 0}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, -18}, {0, -47}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{0, 43}, {90, 13}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{4, 9}, {40, 20}, {8, 7}, {4, 9}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-7, -11}, {-34, -28}, {-1, -11}, {-7, -11}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{8, -2}, {22, -37}, {8, -10}, {8, -2}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-10, -1}, {-24, 34}, {-10, 6}, {-10, -1}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{10, 3}, {41, -13}, {11, -1}, {10, 3}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Polygon(points = {{6, -10}, {-8, -43}, {1, -12}, {6, -10}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-7, -8}, {-43, 8}, {-9, -3}, {-7, -8}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-7, 3}, {8, 41}, {-2, 6}, {-7, 3}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-14, 11}, {13, -14}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-3, 1}, {2, -4}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {0, 0, 0}), Rectangle(extent = {{-40, -60}, {-20, -55}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{20, -60}, {40, -55}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-70, 41}, {-60, 9}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Polygon(points = {{-80, -30}, {-63, -30}, {-65, -22}, {-48, -32}, {-65, -43}, {-63, -35}, {-80, -35}, {-80, -30}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-40, 55}, {-20, 60}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{20, 55}, {40, 60}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-74, 28}, {-70, 22}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {128, 128, 128}), Rectangle(extent = {{-78, 33}, {-74, 17}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Polygon(points = {{51, 31}, {68, 31}, {65, 40}, {83, 29}, {65, 17}, {68, 26}, {51, 26}, {51, 31}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-90, 99}, {90, 59}}, textString = "%name"), Rectangle(extent = {{-60, 55}, {60, -55}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {160, 160, 164}), Rectangle(extent = {{-90, -50}, {0, -15}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {0, 0, 0}), Rectangle(extent = {{0, 46}, {90, 10}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {0, 0, 0}), Ellipse(extent = {{-51, 50}, {51, -51}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Ellipse(extent = {{-48, 47}, {48, -48}}, lineColor = {0, 0, 0}, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, -18}, {0, -47}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{0, 43}, {90, 13}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{4, 9}, {40, 20}, {8, 7}, {4, 9}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-7, -11}, {-34, -28}, {-1, -11}, {-7, -11}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{8, -2}, {22, -37}, {8, -10}, {8, -2}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-10, -1}, {-24, 34}, {-10, 6}, {-10, -1}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{10, 3}, {41, -13}, {11, -1}, {10, 3}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Polygon(points = {{6, -10}, {-8, -43}, {1, -12}, {6, -10}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-7, -8}, {-43, 8}, {-9, -3}, {-7, -8}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-7, 3}, {8, 41}, {-2, 6}, {-7, 3}}, lineColor = {128, 128, 128}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-14, 11}, {13, -14}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-3, 1}, {2, -4}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {0, 0, 0}), Rectangle(extent = {{-40, -60}, {-20, -55}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{20, -60}, {40, -55}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-70, 41}, {-60, 9}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Polygon(points = {{-80, -30}, {-63, -30}, {-65, -22}, {-48, -32}, {-65, -43}, {-63, -35}, {-80, -35}, {-80, -30}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-40, 55}, {-20, 60}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{20, 55}, {40, 60}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-74, 28}, {-70, 22}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {128, 128, 128}), Rectangle(extent = {{-78, 33}, {-74, 17}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {128, 128, 128}), Polygon(points = {{51, 31}, {68, 31}, {65, 40}, {83, 29}, {65, 17}, {68, 26}, {51, 26}, {51, 31}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid)}), Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49));
    end pump;

    partial model WWSource 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-54, 54}, {56, -54}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-8, -54}, {-14, -52}, {-24, -48}, {-32, -44}, {-36, -40}, {-42, -34}, {-48, -26}, {-50, -20}, {52, -20}, {50, -26}, {46, -32}, {42, -36}, {38, -40}, {34, -44}, {30, -46}, {26, -48}, {22, -50}, {16, -52}, {10, -54}, {4, -54}, {0, -54}, {-8, -54}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-54, 54}, {56, -54}}, lineColor = {0, 0, 0}, lineThickness = 0.5), Rectangle(extent = {{-4, -52}, {4, -74}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{4, -74}, {88, -68}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Line(points = {{-4, -54}, {-4, -74}, {88, -74}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{4, -54}, {4, -68}, {88, -68}}, color = {0, 0, 0}, thickness = 0.5), Text(extent = {{-98, 98}, {98, 60}}, textString = "%name")}), Documentation(info = "Icon for a wastewater source"), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-54, 54}, {56, -54}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Polygon(points = {{-8, -54}, {-14, -52}, {-24, -48}, {-32, -44}, {-36, -40}, {-42, -34}, {-48, -26}, {-50, -20}, {52, -20}, {50, -26}, {46, -32}, {42, -36}, {38, -40}, {34, -44}, {30, -46}, {26, -48}, {22, -50}, {16, -52}, {10, -54}, {4, -54}, {0, -54}, {-8, -54}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-54, 54}, {56, -54}}, lineColor = {0, 0, 0}, lineThickness = 0.5), Rectangle(extent = {{-4, -52}, {4, -74}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{4, -74}, {88, -68}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Line(points = {{-4, -54}, {-4, -74}, {88, -74}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{4, -54}, {4, -68}, {88, -68}}, color = {0, 0, 0}, thickness = 0.5), Text(extent = {{-98, 98}, {98, 60}}, textString = "%name")}));
    end WWSource;

    partial model EffluentSink 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-90, 38}, {0, 0}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Polygon(points = {{-80, -20}, {-60, -80}, {60, -80}, {80, -20}, {-80, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Line(points = {{0, 14}, {8, 4}, {10, 0}, {12, -8}, {12, -16}, {12, -22}}, color = {0, 127, 255}, thickness = 1), Line(points = {{0, 20}, {6, 14}, {8, 12}, {10, 8}, {12, 2}, {14, -2}, {16, -6}, {16, -10}, {16, -16}, {16, -22}}, color = {0, 127, 255}, thickness = 1), Line(points = {{0, 8}, {4, 6}, {6, 2}, {8, -2}, {8, -6}, {10, -14}, {10, -20}}, color = {0, 127, 255}, thickness = 1), Polygon(points = {{-80, -20}, {-94, -20}, {-94, -80}, {-60, -80}, {-80, -20}, {-80, -20}}, lineColor = {0, 191, 0}, fillColor = {0, 191, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{80, -20}, {94, -20}, {94, -80}, {60, -80}, {80, -20}}, lineColor = {0, 191, 0}, fillColor = {0, 191, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-94, -80}, {94, -94}}, lineColor = {0, 191, 0}, fillColor = {0, 191, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-52, 58}, {-52, 42}, {-40, 50}, {-52, 58}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-40, 50}, {-30, 58}, {-30, 42}, {-40, 50}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Line(points = {{-40, 50}, {-40, 38}}, color = {0, 0, 0}, thickness = 1), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name")}), Documentation(info = "Icon for a water sink e.g. receiving water "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-90, 38}, {0, 0}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Polygon(points = {{-80, -20}, {-60, -80}, {60, -80}, {80, -20}, {-80, -20}}, lineColor = {0, 127, 255}, fillColor = {0, 127, 255}, fillPattern = FillPattern.Solid), Line(points = {{0, 14}, {8, 4}, {10, 0}, {12, -8}, {12, -16}, {12, -22}}, color = {0, 127, 255}, thickness = 1), Line(points = {{0, 20}, {6, 14}, {8, 12}, {10, 8}, {12, 2}, {14, -2}, {16, -6}, {16, -10}, {16, -16}, {16, -22}}, color = {0, 127, 255}, thickness = 1), Line(points = {{0, 8}, {4, 6}, {6, 2}, {8, -2}, {8, -6}, {10, -14}, {10, -20}}, color = {0, 127, 255}, thickness = 1), Polygon(points = {{-80, -20}, {-94, -20}, {-94, -80}, {-60, -80}, {-80, -20}, {-80, -20}}, lineColor = {0, 191, 0}, fillColor = {0, 191, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{80, -20}, {94, -20}, {94, -80}, {60, -80}, {80, -20}}, lineColor = {0, 191, 0}, fillColor = {0, 191, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-94, -80}, {94, -94}}, lineColor = {0, 191, 0}, fillColor = {0, 191, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-52, 58}, {-52, 42}, {-40, 50}, {-52, 58}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-40, 50}, {-30, 58}, {-30, 42}, {-40, 50}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Line(points = {{-40, 50}, {-40, 38}}, color = {0, 0, 0}, thickness = 1), Text(extent = {{-100, 100}, {98, 60}}, textString = "%name")}));
    end EffluentSink;

    partial model SludgeSink 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Rectangle(extent = {{-90, 6}, {-36, -30}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Polygon(points = {{-72, 26}, {-72, 10}, {-60, 18}, {-72, 26}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-60, 18}, {-50, 26}, {-50, 10}, {-60, 18}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Line(points = {{-60, 18}, {-60, 6}}, color = {0, 0, 0}, thickness = 1), Text(extent = {{-100, 100}, {99, 60}}, textString = "%name"), Ellipse(extent = {{-36, 60}, {80, -40}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-36, 6}, {81, -81}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-36, -44}, {80, -80}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Line(points = {{-36, -30}, {-36, -80}, {80, -80}, {80, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, -14}, {-32, -16}, {-28, -18}, {-26, -22}, {-24, -26}, {-22, -30}, {-20, -34}, {-18, -40}, {-18, -44}}, color = {191, 95, 0}, thickness = 1), Line(points = {{-36, -10}, {-28, -12}, {-24, -16}, {-20, -20}, {-16, -26}, {-12, -34}, {-12, -44}}, color = {191, 95, 0}, thickness = 1), Polygon(points = {{42, -36}, {52, -50}, {62, -36}, {42, -36}}, lineColor = {0, 0, 0}, fillColor = {160, 160, 164}, fillPattern = FillPattern.Solid), Line(points = {{62, -36}, {80, -36}}, color = {0, 0, 0}), Rectangle(extent = {{-100, -80}, {100, -100}}, lineColor = {160, 160, 164}, fillColor = {160, 160, 164}, fillPattern = FillPattern.Solid)}), Documentation(info = "Icon for a wastewater sink"), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Rectangle(extent = {{-90, 6}, {-36, -30}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Polygon(points = {{-72, 26}, {-72, 10}, {-60, 18}, {-72, 26}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-60, 18}, {-50, 26}, {-50, 10}, {-60, 18}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Line(points = {{-60, 18}, {-60, 6}}, color = {0, 0, 0}, thickness = 1), Text(extent = {{-99, 100}, {100, 59}}, textString = "%name"), Ellipse(extent = {{-36, 60}, {80, -40}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-36, 6}, {81, -81}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-36, -44}, {80, -80}}, lineColor = {191, 95, 0}, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Line(points = {{-36, -30}, {-36, -80}, {80, -80}, {80, 6}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, -14}, {-32, -16}, {-28, -18}, {-26, -22}, {-24, -26}, {-22, -30}, {-20, -34}, {-18, -40}, {-18, -44}}, color = {191, 95, 0}, thickness = 1), Line(points = {{-36, -10}, {-28, -12}, {-24, -16}, {-20, -20}, {-16, -26}, {-12, -34}, {-12, -44}}, color = {191, 95, 0}, thickness = 1), Polygon(points = {{42, -36}, {52, -50}, {62, -36}, {42, -36}}, lineColor = {0, 0, 0}, fillColor = {160, 160, 164}, fillPattern = FillPattern.Solid), Line(points = {{62, -36}, {80, -36}}, color = {0, 0, 0}), Rectangle(extent = {{-100, -80}, {100, -100}}, lineColor = {160, 160, 164}, fillColor = {160, 160, 164}, fillPattern = FillPattern.Solid)}));
    end SludgeSink;

    partial model divider2 
       annotation(Documentation(info = "Icon for a flow divider into 2 flows"), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-99, 98}, {97, 53}}, textString = "%name"), Rectangle(extent = {{-45, 47}, {47, -37}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {128, 128, 128}), Rectangle(extent = {{90, 40}, {2, 10}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{90, 0}, {3, -30}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-40, 40}, {40, -30}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-91, 20}, {8, -14}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-38, 38}, {38, -28}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{3, 37}, {90, 13}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{5, -3}, {90, -27}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-91, 16}, {-32, -10}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{3, -11}, {22, -11}, {21, -9}, {27, -12}, {21, -15}, {22, -13}, {3, -13}, {3, -11}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{47, 31}, {66, 31}, {65, 33}, {71, 30}, {65, 27}, {66, 29}, {47, 29}, {47, 31}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{48, -18}, {67, -18}, {66, -16}, {72, -19}, {66, -22}, {67, -20}, {48, -20}, {48, -18}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-42, 0}, {-23, 0}, {-24, 2}, {-18, -1}, {-24, -4}, {-23, -2}, {-42, -2}, {-42, 0}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-80, 12}, {-61, 12}, {-62, 14}, {-56, 11}, {-62, 8}, {-61, 10}, {-80, 10}, {-80, 12}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-12, 24}, {7, 24}, {6, 26}, {12, 23}, {6, 20}, {7, 22}, {-12, 22}, {-12, 24}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid)}), Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-99, 98}, {98, 52}}, textString = "%name"), Rectangle(extent = {{-45, 47}, {47, -37}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {128, 128, 128}), Rectangle(extent = {{90, 40}, {2, 10}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{90, 0}, {3, -30}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-40, 40}, {40, -30}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Rectangle(extent = {{-91, 20}, {8, -14}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-38, 38}, {38, -28}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{3, 37}, {90, 13}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{5, -3}, {90, -27}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-91, 16}, {-32, -10}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{3, -11}, {22, -11}, {21, -9}, {27, -12}, {21, -15}, {22, -13}, {3, -13}, {3, -11}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{47, 31}, {66, 31}, {65, 33}, {71, 30}, {65, 27}, {66, 29}, {47, 29}, {47, 31}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{48, -18}, {67, -18}, {66, -16}, {72, -19}, {66, -22}, {67, -20}, {48, -20}, {48, -18}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-42, 0}, {-23, 0}, {-24, 2}, {-18, -1}, {-24, -4}, {-23, -2}, {-42, -2}, {-42, 0}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-80, 12}, {-61, 12}, {-62, 14}, {-56, 11}, {-62, 8}, {-61, 10}, {-80, 10}, {-80, 12}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-12, 24}, {7, 24}, {6, 26}, {12, 23}, {6, 20}, {7, 22}, {-12, 22}, {-12, 24}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid)}));
    end divider2;

    partial model mixer3 
       annotation(Documentation(info = "Icon for a mixer of 3 flows"), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-100, 99}, {99, 59}}, textString = "%name"), Rectangle(extent = {{-60, 56}, {60, -65}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {160, 160, 164}), Rectangle(extent = {{30, 20}, {90, -30}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-90, 50}, {0, 20}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-90, -30}, {0, -60}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-90, 10}, {0, -20}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-50, 50}, {50, -41}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-50, 31}, {50, -60}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-47, 48}, {47, -38}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-47, 28}, {47, -58}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, 47}, {2, 23}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, 7}, {2, -17}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, -33}, {0, -57}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{11, 16}, {90, -26}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{-20, -7}, {-1, -7}, {-2, -5}, {4, -8}, {-2, -11}, {-1, -9}, {-20, -9}, {-20, -7}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{10, 14}, {29, 14}, {28, 16}, {34, 13}, {28, 10}, {29, 12}, {10, 12}, {10, 14}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{11, -28}, {30, -28}, {29, -26}, {35, -29}, {29, -32}, {30, -30}, {11, -30}, {11, -28}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-75, -1}, {-56, -1}, {-57, 1}, {-51, -2}, {-57, -5}, {-56, -3}, {-75, -3}, {-75, -1}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-74, 41}, {-55, 41}, {-56, 43}, {-50, 40}, {-56, 37}, {-55, 39}, {-74, 39}, {-74, 41}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-71, -49}, {-52, -49}, {-53, -47}, {-47, -50}, {-53, -53}, {-52, -51}, {-71, -51}, {-71, -49}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-30, 33}, {-11, 33}, {-12, 35}, {-6, 32}, {-12, 29}, {-11, 31}, {-30, 31}, {-30, 33}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-28, -42}, {-9, -42}, {-10, -40}, {-4, -43}, {-10, -46}, {-9, -44}, {-28, -44}, {-28, -42}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{44, -7}, {63, -7}, {62, -5}, {68, -8}, {62, -11}, {63, -9}, {44, -9}, {44, -7}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid)}), Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Text(extent = {{-99, 98}, {99, 60}}, textString = "%name"), Rectangle(extent = {{-60, 56}, {60, -65}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder, fillColor = {160, 160, 164}), Rectangle(extent = {{30, 20}, {90, -30}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-90, 50}, {0, 20}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-90, -30}, {0, -60}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Rectangle(extent = {{-90, 10}, {0, -20}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {192, 192, 192}), Ellipse(extent = {{-50, 50}, {50, -41}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-50, 31}, {50, -60}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {192, 192, 192}), Ellipse(extent = {{-47, 48}, {47, -38}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Ellipse(extent = {{-47, 28}, {47, -58}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, 47}, {2, 23}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, 7}, {2, -17}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, -33}, {0, -57}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Rectangle(extent = {{11, 16}, {90, -26}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {223, 191, 159}, fillPattern = FillPattern.Solid), Polygon(points = {{-20, -7}, {-1, -7}, {-2, -5}, {4, -8}, {-2, -11}, {-1, -9}, {-20, -9}, {-20, -7}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{10, 14}, {29, 14}, {28, 16}, {34, 13}, {28, 10}, {29, 12}, {10, 12}, {10, 14}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{11, -28}, {30, -28}, {29, -26}, {35, -29}, {29, -32}, {30, -30}, {11, -30}, {11, -28}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-75, -1}, {-56, -1}, {-57, 1}, {-51, -2}, {-57, -5}, {-56, -3}, {-75, -3}, {-75, -1}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-74, 41}, {-55, 41}, {-56, 43}, {-50, 40}, {-56, 37}, {-55, 39}, {-74, 39}, {-74, 41}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-71, -49}, {-52, -49}, {-53, -47}, {-47, -50}, {-53, -53}, {-52, -51}, {-71, -51}, {-71, -49}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-30, 33}, {-11, 33}, {-12, 35}, {-6, 32}, {-12, 29}, {-11, 31}, {-30, 31}, {-30, 33}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-28, -42}, {-9, -42}, {-10, -40}, {-4, -43}, {-10, -46}, {-9, -44}, {-28, -44}, {-28, -42}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{44, -7}, {63, -7}, {62, -5}, {68, -8}, {62, -11}, {63, -9}, {44, -9}, {44, -7}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {191, 95, 0}, fillPattern = FillPattern.Solid)}));
    end mixer3;

    partial model sensor_COD 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "COD"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}), Documentation(info = "Icon for a chemical oxygen demand sensor "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "COD"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}));
    end sensor_COD;

    partial model sensor_NH 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-98, 98}, {98, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "NH"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}), Documentation(info = "Icon for an ammonium nitrogen sensor "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "NH"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}));
    end sensor_NH;

    partial model sensor_NO 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "NO"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}), Documentation(info = "Icon for a nitrate nitrogen sensor "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "NO"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}));
    end sensor_NO;

    partial model sensor_O2 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-98, 98}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "O2"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}), Documentation(info = "Icon for an oxygen sensor "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-98, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "O2"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}));
    end sensor_O2;

    partial model sensor_TKN 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-98, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "TKN"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}), Documentation(info = "Icon for a Kjeldal nitrogen and total nitrogen sensor "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-98, 98}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "TKN"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}));
    end sensor_TKN;

    partial model sensor_TSS 
       annotation(Window(x = 0.45, y = 0.01, width = 0.35, height = 0.49), Icon(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-100, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "TSS"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}), Documentation(info = "Icon for a total suspended solids sensor "), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-50, 50}, {50, -50}}, lineColor = {0, 0, 0}, lineThickness = 0.5, fillColor = {223, 223, 159}, fillPattern = FillPattern.Solid), Text(extent = {{-98, 100}, {100, 60}}, textString = "%name"), Line(points = {{0, 50}, {0, 38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {38, 0}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-36, 34}, {-28, 26}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{34, 36}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{0, 0}, {26, 28}}, color = {0, 0, 0}, thickness = 0.5), Polygon(points = {{30, 32}, {10, 24}, {24, 12}, {30, 32}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-36, -10}, {36, -32}}, textString = "TSS"), Line(points = {{0, -50}, {0, -90}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{50, 0}, {88, 0}}, color = {0, 0, 0})}));
    end sensor_TSS;
     annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This package is part of the WasteWater Library and contains
      definitions for the graphical layout of components which are
      used within the sub-libraries ASM1, ASM2d and ASM3.

      Main Author:
         Gerald Reichl
         Technische Universitaet Ilmenau
         Faculty of Informatics and Automation
         Department Dynamics and Simulation of ecological Systems
         P.O. Box 10 05 65
         98684 Ilmenau
         Germany
         email: gerald.reichl@tu-ilmenau.de


      This package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
      disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

      Copyright (C) 2003, Gerald Reichl
      "));
  end Icons;

  package WasteWaterUnits 
    extends Modelica.Icons.Library;
    type VolumeFlowRate = Real(final quantity = "VolumeFlowRate", final unit = "m3/d");
    type MassConcentration = Real(final quantity = "MassConcentration", final unit = "mg/l", final min = 0.0);
    type Alkalinity = Real(final quantity = "Alkalinity", final unit = "mmol/l", final min = 0.0);
    type SludgeVolumeIndex = Real(final quantity = "SludgeVolumeIndex", final unit = "ml/g");
     annotation(Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Invisible = true, Documentation(info = "This package provides special convention for units and types
      for use with the activated sludge models.

      These are:
        - VolumeFlowRate with unit [m3/d],
        - MassConcentration with unit [mg/l] and min=0.0,
        - Alkalinity with unit [mmol/l] and min=0.0,
        - SedimentationVelocity with unit [m/d],
        - SedimentationFlux with unit [g/(m2*d)],
        - SludgeVolumeIndex with unit [ml/g],


      Main Author:
         Gerald Reichl
         Technische Universitaet Ilmenau
         Faculty of Informatics and Automation
         Department Dynamics and Simulation of ecological Systems
         P.O. Box 10 05 65
         98684 Ilmenau
         Germany
         email: gerald.reichl@tu-ilmenau.de


      The package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
      disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

      Copyright (C) 2000 - 2001, Gerald Reichl
      "));
  end WasteWaterUnits;

  package Blocks 
    block SimulationTime 
      extends Modelica.Blocks.Interfaces.SO;
    equation
      y = time;
    end SimulationTime;
  end Blocks;
   annotation(uses(grid = [2, 2], component = [20, 20], Modelica(version = "3.2")), Window(x = 0.45, y = 0.01, width = 0.44, height = 0.65, library = 1, autolayout = 1), Documentation(info = "This package can be used to model and simulate biological municipal
    wastewater treatment plants.

    The package WasteWater provides a collection of 3 different
    international accepted (a)ctivated (s)ludge (m)odels (ASM) of different complexity,
    including models for preclarifiers and 5 different models for
    the secondary settling process.

    PS: SimulationTimeUnit for all models built with the WasteWater library is days [d].

    The WasteWater package is free software; it can be redistributed and/or modified under the terms of the Modelica license, see the license conditions and the accompanying
    disclaimer in the documentation of package Modelica in file \"Modelica/package.mo\".

    The WasteWater package currently consists of the following subpackages

      ASM1            - Activated Sludge Model No.1  (models 13 wastewater components and 8 biological processes)
      ASM2d           - Activated sludge Model No.2d (models 19 wastewater components and 21 biological processes)
      ASM3            - Activated Sludge Model No.3  (models 13 wastewater components and 12 biological processes)
      Icons           - Icon definitions for wastewater treatment components
      WasteWaterUnits - unit type definitions for use with the WasteWater library


    Main Author:
       Gerald Reichl
       Technische Universitaet Ilmenau
       Faculty of Informatics and Automation
       Department Dynamics and Simulation of ecological Systems
       P.O. Box 10 05 65
       98684 Ilmenau
       Germany
       email: gerald.reichl@tu-ilmenau.de

    Release Notes:
       version 2.0.1 (2003)
       First official version of the library released at 3rd Modelica Conference.

    Copyright (C) 2000 - 2003, Gerald Reichl
    "), uses(Modelica(version = "2.2.2")), version = "1", conversion(noneFromVersion = ""));
end WasteWater;

package Modelica 
  extends Modelica.Icons.Library;

  package Blocks 
    extends Modelica.Icons.Library2;

    package Continuous 
      extends Modelica.Icons.Library;

      block PI 
        parameter Real k = 1 "Gain";
        parameter .Modelica.SIunits.Time T(start = 1, min = Modelica.Constants.small) "Time Constant (T>0 required)";
        parameter Modelica.Blocks.Types.Init initType = Modelica.Blocks.Types.Init.NoInit "Type of initialization (1: no init, 2: steady state, 3: initial state, 4: initial output)" annotation(Evaluate = true, Dialog(group = "Initialization"));
        parameter Real x_start = 0 "Initial or guess value of state" annotation(Dialog(group = "Initialization"));
        parameter Real y_start = 0 "Initial value of output" annotation(Dialog(enable = initType == Init.SteadyState or initType == Init.InitialOutput, group = "Initialization"));
        extends .Modelica.Blocks.Interfaces.SISO;
        output Real x(start = x_start) "State of block";
         annotation(defaultComponentName = "PI", Documentation(info = "
          <HTML>
          <p>
          This blocks defines the transfer function between the input u and
          the output y (element-wise) as <i>PI</i> system:
          </p>
          <pre>
                           1
             y = k * (1 + ---) * u
                          T*s
                     T*s + 1
               = k * ------- * u
                       T*s
          </pre>
          <p>
          If you would like to be able to change easily between different
          transfer functions (FirstOrder, SecondOrder, ... ) by changing
          parameters, use the general model class <b>TransferFunction</b>
          instead and model a PI SISO system with parameters<br>
          b = {k*T, k}, a = {T, 0}.
          </p>
          <pre>
          Example:

             parameter: k = 0.3,  T = 0.4

             results in:
                         0.4 s + 1
                y = 0.3 ----------- * u
                           0.4 s
          </pre>

          <p>
          It might be difficult to initialize the PI component in steady state
          due to the integrator part.
          This is discussed in the description of package
          <a href=\"Modelica://Modelica.Blocks.Continuous#info\">Continuous</a>.
          </p>

          </HTML>
          "), Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Line(points = {{-80, 78}, {-80, -90}}, color = {192, 192, 192}), Polygon(points = {{-80, 90}, {-88, 68}, {-72, 68}, {-80, 88}, {-80, 90}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-90, -80}, {82, -80}}, color = {192, 192, 192}), Polygon(points = {{90, -80}, {68, -72}, {68, -88}, {90, -80}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-80, -80}, {-80, -20}, {60, 80}}, color = {0, 0, 127}), Text(extent = {{0, 6}, {60, -56}}, lineColor = {192, 192, 192}, textString = "PI"), Text(extent = {{-150, -150}, {150, -110}}, lineColor = {0, 0, 0}, textString = "T=%T")}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Rectangle(extent = {{-60, 60}, {60, -60}}, lineColor = {0, 0, 255}), Text(extent = {{-68, 24}, {-24, -18}}, lineColor = {0, 0, 0}, textString = "k"), Text(extent = {{-32, 48}, {60, 0}}, lineColor = {0, 0, 0}, textString = "T s + 1"), Text(extent = {{-30, -8}, {52, -40}}, lineColor = {0, 0, 0}, textString = "T s"), Line(points = {{-24, 0}, {54, 0}}, color = {0, 0, 0}), Line(points = {{-100, 0}, {-60, 0}}, color = {0, 0, 255}), Line(points = {{62, 0}, {100, 0}}, color = {0, 0, 255})}));
      initial equation
        if initType == .Modelica.Blocks.Types.Init.SteadyState then
          der(x) = 0;
        elseif initType == .Modelica.Blocks.Types.Init.InitialState then
          x = x_start;
        elseif initType == .Modelica.Blocks.Types.Init.InitialOutput then
          y = y_start;
        end if;
      equation
        der(x) = u / T;
        y = k * (x + u);
      end PI;
       annotation(Documentation(info = "<html>
        <p>
        This package contains basic <b>continuous</b> input/output blocks
        described by differential equations.
        </p>

        <p>
        All blocks of this package can be initialized in different
        ways controlled by parameter <b>initType</b>. The possible
        values of initType are defined in
        <a href=\"Modelica://Modelica.Blocks.Types.Init\">Modelica.Blocks.Types.Init</a>:
        </p>

        <table border=1 cellspacing=0 cellpadding=2>
          <tr><td valign=\"top\"><b>Name</b></td>
              <td valign=\"top\"><b>Description</b></td></tr>

          <tr><td valign=\"top\"><b>Init.NoInit</b></td>
              <td valign=\"top\">no initialization (start values are used as guess values with fixed=false)</td></tr>

          <tr><td valign=\"top\"><b>Init.SteadyState</b></td>
              <td valign=\"top\">steady state initialization (derivatives of states are zero)</td></tr>

          <tr><td valign=\"top\"><b>Init.InitialState</b></td>
              <td valign=\"top\">Initialization with initial states</td></tr>

          <tr><td valign=\"top\"><b>Init.InitialOutput</b></td>
              <td valign=\"top\">Initialization with initial outputs (and steady state of the states if possibles)</td></tr>
        </table>

        <p>
        For backward compatibility reasons the default of all blocks is
        <b>Init.NoInit</b>, with the exception of Integrator and LimIntegrator
        where the default is <b>Init.InitialState</b> (this was the initialization
        defined in version 2.2 of the Modelica standard library).
        </p>

        <p>
        In many cases, the most useful initial condition is
        <b>Init.SteadyState</b> because initial transients are then no longer
        present. The drawback is that in combination with a non-linear
        plant, non-linear algebraic equations occur that might be
        difficult to solve if appropriate guess values for the
        iteration variables are not provided (i.e. start values with fixed=false).
        However, it is often already useful to just initialize
        the linear blocks from the Continuous blocks library in SteadyState.
        This is uncritical, because only linear algebraic equations occur.
        If Init.NoInit is set, then the start values for the states are
        interpreted as <b>guess</b> values and are propagated to the
        states with fixed=<b>false</b>.
        </p>

        <p>
        Note, initialization with Init.SteadyState is usually difficult
        for a block that contains an integrator
        (Integrator, LimIntegrator, PI, PID, LimPID).
        This is due to the basic equation of an integrator:
        </p>

        <pre>
          <b>initial equation</b>
             <b>der</b>(y) = 0;   // Init.SteadyState
          <b>equation</b>
             <b>der</b>(y) = k*u;
        </pre>

        <p>
        The steady state equation leads to the condition that the input to the
        integrator is zero. If the input u is already (directly or indirectly) defined
        by another initial condition, then the initialization problem is <b>singular</b>
        (has none or infinitely many solutions). This situation occurs often
        for mechanical systems, where, e.g., u = desiredSpeed - measuredSpeed and
        since speed is both a state and a derivative, it is always defined by
        Init.InitialState or Init.SteadyState initializtion.
        </p>

        <p>
        In such a case, <b>Init.NoInit</b> has to be selected for the integrator
        and an additional initial equation has to be added to the system
        to which the integrator is connected. E.g., useful initial conditions
        for a 1-dim. rotational inertia controlled by a PI controller are that
        <b>angle</b>, <b>speed</b>, and <b>acceleration</b> of the inertia are zero.
        </p>

        </html>
        "));
    end Continuous;

    package Interfaces 
      extends Modelica.Icons.Library;
      connector RealInput = input Real "'input Real' as connector" annotation(defaultComponentName = "u", Icon(graphics = {Polygon(points = {{-100, 100}, {100, 0}, {-100, -100}, {-100, 100}}, lineColor = {0, 0, 127}, fillColor = {0, 0, 127}, fillPattern = FillPattern.Solid)}, coordinateSystem(extent = {{-100, -100}, {100, 100}}, preserveAspectRatio = true, initialScale = 0.2)), Diagram(coordinateSystem(preserveAspectRatio = true, initialScale = 0.2, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Polygon(points = {{0, 50}, {100, 0}, {0, -50}, {0, 50}}, lineColor = {0, 0, 127}, fillColor = {0, 0, 127}, fillPattern = FillPattern.Solid), Text(extent = {{-10, 85}, {-10, 60}}, lineColor = {0, 0, 127}, textString = "%name")}), Documentation(info = "<html>
        <p>
        Connector with one input signal of type Real.
        </p>
        </html>"));
      connector RealOutput = output Real "'output Real' as connector" annotation(defaultComponentName = "y", Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Polygon(points = {{-100, 100}, {100, 0}, {-100, -100}, {-100, 100}}, lineColor = {0, 0, 127}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Polygon(points = {{-100, 50}, {0, 0}, {-100, -50}, {-100, 50}}, lineColor = {0, 0, 127}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid), Text(extent = {{30, 110}, {30, 60}}, lineColor = {0, 0, 127}, textString = "%name")}), Documentation(info = "<html>
        <p>
        Connector with one output signal of type Real.
        </p>
        </html>"));

      partial block BlockIcon 
         annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, -100}, {100, 100}}, lineColor = {0, 0, 127}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid), Text(extent = {{-150, 150}, {150, 110}}, textString = "%name", lineColor = {0, 0, 255})}), Documentation(info = "<html>
          <p>
          Block that has only the basic icon for an input/output
          block (no declarations, no equations). Most blocks
          of package Modelica.Blocks inherit directly or indirectly
          from this block.
          </p>
          </html>"));
      end BlockIcon;

      partial block SO 
        extends BlockIcon;
        RealOutput y "Connector of Real output signal" annotation(Placement(transformation(extent = {{100, -10}, {120, 10}}, rotation = 0)));
         annotation(Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {2, 2})), Documentation(info = "<html>
          <p>
          Block has one continuous Real output signal.
          </p>
          </html>"));
      end SO;

      partial block SISO 
        extends BlockIcon;
        RealInput u "Connector of Real input signal" annotation(Placement(transformation(extent = {{-140, -20}, {-100, 20}}, rotation = 0)));
        RealOutput y "Connector of Real output signal" annotation(Placement(transformation(extent = {{100, -10}, {120, 10}}, rotation = 0)));
         annotation(Documentation(info = "<html>
          <p>
          Block has one continuous Real input and one continuous Real output signal.
          </p>
          </html>"));
      end SISO;

      partial block SIMO 
        extends BlockIcon;
        parameter Integer nout = 1 "Number of outputs";
        RealInput u "Connector of Real input signal" annotation(Placement(transformation(extent = {{-140, -20}, {-100, 20}}, rotation = 0)));
        RealOutput[nout] y "Connector of Real output signals" annotation(Placement(transformation(extent = {{100, -10}, {120, 10}}, rotation = 0)));
         annotation(Documentation(info = "<HTML>
          <p> Block has one continuous Real input signal and a
              vector of continuous Real output signals.</p>

          </HTML>
          "));
      end SIMO;

      partial block SignalSource 
        extends SO;
        parameter Real offset = 0 "Offset of output signal y";
        parameter .Modelica.SIunits.Time startTime = 0 "Output y = offset for time < startTime";
         annotation(Documentation(info = "<html>
          <p>
          Basic block for Real sources of package Blocks.Sources.
          This component has one continuous Real output signal y
          and two parameters (offset, startTime) to shift the
          generated signal.
          </p>
          </html>"));
      end SignalSource;
       annotation(Documentation(info = "<HTML>
        <p>
        This package contains interface definitions for
        <b>continuous</b> input/output blocks with Real,
        Integer and Boolean signals. Furthermore, it contains
        partial models for continuous and discrete blocks.
        </p>

        </HTML>
        ", revisions = "<html>
        <ul>
        <li><i>Oct. 21, 2002</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
               and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
               Added several new interfaces. <a href=\"../Documentation/ChangeNotes1.5.html\">Detailed description</a> available.
        <li><i>Oct. 24, 1999</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
               RealInputSignal renamed to RealInput. RealOutputSignal renamed to
               output RealOutput. GraphBlock renamed to BlockIcon. SISOreal renamed to
               SISO. SOreal renamed to SO. I2SOreal renamed to M2SO.
               SignalGenerator renamed to SignalSource. Introduced the following
               new models: MIMO, MIMOs, SVcontrol, MVcontrol, DiscreteBlockIcon,
               DiscreteBlock, DiscreteSISO, DiscreteMIMO, DiscreteMIMOs,
               BooleanBlockIcon, BooleanSISO, BooleanSignalSource, MI2BooleanMOs.</li>
        <li><i>June 30, 1999</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
               Realized a first version, based on an existing Dymola library
               of Dieter Moormann and Hilding Elmqvist.</li>
        </ul>
        </html>
        "));
    end Interfaces;

    package Math 
      extends Modelica.Icons.Library;

      block Feedback 
        input .Modelica.Blocks.Interfaces.RealInput u1 annotation(Placement(transformation(extent = {{-100, -20}, {-60, 20}}, rotation = 0)));
        input .Modelica.Blocks.Interfaces.RealInput u2 annotation(Placement(transformation(origin = {0, -80}, extent = {{-20, -20}, {20, 20}}, rotation = 90)));
        output .Modelica.Blocks.Interfaces.RealOutput y annotation(Placement(transformation(extent = {{80, -10}, {100, 10}}, rotation = 0)));
         annotation(Documentation(info = "
          <HTML>
          <p>
          This blocks computes output <b>y</b> as <i>difference</i> of the
          commanded input <b>u1</b> and the feedback
          input <b>u2</b>:
          </p>
          <pre>
              <b>y</b> = <b>u1</b> - <b>u2</b>;
          </pre>
          <p>
          Example:
          </p>
          <pre>
               parameter:   n = 2

            results in the following equations:

               y = u1 - u2
          </pre>

          </HTML>
          "), Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-20, 20}, {20, -20}}, lineColor = {0, 0, 127}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid), Line(points = {{-60, 0}, {-20, 0}}, color = {0, 0, 127}), Line(points = {{20, 0}, {80, 0}}, color = {0, 0, 127}), Line(points = {{0, -20}, {0, -60}}, color = {0, 0, 127}), Text(extent = {{-14, 0}, {82, -94}}, lineColor = {0, 0, 0}, textString = "-"), Text(extent = {{-150, 94}, {150, 44}}, textString = "%name", lineColor = {0, 0, 255})}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Ellipse(extent = {{-20, 20}, {20, -20}}, pattern = LinePattern.Solid, lineThickness = 0.25, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Line(points = {{-60, 0}, {-20, 0}}, color = {0, 0, 255}), Line(points = {{20, 0}, {80, 0}}, color = {0, 0, 255}), Line(points = {{0, -20}, {0, -60}}, color = {0, 0, 255}), Text(extent = {{-12, 10}, {84, -84}}, lineColor = {0, 0, 0}, textString = "-")}));
      equation
        y = u1 - u2;
      end Feedback;
       annotation(Documentation(info = "
        <HTML>
        <p>
        This package contains basic <b>mathematical operations</b>,
        such as summation and multiplication, and basic <b>mathematical
        functions</b>, such as <b>sqrt</b> and <b>sin</b>, as
        input/output blocks. All blocks of this library can be either
        connected with continuous blocks or with sampled-data blocks.
        </p>
        </HTML>
        ", revisions = "<html>
        <ul>
        <li><i>October 21, 2002</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
               and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
               New blocks added: RealToInteger, IntegerToReal, Max, Min, Edge, BooleanChange, IntegerChange.</li>
        <li><i>August 7, 1999</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
               Realized (partly based on an existing Dymola library
               of Dieter Moormann and Hilding Elmqvist).
        </li>
        </ul>
        </html>"));
    end Math;

    package Sources 
      extends Modelica.Icons.Library;

      block Constant 
        parameter Real k(start = 1) "Constant output value";
        extends .Modelica.Blocks.Interfaces.SO;
         annotation(defaultComponentName = "const", Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Line(points = {{-80, 68}, {-80, -80}}, color = {192, 192, 192}), Polygon(points = {{-80, 90}, {-88, 68}, {-72, 68}, {-80, 90}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-90, -70}, {82, -70}}, color = {192, 192, 192}), Polygon(points = {{90, -70}, {68, -62}, {68, -78}, {90, -70}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-80, 0}, {80, 0}}, color = {0, 0, 0}), Text(extent = {{-150, -150}, {150, -110}}, lineColor = {0, 0, 0}, textString = "k=%k")}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {2, 2}), graphics = {Polygon(points = {{-80, 90}, {-86, 68}, {-74, 68}, {-80, 90}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Line(points = {{-80, 68}, {-80, -80}}, color = {95, 95, 95}), Line(points = {{-80, 0}, {80, 0}}, color = {0, 0, 255}, thickness = 0.5), Line(points = {{-90, -70}, {82, -70}}, color = {95, 95, 95}), Polygon(points = {{90, -70}, {68, -64}, {68, -76}, {90, -70}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Text(extent = {{-83, 92}, {-30, 74}}, lineColor = {0, 0, 0}, textString = "y"), Text(extent = {{70, -80}, {94, -100}}, lineColor = {0, 0, 0}, textString = "time"), Text(extent = {{-101, 8}, {-81, -12}}, lineColor = {0, 0, 0}, textString = "k")}), Documentation(info = "<html>
          <p>
          The Real output y is a constant signal:
          </p>

          <p>
          <img src=\"../Images/Blocks/Sources/Constant.png\">
          </p>
          </html>"));
      equation
        y = k;
      end Constant;

      block Step 
        parameter Real height = 1 "Height of step";
        extends .Modelica.Blocks.Interfaces.SignalSource;
         annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Line(points = {{-80, 68}, {-80, -80}}, color = {192, 192, 192}), Polygon(points = {{-80, 90}, {-88, 68}, {-72, 68}, {-80, 90}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-90, -70}, {82, -70}}, color = {192, 192, 192}), Polygon(points = {{90, -70}, {68, -62}, {68, -78}, {90, -70}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Line(points = {{-80, -70}, {0, -70}, {0, 50}, {80, 50}}, color = {0, 0, 0}), Text(extent = {{-150, -150}, {150, -110}}, lineColor = {0, 0, 0}, textString = "startTime=%startTime")}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Polygon(points = {{-80, 90}, {-86, 68}, {-74, 68}, {-80, 90}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Line(points = {{-80, 68}, {-80, -80}}, color = {95, 95, 95}), Line(points = {{-80, -18}, {0, -18}, {0, 50}, {80, 50}}, color = {0, 0, 255}, thickness = 0.5), Line(points = {{-90, -70}, {82, -70}}, color = {95, 95, 95}), Polygon(points = {{90, -70}, {68, -64}, {68, -76}, {90, -70}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Text(extent = {{70, -80}, {94, -100}}, lineColor = {0, 0, 0}, textString = "time"), Text(extent = {{-21, -72}, {25, -90}}, lineColor = {0, 0, 0}, textString = "startTime"), Line(points = {{0, -17}, {0, -71}}, color = {95, 95, 95}), Text(extent = {{-68, -36}, {-22, -54}}, lineColor = {0, 0, 0}, textString = "offset"), Line(points = {{-13, 50}, {-13, -17}}, color = {95, 95, 95}), Polygon(points = {{2, 50}, {-19, 50}, {2, 50}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Polygon(points = {{-13, -17}, {-16, -4}, {-10, -4}, {-13, -17}, {-13, -17}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Polygon(points = {{-13, 50}, {-16, 37}, {-9, 37}, {-13, 50}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Text(extent = {{-68, 26}, {-22, 8}}, lineColor = {0, 0, 0}, textString = "height"), Polygon(points = {{-13, -69}, {-16, -56}, {-10, -56}, {-13, -69}, {-13, -69}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Line(points = {{-13, -18}, {-13, -70}}, color = {95, 95, 95}), Polygon(points = {{-13, -18}, {-16, -31}, {-9, -31}, {-13, -18}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid), Text(extent = {{-72, 100}, {-31, 80}}, lineColor = {0, 0, 0}, textString = "y")}), Documentation(info = "<html>
          <p>
          The Real output y is a step signal:
          </p>

          <p>
          <img src=\"../Images/Blocks/Sources/Step.png\">
          </p>

          </html>"));
      equation
        y = offset + (if time < startTime then 0 else height);
      end Step;
       annotation(Documentation(info = "<HTML>
        <p>
        This package contains <b>source</b> components, i.e., blocks which
        have only output signals. These blocks are used as signal generators
        for Real, Integer and Boolean signals.
        </p>

        <p>
        All Real source signals (with the exception of the Constant source)
        have at least the following two parameters:
        </p>

        <table border=1 cellspacing=0 cellpadding=2>
          <tr><td valign=\"top\"><b>offset</b></td>
              <td valign=\"top\">Value which is added to the signal</td>
          </tr>
          <tr><td valign=\"top\"><b>startTime</b></td>
              <td valign=\"top\">Start time of signal. For time &lt; startTime,
                        the output y is set to offset.</td>
          </tr>
        </table>

        <p>
        The <b>offset</b> parameter is especially useful in order to shift
        the corresponding source, such that at initial time the system
        is stationary. To determine the corresponding value of offset,
        usually requires a trimming calculation.
        </p>
        </HTML>
        ", revisions = "<html>
        <ul>
        <li><i>October 21, 2002</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
               and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
               Integer sources added. Step, TimeTable and BooleanStep slightly changed.</li>
        <li><i>Nov. 8, 1999</i>
               by <a href=\"mailto:clauss@eas.iis.fhg.de\">Christoph Clau&szlig;</a>,
               <a href=\"mailto:Andre.Schneider@eas.iis.fraunhofer.de\">Andre.Schneider@eas.iis.fraunhofer.de</a>,
               <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
               New sources: Exponentials, TimeTable. Trapezoid slightly enhanced
               (nperiod=-1 is an infinite number of periods).</li>
        <li><i>Oct. 31, 1999</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
               <a href=\"mailto:clauss@eas.iis.fhg.de\">Christoph Clau&szlig;</a>,
               <a href=\"mailto:Andre.Schneider@eas.iis.fraunhofer.de\">Andre.Schneider@eas.iis.fraunhofer.de</a>,
               All sources vectorized. New sources: ExpSine, Trapezoid,
               BooleanConstant, BooleanStep, BooleanPulse, SampleTrigger.
               Improved documentation, especially detailed description of
               signals in diagram layer.</li>
        <li><i>June 29, 1999</i>
               by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
               Realized a first version, based on an existing Dymola library
               of Dieter Moormann and Hilding Elmqvist.</li>
        </ul>
        </html>"));
    end Sources;

    package Tables 
      extends Modelica.Icons.Library;

      model CombiTable1Ds 
        parameter Boolean tableOnFile = false "true, if table is defined on file or in function usertab" annotation(Dialog(group = "table data definition"));
        parameter Real[:, :] table = fill(0.0, 0, 2) "table matrix (grid = first column; e.g., table=[0,2])" annotation(Dialog(group = "table data definition", enable = not tableOnFile));
        parameter String tableName = "NoName" "table name on file or in function usertab (see docu)" annotation(Dialog(group = "table data definition", enable = tableOnFile));
        parameter String fileName = "NoName" "file where matrix is stored" annotation(Dialog(group = "table data definition", enable = tableOnFile, __Dymola_loadSelector(filter = "Text files (*.txt);;Matlab files (*.mat)", caption = "Open file in which table is present")));
        parameter Integer[:] columns = 2:size(table, 2) "columns of table to be interpolated" annotation(Dialog(group = "table data interpretation"));
        parameter Modelica.Blocks.Types.Smoothness smoothness = .Modelica.Blocks.Types.Smoothness.LinearSegments "smoothness of table interpolation" annotation(Dialog(group = "table data interpretation"));
        extends Modelica.Blocks.Interfaces.SIMO(final nout = size(columns, 1));
      protected
        Integer tableID;

        function tableInit 
          input String tableName;
          input String fileName;
          input Real[:, :] table;
          input Modelica.Blocks.Types.Smoothness smoothness;
          output Integer tableID;
          external "C" tableID = ModelicaTables_CombiTable1D_init(tableName, fileName, table, size(table, 1), size(table, 2), smoothness);
        end tableInit;

        function tableIpo 
          input Integer tableID;
          input Integer icol;
          input Real u;
          output Real value;
          external "C" value = ModelicaTables_CombiTable1D_interpolate(tableID, icol, u);
        end tableIpo;
         annotation(Documentation(info = "<html>
          <p>
          <b>Linear interpolation</b> in <b>one</b> dimension of a <b>table</b>.
          Via parameter <b>columns</b> it can be defined how many columns of the
          table are interpolated. If, e.g., icol={2,4}, it is assumed that one input
          and 2 output signals are present and that the first output interpolates
          via column 2 and the second output interpolates via column 4 of the
          table matrix.
          </p>
          <p>
          The grid points and function values are stored in a matrix \"table[i,j]\",
          where the first column \"table[:,1]\" contains the grid points and the
          other columns contain the data to be interpolated. Example:
          </p>
          <pre>
             table = [0,  0;
                      1,  1;
                      2,  4;
                      4, 16]
             If, e.g., the input u = 1.0, the output y =  1.0,
                 e.g., the input u = 1.5, the output y =  2.5,
                 e.g., the input u = 2.0, the output y =  4.0,
                 e.g., the input u =-1.0, the output y = -1.0 (i.e. extrapolation).
          </pre>
          <ul>
          <li> The interpolation is <b>efficient</b>, because a search for a new interpolation
               starts at the interval used in the last call.</li>
          <li> If the table has only <b>one row</b>, the table value is returned,
               independent of the value of the input signal.</li>
          <li> If the input signal <b>u</b> is <b>outside</b> of the defined <b>interval</b>, i.e.,
               u &gt; table[size(table,1),1] or u &lt; table[1,1], the corresponding
               value is also determined by linear
               interpolation through the last or first two points of the table.</li>
          <li> The grid values (first column) have to be <b>strict</b>
               monotonically increasing.</li>
          </ul>
          <p>
          The table matrix can be defined in the following ways:
          </p>
          <ol>
          <li> Explicitly supplied as <b>parameter matrix</b> \"table\",
               and the other parameters have the following values:
          <pre>
             tableName is \"NoName\" or has only blanks,
             fileName  is \"NoName\" or has only blanks.
          </pre></li>
          <li> <b>Read</b> from a <b>file</b> \"fileName\" where the matrix is stored as
                \"tableName\". Both ASCII and binary file format is possible.
                (the ASCII format is described below).
                It is most convenient to generate the binary file from Matlab
                (Matlab 4 storage format), e.g., by command
          <pre>
             save tables.mat tab1 tab2 tab3 -V4
          </pre>
                when the three tables tab1, tab2, tab3 should be
                used from the model.</li>
          <li>  Statically stored in function \"usertab\" in file \"usertab.c\".
                The matrix is identified by \"tableName\". Parameter
                fileName = \"NoName\" or has only blanks.</li>
          </ol>
          <p>
          Table definition methods (1) and (3) do <b>not</b> allocate dynamic memory,
          and do not access files, whereas method (2) does. Therefore (1) and (3)
          are suited for hardware-in-the-loop simulation (e.g. with dSpace hardware).
          When the constant \"NO_FILE\" is defined, all parts of the
          source code of method (2) are removed by the C-preprocessor, such that
          no dynamic memory allocation and no access to files takes place.
          </p>
          <p>
          If tables are read from an ASCII-file, the file need to have the
          following structure (\"-----\" is not part of the file content):
          </p>
          <pre>
          -----------------------------------------------------
          #1
          double tab1(5,2)   # comment line
            0   0
            1   1
            2   4
            3   9
            4  16
          double tab2(5,2)   # another comment line
            0   0
            2   2
            4   8
            6  18
            8  32
          -----------------------------------------------------
          </pre>
          <p>
          Note, that the first two characters in the file need to be
          \"#1\". Afterwards, the corresponding matrix has to be declared
          with type, name and actual dimensions. Finally, in successive
          rows of the file, the elements of the matrix have to be given.
          Several matrices may be defined one after another.
          </p>
          </HTML>
          "), Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Line(points = {{-60, 40}, {-60, -40}, {60, -40}, {60, 40}, {30, 40}, {30, -40}, {-30, -40}, {-30, 40}, {-60, 40}, {-60, 20}, {60, 20}, {60, 0}, {-60, 0}, {-60, -20}, {60, -20}, {60, -40}, {-60, -40}, {-60, 40}, {60, 40}, {60, -40}}, color = {0, 0, 0}), Line(points = {{0, 40}, {0, -40}}, color = {0, 0, 0}), Rectangle(extent = {{-60, 40}, {-30, 20}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-60, 20}, {-30, 0}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-60, 0}, {-30, -20}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-60, -20}, {-30, -40}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-60, 60}, {60, -60}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Line(points = {{-100, 0}, {-58, 0}}, color = {0, 0, 255}), Line(points = {{60, 0}, {100, 0}}, color = {0, 0, 255}), Text(extent = {{-100, 100}, {100, 64}}, textString = "1 dimensional linear table interpolation", lineColor = {0, 0, 255}), Line(points = {{-54, 40}, {-54, -40}, {54, -40}, {54, 40}, {28, 40}, {28, -40}, {-28, -40}, {-28, 40}, {-54, 40}, {-54, 20}, {54, 20}, {54, 0}, {-54, 0}, {-54, -20}, {54, -20}, {54, -40}, {-54, -40}, {-54, 40}, {54, 40}, {54, -40}}, color = {0, 0, 0}), Line(points = {{0, 40}, {0, -40}}, color = {0, 0, 0}), Rectangle(extent = {{-54, 40}, {-28, 20}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-54, 20}, {-28, 0}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-54, 0}, {-28, -20}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-54, -20}, {-28, -40}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-52, 56}, {-34, 44}}, textString = "u", lineColor = {0, 0, 255}), Text(extent = {{-22, 54}, {2, 42}}, textString = "y[1]", lineColor = {0, 0, 255}), Text(extent = {{4, 54}, {28, 42}}, textString = "y[2]", lineColor = {0, 0, 255}), Text(extent = {{0, -40}, {32, -54}}, textString = "columns", lineColor = {0, 0, 255})}));
      equation
        if tableOnFile then
          assert(tableName <> "NoName", "tableOnFile = true and no table name given");
        end if;
        if not tableOnFile then
          assert(size(table, 1) > 0 and size(table, 2) > 0, "tableOnFile = false and parameter table is an empty matrix");
        end if;
        for i in 1:nout loop
          y[i] = if not tableOnFile and size(table, 1) == 1 then table[1, columns[i]] else tableIpo(tableID, columns[i], u);
        end for;
        when initial() then
          tableID = tableInit(if tableOnFile then tableName else "NoName", if tableOnFile then fileName else "NoName", table, smoothness);
        end when;
      end CombiTable1Ds;
       annotation(Documentation(info = "<html>
        <p>
        This package contains blocks for one- and two-dimensional
        interpolation in tables.
        </p>
        </html>"));
    end Tables;

    package Types 
      extends Modelica.Icons.Library;
      type Smoothness = enumeration(LinearSegments "Table points are linearly interpolated", ContinuousDerivative "Table points are interpolated such that the first derivative is continuous") "Enumeration defining the smoothness of table interpolation" annotation(Documentation(info = "<html>

        </html>"));
      type Init = enumeration(NoInit "No initialization (start values are used as guess values with fixed=false)", SteadyState "Steady state initialization (derivatives of states are zero)", InitialState "Initialization with initial states", InitialOutput "Initialization with initial outputs (and steady state of the states if possibles)") "Enumeration defining initialization of a block" annotation(Evaluate = true, Documentation(info = "<html>

        </html>"));
       annotation(Documentation(info = "<HTML>
        <p>
        In this package <b>types</b> and <b>constants</b> are defined that are used
        in library Modelica.Blocks. The types have additional annotation choices
        definitions that define the menus to be built up in the graphical
        user interface when the type is used as parameter in a declaration.
        </p>
        </HTML>"));
    end Types;
     annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-32, -6}, {16, -35}}, lineColor = {0, 0, 0}), Rectangle(extent = {{-32, -56}, {16, -85}}, lineColor = {0, 0, 0}), Line(points = {{16, -20}, {49, -20}, {49, -71}, {16, -71}}, color = {0, 0, 0}), Line(points = {{-32, -72}, {-64, -72}, {-64, -21}, {-32, -21}}, color = {0, 0, 0}), Polygon(points = {{16, -71}, {29, -67}, {29, -74}, {16, -71}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Polygon(points = {{-32, -21}, {-46, -17}, {-46, -25}, {-32, -21}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid)}), Documentation(info = "<html>
      <p>
      This library contains input/output blocks to build up block diagrams.
      </p>

      <dl>
      <dt><b>Main Author:</b>
      <dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
          Deutsches Zentrum f&uuml;r Luft und Raumfahrt e. V. (DLR)<br>
          Oberpfaffenhofen<br>
          Postfach 1116<br>
          D-82230 Wessling<br>
          email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
      </dl>
      <p>
      Copyright &copy; 1998-2009, Modelica Association and DLR.
      </p>
      <p>
      <i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
      under the terms of the <b>Modelica license</b>, see the license conditions
      and the accompanying <b>disclaimer</b>
      <a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense2\">here</a>.</i>
      </p>
      </HTML>
      ", revisions = "<html>
      <ul>
      <li><i>June 23, 2004</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Introduced new block connectors and adapated all blocks to the new connectors.
             Included subpackages Continuous, Discrete, Logical, Nonlinear from
             package ModelicaAdditions.Blocks.
             Included subpackage ModelicaAdditions.Table in Modelica.Blocks.Sources
             and in the new package Modelica.Blocks.Tables.
             Added new blocks to Blocks.Sources and Blocks.Logical.
             </li>
      <li><i>October 21, 2002</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
             and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
             New subpackage Examples, additional components.
             </li>
      <li><i>June 20, 2000</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a> and
             Michael Tiller:<br>
             Introduced a replaceable signal type into
             Blocks.Interfaces.RealInput/RealOutput:
      <pre>
         replaceable type SignalType = Real
      </pre>
             in order that the type of the signal of an input/output block
             can be changed to a physical type, for example:
      <pre>
         Sine sin1(outPort(redeclare type SignalType=Modelica.SIunits.Torque))
      </pre>
            </li>
      <li><i>Sept. 18, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Renamed to Blocks. New subpackages Math, Nonlinear.
             Additional components in subpackages Interfaces, Continuous
             and Sources. </li>
      <li><i>June 30, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Realized a first version, based on an existing Dymola library
             of Dieter Moormann and Hilding Elmqvist.</li>
      </ul>
      </html>"));
  end Blocks;

  package Math 
    extends Modelica.Icons.Library2;

    function asin 
      extends baseIcon2;
      input Real u;
      output .Modelica.SIunits.Angle y;
      external "C" y = asin(u);
    end asin;

    function exp 
      extends baseIcon2;
      input Real u;
      output Real y;
      external "C" y = exp(u);
    end exp;

    partial function baseIcon2 
       annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid), Line(points = {{0, -80}, {0, 68}}, color = {192, 192, 192}), Polygon(points = {{0, 90}, {-8, 68}, {8, 68}, {0, 90}}, lineColor = {192, 192, 192}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Text(extent = {{-150, 150}, {150, 110}}, textString = "%name", lineColor = {0, 0, 255})}), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}), graphics = {Line(points = {{0, 80}, {-8, 80}}, color = {95, 95, 95}), Line(points = {{0, -80}, {-8, -80}}, color = {95, 95, 95}), Line(points = {{0, -90}, {0, 84}}, color = {95, 95, 95}), Text(extent = {{5, 104}, {25, 84}}, lineColor = {95, 95, 95}, textString = "y"), Polygon(points = {{0, 98}, {-6, 82}, {6, 82}, {0, 98}}, lineColor = {95, 95, 95}, fillColor = {95, 95, 95}, fillPattern = FillPattern.Solid)}), Documentation(revisions = "<html>
        <p>
        Icon for a mathematical function, consisting of an y-axis in the middle.
        It is expected, that an x-axis is added and a plot of the function.
        </p>
        </html>"));
    end baseIcon2;
     annotation(Invisible = true, Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Text(extent = {{-59, -9}, {42, -56}}, lineColor = {0, 0, 0}, textString = "f(x)")}), Documentation(info = "<HTML>
      <p>
      This package contains <b>basic mathematical functions</b> (such as sin(..)),
      as well as functions operating on <b>vectors</b> and <b>matrices</b>.
      </p>

      <dl>
      <dt><b>Main Author:</b>
      <dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
          Deutsches Zentrum f&uuml;r Luft und Raumfahrt e.V. (DLR)<br>
          Institut f&uuml;r Robotik und Mechatronik<br>
          Postfach 1116<br>
          D-82230 Wessling<br>
          Germany<br>
          email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
      </dl>

      <p>
      Copyright &copy; 1998-2009, Modelica Association and DLR.
      </p>
      <p>
      <i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
      under the terms of the <b>Modelica license</b>, see the license conditions
      and the accompanying <b>disclaimer</b>
      <a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense2\">here</a>.</i>
      </p><br>
      </HTML>
      ", revisions = "<html>
      <ul>
      <li><i>October 21, 2002</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
             and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
             Function tempInterpol2 added.</li>
      <li><i>Oct. 24, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Icons for icon and diagram level introduced.</li>
      <li><i>June 30, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Realized.</li>
      </ul>

      </html>"));
  end Math;

  package Constants 
    extends Modelica.Icons.Library2;
    final constant Real pi = 2 * Modelica.Math.asin(1.0);
    final constant Real small = 1e-60 "Smallest number such that small and -small are representable on the machine";
    final constant .Modelica.SIunits.Velocity c = 299792458 "Speed of light in vacuum";
    final constant Real mue_0(final unit = "N/A2") = 4 * pi * 1e-07 "Magnetic constant";
     annotation(Documentation(info = "<html>
      <p>
      This package provides often needed constants from mathematics, machine
      dependent constants and constants from nature. The latter constants
      (name, value, description) are from the following source:
      </p>

      <dl>
      <dt>Peter J. Mohr and Barry N. Taylor (1999):</dt>
      <dd><b>CODATA Recommended Values of the Fundamental Physical Constants: 1998</b>.
          Journal of Physical and Chemical Reference Data, Vol. 28, No. 6, 1999 and
          Reviews of Modern Physics, Vol. 72, No. 2, 2000. See also <a href=
      \"http://physics.nist.gov/cuu/Constants/\">http://physics.nist.gov/cuu/Constants/</a></dd>
      </dl>

      <p>CODATA is the Committee on Data for Science and Technology.</p>

      <dl>
      <dt><b>Main Author:</b></dt>
      <dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
          Deutsches Zentrum f&uuml;r Luft und Raumfahrt e. V. (DLR)<br>
          Oberpfaffenhofen<br>
          Postfach 11 16<br>
          D-82230 We&szlig;ling<br>
          email: <a href=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</a></dd>
      </dl>


      <p>
      Copyright &copy; 1998-2009, Modelica Association and DLR.
      </p>
      <p>
      <i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
      under the terms of the <b>Modelica license</b>, see the license conditions
      and the accompanying <b>disclaimer</b>
      <a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense2\">here</a>.</i>
      </p><br>
      </html>
      ", revisions = "<html>
      <ul>
      <li><i>Nov 8, 2004</i>
             by <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
             Constants updated according to 2002 CODATA values.</li>
      <li><i>Dec 9, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Constants updated according to 1998 CODATA values. Using names, values
             and description text from this source. Included magnetic and
             electric constant.</li>
      <li><i>Sep 18, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Constants eps, inf, small introduced.</li>
      <li><i>Nov 15, 1997</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Realized.</li>
      </ul>
      </html>"), Invisible = true, Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Line(points = {{-34, -38}, {12, -38}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-20, -38}, {-24, -48}, {-28, -56}, {-34, -64}}, color = {0, 0, 0}, thickness = 0.5), Line(points = {{-2, -38}, {2, -46}, {8, -56}, {14, -64}}, color = {0, 0, 0}, thickness = 0.5)}), Diagram(coordinateSystem(preserveAspectRatio = false, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{200, 162}, {380, 312}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{200, 312}, {220, 332}, {400, 332}, {380, 312}, {200, 312}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{400, 332}, {400, 182}, {380, 162}, {380, 312}, {400, 332}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Text(extent = {{210, 302}, {370, 272}}, lineColor = {160, 160, 164}, textString = "Library"), Line(points = {{266, 224}, {312, 224}}, color = {0, 0, 0}, thickness = 1), Line(points = {{280, 224}, {276, 214}, {272, 206}, {266, 198}}, color = {0, 0, 0}, thickness = 1), Line(points = {{298, 224}, {302, 216}, {308, 206}, {314, 198}}, color = {0, 0, 0}, thickness = 1), Text(extent = {{152, 412}, {458, 334}}, lineColor = {255, 0, 0}, textString = "Modelica.Constants")}));
  end Constants;

  package Icons 
    partial package Library 
       annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Rectangle(extent = {{-100, -100}, {80, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{-100, 50}, {-80, 70}, {100, 70}, {80, 50}, {-100, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{100, 70}, {100, -80}, {80, -100}, {80, 50}, {100, 70}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Text(extent = {{-85, 35}, {65, -85}}, lineColor = {0, 0, 255}, textString = "Library"), Text(extent = {{-120, 122}, {120, 73}}, lineColor = {255, 0, 0}, textString = "%name")}), Documentation(info = "<html>
        <p>
        This icon is designed for a <b>library</b>.
        </p>
        </html>"));
    end Library;

    partial package Library2 
       annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Rectangle(extent = {{-100, -100}, {80, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{-100, 50}, {-80, 70}, {100, 70}, {80, 50}, {-100, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{100, 70}, {100, -80}, {80, -100}, {80, 50}, {100, 70}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Text(extent = {{-120, 125}, {120, 70}}, lineColor = {255, 0, 0}, textString = "%name"), Text(extent = {{-90, 40}, {70, 10}}, lineColor = {160, 160, 164}, textString = "Library")}), Documentation(info = "<html>
        <p>
        This icon is designed for a <b>package</b> where a package
        specific graphic is additionally included in the icon.
        </p>
        </html>"));
    end Library2;

    partial model Example 
       annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}, grid = {1, 1}), graphics = {Rectangle(extent = {{-100, -100}, {80, 50}}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{-100, 50}, {-80, 70}, {100, 70}, {80, 50}, {-100, 50}}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{100, 70}, {100, -80}, {80, -100}, {80, 50}, {100, 70}}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Text(extent = {{-96, 3}, {77, -49}}, lineColor = {0, 0, 255}, textString = "Example"), Text(extent = {{-120, 132}, {120, 73}}, lineColor = {255, 0, 0}, textString = "%name")}), Documentation(info = "<html>
        <p>
        This icon is designed for an <b>Example package</b>,
        i.e. a package containing executable demo models.
        </p>
        </html>"));
    end Example;
     annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, -100}, {80, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{-100, 50}, {-80, 70}, {100, 70}, {80, 50}, {-100, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{100, 70}, {100, -80}, {80, -100}, {80, 50}, {100, 70}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Text(extent = {{-120, 135}, {120, 70}}, lineColor = {255, 0, 0}, textString = "%name"), Text(extent = {{-90, 40}, {70, 10}}, lineColor = {160, 160, 164}, textString = "Library"), Rectangle(extent = {{-100, -100}, {80, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{-100, 50}, {-80, 70}, {100, 70}, {80, 50}, {-100, 50}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{100, 70}, {100, -80}, {80, -100}, {80, 50}, {100, 70}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Text(extent = {{-90, 40}, {70, 10}}, lineColor = {160, 160, 164}, textString = "Library"), Polygon(points = {{-64, -20}, {-50, -4}, {50, -4}, {36, -20}, {-64, -20}, {-64, -20}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-64, -20}, {36, -84}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid), Text(extent = {{-60, -24}, {32, -38}}, lineColor = {128, 128, 128}, textString = "Library"), Polygon(points = {{50, -4}, {50, -70}, {36, -84}, {36, -20}, {50, -4}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid)}), Documentation(info = "<html>
      <p>
      This package contains definitions for the graphical layout of
      components which may be used in different libraries.
      The icons can be utilized by inheriting them in the desired class
      using \"extends\" or by directly copying the \"icon\" layer.
      </p>

      <dl>
      <dt><b>Main Author:</b>
      <dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
          Deutsches Zentrum fuer Luft und Raumfahrt e.V. (DLR)<br>
          Oberpfaffenhofen<br>
          Postfach 1116<br>
          D-82230 Wessling<br>
          email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
      </dl>

      <p>
      Copyright &copy; 1998-2009, Modelica Association and DLR.
      </p>
      <p>
      <i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
      under the terms of the <b>Modelica license</b>, see the license conditions
      and the accompanying <b>disclaimer</b>
      <a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense2\">here</a>.</i>
      </p><br>
      </HTML>
      ", revisions = "<html>
      <ul>
      <li><i>October 21, 2002</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
             and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
             Added new icons <b>Function</b>, <b>Enumerations</b> and <b>Record</b>.</li>
      <li><i>June 6, 2000</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Replaced <b>model</b> keyword by <b>package</b> if the main
             usage is for inheriting from a package.<br>
             New icons <b>GearIcon</b> and <b>MotorIcon</b>.</li>
      <li><i>Sept. 18, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Renaming package Icon to Icons.
             Model Advanced removed (icon not accepted on the Modelica meeting).
             New model Library2, which is the Library icon with enough place
             to add library specific elements in the icon. Icon also used in diagram
             level for models Info, TranslationalSensor, RotationalSensor.</li>
      <li><i>July 15, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Model Caution renamed to Advanced, model Sensor renamed to
             TranslationalSensor, new model RotationalSensor.</li>
      <li><i>June 30, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Realized a first version.</li>
      </ul>
      <br>
      </html>"));
  end Icons;

  package SIunits 
    extends Modelica.Icons.Library2;

    package Conversions 
      extends Modelica.Icons.Library2;

      package NonSIunits 
        extends Modelica.Icons.Library2;
        type Temperature_degC = Real(final quantity = "ThermodynamicTemperature", final unit = "degC") "Absolute temperature in degree Celsius (for relative temperature use SIunits.TemperatureDifference)" annotation(__Dymola_absoluteValue = true);
         annotation(Documentation(info = "<HTML>
          <p>
          This package provides predefined types, such as <b>Angle_deg</b> (angle in
          degree), <b>AngularVelocity_rpm</b> (angular velocity in revolutions per
          minute) or <b>Temperature_degF</b> (temperature in degree Fahrenheit),
          which are in common use but are not part of the international standard on
          units according to ISO 31-1992 \"General principles concerning quantities,
          units and symbols\" and ISO 1000-1992 \"SI units and recommendations for
          the use of their multiples and of certain other units\".</p>
          <p>If possible, the types in this package should not be used. Use instead
          types of package Modelica.SIunits. For more information on units, see also
          the book of Francois Cardarelli <b>Scientific Unit Conversion - A
          Practical Guide to Metrication</b> (Springer 1997).</p>
          <p>Some units, such as <b>Temperature_degC/Temp_C</b> are both defined in
          Modelica.SIunits and in Modelica.Conversions.NonSIunits. The reason is that these
          definitions have been placed erroneously in Modelica.SIunits although they
          are not SIunits. For backward compatibility, these type definitions are
          still kept in Modelica.SIunits.</p>
          </HTML>
          "), Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Text(extent = {{-66, -13}, {52, -67}}, lineColor = {0, 0, 0}, textString = "[km/h]")}));
      end NonSIunits;
       annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Text(extent = {{-33, -7}, {-92, -67}}, lineColor = {0, 0, 0}, lineThickness = 1, textString = "°C"), Text(extent = {{82, -7}, {22, -67}}, lineColor = {0, 0, 0}, textString = "K"), Line(points = {{-26, -36}, {6, -36}}, color = {0, 0, 0}), Polygon(points = {{6, -28}, {6, -45}, {26, -37}, {6, -28}}, pattern = LinePattern.None, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255})}), Documentation(info = "<HTML>
        <p>This package provides conversion functions from the non SI Units
        defined in package Modelica.SIunits.Conversions.NonSIunits to the
        corresponding SI Units defined in package Modelica.SIunits and vice
        versa. It is recommended to use these functions in the following
        way (note, that all functions have one Real input and one Real output
        argument):</p>
        <pre>
          <b>import</b> SI = Modelica.SIunits;
          <b>import</b> Modelica.SIunits.Conversions.*;
             ...
          <b>parameter</b> SI.Temperature     T   = from_degC(25);   // convert 25 degree Celsius to Kelvin
          <b>parameter</b> SI.Angle           phi = from_deg(180);   // convert 180 degree to radian
          <b>parameter</b> SI.AngularVelocity w   = from_rpm(3600);  // convert 3600 revolutions per minutes
                                                              // to radian per seconds
        </pre>

        </HTML>
        "));
    end Conversions;

    type Angle = Real(final quantity = "Angle", final unit = "rad", displayUnit = "deg");
    type Length = Real(final quantity = "Length", final unit = "m");
    type Area = Real(final quantity = "Area", final unit = "m2");
    type Volume = Real(final quantity = "Volume", final unit = "m3");
    type Time = Real(final quantity = "Time", final unit = "s");
    type Velocity = Real(final quantity = "Velocity", final unit = "m/s");
    type Acceleration = Real(final quantity = "Acceleration", final unit = "m/s2");
    type FaradayConstant = Real(final quantity = "FaradayConstant", final unit = "C/mol");
     annotation(Invisible = true, Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Text(extent = {{-63, -13}, {45, -67}}, lineColor = {0, 0, 0}, textString = "[kg.m2]")}), Documentation(info = "<html>
      <p>This package provides predefined types, such as <i>Mass</i>,
      <i>Angle</i>, <i>Time</i>, based on the international standard
      on units, e.g.,
      </p>

      <pre>   <b>type</b> Angle = Real(<b>final</b> quantity = \"Angle\",
                           <b>final</b> unit     = \"rad\",
                           displayUnit    = \"deg\");
      </pre>

      <p>
      as well as conversion functions from non SI-units to SI-units
      and vice versa in subpackage
      <a href=\"Modelica://Modelica.SIunits.Conversions\">Conversions</a>.
      </p>

      <p>
      For an introduction how units are used in the Modelica standard library
      with package SIunits, have a look at:
      <a href=\"Modelica://Modelica.SIunits.UsersGuide.HowToUseSIunits\">How to use SIunits</a>.
      </p>

      <p>
      Copyright &copy; 1998-2009, Modelica Association and DLR.
      </p>
      <p>
      <i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
      under the terms of the <b>Modelica license</b>, see the license conditions
      and the accompanying <b>disclaimer</b>
      <a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense2\">here</a>.</i>
      </p>

      </html>", revisions = "<html>
      <ul>
      <li><i>Dec. 14, 2005</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Add User's Guide and removed \"min\" values for Resistance and Conductance.</li>
      <li><i>October 21, 2002</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
             and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
             Added new package <b>Conversions</b>. Corrected typo <i>Wavelenght</i>.</li>
      <li><i>June 6, 2000</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Introduced the following new types<br>
             type Temperature = ThermodynamicTemperature;<br>
             types DerDensityByEnthalpy, DerDensityByPressure,
             DerDensityByTemperature, DerEnthalpyByPressure,
             DerEnergyByDensity, DerEnergyByPressure<br>
             Attribute \"final\" removed from min and max values
             in order that these values can still be changed to narrow
             the allowed range of values.<br>
             Quantity=\"Stress\" removed from type \"Stress\", in order
             that a type \"Stress\" can be connected to a type \"Pressure\".</li>
      <li><i>Oct. 27, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             New types due to electrical library: Transconductance, InversePotential,
             Damping.</li>
      <li><i>Sept. 18, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Renamed from SIunit to SIunits. Subpackages expanded, i.e., the
             SIunits package, does no longer contain subpackages.</li>
      <li><i>Aug 12, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Type \"Pressure\" renamed to \"AbsolutePressure\" and introduced a new
             type \"Pressure\" which does not contain a minimum of zero in order
             to allow convenient handling of relative pressure. Redefined
             BulkModulus as an alias to AbsolutePressure instead of Stress, since
             needed in hydraulics.</li>
      <li><i>June 29, 1999</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
             Bug-fix: Double definition of \"Compressibility\" removed
             and appropriate \"extends Heat\" clause introduced in
             package SolidStatePhysics to incorporate ThermodynamicTemperature.</li>
      <li><i>April 8, 1998</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
             and Astrid Jaschinski:<br>
             Complete ISO 31 chapters realized.</li>
      <li><i>Nov. 15, 1997</i>
             by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
             and <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>:<br>
             Some chapters realized.</li>
      </ul>
      </html>"), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{169, 86}, {349, 236}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{169, 236}, {189, 256}, {369, 256}, {349, 236}, {169, 236}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Polygon(points = {{369, 256}, {369, 106}, {349, 86}, {349, 236}, {369, 256}}, fillColor = {235, 235, 235}, fillPattern = FillPattern.Solid, lineColor = {0, 0, 255}), Text(extent = {{179, 226}, {339, 196}}, lineColor = {160, 160, 164}, textString = "Library"), Text(extent = {{206, 173}, {314, 119}}, lineColor = {0, 0, 0}, textString = "[kg.m2]"), Text(extent = {{163, 320}, {406, 264}}, lineColor = {255, 0, 0}, textString = "Modelica.SIunits")}));
  end SIunits;
   annotation(preferredView = "info", version = "3.1", versionBuild = 9, versionDate = "2009-08-14", uses(ModelicaServices(version = "1.0")), dateModified = "2012-03-06 21:00:00Z", revisionId = "$Id:: package.mo 4972 2012-03-06 19:49:17Z #$", conversion(noneFromVersion = "3.0.1", noneFromVersion = "3.0", from(version = "2.1", script = "Scripts/ConvertModelica_from_2.2.2_to_3.0.mos"), from(version = "2.2", script = "Scripts/ConvertModelica_from_2.2.2_to_3.0.mos"), from(version = "2.2.1", script = "Scripts/ConvertModelica_from_2.2.2_to_3.0.mos"), from(version = "2.2.2", script = "Scripts/ConvertModelica_from_2.2.2_to_3.0.mos")), __Dymola_classOrder = {"UsersGuide", "Blocks", "StateGraph", "Electrical", "Magnetic", "Mechanics", "Fluid", "Media", "Thermal", "Math", "Utilities", "Constants", "Icons", "SIunits"}, Settings(NewStateSelection = true), Documentation(info = "<HTML>
    <p>
    Package <b>Modelica</b> is a <b>standardized</b> and <b>free</b> package
    that is developed together with the Modelica language from the
    Modelica Association, see
    <a href=\"http://www.Modelica.org\">http://www.Modelica.org</a>.
    It is also called <b>Modelica Standard Library</b>.
    It provides model components in many domains that are based on
    standardized interface definitions. Some typical examples are shown
    in the next figure:
    </p>

    <p>
    <img src=\"../Images/UsersGuide/ModelicaLibraries.png\">
    </p>

    <p>
    For an introduction, have especially a look at:
    </p>
    <ul>
    <li> <a href=\"Modelica://Modelica.UsersGuide.Overview\">Overview</a>
      provides an overview of the Modelica Standard Library
      inside the <a href=\"Modelica://Modelica.UsersGuide\">User's Guide</a>.</li>
    <li><a href=\"Modelica://Modelica.UsersGuide.ReleaseNotes\">Release Notes</a>
     summarizes the changes of new versions of this package.</li>
    <li> <a href=\"Modelica://Modelica.UsersGuide.Contact\">Contact</a>
      lists the contributors of the Modelica Standard Library.</li>
    <li> The <b>Examples</b> packages in the various libraries, demonstrate
      how to use the components of the corresponding sublibrary.</li>
    </ul>

    <p>
    This version of the Modelica Standard Library consists of
    </p>
    <ul>
    <li> <b>922</b> models and blocks, and</li>
    <li> <b>615</b> functions
    </ul>
    <p>
    that are directly usable (= number of public, non-partial classes).
    </p>


    <p>
    <b>Licensed by the Modelica Association under the Modelica License 2</b><br>
    Copyright &copy; 1998-2009, ABB, arsenal research, T.&nbsp;B&ouml;drich, DLR, Dynasim, Fraunhofer, Modelon,
    TU Hamburg-Harburg, Politecnico di Milano.
    </p>

    <p>
    <i>This Modelica package is <u>free</u> software and
    the use is completely at <u>your own risk</u>;
    it can be redistributed and/or modified under the terms of the
    Modelica license 2, see the license conditions (including the
    disclaimer of warranty)
    <a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense2\">here</a></u>
    or at
    <a href=\"http://www.Modelica.org/licenses/ModelicaLicense2\">
    http://www.Modelica.org/licenses/ModelicaLicense2</a>.
    </p>

    </HTML>
    "));
end Modelica;
