package HelmholtzMedia
  "Data and models of real pure fluids (liquid, two-phase and gas)"
  extends Modelica.Icons.MaterialPropertiesPackage;

  package Interfaces

    partial package PartialHelmholtzMedium
      extends HelmholtzMedia.Interfaces.Types;

      extends HelmholtzMedia.Interfaces.Choices;

      extends Modelica.Media.Interfaces.PartialTwoPhaseMedium(
        onePhase=false,
        singleState=false,
        smoothModel=true,
        DipoleMoment(min=0, max=5),
        AbsolutePressure(min=Modelica.Constants.small, max=1e12),
        SpecificEntropy(min=-Modelica.Constants.inf, max=Modelica.Constants.inf),
        ThermoStates = Choices.IndependentVariables.ph);

      package EoS

        record HelmholtzDerivs
        "dimensionless Helmholtz energy and its derivatives"
          extends Modelica.Icons.Record;

          MolarMass MM = fluidConstants[1].molarMass;
          SpecificHeatCapacity R=Modelica.Constants.R/MM
          "specific gas constant";
          Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          Temperature T_crit=fluidConstants[1].criticalTemperature;

          Density d(min=0);
          Real delta(unit="1")=d/d_crit "reduced density";
          Temperature T;
          Real tau(unit="1")=T_crit/T "inverse reduced temperature";

          Real i(unit="1") "f_i";
        //Real id(unit="1") "(df_i/ddelta)@tau=const";
        //Real idd(unit="1") "(d2f_i/ddelta2)@tau=const";
          Real it(unit="1") "(df_i/dtau)@delta=const";
          Real itt(unit="1") "(d2f_i/dtau2)@delta=const";
        //Real itd(unit="1") "(d2f_i/dtau ddelta)";
          Real ittt(unit="1") "(d3f_i/dtau3)@delta=const";

          Real r(unit="1") "f_r";
          Real rd(unit="1") "(df_r/ddelta)@tau=const";
          Real rdd(unit="1") "(d2f_r/ddelta2)@tau=const";
          Real rt(unit="1") "(df_r/dtau)@delta=const";
          Real rtt(unit="1") "(d2f_r/dtau2)@delta=const";
          Real rtd(unit="1") "(d2f_r/dtau ddelta)";

          Real rttt(unit="1") "(d3f_r/dtau3)@delta=const";
          Real rttd(unit="1") "(d3f_r/dtau2 ddelta)";
          Real rtdd(unit="1") "(d3f_r/dtau ddelta2)";
          Real rddd(unit="1") "(d3f_r/ddelta3)@tau=const";
        end HelmholtzDerivs;

        record HelmholtzCoefficients
        "Coefficients for Helmholtz energy equations of state"

          //ideal gas part: substance specific coefficients
          constant Real[:,2] idealLog = fill(0.0, 0, 2);
          constant Real[:,2] idealPower = fill(0.0, 0, 2);
          constant Real[:,2] idealEinstein = fill(0.0, 0, 2);
          constant Real[:,2] idealCosh = fill(0.0, 0, 2);
          constant Real[:,2] idealSinh = fill(0.0, 0, 2);

          //residual part: substance specific coefficients
          constant Real[:,4] residualPoly = fill(0.0, 0, 4);
          constant Real[:,4] residualBwr = fill(0.0, 0, 4);
          constant Real[:,9] residualGauss = fill(0.0, 0, 9);
          constant Real[:,12] residualNonAnalytical = fill(0.0, 0, 12);

          Boolean useLineSearch=false;
        end HelmholtzCoefficients;

        function setHelmholtzDerivsZero

          input Density d;
          input Temperature T;
          input FixedPhase phase=1;
          output HelmholtzDerivs f;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant Real delta(unit="1")=d/d_crit "reduced density";
          constant Real tau(unit="1")=T_crit/T "inverse reduced temperature";

        algorithm
          f.d := d;
          f.T := T;
          f.delta := delta;
          f.tau := tau;

          if (phase==1) then
            f.i   := f_i(tau=tau, delta=delta);

            f.r   := f_r(tau=tau, delta=delta);
          else
            assert(false, "This function will return valid values for single phase input only!");
          end if;

        end setHelmholtzDerivsZero;

        function setHelmholtzDerivsFirst

          input Density d;
          input Temperature T;
          input FixedPhase phase=1;
          output HelmholtzDerivs f;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant Real delta(unit="1")=d/d_crit "reduced density";
          constant Real tau(unit="1")=T_crit/T "inverse reduced temperature";

        algorithm
          f.d := d;
          f.T := T;
          f.delta := delta;
          f.tau := tau;

          if (phase==1) then
            f.i   := f_i(tau=tau, delta=delta);
            f.it  := f_it(tau=tau, delta=delta);

            f.r   := f_r(tau=tau, delta=delta);
            f.rt  := f_rt(tau=tau, delta=delta);
            f.rd  := f_rd(tau=tau, delta=delta);
          else
            assert(false, "This function will return valid values for single phase input only!");
          end if;

        end setHelmholtzDerivsFirst;

        function setHelmholtzDerivsSecond

          input Density d;
          input Temperature T;
          input FixedPhase phase=1;
          output HelmholtzDerivs f;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant Real delta(unit="1")=d/d_crit "reduced density";
          constant Real tau(unit="1")=T_crit/T "inverse reduced temperature";

        algorithm
          f.d := d;
          f.T := T;
          f.delta := delta;
          f.tau := tau;

          if (phase==1) then
            f.i   := f_i(tau=tau, delta=delta);
            f.it  := f_it(tau=tau, delta=delta);
            f.itt := f_itt(tau=tau, delta=delta);

            f.r   := f_r(tau=tau, delta=delta);
            f.rt  := f_rt(tau=tau, delta=delta);
            f.rtt := f_rtt(tau=tau, delta=delta);
            f.rtd := f_rtd(tau=tau, delta=delta);
            f.rd  := f_rd(tau=tau, delta=delta);
            f.rdd := f_rdd(tau=tau, delta=delta);
          else
            assert(false, "This function will return valid values for single phase input only!");
          end if;

        end setHelmholtzDerivsSecond;

        function setHelmholtzDerivsThird

          input Density d;
          input Temperature T;
          input FixedPhase phase=1;
          output HelmholtzDerivs f;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant Real delta(unit="1")=d/d_crit "reduced density";
          constant Real tau(unit="1")=T_crit/T "inverse reduced temperature";

        algorithm
          f.d := d;
          f.T := T;
          f.delta := delta;
          f.tau := tau;

          if (phase==1) then
            f.i    := f_i(tau=tau, delta=delta);
            f.it   := f_it(tau=tau, delta=delta);
            f.itt  := f_itt(tau=tau, delta=delta);
            f.ittt  := f_ittt(tau=tau, delta=delta);

            f.r    := f_r(tau=tau, delta=delta);
            f.rt   := f_rt(tau=tau, delta=delta);
            f.rtt  := f_rtt(tau=tau, delta=delta);
            f.rttt  := f_rttt(tau=tau, delta=delta);
            f.rtd  := f_rtd(tau=tau, delta=delta);
            f.rttd := f_rttd(tau=tau, delta=delta);
            f.rtdd := f_rtdd(tau=tau, delta=delta);
            f.rd   := f_rd(tau=tau, delta=delta);
            f.rdd  := f_rdd(tau=tau, delta=delta);
            f.rddd := f_rddd(tau=tau, delta=delta);
          else
            assert(false, "This function will return valid values for single phase input only!");
          end if;

        end setHelmholtzDerivsThird;

        function f_i "ideal part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_ideal "ideal part of dimensionless Helmholtz energy";

      protected
          final constant Integer nLog=size(helmholtzCoefficients.idealLog, 1);
          final constant Integer nPower=size(helmholtzCoefficients.idealPower, 1);
          final constant Integer nEinstein=size(helmholtzCoefficients.idealEinstein, 1);
          final constant Integer nCosh=size(helmholtzCoefficients.idealCosh, 1);
          final constant Integer nSinh=size(helmholtzCoefficients.idealSinh, 1);

          final constant Real[nLog, 2] l=helmholtzCoefficients.idealLog;
          final constant Real[nPower, 2] p=helmholtzCoefficients.idealPower;
          final constant Real[nEinstein, 2] e=helmholtzCoefficients.idealEinstein;
          final constant Real[nCosh, 2] c=helmholtzCoefficients.idealCosh;
          final constant Real[nSinh, 2] s=helmholtzCoefficients.idealSinh;

        algorithm
          if (delta>0) and (tau>0) then
            f_ideal :=
              log(delta)
              + sum(l[i,1]*log(tau^l[i,2]) for i in 1:nLog)
              + sum(p[i,1]*tau^p[i,2] for i in 1:nPower)
              + sum(e[i,1]*log(1 - exp(e[i,2]*tau)) for i in 1:nEinstein)
              - sum(c[i,1]*log(abs(cosh(c[i,2]*tau))) for i in 1:nCosh)
              + sum(s[i,1]*log(abs(sinh(s[i,2]*tau))) for i in 1:nSinh);
          else
            f_ideal := -Modelica.Constants.inf;
          end if;
        end f_i;

        function f_it "ideal part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_ideal_tau
          "ideal part of dimensionless Helmholtz energy";

      protected
          final constant Integer nLog=size(helmholtzCoefficients.idealLog, 1);
          final constant Integer nPower=size(helmholtzCoefficients.idealPower, 1);
          final constant Integer nEinstein=size(helmholtzCoefficients.idealEinstein, 1);
          final constant Integer nCosh=size(helmholtzCoefficients.idealCosh, 1);
          final constant Integer nSinh=size(helmholtzCoefficients.idealSinh, 1);

          final constant Real[nLog, 2] l=helmholtzCoefficients.idealLog;
          final constant Real[nPower, 2] p=helmholtzCoefficients.idealPower;
          final constant Real[nEinstein, 2] e=helmholtzCoefficients.idealEinstein;
          final constant Real[nCosh, 2] c=helmholtzCoefficients.idealCosh;
          final constant Real[nSinh, 2] s=helmholtzCoefficients.idealSinh;

        algorithm
          f_ideal_tau :=
              sum((l[i,1]*l[i,2])/tau for i in 1:nLog)
            + sum(p[i,1]*tau^(p[i,2]-1)*p[i,2] for i in 1:nPower)
            + sum(e[i,1]*(-e[i,2])*((1 - exp(e[i,2]*tau))^(-1) - 1) for i in 1:nEinstein)
            - sum(c[i,1]*c[i,2]*tanh(c[i,2]*tau) for i in 1:nCosh)
            + sum(s[i,1]*s[i,2]/tanh(s[i,2]*tau) for i in 1:nSinh);
        end f_it;

        function f_itt "ideal part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_ideal_tau_tau
          "ideal part of dimensionless Helmholtz energy";

      protected
          final constant Integer nLog=size(helmholtzCoefficients.idealLog, 1);
          final constant Integer nPower=size(helmholtzCoefficients.idealPower, 1);
          final constant Integer nEinstein=size(helmholtzCoefficients.idealEinstein, 1);
          final constant Integer nCosh=size(helmholtzCoefficients.idealCosh, 1);
          final constant Integer nSinh=size(helmholtzCoefficients.idealSinh, 1);

          final constant Real[nLog, 2] l=helmholtzCoefficients.idealLog;
          final constant Real[nPower, 2] p=helmholtzCoefficients.idealPower;
          final constant Real[nEinstein, 2] e=helmholtzCoefficients.idealEinstein;
          final constant Real[nCosh, 2] c=helmholtzCoefficients.idealCosh;
          final constant Real[nSinh, 2] s=helmholtzCoefficients.idealSinh;

        algorithm
          f_ideal_tau_tau :=
              sum(-(l[i,1]*l[i,2])/tau^2 for i in 1:nLog)
            + sum(p[i,1]*tau^(p[i,2]-2)*p[i,2]*(p[i,2]-1) for i in 1:nPower)
            - sum(e[i,1]*(-e[i,2])^2*exp(e[i,2]*tau)*(1 - exp(e[i,2]*tau))^(-2) for i in 1:nEinstein)
            - sum(c[i,1]*c[i,2]^2/(cosh(c[i,2]*tau))^2 for i in 1:nCosh)
            - sum(s[i,1]*s[i,2]^2/(sinh(s[i,2]*tau))^2 for i in 1:nSinh);
        end f_itt;

        function f_ittt "ideal part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_ideal_tau_tau
          "ideal part of dimensionless Helmholtz energy";

      protected
          final constant Integer nLog=size(helmholtzCoefficients.idealLog, 1);
          final constant Integer nPower=size(helmholtzCoefficients.idealPower, 1);
          final constant Integer nEinstein=size(helmholtzCoefficients.idealEinstein, 1);
          final constant Integer nCosh=size(helmholtzCoefficients.idealCosh, 1);
          final constant Integer nSinh=size(helmholtzCoefficients.idealSinh, 1);

          final constant Real[nLog, 2] l=helmholtzCoefficients.idealLog;
          final constant Real[nPower, 2] p=helmholtzCoefficients.idealPower;
          final constant Real[nEinstein, 2] e=helmholtzCoefficients.idealEinstein;
          final constant Real[nCosh, 2] c=helmholtzCoefficients.idealCosh;
          final constant Real[nSinh, 2] s=helmholtzCoefficients.idealSinh;

        algorithm
          f_ideal_tau_tau :=
              sum((2*l[i,1]*l[i,2])/tau^3 for i in 1:nLog)
            + sum(p[i,1]*p[i,2]*tau^(p[i,2] - 3)*(p[i,2] - 1)*(p[i,2] - 2) for i in 1:nPower)
            + sum((e[i,1]*e[i,2]^3*exp(e[i,2]*tau)*(exp(e[i,2]*tau) + 1))/(exp(e[i,2]*tau) - 1)^3 for i in 1:nEinstein)
            - sum(2*c[i,1]*c[i,2]^3*tanh(c[i,2]*tau)*(tanh(c[i,2]*tau)^2 - 1) for i in 1:nCosh)
            + sum(2*s[i,1]*s[i,2]^3*1/tanh(s[i,2]*tau)*(1/tanh(s[i,2]*tau)^2 - 1) for i in 1:nSinh);  //  coth(x) = 1/tanh(x)
        end f_ittt;

        function f_id "ideal part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_ideal_delta
          "ideal part of dimensionless Helmholtz energy";

        algorithm
            f_ideal_delta := 1.0/delta;
        end f_id;

        function f_idd "ideal part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_ideal_delta_delta
          "ideal part of dimensionless Helmholtz energy";

        algorithm
            f_ideal_delta_delta := -1.0/(delta*delta);
        end f_idd;

        function f_r "residual part of dimensionless Helmholtz energy"

          input Real delta(min=0);
          input Real tau(min=0);
          output Real f_residual
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual :=
            sum(p[i,1]*tau^p[i,2]*delta^p[i,3] for i in 1:nPoly)
          + sum(b[i,1]*tau^b[i,2]*delta^b[i,3]*exp(-delta^b[i,4]) for i in 1:nBwr)
          + sum(g[i,1]*tau^g[i,2]*delta^g[i,3]*exp(g[i,6]*(delta - g[i,9])^g[i,4] + g[i,7]*(tau - g[i,8])^g[i,5]) for i in 1:nGauss);
        end f_r;

        function f_rt "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_tau
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_tau :=
              sum(p[i,1]*p[i,2]*tau^(p[i,2] - 1)*delta^p[i,3] for i in 1:nPoly)
            + sum(b[i,1]*b[i,2]*tau^(b[i,2] - 1)*delta^b[i,3]*exp(-delta^b[i,4]) for i in 1:nBwr)
            + sum(g[i,1]*tau^g[i,2]*delta^g[i,3]*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(tau - g[i,8])^2)*(g[i,2]/tau + 2*g[i,7]*(tau - g[i,8])) for i in 1:nGauss);

        end f_rt;

        function f_rtt "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_tau_tau
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_tau_tau :=
              sum(p[i,1]*p[i,2]*(p[i,2] - 1)*delta^p[i,3]*tau^(p[i,2] - 2) for i in 1:nPoly)
            + sum(b[i,1]*b[i,2]*(b[i,2] - 1)*delta^b[i,3]*tau^(b[i,2] - 2)*exp(-delta^b[i,4]) for i in 1:nBwr)
            + sum(g[i,1]*delta^g[i,3]*tau^g[i,2]*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(tau - g[i,8])^2)*((g[i,2]/tau + 2*g[i,7]*(tau - g[i,8]))^2 - g[i,2]/tau^2 + 2*g[i,7]) for i in 1:nGauss);

        end f_rtt;

        function f_rtd "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_delta_tau
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_delta_tau :=
              sum(p[i,1]*p[i,3]*p[i,2]*delta^(p[i,3] - 1)*tau^(p[i,2] - 1) for i in 1:nPoly)
            + sum(b[i,1]*b[i,2]*delta^(b[i,3] - 1)*tau^(b[i,2] - 1)*(b[i,3] - b[i,4]*delta^b[i,4])*exp(-delta^b[i,4]) for i in 1:nBwr)
            + sum(g[i,1]*delta^g[i,3]*tau^g[i,2]*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(tau - g[i,8])^2)*(g[i,3]/delta + 2*g[i,6]*(delta - g[i,9]))*(g[i,2]/tau + 2*g[i,7]*(tau - g[i,8])) for i in 1:nGauss);

        end f_rtd;

        function f_rttd "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_tau_tau_delta
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_tau_tau_delta :=
              sum(delta^(p[i,3] - 1)*p[i,1]*p[i,2]*p[i,3]*tau^(p[i,2] - 2)*(p[i,2] - 1) for i in 1:nPoly)
            + sum(b[i,1]*b[i,2]*delta^(b[i,3] - 1)*tau^(b[i,2] - 2)*exp(-delta^b[i,4])*(b[i,3] - b[i,4]*delta^b[i,4])*(b[i,2] - 1) for i in 1:nBwr)
            + sum(delta^(g[i,3] - 1)*g[i,1]*tau^(g[i,2] - 2)*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(g[i,8] - tau)^2)*(2*g[i,6]*delta^2 - 2*g[i,6]*g[i,9]*delta + g[i,3])*(g[i,2]^2 - 4*g[i,2]*g[i,7]*g[i,8]*tau + 4*g[i,2]*g[i,7]*tau^2 - g[i,2] + 4*g[i,7]^2*g[i,8]^2*tau^2 - 8*g[i,7]^2*g[i,8]*tau^3 + 4*g[i,7]^2*tau^4 + 2*g[i,7]*tau^2) for i in 1:nGauss);

        end f_rttd;

        function f_rttt "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_tau_tau
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_tau_tau :=
              sum(delta^p[i,3]*p[i,1]*p[i,2]*tau^(p[i,2] - 3)*(p[i,2] - 1)*(p[i,2] - 2) for i in 1:nPoly)
            + sum(b[i,1]*b[i,2]*delta^b[i,3]*tau^(b[i,2] - 3)*exp(-delta^b[i,4])*(b[i,2] - 1)*(b[i,2] - 2) for i in 1:nBwr)
            + sum(delta^g[i,3]*g[i,1]*tau^(g[i,2] - 3)*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(g[i,8] - tau)^2)*(g[i,2]^3 - 6*g[i,2]^2*g[i,7]*g[i,8]*tau + 6*g[i,2]^2*g[i,7]*tau^2 - 3*g[i,2]^2 + 12*g[i,2]*g[i,7]^2*g[i,8]^2*tau^2 - 24*g[i,2]*g[i,7]^2*g[i,8]*tau^3 + 12*g[i,2]*g[i,7]^2*tau^4 + 6*g[i,2]*g[i,7]*g[i,8]*tau + 2*g[i,2] - 8*g[i,7]^3*g[i,8]^3*tau^3 + 24*g[i,7]^3*g[i,8]^2*tau^4 - 24*g[i,7]^3*g[i,8]*tau^5 + 8*g[i,7]^3*tau^6 - 12*g[i,7]^2*g[i,8]*tau^3 + 12*g[i,7]^2*tau^4) for i in 1:nGauss);

        end f_rttt;

        function f_rtdd "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_tau_delta_delta
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_tau_delta_delta :=
              sum(delta^(p[i,3] - 2)*p[i,1]*p[i,2]*p[i,3]*tau^(p[i,2] - 1)*(p[i,3] - 1) for i in 1:nPoly)
            + sum(-b[i,1]*b[i,2]*delta^(b[i,3] - 2)*tau^(b[i,2] - 1)*exp(-delta^b[i,4])*(b[i,3] + b[i,4]^2*delta^b[i,4] - b[i,4]^2*delta^(2*b[i,4]) - b[i,3]^2 - b[i,4]*delta^b[i,4] + 2*b[i,3]*b[i,4]*delta^b[i,4]) for i in 1:nBwr)
            + sum(delta^(g[i,3] - 2)*g[i,1]*tau^(g[i,2] - 1)*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(g[i,8] - tau)^2)*(2*g[i,7]*tau^2 - 2*g[i,7]*g[i,8]*tau + g[i,2])*(4*delta^4*g[i,6]^2 - 8*delta^3*g[i,6]^2*g[i,9] + 4*delta^2*g[i,3]*g[i,6] + 4*delta^2*g[i,6]^2*g[i,9]^2 + 2*delta^2*g[i,6] - 4*delta*g[i,3]*g[i,6]*g[i,9] + g[i,3]^2 - g[i,3]) for i in 1:nGauss);

        end f_rtdd;

        function f_rd "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_delta
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_delta :=
              sum(p[i,1]*p[i,3]*delta^(p[i,3] - 1)*tau^p[i,2] for i in 1:nPoly)
            + sum(b[i,1]*exp(-delta^b[i,4])*(delta^(b[i,3] - 1)*tau^b[i,2]*(b[i,3] - b[i,4]*delta^b[i,4])) for i in 1:nBwr)
            + sum(g[i,1]*delta^g[i,3]*tau^g[i,2]*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(tau - g[i,8])^2)*(g[i,3]/delta + 2*g[i,6]*(delta - g[i,9])) for i in 1:nGauss);

        end f_rd;

        function f_rdd "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_delta_delta
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_delta_delta :=
              sum(p[i,1]*p[i,3]*(p[i,3] - 1)*delta^(p[i,3] - 2)*tau^p[i,2] for i in 1:nPoly)
            + sum(b[i,1]*exp(-delta^b[i,4])*(delta^(b[i,3] - 2)*tau^b[i,2]*((b[i,3] - b[i,4]*delta^b[i,4])*(b[i,3] - 1 - b[i,4]*delta^b[i,4]) - b[i,4]^2*delta^b[i,4])) for i in 1:nBwr)
            + sum(g[i,1]*tau^g[i,2]*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(tau - g[i,8])^2)*(  2*g[i,6]*delta^g[i,3] + 4*g[i,6]^2*delta^g[i,3]*(delta - g[i,9])^2 + 4*g[i,3]*g[i,6]*delta^(g[i,3] - 1)*(delta - g[i,9]) +   g[i,3]*(g[i,3] - 1)*delta^(g[i,3] - 2)) for i in 1:nGauss);

        end f_rdd;

        function f_rddd "residual part of dimensionless Helmholtz energy"

          input Real delta;
          input Real tau;
          output Real f_residual_delta_delta_delta
          "residual part of dimensionless Helmholtz energy";

      protected
          final constant Integer nPoly = size(helmholtzCoefficients.residualPoly,1);
          final constant Integer nBwr = size(helmholtzCoefficients.residualBwr,1);
          final constant Integer nGauss = size(helmholtzCoefficients.residualGauss,1);

          final constant Real[nPoly,4] p = helmholtzCoefficients.residualPoly;
          final constant Real[nBwr,4] b = helmholtzCoefficients.residualBwr;
          final constant Real[nGauss,9] g = helmholtzCoefficients.residualGauss;

        algorithm
          f_residual_delta_delta_delta :=
              sum(delta^(p[i,3] - 3)*p[i,1]*p[i,3]*tau^p[i,2]*(p[i,3] - 1)*(p[i,3] - 2) for i in 1:nPoly)
            + sum(-b[i,1]*delta^(b[i,3] - 3)*tau^b[i,2]*exp(-delta^b[i,4])*(b[i,4]^3*delta^b[i,4] - 3*b[i,4]^2*delta^b[i,4] - 2*b[i,3] + 3*b[i,4]^2*delta^(2*b[i,4]) - 3*b[i,4]^3*delta^(2*b[i,4]) + b[i,4]^3*delta^(3*b[i,4]) + 3*b[i,3]^2 - b[i,3]^3 + 2*b[i,4]*delta^b[i,4] - 3*b[i,3]*b[i,4]^2*delta^(2*b[i,4]) - 6*b[i,3]*b[i,4]*delta^b[i,4] + 3*b[i,3]*b[i,4]^2*delta^b[i,4] + 3*b[i,3]^2*b[i,4]*delta^b[i,4]) for i in 1:nBwr)
            + sum(delta^(g[i,3] - 3)*g[i,1]*tau^g[i,2]*exp(g[i,6]*(delta - g[i,9])^2 + g[i,7]*(g[i,8] - tau)^2)*(8*delta^6*g[i,6]^3 - 24*delta^5*g[i,6]^3*g[i,9] + 12*delta^4*g[i,3]*g[i,6]^2 + 24*delta^4*g[i,6]^3*g[i,9]^2 + 12*delta^4*g[i,6]^2 - 24*delta^3*g[i,3]*g[i,6]^2*g[i,9] - 8*delta^3*g[i,6]^3*g[i,9]^3 - 12*delta^3*g[i,6]^2*g[i,9] + 6*delta^2*g[i,3]^2*g[i,6] + 12*delta^2*g[i,3]*g[i,6]^2*g[i,9]^2 - 6*delta*g[i,3]^2*g[i,6]*g[i,9] + 6*delta*g[i,3]*g[i,6]*g[i,9] + g[i,3]^3 - 3*g[i,3]^2 + 2*g[i,3]) for i in 1:nGauss);

        end f_rddd;

        function p "returns pressure p from EoS"
          input HelmholtzDerivs f;
          output AbsolutePressure p;

        algorithm
          p := f.d*f.T*f.R*(1+f.delta*f.rd);
        end p;

        function s "returns specifc entropy s from EoS"
          input HelmholtzDerivs f;
          output SpecificEntropy s;

        algorithm
          s := f.R*(f.tau*(f.it + f.rt) - f.i - f.r);
        end s;

        function u "returns specifc internal energy u from EoS"
          input HelmholtzDerivs f;
          output SpecificEnergy u;

        algorithm
          u := f.T*f.R*(f.tau*(f.it + f.rt));
        end u;

        function h "returns specifc enthalpy h from EoS"
          input HelmholtzDerivs f;
          output SpecificEnthalpy h;

        algorithm
          h := f.T*f.R*(f.tau*(f.it + f.rt) + (1+f.delta*f.rd));
        end h;

        function g "returns specifc Gibbs energy g from EoS"
          input HelmholtzDerivs f;
          output SpecificEnergy g;

        algorithm
          g := f.T*f.R*((f.i+f.r)+(1+f.delta*f.rd));
        end g;

        function cp0 "returns ideal gas heat capacity from EoS"
          input HelmholtzDerivs f;
          output SpecificHeatCapacity cp0;

        algorithm
          cp0 := f.R*(1 - f.tau*f.tau*f.itt);
        end cp0;

        function dpdT "returns pressure derivative (dp/dd)@T=const"
          input EoS.HelmholtzDerivs f;
          output DerPressureByDensity dpdT;

        algorithm
          dpdT := f.T*f.R*(1 + 2*f.delta*f.rd + f.delta*f.delta*f.rdd);
        end dpdT;

        function dpTd "returns pressure derivative (dp/dT)@d=const"
          input EoS.HelmholtzDerivs f;
          output DerPressureByTemperature dpTd;

        algorithm
          dpTd := f.d*f.R*(1 + f.delta*f.rd - f.delta*f.tau*f.rtd);
        end dpTd;

        function dsdT "returns entropy derivative (ds/dd)@T=const"
          input EoS.HelmholtzDerivs f;
          output DerEntropyByDensity dsdT;

        algorithm
          dsdT := -f.R/f.d*(1+f.delta*f.rd -f.tau*f.delta*f.rtd);
        end dsdT;

        function dsTd "returns entropy derivative (ds/dT)@d=const"
          input EoS.HelmholtzDerivs f;
          output DerEntropyByTemperature dsTd;

        algorithm
          dsTd := f.R/f.T*(-f.tau*f.tau*(f.itt+f.rtt));
        end dsTd;

        function dudT "returns internal energy derivative (du/dd)@T=const"
          input EoS.HelmholtzDerivs f;
          output DerEnergyByDensity dudT;

        algorithm
          dudT := f.R*f.T/f.d*(f.tau*f.delta*f.rtd);
        end dudT;

        function duTd "returns internal energy derivative (du/dT)@d=const"
          input EoS.HelmholtzDerivs f;
          output DerEnergyByTemperature duTd;

        algorithm
          duTd := f.R*(-f.tau*f.tau*(f.itt + f.rtt));
        end duTd;

        function dhdT "returns enthalpy derivative (dh/dd)@T=const"
          input EoS.HelmholtzDerivs f;
          output DerEnthalpyByDensity dhdT;

        algorithm
          dhdT := f.T*f.R/f.d*(0+f.tau*f.delta*f.rtd + f.delta*f.rd + f.delta*f.delta*f.rdd);
        end dhdT;

        function dhTd "returns enthalpy derivative (dh/dT)@d=const"
          input EoS.HelmholtzDerivs f;
          output DerEnthalpyByTemperature dhTd;

        algorithm
          dhTd := f.R*(1 - f.tau*f.tau*(f.itt+f.rtt) + f.delta*f.rd - f.tau*f.delta*f.rtd);
        end dhTd;

        function dgdT "returns Gibbs energy derivative (dg/dd)@T=const"
          input EoS.HelmholtzDerivs f;
          output DerEnergyByDensity dgdT;

        algorithm
          dgdT := f.T*f.R/f.d*(1+2*f.delta*f.rd + f.delta*f.delta*f.rdd);
        end dgdT;

        function dgTd "returns Gibbs energy derivative (dg/dT)@d=const"
          input EoS.HelmholtzDerivs f;
          output DerEnergyByTemperature dgTd;

        algorithm
          dgTd := f.R*(-f.tau*(f.it+f.rt) +(f.i+f.r) +(1+f.delta*f.rd-f.delta*f.tau*f.rtd));
        end dgTd;

        function d2pd2T "returns pressure derivative (d2p/dd2)@T=const"
          input EoS.HelmholtzDerivs f;
          output Der2PressureByDensity2 d2pd2T;

        algorithm
          d2pd2T := f.T*f.R/f.d*(2*f.delta*f.rd + 4*f.delta*f.delta*f.rdd + f.delta*f.delta*f.delta*f.rddd);
        end d2pd2T;

        function d2pT2d "returns pressure derivative (d2p/dT2)@d=const"
          input EoS.HelmholtzDerivs f;
          output Der2PressureByTemperature2 d2pT2d;

        algorithm
          d2pT2d := f.d*f.R/f.T*(f.tau^2*f.delta*f.rttd);
        end d2pT2d;

        function d2pTd "returns pressure derivative (d2p/dTdd)"
          input EoS.HelmholtzDerivs f;
          output Der2PressureByTemperatureDensity d2pTd;

        algorithm
          d2pTd := f.R*(1 + 2*f.delta*f.rd + f.delta*f.delta*f.rdd - 2*f.delta*f.tau*f.rtd - f.tau*f.delta*f.delta*f.rtdd);
        end d2pTd;

        function d2sd2T "returns entropy derivative (d2s/dd2)@T=const"
          input EoS.HelmholtzDerivs f;
          output Der2EntropyByDensity2 d2sd2T;

        algorithm
          d2sd2T := f.R/f.d^2*(1 - f.delta*f.delta*f.rdd + f.tau*f.delta*f.delta*f.rtdd);
        end d2sd2T;

        function d2sT2d "returns entropy derivative (d2s/dT2)@d=const"
          input EoS.HelmholtzDerivs f;
          output Der2EntropyByTemperature2 d2sT2d;

        algorithm
          d2sT2d := f.R/f.T^2*(f.tau*f.tau*f.tau*(f.ittt + f.rttt) + 3*f.tau*f.tau*(f.itt + f.rtt));
        end d2sT2d;

        function d2sTd "returns entropy derivative (d2s/dT dd)"
          input EoS.HelmholtzDerivs f;
          output Der2EntropyByTemperatureDensity d2sTd;

        algorithm
          d2sTd := f.R/(f.T*f.d)*(-f.tau*f.tau*f.delta*f.rttd);
        end d2sTd;

        function d2ud2T "returns internal energy derivative (d2u/dd2)@T=const"
          input EoS.HelmholtzDerivs f;
          output Der2EnergyByDensity2 d2ud2T;

        algorithm
          d2ud2T := f.R*f.T/f.d^2*(f.tau*f.delta*f.delta*f.rtdd);
        end d2ud2T;

        function d2uT2d "returns internal energy derivative (d2u/dT2)@d=const"
          input EoS.HelmholtzDerivs f;
          output Der2EnergyByTemperature2 d2uT2d;

        algorithm
          d2uT2d := f.R/f.T*(f.tau*f.tau*f.tau*(f.ittt + f.rttt) +2*f.tau*f.tau*(f.itt + f.rtt));
        end d2uT2d;

        function d2uTd "returns internal energy derivative (d2u/dT dd)"
          input EoS.HelmholtzDerivs f;
          output Der2EnergyByTemperatureDensity d2uTd;

        algorithm
          d2uTd := f.R/f.d*(-f.tau*f.tau*f.delta*f.rttd);
        end d2uTd;

        function d2hd2T "returns enthalpy derivative (d2h/dd2)@T=const"
          input EoS.HelmholtzDerivs f;
          output Der2EnthalpyByDensity2 d2hd2T;

        algorithm
          d2hd2T := f.R*f.T/f.d^2*((f.tau*f.delta*f.delta*f.rtdd) + (2*f.delta*f.delta*f.rdd + f.delta*f.delta*f.delta*f.rddd));
        end d2hd2T;

        function d2hT2d "returns enthalpy derivative (d2h/dT2)@d=const"
          input EoS.HelmholtzDerivs f;
          output Der2EnthalpyByTemperature2 d2hT2d;

        algorithm
          d2hT2d := f.R/f.T*(f.tau*f.tau*f.tau*(f.ittt + f.rttt) + 2*f.tau*f.tau*(f.itt + f.rtt) + f.tau^2*f.delta*f.rttd);
        end d2hT2d;

        function d2hTd "returns enthalpy derivative (d2h/dT dd)"
          input EoS.HelmholtzDerivs f;
          output Der2EnthalpyByTemperatureDensity d2hTd;

        algorithm
          d2hTd := f.R/f.d*((-f.tau*f.tau*f.delta*f.rttd) + (f.delta*f.delta*f.rdd + f.delta*f.rd - f.tau*f.delta*f.delta*f.rtdd - f.tau*f.delta*f.rtd));
        end d2hTd;

        function d2gd2T "returns Gibbs energy derivative (d2g/dd2)@T=const"
          input EoS.HelmholtzDerivs f;
          output Der2EnergyByDensity2 d2gd2T;

        algorithm
          d2gd2T := f.T*f.R/f.d^2*(-1 + (3*f.delta*f.delta*f.rdd + f.delta*f.delta*f.delta*f.rddd));
        end d2gd2T;

        function d2gT2d "returns Gibbs energy derivative (d2g/dT2)@d=const"
          input EoS.HelmholtzDerivs f;
          output Der2EnergyByTemperature2 d2gT2d;

        algorithm
          d2gT2d := f.R/f.T*(f.tau*f.tau*(f.itt+f.rtt) + (f.tau^2*f.delta*f.rttd));
        end d2gT2d;

        function d2gTd "returns Gibbs energy derivative (d2g/dT dd)"
          input EoS.HelmholtzDerivs f;
          output Der2EnergyByTemperatureDensity d2gTd;

        algorithm
          d2gTd := f.R/f.d*(1 + 2*f.delta*f.rd - 2*f.tau*f.delta*f.rtd + f.delta*f.delta*f.rdd - f.tau*f.delta*f.delta*f.rtdd);
        end d2gTd;

        function dpvT "returns pressure derivative (dp/dv)@T=const"
          input EoS.HelmholtzDerivs f;
          output DerPressureByVolume dpvT;

        algorithm
          dpvT := -f.d*f.d*f.T*f.R*(1 + 2*f.delta*f.rd + f.delta*f.delta*f.rdd);
        end dpvT;

        function d2pv2T "returns pressure derivative (d2p/dv2)@T=const"
          input EoS.HelmholtzDerivs f;
          output Der2PressureByVolume2 d2pv2T;

        algorithm
          d2pv2T := f.d*f.d*f.d*f.T*f.R*(2 + 6*f.delta*f.rd + 6*f.delta*f.delta*f.rdd + f.delta*f.delta*f.delta*f.rddd);
        end d2pv2T;

        function d2pTv "returns pressure derivative (d2p/dT dv)"
          input EoS.HelmholtzDerivs f;
          output Der2PressureByTemperatureVolume d2pTv;

        algorithm
          d2pTv := -f.d*f.d*f.R*(1 + 2*f.delta*f.rd + f.delta*f.delta*f.rdd - 2*f.delta*f.tau*f.rtd - f.tau*f.delta*f.delta*f.rtdd);
        end d2pTv;
      end EoS;

      package Ancillary

        record AncillaryCoefficients
        "Coefficients for anciallry equations (psat, dliq, dvap)"

          constant PressureSaturationModel pressureSaturationModel=PressureSaturationModel.PS5;
          constant Real[:,2] pressureSaturation = fill(0.0, 0, 2)
          "vapor pressure coefficients";

          constant DensityLiquidModel densityLiquidModel=DensityLiquidModel.DL1;
          constant Real[:,2] densityLiquid = fill(0.0, 0, 2)
          "saturated liquid density coefficients";

          constant DensityVaporModel densityVaporModel=DensityVaporModel.DV3;
          constant Real[:,2] densityVapor = fill(0.0, 0, 2)
          "saturated vapor density coefficients";

          constant PressureMeltingModel pressureMeltingModel=PressureMeltingModel.ML1;
          constant Temperature T_reducing=273.15;
          constant AbsolutePressure p_reducing=101325;
          constant Real[:,2] pressureMelting1 = fill(0.0, 0, 2)
          "melting pressure coefficients";
          constant Real[:,2] pressureMelting2 = fill(0.0, 0, 2)
          "melting pressure coefficients";
          constant Real[:,2] pressureMelting3 = fill(0.0, 0, 2)
          "melting pressure coefficients";
        end AncillaryCoefficients;

        function bubbleDensity_T
        "ancillary function: calculate density of saturated liquid for a given T"

          input Modelica.SIunits.Temperature T;
          output Modelica.SIunits.Density dliq;

      protected
          DensityLiquidModel densityLiquidModel=ancillaryCoefficients.densityLiquidModel;
          Density d_crit=fluidConstants[1].molarMass/fluidConstants[1].criticalMolarVolume;
          Real delta "reduced density";
          Temperature T_crit=fluidConstants[1].criticalTemperature;
          Real T_theta;
          Real tau=T_crit/T "inverse reduced temperature";

          Integer nDliq=size(ancillaryCoefficients.densityLiquid, 1);
          Real[nDliq] n=ancillaryCoefficients.densityLiquid[:, 1];
          Real[nDliq] theta=ancillaryCoefficients.densityLiquid[:, 2];

        algorithm
          if (densityLiquidModel == DensityLiquidModel.DL1) then
            T_theta := max((1 - T/T_crit), Modelica.Constants.small); // odd
            delta   := sum(n[i]*T_theta^theta[i] for i in 1:nDliq);
            delta   := 1 + delta; // DL1 or DL2
          elseif (densityLiquidModel == DensityLiquidModel.DL2) then
            T_theta := max((1 - T/T_crit)^(1/3), Modelica.Constants.small); // even
            delta   := sum(n[i]*T_theta^theta[i] for i in 1:nDliq);
            delta   := 1 + delta; // DL1 or DL2

          elseif (densityLiquidModel == DensityLiquidModel.DL3) then
            T_theta := max((1 - T/T_crit), Modelica.Constants.small); // odd
            delta   := sum(n[i]*T_theta^theta[i] for i in 1:nDliq);
            delta   := exp(delta);   // DL3 or DL4
          elseif (densityLiquidModel == DensityLiquidModel.DL4) then
            T_theta := max((1 - T/T_crit)^(1/3), Modelica.Constants.small); // even
            delta   := sum(n[i]*T_theta^theta[i] for i in 1:nDliq);
            delta   := exp(delta);   // DL3 or DL4

          elseif (densityLiquidModel == DensityLiquidModel.DL5) then
            T_theta := max((1 - T/T_crit), Modelica.Constants.small); // odd
            delta   := sum(n[i]*T_theta^theta[i] for i in 1:nDliq);
            delta   := exp(tau*delta);   // DL5 or DL6
          elseif (densityLiquidModel == DensityLiquidModel.DL6) then
            T_theta := max((1 - T/T_crit)^(1/3), Modelica.Constants.small); // even
            delta   := sum(n[i]*T_theta^theta[i] for i in 1:nDliq);
            delta   := exp(tau*delta);   // DL5 or DL6

          else
            assert(false, "bubbleDensity_d: this should not happen, check DensityLiquidModel");
          end if;

          dliq := d_crit*delta;

        end bubbleDensity_T;

        function dewDensity_T
        "ancillary function: calculate density of saturated vapor for a given T"

          input Modelica.SIunits.Temperature T;
          output Modelica.SIunits.Density dvap;

      protected
          DensityVaporModel densityVaporModel=ancillaryCoefficients.densityVaporModel;
          Density d_crit=fluidConstants[1].molarMass/fluidConstants[1].criticalMolarVolume;
          Real delta "reduced density";
          Temperature T_crit=fluidConstants[1].criticalTemperature;
          Real T_theta;
          Real tau=T_crit/T "inverse reduced temperature";

          Integer nDvap=size(ancillaryCoefficients.densityVapor, 1);
          Real[nDvap] n=ancillaryCoefficients.densityVapor[:, 1];
          Real[nDvap] theta=ancillaryCoefficients.densityVapor[:, 2];

        algorithm
          if (densityVaporModel == DensityVaporModel.DV1) then
            T_theta := max((1 - T/T_crit), Modelica.Constants.small); // odd
            delta := sum(n[i]*T_theta^theta[i] for i in 1:nDvap);
            delta := 1 + delta; // DV1 or DV2
          elseif (densityVaporModel == DensityVaporModel.DV2) then
            T_theta := max((1 - T/T_crit)^(1/3), Modelica.Constants.small); // even
            delta := sum(n[i]*T_theta^theta[i] for i in 1:nDvap);
            delta := 1 + delta; // DV1 or DV2

          elseif (densityVaporModel == DensityVaporModel.DV3) then
            T_theta := max((1 - T/T_crit), Modelica.Constants.small); // odd
            delta := sum(n[i]*T_theta^theta[i] for i in 1:nDvap);
            delta := exp(delta);   // DV3 or DV4
          elseif (densityVaporModel == DensityVaporModel.DV4) then
            T_theta := max((1 - T/T_crit)^(1/3), Modelica.Constants.small); // even
            delta := sum(n[i]*T_theta^theta[i] for i in 1:nDvap);
            delta := exp(delta);   // DV3 or DV4

          elseif (densityVaporModel == DensityVaporModel.DV5) then
            T_theta := max((1 - T/T_crit), Modelica.Constants.small); // odd
            delta := sum(n[i]*T_theta^theta[i] for i in 1:nDvap);
            delta := exp(tau*delta);   // DV5 or DV6
          elseif (densityVaporModel == DensityVaporModel.DV6) then
            T_theta := max((1 - T/T_crit)^(1/3), Modelica.Constants.small); // even
            delta := sum(n[i]*T_theta^theta[i] for i in 1:nDvap);
            delta := exp(tau*delta);   // DV5 or DV6

          else
            assert(false, "dewDensity_d: this should not happen, check DensityVaporModel");
          end if;

          dvap := d_crit*delta;

        end dewDensity_T;

        function saturationPressure_T
        "ancillary function: calculate saturation pressure for a given Temperature"
          input Modelica.SIunits.Temperature T;
          output Modelica.SIunits.AbsolutePressure p;

      protected
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant Real tau=T_crit/T "inverse reduced temperature";
          constant Real T_theta=max((1 - T/T_crit), Modelica.Constants.small);
          constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;

          Integer nPressureSaturation=size(ancillaryCoefficients.pressureSaturation, 1);
          Real[nPressureSaturation] n=ancillaryCoefficients.pressureSaturation[:, 1];
          Real[nPressureSaturation] theta=ancillaryCoefficients.pressureSaturation[:, 2];

          constant Real eps=1e-9;

        algorithm
          assert(T <= T_crit+eps, "saturationPressure error: Temperature is higher than critical temperature");
          p := p_crit*exp(tau*sum(n[i]*T_theta^theta[i] for i in 1:nPressureSaturation));

          // this is an ancillary forward function
          // the corresponding iterative backward function is saturationTemperature(p)
          annotation (inverse(T=saturationTemperature_p(p=p)), Documentation(info="<html>
      <p>
      This algorithm returns the saturation pressure as a function of Temperature: psat=psat(T).
      This type of vapor pressure equation was developed by W. Wagner.
      Because it cannot be solved for temperature analytically, 
      the inverse function Tsat=Tsat(p) has to find Tsat iteratively.
      </p>
      
      <dl>
      <dt>Wagner, W.</dt>
      <dd> <b>Eine mathematisch statistische Methode zum Aufstellen thermodynamischer Gleichungen - gezeigt am Beispiel der Dampfdruckkurve reiner fluider Stoffe.</b><br>
           Forschrittberichte der VDI Zeitschriften, Reihe 3, Nr. 39 (1974)
      </dd>
      </dl>
      </html>"));
        end saturationPressure_T;

        function meltingPressure_T
        "ancillary function: calculate melting pressure for a given Temperature"
          input Temperature T;
          output AbsolutePressure p_melt;

      protected
          PressureMeltingModel pressureMeltingModel=ancillaryCoefficients.pressureMeltingModel;
          Temperature T_reducing=ancillaryCoefficients.T_reducing;
          AbsolutePressure p_reducing=ancillaryCoefficients.p_reducing;

          Integer nPressureMelting1=size(ancillaryCoefficients.pressureMelting1, 1);
          Real[nPressureMelting1,2] n1=ancillaryCoefficients.pressureMelting1;

          Integer nPressureMelting2=size(ancillaryCoefficients.pressureMelting2, 1);
          Real[nPressureMelting2,2] n2=ancillaryCoefficients.pressureMelting2;

          Integer nPressureMelting3=size(ancillaryCoefficients.pressureMelting3, 1);
          Real[nPressureMelting3,2] n3=ancillaryCoefficients.pressureMelting3;

          constant Real Tr=max((T/T_reducing), Modelica.Constants.small)
          "reduced temperature";
          Real pr "reduced pressure";

        algorithm
          pr := 0;
          pr := sum(n1[i,1]*   (Tr)  ^n1[i,2] for i in 1:nPressureMelting1)
              + sum(n2[i,1]*   (Tr-1)^n2[i,2] for i in 1:nPressureMelting2)
              + sum(n3[i,1]*log(Tr)  ^n3[i,2] for i in 1:nPressureMelting3);

          if (pressureMeltingModel == PressureMeltingModel.ML1) then
            p_melt := p_reducing*pr;
          elseif (pressureMeltingModel == PressureMeltingModel.ML2) then
            p_melt := p_reducing*exp(pr);
          end if;

          if (pr<1e-9) then
            p_melt := fluidLimits.PMAX;
          end if;

        end meltingPressure_T;

        function saturationTemperature_p
        "ancillary iterative function: calculate saturation temperature for a given pressure by iterating the ancillary function"

          input Modelica.SIunits.AbsolutePressure p;
          output Modelica.SIunits.Temperature T;

      protected
          constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          Real tau "inverse reduced temperature";
          Real T_theta;
          constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
          constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;

          Integer nPressureSaturation=size(ancillaryCoefficients.pressureSaturation, 1);
          Real[nPressureSaturation] n=ancillaryCoefficients.pressureSaturation[:, 1];
          Real[nPressureSaturation] theta=ancillaryCoefficients.pressureSaturation[:, 2];

          Real RES_p;
          Real dpdT;
          constant Real gamma(min=0,max=1) = 1 "convergence speed, default=1";
          constant Real tolerance=1e-6 "relative tolerance for RES_p";
          Integer iter=0;
          constant Integer iter_max = 200;

        algorithm
        if (p<p_crit) and (p>p_trip) then
          // calculate start value from the log(p) vs. 1/T diagram
          // see Span (2000) page 52 / equation 3.98
          T := 1/(1/T_crit - (1/T_trip-1/T_crit)/log(p_crit/p_trip)*log(p/p_crit));
          T := min(T,T_crit- Modelica.Constants.eps);

          // calculate RES_p
          tau := T_crit/T;
          T_theta := max((1 - T/T_crit), Modelica.Constants.small);
          RES_p   := p_crit*exp(tau*sum(n[i]*T_theta^theta[i] for i in 1:nPressureSaturation)) - p;

          while ((abs(RES_p/p)>tolerance) and (iter<iter_max)) loop
            iter:=iter + 1;

            // calculate gradient of RES_p (= gradient of Wagner equation)
            dpdT := -p_crit*exp(tau*sum(n[i]*T_theta^theta[i] for i in 1:nPressureSaturation))
                    *(1/T*sum(n[i]*theta[i]*T_theta^(theta[i]-1) for i in 1:nPressureSaturation)
                    +tau/T*sum(n[i]*T_theta^theta[i] for i in 1:nPressureSaturation));

            /* // print for debugging
    Modelica.Utilities.Streams.print(" ", "printlog.txt");
    Modelica.Utilities.Streams.print("Iteration step " +String(iter), "printlog.txt");
    Modelica.Utilities.Streams.print("T=" + String(T) + " and dpdT=" + String(dpdT), "printlog.txt"); */

            // calculate better T
            T := T - gamma/dpdT*RES_p;

            // check bounds
            T := max(T,T_trip*0.99);
            T := min(T,T_crit);

            // calculate new RES_p
            tau := T_crit/T;
            T_theta := max((1 - T/T_crit), Modelica.Constants.small);
            RES_p := p_crit*exp(tau*sum(n[i]*T_theta^theta[i] for i in 1:nPressureSaturation)) - p;
          end while;
          // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_p total iteration steps " + String(iter), "printlog.txt");
          assert(iter<iter_max, "Ancillary.saturationTemperature_p did not converge, input was p=" + String(p));

        elseif (p<=p_trip) then
          T := T_trip;
        elseif (p>=p_crit) then
          T := T_crit;
        else
          assert(false, "Ancillary.saturationTemperature_p: this should not happen, check p");
        end if;

          // this is an iterative backward function
          // the corresponding ancillary forward function is saturationPressure(T)

          annotation (inverse(p=saturationPressure_T(T=T)));
        end saturationTemperature_p;

        function saturationTemperature_d
        "ancillary iterative function: calculate saturation temperature for a given density by iterating the ancillary function"

          input Modelica.SIunits.Density d;
          output Modelica.SIunits.Temperature T;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant SpecificHeatCapacity R=Modelica.Constants.R/MM
          "specific gas constant";
          constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;

          constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;

          constant Density dv_trip = Ancillary.dewDensity_T(T_trip);
          constant Density dl_trip = Ancillary.bubbleDensity_T(T_trip);

          Temperature T1=0.98*T_trip;
          Temperature T2=T_crit;
          Temperature T3;
          Temperature T4;

          Density R1 "residual of T1";
          Density R2 "residual of T2";
          Density R3 "residual of T3";
          Density R4= Modelica.Constants.inf "residual of T4";

          constant Real tolerance=1e-6 "relative tolerance for RES_d";
          Integer iter=0;
          constant Integer iter_max = 200;

        algorithm
          // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_d, d=" + String(d), "printlog.txt");

          if (d<d_crit) and (d>dv_trip) then
            // Modelica.Utilities.Streams.print("d<d_crit: vapour side", "printlog.txt");
            R1 := Ancillary.dewDensity_T(T1)-d;
            R2 := d_crit-d;
            if (R1*R2<0) then
              while (abs(R4/d)>tolerance) and (iter<iter_max) and (abs(T1-T2)>tolerance) loop
                iter:=iter+1;
                T3 := (T1+T2)/2;
                R3 := Ancillary.dewDensity_T(T3)-d;
                // calculate better T from Ridder's method
                T4  := T3 + (T3 - T1)*sign(R1-R2)*R3/sqrt(R3*R3 - R1*R2);
                R4 := Ancillary.dewDensity_T(T4)-d;
                // Modelica.Utilities.Streams.print("Ridders' method: current residuals: R1=" + String(R1) + ", R2=" + String(R2) + ", R3=" + String(R3) + ", R4=" + String(R4), "printlog.txt");
                if (R4*R3<=0) then
                  T1 := T3;
                  R1 := R3;
                  T2 := T4;
                  R2 := R4;
                else
                  if (R4*R1<0) then
                    T2 := T4;
                    R2 := R4;
                  elseif (R4*R2<0) then
                    T1 := T4;
                    R1 := R4;
                  else
                    assert(false, "Ancillary.saturationTemperature_d (vapour side): this should not happen");
                  end if;
                end if;
                // Modelica.Utilities.Streams.print("Ridders' method: new brackets T1=" + String(T1) + " and T2=" + String(T2), "printlog.txt");
              end while;
              assert(iter<iter_max, "saturationTemperature_d_vap did not converge, input was d_vap=" + String(d));
              // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_d_vap total iteration steps " + String(iter) + " for d_vap=" + String(d), "printlog.txt");
              // Modelica.Utilities.Streams.print(" ", "printlog.txt");
              T := T4;
            else
              if (abs(R1/d)<tolerance) then
                T:= T1;
              elseif (abs(R2/d)<tolerance) then
                T:=T2;
              else
                assert(false, "Ancillary.saturationTemperature_d (vapour side): T1=" + String(T1) + " and T2=" + String(T2) + " did not bracket the root");
              end if;
            end if;

          elseif (d>d_crit) and (d<dl_trip) then
            // Modelica.Utilities.Streams.print("d>d_crit: liquid side", "printlog.txt");
            R1 := Ancillary.bubbleDensity_T(T1)-d;
            R2 := d_crit-d;
            if (R1*R2<0) then
              while (abs(R4/d)>tolerance) and (iter<iter_max) and (abs(T1-T2)>tolerance) loop
                iter:=iter+1;
                T3 := (T1+T2)/2;
                R3 := Ancillary.bubbleDensity_T(T3)-d;
                // calculate better T from Ridder's method
                T4  := T3 + (T3 - T1)*sign(R1-R2)*R3/sqrt(R3*R3 - R1*R2);
                R4 := Ancillary.bubbleDensity_T(T4)-d;
                // Modelica.Utilities.Streams.print("Ridders' method: current temperatures: T1=" + String(T1,significantDigits=12) + ", T2=" + String(T2,significantDigits=12) + ", T3=" + String(T3,significantDigits=12) + ", T4=" + String(T4,significantDigits=12), "printlog.txt");
                // Modelica.Utilities.Streams.print("Ridders' method: current    residuals: R1=" + String(R1) + ", R2=" + String(R2) + ", R3=" + String(R3) + ", R4=" + String(R4), "printlog.txt");
                if (R4*R3<=0) then
                  T1 := T3;
                  R1 := R3;
                  T2 := T4;
                  R2 := R4;
                else
                  if (R4*R1<0) then
                    T2 := T4;
                    R2 := R4;
                  elseif (R4*R2<0) then
                    T1 := T4;
                    R1 := R4;
                  else
                    assert(false, "Ancillary.saturationTemperature_d (liquid side): this should not happen");
                  end if;
                end if;
                // Modelica.Utilities.Streams.print("Ridders' method: new brackets T1=" + String(T1) + " and T2=" + String(T2), "printlog.txt");
              end while;
              assert(iter<iter_max, "saturationTemperature_d_liq did not converge, remaining residuum is R4=" + String(R4) + ", input was d_liq=" + String(d));
              // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_d_liq total iteration steps " + String(iter) + " for d_liq=" + String(d), "printlog.txt");
              // Modelica.Utilities.Streams.print(" ", "printlog.txt");
              T := T4;
            else
              if (abs(R1/d)<tolerance) then
                T:= T1;
              elseif (abs(R2/d)<tolerance) then
                T:=T2;
              else
                assert(false, "Ancillary.saturationTemperature_d (liquid side): T1=" + String(T1) + " and T2=" + String(T2) + " did not bracket the root");
              end if;
            end if;

          elseif (d>=dl_trip) or (d<=dv_trip) then
            T := T_trip;
          else
            T := T_crit;
          end if;

        end saturationTemperature_d;

        function meltingTemperature_p
        "ancillary iterative function: calculate melting temperature for a given pressure by iterating the ancillary function"
          input AbsolutePressure p;
          output Temperature T;

      protected
          constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
          constant Temperature T_max=fluidLimits.TMAX;

          constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
          constant AbsolutePressure p_max=fluidLimits.PMAX;

          Temperature T1=0.98*T_trip "low temperature";
          Temperature T2=T_max "high temperature";
          Temperature T3 "intermediate temperature";
          Temperature T4 "new temperature";

          Real R1 "residual of T1";
          Real R2 "residual of T2";
          Real R3 "residual of T3";
          Real R4= Modelica.Constants.inf "residual of T4";

          constant Real tolerance=1e-6 "relative tolerance for RES_p";
          Integer iter=0;
          constant Integer iter_max = 200;

        algorithm
          // Modelica.Utilities.Streams.print("Ancillary.meltingTemperature_p, p=" + String(p), "printlog.txt");

          if (p<p_max) and (p>p_trip) then
            R1 := Ancillary.meltingPressure_T(T1)-p;
            R2 := Ancillary.meltingPressure_T(T2)-p;
            if (R1*R2<0) then
              while (abs(R4/p)>tolerance) and (iter<iter_max) and (abs(T1-T2)>tolerance) loop
                iter:=iter+1;
                T3 := (T1+T2)/2;
                R3 := Ancillary.meltingPressure_T(T3)-p;
                // calculate better T from Ridder's method
                T4  := T3 + (T3 - T1)*sign(R1-R2)*R3/sqrt(R3*R3 - R1*R2);
                R4 := Ancillary.meltingPressure_T(T4)-p;
                // Modelica.Utilities.Streams.print("Ridders' method: current residuals: R1=" + String(R1) + ", R2=" + String(R2) + ", R3=" + String(R3) + ", R4=" + String(R4), "printlog.txt");
                if (R4*R3<=0) then
                  T1 := T3;
                  R1 := R3;
                  T2 := T4;
                  R2 := R4;
                else
                  if (R4*R1<0) then
                    T2 := T4;
                    R2 := R4;
                  elseif (R4*R2<0) then
                    T1 := T4;
                    R1 := R4;
                  else
                    assert(false, "Ancillary.meltingTemperature_p: this should not happen");
                  end if;
                end if;
                // Modelica.Utilities.Streams.print("Ridders' method: new brackets T1=" + String(T1) + " and T2=" + String(T2), "printlog.txt");
              end while;
              assert(iter<iter_max, "meltingTemperature_p did not converge, input was p=" + String(p));
              // Modelica.Utilities.Streams.print("Ancillary.meltingTemperature_p total iteration steps " + String(iter) + " for p=" + String(p), "printlog.txt");
              // Modelica.Utilities.Streams.print(" ", "printlog.txt");
              T := T4;
            else
              if (abs(R1/p)<tolerance) then
                T:= T1;
              elseif (abs(R2/p)<tolerance) then
                T:=T2;
              else
                assert(false, "Ancillary.meltingTemperature_p: T1=" + String(T1) + " and T2=" + String(T2) + " did not bracket the root");
              end if;
            end if;

          elseif (p<=p_trip) then
            T := T_trip;
          else
            T := T_max;
          end if;

          annotation (inverse(p=meltingPressure_T(T=T)));
        end meltingTemperature_p;

        function saturationTemperature_h_liq
        "ancillary iterative function: calculate saturation temperature for a given enthalpy by iterating the ancillary function"

          input Modelica.SIunits.SpecificEnthalpy h;
          output Modelica.SIunits.Temperature T;

      protected
          constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant SpecificEnthalpy h_crit=fluidConstants[1].HCRIT0;

          constant Density dl_trip = Ancillary.bubbleDensity_T(T_trip);
          constant EoS.HelmholtzDerivs fl_trip = EoS.setHelmholtzDerivsFirst(d=dl_trip, T=T_trip);
          constant SpecificEnthalpy hl_trip = EoS.h(fl_trip);

          Temperature T1=0.98*T_trip;
          Temperature T2=T_crit;
          Temperature T3;
          Temperature T4;

          Density d1;
          Density d2;
          Density d3;
          Density d4;

          EoS.HelmholtzDerivs f1;
          EoS.HelmholtzDerivs f2;
          EoS.HelmholtzDerivs f3;
          EoS.HelmholtzDerivs f4;

          SpecificEnthalpy R1 "residual of T1";
          SpecificEnthalpy R2 "residual of T2";
          SpecificEnthalpy R3 "residual of T3";
          SpecificEnthalpy R4= Modelica.Constants.inf "residual of T4";

          constant Real tolerance=1e-6 "relative tolerance for RES";
          Integer iter=0;
          constant Integer iter_max = 200;

        algorithm
          // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_h, h=" + String(h), "printlog.txt");
          // assert(h<=h_crit+tolerance,"bubble enthalpy cannot be higher than critical enthalpy, invalid input");

          if (h<0.98*h_crit) and (h>hl_trip) then
            // Modelica.Utilities.Streams.print("h<h_crit: liquid side", "printlog.txt");
            R1 := hl_trip-h;
            R2 := h_crit-h;
            if (R1*R2<0) then
              while (abs(R4/h)>tolerance) and (iter<iter_max) and (abs(T1-T2)>tolerance) loop
                iter:=iter+1;
                T3 := (T1+T2)/2;
                d3 := Ancillary.bubbleDensity_T(T3);
                f3 := EoS.setHelmholtzDerivsFirst(d=d3,T=T3);
                R3 := EoS.h(f3)-h;
                // caclutate better T from Ridder's method
                T4  := T3 + (T3 - T1)*sign(R1-R2)*R3/sqrt(R3*R3 - R1*R2);
                d4 := Ancillary.bubbleDensity_T(T4);
                f4 := EoS.setHelmholtzDerivsFirst(d=d4,T=T4);
                R4 := EoS.h(f4)-h;
                // Modelica.Utilities.Streams.print("Ridders' method: current residuals: R1=" + String(R1) + ", R2=" + String(R2) + ", R3=" + String(R3) + ", R4=" + String(R4), "printlog.txt");
                if (R4*R3<=0) then
                  T1 := T3;
                  R1 := R3;
                  T2 := T4;
                  R2 := R4;
                else
                  if (R4*R1<0) then
                    T2 := T4;
                    R2 := R4;
                  elseif (R4*R2<0) then
                    T1 := T4;
                    R1 := R4;
                  else
                    assert(false, "Ancillary.saturationTemperature_h (liquid side): this should not happen");
                  end if;
                end if;
                // Modelica.Utilities.Streams.print("Ridders' method: new brackets T1=" + String(T1) + " and T2=" + String(T2), "printlog.txt");
              end while;
              assert(iter<iter_max, "saturationTemperature_h_liq did not converge, input was h_liq=" + String(h));
              // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_h_liq total iteration steps " + String(iter) + " for h_liq=" + String(h), "printlog.txt");
              // Modelica.Utilities.Streams.print(" ", "printlog.txt");
              T := T4;
            else
              if (abs(R1/h)<tolerance) then
                T:= T1;
              elseif (abs(R2/h)<tolerance) then
                T:=T2;
              else
                assert(false, "Ancillary.saturationTemperature_h (liquid side): T1=" + String(T1) + " and T2=" + String(T2) + " did not bracket the root");
              end if;
            end if;

          elseif (h>=0.98*h_crit) then
            T := T_crit;
          elseif (h<=hl_trip) then
            T := T_trip;
          else
            assert(false, "Ancillary.saturationTemperature_h (liquid side): this should also not happen");
          end if;

        end saturationTemperature_h_liq;

        function saturationTemperature_s_liq
        "ancillary iterative function: calculate saturation temperature for a given entropy by iterating the ancillary function"

          input Modelica.SIunits.SpecificEntropy s;
          output Modelica.SIunits.Temperature T;

      protected
          constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant SpecificEntropy s_crit=fluidConstants[1].SCRIT0;

          constant Density dl_trip = Ancillary.bubbleDensity_T(T_trip);
          constant EoS.HelmholtzDerivs fl_trip = EoS.setHelmholtzDerivsFirst(d=dl_trip, T=T_trip);
          constant SpecificEntropy sl_trip = EoS.s(fl_trip);

          Temperature T1=0.98*T_trip;
          Temperature T2=T_crit;
          Temperature T3;
          Temperature T4;

          Density d1;
          Density d2;
          Density d3;
          Density d4;

          EoS.HelmholtzDerivs f1;
          EoS.HelmholtzDerivs f2;
          EoS.HelmholtzDerivs f3;
          EoS.HelmholtzDerivs f4;

          SpecificEntropy R1 "residual of T1";
          SpecificEntropy R2 "residual of T2";
          SpecificEntropy R3 "residual of T3";
          SpecificEntropy R4= Modelica.Constants.inf "residual of T4";

          constant Real tolerance=1e-6 "relative tolerance for RES";
          Integer iter=0;
          constant Integer iter_max = 200;

        algorithm
          // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_h, h=" + String(h), "printlog.txt");
          // assert(s<=s_crit+tolerance,"bubble entropy cannot be higher than critical entropy, invalid input");

          if (s<0.98*s_crit) and (s>sl_trip) then
            // Modelica.Utilities.Streams.print("s=" + String(s,significantDigits=9) + "<s_crit: liquid side", "printlog.txt");
            R1 := sl_trip-s;
            R2 := s_crit-s;
            if (R1*R2<0) then
              while (abs(R4/s)>tolerance) and (iter<iter_max) and (abs(T1-T2)>tolerance) loop
                iter:=iter+1;
                T3 := (T1+T2)/2;
                d3 := Ancillary.bubbleDensity_T(T3);
                f3 := EoS.setHelmholtzDerivsFirst(d=d3,T=T3);
                R3 := EoS.s(f3)-s;
                // caclutate better T from Ridder's method
                T4  := T3 + (T3 - T1)*sign(R1-R2)*R3/sqrt(R3*R3 - R1*R2);
                d4 := Ancillary.bubbleDensity_T(T4);
                f4 := EoS.setHelmholtzDerivsFirst(d=d4,T=T4);
                R4 := EoS.s(f4)-s;
                // Modelica.Utilities.Streams.print("Ridders' method: current residuals: R1=" + String(R1) + ", R2=" + String(R2) + ", R3=" + String(R3) + ", R4=" + String(R4), "printlog.txt");
                if (R4*R3<=0) then
                  T1 := T3;
                  R1 := R3;
                  T2 := T4;
                  R2 := R4;
                else
                  if (R4*R1<0) then
                    T2 := T4;
                    R2 := R4;
                  elseif (R4*R2<0) then
                    T1 := T4;
                    R1 := R4;
                  else
                    assert(false, "Ancillary.saturationTemperature_s (liquid side): this should not happen");
                  end if;
                end if;
                // Modelica.Utilities.Streams.print("Ridders' method: new brackets T1=" + String(T1) + " and T2=" + String(T2), "printlog.txt");
              end while;
              assert(iter<iter_max, "saturationTemperature_s_liq did not converge, input was s_liq=" + String(s));
              // Modelica.Utilities.Streams.print("Ancillary.saturationTemperature_s_liq total iteration steps " + String(iter) + " for s_liq=" + String(s), "printlog.txt");
              // Modelica.Utilities.Streams.print(" ", "printlog.txt");
              T := T4;
            else
              if (abs(R1/s)<tolerance) then
                T:= T1;
              elseif (abs(R2/s)<tolerance) then
                T:=T2;
              else
                assert(false, "Ancillary.saturationTemperature_s (liquid side): T1=" + String(T1) + " and T2=" + String(T2) + " did not bracket the root");
              end if;
            end if;

          elseif (s>=0.98*s_crit) then
            T := T_crit;
          elseif (s<=sl_trip) then
            T := T_trip;
          else
            assert(false, "Ancillary.saturationTemperature_s (liquid side): this should also not happen");
          end if;

        end saturationTemperature_s_liq;

        function density_pT_Soave "solve Redlich-Kwong-Soave for density"
          input Temperature T;
          input AbsolutePressure p;
          input AbsolutePressure psat=Ancillary.saturationPressure_T(T=T);
          output Density d;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant SpecificHeatCapacity R=Modelica.Constants.R/MM
          "specific gas constant";
          constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
          constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;

          // RKS: Redlich-Kwong-Soave (see Soave 1979)
          constant Real omega = fluidConstants[1].acentricFactor;
          constant Real m = 0.480 + 1.574*omega - 0.176*omega^2;
          Real a = 0.42747*R^2*T_crit^2/p_crit*(1 + m*(1 - sqrt(T/T_crit)))^2;
          Real b = 0.08664*R*T_crit/p_crit;
          Real A = a*p/(R^2*T^2);
          Real B = b*p/(R*T);
          Real r = (A-B-B^2)-1/3;
          Real q = -2/27 + 1/3*(A-B-B^2) - A*B;
          Real D = (r/3)^3 + (q/2)^2 "discriminant";
          Real u;
          Real u3;
          Real Y1;
          Real Y2;
          Real Y3;
          Real Theta;
          Real phi;
          import Modelica.Constants.pi;

        algorithm
          // get density start value from Redlich-Kwong-Soave (see Span 2000, section "3.3.1 Calculations based on pT" )
          // Modelica.Utilities.Streams.print("", "printlog.txt");
          // Modelica.Utilities.Streams.print("Redlich-Kwong-Soave called with p=" + String(p) + " and T=" +String(T), "printlog.txt");
          // Modelica.Utilities.Streams.print("RKS discriminant D=" + String(D) + " and r=" + String(r), "printlog.txt");
          if (D >= 0) then
            u3 := -q/2 + sqrt(D);
            u := sign(u3)*abs(u3)^(1/3);
            Y1 := u-r/(3*u);
            // Modelica.Utilities.Streams.print("RKS has one root (Y1=" + String(Y1) + ")", "printlog.txt");
            d := p/(R*T*(Y1+1/3));
          elseif ((abs(D) < 1e-8) and (abs(r) < 1e-3)) then
            // Modelica.Utilities.Streams.print("close to critical region, use critical density");
            d := d_crit;
          else
            // case D<0
            Theta := sqrt(-r^3/27);
            phi := acos(-q/(2*Theta));
            Y1 := 2*Theta^(1/3)*cos(phi/3);
            Y2 := 2*Theta^(1/3)*cos(phi/3+2*pi/3);
            Y3 := 2*Theta^(1/3)*cos(phi/3+4*pi/3);
            // Modelica.Utilities.Streams.print("RKS has three possible roots(Y1=" + String(Y1) + ", Y2=" + String(Y2) + ", Y3=" + String(Y3), "printlog.txt");
            if (T <= T_crit) then
              // Modelica.Utilities.Streams.print("T<T_crit: multiple roots due to phase boundary", "printlog.txt");
              // Modelica.Utilities.Streams.print("d(Y1)=" + String(p/(R*T*(Y1+1/3))) + ", d(Y2)=" + String(p/(R*T*(Y2+1/3))) + ", d(p/(R*T*(Y3+1/3)))=" + String(Y3), "printlog.txt");
              if (p > psat) then
                // liquid, use smallest Y
                Y1 := min({Y1,Y2,Y3});
                d := p/(R*T*(Y1+1/3));
              elseif (p < psat) then
                // vapor, use largest Y
                Y1 := max({Y1,Y2,Y3});
                d := p/(R*T*(Y1+1/3));
              else
                // this is very unlikely, but not impossible
                assert(p <> psat, "Ancillary.density_pT_Soave error: pressure equals saturation pressure");
              end if;
            else
              // Modelica.Utilities.Streams.print("T>T_crit: multiple roots can occur, but two of the roots result in negative densities", "printlog.txt");
              // Modelica.Utilities.Streams.print("d(Y1)=" + String(p/(R*T*(Y1+1/3))) + ", d(Y2)=" + String(p/(R*T*(Y2+1/3))) + ", d(Y3)=" + String(p/(R*T*(Y3+1/3))), "printlog.txt");
              d := max({p/(R*T*(Y1+1/3)), p/(R*T*(Y2+1/3)), p/(R*T*(Y3+1/3))});
            end if;
          end if;
          // Modelica.Utilities.Streams.print("RKS finished, d=" + String(d), "printlog.txt");

        end density_pT_Soave;

        function temperature_pd_Waals
          input AbsolutePressure p;
          input Density d;
          output Temperature T;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant SpecificHeatCapacity R=Modelica.Constants.R/MM
          "specific gas constant";
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;

          // van der Waals, as described by Span (2000)
          Real a = 27/64 * (R*R*T_crit*T_crit) /p_crit
          "correction for attraction";
          Real b = (R*T_crit)/(8*p_crit) "correction for volume";

        algorithm
          // T := (p+a/v^2)*(v-b)/R;
          T := (p+a*d^2)*(1/d-b)/R;
          // Modelica.Utilities.Streams.print("van der Waals finished, T=" + String(T), "printlog.txt");

        end temperature_pd_Waals;

        function pressure_dT_Waals
          input Density d;
          input Temperature T;
          output AbsolutePressure p;

      protected
          constant MolarMass MM = fluidConstants[1].molarMass;
          constant SpecificHeatCapacity R=Modelica.Constants.R/MM
          "specific gas constant";
          constant Temperature T_crit=fluidConstants[1].criticalTemperature;
          constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;

          // van der Waals, as described by Span (2000)
          Real a = 27/64 * (R*R*T_crit*T_crit) /p_crit
          "correction for attraction";
          Real b = (R*T_crit)/(8*p_crit) "correction for volume";

        algorithm
          // p := R*T/(v-b) - a/v^2;
          p := R*T/(1/d-b) - a*d^2;
          // Modelica.Utilities.Streams.print("van der Waals finished, p=" + String(p), "printlog.txt");

        end pressure_dT_Waals;
      end Ancillary;

      package Transport

        record ThermalConductivityCoefficients
          // dilute gas / zero density terms
          constant Temperature reducingTemperature_0=1 "reducing temperature";
          constant Real reducingThermalConductivity_0=1 "usually unity";
          constant Real[:,2] lambda_0_num_coeffs = fill(0.0, 0, 2)
          "coeffs for dilute contribution numerator";
          constant Real[:,2] lambda_0_den_coeffs = fill(0.0, 0, 2)
          "coeffs for dilute contribution denominator";

          // residual / background terms
          constant Temperature reducingTemperature_residual=1
          "reducing temperature";
          constant MolarVolume reducingMolarVolume_residual
          "reducing molar volume";
          constant Real reducingThermalConductivity_residual=1 "usually unity";
          constant Real[:,4] lambda_r_coeffs = fill(0.0, 0, 4)
          "coeffs for residual contribution";

          // critical enhancement terms
          constant Real nu=0.63 "universal exponent";
          constant Real gamma=1.239 "universal exponent";
          constant Real R0=1.03 "universal amplitude";
          constant Real z=0.063 "universal exponent, used for viscosity";
          constant Real c=1 "constant in viscosity, often set to 1";

          constant Real xi_0 "amplitude";
          constant Real Gamma_0 "amplitude";
          constant Real qd_inverse "modified effective cutoff parameter";
          constant Temperature T_ref "reference temperature";
        end ThermalConductivityCoefficients;

        record DynamicViscosityCoefficients
          import HelmholtzMedia;

          constant DynamicViscosityModel dynamicViscosityModel;
          constant CollisionIntegralModel collisionIntegralModel;

          // collision integral S_mathfrak
          constant Temperature epsilon_kappa "Lennard-Jones energy parameter";
          constant Real sigma "Lennard-Jones size parameter";
          constant Real[:,2] a = fill(0.0, 0, 2)
          "coefficients for collision integral";

          // zero density dependence, via Chapman-Enskog-Theory
          constant Real[:,2] CET = fill(0.0, 0, 2);
          constant Temperature reducingTemperature_0=1 "reducing temperature";
          constant Real reducingViscosity_0=1 "reducing viscosity";

          // initial density dependence, via Rainwater-Friend-Theory
          constant Temperature reducingTemperature_1=1 "reducing temperature";
          constant Real reducingViscosity_1=1 "reducing viscosity";
          constant Real[:,2] b = fill(0.0, 0, 2)
          "coefficients for second viscosity virial coefficent B";

          // residual / high density viscosity contribution
          constant Real[:,1] c =  fill(0.0, 0, 1)
          "coefficients for residual viscosity contribution in VS2 model";
          constant Temperature reducingTemperature_residual=1
          "reducing temperature";
          constant MolarVolume reducingMolarVolume_residual=1
          "reducing molar volume";
          constant Real reducingViscosity_residual=1 "reducing viscosity";
          constant Real[:,2] g = fill(0.0, 0, 2) "close-packed density delta_0";
          constant Real[:,5] e = fill(0.0, 0, 5) "simple poly";
          constant Real[:,5] nu_po = fill(0.0, 0, 5)
          "  numerator of rational poly";
          constant Real[:,5] de_po = fill(0.0, 0, 5)
          "denominator of rational poly";
          // constant Real[:,5] nu_ex = fill(0.0, 0, 5) "  numerator of exponential";
          // constant Real[:,5] de_ex = fill(0.0, 0, 5) "denominator of exponential";
        end DynamicViscosityCoefficients;

        record SurfaceTensionCoefficients
          Real[:,2] coeffs = fill(0.0, 0, 2) "sigma0 and n";
        end SurfaceTensionCoefficients;

        function dynamicViscosity_dilute
        "Returns dynamic Viscosity dilute contribution"
          input ThermodynamicState state;
          output DynamicViscosity eta_0;

      protected
          DynamicViscosityModel dynamicViscosityModel=dynamicViscosityCoefficients.dynamicViscosityModel;
          CollisionIntegralModel collisionIntegralModel=dynamicViscosityCoefficients.collisionIntegralModel;

          Temperature T_crit=fluidConstants[1].criticalTemperature;
          Temperature T_red_0=dynamicViscosityCoefficients.reducingTemperature_0;
          Temperature T_red_residual=dynamicViscosityCoefficients.reducingTemperature_residual;
          Real T_star "reduced temperature";
          Real tau "reduced temperature";

          Real[size(dynamicViscosityCoefficients.a, 1),2] a=dynamicViscosityCoefficients.a;

          Real[size(dynamicViscosityCoefficients.CET, 1),2] CET=dynamicViscosityCoefficients.CET; // Chapman-Enskog-Term
          Real Omega=0
          "reduced effective cross section / Omega collision integral";
          Real sigma=dynamicViscosityCoefficients.sigma;

          Real eta_red_0=dynamicViscosityCoefficients.reducingViscosity_0;

        algorithm
          // collision integral
          if (collisionIntegralModel == CollisionIntegralModel.CI0) then
            T_star := (state.T/dynamicViscosityCoefficients.epsilon_kappa);
            Omega := 1.16145/T_star^0.14874 + 0.52487*exp(-0.77320*T_star) + 2.16178*exp(-2.43787*T_star);
          elseif (collisionIntegralModel == CollisionIntegralModel.CI1) then
            T_star := Modelica.Math.log(state.T/dynamicViscosityCoefficients.epsilon_kappa);
            Omega := exp(sum(a[i, 1]*(T_star)^a[i, 2] for i in 1:size(a, 1)));
          elseif (collisionIntegralModel == CollisionIntegralModel.CI2) then
            T_star := (dynamicViscosityCoefficients.epsilon_kappa/state.T)^(1/3);
            Omega := 1/(sum(a[i, 1]*(T_star)^(4-i) for i in 1:size(a, 1)));
          end if;

          // dilute gas (zero density) contribution
          // using the Chapman-Enskog-Term and the collision integral Omega
          if ((dynamicViscosityModel == DynamicViscosityModel.VS1)
          or  (dynamicViscosityModel == DynamicViscosityModel.VS1_alternative)) then
            tau := state.T/T_red_0;
            // first term is the Chapman-Enskog-Term
            eta_0 := CET[1, 1]*sqrt(tau)/(sigma^2*Omega);
            // possibly further empirical terms
            eta_0 := eta_0 + sum(CET[i, 1]*(tau)^CET[i, 2] for i in 2:size(CET, 1));
          elseif (dynamicViscosityModel == DynamicViscosityModel.VS2) then
            eta_0 := CET[1, 1]*state.T^CET[1,2]/(sigma^2*Omega);
          elseif (dynamicViscosityModel == DynamicViscosityModel.VS4) then
          end if;
          eta_0 := eta_0*eta_red_0;

          /* // following lines are for debugging only
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("        T = " + String(state.T));
  Modelica.Utilities.Streams.print("   T_star = " + String(T_star));
  Modelica.Utilities.Streams.print("      tau = " + String(tau));
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("    Omega = " + String(Omega));
  Modelica.Utilities.Streams.print("    eta_0 = " + String(eta_0));
  Modelica.Utilities.Streams.print("===========================================");
  */

          annotation (Documentation(info="<html>
<p>
This model is identical to the RefProp VS1 or VS2 model.

The viscosity is split into three contributions: 
zero density (dilute gas) viscosity eta_0, 
initial density contribution eta_1
and residual contribution eta_r.

This allows to develop functions for each contribution seperately.
The so called background viscosity is the sum of initial and residual viscosity.

At the critical point and a small region around the critical point, the viscosity is enhanced. 
As this critical enhancement is small, it is neglected here.

Special thanks go to Eric W. Lemmon for answering all my emails 
and programming a special version of RefProp that outputs also intermediate values.

</p>

<dl>
<dt> Lemmon, Eric W.; Huber, M. L. and McLinden, M. O.</dt>
<dd> <b>NIST Standard Reference Database 23: Reference Fluid Thermodynamic and Transport Properties - REFPROP. 9.0</b><br>
     National Institute of Standards and Technology, Standard Reference Data Program. Gaithersburg<br>
     URL: <a href=\"http://www.nist.gov/srd/nist23.cfm\">http://www.nist.gov/srd/nist23.cfm</a>
</dd>
<dt>Vogel, E.; Küchenmeister, C. and Birch, E.</dt>
<dd> <b>Reference correlation of the viscosity of propane</b>.<br>
     Journal of Thermophysics (1998) 10, 417-426.<br>
     DOI: <a href=\"http://dx.doi.org/10.1007/BF01133538\">10.1007/BF01133538</a>
</dd>
</dl>
</html>"));
        end dynamicViscosity_dilute;

        function dynamicViscosity_initial
        "Returns dynamic Viscosity initial contribution"
          input ThermodynamicState state;
          output DynamicViscosity eta_1;

      protected
          DynamicViscosityModel dynamicViscosityModel=dynamicViscosityCoefficients.dynamicViscosityModel;
          CollisionIntegralModel collisionIntegralModel=dynamicViscosityCoefficients.collisionIntegralModel;

          Temperature T_crit=fluidConstants[1].criticalTemperature;
          Temperature T_red_0=dynamicViscosityCoefficients.reducingTemperature_0;
          Temperature T_red_residual=dynamicViscosityCoefficients.reducingTemperature_residual;
          Real T_star "reduced temperature";
          Real tau "reduced temperature";

          MolarMass MM = fluidConstants[1].molarMass;
          Real dm=state.d/(1000*MM) "molar density in mol/l";     // 1 m3=1000 l
          //Real dm_crit=d_crit/(1000*MM) "molar density in mol/l"; // 1 m3=1000 l

          Real[size(dynamicViscosityCoefficients.b, 1),2] b=dynamicViscosityCoefficients.b;
          Real B_star=0 "reduced second viscosity virial coefficient";
          Real B=0 "second viscosity virial coefficient, l/mol";
          Real sigma=dynamicViscosityCoefficients.sigma;

          Real eta_red_1=dynamicViscosityCoefficients.reducingViscosity_1;

        algorithm
          // inital density contribution
          if ((dynamicViscosityModel == DynamicViscosityModel.VS1)
          or  (dynamicViscosityModel == DynamicViscosityModel.VS1_alternative)
          or  (dynamicViscosityModel == DynamicViscosityModel.VS4)) then
            // use the second viscosity virial coefficient B according to Rainwater and Friend theory
            T_star := (state.T/dynamicViscosityCoefficients.epsilon_kappa);
            B_star := sum(b[i, 1]*T_star^b[i, 2] for i in 1:size(b, 1));
            B := B_star*0.6022137*sigma^3;
            eta_1 := dynamicViscosity_dilute(state)*B*dm;
          elseif (dynamicViscosityModel == DynamicViscosityModel.VS2) then
            eta_1 := dm * (b[1,1] + b[2,1]*(b[3,1]-log(state.T/b[4,1]))^2);
          end if;
          eta_1 := eta_1*eta_red_1;

          /* // following lines are for debugging only
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("        T = " + String(state.T));
  Modelica.Utilities.Streams.print("   T_star = " + String(T_star));
  Modelica.Utilities.Streams.print("      tau = " + String(tau));
  Modelica.Utilities.Streams.print("        d = " + String(state.d));
  Modelica.Utilities.Streams.print("       dm = " + String(dm));
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("   B_star = " + String(B_star));
  Modelica.Utilities.Streams.print("        B = " + String(B));
  Modelica.Utilities.Streams.print("    eta_1 = " + String(eta_1));
  Modelica.Utilities.Streams.print("===========================================");
  */

          annotation (Documentation(info="<html>
<p>
This model is identical to the RefProp VS1 or VS2 model.

The viscosity is split into three contributions: 
zero density (dilute gas) viscosity eta_0, 
initial density contribution eta_1
and residual contribution eta_r.

This allows to develop functions for each contribution seperately.
The so called background viscosity is the sum of initial and residual viscosity.

At the critical point and a small region around the critical point, the viscosity is enhanced. 
As this critical enhancement is small, it is neglected here.

Special thanks go to Eric W. Lemmon for answering all my emails 
and programming a special version of RefProp that outputs also intermediate values.

</p>

<dl>
<dt> Lemmon, Eric W.; Huber, M. L. and McLinden, M. O.</dt>
<dd> <b>NIST Standard Reference Database 23: Reference Fluid Thermodynamic and Transport Properties - REFPROP. 9.0</b><br>
     National Institute of Standards and Technology, Standard Reference Data Program. Gaithersburg<br>
     URL: <a href=\"http://www.nist.gov/srd/nist23.cfm\">http://www.nist.gov/srd/nist23.cfm</a>
</dd>
<dt>Vogel, E.; Küchenmeister, C. and Birch, E.</dt>
<dd> <b>Reference correlation of the viscosity of propane</b>.<br>
     Journal of Thermophysics (1998) 10, 417-426.<br>
     DOI: <a href=\"http://dx.doi.org/10.1007/BF01133538\">10.1007/BF01133538</a>
</dd>
</dl>
</html>"));
        end dynamicViscosity_initial;

        function dynamicViscosity_residual
        "Returns dynamic Viscosity residual contribution"
          input ThermodynamicState state;
          output DynamicViscosity eta_r;

      protected
          DynamicViscosityModel dynamicViscosityModel=dynamicViscosityCoefficients.dynamicViscosityModel;
          CollisionIntegralModel collisionIntegralModel=dynamicViscosityCoefficients.collisionIntegralModel;

          Temperature T_crit=fluidConstants[1].criticalTemperature;
          Temperature T_red_0=dynamicViscosityCoefficients.reducingTemperature_0;
          Temperature T_red_residual=dynamicViscosityCoefficients.reducingTemperature_residual;
          Real T_star "reduced temperature";
          Real tau "reduced temperature";

          MolarMass MM = fluidConstants[1].molarMass;
          Real dm=state.d/(1000*MM) "molar density in mol/l";     // 1 m3=1000 l
          //Real dm_crit=d_crit/(1000*MM) "molar density in mol/l"; // 1 m3=1000 l

          Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          Density d_red_residual=MM/dynamicViscosityCoefficients.reducingMolarVolume_residual;
          Real delta=0 "reduced density";
          Real delta_exp=0 "reduced density in exponential term";
          Real delta_0=0 "close packed density";

        Real[size(dynamicViscosityCoefficients.c, 1),1] c=dynamicViscosityCoefficients.c;

          Real[size(dynamicViscosityCoefficients.g, 1),2] g=dynamicViscosityCoefficients.g;
          Real[size(dynamicViscosityCoefficients.e, 1),5] e=dynamicViscosityCoefficients.e;
          Real[size(dynamicViscosityCoefficients.nu_po, 1),5] nu_po=dynamicViscosityCoefficients.nu_po;
          Real[size(dynamicViscosityCoefficients.de_po, 1),5] de_po=dynamicViscosityCoefficients.de_po;
          // Real[size(dynamicViscosityCoefficients.nu_ex,1),5] nu_ex=dynamicViscosityCoefficients.nu_ex;
          // Real[size(dynamicViscosityCoefficients.de_ex,1),5] de_ex=dynamicViscosityCoefficients.de_ex;

          Real visci=0 "RefProp      visci temporary variable";
          Real xnum=0 "RefProp   numerator temporary variable";
          Real xden=0 "RefProp denominator temporary variable";
          Real G=0 "RefProp temporary variable";
          Real H=0 "RefProp temporary variable";
          Real F=0 "RefProp temporary variable";

          Real eta_red_residual=dynamicViscosityCoefficients.reducingViscosity_residual;

        algorithm
          // residual contribution
          if ((dynamicViscosityModel == DynamicViscosityModel.VS1)
          or  (dynamicViscosityModel == DynamicViscosityModel.VS1_alternative)) then
            // use the reduced close-packed density delta_0,
            // a simple polynominal, a rational polynominal and an exponential term
            tau := state.T/T_red_residual;
            delta := state.d/d_red_residual;
            if (abs(d_red_residual - 1) > 0.001) then
              delta_exp := state.d/d_crit;
            else
              delta_exp := delta;
            end if;

            if (dynamicViscosityModel == DynamicViscosityModel.VS1) then
              // generalized RefProp algorithm, be careful with coeffs: they may differ from article
              delta_0 := sum(g[i, 1]*tau^g[i, 2] for i in 1:size(g, 1));
            elseif (dynamicViscosityModel == DynamicViscosityModel.VS1_alternative) then
              // alternative inverse form
              delta_0 := g[1, 1]/(1 + sum(g[i, 1]*tau^g[i, 2] for i in 2:size(g, 1)));
            end if;
            for i in 1:size(e, 1) loop
              visci := e[i, 1]*tau^e[i, 2]*delta^e[i, 3]*delta_0^e[i, 4]; // simple polynominal terms
              if (e[i, 5] > 0) then
                visci := visci*exp(-delta_exp^e[i, 5]);
              end if;
              eta_r := eta_r + visci;
            end for;

            for i in 1:size(nu_po, 1) loop
              // numerator of rational poly terms, RefProp algorithm
              xnum := xnum + (nu_po[i, 1]*tau^nu_po[i, 2]*delta^nu_po[i, 3]*delta_0^nu_po[i, 4]);
              if (nu_po[i, 5] > 0) then
                xnum := xnum*exp(-delta_exp^nu_po[i, 5]);
              end if;
            end for;
            for i in 1:size(de_po, 1) loop
              // denominator of rational poly terms, RefProp algorithm
              xden := xden + (de_po[i, 1]*tau^de_po[i, 2]*delta^de_po[i, 3]*delta_0^de_po[i, 4]);
              if (de_po[i, 5] > 0) then
                xden := xden*exp(-delta_exp^de_po[i, 5]);
              end if;
            end for;
            eta_r := eta_r + xnum/xden;
            // exponential terms not yet implemented!!

          elseif (dynamicViscosityModel == DynamicViscosityModel.VS2) then
            G := c[1,1] + c[2,1]/state.T;
          //H := sqrt(dm)*(dm-dm_crit)/dm_crit;
            H := sqrt(dm)*(dm- c[8,1])/c[8,1];
            F := G + (c[3,1] + c[4,1]*state.T^(-3/2))*dm^0.1
                   + (c[5,1] + c[6,1]/state.T + c[7,1]/state.T^2)*H;
            eta_r :=exp(F) - exp(G);

          elseif (dynamicViscosityModel == DynamicViscosityModel.VS4) then
            // not yet implemented!!
          end if;
          eta_r := eta_r*eta_red_residual;

          /* // following lines are for debugging only
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("        T = " + String(state.T));
  Modelica.Utilities.Streams.print("      tau = " + String(tau));
  Modelica.Utilities.Streams.print("        d = " + String(state.d));
  Modelica.Utilities.Streams.print("       dm = " + String(dm));
  Modelica.Utilities.Streams.print("    delta = " + String(delta));
  Modelica.Utilities.Streams.print("delta_exp = " + String(delta_exp));
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("  delta_0 = " + String(delta_0));
  Modelica.Utilities.Streams.print("     xnum = " + String(xnum) + " and xden = " + String(xden));
  Modelica.Utilities.Streams.print("    eta_r = " + String(eta_r));
  Modelica.Utilities.Streams.print("===========================================");
  */

          annotation (Documentation(info="<html>
<p>
This model is identical to the RefProp VS1 or VS2 model.

The viscosity is split into three contributions: 
zero density (dilute gas) viscosity eta_0, 
initial density contribution eta_1
and residual contribution eta_r.

This allows to develop functions for each contribution seperately.
The so called background viscosity is the sum of initial and residual viscosity.

At the critical point and a small region around the critical point, the viscosity is enhanced. 
As this critical enhancement is small, it is neglected here.

Special thanks go to Eric W. Lemmon for answering all my emails 
and programming a special version of RefProp that outputs also intermediate values.

</p>

<dl>
<dt> Lemmon, Eric W.; Huber, M. L. and McLinden, M. O.</dt>
<dd> <b>NIST Standard Reference Database 23: Reference Fluid Thermodynamic and Transport Properties - REFPROP. 9.0</b><br>
     National Institute of Standards and Technology, Standard Reference Data Program. Gaithersburg<br>
     URL: <a href=\"http://www.nist.gov/srd/nist23.cfm\">http://www.nist.gov/srd/nist23.cfm</a>
</dd>
<dt>Vogel, E.; Küchenmeister, C. and Birch, E.</dt>
<dd> <b>Reference correlation of the viscosity of propane</b>.<br>
     Journal of Thermophysics (1998) 10, 417-426.<br>
     DOI: <a href=\"http://dx.doi.org/10.1007/BF01133538\">10.1007/BF01133538</a>
</dd>
</dl>
</html>"));
        end dynamicViscosity_residual;

        function thermalConductivity_dilute
        "Return thermal conductivity dilute contribution"
          // depends on dynamicViscosity, specificHeatCapacityCp, specificHeatCapacityCv and dpdd=1/dddp
          input ThermodynamicState state;
          output ThermalConductivity lambda_0;

      protected
          MolarMass MM = fluidConstants[1].molarMass;
          Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          Density d_red_residual=fluidConstants[1].molarMass/
              thermalConductivityCoefficients.reducingMolarVolume_residual;
          Real delta "reduced density";

          Temperature T_red_0=thermalConductivityCoefficients.reducingTemperature_0;
          Real tau "reduced temperature";

          // coeffs for dilute contribution
          Integer nDilute_num = size(thermalConductivityCoefficients.lambda_0_num_coeffs,1);
          Real[nDilute_num, 2] A_num= thermalConductivityCoefficients.lambda_0_num_coeffs;
          Integer nDilute_den = size(thermalConductivityCoefficients.lambda_0_den_coeffs,1);
          Real[nDilute_den, 2] A_den= thermalConductivityCoefficients.lambda_0_den_coeffs;
          Real denom= 1;
          Real lambda_red_0=thermalConductivityCoefficients.reducingThermalConductivity_0;

          // interim variables for flagged algorithms
          constant Real eps = Modelica.Constants.eps;
          constant Real kilo = 1e3;
          EoS.HelmholtzDerivs f;
          Real cp0;  // ideal gas cp
          Real eta_0; // dilute contribution only

        algorithm
          // Modelica.Utilities.Streams.print("thermalConductivity_dilute: d = " + String(state.d) + " and T = " + String(state.T));
          tau := state.T/T_red_0;

          // numerator terms, check exponent
          if (A_num[nDilute_num,2]>-90) then
            lambda_0 := sum(A_num[i, 1]*tau^A_num[i, 2] for i in 1:nDilute_num);
          else
            lambda_0 := sum(A_num[i, 1]*tau^A_num[i, 2] for i in 1:nDilute_num-1);
            f := EoS.setHelmholtzDerivsSecond(d=state.d,T=state.T);
            cp0 := EoS.cp0(f);
            eta_0 := dynamicViscosity_dilute(state);
            // Modelica.Utilities.Streams.print("flagged algorithms, cp0=" + String(cp0) + " and eta_0=" + String(eta_0));
            if (abs(A_num[nDilute_num,2]+99)<eps) then
              Modelica.Utilities.Streams.print("thermalConductivity_dilute: flag 99");
              assert(false, "not yet implemented");
            elseif (abs(A_num[nDilute_num,2]+98)<eps) then
              Modelica.Utilities.Streams.print("thermalConductivity_dilute: flag 98");
              assert(false, "not yet implemented");
            elseif (abs(A_num[nDilute_num,2]+97)<eps) then
              Modelica.Utilities.Streams.print("thermalConductivity_dilute: flag 97");
              assert(false, "not yet implemented");
            elseif (abs(A_num[nDilute_num,2]+96)<eps) then
              // Modelica.Utilities.Streams.print("thermalConductivity_dilute: flag 96");
              // remember: RefProp uses molar units and g/mol!
              cp0 := cp0/f.R-2.5;
              lambda_0 := (lambda_0*cp0+15.0/4.0)*f.R*eta_0/kilo;
            end if;
          end if;

          if (nDilute_den>1) then
            // Modelica.Utilities.Streams.print("thermalConductivity_dilute: use denominator term");
            denom := sum(A_den[i, 1]*tau^A_den[i, 2] for i in 1:nDilute_den);
          else
            denom := 1;
          end if;

          lambda_0 := lambda_0/denom*lambda_red_0;
          // Modelica.Utilities.Streams.print(" lambda_0 = " + String(lambda_0));

        end thermalConductivity_dilute;

        function thermalConductivity_residual
        "Return thermal conductivity residual contribution"
          input ThermodynamicState state;
          output ThermalConductivity lambda_r;

      protected
          MolarMass MM = fluidConstants[1].molarMass;
          Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          Density d_red_residual=fluidConstants[1].molarMass/
              thermalConductivityCoefficients.reducingMolarVolume_residual;
          Real delta "reduced density";

          Temperature T_red_residual=thermalConductivityCoefficients.reducingTemperature_residual;
          Real tau "reduced temperature";

          // coeffs for residual contribution
          Integer nResidual = size(thermalConductivityCoefficients.lambda_r_coeffs, 1);
          Real[nResidual, 4] B= thermalConductivityCoefficients.lambda_r_coeffs;

          Real lambda_red_residual=thermalConductivityCoefficients.reducingThermalConductivity_residual;

        algorithm
          // residual contribution; RefProp uses the name background contribution
          tau := state.T/T_red_residual;
          delta := state.d/d_red_residual;
          lambda_r := sum((B[i, 1]*tau^B[i, 2])*(delta)^B[i, 3] for i in 1:nResidual);
          lambda_r := lambda_r*lambda_red_residual;

          /* // following lines are for debugging only
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("        d = " + String(state.d) + " and T = " + String(state.T));
  Modelica.Utilities.Streams.print(" lambda_r = " + String(lambda_r));
  Modelica.Utilities.Streams.print("===========================================");
  */

          annotation (Documentation(info="<html>
  <p>
The thermal conductivity (TC) is split into three parts: ideal gas TC lamda_0, residual TC lambda_r and critical TC enhancement lambda_c.
Sometimes the residual TC is again split into two parts.
This allows to develop functions for each contribution seperately.
The sum of ideal gas TC and residual TC is called background TC.
Ideal gas TC depends on Temperature only and can be modelled by a quadratic function.
Residual TC is also modeled by a polynominal.
At the critical point TC becomes infinite; TC is enhanced for a large region around the critical point.
The critical enhancement can be described by various alternative approaches.
Here, the simplified approach as suggested by Olchowy and Sengers is implemented.

Special thanks go to Eric W. Lemmon for answering all my emails 
and programming a special version of RefProp that outputs also intermediate values.

</p>
<dl>
<dt>Olchowy, G.A. and Sengers, J.V</dt>
<dd> <b>A simplified representation for the thermal conductivity of fluids in the critical region</b>.<br>
     International Journal of Thermophysics (1998) 10, 417-426.<br>
     DOI: <a href=\"http://dx.doi.org/10.1007/BF01133538\">10.1007/BF01133538</a>
</dd>
</dl>
</html>"));
        end thermalConductivity_residual;

        function thermalConductivity_critical
        "Return thermal conductivity critical enhancement"
          input ThermodynamicState state;
          output ThermalConductivity lambda_c;

      protected
          MolarMass MM = fluidConstants[1].molarMass;
          Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
          Density d_red_residual=fluidConstants[1].molarMass/
              thermalConductivityCoefficients.reducingMolarVolume_residual;

          Temperature T_crit=fluidConstants[1].criticalTemperature;
          Real tau "reduced temperature";

          AbsolutePressure p_crit=fluidConstants[1].criticalPressure;

          // coeffs for critical enhancement
          Real nu=thermalConductivityCoefficients.nu;
          Real gamma=thermalConductivityCoefficients.gamma;
          Real R0=thermalConductivityCoefficients.R0;
          Real z=thermalConductivityCoefficients.z;
          Real c=thermalConductivityCoefficients.c;
          Real xi_0=thermalConductivityCoefficients.xi_0;
          Real Gamma_0=thermalConductivityCoefficients.Gamma_0;
          Real q_D=1/thermalConductivityCoefficients.qd_inverse;
          Temperature T_ref=thermalConductivityCoefficients.T_ref;

          // interim variables for critical enhancement
          constant Real pi=Modelica.Constants.pi;
          constant Real k_b=Modelica.Constants.k;
          EoS.HelmholtzDerivs f;
          EoS.HelmholtzDerivs f_ref;
          Real ddpT;
          Real ddpT_ref;
          Real chi;
          Real chi_ref;
          Real Delta_chi;
          Real xi;
          Real Omega_0;
          Real Omega;

          SpecificHeatCapacity Cp;
          SpecificHeatCapacity Cv;
          DynamicViscosity eta_b;

        algorithm
          // critical enhancement by the simplified crossover model by Olchowy and Sengers
          if ((state.T > T_ref) or (state.d < d_crit/100)) then
            lambda_c := 0; // far away from critical point
          else
            // use critical values from EoS to calculate chi, Omega and lambda_c
            // watch out: algorithm for chi and chi_ref are different (chi_ref is multiplied with T_ref/state.T)
            f     := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
            f_ref := EoS.setHelmholtzDerivsSecond(T=T_ref,   d=state.d, phase=1);
            ddpT     := 1.0/EoS.dpdT(f);
            ddpT_ref := 1.0/EoS.dpdT(f_ref);
            chi     := p_crit/d_crit^2*state.d*ddpT;
            chi_ref := p_crit/d_crit^2*state.d*ddpT_ref*T_ref/state.T;

            Delta_chi := chi - chi_ref;

            if (Delta_chi < 0) then
              lambda_c := 0;
            else
              xi := xi_0*(Delta_chi/Gamma_0)^(nu/gamma);

              Cp := specificHeatCapacityCp(state=state);
              Cv := specificHeatCapacityCv(state=state);
              Omega := 2/pi*((Cp - Cv)/Cp*atan(q_D*xi) + Cv/Cp*q_D*xi);
              Omega_0 := 2/pi*(1 - exp(-1/(1/(q_D*xi) + ((q_D*xi*d_crit/state.d)^2)/3)));

              eta_b := dynamicViscosity(state=state);
              lambda_c := (state.d*Cp*R0*k_b*state.T)/(6*pi*eta_b*xi)*(Omega - Omega_0);
              lambda_c := max(0, lambda_c);
            end if;
          end if;

          /* // following lines are for debugging only
  Modelica.Utilities.Streams.print("===========================================");
  Modelica.Utilities.Streams.print("        d = " + String(state.d) + " and T = " + String(state.T));
  Modelica.Utilities.Streams.print("   ddpT   = " + String(ddpT) + " and ddpT_ref = " + String(ddpT_ref));
  Modelica.Utilities.Streams.print("   chi    = " + String(chi) + "  and  chi_ref = " + String(chi_ref));
  Modelica.Utilities.Streams.print("Delta_chi = " + String(Delta_chi));
  Modelica.Utilities.Streams.print("       xi = " + String(xi));
  Modelica.Utilities.Streams.print("       Cp = " + String(Cp) + "  and  Cv = " + String(Cv));
  Modelica.Utilities.Streams.print("  Omega_0 = " + String(Omega_0));
  Modelica.Utilities.Streams.print("    Omega = " + String(Omega));
  Modelica.Utilities.Streams.print("    eta_b = " + String(eta_b));
  Modelica.Utilities.Streams.print(" lambda_c = " + String(lambda_c));
  Modelica.Utilities.Streams.print("===========================================");
  */

          annotation (Documentation(info="<html>
  <p>
The thermal conductivity (TC) is split into three parts: ideal gas TC lamda_0, residual TC lambda_r and critical TC enhancement lambda_c.
Sometimes the residual TC is again split into two parts.
This allows to develop functions for each contribution seperately.
The sum of ideal gas TC and residual TC is called background TC.
Ideal gas TC depends on Temperature only and can be modelled by a quadratic function.
Residual TC is also modeled by a polynominal.
At the critical point TC becomes infinite; TC is enhanced for a large region around the critical point.
The critical enhancement can be described by various alternative approaches.
Here, the simplified approach as suggested by Olchowy and Sengers is implemented.

Special thanks go to Eric W. Lemmon for answering all my emails 
and programming a special version of RefProp that outputs also intermediate values.

</p>
<dl>
<dt>Olchowy, G.A. and Sengers, J.V</dt>
<dd> <b>A simplified representation for the thermal conductivity of fluids in the critical region</b>.<br>
     International Journal of Thermophysics (1998) 10, 417-426.<br>
     DOI: <a href=\"http://dx.doi.org/10.1007/BF01133538\">10.1007/BF01133538</a>
</dd>
</dl>
</html>"));
        end thermalConductivity_critical;
      end Transport;
      constant HelmholtzMedia.Interfaces.Types.FluidLimits fluidLimits;

      constant EoS.HelmholtzCoefficients helmholtzCoefficients;

      constant Transport.ThermalConductivityCoefficients thermalConductivityCoefficients;

      constant Transport.DynamicViscosityCoefficients dynamicViscosityCoefficients;

      constant Transport.SurfaceTensionCoefficients surfaceTensionCoefficients;

      constant Ancillary.AncillaryCoefficients ancillaryCoefficients;

      // constant IndependentVariables independentVariables=IndependentVariables.dTX;
      constant InputChoice inputChoice=InputChoice.ph
      "Default choice of input variables for property computations";

      redeclare record extends ThermodynamicState(phase(start=0))
        // inherits phase integer
        Temperature T "Temperature of medium";
        AbsolutePressure p "Absolute pressure of medium";
        Density d "Density of medium";
        SpecificEnergy u "Specific inner energy of medium";
        SpecificEnthalpy h "Specific enthalpy of medium";
        SpecificEntropy s "Specific entropy of medium";
      end ThermodynamicState;

      redeclare record extends SaturationProperties
        // inherits Tsat and psat
        ThermodynamicState liq;
        ThermodynamicState vap;
      end SaturationProperties;

      redeclare model extends BaseProperties(
          p(stateSelect = if preferredMediumStates and
                             (componentInputChoice == InputChoice.ph or
                              componentInputChoice == InputChoice.pT or
                              componentInputChoice == InputChoice.ps) then
                                  StateSelect.prefer else StateSelect.default),
          T(stateSelect = if preferredMediumStates and
                             (componentInputChoice == InputChoice.pT or
                             componentInputChoice == InputChoice.dT) then
                               StateSelect.prefer else StateSelect.default),
          h(stateSelect = if preferredMediumStates and
                             componentInputChoice == InputChoice.ph then
                               StateSelect.prefer else StateSelect.default),
          d(stateSelect = if preferredMediumStates and
                             componentInputChoice == InputChoice.dT then
                               StateSelect.prefer else StateSelect.default))
      "Base properties (p, d, T, h, u, s) of a medium"

        SpecificEntropy s;
        parameter InputChoice componentInputChoice=inputChoice
        "Choice of input variables for property computations";

      equation
        MM = fluidConstants[1].molarMass;
        R = Modelica.Constants.R/MM;

        // use functions to calculate properties
        if (componentInputChoice == InputChoice.ph) then
          // state = setState_ph(p=p, h=h);
          d = density_ph(p=p, h=h);
          T = temperature_ph(p=p, h=h);
          s = specificEntropy_ph(p=p, h=h);
        elseif (componentInputChoice == InputChoice.ps) then
          // state = setState_ps(p=p, s=s);
          d = density_ps(p=p, s=s);
          T = temperature_ps(p=p, s=s);
          h = specificEnthalpy_ps(p=p, s=s);
        elseif (componentInputChoice == InputChoice.pT) then
          // state = setState_pT(p=p, T=T);
          d = density_pT(p=p, T=T);
          h = specificEnthalpy_pT(p=p, T=T);
          s = specificEntropy_pT(p=p, T=T);
        elseif (componentInputChoice == InputChoice.dT) then
          // state = setState_dT(d=d, T=T);
          p = pressure_dT(d=d, T=T);
          h = specificEnthalpy_dT(d=d, T=T);
          s = specificEntropy_dT(d=d, T=T);
        else
          assert(false, "Invalid choice for basePropertiesInput");
        end if;

      // calculate u
        u = h - p/d;

      // set SaturationProperties
        sat = setSat_p(p=p);

      // connect state with BaseProperties
        state.d = d;
        state.T = T;
        state.p = p;
        state.h = h;
        state.s = s;
        state.u = u;
        state.phase = if ((p < fluidConstants[1].criticalPressure) and (p > fluidConstants[1].triplePointPressure) and (h > sat.liq.h) and (h < sat.vap.h)) then 2 else 1;

      end BaseProperties;

      redeclare function setSat_T
      "iterative calculation of saturation properties from EoS with Newton-Raphson algorithm"
        // does not extend, because base class already defines an algorithm
        input Temperature T;
        output SaturationProperties sat;

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        constant Real tau(unit="1")=T_crit/T "inverse reduced temperature";

        EoS.HelmholtzDerivs fl(T=T);
        EoS.HelmholtzDerivs fv(T=T);

        Real delta_liq(unit="1", min=0);
        Real delta_vap(unit="1", min=0);
        Real J_liq;
        Real J_liq_delta;
        Real J_vap;
        Real J_vap_delta;
        Real K_liq;
        Real K_liq_delta;
        Real K_vap;
        Real K_vap_delta;

        Real RES[2] "residual function vector";
        Real RSS "residual sum of squares";
        Real Jacobian[2,2] "Jacobian matrix";
        Real NS[2] "Newton step vector";

        constant Real lambda(min=0.1,max=1) = 1 "convergence speed, default=1";
        constant Real tolerance=1e-18 "tolerance for RSS";
        Integer iter = 0;
        constant Integer iter_max = 200;

      algorithm
        // Modelica.Utilities.Streams.print("setSat_T: T="+String(T),"printlog.txt");
        sat.Tsat := T;

      if ((T>=T_trip) and (T<T_crit)) then
        // calculate guess values for reduced density delta
        delta_liq := 1.02*Ancillary.bubbleDensity_T(T=T)/d_crit;
        delta_vap := 0.98*Ancillary.dewDensity_T(T=T)/d_crit;

        // set necessary parts of fl and fv
        fl.r   := EoS.f_r(tau=tau, delta=delta_liq);
        fv.r   := EoS.f_r(tau=tau, delta=delta_vap);
        fl.rd  := EoS.f_rd(tau=tau, delta=delta_liq);
        fv.rd  := EoS.f_rd(tau=tau, delta=delta_vap);

        // dimensionless pressure
        J_liq := delta_liq*(1 + delta_liq*fl.rd);
        J_vap := delta_vap*(1 + delta_vap*fv.rd);
        // dimensionless Gibbs energy
        K_liq := delta_liq*fl.rd + fl.r + log(delta_liq);
        K_vap := delta_vap*fv.rd + fv.r + log(delta_vap);
        // residual vector
        RES := {(J_vap-J_liq), (K_vap-K_liq)};
        RSS := RES*RES/2;

        while (RSS>tolerance) and (iter<iter_max) loop
          iter := iter+1;

          // calculate gradients of J and K, set up Jacobian matrix, get Newton step
          fl.rdd := EoS.f_rdd(tau=tau, delta=delta_liq);
          fv.rdd := EoS.f_rdd(tau=tau, delta=delta_vap);
          J_liq_delta := 1 + 2*delta_liq*fl.rd + delta_liq*delta_liq*fl.rdd;
          J_vap_delta := 1 + 2*delta_vap*fv.rd + delta_vap*delta_vap*fv.rdd;
          K_liq_delta := 2*fl.rd + delta_liq*fl.rdd + 1/delta_liq;
          K_vap_delta := 2*fv.rd + delta_vap*fv.rdd + 1/delta_vap;
          Jacobian := [-J_liq_delta, J_vap_delta;
                       -K_liq_delta, K_vap_delta];
          NS := -Modelica.Math.Matrices.solve(Jacobian,RES);

          // calculate better values for reduced density delta
          delta_liq := delta_liq + lambda*NS[1];
          delta_vap := delta_vap + lambda*NS[2];

          // check bounds
          delta_liq := max(delta_liq, 1);
          delta_liq := min(delta_liq, Modelica.Constants.inf);
          delta_vap := max(delta_vap, Modelica.Constants.small);
          delta_vap := min(delta_vap, 1);

          // update fl and fv
          fl.r   := EoS.f_r(tau=tau, delta=delta_liq);
          fv.r   := EoS.f_r(tau=tau, delta=delta_vap);
          fl.rd  := EoS.f_rd(tau=tau, delta=delta_liq);
          fv.rd  := EoS.f_rd(tau=tau, delta=delta_vap);

          // dimensionless pressure
          J_liq := delta_liq*(1 + delta_liq*fl.rd);
          J_vap := delta_vap*(1 + delta_vap*fv.rd);
          // dimensionless Gibbs energy
          K_liq := delta_liq*fl.rd + fl.r + log(delta_liq);
          K_vap := delta_vap*fv.rd + fv.r + log(delta_vap);
          // residual vector
          RES := {(J_vap-J_liq), (K_vap-K_liq)};
          RSS := RES*RES/2;

        end while;
        // Modelica.Utilities.Streams.print("setSat_T total iteration steps " + String(iter), "printlog.txt");
        assert(iter<iter_max, "setSat_T did not converge, input was T=" + String(T) +
                              "; the remaining residuals are RES_J=" + String(RES[1]) +
                              " and RES_K=" + String(RES[2]));

        sat.liq := setState_dTX(d=delta_liq*d_crit, T=T, phase=1);
        sat.vap := setState_dTX(d=delta_vap*d_crit, T=T, phase=1);
        sat.psat := (sat.liq.p+sat.vap.p)/2;

      elseif (T>=T_crit) then
        // assert(T <= T_crit, "setSat_T error: Temperature is higher than critical temperature", level=AssertionLevel.warning);
        // above critical temperature, no stable two-phase state exists
        // anyway, it is possible to extend the vapour-pressure curve into this region
        // this can happen when called from BaseProperties
        // one possibility is use the state where ds/dT=max or ds/dp=max or dcp/dT=max or dcp/dp=max
        // here, the critical isochore is returned
        sat.liq := setState_dTX(d=d_crit, T=T, phase=1);
        sat.vap := setState_dTX(d=d_crit, T=T, phase=1);
        sat.psat := (sat.liq.p+sat.vap.p)/2;
      else
        // assert(T >= T_trip, "setSat_T error: Temperature is lower than triple-point temperature", level=AssertionLevel.warning);
        // T<T_trip: this does not make sense: if T is below the triple temperature, the medium is solid, not fluid
        // anyway, during initialization (at time=0) T=0 may happen
        // density values are extrapolated linearly, fantasy values are returned
        sat.Tsat := max(T, Modelica.Constants.small);
        delta_liq := (T_trip/sat.Tsat)*Ancillary.bubbleDensity_T(T=T_trip)/d_crit;
        delta_vap := (sat.Tsat/T_trip)*Ancillary.dewDensity_T(T=T_trip)/d_crit;

        sat.liq := setState_dTX(d=delta_liq*d_crit, T=T, phase=1);
        sat.vap := setState_dTX(d=delta_vap*d_crit, T=T, phase=1);
        sat.psat := (sat.liq.p+sat.vap.p)/2;
      end if;

      annotation (Documentation(info="
<html>
This function iteratively determines the saturation state  for a given temperature 
by varying the density of saturated liquid and saturated vapor 
with a Newton-Raphson approach for simultaneous equations.

<dl>
<dt> Ryo Akasaka:</dt>
<dd> <b>A reliable and useful method to determine the saturation state from Helmholtz energy equations of state</b>.<br>
     Journal of Thermal Science and Technology 3 (3) , 442-451.<br>
     DOI: <a href=\"http://dx.doi.org/10.1299/jtst.3.442\">10.1299/jtst.3.442</a>
</dd>
</dl>
</html>"));
      end setSat_T;

      redeclare function setSat_p
      "iterative calculation of saturation properties from EoS for a given pressure"
        input AbsolutePressure p;
        output SaturationProperties sat;

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
        constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;
        constant Density dv_trip = Ancillary.dewDensity_T(T_trip);
        constant Density dl_trip = Ancillary.bubbleDensity_T(T_trip);

        EoS.HelmholtzDerivs fl;
        EoS.HelmholtzDerivs fv;

        Real RES[3] "residual function vector";
        Real RSS "residual sum of squares";
        Real Jacobian[3,3] "Jacobian matrix";
        Real NS[3] "Newton step vector";

        constant Real lambda(min=0.1,max=1) = 1 "convergence speed, default=1";
        constant Real tolerance=1e-9 "tolerance for RSS";
        Integer iter = 0;
        constant Integer iter_max = 200;

      algorithm
        // Modelica.Utilities.Streams.print(" ", "printlog.txt");
        // Modelica.Utilities.Streams.print("setSat_p: p=" + String(p), "printlog.txt");
        sat.psat := p;

      if ((p>p_trip) and (p<p_crit)) then
        // calculate start values, density should be outside of two-phase dome
        sat.Tsat := Ancillary.saturationTemperature_p(p=p);
        sat.liq.d := 1.02*Ancillary.bubbleDensity_T(T=sat.Tsat);
        sat.vap.d := 0.98*Ancillary.dewDensity_T(T=sat.Tsat);

        // calculate residuals: RES=calc-input
        fl := EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.Tsat, phase=1);
        fv := EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.Tsat, phase=1);
        RES := {EoS.p(fl)-p, EoS.p(fv)-p, EoS.g(fl)-EoS.g(fv)};
        RSS := RES*RES/2;

        while (RSS>tolerance) and (iter<iter_max) loop
          iter := iter+1;

          // calculate Jacobian matrix and Newton Step vector
          Jacobian := [+EoS.dpdT(fl), +0,            +EoS.dpTd(fl);
                       +0,            +EoS.dpdT(fv), +EoS.dpTd(fv);
                       +EoS.dgdT(fl), -EoS.dgdT(fv), EoS.dgTd(fl)-EoS.dgTd(fv)];
          NS := -Modelica.Math.Matrices.solve(Jacobian,RES);

          // calculate better sat.liq.d, sat.vap.d and sat.Tsat
          sat.liq.d := sat.liq.d + lambda*NS[1];
          sat.vap.d := sat.vap.d + lambda*NS[2];
          sat.Tsat  := sat.Tsat  + lambda*NS[3];

          // check bounds
          sat.liq.d := max(sat.liq.d, 0.98*d_crit);
          sat.liq.d := min(sat.liq.d, 1.02*dl_trip);
          sat.vap.d := max(sat.vap.d, 0.98*dv_trip);
          sat.vap.d := min(sat.vap.d, 1.02*d_crit);
          sat.Tsat  := max(sat.Tsat,  0.98*T_trip);
          sat.Tsat  := min(sat.Tsat,  1.02*T_crit);

          // calculate new residuals: RES=calc-input
          fl := EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.Tsat, phase=1);
          fv := EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.Tsat, phase=1);
          RES := {EoS.p(fl)-p, EoS.p(fv)-p, EoS.g(fl)-EoS.g(fv)};
          RSS := RES*RES/2;
        end while;
        // if verbose then Modelica.Utilities.Streams.print("setSat_p total iteration steps " + String(iter), "printlog.txt"); end if;
        // Modelica.Utilities.Streams.print("setSat_p total iteration steps " + String(iter), "printlog.txt");
        assert(iter<iter_max, "setSat_p did not converge, input was p=" + String(p) +
                              "; the remaining residuals are RES_pl=" + String(RES[1]) +
                              " and RES_pv=" + String(RES[2]) +
                              " and RES_g=" + String(RES[3]));

        // check bounds, more strict
        sat.liq.d := max(sat.liq.d, d_crit);
        sat.liq.d := min(sat.liq.d, dl_trip);
        sat.vap.d := max(sat.vap.d, dv_trip);
        sat.vap.d := min(sat.vap.d, d_crit);
        sat.Tsat  := max(sat.Tsat,  T_trip);
        sat.Tsat  := min(sat.Tsat,  T_crit);

        sat.liq := setState_dTX(d=sat.liq.d, T=sat.Tsat, phase=1);
        sat.vap := setState_dTX(d=sat.vap.d, T=sat.Tsat, phase=1);

      elseif (p>=p_crit) then
        // assert(p <= p_crit, "setSat_p error: pressure is higher than critical pressure", level=AssertionLevel.warning);
        // above critical pressure, no stable two-phase state exists
        // anyway, it is possible to extend the vapour-pressure curve into this region
        // this can happen when called from BaseProperties
        // one possibility is to use the state where ds/dT=max or ds/dp=max or dcp/dT=max or dcp/dp=max
        // here, the critical isochore is returned
        sat.psat  := p;
        sat.liq := setState_pd(p=p, d=d_crit, phase=1);
        sat.vap := sat.liq;
        sat.Tsat := sat.liq.T;
      elseif (p<=p_trip) then
        // at pressures below triple point pressure, only single-phase vapor is possible
        // just return triple-point values
        sat.psat  := p;
        sat.liq := setState_dT(d=dl_trip, T=T_trip, phase=1);
        sat.vap := setState_dT(d=dv_trip, T=T_trip, phase=1);
        sat.Tsat := T_trip;
      else
        assert(false, "setSat_p: this should not happen, check p");
      end if;

      end setSat_p;

      function setSat_d
      "iterative calculation of saturation properties from EoS with Newton-Raphson algorithm"
        input Density d;
        output SaturationProperties sat;

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
        constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;
        constant Density dv_trip = Ancillary.dewDensity_T(T_trip);
        constant Density dl_trip = Ancillary.bubbleDensity_T(T_trip);

        EoS.HelmholtzDerivs fl;
        EoS.HelmholtzDerivs fv;

        Real RES[2] "residual function vector";
        Real RSS "residual sum of squares";
        Real Jacobian[2,2] "Jacobian matrix";
        Real NS[2] "Newton step vector";

        constant Real lambda(min=0.1,max=1) = 1 "convergence speed, default=1";
        constant Real tolerance=1e-9 "tolerance for RSS";
        Integer iter = 0;
        constant Integer iter_max = 200;

      algorithm
        // Modelica.Utilities.Streams.print(" ", "printlog.txt");
        // Modelica.Utilities.Streams.print("setSat_d: d="+String(d),"printlog.txt");

        sat.Tsat := Ancillary.saturationTemperature_d(d=d);
        if (d<d_crit) and (d>dv_trip) then
          // Modelica.Utilities.Streams.print("d<d_crit: input is on vapour side: find d_liq and T_sat", "printlog.txt");
          sat.liq.d := 1.02*Ancillary.bubbleDensity_T(sat.Tsat);
          sat.vap.d := d; // d_vap is a constant

          // calculate residuals: liq-vap (=var-const)
          fl := EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.Tsat, phase=1);
          fv := EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.Tsat, phase=1);
          RES := {EoS.p(fl)-EoS.p(fv), EoS.g(fl)-EoS.g(fv)};
          RSS := RES*RES/2;

          while (RSS>tolerance) and (iter<iter_max) loop
            iter := iter+1;

            // calculate Jacobian matrix and Newton Step vector
            Jacobian := [EoS.dpdT(fl), EoS.dpTd(fl)-EoS.dpTd(fv);
                         EoS.dgdT(fl), EoS.dgTd(fl)-EoS.dgTd(fv)];
            NS := -Modelica.Math.Matrices.solve(Jacobian,RES);

            // calculate better values for sat.liq.d and sat.Tsat
            sat.liq.d := sat.liq.d +lambda*NS[1];
            sat.Tsat  := sat.Tsat  +lambda*NS[2];

            // check bounds
            sat.liq.d := max(sat.liq.d, 0.98*d_crit);
            sat.liq.d := min(sat.liq.d, 1.02*dl_trip);
            sat.Tsat  := max(sat.Tsat,  0.98*T_trip);
            sat.Tsat  := min(sat.Tsat,  1.02*T_crit);

            // calculate new residuals: liq-vap
            fl := EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.Tsat, phase=1);
            fv := EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.Tsat, phase=1);
            RES := {EoS.p(fl)-EoS.p(fv), EoS.g(fl)-EoS.g(fv)};
            RSS := RES*RES/2;
          end while;
          sat.liq  := setState_dTX(d=sat.liq.d, T=sat.Tsat, phase=1);
          sat.vap  := setState_dTX(d=sat.vap.d, T=sat.Tsat, phase=1);
          sat.psat := sat.liq.p;

        elseif (d>d_crit) and (d<dl_trip) then
          // Modelica.Utilities.Streams.print("d>d_crit: input is on liquid side: find d_vap and T_sat", "printlog.txt");
          sat.vap.d := 0.98*Ancillary.dewDensity_T(sat.Tsat);
          sat.liq.d := d; // d_liq is a constant

          // calculate residuals: vap-liq (=var-const)
          fv := EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.Tsat, phase=1);
          fl := EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.Tsat, phase=1);
          RES := {EoS.p(fv)-EoS.p(fl), EoS.g(fv)-EoS.g(fl)};
          RSS := RES*RES/2;

          while (RSS>tolerance) and (iter<iter_max) loop
            iter := iter+1;

            // calculate Jacobian matrix and Newton Step vector
            Jacobian := [EoS.dpdT(fv), EoS.dpTd(fv)-EoS.dpTd(fl);
                         EoS.dgdT(fv), EoS.dgTd(fv)-EoS.dgTd(fl)];
            NS := -Modelica.Math.Matrices.solve(Jacobian,RES);

            // calculate better values for sat.vap.d and sat.Tsat
            sat.vap.d := sat.vap.d +lambda*NS[1];
            sat.Tsat  := sat.Tsat  +lambda*NS[2];

            // check bounds
            sat.vap.d := max(sat.vap.d, 0.98*dv_trip);
            sat.vap.d := min(sat.vap.d, 1.02*d_crit);
            sat.Tsat  := max(sat.Tsat,  0.98*T_trip);
            sat.Tsat  := min(sat.Tsat,  1.02*T_crit);

            // calculate new residuals: vap-liq
            fv := EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.Tsat, phase=1);
            fl := EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.Tsat, phase=1);
            RES := {EoS.p(fv)-EoS.p(fl), EoS.g(fv)-EoS.g(fl)};
            RSS := RES*RES/2;
          end while;
          sat.liq  := setState_dTX(d=sat.liq.d, T=sat.Tsat, phase=1);
          sat.vap  := setState_dTX(d=sat.vap.d, T=sat.Tsat, phase=1);
          sat.psat := sat.liq.p;

        elseif (d>=dl_trip) or (d<=dv_trip) then
          // Modelica.Utilities.Streams.print("d out of two-phase range, return triple point values", "printlog.txt");
          sat.Tsat:= T_trip;
          sat.psat:= p_trip;
          sat.liq := setState_dTX(d=dl_trip, T=T_trip, phase=1);
          sat.vap := setState_dTX(d=dv_trip, T=T_trip, phase=1);
        else
          // Modelica.Utilities.Streams.print("d=d_crit: return critical values", "printlog.txt");
          sat.Tsat := T_crit;
          sat.psat := p_crit;
          sat.liq := setState_dTX(d=d_crit, T=T_crit, phase=1);
          sat.vap := setState_dTX(d=d_crit, T=T_crit, phase=1);
        end if;
        // Modelica.Utilities.Streams.print("setSat_d total iteration steps " + String(iter), "printlog.txt");
        assert(iter<iter_max, "setSat_d did not converge, input was d=" + String(d)+
                              "; the remaining residuals are RES[1]=" + String(RES[1]) +
                              " and RES[2]=" + String(RES[2]));

      end setSat_d;

      redeclare function extends setBubbleState
      "returns bubble ThermodynamicState from given saturation properties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat, input phase and output state
      algorithm
        state := sat.liq;
      end setBubbleState;

      redeclare function extends setDewState
      "returns dew ThermodynamicState from given saturation properties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat, input phase and output state
      algorithm
        state := sat.vap;
      end setDewState;

      redeclare function extends setSmoothState
      "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
        import Modelica.Media.Common.smoothStep;

      algorithm
        state := ThermodynamicState(
          p=smoothStep(x, state_a.p, state_b.p, x_small),
          h=smoothStep(x, state_a.h, state_b.h, x_small),
          d=density_ph(p=smoothStep(x, state_a.p, state_b.p, x_small),
                       h=smoothStep(x, state_a.h, state_b.h, x_small)),
          T=temperature_ph(p=smoothStep(x, state_a.p, state_b.p, x_small),
                           h=smoothStep(x, state_a.h, state_b.h, x_small)),
          s=specificEntropy_ph(p=smoothStep(x, state_a.p, state_b.p, x_small),
                               h=smoothStep(x, state_a.h, state_b.h, x_small)),
          u=state.h-state.p/state.d,
          phase=0);
      annotation (Inline=true);
      end setSmoothState;

      redeclare function extends setState_dTX
      "Return thermodynamic state as function of (d, T)"

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        constant Temperature T_trip=fluidConstants[1].triplePointTemperature;

        EoS.HelmholtzDerivs f;
        SaturationProperties sat;
        MassFraction x "vapour quality";

      algorithm
        state.phase := phase;

        if (state.phase == 0) then
          //phase unknown, check phase first
          if ((T>=T_trip) and (T<T_crit)) then
            // two-phase possible, do simple density check
            // Modelica.Utilities.Streams.print("setState_dT: dliq=" + String(bubbleDensity_T_ANC(T=T)) + " dvap=" + String(dewDensity_T_ANC(T=T)) + ", simple check only");
            if ((d > 1.05*Ancillary.bubbleDensity_T(T=T)) or (d < 0.98*Ancillary.dewDensity_T(T=T))) then
              state.phase := 1;
            else
              // Modelica.Utilities.Streams.print("setState_dT: d=" + String(d) + " T=" + String(T) + ", two-phase state or close to it");
              // get saturation properties from EoS, use Tsat as starting value
              sat := setSat_T(T=T);
              if ((d < sat.liq.d) and (d > sat.vap.d)) then
                state.phase := 2;
              else
                state.phase := 1;
              end if;
            end if;
          else
            // T>=T_crit
            state.phase := 1;
          end if;
        elseif (state.phase == 2) then
          assert(T <= T_crit, "setState_dTX_error: Temperature is higher than critical temperature");
          sat := setSat_T(T=T);
          assert(d >= sat.vap.d, "setState_dTX_error: density is lower than saturated vapor density: this is single phase vapor");
          assert(d <= sat.liq.d, "setState_dTX_error: density is higher than saturated liquid density: this is single phase liquid");
        end if;

        state.d := d;
        state.T := T;
        if (state.phase == 2) then
          // force two-phase
          x := (1.0/d - 1.0/sat.liq.d)/(1.0/sat.vap.d - 1.0/sat.liq.d);
          state.p := sat.psat;
          state.h := sat.liq.h + x*(sat.vap.h - sat.liq.h);
          state.u := sat.liq.u + x*(sat.vap.u - sat.liq.u);
          state.s := sat.liq.s + x*(sat.vap.s - sat.liq.s);
        else
          // force single-phase
          f := EoS.setHelmholtzDerivsFirst(d=d,T=T);
          state.p := EoS.p(f=f);
          state.s := EoS.s(f=f);
          state.h := EoS.h(f=f);
          state.u := EoS.u(f=f);
        end if;

      end setState_dTX;

      redeclare function setState_Tx
      "Return thermodynamic state as function of (T, x)"
        input Temperature T "Temperature";
        input MassFraction x "Vapour quality";
        output ThermodynamicState state "Thermodynamic state record";

    protected
        SaturationProperties sat=setSat_T(T=T);

      algorithm
        state.phase := 2;
        state.p := sat.psat;
        state.T := sat.Tsat;
        state.d := 1.0/(1.0/sat.liq.d + x*(1.0/sat.vap.d - 1.0/sat.liq.d));
        state.h := sat.liq.h + x*(sat.vap.h - sat.liq.h);
        state.u := sat.liq.u + x*(sat.vap.u - sat.liq.u);
        state.s := sat.liq.s + x*(sat.vap.s - sat.liq.s);
      end setState_Tx;

      redeclare function setState_px
      "Return thermodynamic state as function of (p, x)"
        input AbsolutePressure p "Pressure";
        input MassFraction x "Vapour quality";
        output ThermodynamicState state "Thermodynamic state record";

    protected
        SaturationProperties sat=setSat_p(p=p);

      algorithm
        state.phase := 2;
        state.p := sat.psat;
        state.T := sat.Tsat;
        state.d := 1.0/(1.0/sat.liq.d + x*(1.0/sat.vap.d - 1.0/sat.liq.d));
        state.h := sat.liq.h + x*(sat.vap.h - sat.liq.h);
        state.u := sat.liq.u + x*(sat.vap.u - sat.liq.u);
        state.s := sat.liq.s + x*(sat.vap.s - sat.liq.s);
      end setState_px;

      redeclare function extends setState_pTX
      "Return thermodynamic state as function of (p, T)"

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
        constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;
        // constant Real Z_crit=1/(d_crit*R*T_crit/p_crit);

        EoS.HelmholtzDerivs f(T=T);
        SaturationProperties sat;

        Density d_min;
        Density d_max;
        Density d_iter;
        AbsolutePressure RES_p;
        AbsolutePressure RES_min;
        AbsolutePressure RES_max;
        DerPressureByDensity dpdT "(dp/dd)@T=const";
        Real gamma(min=0.1,max=1) = 1 "convergence speed, default=1";
        constant Real tolerance=1e-9 "relativ tolerance for RES_p";
        Integer iter = 0;
        constant Integer iter_max=200;

      algorithm
        // Modelica.Utilities.Streams.print(" ", "printlog.txt");
        // Modelica.Utilities.Streams.print("setState_pTX: p=" + String(p) + " and T=" + String(T), "printlog.txt");
        assert(phase <> 2, "setState_pTX_error: pressure and temperature are not independent variables in two-phase state");
        state.phase := 1;

        if (T < T_crit) then
          // determine p_sat
          sat.psat := Ancillary.saturationPressure_T(T=T);
          if (p > 1.02*sat.psat) then
            sat.liq.d := Ancillary.bubbleDensity_T(T=T);
          elseif (p < 0.98*sat.psat) then
            sat.vap.d := Ancillary.dewDensity_T(T=T);
          else
            // Modelica.Utilities.Streams.print("close to saturation boundary, get saturation properties from EoS", "printlog.txt");
            sat := setSat_T(T=T);
          end if;
          // Modelica.Utilities.Streams.print("sat.psat=" + String(sat.psat), "printlog.txt");

          // determine region
          if (p > sat.psat) then
            // Modelica.Utilities.Streams.print("single-phase liquid region", "printlog.txt");
            d_min  := 0.98*sat.liq.d;
            d_max  := 1.10*fluidLimits.DMAX;
            d_iter := 1.10*sat.liq.d;
            // d_iter := Ancillary.density_pT_Soave(T=T, p=p, psat=sat.psat);
            // d_iter := 1/(R*T_crit/p_crit*Z_crit^(1+min(0,1-T/T_crit)^(2/7))) "Rackett";
          elseif (p < sat.psat) then
            // Modelica.Utilities.Streams.print("single-phase vapor region", "printlog.txt");
            d_min  := fluidLimits.DMIN;
            d_max  := 1.02*sat.vap.d;
            // d_iter := p/(R*T);
            d_iter := Ancillary.density_pT_Soave(T=T, p=p, psat=sat.psat);
          else
            // Modelica.Utilities.Streams.print("two-phase region", "printlog.txt");
            assert(p <> sat.psat, "setState_pTX_error: pressure equals saturation pressure");
          end if;
        else
          // Modelica.Utilities.Streams.print("single-phase super-critical region", "printlog.txt");
          d_min  := fluidLimits.DMIN;
          d_max  := 1.1*fluidLimits.DMAX;
          // d_iter := p/(R*T);
          d_iter := Ancillary.density_pT_Soave(T=T, p=p, psat=sat.psat);
        end if;

        // Modelica.Utilities.Streams.print("phase and region determined, d_min=" + String(d_min) + ", d_max=" + String(d_max) + " and d_iter=" + String(d_iter), "printlog.txt");

        // check bounds, ideal gas and Soave are not very accurate
        d_iter := max(d_iter, d_min);
        d_iter := min(d_iter, d_max);

        // Modelica.Utilities.Streams.print("initialize bisection", "printlog.txt");
        // min
        f.d := d_min;
        f.delta := f.d/d_crit;
        f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
        RES_min := EoS.p(f) - p;
        // max
        f.d := d_max;
        f.delta := f.d/d_crit;
        f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
        RES_max := EoS.p(f) - p;
        // iter
        f.d := d_iter;
        f.delta := f.d/d_crit;
        f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
        RES_p := EoS.p(f) - p;

        assert((RES_min*RES_max<0), "setState_pTX: d_min and d_max did not bracket the root", level=AssertionLevel.warning);
        // thighten the bounds
        // opposite sign brackets the root
        if (RES_p*RES_min<0) then
          d_max := d_iter;
          RES_max := RES_p;
        elseif (RES_p*RES_max<0) then
          d_min := d_iter;
          RES_min := RES_p;
        end if;

        while ((abs(RES_p/p) > tolerance) and (iter<iter_max)) loop
          iter := iter+1;
          // gamma := iter/(iter+1);

          // calculate gradient with respect to density
          f.rdd := EoS.f_rdd(delta=f.delta, tau=f.tau);
          dpdT := EoS.dpdT(f);

          // print for Newton debugging
          // Modelica.Utilities.Streams.print("Iteration step " +String(iter) + ", current d_iter=" + String(d_iter), "printlog.txt");
          // Modelica.Utilities.Streams.print("RES_p=" + String(RES_p) + " and dpdT=" + String(dpdT), "printlog.txt");

          // calculate better d_iter using Newton
          d_iter := d_iter - gamma/dpdT*RES_p;

          // check bounds, if out of bounds use bisection
          if (d_iter<d_min) or (d_iter>d_max) then
            // Modelica.Utilities.Streams.print("d_iter out of bounds, fallback to bisection method, step=" + String(iter) + ", d_iter=" + String(d_iter) + ", input was p=" + String(p) + " and T=" + String(T), "printlog.txt");
            d_iter := (d_min+d_max)/2;
          end if;

          // set necessary parts of f, then calculate new RES_p
          f.d := d_iter;
          f.delta := f.d/d_crit;
          f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
          RES_p := EoS.p(f) - p;

          // thighten the bounds
          // opposite sign brackets the root
          if (RES_p*RES_min<0) then
            d_max := d_iter;
            RES_max := RES_p;
          elseif (RES_p*RES_max<0) then
            d_min := d_iter;
            RES_min := RES_p;
          end if;
          // Modelica.Utilities.Streams.print("d_min=" + String(d_min) + ", d_max=" + String(d_max) + " and d_iter=" + String(d_iter), "printlog.txt");
        end while;
        // Modelica.Utilities.Streams.print("setState_pT required " + String(iter) + " iterations to find d=" + String(d_iter) + " for input T=" + String(T) + " and p=" + String(p), "printlog.txt");
        assert(iter<iter_max, "setState_pTX did not converge, input was p=" + String(p) + " and T=" + String(T));

        state.p := p;
        state.T := T;
        state.d := d_iter;
        f := EoS.setHelmholtzDerivsFirst(d=state.d, T=state.T);
        state.s := EoS.s(f=f);
        state.h := EoS.h(f=f);
        state.u := EoS.u(f=f);
      end setState_pTX;

      redeclare function extends setState_phX
      "Return thermodynamic state as function of (p, h)"

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        Real delta "reduced density";
        Real tau "inverse reduced temperature";

        constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
        constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;
        constant SpecificEnthalpy h_crit=fluidConstants[1].HCRIT0;

        EoS.HelmholtzDerivs f;
        SaturationProperties sat;
        MassFraction x "vapour quality";

        Density d_min;
        Density d_max;
        Density d_iter;
        Density d_iter_old;
        Temperature T_min;
        Temperature T_max;
        Temperature T_iter;
        Temperature T_iter_old;

        Real RES[2] "residual function vector";
        Real RSS "residual sum of squares";
        Real RSS_old "residual sum of squares";
        Real Jacobian[2,2] "Jacobian matrix";
        Real NS[2] "Newton step vector";
        Real grad[2] "gradient vector";
        Real slope;

        constant Real tolerance=1e-9 "tolerance for RSS";
        Integer iter = 0;
        Integer iter_max = 200;
        Real lambda(min=1e-3,max=1) = 1 "convergence speed, default=1";

        Boolean useLineSearch=helmholtzCoefficients.useLineSearch;
        Integer iterLineSearch = 0;
        Real RSS_ls;
        Real lambda_ls;
        constant Real lambda_min = 0.01 "minimum for convergence speed";
        Real lambda_temp = 1 "temporary variable for convergence speed";
        constant Real alpha(min=0,max=1)=1e-4;
        Real rhs1;
        Real rhs2;
        Real a;
        Real b;
        Real Discriminant;

      algorithm
        // Modelica.Utilities.Streams.print(" ", "printlog.txt");
        // Modelica.Utilities.Streams.print("setState_phX: p=" + String(p) + " and h=" + String(h), "printlog.txt");
        state.phase := phase;

        if (state.phase == 2) then
          assert(p <= p_crit, "setState_phX_error: pressure is higher than critical pressure");
          sat := setSat_p(p=p);
          assert(h >= sat.liq.h, "setState_phX_error: enthalpy is lower than saturated liquid enthalpy: this is single phase liquid");
          assert(h <= sat.vap.h, "setState_phX_error: enthalpy is higher than saturated vapor enthalpy: this is single phase vapor");
        else
          if (p < p_crit) then
            // two-phase possible, check region
            if (p>0.98*p_crit) or (p<300*p_trip) then
              // Modelica.Utilities.Streams.print("close to critical or triple point, get saturation properties from EoS", "printlog.txt");
              sat := setSat_p(p=p);
            else
              // do a simple check first, quite often this is sufficient
              sat.Tsat := Ancillary.saturationTemperature_p(p=p);

              sat.liq.d := Ancillary.bubbleDensity_T(T=sat.Tsat);
              f := EoS.setHelmholtzDerivsFirst(d=sat.liq.d, T=sat.Tsat, phase=1);
              sat.liq.h := EoS.h(f);

              sat.vap.d := Ancillary.dewDensity_T(T=sat.Tsat);
              f := EoS.setHelmholtzDerivsFirst(d=sat.vap.d, T=sat.Tsat, phase=1);
              sat.vap.h := EoS.h(f);

              if ((h > sat.liq.h - abs(0.02*sat.liq.h)) and (h < sat.vap.h + abs(0.02*sat.vap.h))) then
                // Modelica.Utilities.Streams.print("two-phase state or close to it, get saturation properties from EoS", "printlog.txt");
                sat := setSat_p(p=p);
              end if;
            end if;

            // Modelica.Utilities.Streams.print("phase boundary determined; sat.liq.h=" + String(sat.liq.h) + " and sat.vap.h=" + String(sat.vap.h), "printlog.txt");
            if (h < sat.liq.h) then
              // Modelica.Utilities.Streams.print("single phase liquid", "printlog.txt");
              state.phase := 1;

              d_min := sat.liq.d*0.98;
              d_iter:= sat.liq.d*1.02;
              d_max := fluidLimits.DMAX*1.10;

              T_min := fluidLimits.TMIN*0.99;
              T_iter:= sat.Tsat*0.98;
              T_max := sat.Tsat*1.02;
            elseif (h > sat.vap.h) then
              // Modelica.Utilities.Streams.print("single phase vapor", "printlog.txt");
              state.phase := 1;

              d_min := fluidLimits.DMIN;
              d_iter:= sat.vap.d/10;
              d_max := sat.vap.d*1.02;

              T_min := sat.Tsat*0.98;
              T_iter:= sat.Tsat*1.02;
              T_max := fluidLimits.TMAX*1.10;
            else
              // Modelica.Utilities.Streams.print("two-phase, all properties can be calculated from sat record", "printlog.txt");
              state.phase := 2;
            end if;

          else
            state.phase := 1;
            // Modelica.Utilities.Streams.print("p>=p_crit, only single-phase possible", "printlog.txt");
            if (h<=h_crit) then
              // Modelica.Utilities.Streams.print("h<=h_crit, single-phase super-critical liquid-like region", "printlog.txt");
              d_min := d_crit*0.99;
              d_iter:= fluidLimits.DMAX*0.85;
              d_max := fluidLimits.DMAX*1.1;

              T_min := fluidLimits.TMIN;
              T_max := 1.25*Ancillary.saturationTemperature_h_liq(h=h);
              T_iter:= (1*T_min + 5*T_max/1.25)/6;
            else
              // Modelica.Utilities.Streams.print("h>h_crit, single-phase super-critical vapour-like region", "printlog.txt");
              // due to the curvature, Newton will converge better when starting from the ideal gas region (low d, high T)
              d_min := fluidLimits.DMIN;
              d_iter:= d_crit/10;
              d_max := fluidLimits.DMAX*1.1;

              T_min := fluidLimits.TMIN;
              T_iter:= T_crit*1.3;
              T_max := fluidLimits.TMAX*1.1;
            end if;
          end if;
        end if;

        // phase and region determination finished !

        if (state.phase == 2) then
          // Modelica.Utilities.Streams.print("two-phase, SaturationProperties are already known", "printlog.txt");
          state.p := p;
          state.h := h;
          state.T := sat.Tsat;
          x := (h - sat.liq.h)/(sat.vap.h - sat.liq.h);
          state.d := 1/(1/sat.liq.d + x*(1/sat.vap.d - 1/sat.liq.d));
          state.u := sat.liq.u + x*(sat.vap.u - sat.liq.u);
          state.s := sat.liq.s + x*(sat.vap.s - sat.liq.s);
        else
          // Modelica.Utilities.Streams.print("single-phase, use 2D Newton-Raphson, start with d_iter=" + String(d_iter) + " (d_min=" + String(d_min) + " and d_max=" + String(d_max) + ") and T_iter=" + String(T_iter)+ " (T_min=" + String(T_min) + " and T_max=" + String(T_max) + ")", "printlog.txt");
          f := EoS.setHelmholtzDerivsSecond(d=d_iter, T=T_iter, phase=1);
          RES := {EoS.p(f)-p, EoS.h(f)-h};
          RSS := RES*RES/2;

          while ((RSS>tolerance) and (iter<iter_max)) loop
            iter := iter+1;

            // calculate Jacobian matrix, Newton Step vector, gradient and slope
            Jacobian := [EoS.dpdT(f), EoS.dpTd(f);
                         EoS.dhdT(f), EoS.dhTd(f)];
            NS := -Modelica.Math.Matrices.solve(Jacobian,RES) "-J^(-1)·F";
            grad := RES*Jacobian "F·J";
            slope := grad*NS;
            // Modelica.Utilities.Streams.print("  Jacobian=" + Modelica.Math.Matrices.toString(Jacobian) + "  NS=" + Modelica.Math.Vectors.toString(NS) + "  grad=" + Modelica.Math.Vectors.toString(grad) + "  slope=" + String(slope), "printlog.txt");
            assert(slope<0,"roundoff problem, input was p=" + String(p) + " and h=" + String(h));

            // store old d_iter, T_iter and RSS
            d_iter_old := d_iter;
            T_iter_old := T_iter;
            RSS_old := RSS;

            // calculate new d_iter and T_iter using full Newton step (lambda=1)
            d_iter := d_iter_old + NS[1];
            T_iter := T_iter_old + NS[2];

            // check bounds
            d_iter := max(d_iter, d_min);
            d_iter := min(d_iter, d_max);
            T_iter := max(T_iter, T_min);
            T_iter := min(T_iter, T_max);

            // calculate new residual vector and residual sum of squares
            f := EoS.setHelmholtzDerivsSecond(d=d_iter, T=T_iter, phase=1);
            RES := {EoS.p(f)-p, EoS.h(f)-h};
            RSS := RES*RES/2;
            // Modelica.Utilities.Streams.print("iter=" + String(iter) + " d_iter=" + String(d_iter) + " T_iter=" + String(T_iter) + " RES_p=" + String(RES[1]) + " RES_h=" + String(RES[2]) + " RSS=" + String(RSS), "printlog.txt");

            // if RSS is not decreasing fast enough, the full Newton step is not used
            // instead, the linesearching / backtracking loop tries to find lambda such that RSS decreases
            while useLineSearch and (lambda>=lambda_min) and not (RSS<=(RSS_old+alpha*lambda*slope)) loop
              iterLineSearch := iterLineSearch+1;
              iter_max := iter_max+1;

              // decrease lambda
              if (iterLineSearch<2) then
                // first attempt
                lambda_temp := -slope/(2*(RSS-RSS_old-slope));
              else
                // subsequent attempts
                rhs1 := RSS   -RSS_old-lambda   *slope;
                rhs2 := RSS_ls-RSS_old-lambda_ls*slope;
                a := (           rhs1/lambda^2 -        rhs2/lambda_ls^2) / (lambda-lambda_ls);
                b := (-lambda_ls*rhs1/lambda^2 + lambda*rhs2/lambda_ls^2) / (lambda-lambda_ls);
                if (a==0) then
                  lambda_temp := -slope/(2*b);
                else
                  Discriminant := b*b-3*a*slope;
                  if (Discriminant<0) then
                    lambda_temp := 0.5*lambda;
                  elseif (b<=0) then
                    lambda_temp := (-b+sqrt(Discriminant))/(3*a);
                  else
                    lambda_temp := -slope/(b+sqrt(Discriminant));
                  end if;
                  // new lambda should be less or equal 0.5*previous lambda
                  lambda_temp := if (lambda_temp>0.5*lambda) then 0.5*lambda else lambda_temp;
                end if;
              end if;
              // store values for subsequent linesearch attempt
              lambda_ls := lambda;
              RSS_ls := RSS;

              // new lambda should be greater or equal 0.1*previous lambda
              lambda := max({lambda_temp, 0.1*lambda});

              // calculate new d_iter and T_iter using partial Newton step (lambda<1)
              d_iter := d_iter_old +lambda*NS[1];
              T_iter := T_iter_old +lambda*NS[2];
              // check bounds
              d_iter := max(d_iter, d_min);
              d_iter := min(d_iter, d_max);
              T_iter := max(T_iter, T_min);
              T_iter := min(T_iter, T_max);

              // calculate new residual vector and residual sum of squares
              f := EoS.setHelmholtzDerivsSecond(d=d_iter, T=T_iter, phase=1);
              RES := {EoS.p(f)-p, EoS.h(f)-h};
              RSS := RES*RES/2;
              // Modelica.Utilities.Streams.print("    linesearch attempt " + String(iterLineSearch) + ": lambda= " + String(lambda) + " d_iter=" + String(d_iter) + " T_iter= " + String(T_iter) + " RSS= " + String(RSS), "printlog.txt");

            end while;
            // reset iterLineSearch and lambda
            iterLineSearch := 0;
            lambda_temp := 1;
            lambda := 1;

          end while;
          // Modelica.Utilities.Streams.print("setState_ph total iteration steps " + String(iter), "printlog.txt");
          assert(iter<iter_max, "setState_phX did not converge, input was p=" + String(p) + " and h=" + String(h));

          state.p := p;
          state.h := h;
          state.d := d_iter;
          state.T := T_iter;
          state.u := EoS.u(f);
          state.s := EoS.s(f);
        end if;

      end setState_phX;

      redeclare function extends setState_psX
      "Return thermodynamic state as function of (p, s)"

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        Real delta "reduced density";
        Real tau "inverse reduced temperature";

        constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
        constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;
        constant SpecificEntropy s_crit=fluidConstants[1].SCRIT0;

        EoS.HelmholtzDerivs f;
        SaturationProperties sat;
        MassFraction x "vapour quality";

        Density d_min;
        Density d_max;
        Density d_iter;
        Density d_iter_old;
        Temperature T_min;
        Temperature T_max;
        Temperature T_iter;
        Temperature T_iter_old;

        Real[2] RES "residual function vector";
        Real RSS "residual sum of squares (divided by 2)";
        Real RSS_old "residual sum of squares (divided by 2)";
        Real[2,2] Jacobian "Jacobian matrix";
        Real[2] NS "Newton step vector";
        Real grad[2] "gradient vector";
        Real slope;

        EoS.HelmholtzDerivs f_med;
        Real[2] RES_med "residual function vector";
        Real[2] xy;
        Real[2] xy_old;
        Real[2] xy_med;
        Real[2,2] Jacobian_med "Jacobian matrix";

        constant Real tolerance=1e-7 "tolerance for RSS";
        Integer iter = 0;
        Integer iter_max = 200;
        Real lambda(min=1e-3,max=1) = 1 "convergence speed, default=1";

        Boolean useLineSearch=helmholtzCoefficients.useLineSearch;
        Integer iterLineSearch = 0;
        Real RSS_ls;
        Real lambda_ls;
        constant Real lambda_min = 0.01 "minimum for convergence speed";
        Real lambda_temp = 1 "temporary variable for convergence speed";
        constant Real alpha(min=0,max=1)=1e-4;
        Real rhs1;
        Real rhs2;
        Real a;
        Real b;
        Real Discriminant;

        ThermodynamicState state_ref;

      algorithm
        // Modelica.Utilities.Streams.print(" ", "printlog.txt");
        // Modelica.Utilities.Streams.print("setState_psX: p=" + String(p) + " and s=" + String(s), "printlog.txt");
        state.phase := phase;

        if (state.phase == 2) then
          assert(p <= p_crit, "setState_psX_error: pressure is higher than critical pressure");
          sat := setSat_p(p=p);
          assert(s >= sat.liq.s, "setState_psX_error: entropy is lower than saturated liquid entropy: this is single phase liquid");
          assert(s <= sat.vap.s, "setState_psX_error: entropy is higher than saturated vapor entropy: this is single phase vapor");
        else
          if (p <= p_crit) then
            // two-phase possible, check region
            if (p>0.98*p_crit) or (p<300*p_trip) then
              // Modelica.Utilities.Streams.print("close to critical or triple point, get saturation properties from EoS", "printlog.txt");
              sat := setSat_p(p=p);
            else
              // do a simple check first, quite often this is sufficient
              sat.Tsat := Ancillary.saturationTemperature_p(p=p);

              sat.liq.d := Ancillary.bubbleDensity_T(T=sat.Tsat);
              f := EoS.setHelmholtzDerivsFirst(d=sat.liq.d, T=sat.Tsat, phase=1);
              sat.liq.s := EoS.s(f);

              sat.vap.d := Ancillary.dewDensity_T(T=sat.Tsat);
              f := EoS.setHelmholtzDerivsFirst(d=sat.vap.d, T=sat.Tsat, phase=1);
              sat.vap.s := EoS.s(f);

              if ((s > sat.liq.s - abs(0.05*sat.liq.s)) and (s < sat.vap.s + abs(0.05*sat.vap.s))) then
                // Modelica.Utilities.Streams.print("two-phase state or close to it, get saturation properties from EoS", "printlog.txt");
                sat := setSat_p(p=p);
              end if;
            end if;

            // Modelica.Utilities.Streams.print("phase boundary determined; sat.liq.s=" + String(sat.liq.s) + " and sat.vap.s=" + String(sat.vap.s), "printlog.txt");
            if (s < sat.liq.s) then
              // Modelica.Utilities.Streams.print("single phase liquid", "printlog.txt");
              state.phase := 1;

              d_min := sat.liq.d*0.98;
              d_max := fluidLimits.DMAX*1.10;
              d_iter := sat.liq.d*1.10;

              T_min := fluidLimits.TMIN*0.99;
              T_max := sat.Tsat*1.02;
              T_iter:= sat.Tsat*0.95;
            elseif (s > sat.vap.s) then
              // Modelica.Utilities.Streams.print("single phase vapor", "printlog.txt");
              state.phase := 1;

              d_min := fluidLimits.DMIN;
              d_max := sat.vap.d*1.02;
              d_iter:= sat.vap.d/10;

              T_min := sat.Tsat*0.98;
              T_max := fluidLimits.TMAX*1.10;
              T_iter:= sat.Tsat*1.02;
            else
              // Modelica.Utilities.Streams.print("two-phase, all properties can be calculated from sat record", "printlog.txt");
              state.phase := 2;
            end if;

          else
            state.phase := 1;
            // Modelica.Utilities.Streams.print("p>=p_crit, only single-phase possible", "printlog.txt");
            if (s<=s_crit) then
              // Modelica.Utilities.Streams.print("s<=s_crit, single-phase super-critical liquid-like region", "printlog.txt");
              state_ref := setState_pT(p=p, T=T_crit);
              if (s<state_ref.s) then
                T_min := max(fluidLimits.TMIN*0.99, Ancillary.saturationTemperature_s_liq(s=s)*0.95);
                T_max := T_crit;
                T_iter := 0.95*T_crit;
              elseif (s>state_ref.s) then
                T_min := T_crit;
                T_max := fluidLimits.TMAX*1.1;
                T_iter := 1.05*T_crit;
              else
                T_min := 0.95*T_crit;
                T_max := 1.05*T_crit;
                T_iter := T_crit;
              end if;
              // T_iter := min(T_min*1.5, T_max);
              // T_iter := (T_min+T_max)/2;
              // T_iter := min(T_min*1.5, T_crit);
              // T_iter:= (T_min+T_crit)/2;
              // T_iter := T_crit;

              d_min := d_crit*0.99;
              d_max := fluidLimits.DMAX*1.1;
              d_iter := fluidLimits.DMAX*0.9;
              // d_iter := min({state_ref.d, fluidLimits.DMAX*0.7});
              // d_iter := d_crit*1.02;
            else
              // Modelica.Utilities.Streams.print("s>s_crit, single-phase super-critical vapour-like region", "printlog.txt");
              state_ref := setState_pd(p=p, d=d_crit);
              if (s>state_ref.s) then
                d_min := fluidLimits.DMIN*0.99;
                d_max := d_crit;
                d_iter := 0.95*d_crit;
              elseif (s<state_ref.s) then
                d_min := d_crit;
                d_max := fluidLimits.DMAX*1.1;
                d_iter := 1.05*d_crit;
              else
                d_min := 0.95*d_crit;
                d_max := 1.05*d_crit;
                d_iter := d_crit;
              end if;
              // d_iter:= d_crit;
              // d_iter := p/(R*T_crit);
              // d_iter:= (d_min+d_max)/2;

              T_min := T_crit*0.98;
              T_max := fluidLimits.TMAX*1.1;
              // T_iter := state_ref.T;
              // T_iter := min(T_crit*2, T_max);
              T_iter:= T_crit*1.3;
              // T_iter := (T_min+T_max)/2;
            end if;
          end if;
        end if;

        // phase and region determination finished !

        if (state.phase == 2) then
          // Modelica.Utilities.Streams.print("two-phase, SaturationProperties are already known", "printlog.txt");
          state.p := p;
          state.s := s;
          state.T := sat.Tsat;
          x := (s - sat.liq.s)/(sat.vap.s - sat.liq.s);
          state.d := 1/(1/sat.liq.d + x*(1/sat.vap.d - 1/sat.liq.d));
          state.u := sat.liq.u + x*(sat.vap.u - sat.liq.u);
          state.h := sat.liq.h + x*(sat.vap.h - sat.liq.h);
        else
          // Modelica.Utilities.Streams.print("single-phase, use 2D Newton-Raphson, start with d_iter=" + String(d_iter) + " (d_min=" + String(d_min) + " and d_max=" + String(d_max) + ") and T_iter=" + String(T_iter)+ " (T_min=" + String(T_min) + " and T_max=" + String(T_max) + ")", "printlog.txt");
          f := EoS.setHelmholtzDerivsSecond(d=d_iter, T=T_iter, phase=1);
          RES := {EoS.p(f)-p, EoS.s(f)-s};
          RSS := RES*RES/2;

          while ((RSS>tolerance) and (iter<iter_max)) loop
            iter := iter+1;

            // calculate Jacobian matrix, Newton Step vector, gradient and slope
            Jacobian := [EoS.dpdT(f), EoS.dpTd(f);
                         EoS.dsdT(f), EoS.dsTd(f)];
            NS := -Modelica.Math.Matrices.solve(Jacobian,RES) "-J^(-1)·F";
            grad := RES*Jacobian "F·J";
            slope := grad*NS;
            // Modelica.Utilities.Streams.print("  Jacobian=" + Modelica.Math.Matrices.toString(Jacobian) + "  NS=" + Modelica.Math.Vectors.toString(NS) + "  grad=" + Modelica.Math.Vectors.toString(grad) + "  slope=" + String(slope), "printlog.txt");
            assert(slope<0,"roundoff problem, input was p=" + String(p) + " and s=" + String(s), level=AssertionLevel.warning);

            // store old d_iter, T_iter and RSS
            d_iter_old := d_iter;
            T_iter_old := T_iter;
            RSS_old := RSS;

            // calculate new d_iter and T_iter using full Newton step
            d_iter := d_iter_old + NS[1];
            T_iter := T_iter_old + NS[2];

            /* // Babajee
      xy_old := {d_iter,T_iter};
      xy_med := xy_old + NS;
      xy_med[1] := max(d_iter, d_min);
      xy_med[1] := min(d_iter, d_max);
      xy_med[2] := max(T_iter, T_min);
      xy_med[2] := min(T_iter, T_max);
      f_med := EoS.setHelmholtzDerivsSecond(d=xy_med[1], T=xy_med[2], phase=1);
      RES_med := {EoS.p(f_med)-p, EoS.s(f_med)-s};
      Jacobian_med := [EoS.dpdT(f_med), EoS.dpTd(f_med);
                       EoS.dsdT(f_med), EoS.dsTd(f_med)];
      xy := xy_med - Modelica.Math.Matrices.inv(Jacobian)*RES_med;
      d_iter:=xy[1];
      T_iter:=xy[2]; */

            // check bounds
            d_iter := max(d_iter, d_min);
            d_iter := min(d_iter, d_max);
            T_iter := max(T_iter, T_min);
            T_iter := min(T_iter, T_max);
            /*if (d_iter>d_max) then
        NS := NS*(d_iter-d_iter_old)/(d_max-d_iter_old);
        d_iter := d_iter_old + NS[1];
        T_iter := T_iter_old + NS[2];
      end if;
      if (d_iter<d_min) then
        NS := NS*(d_min-d_iter_old)/(d_iter-d_iter_old);
        d_iter := d_iter_old + NS[1];
        T_iter := T_iter_old + NS[2];
      end if;
      if (T_iter>T_max) then
        NS := NS*(T_iter-T_iter_old)/(T_max-T_iter_old);
        d_iter := d_iter_old + NS[1];
        T_iter := T_iter_old + NS[2];
      end if;
      if (T_iter<T_min) then
        NS := NS*(T_min-T_iter_old)/(T_iter-T_iter_old);
        d_iter := d_iter_old + NS[1];
        T_iter := T_iter_old + NS[2];
      end if;*/

            // calculate new residual vector and residual sum of squares
            f := EoS.setHelmholtzDerivsSecond(d=d_iter, T=T_iter, phase=1);
            RES := {EoS.p(f)-p, EoS.s(f)-s};
            RSS := RES*RES/2;
            // Modelica.Utilities.Streams.print("iter=" + String(iter) + " d_iter=" + String(d_iter) + " T_iter=" + String(T_iter) + " RES_p=" + String(RES[1]) + " RES_s=" + String(RES[2]) + " RSS=" + String(RSS), "printlog.txt");

            // if RSS is not decreasing fast enough, the full Newton step is not used
            // instead, the linesearching / backtracking loop tries to find lambda such that RSS decreases
            while useLineSearch and (lambda>=lambda_min) and not (RSS<=(RSS_old+alpha*lambda*slope)) loop
              iterLineSearch := iterLineSearch+1;
              iter_max := iter_max+1;

              // decrease lambda
              if (iterLineSearch<2) then
                // first attempt
                lambda_temp := -slope/(2*(RSS-RSS_old-slope));
              else
                // subsequent attempts
                rhs1 := RSS   -RSS_old-lambda   *slope;
                rhs2 := RSS_ls-RSS_old-lambda_ls*slope;
                a := (           rhs1/lambda^2 -        rhs2/lambda_ls^2) / (lambda-lambda_ls);
                b := (-lambda_ls*rhs1/lambda^2 + lambda*rhs2/lambda_ls^2) / (lambda-lambda_ls);
                if (a==0) then
                  lambda_temp := -slope/(2*b);
                else
                  Discriminant := b*b-3*a*slope;
                  if (Discriminant<0) then
                    lambda_temp := 0.5*lambda;
                  elseif (b<=0) then
                    lambda_temp := (-b+sqrt(Discriminant))/(3*a);
                  else
                    lambda_temp := -slope/(b+sqrt(Discriminant));
                  end if;
                  // new lambda should be less or equal 0.5*previous lambda
                  lambda_temp := if (lambda_temp>0.5*lambda) then 0.5*lambda else lambda_temp;
                end if;
              end if;
              // store values for subsequent linesearch attempt
              lambda_ls := lambda;
              RSS_ls := RSS;

              // new lambda should be greater or equal 0.1*previous lambda
              lambda := max({lambda_temp, 0.1*lambda});

              // calculate new d_iter and T_iter using partial Newton step (lambda<1)
              d_iter := d_iter_old +lambda*NS[1];
              T_iter := T_iter_old +lambda*NS[2];

              // check bounds
              d_iter := max(d_iter, d_min);
              d_iter := min(d_iter, d_max);
              T_iter := max(T_iter, T_min);
              T_iter := min(T_iter, T_max);

              // calculate new residual vector and residual sum of squares
              f := EoS.setHelmholtzDerivsSecond(d=d_iter, T=T_iter, phase=1);
              RES := {EoS.p(f)-p, EoS.s(f)-s};
              RSS := RES*RES/2;
              // Modelica.Utilities.Streams.print("    linesearch attempt " + String(iterLineSearch) + ": lambda= " + String(lambda) + " d_iter=" + String(d_iter) + " T_iter= " + String(T_iter) + " RSS= " + String(RSS), "printlog.txt");

            end while;
            // reset iterLineSearch and lambda
            iterLineSearch := 0;
            lambda_temp := 1;
            lambda := 1;

          end while;
          // Modelica.Utilities.Streams.print("setState_ps total iteration steps " + String(iter), "printlog.txt");
          assert(iter<iter_max, "setState_psX did not converge, input was p=" + String(p) + " and s=" + String(s) + ", remaining residual is RES_p=" +String(RES[1]) + " and RES_s=" + String(RES[2]));

          state.p := p;
          state.s := s;
          state.d := d_iter;
          state.T := T_iter;
          state.u := EoS.u(f);
          state.h := EoS.h(f);
        end if;

      end setState_psX;

      function setState_pd "Return thermodynamic state as function of (p, d)"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Density d "Density";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output ThermodynamicState state "thermodynamic state record";

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant SpecificHeatCapacity R=Modelica.Constants.R/MM
        "specific gas constant";
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_trip=fluidConstants[1].triplePointTemperature;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;
        constant AbsolutePressure p_trip=fluidConstants[1].triplePointPressure;
        constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;
        constant Density dv_trip = Ancillary.dewDensity_T(T_trip);
        constant Density dl_trip = Ancillary.bubbleDensity_T(T_trip);

        EoS.HelmholtzDerivs f(d=d);
        SaturationProperties sat;
        MassFraction x "vapour quality";

        Temperature T_min=0.98*fluidLimits.TMIN;
        Temperature T_max=2*max(fluidLimits.TMAX, p/(R*d));
        Temperature T_iter=T_crit;
        AbsolutePressure RES_min;
        AbsolutePressure RES_max;
        AbsolutePressure RES_p;
        DerPressureByTemperature dpTd "(dp/dT)@d=const";
        Real gamma(min=0.1,max=1) = 1 "convergence speed, default=1";
        constant Real tolerance=1e-9 "relative tolerance for RES_p";
        Integer iter = 0;
        constant Integer iter_max = 200;

      algorithm
        // Modelica.Utilities.Streams.print(" ", "printlog.txt");
        // Modelica.Utilities.Streams.print("setState_pdX: p=" + String(p) + " and d=" + String(d), "printlog.txt");
        state.phase := phase;

        if (state.phase == 2) then
          assert(p <= p_crit, "setState_pdX_error: pressure is higher than critical pressure");
          sat := setSat_p(p=p);
          assert(d <= sat.liq.d, "setState_pdX_error: density is higher than saturated liquid density: this is single phase liquid");
          assert(d >= sat.vap.d, "setState_pdX_error: density is lower than saturated vapor density: this is single phase vapor");
        else
          if (p < p_crit) and (p>=p_trip) then
            // two-phase possible, do a region check
            if (p>0.98*p_crit) or (p<300*p_trip) then
              // Modelica.Utilities.Streams.print("close to critical or triple point", "printlog.txt");
              // get saturation properties from EoS
              sat := setSat_p(p=p);
            else
              // do a simple check first, quite often this is sufficient
              sat.Tsat := Ancillary.saturationTemperature_p(p=p);
              sat.liq.d := Ancillary.bubbleDensity_T(T=sat.Tsat);
              sat.vap.d := Ancillary.dewDensity_T(T=sat.Tsat);
              // Modelica.Utilities.Streams.print("setState_pd: sat.Tsat=" + String(sat.Tsat) + " and sat.liq.d=" + String(sat.liq.d) + " sat.vap.d=" + String(sat.vap.d) + ", simple check only", "printlog.txt");
              if ((d < sat.liq.d + abs(0.05*sat.liq.d)) and (d > sat.vap.d - abs(0.05*sat.vap.d))) then
                // Modelica.Utilities.Streams.print("setState_pd: p = " + String(p) + "d = " + String(d) + ", two-phase state or close to it", "printlog.txt");
                // get saturation properties from EoS
                sat := setSat_p(p=p);
              end if;
            end if;

            // Modelica.Utilities.Streams.print("setState_pd: phase boundary from EoS: sat.liq.d=" + String(sat.liq.d) + " sat.vap.d=" + String(sat.vap.d), "printlog.txt");
            if (d > sat.liq.d) then
              // Modelica.Utilities.Streams.print("single-phase liquid region", "printlog.txt");
              state.phase := 1;
              T_min := 0.98*Ancillary.saturationTemperature_d(d=d); // look at isobars in T,d-Diagram !!
              T_max := 1.02*sat.Tsat;
              T_iter := 1.1*T_min;
              // T_iter:= Ancillary.temperature_pd_Waals(p=p, d=d);
            elseif (d < sat.vap.d) then
              // Modelica.Utilities.Streams.print("single-phase vapour region", "printlog.txt");
              state.phase := 1;
              T_min := 0.99*sat.Tsat;
              T_iter:= Ancillary.temperature_pd_Waals(p=p, d=d);
            else
              // Modelica.Utilities.Streams.print("two-phase region, all properties can be calculated from sat record", "printlog.txt");
              state.phase := 2;
            end if;

          elseif (p < p_trip) then
            state.phase := 1;
            // very low pressure, behaves like an ideal gas
            T_iter := p/(d*R);
            // T_iter:= Ancillary.temperature_pd_Waals(p=p, d=d);

          elseif (p >= p_crit) then
            state.phase := 1;
            if (d>d_crit) then
              // Modelica.Utilities.Streams.print("p>p_crit and d>d_crit, single-phase super-critical liquid-like region", "printlog.txt");
              T_min := 0.98*Ancillary.saturationTemperature_d(d=d); // look at isobars in T,d-Diagram !!
              T_iter := 1.05*T_min;
              // T_iter:= Ancillary.temperature_pd_Waals(p=p, d=d);
            else
              // Modelica.Utilities.Streams.print("p>p_crit and d<=d_crit, single-phase super-critical vapour-like region", "printlog.txt");
              T_min := 0.98*T_crit;
              T_iter:= Ancillary.temperature_pd_Waals(p=p, d=d);
              T_max := 25*fluidLimits.TMAX;
            end if;
          else
            assert(false, "setState_pd: this should not happen, check p");
          end if;
        end if;
        // Modelica.Utilities.Streams.print("phase and region determination finshed, phase=" + String(state.phase) + ", T_min=" + String(T_min) + ", T_max=" + String(T_max) + ", T_iter=" + String(T_iter), "printlog.txt");

        // check bounds, van der Waals is not very accurate
        T_iter := max({T_iter, T_min, fluidLimits.TMIN});
        T_iter := min({T_iter, T_max, fluidLimits.TMAX});

        if (state.phase == 2) then
          // force two-phase, SaturationProperties are already known
          state.p := p;
          state.d := d;
          x := (1.0/d - 1.0/sat.liq.d)/(1.0/sat.vap.d - 1.0/sat.liq.d);
          state.T := sat.Tsat;
          state.h := sat.liq.h + x*(sat.vap.h - sat.liq.h);
          state.u := sat.liq.u + x*(sat.vap.u - sat.liq.u);
          state.s := sat.liq.s + x*(sat.vap.s - sat.liq.s);
        else
          // force single-phase

          // Modelica.Utilities.Streams.print("initialize bisection", "printlog.txt");
          // min
          f.T := T_min;
          f.tau := T_crit/f.T;
          f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
          RES_min := EoS.p(f) - p;
          // max
          f.T := T_max;
          f.tau := T_crit/f.T;
          f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
          RES_max := EoS.p(f) - p;
          // iter
          f.T := T_iter;
          f.tau := T_crit/f.T;
          f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
          RES_p := EoS.p(f) - p;

          assert((RES_min*RES_max<0), "setState_pd: T_min=" + String(T_min) + " and T_max=" + String(T_max) + " did not bracket the root, input was p=" + String(p) + " and d=" + String(d), level=AssertionLevel.warning);
          // thighten the bounds
          // opposite sign brackets the root
          if (RES_p*RES_min<0) then
            T_max := T_iter;
            RES_max := RES_p;
          elseif (RES_p*RES_max<0) then
            T_min := T_iter;
            RES_min := RES_p;
          end if;

          while ((abs(RES_p/p) > tolerance) and (iter<iter_max)) loop
            iter := iter+1;
            // gamma := iter/(iter+1);

            // calculate gradients with respect to temperature
            f.rtd := EoS.f_rtd(delta=f.delta, tau=f.tau);
            dpTd := EoS.dpTd(f);

            // print for debugging
            // Modelica.Utilities.Streams.print("Iteration step " +String(iter), "printlog.txt");
            // Modelica.Utilities.Streams.print("T_iter=" + String(T_iter) + " and dpdT=" + String(dpdT), "printlog.txt");

            // calculate better T_iter
            T_iter := T_iter - gamma/dpTd*RES_p;

            // check bounds, if out of bounds use bisection
            if (T_iter<0.95*T_min) or (T_iter>1.05*T_max) then
              // Modelica.Utilities.Streams.print("T_iter out of bounds, fallback to bisection method, step=" + String(iter) + ", T_iter=" + String(T_iter), "printlog.txt");
              T_iter := (T_min+T_max)/2;
            end if;

            // calculate new RES_p
            f.T := T_iter;
            f.tau := T_crit/f.T;
            f.rd  := EoS.f_rd(delta=f.delta, tau=f.tau);
            RES_p := EoS.p(f) - p;

            // thighten the bounds
            // opposite sign brackets the root
            if (RES_p*RES_min<0) then
              T_max := T_iter;
              RES_max := RES_p;
            elseif (RES_p*RES_max<0) then
              T_min := T_iter;
              RES_min := RES_p;
            end if;

          end while;
          // Modelica.Utilities.Streams.print("setState_pd required " + String(iter) + " iterations to find T=" + String(T_iter,significantDigits=6) + " for input p=" + String(p) + " and d=" + String(d), "printlog.txt");
          assert(iter<iter_max, "setState_pdX did not converge (RES_p=" + String(RES_p) + "), input was p=" + String(p) + " and d=" + String(d));

          state.p := p;
          state.d := d;
          state.T := T_iter;
          f := EoS.setHelmholtzDerivsFirst(d=state.d, T=state.T);
          state.h := EoS.h(f);
          state.u := EoS.u(f);
          state.s := EoS.s(f);
        end if;

      end setState_pd;

      function setState_Ts "Return thermodynamic state as function of (T, s)"
        extends Modelica.Icons.Function;
        input Temperature T "Temperature";
        input SpecificEntropy s "Entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output ThermodynamicState state "thermodynamic state record";

    protected
        MolarMass MM = fluidConstants[1].molarMass;
        SpecificHeatCapacity R=Modelica.Constants.R/MM "specific gas constant";
        Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        Temperature T_crit=fluidConstants[1].criticalTemperature;
        Temperature T_trip=fluidConstants[1].triplePointTemperature;

        EoS.HelmholtzDerivs f(T=T);
        SaturationProperties sat;
        MassFraction x "vapour quality";

        Density d_min=fluidLimits.DMIN;
        Density d_max=2*fluidLimits.DMAX; // extrapolation to higher densities should return reasonable values
        Density d_iter=d_crit;
        SpecificEntropy RES_s;
        SpecificEntropy RES_min;
        SpecificEntropy RES_max;
        DerEntropyByDensity dsdT "(ds/dd)@T=const";
        Real gamma(min=0.1,max=1) = 1 "convergence speed, default=1";
        constant Real tolerance=1e-9 "relative tolerance for RES_s (in J/kgK)";
        Integer iter = 0;
        constant Integer iter_max = 200;

      algorithm
        // Modelica.Utilities.Streams.print(" ", "printlog.txt");
        // Modelica.Utilities.Streams.print("setState_Ts: T=" + String(T) + " and s=" + String(s), "printlog.txt");
        state.phase := phase;

        if (state.phase == 2) then
          assert(T >= T_trip, "setState_Ts_error: pressure is lower than triple point pressure");
          assert(T <= T_crit, "setState_TsX_error: pressure is higher than critical pressure");
          sat := setSat_T(T=T);
          assert(s >= sat.liq.s, "setState_TsX_error: entropy is lower than saturated liquid entropy: this is single phase liquid");
          assert(s <= sat.vap.s, "setState_TsX_error: entropy is higher than saturated vapor entropy: this is single phase vapor");
        else
          if ((T < T_crit) and (T >= T_trip)) then
            // two-phase possible, check region
            if (T>0.95*T_crit) or (T<1.05*T_trip) then
              // Modelica.Utilities.Streams.print("close to critical or triple point, get saturation properties from EoS", "printlog.txt");
              sat := setSat_T(T=T);
            else
              // do a simple check first, quite often this is sufficient
              sat.liq.d := Ancillary.bubbleDensity_T(T=T);
              f := EoS.setHelmholtzDerivsFirst(T=T, d=sat.liq.d);
              sat.liq.s := EoS.s(f);

              sat.vap.d := Ancillary.dewDensity_T(T=T);
              f := EoS.setHelmholtzDerivsFirst(T=T, d=sat.vap.d);
              sat.vap.s := EoS.s(f);

              if ((s > sat.liq.s - abs(0.05*sat.liq.s)) and (s < sat.vap.s + abs(0.05*sat.vap.s))) then
                // Modelica.Utilities.Streams.print("two-phase state or close to it, get saturation properties from EoS", "printlog.txt");
                sat := setSat_T(T=T);
              end if;
            end if;

            if (s < sat.liq.s) then
              // Modelica.Utilities.Streams.print("single phase liquid: d is between dliq and rho_max", "printlog.txt");
              state.phase := 1;
              d_min := 0.98*sat.liq.d;
              d_iter := 1.02*sat.liq.d;
            elseif (s > sat.vap.s) then
              // Modelica.Utilities.Streams.print("single phase vapor: d is between 0 and sat.vap.d=" + String(sat.vap.d), "printlog.txt");
              state.phase := 1;
              d_max := sat.vap.d;
              d_iter := sat.vap.d/100;
            else
              // Modelica.Utilities.Streams.print("two-phase, all properties can be calculated from sat record", "printlog.txt");
              state.phase := 2;
            end if;

          else
            // Modelica.Utilities.Streams.print("T>=T_crit or T<T_trip, only single phase possible, do not change dmin and dmax", "printlog.txt");
            state.phase := 1;
            d_iter := d_crit/100;
          end if;
        end if;

        // Modelica.Utilities.Streams.print("phase and region determination finished, d_min=" + String(d_min) + ", d_max=" + String(d_max) + " and d_iter=" + String(d_iter), "printlog.txt");

        if (state.phase == 2) then
          // force two-phase, SaturationProperties are already known
          state.T := T;
          state.s := s;
          state.p := sat.psat;
          x := (s - sat.liq.s)/(sat.vap.s - sat.liq.s);
          state.d := 1.0/(1.0/sat.liq.d + x*(1.0/sat.vap.d - 1.0/sat.liq.d));
          state.h := sat.liq.h + x*(sat.vap.h - sat.liq.h);
          state.u := sat.liq.u + x*(sat.vap.u - sat.liq.u);
        else
          // force single-phase

          // Modelica.Utilities.Streams.print("initialize bisection", "printlog.txt");
          // min
          f := EoS.setHelmholtzDerivsFirst(T=T, d=d_min);
          RES_min := EoS.s(f) - s;
          // max
          f := EoS.setHelmholtzDerivsFirst(T=T, d=d_max);
          RES_max := EoS.s(f) - s;
          // iter
          f := EoS.setHelmholtzDerivsFirst(T=T, d=d_iter);
          RES_s := EoS.s(f) - s;

          //assert((RES_min*RES_max<0), "setState_Ts: d_min and d_max did not bracket the root, input was T=" + String(T) + " and s=" + String(s));
          // thighten the bounds
          // opposite sign brackets the root
          if (RES_s*RES_min<0) then
            d_max := d_iter;
            RES_max := RES_s;
          elseif (RES_s*RES_max<0) then
            d_min := d_iter;
            RES_min := RES_s;
          end if;

          while ((abs(RES_s/s) > tolerance) and (iter<iter_max)) loop
            iter := iter+1;
            // gamma := iter/(iter+1);

            // calculate missing parts of f, then calculate gradient with respect to density
            f.rtd := EoS.f_rtd(delta=f.delta, tau=f.tau);
            dsdT := EoS.dsdT(f);

            // print for debugging
            // Modelica.Utilities.Streams.print("Iteration step " +String(iter), "printlog.txt");
            // Modelica.Utilities.Streams.print("d_iter=" + String(d_iter) + " and dsdT=" + String(dsdT), "printlog.txt");

            // calculate better d_iter
            d_iter := d_iter - gamma/dsdT*RES_s;

            // check bounds, if out of bounds use bisection
            if (d_iter<d_min) or (d_iter>d_max) then
              // Modelica.Utilities.Streams.print("d_iter out of bounds, fallback to bisection method, step=" + String(iter) + ", d_iter=" + String(d_iter), "printlog.txt");
              d_iter := (d_min+d_max)/2;
            end if;

            // calculate new RES_s
            f := EoS.setHelmholtzDerivsFirst(T=T, d=d_iter);
            RES_s := EoS.s(f) - s;

            // thighten the bounds
            // opposite sign brackets the root
            if (RES_s*RES_min<0) then
              d_max := d_iter;
              RES_max := RES_s;
            elseif (RES_s*RES_max<0) then
              d_min := d_iter;
              RES_min := RES_s;
            end if;

          end while;
          // Modelica.Utilities.Streams.print("setState_Ts total iteration steps " + String(iter), "printlog.txt");
          // Modelica.Utilities.Streams.print(" ", "printlog.txt");
          assert(iter<iter_max, "setState_Ts did not converge, input was T=" + String(T) + " and s=" + String(s));

          state.T := T;
          state.s :=s;
          state.d := d_iter;
          state.p := EoS.p(f);
          state.h := EoS.h(f);
          state.u := EoS.u(f);
        end if;

      end setState_Ts;

      redeclare function extends temperature
      "returns temperature from given ThermodynamicState"
      // inherited from: PartialMedium
      // inherits input state and output T
      algorithm
        T := state.T;
      annotation(Inline = true);
      end temperature;

      redeclare function extends density
      "returns density from given ThermodynamicState"
      // inherited from: PartialMedium
      // inherits input state and output d
      algorithm
        d := state.d;
      annotation(Inline = true);
      end density;

      redeclare function extends pressure
      "returns pressure from given ThermodynamicState"
      // inherited from: PartialMedium
      // inherits input state and output p
      algorithm
        p := state.p;
      annotation(Inline = true);
      end pressure;

      redeclare function extends specificEntropy
      "returns specificEntropy from given ThermodynamicState"
      // inherited from: PartialMedium
      // inherits input state and output h
      algorithm
        s := state.s;
      annotation(Inline = true);
      end specificEntropy;

      redeclare function extends specificEnthalpy
      "returns specificEnthalpy from given ThermodynamicState"
      // inherited from: PartialMedium
      // inherits input state and output h
      algorithm
        h := state.h;
      annotation(Inline = true);
      end specificEnthalpy;

      redeclare function vapourQuality "returns the vapour quality"

      input ThermodynamicState state;
      output MassFraction x;

      algorithm
      x := vapourQuality_sat(state=state, sat=setSat_T(state.T));
      annotation (Inline=true);
      end vapourQuality;

      function vapourQuality_sat "returns the vapour quality"

        input ThermodynamicState state;
        input SaturationProperties sat;
        output MassFraction x;

    protected
        Temperature T_trip=fluidConstants[1].triplePointTemperature;
        Temperature T_crit=fluidConstants[1].criticalTemperature;

      algorithm
        assert(state.T >= T_trip, "vapourQuality warning: Temperature is lower than triple-point temperature", level=AssertionLevel.warning);
        assert(state.T <= T_crit, "vapourQuality warning: Temperature is higher than critical temperature", level=AssertionLevel.warning);

        if state.d <= sat.vap.d then
          x := 1;
        elseif state.d >= sat.liq.d then
          x := 0;
        else
          x := (1.0/state.d - 1.0/sat.liq.d)/(1.0/sat.vap.d - 1.0/sat.liq.d);
        end if;

      end vapourQuality_sat;

      redeclare function extends specificHeatCapacityCp
      "returns the isobaric specific heat capcacity"
      //input state
      //output cp

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          cp := EoS.dhTd(f) - EoS.dhdT(f)*EoS.dpTd(f)/EoS.dpdT(f);
        elseif (state.phase == 2) then
          assert(false, "specificHeatCapacityCp warning: in the two-phase region cp is infinite", level=AssertionLevel.warning);
          cp := Modelica.Constants.inf;
        end if;

      end specificHeatCapacityCp;

      redeclare function extends specificHeatCapacityCv
      "returns the isochoric specific heat capcacity"
      //input state
      //output cv

    protected
        EoS.HelmholtzDerivs f;

        SaturationProperties sat;
        MassFraction x "vapour quality";
        DerPressureByTemperature dpT;
        EoS.HelmholtzDerivs fl;
        EoS.HelmholtzDerivs fv;

        DerEnergyByTemperature duT_liq;
        DerEnergyByTemperature duT_vap;
        DerDensityByTemperature ddT_liq;
        DerDensityByTemperature ddT_vap;
        DerVolumeByTemperature dvT_liq;
        DerVolumeByTemperature dvT_vap;
        DerFractionByTemperature dxTv;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          cv := EoS.duTd(f);

        elseif (state.phase == 2) then
          sat := setSat_T(T=state.T);
          x := vapourQuality_sat(state=state, sat=sat);
          dpT := saturationPressure_derT_sat(sat=sat);

          fl := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.liq.d, phase=1);
          fv := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.vap.d, phase=1);

          duT_liq := EoS.duTd(fl)-EoS.dudT(fl)*EoS.dpTd(fl)/EoS.dpdT(fl) + EoS.dudT(fl)/EoS.dpdT(fl)*dpT;
          duT_vap := EoS.duTd(fv)-EoS.dudT(fv)*EoS.dpTd(fv)/EoS.dpdT(fv) + EoS.dudT(fv)/EoS.dpdT(fv)*dpT;
          ddT_liq := -EoS.dpTd(fl)/EoS.dpdT(fl) + 1.0/EoS.dpdT(fl)*dpT;
          ddT_vap := -EoS.dpTd(fv)/EoS.dpdT(fv) + 1.0/EoS.dpdT(fv)*dpT;
          dvT_liq := -1/sat.liq.d^2 * ddT_liq;
          dvT_vap := -1/sat.vap.d^2 * ddT_vap;
          dxTv :=(x*dvT_vap + (1 - x)*dvT_liq)/(1/sat.liq.d - 1/sat.vap.d);

          cv := duT_liq + dxTv*(sat.vap.u-sat.liq.u) + x*(duT_vap-duT_liq);
        end if;

      end specificHeatCapacityCv;

      redeclare function extends velocityOfSound
      "returns the speed or velocity of sound"
      // a^2 := (dp/dd)@s=const

    protected
        EoS.HelmholtzDerivs f;

        SaturationProperties sat;
        MassFraction x "vapour quality";
        DerTemperatureByPressure dTp;
        EoS.HelmholtzDerivs fl;
        EoS.HelmholtzDerivs fv;

        DerEntropyByPressure dsp_liq;
        DerEntropyByPressure dsp_vap;
        DerDensityByPressure ddp_liq;
        DerDensityByPressure ddp_vap;
        DerVolumeByPressure dvp_liq;
        DerVolumeByPressure dvp_vap;
        DerVolumeByPressure dvps;
        DerFractionByPressure dxps;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          a := sqrt(EoS.dpdT(f)-EoS.dpTd(f)*EoS.dsdT(f)/EoS.dsTd(f));
        elseif (state.phase == 2) then
          sat := setSat_T(T=state.T);
          x := vapourQuality_sat(state=state, sat=sat);
          dTp := saturationTemperature_derp_sat(sat=sat);

          fl := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.liq.d, phase=1);
          fv := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.vap.d, phase=1);

          dsp_liq := EoS.dsdT(fl)/EoS.dpdT(fl) + (EoS.dsTd(fl)-EoS.dsdT(fl)*EoS.dpTd(fl)/EoS.dpdT(fl))*dTp;
          dsp_vap := EoS.dsdT(fv)/EoS.dpdT(fv) + (EoS.dsTd(fv)-EoS.dsdT(fv)*EoS.dpTd(fv)/EoS.dpdT(fv))*dTp;
          ddp_liq := 1.0/EoS.dpdT(fl) - EoS.dpTd(fl)/EoS.dpdT(fl)*dTp;
          ddp_vap := 1.0/EoS.dpdT(fv) - EoS.dpTd(fv)/EoS.dpdT(fv)*dTp;
          dvp_liq := -1.0/sat.liq.d^2 * ddp_liq;
          dvp_vap := -1.0/sat.vap.d^2 * ddp_vap;
          dxps :=(x*dsp_vap + (1 - x)*dsp_liq)/(sat.liq.s - sat.vap.s);

          dvps := dvp_liq + dxps*(1.0/sat.vap.d-1.0/sat.liq.d) + x*(dvp_vap-dvp_liq);
          a := sqrt(-1.0/(state.d*state.d*dvps));
        end if;
      end velocityOfSound;

      redeclare function extends isobaricExpansionCoefficient
      "returns 1/v*(dv/dT)@p=const"
      //input state
      //output beta

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f:=EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          beta := 1.0/state.d*EoS.dpTd(f)/EoS.dpdT(f);
        elseif (state.phase == 2) then
          assert(false, "isobaricExpansionCoefficient warning: in the two-phase region beta is zero", level=AssertionLevel.warning);
          beta := Modelica.Constants.small;
        end if;
      end isobaricExpansionCoefficient;

      redeclare function extends isothermalCompressibility
      "returns -1/v*(dv/dp)@T=const"
      //input state and
      //output kappa are inherited from PartialMedium

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f:=EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          kappa := 1.0/state.d/EoS.dpdT(f);
        elseif (state.phase == 2) then
          assert(false, "isothermalCompressibility warning: in the two-phase region kappa is infinite", level=AssertionLevel.warning);
          kappa := Modelica.Constants.inf;
        end if;
      end isothermalCompressibility;

      redeclare function extends isentropicExponent
      "returns -v/p*(dp/dv)@s=const"
      // also known as isentropic expansion coefficient

      algorithm
        gamma := state.d/state.p*velocityOfSound(state)^2;
        annotation(Inline = true);
      end isentropicExponent;

      redeclare function extends isentropicEnthalpy
      "returns isentropic enthalpy"

      algorithm
        h_is := specificEnthalpy(setState_ps(p=p_downstream, s=specificEntropy(refState)));
        annotation(Inline = true);
      end isentropicEnthalpy;

      function jouleThomsonCoefficient
      "returns Joule-Thomson-Coefficient (dT/dp)@h=const"
        input ThermodynamicState state;
        output DerTemperatureByPressure mu;

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=state.phase);
          mu := 1.0/(EoS.dpTd(f)-EoS.dpdT(f)*EoS.dhTd(f)/EoS.dhdT(f));
        elseif (state.phase == 2) then
          mu := saturationTemperature_derp_sat(sat=setSat_T(T=state.T));
        end if;
      end jouleThomsonCoefficient;

      function isothermalThrottlingCoefficient
      "returns isothermal throttling coefficient (dh/dp)@T=const"
        input ThermodynamicState state;
        output DerEnthalpyByPressure delta_T;

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          delta_T := EoS.dhdT(f)/EoS.dpdT(f);
        elseif (state.phase == 2) then
          assert(false, "isothermalThrottlingCoefficient warning: in the two-phase region delta_T is infinite", level=AssertionLevel.warning);
          delta_T := Modelica.Constants.inf;
        end if;
      end isothermalThrottlingCoefficient;

      redeclare replaceable function extends dynamicViscosity
      "Returns dynamic Viscosity"
        // inherits input state and output eta

    protected
        constant Real micro=1e-6;

      algorithm
        // assert(state.phase <> 2, "dynamicViscosity warning: property not defined in two-phase region", level=AssertionLevel.warning);

        // RefProp results are in µPa·s where µ means micro or 1E-6 but SI default is Pa·s
        eta := micro*(Transport.dynamicViscosity_dilute(state)
                    + Transport.dynamicViscosity_initial(state)
                    + Transport.dynamicViscosity_residual(state));

        annotation (Documentation(info="<html>
<p>
This model is identical to the RefProp VS1 or VS2 model.

The viscosity is split into three contributions: 
zero density (dilute gas) viscosity eta_0, 
initial density contribution eta_1
and residual contribution eta_r.

This allows to develop functions for each contribution seperately.
The so called background viscosity is the sum of initial and residual viscosity.

At the critical point and a small region around the critical point, the viscosity is enhanced. 
As this critical enhancement is small, it is neglected here.

Special thanks go to Eric W. Lemmon for answering all my emails 
and programming a special version of RefProp that outputs also intermediate values.

</p>

<dl>
<dt> Lemmon, Eric W.; Huber, M. L. and McLinden, M. O.</dt>
<dd> <b>NIST Standard Reference Database 23: Reference Fluid Thermodynamic and Transport Properties - REFPROP. 9.0</b><br>
     National Institute of Standards and Technology, Standard Reference Data Program. Gaithersburg<br>
     URL: <a href=\"http://www.nist.gov/srd/nist23.cfm\">http://www.nist.gov/srd/nist23.cfm</a>
</dd>
<dt>Vogel, E.; Küchenmeister, C. and Birch, E.</dt>
<dd> <b>Reference correlation of the viscosity of propane</b>.<br>
     Journal of Thermophysics (1998) 10, 417-426.<br>
     DOI: <a href=\"http://dx.doi.org/10.1007/BF01133538\">10.1007/BF01133538</a>
</dd>
</dl>
</html>"));
      end dynamicViscosity;

      redeclare replaceable function extends thermalConductivity
      "Return thermal conductivity"
        // inherits input state and output lambda

      algorithm
        assert(state.phase <> 2, "thermalConductivity warning: property not defined in two-phase region", level=AssertionLevel.warning);

        lambda := ( Transport.thermalConductivity_dilute(state)
                  + Transport.thermalConductivity_residual(state)
                  + Transport.thermalConductivity_critical(state));

        annotation (Documentation(info="<html>
  <p>
The thermal conductivity (TC) is split into three parts: ideal gas TC lamda_0, residual TC lambda_r and critical TC enhancement lambda_c.
Sometimes the residual TC is again split into two parts.
This allows to develop functions for each contribution seperately.
The sum of ideal gas TC and residual TC is called background TC.
Ideal gas TC depends on Temperature only and can be modelled by a quadratic function.
Residual TC is also modeled by a polynominal.
At the critical point TC becomes infinite; TC is enhanced for a large region around the critical point.
The critical enhancement can be described by various alternative approaches.
Here, the simplified approach as suggested by Olchowy and Sengers is implemented.

Special thanks go to Eric W. Lemmon for answering all my emails 
and programming a special version of RefProp that outputs also intermediate values.

</p>

</html>"));
      end thermalConductivity;

      redeclare replaceable function extends surfaceTension
      "Return surface tension sigma in the two phase region"
          // inherits input saturationProperties sat and output SurfaceTension sigma
          // this algorithm uses T only
          // liquid and vapour density are used in some mixture models

    protected
        Temperature T_trip=fluidConstants[1].triplePointTemperature;
        Temperature T_crit=fluidConstants[1].criticalTemperature;

        Real[size(surfaceTensionCoefficients.coeffs, 1)] a=
            surfaceTensionCoefficients.coeffs[:, 1];
        Real[size(surfaceTensionCoefficients.coeffs, 1)] n=
            surfaceTensionCoefficients.coeffs[:, 2];
        Real X "reduced temperature difference";

      algorithm
        assert(sat.Tsat >= T_trip, "vapourQuality error: Temperature is lower than triple-point temperature");
        assert(sat.Tsat <= T_crit, "vapourQuality error: Temperature is higher than critical temperature");

        X := (T_crit - sat.Tsat)/T_crit;
        sigma := sum(a[i]*X^n[i] for i in 1:size(a, 1));

        annotation (Documentation(info="<html>
  <p>
This is an implementation of the model as suggested by Somayajulu, G.R., 
which is an extension of the van der Waals surface tension correlation. 
The extended version has up to three terms with two parameters each.
</p>
<dl>
<dt>Somayajulu, G.R.</dt>
<dd> <b>A generalized equation for surface tension from the triple point to the critical point</b>.<br>
     International Journal of Thermophysics (1988) 9, 559-566.<br>
     DOI: <a href=\"http://dx.doi.org/10.1007/BF00503154\">10.1007/BF00503154</a>
</dd>
<dt>Van der Waals, J.D.</dt>
<dd> <b>Thermodynamische Theorie der Kapillarität unter Voraussetzung stetiger Dichteänderung</b>.<br>
     Zeitschrift für Physikalische Chemie (1894) 13, 657-725.
</dd>
</dl>
</html>"));
      end surfaceTension;

      redeclare function extends bubbleEnthalpy
      "returns specificEnthalpy from given SaturationProperties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output hl
      algorithm
        hl := sat.liq.h;
      annotation(Inline = true);
      end bubbleEnthalpy;

      redeclare function extends dewEnthalpy
      "returns specificEnthalpy from given SaturationProperties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output hl
      algorithm
        hv := sat.vap.h;
      annotation(Inline = true);
      end dewEnthalpy;

      redeclare function extends dewEntropy
      "returns specificEntropy from given SaturationProperties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output hl
      algorithm
        sv := sat.vap.s;
      annotation(Inline = true);
      end dewEntropy;

      redeclare function extends bubbleEntropy
      "returns specificEntropy from given SaturationProperties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output hl
      algorithm
        sl := sat.liq.s;
      annotation(Inline = true);
      end bubbleEntropy;

      redeclare function extends dewDensity
      "returns density from given SaturationProperties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output hl
      algorithm
        dv := sat.vap.d;
      annotation(Inline = true);
      end dewDensity;

      redeclare function extends bubbleDensity
      "returns density from given SaturationProperties"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output hl
      algorithm
        dl := sat.liq.d;
      annotation(Inline = true);
      end bubbleDensity;

      redeclare function extends saturationTemperature

      algorithm
        T := saturationTemperature_sat(setSat_p(p=p));
      annotation(Inline = true);
      end saturationTemperature;

      redeclare function extends saturationTemperature_derp
      "returns (dT/dp)@sat"

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant AbsolutePressure p_crit=fluidConstants[1].criticalPressure;

      algorithm
        dTp := if p<p_crit then saturationTemperature_derp_sat(sat=setSat_p(p=p)) else 1.0/pressure_derT_d(state=setState_pd(p=p, d=d_crit));
      annotation(Inline = true);
      end saturationTemperature_derp;

      redeclare function saturationTemperature_derp_sat "returns (dT/dp)@sat"
      // does not extend, because base class already defines an algorithm
      input SaturationProperties sat;
      output DerTemperatureByPressure dTp;

      algorithm
        // Clausius-Clapeyron equation
        dTp := (1.0/sat.vap.d-1.0/sat.liq.d)/(sat.vap.s-sat.liq.s);
      annotation(Inline = true);
      end saturationTemperature_derp_sat;

      redeclare function extends saturationPressure

      algorithm
        p := saturationPressure_sat(setSat_T(T=T));
      annotation(Inline = true);
      end saturationPressure;

      function saturationPressure_derT "returns (dp/dT)@sat"

      input Temperature T;
      output DerPressureByTemperature dpT;

    protected
        constant MolarMass MM = fluidConstants[1].molarMass;
        constant Density d_crit=MM/fluidConstants[1].criticalMolarVolume;
        constant Temperature T_crit=fluidConstants[1].criticalTemperature;

      algorithm
        dpT := if T< T_crit then saturationPressure_derT_sat(sat=setSat_T(T=T)) else pressure_derT_d(state=setState_dT(T=T, d=d_crit));
      annotation(Inline = true);
      end saturationPressure_derT;

      function saturationPressure_derT_sat "returns (dp/dT)@sat"

      input SaturationProperties sat;
      output DerPressureByTemperature dpT;

      algorithm
        // Clausius-Clapeyron equation
        dpT := (sat.vap.s-sat.liq.s)/(1.0/sat.vap.d-1.0/sat.liq.d);
      annotation(Inline = true);
      end saturationPressure_derT_sat;

      redeclare function extends dBubbleDensity_dPressure
      "Return bubble point density derivative"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output ddldp

    protected
        EoS.HelmholtzDerivs f = EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.liq.T);
        DerDensityByPressure ddpT = 1.0/EoS.dpdT(f);
        DerDensityByTemperature ddTp = -EoS.dpTd(f)/EoS.dpdT(f);
        DerTemperatureByPressure dTp = saturationTemperature_derp(p=sat.psat);

      algorithm
        ddldp := ddpT + ddTp*dTp;
      annotation(Inline = true);
      end dBubbleDensity_dPressure;

      redeclare function extends dDewDensity_dPressure
      "Return dew point density derivative"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output ddvdp

    protected
        EoS.HelmholtzDerivs f = EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.vap.T);
        DerDensityByPressure ddpT = 1.0/EoS.dpdT(f);
        DerDensityByTemperature ddTp = -EoS.dpTd(f)/EoS.dpdT(f);
        DerTemperatureByPressure dTp = saturationTemperature_derp(p=sat.psat);

      algorithm
        ddvdp := ddpT + ddTp*dTp;
      annotation(Inline = true);
      end dDewDensity_dPressure;

      redeclare function extends dBubbleEnthalpy_dPressure
      "Return bubble point enthalpy derivative"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output dhldp

    protected
        EoS.HelmholtzDerivs f = EoS.setHelmholtzDerivsSecond(d=sat.liq.d, T=sat.liq.T);
        DerEnthalpyByPressure dhpT = EoS.dhdT(f)/EoS.dpdT(f);
        DerEnthalpyByTemperature dhTp = EoS.dhTd(f) - EoS.dhdT(f)*EoS.dpTd(f)/EoS.dpdT(f);
        DerTemperatureByPressure dTp = saturationTemperature_derp(p=sat.psat);

      algorithm
        dhldp := dhpT + dhTp*dTp;
      annotation(Inline = true);
      end dBubbleEnthalpy_dPressure;

      redeclare function extends dDewEnthalpy_dPressure
      "Return dew point enthalpy derivative"
      // inherited from: PartialTwoPhaseMedium
      // inherits input sat and output dhvdp

    protected
        EoS.HelmholtzDerivs f = EoS.setHelmholtzDerivsSecond(d=sat.vap.d, T=sat.vap.T);
        DerEnthalpyByPressure dhpT = EoS.dhdT(f)/EoS.dpdT(f);
        DerEnthalpyByTemperature dhTp = EoS.dhTd(f) - EoS.dhdT(f)*EoS.dpTd(f)/EoS.dpdT(f);
        DerTemperatureByPressure dTp = saturationTemperature_derp(p=sat.psat);

      algorithm
        dhvdp := dhpT + dhTp*dTp;
      annotation(Inline = true);
      end dDewEnthalpy_dPressure;

      redeclare function density_ph "returns density for given p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Enthalpy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output Density d "density";

      algorithm
        d := density_ph_state(p=p, h=h, state=setState_ph(p=p, h=h, phase=phase));

      annotation (
        Inline=true,
        inverse(h=specificEnthalpy_pd(p=p, d=d, phase=phase)));
      end density_ph;

      function density_ph_state "returns density for given p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Enthalpy";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        output Density d "density";

      algorithm
        d := density(state);

      annotation (
        Inline=false,
        LateInline=true,
        derivative(noDerivative=state)=density_ph_der);
      end density_ph_state;

      function density_ph_der "time derivative of density_ph"

        input AbsolutePressure p;
        input SpecificEnthalpy h;
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        input Real p_der "time derivative of pressure";
        input Real h_der "time derivative of specific enthalpy";
        output Real d_der "time derivative of density";

      algorithm
        d_der := p_der*density_derp_h(state=state)
               + h_der*density_derh_p(state=state);

      annotation (Inline=true);
      end density_ph_der;

      redeclare function extends density_derp_h
      "returns density derivative (dd/dp)@h=const"
      //input state
      //output ddph

    protected
        EoS.HelmholtzDerivs f;

        SaturationProperties sat;
        MassFraction x "vapour quality";
        DerTemperatureByPressure dTp;
        EoS.HelmholtzDerivs fl;
        EoS.HelmholtzDerivs fv;

        DerEnthalpyByPressure dhp_liq;
        DerEnthalpyByPressure dhp_vap;
        DerDensityByPressure ddp_liq;
        DerDensityByPressure ddp_vap;
        DerVolumeByPressure dvp_liq;
        DerVolumeByPressure dvp_vap;
        DerVolumeByPressure dvph;
        DerFractionByPressure dxph;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=state.phase);
          ddph := 1.0/(EoS.dpdT(f) - EoS.dpTd(f)*EoS.dhdT(f)/EoS.dhTd(f));
        elseif (state.phase == 2) then
          sat := setSat_T(T=state.T);
          x := vapourQuality_sat(state=state, sat=sat);
          dTp := saturationTemperature_derp_sat(sat=sat);

          fl := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.liq.d, phase=1);
          fv := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.vap.d, phase=1);

          dhp_liq := EoS.dhdT(fl)/EoS.dpdT(fl) + (EoS.dhTd(fl)-EoS.dhdT(fl)*EoS.dpTd(fl)/EoS.dpdT(fl))*dTp;
          dhp_vap := EoS.dhdT(fv)/EoS.dpdT(fv) + (EoS.dhTd(fv)-EoS.dhdT(fv)*EoS.dpTd(fv)/EoS.dpdT(fv))*dTp;
          ddp_liq := 1.0/EoS.dpdT(fl) - EoS.dpTd(fl)/EoS.dpdT(fl)*dTp;
          ddp_vap := 1.0/EoS.dpdT(fv) - EoS.dpTd(fv)/EoS.dpdT(fv)*dTp;
          dvp_liq := -1.0/sat.liq.d^2 * ddp_liq;
          dvp_vap := -1.0/sat.vap.d^2 * ddp_vap;
          dxph :=(x*dhp_vap + (1 - x)*dhp_liq)/(sat.liq.h - sat.vap.h);

          dvph := dvp_liq + dxph*(1.0/sat.vap.d-1.0/sat.liq.d) + x*(dvp_vap-dvp_liq);
          ddph := -state.d*state.d*dvph;
        end if;
      end density_derp_h;

      redeclare function extends density_derh_p
      "returns density derivative (dd/dh)@p=const"
      //input state
      //output ddhp

    protected
        EoS.HelmholtzDerivs f;
        SaturationProperties sat;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=state.phase);
          ddhp := 1.0/(EoS.dhdT(f) - EoS.dhTd(f)*EoS.dpdT(f)/EoS.dpTd(f));
        elseif (state.phase == 2) then
          sat:=setSat_T(T=state.T);
          // dvhp = (v"-v')/(h"-h')
          // ddhp = -d^2 * dvhp
          ddhp := -state.d^2*(1/sat.liq.d-1/sat.vap.d)/(sat.liq.h-sat.vap.h);
        end if;
      end density_derh_p;

      redeclare function temperature_ph "returns temperature for given p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Enthalpy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output Temperature T "Temperature";

      algorithm
        T := temperature_ph_state(p=p, h=h, state=setState_ph(p=p, h=h, phase=phase));

      annotation (
        Inline=true,
        inverse(h=specificEnthalpy_pT(p=p, T=T, phase=phase)));
      end temperature_ph;

      function temperature_ph_state "returns temperature for given p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Enthalpy";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        output Temperature T "Temperature";

      algorithm
        T := temperature(state);

      annotation (
        Inline=false,
        LateInline=true,
        inverse(h=specificEnthalpy_pT_state(p=p, T=T, state=state)),
        derivative(noDerivative=state)=temperature_ph_der);
      end temperature_ph_state;

      function temperature_ph_der "time derivative of temperature_ph"

        input AbsolutePressure p;
        input SpecificEnthalpy h;
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        input Real p_der "time derivative of pressure";
        input Real h_der "time derivative of specific enthalpy";
        output Real T_der "time derivative of temperature";

      algorithm
        T_der := p_der*jouleThomsonCoefficient(state=state)
               + h_der/specificHeatCapacityCp(state=state);

      annotation (Inline=true);
      end temperature_ph_der;

      function specificEntropy_ph
      "returns specific entropy for a given p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific Enthalpy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output SpecificEntropy s "Specific Entropy";

      algorithm
        s := specificEntropy_ph_state(p=p, h=h, state=setState_ph(p=p, h=h, phase=phase));

      annotation (
        Inline=true,
        inverse(h=specificEnthalpy_ps(p=p, s=s, phase=phase)));
      end specificEntropy_ph;

      function specificEntropy_ph_state
      "returns specific entropy for a given p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific Enthalpy";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        output SpecificEntropy s "Specific Entropy";

      algorithm
        s := specificEntropy(state);

      annotation (
        Inline=false,
        LateInline=true,
        derivative(noDerivative=state)=specificEntropy_ph_der);
      end specificEntropy_ph_state;

      function specificEntropy_ph_der

        input AbsolutePressure p;
        input SpecificEnthalpy h;
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        input Real p_der "time derivative of pressure";
        input Real h_der "time derivative of specific enthalpy";
        output Real s_der "time derivative of specific entropy";

      algorithm
        s_der := p_der*(-1.0/(state.d*state.T))
               + h_der*(1.0/state.T);

      annotation (
        Inline=true);
      end specificEntropy_ph_der;

      redeclare function density_pT "Return density from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output Density d "Density";

      algorithm
        d := density_pT_state(p=p, T=T, state=setState_pT(p=p, T=T, phase=phase));

      annotation (
        Inline=true,
        inverse(p=pressure_dT(d=d, T=T, phase=phase),
                T=temperature_pd(p=p, d=d, phase=phase)));
      end density_pT;

      function density_pT_state
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        output Density d "Density";

      algorithm
        d := density(state);

      annotation (
        Inline=false,
        LateInline=true,
        inverse(p=pressure_dT_state(d=d, T=T, state=state)),
        derivative(noDerivative=state)=density_pT_der);
      end density_pT_state;

      function density_pT_der "time derivative of density_pT"

        input AbsolutePressure p;
        input Temperature T;
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        input Real p_der "time derivative of pressure";
        input Real T_der "time derivative of temperature";
        output Real d_der "time derivative of density";

      algorithm
        d_der := p_der*density_derp_T(state=state)
               + T_der*density_derT_p(state=state);

      annotation (Inline=true);
      end density_pT_der;

      redeclare function extends density_derp_T
      "returns density derivative (dd/dp)@T=const"
      //input state and output ddpT are inherited

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          ddpT := 1.0/EoS.dpdT(f);
        elseif (state.phase == 2) then
          ddpT := Modelica.Constants.inf; // divide by zero
        end if;
      end density_derp_T;

      redeclare function extends density_derT_p
      "returns density derivative (dd/dT)@p=const"
      //input state and output ddTp are inherited

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=1);
          ddTp := -EoS.dpTd(f)/EoS.dpdT(f);
        elseif (state.phase == 2) then
          ddTp := Modelica.Constants.inf; // divide by zero
        end if;
      end density_derT_p;

      redeclare function specificEnthalpy_pT
      "returns specific enthalpy for given p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output SpecificEnthalpy h "specific enthalpy";

      algorithm
        h := specificEnthalpy_pT_state(p=p, T=T, state=setState_pT(p=p, T=T, phase=phase));

      annotation (
        Inline=true,
        inverse(T=temperature_ph(p=p, h=h, phase=phase)));
      end specificEnthalpy_pT;

      function specificEnthalpy_pT_state
      "returns specific enthalpy for given p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        output SpecificEnthalpy h "specific enthalpy";

      algorithm
        h := specificEnthalpy(state);

      annotation (
        Inline=false,
        LateInline=true,
        inverse(T=temperature_ph_state(p=p, h=h, state=state)),
        derivative(noDerivative=state)=specificEnthalpy_pT_der);
      end specificEnthalpy_pT_state;

      function specificEnthalpy_pT_der "time derivative of specificEnthalpy_pT"
        extends Modelica.Icons.Function;
        input AbsolutePressure p;
        input Temperature T;
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        input Real p_der "time derivative of pressure";
        input Real T_der "time derivative of temperature";
        output Real h_der "time derivative of specific Enthalpy";

      algorithm
        h_der := p_der*isothermalThrottlingCoefficient(state=state)
               + T_der*specificHeatCapacityCp(state=state);

      annotation (Inline=true);
      end specificEnthalpy_pT_der;

      redeclare function pressure_dT
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output AbsolutePressure p "pressure";

      algorithm
        p := pressure_dT_state(d=d, T=T, state=setState_dT(d=d, T=T, phase=phase));

      annotation (
        Inline=true,
        inverse(d=density_pT(p=p, T=T, phase=phase),
                T=temperature_pd(p=p, d=d, phase=phase)));
      end pressure_dT;

      function pressure_dT_state
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        output AbsolutePressure p "pressure";

      algorithm
        p := pressure(state);

      annotation (
        Inline=false,
        LateInline=true,
        inverse(d=density_pT_state(p=p, T=T, state=state)),
        derivative(noDerivative=state)=pressure_dT_der);
      end pressure_dT_state;

      function pressure_dT_der "time derivative of pressure_dT"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        input Real der_d "Density derivative";
        input Real der_T "Temperature derivative";
        output Real der_p "Time derivative of pressure";

      algorithm
        der_p := der_d*pressure_derd_T(state=state)
               + der_T*pressure_derT_d(state=state);
      end pressure_dT_der;

      function pressure_derd_T "returns pressure derivative (dp/dd)@T=const"
        input ThermodynamicState state;
        output DerPressureByDensity dpdT;

    protected
        EoS.HelmholtzDerivs f;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=state.phase);
          dpdT := EoS.dpdT(f);
        elseif (state.phase == 2) then
          dpdT := Modelica.Constants.small; // zero
        end if;
      end pressure_derd_T;

      function pressure_derT_d "returns pressure derivative (dp/dT)@d=const"
        input ThermodynamicState state;
        output DerPressureByTemperature dpTd;

    protected
        EoS.HelmholtzDerivs f;
        SaturationProperties sat;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=state.phase);
          dpTd := EoS.dpTd(f);
        elseif (state.phase == 2) then
          sat := setSat_T(T=state.T);
          dpTd := (sat.vap.s-sat.liq.s)/(1.0/sat.vap.d-1.0/sat.liq.d);
        end if;
      end pressure_derT_d;

      redeclare function specificEnthalpy_dT
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output SpecificEnthalpy h "Specific Enthalpy";

      algorithm
        h := specificEnthalpy_dT_state(d=d, T=T, state=setState_dT(d=d, T=T, phase=phase));

      annotation (
        Inline=true);
      end specificEnthalpy_dT;

      function specificEnthalpy_dT_state
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        output SpecificEnthalpy h "SpecificEnthalpy";

      algorithm
        h := specificEnthalpy(state);

      annotation (
        Inline=false,
        LateInline=true,
        derivative(noDerivative=state)=specificEnthalpy_dT_der);
      end specificEnthalpy_dT_state;

      function specificEnthalpy_dT_der
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
      //input FixedPhase phase=0 "2 for two-phase, 1 for one-phase, 0 if not known";
        input ThermodynamicState state;
        input Real der_d "Density derivative";
        input Real der_T "Temperature derivative";
        output Real der_h "Time derivative of enthalpy";

      algorithm
        der_h := der_d*specificEnthalpy_derd_T(state=state)
               + der_T*specificEnthalpy_derT_d(state=state);

      end specificEnthalpy_dT_der;

      function specificEnthalpy_derd_T
      "returns enthalpy derivative (dh/dd)@T=const"
        input ThermodynamicState state;
        output DerEnthalpyByDensity dhdT;

    protected
        EoS.HelmholtzDerivs f;
        SaturationProperties sat;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=state.phase);
          dhdT := EoS.dhdT(f);
        elseif (state.phase == 2) then
          // dhvT = (h"-h')/(v"-v')
          // dhdT = -1/d^2 * dhvT
          sat:=setSat_T(T=state.T);
          dhdT := -1/state.d^2*(sat.liq.h-sat.vap.h)/(1/sat.liq.d-1/sat.vap.d);
        end if;
      end specificEnthalpy_derd_T;

      function specificEnthalpy_derT_d
      "returns enthalpy derivative (dh/dT)@d=const"
        input ThermodynamicState state;
        output DerEnthalpyByTemperature dhTd;

    protected
        EoS.HelmholtzDerivs f;

        SaturationProperties sat;
        MassFraction x "vapour quality";
        DerPressureByTemperature dpT;
        EoS.HelmholtzDerivs fl;
        EoS.HelmholtzDerivs fv;

        DerEnthalpyByTemperature dhT_liq;
        DerEnthalpyByTemperature dhT_vap;
        DerDensityByTemperature ddT_liq;
        DerDensityByTemperature ddT_vap;
        DerVolumeByTemperature dvT_liq;
        DerVolumeByTemperature dvT_vap;
        DerFractionByTemperature dxTv;

      algorithm
        if (state.phase == 1) then
          f := EoS.setHelmholtzDerivsSecond(T=state.T, d=state.d, phase=state.phase);
          dhTd := EoS.dhTd(f);

        elseif (state.phase == 2) then
          sat:=setSat_T(T=state.T);
          x := (1/state.d - 1/sat.liq.d)/(1/sat.vap.d - 1/sat.liq.d);
          dpT := (sat.vap.s-sat.liq.s)/(1.0/sat.vap.d-1.0/sat.liq.d);

          fl := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.liq.d, phase=1);
          fv := EoS.setHelmholtzDerivsSecond(T=state.T, d=sat.vap.d, phase=1);

          dhT_liq := EoS.dhTd(fl)-EoS.dhdT(fl)*EoS.dpTd(fl)/EoS.dpdT(fl) + EoS.dhdT(fl)/EoS.dpdT(fl)*dpT;
          dhT_vap := EoS.dhTd(fv)-EoS.dhdT(fv)*EoS.dpTd(fv)/EoS.dpdT(fv) + EoS.dhdT(fv)/EoS.dpdT(fv)*dpT;
          ddT_liq := -EoS.dpTd(fl)/EoS.dpdT(fl) + 1.0/EoS.dpdT(fl)*dpT;
          ddT_vap := -EoS.dpTd(fv)/EoS.dpdT(fv) + 1.0/EoS.dpdT(fv)*dpT;
          dvT_liq := -1/sat.liq.d^2 * ddT_liq;
          dvT_vap := -1/sat.vap.d^2 * ddT_vap;
          dxTv :=(x*dvT_vap + (1 - x)*dvT_liq)/(1/sat.liq.d - 1/sat.vap.d);

          dhTd := dhT_liq + dxTv*(sat.vap.h-sat.liq.h) + x*(dhT_vap-dhT_liq);
        end if;

      end specificEnthalpy_derT_d;

      function specificEntropy_pT
      "iteratively finds the specific entropy for a given p and T"
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEntropy s "Specific Entropy";

      algorithm
        s := specificEntropy(setState_pTX(p=p, T=T, phase=phase));

      annotation (
        inverse(T=temperature_ps(p=p, s=s, phase=phase),
                p=pressure_Ts(T=T, s=s, phase=phase)));
      end specificEntropy_pT;

      function specificEntropy_dT "return specific enthalpy for given d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output SpecificEntropy s "specific entropy";

      algorithm
        s := specificEntropy(setState_dTX(d=d,T=T,phase=phase));

      annotation (
        inverse(d=density_Ts(T=T, s=s, phase=phase)));
      end specificEntropy_dT;

      function temperature_pd "returns temperature for given p and d"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Density d "Density";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output Temperature T "Temperature";

      algorithm
        T := temperature(setState_pd(p=p, d=d, phase=phase));

      annotation (
        inverse(p=pressure_dT(d=d, T=T, phase=phase),
                d=density_pT(p=p, T=T, phase=phase)));
      end temperature_pd;

      function pressure_Ts "returns pressure for given T and s"
        extends Modelica.Icons.Function;
        input Temperature T "Temperature";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output AbsolutePressure p "Pressure";

      algorithm
        p := pressure(setState_Ts(T=T, s=s, phase=phase));

      annotation (
        inverse(s=specificEntropy_pT(p=p, T=T, phase=phase),
                T=temperature_ps(p=p, s=s, phase=phase)));
      end pressure_Ts;

      redeclare function temperature_ps "returns temperature for given p and d"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output Temperature T "Temperature";

      algorithm
        T := temperature(setState_ps(p=p, s=s, phase=phase));

      annotation (
        inverse(p=pressure_Ts(T=T, s=s, phase=phase),
                s=specificEntropy_pT(p=p, T=T, phase=phase)));
      end temperature_ps;

      function density_Ts "returns density for given T and s"
        extends Modelica.Icons.Function;
        input Temperature T "Temperature";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output Density d "Density";

      algorithm
        d := density(setState_Ts(T=T, s=s, phase=phase));

      annotation (
        inverse=specificEntropy_dT(d=d, T=T, phase=phase));
      end density_Ts;

      redeclare function specificEnthalpy_ps
      "returns specific enthalpy for a given p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output SpecificEnthalpy h "specific enthalpy";

      algorithm
        h := specificEnthalpy(setState_psX(p=p, s=s, phase=phase));

      annotation (
        inverse(s=specificEntropy_ph(p=p, h=h, phase=phase)));
      end specificEnthalpy_ps;

      function specificEnthalpy_pd
      "returns specific enthalpy for a given p and d"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Density d "Density";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      //input ThermodynamicState state;
        output SpecificEnthalpy h "specific enthalpy";

      algorithm
        h := specificEnthalpy(setState_pd(p=p, d=d, phase=phase));

      annotation (
        inverse(d=density_ph(p=p, h=h, phase=phase)));
      end specificEnthalpy_pd;

      package Types

        type ReferenceState = enumeration(
          IIR "IIR",
          ASHRAE "ASHRAE",
          NBP "NBP");
        type DynamicViscosityModel = enumeration(
          VS0 "VS0 model (water)",
          VS1 "VS1 model",
          VS1_alternative "VS1 with alternative first term",
          VS2 "VS2 model",
          VS4 "VS4 model (Quinones-Cisneros and Deiters)");
        type ThermalConductivityModel = enumeration(
          TC0 "TC0 model (water)",
          TC1 "TC1 model",
          TC2 "TC2 model");
        type CollisionIntegralModel = enumeration(
          CI0 "CI0 model",
          CI1 "CI1 model",
          CI2 "CI2 model");
        type PressureSaturationModel = enumeration(
          PS1,
          PS2,
          PS3,
          PS4,
          PS5 "most common",
          PS6);
        type DensityLiquidModel = enumeration(
          DL1 "most common",
          DL2,
          DL3,
          DL4,
          DL5,
          DL6);
        type DensityVaporModel = enumeration(
          DV1,
          DV2,
          DV3 "most common",
          DV4,
          DV5,
          DV6);
        type PressureMeltingModel = enumeration(
          ML1 "most common",
          ML2);
        type InputChoice = enumeration(
          dT "(d,T) as inputs",
          ph "(p,h) as inputs",
          ps "(p,s) as inputs",
          pT "(p,T) as inputs");
        type DerPressureByDensity = Real (final unit="Pa/(kg/m3)", start=1);
        type Der2PressureByDensity2 = Real (final unit="Pa/(kg2/m6)");
        type DerPressureByTemperature = Real (final unit="Pa/K");
        type Der2PressureByTemperature2 = Real (final unit="Pa/(K.K)");
        type Der2PressureByTemperatureDensity = Real (final unit="(Pa.m3)/(K.kg)");
        type DerEnthalpyByDensity = Real (final unit="(J/kg)/(kg/m3)");
        type Der2EnthalpyByDensity2 = Real (final unit="(J/kg)/(kg2/m6)");
        type DerEnthalpyByTemperature = Real (final unit="(J/kg)/K");
        type Der2EnthalpyByTemperature2 = Real (final unit="(J/kg)/(K.K)");
        type Der2EnthalpyByTemperatureDensity = Real (final unit="(J/kg)/(K.kg/m3)");
        type Der2EnthalpyByTemperaturePressure = Real (final unit="(J/kg)/(K.Pa)");
        type DerEnthalpyByPressure = Real (final unit="(J/kg)/Pa");
        type DerEnergyByDensity = Real (final unit="(J/kg)/(kg/m3)");
        type Der2EnergyByDensity2 = Real (final unit="(J/kg)/(kg2/m6)");
        type DerEnergyByTemperature = Real (final unit="(J/kg)/K");
        type Der2EnergyByTemperature2 = Real (final unit="(J/kg)/(K.K)");
        type Der2EnergyByTemperatureDensity = Real (final unit="(J/kg)/(K.kg/m3)");
        type DerEnergyByPressure = Real (final unit="(J/kg)/Pa");
        type DerEntropyByDensity = Real (final unit="(J/(kg.K))/(kg/m3)");
        type Der2EntropyByDensity2 = Real (final unit="(J/(kg.K))/(kg2/m6)");
        type DerEntropyByTemperature = Real (final unit="(J/(kg.K))/K");
        type Der2EntropyByTemperature2 = Real (final unit="(J/(kg.K))/K2");
        type Der2EntropyByTemperatureDensity = Real (final unit="(J/(kg.K))/(K.kg/m3)");
        type DerEntropyByPressure = Real (final unit="(J/(kg.K))/Pa");
        type Der2DensityByPressure2 = Real (final unit="(kg/m3)/(Pa2)");
        type Der2DensityByTemperature2 = Real (final unit="(kg/m3)/(K2)");
        type Der2DensityByTemperaturePressure = Real (final unit="(kg/m3)/(K.Pa)");
        type DerVolumeByTemperature = Real (final unit="(m3/kg)/K");
        type DerVolumeByPressure = Real (final unit="(m3/kg)/Pa");
        type DerPressureByVolume = Real (final unit="Pa/(m3/kg)");
        type Der2PressureByVolume2 = Real (final unit="Pa/(m6/kg2)");
        type Der2PressureByTemperatureVolume = Real (final unit="Pa/(K.m3/kg)");
        type DerTemperatureByEnthalpy = Real (final unit="K/(J/kg)");
        type DerTemperatureByDensity = Real (final unit="K/(kg/m3)");
        type Der2TemperatureByDensity2 = Real (final unit="K/(kg2/m6)");
        type DerTemperatureByPressure = Real (final unit="K/Pa");
        type Der2TemperatureByPressure2 = Real (final unit="K/Pa2");
        type Der2TemperatureByPressureDensity = Real (final unit="K.m3/(Pa.kg)");
        type DerFractionByTemperature = Real (final unit="1/K");
        type DerFractionByPressure = Real (final unit="1/Pa");
      end Types;
    end PartialHelmholtzMedium;

    package Choices
      extends Modelica.Media.Interfaces.Choices;
      type ReferenceState = enumeration(
        IIR "IIR",
        ASHRAE "ASHRAE",
        NBP "NBP");
      type DynamicViscosityModel = enumeration(
        VS0 "VS0 model (water)",
        VS1 "VS1 model",
        VS1_alternative "VS1 with alternative first term",
        VS2 "VS2 model",
        VS4 "VS4 model (Quinones-Cisneros and Deiters)");
      type ThermalConductivityModel = enumeration(
        TC0 "TC0 model (water)",
        TC1 "TC1 model",
        TC2 "TC2 model");
      type CollisionIntegralModel = enumeration(
        CI0 "CI0 model",
        CI1 "CI1 model",
        CI2 "CI2 model");
      type PressureSaturationModel = enumeration(
        PS1,
        PS2,
        PS3,
        PS4,
        PS5 "most common",
        PS6);
      type DensityLiquidModel = enumeration(
        DL1 "most common",
        DL2,
        DL3,
        DL4,
        DL5,
        DL6);
      type DensityVaporModel = enumeration(
        DV1,
        DV2,
        DV3 "most common",
        DV4,
        DV5,
        DV6);
      type PressureMeltingModel = enumeration(
        ML1 "most common",
        ML2);
      type InputChoice = enumeration(
        dT "(d,T) as inputs",
        ph "(p,h) as inputs",
        ps "(p,s) as inputs",
        pT "(p,T) as inputs");
    end Choices;

    package Types
      extends Modelica.Media.Interfaces.Types;
      //type DerEnthalpyByPressure = Real (final unit="(J/kg)/Pa");
      //type DerTemperatureByPressure = Real (final unit="K/Pa");
      type DerPressureByDensity = Real (final unit="Pa/(kg/m3)", start=1);
      type Der2PressureByDensity2 = Real (final unit="Pa/(kg2/m6)");
      type DerPressureByTemperature = Real (final unit="Pa/K");
      type Der2PressureByTemperature2 = Real (final unit="Pa/(K.K)");
      type Der2PressureByTemperatureDensity = Real (final unit="(Pa.m3)/(K.kg)");
      type DerEnthalpyByDensity = Real (final unit="(J/kg)/(kg/m3)");
      type Der2EnthalpyByDensity2 = Real (final unit="(J/kg)/(kg2/m6)");
      type DerEnthalpyByTemperature = Real (final unit="(J/kg)/K");
      type Der2EnthalpyByTemperature2 = Real (final unit="(J/kg)/(K.K)");
      type Der2EnthalpyByTemperatureDensity = Real (final unit="(J/kg)/(K.kg/m3)");
      type Der2EnthalpyByTemperaturePressure = Real (final unit="(J/kg)/(K.Pa)");
      type DerEnergyByDensity = Real (final unit="(J/kg)/(kg/m3)");
      type Der2EnergyByDensity2 = Real (final unit="(J/kg)/(kg2/m6)");
      type DerEnergyByTemperature = Real (final unit="(J/kg)/K");
      type Der2EnergyByTemperature2 = Real (final unit="(J/kg)/(K.K)");
      type Der2EnergyByTemperatureDensity = Real (final unit="(J/kg)/(K.kg/m3)");
      type DerEnergyByPressure = Real (final unit="(J/kg)/Pa");
      type DerEntropyByDensity = Real (final unit="(J/(kg.K))/(kg/m3)");
      type Der2EntropyByDensity2 = Real (final unit="(J/(kg.K))/(kg2/m6)");
      type DerEntropyByTemperature = Real (final unit="(J/(kg.K))/K");
      type Der2EntropyByTemperature2 = Real (final unit="(J/(kg.K))/K2");
      type Der2EntropyByTemperatureDensity = Real (final unit="(J/(kg.K))/(K.kg/m3)");
      type DerEntropyByPressure = Real (final unit="(J/(kg.K))/Pa");
      type Der2DensityByPressure2 = Real (final unit="(kg/m3)/(Pa2)");
      type Der2DensityByTemperature2 = Real (final unit="(kg/m3)/(K2)");
      type Der2DensityByTemperaturePressure = Real (final unit="(kg/m3)/(K.Pa)");
      type DerVolumeByTemperature = Real (final unit="(m3/kg)/K");
      type DerVolumeByPressure = Real (final unit="(m3/kg)/Pa");
      type DerPressureByVolume = Real (final unit="Pa/(m3/kg)");
      type Der2PressureByVolume2 = Real (final unit="Pa/(m6/kg2)");
      type Der2PressureByTemperatureVolume = Real (final unit="Pa/(K.m3/kg)");
      type DerTemperatureByEnthalpy = Real (final unit="K/(J/kg)");
      type DerTemperatureByDensity = Real (final unit="K/(kg/m3)");
      type Der2TemperatureByDensity2 = Real (final unit="K/(kg2/m6)");
      type Der2TemperatureByPressure2 = Real (final unit="K/Pa2");
      type Der2TemperatureByPressureDensity = Real (final unit="K.m3/(Pa.kg)");
      type DerFractionByTemperature = Real (final unit="1/K");
      type DerFractionByPressure = Real (final unit="1/Pa");
    end Types;
  end Interfaces;

  package HelmholtzFluids

    package Helium "Helium"
    extends Interfaces.PartialHelmholtzMedium(
      mediumName="helium" "short name",
      fluidConstants={fluidConstantsHelium},
      helmholtzCoefficients=helmholtzCoefficientsHelium,
      thermalConductivityCoefficients=thermalConductivityCoefficientsHelium,
      dynamicViscosityCoefficients=dynamicViscosityCoefficientsHelium,
      surfaceTensionCoefficients=surfaceTensionCoefficientsHelium,
      ancillaryCoefficients=ancillaryCoefficientsHelium,
      fluidLimits=fluidLimitsHelium,
      Density(min=fluidLimitsHelium.DMIN, max=fluidLimitsHelium.DMAX, start=fluidConstantsHelium.molarMass/fluidConstantsHelium.criticalMolarVolume),
      Temperature(min=fluidLimitsHelium.TMIN, max=fluidLimitsHelium.TMAX, start=298.15),
      AbsolutePressure(min=0, max=1000e6, start=101325),
      SpecificEnthalpy(min=fluidLimitsHelium.HMIN, max=fluidLimitsHelium.HMAX, start=0),
      SpecificEntropy(min=fluidLimitsHelium.SMIN, max=fluidLimitsHelium.SMAX, start=0));

      final constant FluidConstants
      fluidConstantsHelium(
        casRegistryNumber="7440-59-7" "CAS number",
        iupacName="helium-4" "full name",
        structureFormula="He",
        chemicalFormula="He",
        molarMass=0.004002602,
        triplePointTemperature=2.1768,
        normalBoilingPoint=4.2226,
        hasCriticalData=true,
           criticalTemperature=5.1953,
           criticalPressure=227610,
           criticalMolarVolume=1.0/17.3837e3,
           HCRIT0=11229.8913760996,
           SCRIT0=2054.84033693702,
        hasAcentricFactor=true,
           acentricFactor=-0.385,
        hasIdealGasHeatCapacity=false,
        hasDipoleMoment=true,
           dipoleMoment=0.0,
        hasFundamentalEquation=true,
        hasLiquidHeatCapacity=true,
        hasSolidHeatCapacity=false,
        hasAccurateViscosityData=true,
        hasAccurateConductivityData=true,
        hasVapourPressureCurve=true,
        triplePointPressure=5033.5,
        meltingPoint=2.1768) "Fluid Constants";

      final constant FluidLimits
      fluidLimitsHelium(
        TMIN=fluidConstantsHelium.triplePointTemperature,
        TMAX=2000,
        DMIN=Modelica.Constants.small,
        DMAX=565.247,
        PMIN=Modelica.Constants.small,
        PMAX=1000e6,
        HMIN=-6.6877e3,
        HMAX=+12000e3,
        SMIN=-1.8782e3,
        SMAX=120e3) "Helmholtz EoS Limits";

      final constant EoS.HelmholtzCoefficients
      helmholtzCoefficientsHelium(
        useLineSearch=true,
        idealLog=[
              1.5,          1],
        idealPower=[
            0.159269591430361000,  0;
            0.476532113807410000,  1],
        idealEinstein=fill(0.0, 0, 2),
        idealCosh=fill(0.0, 0, 2),
        idealSinh=fill(0.0, 0, 2),
        residualPoly=[
              0.014799269,   1.0,     4,  0;
              3.06281562,    0.426,   1,  0;
             -4.25338698,    0.631,   1,  0;
              0.05192797,    0.596,   2,  0;
             -0.165087335,   1.705,   2,  0;
              0.087236897,   0.568,   3,  0],
        residualBwr=[
              2.10653786,    0.9524,  1,  1;
             -0.62835030,    1.471,   1,  2;
             -0.28200301,    1.48,    3,  2;
              1.04234019,    1.393,   2,  1;
             -0.07620555,    3.863,   2,  2;
             -1.35006365,    0.803,   1,  1],
         residualGauss=[
              0.11997252,    3.273,   1,  2, 2,   -8.674,   -8.005,  1.1475,  0.912;
              0.10724500,    0.66,    1,  2, 2,   -4.006,   -1.15,   1.7036,  0.79;
             -0.35374839,    2.629,   1,  2, 2,   -8.1099,  -2.143,  1.6795,  0.90567;
              0.75348862,    1.4379,  2,  2, 2,   -0.1449,  -0.147,  0.9512,  5.1136;
              0.00701871,    3.317,   2,  2, 2,   -0.1784,  -0.154,  4.475,   3.6022;
              0.226283167,   2.3676,  2,  2, 2,   -2.432,   -0.701,  2.7284,  0.6488;
             -0.22464733,    0.7545,  3,  2, 2,   -0.0414,  -0.21,   1.7167,  4.2753;
              0.12413584,    1.353,   2,  2, 2,   -0.421,   -0.134,  1.5237,  2.744;
              0.00901399,    1.982,   2,  2, 2,   -5.8575, -19.256,  0.7649,  0.8736])
      "Coefficients of the Helmholtz EoS";

      final constant Transport.ThermalConductivityCoefficients
      thermalConductivityCoefficientsHelium(
        reducingTemperature_0=10,
        reducingThermalConductivity_0=1,
        lambda_0_num_coeffs=fill(0.0, 0, 2),
        reducingTemperature_residual=1,
        reducingMolarVolume_residual=1,
        reducingThermalConductivity_residual=1,
        lambda_r_coeffs=fill(0.0, 0, 4),
        xi_0=0.194E-9,
        Gamma_0=0.0496,
        qd_inverse=0.875350E-9,
        T_ref=637.68) "Coefficients for the thermal conductivity";

      final constant Transport.DynamicViscosityCoefficients
      dynamicViscosityCoefficientsHelium(
        dynamicViscosityModel=DynamicViscosityModel.VS1,
        collisionIntegralModel=CollisionIntegralModel.CI1,
        sigma=1,
        epsilon_kappa=1,
        CET=fill(0.0, 0, 2),
        a=fill(0.0, 0, 2),
        b=fill(0.0, 0, 2),
        reducingTemperature_residual=1,
        reducingMolarVolume_residual=1,
        reducingViscosity_residual=1,
        g=fill(0.0, 0, 2),
        e=fill(0.0, 0, 5),
        nu_po=fill(0.0, 0, 5),
        de_po=fill(0.0, 0, 5)) "Coefficients for the dynamic viscosity";

      final constant Transport.SurfaceTensionCoefficients
      surfaceTensionCoefficientsHelium(
        coeffs=[
           0.0004656,   1.04;
           0.001889,    2.468;
          -0.002006,    2.661]) "Coefficients for the surface tension";

      final constant Ancillary.AncillaryCoefficients
      ancillaryCoefficientsHelium(
        pressureMeltingModel=PressureMeltingModel.ML1,
        T_reducing=10,
        p_reducing=1000e3,
        pressureMelting1=[
          -1.7455837,      0.000000;
           1.6979793,      1.555414],
        pressureMelting2=fill(0.0, 0, 2),
        pressureMelting3=fill(0.0, 0, 2),
        pressureSaturationModel=PressureSaturationModel.PS5,
        pressureSaturation=[
          -3.8357,   1.0;
           1.7062,   1.5;
          -0.71231,  1.25;
           1.0862,   2.8],
        densityLiquidModel=DensityLiquidModel.DL1,
        densityLiquid=[
           1.0929,   0.286;
           1.6584,   1.2;
          -3.6477,   2.0;
           2.7440,   2.8;
          -2.3859,   6.5],
        densityVaporModel=DensityVaporModel.DV3,
        densityVapor=[
          -1.5789,   0.333;
          -10.749,   1.5;
           17.711,   2.1;
          -15.413,   2.7;
          -14.352,   9.0])
      "Coefficients for the ancillary equations (PS5, DL1, DV3, ML1)";

      annotation (Documentation(info="<html>
These are the coefficients for Helium. 

<dl>
<dt> Ortiz-Vega, D.O., Hall, K.R., Arp, V.D., and Lemmon, E.W.</dt>
<dd> <b>Interim equation (final version) for the properties of helium</b><br>
     to be published in J. Phys. Chem. Ref. Data, 2013<br>
     DOI: <a href=\"http://dx.doi.org/\"></a>
</dd>
<dt> McCarty, R.D. and Arp, V.D.</dt>
<dd> <b>A new wide rand equation of state for helium</b><br>
     Adv. Cryo. Eng. 35:1465-1475 (1990)<br>
     DOI: <a href=\"http://dx.doi.org/\"></a>
</dd>
<dt> Lemmon, Eric W.; Huber, M. L. and McLinden, M. O.</dt>
<dd> <b>NIST Standard Reference Database 23: Reference Fluid Thermodynamic and Transport Properties - REFPROP. 9.0</b><br>
     National Institute of Standards and Technology, Standard Reference Data Program. Gaithersburg<br>
     URL: <a href=\"http://www.nist.gov/srd/nist23.cfm\">http://www.nist.gov/srd/nist23.cfm</a>
</dd>
</dl>
</html>"));

    end Helium;
  end HelmholtzFluids;
  annotation (uses(Modelica(version="3.2.1")));
end HelmholtzMedia;

package ExternalMedia
  extends Modelica.Icons.Package;
  import SI = Modelica.SIunits;

  package Common "Package with common definitions"
    extends Modelica.Icons.Package;

    type InputChoice = enumeration(
      dT "(d,T) as inputs",
      hs "(h,s) as inputs",
      ph "(p,h) as inputs",
      ps "(p,s) as inputs",
      pT "(p,T) as inputs");
  end Common;

  package Media "Medium packages compatible with Modelica.Media"
    extends Modelica.Icons.Package;

    package FluidPropMedium "Medium package accessing the FluidProp solver"
      extends BaseClasses.ExternalTwoPhaseMedium;
      redeclare replaceable function setBubbleState
      "Set the thermodynamic state on the bubble line"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "saturation point";
        input FixedPhase phase = 0 "phase flag";
        output ThermodynamicState state "complete thermodynamic state info";
        // Standard definition
        external "C" TwoPhaseMedium_setBubbleState_C_impl(sat, phase, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        annotation(Inline = true);
      end setBubbleState;

      redeclare replaceable function setDewState
      "Set the thermodynamic state on the dew line"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "saturation point";
        input FixedPhase phase = 0 "phase flag";
        output ThermodynamicState state "complete thermodynamic state info";
        // Standard definition
        external "C" TwoPhaseMedium_setDewState_C_impl(sat, phase, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        annotation(Inline = true);
      end setDewState;

      redeclare function bubbleEntropy "Return bubble point specific entropy"
        input SaturationProperties sat "saturation property record";
        output SI.SpecificEntropy sl "boiling curve specific entropy";
      algorithm
        sl := specificEntropy(setBubbleState(sat));
      end bubbleEntropy;

      redeclare function dewEntropy "Return dew point specific entropy"
        input SaturationProperties sat "saturation property record";
        output SI.SpecificEntropy sv "dew curve specific entropy";
      algorithm
        sv := specificEntropy(setDewState(sat));
      end dewEntropy;

      redeclare function surfaceTension
        extends Modelica.Icons.Function;
        input SaturationProperties sat "saturation property record";
        output SurfaceTension sigma
        "Surface tension sigma in the two phase region";
      algorithm
        assert(false, "The FluidProp solver does not provide surface tension");
      end surfaceTension;
    end FluidPropMedium;

    package BaseClasses "Base classes for external media packages"
      extends Modelica.Icons.BasesPackage;

      package ExternalTwoPhaseMedium
      "Generic external two phase medium package"
        extends Modelica.Media.Interfaces.PartialTwoPhaseMedium(
          singleState = false,
          onePhase = false,
          smoothModel = false,
          fluidConstants = {externalFluidConstants});
        import ExternalMedia.Common.InputChoice;
        // mediumName is declared here instead of in the extends clause
        // to break a circular dependency in redeclaration that OpenModelica
        // cannot yet handle
        constant String mediumName="unusablePartialMedium" "Name of the medium";
        constant String libraryName = "UnusableExternalMedium"
        "Name of the external fluid property computation library";
        constant String substanceName = substanceNames[1]
        "Only one substance can be specified";
        constant FluidConstants externalFluidConstants = FluidConstants(
          iupacName=  "unknown",
          casRegistryNumber=  "unknown",
          chemicalFormula=  "unknown",
          structureFormula=  "unknown",
          molarMass=  getMolarMass(),
          criticalTemperature=  getCriticalTemperature(),
          criticalPressure=  getCriticalPressure(),
          criticalMolarVolume=  getCriticalMolarVolume(),
          acentricFactor=  0,
          triplePointTemperature=  280.0,
          triplePointPressure=  500.0,
          meltingPoint=  280,
          normalBoilingPoint=  380.0,
          dipoleMoment=  2.0);

        constant InputChoice inputChoice=InputChoice.ph
        "Default choice of input variables for property computations";
        redeclare replaceable record ThermodynamicState
          // Fields in ASCII lexicographical order to work in Dymola
          Temperature T "temperature";
          VelocityOfSound a "velocity of sound";
          Modelica.SIunits.CubicExpansionCoefficient beta
          "isobaric expansion coefficient";
          SpecificHeatCapacity cp "specific heat capacity cp";
          SpecificHeatCapacity cv "specific heat capacity cv";
          Density d "density";
          DerDensityByEnthalpy ddhp
          "derivative of density wrt enthalpy at constant pressure";
          DerDensityByPressure ddph
          "derivative of density wrt pressure at constant enthalpy";
          DynamicViscosity eta "dynamic viscosity";
          SpecificEnthalpy h "specific enthalpy";
          Modelica.SIunits.Compressibility kappa "compressibility";
          ThermalConductivity lambda "thermal conductivity";
          AbsolutePressure p "pressure";
          FixedPhase phase(min=0, max=2)
          "phase flag: 2 for two-phase, 1 for one-phase";
          SpecificEntropy s "specific entropy";
        end ThermodynamicState;

        redeclare record SaturationProperties
          // Fields in ASCII lexicographical order to work in Dymola
          Temperature Tsat "saturation temperature";
          Real dTp "derivative of Ts wrt pressure";
          DerDensityByPressure ddldp "derivative of dls wrt pressure";
          DerDensityByPressure ddvdp "derivative of dvs wrt pressure";
          DerEnthalpyByPressure dhldp "derivative of hls wrt pressure";
          DerEnthalpyByPressure dhvdp "derivative of hvs wrt pressure";
          Density dl "density at bubble line (for pressure ps)";
          Density dv "density at dew line (for pressure ps)";
          SpecificEnthalpy hl
          "specific enthalpy at bubble line (for pressure ps)";
          SpecificEnthalpy hv "specific enthalpy at dew line (for pressure ps)";
          AbsolutePressure psat "saturation pressure";
          SurfaceTension sigma "surface tension";
          SpecificEntropy sl
          "specific entropy at bubble line (for pressure ps)";
          SpecificEntropy sv "specific entropy at dew line (for pressure ps)";
        end SaturationProperties;

        redeclare replaceable model extends BaseProperties(
          p(stateSelect = if preferredMediumStates and
                             (basePropertiesInputChoice == InputChoice.ph or
                              basePropertiesInputChoice == InputChoice.pT or
                              basePropertiesInputChoice == InputChoice.ps) then
                                  StateSelect.prefer else StateSelect.default),
          T(stateSelect = if preferredMediumStates and
                             (basePropertiesInputChoice == InputChoice.pT or
                              basePropertiesInputChoice == InputChoice.dT) then
                               StateSelect.prefer else StateSelect.default),
          h(stateSelect = if preferredMediumStates and
                             (basePropertiesInputChoice == InputChoice.hs or
                              basePropertiesInputChoice == InputChoice.ph) then
                               StateSelect.prefer else StateSelect.default),
          d(stateSelect = if preferredMediumStates and
                             basePropertiesInputChoice == InputChoice.dT then
                               StateSelect.prefer else StateSelect.default))
          import ExternalMedia.Common.InputChoice;
          parameter InputChoice basePropertiesInputChoice=inputChoice
          "Choice of input variables for property computations";
          FixedPhase phaseInput
          "Phase input for property computation functions, 2 for two-phase, 1 for one-phase, 0 if not known";
          Integer phaseOutput
          "Phase output for medium, 2 for two-phase, 1 for one-phase";
          SpecificEntropy s(
            stateSelect = if (basePropertiesInputChoice == InputChoice.hs or
                              basePropertiesInputChoice == InputChoice.ps) then
                             StateSelect.prefer else StateSelect.default)
          "Specific entropy";
          SaturationProperties sat "saturation property record";
        equation
          MM = externalFluidConstants.molarMass;
          R = Modelica.Constants.R/MM;
          if (onePhase or (basePropertiesInputChoice == InputChoice.pT)) then
            phaseInput = 1 "Force one-phase property computation";
          else
            phaseInput = 0 "Unknown phase";
          end if;
          if (basePropertiesInputChoice == InputChoice.ph) then
            // Compute the state record (including the unique ID)
            state = setState_ph(p, h, phaseInput);
            // Compute the remaining variables.
            // It is not possible to use the standard functions like
            // d = density(state), because differentiation for index
            // reduction and change of state variables would not be supported
            // density_ph(), which has an appropriate derivative annotation,
            // is used instead. The implementation of density_ph() uses
            // setState with the same inputs, so there's no actual overhead
            d = density_ph(p, h, phaseInput);
            s = specificEntropy_ph(p, h, phaseInput);
            T = temperature_ph(p, h, phaseInput);
          elseif (basePropertiesInputChoice == InputChoice.dT) then
            state = setState_dT(d, T, phaseInput);
            h = specificEnthalpy(state);
            p = pressure(state);
            s = specificEntropy(state);
          elseif (basePropertiesInputChoice == InputChoice.pT) then
            state = setState_pT(p, T, phaseInput);
            d = density(state);
            h = specificEnthalpy(state);
            s = specificEntropy(state);
          elseif (basePropertiesInputChoice == InputChoice.ps) then
            state = setState_ps(p, s, phaseInput);
            d = density(state);
            h = specificEnthalpy(state);
            T = temperature(state);
          elseif (basePropertiesInputChoice == InputChoice.hs) then
            state = setState_hs(h, s, phaseInput);
            d = density(state);
            p = pressure(state);
            T = temperature(state);
          end if;
          // Compute the internal energy
          u = h - p/d;
          // Compute the saturation properties record only if below critical point
          //sat = setSat_p(min(p,fluidConstants[1].criticalPressure));
          sat = setSat_p_state(state);
          // Event generation for phase boundary crossing
          if smoothModel then
            // No event generation
            phaseOutput = state.phase;
          else
            // Event generation at phase boundary crossing
            if basePropertiesInputChoice == InputChoice.ph then
              phaseOutput = if ((h > bubbleEnthalpy(sat) and h < dewEnthalpy(sat)) and
                                 p < fluidConstants[1].criticalPressure) then 2 else 1;
            elseif basePropertiesInputChoice == InputChoice.dT then
              phaseOutput = if  ((d < bubbleDensity(sat) and d > dewDensity(sat)) and
                                  T < fluidConstants[1].criticalTemperature) then 2 else 1;
            elseif basePropertiesInputChoice == InputChoice.ps then
              phaseOutput = if ((s > bubbleEntropy(sat) and s < dewEntropy(sat)) and
                                 p < fluidConstants[1].criticalPressure) then 2 else 1;
            elseif basePropertiesInputChoice == InputChoice.hs then
              phaseOutput = if ((s > bubbleEntropy(sat)  and s < dewEntropy(sat)) and
                                (h > bubbleEnthalpy(sat) and h < dewEnthalpy(sat))) then 2 else 1;
            elseif basePropertiesInputChoice == InputChoice.pT then
              phaseOutput = 1;
            else
              assert(false, "You are using an unsupported pair of inputs.");
            end if;
          end if;
        end BaseProperties;

        redeclare function molarMass "Return the molar mass of the medium"
            input ThermodynamicState state;
            output MolarMass MM "Mixture molar mass";
        algorithm
          MM := fluidConstants[1].molarMass;
        end molarMass;

        replaceable function getMolarMass
          output MolarMass MM "molar mass";
          external "C" MM = TwoPhaseMedium_getMolarMass_C_impl(mediumName, libraryName, substanceName)
            annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end getMolarMass;

        replaceable function getCriticalTemperature
          output Temperature Tc "Critical temperature";
          external "C" Tc = TwoPhaseMedium_getCriticalTemperature_C_impl(mediumName, libraryName, substanceName)
            annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end getCriticalTemperature;

        replaceable function getCriticalPressure
          output AbsolutePressure pc "Critical temperature";
          external "C" pc = TwoPhaseMedium_getCriticalPressure_C_impl(mediumName, libraryName, substanceName)
            annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end getCriticalPressure;

        replaceable function getCriticalMolarVolume
          output MolarVolume vc "Critical molar volume";
          external "C" vc = TwoPhaseMedium_getCriticalMolarVolume_C_impl(mediumName, libraryName, substanceName)
            annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end getCriticalMolarVolume;

        redeclare replaceable function setState_ph
        "Return thermodynamic state record from p and h"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "pressure";
          input SpecificEnthalpy h "specific enthalpy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output ThermodynamicState state;
        external "C" TwoPhaseMedium_setState_ph_C_impl(p, h, phase, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end setState_ph;

        redeclare replaceable function setState_pT
        "Return thermodynamic state record from p and T"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "pressure";
          input Temperature T "temperature";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output ThermodynamicState state;
        external "C" TwoPhaseMedium_setState_pT_C_impl(p, T, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end setState_pT;

        redeclare replaceable function setState_dT
        "Return thermodynamic state record from d and T"
          extends Modelica.Icons.Function;
          input Density d "density";
          input Temperature T "temperature";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output ThermodynamicState state;
        external "C" TwoPhaseMedium_setState_dT_C_impl(d, T, phase, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end setState_dT;

        redeclare replaceable function setState_ps
        "Return thermodynamic state record from p and s"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "pressure";
          input SpecificEntropy s "specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output ThermodynamicState state;
        external "C" TwoPhaseMedium_setState_ps_C_impl(p, s, phase, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end setState_ps;

        replaceable function setState_hs
        "Return thermodynamic state record from h and s"
          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "specific enthalpy";
          input SpecificEntropy s "specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output ThermodynamicState state;
        external "C" TwoPhaseMedium_setState_hs_C_impl(h, s, phase, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end setState_hs;

        replaceable function partialDeriv_state
        "Return partial derivative from a thermodynamic state record"
          extends Modelica.Icons.Function;
          input String of "The property to differentiate";
          input String wrt "Differentiate with respect to this";
          input String cst "Keep this constant";
          input ThermodynamicState  state;
          output Real partialDerivative;
          external "C" partialDerivative = TwoPhaseMedium_partialDeriv_state_C_impl(of, wrt, cst, state, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end partialDeriv_state;

        redeclare function extends setState_phX
        algorithm
          // The composition is an empty vector
          state :=setState_ph(p, h, phase);
        end setState_phX;

        redeclare function extends setState_pTX
        algorithm
          // The composition is an empty vector
          state :=setState_pT(p, T, phase);
        end setState_pTX;

        redeclare function extends setState_dTX
        algorithm
          // The composition is an empty vector
          state :=setState_dT(d, T, phase);
        end setState_dTX;

        redeclare function extends setState_psX
        algorithm
          // The composition is an empty vector
          state :=setState_ps(p, s, phase);
        end setState_psX;

        replaceable function setState_hsX
                                          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "specific enthalpy";
          input SpecificEntropy s "specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output ThermodynamicState state;
        algorithm
          // The composition is an empty vector
          state :=setState_hs(h, s, phase);
        end setState_hsX;

        redeclare replaceable function density_ph "Return density from p and h"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Specific enthalpy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output Density d "Density";
        algorithm
          d := density_ph_state(p=p, h=h, state=setState_ph(p=p, h=h, phase=phase));
        annotation (Inline = true);
        end density_ph;

        function density_ph_state "returns density for given p and h"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Enthalpy";
          input ThermodynamicState state;
          output Density d "density";
        algorithm
          d := density(state);
        annotation (
          Inline=false,
          LateInline=true,
          derivative(noDerivative=state)=density_ph_der);
        end density_ph_state;

        replaceable function density_ph_der "Total derivative of density_ph"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Specific enthalpy";
          input ThermodynamicState state;
          input Real p_der "time derivative of pressure";
          input Real h_der "time derivative of specific enthalpy";
          output Real d_der "time derivative of density";
        algorithm
          d_der := p_der*density_derp_h(state=state)
                 + h_der*density_derh_p(state=state);
        annotation (Inline=true);
        end density_ph_der;

        redeclare replaceable function temperature_ph
        "Return temperature from p and h"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Specific enthalpy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output Temperature T "Temperature";
        algorithm
          T := temperature_ph_state(p=p, h=h, state=setState_ph(p=p, h=h, phase=phase));
        annotation (
          Inline=true,
          inverse(h=specificEnthalpy_pT(p=p, T=T, phase=phase)));
        end temperature_ph;

        function temperature_ph_state "returns temperature for given p and h"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Enthalpy";
          input ThermodynamicState state;
          output Temperature T "Temperature";
        algorithm
          T := temperature(state);
        annotation (
          Inline=false,
          LateInline=true,
          inverse(h=specificEnthalpy_pT_state(p=p, T=T, state=state)));
        end temperature_ph_state;

        replaceable function specificEntropy_ph
        "Return specific entropy from p and h"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Specific enthalpy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output SpecificEntropy s "specific entropy";
        algorithm
          s := specificEntropy_ph_state(p=p, h=h, state=setState_ph(p=p, h=h, phase=phase));
          annotation (
          Inline=true,
          inverse(h=specificEnthalpy_ps(p=p, s=s, phase=phase)));
        end specificEntropy_ph;

        function specificEntropy_ph_state
        "returns specific entropy for a given p and h"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Specific Enthalpy";
          input ThermodynamicState state;
          output SpecificEntropy s "Specific Entropy";
        algorithm
          s := specificEntropy(state);
        annotation (
          Inline=false,
          LateInline=true,
          derivative(noDerivative=state)=specificEntropy_ph_der);
        end specificEntropy_ph_state;

        function specificEntropy_ph_der "time derivative of specificEntropy_ph"
          extends Modelica.Icons.Function;
          input AbsolutePressure p;
          input SpecificEnthalpy h;
          input ThermodynamicState state;
          input Real p_der "time derivative of pressure";
          input Real h_der "time derivative of specific enthalpy";
          output Real s_der "time derivative of specific entropy";
        algorithm
          s_der := p_der*(-1.0/(state.d*state.T))
                 + h_der*( 1.0/state.T);
        annotation (Inline = true);
        end specificEntropy_ph_der;

        redeclare replaceable function density_pT "Return density from p and T"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output Density d "Density";
        algorithm
          d := density_pT_state(p=p, T=T, state=setState_pT(p=p, T=T, phase=phase));
        annotation (
          Inline=true,
          inverse(p=pressure_dT(d=d, T=T, phase=phase)));
        end density_pT;

        function density_pT_state
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input ThermodynamicState state;
          output Density d "Density";
        algorithm
          d := density(state);
        annotation (
          Inline=false,
          LateInline=true,
          inverse(p=pressure_dT_state(d=d, T=T, state=state)));
        end density_pT_state;

        replaceable function density_pT_der "Total derivative of density_pT"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input FixedPhase phase
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Real p_der;
          input Real T_der;
          output Real d_der;
        algorithm
          d_der:=density_derp_T(setState_pT(p, T))*p_der +
                 density_derT_p(setState_pT(p, T))*T_der;
          /*  // If special definition in "C"
    external "C" d_der=  TwoPhaseMedium_density_pT_der_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
    */
          annotation(Inline = true);
        end density_pT_der;

        redeclare replaceable function specificEnthalpy_pT
        "Return specific enthalpy from p and T"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output SpecificEnthalpy h "specific enthalpy";
        algorithm
          h := specificEnthalpy_pT_state(p=p, T=T, state=setState_pT(p=p, T=T, phase=phase));
        annotation (
          Inline=true,
          inverse(T=temperature_ph(p=p, h=h, phase=phase)));
        end specificEnthalpy_pT;

        function specificEnthalpy_pT_state
        "returns specific enthalpy for given p and T"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input ThermodynamicState state;
          output SpecificEnthalpy h "specific enthalpy";
        algorithm
          h := specificEnthalpy(state);
        annotation (
          Inline=false,
          LateInline=true,
          inverse(T=temperature_ph_state(p=p, h=h, state=state)));
        end specificEnthalpy_pT_state;

        function specificEntropy_pT
        "returns specific entropy for a given p and T"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output SpecificEntropy s "Specific Entropy";
        algorithm
          s := specificEntropy_pT_state(p=p, T=T, state=setState_pT(p=p, T=T, phase=phase));
        annotation (
          Inline=true,
          inverse(T=temperature_ps(p=p, s=s, phase=phase)));
        end specificEntropy_pT;

        function specificEntropy_pT_state
        "returns specific entropy for a given p and T"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input ThermodynamicState state;
          output SpecificEntropy s "Specific Entropy";
        algorithm
          s := specificEntropy(state);
        annotation (
          Inline=false,
          LateInline=true);
        end specificEntropy_pT_state;

        redeclare replaceable function pressure_dT
        "Return pressure from d and T"
          extends Modelica.Icons.Function;
          input Density d "Density";
          input Temperature T "Temperature";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output AbsolutePressure p "Pressure";
        algorithm
          p := pressure_dT_state(d=d, T=T, state=setState_dT(d=d, T=T, phase=phase));
          annotation (
          Inline=true,
          inverse(d=density_pT(p=p, T=T, phase=phase)));
        end pressure_dT;

        function pressure_dT_state
          extends Modelica.Icons.Function;
          input Density d "Density";
          input Temperature T "Temperature";
          input ThermodynamicState state;
          output AbsolutePressure p "pressure";
        algorithm
          p := pressure(state);
        annotation (
          Inline=false,
          LateInline=true,
          inverse(d=density_pT_state(p=p, T=T, state=state)));
        end pressure_dT_state;

        redeclare replaceable function specificEnthalpy_dT
        "Return specific enthalpy from d and T"
          extends Modelica.Icons.Function;
          input Density d "Density";
          input Temperature T "Temperature";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output SpecificEnthalpy h "specific enthalpy";
        algorithm
          h := specificEnthalpy_dT_state(d=d, T=T, state=setState_dT(d=d, T=T, phase=phase));
        annotation (
          Inline=true);
        end specificEnthalpy_dT;

        function specificEnthalpy_dT_state
          extends Modelica.Icons.Function;
          input Density d "Density";
          input Temperature T "Temperature";
          input ThermodynamicState state;
          output SpecificEnthalpy h "SpecificEnthalpy";
        algorithm
          h := specificEnthalpy(state);
        annotation (
          Inline=false,
          LateInline=true);
        end specificEnthalpy_dT_state;

        function specificEntropy_dT
        "returns specific entropy for a given d and T"
          extends Modelica.Icons.Function;
          input Density d "Density";
          input Temperature T "Temperature";
          input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEntropy s "Specific Entropy";
        algorithm
          s := specificEntropy_dT_state(d=d, T=T, state=setState_dT(d=d, T=T, phase=phase));
        annotation (Inline=true);
        end specificEntropy_dT;

        function specificEntropy_dT_state
        "returns specific entropy for a given d and T"
          extends Modelica.Icons.Function;
          input Density d "Density";
          input Temperature T "Temperature";
          input ThermodynamicState state;
          output SpecificEntropy s "Specific Entropy";
        algorithm
          s := specificEntropy(state);
        annotation (
          Inline=false,
          LateInline=true);
        end specificEntropy_dT_state;

        redeclare replaceable function density_ps "Return density from p and s"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output Density d "Density";
        algorithm
          d := density_ps_state(p=p, s=s, state=setState_ps(p=p, s=s, phase=phase));
        annotation (
          Inline=true);
        end density_ps;

        function density_ps_state "Return density from p and s"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          input ThermodynamicState state;
          output Density d "Density";
        algorithm
          d := density(state);
        annotation (
          Inline=false,
          LateInline=true,
          derivative(noDerivative=state) = density_ps_der);
        end density_ps_state;

        replaceable partial function density_ps_der
        "Total derivative of density_ps"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          input ThermodynamicState state;
          input Real p_der;
          input Real h_der;
          output Real d_der;
          // To be implemented
          annotation(Inline = true);
        end density_ps_der;

        redeclare replaceable function temperature_ps
        "Return temperature from p and s"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output Temperature T "Temperature";
        algorithm
          T := temperature_ps_state(p=p, s=s, state=setState_ps(p=p, s=s, phase=phase));
        annotation (
          Inline=true,
          inverse(s=specificEntropy_pT(p=p, T=T, phase=phase)));
        end temperature_ps;

        function temperature_ps_state "returns temperature for given p and s"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          input ThermodynamicState state;
          output Temperature T "Temperature";
        algorithm
          T := temperature(state);
        annotation (
          Inline=false,
          LateInline=true,
          inverse(s=specificEntropy_pT_state(p=p, T=T, state=state)));
        end temperature_ps_state;

        redeclare replaceable function specificEnthalpy_ps
        "Return specific enthalpy from p and s"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output SpecificEnthalpy h "specific enthalpy";
        algorithm
          h := specificEnthalpy_ps_state(p=p, s=s, state=setState_ps(p=p, s=s, phase=phase));
          annotation (
          Inline = true,
          inverse(s=specificEntropy_ph(p=p, h=h, phase=phase)));
        end specificEnthalpy_ps;

        function specificEnthalpy_ps_state "Return enthalpy from p and s"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          input ThermodynamicState state;
          output SpecificEnthalpy h "Enthalpy";
        algorithm
          h := specificEnthalpy(state);
        annotation (
          Inline=false,
          LateInline=true,
          inverse(s=specificEntropy_ph_state(p=p, h=h, state=state)));
        end specificEnthalpy_ps_state;

        function density_hs "Return density for given h and s"
          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "Enthalpy";
          input SpecificEntropy s "Specific entropy";
          input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output Density d "density";
        algorithm
          d := density_hs_state(h=h, s=s, state=setState_hs(h=h, s=s, phase=phase));
        annotation (
          Inline=true);
        end density_hs;

        function density_hs_state "Return density for given h and s"
          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "Enthalpy";
          input SpecificEntropy s "Specific entropy";
          input ThermodynamicState state;
          output Density d "density";
        algorithm
          d := density(state);
        annotation (
          Inline=false,
          LateInline=true);
        end density_hs_state;

        replaceable function pressure_hs "Return pressure from h and s"
          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "Specific enthalpy";
          input SpecificEntropy s "Specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output AbsolutePressure p "Pressure";
        algorithm
          p := pressure_hs_state(h=h, s=s, state=setState_hs(h=h, s=s, phase=phase));
          annotation (
            Inline = true,
            inverse(
              h=specificEnthalpy_ps(p=p, s=s, phase=phase),
              s=specificEntropy_ph(p=p, h=h, phase=phase)));
        end pressure_hs;

        function pressure_hs_state "Return pressure for given h and s"
          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "Enthalpy";
          input SpecificEntropy s "Specific entropy";
          input ThermodynamicState state;
          output AbsolutePressure p "Pressure";
        algorithm
          p := pressure(state);
        annotation (
          Inline=false,
          LateInline=true);
        end pressure_hs_state;

        replaceable function temperature_hs "Return temperature from h and s"
          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "Specific enthalpy";
          input SpecificEntropy s "Specific entropy";
          input FixedPhase phase = 0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          output Temperature T "Temperature";
        algorithm
          T := temperature_hs_state(h=h, s=s, state=setState_hs(h=h, s=s, phase=phase));
          annotation (
            Inline = true);
        end temperature_hs;

        function temperature_hs_state "Return temperature for given h and s"
          extends Modelica.Icons.Function;
          input SpecificEnthalpy h "Enthalpy";
          input SpecificEntropy s "Specific entropy";
          input ThermodynamicState state;
          output Temperature T "Temperature";
        algorithm
          T := temperature(state);
        annotation (
          Inline=false,
          LateInline=true);
        end temperature_hs_state;

        redeclare function extends prandtlNumber "Returns Prandtl number"
          /*  // If special definition in "C"
  external "C" T=  TwoPhaseMedium_prandtlNumber_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end prandtlNumber;

        redeclare replaceable function extends temperature
        "Return temperature from state"
          // Standard definition
        algorithm
          T := state.T;
          /*  // If special definition in "C"
  external "C" T=  TwoPhaseMedium_temperature_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end temperature;

        redeclare replaceable function extends velocityOfSound
        "Return velocity of sound from state"
          // Standard definition
        algorithm
          a := state.a;
          /*  // If special definition in "C"
  external "C" a=  TwoPhaseMedium_velocityOfSound_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end velocityOfSound;

        redeclare replaceable function extends isobaricExpansionCoefficient
        "Return isobaric expansion coefficient from state"
          // Standard definition
        algorithm
          beta := state.beta;
          /*  // If special definition in "C"
  external "C" beta=  TwoPhaseMedium_isobaricExpansionCoefficient_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end isobaricExpansionCoefficient;

        redeclare replaceable function extends isentropicExponent
        "Return isentropic exponent"
          extends Modelica.Icons.Function;
        algorithm
          gamma := density(state) / pressure(state) * velocityOfSound(state) * velocityOfSound(state);
        end isentropicExponent;

        redeclare replaceable function extends specificHeatCapacityCp
        "Return specific heat capacity cp from state"
          // Standard definition
        algorithm
          cp := state.cp;
          /*  // If special definition in "C"
  external "C" cp=  TwoPhaseMedium_specificHeatCapacityCp_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end specificHeatCapacityCp;

        redeclare replaceable function extends specificHeatCapacityCv
        "Return specific heat capacity cv from state"
          // Standard definition
        algorithm
          cv := state.cv;
          /*  // If special definition in "C"
  external "C" cv=  TwoPhaseMedium_specificHeatCapacityCv_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end specificHeatCapacityCv;

        redeclare replaceable function extends density
        "Return density from state"
          // Standard definition
        algorithm
          d := state.d;
          /*  // If special definition in "C"
  external "C" d=  TwoPhaseMedium_density_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end density;

        redeclare replaceable function extends density_derh_p
        "Return derivative of density wrt enthalpy at constant pressure from state"
          // Standard definition
        algorithm
          ddhp := state.ddhp;
          /*  // If special definition in "C"
  external "C" ddhp=  TwoPhaseMedium_density_derh_p_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end density_derh_p;

        redeclare replaceable function extends density_derp_h
        "Return derivative of density wrt pressure at constant enthalpy from state"
          // Standard definition
        algorithm
          ddph := state.ddph;
          /*  // If special definition in "C"
  external "C" ddph=  TwoPhaseMedium_density_derp_h_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end density_derp_h;

        redeclare replaceable function extends density_derp_T
        algorithm
          ddpT := state.kappa*state.d;
        end density_derp_T;

        redeclare replaceable function extends density_derT_p
        algorithm
          ddTp :=-state.beta*state.d;
        end density_derT_p;

        redeclare replaceable function extends dynamicViscosity
        "Return dynamic viscosity from state"
          // Standard definition
        algorithm
          eta := state.eta;
          /*  // If special definition in "C"
  external "C" eta=  TwoPhaseMedium_dynamicViscosity_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dynamicViscosity;

        redeclare replaceable function extends specificEnthalpy
        "Return specific enthalpy from state"
          // Standard definition
        algorithm
          h := state.h;
          /*  // If special definition in "C"
  external "C" h=  TwoPhaseMedium_specificEnthalpy_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end specificEnthalpy;

        redeclare replaceable function extends specificInternalEnergy
        "Returns specific internal energy"
          extends Modelica.Icons.Function;
        algorithm
          u := specificEnthalpy(state) - pressure(state)/density(state);
        end specificInternalEnergy;

        redeclare replaceable function extends isothermalCompressibility
        "Return isothermal compressibility from state"
          // Standard definition
        algorithm
          kappa := state.kappa;
          /*  // If special definition in "C"
  external "C" kappa=  TwoPhaseMedium_isothermalCompressibility_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end isothermalCompressibility;

        redeclare replaceable function extends thermalConductivity
        "Return thermal conductivity from state"
          // Standard definition
        algorithm
          lambda := state.lambda;
          /*  // If special definition in "C"
  external "C" lambda=  TwoPhaseMedium_thermalConductivity_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end thermalConductivity;

        redeclare replaceable function extends pressure
        "Return pressure from state"
          // Standard definition
        algorithm
          p := state.p;
          /*  // If special definition in "C"
  external "C" p=  TwoPhaseMedium_pressure_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end pressure;

        redeclare replaceable function extends specificEntropy
        "Return specific entropy from state"
          // Standard definition
        algorithm
          s := state.s;
          /*  // If special definition in "C"
    external "C" s=  TwoPhaseMedium_specificEntropy_C_impl(state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end specificEntropy;

        redeclare replaceable function extends isentropicEnthalpy
        external "C" h_is = TwoPhaseMedium_isentropicEnthalpy_C_impl(p_downstream, refState,
         mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end isentropicEnthalpy;

        redeclare replaceable function setSat_p
        "Return saturation properties from p"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "pressure";
          output SaturationProperties sat "saturation property record";
        external "C" TwoPhaseMedium_setSat_p_C_impl(p, sat, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end setSat_p;

        replaceable function setSat_p_state
        "Return saturation properties from the state"
          extends Modelica.Icons.Function;
          input ThermodynamicState state;
          output SaturationProperties sat "saturation property record";
          // Standard definition
        algorithm
          sat:=setSat_p(state.p);
          //Redeclare this function for more efficient implementations avoiding the repeated computation of saturation properties
        /*  // If special definition in "C"
  external "C" TwoPhaseMedium_setSat_p_state_C_impl(state, sat)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end setSat_p_state;

        redeclare replaceable function setSat_T
        "Return saturation properties from p"
          extends Modelica.Icons.Function;
          input Temperature T "temperature";
          output SaturationProperties sat "saturation property record";
        external "C" TwoPhaseMedium_setSat_T_C_impl(T, sat, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end setSat_T;

        replaceable function setSat_T_state
        "Return saturation properties from the state"
          extends Modelica.Icons.Function;
          input ThermodynamicState state;
          output SaturationProperties sat "saturation property record";
          // Standard definition
        algorithm
          sat:=setSat_T(state.T);
          //Redeclare this function for more efficient implementations avoiding the repeated computation of saturation properties
        /*  // If special definition in "C"
  external "C" TwoPhaseMedium_setSat_T_state_C_impl(state, sat)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end setSat_T_state;

        redeclare replaceable function extends setBubbleState
        "set the thermodynamic state on the bubble line"
          extends Modelica.Icons.Function;
          input SaturationProperties sat "saturation point";
          input FixedPhase phase(min = 1, max = 2) =  1
          "phase: default is one phase";
          output ThermodynamicState state "complete thermodynamic state info";
          // Standard definition
        algorithm
          state :=setState_ph(sat.psat, sat.hl, phase);
          /*  // If special definition in "C"
  external "C" TwoPhaseMedium_setBubbleState_C_impl(sat, phase, state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end setBubbleState;

        redeclare replaceable function extends setDewState
        "set the thermodynamic state on the dew line"
          extends Modelica.Icons.Function;
          input SaturationProperties sat "saturation point";
          input FixedPhase phase(min = 1, max = 2) = 1
          "phase: default is one phase";
          output ThermodynamicState state "complete thermodynamic state info";
          // Standard definition
        algorithm
          state :=setState_ph(sat.psat, sat.hv, phase);
          /*  // If special definition in "C"
  external "C" TwoPhaseMedium_setDewState_C_impl(sat, phase, state, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end setDewState;

        redeclare replaceable function extends saturationTemperature
          // Standard definition
        algorithm
          T :=saturationTemperature_sat(setSat_p(p));
          /*  // If special definition in "C"
  external "C" T=  TwoPhaseMedium_saturationTemperature_C_impl(p, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end saturationTemperature;

        redeclare function extends saturationTemperature_sat

          annotation(Inline = true);
        end saturationTemperature_sat;

        redeclare replaceable function extends saturationTemperature_derp "Returns derivative of saturation temperature w.r.t.. pressureBeing this function inefficient, it is strongly recommended to use saturationTemperature_derp_sat
     and never use saturationTemperature_derp directly"
        external "C" dTp = TwoPhaseMedium_saturationTemperature_derp_C_impl(p, mediumName, libraryName, substanceName)
          annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
        end saturationTemperature_derp;

        redeclare replaceable function saturationTemperature_derp_sat
        "Returns derivative of saturation temperature w.r.t.. pressure"
          extends Modelica.Icons.Function;
          input SaturationProperties sat "saturation property record";
          output Real dTp
          "derivative of saturation temperature w.r.t. pressure";
          // Standard definition
        algorithm
          dTp := sat.dTp;
          /*  // If special definition in "C"
  external "C" dTp=  TwoPhaseMedium_saturationTemperature_derp_sat_C_impl(sat.psat, sat.Tsat, sat.uniqueID, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end saturationTemperature_derp_sat;

        redeclare replaceable function extends dBubbleDensity_dPressure
        "Returns bubble point density derivative"
          // Standard definition
        algorithm
          ddldp := sat.ddldp;
          /*  // If special definition in "C"
  external "C" ddldp=  TwoPhaseMedium_dBubbleDensity_dPressure_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dBubbleDensity_dPressure;

        redeclare replaceable function extends dDewDensity_dPressure
        "Returns dew point density derivative"
          // Standard definition
        algorithm
          ddvdp := sat.ddvdp;
          /*  // If special definition in "C"
  external "C" ddvdp=  TwoPhaseMedium_dDewDensity_dPressure_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dDewDensity_dPressure;

        redeclare replaceable function extends dBubbleEnthalpy_dPressure
        "Returns bubble point specific enthalpy derivative"
          // Standard definition
        algorithm
          dhldp := sat.dhldp;
          /*  // If special definition in "C"
  external "C" dhldp=  TwoPhaseMedium_dBubbleEnthalpy_dPressure_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dBubbleEnthalpy_dPressure;

        redeclare replaceable function extends dDewEnthalpy_dPressure
        "Returns dew point specific enthalpy derivative"
          // Standard definition
        algorithm
          dhvdp := sat.dhvdp;
          /*  // If special definition in "C"
  external "C" dhvdp=  TwoPhaseMedium_dDewEnthalpy_dPressure_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dDewEnthalpy_dPressure;

        redeclare replaceable function extends bubbleDensity
        "Returns bubble point density"
          // Standard definition
        algorithm
          dl := sat.dl;
          /*  // If special definition in "C"
  external "C" dl=  TwoPhaseMedium_bubbleDensity_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end bubbleDensity;

        redeclare replaceable function extends dewDensity
        "Returns dew point density"
          // Standard definition
        algorithm
          dv := sat.dv;
          /*  // If special definition in "C"
  external "C" dv=  TwoPhaseMedium_dewDensity_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dewDensity;

        redeclare replaceable function extends bubbleEnthalpy
        "Returns bubble point specific enthalpy"
          // Standard definition
        algorithm
          hl := sat.hl;
          /*  // If special definition in "C"
  external "C" hl=  TwoPhaseMedium_bubbleEnthalpy_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end bubbleEnthalpy;

        redeclare replaceable function extends dewEnthalpy
        "Returns dew point specific enthalpy"
          // Standard definition
        algorithm
          hv := sat.hv;
          /*  // If special definition in "C"
  external "C" hv=  TwoPhaseMedium_dewEnthalpy_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dewEnthalpy;

        redeclare replaceable function extends saturationPressure
          // Standard definition
        algorithm
          p :=saturationPressure_sat(setSat_T(T));
          /*  // If special definition in "C"
  external "C" p=  TwoPhaseMedium_saturationPressure_C_impl(T, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = false,
                     LateInline = true,
                     derivative = saturationPressure_der);
        end saturationPressure;

        function saturationPressure_der
        "Return saturation pressure time derivative"
          extends Modelica.Icons.Function;
          input Temperature T "temperature";
          input Real T_der "Temperature derivative";
          output Real p_der "saturation pressure derivative";
          // Standard definition
        algorithm
          p_der :=T_der/saturationTemperature_derp_sat(setSat_T(T));
          annotation(Inline = true);
        end saturationPressure_der;

        redeclare function extends saturationPressure_sat

          annotation(Inline = true);
        end saturationPressure_sat;

        redeclare replaceable function extends surfaceTension
        "Returns surface tension sigma in the two phase region"
          //Standard definition
        algorithm
          sigma := sat.sigma;
          /*  //If special definition in "C"
  external "C" sigma=  TwoPhaseMedium_surfaceTension_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end surfaceTension;

        redeclare replaceable function extends bubbleEntropy
        "Returns bubble point specific entropy"
          //Standard definition
        algorithm
          sl := specificEntropy(setBubbleState(sat));
          /*  //If special definition in "C"
  external "C" sl=  TwoPhaseMedium_bubbleEntropy_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end bubbleEntropy;

        redeclare replaceable function extends dewEntropy
        "Returns dew point specific entropy"
          //Standard definition
        algorithm
          sv := specificEntropy(setDewState(sat));
          /*  //If special definition in "C"
  external "C" sv=  TwoPhaseMedium_dewEntropy_C_impl(sat, mediumName, libraryName, substanceName)
    annotation(Include="#include \"externalmedialib.h\"", Library="ExternalMediaLib", IncludeDirectory="modelica://ExternalMedia/Resources/Include", LibraryDirectory="modelica://ExternalMedia/Resources/Library");
*/
          annotation(Inline = true);
        end dewEntropy;
      end ExternalTwoPhaseMedium;
    end BaseClasses;
  end Media;
  annotation(uses(Modelica(version="3.2.1")),
  Documentation(info="<html>
<p>The <b>ExternalMedia</b> library provides a framework for interfacing external codes computing fluid properties to Modelica.Media-compatible component models. The library has been designed with two main goals: maximizing the efficiency of the code, while minimizing the amount of extra code required to interface existing external codes to the library.</p>
<p>The library covers pure fluids models, possibly two-phase, compliant with the <a href=\"modelica://Modelica.Media.Interfaces.PartialTwoPhaseMedium\">Modelica.Media.Interfaces.PartialTwoPhaseMedium</a> interface. </p>
<p>Two external softwares for fluid property computation are currently suppored by the ExternalMedia library:</p>
<ul>
<li><a href=\"http://www.fluidprop.com\">FluidProp</a>, formerly developed at TU Delft and currently devloped and maintained by Asimptote</li>
<li><a href=\"http://coolprop.org\">CoolProp</a>, developed at the University of Liege and at the Technical University of Denmark (DTU)</li>
</ul>
<p>The library has been tested with the Dymola and OpenModelica tools under the Windows operating system. If you are interested in the support of other tools, operating systems, and external fluid property computation codes, please contact the developers.</p>
<p>Main contributors: Francesco Casella, Christoph Richter, Roberto Bonifetto, Ian Bell.</p>
<p><b>The code is licensed under the Modelica License 2. </b>For license conditions (including the disclaimer of warranty) visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\">https://www.modelica.org/licenses/ModelicaLicense2</a>. </p>
<p>Copyright &copy; 2006-2014, Politecnico di Milano, TU Braunschweig, Politecnico di Torino, Universit&eacute; de Liege.</p>
</html>"));
end ExternalMedia;

package ModelicaServices
  "(version = 3.2.1, target = \"Dymola\") Models and functions used in the Modelica Standard Library requiring a tool specific implementation"

package Machine

  final constant Real eps=1.e-15 "Biggest number such that 1.0 + eps = 1.0";

  final constant Real small=1.e-60
  "Smallest number such that small and -small are representable on the machine";

  final constant Real inf=1.e+60
  "Biggest Real number such that inf and -inf are representable on the machine";
  annotation (Documentation(info="<html>
<p>
Package in which processor specific constants are defined that are needed
by numerical algorithms. Typically these constants are not directly used,
but indirectly via the alias definition in
<a href=\"modelica://Modelica.Constants\">Modelica.Constants</a>.
</p>
</html>"));
end Machine;
annotation (
  Protection(access=Access.hide),
  preferredView="info",
  version="3.2.1",
  versionDate="2013-01-17",
  versionBuild=1,
  uses(Modelica(version="3.2.1")),
  conversion(
    noneFromVersion="1.0",
    noneFromVersion="1.1",
    noneFromVersion="1.2"),
  Documentation(info="<html>
<p>
This package contains a set of functions and models to be used in the
Modelica Standard Library that requires a tool specific implementation.
These are:
</p>

<ul>
<li> <a href=\"modelica://ModelicaServices.Animation.Shape\">Shape</a>
     provides a 3-dim. visualization of elementary
     mechanical objects. It is used in
<a href=\"modelica://Modelica.Mechanics.MultiBody.Visualizers.Advanced.Shape\">Modelica.Mechanics.MultiBody.Visualizers.Advanced.Shape</a>
     via inheritance.</li>

<li> <a href=\"modelica://ModelicaServices.Animation.Surface\">Surface</a>
     provides a 3-dim. visualization of
     moveable parameterized surface. It is used in
<a href=\"modelica://Modelica.Mechanics.MultiBody.Visualizers.Advanced.Surface\">Modelica.Mechanics.MultiBody.Visualizers.Advanced.Surface</a>
     via inheritance.</li>

<li> <a href=\"modelica://ModelicaServices.ExternalReferences.loadResource\">loadResource</a>
     provides a function to return the absolute path name of an URI or a local file name. It is used in
<a href=\"modelica://Modelica.Utilities.Files.loadResource\">Modelica.Utilities.Files.loadResource</a>
     via inheritance.</li>

<li> <a href=\"modelica://ModelicaServices.Machine\">ModelicaServices.Machine</a>
     provides a package of machine constants. It is used in
<a href=\"modelica://Modelica.Constants\">Modelica.Constants</a>.</li>

<li> <a href=\"modelica://ModelicaServices.Types.SolverMethod\">Types.SolverMethod</a>
     provides a string defining the integration method to solve differential equations in
     a clocked discretized continuous-time partition (see Modelica 3.3 language specification).
     It is not yet used in the Modelica Standard Library, but in the Modelica_Synchronous library
     that provides convenience blocks for the clock operators of Modelica version &ge; 3.3.</li>
</ul>

<p>
This implementation is targeted for Dymola.
</p>

<p>
<b>Licensed by DLR and Dassault Syst&egrave;mes AB under the Modelica License 2</b><br>
Copyright &copy; 2009-2013, DLR and Dassault Syst&egrave;mes AB.
</p>

<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\"> http://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>

</html>"));
end ModelicaServices;

package ThermoPower "Open library for thermal power plant simulation"
  extends Modelica.Icons.Package;
  import SI = Modelica.SIunits;
  import NonSI = Modelica.SIunits.Conversions.NonSIunits;

model System "System wide properties and defaults"
  // Assumptions
  parameter Boolean allowFlowReversal=true
    "= false to restrict to design flow direction (flangeA -> flangeB)"
    annotation (Evaluate=true);
  parameter Choices.Init.Options initOpt = ThermoPower.Choices.Init.Options.fixedState;
  // parameter ThermoPower.Choices.System.Dynamics Dynamics=ThermoPower.Choices.System.Dynamics.DynamicFreeInitial;
  annotation (
    defaultComponentName="system",
    defaultComponentPrefixes="inner",
    missingInnerMessage="The System object is missing, please drag it on the top layer of your model",
    Icon(graphics={Polygon(
          points={{-100,60},{-60,100},{60,100},{100,60},{100,-60},{60,-100},{-60,
              -100},{-100,-60},{-100,60}},
          lineColor={0,0,255},
          smooth=Smooth.None,
          fillColor={170,213,255},
          fillPattern=FillPattern.Solid), Text(
          extent={{-80,40},{80,-20}},
          lineColor={0,0,255},
          textString="system")}));
end System;

package Icons "Icons for ThermoPower library"
  extends Modelica.Icons.Package;

  package Water "Icons for component using water/steam as working fluid"
    extends Modelica.Icons.Package;

    partial model Pump

      annotation (Icon(graphics={
            Polygon(
              points={{-40,-24},{-60,-60},{60,-60},{40,-24},{-40,-24}},
              lineColor={0,0,255},
              pattern=LinePattern.None,
              fillColor={0,0,191},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{-60,80},{60,-40}},
              lineColor={0,0,0},
              fillPattern=FillPattern.Sphere),
            Polygon(
              points={{-30,52},{-30,-8},{48,20},{-30,52}},
              lineColor={0,0,0},
              pattern=LinePattern.None,
              fillPattern=FillPattern.HorizontalCylinder,
              fillColor={255,255,255}),
            Text(extent={{-100,-64},{100,-90}}, textString="%name")}));
    end Pump;
  end Water;
end Icons;

  package Choices "Choice enumerations for ThermoPower models"
    extends Modelica.Icons.Package;

    package Init "Options for initialisation"

      type Options = enumeration(
        noInit "No initial equations",
        fixedState "Fixed initial state variables",
        steadyState "Steady-state initialization",
        steadyStateNoP
          "Steady-state initialization except pressures (deprecated)",
        steadyStateNoT
          "Steady-state initialization except temperatures (deprecated)",
        steadyStateNoPT
          "Steady-state initialization except pressures and temperatures (deprecated)")
      "Type, constants and menu choices to select the initialisation options";
    end Init;
  end Choices;

package Functions "Miscellaneous functions"
  extends Modelica.Icons.Package;

  package PumpCharacteristics "Functions for pump characteristics"

    partial function baseFlow "Base class for pump flow characteristics"
      extends Modelica.Icons.Function;
      input SI.VolumeFlowRate q_flow "Volumetric flow rate";
      output SI.Height head "Pump head";
    end baseFlow;

    partial function basePower
    "Base class for pump power consumption characteristics"
      extends Modelica.Icons.Function;
      input SI.VolumeFlowRate q_flow "Volumetric flow rate";
      output SI.Power consumption "Power consumption at nominal density";
    end basePower;

    partial function baseEfficiency "Base class for efficiency characteristics"
      extends Modelica.Icons.Function;
      input SI.VolumeFlowRate q_flow "Volumetric flow rate";
      output SI.PerUnit eta "Efficiency";
    end baseEfficiency;

    function constantPower "Constant power consumption characteristic"
      extends basePower;
      input SI.Power power=0 "Constant power consumption" annotation(Dialog);
    algorithm
      consumption := power;
    end constantPower;

    function constantEfficiency "Constant efficiency characteristic"
      extends baseEfficiency;
      input Real eta_nom "Nominal efficiency" annotation(Dialog);
    algorithm
      eta := eta_nom;
    end constantEfficiency;
  end PumpCharacteristics;
  annotation (Documentation(info="<HTML>
This package contains general-purpose functions and models
</HTML>"));
end Functions;

  package Water "Models of components with water/steam as working fluid"

    connector Flange "Flange connector for water/steam flows"
      replaceable package Medium = StandardWater constrainedby
      Modelica.Media.Interfaces.PartialMedium "Medium model";
      flow Medium.MassFlowRate m_flow
      "Mass flow rate from the connection point into the component";
      Medium.AbsolutePressure p
      "Thermodynamic pressure in the connection point";
      stream Medium.SpecificEnthalpy h_outflow
      "Specific thermodynamic enthalpy close to the connection point if m_flow < 0";
      stream Medium.MassFraction Xi_outflow[Medium.nXi]
      "Independent mixture mass fractions m_i/m close to the connection point if m_flow < 0";
      stream Medium.ExtraProperty C_outflow[Medium.nC]
      "Properties c_i/m close to the connection point if m_flow < 0";
      annotation (
        Documentation(info="<HTML>.
</HTML>",   revisions="<html>
<ul>
<li><i>16 Dec 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Medium model added.</li>
<li><i>1 Oct 2003</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       First release.</li>
</ul>
</html>"),
        Diagram(graphics),
        Icon(graphics));
    end Flange;

    connector FlangeA "A-type flange connector for water/steam flows"
      extends ThermoPower.Water.Flange;
      annotation (Icon(graphics={Ellipse(
              extent={{-100,100},{100,-100}},
              lineColor={0,0,255},
              fillColor={0,0,255},
              fillPattern=FillPattern.Solid)}));
    end FlangeA;

    connector FlangeB "B-type flange connector for water/steam flows"
      extends ThermoPower.Water.Flange;
      annotation (Icon(graphics={Ellipse(
              extent={{-100,100},{100,-100}},
              lineColor={0,0,255},
              fillColor={0,0,255},
              fillPattern=FillPattern.Solid), Ellipse(
              extent={{-40,40},{40,-40}},
              lineColor={0,0,255},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid)}));
    end FlangeB;
    extends Modelica.Icons.Package;

    package StandardWater = Modelica.Media.Water.StandardWater;

    package BaseClasses "Contains partial models"
      extends Modelica.Icons.BasesPackage;

      partial model PumpBase "Base model for centrifugal pumps"
        extends Icons.Water.Pump;
        replaceable package Medium = StandardWater constrainedby
        Modelica.Media.Interfaces.PartialMedium "Medium model"
          annotation(choicesAllMatching = true);
        Medium.ThermodynamicState inletFluidState
        "Thermodynamic state of the fluid at the inlet";
        replaceable function flowCharacteristic =
            ThermoPower.Functions.PumpCharacteristics.baseFlow
        "Head vs. q_flow characteristic at nominal speed and density"
          annotation (Dialog(group="Characteristics"),choicesAllMatching=true);
        parameter Boolean usePowerCharacteristic=false
        "Use powerCharacteristic (vs. efficiencyCharacteristic)"
          annotation (Dialog(group="Characteristics"));
        replaceable function powerCharacteristic =
            Functions.PumpCharacteristics.constantPower constrainedby
        ThermoPower.Functions.PumpCharacteristics.basePower
        "Power consumption vs. q_flow at nominal speed and density"   annotation (
           Dialog(group="Characteristics", enable=usePowerCharacteristic),
            choicesAllMatching=true);
        replaceable function efficiencyCharacteristic =
            Functions.PumpCharacteristics.constantEfficiency (eta_nom=0.8)
          constrainedby
        ThermoPower.Functions.PumpCharacteristics.baseEfficiency
        "Efficiency vs. q_flow at nominal speed and density"   annotation (Dialog(
              group="Characteristics", enable=not usePowerCharacteristic),
            choicesAllMatching=true);
        parameter Integer Np0(min=1) = 1 "Nominal number of pumps in parallel";
        parameter Boolean use_in_Np = false
        "Use connector input for the pressure"
          annotation(Dialog(group="External inputs"), choices(checkBox=true));
        parameter Units.LiquidDensity rho0=1000 "Nominal Liquid Density"
          annotation (Dialog(group="Characteristics"));
        parameter NonSI.AngularVelocity_rpm n0 "Nominal rotational speed"
          annotation (Dialog(group="Characteristics"));
        parameter SI.Volume V=0 "Pump Internal Volume" annotation (Evaluate=true);
        parameter Boolean CheckValve=false "Reverse flow stopped";
        parameter Boolean allowFlowReversal=system.allowFlowReversal
        "= true to allow flow reversal, false restricts to design direction";
        outer ThermoPower.System system "System wide properties";
        parameter Medium.MassFlowRate wstart=w0 "Mass Flow Rate Start Value"
          annotation (Dialog(tab="Initialisation"));
        parameter Medium.SpecificEnthalpy hstart=1e5
        "Specific Enthalpy Start Value"
          annotation (Dialog(tab="Initialisation"));
        parameter Choices.Init.Options initOpt=system.initOpt
        "Initialisation option"
          annotation (Dialog(tab="Initialisation"));
        parameter Boolean noInitialPressure=false
        "Remove initial equation on pressure"
          annotation (Dialog(tab="Initialisation"),choices(checkBox=true));
        constant SI.Acceleration g=Modelica.Constants.g_n;
        parameter Medium.MassFlowRate w0 "Nominal mass flow rate"
          annotation (Dialog(group="Characteristics"));
        parameter SI.Pressure dp0 "Nominal pressure increase"
          annotation (Dialog(group="Characteristics"));
        final parameter SI.VolumeFlowRate q_single0=w0/(Np0*rho0)
        "Nominal volume flow rate (single pump)"
          annotation(Evaluate = true);
        final parameter SI.Height head0=dp0/(rho0*g) "Nominal pump head"
          annotation(Evaluate = true);
        final parameter Real d_head_dq_0=
          (flowCharacteristic(q_single0*1.05) - flowCharacteristic(q_single0*0.95))/
          (q_single0*0.1)
        "Approximate derivative of flow characteristic w.r.t. volume flow"
          annotation(Evaluate = true);
        final parameter Real d_head_dn_0 = 2/n0*head0 - q_single0/n0*d_head_dq_0
        "Approximate derivative of the flow characteristic w.r.t. rotational speed"
          annotation(Evaluate = true);

        Medium.MassFlowRate w_single(start=wstart/Np0)
        "Mass flow rate (single pump)";
        Medium.MassFlowRate w=Np*w_single "Mass flow rate (total)";
        SI.VolumeFlowRate q_single(start=wstart/(Np0*rho0))
        "Volume flow rate (single pump)";
        SI.VolumeFlowRate q=Np*q_single "Volume flow rate (total)";
        SI.Pressure dp "Outlet pressure minus inlet pressure";
        SI.Height head "Pump head";
        Medium.SpecificEnthalpy h(start=hstart) "Fluid specific enthalpy";
        Medium.SpecificEnthalpy hin "Enthalpy of entering fluid";
        Medium.SpecificEnthalpy hout "Enthalpy of outgoing fluid";
        Units.LiquidDensity rho "Liquid density";
        Medium.Temperature Tin "Liquid inlet temperature";
        NonSI.AngularVelocity_rpm n "Shaft r.p.m.";
        Integer Np(min=1) "Number of pumps in parallel";
        SI.Power W_single "Power Consumption (single pump)";
        SI.Power W=Np*W_single "Power Consumption (total)";
        SI.Power Qloss = 0 "Heat loss (single pump)";
        constant SI.Power W_eps=1e-8
        "Small coefficient to avoid numerical singularities";
        constant NonSI.AngularVelocity_rpm n_eps=1e-6;
        SI.PerUnit eta "Pump efficiency";
        SI.PerUnit s(start = 1) "Auxiliary non-dimensional variable";
        FlangeA infl(redeclare package Medium = Medium, m_flow(min=if
                allowFlowReversal then -Modelica.Constants.inf else 0))
          annotation (Placement(transformation(extent={{-100,0},{-60,40}},
                rotation=0)));
        FlangeB outfl(redeclare package Medium = Medium, m_flow(max=if
                allowFlowReversal then +Modelica.Constants.inf else 0))
          annotation (Placement(transformation(extent={{40,50},{80,90}}, rotation=
                 0)));
        Modelica.Blocks.Interfaces.IntegerInput in_Np if use_in_Np
        "Number of  parallel pumps"
          annotation (Placement(transformation(
              origin={28,80},
              extent={{-10,-10},{10,10}},
              rotation=270)));
    protected
        Modelica.Blocks.Interfaces.IntegerInput int_Np
        "Internal connector with number of parallel pumps";

      equation
        // Flow equations
        q_single = w_single/homotopy(rho, rho0);
        head = dp/(homotopy(rho, rho0)*g);
        if noEvent(s > 0 or (not CheckValve)) then
          // Flow characteristics when check valve is open
          q_single = s*q_single0;
          head = homotopy((n/n0)^2*flowCharacteristic(q_single*n0/(n + n_eps)),
                           head0 + d_head_dq_0*(q_single - q_single0) +
                                   d_head_dn_0*(n - n0));
        else
          // Flow characteristics when check valve is closed
          head = homotopy((n/n0)^2*flowCharacteristic(0) - s*head0,
                           head0 + d_head_dq_0*(q_single - q_single0) +
                                   d_head_dn_0*(n - n0));
          q_single = 0;
        end if;

        // Power consumption
        if usePowerCharacteristic then
          W_single = (n/n0)^3*(rho/rho0)*
                      powerCharacteristic(q_single*n0/(n + n_eps))
          "Power consumption (single pump)";
          eta = (dp*q_single)/(W_single + W_eps) "Hydraulic efficiency";
        else
          eta = efficiencyCharacteristic(q_single*n0/(n + n_eps));
          W_single = dp*q_single/eta;
        end if;

        // Fluid properties
        inletFluidState = Medium.setState_ph(infl.p, hin);
        rho = Medium.density(inletFluidState);
        Tin = Medium.temperature(inletFluidState);

        // Boundary conditions
        dp = outfl.p - infl.p;
        w = infl.m_flow "Pump total flow rate";
        hin = homotopy(if not allowFlowReversal then inStream(infl.h_outflow)
                       else if w >= 0 then inStream(infl.h_outflow)
                       else inStream(outfl.h_outflow),
                       inStream(infl.h_outflow));
        infl.h_outflow = hout;
        outfl.h_outflow = hout;
        h = hout;

        // Mass and energy balances
        infl.m_flow + outfl.m_flow = 0 "Mass balance";
        if V > 0 then
          (rho*V*der(h)) = (outfl.m_flow/Np)*hout + (infl.m_flow/Np)*hin +
            W_single - Qloss "Energy balance";
        else
          0 = (outfl.m_flow/Np)*hout + (infl.m_flow/Np)*hin + W_single - Qloss
          "Energy balance";
        end if;

        // Number of pumps in parallel
        connect(in_Np, int_Np);
        if not use_in_Np then
          int_Np = Np0;
        end if;
        Np = int_Np;

      initial equation
        if initOpt == Choices.Init.Options.noInit then
          // do nothing
        elseif initOpt == Choices.Init.Options.fixedState then
          if V > 0 then
            h = hstart;
          end if;
        elseif initOpt == Choices.Init.Options.steadyState then
          if V > 0 then
            der(h) = 0;
          end if;
        else
          assert(false, "Unsupported initialisation option");
        end if;

        annotation (
          Icon(graphics),
          Diagram(graphics),
          Documentation(info="<HTML>
<p>This is the base model for the <tt>Pump</tt> and <tt>
PumpMech</tt> pump models.
<p>The model describes a centrifugal pump, or a group of <tt>Np</tt> identical pumps in parallel. The pump model is based on the theory of kinematic similarity: the pump characteristics are given for nominal operating conditions (rotational speed and fluid density), and then adapted to actual operating condition, according to the similarity equations.
<p>In order to avoid singularities in the computation of the outlet enthalpy at zero flowrate, the thermal capacity of the fluid inside the pump body can be taken into account.
<p>The model can either support reverse flow conditions or include a built-in check valve to avoid flow reversal.
<p><b>Modelling options</b></p>
<p> The nominal hydraulic characteristic (head vs. volume flow rate) is given by the the replaceable function <tt>flowCharacteristic</tt>.
<p> The pump energy balance can be specified in two alternative ways:
<ul>
<li><tt>usePowerCharacteristic = false</tt> (default option): the replaceable function <tt>efficiencyCharacteristic</tt> (efficiency vs. volume flow rate in nominal conditions) is used to determine the efficiency, and then the power consumption. The default is a constant efficiency of 0.8.
<li><tt>usePowerCharacteristic = true</tt>: the replaceable function <tt>powerCharacteristic</tt> (power consumption vs. volume flow rate in nominal conditions) is used to determine the power consumption, and then the efficiency.
</ul>
<p>
Several functions are provided in the package <tt>Functions.PumpCharacteristics</tt> to specify the characteristics as a function of some operating points at nominal conditions.
<p>Depending on the value of the <tt>checkValve</tt> parameter, the model either supports reverse flow conditions, or includes a built-in check valve to avoid flow reversal.

<p>If the <tt>in_Np</tt> input connector is wired, it provides the number of pumps in parallel; otherwise,  <tt>Np0</tt> parallel pumps are assumed.</p>
<p>It is possible to take into account the heat capacity of the fluid inside the pump by specifying its volume <tt>V</tt> at nominal conditions; this is necessary to avoid singularities in the computation of the outlet enthalpy in case of zero flow rate. If zero flow rate conditions are always avoided, this dynamic effect can be neglected by leaving the default value <tt>V = 0</tt>, thus avoiding a fast state variable in the model.
<p>The <tt>CheckValve</tt> parameter determines whether the pump has a built-in check valve or not.
<p>If <tt>computeNPSHa = true</tt>, the available net positive suction head is also computed; this requires a two-phase medium model to provide the fluid saturation pressure.
</HTML>",   revisions="<html>
<ul>
<li><i>31 Oct 2006</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
      Added initialisation parameter <tt>wstart</tt>.</li>
<li><i>5 Nov 2005</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
      Model restructured according to kinematic similarity theory.<br>
      Characteristics now specified by replaceable functions.</li>
<li><i>6 Apr 2005</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       <tt>CharData</tt> substituted by <tt>OpPoints</tt></li>
<li><i>16 Dec 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Standard medium definition added.</li>
<li><i>2 Aug 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Optional NPSHa computation added. Changed parameter names</li>
<li><i>5 Jul 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Model restructured by using inheritance. Adapted to Modelica.Media.</li>
<li><i>15 Jan 2004</i>
    by <a href=\"mailto:francesco.schiavo@polimi.it\">Francesco Schiavo</a>:<br>
       <tt>ThermalCapacity</tt> and <tt>CheckValve</tt> added.</li>
<li><i>15 Dec 2003</i>
    by <a href=\"mailto:francesco.schiavo@polimi.it\">Francesco Schiavo</a>:<br>
       First release.</li>
</ul>
</html>"));
      end PumpBase;
    end BaseClasses;
    annotation (Documentation(info="<HTML>
This package contains models of physical processes and components using water or steam as working fluid.
<p>All models use the <tt>StandardWater</tt> medium model by default, which is in turn set to <tt>Modelica.Media.Water.StandardWater</tt> at the library level. It is of course possible to redeclare the medium model to any model extending <tt>Modelica.Media.Interfaces.PartialMedium</tt> (or <tt>PartialTwoPhaseMedium</tt> for 2-phase models). This can be done by directly setting Medium in the parameter dialog, or through a local package definition, as shown e.g. in <tt>Test.TestMixerSlowFast</tt>. The latter solution allows to easily replace the medium model for an entire set of components.
<p>All models with dynamic equations provide initialisation support. Set the <tt>initOpt</tt> parameter to the appropriate value:
<ul>
<li><tt>Choices.Init.Options.noInit</tt>: no initialisation
<li><tt>Choices.Init.Options.steadyState</tt>: full steady-state initialisation
<li><tt>Choices.Init.Options.steadyStateNoP</tt>: steady-state initialisation (except pressure)
<li><tt>Choices.Init.Options.steadyStateNoT</tt>: steady-state initialisation (except temperature)
</ul>
The latter options can be useful when two or more components are connected directly so that they will have the same pressure or temperature, to avoid over-specified systems of initial equations.
</HTML>"));
  end Water;

  package Units "Types with custom units"
    extends Modelica.Icons.Package;

    type LiquidDensity = SI.Density (start=1000, nominal = 1000)
    "start value for liquids";
  end Units;
annotation (
  Documentation(info="<html>
<h2>General Information</h2>
<p>The ThermoPower library is an open-source <a href=\"http://www.modelica.org/libraries\">Modelica library</a> for the dynamic modelling of thermal power plants and energy conversion systems. It provides basic components for system-level modelling, in particular for the study of control systems in traditional and innovative power plants and energy conversion systems.</p>
<p>The libray has been under continuous development at Politecnico di Milano since 2002. It has been applied to the dynamic modelling of steam generators, combined-cycle power plants, III- and IV-generation nuclear power plants, direct steam generation solar plants, organic Rankine cycle plants, and cryogenic circuits for nuclear fusion applications. The main author is Francesco Casella, with contributions from Alberto Leva, Matilde Ratti, Luca Savoldelli, Roberto Bonifetto, Stefano Boni, Leonardo Pierobon, and many others. The library is licensed under the <b><a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">Modelica License 2</a></b>. The library has been developed as a tool for research in the field of energy system control at the Dipartimento di Elettronica, Informazione e Bioingegneria of Politecnico di Milano and progressively enhanced as new projects were undertaken there. It has been released as open source for the benefit of the community, but without any guarantee of support or completeness of documentation.</p>
<p>The latest released version is 3.1 Beta 0. which uses Modelica 3.2 revision 2 and Modelica Standard Library 3.2.1. If you have used the development version of ThermoPower since 2011 to develop your models, then they should run with version 3.1 of the library with little or no modification.</p>
<p>At some point in the future, changes might be introduced to improve the handling of initial conditions, that could break your models developed with ThermoPower 3.1. These changes will be incorporated in version 3.2.</p>
<p>The library has been mainly developed using the tool <a href=\"http://www.dynasim.se\">Dymola</a>, but it is designed to also run with any other tool that fully supports Modelica 3.2 revision 2 or later. The current coverage of the library by the latest nightly build of the <a href=\"https://openmodelica.org\">OpenModelica</a> compiler is reported <a href=\"https://test.openmodelica.org/libraries/ThermoPower/BuildModelRecursive.html\"> here</a>.</p>
<p>You can download the released versions from the <a href=\"https://github.com/modelica-3rdparty/ThermoPower\">GitHub mirror</a>. The current development version of the source code can be checked out anonymously using an SVN client using this URL:<br><br><a href=\"svn://svn.code.sf.net/p/thermopower/svn/trunk\">http://svn.code.sf.net/p/thermopower/svn/trunk</a><br><br>If you are running Windows, we recommend using the excellent <a href=\"http://tortoisesvn.net/\">TortoiseSVN</a> to do so.</p>
<p>Please note that since 2013 the structure of the Flow1D models, which are the backbone of heat exchanger models, has been revised for greater flexibility and ease of use. New thermal ports are used and the heat transfer model is embedded inside the Flow1D model as a replaceable model. Old Flow1D models (with their thermal counterparts) have been kept in the library for backwards compatibility, but they are deprecated and should not be used to build new models. They are identified by an obsolete marker on the icon. Also the old source and sink components using the deprecated cardinality operator are kept for backwards compatibility, but have been replaced by new components using conditional input connectors.</p>
<p>If you want to get involved in the development, or you need some further information, please contact the main developer <a href=\"mailto://francesco.casella@polimi.it\">francesco.casella@polimi.it</a>.</p>
<h2>References</h2>
<p>A general description of the library and on the modelling principles can be found in the papers: </p>
<p><ul>
<li>F. Casella, A. Leva, &QUOT;Modelling of distributed thermo-hydraulic processes using Modelica&QUOT;, <i>Proceedings of the MathMod &apos;03 Conference</i>, Wien , Austria, February 2003. </li>
<li>F. Casella, A. Leva, &QUOT;Modelica open library for power plant simulation: design and experimental validation&QUOT;, <i>Proceedings of the 2003 Modelica Conference</i>, Link&ouml;ping, Sweden, November 2003, pp. 41-50. (<a href=\"http://www.modelica.org/Conference2003/papers/h08_Leva.pdf\">Available online</a>) </li>
<li>F. Casella, A. Leva, &QUOT;Simulazione di impianti termoidraulici con strumenti object-oriented&QUOT;, <i>Atti convegno ANIPLA Enersis 2004,</i> Milano, Italy, April 2004 (in Italian). </li>
<li>F. Casella, A. Leva, &QUOT;Object-oriented library for thermal power plant simulation&QUOT;, <i>Proceedings of the Eurosis Industrial Simulation Conference 2004 (ISC-2004)</i>, Malaga, Spain, June 2004. </li>
<li>F. Casella, A. Leva, &QUOT;Simulazione object-oriented di impianti di generazione termoidraulici per studi di sistema&QUOT;, <i>Atti convegno nazionale ANIPLA 2004</i>, Milano, Italy, September 2004 (in Italian).</li>
<li>Francesco Casella and Alberto Leva, &ldquo;Modelling of Thermo-Hydraulic Power Generation Processes Using Modelica&rdquo;. <i>Mathematical and Computer Modeling of Dynamical Systems</i>, vol. 12, n. 1, pp. 19-33, Feb. 2006. <a href=\"http://dx.doi.org/10.1080/13873950500071082\">Online</a>. </li>
<li>Francesco Casella, J. G. van Putten and Piero Colonna, &ldquo;Dynamic Simulation of a Biomass-Fired Power Plant: a Comparison Between Causal and A-Causal Modular Modeling&rdquo;. In <i>Proceedings of 2007 ASME International Mechanical Engineering Congress and Exposition</i>, Seattle, Washington, USA, Nov. 11-15, 2007, paper IMECE2007-41091 (Best paper award). </li>
</ul></p>
<p><br/>Other papers about the library and its applications:</p>
<p><ul>
<li>F. Casella, F. Schiavo, &QUOT;Modelling and Simulation of Heat Exchangers in Modelica with Finite Element Methods&QUOT;, <i>Proceedings of the 2003 Modelica Conference</i>, Link&ouml;ping, Sweden, 2003, pp. 343-352. (<a href=\"http://www.modelica.org/Conference2003/papers/h22_Schiavo.pdf\">Available online</a>) </li>
<li>A. Cammi, M.E. Ricotti, F. Casella, F. Schiavo, &QUOT;New modelling strategy for IRIS dynamic response simulation&QUOT;, <i>Proc. 5th International Conference on Nuclear Option in Countries with Small and Medium Electricity Grids</i>, Dubrovnik, Croatia, May 2004.</li>
<li>A. Cammi, F. Casella, M.E. Ricotti, F. Schiavo, &QUOT;Object-oriented Modelling for Integral Nuclear Reactors Dynamic Dimulation&QUOT;, <i>Proceedings of the International Conference on Integrated Modeling &AMP; Analysis in Applied Control &AMP; Automation</i>, Genova, Italy, October 2004. </li>
<li>Antonio Cammi, Francesco Casella, Marco Ricotti and Francesco Schiavo, &ldquo;Object-Oriented Modeling, Simulation and Control of the IRIS Nuclear Power Plant with Modelica&rdquo;. In <i>Proceedings 4th International Modelica Conference</i>, Hamburg, Germany,Mar. 7-8, 2005, pp. 423-432. <a href=\"http://www.modelica.org/events/Conference2005/online_proceedings/Session5/Session5b3.pdf\">Online</a>. </li>
<li>A. Cammi, F. Casella, M. E. Ricotti, F. Schiavo, G. D. Storrick, &QUOT;Object-oriented Simulation for the Control of the IRIS Nuclear Power Plant&QUOT;, <i>Proceedings of the IFAC World Congress, </i>Prague, Czech Republic, July 2005 </li>
<li>Francesco Casella and Francesco Pretolani, &ldquo;Fast Start-up of a Combined-Cycle Power Plant: a Simulation Study with Modelica&rdquo;. In <i>Proceedings 5th International Modelica Conference</i>, Vienna, Austria, Sep. 6-8, 2006, pp. 3-10. <a href=\"http://www.modelica.org/events/modelica2006/Proceedings/sessions/Session1a1.pdf\">Online</a>. </li>
<li>Francesco Casella, &ldquo;Object-Oriented Modelling of Two-Phase Fluid Flows by the Finite Volume Method&rdquo;. In <i>Proceedings 5th Mathmod Vienna</i>, Vienna, Austria, Feb. 8-10, 2006. </li>
<li>Andrea Bartolini, Francesco Casella, Alberto Leva and Valeria Motterle, &ldquo;A Simulation Study of the Flue Gas Path Control System in a Coal-Fired Power Plant&rdquo;. In <i>Proceedings ANIPLA International Congress 2006</i>, Rome, Italy,vNov. 13-15, 2006. </li>
<li>Francesco Schiavo and Francesco Casella, &ldquo;Object-oriented modelling and simulation of heat exchangers with finite element methods&rdquo;. <i>Mathematical and Computer Modeling of Dynamical Sytems</i>, vol. 13, n. 3, pp. 211-235, Jun. 2007. <a href=\"http://dx.doi.org/10.1080/13873950600821766\">Online</a>. </li>
<li>Laura Savoldi Richard, Francesco Casella, Barbara Fiori and Roberto Zanino, &ldquo;Development of the Cryogenic Circuit Conductor and Coil (4C) Code for thermal-hydraulic modelling of ITER superconducting coils&rdquo;. In <i>Presented at the 22nd International Cryogenic Engineering Conference ICEC22</i>, Seoul, Korea, July 21-25, 2008. </li>
<li>Francesco Casella, &ldquo;Object-Oriented Modelling of Power Plants: a Structured Approach&rdquo;. In <i>Proceedings of the IFAC Symposium on Power Plants and Power Systems Control</i>, Tampere, Finland, July 5-8, 2009. </li>
<li>Laura Savoldi Richard, Francesco Casella, Barbara Fiori and Roberto Zanino, &ldquo;The 4C code for the cryogenic circuit conductor and coil modeling in ITER&rdquo;. <i>Cryogenics</i>, vol. 50, n. 3, pp. 167-176, Mar 2010. <a href=\"http://dx.doi.org/10.1016/j.cryogenics.2009.07.008\">Online</a>. </li>
<li>Antonio Cammi, Francesco Casella, Marco Enrico Ricotti and Francesco Schiavo, &ldquo;An object-oriented approach to simulation of IRIS dynamic response&rdquo;. <i>Progress in Nuclear Energy</i>, vol. 53, n. 1, pp. 48-58, Jan. 2011. <a href=\"http://dx.doi.org/10.1016/j.pnucene.2010.09.004\">Online</a>. </li>
<li>Francesco Casella and Piero Colonna, &ldquo;Development of a Modelica dynamic model of solar supercritical CO2 Brayton cycle power plants for control studies&rdquo;. In <i>Proceedings of the Supercritical CO2 Power Cycle Symposium</i>, Boulder, Colorado, USA, May 24-25, 2011, pp. 1-7. <a href=\"http://www.sco2powercyclesymposium.org/resource_center/system_modeling_control\">Online</a>. </li>
<li>Roberto Bonifetto, Francesco Casella, Laura Savoldi Richard and Roberto Zanino, &ldquo;Dynamic modeling of a SHe closed loop with the 4C code&rdquo;. In <i>Transactions of the Cryogenic Engineering Conference - CEC: Advances in Cryogenic Engineering</i>, Spokane, Washington, USA, Jun. 13-17, 2011, pp. 1-8. </li>
<li>Roberto Zanino, Roberto Bonifetto, Francesco Casella and Laura Savoldi Richard, &ldquo;Validation of the 4C code against data from the HELIOS loop at CEA Grenoble&rdquo;. <i>Cryogenics</i>, vol. 0, pp. 1-6, 2012. In press; available online 6 May 2012. <a href=\"http://dx.doi.org/10.1016/j.cryogenics.2012.04.010\">Online</a>. </li>
<li>Francesco Casella and Piero Colonna, &ldquo;Dynamic modelling of IGCC power plants&rdquo;. <i>Applied Thermal Engineering</i>, vol. 35, pp. 91-111, 2012. <a href=\"http://dx.doi.org/10.1016/j.applthermaleng.2011.10.011\">Online</a>. </li>
</ul></p>

<h2>Release notes:</h2>
<p><b>Version 3.1</b></p>
<p>
This is a major new release, that has been in the making for 5 years. The new release is not compatible with 2.1. However, models built using the development version of the library after 2011 should compile with little or no adjustments. It has many new features:
<ul>
  <li>Use of Modelica 3.2 revision 2 and Modelica Standard Library 3.2.1, ensuring full compatibility with all compliant Modelica tools</li>
  <li>Tested with Dymola and OpenModelica./li>
  <li>Use of stream connectors, compatible with the Modelica.Fluid library, allowing multiple-way connections (see <a href=\"http://dx.doi.org/10.3384/ecp09430078\">paper</a>).</li>
  <li>Use of the homotopy operator for improved convegence of steady-state initialization problems(see <a href=\"https://www.modelica.org/events/modelica2011/Proceedings/pages/papers/04_2_ID_131_a_fv.pdf\">paper</a>).</li>
  <li>Improved Flow1D models with embedded replaceable heat transfer models, allowing a much easier customization of heat transfer correlations</li>
  <li>Many bug fixes</li>
</ul></p>

<p><b>Version 2.1 (<i>6 Jul 2009</i>)</b></p>
<p>The 2.1 release of ThermoPower contains several additions and a few bug fixes with respect to version 2.0. We tried to keep the new version backwards-compatible with the old one, but there might be a few cases where small adaptations could be required.</p><p>ThermoPower 2.1 requires the Modelica Standard Library version 2.2.1 or 2.2.2. It has been tested with Dymola 6.1 (using MSL 2.2.1) and with Dymola 7.1 (using MSL 2.2.2). It is planned to be usable also with other tools, in particular OpenModelica, MathModelica and SimulationX, but this is not possible with the currently released versions of those tools. It is expected that this should become at least partially possible within the year 2009. </p><p>ThermoPower 2.1 is the last major revision compatible with Modelica 2.1 and the Modelica Standard Library 2.2.x. The next version is planned to use Modelica 3.1 and the Modelica Standard Library 3.1. It will use use stream connectors, which generalize the concept of Flange connectors, lifting the restrictions that only two complementary connectors can be bound.</p>
<p>This is a list of the main changes with respect to v. 2.0</p>
<p><ul>
<li>New PowerPlants package, containing a library of high-level reusable components for the modelling of combined-cycle power plants, including full models that can be simulated. </li>
<li>New examples cases in the Examples package. </li>
<li>New components in the Electrical package, to model the generator-grid connection by the swing equation </li>
<li>Three-way junctions (FlowJoin and FlowSplit) now have an option to describe unidirectional flow at each flange. This feature can substantially enhance numerical robustness and simulation performance in all cases when it is known a priori that no flow reversal will occur. </li>
<li>The Flow1D and Flow1D2ph models are now restricted to positive flow direction, since it was found that it is not possible to describe flow reversal consistently with the average-density approach adopted in this library. For full flow reversal support please use the Flow1Dfem model, which does not have any restriction in this respect. </li>
<li>A bug in Flow1D and Flow1D2ph has been corrected, which caused significant errors in the mass balance under dynamic conditions; this was potentially critical in closed-loop models, but has now been resolved.&nbsp; </li>
<li>The connectors for lumped- and distribute-parameters heat transfer with variable heat transfer coefficients have been split: HThtc and DHThtc now have an output qualifier on the h.t.c., while HThtc_in and DHThtc_in have an input qualifier. This was necessary to avoid incorrect connections, and is also required by tools to correctly checked if a model is balanced. This change should have no impact on most user-developed models. </li>
</ul></p>
<p><b>Version 2.0 (<i>10 Jun 2005</i>)</b></p>
<p><ul>
<li>The new Modelica 2.2 standard library is used. </li>
<li>The ThermoPower library is now based on the Modelica.Media standard library for fluid property calculations. All the component models use a Modelica.Media compliant interface to specify the medium model. Standard water and gas models from the Modelica.Media library can be used, as well as custom-built water and gas models, compliant with the Modelica.Media interfaces. </li>
<li>Fully functional gas components are now available, including model for gas compressors and turbines, as well as compact gas turbine unit models. </li>
<li>Steady-state initialisation support has been added to all dynamic models. </li>
<li>Some components are still under development, and could be changed in the final 2.0 release: </li>
<li>Moving boundary model for two-phase flow in once-through evaporators. </li>
<li>Stress models for headers and turbines. </li>
</ul></p>
<p><b>Version 1.2 (<i>18 Nov 2004</i>)</b></p>
<p><ul>
<li>Valve and pump models restructured using inheritance. </li>
<li>Simple model of a steam turbine unit added (requires the Modelica.Media library). </li>
<li>CISE example restructured and moved to the <code>Examples</code> package. </li>
<li>Preliminary version of gas components added in the <code>Gas</code> package. </li>
<li>Finite element model of thermohydraulic two-phase flow added. </li>
<li>Simplified models for the connection to the power system added in the <code>Electrical</code> package. </li>
</ul></p>
<p><b>Version 1.1 (<i>15 Feb 2004</i>)</b></p>
<p><ul>
<li>No default values for parameters whose values must be set explicitly by the user. </li>
<li>Description of the meaning of the model variables added. </li>
<li><code>Pump</code>, <code>PumpMech</code>, <code>Accumulator</code> models added. </li>
<li>More rational geometric parameters for <code>Flow1D*</code> models. </li>
<li><code>Flow1D</code> model corrected to avoid numerical problems when the phase transition boundaries cross the nodes. </li>
<li><code>Flow1D2phDB</code> model updated. </li>
<li><code>Flow1D2phChen</code> models with two-phase heat transfer added. </li>
</ul></p>
<p><b>Version 1.0 (<i>20 Oct 2003</i>)</b></p>
<p><ul>
<li>First release in the public domain</li>
</ul></p>
<p><b></font><font style=\"font-size: 12pt; \">License agreement</b></p>
<p>The ThermoPower package is licensed by Politecnico di Milano under the <b><a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">Modelica License 2</a></b>. </p>
<p><h4>Copyright &copy; 2002-2014, Politecnico di Milano.</h4></p>
</html>"),
  uses(Modelica(version="3.2.1")),
  version="3.1");
end ThermoPower;

package Modelica "Modelica Standard Library - Version 3.2.1 (Build 2)"
extends Modelica.Icons.Package;

  package Blocks
  "Library of basic input/output control blocks (continuous, discrete, logical, table blocks)"
  import SI = Modelica.SIunits;
  extends Modelica.Icons.Package;

    package Interfaces
    "Library of connectors and partial models for input/output blocks"
      import Modelica.SIunits;
      extends Modelica.Icons.InterfacesPackage;

      connector RealInput = input Real "'input Real' as connector" annotation (
        defaultComponentName="u",
        Icon(graphics={
          Polygon(
            lineColor={0,0,127},
            fillColor={0,0,127},
            fillPattern=FillPattern.Solid,
            points={{-100.0,100.0},{100.0,0.0},{-100.0,-100.0}})},
          coordinateSystem(extent={{-100.0,-100.0},{100.0,100.0}},
            preserveAspectRatio=true,
            initialScale=0.2)),
        Diagram(
          coordinateSystem(preserveAspectRatio=true,
            initialScale=0.2,
            extent={{-100.0,-100.0},{100.0,100.0}}),
            graphics={
          Polygon(
            lineColor={0,0,127},
            fillColor={0,0,127},
            fillPattern=FillPattern.Solid,
            points={{0.0,50.0},{100.0,0.0},{0.0,-50.0},{0.0,50.0}}),
          Text(
            lineColor={0,0,127},
            extent={{-10.0,60.0},{-10.0,85.0}},
            textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one input signal of type Real.
</p>
</html>"));

      connector RealOutput = output Real "'output Real' as connector" annotation (
        defaultComponentName="y",
        Icon(
          coordinateSystem(preserveAspectRatio=true,
            extent={{-100.0,-100.0},{100.0,100.0}},
            initialScale=0.1),
            graphics={
          Polygon(
            lineColor={0,0,127},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid,
            points={{-100.0,100.0},{100.0,0.0},{-100.0,-100.0}})}),
        Diagram(
          coordinateSystem(preserveAspectRatio=true,
            extent={{-100.0,-100.0},{100.0,100.0}},
            initialScale=0.1),
            graphics={
          Polygon(
            lineColor={0,0,127},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid,
            points={{-100.0,50.0},{0.0,0.0},{-100.0,-50.0}}),
          Text(
            lineColor={0,0,127},
            extent={{30.0,60.0},{30.0,110.0}},
            textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one output signal of type Real.
</p>
</html>"));

      connector IntegerInput = input Integer "'input Integer' as connector"
        annotation (
        defaultComponentName="u",
        Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={255,127,0},
              fillColor={255,127,0},
              fillPattern=FillPattern.Solid)}, coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=true,
            initialScale=0.2)),
        Diagram(coordinateSystem(
            preserveAspectRatio=true,
            initialScale=0.2,
            extent={{-100,-100},{100,100}}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={255,127,0},
              fillColor={255,127,0},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={255,127,0},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one input signal of type Integer.
</p>
</html>"));

      partial block SO "Single Output continuous control block"
        extends Modelica.Blocks.Icons.Block;

        RealOutput y "Connector of Real output signal" annotation (Placement(
              transformation(extent={{100,-10},{120,10}}, rotation=0)));
        annotation (Documentation(info="<html>
<p>
Block has one continuous Real output signal.
</p>
</html>"));

      end SO;
      annotation (Documentation(info="<HTML>
<p>
This package contains interface definitions for
<b>continuous</b> input/output blocks with Real,
Integer and Boolean signals. Furthermore, it contains
partial models for continuous and discrete blocks.
</p>

</html>",     revisions="<html>
<ul>
<li><i>Oct. 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Added several new interfaces.
<li><i>Oct. 24, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       RealInputSignal renamed to RealInput. RealOutputSignal renamed to
       output RealOutput. GraphBlock renamed to BlockIcon. SISOreal renamed to
       SISO. SOreal renamed to SO. I2SOreal renamed to M2SO.
       SignalGenerator renamed to SignalSource. Introduced the following
       new models: MIMO, MIMOs, SVcontrol, MVcontrol, DiscreteBlockIcon,
       DiscreteBlock, DiscreteSISO, DiscreteMIMO, DiscreteMIMOs,
       BooleanBlockIcon, BooleanSISO, BooleanSignalSource, MI2BooleanMOs.</li>
<li><i>June 30, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized a first version, based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist.</li>
</ul>
</html>"));
    end Interfaces;

    package Math
    "Library of Real mathematical functions as input/output blocks"
      import Modelica.SIunits;
      import Modelica.Blocks.Interfaces;
      extends Modelica.Icons.Package;

          block Gain "Output the product of a gain value with the input signal"

            parameter Real k(start=1, unit="1")
        "Gain value multiplied with input signal";
    public
            Interfaces.RealInput u "Input signal connector"
              annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
            Interfaces.RealOutput y "Output signal connector"
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

          equation
            y = k*u;
            annotation (
              Documentation(info="<html>
<p>
This block computes output <i>y</i> as
<i>product</i> of gain <i>k</i> with the
input <i>u</i>:
</p>
<pre>
    y = k * u;
</pre>

</html>"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics={
              Polygon(
                points={{-100,-100},{-100,100},{100,0},{-100,-100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-140},{150,-100}},
                lineColor={0,0,0},
                textString="k=%k"),
              Text(
                extent={{-150,140},{150,100}},
                textString="%name",
                lineColor={0,0,255})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics={Polygon(
                points={{-100,-100},{-100,100},{100,0},{-100,-100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-76,38},{0,-34}},
                textString="k",
                lineColor={0,0,255})}));
          end Gain;
      annotation (
        Documentation(info="<html>
<p>
This package contains basic <b>mathematical operations</b>,
such as summation and multiplication, and basic <b>mathematical
functions</b>, such as <b>sqrt</b> and <b>sin</b>, as
input/output blocks. All blocks of this library can be either
connected with continuous blocks or with sampled-data blocks.
</p>
</html>",     revisions="<html>
<ul>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       New blocks added: RealToInteger, IntegerToReal, Max, Min, Edge, BooleanChange, IntegerChange.</li>
<li><i>August 7, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized (partly based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist).
</li>
</ul>
</html>"),     Icon(graphics={Line(
              points={{-80,-2},{-68.7,32.2},{-61.5,51.1},{-55.1,64.4},{-49.4,72.6},
                  {-43.8,77.1},{-38.2,77.8},{-32.6,74.6},{-26.9,67.7},{-21.3,57.4},
                  {-14.9,42.1},{-6.83,19.2},{10.1,-32.8},{17.3,-52.2},{23.7,-66.2},
                  {29.3,-75.1},{35,-80.4},{40.6,-82},{46.2,-79.6},{51.9,-73.5},{
                  57.5,-63.9},{63.9,-49.2},{72,-26.8},{80,-2}},
              color={95,95,95},
              smooth=Smooth.Bezier)}));
    end Math;

    package Sources
    "Library of signal source blocks generating Real and Boolean signals"
      import Modelica.Blocks.Interfaces;
      import Modelica.SIunits;
      extends Modelica.Icons.SourcesPackage;

      block Constant "Generate constant signal of type Real"
        parameter Real k(start=1) "Constant output value";
        extends Interfaces.SO;

      equation
        y = k;
        annotation (
          defaultComponentName="const",
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics={
              Line(points={{-80,68},{-80,-80}}, color={192,192,192}),
              Polygon(
                points={{-80,90},{-88,68},{-72,68},{-80,90}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,-70},{82,-70}}, color={192,192,192}),
              Polygon(
                points={{90,-70},{68,-62},{68,-78},{90,-70}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,0},{80,0}}, color={0,0,0}),
              Text(
                extent={{-150,-150},{150,-110}},
                lineColor={0,0,0},
                textString="k=%k")}),
          Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics={
              Polygon(
                points={{-80,90},{-86,68},{-74,68},{-80,90}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,68},{-80,-80}}, color={95,95,95}),
              Line(
                points={{-80,0},{80,0}},
                color={0,0,255},
                thickness=0.5),
              Line(points={{-90,-70},{82,-70}}, color={95,95,95}),
              Polygon(
                points={{90,-70},{68,-64},{68,-76},{90,-70}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-83,92},{-30,74}},
                lineColor={0,0,0},
                textString="y"),
              Text(
                extent={{70,-80},{94,-100}},
                lineColor={0,0,0},
                textString="time"),
              Text(
                extent={{-101,8},{-81,-12}},
                lineColor={0,0,0},
                textString="k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/Blocks/Sources/Constant.png\"
     alt=\"Constant.png\">
</p>
</html>"));
      end Constant;

      block Ramp "Generate ramp signal"
        parameter Real height=1 "Height of ramps";
        parameter Modelica.SIunits.Time duration(min=0.0, start=2)
        "Duration of ramp (= 0.0 gives a Step)";
        parameter Real offset=0 "Offset of output signal";
        parameter Modelica.SIunits.Time startTime=0
        "Output = offset for time < startTime";
        extends Interfaces.SO;

      equation
        y = offset + (if time < startTime then 0 else if time < (startTime +
          duration) then (time - startTime)*height/duration else height);
        annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics={
              Line(points={{-80,68},{-80,-80}}, color={192,192,192}),
              Polygon(
                points={{-80,90},{-88,68},{-72,68},{-80,90}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,-70},{82,-70}}, color={192,192,192}),
              Polygon(
                points={{90,-70},{68,-62},{68,-78},{90,-70}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,-70},{-40,-70},{31,38}}, color={0,0,0}),
              Text(
                extent={{-150,-150},{150,-110}},
                lineColor={0,0,0},
                textString="duration=%duration"),
              Line(points={{31,38},{86,38}}, color={0,0,0})}),
          Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics={
              Polygon(
                points={{-80,90},{-86,68},{-74,68},{-80,90}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,68},{-80,-80}}, color={95,95,95}),
              Line(
                points={{-80,-20},{-20,-20},{50,50}},
                color={0,0,255},
                thickness=0.5),
              Line(points={{-90,-70},{82,-70}}, color={95,95,95}),
              Polygon(
                points={{90,-70},{68,-64},{68,-76},{90,-70}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-40,-20},{-42,-30},{-38,-30},{-40,-20}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Line(
                points={{-40,-20},{-40,-70}},
                color={95,95,95},
                thickness=0.25,
                arrow={Arrow.None,Arrow.None}),
              Polygon(
                points={{-40,-70},{-42,-60},{-38,-60},{-40,-70},{-40,-70}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-72,-39},{-34,-50}},
                lineColor={0,0,0},
                textString="offset"),
              Text(
                extent={{-38,-72},{6,-83}},
                lineColor={0,0,0},
                textString="startTime"),
              Text(
                extent={{-78,92},{-37,72}},
                lineColor={0,0,0},
                textString="y"),
              Text(
                extent={{70,-80},{94,-91}},
                lineColor={0,0,0},
                textString="time"),
              Line(points={{-20,-20},{-20,-70}}, color={95,95,95}),
              Line(
                points={{-19,-20},{50,-20}},
                color={95,95,95},
                thickness=0.25,
                arrow={Arrow.None,Arrow.None}),
              Line(
                points={{50,50},{101,50}},
                color={0,0,255},
                thickness=0.5),
              Line(
                points={{50,50},{50,-20}},
                color={95,95,95},
                thickness=0.25,
                arrow={Arrow.None,Arrow.None}),
              Polygon(
                points={{50,-20},{42,-18},{42,-22},{50,-20}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-20,-20},{-11,-18},{-11,-22},{-20,-20}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{50,50},{48,40},{52,40},{50,50}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{50,-20},{48,-10},{52,-10},{50,-20},{50,-20}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{53,23},{82,10}},
                lineColor={0,0,0},
                textString="height"),
              Text(
                extent={{-2,-21},{37,-33}},
                lineColor={0,0,0},
                textString="duration")}),
          Documentation(info="<html>
<p>
The Real output y is a ramp signal:
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/Blocks/Sources/Ramp.png\"
     alt=\"Ramp.png\">
</p>

<p>
If parameter duration is set to 0.0, the limiting case of a Step signal is achieved.
</p>
</html>"));
      end Ramp;
      annotation (Documentation(info="<HTML>
<p>
This package contains <b>source</b> components, i.e., blocks which
have only output signals. These blocks are used as signal generators
for Real, Integer and Boolean signals.
</p>

<p>
All Real source signals (with the exception of the Constant source)
have at least the following two parameters:
</p>

<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>offset</b></td>
      <td valign=\"top\">Value which is added to the signal</td>
  </tr>
  <tr><td valign=\"top\"><b>startTime</b></td>
      <td valign=\"top\">Start time of signal. For time &lt; startTime,
                the output y is set to offset.</td>
  </tr>
</table>

<p>
The <b>offset</b> parameter is especially useful in order to shift
the corresponding source, such that at initial time the system
is stationary. To determine the corresponding value of offset,
usually requires a trimming calculation.
</p>
</html>",     revisions="<html>
<ul>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Integer sources added. Step, TimeTable and BooleanStep slightly changed.</li>
<li><i>Nov. 8, 1999</i>
       by <a href=\"mailto:clauss@eas.iis.fhg.de\">Christoph Clau&szlig;</a>,
       <a href=\"mailto:Andre.Schneider@eas.iis.fraunhofer.de\">Andre.Schneider@eas.iis.fraunhofer.de</a>,
       <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       New sources: Exponentials, TimeTable. Trapezoid slightly enhanced
       (nperiod=-1 is an infinite number of periods).</li>
<li><i>Oct. 31, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       <a href=\"mailto:clauss@eas.iis.fhg.de\">Christoph Clau&szlig;</a>,
       <a href=\"mailto:Andre.Schneider@eas.iis.fraunhofer.de\">Andre.Schneider@eas.iis.fraunhofer.de</a>,
       All sources vectorized. New sources: ExpSine, Trapezoid,
       BooleanConstant, BooleanStep, BooleanPulse, SampleTrigger.
       Improved documentation, especially detailed description of
       signals in diagram layer.</li>
<li><i>June 29, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized a first version, based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist.</li>
</ul>
</html>"));
    end Sources;

    package Icons "Icons for Blocks"
        extends Modelica.Icons.IconsPackage;

        partial block Block "Basic graphical layout of input/output block"

          annotation (
            Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255})}),
          Documentation(info="<html>
<p>
Block that has only the basic icon for an input/output
block (no declarations, no equations). Most blocks
of package Modelica.Blocks inherit directly or indirectly
from this block.
</p>
</html>"));

        end Block;
    end Icons;
  annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100.0,-100.0},{100.0,100.0}}, initialScale=0.1), graphics={
        Rectangle(
          origin={0.0,35.1488},
          fillColor={255,255,255},
          extent={{-30.0,-20.1488},{30.0,20.1488}}),
        Rectangle(
          origin={0.0,-34.8512},
          fillColor={255,255,255},
          extent={{-30.0,-20.1488},{30.0,20.1488}}),
        Line(
          origin={-51.25,0.0},
          points={{21.25,-35.0},{-13.75,-35.0},{-13.75,35.0},{6.25,35.0}}),
        Polygon(
          origin={-40.0,35.0},
          pattern=LinePattern.None,
          fillPattern=FillPattern.Solid,
          points={{10.0,0.0},{-5.0,5.0},{-5.0,-5.0}}),
        Line(
          origin={51.25,0.0},
          points={{-21.25,35.0},{13.75,35.0},{13.75,-35.0},{-6.25,-35.0}}),
        Polygon(
          origin={40.0,-35.0},
          pattern=LinePattern.None,
          fillPattern=FillPattern.Solid,
          points={{-10.0,0.0},{5.0,5.0},{5.0,-5.0}})}), Documentation(info="<html>
<p>
This library contains input/output blocks to build up block diagrams.
</p>

<dl>
<dt><b>Main Author:</b>
<dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
    Deutsches Zentrum f&uuml;r Luft und Raumfahrt e. V. (DLR)<br>
    Oberpfaffenhofen<br>
    Postfach 1116<br>
    D-82230 Wessling<br>
    email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
</dl>
<p>
Copyright &copy; 1998-2013, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>
</html>",   revisions="<html>
<ul>
<li><i>June 23, 2004</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Introduced new block connectors and adapted all blocks to the new connectors.
       Included subpackages Continuous, Discrete, Logical, Nonlinear from
       package ModelicaAdditions.Blocks.
       Included subpackage ModelicaAdditions.Table in Modelica.Blocks.Sources
       and in the new package Modelica.Blocks.Tables.
       Added new blocks to Blocks.Sources and Blocks.Logical.
       </li>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       New subpackage Examples, additional components.
       </li>
<li><i>June 20, 2000</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a> and
       Michael Tiller:<br>
       Introduced a replaceable signal type into
       Blocks.Interfaces.RealInput/RealOutput:
<pre>
   replaceable type SignalType = Real
</pre>
       in order that the type of the signal of an input/output block
       can be changed to a physical type, for example:
<pre>
   Sine sin1(outPort(redeclare type SignalType=Modelica.SIunits.Torque))
</pre>
      </li>
<li><i>Sept. 18, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Renamed to Blocks. New subpackages Math, Nonlinear.
       Additional components in subpackages Interfaces, Continuous
       and Sources. </li>
<li><i>June 30, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized a first version, based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist.</li>
</ul>
</html>"));
  end Blocks;

  package Fluid
  "Library of 1-dim. thermo-fluid flow models using the Modelica.Media media description"
    extends Modelica.Icons.Package;
  import SI = Modelica.SIunits;
  import Cv = Modelica.SIunits.Conversions;

    model System
    "System properties and default values (ambient, flow direction, initialization)"

      package Medium = Modelica.Media.Interfaces.PartialMedium
      "Medium model for default start values"
          annotation (choicesAllMatching = true);
      parameter Modelica.SIunits.AbsolutePressure p_ambient=101325
      "Default ambient pressure"
        annotation(Dialog(group="Environment"));
      parameter Modelica.SIunits.Temperature T_ambient=293.15
      "Default ambient temperature"
        annotation(Dialog(group="Environment"));
      parameter Modelica.SIunits.Acceleration g=Modelica.Constants.g_n
      "Constant gravity acceleration"
        annotation(Dialog(group="Environment"));

      // Assumptions
      parameter Boolean allowFlowReversal = true
      "= false to restrict to design flow direction (port_a -> port_b)"
        annotation(Dialog(tab="Assumptions"), Evaluate=true);
      parameter Modelica.Fluid.Types.Dynamics energyDynamics=
        Modelica.Fluid.Types.Dynamics.DynamicFreeInitial
      "Default formulation of energy balances"
        annotation(Evaluate=true, Dialog(tab = "Assumptions", group="Dynamics"));
      parameter Modelica.Fluid.Types.Dynamics massDynamics=
        energyDynamics "Default formulation of mass balances"
        annotation(Evaluate=true, Dialog(tab = "Assumptions", group="Dynamics"));
      final parameter Modelica.Fluid.Types.Dynamics substanceDynamics=
        massDynamics "Default formulation of substance balances"
        annotation(Evaluate=true, Dialog(tab = "Assumptions", group="Dynamics"));
      final parameter Modelica.Fluid.Types.Dynamics traceDynamics=
        massDynamics "Default formulation of trace substance balances"
        annotation(Evaluate=true, Dialog(tab = "Assumptions", group="Dynamics"));
      parameter Modelica.Fluid.Types.Dynamics momentumDynamics=
        Modelica.Fluid.Types.Dynamics.SteadyState
      "Default formulation of momentum balances, if options available"
        annotation(Evaluate=true, Dialog(tab = "Assumptions", group="Dynamics"));

      // Initialization
      parameter Modelica.SIunits.MassFlowRate m_flow_start = 0
      "Default start value for mass flow rates"
        annotation(Dialog(tab = "Initialization"));
      parameter Modelica.SIunits.AbsolutePressure p_start = p_ambient
      "Default start value for pressures"
        annotation(Dialog(tab = "Initialization"));
      parameter Modelica.SIunits.Temperature T_start = T_ambient
      "Default start value for temperatures"
        annotation(Dialog(tab = "Initialization"));
      // Advanced
      parameter Boolean use_eps_Re = false
      "= true to determine turbulent region automatically using Reynolds number"
        annotation(Evaluate=true, Dialog(tab = "Advanced"));
      parameter Modelica.SIunits.MassFlowRate m_flow_nominal = if use_eps_Re then 1 else 1e2*m_flow_small
      "Default nominal mass flow rate"
        annotation(Dialog(tab="Advanced", enable = use_eps_Re));
      parameter Real eps_m_flow(min=0) = 1e-4
      "Regularization of zero flow for |m_flow| < eps_m_flow*m_flow_nominal"
        annotation(Dialog(tab = "Advanced", enable = use_eps_Re));
      parameter Modelica.SIunits.AbsolutePressure dp_small(min=0) = 1
      "Default small pressure drop for regularization of laminar and zero flow"
        annotation(Dialog(tab="Advanced", group="Classic", enable = not use_eps_Re));
      parameter Modelica.SIunits.MassFlowRate m_flow_small(min=0) = 1e-2
      "Default small mass flow rate for regularization of laminar and zero flow"
        annotation(Dialog(tab = "Advanced", group="Classic", enable = not use_eps_Re));
    initial equation
      //assert(use_eps_Re, "*** Using classic system.m_flow_small and system.dp_small."
      //       + " They do not distinguish between laminar flow and regularization of zero flow."
      //       + " Absolute small values are error prone for models with local nominal values."
      //       + " Moreover dp_small can generally be obtained automatically."
      //       + " Please update the model to new system.use_eps_Re = true  (see system, Advanced tab). ***",
      //       level=AssertionLevel.warning);
      annotation (
        defaultComponentName="system",
        defaultComponentPrefixes="inner",
        missingInnerMessage="
Your model is using an outer \"system\" component but
an inner \"system\" component is not defined.
For simulation drag Modelica.Fluid.System into your model
to specify system properties.
",      Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{100,
                100}}), graphics={
            Rectangle(
              extent={{-100,100},{100,-100}},
              lineColor={0,0,255},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{-150,150},{150,110}},
              lineColor={0,0,255},
              textString="%name"),
            Line(points={{-86,-30},{82,-30}}, color={0,0,0}),
            Line(points={{-82,-68},{-52,-30}}, color={0,0,0}),
            Line(points={{-48,-68},{-18,-30}}, color={0,0,0}),
            Line(points={{-14,-68},{16,-30}}, color={0,0,0}),
            Line(points={{22,-68},{52,-30}}, color={0,0,0}),
            Line(points={{74,84},{74,14}}, color={0,0,0}),
            Polygon(
              points={{60,14},{88,14},{74,-18},{60,14}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{16,20},{60,-18}},
              lineColor={0,0,0},
              textString="g"),
            Text(
              extent={{-90,82},{74,50}},
              lineColor={0,0,0},
              textString="defaults"),
            Line(
              points={{-82,14},{-42,-20},{2,30}},
              color={0,0,0},
              thickness=0.5),
            Ellipse(
              extent={{-10,40},{12,18}},
              pattern=LinePattern.None,
              lineColor={0,0,0},
              fillColor={255,0,0},
              fillPattern=FillPattern.Solid)}),
        Documentation(info="<html>
<p>
 A system component is needed in each fluid model to provide system-wide settings, such as ambient conditions and overall modeling assumptions.
 The system settings are propagated to the fluid models using the inner/outer mechanism.
</p>
<p>
 A model should never directly use system parameters.
 Instead a local parameter should be declared, which uses the global setting as default.
 The only exceptions are:</p>
 <ul>
  <li>the gravity system.g,</li>
  <li>the global system.eps_m_flow, which is used to define a local m_flow_small for the local m_flow_nominal:
      <pre>m_flow_small = system.eps_m_flow*m_flow_nominal</pre>
  </li>
 </ul>
<p>
 The global system.m_flow_small and system.dp_small are classic parameters.
 They do not distinguish between laminar flow and regularization of zero flow.
 Absolute small values are error prone for models with local nominal values.
 Moreover dp_small can generally be obtained automatically.
 Consider using the new system.use_eps_Re = true (see Advanced tab).
</p>
</html>"));
    end System;

    package Sources "Define fixed or prescribed boundary conditions"
      extends Modelica.Icons.SourcesPackage;

      model Boundary_pT
      "Boundary with prescribed pressure, temperature, composition and trace substances"
        import Modelica.Media.Interfaces.Choices.IndependentVariables;

        extends Sources.BaseClasses.PartialSource;
        parameter Boolean use_p_in = false
        "Get the pressure from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Boolean use_T_in= false
        "Get the temperature from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Boolean use_X_in = false
        "Get the composition from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Boolean use_C_in = false
        "Get the trace substances from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Medium.AbsolutePressure p = Medium.p_default
        "Fixed value of pressure"
          annotation (Evaluate = true,
                      Dialog(enable = not use_p_in));
        parameter Medium.Temperature T = Medium.T_default
        "Fixed value of temperature"
          annotation (Evaluate = true,
                      Dialog(enable = not use_T_in));
        parameter Medium.MassFraction X[Medium.nX] = Medium.X_default
        "Fixed value of composition"
          annotation (Evaluate = true,
                      Dialog(enable = (not use_X_in) and Medium.nXi > 0));
        parameter Medium.ExtraProperty C[Medium.nC](
             quantity=Medium.extraPropertiesNames)=fill(0, Medium.nC)
        "Fixed values of trace substances"
          annotation (Evaluate=true,
                      Dialog(enable = (not use_C_in) and Medium.nC > 0));
        Modelica.Blocks.Interfaces.RealInput p_in if use_p_in
        "Prescribed boundary pressure"
          annotation (Placement(transformation(extent={{-140,60},{-100,100}},
                rotation=0)));
        Modelica.Blocks.Interfaces.RealInput T_in if use_T_in
        "Prescribed boundary temperature"
          annotation (Placement(transformation(extent={{-140,20},{-100,60}},
                rotation=0)));
        Modelica.Blocks.Interfaces.RealInput X_in[Medium.nX] if use_X_in
        "Prescribed boundary composition"
          annotation (Placement(transformation(extent={{-140,-60},{-100,-20}},
                rotation=0)));
        Modelica.Blocks.Interfaces.RealInput C_in[Medium.nC] if use_C_in
        "Prescribed boundary trace substances"
          annotation (Placement(transformation(extent={{-140,-100},{-100,-60}},
                rotation=0)));
    protected
        Modelica.Blocks.Interfaces.RealInput p_in_internal
        "Needed to connect to conditional connector";
        Modelica.Blocks.Interfaces.RealInput T_in_internal
        "Needed to connect to conditional connector";
        Modelica.Blocks.Interfaces.RealInput X_in_internal[Medium.nX]
        "Needed to connect to conditional connector";
        Modelica.Blocks.Interfaces.RealInput C_in_internal[Medium.nC]
        "Needed to connect to conditional connector";
      equation
        Modelica.Fluid.Utilities.checkBoundary(Medium.mediumName, Medium.substanceNames,
          Medium.singleState, true, X_in_internal, "Boundary_pT");
        connect(p_in, p_in_internal);
        connect(T_in, T_in_internal);
        connect(X_in, X_in_internal);
        connect(C_in, C_in_internal);
        if not use_p_in then
          p_in_internal = p;
        end if;
        if not use_T_in then
          T_in_internal = T;
        end if;
        if not use_X_in then
          X_in_internal = X;
        end if;
        if not use_C_in then
          C_in_internal = C;
        end if;
        medium.p = p_in_internal;
        if Medium.ThermoStates == IndependentVariables.ph or
           Medium.ThermoStates == IndependentVariables.phX then
           medium.h = Medium.specificEnthalpy(Medium.setState_pTX(p_in_internal, T_in_internal, X_in_internal));
        else
           medium.T = T_in_internal;
        end if;
        medium.Xi = X_in_internal[1:Medium.nXi];
        ports.C_outflow = fill(C_in_internal, nPorts);
        annotation (defaultComponentName="boundary",
          Icon(coordinateSystem(
              preserveAspectRatio=false,
              extent={{-100,-100},{100,100}}), graphics={
              Ellipse(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillPattern=FillPattern.Sphere,
                fillColor={0,127,255}),
              Text(
                extent={{-150,120},{150,160}},
                textString="%name",
                lineColor={0,0,255}),
              Line(
                visible=use_p_in,
                points={{-100,80},{-58,80}},
                color={0,0,255}),
              Line(
                visible=use_T_in,
                points={{-100,40},{-92,40}},
                color={0,0,255}),
              Line(
                visible=use_X_in,
                points={{-100,-40},{-92,-40}},
                color={0,0,255}),
              Line(
                visible=use_C_in,
                points={{-100,-80},{-60,-80}},
                color={0,0,255}),
              Text(
                visible=use_p_in,
                extent={{-152,134},{-68,94}},
                lineColor={0,0,0},
                textString="p"),
              Text(
                visible=use_X_in,
                extent={{-164,4},{-62,-36}},
                lineColor={0,0,0},
                textString="X"),
              Text(
                visible=use_C_in,
                extent={{-164,-90},{-62,-130}},
                lineColor={0,0,0},
                textString="C"),
              Text(
                visible=use_T_in,
                extent={{-162,34},{-60,-6}},
                lineColor={0,0,0},
                textString="T")}),
          Documentation(info="<html>
<p>
Defines prescribed values for boundary conditions:
</p>
<ul>
<li> Prescribed boundary pressure.</li>
<li> Prescribed boundary temperature.</li>
<li> Boundary composition (only for multi-substance or trace-substance flow).</li>
</ul>
<p>If <code>use_p_in</code> is false (default option), the <code>p</code> parameter
is used as boundary pressure, and the <code>p_in</code> input connector is disabled; if <code>use_p_in</code> is true, then the <code>p</code> parameter is ignored, and the value provided by the input connector is used instead.</p>
<p>The same thing goes for the temperature, composition and trace substances.</p>
<p>
Note, that boundary temperature,
mass fractions and trace substances have only an effect if the mass flow
is from the boundary into the port. If mass is flowing from
the port into the boundary, the boundary definitions,
with exception of boundary pressure, do not have an effect.
</p>
</html>"));
      end Boundary_pT;

      model MassFlowSource_T
      "Ideal flow source that produces a prescribed mass flow with prescribed temperature, mass fraction and trace substances"
        import Modelica.Media.Interfaces.Choices.IndependentVariables;
        extends Sources.BaseClasses.PartialFlowSource;
        parameter Boolean use_m_flow_in = false
        "Get the mass flow rate from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Boolean use_T_in= false
        "Get the temperature from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Boolean use_X_in = false
        "Get the composition from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Boolean use_C_in = false
        "Get the trace substances from the input connector"
          annotation(Evaluate=true, HideResult=true, choices(checkBox=true));
        parameter Medium.MassFlowRate m_flow = 0
        "Fixed mass flow rate going out of the fluid port"
          annotation (Evaluate = true,
                      Dialog(enable = not use_m_flow_in));
        parameter Medium.Temperature T = Medium.T_default
        "Fixed value of temperature"
          annotation (Evaluate = true,
                      Dialog(enable = not use_T_in));
        parameter Medium.MassFraction X[Medium.nX] = Medium.X_default
        "Fixed value of composition"
          annotation (Evaluate = true,
                      Dialog(enable = (not use_X_in) and Medium.nXi > 0));
        parameter Medium.ExtraProperty C[Medium.nC](
             quantity=Medium.extraPropertiesNames)=fill(0, Medium.nC)
        "Fixed values of trace substances"
          annotation (Evaluate=true,
                      Dialog(enable = (not use_C_in) and Medium.nC > 0));
        Modelica.Blocks.Interfaces.RealInput m_flow_in if     use_m_flow_in
        "Prescribed mass flow rate"
          annotation (Placement(transformation(extent={{-120,60},{-80,100}},
                rotation=0), iconTransformation(extent={{-120,60},{-80,100}})));
        Modelica.Blocks.Interfaces.RealInput T_in if         use_T_in
        "Prescribed fluid temperature"
          annotation (Placement(transformation(extent={{-140,20},{-100,60}},
                rotation=0), iconTransformation(extent={{-140,20},{-100,60}})));
        Modelica.Blocks.Interfaces.RealInput X_in[Medium.nX] if
                                                              use_X_in
        "Prescribed fluid composition"
          annotation (Placement(transformation(extent={{-140,-60},{-100,-20}},
                rotation=0)));
        Modelica.Blocks.Interfaces.RealInput C_in[Medium.nC] if
                                                              use_C_in
        "Prescribed boundary trace substances"
          annotation (Placement(transformation(extent={{-120,-100},{-80,-60}},
                rotation=0)));
    protected
        Modelica.Blocks.Interfaces.RealInput m_flow_in_internal
        "Needed to connect to conditional connector";
        Modelica.Blocks.Interfaces.RealInput T_in_internal
        "Needed to connect to conditional connector";
        Modelica.Blocks.Interfaces.RealInput X_in_internal[Medium.nX]
        "Needed to connect to conditional connector";
        Modelica.Blocks.Interfaces.RealInput C_in_internal[Medium.nC]
        "Needed to connect to conditional connector";
      equation
        Utilities.checkBoundary(Medium.mediumName, Medium.substanceNames,
          Medium.singleState, true, X_in_internal, "MassFlowSource_T");
        connect(m_flow_in, m_flow_in_internal);
        connect(T_in, T_in_internal);
        connect(X_in, X_in_internal);
        connect(C_in, C_in_internal);
        if not use_m_flow_in then
          m_flow_in_internal = m_flow;
        end if;
        if not use_T_in then
          T_in_internal = T;
        end if;
        if not use_X_in then
          X_in_internal = X;
        end if;
        if not use_C_in then
          C_in_internal = C;
        end if;
        if Medium.ThermoStates == IndependentVariables.ph or
           Medium.ThermoStates == IndependentVariables.phX then
           medium.h = Medium.specificEnthalpy(Medium.setState_pTX(medium.p, T_in_internal, X_in_internal));
        else
           medium.T = T_in_internal;
        end if;
        sum(ports.m_flow) = -m_flow_in_internal;
        medium.Xi = X_in_internal[1:Medium.nXi];
        ports.C_outflow = fill(C_in_internal, nPorts);
        annotation (defaultComponentName="boundary",
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics={
              Rectangle(
                extent={{35,45},{100,-45}},
                lineColor={0,0,0},
                fillPattern=FillPattern.HorizontalCylinder,
                fillColor={0,127,255}),
              Ellipse(
                extent={{-100,80},{60,-80}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-60,70},{60,0},{-60,-68},{-60,70}},
                lineColor={0,0,255},
                fillColor={0,0,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-54,32},{16,-30}},
                lineColor={255,0,0},
                textString="m"),
              Text(
                extent={{-150,130},{150,170}},
                textString="%name",
                lineColor={0,0,255}),
              Ellipse(
                extent={{-26,30},{-18,22}},
                lineColor={255,0,0},
                fillColor={255,0,0},
                fillPattern=FillPattern.Solid),
              Text(
                visible=use_m_flow_in,
                extent={{-185,132},{-45,100}},
                lineColor={0,0,0},
                textString="m_flow"),
              Text(
                visible=use_T_in,
                extent={{-111,71},{-71,37}},
                lineColor={0,0,0},
                textString="T"),
              Text(
                visible=use_X_in,
                extent={{-153,-44},{-33,-72}},
                lineColor={0,0,0},
                textString="X"),
              Text(
                visible=use_C_in,
                extent={{-155,-98},{-35,-126}},
                lineColor={0,0,0},
                textString="C")}),
          Documentation(info="<html>
<p>
Models an ideal flow source, with prescribed values of flow rate, temperature, composition and trace substances:
</p>
<ul>
<li> Prescribed mass flow rate.</li>
<li> Prescribed temperature.</li>
<li> Boundary composition (only for multi-substance or trace-substance flow).</li>
</ul>
<p>If <code>use_m_flow_in</code> is false (default option), the <code>m_flow</code> parameter
is used as boundary pressure, and the <code>m_flow_in</code> input connector is disabled; if <code>use_m_flow_in</code> is true, then the <code>m_flow</code> parameter is ignored, and the value provided by the input connector is used instead.</p>
<p>The same thing goes for the temperature and composition</p>
<p>
Note, that boundary temperature,
mass fractions and trace substances have only an effect if the mass flow
is from the boundary into the port. If mass is flowing from
the port into the boundary, the boundary definitions,
with exception of boundary flow rate, do not have an effect.
</p>
</html>"));
      end MassFlowSource_T;

      package BaseClasses
      "Base classes used in the Sources package (only of interest to build new component models)"
        extends Modelica.Icons.BasesPackage;

      partial model PartialSource
        "Partial component source with one fluid connector"
          import Modelica.Constants;

        parameter Integer nPorts=0 "Number of ports" annotation(Dialog(connectorSizing=true));

        replaceable package Medium =
            Modelica.Media.Interfaces.PartialMedium
          "Medium model within the source"
           annotation (choicesAllMatching=true);

        Medium.BaseProperties medium "Medium in the source";

        Interfaces.FluidPorts_b ports[nPorts](
                           redeclare each package Medium = Medium,
                           m_flow(each max=if flowDirection==Types.PortFlowDirection.Leaving then 0 else
                                           +Constants.inf,
                                  each min=if flowDirection==Types.PortFlowDirection.Entering then 0 else
                                           -Constants.inf))
          annotation (Placement(transformation(extent={{90,40},{110,-40}})));
      protected
        parameter Types.PortFlowDirection flowDirection=
                         Types.PortFlowDirection.Bidirectional
          "Allowed flow direction"               annotation(Evaluate=true, Dialog(tab="Advanced"));
      equation
        // Only one connection allowed to a port to avoid unwanted ideal mixing
        for i in 1:nPorts loop
          assert(cardinality(ports[i]) <= 1,"
each ports[i] of boundary shall at most be connected to one component.
If two or more connections are present, ideal mixing takes
place with these connections, which is usually not the intention
of the modeller. Increase nPorts to add an additional port.
");

           ports[i].p          = medium.p;
           ports[i].h_outflow  = medium.h;
           ports[i].Xi_outflow = medium.Xi;
        end for;

        annotation (defaultComponentName="boundary", Documentation(info="<html>
<p>
Partial component to model the <b>volume interface</b> of a <b>source</b>
component, such as a mass flow source. The essential
features are:
</p>
<ul>
<li> The pressure in the connection port (= ports.p) is identical to the
     pressure in the volume.</li>
<li> The outflow enthalpy rate (= port.h_outflow) and the composition of the
     substances (= port.Xi_outflow) are identical to the respective values in the volume.</li>
</ul>
</html>"));
      end PartialSource;

      partial model PartialFlowSource
        "Partial component source with one fluid connector"
        import Modelica.Constants;

        parameter Integer nPorts=0 "Number of ports" annotation(Dialog(connectorSizing=true));

        replaceable package Medium =
            Modelica.Media.Interfaces.PartialMedium
          "Medium model within the source"
           annotation (choicesAllMatching=true);

        Medium.BaseProperties medium "Medium in the source";

        Interfaces.FluidPort_b ports[nPorts](
                           redeclare each package Medium = Medium,
                           m_flow(each max=if flowDirection==Types.PortFlowDirection.Leaving then 0 else
                                           +Constants.inf,
                                  each min=if flowDirection==Types.PortFlowDirection.Entering then 0 else
                                           -Constants.inf))
          annotation (Placement(transformation(extent={{90,10},{110,-10}})));
      protected
        parameter Types.PortFlowDirection flowDirection=
                         Types.PortFlowDirection.Bidirectional
          "Allowed flow direction"               annotation(Evaluate=true, Dialog(tab="Advanced"));
      equation
        assert(abs(sum(abs(ports.m_flow)) - max(abs(ports.m_flow))) <= Modelica.Constants.small, "FlowSource only supports one connection with flow");
        // Only one connection allowed to a port to avoid unwanted ideal mixing
        for i in 1:nPorts loop
          assert(cardinality(ports[i]) <= 1,"
each ports[i] of boundary shall at most be connected to one component.
If two or more connections are present, ideal mixing takes
place with these connections, which is usually not the intention
of the modeller. Increase nPorts to add an additional port.
");
           ports[i].p          = medium.p;
           ports[i].h_outflow  = medium.h;
           ports[i].Xi_outflow = medium.Xi;
        end for;

        annotation (defaultComponentName="boundary", Documentation(info="<html>
<p>
Partial component to model the <b>volume interface</b> of a <b>source</b>
component, such as a mass flow source. The essential
features are:
</p>
<ul>
<li> The pressure in the connection port (= ports.p) is identical to the
     pressure in the volume.</li>
<li> The outflow enthalpy rate (= port.h_outflow) and the composition of the
     substances (= port.Xi_outflow) are identical to the respective values in the volume.</li>
</ul>
</html>"));
      end PartialFlowSource;
      end BaseClasses;
      annotation (Documentation(info="<html>
<p>
Package <b>Sources</b> contains generic sources for fluid connectors
to define fixed or prescribed ambient conditions.
</p>
</html>"));
    end Sources;

    package Interfaces
    "Interfaces for steady state and unsteady, mixed-phase, multi-substance, incompressible and compressible flow"
      extends Modelica.Icons.InterfacesPackage;

      connector FluidPort
      "Interface for quasi one-dimensional fluid flow in a piping network (incompressible or compressible, one or more phases, one or more substances)"

        replaceable package Medium = Modelica.Media.Interfaces.PartialMedium
        "Medium model"   annotation (choicesAllMatching=true);

        flow Medium.MassFlowRate m_flow
        "Mass flow rate from the connection point into the component";
        Medium.AbsolutePressure p
        "Thermodynamic pressure in the connection point";
        stream Medium.SpecificEnthalpy h_outflow
        "Specific thermodynamic enthalpy close to the connection point if m_flow < 0";
        stream Medium.MassFraction Xi_outflow[Medium.nXi]
        "Independent mixture mass fractions m_i/m close to the connection point if m_flow < 0";
        stream Medium.ExtraProperty C_outflow[Medium.nC]
        "Properties c_i/m close to the connection point if m_flow < 0";
      end FluidPort;

      connector FluidPort_b "Generic fluid connector at design outlet"
        extends FluidPort;
        annotation (defaultComponentName="port_b",
                    Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics={
              Ellipse(
                extent={{-40,40},{40,-40}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-30,30},{30,-30}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(extent={{-150,110},{150,50}}, textString="%name")}),
             Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{
                  100,100}}), graphics={
              Ellipse(
                extent={{-100,100},{100,-100}},
                lineColor={0,127,255},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-80,80},{80,-80}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
      end FluidPort_b;

      connector FluidPorts_b
      "Fluid connector with outlined, large icon to be used for vectors of FluidPorts (vector dimensions must be added after dragging)"
        extends FluidPort;
        annotation (defaultComponentName="ports_b",
                    Diagram(coordinateSystem(
              preserveAspectRatio=false,
              extent={{-50,-200},{50,200}},
              initialScale=0.2), graphics={
              Text(extent={{-75,130},{75,100}}, textString="%name"),
              Rectangle(
                extent={{-25,100},{25,-100}},
                lineColor={0,0,0}),
              Ellipse(
                extent={{-25,90},{25,40}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-25,25},{25,-25}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-25,-40},{25,-90}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-15,-50},{15,-80}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-15,15},{15,-15}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-15,50},{15,80}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}),
             Icon(coordinateSystem(
              preserveAspectRatio=false,
              extent={{-50,-200},{50,200}},
              initialScale=0.2), graphics={
              Rectangle(
                extent={{-50,200},{50,-200}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-50,180},{50,80}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-50,50},{50,-50}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-50,-80},{50,-180}},
                lineColor={0,0,0},
                fillColor={0,127,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-30,30},{30,-30}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-30,100},{30,160}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-30,-100},{30,-160}},
                lineColor={0,127,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
      end FluidPorts_b;
      annotation (Documentation(info="<html>

</html>",     revisions="<html>
<ul>
<li><i>June 9th, 2008</i>
       by Michael Sielemann: Introduced stream keyword after decision at 57th Design Meeting (Lund).</li>
<li><i>May 30, 2007</i>
       by Christoph Richter: moved everything back to its original position in Modelica.Fluid.</li>
<li><i>Apr. 20, 2007</i>
       by Christoph Richter: moved parts of the original package from Modelica.Fluid
       to the development branch of Modelica 2.2.2.</li>
<li><i>Nov. 2, 2005</i>
       by Francesco Casella: restructured after 45th Design Meeting.</li>
<li><i>Nov. 20-21, 2002</i>
       by Hilding Elmqvist, Mike Tiller, Allan Watson, John Batteh, Chuck Newman,
       Jonas Eborn: Improved at the 32nd Modelica Design Meeting.
<li><i>Nov. 11, 2002</i>
       by Hilding Elmqvist, Martin Otter: improved version.</li>
<li><i>Nov. 6, 2002</i>
       by Hilding Elmqvist: first version.</li>
<li><i>Aug. 11, 2002</i>
       by Martin Otter: Improved according to discussion with Hilding
       Elmqvist and Hubertus Tummescheit.<br>
       The PortVicinity model is manually
       expanded in the base models.<br>
       The Volume used for components is renamed
       PartialComponentVolume.<br>
       A new volume model \"Fluid.Components.PortVolume\"
       introduced that has the medium properties of the port to which it is
       connected.<br>
       Fluid.Interfaces.PartialTwoPortTransport is a component
       for elementary two port transport elements, whereas PartialTwoPort
       is a component for a container component.</li>
</ul>
</html>"));
    end Interfaces;

    package Types "Common types for fluid models"
      extends Modelica.Icons.TypesPackage;

      type Dynamics = enumeration(
        DynamicFreeInitial
          "DynamicFreeInitial -- Dynamic balance, Initial guess value",
        FixedInitial "FixedInitial -- Dynamic balance, Initial value fixed",
        SteadyStateInitial
          "SteadyStateInitial -- Dynamic balance, Steady state initial with guess value",

        SteadyState "SteadyState -- Steady state balance, Initial guess value")
      "Enumeration to define definition of balance equations"
      annotation (Documentation(info="<html>
<p>
Enumeration to define the formulation of balance equations
(to be selected via choices menu):
</p>

<table border=1 cellspacing=0 cellpadding=2>
<tr><th><b>Dynamics.</b></th><th><b>Meaning</b></th></tr>
<tr><td>DynamicFreeInitial</td><td>Dynamic balance, Initial guess value</td></tr>

<tr><td>FixedInitial</td><td>Dynamic balance, Initial value fixed</td></tr>

<tr><td>SteadyStateInitial</td><td>Dynamic balance, Steady state initial with guess value</td></tr>

<tr><td>SteadyState</td><td>Steady state balance, Initial guess value</td></tr>
</table>

<p>
The enumeration \"Dynamics\" is used for the mass, energy and momentum balance equations
respectively. The exact meaning for the three balance equations is stated in the following
tables:
</p>

<table border=1 cellspacing=0 cellpadding=2>
<tr><td colspan=\"3\"><b>Mass balance</b> </td>
<tr><td><b>Dynamics.</b></td>
    <td><b>Balance equation</b></td>
    <td><b>Initial condition</b></td></tr>

<tr><td> DynamicFreeInitial</td>
    <td> no restrictions </td>
    <td> no initial conditions </td></tr>

<tr><td> FixedInitial</td>
    <td> no restrictions </td>
    <td> <b>if</b> Medium.singleState <b>then</b> <br>
         &nbsp;&nbsp;no initial condition<br>
         <b>else</b> p=p_start </td></tr>

<tr><td> SteadyStateInitial</td>
    <td> no restrictions </td>
    <td> <b>if</b> Medium.singleState <b>then</b> <br>
         &nbsp;&nbsp;no initial condition<br>
         <b>else</b> <b>der</b>(p)=0 </td></tr>

<tr><td> SteadyState</td>
    <td> <b>der</b>(m)=0  </td>
    <td> no initial conditions </td></tr>
</table>

&nbsp;<br>

<table border=1 cellspacing=0 cellpadding=2>
<tr><td colspan=\"3\"><b>Energy balance</b> </td>
<tr><td><b>Dynamics.</b></td>
    <td><b>Balance equation</b></td>
    <td><b>Initial condition</b></td></tr>

<tr><td> DynamicFreeInitial</td>
    <td> no restrictions </td>
    <td> no initial conditions </td></tr>

<tr><td> FixedInitial</td>
    <td> no restrictions </td>
    <td> T=T_start or h=h_start </td></tr>

<tr><td> SteadyStateInitial</td>
    <td> no restrictions </td>
    <td> <b>der</b>(T)=0 or <b>der</b>(h)=0 </td></tr>

<tr><td> SteadyState</td>
    <td> <b>der</b>(U)=0  </td>
    <td> no initial conditions </td></tr>
</table>

&nbsp;<br>

<table border=1 cellspacing=0 cellpadding=2>
<tr><td colspan=\"3\"><b>Momentum balance</b> </td>
<tr><td><b>Dynamics.</b></td>
    <td><b>Balance equation</b></td>
    <td><b>Initial condition</b></td></tr>

<tr><td> DynamicFreeInitial</td>
    <td> no restrictions </td>
    <td> no initial conditions </td></tr>

<tr><td> FixedInitial</td>
    <td> no restrictions </td>
    <td> m_flow = m_flow_start </td></tr>

<tr><td> SteadyStateInitial</td>
    <td> no restrictions </td>
    <td> <b>der</b>(m_flow)=0 </td></tr>

<tr><td> SteadyState</td>
    <td> <b>der</b>(m_flow)=0 </td>
    <td> no initial conditions </td></tr>
</table>

<p>
In the tables above, the equations are given for one-substance fluids. For multiple-substance
fluids and for trace substances, equivalent equations hold.
</p>

<p>
Medium.singleState is a medium property and defines whether the medium is only
described by one state (+ the mass fractions in case of a multi-substance fluid). In such
a case one initial condition less must be provided. For example, incompressible
media have Medium.singleState = <b>true</b>.
</p>

</html>"));

      type PortFlowDirection = enumeration(
        Entering "Fluid flow is only entering",
        Leaving "Fluid flow is only leaving",
        Bidirectional "No restrictions on fluid flow (flow reversal possible)")
      "Enumeration to define whether flow reversal is allowed"   annotation (
          Documentation(info="<html>

<p>
Enumeration to define the assumptions on the model for the
direction of fluid flow at a port (to be selected via choices menu):
</p>

<table border=1 cellspacing=0 cellpadding=2>
<tr><th><b>PortFlowDirection.</b></th>
    <th><b>Meaning</b></th></tr>

<tr><td>Entering</td>
    <td>Fluid flow is only entering the port from the outside</td></tr>

<tr><td>Leaving</td>
    <td>Fluid flow is only leaving the port to the outside</td></tr>

<tr><td>Bidirectional</td>
    <td>No restrictions on fluid flow (flow reversal possible)</td></tr>
</table>

<p>
The default is \"PortFlowDirection.Bidirectional\". If you are completely sure that
the flow is only in one direction, then the other settings may
make the simulation of your model faster.
</p>

</html>"));
      annotation (preferredView="info",
                  Documentation(info="<html>

</html>"));
    end Types;

    package Utilities
    "Utility models to construct fluid components (should not be used directly)"
      extends Modelica.Icons.UtilitiesPackage;

      function checkBoundary "Check whether boundary definition is correct"
        extends Modelica.Icons.Function;
        input String mediumName;
        input String substanceNames[:] "Names of substances";
        input Boolean singleState;
        input Boolean define_p;
        input Real X_boundary[:];
        input String modelName = "??? boundary ???";
    protected
        Integer nX = size(X_boundary,1);
        String X_str;
      algorithm
        assert(not singleState or singleState and define_p, "
Wrong value of parameter define_p (= false) in model \""     + modelName + "\":
The selected medium \""     + mediumName + "\" has Medium.singleState=true.
Therefore, an boundary density cannot be defined and
define_p = true is required.
");

        for i in 1:nX loop
          assert(X_boundary[i] >= 0.0, "
Wrong boundary mass fractions in medium \""
      + mediumName + "\" in model \"" + modelName + "\":
The boundary value X_boundary("   + String(i) + ") = " + String(
            X_boundary[i]) + "
is negative. It must be positive.
");
        end for;

        if nX > 0 and abs(sum(X_boundary) - 1.0) > 1.e-10 then
           X_str :="";
           for i in 1:nX loop
              X_str :=X_str + "   X_boundary[" + String(i) + "] = " + String(X_boundary[
              i]) + " \"" + substanceNames[i] + "\"\n";
           end for;
           Modelica.Utilities.Streams.error(
              "The boundary mass fractions in medium \"" + mediumName + "\" in model \"" + modelName + "\"\n" +
              "do not sum up to 1. Instead, sum(X_boundary) = " + String(sum(X_boundary)) + ":\n"
              + X_str);
        end if;
      end checkBoundary;
      annotation (Documentation(info="<html>

</html>"));
    end Utilities;
  annotation (Icon(graphics={
          Polygon(points={{-70,26},{68,-44},{68,26},{2,-10},{-70,-42},{-70,26}},
              lineColor={0,0,0}),
          Line(points={{2,42},{2,-10}}, color={0,0,0}),
          Rectangle(
            extent={{-18,50},{22,42}},
            lineColor={0,0,0},
            fillColor={0,0,0},
            fillPattern=FillPattern.Solid)}), preferredView="info",
    Documentation(info="<html>
<p>
Library <b>Modelica.Fluid</b> is a <b>free</b> Modelica package providing components for
<b>1-dimensional thermo-fluid flow</b> in networks of vessels, pipes, fluid machines, valves and fittings.
A unique feature is that the component equations and the media models
as well as pressure loss and heat transfer correlations are decoupled from each other.
All components are implemented such that they can be used for
media from the Modelica.Media library. This means especially that an
incompressible or compressible medium, a single or a multiple
substance medium with one or more phases might be used.
</p>

<p>
In the next figure, several features of the library are demonstrated with
a simple heating system with a closed flow cycle. By just changing one configuration parameter in the system object the equations are changed between steady-state and dynamic simulation with fixed or steady-state initial conditions.
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/Fluid/UsersGuide/HeatingSystem.png\" border=\"1\"
     alt=\"HeatingSystem.png\">
</p>

<p>
With respect to previous versions, the design
of the connectors has been changed in a non-backward compatible way,
using the recently developed concept
of stream connectors that results in much more reliable simulations
(see also <a href=\"modelica://Modelica/Resources/Documentation/Fluid/Stream-Connectors-Overview-Rationale.pdf\">Stream-Connectors-Overview-Rationale.pdf</a>).
This extension was included in Modelica 3.1.
As of Jan. 2009, the stream concept is supported in Dymola 7.1.
It is recommended to use Dymola 7.2 (available since Feb. 2009), or a later Dymola version,
since this version supports a new annotation to connect very
conveniently to vectors of connectors.
Other tool vendors will support the stream concept as well.
</p>

<p>
The following parts are useful, when newly starting with this library:
</p>
<ul>
<li> <a href=\"modelica://Modelica.Fluid.UsersGuide\">Modelica.Fluid.UsersGuide</a>.</li>
<li> <a href=\"modelica://Modelica.Fluid.UsersGuide.ReleaseNotes\">Modelica.Fluid.UsersGuide.ReleaseNotes</a>
     summarizes the changes of the library releases.</li>
<li> <a href=\"modelica://Modelica.Fluid.Examples\">Modelica.Fluid.Examples</a>
     contains examples that demonstrate the usage of this library.</li>
</ul>
<p>
<b>Licensed by the Modelica Association under the Modelica License 2</b><br>
Copyright &copy; 2002-2013, ABB, DLR, Dassault Syst&egrave;mes AB, Modelon, TU Braunschweig, TU Hamburg-Harburg, Politecnico di Milano.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>
</html>"));
  end Fluid;

  package Media "Library of media property models"
  extends Modelica.Icons.Package;
  import SI = Modelica.SIunits;
  import Cv = Modelica.SIunits.Conversions;

  package Interfaces "Interfaces for media models"
    extends Modelica.Icons.InterfacesPackage;

    partial package PartialMedium
    "Partial medium properties (base package of all media packages)"
      extends Modelica.Media.Interfaces.Types;
      extends Modelica.Icons.MaterialPropertiesPackage;

      // Constants to be set in Medium
      constant Modelica.Media.Interfaces.Choices.IndependentVariables
        ThermoStates "Enumeration type for independent variables";
      constant String mediumName="unusablePartialMedium" "Name of the medium";
      constant String substanceNames[:]={mediumName}
      "Names of the mixture substances. Set substanceNames={mediumName} if only one substance.";
      constant String extraPropertiesNames[:]=fill("", 0)
      "Names of the additional (extra) transported properties. Set extraPropertiesNames=fill(\"\",0) if unused";
      constant Boolean singleState
      "= true, if u and d are not a function of pressure";
      constant Boolean reducedX=true
      "= true if medium contains the equation sum(X) = 1.0; set reducedX=true if only one substance (see docu for details)";
      constant Boolean fixedX=false
      "= true if medium contains the equation X = reference_X";
      constant AbsolutePressure reference_p=101325
      "Reference pressure of Medium: default 1 atmosphere";
      constant Temperature reference_T=298.15
      "Reference temperature of Medium: default 25 deg Celsius";
      constant MassFraction reference_X[nX]=fill(1/nX, nX)
      "Default mass fractions of medium";
      constant AbsolutePressure p_default=101325
      "Default value for pressure of medium (for initialization)";
      constant Temperature T_default=Modelica.SIunits.Conversions.from_degC(20)
      "Default value for temperature of medium (for initialization)";
      constant SpecificEnthalpy h_default=specificEnthalpy_pTX(
              p_default,
              T_default,
              X_default)
      "Default value for specific enthalpy of medium (for initialization)";
      constant MassFraction X_default[nX]=reference_X
      "Default value for mass fractions of medium (for initialization)";

      final constant Integer nS=size(substanceNames, 1) "Number of substances"
        annotation (Evaluate=true);
      constant Integer nX=nS "Number of mass fractions" annotation (Evaluate=true);
      constant Integer nXi=if fixedX then 0 else if reducedX then nS - 1 else nS
      "Number of structurally independent mass fractions (see docu for details)"
        annotation (Evaluate=true);

      final constant Integer nC=size(extraPropertiesNames, 1)
      "Number of extra (outside of standard mass-balance) transported properties"
        annotation (Evaluate=true);
      constant Real C_nominal[nC](min=fill(Modelica.Constants.eps, nC)) = 1.0e-6*
        ones(nC) "Default for the nominal values for the extra properties";
      replaceable record FluidConstants =
          Modelica.Media.Interfaces.Types.Basic.FluidConstants
      "Critical, triple, molecular and other standard data of fluid";

      replaceable record ThermodynamicState
      "Minimal variable set that is available as input argument to every medium function"
        extends Modelica.Icons.Record;
      end ThermodynamicState;

      replaceable partial model BaseProperties
      "Base properties (p, d, T, h, u, R, MM and, if applicable, X and Xi) of a medium"
        InputAbsolutePressure p "Absolute pressure of medium";
        InputMassFraction[nXi] Xi(start=reference_X[1:nXi])
        "Structurally independent mass fractions";
        InputSpecificEnthalpy h "Specific enthalpy of medium";
        Density d "Density of medium";
        Temperature T "Temperature of medium";
        MassFraction[nX] X(start=reference_X)
        "Mass fractions (= (component mass)/total mass  m_i/m)";
        SpecificInternalEnergy u "Specific internal energy of medium";
        SpecificHeatCapacity R "Gas constant (of mixture if applicable)";
        MolarMass MM "Molar mass (of mixture or single fluid)";
        ThermodynamicState state
        "Thermodynamic state record for optional functions";
        parameter Boolean preferredMediumStates=false
        "= true if StateSelect.prefer shall be used for the independent property variables of the medium"
          annotation (Evaluate=true, Dialog(tab="Advanced"));
        parameter Boolean standardOrderComponents=true
        "If true, and reducedX = true, the last element of X will be computed from the other ones";
        SI.Conversions.NonSIunits.Temperature_degC T_degC=
            Modelica.SIunits.Conversions.to_degC(T)
        "Temperature of medium in [degC]";
        SI.Conversions.NonSIunits.Pressure_bar p_bar=
            Modelica.SIunits.Conversions.to_bar(p)
        "Absolute pressure of medium in [bar]";

        // Local connector definition, used for equation balancing check
        connector InputAbsolutePressure = input SI.AbsolutePressure
        "Pressure as input signal connector";
        connector InputSpecificEnthalpy = input SI.SpecificEnthalpy
        "Specific enthalpy as input signal connector";
        connector InputMassFraction = input SI.MassFraction
        "Mass fraction as input signal connector";

      equation
        if standardOrderComponents then
          Xi = X[1:nXi];

          if fixedX then
            X = reference_X;
          end if;
          if reducedX and not fixedX then
            X[nX] = 1 - sum(Xi);
          end if;
          for i in 1:nX loop
            assert(X[i] >= -1.e-5 and X[i] <= 1 + 1.e-5, "Mass fraction X[" +
              String(i) + "] = " + String(X[i]) + "of substance " +
              substanceNames[i] + "\nof medium " + mediumName +
              " is not in the range 0..1");
          end for;

        end if;

        assert(p >= 0.0, "Pressure (= " + String(p) + " Pa) of medium \"" +
          mediumName + "\" is negative\n(Temperature = " + String(T) + " K)");
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                lineColor={0,0,255}), Text(
                extent={{-152,164},{152,102}},
                textString="%name",
                lineColor={0,0,255})}), Documentation(info="<html>
<p>
Model <b>BaseProperties</b> is a model within package <b>PartialMedium</b>
and contains the <b>declarations</b> of the minimum number of
variables that every medium model is supposed to support.
A specific medium inherits from model <b>BaseProperties</b> and provides
the equations for the basic properties.</p>
<p>
The BaseProperties model contains the following <b>7+nXi variables</b>
(nXi is the number of independent mass fractions defined in package
PartialMedium):
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>Variable</b></td>
      <td valign=\"top\"><b>Unit</b></td>
      <td valign=\"top\"><b>Description</b></td></tr>
  <tr><td valign=\"top\">T</td>
      <td valign=\"top\">K</td>
      <td valign=\"top\">temperature</td></tr>
  <tr><td valign=\"top\">p</td>
      <td valign=\"top\">Pa</td>
      <td valign=\"top\">absolute pressure</td></tr>
  <tr><td valign=\"top\">d</td>
      <td valign=\"top\">kg/m3</td>
      <td valign=\"top\">density</td></tr>
  <tr><td valign=\"top\">h</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">specific enthalpy</td></tr>
  <tr><td valign=\"top\">u</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">specific internal energy</td></tr>
  <tr><td valign=\"top\">Xi[nXi]</td>
      <td valign=\"top\">kg/kg</td>
      <td valign=\"top\">independent mass fractions m_i/m</td></tr>
  <tr><td valign=\"top\">R</td>
      <td valign=\"top\">J/kg.K</td>
      <td valign=\"top\">gas constant</td></tr>
  <tr><td valign=\"top\">M</td>
      <td valign=\"top\">kg/mol</td>
      <td valign=\"top\">molar mass</td></tr>
</table>
<p>
In order to implement an actual medium model, one can extend from this
base model and add <b>5 equations</b> that provide relations among
these variables. Equations will also have to be added in order to
set all the variables within the ThermodynamicState record state.</p>
<p>
If standardOrderComponents=true, the full composition vector X[nX]
is determined by the equations contained in this base class, depending
on the independent mass fraction vector Xi[nXi].</p>
<p>Additional <b>2 + nXi</b> equations will have to be provided
when using the BaseProperties model, in order to fully specify the
thermodynamic conditions. The input connector qualifier applied to
p, h, and nXi indirectly declares the number of missing equations,
permitting advanced equation balance checking by Modelica tools.
Please note that this doesn't mean that the additional equations
should be connection equations, nor that exactly those variables
should be supplied, in order to complete the model.
For further information, see the Modelica.Media User's guide, and
Section 4.7 (Balanced Models) of the Modelica 3.0 specification.</p>
</html>"));
      end BaseProperties;

      replaceable partial function setState_pTX
      "Return thermodynamic state as function of p, T and composition X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      end setState_pTX;

      replaceable partial function setState_phX
      "Return thermodynamic state as function of p, h and composition X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      end setState_phX;

      replaceable partial function setState_psX
      "Return thermodynamic state as function of p, s and composition X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      end setState_psX;

      replaceable partial function setState_dTX
      "Return thermodynamic state as function of d, T and composition X or Xi"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      end setState_dTX;

      replaceable partial function setSmoothState
      "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
        extends Modelica.Icons.Function;
        input Real x "m_flow or dp";
        input ThermodynamicState state_a "Thermodynamic state if x > 0";
        input ThermodynamicState state_b "Thermodynamic state if x < 0";
        input Real x_small(min=0)
        "Smooth transition in the region -x_small < x < x_small";
        output ThermodynamicState state
        "Smooth thermodynamic state for all x (continuous and differentiable)";
        annotation (Documentation(info="<html>
<p>
This function is used to approximate the equation
</p>
<pre>
    state = <b>if</b> x &gt; 0 <b>then</b> state_a <b>else</b> state_b;
</pre>

<p>
by a smooth characteristic, so that the expression is continuous and differentiable:
</p>

<pre>
   state := <b>smooth</b>(1, <b>if</b> x &gt;  x_small <b>then</b> state_a <b>else</b>
                      <b>if</b> x &lt; -x_small <b>then</b> state_b <b>else</b> f(state_a, state_b));
</pre>

<p>
This is performed by applying function <b>Media.Common.smoothStep</b>(..)
on every element of the thermodynamic state record.
</p>

<p>
If <b>mass fractions</b> X[:] are approximated with this function then this can be performed
for all <b>nX</b> mass fractions, instead of applying it for nX-1 mass fractions and computing
the last one by the mass fraction constraint sum(X)=1. The reason is that the approximating function has the
property that sum(state.X) = 1, provided sum(state_a.X) = sum(state_b.X) = 1.
This can be shown by evaluating the approximating function in the abs(x) &lt; x_small
region (otherwise state.X is either state_a.X or state_b.X):
</p>

<pre>
    X[1]  = smoothStep(x, X_a[1] , X_b[1] , x_small);
    X[2]  = smoothStep(x, X_a[2] , X_b[2] , x_small);
       ...
    X[nX] = smoothStep(x, X_a[nX], X_b[nX], x_small);
</pre>

<p>
or
</p>

<pre>
    X[1]  = c*(X_a[1]  - X_b[1])  + (X_a[1]  + X_b[1])/2
    X[2]  = c*(X_a[2]  - X_b[2])  + (X_a[2]  + X_b[2])/2;
       ...
    X[nX] = c*(X_a[nX] - X_b[nX]) + (X_a[nX] + X_b[nX])/2;
    c     = (x/x_small)*((x/x_small)^2 - 3)/4
</pre>

<p>
Summing all mass fractions together results in
</p>

<pre>
    sum(X) = c*(sum(X_a) - sum(X_b)) + (sum(X_a) + sum(X_b))/2
           = c*(1 - 1) + (1 + 1)/2
           = 1
</pre>

</html>"));
      end setSmoothState;

      replaceable partial function dynamicViscosity "Return dynamic viscosity"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output DynamicViscosity eta "Dynamic viscosity";
      end dynamicViscosity;

      replaceable partial function thermalConductivity
      "Return thermal conductivity"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output ThermalConductivity lambda "Thermal conductivity";
      end thermalConductivity;

      replaceable function prandtlNumber "Return the Prandtl number"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output PrandtlNumber Pr "Prandtl number";
      algorithm
        Pr := dynamicViscosity(state)*specificHeatCapacityCp(state)/
          thermalConductivity(state);
      end prandtlNumber;

      replaceable partial function pressure "Return pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output AbsolutePressure p "Pressure";
      end pressure;

      replaceable partial function temperature "Return temperature"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output Temperature T "Temperature";
      end temperature;

      replaceable partial function density "Return density"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output Density d "Density";
      end density;

      replaceable partial function specificEnthalpy "Return specific enthalpy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SpecificEnthalpy h "Specific enthalpy";
      end specificEnthalpy;

      replaceable partial function specificInternalEnergy
      "Return specific internal energy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SpecificEnergy u "Specific internal energy";
      end specificInternalEnergy;

      replaceable partial function specificEntropy "Return specific entropy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SpecificEntropy s "Specific entropy";
      end specificEntropy;

      replaceable partial function specificGibbsEnergy
      "Return specific Gibbs energy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SpecificEnergy g "Specific Gibbs energy";
      end specificGibbsEnergy;

      replaceable partial function specificHelmholtzEnergy
      "Return specific Helmholtz energy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SpecificEnergy f "Specific Helmholtz energy";
      end specificHelmholtzEnergy;

      replaceable partial function specificHeatCapacityCp
      "Return specific heat capacity at constant pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SpecificHeatCapacity cp
        "Specific heat capacity at constant pressure";
      end specificHeatCapacityCp;

      function heatCapacity_cp = specificHeatCapacityCp
      "Alias for deprecated name";

      replaceable partial function specificHeatCapacityCv
      "Return specific heat capacity at constant volume"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SpecificHeatCapacity cv
        "Specific heat capacity at constant volume";
      end specificHeatCapacityCv;

      function heatCapacity_cv = specificHeatCapacityCv
      "Alias for deprecated name";

      replaceable partial function isentropicExponent
      "Return isentropic exponent"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output IsentropicExponent gamma "Isentropic exponent";
      end isentropicExponent;

      replaceable partial function isentropicEnthalpy
      "Return isentropic enthalpy"
        extends Modelica.Icons.Function;
        input AbsolutePressure p_downstream "Downstream pressure";
        input ThermodynamicState refState "Reference state for entropy";
        output SpecificEnthalpy h_is "Isentropic enthalpy";
        annotation (Documentation(info="<html>
<p>
This function computes an isentropic state transformation:
</p>
<ol>
<li> A medium is in a particular state, refState.</li>
<li> The enthalpy at another state (h_is) shall be computed
     under the assumption that the state transformation from refState to h_is
     is performed with a change of specific entropy ds = 0 and the pressure of state h_is
     is p_downstream and the composition X upstream and downstream is assumed to be the same.</li>
</ol>

</html>"));
      end isentropicEnthalpy;

      replaceable partial function velocityOfSound "Return velocity of sound"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output VelocityOfSound a "Velocity of sound";
      end velocityOfSound;

      replaceable partial function isobaricExpansionCoefficient
      "Return overall the isobaric expansion coefficient beta"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output IsobaricExpansionCoefficient beta
        "Isobaric expansion coefficient";
        annotation (Documentation(info="<html>
<pre>
beta is defined as  1/v * der(v,T), with v = 1/d, at constant pressure p.
</pre>
</html>"));
      end isobaricExpansionCoefficient;

      function beta = isobaricExpansionCoefficient
      "Alias for isobaricExpansionCoefficient for user convenience";

      replaceable partial function isothermalCompressibility
      "Return overall the isothermal compressibility factor"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output SI.IsothermalCompressibility kappa "Isothermal compressibility";
        annotation (Documentation(info="<html>
<pre>

kappa is defined as - 1/v * der(v,p), with v = 1/d at constant temperature T.

</pre>
</html>"));
      end isothermalCompressibility;

      function kappa = isothermalCompressibility
      "Alias of isothermalCompressibility for user convenience";

      // explicit derivative functions for finite element models
      replaceable partial function density_derp_h
      "Return density derivative w.r.t. pressure at const specific enthalpy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output DerDensityByPressure ddph "Density derivative w.r.t. pressure";
      end density_derp_h;

      replaceable partial function density_derh_p
      "Return density derivative w.r.t. specific enthalpy at constant pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output DerDensityByEnthalpy ddhp
        "Density derivative w.r.t. specific enthalpy";
      end density_derh_p;

      replaceable partial function density_derp_T
      "Return density derivative w.r.t. pressure at const temperature"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output DerDensityByPressure ddpT "Density derivative w.r.t. pressure";
      end density_derp_T;

      replaceable partial function density_derT_p
      "Return density derivative w.r.t. temperature at constant pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output DerDensityByTemperature ddTp
        "Density derivative w.r.t. temperature";
      end density_derT_p;

      replaceable partial function density_derX
      "Return density derivative w.r.t. mass fraction"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output Density[nX] dddX "Derivative of density w.r.t. mass fraction";
      end density_derX;

      replaceable partial function molarMass
      "Return the molar mass of the medium"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output MolarMass MM "Mixture molar mass";
      end molarMass;

      replaceable function specificEnthalpy_pTX
      "Return specific enthalpy from p, T, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_pTX(
                p,
                T,
                X));
        annotation (inverse(T=temperature_phX(
                      p,
                      h,
                      X)));
      end specificEnthalpy_pTX;

      replaceable function specificEntropy_pTX
      "Return specific enthalpy from p, T, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output SpecificEntropy s "Specific entropy";
      algorithm
        s := specificEntropy(setState_pTX(
                p,
                T,
                X));

        annotation (inverse(T=temperature_psX(
                      p,
                      s,
                      X)));
      end specificEntropy_pTX;

      replaceable function density_pTX "Return density from p, T, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:] "Mass fractions";
        output Density d "Density";
      algorithm
        d := density(setState_pTX(
                p,
                T,
                X));
      end density_pTX;

      replaceable function temperature_phX
      "Return temperature from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Temperature T "Temperature";
      algorithm
        T := temperature(setState_phX(
                p,
                h,
                X));
      end temperature_phX;

      replaceable function density_phX "Return density from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Density d "Density";
      algorithm
        d := density(setState_phX(
                p,
                h,
                X));
      end density_phX;

      replaceable function temperature_psX
      "Return temperature from p,s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Temperature T "Temperature";
      algorithm
        T := temperature(setState_psX(
                p,
                s,
                X));
        annotation (inverse(s=specificEntropy_pTX(
                      p,
                      T,
                      X)));
      end temperature_psX;

      replaceable function density_psX "Return density from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Density d "Density";
      algorithm
        d := density(setState_psX(
                p,
                s,
                X));
      end density_psX;

      replaceable function specificEnthalpy_psX
      "Return specific enthalpy from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_psX(
                p,
                s,
                X));
      end specificEnthalpy_psX;

      type MassFlowRate = SI.MassFlowRate (
          quantity="MassFlowRate." + mediumName,
          min=-1.0e5,
          max=1.e5) "Type for mass flow rate with medium specific attributes";

      // Only for backwards compatibility to version 3.2 (
      // (do not use these definitions in new models, but use Modelica.Media.Interfaces.Choices instead)
      package Choices = Modelica.Media.Interfaces.Choices annotation (obsolete=
            "Use Modelica.Media.Interfaces.Choices");

      annotation (Documentation(info="<html>
<p>
<b>PartialMedium</b> is a package and contains all <b>declarations</b> for
a medium. This means that constants, models, and functions
are defined that every medium is supposed to support
(some of them are optional). A medium package
inherits from <b>PartialMedium</b> and provides the
equations for the medium. The details of this package
are described in
<a href=\"modelica://Modelica.Media.UsersGuide\">Modelica.Media.UsersGuide</a>.
</p>
</html>",   revisions="<html>

</html>"));
    end PartialMedium;

    partial package PartialPureSubstance
    "Base class for pure substances of one chemical substance"
      extends PartialMedium(final reducedX=true, final fixedX=true);

      replaceable function setState_pT
      "Return thermodynamic state from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_pTX(
                p,
                T,
                fill(0, 0));
      end setState_pT;

      replaceable function setState_ph
      "Return thermodynamic state from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_phX(
                p,
                h,
                fill(0, 0));
      end setState_ph;

      replaceable function setState_ps
      "Return thermodynamic state from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_psX(
                p,
                s,
                fill(0, 0));
      end setState_ps;

      replaceable function setState_dT
      "Return thermodynamic state from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_dTX(
                d,
                T,
                fill(0, 0));
      end setState_dT;

      replaceable function density_ph "Return density from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        output Density d "Density";
      algorithm
        d := density_phX(
                p,
                h,
                fill(0, 0));
      end density_ph;

      replaceable function temperature_ph "Return temperature from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        output Temperature T "Temperature";
      algorithm
        T := temperature_phX(
                p,
                h,
                fill(0, 0));
      end temperature_ph;

      replaceable function pressure_dT "Return pressure from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        output AbsolutePressure p "Pressure";
      algorithm
        p := pressure(setState_dTX(
                d,
                T,
                fill(0, 0)));
      end pressure_dT;

      replaceable function specificEnthalpy_dT
      "Return specific enthalpy from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_dTX(
                d,
                T,
                fill(0, 0)));
      end specificEnthalpy_dT;

      replaceable function specificEnthalpy_ps
      "Return specific enthalpy from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy_psX(
                p,
                s,
                fill(0, 0));
      end specificEnthalpy_ps;

      replaceable function temperature_ps "Return temperature from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output Temperature T "Temperature";
      algorithm
        T := temperature_psX(
                p,
                s,
                fill(0, 0));
      end temperature_ps;

      replaceable function density_ps "Return density from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output Density d "Density";
      algorithm
        d := density_psX(
                p,
                s,
                fill(0, 0));
      end density_ps;

      replaceable function specificEnthalpy_pT
      "Return specific enthalpy from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy_pTX(
                p,
                T,
                fill(0, 0));
      end specificEnthalpy_pT;

      replaceable function density_pT "Return density from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        output Density d "Density";
      algorithm
        d := density(setState_pTX(
                p,
                T,
                fill(0, 0)));
      end density_pT;

      redeclare replaceable partial model extends BaseProperties(final
          standardOrderComponents=true)
      end BaseProperties;
    end PartialPureSubstance;

    partial package PartialTwoPhaseMedium
    "Base class for two phase medium of one substance"
      extends PartialPureSubstance(redeclare record FluidConstants =
            Modelica.Media.Interfaces.Types.TwoPhase.FluidConstants);
      constant Boolean smoothModel=false
      "True if the (derived) model should not generate state events";
      constant Boolean onePhase=false
      "True if the (derived) model should never be called with two-phase inputs";

      constant FluidConstants[nS] fluidConstants "Constant data for the fluid";

      redeclare replaceable record extends ThermodynamicState
      "Thermodynamic state of two phase medium"
        FixedPhase phase(min=0, max=2)
        "Phase of the fluid: 1 for 1-phase, 2 for two-phase, 0 for not known, e.g., interactive use";
      end ThermodynamicState;

      redeclare replaceable partial model extends BaseProperties
      "Base properties (p, d, T, h, u, R, MM, sat) of two phase medium"
        SaturationProperties sat "Saturation properties at the medium pressure";
      end BaseProperties;

      replaceable partial function setDewState
      "Return the thermodynamic state on the dew line"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation point";
        input FixedPhase phase(
          min=1,
          max=2) = 1 "Phase: default is one phase";
        output ThermodynamicState state "Complete thermodynamic state info";
      end setDewState;

      replaceable partial function setBubbleState
      "Return the thermodynamic state on the bubble line"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation point";
        input FixedPhase phase(
          min=1,
          max=2) = 1 "Phase: default is one phase";
        output ThermodynamicState state "Complete thermodynamic state info";
      end setBubbleState;

      redeclare replaceable partial function extends setState_dTX
      "Return thermodynamic state as function of d, T and composition X or Xi"
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      end setState_dTX;

      redeclare replaceable partial function extends setState_phX
      "Return thermodynamic state as function of p, h and composition X or Xi"
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      end setState_phX;

      redeclare replaceable partial function extends setState_psX
      "Return thermodynamic state as function of p, s and composition X or Xi"
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      end setState_psX;

      redeclare replaceable partial function extends setState_pTX
      "Return thermodynamic state as function of p, T and composition X or Xi"
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
      end setState_pTX;

      replaceable function setSat_T
      "Return saturation property record from temperature"
        extends Modelica.Icons.Function;
        input Temperature T "Temperature";
        output SaturationProperties sat "Saturation property record";
      algorithm
        sat.Tsat := T;
        sat.psat := saturationPressure(T);
      end setSat_T;

      replaceable function setSat_p
      "Return saturation property record from pressure"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        output SaturationProperties sat "Saturation property record";
      algorithm
        sat.psat := p;
        sat.Tsat := saturationTemperature(p);
      end setSat_p;

      replaceable partial function bubbleEnthalpy
      "Return bubble point specific enthalpy"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output SI.SpecificEnthalpy hl "Boiling curve specific enthalpy";
      end bubbleEnthalpy;

      replaceable partial function dewEnthalpy
      "Return dew point specific enthalpy"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output SI.SpecificEnthalpy hv "Dew curve specific enthalpy";
      end dewEnthalpy;

      replaceable partial function bubbleEntropy
      "Return bubble point specific entropy"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output SI.SpecificEntropy sl "Boiling curve specific entropy";
      end bubbleEntropy;

      replaceable partial function dewEntropy
      "Return dew point specific entropy"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output SI.SpecificEntropy sv "Dew curve specific entropy";
      end dewEntropy;

      replaceable partial function bubbleDensity "Return bubble point density"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output Density dl "Boiling curve density";
      end bubbleDensity;

      replaceable partial function dewDensity "Return dew point density"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output Density dv "Dew curve density";
      end dewDensity;

      replaceable partial function saturationPressure
      "Return saturation pressure"
        extends Modelica.Icons.Function;
        input Temperature T "Temperature";
        output AbsolutePressure p "Saturation pressure";
      end saturationPressure;

      replaceable partial function saturationTemperature
      "Return saturation temperature"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        output Temperature T "Saturation temperature";
      end saturationTemperature;

      replaceable function saturationPressure_sat
      "Return saturation temperature"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output AbsolutePressure p "Saturation pressure";
      algorithm
        p := sat.psat;
      end saturationPressure_sat;

      replaceable function saturationTemperature_sat
      "Return saturation temperature"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output Temperature T "Saturation temperature";
      algorithm
        T := sat.Tsat;
      end saturationTemperature_sat;

      replaceable partial function saturationTemperature_derp
      "Return derivative of saturation temperature w.r.t. pressure"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        output DerTemperatureByPressure dTp
        "Derivative of saturation temperature w.r.t. pressure";
      end saturationTemperature_derp;

      replaceable function saturationTemperature_derp_sat
      "Return derivative of saturation temperature w.r.t. pressure"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output DerTemperatureByPressure dTp
        "Derivative of saturation temperature w.r.t. pressure";
      algorithm
        dTp := saturationTemperature_derp(sat.psat);
      end saturationTemperature_derp_sat;

      replaceable partial function surfaceTension
      "Return surface tension sigma in the two phase region"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output SurfaceTension sigma
        "Surface tension sigma in the two phase region";
      end surfaceTension;

      redeclare replaceable function extends molarMass
      "Return the molar mass of the medium"
      algorithm
        MM := fluidConstants[1].molarMass;
      end molarMass;

      replaceable partial function dBubbleDensity_dPressure
      "Return bubble point density derivative"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output DerDensityByPressure ddldp "Boiling curve density derivative";
      end dBubbleDensity_dPressure;

      replaceable partial function dDewDensity_dPressure
      "Return dew point density derivative"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output DerDensityByPressure ddvdp "Saturated steam density derivative";
      end dDewDensity_dPressure;

      replaceable partial function dBubbleEnthalpy_dPressure
      "Return bubble point specific enthalpy derivative"
        extends Modelica.Icons.Function;
        input SaturationProperties sat "Saturation property record";
        output DerEnthalpyByPressure dhldp
        "Boiling curve specific enthalpy derivative";
      end dBubbleEnthalpy_dPressure;

      replaceable partial function dDewEnthalpy_dPressure
      "Return dew point specific enthalpy derivative"
        extends Modelica.Icons.Function;

        input SaturationProperties sat "Saturation property record";
        output DerEnthalpyByPressure dhvdp
        "Saturated steam specific enthalpy derivative";
      end dDewEnthalpy_dPressure;

      redeclare replaceable function specificEnthalpy_pTX
      "Return specific enthalpy from pressure, temperature and mass fraction"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:] "Mass fractions";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy at p, T, X";
      algorithm
        h := specificEnthalpy(setState_pTX(
                p,
                T,
                X,
                phase));
      end specificEnthalpy_pTX;

      redeclare replaceable function temperature_phX
      "Return temperature from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:] "Mass fractions";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Temperature T "Temperature";
      algorithm
        T := temperature(setState_phX(
                p,
                h,
                X,
                phase));
      end temperature_phX;

      redeclare replaceable function density_phX
      "Return density from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:] "Mass fractions";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := density(setState_phX(
                p,
                h,
                X,
                phase));
      end density_phX;

      redeclare replaceable function temperature_psX
      "Return temperature from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:] "Mass fractions";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Temperature T "Temperature";
      algorithm
        T := temperature(setState_psX(
                p,
                s,
                X,
                phase));
      end temperature_psX;

      redeclare replaceable function density_psX
      "Return density from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:] "Mass fractions";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := density(setState_psX(
                p,
                s,
                X,
                phase));
      end density_psX;

      redeclare replaceable function specificEnthalpy_psX
      "Return specific enthalpy from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:] "Mass fractions";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_psX(
                p,
                s,
                X,
                phase));
      end specificEnthalpy_psX;

      redeclare replaceable function setState_pT
      "Return thermodynamic state from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_pTX(
                p,
                T,
                fill(0, 0),
                phase);
      end setState_pT;

      redeclare replaceable function setState_ph
      "Return thermodynamic state from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_phX(
                p,
                h,
                fill(0, 0),
                phase);
      end setState_ph;

      redeclare replaceable function setState_ps
      "Return thermodynamic state from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_psX(
                p,
                s,
                fill(0, 0),
                phase);
      end setState_ps;

      redeclare replaceable function setState_dT
      "Return thermodynamic state from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_dTX(
                d,
                T,
                fill(0, 0),
                phase);
      end setState_dT;

      replaceable function setState_px
      "Return thermodynamic state from pressure and vapour quality"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input MassFraction x "Vapour quality";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_ph(
                p,
                (1 - x)*bubbleEnthalpy(setSat_p(p)) + x*dewEnthalpy(setSat_p(p)),
                2);
      end setState_px;

      replaceable function setState_Tx
      "Return thermodynamic state from temperature and vapour quality"
        extends Modelica.Icons.Function;
        input Temperature T "Temperature";
        input MassFraction x "Vapour quality";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := setState_ph(
                saturationPressure_sat(setSat_T(T)),
                (1 - x)*bubbleEnthalpy(setSat_T(T)) + x*dewEnthalpy(setSat_T(T)),
                2);
      end setState_Tx;

      replaceable function vapourQuality "Return vapour quality"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output MassFraction x "Vapour quality";
    protected
        constant SpecificEnthalpy eps=1e-8;
      algorithm
        x := min(max((specificEnthalpy(state) - bubbleEnthalpy(setSat_p(pressure(
          state))))/(dewEnthalpy(setSat_p(pressure(state))) - bubbleEnthalpy(
          setSat_p(pressure(state))) + eps), 0), 1);
      end vapourQuality;

      redeclare replaceable function density_ph "Return density from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := density_phX(
                p,
                h,
                fill(0, 0),
                phase);
      end density_ph;

      redeclare replaceable function temperature_ph
      "Return temperature from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Temperature T "Temperature";
      algorithm
        T := temperature_phX(
                p,
                h,
                fill(0, 0),
                phase);
      end temperature_ph;

      redeclare replaceable function pressure_dT "Return pressure from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output AbsolutePressure p "Pressure";
      algorithm
        p := pressure(setState_dTX(
                d,
                T,
                fill(0, 0),
                phase));
      end pressure_dT;

      redeclare replaceable function specificEnthalpy_dT
      "Return specific enthalpy from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_dTX(
                d,
                T,
                fill(0, 0),
                phase));
      end specificEnthalpy_dT;

      redeclare replaceable function specificEnthalpy_ps
      "Return specific enthalpy from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy_psX(
                p,
                s,
                fill(0, 0));
      end specificEnthalpy_ps;

      redeclare replaceable function temperature_ps
      "Return temperature from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Temperature T "Temperature";
      algorithm
        T := temperature_psX(
                p,
                s,
                fill(0, 0),
                phase);
      end temperature_ps;

      redeclare replaceable function density_ps "Return density from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := density_psX(
                p,
                s,
                fill(0, 0),
                phase);
      end density_ps;

      redeclare replaceable function specificEnthalpy_pT
      "Return specific enthalpy from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy_pTX(
                p,
                T,
                fill(0, 0),
                phase);
      end specificEnthalpy_pT;

      redeclare replaceable function density_pT "Return density from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
        "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := density(setState_pTX(
                p,
                T,
                fill(0, 0),
                phase));
      end density_pT;
    end PartialTwoPhaseMedium;

    partial package PartialSimpleMedium
    "Medium model with linear dependency of u, h from temperature. All other quantities, especially density, are constant."

      extends Interfaces.PartialPureSubstance(final ThermoStates=Modelica.Media.Interfaces.Choices.IndependentVariables.pT,
          final singleState=true);

      constant SpecificHeatCapacity cp_const
      "Constant specific heat capacity at constant pressure";
      constant SpecificHeatCapacity cv_const
      "Constant specific heat capacity at constant volume";
      constant Density d_const "Constant density";
      constant DynamicViscosity eta_const "Constant dynamic viscosity";
      constant ThermalConductivity lambda_const "Constant thermal conductivity";
      constant VelocityOfSound a_const "Constant velocity of sound";
      constant Temperature T_min "Minimum temperature valid for medium model";
      constant Temperature T_max "Maximum temperature valid for medium model";
      constant Temperature T0=reference_T "Zero enthalpy temperature";
      constant MolarMass MM_const "Molar mass";

      constant FluidConstants[nS] fluidConstants "Fluid constants";

      redeclare record extends ThermodynamicState "Thermodynamic state"
        AbsolutePressure p "Absolute pressure of medium";
        Temperature T "Temperature of medium";
      end ThermodynamicState;

      redeclare replaceable model extends BaseProperties(T(stateSelect=if
              preferredMediumStates then StateSelect.prefer else StateSelect.default),
          p(stateSelect=if preferredMediumStates then StateSelect.prefer else
              StateSelect.default)) "Base properties"
      equation
        assert(T >= T_min and T <= T_max, "
Temperature T (= "   + String(T) + " K) is not
in the allowed range ("   + String(T_min) + " K <= T <= " + String(T_max) + " K)
required from medium model \""   + mediumName + "\".
");

        // h = cp_const*(T-T0);
        h = specificEnthalpy_pTX(
                p,
                T,
                X);
        u = cv_const*(T - T0);
        d = d_const;
        R = 0;
        MM = MM_const;
        state.T = T;
        state.p = p;
        annotation (Documentation(info="<HTML>
<p>
This is the most simple incompressible medium model, where
specific enthalpy h and specific internal energy u are only
a function of temperature T and all other provided medium
quantities are assumed to be constant.
Note that the (small) influence of the pressure term p/d is neglected.
</p>
</HTML>"));
      end BaseProperties;

      redeclare function setState_pTX
      "Return thermodynamic state from p, T, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := ThermodynamicState(p=p, T=T);
      end setState_pTX;

      redeclare function setState_phX
      "Return thermodynamic state from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := ThermodynamicState(p=p, T=T0 + h/cp_const);
      end setState_phX;

      redeclare replaceable function setState_psX
      "Return thermodynamic state from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        state := ThermodynamicState(p=p, T=Modelica.Math.exp(s/cp_const +
          Modelica.Math.log(reference_T)))
        "Here the incompressible limit is used, with cp as heat capacity";
      end setState_psX;

      redeclare function setState_dTX
      "Return thermodynamic state from d, T, and X or Xi"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "Thermodynamic state record";
      algorithm
        assert(false,
          "Pressure can not be computed from temperature and density for an incompressible fluid!");
      end setState_dTX;

      redeclare function extends setSmoothState
      "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
      algorithm
        state := ThermodynamicState(p=Media.Common.smoothStep(
                x,
                state_a.p,
                state_b.p,
                x_small), T=Media.Common.smoothStep(
                x,
                state_a.T,
                state_b.T,
                x_small));
      end setSmoothState;

      redeclare function extends dynamicViscosity "Return dynamic viscosity"

      algorithm
        eta := eta_const;
      end dynamicViscosity;

      redeclare function extends thermalConductivity
      "Return thermal conductivity"

      algorithm
        lambda := lambda_const;
      end thermalConductivity;

      redeclare function extends pressure "Return pressure"

      algorithm
        p := state.p;
      end pressure;

      redeclare function extends temperature "Return temperature"

      algorithm
        T := state.T;
      end temperature;

      redeclare function extends density "Return density"

      algorithm
        d := d_const;
      end density;

      redeclare function extends specificEnthalpy "Return specific enthalpy"

      algorithm
        h := cp_const*(state.T - T0);
      end specificEnthalpy;

      redeclare function extends specificHeatCapacityCp
      "Return specific heat capacity at constant pressure"

      algorithm
        cp := cp_const;
      end specificHeatCapacityCp;

      redeclare function extends specificHeatCapacityCv
      "Return specific heat capacity at constant volume"

      algorithm
        cv := cv_const;
      end specificHeatCapacityCv;

      redeclare function extends isentropicExponent
      "Return isentropic exponent"

      algorithm
        gamma := cp_const/cv_const;
      end isentropicExponent;

      redeclare function extends velocityOfSound "Return velocity of sound"

      algorithm
        a := a_const;
      end velocityOfSound;

      redeclare function specificEnthalpy_pTX
      "Return specific enthalpy from p, T, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[nX] "Mass fractions";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := cp_const*(T - T0);
        annotation (Documentation(info="<html>
<p>
This function computes the specific enthalpy of the fluid, but neglects the (small) influence of the pressure term p/d.
</p>
</html>"));
      end specificEnthalpy_pTX;

      redeclare function temperature_phX
      "Return temperature from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[nX] "Mass fractions";
        output Temperature T "Temperature";
      algorithm
        T := T0 + h/cp_const;
      end temperature_phX;

      redeclare function density_phX "Return density from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[nX] "Mass fractions";
        output Density d "Density";
      algorithm
        d := density(setState_phX(
                p,
                h,
                X));
      end density_phX;

      redeclare function extends specificInternalEnergy
      "Return specific internal energy"
        extends Modelica.Icons.Function;
      algorithm
        //  u := cv_const*(state.T - T0) - reference_p/d_const;
        u := cv_const*(state.T - T0);
        annotation (Documentation(info="<html>
<p>
This function computes the specific internal energy of the fluid, but neglects the (small) influence of the pressure term p/d.
</p>
</html>"));
      end specificInternalEnergy;

      redeclare function extends specificEntropy "Return specific entropy"
        extends Modelica.Icons.Function;
      algorithm
        s := cv_const*Modelica.Math.log(state.T/T0);
      end specificEntropy;

      redeclare function extends specificGibbsEnergy
      "Return specific Gibbs energy"
        extends Modelica.Icons.Function;
      algorithm
        g := specificEnthalpy(state) - state.T*specificEntropy(state);
      end specificGibbsEnergy;

      redeclare function extends specificHelmholtzEnergy
      "Return specific Helmholtz energy"
        extends Modelica.Icons.Function;
      algorithm
        f := specificInternalEnergy(state) - state.T*specificEntropy(state);
      end specificHelmholtzEnergy;

      redeclare function extends isentropicEnthalpy
      "Return isentropic enthalpy"
      algorithm
        h_is := cp_const*(temperature(refState) - T0);
      end isentropicEnthalpy;

      redeclare function extends isobaricExpansionCoefficient
      "Returns overall the isobaric expansion coefficient beta"
      algorithm
        beta := 0.0;
      end isobaricExpansionCoefficient;

      redeclare function extends isothermalCompressibility
      "Returns overall the isothermal compressibility factor"
      algorithm
        kappa := 0;
      end isothermalCompressibility;

      redeclare function extends density_derp_T
      "Returns the partial derivative of density with respect to pressure at constant temperature"
      algorithm
        ddpT := 0;
      end density_derp_T;

      redeclare function extends density_derT_p
      "Returns the partial derivative of density with respect to temperature at constant pressure"
      algorithm
        ddTp := 0;
      end density_derT_p;

      redeclare function extends density_derX
      "Returns the partial derivative of density with respect to mass fractions at constant pressure and temperature"
      algorithm
        dddX := fill(0, nX);
      end density_derX;

      redeclare function extends molarMass
      "Return the molar mass of the medium"
      algorithm
        MM := MM_const;
      end molarMass;
    end PartialSimpleMedium;

    package Choices "Types, constants to define menu choices"
      extends Modelica.Icons.Package;

      type IndependentVariables = enumeration(
        T "Temperature",
        pT "Pressure, Temperature",
        ph "Pressure, Specific Enthalpy",
        phX "Pressure, Specific Enthalpy, Mass Fraction",
        pTX "Pressure, Temperature, Mass Fractions",
        dTX "Density, Temperature, Mass Fractions")
      "Enumeration defining the independent variables of a medium";
      annotation (Documentation(info="<html>
<p>
Enumerations and data types for all types of fluids
</p>

<p>
Note: Reference enthalpy might have to be extended with enthalpy of formation.
</p>
</html>"));
    end Choices;

    package Types "Types to be used in fluid models"
      extends Modelica.Icons.Package;

      type AbsolutePressure = SI.AbsolutePressure (
          min=0,
          max=1.e8,
          nominal=1.e5,
          start=1.e5)
      "Type for absolute pressure with medium specific attributes";

      type Density = SI.Density (
          min=0,
          max=1.e5,
          nominal=1,
          start=1) "Type for density with medium specific attributes";

      type DynamicViscosity = SI.DynamicViscosity (
          min=0,
          max=1.e8,
          nominal=1.e-3,
          start=1.e-3)
      "Type for dynamic viscosity with medium specific attributes";

      type MassFraction = Real (
          quantity="MassFraction",
          final unit="kg/kg",
          min=0,
          max=1,
          nominal=0.1) "Type for mass fraction with medium specific attributes";

      type MolarMass = SI.MolarMass (
          min=0.001,
          max=0.25,
          nominal=0.032) "Type for molar mass with medium specific attributes";

      type MolarVolume = SI.MolarVolume (
          min=1e-6,
          max=1.0e6,
          nominal=1.0) "Type for molar volume with medium specific attributes";

      type IsentropicExponent = SI.RatioOfSpecificHeatCapacities (
          min=1,
          max=500000,
          nominal=1.2,
          start=1.2)
      "Type for isentropic exponent with medium specific attributes";

      type SpecificEnergy = SI.SpecificEnergy (
          min=-1.0e8,
          max=1.e8,
          nominal=1.e6)
      "Type for specific energy with medium specific attributes";

      type SpecificInternalEnergy = SpecificEnergy
      "Type for specific internal energy with medium specific attributes";

      type SpecificEnthalpy = SI.SpecificEnthalpy (
          min=-1.0e10,
          max=1.e10,
          nominal=1.e6)
      "Type for specific enthalpy with medium specific attributes";

      type SpecificEntropy = SI.SpecificEntropy (
          min=-1.e7,
          max=1.e7,
          nominal=1.e3)
      "Type for specific entropy with medium specific attributes";

      type SpecificHeatCapacity = SI.SpecificHeatCapacity (
          min=0,
          max=1.e7,
          nominal=1.e3,
          start=1.e3)
      "Type for specific heat capacity with medium specific attributes";

      type SurfaceTension = SI.SurfaceTension
      "Type for surface tension with medium specific attributes";

      type Temperature = SI.Temperature (
          min=1,
          max=1.e4,
          nominal=300,
          start=300) "Type for temperature with medium specific attributes";

      type ThermalConductivity = SI.ThermalConductivity (
          min=0,
          max=500,
          nominal=1,
          start=1)
      "Type for thermal conductivity with medium specific attributes";

      type PrandtlNumber = SI.PrandtlNumber (
          min=1e-3,
          max=1e5,
          nominal=1.0)
      "Type for Prandtl number with medium specific attributes";

      type VelocityOfSound = SI.Velocity (
          min=0,
          max=1.e5,
          nominal=1000,
          start=1000)
      "Type for velocity of sound with medium specific attributes";

      type ExtraProperty = Real (min=0.0, start=1.0)
      "Type for unspecified, mass-specific property transported by flow";

      type IsobaricExpansionCoefficient = Real (
          min=0,
          max=1.0e8,
          unit="1/K")
      "Type for isobaric expansion coefficient with medium specific attributes";

      type DipoleMoment = Real (
          min=0.0,
          max=2.0,
          unit="debye",
          quantity="ElectricDipoleMoment")
      "Type for dipole moment with medium specific attributes";

      type DerDensityByPressure = SI.DerDensityByPressure
      "Type for partial derivative of density with respect to pressure with medium specific attributes";

      type DerDensityByEnthalpy = SI.DerDensityByEnthalpy
      "Type for partial derivative of density with respect to enthalpy with medium specific attributes";

      type DerEnthalpyByPressure = SI.DerEnthalpyByPressure
      "Type for partial derivative of enthalpy with respect to pressure with medium specific attributes";

      type DerDensityByTemperature = SI.DerDensityByTemperature
      "Type for partial derivative of density with respect to temperature with medium specific attributes";

      type DerTemperatureByPressure = Real (final unit="K/Pa")
      "Type for partial derivative of temperature with respect to pressure with medium specific attributes";

      replaceable record SaturationProperties
      "Saturation properties of two phase medium"
        extends Modelica.Icons.Record;
        AbsolutePressure psat "Saturation pressure";
        Temperature Tsat "Saturation temperature";
      end SaturationProperties;

      record FluidLimits "Validity limits for fluid model"
        extends Modelica.Icons.Record;
        Temperature TMIN "Minimum temperature";
        Temperature TMAX "Maximum temperature";
        Density DMIN "Minimum density";
        Density DMAX "Maximum density";
        AbsolutePressure PMIN "Minimum pressure";
        AbsolutePressure PMAX "Maximum pressure";
        SpecificEnthalpy HMIN "Minimum enthalpy";
        SpecificEnthalpy HMAX "Maximum enthalpy";
        SpecificEntropy SMIN "Minimum entropy";
        SpecificEntropy SMAX "Maximum entropy";
        annotation (Documentation(info="<html>
          <p>The minimum pressure mostly applies to the liquid state only.
          The minimum density is also arbitrary, but is reasonable for technical
          applications to limit iterations in non-linear systems. The limits in
          enthalpy and entropy are used as safeguards in inverse iterations.</p>
          </html>"));
      end FluidLimits;

      type FixedPhase = Integer (min=0, max=2)
      "Phase of the fluid: 1 for 1-phase, 2 for two-phase, 0 for not known, e.g., interactive use";

      package Basic
      "The most basic version of a record used in several degrees of detail"
        extends Icons.Package;

        record FluidConstants
        "Critical, triple, molecular and other standard data of fluid"
          extends Modelica.Icons.Record;
          String iupacName
          "Complete IUPAC name (or common name, if non-existent)";
          String casRegistryNumber
          "Chemical abstracts sequencing number (if it exists)";
          String chemicalFormula
          "Chemical formula, (brutto, nomenclature according to Hill";
          String structureFormula "Chemical structure formula";
          MolarMass molarMass "Molar mass";
        end FluidConstants;
      end Basic;

      package TwoPhase
      "The two phase fluid version of a record used in several degrees of detail"
        extends Icons.Package;

        record FluidConstants "Extended fluid constants"
          extends Modelica.Media.Interfaces.Types.Basic.FluidConstants;
          Temperature criticalTemperature "Critical temperature";
          AbsolutePressure criticalPressure "Critical pressure";
          MolarVolume criticalMolarVolume "Critical molar Volume";
          Real acentricFactor "Pitzer acentric factor";
          Temperature triplePointTemperature "Triple point temperature";
          AbsolutePressure triplePointPressure "Triple point pressure";
          Temperature meltingPoint "Melting point at 101325 Pa";
          Temperature normalBoilingPoint "Normal boiling point (at 101325 Pa)";
          DipoleMoment dipoleMoment
          "Dipole moment of molecule in Debye (1 debye = 3.33564e10-30 C.m)";
          Boolean hasIdealGasHeatCapacity=false
          "True if ideal gas heat capacity is available";
          Boolean hasCriticalData=false "True if critical data are known";
          Boolean hasDipoleMoment=false "True if a dipole moment known";
          Boolean hasFundamentalEquation=false "True if a fundamental equation";
          Boolean hasLiquidHeatCapacity=false
          "True if liquid heat capacity is available";
          Boolean hasSolidHeatCapacity=false
          "True if solid heat capacity is available";
          Boolean hasAccurateViscosityData=false
          "True if accurate data for a viscosity function is available";
          Boolean hasAccurateConductivityData=false
          "True if accurate data for thermal conductivity is available";
          Boolean hasVapourPressureCurve=false
          "True if vapour pressure data, e.g., Antoine coefficents are known";
          Boolean hasAcentricFactor=false
          "True if Pitzer accentric factor is known";
          SpecificEnthalpy HCRIT0=0.0
          "Critical specific enthalpy of the fundamental equation";
          SpecificEntropy SCRIT0=0.0
          "Critical specific entropy of the fundamental equation";
          SpecificEnthalpy deltah=0.0
          "Difference between specific enthalpy model (h_m) and f.eq. (h_f) (h_m - h_f)";
          SpecificEntropy deltas=0.0
          "Difference between specific enthalpy model (s_m) and f.eq. (s_f) (s_m - s_f)";
        end FluidConstants;
      end TwoPhase;
    end Types;
    annotation (Documentation(info="<HTML>
<p>
This package provides basic interfaces definitions of media models for different
kind of media.
</p>
</HTML>"));
  end Interfaces;

  package Common
    "Data structures and fundamental functions for fluid properties"
    extends Modelica.Icons.Package;

    type DerPressureByDensity = Real (final quantity="DerPressureByDensity",
          final unit="Pa.m3/kg");

    type DerPressureByTemperature = Real (final quantity=
            "DerPressureByTemperature", final unit="Pa/K");

    record IF97BaseTwoPhase "Intermediate property data record for IF 97"
      extends Modelica.Icons.Record;
      Integer phase=0 "Phase: 2 for two-phase, 1 for one phase, 0 if unknown";
      Integer region(min=1, max=5) "IF 97 region";
      SI.Pressure p "Pressure";
      SI.Temperature T "Temperature";
      SI.SpecificEnthalpy h "Specific enthalpy";
      SI.SpecificHeatCapacity R "Gas constant";
      SI.SpecificHeatCapacity cp "Specific heat capacity";
      SI.SpecificHeatCapacity cv "Specific heat capacity";
      SI.Density rho "Density";
      SI.SpecificEntropy s "Specific entropy";
      DerPressureByTemperature pt "Derivative of pressure w.r.t. temperature";
      DerPressureByDensity pd "Derivative of pressure w.r.t. density";
      Real vt "Derivative of specific volume w.r.t. temperature";
      Real vp "Derivative of specific volume w.r.t. pressure";
      Real x "Dryness fraction";
      Real dpT "dp/dT derivative of saturation curve";
    end IF97BaseTwoPhase;

    record IF97PhaseBoundaryProperties
    "Thermodynamic base properties on the phase boundary for IF97 steam tables"

      extends Modelica.Icons.Record;
      Boolean region3boundary "True if boundary between 2-phase and region 3";
      SI.SpecificHeatCapacity R "Specific heat capacity";
      SI.Temperature T "Temperature";
      SI.Density d "Density";
      SI.SpecificEnthalpy h "Specific enthalpy";
      SI.SpecificEntropy s "Specific entropy";
      SI.SpecificHeatCapacity cp "Heat capacity at constant pressure";
      SI.SpecificHeatCapacity cv "Heat capacity at constant volume";
      DerPressureByTemperature dpT "dp/dT derivative of saturation curve";
      DerPressureByTemperature pt "Derivative of pressure w.r.t. temperature";
      DerPressureByDensity pd "Derivative of pressure w.r.t. density";
      Real vt(unit="m3/(kg.K)")
      "Derivative of specific volume w.r.t. temperature";
      Real vp(unit="m3/(kg.Pa)")
      "Derivative of specific volume w.r.t. pressure";
    end IF97PhaseBoundaryProperties;

    record GibbsDerivs
    "Derivatives of dimensionless Gibbs-function w.r.t. dimensionless pressure and temperature"

      extends Modelica.Icons.Record;
      SI.Pressure p "Pressure";
      SI.Temperature T "Temperature";
      SI.SpecificHeatCapacity R "Specific heat capacity";
      Real pi(unit="1") "Dimensionless pressure";
      Real tau(unit="1") "Dimensionless temperature";
      Real g(unit="1") "Dimensionless Gibbs-function";
      Real gpi(unit="1") "Derivative of g w.r.t. pi";
      Real gpipi(unit="1") "2nd derivative of g w.r.t. pi";
      Real gtau(unit="1") "Derivative of g w.r.t. tau";
      Real gtautau(unit="1") "2nd derivative of g w.r.t. tau";
      Real gtaupi(unit="1") "Mixed derivative of g w.r.t. pi and tau";
    end GibbsDerivs;

    record HelmholtzDerivs
    "Derivatives of dimensionless Helmholtz-function w.r.t. dimensionless pressure, density and temperature"
      extends Modelica.Icons.Record;
      SI.Density d "Density";
      SI.Temperature T "Temperature";
      SI.SpecificHeatCapacity R "Specific heat capacity";
      Real delta(unit="1") "Dimensionless density";
      Real tau(unit="1") "Dimensionless temperature";
      Real f(unit="1") "Dimensionless Helmholtz-function";
      Real fdelta(unit="1") "Derivative of f w.r.t. delta";
      Real fdeltadelta(unit="1") "2nd derivative of f w.r.t. delta";
      Real ftau(unit="1") "Derivative of f w.r.t. tau";
      Real ftautau(unit="1") "2nd derivative of f w.r.t. tau";
      Real fdeltatau(unit="1") "Mixed derivative of f w.r.t. delta and tau";
    end HelmholtzDerivs;

    record PhaseBoundaryProperties
    "Thermodynamic base properties on the phase boundary"
      extends Modelica.Icons.Record;
      SI.Density d "Density";
      SI.SpecificEnthalpy h "Specific enthalpy";
      SI.SpecificEnergy u "Inner energy";
      SI.SpecificEntropy s "Specific entropy";
      SI.SpecificHeatCapacity cp "Heat capacity at constant pressure";
      SI.SpecificHeatCapacity cv "Heat capacity at constant volume";
      DerPressureByTemperature pt "Derivative of pressure w.r.t. temperature";
      DerPressureByDensity pd "Derivative of pressure w.r.t. density";
    end PhaseBoundaryProperties;

    record NewtonDerivatives_ph
    "Derivatives for fast inverse calculations of Helmholtz functions: p & h"

      extends Modelica.Icons.Record;
      SI.Pressure p "Pressure";
      SI.SpecificEnthalpy h "Specific enthalpy";
      DerPressureByDensity pd "Derivative of pressure w.r.t. density";
      DerPressureByTemperature pt "Derivative of pressure w.r.t. temperature";
      Real hd "Derivative of specific enthalpy w.r.t. density";
      Real ht "Derivative of specific enthalpy w.r.t. temperature";
    end NewtonDerivatives_ph;

    record NewtonDerivatives_ps
    "Derivatives for fast inverse calculation of Helmholtz functions: p & s"

      extends Modelica.Icons.Record;
      SI.Pressure p "Pressure";
      SI.SpecificEntropy s "Specific entropy";
      DerPressureByDensity pd "Derivative of pressure w.r.t. density";
      DerPressureByTemperature pt "Derivative of pressure w.r.t. temperature";
      Real sd "Derivative of specific entropy w.r.t. density";
      Real st "Derivative of specific entropy w.r.t. temperature";
    end NewtonDerivatives_ps;

    record NewtonDerivatives_pT
    "Derivatives for fast inverse calculations of Helmholtz functions:p & T"

      extends Modelica.Icons.Record;
      SI.Pressure p "Pressure";
      DerPressureByDensity pd "Derivative of pressure w.r.t. density";
    end NewtonDerivatives_pT;

    function gibbsToBoundaryProps
    "Calculate phase boundary property record from dimensionless Gibbs function"

      extends Modelica.Icons.Function;
      input GibbsDerivs g "Dimensionless derivatives of Gibbs function";
      output PhaseBoundaryProperties sat "Phase boundary properties";
  protected
      Real vt "Derivative of specific volume w.r.t. temperature";
      Real vp "Derivative of specific volume w.r.t. pressure";
    algorithm
      sat.d := g.p/(g.R*g.T*g.pi*g.gpi);
      sat.h := g.R*g.T*g.tau*g.gtau;
      sat.u := g.T*g.R*(g.tau*g.gtau - g.pi*g.gpi);
      sat.s := g.R*(g.tau*g.gtau - g.g);
      sat.cp := -g.R*g.tau*g.tau*g.gtautau;
      sat.cv := g.R*(-g.tau*g.tau*g.gtautau + (g.gpi - g.tau*g.gtaupi)*(g.gpi - g.tau
        *g.gtaupi)/(g.gpipi));
      vt := g.R/g.p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
      vp := g.R*g.T/(g.p*g.p)*g.pi*g.pi*g.gpipi;
      // sat.kappa := -1/(sat.d*g.p)*sat.cp/(vp*sat.cp + vt*vt*g.T);
      sat.pt := -g.p/g.T*(g.gpi - g.tau*g.gtaupi)/(g.gpipi*g.pi);
      sat.pd := -g.R*g.T*g.gpi*g.gpi/(g.gpipi);
    end gibbsToBoundaryProps;

    function helmholtzToBoundaryProps
    "Calculate phase boundary property record from dimensionless Helmholtz function"

      extends Modelica.Icons.Function;
      input HelmholtzDerivs f "Dimensionless derivatives of Helmholtz function";
      output PhaseBoundaryProperties sat "Phase boundary property record";
  protected
      SI.Pressure p "Pressure";
    algorithm
      p := f.R*f.d*f.T*f.delta*f.fdelta;
      sat.d := f.d;
      sat.h := f.R*f.T*(f.tau*f.ftau + f.delta*f.fdelta);
      sat.s := f.R*(f.tau*f.ftau - f.f);
      sat.u := f.R*f.T*f.tau*f.ftau;
      sat.cp := f.R*(-f.tau*f.tau*f.ftautau + (f.delta*f.fdelta - f.delta*f.tau*f.fdeltatau)
        ^2/(2*f.delta*f.fdelta + f.delta*f.delta*f.fdeltadelta));
      sat.cv := f.R*(-f.tau*f.tau*f.ftautau);
      sat.pt := f.R*f.d*f.delta*(f.fdelta - f.tau*f.fdeltatau);
      sat.pd := f.R*f.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
    end helmholtzToBoundaryProps;

    function cv2Phase
    "Compute isochoric specific heat capacity inside the two-phase region"

      extends Modelica.Icons.Function;
      input PhaseBoundaryProperties liq "Properties on the boiling curve";
      input PhaseBoundaryProperties vap "Properties on the condensation curve";
      input SI.MassFraction x "Vapour mass fraction";
      input SI.Temperature T "Temperature";
      input SI.Pressure p "Properties";
      output SI.SpecificHeatCapacity cv "Isochoric specific heat capacity";
  protected
      Real dpT "Derivative of pressure w.r.t. temperature";
      Real dxv "Derivative of vapour mass fraction w.r.t. specific volume";
      Real dvTl "Derivative of liquid specific volume w.r.t. temperature";
      Real dvTv "Derivative of vapour specific volume w.r.t. temperature";
      Real duTl "Derivative of liquid specific inner energy w.r.t. temperature";
      Real duTv "Derivative of vapour specific inner energy w.r.t. temperature";
      Real dxt "Derivative of vapour mass fraction w.r.t. temperature";
    algorithm
      dxv := if (liq.d <> vap.d) then liq.d*vap.d/(liq.d - vap.d) else 0.0;
      dpT := (vap.s - liq.s)*dxv;
      // wrong at critical point
      dvTl := (liq.pt - dpT)/liq.pd/liq.d/liq.d;
      dvTv := (vap.pt - dpT)/vap.pd/vap.d/vap.d;
      dxt := -dxv*(dvTl + x*(dvTv - dvTl));
      duTl := liq.cv + (T*liq.pt - p)*dvTl;
      duTv := vap.cv + (T*vap.pt - p)*dvTv;
      cv := duTl + x*(duTv - duTl) + dxt*(vap.u - liq.u);
    end cv2Phase;

    function Helmholtz_ph
    "Function to calculate analytic derivatives for computing d and t given p and h"
      extends Modelica.Icons.Function;
      input HelmholtzDerivs f "Dimensionless derivatives of Helmholtz function";
      output NewtonDerivatives_ph nderivs
      "Derivatives for Newton iteration to calculate d and t from p and h";
  protected
      SI.SpecificHeatCapacity cv "Isochoric heat capacity";
    algorithm
      cv := -f.R*(f.tau*f.tau*f.ftautau);
      nderivs.p := f.d*f.R*f.T*f.delta*f.fdelta;
      nderivs.h := f.R*f.T*(f.tau*f.ftau + f.delta*f.fdelta);
      nderivs.pd := f.R*f.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
      nderivs.pt := f.R*f.d*f.delta*(f.fdelta - f.tau*f.fdeltatau);
      nderivs.ht := cv + nderivs.pt/f.d;
      nderivs.hd := (nderivs.pd - f.T*nderivs.pt/f.d)/f.d;
    end Helmholtz_ph;

    function Helmholtz_pT
    "Function to calculate analytic derivatives for computing d and t given p and t"

      extends Modelica.Icons.Function;
      input HelmholtzDerivs f "Dimensionless derivatives of Helmholtz function";
      output NewtonDerivatives_pT nderivs
      "Derivatives for Newton iteration to compute d and t from p and t";
    algorithm
      nderivs.p := f.d*f.R*f.T*f.delta*f.fdelta;
      nderivs.pd := f.R*f.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
    end Helmholtz_pT;

    function Helmholtz_ps
    "Function to calculate analytic derivatives for computing d and t given p and s"

      extends Modelica.Icons.Function;
      input HelmholtzDerivs f "Dimensionless derivatives of Helmholtz function";
      output NewtonDerivatives_ps nderivs
      "Derivatives for Newton iteration to compute d and t from p and s";
  protected
      SI.SpecificHeatCapacity cv "Isochoric heat capacity";
    algorithm
      cv := -f.R*(f.tau*f.tau*f.ftautau);
      nderivs.p := f.d*f.R*f.T*f.delta*f.fdelta;
      nderivs.s := f.R*(f.tau*f.ftau - f.f);
      nderivs.pd := f.R*f.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
      nderivs.pt := f.R*f.d*f.delta*(f.fdelta - f.tau*f.fdeltatau);
      nderivs.st := cv/f.T;
      nderivs.sd := -nderivs.pt/(f.d*f.d);
    end Helmholtz_ps;

    function smoothStep
    "Approximation of a general step, such that the characteristic is continuous and differentiable"
      extends Modelica.Icons.Function;
      input Real x "Abscissa value";
      input Real y1 "Ordinate value for x > 0";
      input Real y2 "Ordinate value for x < 0";
      input Real x_small(min=0) = 1e-5
      "Approximation of step for -x_small <= x <= x_small; x_small > 0 required";
      output Real y
      "Ordinate value to approximate y = if x > 0 then y1 else y2";
    algorithm
      y := smooth(1, if x > x_small then y1 else if x < -x_small then y2 else if
        abs(x_small) > 0 then (x/x_small)*((x/x_small)^2 - 3)*(y2 - y1)/4 + (y1
         + y2)/2 else (y1 + y2)/2);

      annotation (
        Inline=true,
        smoothOrder=1,
        Documentation(revisions="<html>
<ul>
<li><i>April 29, 2008</i>
    by <a href=\"mailto:Martin.Otter@DLR.de\">Martin Otter</a>:<br>
    Designed and implemented.</li>
<li><i>August 12, 2008</i>
    by <a href=\"mailto:Michael.Sielemann@dlr.de\">Michael Sielemann</a>:<br>
    Minor modification to cover the limit case <code>x_small -> 0</code> without division by zero.</li>
</ul>
</html>",   info="<html>
<p>
This function is used to approximate the equation
</p>
<pre>
    y = <b>if</b> x &gt; 0 <b>then</b> y1 <b>else</b> y2;
</pre>

<p>
by a smooth characteristic, so that the expression is continuous and differentiable:
</p>

<pre>
   y = <b>smooth</b>(1, <b>if</b> x &gt;  x_small <b>then</b> y1 <b>else</b>
                 <b>if</b> x &lt; -x_small <b>then</b> y2 <b>else</b> f(y1, y2));
</pre>

<p>
In the region -x_small &lt; x &lt; x_small a 2nd order polynomial is used
for a smooth transition from y1 to y2.
</p>

<p>
If <b>mass fractions</b> X[:] are approximated with this function then this can be performed
for all <b>nX</b> mass fractions, instead of applying it for nX-1 mass fractions and computing
the last one by the mass fraction constraint sum(X)=1. The reason is that the approximating function has the
property that sum(X) = 1, provided sum(X_a) = sum(X_b) = 1
(and y1=X_a[i], y2=X_b[i]).
This can be shown by evaluating the approximating function in the abs(x) &lt; x_small
region (otherwise X is either X_a or X_b):
</p>

<pre>
    X[1]  = smoothStep(x, X_a[1] , X_b[1] , x_small);
    X[2]  = smoothStep(x, X_a[2] , X_b[2] , x_small);
       ...
    X[nX] = smoothStep(x, X_a[nX], X_b[nX], x_small);
</pre>

<p>
or
</p>

<pre>
    X[1]  = c*(X_a[1]  - X_b[1])  + (X_a[1]  + X_b[1])/2
    X[2]  = c*(X_a[2]  - X_b[2])  + (X_a[2]  + X_b[2])/2;
       ...
    X[nX] = c*(X_a[nX] - X_b[nX]) + (X_a[nX] + X_b[nX])/2;
    c     = (x/x_small)*((x/x_small)^2 - 3)/4
</pre>

<p>
Summing all mass fractions together results in
</p>

<pre>
    sum(X) = c*(sum(X_a) - sum(X_b)) + (sum(X_a) + sum(X_b))/2
           = c*(1 - 1) + (1 + 1)/2
           = 1
</pre>
</html>"));
    end smoothStep;
    annotation (Documentation(info="<HTML><h4>Package description</h4>
      <p>Package Modelica.Media.Common provides records and functions shared by many of the property sub-packages.
      High accuracy fluid property models share a lot of common structure, even if the actual models are different.
      Common data structures and computations shared by these property models are collected in this library.
   </p>

</html>",   revisions="<html>
      <ul>
      <li>First implemented: <i>July, 2000</i>
      by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>
      for the ThermoFluid Library with help from Jonas Eborn and Falko Jens Wagner
      </li>
      <li>Code reorganization, enhanced documentation, additional functions: <i>December, 2002</i>
      by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a> and move to Modelica
                            properties library.</li>
      <li>Inclusion into Modelica.Media: September 2003 </li>
      </ul>

      <address>Author: Hubertus Tummescheit, <br>
      Lund University<br>
      Department of Automatic Control<br>
      Box 118, 22100 Lund, Sweden<br>
      email: hubertus@control.lth.se
      </address>
</html>"));
  end Common;

    package Water "Medium models for water"
    extends Modelica.Icons.VariantsPackage;
    import Modelica.Media.Water.ConstantPropertyLiquidWater.simpleWaterConstants;

    constant Modelica.Media.Interfaces.Types.TwoPhase.FluidConstants[1]
      waterConstants(
      each chemicalFormula="H2O",
      each structureFormula="H2O",
      each casRegistryNumber="7732-18-5",
      each iupacName="oxidane",
      each molarMass=0.018015268,
      each criticalTemperature=647.096,
      each criticalPressure=22064.0e3,
      each criticalMolarVolume=1/322.0*0.018015268,
      each normalBoilingPoint=373.124,
      each meltingPoint=273.15,
      each triplePointTemperature=273.16,
      each triplePointPressure=611.657,
      each acentricFactor=0.344,
      each dipoleMoment=1.8,
      each hasCriticalData=true);

    package ConstantPropertyLiquidWater
      "Water: Simple liquid water medium (incompressible, constant data)"

      constant Modelica.Media.Interfaces.Types.Basic.FluidConstants[1]
        simpleWaterConstants(
        each chemicalFormula="H2O",
        each structureFormula="H2O",
        each casRegistryNumber="7732-18-5",
        each iupacName="oxidane",
        each molarMass=0.018015268);
      extends Interfaces.PartialSimpleMedium(
        mediumName="SimpleLiquidWater",
        cp_const=4184,
        cv_const=4184,
        d_const=995.586,
        eta_const=1.e-3,
        lambda_const=0.598,
        a_const=1484,
        T_min=Cv.from_degC(-1),
        T_max=Cv.from_degC(130),
        T0=273.15,
        MM_const=0.018015268,
        fluidConstants=simpleWaterConstants);
      annotation (Documentation(info="<html>

</html>"));
    end ConstantPropertyLiquidWater;

    package StandardWater = WaterIF97_ph
      "Water using the IF97 standard, explicit in p and h. Recommended for most applications";

    package WaterIF97_ph "Water using the IF97 standard, explicit in p and h"
      extends WaterIF97_base(
        ThermoStates=Modelica.Media.Interfaces.Choices.IndependentVariables.ph,
        final ph_explicit=true,
        final dT_explicit=false,
        final pT_explicit=false,
        smoothModel=false,
        onePhase=false);
      annotation (Documentation(info="<html>

</html>"));
    end WaterIF97_ph;

    partial package WaterIF97_base
      "Water: Steam properties as defined by IAPWS/IF97 standard"

      extends Interfaces.PartialTwoPhaseMedium(
        mediumName="WaterIF97",
        substanceNames={"water"},
        singleState=false,
        SpecificEnthalpy(start=1.0e5, nominal=5.0e5),
        Density(start=150, nominal=500),
        AbsolutePressure(
          start=50e5,
          nominal=10e5,
          min=611.657,
          max=100e6),
        Temperature(
          start=500,
          nominal=500,
          min=273.15,
          max=2273.15),
        smoothModel=false,
        onePhase=false,
        fluidConstants=waterConstants);

      redeclare record extends ThermodynamicState "Thermodynamic state"
        SpecificEnthalpy h "Specific enthalpy";
        Density d "Density";
        Temperature T "Temperature";
        AbsolutePressure p "Pressure";
      end ThermodynamicState;

      constant Boolean ph_explicit
        "True if explicit in pressure and specific enthalpy";
      constant Boolean dT_explicit
        "True if explicit in density and temperature";
      constant Boolean pT_explicit
        "True if explicit in pressure and temperature";

      redeclare replaceable model extends BaseProperties(
        h(stateSelect=if ph_explicit and preferredMediumStates then StateSelect.prefer
               else StateSelect.default),
        d(stateSelect=if dT_explicit and preferredMediumStates then StateSelect.prefer
               else StateSelect.default),
        T(stateSelect=if (pT_explicit or dT_explicit) and preferredMediumStates
               then StateSelect.prefer else StateSelect.default),
        p(stateSelect=if (pT_explicit or ph_explicit) and preferredMediumStates
               then StateSelect.prefer else StateSelect.default))
        "Base properties of water"
        Integer phase(
          min=0,
          max=2,
          start=1,
          fixed=false) "2 for two-phase, 1 for one-phase, 0 if not known";
      equation
        MM = fluidConstants[1].molarMass;
        if smoothModel then
          if onePhase then
            phase = 1;
            if ph_explicit then
              assert(((h < bubbleEnthalpy(sat) or h > dewEnthalpy(sat)) or p >
                fluidConstants[1].criticalPressure),
                "With onePhase=true this model may only be called with one-phase states h < hl or h > hv!"
                 + "(p = " + String(p) + ", h = " + String(h) + ")");
            else
              if dT_explicit then
                assert(not ((d < bubbleDensity(sat) and d > dewDensity(sat)) and T
                   < fluidConstants[1].criticalTemperature),
                  "With onePhase=true this model may only be called with one-phase states d > dl or d < dv!"
                   + "(d = " + String(d) + ", T = " + String(T) + ")");
              end if;
            end if;
          else
            phase = 0;
          end if;
        else
          if ph_explicit then
            phase = if ((h < bubbleEnthalpy(sat) or h > dewEnthalpy(sat)) or p >
              fluidConstants[1].criticalPressure) then 1 else 2;
          elseif dT_explicit then
            phase = if not ((d < bubbleDensity(sat) and d > dewDensity(sat)) and T
               < fluidConstants[1].criticalTemperature) then 1 else 2;
          else
            phase = 1;
            //this is for the one-phase only case pT
          end if;
        end if;
        if dT_explicit then
          p = pressure_dT(
                d,
                T,
                phase);
          h = specificEnthalpy_dT(
                d,
                T,
                phase);
          sat.Tsat = T;
          sat.psat = saturationPressure(T);
        elseif ph_explicit then
          d = density_ph(
                p,
                h,
                phase);
          T = temperature_ph(
                p,
                h,
                phase);
          sat.Tsat = saturationTemperature(p);
          sat.psat = p;
        else
          h = specificEnthalpy_pT(p, T);
          d = density_pT(p, T);
          sat.psat = p;
          sat.Tsat = saturationTemperature(p);
        end if;
        u = h - p/d;
        R = Modelica.Constants.R/fluidConstants[1].molarMass;
        h = state.h;
        p = state.p;
        T = state.T;
        d = state.d;
        phase = state.phase;
      end BaseProperties;

      redeclare function density_ph
        "Computes density as a function of pressure and specific enthalpy"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := IF97_Utilities.rho_ph(
              p,
              h,
              phase);
        annotation (Inline=true);
      end density_ph;

      redeclare function temperature_ph
        "Computes temperature as a function of pressure and specific enthalpy"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output Temperature T "Temperature";
      algorithm
        T := IF97_Utilities.T_ph(
              p,
              h,
              phase);
        annotation (Inline=true);
      end temperature_ph;

      redeclare function temperature_ps
        "Compute temperature from pressure and specific enthalpy"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output Temperature T "Temperature";
      algorithm
        T := IF97_Utilities.T_ps(
              p,
              s,
              phase);
        annotation (Inline=true);
      end temperature_ps;

      redeclare function density_ps
        "Computes density as a function of pressure and specific enthalpy"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := IF97_Utilities.rho_ps(
              p,
              s,
              phase);
        annotation (Inline=true);
      end density_ps;

      redeclare function pressure_dT
        "Computes pressure as a function of density and temperature"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output AbsolutePressure p "Pressure";
      algorithm
        p := IF97_Utilities.p_dT(
              d,
              T,
              phase);
        annotation (Inline=true);
      end pressure_dT;

      redeclare function specificEnthalpy_dT
        "Computes specific enthalpy as a function of density and temperature"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := IF97_Utilities.h_dT(
              d,
              T,
              phase);
        annotation (Inline=true);
      end specificEnthalpy_dT;

      redeclare function specificEnthalpy_pT
        "Computes specific enthalpy as a function of pressure and temperature"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := IF97_Utilities.h_pT(p, T);
        annotation (Inline=true);
      end specificEnthalpy_pT;

      redeclare function specificEnthalpy_ps
        "Computes specific enthalpy as a function of pressure and temperature"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := IF97_Utilities.h_ps(
              p,
              s,
              phase);
        annotation (Inline=true);
      end specificEnthalpy_ps;

      redeclare function density_pT
        "Computes density as a function of pressure and temperature"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input FixedPhase phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
        output Density d "Density";
      algorithm
        d := IF97_Utilities.rho_pT(p, T);
        annotation (Inline=true);
      end density_pT;

      redeclare function extends setDewState
        "Set the thermodynamic state on the dew line"
      algorithm
        state := ThermodynamicState(
              phase=phase,
              p=sat.psat,
              T=sat.Tsat,
              h=dewEnthalpy(sat),
              d=dewDensity(sat));
        annotation (Inline=true);
      end setDewState;

      redeclare function extends setBubbleState
        "Set the thermodynamic state on the bubble line"
      algorithm
        state := ThermodynamicState(
              phase=phase,
              p=sat.psat,
              T=sat.Tsat,
              h=bubbleEnthalpy(sat),
              d=bubbleDensity(sat));
        annotation (Inline=true);
      end setBubbleState;

      redeclare function extends dynamicViscosity "Dynamic viscosity of water"
      algorithm
        eta := IF97_Utilities.dynamicViscosity(
              state.d,
              state.T,
              state.p,
              state.phase);
        annotation (Inline=true);
      end dynamicViscosity;

      redeclare function extends thermalConductivity
        "Thermal conductivity of water"
      algorithm
        lambda := IF97_Utilities.thermalConductivity(
              state.d,
              state.T,
              state.p,
              state.phase);
        annotation (Inline=true);
      end thermalConductivity;

      redeclare function extends surfaceTension
        "Surface tension in two phase region of water"
      algorithm
        sigma := IF97_Utilities.surfaceTension(sat.Tsat);
        annotation (Inline=true);
      end surfaceTension;

      redeclare function extends pressure "Return pressure of ideal gas"
      algorithm
        p := state.p;
        annotation (Inline=true);
      end pressure;

      redeclare function extends temperature "Return temperature of ideal gas"
      algorithm
        T := state.T;
        annotation (Inline=true);
      end temperature;

      redeclare function extends density "Return density of ideal gas"
      algorithm
        d := state.d;
        annotation (Inline=true);
      end density;

      redeclare function extends specificEnthalpy "Return specific enthalpy"
        extends Modelica.Icons.Function;
      algorithm
        h := state.h;
        annotation (Inline=true);
      end specificEnthalpy;

      redeclare function extends specificInternalEnergy
        "Return specific internal energy"
        extends Modelica.Icons.Function;
      algorithm
        u := state.h - state.p/state.d;
        annotation (Inline=true);
      end specificInternalEnergy;

      redeclare function extends specificGibbsEnergy
        "Return specific Gibbs energy"
        extends Modelica.Icons.Function;
      algorithm
        g := state.h - state.T*specificEntropy(state);
        annotation (Inline=true);
      end specificGibbsEnergy;

      redeclare function extends specificHelmholtzEnergy
        "Return specific Helmholtz energy"
        extends Modelica.Icons.Function;
      algorithm
        f := state.h - state.p/state.d - state.T*specificEntropy(state);
        annotation (Inline=true);
      end specificHelmholtzEnergy;

      redeclare function extends specificEntropy "Specific entropy of water"
      algorithm
        s := if dT_explicit then IF97_Utilities.s_dT(
              state.d,
              state.T,
              state.phase) else if pT_explicit then IF97_Utilities.s_pT(state.p,
          state.T) else IF97_Utilities.s_ph(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true);
      end specificEntropy;

      redeclare function extends specificHeatCapacityCp
        "Specific heat capacity at constant pressure of water"
      algorithm
        cp := if dT_explicit then IF97_Utilities.cp_dT(
              state.d,
              state.T,
              state.phase) else if pT_explicit then IF97_Utilities.cp_pT(state.p,
          state.T) else IF97_Utilities.cp_ph(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true, Documentation(info="<html>
                                <p>In the two phase region this function returns the interpolated heat capacity between the
                                liquid and vapour state heat capacities.</p>
                                </html>"));
      end specificHeatCapacityCp;

      redeclare function extends specificHeatCapacityCv
        "Specific heat capacity at constant volume of water"
      algorithm
        cv := if dT_explicit then IF97_Utilities.cv_dT(
              state.d,
              state.T,
              state.phase) else if pT_explicit then IF97_Utilities.cv_pT(state.p,
          state.T) else IF97_Utilities.cv_ph(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true);
      end specificHeatCapacityCv;

      redeclare function extends isentropicExponent
        "Return isentropic exponent"
      algorithm
        gamma := if dT_explicit then IF97_Utilities.isentropicExponent_dT(
              state.d,
              state.T,
              state.phase) else if pT_explicit then
          IF97_Utilities.isentropicExponent_pT(state.p, state.T) else
          IF97_Utilities.isentropicExponent_ph(
              state.p,
              state.h,
              state.phase);
      end isentropicExponent;

      redeclare function extends isothermalCompressibility
        "Isothermal compressibility of water"
      algorithm
        kappa := if dT_explicit then IF97_Utilities.kappa_dT(
              state.d,
              state.T,
              state.phase) else if pT_explicit then IF97_Utilities.kappa_pT(state.p,
          state.T) else IF97_Utilities.kappa_ph(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true);
      end isothermalCompressibility;

      redeclare function extends isobaricExpansionCoefficient
        "Isobaric expansion coefficient of water"
      algorithm
        beta := if dT_explicit then IF97_Utilities.beta_dT(
              state.d,
              state.T,
              state.phase) else if pT_explicit then IF97_Utilities.beta_pT(state.p,
          state.T) else IF97_Utilities.beta_ph(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true);
      end isobaricExpansionCoefficient;

      redeclare function extends velocityOfSound
        "Return velocity of sound as a function of the thermodynamic state record"
      algorithm
        a := if dT_explicit then IF97_Utilities.velocityOfSound_dT(
              state.d,
              state.T,
              state.phase) else if pT_explicit then
          IF97_Utilities.velocityOfSound_pT(state.p, state.T) else
          IF97_Utilities.velocityOfSound_ph(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true);
      end velocityOfSound;

      redeclare function extends isentropicEnthalpy "Compute h(p,s)"
      algorithm
        h_is := IF97_Utilities.isentropicEnthalpy(
              p_downstream,
              specificEntropy(refState),
              0);
        annotation (Inline=true);
      end isentropicEnthalpy;

      redeclare function extends density_derh_p
        "Density derivative by specific enthalpy"
      algorithm
        ddhp := IF97_Utilities.ddhp(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true);
      end density_derh_p;

      redeclare function extends density_derp_h
        "Density derivative by pressure"
      algorithm
        ddph := IF97_Utilities.ddph(
              state.p,
              state.h,
              state.phase);
        annotation (Inline=true);
      end density_derp_h;

      //   redeclare function extends density_derT_p
      //     "Density derivative by temperature"
      //   algorithm
      //     ddTp := IF97_Utilities.ddTp(state.p, state.h, state.phase);
      //   end density_derT_p;
      //
      //   redeclare function extends density_derp_T
      //     "Density derivative by pressure"
      //   algorithm
      //     ddpT := IF97_Utilities.ddpT(state.p, state.h, state.phase);
      //   end density_derp_T;

      redeclare function extends bubbleEnthalpy
        "Boiling curve specific enthalpy of water"
      algorithm
        hl := IF97_Utilities.BaseIF97.Regions.hl_p(sat.psat);
        annotation (Inline=true);
      end bubbleEnthalpy;

      redeclare function extends dewEnthalpy
        "Dew curve specific enthalpy of water"
      algorithm
        hv := IF97_Utilities.BaseIF97.Regions.hv_p(sat.psat);
        annotation (Inline=true);
      end dewEnthalpy;

      redeclare function extends bubbleEntropy
        "Boiling curve specific entropy of water"
      algorithm
        sl := IF97_Utilities.BaseIF97.Regions.sl_p(sat.psat);
        annotation (Inline=true);
      end bubbleEntropy;

      redeclare function extends dewEntropy
        "Dew curve specific entropy of water"
      algorithm
        sv := IF97_Utilities.BaseIF97.Regions.sv_p(sat.psat);
        annotation (Inline=true);
      end dewEntropy;

      redeclare function extends bubbleDensity
        "Boiling curve specific density of water"
      algorithm
        dl := if ph_explicit then IF97_Utilities.BaseIF97.Regions.rhol_p(sat.psat)
           else IF97_Utilities.BaseIF97.Regions.rhol_T(sat.Tsat);
        annotation (Inline=true);
      end bubbleDensity;

      redeclare function extends dewDensity
        "Dew curve specific density of water"
      algorithm
        dv := if ph_explicit or pT_explicit then
          IF97_Utilities.BaseIF97.Regions.rhov_p(sat.psat) else
          IF97_Utilities.BaseIF97.Regions.rhov_T(sat.Tsat);
        annotation (Inline=true);
      end dewDensity;

      redeclare function extends saturationTemperature
        "Saturation temperature of water"
      algorithm
        T := IF97_Utilities.BaseIF97.Basic.tsat(p);
        annotation (Inline=true);
      end saturationTemperature;

      redeclare function extends saturationTemperature_derp
        "Derivative of saturation temperature w.r.t. pressure"
      algorithm
        dTp := IF97_Utilities.BaseIF97.Basic.dtsatofp(p);
        annotation (Inline=true);
      end saturationTemperature_derp;

      redeclare function extends saturationPressure
        "Saturation pressure of water"
      algorithm
        p := IF97_Utilities.BaseIF97.Basic.psat(T);
        annotation (Inline=true);
      end saturationPressure;

      redeclare function extends dBubbleDensity_dPressure
        "Bubble point density derivative"
      algorithm
        ddldp := IF97_Utilities.BaseIF97.Regions.drhol_dp(sat.psat);
        annotation (Inline=true);
      end dBubbleDensity_dPressure;

      redeclare function extends dDewDensity_dPressure
        "Dew point density derivative"
      algorithm
        ddvdp := IF97_Utilities.BaseIF97.Regions.drhov_dp(sat.psat);
        annotation (Inline=true);
      end dDewDensity_dPressure;

      redeclare function extends dBubbleEnthalpy_dPressure
        "Bubble point specific enthalpy derivative"
      algorithm
        dhldp := IF97_Utilities.BaseIF97.Regions.dhl_dp(sat.psat);
        annotation (Inline=true);
      end dBubbleEnthalpy_dPressure;

      redeclare function extends dDewEnthalpy_dPressure
        "Dew point specific enthalpy derivative"
      algorithm
        dhvdp := IF97_Utilities.BaseIF97.Regions.dhv_dp(sat.psat);
        annotation (Inline=true);
      end dDewEnthalpy_dPressure;

      redeclare function extends setState_dTX
        "Return thermodynamic state of water as function of d and T"
      algorithm
        state := ThermodynamicState(
              d=d,
              T=T,
              phase=0,
              h=specificEnthalpy_dT(d, T),
              p=pressure_dT(d, T));
        annotation (Inline=true);
      end setState_dTX;

      redeclare function extends setState_phX
        "Return thermodynamic state of water as function of p and h"
      algorithm
        state := ThermodynamicState(
              d=density_ph(p, h),
              T=temperature_ph(p, h),
              phase=0,
              h=h,
              p=p);
        annotation (Inline=true);
      end setState_phX;

      redeclare function extends setState_psX
        "Return thermodynamic state of water as function of p and s"
      algorithm
        state := ThermodynamicState(
              d=density_ps(p, s),
              T=temperature_ps(p, s),
              phase=0,
              h=specificEnthalpy_ps(p, s),
              p=p);
        annotation (Inline=true);
      end setState_psX;

      redeclare function extends setState_pTX
        "Return thermodynamic state of water as function of p and T"
      algorithm
        state := ThermodynamicState(
              d=density_pT(p, T),
              T=T,
              phase=1,
              h=specificEnthalpy_pT(p, T),
              p=p);
        annotation (Inline=true);
      end setState_pTX;

      redeclare function extends setSmoothState
        "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
        import Modelica.Media.Common.smoothStep;
      algorithm
        state := ThermodynamicState(
              p=smoothStep(
                x,
                state_a.p,
                state_b.p,
                x_small),
              h=smoothStep(
                x,
                state_a.h,
                state_b.h,
                x_small),
              d=density_ph(smoothStep(
                x,
                state_a.p,
                state_b.p,
                x_small), smoothStep(
                x,
                state_a.h,
                state_b.h,
                x_small)),
              T=temperature_ph(smoothStep(
                x,
                state_a.p,
                state_b.p,
                x_small), smoothStep(
                x,
                state_a.h,
                state_b.h,
                x_small)),
              phase=0);
        annotation (Inline=true);
      end setSmoothState;
      annotation (Documentation(info="<HTML>
<p>
This model calculates medium properties
for water in the <b>liquid</b>, <b>gas</b> and <b>two phase</b> regions
according to the IAPWS/IF97 standard, i.e., the accepted industrial standard
and best compromise between accuracy and computation time.
For more details see <a href=\"modelica://Modelica.Media.Water.IF97_Utilities\">
Modelica.Media.Water.IF97_Utilities</a>. Three variable pairs can be the
independent variables of the model:
</p>
<ol>
<li>Pressure <b>p</b> and specific enthalpy <b>h</b> are the most natural choice for general applications. This is the recommended choice for most general purpose applications, in particular for power plants.</li>
<li>Pressure <b>p</b> and temperature <b>T</b> are the most natural choice for applications where water is always in the same phase, both for liquid water and steam.</li>
<li>Density <b>d</b> and temperature <b>T</b> are explicit variables of the Helmholtz function in the near-critical region and can be the best choice for applications with super-critical or near-critical states.</li>
</ol>
<p>
The following quantities are always computed:
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>Variable</b></td>
      <td valign=\"top\"><b>Unit</b></td>
      <td valign=\"top\"><b>Description</b></td></tr>
  <tr><td valign=\"top\">T</td>
      <td valign=\"top\">K</td>
      <td valign=\"top\">temperature</td></tr>
  <tr><td valign=\"top\">u</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">specific internal energy</td></tr>
  <tr><td valign=\"top\">d</td>
      <td valign=\"top\">kg/m^3</td>
      <td valign=\"top\">density</td></tr>
  <tr><td valign=\"top\">p</td>
      <td valign=\"top\">Pa</td>
      <td valign=\"top\">pressure</td></tr>
  <tr><td valign=\"top\">h</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">specific enthalpy</td></tr>
</table>
<p>
In some cases additional medium properties are needed.
A component that needs these optional properties has to call
one of the functions listed in
<a href=\"modelica://Modelica.Media.UsersGuide.MediumUsage.OptionalProperties\">
Modelica.Media.UsersGuide.MediumUsage.OptionalProperties</a> and in
<a href=\"modelica://Modelica.Media.UsersGuide.MediumUsage.TwoPhase\">
Modelica.Media.UsersGuide.MediumUsage.TwoPhase</a>.
</p>
<p>Many further properties can be computed. Using the well-known Bridgman's Tables, all first partial derivatives of the standard thermodynamic variables can be computed easily.</p>
</html>"));
    end WaterIF97_base;

      package IF97_Utilities
      "Low level and utility computation for high accuracy water properties according to the IAPWS/IF97 standard"
        extends Modelica.Icons.UtilitiesPackage;

        package BaseIF97
        "Modelica Physical Property Model: the new industrial formulation IAPWS-IF97"
          extends Modelica.Icons.Package;

          record IterationData
          "Constants for iterations internal to some functions"

            extends Modelica.Icons.Record;
            constant Integer IMAX=50
            "Maximum number of iterations for inverse functions";
            constant Real DELP=1.0e-6 "Maximum iteration error in pressure, Pa";
            constant Real DELS=1.0e-8
            "Maximum iteration error in specific entropy, J/{kg.K}";
            constant Real DELH=1.0e-8
            "Maximum iteration error in specific enthalpy, J/kg";
            constant Real DELD=1.0e-8
            "Maximum iteration error in density, kg/m^3";
          end IterationData;

          record data "Constant IF97 data and region limits"
            extends Modelica.Icons.Record;
            constant SI.SpecificHeatCapacity RH2O=461.526
            "Specific gas constant of water vapour";
            constant SI.MolarMass MH2O=0.01801528 "Molar weight of water";
            constant SI.Temperature TSTAR1=1386.0
            "Normalization temperature for region 1 IF97";
            constant SI.Pressure PSTAR1=16.53e6
            "Normalization pressure for region 1 IF97";
            constant SI.Temperature TSTAR2=540.0
            "Normalization temperature for region 2 IF97";
            constant SI.Pressure PSTAR2=1.0e6
            "Normalization pressure for region 2 IF97";
            constant SI.Temperature TSTAR5=1000.0
            "Normalization temperature for region 5 IF97";
            constant SI.Pressure PSTAR5=1.0e6
            "Normalization pressure for region 5 IF97";
            constant SI.SpecificEnthalpy HSTAR1=2.5e6
            "Normalization specific enthalpy for region 1 IF97";
            constant Real IPSTAR=1.0e-6
            "Normalization pressure for inverse function in region 2 IF97";
            constant Real IHSTAR=5.0e-7
            "Normalization specific enthalpy for inverse function in region 2 IF97";
            constant SI.Temperature TLIMIT1=623.15
            "Temperature limit between regions 1 and 3";
            constant SI.Temperature TLIMIT2=1073.15
            "Temperature limit between regions 2 and 5";
            constant SI.Temperature TLIMIT5=2273.15
            "Upper temperature limit of 5";
            constant SI.Pressure PLIMIT1=100.0e6
            "Upper pressure limit for regions 1, 2 and 3";
            constant SI.Pressure PLIMIT4A=16.5292e6
            "Pressure limit between regions 1 and 2, important for for two-phase (region 4)";
            constant SI.Pressure PLIMIT5=10.0e6
            "Upper limit of valid pressure in region 5";
            constant SI.Pressure PCRIT=22064000.0 "The critical pressure";
            constant SI.Temperature TCRIT=647.096 "The critical temperature";
            constant SI.Density DCRIT=322.0 "The critical density";
            constant SI.SpecificEntropy SCRIT=4412.02148223476
            "The calculated specific entropy at the critical point";
            constant SI.SpecificEnthalpy HCRIT=2087546.84511715
            "The calculated specific enthalpy at the critical point";
            constant Real[5] n=array(
                    0.34805185628969e3,
                    -0.11671859879975e1,
                    0.10192970039326e-2,
                    0.57254459862746e3,
                    0.13918839778870e2)
            "Polynomial coefficients for boundary between regions 2 and 3";
            annotation (Documentation(info="<HTML>
 <h4>Record description</h4>
                           <p>Constants needed in the international steam properties IF97.
                           SCRIT and HCRIT are calculated from Helmholtz function for region 3.</p>
<h4>Version Info and Revision history
</h4>
<ul>
<li>First implemented: <i>July, 2000</i>
       by Hubertus Tummescheit
       </li>
</ul>
 <address>Author: Hubertus Tummescheit, <br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
 </address>
<ul>
 <li>Initial version: July 2000</li>
 <li>Documentation added: December 2002</li>
</ul>
</html>"));
          end data;

          record triple "Triple point data"
            extends Modelica.Icons.Record;
            constant SI.Temperature Ttriple=273.16
            "The triple point temperature";
            constant SI.Pressure ptriple=611.657 "The triple point temperature";
            constant SI.Density dltriple=999.792520031617642
            "The triple point liquid density";
            constant SI.Density dvtriple=0.485457572477861372e-2
            "The triple point vapour density";
            annotation (Documentation(info="<HTML>
 <h4>Record description</h4>
 <p>Vapour/liquid/ice triple point data for IF97 steam properties.</p>
<h4>Version Info and Revision history
</h4>
<ul>
<li>First implemented: <i>July, 2000</i>
       by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>
       </li>
</ul>
 <address>Author: Hubertus Tummescheit, <br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
 </address>
<ul>
 <li>Initial version: July 2000</li>
 <li>Documentation added: December 2002</li>
</ul>
</html>"));
          end triple;

          package Regions
          "Functions to find the current region for given pairs of input variables"
            extends Modelica.Icons.Package;

            function boundary23ofT
            "Boundary function for region boundary between regions 2 and 3 (input temperature)"

              extends Modelica.Icons.Function;
              input SI.Temperature t "Temperature (K)";
              output SI.Pressure p "Pressure";
          protected
              constant Real[5] n=data.n;
            algorithm
              p := 1.0e6*(n[1] + t*(n[2] + t*n[3]));
            end boundary23ofT;

            function boundary23ofp
            "Boundary function for region boundary between regions 2 and 3 (input pressure)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.Temperature t "Temperature (K)";
          protected
              constant Real[5] n=data.n;
              Real pi "Dimensionless pressure";
            algorithm
              pi := p/1.0e6;
              assert(p > triple.ptriple,
                "IF97 medium function boundary23ofp called with too low pressure\n"
                 + "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              t := n[4] + ((pi - n[5])/n[3])^0.5;
            end boundary23ofp;

            function hlowerofp5
            "Explicit lower specific enthalpy limit of region 5 as function of pressure"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real pi "Dimensionless pressure";
            algorithm
              pi := p/data.PSTAR5;
              assert(p > triple.ptriple,
                "IF97 medium function hlowerofp5 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              h := 461526.*(9.01505286876203 + pi*(-0.00979043490246092 + (-0.0000203245575263501
                 + 3.36540214679088e-7*pi)*pi));
            end hlowerofp5;

            function hupperofp5
            "Explicit upper specific enthalpy limit of region 5 as function of pressure"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real pi "Dimensionless pressure";
            algorithm
              pi := p/data.PSTAR5;
              assert(p > triple.ptriple,
                "IF97 medium function hupperofp5 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              h := 461526.*(15.9838891400332 + pi*(-0.000489898813722568 + (-5.01510211858761e-8
                 + 7.5006972718273e-8*pi)*pi));
            end hupperofp5;

            function slowerofp5
            "Explicit lower specific entropy limit of region 5 as function of pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real pi "Dimensionless pressure";
            algorithm
              pi := p/data.PSTAR5;
              assert(p > triple.ptriple,
                "IF97 medium function slowerofp5 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              s := 461.526*(18.4296209980112 + pi*(-0.00730911805860036 + (-0.0000168348072093888
                 + 2.09066899426354e-7*pi)*pi) - Modelica.Math.log(pi));
            end slowerofp5;

            function supperofp5
            "Explicit upper specific entropy limit of region 5 as function of pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real pi "Dimensionless pressure";
            algorithm
              pi := p/data.PSTAR5;
              assert(p > triple.ptriple,
                "IF97 medium function supperofp5 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              s := 461.526*(22.7281531474243 + pi*(-0.000656650220627603 + (-1.96109739782049e-8
                 + 2.19979537113031e-8*pi)*pi) - Modelica.Math.log(pi));
            end supperofp5;

            function hlowerofp1
            "Explicit lower specific enthalpy limit of region 1 as function of pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real pi1 "Dimensionless pressure";
              Real[3] o "Vector of auxiliary variables";
            algorithm
              pi1 := 7.1 - p/data.PSTAR1;
              assert(p > triple.ptriple,
                "IF97 medium function hlowerofp1 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              o[1] := pi1*pi1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];

              h := 639675.036*(0.173379420894777 + pi1*(-0.022914084306349 + pi1*(-0.00017146768241932
                 + pi1*(-4.18695814670391e-6 + pi1*(-2.41630417490008e-7 + pi1*(
                1.73545618580828e-11 + o[1]*pi1*(8.43755552264362e-14 + o[2]*o[3]*pi1
                *(5.35429206228374e-35 + o[1]*(-8.12140581014818e-38 + o[1]*o[2]*(-1.43870236842915e-44
                 + pi1*(1.73894459122923e-45 + (-7.06381628462585e-47 +
                9.64504638626269e-49*pi1)*pi1)))))))))));
            end hlowerofp1;

            function hupperofp1
            "Explicit upper specific enthalpy limit of region 1 as function of pressure (meets region 4 saturation pressure curve at 623.15 K)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real pi1 "Dimensionless pressure";
              Real[3] o "Vector of auxiliary variables";
            algorithm
              pi1 := 7.1 - p/data.PSTAR1;
              assert(p > triple.ptriple,
                "IF97 medium function hupperofp1 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              o[1] := pi1*pi1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              h := 639675.036*(2.42896927729349 + pi1*(-0.00141131225285294 + pi1*(
                0.00143759406818289 + pi1*(0.000125338925082983 + pi1*(
                0.0000123617764767172 + pi1*(3.17834967400818e-6 + o[1]*pi1*(
                1.46754947271665e-8 + o[2]*o[3]*pi1*(1.86779322717506e-17 + o[1]*(-4.18568363667416e-19
                 + o[1]*o[2]*(-9.19148577641497e-22 + pi1*(4.27026404402408e-22 + (-6.66749357417962e-23
                 + 3.49930466305574e-24*pi1)*pi1)))))))))));
            end hupperofp1;

            function supperofp1
            "Explicit upper specific entropy limit of region 1 as function of pressure (meets region 4 saturation pressure curve at 623.15 K)"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real pi1 "Dimensionless pressure";
              Real[3] o "Vector of auxiliary variables";
            algorithm
              pi1 := 7.1 - p/data.PSTAR1;
              assert(p > triple.ptriple,
                "IF97 medium function supperofp1 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              o[1] := pi1*pi1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              s := 461.526*(7.28316418503422 + pi1*(0.070602197808399 + pi1*(
                0.0039229343647356 + pi1*(0.000313009170788845 + pi1*(
                0.0000303619398631619 + pi1*(7.46739440045781e-6 + o[1]*pi1*(
                3.40562176858676e-8 + o[2]*o[3]*pi1*(4.21886233340801e-17 + o[1]*(-9.44504571473549e-19
                 + o[1]*o[2]*(-2.06859611434475e-21 + pi1*(9.60758422254987e-22 + (-1.49967810652241e-22
                 + 7.86863124555783e-24*pi1)*pi1)))))))))));
            end supperofp1;

            function hlowerofp2
            "Explicit lower specific enthalpy limit of region 2 as function of pressure (meets region 4 saturation pressure curve at 623.15 K)"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real pi "Dimensionless pressure";
              Real q1 "Auxiliary variable";
              Real q2 "Auxiliary variable";
              Real[18] o "Vector of auxiliary variables";
            algorithm
              pi := p/data.PSTAR2;
              assert(p > triple.ptriple,
                "IF97 medium function hlowerofp2 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              q1 := 572.54459862746 + 31.3220101646784*(-13.91883977887 + pi)^0.5;
              q2 := -0.5 + 540./q1;
              o[1] := q1*q1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              o[4] := pi*pi;
              o[5] := o[4]*o[4];
              o[6] := q2*q2;
              o[7] := o[6]*o[6];
              o[8] := o[6]*o[7];
              o[9] := o[5]*o[5];
              o[10] := o[7]*o[7];
              o[11] := o[9]*o[9];
              o[12] := o[10]*o[10];
              o[13] := o[12]*o[12];
              o[14] := o[7]*q2;
              o[15] := o[6]*q2;
              o[16] := o[10]*o[6];
              o[17] := o[13]*o[6];
              o[18] := o[13]*o[6]*q2;
              h := (4.63697573303507e9 + 3.74686560065793*o[2] + 3.57966647812489e-6*
                o[1]*o[2] + 2.81881548488163e-13*o[3] - 7.64652332452145e7*q1 -
                0.00450789338787835*o[2]*q1 - 1.55131504410292e-9*o[1]*o[2]*q1 + o[1]
                *(2.51383707870341e6 - 4.78198198764471e6*o[10]*o[11]*o[12]*o[13]*o[4]
                 + 49.9651389369988*o[11]*o[12]*o[13]*o[4]*o[5]*o[7] + o[15]*o[4]*(
                1.03746636552761e-13 - 0.00349547959376899*o[16] -
                2.55074501962569e-7*o[8])*o[9] + (-242662.235426958*o[10]*o[12] -
                3.46022402653609*o[16])*o[4]*o[5]*pi + o[4]*(0.109336249381227 -
                2248.08924686956*o[14] - 354742.725841972*o[17] - 24.1331193696374*o[
                6])*pi - 3.09081828396912e-19*o[11]*o[12]*o[5]*o[7]*pi -
                1.24107527851371e-8*o[11]*o[13]*o[4]*o[5]*o[6]*o[7]*pi +
                3.99891272904219*o[5]*o[8]*pi + 0.0641817365250892*o[10]*o[7]*o[9]*pi
                 + pi*(-4444.87643334512 - 75253.6156722047*o[14] - 43051.9020511789*
                o[6] - 22926.6247146068*q2) + o[4]*(-8.23252840892034 -
                3927.0508365636*o[15] - 239.325789467604*o[18] - 76407.3727417716*o[8]
                 - 94.4508644545118*q2) + 0.360567666582363*o[5]*(-0.0161221195808321
                 + q2)*(0.0338039844460968 + q2) + o[11]*(-0.000584580992538624*o[10]
                *o[12]*o[7] + 1.33248030241755e6*o[12]*o[13]*q2) + o[9]*(-7.38502736990986e7
                *o[18] + 0.0000224425477627799*o[6]*o[7]*q2) + o[4]*o[5]*(-2.08438767026518e8
                *o[17] - 0.0000124971648677697*o[6] - 8442.30378348203*o[10]*o[6]*o[7]
                *q2) + o[11]*o[9]*(4.73594929247646e-22*o[10]*o[12]*q2 -
                13.6411358215175*o[10]*o[12]*o[13]*q2 + 5.52427169406836e-10*o[13]*o[
                6]*o[7]*q2) + o[11]*o[5]*(2.67174673301715e-6*o[17] +
                4.44545133805865e-18*o[12]*o[6]*q2 - 50.2465185106411*o[10]*o[13]*o[6]
                *o[7]*q2)))/o[1];
            end hlowerofp2;

            function hupperofp2
            "Explicit upper specific enthalpy limit of region 2 as function of pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real pi "Dimensionless pressure";
              Real[2] o "Vector of auxiliary variables";
            algorithm
              pi := p/data.PSTAR2;
              assert(p > triple.ptriple,
                "IF97 medium function hupperofp2 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              o[1] := pi*pi;
              o[2] := o[1]*o[1]*o[1];
              h := 4.16066337647071e6 + pi*(-4518.48617188327 + pi*(-8.53409968320258
                 + pi*(0.109090430596056 + pi*(-0.000172486052272327 + pi*(
                4.2261295097284e-15 + pi*(-1.27295130636232e-10 + pi*(-3.79407294691742e-25
                 + pi*(7.56960433802525e-23 + pi*(7.16825117265975e-32 + pi*(
                3.37267475986401e-21 + (-7.5656940729795e-74 + o[1]*(-8.00969737237617e-134
                 + (1.6746290980312e-65 + pi*(-3.71600586812966e-69 + pi*(
                8.06630589170884e-129 + (-1.76117969553159e-103 +
                1.88543121025106e-84*pi)*pi)))*o[1]))*o[2]))))))))));
            end hupperofp2;

            function slowerofp2
            "Explicit lower specific entropy limit of region 2 as function of pressure (meets region 4 saturation pressure curve at 623.15 K)"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real pi "Dimensionless pressure";
              Real q1 "Auxiliary variable";
              Real q2 "Auxiliary variable";
              Real[40] o "Vector of auxiliary variables";
            algorithm
              pi := p/data.PSTAR2;
              assert(p > triple.ptriple,
                "IF97 medium function slowerofp2 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              q1 := 572.54459862746 + 31.3220101646784*(-13.91883977887 + pi)^0.5;
              q2 := -0.5 + 540.0/q1;
              o[1] := pi*pi;
              o[2] := o[1]*pi;
              o[3] := o[1]*o[1];
              o[4] := o[1]*o[3]*pi;
              o[5] := q1*q1;
              o[6] := o[5]*q1;
              o[7] := 1/o[5];
              o[8] := 1/q1;
              o[9] := o[5]*o[5];
              o[10] := o[9]*q1;
              o[11] := q2*q2;
              o[12] := o[11]*q2;
              o[13] := o[1]*o[3];
              o[14] := o[11]*o[11];
              o[15] := o[3]*o[3];
              o[16] := o[1]*o[15];
              o[17] := o[11]*o[14];
              o[18] := o[11]*o[14]*q2;
              o[19] := o[3]*pi;
              o[20] := o[14]*o[14];
              o[21] := o[11]*o[20];
              o[22] := o[15]*pi;
              o[23] := o[14]*o[20]*q2;
              o[24] := o[20]*o[20];
              o[25] := o[15]*o[15];
              o[26] := o[25]*o[3];
              o[27] := o[14]*o[24];
              o[28] := o[25]*o[3]*pi;
              o[29] := o[20]*o[24]*q2;
              o[30] := o[15]*o[25];
              o[31] := o[24]*o[24];
              o[32] := o[11]*o[31]*q2;
              o[33] := o[14]*o[31];
              o[34] := o[1]*o[25]*o[3]*pi;
              o[35] := o[11]*o[14]*o[31]*q2;
              o[36] := o[1]*o[25]*o[3];
              o[37] := o[1]*o[25];
              o[38] := o[20]*o[24]*o[31]*q2;
              o[39] := o[14]*q2;
              o[40] := o[11]*o[31];

              s := 461.526*(9.692768600217 + 1.22151969114703e-16*o[10] +
                0.00018948987516315*o[1]*o[11] + 1.6714766451061e-11*o[12]*o[13] +
                0.0039392777243355*o[1]*o[14] - 1.0406965210174e-19*o[14]*o[16] +
                0.043797295650573*o[1]*o[18] - 2.2922076337661e-6*o[18]*o[19] -
                2.0481737692309e-8*o[2] + 0.00003227767723857*o[12]*o[2] +
                0.0015033924542148*o[17]*o[2] - 1.1256211360459e-11*o[15]*o[20] +
                1.0018179379511e-9*o[11]*o[14]*o[16]*o[20] + 1.0234747095929e-13*o[16]
                *o[21] - 1.9809712802088e-8*o[22]*o[23] + 0.0021171472321355*o[13]*o[
                24] - 8.9185845355421e-25*o[26]*o[27] - 1.2790717852285e-8*o[11]*o[3]
                 - 4.8225372718507e-7*o[12]*o[3] - 7.3087610595061e-29*o[11]*o[20]*o[
                24]*o[30] - 0.10693031879409*o[11]*o[24]*o[25]*o[31] +
                4.2002467698208e-6*o[24]*o[26]*o[31] - 5.5414715350778e-17*o[20]*o[30]
                *o[31] + 9.436970724121e-7*o[11]*o[20]*o[24]*o[30]*o[31] +
                23.895741934104*o[13]*o[32] + 0.040668253562649*o[2]*o[32] -
                3.0629316876232e-13*o[26]*o[32] + 0.000026674547914087*o[1]*o[33] +
                8.2311340897998*o[15]*o[33] + 1.2768608934681e-15*o[34]*o[35] +
                0.33662250574171*o[37]*o[38] + 5.905956432427e-18*o[4] +
                0.038946842435739*o[29]*o[4] - 4.88368302964335e-6*o[5] -
                3.34901734177133e6/o[6] + 2.58538448402683e-9*o[6] + 82839.5726841115
                *o[7] - 5446.7940672972*o[8] - 8.40318337484194e-13*o[9] +
                0.0017731742473213*pi + 0.045996013696365*o[11]*pi +
                0.057581259083432*o[12]*pi + 0.05032527872793*o[17]*pi + o[8]*pi*(
                9.63082563787332 - 0.008917431146179*q1) + 0.00811842799898148*q1 +
                0.000033032641670203*o[1]*q2 - 4.3870667284435e-7*o[2]*q2 +
                8.0882908646985e-11*o[14]*o[20]*o[24]*o[25]*q2 + 5.9056029685639e-26*
                o[14]*o[24]*o[28]*q2 + 7.8847309559367e-10*o[3]*q2 -
                3.7826947613457e-6*o[14]*o[24]*o[31]*o[36]*q2 + 1.2621808899101e-6*o[
                11]*o[20]*o[4]*q2 + 540.*o[8]*(10.08665568018 - 0.000033032641670203*
                o[1] - 6.2245802776607e-15*o[10] - 0.015757110897342*o[1]*o[12] -
                5.0144299353183e-11*o[11]*o[13] + 4.1627860840696e-19*o[12]*o[16] -
                0.306581069554011*o[1]*o[17] + 9.0049690883672e-11*o[15]*o[18] +
                0.0000160454534363627*o[17]*o[19] + 4.3870667284435e-7*o[2] -
                0.00009683303171571*o[11]*o[2] + 2.57526266427144e-7*o[14]*o[20]*o[22]
                 - 1.40254511313154e-8*o[16]*o[23] - 2.34560435076256e-9*o[14]*o[20]*
                o[24]*o[25] - 1.24017662339842e-24*o[27]*o[28] - 7.8847309559367e-10*
                o[3] + 1.44676118155521e-6*o[11]*o[3] + 1.90027787547159e-27*o[29]*o[
                30] - 0.000960283724907132*o[1]*o[32] - 296.320827232793*o[15]*o[32]
                 - 4.97975748452559e-14*o[11]*o[14]*o[31]*o[34] +
                2.21658861403112e-15*o[30]*o[35] + 0.000200482822351322*o[14]*o[24]*o[
                31]*o[36] - 19.1874828272775*o[20]*o[24]*o[31]*o[37] -
                0.0000547344301999018*o[30]*o[38] - 0.0090203547252888*o[2]*o[39] -
                0.0000138839897890111*o[21]*o[4] - 0.973671060893475*o[20]*o[24]*o[4]
                 - 836.35096769364*o[13]*o[40] - 1.42338887469272*o[2]*o[40] +
                1.07202609066812e-11*o[26]*o[40] + 0.0000150341259240398*o[5] -
                1.8087714924605e-8*o[6] + 18605.6518987296*o[7] - 306.813232163376*o[
                8] + 1.43632471334824e-11*o[9] + 1.13103675106207e-18*o[5]*o[9] -
                0.017834862292358*pi - 0.172743777250296*o[11]*pi - 0.30195167236758*
                o[39]*pi + o[8]*pi*(-49.6756947920742 + 0.045996013696365*q1) -
                0.0003789797503263*o[1]*q2 - 0.033874355714168*o[11]*o[13]*o[14]*o[20]
                *q2 - 1.0234747095929e-12*o[16]*o[20]*q2 + 1.78371690710842e-23*o[11]
                *o[24]*o[26]*q2 + 2.558143570457e-8*o[3]*q2 + 5.3465159397045*o[24]*o[
                25]*o[31]*q2 - 0.000201611844951398*o[11]*o[14]*o[20]*o[26]*o[31]*q2)
                 - Modelica.Math.log(pi));
            end slowerofp2;

            function supperofp2
            "Explicit upper specific entropy limit of region 2 as function of pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real pi "Dimensionless pressure";
              Real[2] o "Vector of auxiliary variables";
            algorithm
              pi := p/data.PSTAR2;
              assert(p > triple.ptriple,
                "IF97 medium function supperofp2 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              o[1] := pi*pi;
              o[2] := o[1]*o[1]*o[1];
              s := 8505.73409708683 - 461.526*Modelica.Math.log(pi) + pi*(-3.36563543302584
                 + pi*(-0.00790283552165338 + pi*(0.0000915558349202221 + pi*(-1.59634706513e-7
                 + pi*(3.93449217595397e-18 + pi*(-1.18367426347994e-13 + pi*(
                2.72575244843195e-15 + pi*(7.04803892603536e-26 + pi*(
                6.67637687381772e-35 + pi*(3.1377970315132e-24 + (-7.04844558482265e-77
                 + o[1]*(-7.46289531275314e-137 + (1.55998511254305e-68 + pi*(-3.46166288915497e-72
                 + pi*(7.51557618628583e-132 + (-1.64086406733212e-106 +
                1.75648443097063e-87*pi)*pi)))*o[1]))*o[2]*o[2]))))))))));
            end supperofp2;

            function d1n "Density in region 1 as function of p and T"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output SI.Density d "Density";
          protected
              Real pi "Dimensionless pressure";
              Real pi1 "Dimensionless pressure";
              Real tau "Dimensionless temperature";
              Real tau1 "Dimensionless temperature";
              Real gpi "Dimensionless Gibbs-derivative w.r.t. pi";
              Real[11] o "Auxiliary variables";
            algorithm
              pi := p/data.PSTAR1;
              tau := data.TSTAR1/T;
              pi1 := 7.1 - pi;
              tau1 := tau - 1.222;
              o[1] := tau1*tau1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              o[4] := o[1]*o[2];
              o[5] := o[1]*tau1;
              o[6] := o[2]*tau1;
              o[7] := pi1*pi1;
              o[8] := o[7]*o[7];
              o[9] := o[8]*o[8];
              o[10] := o[3]*o[3];
              o[11] := o[10]*o[10];
              gpi := pi1*(pi1*((0.000095038934535162 + o[2]*(8.4812393955936e-6 +
                2.55615384360309e-9*o[4]))/o[2] + pi1*((8.9701127632e-6 + (
                2.60684891582404e-6 + 5.7366919751696e-13*o[2]*o[3])*o[5])/o[6] + pi1
                *(2.02584984300585e-6/o[3] + o[7]*pi1*(o[8]*o[9]*pi1*(o[7]*(o[7]*o[8]
                *(-7.63737668221055e-22/(o[1]*o[11]*o[2]) + pi1*(pi1*(-5.65070932023524e-23
                /(o[11]*o[3]) + (2.99318679335866e-24*pi1)/(o[11]*o[3]*tau1)) +
                3.5842867920213e-22/(o[1]*o[11]*o[2]*tau1))) - 3.33001080055983e-19/(
                o[1]*o[10]*o[2]*o[3]*tau1)) + 1.44400475720615e-17/(o[10]*o[2]*o[3]*
                tau1)) + (1.01874413933128e-8 + 1.39398969845072e-9*o[6])/(o[1]*o[3]*
                tau1))))) + (0.00094368642146534 + o[5]*(0.00060003561586052 + (-0.000095322787813974
                 + o[1]*(8.8283690661692e-6 + 1.45389992595188e-15*o[1]*o[2]*o[3]))*
                tau1))/o[5]) + (-0.00028319080123804 + o[1]*(0.00060706301565874 + o[
                4]*(0.018990068218419 + tau1*(0.032529748770505 + (0.021841717175414
                 + 0.00005283835796993*o[1])*tau1))))/(o[3]*tau1);
              d := p/(data.RH2O*T*pi*gpi);
            end d1n;

            function d2n "Density in region 2 as function of p and T"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output SI.Density d "Density";
          protected
              Real pi "Dimensionless pressure";
              Real tau "Dimensionless temperature";
              Real tau2 "Dimensionless temperature";
              Real gpi "Dimensionless Gibbs-derivative w.r.t. pi";
              Real[12] o "Auxiliary variables";
            algorithm
              pi := p/data.PSTAR2;
              tau := data.TSTAR2/T;
              tau2 := tau - 0.5;
              o[1] := tau2*tau2;
              o[2] := o[1]*tau2;
              o[3] := o[1]*o[1];
              o[4] := o[3]*o[3];
              o[5] := o[4]*o[4];
              o[6] := o[3]*o[4]*o[5]*tau2;
              o[7] := o[3]*o[4]*tau2;
              o[8] := o[1]*o[3]*o[4];
              o[9] := pi*pi;
              o[10] := o[9]*o[9];
              o[11] := o[3]*o[5]*tau2;
              o[12] := o[5]*o[5];
              gpi := (1. + pi*(-0.0017731742473213 + tau2*(-0.017834862292358 + tau2*
                (-0.045996013696365 + (-0.057581259083432 - 0.05032527872793*o[2])*
                tau2)) + pi*(tau2*(-0.000066065283340406 + (-0.0003789797503263 + o[1]
                *(-0.007878555448671 + o[2]*(-0.087594591301146 -
                0.000053349095828174*o[6])))*tau2) + pi*(6.1445213076927e-8 + (
                1.31612001853305e-6 + o[1]*(-0.00009683303171571 + o[2]*(-0.0045101773626444
                 - 0.122004760687947*o[6])))*tau2 + pi*(tau2*(-3.15389238237468e-9 +
                (5.116287140914e-8 + 1.92901490874028e-6*tau2)*tau2) + pi*(
                0.0000114610381688305*o[1]*o[3]*tau2 + pi*(o[2]*(-1.00288598706366e-10
                 + o[7]*(-0.012702883392813 - 143.374451604624*o[1]*o[5]*tau2)) + pi*
                (-4.1341695026989e-17 + o[1]*o[4]*(-8.8352662293707e-6 -
                0.272627897050173*o[8])*tau2 + pi*(o[4]*(9.0049690883672e-11 -
                65.8490727183984*o[3]*o[4]*o[5]) + pi*(1.78287415218792e-7*o[7] + pi*
                (o[3]*(1.0406965210174e-18 + o[1]*(-1.0234747095929e-12 -
                1.0018179379511e-8*o[3])*o[3]) + o[10]*o[9]*((-1.29412653835176e-9 +
                1.71088510070544*o[11])*o[6] + o[9]*(-6.05920510335078*o[12]*o[4]*o[5]
                *tau2 + o[9]*(o[3]*o[5]*(1.78371690710842e-23 + o[1]*o[3]*o[4]*(
                6.1258633752464e-12 - 0.000084004935396416*o[7])*tau2) + pi*(-1.24017662339842e-24
                *o[11] + pi*(0.0000832192847496054*o[12]*o[3]*o[5]*tau2 + pi*(o[1]*o[
                4]*o[5]*(1.75410265428146e-27 + (1.32995316841867e-15 -
                0.0000226487297378904*o[1]*o[5])*o[8])*pi - 2.93678005497663e-14*o[1]
                *o[12]*o[3]*tau2)))))))))))))))))/pi;
              d := p/(data.RH2O*T*pi*gpi);
            end d2n;

            function hl_p_R4b
            "Explicit approximation of liquid specific enthalpy on the boundary between regions 4 and 3"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real x "Auxiliary variable";
            algorithm
              // documentation of accuracy in notebook ~hubertus/props/IAPWS/R3Approx.nb
              // boundary between region IVa and III
              x := Modelica.Math.acos(p/data.PCRIT);
              h := (1 + x*(-0.4945586958175176 + x*(1.346800016564904 + x*(-3.889388153209752
                 + x*(6.679385472887931 + x*(-6.75820241066552 + x*(3.558919744656498
                 + (-0.7179818554978939 - 0.0001152032945617821*x)*x)))))))*data.HCRIT;
            end hl_p_R4b;

            function hv_p_R4b
            "Explicit approximation of vapour specific enthalpy on the boundary between regions 4 and 3"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real x "Auxiliary variable";
            algorithm
              // boundary between region IVa and III
              x := Modelica.Math.acos(p/data.PCRIT);
              h := (1 + x*(0.4880153718655694 + x*(0.2079670746250689 + x*(-6.084122698421623
                 + x*(25.08887602293532 + x*(-48.38215180269516 + x*(
                45.66489164833212 + (-16.98555442961553 + 0.0006616936460057691*x)*x)))))))
                *data.HCRIT;
            end hv_p_R4b;

            function sl_p_R4b
            "Explicit approximation of liquid specific entropy on the boundary between regions 4 and 3"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real x "Auxiliary variable";
            algorithm
              // boundary between region IVa and III
              x := Modelica.Math.acos(p/data.PCRIT);
              s := (1 + x*(-0.36160692245648063 + x*(0.9962778630486647 + x*(-2.8595548144171103
                 + x*(4.906301159555333 + x*(-4.974092309614206 + x*(
                2.6249651699204457 + (-0.5319954375299023 - 0.00008064497431880644*x)
                *x)))))))*data.SCRIT;
            end sl_p_R4b;

            function sv_p_R4b
            "Explicit approximation of vapour specific entropy on the boundary between regions 4 and 3"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s;
          protected
              Real x "Auxiliary variable";
            algorithm

              // documentation of accuracy in notebook ~hubertus/props/IAPWS/R3Approx.nb
              // boundary between region IVa and III
              x := Modelica.Math.acos(p/data.PCRIT);
              s := (1 + x*(0.35682641826674344 + x*(0.1642457027815487 + x*(-4.425350377422446
                 + x*(18.324477859983133 + x*(-35.338631625948665 + x*(
                33.36181025816282 + (-12.408711490585757 + 0.0004810049834109226*x)*x)))))))
                *data.SCRIT;
            end sv_p_R4b;

            function rhol_p_R4b
            "Explicit approximation of liquid density on the boundary between regions 4 and 3"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.Density dl "Liquid density";
          protected
              Real x "Auxiliary variable";
            algorithm
              if (p < data.PCRIT) then
                x := Modelica.Math.acos(p/data.PCRIT);
                dl := (1 + x*(1.903224079094824 + x*(-2.5314861802401123 + x*(-8.191449323843552
                   + x*(94.34196116778385 + x*(-369.3676833623383 + x*(
                  796.6627910598293 + x*(-994.5385383600702 + x*(673.2581177021598 +
                  (-191.43077336405156 + 0.00052536560808895*x)*x)))))))))*data.DCRIT;
              else
                dl := data.DCRIT;
              end if;
            end rhol_p_R4b;

            function rhov_p_R4b
            "Explicit approximation of vapour density on the boundary between regions 4 and 2"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.Density dv "Vapour density";
          protected
              Real x "Auxiliary variable";
            algorithm
              if (p < data.PCRIT) then
                x := Modelica.Math.acos(p/data.PCRIT);
                dv := (1 + x*(-1.8463850803362596 + x*(-1.1447872718878493 + x*(
                  59.18702203076563 + x*(-403.5391431811611 + x*(1437.2007245332388
                   + x*(-3015.853540307519 + x*(3740.5790348670057 + x*(-2537.375817253895
                   + (725.8761975803782 - 0.0011151111658332337*x)*x)))))))))*data.DCRIT;
              else
                dv := data.DCRIT;
              end if;
            end rhov_p_R4b;

            function boilingcurve_p "Properties on the boiling curve"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output Common.IF97PhaseBoundaryProperties bpro "Property record";
          protected
              Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives";
              Common.HelmholtzDerivs f
              "Dimensionless Helmholtz function and derivatives";
              SI.Pressure plim=min(p, data.PCRIT - 1e-7)
              "Pressure limited to critical pressure - epsilon";
            algorithm
              bpro.R := data.RH2O;
              bpro.T := Basic.tsat(plim);
              bpro.dpT := Basic.dptofT(bpro.T);
              bpro.region3boundary := bpro.T > data.TLIMIT1;
              if not bpro.region3boundary then
                g := Basic.g1(p, bpro.T);
                bpro.d := p/(bpro.R*bpro.T*g.pi*g.gpi);
                bpro.h := if p > plim then data.HCRIT else bpro.R*bpro.T*g.tau*g.gtau;
                bpro.s := g.R*(g.tau*g.gtau - g.g);
                bpro.cp := -bpro.R*g.tau*g.tau*g.gtautau;
                bpro.vt := bpro.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
                bpro.vp := bpro.R*bpro.T/(p*p)*g.pi*g.pi*g.gpipi;
                bpro.pt := -p/bpro.T*(g.gpi - g.tau*g.gtaupi)/(g.gpipi*g.pi);
                bpro.pd := -bpro.R*bpro.T*g.gpi*g.gpi/(g.gpipi);
              else
                bpro.d := rhol_p_R4b(plim);
                f := Basic.f3(bpro.d, bpro.T);
                bpro.h := hl_p_R4b(plim);
                // bpro.R*bpro.T*(f.tau*f.ftau + f.delta*f.fdelta);
                bpro.s := f.R*(f.tau*f.ftau - f.f);
                bpro.cv := bpro.R*(-f.tau*f.tau*f.ftautau);
                bpro.pt := bpro.R*bpro.d*f.delta*(f.fdelta - f.tau*f.fdeltatau);
                bpro.pd := bpro.R*bpro.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
              end if;
            end boilingcurve_p;

            function dewcurve_p "Properties on the dew curve"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output Common.IF97PhaseBoundaryProperties bpro "Property record";
          protected
              Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives";
              Common.HelmholtzDerivs f
              "Dimensionless Helmholtz function and derivatives";
              SI.Pressure plim=min(p, data.PCRIT - 1e-7)
              "Pressure limited to critical pressure - epsilon";
            algorithm
              bpro.R := data.RH2O;
              bpro.T := Basic.tsat(plim);
              bpro.dpT := Basic.dptofT(bpro.T);
              bpro.region3boundary := bpro.T > data.TLIMIT1;
              if not bpro.region3boundary then
                g := Basic.g2(p, bpro.T);
                bpro.d := p/(bpro.R*bpro.T*g.pi*g.gpi);
                bpro.h := if p > plim then data.HCRIT else bpro.R*bpro.T*g.tau*g.gtau;
                bpro.s := g.R*(g.tau*g.gtau - g.g);
                bpro.cp := -bpro.R*g.tau*g.tau*g.gtautau;
                bpro.vt := bpro.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
                bpro.vp := bpro.R*bpro.T/(p*p)*g.pi*g.pi*g.gpipi;
                bpro.pt := -p/bpro.T*(g.gpi - g.tau*g.gtaupi)/(g.gpipi*g.pi);
                bpro.pd := -bpro.R*bpro.T*g.gpi*g.gpi/(g.gpipi);
              else
                bpro.d := rhov_p_R4b(plim);
                f := Basic.f3(bpro.d, bpro.T);
                bpro.h := hv_p_R4b(plim);
                // bpro.R*bpro.T*(f.tau*f.ftau + f.delta*f.fdelta);
                bpro.s := f.R*(f.tau*f.ftau - f.f);
                bpro.cv := bpro.R*(-f.tau*f.tau*f.ftautau);
                bpro.pt := bpro.R*bpro.d*f.delta*(f.fdelta - f.tau*f.fdeltatau);
                bpro.pd := bpro.R*bpro.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
              end if;
            end dewcurve_p;

            function hvl_p
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input Common.IF97PhaseBoundaryProperties bpro "Property record";
              output SI.SpecificEnthalpy h "Specific enthalpy";
            algorithm
              h := bpro.h;
              annotation (
                derivative(noDerivative=bpro) = hvl_p_der,
                Inline=false,
                LateInline=true);
            end hvl_p;

            function hl_p
            "Liquid specific enthalpy on the boundary between regions 4 and 3 or 1"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
            algorithm
              h := hvl_p(p, boilingcurve_p(p));
              annotation (Inline=true);
            end hl_p;

            function hv_p
            "Vapour specific enthalpy on the boundary between regions 4 and 3 or 2"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEnthalpy h "Specific enthalpy";
            algorithm
              h := hvl_p(p, dewcurve_p(p));
              annotation (Inline=true);
            end hv_p;

            function hvl_p_der
            "Derivative function for the specific enthalpy along the phase boundary"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input Common.IF97PhaseBoundaryProperties bpro "Property record";
              input Real p_der "Derivative of pressure";
              output Real h_der
              "Time derivative of specific enthalpy along the phase boundary";
            algorithm
              if bpro.region3boundary then
                h_der := ((bpro.d*bpro.pd - bpro.T*bpro.pt)*p_der + (bpro.T*bpro.pt*
                  bpro.pt + bpro.d*bpro.d*bpro.pd*bpro.cv)/bpro.dpT*p_der)/(bpro.pd*
                  bpro.d*bpro.d);
              else
                h_der := (1/bpro.d - bpro.T*bpro.vt)*p_der + bpro.cp/bpro.dpT*p_der;
              end if;
              annotation (Inline=true);
            end hvl_p_der;

            function rhovl_p
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input Common.IF97PhaseBoundaryProperties bpro "Property record";
              output SI.Density rho "Density";
            algorithm
              rho := bpro.d;
              annotation (
                derivative(noDerivative=bpro) = rhovl_p_der,
                Inline=false,
                LateInline=true);
            end rhovl_p;

            function rhol_p "Density of saturated water"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Saturation pressure";
              output SI.Density rho
              "Density of steam at the condensation point";
            algorithm
              rho := rhovl_p(p, boilingcurve_p(p));
              annotation (Inline=true);
            end rhol_p;

            function rhov_p "Density of saturated vapour"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Saturation pressure";
              output SI.Density rho
              "Density of steam at the condensation point";
            algorithm
              rho := rhovl_p(p, dewcurve_p(p));
              annotation (Inline=true);
            end rhov_p;

            function rhovl_p_der
              extends Modelica.Icons.Function;
              input SI.Pressure p "Saturation pressure";
              input Common.IF97PhaseBoundaryProperties bpro "Property record";
              input Real p_der "Derivative of pressure";
              output Real d_der
              "Time derivative of density along the phase boundary";
            algorithm
              d_der := if bpro.region3boundary then (p_der - bpro.pt*p_der/bpro.dpT)/
                bpro.pd else -bpro.d*bpro.d*(bpro.vp + bpro.vt/bpro.dpT)*p_der;
              annotation (Inline=true);
            end rhovl_p_der;

            function sl_p
            "Liquid specific entropy on the boundary between regions 4 and 3 or 1"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              SI.Temperature Tsat "Saturation temperature";
              SI.SpecificEnthalpy h "Specific enthalpy";
            algorithm
              if (p < data.PLIMIT4A) then
                Tsat := Basic.tsat(p);
                (h,s) := Isentropic.handsofpT1(p, Tsat);
              elseif (p < data.PCRIT) then
                s := sl_p_R4b(p);
              else
                s := data.SCRIT;
              end if;
            end sl_p;

            function sv_p
            "Vapour specific entropy on the boundary between regions 4 and 3 or 2"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              SI.Temperature Tsat "Saturation temperature";
              SI.SpecificEnthalpy h "Specific enthalpy";
            algorithm
              if (p < data.PLIMIT4A) then
                Tsat := Basic.tsat(p);
                (h,s) := Isentropic.handsofpT2(p, Tsat);
              elseif (p < data.PCRIT) then
                s := sv_p_R4b(p);
              else
                s := data.SCRIT;
              end if;
            end sv_p;

            function rhol_T "Density of saturated water"
              extends Modelica.Icons.Function;
              input SI.Temperature T "Temperature";
              output SI.Density d "Density of water at the boiling point";
          protected
              SI.Pressure p "Saturation pressure";
            algorithm
              p := Basic.psat(T);
              if T < data.TLIMIT1 then
                d := d1n(p, T);
              elseif T < data.TCRIT then
                d := rhol_p_R4b(p);
              else
                d := data.DCRIT;
              end if;
            end rhol_T;

            function rhov_T "Density of saturated vapour"
              extends Modelica.Icons.Function;
              input SI.Temperature T "Temperature";
              output SI.Density d "Density of steam at the condensation point";
          protected
              SI.Pressure p "Saturation pressure";
            algorithm

              // assert(T <= data.TCRIT,"input temperature has to be below the critical temperature");
              p := Basic.psat(T);
              if T < data.TLIMIT1 then
                d := d2n(p, T);
              elseif T < data.TCRIT then
                d := rhov_p_R4b(p);
              else
                d := data.DCRIT;
              end if;
            end rhov_T;

            function region_ph
            "Return the current region (valid values: 1,2,3,4,5) in IF97 for given pressure and specific enthalpy"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              input Integer phase=0
              "Phase: 2 for two-phase, 1 for one phase, 0 if not known";
              input Integer mode=0
              "Mode: 0 means check, otherwise assume region=mode";
              output Integer region "Region (valid values: 1,2,3,4,5) in IF97";
              // If mode is different from 0, no checking for the region is done and
              // the mode is assumed to be the correct region. This can be used to
              // implement e.g., water-only steam tables when mode == 1
          protected
              Boolean hsubcrit;
              SI.Temperature Ttest;
              constant Real[5] n=data.n;
              SI.SpecificEnthalpy hl "Bubble enthalpy";
              SI.SpecificEnthalpy hv "Dew enthalpy";
            algorithm
              if (mode <> 0) then
                region := mode;
              else
                // check for regions 1, 2, 3 and 4
                hl := hl_p(p);
                hv := hv_p(p);
                if (phase == 2) then
                  region := 4;
                else
                  // phase == 1 or 0, now check if we are in the legal area
                  if (p < triple.ptriple) or (p > data.PLIMIT1) or (h < hlowerofp1(p))
                       or ((p < 10.0e6) and (h > hupperofp5(p))) or ((p >= 10.0e6)
                       and (h > hupperofp2(p))) then
                    // outside of valid range
                    region := -1;
                  else
                    //region 5 and -1 check complete
                    hsubcrit := (h < data.HCRIT);
                    // simple precheck: very simple if pressure < PLIMIT4A
                    if (p < data.PLIMIT4A) then
                      // we can never be in region 3, so test for others
                      if hsubcrit then
                        if (phase == 1) then
                          region := 1;
                        else
                          if (h < Isentropic.hofpT1(p, Basic.tsat(p))) then
                            region := 1;
                          else
                            region := 4;
                          end if;
                        end if;
                        // external or internal phase check
                      else
                        if (h > hlowerofp5(p)) then
                          // check for region 5
                          if ((p < data.PLIMIT5) and (h < hupperofp5(p))) then
                            region := 5;
                          else
                            region := -2;
                            // pressure and specific enthalpy too high, but this should
                          end if;
                          // never happen
                        else
                          if (phase == 1) then
                            region := 2;
                          else
                            if (h > Isentropic.hofpT2(p, Basic.tsat(p))) then
                              region := 2;
                            else
                              region := 4;
                            end if;
                          end if;
                          // external or internal phase check
                        end if;
                        // tests for region 2 or 5
                      end if;
                      // tests for sub or supercritical
                    else
                      // the pressure is over data.PLIMIT4A
                      if hsubcrit then
                        // region 1 or 3 or 4
                        if h < hupperofp1(p) then
                          region := 1;
                        else
                          if h < hl or p > data.PCRIT then
                            region := 3;
                          else
                            region := 4;
                          end if;
                        end if;
                        // end of test for region 1, 3 or 4
                      else
                        // region 2, 3 or 4
                        if (h > hlowerofp2(p)) then
                          region := 2;
                        else
                          if h > hv or p > data.PCRIT then
                            region := 3;
                          else
                            region := 4;
                          end if;
                        end if;
                        // test for 2 and 3
                      end if;
                      // tests above PLIMIT4A
                    end if;
                    // above or below PLIMIT4A
                  end if;
                  // check for grand limits of p and h
                end if;
                // all tests with phase == 1
              end if;
              // mode was == 0
              // assert(region > 0,"IF97 function called outside the valid range!");
            end region_ph;

            function region_ps
            "Return the current region (valid values: 1,2,3,4,5) in IF97 for given pressure and specific entropy"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              input Integer phase=0
              "Phase: 2 for two-phase, 1 for one phase, 0 if unknown";
              input Integer mode=0
              "Mode: 0 means check, otherwise assume region=mode";
              output Integer region "Region (valid values: 1,2,3,4,5) in IF97";
              //  If mode is different from 0, no checking for the region is done and
              //    the mode is assumed to be the correct region. This can be used to
              //    implement e.g., water-only steam tables when mode == 1
          protected
              Boolean ssubcrit;
              SI.Temperature Ttest;
              constant Real[5] n=data.n;
              SI.SpecificEntropy sl "Bubble entropy";
              SI.SpecificEntropy sv "Dew entropy";
            algorithm
              if (mode <> 0) then
                region := mode;
              else
                // check for regions 1, 2, 3, and 4
                sl := sl_p(p);
                sv := sv_p(p);
                // check all cases two-phase
                if (phase == 2) or (phase == 0 and s > sl and s < sv and p < data.PCRIT) then
                  region := 4;
                else
                  // phase == 1
                  region := 0;
                  if (p < triple.ptriple) then
                    region := -2;
                  end if;
                  if (p > data.PLIMIT1) then
                    region := -3;
                  end if;
                  if ((p < 10.0e6) and (s > supperofp5(p))) then
                    region := -5;
                  end if;
                  if ((p >= 10.0e6) and (s > supperofp2(p))) then
                    region := -6;
                  end if;
                  if region < 0 then
                    assert(false,
                      "Region computation from p and s failed: function called outside the legal region");
                  else
                    ssubcrit := (s < data.SCRIT);
                    // simple precheck: very simple if pressure < PLIMIT4A
                    if (p < data.PLIMIT4A) then
                      // we can never be in region 3, so test for 1 and 2
                      if ssubcrit then
                        region := 1;
                      else
                        if (s > slowerofp5(p)) then
                          // check for region 5
                          if ((p < data.PLIMIT5) and (s < supperofp5(p))) then
                            region := 5;
                          else
                            region := -1;
                            // pressure and specific entropy too high, should never happen!
                          end if;
                        else
                          region := 2;
                        end if;
                        // tests for region 2 or 5
                      end if;
                      // tests for sub or supercritical
                    else
                      // the pressure is over data.PLIMIT4A
                      if ssubcrit then
                        // region 1 or 3
                        if s < supperofp1(p) then
                          region := 1;
                        else
                          if s < sl or p > data.PCRIT then
                            region := 3;
                          else
                            region := 4;
                          end if;
                        end if;
                        // test for region 1, 3 or 4
                      else
                        // region 2, 3 or 4
                        if (s > slowerofp2(p)) then
                          region := 2;
                        else
                          if s > sv or p > data.PCRIT then
                            region := 3;
                          else
                            region := 4;
                          end if;
                        end if;
                        // test for 2,3 and 4
                      end if;
                      // tests above PLIMIT4A
                    end if;
                    // above or below PLIMIT4A
                  end if;
                  // grand test for limits of p and s
                end if;
                // all tests with phase == 1
              end if;
              // mode was == 0
            end region_ps;

            function region_pT
            "Return the current region (valid values: 1,2,3,5) in IF97, given pressure and temperature"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              input Integer mode=0
              "Mode: 0 means check, otherwise assume region=mode";
              output Integer region
              "Region (valid values: 1,2,3,5) in IF97, region 4 is impossible!";
            algorithm
              if (mode <> 0) then
                region := mode;
              else
                if p < data.PLIMIT4A then
                  //test for regions 1,2,5
                  if T > data.TLIMIT2 then
                    region := 5;
                  elseif T > Basic.tsat(p) then
                    region := 2;
                  else
                    region := 1;
                  end if;
                else
                  //test for regions 1,2,3
                  if T < data.TLIMIT1 then
                    region := 1;
                  elseif T < boundary23ofp(p) then
                    region := 3;
                  else
                    region := 2;
                  end if;
                end if;
              end if;
              // mode was == 0
            end region_pT;

            function region_dT
            "Return the current region (valid values: 1,2,3,4,5) in IF97, given density and temperature"
              extends Modelica.Icons.Function;
              input SI.Density d "Density";
              input SI.Temperature T "Temperature (K)";
              input Integer phase=0
              "Phase: 2 for two-phase, 1 for one phase, 0 if not known";
              input Integer mode=0
              "Mode: 0 means check, otherwise assume region=mode";
              output Integer region "(valid values: 1,2,3,4,5) in IF97";
          protected
              Boolean Tovercrit "Flag if overcritical temperature";
              SI.Pressure p23 "Pressure needed to know if region 2 or 3";
            algorithm
              Tovercrit := T > data.TCRIT;
              if (mode <> 0) then
                region := mode;
              else
                p23 := boundary23ofT(T);
                if T > data.TLIMIT2 then
                  if d < 20.5655874106483 then
                    // check for the density in the upper corner of validity!
                    region := 5;
                  else
                    assert(false,
                      "Out of valid region for IF97, pressure above region 5!");
                  end if;
                elseif Tovercrit then
                  //check for regions 1, 2 or 3
                  if d > d2n(p23, T) and T > data.TLIMIT1 then
                    region := 3;
                  elseif T < data.TLIMIT1 then
                    region := 1;
                  else
                    // d  < d2n(p23, T) and T > data.TLIMIT1
                    region := 2;
                  end if;
                  // below critical, check for regions 1, 2, 3 or 4
                elseif (d > rhol_T(T)) then
                  // either 1 or 3
                  if T < data.TLIMIT1 then
                    region := 1;
                  else
                    region := 3;
                  end if;
                elseif (d < rhov_T(T)) then
                  // not liquid, not 2-phase, and not region 5, so either 2 or 3 or illegal
                  if (d > d2n(p23, T) and T > data.TLIMIT1) then
                    region := 3;
                  else
                    region := 2;
                  end if;
                else
                  region := 4;
                end if;
              end if;
            end region_dT;

            function hvl_dp
            "Derivative function for the specific enthalpy along the phase boundary"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input Common.IF97PhaseBoundaryProperties bpro "Property record";
              output Real dh_dp
              "Derivative of specific enthalpy along the phase boundary";
            algorithm
              if bpro.region3boundary then
                dh_dp := ((bpro.d*bpro.pd - bpro.T*bpro.pt) + (bpro.T*bpro.pt*bpro.pt
                   + bpro.d*bpro.d*bpro.pd*bpro.cv)/bpro.dpT)/(bpro.pd*bpro.d*bpro.d);
              else
                dh_dp := (1/bpro.d - bpro.T*bpro.vt) + bpro.cp/bpro.dpT;
              end if;
            end hvl_dp;

            function dhl_dp
            "Derivative of liquid specific enthalpy on the boundary between regions 4 and 3 or 1 w.r.t. pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.DerEnthalpyByPressure dh_dp
              "Specific enthalpy derivative w.r.t. pressure";
            algorithm
              dh_dp := hvl_dp(p, boilingcurve_p(p));
              annotation (Inline=true);
            end dhl_dp;

            function dhv_dp
            "Derivative of vapour specific enthalpy on the boundary between regions 4 and 3 or 1 w.r.t. pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.DerEnthalpyByPressure dh_dp
              "Specific enthalpy derivative w.r.t. pressure";
            algorithm
              dh_dp := hvl_dp(p, dewcurve_p(p));
              annotation (Inline=true);
            end dhv_dp;

            function drhovl_dp
              extends Modelica.Icons.Function;
              input SI.Pressure p "Saturation pressure";
              input Common.IF97PhaseBoundaryProperties bpro "Property record";
              output Real dd_dp(unit="kg/(m3.Pa)")
              "Derivative of density along the phase boundary";
            algorithm
              dd_dp := if bpro.region3boundary then (1.0 - bpro.pt/bpro.dpT)/bpro.pd
                 else -bpro.d*bpro.d*(bpro.vp + bpro.vt/bpro.dpT);
              annotation (Inline=true);
            end drhovl_dp;

            function drhol_dp
            "Derivative of density of saturated water w.r.t. pressure"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Saturation pressure";
              output SI.DerDensityByPressure dd_dp
              "Derivative of density of water at the boiling point";
            algorithm
              dd_dp := drhovl_dp(p, boilingcurve_p(p));
              annotation (Inline=true);
            end drhol_dp;

            function drhov_dp
            "Derivative of density of saturated steam w.r.t. pressure"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Saturation pressure";
              output SI.DerDensityByPressure dd_dp
              "Derivative of density of water at the boiling point";
            algorithm
              dd_dp := drhovl_dp(p, dewcurve_p(p));
              annotation (Inline=true);
            end drhov_dp;
            annotation (Documentation(info="<HTML><h4>Package description</h4>
 <p>Package <b>Regions</b> contains a large number of auxiliary functions which are needed to compute the current region
 of the IAPWS/IF97 for a given pair of input variables as quickly as possible. The focus of this implementation was on
 computational efficiency, not on compact code. Many of the function values calculated in these functions could be obtained
 using the fundamental functions of IAPWS/IF97, but with considerable overhead. If the region of IAPWS/IF97 is known in advance,
 the input variable mode can be set to the region, then the somewhat costly region checks are omitted.
 The checking for the phase has to be done outside the region functions because many properties are not
 differentiable at the region boundary. If the input phase is 2, the output region will be set to 4 immediately.</p>
 <h4>Package contents</h4>
 <p> The main 4 functions in this package are the functions returning the appropriate region for two input variables.</p>
 <ul>
 <li>Function <b>region_ph</b> compute the region of IAPWS/IF97 for input pair pressure and specific enthalpy.</li>
 <li>Function <b>region_ps</b> compute the region of IAPWS/IF97 for input pair pressure and specific entropy</li>
 <li>Function <b>region_dT</b> compute the region of IAPWS/IF97 for input pair density and temperature.</li>
 <li>Function <b>region_pT</b> compute the region of IAPWS/IF97 for input pair pressure and temperature (only in phase region).</li>
 </ul>
 <p>In addition, functions of the boiling and condensation curves compute the specific enthalpy, specific entropy, or density on these
 curves. The functions for the saturation pressure and temperature are included in the package <b>Basic</b> because they are part of
 the original <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IAPWS/IF97 standards document</a>. These functions are also aliased to
 be used directly from package <b>Water</b>.
</p>
 <ul>
 <li>Function <b>hl_p</b> computes the liquid specific enthalpy as a function of pressure. For overcritical pressures,
 the critical specific enthalpy is returned. An approximation is used for temperatures > 623.15 K.</li>
 <li>Function <b>hv_p</b> computes the vapour specific enthalpy as a function of pressure. For overcritical pressures,
 the critical specific enthalpy is returned. An approximation is used for temperatures > 623.15 K.</li>
 <li>Function <b>sl_p</b> computes the liquid specific entropy as a function of pressure. For overcritical pressures,
 the critical  specific entropy is returned. An approximation is used for temperatures > 623.15 K.</li>
 <li>Function <b>sv_p</b> computes the vapour  specific entropy as a function of pressure. For overcritical pressures,
 the critical  specific entropy is returned. An approximation is used for temperatures > 623.15 K.</li>
 <li>Function <b>rhol_T</b> computes the liquid density as a function of temperature. For overcritical temperatures,
 the critical density is returned. An approximation is used for temperatures > 623.15 K.</li>
 <li>Function <b>rhol_T</b> computes the vapour density as a function of temperature. For overcritical temperatures,
 the critical density is returned. An approximation is used for temperatures > 623.15 K.</li>
 </ul>
 <p>All other functions are auxiliary functions called from the region functions to check a specific boundary.</p>
 <ul>
 <li>Function <b>boundary23ofT</b> computes the boundary pressure between regions 2 and 3 (input temperature)</li>
 <li>Function <b>boundary23ofp</b> computes the boundary temperature between regions 2 and 3 (input pressure)</li>
 <li>Function <b>hlowerofp5</b> computes the lower specific enthalpy limit of region 5 (input p, T=1073.15 K)</li>
 <li>Function <b>hupperofp5</b> computes the upper specific enthalpy limit of region 5 (input p, T=2273.15 K)</li>
 <li>Function <b>slowerofp5</b> computes the lower specific entropy limit of region 5 (input p, T=1073.15 K)</li>
 <li>Function <b>supperofp5</b> computes the upper specific entropy limit of region 5 (input p, T=2273.15 K)</li>
 <li>Function <b>hlowerofp1</b> computes the lower specific enthalpy limit of region 1 (input p, T=273.15 K)</li>
 <li>Function <b>hupperofp1</b> computes the upper specific enthalpy limit of region 1 (input p, T=623.15 K)</li>
 <li>Function <b>slowerofp1</b> computes the lower specific entropy limit of region 1 (input p, T=273.15 K)</li>
 <li>Function <b>supperofp1</b> computes the upper specific entropy limit of region 1 (input p, T=623.15 K)</li>
 <li>Function <b>hlowerofp2</b> computes the lower specific enthalpy limit of region 2 (input p, T=623.15 K)</li>
 <li>Function <b>hupperofp2</b> computes the upper specific enthalpy limit of region 2 (input p, T=1073.15 K)</li>
 <li>Function <b>slowerofp2</b> computes the lower specific entropy limit of region 2 (input p, T=623.15 K)</li>
 <li>Function <b>supperofp2</b> computes the upper specific entropy limit of region 2 (input p, T=1073.15 K)</li>
 <li>Function <b>d1n</b> computes the density in region 1 as function of pressure and temperature</li>
 <li>Function <b>d2n</b> computes the density in region 2 as function of pressure and temperature</li>
 <li>Function <b>dhot1ofp</b> computes the hot density limit of region 1 (input p, T=623.15 K)</li>
 <li>Function <b>dupper1ofT</b>computes the high pressure density limit of region 1 (input T, p=100MPa)</li>
 <li>Function <b>hl_p_R4b</b> computes a high accuracy approximation to the liquid enthalpy for temperatures > 623.15 K (input p)</li>
 <li>Function <b>hv_p_R4b</b> computes a high accuracy approximation to the vapour enthalpy for temperatures > 623.15 K (input p)</li>
 <li>Function <b>sl_p_R4b</b> computes a high accuracy approximation to the liquid entropy for temperatures > 623.15 K (input p)</li>
 <li>Function <b>sv_p_R4b</b> computes a high accuracy approximation to the vapour entropy for temperatures > 623.15 K (input p)</li>
 <li>Function <b>rhol_p_R4b</b> computes a high accuracy approximation to the liquid density for temperatures > 623.15 K (input p)</li>
 <li>Function <b>rhov_p_R4b</b> computes a high accuracy approximation to the vapour density for temperatures > 623.15 K (input p)</li>
 </ul>

<h4>Version Info and Revision history</h4>
<ul>
<li>First implemented: <i>July, 2000</i>
       by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>
       </li>
</ul>
<address>Authors: Hubertus Tummescheit, Jonas Eborn and Falko Jens Wagner<br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
 </address>
 <ul>
 <li>Initial version: July 2000</li>
 <li>Revised and extended for inclusion in Modelica.Thermal: December 2002</li>
</ul>
</html>"));
          end Regions;

          package Basic "Base functions as described in IAWPS/IF97"
            extends Modelica.Icons.Package;

            function g1 "Gibbs function for region 1: g(p,T)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output Modelica.Media.Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          protected
              Real pi1 "Dimensionless pressure";
              Real tau1 "Dimensionless temperature";
              Real[45] o "Vector of auxiliary variables";
              Real pl "Auxiliary variable";
            algorithm
              pl := min(p, data.PCRIT - 1);
              assert(p > triple.ptriple,
                "IF97 medium function g1 called with too low pressure\n" + "p = " +
                String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              assert(p <= 100.0e6, "IF97 medium function g1: the input pressure (= "
                 + String(p) + " Pa) is higher than 100 Mpa");
              assert(T >= 273.15, "IF97 medium function g1: the temperature (= " +
                String(T) + " K) is lower than 273.15 K!");
              g.p := p;
              g.T := T;
              g.R := data.RH2O;
              g.pi := p/data.PSTAR1;
              g.tau := data.TSTAR1/T;
              pi1 := 7.1000000000000 - g.pi;
              tau1 := -1.22200000000000 + g.tau;
              o[1] := tau1*tau1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              o[4] := o[3]*tau1;
              o[5] := 1/o[4];
              o[6] := o[1]*o[2];
              o[7] := o[1]*tau1;
              o[8] := 1/o[7];
              o[9] := o[1]*o[2]*o[3];
              o[10] := 1/o[2];
              o[11] := o[2]*tau1;
              o[12] := 1/o[11];
              o[13] := o[2]*o[3];
              o[14] := 1/o[3];
              o[15] := pi1*pi1;
              o[16] := o[15]*pi1;
              o[17] := o[15]*o[15];
              o[18] := o[17]*o[17];
              o[19] := o[17]*o[18]*pi1;
              o[20] := o[15]*o[17];
              o[21] := o[3]*o[3];
              o[22] := o[21]*o[21];
              o[23] := o[22]*o[3]*tau1;
              o[24] := 1/o[23];
              o[25] := o[22]*o[3];
              o[26] := 1/o[25];
              o[27] := o[1]*o[2]*o[22]*tau1;
              o[28] := 1/o[27];
              o[29] := o[1]*o[2]*o[22];
              o[30] := 1/o[29];
              o[31] := o[1]*o[2]*o[21]*o[3]*tau1;
              o[32] := 1/o[31];
              o[33] := o[2]*o[21]*o[3]*tau1;
              o[34] := 1/o[33];
              o[35] := o[1]*o[3]*tau1;
              o[36] := 1/o[35];
              o[37] := o[1]*o[3];
              o[38] := 1/o[37];
              o[39] := 1/o[6];
              o[40] := o[1]*o[22]*o[3];
              o[41] := 1/o[40];
              o[42] := 1/o[22];
              o[43] := o[1]*o[2]*o[21]*o[3];
              o[44] := 1/o[43];
              o[45] := 1/o[13];
              g.g := pi1*(pi1*(pi1*(o[10]*(-0.000031679644845054 + o[2]*(-2.82707979853120e-6
                 - 8.5205128120103e-10*o[6])) + pi1*(o[12]*(-2.24252819080000e-6 + (-6.5171222895601e-7
                 - 1.43417299379240e-13*o[13])*o[7]) + pi1*(-4.0516996860117e-7*o[14]
                 + o[16]*((-1.27343017416410e-9 - 1.74248712306340e-10*o[11])*o[36]
                 + o[19]*(-6.8762131295531e-19*o[34] + o[15]*(1.44783078285210e-20*o[
                32] + o[20]*(2.63357816627950e-23*o[30] + pi1*(-1.19476226400710e-23*
                o[28] + pi1*(1.82280945814040e-24*o[26] - 9.3537087292458e-26*o[24]*
                pi1))))))))) + o[8]*(-0.00047184321073267 + o[7]*(-0.000300017807930260
                 + (0.000047661393906987 + o[1]*(-4.4141845330846e-6 -
                7.2694996297594e-16*o[9]))*tau1))) + o[5]*(0.000283190801238040 + o[1]
                *(-0.00060706301565874 + o[6]*(-0.0189900682184190 + tau1*(-0.032529748770505
                 + (-0.0218417171754140 - 0.000052838357969930*o[1])*tau1))))) + (
                0.146329712131670 + tau1*(-0.84548187169114 + tau1*(-3.7563603672040
                 + tau1*(3.3855169168385 + tau1*(-0.95791963387872 + tau1*(
                0.157720385132280 + (-0.0166164171995010 + 0.00081214629983568*tau1)*
                tau1))))))/o[1];

              g.gpi := pi1*(pi1*(o[10]*(0.000095038934535162 + o[2]*(
                8.4812393955936e-6 + 2.55615384360309e-9*o[6])) + pi1*(o[12]*(
                8.9701127632000e-6 + (2.60684891582404e-6 + 5.7366919751696e-13*o[13])
                *o[7]) + pi1*(2.02584984300585e-6*o[14] + o[16]*((1.01874413933128e-8
                 + 1.39398969845072e-9*o[11])*o[36] + o[19]*(1.44400475720615e-17*o[
                34] + o[15]*(-3.3300108005598e-19*o[32] + o[20]*(-7.6373766822106e-22
                *o[30] + pi1*(3.5842867920213e-22*o[28] + pi1*(-5.6507093202352e-23*o[
                26] + 2.99318679335866e-24*o[24]*pi1))))))))) + o[8]*(
                0.00094368642146534 + o[7]*(0.00060003561586052 + (-0.000095322787813974
                 + o[1]*(8.8283690661692e-6 + 1.45389992595188e-15*o[9]))*tau1))) + o[
                5]*(-0.000283190801238040 + o[1]*(0.00060706301565874 + o[6]*(
                0.0189900682184190 + tau1*(0.032529748770505 + (0.0218417171754140 +
                0.000052838357969930*o[1])*tau1))));

              g.gpipi := pi1*(o[10]*(-0.000190077869070324 + o[2]*(-0.0000169624787911872
                 - 5.1123076872062e-9*o[6])) + pi1*(o[12]*(-0.0000269103382896000 + (
                -7.8205467474721e-6 - 1.72100759255088e-12*o[13])*o[7]) + pi1*(-8.1033993720234e-6
                *o[14] + o[16]*((-7.1312089753190e-8 - 9.7579278891550e-9*o[11])*o[36]
                 + o[19]*(-2.88800951441230e-16*o[34] + o[15]*(7.3260237612316e-18*o[
                32] + o[20]*(2.13846547101895e-20*o[30] + pi1*(-1.03944316968618e-20*
                o[28] + pi1*(1.69521279607057e-21*o[26] - 9.2788790594118e-23*o[24]*
                pi1))))))))) + o[8]*(-0.00094368642146534 + o[7]*(-0.00060003561586052
                 + (0.000095322787813974 + o[1]*(-8.8283690661692e-6 -
                1.45389992595188e-15*o[9]))*tau1));

              g.gtau := pi1*(o[38]*(-0.00254871721114236 + o[1]*(0.0042494411096112
                 + (0.0189900682184190 + (-0.0218417171754140 - 0.000158515073909790*
                o[1])*o[1])*o[6])) + pi1*(o[10]*(0.00141552963219801 + o[2]*(
                0.000047661393906987 + o[1]*(-0.0000132425535992538 -
                1.23581493705910e-14*o[9]))) + pi1*(o[12]*(0.000126718579380216 -
                5.1123076872062e-9*o[37]) + pi1*(o[39]*(0.0000112126409540000 + (
                1.30342445791202e-6 - 1.43417299379240e-12*o[13])*o[7]) + pi1*(
                3.2413597488094e-6*o[5] + o[16]*((1.40077319158051e-8 +
                1.04549227383804e-9*o[11])*o[45] + o[19]*(1.99410180757040e-17*o[44]
                 + o[15]*(-4.4882754268415e-19*o[42] + o[20]*(-1.00075970318621e-21*o[
                28] + pi1*(4.6595728296277e-22*o[26] + pi1*(-7.2912378325616e-23*o[24]
                 + 3.8350205789908e-24*o[41]*pi1))))))))))) + o[8]*(-0.292659424263340
                 + tau1*(0.84548187169114 + o[1]*(3.3855169168385 + tau1*(-1.91583926775744
                 + tau1*(0.47316115539684 + (-0.066465668798004 + 0.0040607314991784*
                tau1)*tau1)))));

              g.gtautau := pi1*(o[36]*(0.0254871721114236 + o[1]*(-0.033995528876889
                 + (-0.037980136436838 - 0.00031703014781958*o[2])*o[6])) + pi1*(o[12]
                *(-0.0056621185287920 + o[6]*(-0.0000264851071985076 -
                1.97730389929456e-13*o[9])) + pi1*((-0.00063359289690108 -
                2.55615384360309e-8*o[37])*o[39] + pi1*(pi1*(-0.0000291722377392842*o[
                38] + o[16]*(o[19]*(-5.9823054227112e-16*o[32] + o[15]*(o[20]*(
                3.9029628424262e-20*o[26] + pi1*(-1.86382913185108e-20*o[24] + pi1*(
                2.98940751135026e-21*o[41] - (1.61070864317613e-22*pi1)/(o[1]*o[22]*o[
                3]*tau1)))) + 1.43624813658928e-17/(o[22]*tau1))) + (-1.68092782989661e-7
                 - 7.3184459168663e-9*o[11])/(o[2]*o[3]*tau1))) + (-0.000067275845724000
                 + (-3.9102733737361e-6 - 1.29075569441316e-11*o[13])*o[7])/(o[1]*o[2]
                *tau1))))) + o[10]*(0.87797827279002 + tau1*(-1.69096374338228 + o[7]
                *(-1.91583926775744 + tau1*(0.94632231079368 + (-0.199397006394012 +
                0.0162429259967136*tau1)*tau1))));

              g.gtaupi := o[38]*(0.00254871721114236 + o[1]*(-0.0042494411096112 + (-0.0189900682184190
                 + (0.0218417171754140 + 0.000158515073909790*o[1])*o[1])*o[6])) +
                pi1*(o[10]*(-0.00283105926439602 + o[2]*(-0.000095322787813974 + o[1]
                *(0.0000264851071985076 + 2.47162987411820e-14*o[9]))) + pi1*(o[12]*(
                -0.00038015573814065 + 1.53369230616185e-8*o[37]) + pi1*(o[39]*(-0.000044850563816000
                 + (-5.2136978316481e-6 + 5.7366919751696e-12*o[13])*o[7]) + pi1*(-0.0000162067987440468
                *o[5] + o[16]*((-1.12061855326441e-7 - 8.3639381907043e-9*o[11])*o[45]
                 + o[19]*(-4.1876137958978e-16*o[44] + o[15]*(1.03230334817355e-17*o[
                42] + o[20]*(2.90220313924001e-20*o[28] + pi1*(-1.39787184888831e-20*
                o[26] + pi1*(2.26028372809410e-21*o[24] - 1.22720658527705e-22*o[41]*
                pi1))))))))));
            end g1;

            function g2 "Gibbs function for region 2: g(p,T)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output Modelica.Media.Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          protected
              Real tau2 "Dimensionless temperature";
              Real[55] o "Vector of auxiliary variables";
            algorithm
              g.p := p;
              g.T := T;
              g.R := data.RH2O;
              assert(p > 0.0,
                "IF97 medium function g2 called with too low pressure\n" + "p = " +
                String(p) + " Pa <=  0.0 Pa");
              assert(p <= 100.0e6, "IF97 medium function g2: the input pressure (= "
                 + String(p) + " Pa) is higher than 100 Mpa");
              assert(T >= 273.15, "IF97 medium function g2: the temperature (= " +
                String(T) + " K) is lower than 273.15 K!");
              assert(T <= 1073.15,
                "IF97 medium function g2: the input temperature (= " + String(T) +
                " K) is higher than the limit of 1073.15 K");
              g.pi := p/data.PSTAR2;
              g.tau := data.TSTAR2/T;
              tau2 := -0.5 + g.tau;
              o[1] := tau2*tau2;
              o[2] := o[1]*tau2;
              o[3] := -0.050325278727930*o[2];
              o[4] := -0.057581259083432 + o[3];
              o[5] := o[4]*tau2;
              o[6] := -0.045996013696365 + o[5];
              o[7] := o[6]*tau2;
              o[8] := -0.0178348622923580 + o[7];
              o[9] := o[8]*tau2;
              o[10] := o[1]*o[1];
              o[11] := o[10]*o[10];
              o[12] := o[11]*o[11];
              o[13] := o[10]*o[11]*o[12]*tau2;
              o[14] := o[1]*o[10]*tau2;
              o[15] := o[10]*o[11]*tau2;
              o[16] := o[1]*o[12]*tau2;
              o[17] := o[1]*o[11]*tau2;
              o[18] := o[1]*o[10]*o[11];
              o[19] := o[10]*o[11]*o[12];
              o[20] := o[1]*o[10];
              o[21] := g.pi*g.pi;
              o[22] := o[21]*o[21];
              o[23] := o[21]*o[22];
              o[24] := o[10]*o[12]*tau2;
              o[25] := o[12]*o[12];
              o[26] := o[11]*o[12]*o[25]*tau2;
              o[27] := o[10]*o[12];
              o[28] := o[1]*o[10]*o[11]*tau2;
              o[29] := o[10]*o[12]*o[25]*tau2;
              o[30] := o[1]*o[10]*o[25]*tau2;
              o[31] := o[1]*o[11]*o[12];
              o[32] := o[1]*o[12];
              o[33] := g.tau*g.tau;
              o[34] := o[33]*o[33];
              o[35] := -0.000053349095828174*o[13];
              o[36] := -0.087594591301146 + o[35];
              o[37] := o[2]*o[36];
              o[38] := -0.0078785554486710 + o[37];
              o[39] := o[1]*o[38];
              o[40] := -0.00037897975032630 + o[39];
              o[41] := o[40]*tau2;
              o[42] := -0.000066065283340406 + o[41];
              o[43] := o[42]*tau2;
              o[44] := 5.7870447262208e-6*tau2;
              o[45] := -0.301951672367580*o[2];
              o[46] := -0.172743777250296 + o[45];
              o[47] := o[46]*tau2;
              o[48] := -0.091992027392730 + o[47];
              o[49] := o[48]*tau2;
              o[50] := o[1]*o[11];
              o[51] := o[10]*o[11];
              o[52] := o[11]*o[12]*o[25];
              o[53] := o[10]*o[12]*o[25];
              o[54] := o[1]*o[10]*o[25];
              o[55] := o[11]*o[12]*tau2;

              g.g := g.pi*(-0.00177317424732130 + o[9] + g.pi*(tau2*(-0.000033032641670203
                 + (-0.000189489875163150 + o[1]*(-0.0039392777243355 + (-0.043797295650573
                 - 0.0000266745479140870*o[13])*o[2]))*tau2) + g.pi*(
                2.04817376923090e-8 + (4.3870667284435e-7 + o[1]*(-0.000032277677238570
                 + (-0.00150339245421480 - 0.040668253562649*o[13])*o[2]))*tau2 + g.pi
                *(g.pi*(2.29220763376610e-6*o[14] + g.pi*((-1.67147664510610e-11 + o[
                15]*(-0.00211714723213550 - 23.8957419341040*o[16]))*o[2] + g.pi*(-5.9059564324270e-18
                 + o[17]*(-1.26218088991010e-6 - 0.038946842435739*o[18]) + g.pi*(o[
                11]*(1.12562113604590e-11 - 8.2311340897998*o[19]) + g.pi*(
                1.98097128020880e-8*o[15] + g.pi*(o[10]*(1.04069652101740e-19 + (-1.02347470959290e-13
                 - 1.00181793795110e-9*o[10])*o[20]) + o[23]*(o[13]*(-8.0882908646985e-11
                 + 0.106930318794090*o[24]) + o[21]*(-0.33662250574171*o[26] + o[21]*
                (o[27]*(8.9185845355421e-25 + (3.06293168762320e-13 -
                4.2002467698208e-6*o[15])*o[28]) + g.pi*(-5.9056029685639e-26*o[24]
                 + g.pi*(3.7826947613457e-6*o[29] + g.pi*(-1.27686089346810e-15*o[30]
                 + o[31]*(7.3087610595061e-29 + o[18]*(5.5414715350778e-17 -
                9.4369707241210e-7*o[32]))*g.pi)))))))))))) + tau2*(-7.8847309559367e-10
                 + (1.27907178522850e-8 + 4.8225372718507e-7*tau2)*tau2))))) + (-0.0056087911830200
                 + g.tau*(0.071452738814550 + g.tau*(-0.40710498239280 + g.tau*(
                1.42408197144400 + g.tau*(-4.3839511194500 + g.tau*(-9.6927686002170
                 + g.tau*(10.0866556801800 + (-0.284086326077200 + 0.0212684635330700
                *g.tau)*g.tau) + Modelica.Math.log(g.pi)))))))/(o[34]*g.tau);

              g.gpi := (1.00000000000000 + g.pi*(-0.00177317424732130 + o[9] + g.pi*(
                o[43] + g.pi*(6.1445213076927e-8 + (1.31612001853305e-6 + o[1]*(-0.000096833031715710
                 + (-0.0045101773626444 - 0.122004760687947*o[13])*o[2]))*tau2 + g.pi
                *(g.pi*(0.0000114610381688305*o[14] + g.pi*((-1.00288598706366e-10 +
                o[15]*(-0.0127028833928130 - 143.374451604624*o[16]))*o[2] + g.pi*(-4.1341695026989e-17
                 + o[17]*(-8.8352662293707e-6 - 0.272627897050173*o[18]) + g.pi*(o[11]
                *(9.0049690883672e-11 - 65.849072718398*o[19]) + g.pi*(
                1.78287415218792e-7*o[15] + g.pi*(o[10]*(1.04069652101740e-18 + (-1.02347470959290e-12
                 - 1.00181793795110e-8*o[10])*o[20]) + o[23]*(o[13]*(-1.29412653835176e-9
                 + 1.71088510070544*o[24]) + o[21]*(-6.0592051033508*o[26] + o[21]*(o[
                27]*(1.78371690710842e-23 + (6.1258633752464e-12 -
                0.000084004935396416*o[15])*o[28]) + g.pi*(-1.24017662339842e-24*o[24]
                 + g.pi*(0.000083219284749605*o[29] + g.pi*(-2.93678005497663e-14*o[
                30] + o[31]*(1.75410265428146e-27 + o[18]*(1.32995316841867e-15 -
                0.0000226487297378904*o[32]))*g.pi)))))))))))) + tau2*(-3.15389238237468e-9
                 + (5.1162871409140e-8 + 1.92901490874028e-6*tau2)*tau2))))))/g.pi;

              g.gpipi := (-1.00000000000000 + o[21]*(o[43] + g.pi*(
                1.22890426153854e-7 + (2.63224003706610e-6 + o[1]*(-0.000193666063431420
                 + (-0.0090203547252888 - 0.244009521375894*o[13])*o[2]))*tau2 + g.pi
                *(g.pi*(0.000045844152675322*o[14] + g.pi*((-5.0144299353183e-10 + o[
                15]*(-0.063514416964065 - 716.87225802312*o[16]))*o[2] + g.pi*(-2.48050170161934e-16
                 + o[17]*(-0.000053011597376224 - 1.63576738230104*o[18]) + g.pi*(o[
                11]*(6.3034783618570e-10 - 460.94350902879*o[19]) + g.pi*(
                1.42629932175034e-6*o[15] + g.pi*(o[10]*(9.3662686891566e-18 + (-9.2112723863361e-12
                 - 9.0163614415599e-8*o[10])*o[20]) + o[23]*(o[13]*(-1.94118980752764e-8
                 + 25.6632765105816*o[24]) + o[21]*(-103.006486756963*o[26] + o[21]*(
                o[27]*(3.3890621235060e-22 + (1.16391404129682e-10 -
                0.00159609377253190*o[15])*o[28]) + g.pi*(-2.48035324679684e-23*o[24]
                 + g.pi*(0.00174760497974171*o[29] + g.pi*(-6.4609161209486e-13*o[30]
                 + o[31]*(4.0344361048474e-26 + o[18]*(3.05889228736295e-14 -
                0.00052092078397148*o[32]))*g.pi)))))))))))) + tau2*(-9.4616771471240e-9
                 + (1.53488614227420e-7 + o[44])*tau2)))))/o[21];

              g.gtau := (0.0280439559151000 + g.tau*(-0.285810955258200 + g.tau*(
                1.22131494717840 + g.tau*(-2.84816394288800 + g.tau*(4.3839511194500
                 + o[33]*(10.0866556801800 + (-0.56817265215440 + 0.063805390599210*g.tau)
                *g.tau))))))/(o[33]*o[34]) + g.pi*(-0.0178348622923580 + o[49] + g.pi
                *(-0.000033032641670203 + (-0.00037897975032630 + o[1]*(-0.0157571108973420
                 + (-0.306581069554011 - 0.00096028372490713*o[13])*o[2]))*tau2 + g.pi
                *(4.3870667284435e-7 + o[1]*(-0.000096833031715710 + (-0.0090203547252888
                 - 1.42338887469272*o[13])*o[2]) + g.pi*(-7.8847309559367e-10 + g.pi*
                (0.0000160454534363627*o[20] + g.pi*(o[1]*(-5.0144299353183e-11 + o[
                15]*(-0.033874355714168 - 836.35096769364*o[16])) + g.pi*((-0.0000138839897890111
                 - 0.97367106089347*o[18])*o[50] + g.pi*(o[14]*(9.0049690883672e-11
                 - 296.320827232793*o[19]) + g.pi*(2.57526266427144e-7*o[51] + g.pi*(
                o[2]*(4.1627860840696e-19 + (-1.02347470959290e-12 -
                1.40254511313154e-8*o[10])*o[20]) + o[23]*(o[19]*(-2.34560435076256e-9
                 + 5.3465159397045*o[24]) + o[21]*(-19.1874828272775*o[52] + o[21]*(o[
                16]*(1.78371690710842e-23 + (1.07202609066812e-11 -
                0.000201611844951398*o[15])*o[28]) + g.pi*(-1.24017662339842e-24*o[27]
                 + g.pi*(0.000200482822351322*o[53] + g.pi*(-4.9797574845256e-14*o[54]
                 + (1.90027787547159e-27 + o[18]*(2.21658861403112e-15 -
                0.000054734430199902*o[32]))*o[55]*g.pi)))))))))))) + (
                2.55814357045700e-8 + 1.44676118155521e-6*tau2)*tau2))));

              g.gtautau := (-0.168263735490600 + g.tau*(1.42905477629100 + g.tau*(-4.8852597887136
                 + g.tau*(8.5444918286640 + g.tau*(-8.7679022389000 + o[33]*(-0.56817265215440
                 + 0.127610781198420*g.tau)*g.tau)))))/(o[33]*o[34]*g.tau) + g.pi*(-0.091992027392730
                 + (-0.34548755450059 - 1.50975836183790*o[2])*tau2 + g.pi*(-0.00037897975032630
                 + o[1]*(-0.047271332692026 + (-1.83948641732407 - 0.033609930371750*
                o[13])*o[2]) + g.pi*((-0.000193666063431420 + (-0.045101773626444 -
                48.395221739552*o[13])*o[2])*tau2 + g.pi*(2.55814357045700e-8 +
                2.89352236311042e-6*tau2 + g.pi*(0.000096272720618176*o[10]*tau2 + g.pi
                *((-1.00288598706366e-10 + o[15]*(-0.50811533571252 -
                28435.9329015838*o[16]))*tau2 + g.pi*(o[11]*(-0.000138839897890111 -
                23.3681054614434*o[18])*tau2 + g.pi*((6.3034783618570e-10 -
                10371.2289531477*o[19])*o[20] + g.pi*(3.09031519712573e-6*o[17] + g.pi
                *(o[1]*(1.24883582522088e-18 + (-9.2112723863361e-12 -
                1.82330864707100e-7*o[10])*o[20]) + o[23]*(o[1]*o[11]*o[12]*(-6.5676921821352e-8
                 + 261.979281045521*o[24])*tau2 + o[21]*(-1074.49903832754*o[1]*o[10]
                *o[12]*o[25]*tau2 + o[21]*((3.3890621235060e-22 + (
                3.6448887082716e-10 - 0.0094757567127157*o[15])*o[28])*o[32] + g.pi*(
                -2.48035324679684e-23*o[16] + g.pi*(0.0104251067622687*o[1]*o[12]*o[
                25]*tau2 + g.pi*(o[11]*o[12]*(4.7506946886790e-26 + o[18]*(
                8.6446955947214e-14 - 0.00311986252139440*o[32]))*g.pi -
                1.89230784411972e-12*o[10]*o[25]*tau2))))))))))))))));

              g.gtaupi := -0.0178348622923580 + o[49] + g.pi*(-0.000066065283340406
                 + (-0.00075795950065260 + o[1]*(-0.0315142217946840 + (-0.61316213910802
                 - 0.00192056744981426*o[13])*o[2]))*tau2 + g.pi*(1.31612001853305e-6
                 + o[1]*(-0.000290499095147130 + (-0.0270610641758664 -
                4.2701666240781*o[13])*o[2]) + g.pi*(-3.15389238237468e-9 + g.pi*(
                0.000080227267181813*o[20] + g.pi*(o[1]*(-3.00865796119098e-10 + o[15]
                *(-0.203246134285008 - 5018.1058061618*o[16])) + g.pi*((-0.000097187928523078
                 - 6.8156974262543*o[18])*o[50] + g.pi*(o[14]*(7.2039752706938e-10 -
                2370.56661786234*o[19]) + g.pi*(2.31773639784430e-6*o[51] + g.pi*(o[2]
                *(4.1627860840696e-18 + (-1.02347470959290e-11 - 1.40254511313154e-7*
                o[10])*o[20]) + o[23]*(o[19]*(-3.7529669612201e-8 + 85.544255035272*o[
                24]) + o[21]*(-345.37469089099*o[52] + o[21]*(o[16]*(
                3.5674338142168e-22 + (2.14405218133624e-10 - 0.0040322368990280*o[15])
                *o[28]) + g.pi*(-2.60437090913668e-23*o[27] + g.pi*(
                0.0044106220917291*o[53] + g.pi*(-1.14534422144089e-12*o[54] + (
                4.5606669011318e-26 + o[18]*(5.3198126736747e-14 -
                0.00131362632479764*o[32]))*o[55]*g.pi)))))))))))) + (
                1.02325742818280e-7 + o[44])*tau2)));
            end g2;

            function f3 "Helmholtz function for region 3: f(d,T)"
              extends Modelica.Icons.Function;
              input SI.Density d "Density";
              input SI.Temperature T "Temperature (K)";
              output Modelica.Media.Common.HelmholtzDerivs f
              "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          protected
              Real[40] o "Vector of auxiliary variables";
            algorithm
              f.T := T;
              f.d := d;
              f.R := data.RH2O;
              f.tau := data.TCRIT/T;
              f.delta := if (d == data.DCRIT and T == data.TCRIT) then 1 - Modelica.Constants.eps
                 else abs(d/data.DCRIT);
              o[1] := f.tau*f.tau;
              o[2] := o[1]*o[1];
              o[3] := o[2]*f.tau;
              o[4] := o[1]*f.tau;
              o[5] := o[2]*o[2];
              o[6] := o[1]*o[5]*f.tau;
              o[7] := o[5]*f.tau;
              o[8] := -0.64207765181607*o[1];
              o[9] := 0.88521043984318 + o[8];
              o[10] := o[7]*o[9];
              o[11] := -1.15244078066810 + o[10];
              o[12] := o[11]*o[2];
              o[13] := -1.26543154777140 + o[12];
              o[14] := o[1]*o[13];
              o[15] := o[1]*o[2]*o[5]*f.tau;
              o[16] := o[2]*o[5];
              o[17] := o[1]*o[5];
              o[18] := o[5]*o[5];
              o[19] := o[1]*o[18]*o[2];
              o[20] := o[1]*o[18]*o[2]*f.tau;
              o[21] := o[18]*o[5];
              o[22] := o[1]*o[18]*o[5];
              o[23] := 0.251168168486160*o[2];
              o[24] := 0.078841073758308 + o[23];
              o[25] := o[15]*o[24];
              o[26] := -6.1005234513930 + o[25];
              o[27] := o[26]*f.tau;
              o[28] := 9.7944563083754 + o[27];
              o[29] := o[2]*o[28];
              o[30] := -1.70429417648412 + o[29];
              o[31] := o[1]*o[30];
              o[32] := f.delta*f.delta;
              o[33] := -10.9153200808732*o[1];
              o[34] := 13.2781565976477 + o[33];
              o[35] := o[34]*o[7];
              o[36] := -6.9146446840086 + o[35];
              o[37] := o[2]*o[36];
              o[38] := -2.53086309554280 + o[37];
              o[39] := o[38]*f.tau;
              o[40] := o[18]*o[5]*f.tau;

              f.f := -15.7328452902390 + f.tau*(20.9443969743070 + (-7.6867707878716
                 + o[3]*(2.61859477879540 + o[4]*(-2.80807811486200 + o[1]*(
                1.20533696965170 - 0.0084566812812502*o[6]))))*f.tau) + f.delta*(o[14]
                 + f.delta*(0.38493460186671 + o[1]*(-0.85214708824206 + o[2]*(
                4.8972281541877 + (-3.05026172569650 + o[15]*(0.039420536879154 +
                0.125584084243080*o[2]))*f.tau)) + f.delta*(-0.279993296987100 + o[1]
                *(1.38997995694600 + o[1]*(-2.01899150235700 + o[16]*(-0.0082147637173963
                 - 0.47596035734923*o[17]))) + f.delta*(0.043984074473500 + o[1]*(-0.44476435428739
                 + o[1]*(0.90572070719733 + 0.70522450087967*o[19])) + f.delta*(f.delta
                *(-0.0221754008730960 + o[1]*(0.094260751665092 + 0.164362784479610*o[
                21]) + f.delta*(-0.0135033722413480*o[1] + f.delta*(-0.0148343453524720
                *o[22] + f.delta*(o[1]*(0.00057922953628084 + 0.0032308904703711*o[21])
                 + f.delta*(0.000080964802996215 - 0.000044923899061815*f.delta*o[22]
                 - 0.000165576797950370*f.tau))))) + (0.107705126263320 + o[1]*(-0.32913623258954
                 - 0.50871062041158*o[20]))*f.tau))))) + 1.06580700285130*
                Modelica.Math.log(f.delta);

              f.fdelta := (1.06580700285130 + f.delta*(o[14] + f.delta*(
                0.76986920373342 + o[31] + f.delta*(-0.83997989096130 + o[1]*(
                4.1699398708380 + o[1]*(-6.0569745070710 + o[16]*(-0.0246442911521889
                 - 1.42788107204769*o[17]))) + f.delta*(0.175936297894000 + o[1]*(-1.77905741714956
                 + o[1]*(3.6228828287893 + 2.82089800351868*o[19])) + f.delta*(f.delta
                *(-0.133052405238576 + o[1]*(0.56556450999055 + 0.98617670687766*o[21])
                 + f.delta*(-0.094523605689436*o[1] + f.delta*(-0.118674762819776*o[
                22] + f.delta*(o[1]*(0.0052130658265276 + 0.0290780142333399*o[21])
                 + f.delta*(0.00080964802996215 - 0.00049416288967996*f.delta*o[22]
                 - 0.00165576797950370*f.tau))))) + (0.53852563131660 + o[1]*(-1.64568116294770
                 - 2.54355310205790*o[20]))*f.tau))))))/f.delta;

              f.fdeltadelta := (-1.06580700285130 + o[32]*(0.76986920373342 + o[31]
                 + f.delta*(-1.67995978192260 + o[1]*(8.3398797416760 + o[1]*(-12.1139490141420
                 + o[16]*(-0.049288582304378 - 2.85576214409538*o[17]))) + f.delta*(
                0.52780889368200 + o[1]*(-5.3371722514487 + o[1]*(10.8686484863680 +
                8.4626940105560*o[19])) + f.delta*(f.delta*(-0.66526202619288 + o[1]*
                (2.82782254995276 + 4.9308835343883*o[21]) + f.delta*(-0.56714163413662
                *o[1] + f.delta*(-0.83072333973843*o[22] + f.delta*(o[1]*(
                0.041704526612220 + 0.232624113866719*o[21]) + f.delta*(
                0.0072868322696594 - 0.0049416288967996*f.delta*o[22] -
                0.0149019118155333*f.tau))))) + (2.15410252526640 + o[1]*(-6.5827246517908
                 - 10.1742124082316*o[20]))*f.tau)))))/o[32];

              f.ftau := 20.9443969743070 + (-15.3735415757432 + o[3]*(
                18.3301634515678 + o[4]*(-28.0807811486200 + o[1]*(14.4640436358204
                 - 0.194503669468755*o[6]))))*f.tau + f.delta*(o[39] + f.delta*(f.tau
                *(-1.70429417648412 + o[2]*(29.3833689251262 + (-21.3518320798755 + o[
                15]*(0.86725181134139 + 3.2651861903201*o[2]))*f.tau)) + f.delta*((
                2.77995991389200 + o[1]*(-8.0759660094280 + o[16]*(-0.131436219478341
                 - 12.3749692910800*o[17])))*f.tau + f.delta*((-0.88952870857478 + o[
                1]*(3.6228828287893 + 18.3358370228714*o[19]))*f.tau + f.delta*(
                0.107705126263320 + o[1]*(-0.98740869776862 - 13.2264761307011*o[20])
                 + f.delta*((0.188521503330184 + 4.2734323964699*o[21])*f.tau + f.delta
                *(-0.0270067444826960*f.tau + f.delta*(-0.38569297916427*o[40] + f.delta
                *(f.delta*(-0.000165576797950370 - 0.00116802137560719*f.delta*o[40])
                 + (0.00115845907256168 + 0.084003152229649*o[21])*f.tau)))))))));

              f.ftautau := -15.3735415757432 + o[3]*(109.980980709407 + o[4]*(-252.727030337580
                 + o[1]*(159.104479994024 - 4.2790807283126*o[6]))) + f.delta*(-2.53086309554280
                 + o[2]*(-34.573223420043 + (185.894192367068 - 174.645121293971*o[1])
                *o[7]) + f.delta*(-1.70429417648412 + o[2]*(146.916844625631 + (-128.110992479253
                 + o[15]*(18.2122880381691 + 81.629654758002*o[2]))*f.tau) + f.delta*
                (2.77995991389200 + o[1]*(-24.2278980282840 + o[16]*(-1.97154329217511
                 - 309.374232277000*o[17])) + f.delta*(-0.88952870857478 + o[1]*(
                10.8686484863680 + 458.39592557179*o[19]) + f.delta*(f.delta*(
                0.188521503330184 + 106.835809911747*o[21] + f.delta*(-0.0270067444826960
                 + f.delta*(-9.6423244791068*o[21] + f.delta*(0.00115845907256168 +
                2.10007880574121*o[21] - 0.0292005343901797*o[21]*o[32])))) + (-1.97481739553724
                 - 330.66190326753*o[20])*f.tau)))));

              f.fdeltatau := o[39] + f.delta*(f.tau*(-3.4085883529682 + o[2]*(
                58.766737850252 + (-42.703664159751 + o[15]*(1.73450362268278 +
                6.5303723806402*o[2]))*f.tau)) + f.delta*((8.3398797416760 + o[1]*(-24.2278980282840
                 + o[16]*(-0.39430865843502 - 37.124907873240*o[17])))*f.tau + f.delta
                *((-3.5581148342991 + o[1]*(14.4915313151573 + 73.343348091486*o[19]))
                *f.tau + f.delta*(0.53852563131660 + o[1]*(-4.9370434888431 -
                66.132380653505*o[20]) + f.delta*((1.13112901998110 +
                25.6405943788192*o[21])*f.tau + f.delta*(-0.189047211378872*f.tau + f.delta
                *(-3.08554383331418*o[40] + f.delta*(f.delta*(-0.00165576797950370 -
                0.0128482351316791*f.delta*o[40]) + (0.0104261316530551 +
                0.75602837006684*o[21])*f.tau))))))));
            end f3;

            function g5 "Base function for region 5: g(p,T)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output Modelica.Media.Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          protected
              Real[11] o "Vector of auxiliary variables";
            algorithm
              assert(p > 0.0,
                "IF97 medium function g5 called with too low pressure\n" + "p = " +
                String(p) + " Pa <=  0.0 Pa");
              assert(p <= data.PLIMIT5, "IF97 medium function g5: input pressure (= "
                 + String(p) + " Pa) is higher than 10 Mpa in region 5");
              assert(T <= 2273.15, "IF97 medium function g5: input temperature (= "
                 + String(T) + " K) is higher than limit of 2273.15K in region 5");
              g.p := p;
              g.T := T;
              g.R := data.RH2O;
              g.pi := p/data.PSTAR5;
              g.tau := data.TSTAR5/T;
              o[1] := g.tau*g.tau;
              o[2] := -0.0045942820899910*o[1];
              o[3] := 0.00217746787145710 + o[2];
              o[4] := o[3]*g.tau;
              o[5] := o[1]*g.tau;
              o[6] := o[1]*o[1];
              o[7] := o[6]*o[6];
              o[8] := o[7]*g.tau;
              o[9] := -7.9449656719138e-6*o[8];
              o[10] := g.pi*g.pi;
              o[11] := -0.0137828462699730*o[1];

              g.g := g.pi*(-0.000125631835895920 + o[4] + g.pi*(-3.9724828359569e-6*o[
                8] + 1.29192282897840e-7*o[5]*g.pi)) + (-0.0248051489334660 + g.tau*(
                0.36901534980333 + g.tau*(-3.11613182139250 + g.tau*(-13.1799836742010
                 + (6.8540841634434 - 0.32961626538917*g.tau)*g.tau +
                Modelica.Math.log(g.pi)))))/o[5];

              g.gpi := (1.0 + g.pi*(-0.000125631835895920 + o[4] + g.pi*(o[9] +
                3.8757684869352e-7*o[5]*g.pi)))/g.pi;

              g.gpipi := (-1.00000000000000 + o[10]*(o[9] + 7.7515369738704e-7*o[5]*g.pi))
                /o[10];

              g.gtau := g.pi*(0.00217746787145710 + o[11] + g.pi*(-0.000035752345523612
                *o[7] + 3.8757684869352e-7*o[1]*g.pi)) + (0.074415446800398 + g.tau*(
                -0.73803069960666 + (3.11613182139250 + o[1]*(6.8540841634434 -
                0.65923253077834*g.tau))*g.tau))/o[6];

              g.gtautau := (-0.297661787201592 + g.tau*(2.21409209881998 + (-6.2322636427850
                 - 0.65923253077834*o[5])*g.tau))/(o[6]*g.tau) + g.pi*(-0.0275656925399460
                *g.tau + g.pi*(-0.000286018764188897*o[1]*o[6]*g.tau +
                7.7515369738704e-7*g.pi*g.tau));

              g.gtaupi := 0.00217746787145710 + o[11] + g.pi*(-0.000071504691047224*o[
                7] + 1.16273054608056e-6*o[1]*g.pi);
            end g5;

            function tph1 "Inverse function for region 1: T(p,h)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.Temperature T "Temperature (K)";
          protected
              Real pi "Dimensionless pressure";
              Real eta1 "Dimensionless specific enthalpy";
              Real[3] o "Vector of auxiliary variables";
            algorithm
              assert(p > triple.ptriple,
                "IF97 medium function tph1 called with too low pressure\n" + "p = "
                 + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              pi := p/data.PSTAR2;
              eta1 := h/data.HSTAR1 + 1.0;
              o[1] := eta1*eta1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              T := -238.724899245210 - 13.3917448726020*pi + eta1*(404.21188637945 +
                43.211039183559*pi + eta1*(113.497468817180 - 54.010067170506*pi +
                eta1*(30.5358922039160*pi + eta1*(-6.5964749423638*pi + o[1]*(-5.8457616048039
                 + o[2]*(pi*(0.0093965400878363 + (-0.0000258586412820730 +
                6.6456186191635e-8*pi)*pi) + o[2]*o[3]*(-0.000152854824131400 + o[1]*
                o[3]*(-1.08667076953770e-6 + pi*(1.15736475053400e-7 + pi*(-4.0644363084799e-9
                 + pi*(8.0670734103027e-11 + pi*(-9.3477771213947e-13 + (
                5.8265442020601e-15 - 1.50201859535030e-17*pi)*pi))))))))))));
            end tph1;

            function tps1 "Inverse function for region 1: T(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.Temperature T "Temperature (K)";
          protected
              constant SI.Pressure pstar=1.0e6;
              constant SI.SpecificEntropy sstar=1.0e3;
              Real pi "Dimensionless pressure";
              Real sigma1 "Dimensionless specific entropy";
              Real[6] o "Vector of auxiliary variables";
            algorithm
              pi := p/pstar;
              assert(p > triple.ptriple,
                "IF97 medium function tps1 called with too low pressure\n" + "p = "
                 + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");

              sigma1 := s/sstar + 2.0;
              o[1] := sigma1*sigma1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              o[4] := o[3]*o[3];
              o[5] := o[4]*o[4];
              o[6] := o[1]*o[2]*o[4];

              T := 174.782680583070 + sigma1*(34.806930892873 + sigma1*(
                6.5292584978455 + (0.33039981775489 + o[3]*(-1.92813829231960e-7 -
                2.49091972445730e-23*o[2]*o[4]))*sigma1)) + pi*(-0.261076364893320 +
                pi*(0.00056608900654837 + pi*(o[1]*o[3]*(2.64004413606890e-13 +
                7.8124600459723e-29*o[6]) - 3.07321999036680e-31*o[5]*pi) + sigma1*(-0.00032635483139717
                 + sigma1*(0.000044778286690632 + o[1]*o[2]*(-5.1322156908507e-10 -
                4.2522657042207e-26*o[6])*sigma1))) + sigma1*(0.225929659815860 +
                sigma1*(-0.064256463395226 + sigma1*(0.0078876289270526 + o[3]*sigma1
                *(3.5672110607366e-10 + 1.73324969948950e-24*o[1]*o[4]*sigma1)))));
            end tps1;

            function tph2 "Reverse function for region 2: T(p,h)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.Temperature T "Temperature (K)";
          protected
              Real pi "Dimensionless pressure";
              Real pi2b "Dimensionless pressure";
              Real pi2c "Dimensionless pressure";
              Real eta "Dimensionless specific enthalpy";
              Real etabc "Dimensionless specific enthalpy";
              Real eta2a "Dimensionless specific enthalpy";
              Real eta2b "Dimensionless specific enthalpy";
              Real eta2c "Dimensionless specific enthalpy";
              Real[8] o "Vector of auxiliary variables";
            algorithm
              pi := p*data.IPSTAR;
              eta := h*data.IHSTAR;
              etabc := h*1.0e-3;
              if (pi < 4.0) then
                eta2a := eta - 2.1;
                o[1] := eta2a*eta2a;
                o[2] := o[1]*o[1];
                o[3] := pi*pi;
                o[4] := o[3]*o[3];
                o[5] := o[3]*pi;
                T := 1089.89523182880 + (1.84457493557900 - 0.0061707422868339*pi)*pi
                   + eta2a*(849.51654495535 - 4.1792700549624*pi + eta2a*(-107.817480918260
                   + (6.2478196935812 - 0.310780466295830*pi)*pi + eta2a*(
                  33.153654801263 - 17.3445631081140*pi + o[2]*(-7.4232016790248 + pi
                  *(-200.581768620960 + 11.6708730771070*pi) + o[1]*(271.960654737960
                  *pi + o[1]*(-455.11318285818*pi + eta2a*(1.38657242832260*o[4] + o[
                  1]*o[2]*(3091.96886047550*pi + o[1]*(11.7650487243560 + o[2]*(-13551.3342407750
                  *o[5] + o[2]*(-62.459855192507*o[3]*o[4]*pi + o[2]*(o[4]*(
                  235988.325565140 + 7399.9835474766*pi) + o[1]*(19127.7292396600*o[3]
                  *o[4] + o[1]*(o[3]*(1.28127984040460e8 - 551966.97030060*o[5]) + o[
                  1]*(-9.8554909623276e8*o[3] + o[1]*(2.82245469730020e9*o[3] + o[1]*
                  (o[3]*(-3.5948971410703e9 + 3.7154085996233e6*o[5]) + o[1]*pi*(
                  252266.403578720 + pi*(1.72273499131970e9 + pi*(1.28487346646500e7
                   + (-1.31052365450540e7 - 415351.64835634*o[3])*pi))))))))))))))))))));
              elseif (pi < (0.12809002730136e-03*etabc - 0.67955786399241)*etabc +
                  0.90584278514723e3) then
                eta2b := eta - 2.6;
                pi2b := pi - 2.0;
                o[1] := pi2b*pi2b;
                o[2] := o[1]*pi2b;
                o[3] := o[1]*o[1];
                o[4] := eta2b*eta2b;
                o[5] := o[4]*o[4];
                o[6] := o[4]*o[5];
                o[7] := o[5]*o[5];
                T := 1489.50410795160 + 0.93747147377932*pi2b + eta2b*(
                  743.07798314034 + o[2]*(0.000110328317899990 - 1.75652339694070e-18
                  *o[1]*o[3]) + eta2b*(-97.708318797837 + pi2b*(3.3593118604916 +
                  pi2b*(-0.0218107553247610 + pi2b*(0.000189552483879020 + (
                  2.86402374774560e-7 - 8.1456365207833e-14*o[2])*pi2b))) + o[5]*(
                  3.3809355601454*pi2b + o[4]*(-0.108297844036770*o[1] + o[5]*(
                  2.47424647056740 + (0.168445396719040 + o[1]*(0.00308915411605370
                   - 0.0000107798573575120*pi2b))*pi2b + o[6]*(-0.63281320016026 +
                  pi2b*(0.73875745236695 + (-0.046333324635812 + o[1]*(-0.000076462712454814
                   + 2.82172816350400e-7*pi2b))*pi2b) + o[6]*(1.13859521296580 + pi2b
                  *(-0.47128737436186 + o[1]*(0.00135555045549490 + (
                  0.0000140523928183160 + 1.27049022719450e-6*pi2b)*pi2b)) + o[5]*(-0.47811863648625
                   + (0.150202731397070 + o[2]*(-0.0000310838143314340 + o[1]*(-1.10301392389090e-8
                   - 2.51805456829620e-11*pi2b)))*pi2b + o[5]*o[7]*(
                  0.0085208123431544 + pi2b*(-0.00217641142197500 + pi2b*(
                  0.000071280351959551 + o[1]*(-1.03027382121030e-6 + (
                  7.3803353468292e-8 + 8.6934156344163e-15*o[3])*pi2b))))))))))));
              else
                eta2c := eta - 1.8;
                pi2c := pi + 25.0;
                o[1] := pi2c*pi2c;
                o[2] := o[1]*o[1];
                o[3] := o[1]*o[2]*pi2c;
                o[4] := 1/o[3];
                o[5] := o[1]*o[2];
                o[6] := eta2c*eta2c;
                o[7] := o[2]*o[2];
                o[8] := o[6]*o[6];
                T := eta2c*((859777.22535580 + o[1]*(482.19755109255 +
                  1.12615974072300e-12*o[5]))/o[1] + eta2c*((-5.8340131851590e11 + (
                  2.08255445631710e10 + 31081.0884227140*o[2])*pi2c)/o[5] + o[6]*(o[8]
                  *(o[6]*(1.23245796908320e-7*o[5] + o[6]*(-1.16069211309840e-6*o[5]
                   + o[8]*(0.0000278463670885540*o[5] + (-0.00059270038474176*o[5] +
                  0.00129185829918780*o[5]*o[6])*o[8]))) - 10.8429848800770*pi2c) + o[
                  4]*(7.3263350902181e12 + o[7]*(3.7966001272486 + (-0.045364172676660
                   - 1.78049822406860e-11*o[2])*pi2c))))) + o[4]*(-3.2368398555242e12
                   + pi2c*(3.5825089945447e11 + pi2c*(-1.07830682174700e10 + o[1]*
                  pi2c*(610747.83564516 + pi2c*(-25745.7236041700 + (1208.23158659360
                   + 1.45591156586980e-13*o[5])*pi2c)))));
              end if;
            end tph2;

            function tps2a "Reverse function for region 2a: T(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.Temperature T "Temperature (K)";
          protected
              Real[12] o "Vector of auxiliary variables";
              constant Real IPSTAR=1.0e-6 "Scaling variable";
              constant Real ISSTAR2A=1/2000.0 "Scaling variable";
              Real pi "Dimensionless pressure";
              Real sigma2a "Dimensionless specific entropy";
            algorithm
              pi := p*IPSTAR;
              sigma2a := s*ISSTAR2A - 2.0;
              o[1] := pi^0.5;
              o[2] := sigma2a*sigma2a;
              o[3] := o[2]*o[2];
              o[4] := o[3]*o[3];
              o[5] := o[4]*o[4];
              o[6] := pi^0.25;
              o[7] := o[2]*o[4]*o[5];
              o[8] := 1/o[7];
              o[9] := o[3]*sigma2a;
              o[10] := o[2]*o[3]*sigma2a;
              o[11] := o[3]*o[4]*sigma2a;
              o[12] := o[2]*sigma2a;
              T := ((-392359.83861984 + (515265.73827270 + o[3]*(40482.443161048 + o[
                2]*o[3]*(-321.93790923902 + o[2]*(96.961424218694 - 22.8678463717730*
                sigma2a))))*sigma2a)/(o[4]*o[5]) + o[6]*((-449429.14124357 + o[3]*(-5011.8336020166
                 + 0.35684463560015*o[4]*sigma2a))/(o[2]*o[5]*sigma2a) + o[6]*(o[8]*(
                44235.335848190 + o[9]*(-13673.3888117080 + o[3]*(421632.60207864 + (
                22516.9258374750 + o[10]*(474.42144865646 - 149.311307976470*sigma2a))
                *sigma2a))) + o[6]*((-197811.263204520 - 23554.3994707600*sigma2a)/(o[
                2]*o[3]*o[4]*sigma2a) + o[6]*((-19070.6163020760 + o[11]*(
                55375.669883164 + (3829.3691437363 - 603.91860580567*o[2])*o[3]))*o[8]
                 + o[6]*((1936.31026203310 + o[2]*(4266.0643698610 + o[2]*o[3]*o[4]*(
                -5978.0638872718 - 704.01463926862*o[9])))/(o[2]*o[4]*o[5]*sigma2a)
                 + o[1]*((338.36784107553 + o[12]*(20.8627866351870 + (
                0.033834172656196 - 0.000043124428414893*o[12])*o[3]))*sigma2a + o[6]
                *(166.537913564120 + sigma2a*(-139.862920558980 + o[3]*(-0.78849547999872
                 + (0.072132411753872 + o[3]*(-0.0059754839398283 + (-0.0000121413589539040
                 + 2.32270967338710e-7*o[2])*o[3]))*sigma2a)) + o[6]*(-10.5384635661940
                 + o[3]*(2.07189254965020 + (-0.072193155260427 + 2.07498870811200e-7
                *o[4])*o[9]) + o[6]*(o[6]*(o[12]*(0.210375278936190 +
                0.000256812397299990*o[3]*o[4]) + (-0.0127990029337810 -
                8.2198102652018e-6*o[11])*o[6]*o[9]) + o[10]*(-0.0183406579113790 +
                2.90362723486960e-7*o[2]*o[4]*sigma2a)))))))))))/(o[1]*pi);
            end tps2a;

            function tps2b "Reverse function for region 2b: T(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.Temperature T "Temperature (K)";
          protected
              Real[8] o "Vector of auxiliary variables";
              constant Real IPSTAR=1.0e-6 "Scaling variable";
              constant Real ISSTAR2B=1/785.3 "Scaling variable";
              Real pi "Dimensionless pressure";
              Real sigma2b "Dimensionless specific entropy";
            algorithm
              pi := p*IPSTAR;
              sigma2b := 10.0 - s*ISSTAR2B;
              o[1] := pi*pi;
              o[2] := o[1]*o[1];
              o[3] := sigma2b*sigma2b;
              o[4] := o[3]*o[3];
              o[5] := o[4]*o[4];
              o[6] := o[3]*o[5]*sigma2b;
              o[7] := o[3]*o[5];
              o[8] := o[3]*sigma2b;
              T := (316876.65083497 + 20.8641758818580*o[6] + pi*(-398593.99803599 -
                21.8160585188770*o[6] + pi*(223697.851942420 + (-2784.17034458170 +
                9.9207436071480*o[7])*sigma2b + pi*(-75197.512299157 + (
                2970.86059511580 + o[7]*(-3.4406878548526 + 0.38815564249115*sigma2b))
                *sigma2b + pi*(17511.2950857500 + sigma2b*(-1423.71128544490 + (
                1.09438033641670 + 0.89971619308495*o[4])*o[4]*sigma2b) + pi*(-3375.9740098958
                 + (471.62885818355 + o[4]*(-1.91882419936790 + o[8]*(
                0.41078580492196 - 0.33465378172097*sigma2b)))*sigma2b + pi*(
                1387.00347775050 + sigma2b*(-406.63326195838 + sigma2b*(
                41.727347159610 + o[3]*(2.19325494345320 + sigma2b*(-1.03200500090770
                 + (0.35882943516703 + 0.0052511453726066*o[8])*sigma2b)))) + pi*(
                12.8389164507050 + sigma2b*(-2.86424372193810 + sigma2b*(
                0.56912683664855 + (-0.099962954584931 + o[4]*(-0.0032632037778459 +
                0.000233209225767230*sigma2b))*sigma2b)) + pi*(-0.153348098574500 + (
                0.0290722882399020 + 0.00037534702741167*o[4])*sigma2b + pi*(
                0.00172966917024110 + (-0.00038556050844504 - 0.000035017712292608*o[
                3])*sigma2b + pi*(-0.0000145663936314920 + 5.6420857267269e-6*sigma2b
                 + pi*(4.1286150074605e-8 + (-2.06846711188240e-8 +
                1.64093936747250e-9*sigma2b)*sigma2b))))))))))))/(o[1]*o[2]);
            end tps2b;

            function tps2c "Reverse function for region 2c: T(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.Temperature T "Temperature (K)";
          protected
              constant Real IPSTAR=1.0e-6 "Scaling variable";
              constant Real ISSTAR2C=1/2925.1 "Scaling variable";
              Real pi "Dimensionless pressure";
              Real sigma2c "Dimensionless specific entropy";
              Real[3] o "Vector of auxiliary variables";
            algorithm
              pi := p*IPSTAR;
              sigma2c := 2.0 - s*ISSTAR2C;
              o[1] := pi*pi;
              o[2] := sigma2c*sigma2c;
              o[3] := o[2]*o[2];
              T := (909.68501005365 + 2404.56670884200*sigma2c + pi*(-591.62326387130
                 + pi*(541.45404128074 + sigma2c*(-270.983084111920 + (
                979.76525097926 - 469.66772959435*sigma2c)*sigma2c) + pi*(
                14.3992746047230 + (-19.1042042304290 + o[2]*(5.3299167111971 -
                21.2529753759340*sigma2c))*sigma2c + pi*(-0.311473344137600 + (
                0.60334840894623 - 0.042764839702509*sigma2c)*sigma2c + pi*(
                0.0058185597255259 + (-0.0145970082847530 + 0.0056631175631027*o[3])*
                sigma2c + pi*(-0.000076155864584577 + sigma2c*(0.000224403429193320
                 - 0.0000125610950134130*o[2]*sigma2c) + pi*(6.3323132660934e-7 + (-2.05419896753750e-6
                 + 3.6405370390082e-8*sigma2c)*sigma2c + pi*(-2.97598977892150e-9 +
                1.01366185297630e-8*sigma2c + pi*(5.9925719692351e-12 + sigma2c*(-2.06778701051640e-11
                 + o[2]*(-2.08742781818860e-11 + (1.01621668250890e-10 -
                1.64298282813470e-10*sigma2c)*sigma2c))))))))))))/o[1];
            end tps2c;

            function tps2 "Reverse function for region 2: T(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.Temperature T "Temperature (K)";
          protected
              Real pi "Dimensionless pressure";
              constant SI.SpecificEntropy SLIMIT=5.85e3
              "Subregion boundary specific entropy between regions 2a and 2b";
            algorithm
              if p < 4.0e6 then
                T := tps2a(p, s);
              elseif s > SLIMIT then
                T := tps2b(p, s);
              else
                T := tps2c(p, s);
              end if;
            end tps2;

            function tsat
            "Region 4 saturation temperature as a function of pressure"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.Temperature t_sat "Temperature";
          protected
              Real pi "Dimensionless pressure";
              Real[20] o "Vector of auxiliary variables";
            algorithm
              assert(p > triple.ptriple,
                "IF97 medium function tsat called with too low pressure\n" + "p = "
                 + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              //  assert(p <= data.PCRIT,
              //    "tsat: input pressure is higher than the critical point pressure");
              pi := min(p, data.PCRIT)*data.IPSTAR;
              o[1] := pi^0.25;
              o[2] := -3.2325550322333e6*o[1];
              o[3] := pi^0.5;
              o[4] := -724213.16703206*o[3];
              o[5] := 405113.40542057 + o[2] + o[4];
              o[6] := -17.0738469400920*o[1];
              o[7] := 14.9151086135300 + o[3] + o[6];
              o[8] := -4.0*o[5]*o[7];
              o[9] := 12020.8247024700*o[1];
              o[10] := 1167.05214527670*o[3];
              o[11] := -4823.2657361591 + o[10] + o[9];
              o[12] := o[11]*o[11];
              o[13] := o[12] + o[8];
              o[14] := o[13]^0.5;
              o[15] := -o[14];
              o[16] := -12020.8247024700*o[1];
              o[17] := -1167.05214527670*o[3];
              o[18] := 4823.2657361591 + o[15] + o[16] + o[17];
              o[19] := 1/o[18];
              o[20] := 2.0*o[19]*o[5];

              t_sat := 0.5*(650.17534844798 + o[20] - (-4.0*(-0.238555575678490 +
                1300.35069689596*o[19]*o[5]) + (650.17534844798 + o[20])^2.0)^0.5);
              annotation (derivative=tsat_der);
            end tsat;

            function dtsatofp
            "Derivative of saturation temperature w.r.t. pressure"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output Real dtsat(unit="K/Pa") "Derivative of T w.r.t. p";
          protected
              Real pi "Dimensionless pressure";
              Real[49] o "Vector of auxiliary variables";
            algorithm
              pi := max(Modelica.Constants.small, p*data.IPSTAR);
              o[1] := pi^0.75;
              o[2] := 1/o[1];
              o[3] := -4.268461735023*o[2];
              o[4] := sqrt(pi);
              o[5] := 1/o[4];
              o[6] := 0.5*o[5];
              o[7] := o[3] + o[6];
              o[8] := pi^0.25;
              o[9] := -3.2325550322333e6*o[8];
              o[10] := -724213.16703206*o[4];
              o[11] := 405113.40542057 + o[10] + o[9];
              o[12] := -4*o[11]*o[7];
              o[13] := -808138.758058325*o[2];
              o[14] := -362106.58351603*o[5];
              o[15] := o[13] + o[14];
              o[16] := -17.073846940092*o[8];
              o[17] := 14.91510861353 + o[16] + o[4];
              o[18] := -4*o[15]*o[17];
              o[19] := 3005.2061756175*o[2];
              o[20] := 583.52607263835*o[5];
              o[21] := o[19] + o[20];
              o[22] := 12020.82470247*o[8];
              o[23] := 1167.0521452767*o[4];
              o[24] := -4823.2657361591 + o[22] + o[23];
              o[25] := 2.0*o[21]*o[24];
              o[26] := o[12] + o[18] + o[25];
              o[27] := -4.0*o[11]*o[17];
              o[28] := o[24]*o[24];
              o[29] := o[27] + o[28];
              o[30] := sqrt(o[29]);
              o[31] := 1/o[30];
              o[32] := (-o[30]);
              o[33] := -12020.82470247*o[8];
              o[34] := -1167.0521452767*o[4];
              o[35] := 4823.2657361591 + o[32] + o[33] + o[34];
              o[36] := o[30];
              o[37] := -4823.2657361591 + o[22] + o[23] + o[36];
              o[38] := o[37]*o[37];
              o[39] := 1/o[38];
              o[40] := -1.72207339365771*o[30];
              o[41] := 21592.2055343628*o[8];
              o[42] := o[30]*o[8];
              o[43] := -8192.87114842946*o[4];
              o[44] := -0.510632954559659*o[30]*o[4];
              o[45] := -3100.02526152368*o[1];
              o[46] := pi;
              o[47] := 1295.95640782102*o[46];
              o[48] := 2862.09212505088 + o[40] + o[41] + o[42] + o[43] + o[44] + o[
                45] + o[47];
              o[49] := 1/(o[35]*o[35]);
              dtsat := data.IPSTAR*0.5*((2.0*o[15])/o[35] - 2.*o[11]*(-3005.2061756175
                *o[2] - 0.5*o[26]*o[31] - 583.52607263835*o[5])*o[49] - (
                20953.46356643991*(o[39]*(1295.95640782102 + 5398.05138359071*o[2] +
                0.25*o[2]*o[30] - 0.861036696828853*o[26]*o[31] - 0.255316477279829*o[
                26]*o[31]*o[4] - 4096.43557421473*o[5] - 0.255316477279829*o[30]*o[5]
                 - 2325.01894614276/o[8] + 0.5*o[26]*o[31]*o[8]) - 2.0*(o[19] + o[20]
                 + 0.5*o[26]*o[31])*o[48]*o[37]^(-3)))/sqrt(o[39]*o[48]));
            end dtsatofp;

            function tsat_der "Derivative function for tsat"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input Real der_p(unit="Pa/s") "Pressure derivative";
              output Real der_tsat(unit="K/s") "Temperature derivative";
          protected
              Real dtp;
            algorithm
              dtp := dtsatofp(p);
              der_tsat := dtp*der_p;
            end tsat_der;

            function psat
            "Region 4 saturation pressure as a function of temperature"

              extends Modelica.Icons.Function;
              input SI.Temperature T "Temperature (K)";
              output SI.Pressure p_sat "Pressure";
          protected
              Real[8] o "Vector of auxiliary variables";
              Real Tlim=min(T, data.TCRIT);
            algorithm
              assert(T >= 273.16, "IF97 medium function psat: input temperature (= "
                 + String(triple.ptriple) + " K).\n" +
                "lower than the triple point temperature 273.16 K");
              o[1] := -650.17534844798 + Tlim;
              o[2] := 1/o[1];
              o[3] := -0.238555575678490*o[2];
              o[4] := o[3] + Tlim;
              o[5] := -4823.2657361591*o[4];
              o[6] := o[4]*o[4];
              o[7] := 14.9151086135300*o[6];
              o[8] := 405113.40542057 + o[5] + o[7];
              p_sat := 16.0e6*o[8]*o[8]*o[8]*o[8]*1/(3.2325550322333e6 -
                12020.8247024700*o[4] + 17.0738469400920*o[6] + (-4.0*(-724213.16703206
                 + 1167.05214527670*o[4] + o[6])*o[8] + (-3.2325550322333e6 +
                12020.8247024700*o[4] - 17.0738469400920*o[6])^2.0)^0.5)^4.0;
              annotation (derivative=psat_der);
            end psat;

            function dptofT
            "Derivative of pressure w.r.t. temperature along the saturation pressure curve"

              extends Modelica.Icons.Function;
              input SI.Temperature T "Temperature (K)";
              output Real dpt(unit="Pa/K") "Temperature derivative of pressure";
          protected
              Real[31] o "Vector of auxiliary variables";
              Real Tlim "Temperature limited to TCRIT";
            algorithm
              Tlim := min(T, data.TCRIT);
              o[1] := -650.17534844798 + Tlim;
              o[2] := 1/o[1];
              o[3] := -0.238555575678490*o[2];
              o[4] := o[3] + Tlim;
              o[5] := -4823.2657361591*o[4];
              o[6] := o[4]*o[4];
              o[7] := 14.9151086135300*o[6];
              o[8] := 405113.40542057 + o[5] + o[7];
              o[9] := o[8]*o[8];
              o[10] := o[9]*o[9];
              o[11] := o[1]*o[1];
              o[12] := 1/o[11];
              o[13] := 0.238555575678490*o[12];
              o[14] := 1.00000000000000 + o[13];
              o[15] := 12020.8247024700*o[4];
              o[16] := -17.0738469400920*o[6];
              o[17] := -3.2325550322333e6 + o[15] + o[16];
              o[18] := -4823.2657361591*o[14];
              o[19] := 29.8302172270600*o[14]*o[4];
              o[20] := o[18] + o[19];
              o[21] := 1167.05214527670*o[4];
              o[22] := -724213.16703206 + o[21] + o[6];
              o[23] := o[17]*o[17];
              o[24] := -4.0000000000000*o[22]*o[8];
              o[25] := o[23] + o[24];
              o[26] := sqrt(o[25]);
              o[27] := -12020.8247024700*o[4];
              o[28] := 17.0738469400920*o[6];
              o[29] := 3.2325550322333e6 + o[26] + o[27] + o[28];
              o[30] := o[29]*o[29];
              o[31] := o[30]*o[30];
              dpt := 1e6*((-64.0*o[10]*(-12020.8247024700*o[14] + 34.147693880184*o[
                14]*o[4] + (0.5*(-4.0*o[20]*o[22] + 2.00000000000000*o[17]*(
                12020.8247024700*o[14] - 34.147693880184*o[14]*o[4]) - 4.0*(
                1167.05214527670*o[14] + 2.0*o[14]*o[4])*o[8]))/o[26]))/(o[29]*o[31])
                 + (64.*o[20]*o[8]*o[9])/o[31]);
            end dptofT;

            function psat_der "Derivative function for psat"
              extends Modelica.Icons.Function;
              input SI.Temperature T "Temperature (K)";
              input Real der_T(unit="K/s") "Temperature derivative";
              output Real der_psat(unit="Pa/s") "Pressure";
          protected
              Real dpt;
            algorithm
              dpt := dptofT(T);
              der_psat := dpt*der_T;
            end psat_der;

            function h3ab_p "Region 3 a b boundary for pressure/enthalpy"
              extends Modelica.Icons.Function;
              output SI.SpecificEnthalpy h "Enthalpy";
              input SI.Pressure p "Pressure";
          protected
              constant Real[:] n={0.201464004206875e4,0.374696550136983e1,-0.219921901054187e-1,
                  0.875131686009950e-4};
              constant SI.SpecificEnthalpy hstar=1000 "Normalization enthalpy";
              constant SI.Pressure pstar=1e6 "Normalization pressure";
              Real pi=p/pstar "Normalized specific pressure";

            algorithm
              h := (n[1] + n[2]*pi + n[3]*pi^2 + n[4]*pi^3)*hstar;
              annotation (Documentation(info="<html>
      <p>
      &nbsp;Equation number 1 from:
      </p>
      <div style=\"text-align: center;\">&nbsp;[1] The international Association
      for the Properties of Water and Steam<br>
      &nbsp;Vejle, Denmark<br>
      &nbsp;August 2003<br>
      &nbsp;Supplementary Release on Backward Equations for the Functions
      T(p,h), v(p,h) and T(p,s), <br>
      &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
      the Thermodynamic Properties of<br>
      &nbsp;Water and Steam</div>
      </html>"));
            end h3ab_p;

            function T3a_ph "Region 3 a: inverse function T(p,h)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.Temp_K T "Temperature";
          protected
              constant Real[:] n={-0.133645667811215e-6,0.455912656802978e-5,-0.146294640700979e-4,
                  0.639341312970080e-2,0.372783927268847e3,-0.718654377460447e4,
                  0.573494752103400e6,-0.267569329111439e7,-0.334066283302614e-4,-0.245479214069597e-1,
                  0.478087847764996e2,0.764664131818904e-5,0.128350627676972e-2,
                  0.171219081377331e-1,-0.851007304583213e1,-0.136513461629781e-1,-0.384460997596657e-5,
                  0.337423807911655e-2,-0.551624873066791,0.729202277107470,-0.992522757376041e-2,
                  -0.119308831407288,0.793929190615421,0.454270731799386,
                  0.209998591259910,-0.642109823904738e-2,-0.235155868604540e-1,
                  0.252233108341612e-2,-0.764885133368119e-2,0.136176427574291e-1,-0.133027883575669e-1};
              constant Real[:] I={-12,-12,-12,-12,-12,-12,-12,-12,-10,-10,-10,-8,-8,-8,
                  -8,-5,-3,-2,-2,-2,-1,-1,0,0,1,3,3,4,4,10,12};
              constant Real[:] J={0,1,2,6,14,16,20,22,1,5,12,0,2,4,10,2,0,1,3,4,0,2,0,
                  1,1,0,1,0,3,4,5};
              constant SI.SpecificEnthalpy hstar=2300e3
              "Normalization enthalpy";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.Temp_K Tstar=760 "Normalization temperature";
              Real pi=p/pstar "Normalized specific pressure";
              Real eta=h/hstar "Normalized specific enthalpy";
            algorithm
              T := sum(n[i]*(pi + 0.240)^I[i]*(eta - 0.615)^J[i] for i in 1:31)*Tstar;
              annotation (Documentation(info="<html>
<p>
 &nbsp;Equation number 2 from:
</p>
 <div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
 </html>"));
            end T3a_ph;

            function T3b_ph "Region 3 b: inverse function T(p,h)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.Temp_K T "Temperature";
          protected
              constant Real[:] n={0.323254573644920e-4,-0.127575556587181e-3,-0.475851877356068e-3,
                  0.156183014181602e-2,0.105724860113781,-0.858514221132534e2,
                  0.724140095480911e3,0.296475810273257e-2,-0.592721983365988e-2,-0.126305422818666e-1,
                  -0.115716196364853,0.849000969739595e2,-0.108602260086615e-1,
                  0.154304475328851e-1,0.750455441524466e-1,0.252520973612982e-1,-0.602507901232996e-1,
                  -0.307622221350501e1,-0.574011959864879e-1,0.503471360939849e1,-0.925081888584834,
                  0.391733882917546e1,-0.773146007130190e2,0.949308762098587e4,-0.141043719679409e7,
                  0.849166230819026e7,0.861095729446704,0.323346442811720,
                  0.873281936020439,-0.436653048526683,0.286596714529479,-0.131778331276228,
                  0.676682064330275e-2};
              constant Real[:] I={-12,-12,-10,-10,-10,-10,-10,-8,-8,-8,-8,-8,-6,-6,-6,
                  -4,-4,-3,-2,-2,-1,-1,-1,-1,-1,-1,0,0,1,3,5,6,8};
              constant Real[:] J={0,1,0,1,5,10,12,0,1,2,4,10,0,1,2,0,1,5,0,4,2,4,6,10,
                  14,16,0,2,1,1,1,1,1};
              constant SI.Temp_K Tstar=860 "Normalization temperature";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.SpecificEnthalpy hstar=2800e3
              "Normalization enthalpy";
              Real pi=p/pstar "Normalized specific pressure";
              Real eta=h/hstar "Normalized specific enthalpy";
            algorithm
              T := sum(n[i]*(pi + 0.298)^I[i]*(eta - 0.720)^J[i] for i in 1:33)*Tstar;
              annotation (Documentation(info="<html>
 <p>
 &nbsp;Equation number 3 from:
</p>
 <div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
 </html>"));
            end T3b_ph;

            function v3a_ph "Region 3 a: inverse function v(p,h)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.SpecificVolume v "Specific volume";
          protected
              constant Real[:] n={0.529944062966028e-2,-0.170099690234461,
                  0.111323814312927e2,-0.217898123145125e4,-0.506061827980875e-3,
                  0.556495239685324,-0.943672726094016e1,-0.297856807561527,
                  0.939353943717186e2,0.192944939465981e-1,0.421740664704763,-0.368914126282330e7,
                  -0.737566847600639e-2,-0.354753242424366,-0.199768169338727e1,
                  0.115456297059049e1,0.568366875815960e4,0.808169540124668e-2,
                  0.172416341519307,0.104270175292927e1,-0.297691372792847,
                  0.560394465163593,0.275234661176914,-0.148347894866012,-0.651142513478515e-1,
                  -0.292468715386302e1,0.664876096952665e-1,0.352335014263844e1,-0.146340792313332e-1,
                  -0.224503486668184e1,0.110533464706142e1,-0.408757344495612e-1};
              constant Real[:] I={-12,-12,-12,-12,-10,-10,-10,-8,-8,-6,-6,-6,-4,-4,-3,
                  -2,-2,-1,-1,-1,-1,0,0,1,1,1,2,2,3,4,5,8};
              constant Real[:] J={6,8,12,18,4,7,10,5,12,3,4,22,2,3,7,3,16,0,1,2,3,0,1,
                  0,1,2,0,2,0,2,2,2};
              constant SI.Volume vstar=0.0028 "Normalization temperature";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.SpecificEnthalpy hstar=2100e3
              "Normalization enthalpy";
              Real pi=p/pstar "Normalized specific pressure";
              Real eta=h/hstar "Normalized specific enthalpy";
            algorithm
              v := sum(n[i]*(pi + 0.128)^I[i]*(eta - 0.727)^J[i] for i in 1:32)*vstar;
              annotation (Documentation(info="<html>
<p>
&nbsp;Equation number 4 from:
</p>
 <div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
 </html>"));
            end v3a_ph;

            function v3b_ph "Region 3 b: inverse function v(p,h)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.SpecificVolume v "Specific volume";
          protected
              constant Real[:] n={-0.225196934336318e-8,0.140674363313486e-7,
                  0.233784085280560e-5,-0.331833715229001e-4,0.107956778514318e-2,-0.271382067378863,
                  0.107202262490333e1,-0.853821329075382,-0.215214194340526e-4,
                  0.769656088222730e-3,-0.431136580433864e-2,0.453342167309331,-0.507749535873652,
                  -0.100475154528389e3,-0.219201924648793,-0.321087965668917e1,
                  0.607567815637771e3,0.557686450685932e-3,0.187499040029550,
                  0.905368030448107e-2,0.285417173048685,0.329924030996098e-1,
                  0.239897419685483,0.482754995951394e1,-0.118035753702231e2,
                  0.169490044091791,-0.179967222507787e-1,0.371810116332674e-1,-0.536288335065096e-1,
                  0.160697101092520e1};
              constant Real[:] I={-12,-12,-8,-8,-8,-8,-8,-8,-6,-6,-6,-6,-6,-6,-4,-4,-4,
                  -3,-3,-2,-2,-1,-1,-1,-1,0,1,1,2,2};
              constant Real[:] J={0,1,0,1,3,6,7,8,0,1,2,5,6,10,3,6,10,0,2,1,2,0,1,4,5,
                  0,0,1,2,6};
              constant SI.Volume vstar=0.0088 "Normalization temperature";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.SpecificEnthalpy hstar=2800e3
              "Normalization enthalpy";
              Real pi=p/pstar "Normalized specific pressure";
              Real eta=h/hstar "Normalized specific enthalpy";
            algorithm
              v := sum(n[i]*(pi + 0.0661)^I[i]*(eta - 0.720)^J[i] for i in 1:30)*
                vstar;
              annotation (Documentation(info="<html>
<p>
 &nbsp;Equation number 5 from:
</p>
 <div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
 </html>"));
            end v3b_ph;

            function T3a_ps "Region 3 a: inverse function T(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.Temp_K T "Temperature";
          protected
              constant Real[:] n={0.150042008263875e10,-0.159397258480424e12,
                  0.502181140217975e-3,-0.672057767855466e2,0.145058545404456e4,-0.823889534888890e4,
                  -0.154852214233853,0.112305046746695e2,-0.297000213482822e2,
                  0.438565132635495e11,0.137837838635464e-2,-0.297478527157462e1,
                  0.971777947349413e13,-0.571527767052398e-4,0.288307949778420e5,-0.744428289262703e14,
                  0.128017324848921e2,-0.368275545889071e3,0.664768904779177e16,
                  0.449359251958880e-1,-0.422897836099655e1,-0.240614376434179,-0.474341365254924e1,
                  0.724093999126110,0.923874349695897,0.399043655281015e1,
                  0.384066651868009e-1,-0.359344365571848e-2,-0.735196448821653,
                  0.188367048396131,0.141064266818704e-3,-0.257418501496337e-2,
                  0.123220024851555e-2};
              constant Real[:] I={-12,-12,-10,-10,-10,-10,-8,-8,-8,-8,-6,-6,-6,-5,-5,
                  -5,-4,-4,-4,-2,-2,-1,-1,0,0,0,1,2,2,3,8,8,10};
              constant Real[:] J={28,32,4,10,12,14,5,7,8,28,2,6,32,0,14,32,6,10,36,1,
                  4,1,6,0,1,4,0,0,3,2,0,1,2};
              constant SI.Temp_K Tstar=760 "Normalization temperature";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.SpecificEntropy sstar=4.4e3 "Normalization entropy";
              Real pi=p/pstar "Normalized specific pressure";
              Real sigma=s/sstar "Normalized specific entropy";
            algorithm
              T := sum(n[i]*(pi + 0.240)^I[i]*(sigma - 0.703)^J[i] for i in 1:33)*
                Tstar;
              annotation (Documentation(info="<html>
<p>
 &nbsp;Equation number 6 from:
</p>
 <div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
 </html>"));
            end T3a_ps;

            function T3b_ps "Region 3 b: inverse function T(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.Temp_K T "Temperature";
          protected
              constant Real[:] n={0.527111701601660,-0.401317830052742e2,
                  0.153020073134484e3,-0.224799398218827e4,-0.193993484669048,-0.140467557893768e1,
                  0.426799878114024e2,0.752810643416743,0.226657238616417e2,-0.622873556909932e3,
                  -0.660823667935396,0.841267087271658,-0.253717501764397e2,
                  0.485708963532948e3,0.880531517490555e3,0.265015592794626e7,-0.359287150025783,
                  -0.656991567673753e3,0.241768149185367e1,0.856873461222588,
                  0.655143675313458,-0.213535213206406,0.562974957606348e-2,-0.316955725450471e15,
                  -0.699997000152457e-3,0.119845803210767e-1,0.193848122022095e-4,-0.215095749182309e-4};
              constant Real[:] I={-12,-12,-12,-12,-8,-8,-8,-6,-6,-6,-5,-5,-5,-5,-5,-4,
                  -3,-3,-2,0,2,3,4,5,6,8,12,14};
              constant Real[:] J={1,3,4,7,0,1,3,0,2,4,0,1,2,4,6,12,1,6,2,0,1,1,0,24,0,
                  3,1,2};
              constant SI.Temp_K Tstar=860 "Normalization temperature";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.SpecificEntropy sstar=5.3e3 "Normalization entropy";
              Real pi=p/pstar "Normalized specific pressure";
              Real sigma=s/sstar "Normalized specific entropy";
            algorithm
              T := sum(n[i]*(pi + 0.760)^I[i]*(sigma - 0.818)^J[i] for i in 1:28)*
                Tstar;
              annotation (Documentation(info="<html>
<p>
 &nbsp;Equation number 7 from:
</p>
 <div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
</html>"));
            end T3b_ps;

            function v3a_ps "Region 3 a: inverse function v(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.SpecificVolume v "Specific volume";
          protected
              constant Real[:] n={0.795544074093975e2,-0.238261242984590e4,
                  0.176813100617787e5,-0.110524727080379e-2,-0.153213833655326e2,
                  0.297544599376982e3,-0.350315206871242e8,0.277513761062119,-0.523964271036888,
                  -0.148011182995403e6,0.160014899374266e7,0.170802322663427e13,
                  0.246866996006494e-3,0.165326084797980e1,-0.118008384666987,
                  0.253798642355900e1,0.965127704669424,-0.282172420532826e2,
                  0.203224612353823,0.110648186063513e1,0.526127948451280,
                  0.277000018736321,0.108153340501132e1,-0.744127885357893e-1,
                  0.164094443541384e-1,-0.680468275301065e-1,0.257988576101640e-1,-0.145749861944416e-3};
              constant Real[:] I={-12,-12,-12,-10,-10,-10,-10,-8,-8,-8,-8,-6,-5,-4,-3,
                  -3,-2,-2,-1,-1,0,0,0,1,2,4,5,6};
              constant Real[:] J={10,12,14,4,8,10,20,5,6,14,16,28,1,5,2,4,3,8,1,2,0,1,
                  3,0,0,2,2,0};
              constant SI.Volume vstar=0.0028 "Normalization temperature";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.SpecificEntropy sstar=4.4e3 "Normalization entropy";
              Real pi=p/pstar "Normalized specific pressure";
              Real sigma=s/sstar "Normalized specific entropy";
            algorithm
              v := sum(n[i]*(pi + 0.187)^I[i]*(sigma - 0.755)^J[i] for i in 1:28)*
                vstar;
              annotation (Documentation(info="<html>
<p>
 &nbsp;Equation number 8 from:
</p>
 <div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
</html>"));
            end v3a_ps;

            function v3b_ps "Region 3 b: inverse function v(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              output SI.SpecificVolume v "Specific volume";
          protected
              constant Real[:] n={0.591599780322238e-4,-0.185465997137856e-2,
                  0.104190510480013e-1,0.598647302038590e-2,-0.771391189901699,
                  0.172549765557036e1,-0.467076079846526e-3,0.134533823384439e-1,-0.808094336805495e-1,
                  0.508139374365767,0.128584643361683e-2,-0.163899353915435e1,
                  0.586938199318063e1,-0.292466667918613e1,-0.614076301499537e-2,
                  0.576199014049172e1,-0.121613320606788e2,0.167637540957944e1,-0.744135838773463e1,
                  0.378168091437659e-1,0.401432203027688e1,0.160279837479185e2,
                  0.317848779347728e1,-0.358362310304853e1,-0.115995260446827e7,
                  0.199256573577909,-0.122270624794624,-0.191449143716586e2,-0.150448002905284e-1,
                  0.146407900162154e2,-0.327477787188230e1};
              constant Real[:] I={-12,-12,-12,-12,-12,-12,-10,-10,-10,-10,-8,-5,-5,-5,
                  -4,-4,-4,-4,-3,-2,-2,-2,-2,-2,-2,0,0,0,1,1,2};
              constant Real[:] J={0,1,2,3,5,6,0,1,2,4,0,1,2,3,0,1,2,3,1,0,1,2,3,4,12,
                  0,1,2,0,2,2};
              constant SI.Volume vstar=0.0088 "Normalization temperature";
              constant SI.Pressure pstar=100e6 "Normalization pressure";
              constant SI.SpecificEntropy sstar=5.3e3 "Normalization entropy";
              Real pi=p/pstar "Normalized specific pressure";
              Real sigma=s/sstar "Normalized specific entropy";
            algorithm
              v := sum(n[i]*(pi + 0.298)^I[i]*(sigma - 0.816)^J[i] for i in 1:31)*
                vstar;
              annotation (Documentation(info="<html>
<p>
 &nbsp;Equation number 9 from:
</p>
<div style=\"text-align: center;\">&nbsp;[1] The international Association
 for the Properties of Water and Steam<br>
 &nbsp;Vejle, Denmark<br>
 &nbsp;August 2003<br>
 &nbsp;Supplementary Release on Backward Equations for the Functions
 T(p,h), v(p,h) and T(p,s), <br>
 &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
 the Thermodynamic Properties of<br>
 &nbsp;Water and Steam</div>
</html>"));
            end v3b_ps;
            annotation (Documentation(info="<HTML><h4>Package description</h4>
          <p>Package BaseIF97/Basic computes the the fundamental functions for the 5 regions of the steam tables
          as described in the standards document <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a>. The code of these
          functions has been generated using <b><i>Mathematica</i></b> and the add-on packages \"Format\" and \"Optimize\"
          to generate highly efficient, expression-optimized C-code from a symbolic representation of the thermodynamic
          functions. The C-code has than been transformed into Modelica code. An important feature of this optimization was to
          simultaneously optimize the functions and the directional derivatives because they share many common subexpressions.</p>
          <h4>Package contents</h4>
          <ul>
          <li>Function <b>g1</b> computes the dimensionless Gibbs function for region 1 and all derivatives up
          to order 2 w.r.t. pi and tau. Inputs: p and T.</li>
          <li>Function <b>g2</b> computes the dimensionless Gibbs function  for region 2 and all derivatives up
          to order 2 w.r.t. pi and tau. Inputs: p and T.</li>
          <li>Function <b>g2metastable</b> computes the dimensionless Gibbs function for metastable vapour
          (adjacent to region 2 but 2-phase at equilibrium) and all derivatives up
          to order 2 w.r.t. pi and tau. Inputs: p and T.</li>
          <li>Function <b>f3</b> computes the dimensionless Helmholtz function  for region 3 and all derivatives up
          to order 2 w.r.t. delta and tau. Inputs: d and T.</li>
          <li>Function <b>g5</b>computes the dimensionless Gibbs function for region 5 and all derivatives up
          to order 2 w.r.t. pi and tau. Inputs: p and T.</li>
          <li>Function <b>tph1</b> computes the inverse function T(p,h) in region 1.</li>
          <li>Function <b>tph2</b> computes the inverse function T(p,h) in region 2.</li>
          <li>Function <b>tps2a</b> computes the inverse function T(p,s) in region 2a.</li>
          <li>Function <b>tps2b</b> computes the inverse function T(p,s) in region 2b.</li>
          <li>Function <b>tps2c</b> computes the inverse function T(p,s) in region 2c.</li>
          <li>Function <b>tps2</b> computes the inverse function T(p,s) in region 2.</li>
          <li>Function <b>tsat</b> computes the saturation temperature as a function of pressure.</li>
          <li>Function <b>dtsatofp</b> computes the derivative of the saturation temperature w.r.t. pressure as
          a function of pressure.</li>
          <li>Function <b>tsat_der</b> computes the Modelica derivative function of tsat.</li>
          <li>Function <b>psat</b> computes the saturation pressure as a function of temperature.</li>
          <li>Function <b>dptofT</b>  computes the derivative of the saturation pressure w.r.t. temperature as
          a function of temperature.</li>
          <li>Function <b>psat_der</b> computes the Modelica derivative function of psat.</li>
          </ul>
          <h4>Version Info and Revision history
          </h4>
          <ul>
          <li>First implemented: <i>July, 2000</i>
          by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>
          </li>
          </ul>
          <address>Author: Hubertus Tummescheit, <br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
          </address>
          <ul>
          <li>Initial version: July 2000</li>
          <li>Documentation added: December 2002</li>
          </ul>
       <p>
       Equation from:
       </p>
       <div style=\"text-align: center;\">&nbsp;[1] The international Association
       for the Properties of Water and Steam<br>
       &nbsp;Vejle, Denmark<br>
       &nbsp;August 2003<br>
       &nbsp;Supplementary Release on Backward Equations for the Functions
       T(p,h), v(p,h) and T(p,s), <br>
       &nbsp;v(p,s) for Region 3 of the IAPWS Industrial Formulation 1997 for
       the Thermodynamic Properties of<br>
       &nbsp;Water and Steam</div>
       </html>"));
          end Basic;

          package Transport
          "Transport properties for water according to IAPWS/IF97"
            extends Modelica.Icons.Package;

            function visc_dTp
            "Dynamic viscosity eta(d,T,p), industrial formulation"
              extends Modelica.Icons.Function;
              input SI.Density d "Density";
              input SI.Temperature T "Temperature (K)";
              input SI.Pressure p
              "Pressure (only needed for region of validity)";
              input Integer phase=0
              "2 for two-phase, 1 for one-phase, 0 if not known";
              output SI.DynamicViscosity eta "Dynamic viscosity";
          protected
              constant Real n0=1.0 "Viscosity coefficient";
              constant Real n1=0.978197 "Viscosity coefficient";
              constant Real n2=0.579829 "Viscosity coefficient";
              constant Real n3=-0.202354 "Viscosity coefficient";
              constant Real[42] nn=array(
                        0.5132047,
                        0.3205656,
                        0.0,
                        0.0,
                        -0.7782567,
                        0.1885447,
                        0.2151778,
                        0.7317883,
                        1.241044,
                        1.476783,
                        0.0,
                        0.0,
                        -0.2818107,
                        -1.070786,
                        -1.263184,
                        0.0,
                        0.0,
                        0.0,
                        0.1778064,
                        0.460504,
                        0.2340379,
                        -0.4924179,
                        0.0,
                        0.0,
                        -0.0417661,
                        0.0,
                        0.0,
                        0.1600435,
                        0.0,
                        0.0,
                        0.0,
                        -0.01578386,
                        0.0,
                        0.0,
                        0.0,
                        0.0,
                        0.0,
                        0.0,
                        0.0,
                        -0.003629481,
                        0.0,
                        0.0) "Viscosity coefficients";
              constant SI.Density rhostar=317.763 "Scaling density";
              constant SI.DynamicViscosity etastar=55.071e-6
              "Scaling viscosity";
              constant SI.Temperature tstar=647.226 "Scaling temperature";
              Integer i "Auxiliary variable";
              Integer j "Auxiliary variable";
              Real delta "Dimensionless density";
              Real deltam1 "Dimensionless density";
              Real tau "Dimensionless temperature";
              Real taum1 "Dimensionless temperature";
              Real Psi0 "Auxiliary variable";
              Real Psi1 "Auxiliary variable";
              Real tfun "Auxiliary variable";
              Real rhofun "Auxiliary variable";
              Real Tc=T - 273.15 "Celsius temperature for region check";
              //      Integer region "Region of IF97";
            algorithm
              //      if phase == 0 then
              //        region := BaseIF97.Regions.region_dT(d,T,0);
              //      end if;
              //      if phase == 2 then
              //        region := 4;
              //      end if;
              // assert(phase <> 2, "Viscosity can not be computed for two-phase states");
              delta := d/rhostar;
              assert(d > triple.dvtriple,
                "IF97 medium function visc_dTp for viscosity called with too low density\n"
                 + "d = " + String(d) + " <= " + String(triple.dvtriple) +
                " (triple point density)");
              assert((p <= 500e6 and (Tc >= 0.0 and Tc <= 150)) or (p <= 350e6 and (
                Tc > 150.0 and Tc <= 600)) or (p <= 300e6 and (Tc > 600.0 and Tc <=
                900)),
                "IF97 medium function visc_dTp: viscosity computed outside the range\n"
                 + "of validity of the IF97 formulation: p = " + String(p) +
                " Pa, Tc = " + String(Tc) + " K");
              deltam1 := delta - 1.0;
              tau := tstar/T;
              taum1 := tau - 1.0;
              Psi0 := 1/(n0 + (n1 + (n2 + n3*tau)*tau)*tau)/(tau^0.5);
              Psi1 := 0.0;
              tfun := 1.0;
              for i in 1:6 loop
                if (i <> 1) then
                  tfun := tfun*taum1;
                end if;
                rhofun := 1.;
                for j in 0:6 loop
                  if (j <> 0) then
                    rhofun := rhofun*deltam1;
                  end if;
                  Psi1 := Psi1 + nn[i + j*6]*tfun*rhofun;
                end for;
              end for;
              eta := etastar*Psi0*Modelica.Math.exp(delta*Psi1);
            end visc_dTp;

            function cond_dTp
            "Thermal conductivity lam(d,T,p) (industrial use version) only in one-phase region"
              extends Modelica.Icons.Function;
              input SI.Density d "Density";
              input SI.Temperature T "Temperature (K)";
              input SI.Pressure p "Pressure";
              input Integer phase=0
              "2 for two-phase, 1 for one-phase, 0 if not known";
              input Boolean industrialMethod=true
              "If true, the industrial method is used, otherwise the scientific one";
              output SI.ThermalConductivity lambda "Thermal conductivity";
          protected
              Integer region(min=1, max=5)
              "IF97 region, valid values:1,2,3, and 5";
              constant Real n0=1.0 "Conductivity coefficient";
              constant Real n1=6.978267 "Conductivity coefficient";
              constant Real n2=2.599096 "Conductivity coefficient";
              constant Real n3=-0.998254 "Conductivity coefficient";
              constant Real[30] nn=array(
                        1.3293046,
                        1.7018363,
                        5.2246158,
                        8.7127675,
                        -1.8525999,
                        -0.40452437,
                        -2.2156845,
                        -10.124111,
                        -9.5000611,
                        0.9340469,
                        0.2440949,
                        1.6511057,
                        4.9874687,
                        4.3786606,
                        0.0,
                        0.018660751,
                        -0.76736002,
                        -0.27297694,
                        -0.91783782,
                        0.0,
                        -0.12961068,
                        0.37283344,
                        -0.43083393,
                        0.0,
                        0.0,
                        0.044809953,
                        -0.1120316,
                        0.13333849,
                        0.0,
                        0.0) "Conductivity coefficient";
              constant SI.ThermalConductivity lamstar=0.4945
              "Scaling conductivity";
              constant SI.Density rhostar=317.763 "Scaling density";
              constant SI.Temperature tstar=647.226 "Scaling temperature";
              constant SI.Pressure pstar=22.115e6 "Scaling pressure";
              constant SI.DynamicViscosity etastar=55.071e-6
              "Scaling viscosity";
              Integer i "Auxiliary variable";
              Integer j "Auxiliary variable";
              Real delta "Dimensionless density";
              Real tau "Dimensionless temperature";
              Real deltam1 "Dimensionless density";
              Real taum1 "Dimensionless temperature";
              Real Lam0 "Part of thermal conductivity";
              Real Lam1 "Part of thermal conductivity";
              Real Lam2 "Part of thermal conductivity";
              Real tfun "Auxiliary variable";
              Real rhofun "Auxiliary variable";
              Real dpitau "Auxiliary variable";
              Real ddelpi "Auxiliary variable";
              Real d2 "Auxiliary variable";
              Modelica.Media.Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
              Modelica.Media.Common.HelmholtzDerivs f
              "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
              Real Tc=T - 273.15 "Celsius temperature for region check";
              Real Chi "Symmetrized compressibility";
              // slightly different variables for industrial use
              constant SI.Density rhostar2=317.7 "Reference density";
              constant SI.Temperature Tstar2=647.25 "Reference temperature";
              constant SI.ThermalConductivity lambdastar=1
              "Reference thermal conductivity";
              parameter Real TREL=T/Tstar2 "Relative temperature";
              parameter Real rhoREL=d/rhostar2 "Relative density";
              Real lambdaREL "Relative thermal conductivity";
              Real deltaTREL "Relative temperature increment";
              constant Real[:] C={0.642857,-4.11717,-6.17937,0.00308976,0.0822994,
                  10.0932};
              constant Real[:] dpar={0.0701309,0.0118520,0.00169937,-1.0200};
              constant Real[:] b={-0.397070,0.400302,1.060000};
              constant Real[:] B={-0.171587,2.392190};
              constant Real[:] a={0.0102811,0.0299621,0.0156146,-0.00422464};
              Real Q;
              Real S;
              Real lambdaREL2
              "Function, part of the interpolating equation of the thermal conductivity";
              Real lambdaREL1
              "Function, part of the interpolating equation of the thermal conductivity";
              Real lambdaREL0
              "Function, part of the interpolating equation of the thermal conductivity";
            algorithm
              // region := BaseIF97.Regions.region_dT(d,T,phase);
              // simplified region check, assuming that calling arguments are legal
              //  assert(phase <> 2,
              //   "ThermalConductivity can not be called with 2-phase inputs!");
              assert(d > triple.dvtriple,
                "IF97 medium function cond_dTp called with too low density\n" +
                "d = " + String(d) + " <= " + String(triple.dvtriple) +
                " (triple point density)");
              assert((p <= 100e6 and (Tc >= 0.0 and Tc <= 500)) or (p <= 70e6 and (Tc
                 > 500.0 and Tc <= 650)) or (p <= 40e6 and (Tc > 650.0 and Tc <= 800)),
                "IF97 medium function cond_dTp: thermal conductivity computed outside the range\n"
                 + "of validity of the IF97 formulation: p = " + String(p) +
                " Pa, Tc = " + String(Tc) + " K");
              if industrialMethod == true then
                deltaTREL := abs(TREL - 1) + C[4];
                Q := 2 + C[5]/deltaTREL^(3/5);
                if TREL >= 1 then
                  S := 1/deltaTREL;
                else
                  S := C[6]/deltaTREL^(3/5);
                end if;
                lambdaREL2 := (dpar[1]/TREL^10 + dpar[2])*rhoREL^(9/5)*
                  Modelica.Math.exp(C[1]*(1 - rhoREL^(14/5))) + dpar[3]*S*rhoREL^Q*
                  Modelica.Math.exp((Q/(1 + Q))*(1 - rhoREL^(1 + Q))) + dpar[4]*
                  Modelica.Math.exp(C[2]*TREL^(3/2) + C[3]/rhoREL^5);
                lambdaREL1 := b[1] + b[2]*rhoREL + b[3]*Modelica.Math.exp(B[1]*(
                  rhoREL + B[2])^2);
                lambdaREL0 := TREL^(1/2)*sum(a[i]*TREL^(i - 1) for i in 1:4);
                lambdaREL := lambdaREL0 + lambdaREL1 + lambdaREL2;
                lambda := lambdaREL*lambdastar;
              else
                if p < data.PLIMIT4A then
                  //regions are 1 or 2,
                  if d > data.DCRIT then
                    region := 1;
                  else
                    region := 2;
                  end if;
                else
                  //region is 3, or illegal
                  assert(false,
                    "The scientific method works only for temperature up to 623.15 K");
                end if;
                tau := tstar/T;
                delta := d/rhostar;
                deltam1 := delta - 1.0;
                taum1 := tau - 1.0;
                Lam0 := 1/(n0 + (n1 + (n2 + n3*tau)*tau)*tau)/(tau^0.5);
                Lam1 := 0.0;
                tfun := 1.0;
                for i in 1:5 loop
                  if (i <> 1) then
                    tfun := tfun*taum1;
                  end if;
                  rhofun := 1.0;
                  for j in 0:5 loop
                    if (j <> 0) then
                      rhofun := rhofun*deltam1;
                    end if;
                    Lam1 := Lam1 + nn[i + j*5]*tfun*rhofun;
                  end for;
                end for;
                if (region == 1) then
                  g := Basic.g1(p, T);
                  // dp/dT @ cont d = -g.p/g.T*(g.gpi - g.tau*g.gtaupi)/(g.gpipi*g.pi);
                  dpitau := -tstar/pstar*(data.PSTAR1*(g.gpi - data.TSTAR1/T*g.gtaupi)
                    /g.gpipi/T);
                  ddelpi := -pstar/rhostar*data.RH2O/data.PSTAR1/data.PSTAR1*T*d*d*g.gpipi;
                  Chi := delta*ddelpi;
                elseif (region == 2) then
                  g := Basic.g2(p, T);
                  dpitau := -tstar/pstar*(data.PSTAR2*(g.gpi - data.TSTAR2/T*g.gtaupi)
                    /g.gpipi/T);
                  ddelpi := -pstar/rhostar*data.RH2O/data.PSTAR2/data.PSTAR2*T*d*d*g.gpipi;
                  Chi := delta*ddelpi;
                  //         elseif (region == 3) then
                  //           f := Basic.f3(T, d);
                  //            dpitau := tstar/pstar*(f.R*f.d*f.delta*(f.fdelta - f.tau*f.fdeltatau));
                  //           ddelpi := pstar*d*d/(rhostar*p*p)/(f.R*f.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta));
                  //    Chi := delta*ddelpi;
                else
                  assert(false,
                    "Thermal conductivity can only be called in the one-phase regions below 623.15 K\n"
                     + "(p = " + String(p) + " Pa, T = " + String(T) +
                    " K, region = " + String(region) + ")");
                end if;
                taum1 := 1/tau - 1;
                d2 := deltam1*deltam1;
                Lam2 := 0.0013848*etastar/visc_dTp(
                        d,
                        T,
                        p)/(tau*tau*delta*delta)*dpitau*dpitau*max(Chi, Modelica.Constants.small)
                  ^0.4678*(delta)^0.5*Modelica.Math.exp(-18.66*taum1*taum1 - d2*d2);
                lambda := lamstar*(Lam0*Modelica.Math.exp(delta*Lam1) + Lam2);
              end if;
            end cond_dTp;

            function surfaceTension
            "Surface tension in region 4 between steam and water"
              extends Modelica.Icons.Function;
              input SI.Temperature T "Temperature (K)";
              output SI.SurfaceTension sigma "Surface tension in SI units";
          protected
              Real Theta "Dimensionless temperature";
            algorithm
              Theta := min(1.0, T/data.TCRIT);
              sigma := 235.8e-3*(1 - Theta)^1.256*(1 - 0.625*(1 - Theta));
            end surfaceTension;
            annotation (Documentation(info="<HTML><h4>Package description</h4>
          <h4>Package contents</h4>
          <ul>
          <li>Function <b>visc_dTp</b> implements a function to compute the industrial formulation of the
          dynamic viscosity of water as a function of density and temperature.
          The details are described in the document <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/visc.pdf\">visc.pdf</a>.</li>
          <li>Function <b>cond_dTp</b> implements a function to compute  the industrial formulation of the thermal conductivity of water as
          a function of density, temperature and pressure. <b>Important note</b>: Obviously only two of the three
          inputs are really needed, but using three inputs speeds up the computation and the three variables are known in most models anyways.
          The inputs d,T and p have to be consistent.
          The details are described in the document <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/surf.pdf\">surf.pdf</a>.</li>
          <li>Function <b>surfaceTension</b> implements a function to compute the surface tension between vapour
          and liquid water as a function of temperature.
          The details are described in the document <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/thcond.pdf\">thcond.pdf</a>.</li>
          </ul>
          <h4>Version Info and Revision history
          </h4>
          <ul>
          <li>First implemented: <i>October, 2002</i>
          by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>
          </li>
          </ul>
          <address>Authors: Hubertus Tummescheit and Jonas Eborn<br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
          </address>
          <ul>
          <li>Initial version: October 2002</li>
          </ul>
          </html>"));
          end Transport;

          package Isentropic
          "Functions for calculating the isentropic enthalpy from pressure p and specific entropy s"
            extends Modelica.Icons.Package;

            function hofpT1
            "Intermediate function for isentropic specific enthalpy in region 1"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real[13] o "Vector of auxiliary variables";
              Real pi1 "Dimensionless pressure";
              Real tau "Dimensionless temperature";
              Real tau1 "Dimensionless temperature";
            algorithm
              tau := data.TSTAR1/T;
              pi1 := 7.1 - p/data.PSTAR1;
              assert(p > triple.ptriple,
                "IF97 medium function hofpT1 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              tau1 := -1.222 + tau;
              o[1] := tau1*tau1;
              o[2] := o[1]*tau1;
              o[3] := o[1]*o[1];
              o[4] := o[3]*o[3];
              o[5] := o[1]*o[4];
              o[6] := o[1]*o[3];
              o[7] := o[3]*tau1;
              o[8] := o[3]*o[4];
              o[9] := pi1*pi1;
              o[10] := o[9]*o[9];
              o[11] := o[10]*o[10];
              o[12] := o[4]*o[4];
              o[13] := o[12]*o[12];

              h := data.RH2O*T*tau*(pi1*((-0.00254871721114236 + o[1]*(
                0.00424944110961118 + (0.018990068218419 + (-0.021841717175414 -
                0.00015851507390979*o[1])*o[1])*o[6]))/o[5] + pi1*((
                0.00141552963219801 + o[3]*(0.000047661393906987 + o[1]*(-0.0000132425535992538
                 - 1.2358149370591e-14*o[1]*o[3]*o[4])))/o[3] + pi1*((
                0.000126718579380216 - 5.11230768720618e-9*o[5])/o[7] + pi1*((
                0.000011212640954 + o[2]*(1.30342445791202e-6 - 1.4341729937924e-12*o[
                8]))/o[6] + pi1*(o[9]*pi1*((1.40077319158051e-8 + 1.04549227383804e-9
                *o[7])/o[8] + o[10]*o[11]*pi1*(1.9941018075704e-17/(o[1]*o[12]*o[3]*o[
                4]) + o[9]*(-4.48827542684151e-19/o[13] + o[10]*o[9]*(pi1*(
                4.65957282962769e-22/(o[13]*o[4]) + pi1*((3.83502057899078e-24*pi1)/(
                o[1]*o[13]*o[4]) - 7.2912378325616e-23/(o[13]*o[4]*tau1))) -
                1.00075970318621e-21/(o[1]*o[13]*o[3]*tau1))))) + 3.24135974880936e-6
                /(o[4]*tau1)))))) + (-0.29265942426334 + tau1*(0.84548187169114 + o[1]
                *(3.3855169168385 + tau1*(-1.91583926775744 + tau1*(0.47316115539684
                 + (-0.066465668798004 + 0.0040607314991784*tau1)*tau1)))))/o[2]);
            end hofpT1;

            function handsofpT1
            "Special function for specific enthalpy and specific entropy in region 1"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real[28] o "Vector of auxiliary variables";
              Real pi1 "Dimensionless pressure";
              Real tau "Dimensionless temperature";
              Real tau1 "Dimensionless temperature";
              Real g "Dimensionless Gibbs energy";
              Real gtau "Derivative of dimensionless Gibbs energy w.r.t. tau";
            algorithm
              assert(p > triple.ptriple,
                "IF97 medium function handsofpT1 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              tau := data.TSTAR1/T;
              pi1 := 7.1 - p/data.PSTAR1;
              tau1 := -1.222 + tau;
              o[1] := tau1*tau1;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              o[4] := o[3]*tau1;
              o[5] := 1/o[4];
              o[6] := o[1]*o[2];
              o[7] := o[1]*tau1;
              o[8] := 1/o[7];
              o[9] := o[1]*o[2]*o[3];
              o[10] := 1/o[2];
              o[11] := o[2]*tau1;
              o[12] := 1/o[11];
              o[13] := o[2]*o[3];
              o[14] := pi1*pi1;
              o[15] := o[14]*pi1;
              o[16] := o[14]*o[14];
              o[17] := o[16]*o[16];
              o[18] := o[16]*o[17]*pi1;
              o[19] := o[14]*o[16];
              o[20] := o[3]*o[3];
              o[21] := o[20]*o[20];
              o[22] := o[21]*o[3]*tau1;
              o[23] := 1/o[22];
              o[24] := o[21]*o[3];
              o[25] := 1/o[24];
              o[26] := o[1]*o[2]*o[21]*tau1;
              o[27] := 1/o[26];
              o[28] := o[1]*o[3];

              g := pi1*(pi1*(pi1*(o[10]*(-0.000031679644845054 + o[2]*(-2.8270797985312e-6
                 - 8.5205128120103e-10*o[6])) + pi1*(o[12]*(-2.2425281908e-6 + (-6.5171222895601e-7
                 - 1.4341729937924e-13*o[13])*o[7]) + pi1*(-4.0516996860117e-7/o[3]
                 + o[15]*(o[18]*(o[14]*(o[19]*(2.6335781662795e-23/(o[1]*o[2]*o[21])
                 + pi1*(-1.1947622640071e-23*o[27] + pi1*(1.8228094581404e-24*o[25]
                 - 9.3537087292458e-26*o[23]*pi1))) + 1.4478307828521e-20/(o[1]*o[2]*
                o[20]*o[3]*tau1)) - 6.8762131295531e-19/(o[2]*o[20]*o[3]*tau1)) + (-1.2734301741641e-9
                 - 1.7424871230634e-10*o[11])/(o[1]*o[3]*tau1))))) + o[8]*(-0.00047184321073267
                 + o[7]*(-0.00030001780793026 + (0.000047661393906987 + o[1]*(-4.4141845330846e-6
                 - 7.2694996297594e-16*o[9]))*tau1))) + o[5]*(0.00028319080123804 + o[
                1]*(-0.00060706301565874 + o[6]*(-0.018990068218419 + tau1*(-0.032529748770505
                 + (-0.021841717175414 - 0.00005283835796993*o[1])*tau1))))) + (
                0.14632971213167 + tau1*(-0.84548187169114 + tau1*(-3.756360367204 +
                tau1*(3.3855169168385 + tau1*(-0.95791963387872 + tau1*(
                0.15772038513228 + (-0.016616417199501 + 0.00081214629983568*tau1)*
                tau1))))))/o[1];

              gtau := pi1*((-0.00254871721114236 + o[1]*(0.00424944110961118 + (
                0.018990068218419 + (-0.021841717175414 - 0.00015851507390979*o[1])*o[
                1])*o[6]))/o[28] + pi1*(o[10]*(0.00141552963219801 + o[2]*(
                0.000047661393906987 + o[1]*(-0.0000132425535992538 -
                1.2358149370591e-14*o[9]))) + pi1*(o[12]*(0.000126718579380216 -
                5.11230768720618e-9*o[28]) + pi1*((0.000011212640954 + (
                1.30342445791202e-6 - 1.4341729937924e-12*o[13])*o[7])/o[6] + pi1*(
                3.24135974880936e-6*o[5] + o[15]*((1.40077319158051e-8 +
                1.04549227383804e-9*o[11])/o[13] + o[18]*(1.9941018075704e-17/(o[1]*o[
                2]*o[20]*o[3]) + o[14]*(-4.48827542684151e-19/o[21] + o[19]*(-1.00075970318621e-21
                *o[27] + pi1*(4.65957282962769e-22*o[25] + pi1*(-7.2912378325616e-23*
                o[23] + (3.83502057899078e-24*pi1)/(o[1]*o[21]*o[3])))))))))))) + o[8]
                *(-0.29265942426334 + tau1*(0.84548187169114 + o[1]*(3.3855169168385
                 + tau1*(-1.91583926775744 + tau1*(0.47316115539684 + (-0.066465668798004
                 + 0.0040607314991784*tau1)*tau1)))));

              h := data.RH2O*T*tau*gtau;
              s := data.RH2O*(tau*gtau - g);
            end handsofpT1;

            function hofpT2
            "Intermediate function for isentropic specific enthalpy in region 2"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output SI.SpecificEnthalpy h "Specific enthalpy";
          protected
              Real[16] o "Vector of auxiliary variables";
              Real pi "Dimensionless pressure";
              Real tau "Dimensionless temperature";
              Real tau2 "Dimensionless temperature";
            algorithm
              assert(p > triple.ptriple,
                "IF97 medium function hofpT2 called with too low pressure\n" + "p = "
                 + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              pi := p/data.PSTAR2;
              tau := data.TSTAR2/T;
              tau2 := -0.5 + tau;
              o[1] := tau*tau;
              o[2] := o[1]*o[1];
              o[3] := tau2*tau2;
              o[4] := o[3]*tau2;
              o[5] := o[3]*o[3];
              o[6] := o[5]*o[5];
              o[7] := o[6]*o[6];
              o[8] := o[5]*o[6]*o[7]*tau2;
              o[9] := o[3]*o[5];
              o[10] := o[5]*o[6]*tau2;
              o[11] := o[3]*o[7]*tau2;
              o[12] := o[3]*o[5]*o[6];
              o[13] := o[5]*o[6]*o[7];
              o[14] := pi*pi;
              o[15] := o[14]*o[14];
              o[16] := o[7]*o[7];

              h := data.RH2O*T*tau*((0.0280439559151 + tau*(-0.2858109552582 + tau*(
                1.2213149471784 + tau*(-2.848163942888 + tau*(4.38395111945 + o[1]*(
                10.08665568018 + (-0.5681726521544 + 0.06380539059921*tau)*tau))))))/
                (o[1]*o[2]) + pi*(-0.017834862292358 + tau2*(-0.09199202739273 + (-0.172743777250296
                 - 0.30195167236758*o[4])*tau2) + pi*(-0.000033032641670203 + (-0.0003789797503263
                 + o[3]*(-0.015757110897342 + o[4]*(-0.306581069554011 -
                0.000960283724907132*o[8])))*tau2 + pi*(4.3870667284435e-7 + o[3]*(-0.00009683303171571
                 + o[4]*(-0.0090203547252888 - 1.42338887469272*o[8])) + pi*(-7.8847309559367e-10
                 + (2.558143570457e-8 + 1.44676118155521e-6*tau2)*tau2 + pi*(
                0.0000160454534363627*o[9] + pi*((-5.0144299353183e-11 + o[10]*(-0.033874355714168
                 - 836.35096769364*o[11]))*o[3] + pi*((-0.0000138839897890111 -
                0.973671060893475*o[12])*o[3]*o[6] + pi*((9.0049690883672e-11 -
                296.320827232793*o[13])*o[3]*o[5]*tau2 + pi*(2.57526266427144e-7*o[5]
                *o[6] + pi*(o[4]*(4.1627860840696e-19 + (-1.0234747095929e-12 -
                1.40254511313154e-8*o[5])*o[9]) + o[14]*o[15]*(o[13]*(-2.34560435076256e-9
                 + 5.3465159397045*o[5]*o[7]*tau2) + o[14]*(-19.1874828272775*o[16]*o[
                6]*o[7] + o[14]*(o[11]*(1.78371690710842e-23 + (1.07202609066812e-11
                 - 0.000201611844951398*o[10])*o[3]*o[5]*o[6]*tau2) + pi*(-1.24017662339842e-24
                *o[5]*o[7] + pi*(0.000200482822351322*o[16]*o[5]*o[7] + pi*(-4.97975748452559e-14
                *o[16]*o[3]*o[5] + o[6]*o[7]*(1.90027787547159e-27 + o[12]*(
                2.21658861403112e-15 - 0.0000547344301999018*o[3]*o[7]))*pi*tau2)))))))))))))))));
            end hofpT2;

            function handsofpT2
            "Function for isentropic specific enthalpy and specific entropy in region 2"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              output SI.SpecificEnthalpy h "Specific enthalpy";
              output SI.SpecificEntropy s "Specific entropy";
          protected
              Real[22] o "Vector of auxiliary variables";
              Real pi "Dimensionless pressure";
              Real tau "Dimensionless temperature";
              Real tau2 "Dimensionless temperature";
              Real g "Dimensionless Gibbs energy";
              Real gtau "Derivative of dimensionless Gibbs energy w.r.t. tau";
            algorithm
              assert(p > triple.ptriple,
                "IF97 medium function handsofpT2 called with too low pressure\n" +
                "p = " + String(p) + " Pa <= " + String(triple.ptriple) +
                " Pa (triple point pressure)");
              tau := data.TSTAR2/T;
              pi := p/data.PSTAR2;
              tau2 := tau - 0.5;
              o[1] := tau2*tau2;
              o[2] := o[1]*tau2;
              o[3] := o[1]*o[1];
              o[4] := o[3]*o[3];
              o[5] := o[4]*o[4];
              o[6] := o[3]*o[4]*o[5]*tau2;
              o[7] := o[1]*o[3]*tau2;
              o[8] := o[3]*o[4]*tau2;
              o[9] := o[1]*o[5]*tau2;
              o[10] := o[1]*o[3]*o[4];
              o[11] := o[3]*o[4]*o[5];
              o[12] := o[1]*o[3];
              o[13] := pi*pi;
              o[14] := o[13]*o[13];
              o[15] := o[13]*o[14];
              o[16] := o[3]*o[5]*tau2;
              o[17] := o[5]*o[5];
              o[18] := o[3]*o[5];
              o[19] := o[1]*o[3]*o[4]*tau2;
              o[20] := o[1]*o[5];
              o[21] := tau*tau;
              o[22] := o[21]*o[21];

              g := pi*(-0.0017731742473213 + tau2*(-0.017834862292358 + tau2*(-0.045996013696365
                 + (-0.057581259083432 - 0.05032527872793*o[2])*tau2)) + pi*(tau2*(-0.000033032641670203
                 + (-0.00018948987516315 + o[1]*(-0.0039392777243355 + o[2]*(-0.043797295650573
                 - 0.000026674547914087*o[6])))*tau2) + pi*(2.0481737692309e-8 + (
                4.3870667284435e-7 + o[1]*(-0.00003227767723857 + o[2]*(-0.0015033924542148
                 - 0.040668253562649*o[6])))*tau2 + pi*(tau2*(-7.8847309559367e-10 +
                (1.2790717852285e-8 + 4.8225372718507e-7*tau2)*tau2) + pi*(
                2.2922076337661e-6*o[7] + pi*(o[2]*(-1.6714766451061e-11 + o[8]*(-0.0021171472321355
                 - 23.895741934104*o[9])) + pi*(-5.905956432427e-18 + o[1]*(-1.2621808899101e-6
                 - 0.038946842435739*o[10])*o[4]*tau2 + pi*((1.1256211360459e-11 -
                8.2311340897998*o[11])*o[4] + pi*(1.9809712802088e-8*o[8] + pi*((
                1.0406965210174e-19 + o[12]*(-1.0234747095929e-13 -
                1.0018179379511e-9*o[3]))*o[3] + o[15]*((-8.0882908646985e-11 +
                0.10693031879409*o[16])*o[6] + o[13]*(-0.33662250574171*o[17]*o[4]*o[
                5]*tau2 + o[13]*(o[18]*(8.9185845355421e-25 + o[19]*(
                3.0629316876232e-13 - 4.2002467698208e-6*o[8])) + pi*(-5.9056029685639e-26
                *o[16] + pi*(3.7826947613457e-6*o[17]*o[3]*o[5]*tau2 + pi*(o[1]*(
                7.3087610595061e-29 + o[10]*(5.5414715350778e-17 - 9.436970724121e-7*
                o[20]))*o[4]*o[5]*pi - 1.2768608934681e-15*o[1]*o[17]*o[3]*tau2))))))))))))))))
                 + (-0.00560879118302 + tau*(0.07145273881455 + tau*(-0.4071049823928
                 + tau*(1.424081971444 + tau*(-4.38395111945 + tau*(-9.692768600217
                 + tau*(10.08665568018 + (-0.2840863260772 + 0.02126846353307*tau)*
                tau) + Modelica.Math.log(pi)))))))/(o[22]*tau);

              gtau := (0.0280439559151 + tau*(-0.2858109552582 + tau*(1.2213149471784
                 + tau*(-2.848163942888 + tau*(4.38395111945 + o[21]*(10.08665568018
                 + (-0.5681726521544 + 0.06380539059921*tau)*tau))))))/(o[21]*o[22])
                 + pi*(-0.017834862292358 + tau2*(-0.09199202739273 + (-0.172743777250296
                 - 0.30195167236758*o[2])*tau2) + pi*(-0.000033032641670203 + (-0.0003789797503263
                 + o[1]*(-0.015757110897342 + o[2]*(-0.306581069554011 -
                0.000960283724907132*o[6])))*tau2 + pi*(4.3870667284435e-7 + o[1]*(-0.00009683303171571
                 + o[2]*(-0.0090203547252888 - 1.42338887469272*o[6])) + pi*(-7.8847309559367e-10
                 + (2.558143570457e-8 + 1.44676118155521e-6*tau2)*tau2 + pi*(
                0.0000160454534363627*o[12] + pi*(o[1]*(-5.0144299353183e-11 + o[8]*(
                -0.033874355714168 - 836.35096769364*o[9])) + pi*(o[1]*(-0.0000138839897890111
                 - 0.973671060893475*o[10])*o[4] + pi*((9.0049690883672e-11 -
                296.320827232793*o[11])*o[7] + pi*(2.57526266427144e-7*o[3]*o[4] + pi
                *(o[2]*(4.1627860840696e-19 + o[12]*(-1.0234747095929e-12 -
                1.40254511313154e-8*o[3])) + o[15]*(o[11]*(-2.34560435076256e-9 +
                5.3465159397045*o[16]) + o[13]*(-19.1874828272775*o[17]*o[4]*o[5] + o[
                13]*((1.78371690710842e-23 + o[19]*(1.07202609066812e-11 -
                0.000201611844951398*o[8]))*o[9] + pi*(-1.24017662339842e-24*o[18] +
                pi*(0.000200482822351322*o[17]*o[3]*o[5] + pi*(-4.97975748452559e-14*
                o[1]*o[17]*o[3] + (1.90027787547159e-27 + o[10]*(2.21658861403112e-15
                 - 0.0000547344301999018*o[20]))*o[4]*o[5]*pi*tau2))))))))))))))));

              h := data.RH2O*T*tau*gtau;
              s := data.RH2O*(tau*gtau - g);
            end handsofpT2;
            annotation (Documentation(info="<HTML><h4>Package description</h4>
          <h4>Package contents</h4>
          <ul>
          <li>Function <b>hofpT1</b> computes h(p,T) in region 1.</li>
          <li>Function <b>handsofpT1</b> computes (s,h)=f(p,T) in region 1, needed for two-phase properties.</li>
          <li>Function <b>hofps1</b> computes h(p,s) in region 1.</li>
          <li>Function <b>hofpT2</b> computes h(p,T) in region 2.</li>
          <li>Function <b>handsofpT2</b> computes (s,h)=f(p,T) in region 2, needed for two-phase properties.</li>
          <li>Function <b>hofps2</b> computes h(p,s) in region 2.</li>
          <li>Function <b>hofdT3</b> computes h(d,T) in region 3.</li>
          <li>Function <b>hofpsdt3</b> computes h(p,s,dguess,Tguess) in region 3, where dguess and Tguess are initial guess
          values for the density and temperature consistent with p and s.</li>
          <li>Function <b>hofps4</b> computes h(p,s) in region 4.</li>
          <li>Function <b>hofpT5</b> computes h(p,T) in region 5.</li>
          <li>Function <b>water_hisentropic</b> computes h(p,s,phase) in all regions.
          The phase input is needed due to discontinuous derivatives at the phase boundary.</li>
          <li>Function <b>water_hisentropic_dyn</b> computes h(p,s,dguess,Tguess,phase) in all regions.
          The phase input is needed due to discontinuous derivatives at the phase boundary. Tguess and dguess are initial guess
          values for the density and temperature consistent with p and s. This function should be preferred in
          dynamic simulations where good guesses are often available.</li>
          </ul>
          <h4>Version Info and Revision history
          </h4>
          <ul>
          <li>First implemented: <i>July, 2000</i>
          by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>
          </li>
          </ul>
          <address>Author: Hubertus Tummescheit, <br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
          </address>
          <ul>
          <li>Initial version: July 2000</li>
          <li>Documentation added: December 2002</li>
          </ul>
          </html>"));
          end Isentropic;

          package Inverses "Efficient inverses for selected pairs of variables"
            extends Modelica.Icons.Package;

            function fixdT "Region limits for inverse iteration in region 3"

              extends Modelica.Icons.Function;
              input SI.Density din "Density";
              input SI.Temperature Tin "Temperature";
              output SI.Density dout "Density";
              output SI.Temperature Tout "Temperature";
          protected
              SI.Temperature Tmin "Approximation of minimum temperature";
              SI.Temperature Tmax "Approximation of maximum temperature";
            algorithm
              if (din > 765.0) then
                dout := 765.0;
              elseif (din < 110.0) then
                dout := 110.0;
              else
                dout := din;
              end if;
              if (dout < 390.0) then
                Tmax := 554.3557377 + dout*0.809344262;
              else
                Tmax := 1116.85 - dout*0.632948717;
              end if;
              if (dout < data.DCRIT) then
                Tmin := data.TCRIT*(1.0 - (dout - data.DCRIT)*(dout - data.DCRIT)/
                  1.0e6);
              else
                Tmin := data.TCRIT*(1.0 - (dout - data.DCRIT)*(dout - data.DCRIT)/
                  1.44e6);
              end if;
              if (Tin < Tmin) then
                Tout := Tmin;
              elseif (Tin > Tmax) then
                Tout := Tmax;
              else
                Tout := Tin;
              end if;
            end fixdT;

            function dofp13 "Density at the boundary between regions 1 and 3"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.Density d "Density";
          protected
              Real p2 "Auxiliary variable";
              Real[3] o "Vector of auxiliary variables";
            algorithm
              p2 := 7.1 - 6.04960677555959e-8*p;
              o[1] := p2*p2;
              o[2] := o[1]*o[1];
              o[3] := o[2]*o[2];
              d := 57.4756752485113/(0.0737412153522555 + p2*(0.00145092247736023 +
                p2*(0.000102697173772229 + p2*(0.0000114683182476084 + p2*(
                1.99080616601101e-6 + o[1]*p2*(1.13217858826367e-8 + o[2]*o[3]*p2*(
                1.35549330686006e-17 + o[1]*(-3.11228834832975e-19 + o[1]*o[2]*(-7.02987180039442e-22
                 + p2*(3.29199117056433e-22 + (-5.17859076694812e-23 +
                2.73712834080283e-24*p2)*p2))))))))));

            end dofp13;

            function dofp23 "Density at the boundary between regions 2 and 3"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              output SI.Density d "Density";
          protected
              SI.Temperature T;
              Real[13] o "Vector of auxiliary variables";
              Real taug "Auxiliary variable";
              Real pi "Dimensionless pressure";
              Real gpi23
              "Derivative of g w.r.t. pi on the boundary between regions 2 and 3";
            algorithm
              pi := p/data.PSTAR2;
              T := 572.54459862746 + 31.3220101646784*(-13.91883977887 + pi)^0.5;
              o[1] := (-13.91883977887 + pi)^0.5;
              taug := -0.5 + 540.0/(572.54459862746 + 31.3220101646784*o[1]);
              o[2] := taug*taug;
              o[3] := o[2]*taug;
              o[4] := o[2]*o[2];
              o[5] := o[4]*o[4];
              o[6] := o[5]*o[5];
              o[7] := o[4]*o[5]*o[6]*taug;
              o[8] := o[4]*o[5]*taug;
              o[9] := o[2]*o[4]*o[5];
              o[10] := pi*pi;
              o[11] := o[10]*o[10];
              o[12] := o[4]*o[6]*taug;
              o[13] := o[6]*o[6];

              gpi23 := (1.0 + pi*(-0.0017731742473213 + taug*(-0.017834862292358 +
                taug*(-0.045996013696365 + (-0.057581259083432 - 0.05032527872793*o[3])
                *taug)) + pi*(taug*(-0.000066065283340406 + (-0.0003789797503263 + o[
                2]*(-0.007878555448671 + o[3]*(-0.087594591301146 -
                0.000053349095828174*o[7])))*taug) + pi*(6.1445213076927e-8 + (
                1.31612001853305e-6 + o[2]*(-0.00009683303171571 + o[3]*(-0.0045101773626444
                 - 0.122004760687947*o[7])))*taug + pi*(taug*(-3.15389238237468e-9 +
                (5.116287140914e-8 + 1.92901490874028e-6*taug)*taug) + pi*(
                0.0000114610381688305*o[2]*o[4]*taug + pi*(o[3]*(-1.00288598706366e-10
                 + o[8]*(-0.012702883392813 - 143.374451604624*o[2]*o[6]*taug)) + pi*
                (-4.1341695026989e-17 + o[2]*o[5]*(-8.8352662293707e-6 -
                0.272627897050173*o[9])*taug + pi*(o[5]*(9.0049690883672e-11 -
                65.8490727183984*o[4]*o[5]*o[6]) + pi*(1.78287415218792e-7*o[8] + pi*
                (o[4]*(1.0406965210174e-18 + o[2]*(-1.0234747095929e-12 -
                1.0018179379511e-8*o[4])*o[4]) + o[10]*o[11]*((-1.29412653835176e-9
                 + 1.71088510070544*o[12])*o[7] + o[10]*(-6.05920510335078*o[13]*o[5]
                *o[6]*taug + o[10]*(o[4]*o[6]*(1.78371690710842e-23 + o[2]*o[4]*o[5]*
                (6.1258633752464e-12 - 0.000084004935396416*o[8])*taug) + pi*(-1.24017662339842e-24
                *o[12] + pi*(0.0000832192847496054*o[13]*o[4]*o[6]*taug + pi*(o[2]*o[
                5]*o[6]*(1.75410265428146e-27 + (1.32995316841867e-15 -
                0.0000226487297378904*o[2]*o[6])*o[9])*pi - 2.93678005497663e-14*o[13]
                *o[2]*o[4]*taug)))))))))))))))))/pi;
              d := p/(data.RH2O*T*pi*gpi23);
            end dofp23;

            function dofpt3 "Inverse iteration in region 3: (d) = f(p,T)"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.Temperature T "Temperature (K)";
              input SI.Pressure delp "Iteration converged if (p-pre(p) < delp)";
              output SI.Density d "Density";
              output Integer error=0
              "Error flag: iteration failed if different from 0";
          protected
              SI.Density dguess "Guess density";
              Integer i=0 "Loop counter";
              Real dp "Pressure difference";
              SI.Density deld "Density step";
              Modelica.Media.Common.HelmholtzDerivs f
              "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
              Modelica.Media.Common.NewtonDerivatives_pT nDerivs
              "Derivatives needed in Newton iteration";
              Boolean found=false "Flag for iteration success";
              Boolean supercritical "Flag, true for supercritical states";
              Boolean liquid "Flag, true for liquid states";
              SI.Density dmin "Lower density limit";
              SI.Density dmax "Upper density limit";
              SI.Temperature Tmax "Maximum temperature";
              Real damping "Damping factor";
            algorithm
              found := false;
              assert(p >= data.PLIMIT4A,
                "BaseIF97.dofpt3: function called outside of region 3! p too low\n"
                 + "p = " + String(p) + " Pa < " + String(data.PLIMIT4A) + " Pa");
              assert(T >= data.TLIMIT1,
                "BaseIF97.dofpt3: function called outside of region 3! T too low\n"
                 + "T = " + String(T) + " K < " + String(data.TLIMIT1) + " K");
              assert(p >= Regions.boundary23ofT(T),
                "BaseIF97.dofpt3: function called outside of region 3! T too high\n"
                 + "p = " + String(p) + " Pa, T = " + String(T) + " K");
              supercritical := p > data.PCRIT;
              damping := if supercritical then 1.0 else 1.0;
              Tmax := Regions.boundary23ofp(p);
              if supercritical then
                dmax := dofp13(p);
                dmin := dofp23(p);
                dguess := dmax - (T - data.TLIMIT1)/(data.TLIMIT1 - Tmax)*(dmax -
                  dmin);
                //this may need even further improvement!!
              else
                liquid := T < Basic.tsat(p);
                if liquid then
                  dmax := dofp13(p);
                  dmin := Regions.rhol_p_R4b(p);
                  dguess := 1.1*Regions.rhol_T(T)
                  "Guess: 10 percent more than on the phase boundary for same T";
                  //      dguess := 0.5*(dmax + dmin);
                else
                  dmax := Regions.rhov_p_R4b(p);
                  dmin := dofp23(p);
                  dguess := 0.9*Regions.rhov_T(T)
                  "Guess: 10% less than on the phase boundary for same T";
                  //      dguess := 0.5*(dmax + dmin);
                end if;
              end if;
              while ((i < IterationData.IMAX) and not found) loop
                d := dguess;
                f := Basic.f3(d, T);
                nDerivs := Modelica.Media.Common.Helmholtz_pT(f);
                dp := nDerivs.p - p;
                if (abs(dp/p) <= delp) then
                  found := true;
                end if;
                deld := dp/nDerivs.pd*damping;
                // Can be used for debugging in Dymola: not a standard function
                //          LogVariable(deld);
                d := d - deld;
                if d > dmin and d < dmax then
                  dguess := d;
                else
                  if d > dmax then
                    dguess := dmax - sqrt(Modelica.Constants.eps);
                    // put it on the correct spot just inside the boundary here instead
                  else
                    dguess := dmin + sqrt(Modelica.Constants.eps);
                  end if;
                end if;
                i := i + 1;
              end while;
              if not found then
                error := 1;
              end if;
              assert(error <> 1, "Error in inverse function dofpt3: iteration failed");
            end dofpt3;

            function dtofph3 "Inverse iteration in region 3: (d,T) = f(p,h)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              input SI.Pressure delp "Iteration accuracy";
              input SI.SpecificEnthalpy delh "Iteration accuracy";
              output SI.Density d "Density";
              output SI.Temperature T "Temperature (K)";
              output Integer error
              "Error flag: iteration failed if different from 0";
          protected
              SI.Temperature Tguess "Initial temperature";
              SI.Density dguess "Initial density";
              Integer i "Iteration counter";
              Real dh "Newton-error in h-direction";
              Real dp "Newton-error in p-direction";
              Real det "Determinant of directional derivatives";
              Real deld "Newton-step in d-direction";
              Real delt "Newton-step in T-direction";
              Modelica.Media.Common.HelmholtzDerivs f
              "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
              Modelica.Media.Common.NewtonDerivatives_ph nDerivs
              "Derivatives needed in Newton iteration";
              Boolean found=false "Flag for iteration success";
              Integer subregion "1 for subregion 3a, 2 for subregion 3b";
            algorithm
              if p < data.PCRIT then
                // allow a 10 J margin inside the (well approximated) phase boundary
                subregion := if h < (Regions.hl_p(p) + 10.0) then 1 else if h > (
                  Regions.hv_p(p) - 10.0) then 2 else 0;
                assert(subregion <> 0,
                  "Inverse iteration of dt from ph called in 2 phase region: this can not work");
              else
                //supercritical
                subregion := if h < Basic.h3ab_p(p) then 1 else 2;
              end if;
              T := if subregion == 1 then Basic.T3a_ph(p, h) else Basic.T3b_ph(p, h);
              d := if subregion == 1 then 1/Basic.v3a_ph(p, h) else 1/Basic.v3b_ph(p,
                h);
              i := 0;
              error := 0;
              while ((i < IterationData.IMAX) and not found) loop
                f := Basic.f3(d, T);
                nDerivs := Modelica.Media.Common.Helmholtz_ph(f);
                dh := nDerivs.h - h;
                dp := nDerivs.p - p;
                if ((abs(dh/h) <= delh) and (abs(dp/p) <= delp)) then
                  found := true;
                end if;
                det := nDerivs.ht*nDerivs.pd - nDerivs.pt*nDerivs.hd;
                delt := (nDerivs.pd*dh - nDerivs.hd*dp)/det;
                deld := (nDerivs.ht*dp - nDerivs.pt*dh)/det;
                T := T - delt;
                d := d - deld;
                dguess := d;
                Tguess := T;
                i := i + 1;
                (d,T) := fixdT(dguess, Tguess);
              end while;
              if not found then
                error := 1;
              end if;
              assert(error <> 1,
                "Error in inverse function dtofph3: iteration failed");
            end dtofph3;

            function dtofps3 "Inverse iteration in region 3: (d,T) = f(p,s)"
              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              input SI.Pressure delp "Iteration accuracy";
              input SI.SpecificEntropy dels "Iteration accuracy";
              output SI.Density d "Density";
              output SI.Temperature T "Temperature (K)";
              output Integer error
              "Error flag: iteration failed if different from 0";
          protected
              SI.Temperature Tguess "Initial temperature";
              SI.Density dguess "Initial density";
              Integer i "Iteration counter";
              Real ds "Newton-error in s-direction";
              Real dp "Newton-error in p-direction";
              Real det "Determinant of directional derivatives";
              Real deld "Newton-step in d-direction";
              Real delt "Newton-step in T-direction";
              Modelica.Media.Common.HelmholtzDerivs f
              "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
              Modelica.Media.Common.NewtonDerivatives_ps nDerivs
              "Derivatives needed in Newton iteration";
              Boolean found "Flag for iteration success";
              Integer subregion "1 for subregion 3a, 2 for subregion 3b";
            algorithm
              i := 0;
              error := 0;
              found := false;
              if p < data.PCRIT then
                // allow a 1 J/K margin inside the (well approximated) phase boundary
                subregion := if s < (Regions.sl_p(p) + 10.0) then 1 else if s > (
                  Regions.sv_p(p) - 10.0) then 2 else 0;
                assert(subregion <> 0,
                  "Inverse iteration of dt from ps called in 2 phase region: this is illegal!");
              else
                subregion := if s < data.SCRIT then 1 else 2;
              end if;
              T := if subregion == 1 then Basic.T3a_ps(p, s) else Basic.T3b_ps(p, s);
              d := if subregion == 1 then 1/Basic.v3a_ps(p, s) else 1/Basic.v3b_ps(p,
                s);
              while ((i < IterationData.IMAX) and not found) loop
                f := Basic.f3(d, T);
                nDerivs := Modelica.Media.Common.Helmholtz_ps(f);
                ds := nDerivs.s - s;
                dp := nDerivs.p - p;
                if ((abs(ds/s) <= dels) and (abs(dp/p) <= delp)) then
                  found := true;
                end if;
                det := nDerivs.st*nDerivs.pd - nDerivs.pt*nDerivs.sd;
                delt := (nDerivs.pd*ds - nDerivs.sd*dp)/det;
                deld := (nDerivs.st*dp - nDerivs.pt*ds)/det;
                T := T - delt;
                d := d - deld;
                dguess := d;
                Tguess := T;
                i := i + 1;
                (d,T) := fixdT(dguess, Tguess);
              end while;
              if not found then
                error := 1;
              end if;
              assert(error <> 1,
                "Error in inverse function dtofps3: iteration failed");
            end dtofps3;

            function pofdt125
            "Inverse iteration in region 1,2 and 5: p = g(d,T)"
              extends Modelica.Icons.Function;
              input SI.Density d "Density";
              input SI.Temperature T "Temperature (K)";
              input SI.Pressure reldd "Relative iteration accuracy of density";
              input Integer region
              "Region in IAPWS/IF97 in which inverse should be calculated";
              output SI.Pressure p "Pressure";
              output Integer error
              "Error flag: iteration failed if different from 0";
          protected
              Integer i "Counter for while-loop";
              Modelica.Media.Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
              Boolean found "Flag if iteration has been successful";
              Real dd
              "Difference between density for guessed p and the current density";
              Real delp "Step in p in Newton-iteration";
              Real relerr "Relative error in d";
              SI.Pressure pguess1=1.0e6 "Initial pressure guess in region 1";
              SI.Pressure pguess2 "Initial pressure guess in region 2";
              constant SI.Pressure pguess5=0.5e6
              "Initial pressure guess in region 5";
            algorithm
              i := 0;
              error := 0;
              pguess2 := 42800*d;
              found := false;
              if region == 1 then
                p := pguess1;
              elseif region == 2 then
                p := pguess2;
              else
                p := pguess5;
              end if;
              while ((i < IterationData.IMAX) and not found) loop
                if region == 1 then
                  g := Basic.g1(p, T);
                elseif region == 2 then
                  g := Basic.g2(p, T);
                else
                  g := Basic.g5(p, T);
                end if;
                dd := p/(data.RH2O*T*g.pi*g.gpi) - d;
                relerr := dd/d;
                if (abs(relerr) < reldd) then
                  found := true;
                end if;
                delp := dd*(-p*p/(d*d*data.RH2O*T*g.pi*g.pi*g.gpipi));
                p := p - delp;
                i := i + 1;
                if not found then
                  if p < triple.ptriple then
                    p := 2.0*triple.ptriple;
                  end if;
                  if p > data.PLIMIT1 then
                    p := 0.95*data.PLIMIT1;
                  end if;
                end if;
              end while;

              // print("i = " + i2s(i) + ", p = " + r2s(p/1.0e5) + ", delp = " + r2s(delp*1.0e-5) + "\n");
              if not found then
                error := 1;
              end if;
              assert(error <> 1,
                "Error in inverse function pofdt125: iteration failed");
            end pofdt125;

            function tofph5 "Inverse iteration in region 5: (p,T) = f(p,h)"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEnthalpy h "Specific enthalpy";
              input SI.SpecificEnthalpy reldh "Iteration accuracy";
              output SI.Temperature T "Temperature (K)";
              output Integer error
              "Error flag: iteration failed if different from 0";

          protected
              Modelica.Media.Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
              SI.SpecificEnthalpy proh "H for current guess in T";
              constant SI.Temperature Tguess=1500 "Initial temperature";
              Integer i "Iteration counter";
              Real relerr "Relative error in h";
              Real dh "Newton-error in h-direction";
              Real dT "Newton-step in T-direction";
              Boolean found "Flag for iteration success";
            algorithm
              i := 0;
              error := 0;
              T := Tguess;
              found := false;
              while ((i < IterationData.IMAX) and not found) loop
                g := Basic.g5(p, T);
                proh := data.RH2O*T*g.tau*g.gtau;
                dh := proh - h;
                relerr := dh/h;
                if (abs(relerr) < reldh) then
                  found := true;
                end if;
                dT := dh/(-data.RH2O*g.tau*g.tau*g.gtautau);
                T := T - dT;
                i := i + 1;
              end while;
              if not found then
                error := 1;
              end if;
              assert(error <> 1, "Error in inverse function tofph5: iteration failed");
            end tofph5;

            function tofps5 "Inverse iteration in region 5: (p,T) = f(p,s)"

              extends Modelica.Icons.Function;
              input SI.Pressure p "Pressure";
              input SI.SpecificEntropy s "Specific entropy";
              input SI.SpecificEnthalpy relds "Iteration accuracy";
              output SI.Temperature T "Temperature (K)";
              output Integer error
              "Error flag: iteration failed if different from 0";

          protected
              Modelica.Media.Common.GibbsDerivs g
              "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
              SI.SpecificEntropy pros "S for current guess in T";
              parameter SI.Temperature Tguess=1500 "Initial temperature";
              Integer i "Iteration counter";
              Real relerr "Relative error in s";
              Real ds "Newton-error in s-direction";
              Real dT "Newton-step in T-direction";
              Boolean found "Flag for iteration success";
            algorithm
              i := 0;
              error := 0;
              T := Tguess;
              found := false;
              while ((i < IterationData.IMAX) and not found) loop
                g := Basic.g5(p, T);
                pros := data.RH2O*(g.tau*g.gtau - g.g);
                ds := pros - s;
                relerr := ds/s;
                if (abs(relerr) < relds) then
                  found := true;
                end if;
                dT := ds*T/(-data.RH2O*g.tau*g.tau*g.gtautau);
                T := T - dT;
                i := i + 1;
              end while;
              if not found then
                error := 1;
              end if;
              assert(error <> 1, "Error in inverse function tofps5: iteration failed");
            end tofps5;
            annotation (Documentation(info="<HTML><h4>Package description</h4>
          <h4>Package contents</h4>
          <ul>
          <li>Function <b>fixdT</b> constrains density and temperature to allowed region</li>
          <li>Function <b>dofp13</b> computes d as a function of p at boundary between regions 1 and 3</li>
          <li>Function <b>dofp23</b> computes d as a function of p at boundary between regions 2 and 3</li>
          <li>Function <b>dofpt3</b> iteration to compute d as a function of p and T in region 3</li>
          <li>Function <b>dtofph3</b> iteration to compute d and T as a function of p and h in region 3</li>
          <li>Function <b>dtofps3</b> iteration to compute d and T as a function of p and s in region 3</li>
          <li>Function <b>dtofpsdt3</b> iteration to compute d and T as a function of p and s in region 3,
          with initial guesses</li>
          <li>Function <b>pofdt125</b> iteration to compute p as a function of p and T in regions 1, 2 and 5</li>
          <li>Function <b>tofph5</b> iteration to compute T as a function of p and h in region 5</li>
          <li>Function <b>tofps5</b> iteration to compute T as a function of p and s in region 5</li>
          <li>Function <b>tofpst5</b> iteration to compute T as a function of p and s in region 5, with initial guess in T</li>
          </ul>
          <h4>Version Info and Revision history
          </h4>
          <ul>
          <li>First implemented: <i>July, 2000</i>
          by <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>
          </li>
          </ul>
          <address>Author: Hubertus Tummescheit, <br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
          </address>
          <ul>
          <li>Initial version: July 2000</li>
          <li>Documentation added: December 2002</li>
          </ul>
          </html>"));
          end Inverses;
          annotation (Documentation(info="<HTML>
<style type=\"text/css\">
.nobr
{
white-space:nowrap;
}
</style>
    <h4>Version Info and Revision history</h4>
        <ul>
        <li>First implemented: <i>July, 2000</i>
        by Hubertus Tummescheit
        for the ThermoFluid Library with help from Jonas Eborn and Falko Jens Wagner
        </li>
      <li>Code reorganization, enhanced documentation, additional functions:   <i>December, 2002</i>
      by <a href=\"mailto:Hubertus.Tummescheit@modelon.se\">Hubertus Tummescheit</a> and moved to Modelica
      properties library.</li>
        </ul>
      <address>Author: Hubertus Tummescheit, <br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
      </address>
        <p>In September 1997, the International Association for the Properties
        of Water and Steam (<A HREF=\"http://www.iapws.org\">IAPWS</A>) adopted a
        new formulation for the thermodynamic properties of water and steam for
        industrial use. This new industrial standard is called \"IAPWS Industrial
        Formulation for the Thermodynamic Properties of Water and Steam\" (IAPWS-IF97).
        The formulation IAPWS-IF97 replaces the previous industrial standard IFC-67.
        <p>Based on this new formulation, a new steam table, titled \"<a href=\"http://www.springer.de/cgi-bin/search_book.pl?isbn=3-540-64339-7\">Properties of Water and Steam</a>\" by W. Wagner and A. Kruse, was published by
        the Springer-Verlag, Berlin - New-York - Tokyo in April 1998. This
        steam table, ref. <a href=\"#steamprop\">[1]</a> is bilingual (English /
        German) and contains a complete description of the equations of
        IAPWS-IF97. This reference is the authoritative source of information
        for this implementation. A mostly identical version has been published by the International
        Association for the Properties
        of Water and Steam (<A HREF=\"http://www.iapws.org\">IAPWS</A>) with permission granted to re-publish the
        information if credit is given to IAPWS. This document is distributed with this library as
        <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a>.
        In addition, the equations published by <A HREF=\"http://www.iapws.org\">IAPWS</A> for
        the transport properties dynamic viscosity (standards document: <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/visc.pdf\">visc.pdf</a>)
        and thermal conductivity (standards document: <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/thcond.pdf\">thcond.pdf</a>)
        and equations for the surface tension (standards document: <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/surf.pdf\">surf.pdf</a>)
        are also implemented in this library and included for reference.</p>
        <p>
        The functions in BaseIF97.mo are low level functions which should
        only be used in those exceptions when the standard user level
        functions in Water.mo do not contain the wanted properties.
     </p>
<p>Based on IAPWS-IF97, Modelica functions are available for calculating
the most common thermophysical properties (thermodynamic and transport
properties). The implementation requires part of the common medium
property infrastructure of the Modelica.Thermal.Properties library in the file
Common.mo. There are a few extensions from the version of IF97 as
documented in <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a> in order to improve performance for
dynamic simulations. Input variables for calculating the properties are
only implemented for a limited number of variable pairs which make sense as dynamic states: (p,h), (p,T), (p,s) and (d,T).
</p>
<hr size=3 width=\"70%\">
<h4><a name=\"regions\">1. Structure and Regions of IAPWS-IF97</a></h4>
<p>The IAPWS Industrial Formulation 1997 consists of
a set of equations for different regions which cover the following range
of validity:</p>
<table border=0 cellpadding=4>
<tr>
<td valign=\"top\">273,15 K &lt; <em>T</em> &lt; 1073,15 K</td>
<td valign=\"top\"><em>p</em> &lt; 100 MPa</td>
</tr>
<tr>
<td valign=\"top\">1073,15 K &lt; <em>T</em> &lt; 2273,15 K</td>
<td valign=\"top\"><em>p</em> &lt; 10 MPa</td>
</tr>
</table>
<p>
Figure 1 shows the 5 regions into which the entire range of validity of
IAPWS-IF97 is divided. The boundaries of the regions can be directly taken
from Fig. 1 except for the boundary between regions 2 and 3; this boundary,
which corresponds approximately to the isentropic line <span class=\"nobr\"><em>s</em> = 5.047 kJ kg
<sup>-1</sup>K<sup>-1</sup></span>, is defined
by a corresponding auxiliary equation. Both regions 1 and 2 are individually
covered by a fundamental equation for the specific Gibbs free energy <span class=\"nobr\"><em>g</em>( <em>p</em>,<em>T</em> )</span>, region 3 by a fundamental equation for the specific Helmholtz
free energy <span class=\"nobr\"><em>f </em>(<em> <font face=\"symbol\">r</font></em>,<em>T
</em>)</span>, and the saturation curve, corresponding to region 4, by a saturation-pressure
equation <span><em>p</em><sub>s</sub>( <em>T</em> )</span>. The high-temperature
region 5 is also covered by a <span class=\"nobr\"><em>g</em>( <em>p</em>,<em>T</em> )</span> equation. These
5 equations, shown in rectangular boxes in Fig. 1, form the so-called <em>basic
equations</em>.
</p>
<table border=\"0\" cellspacing=\"0\" cellpadding=\"2\">
  <caption align=\"bottom\">Figure 1: Regions and equations of IAPWS-IF97</caption>
  <tr>
    <td>
    <img src=\"modelica://Modelica/Resources/Images/Media/Water/if97.png\" alt=\"Regions and equations of IAPWS-IF97\">
    </td>
  </tr>
</table>
<p>
In addition to these basic equations, so-called <em>backward
equations</em> are provided for regions 1, 2, and 4 in form of
<span class=\"nobr\"><em>T</em>( <em>p</em>,<em>h</em> )</span> and <span class=\"nobr\"><em>T</em>( <em>
p</em>,<em>s</em> )</span> for regions 1 and 2, and <span class=\"nobr\"><em>T</em><sub>s</sub>( <em>p</em> )</span> for region 4. These
backward equations, marked in grey in Fig. 1, were developed in such a
way that they are numerically very consistent with the corresponding
basic equation. Thus, properties as functions of&nbsp; <em>p</em>,<em>h
</em>and of&nbsp;<em> p</em>,<em>s </em>for regions 1 and 2, and of
<em>p</em> for region 4 can be calculated without any iteration. As a
result of this special concept for the development of the new
industrial standard IAPWS-IF97, the most important properties can be
calculated extremely quickly. All Modelica functions are optimized
with regard to short computing times.
</p>
<p>
The complete description of the individual equations of the new industrial
formulation IAPWS-IF97 is given in <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a>. Comprehensive information on
IAPWS-IF97 (requirements, concept, accuracy, consistency along region boundaries,
and the increase of computing speed in comparison with IFC-67, etc.) can
be taken from <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a> or [2].
</p>
<p>
<a name=\"steamprop\">[1]<em>Wagner, W., Kruse, A.</em> Properties of Water
and Steam / Zustandsgr&ouml;&szlig;en von Wasser und Wasserdampf / IAPWS-IF97.
Springer-Verlag, Berlin, 1998.</a>
</p>
<p>
[2] <em>Wagner, W., Cooper, J. R., Dittmann, A., Kijima,
J., Kretzschmar, H.-J., Kruse, A., Mare&#353; R., Oguchi, K., Sato, H., St&ouml;cker,
I., &#352;ifner, O., Takaishi, Y., Tanishita, I., Tr&uuml;benbach, J., and Willkommen,
Th.</em> The IAPWS Industrial Formulation 1997 for the Thermodynamic Properties
of Water and Steam. ASME Journal of Engineering for Gas Turbines and Power 122 (2000), 150 - 182.
</p>
<HR size=3 width=\"90%\">
<h4>2. Calculable Properties      </h4>
<table border=\"1\" cellpadding=\"2\" cellspacing=\"0\">
       <tbody>
       <tr>
       <td valign=\"top\" bgcolor=\"#cccccc\"><br>
      </td>
      <td valign=\"top\" bgcolor=\"#cccccc\"><b>Common name</b><br>
       </td>
       <td valign=\"top\" bgcolor=\"#cccccc\"><b>Abbreviation </b><br>
       </td>
       <td valign=\"top\" bgcolor=\"#cccccc\"><b>Unit</b><br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;1<br>
      </td>
      <td valign=\"top\">Pressure</td>
       <td valign=\"top\">p<br>
        </td>
       <td valign=\"top\">Pa<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;2<br>
      </td>
      <td valign=\"top\">Temperature</td>
       <td valign=\"top\">T<br>
       </td>
       <td valign=\"top\">K<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;3<br>
      </td>
      <td valign=\"top\">Density</td>
        <td valign=\"top\">d<br>
        </td>
       <td valign=\"top\">kg/m<sup>3</sup><br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;4<br>
      </td>
      <td valign=\"top\">Specific volume</td>
        <td valign=\"top\">v<br>
        </td>
       <td valign=\"top\">m<sup>3</sup>/kg<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;5<br>
      </td>
      <td valign=\"top\">Specific enthalpy</td>
       <td valign=\"top\">h<br>
       </td>
       <td valign=\"top\">J/kg<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;6<br>
      </td>
      <td valign=\"top\">Specific entropy</td>
       <td valign=\"top\">s<br>
       </td>
       <td valign=\"top\">J/(kg K)<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;7<br>
      </td>
      <td valign=\"top\">Specific internal energy<br>
       </td>
       <td valign=\"top\">u<br>
       </td>
       <td valign=\"top\">J/kg<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;8<br>
      </td>
      <td valign=\"top\">Specific isobaric heat capacity</td>
       <td valign=\"top\">c<sub>p</sub><br>
       </td>
       <td valign=\"top\">J/(kg K)<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">&nbsp;9<br>
      </td>
      <td valign=\"top\">Specific isochoric heat capacity</td>
       <td valign=\"top\">c<sub>v</sub><br>
       </td>
       <td valign=\"top\">J/(kg K)<br>
       </td>
       </tr>
       <tr>
       <td valign=\"top\">10<br>
      </td>
      <td valign=\"top\">Isentropic exponent, kappa<span class=\"nobr\">=<font face=\"Symbol\">-</font>(v/p)
(dp/dv)<sub>s</sub></span></td>
     <td valign=\"top\">kappa (<font face=\"Symbol\">k</font>)<br>
     </td>
     <td valign=\"top\">1<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">11<br>
      </td>
      <td valign=\"top\">Speed of sound<br>
     </td>
     <td valign=\"top\">a<br>
     </td>
     <td valign=\"top\">m/s<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">12<br>
      </td>
      <td valign=\"top\">Dryness fraction<br>
     </td>
     <td valign=\"top\">x<br>
     </td>
     <td valign=\"top\">kg/kg<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">13<br>
      </td>
      <td valign=\"top\">Specific Helmholtz free energy,     f = u - Ts</td>
     <td valign=\"top\">f<br>
     </td>
     <td valign=\"top\">J/kg<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">14<br>
      </td>
      <td valign=\"top\">Specific Gibbs free energy,     g = h - Ts</td>
     <td valign=\"top\">g<br>
     </td>
     <td valign=\"top\">J/kg<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">15<br>
      </td>
      <td valign=\"top\">Isenthalpic exponent, <span class=\"nobr\"> theta = -(v/p)(dp/dv)<sub>h</sub></span></td>
     <td valign=\"top\">theta (<font face=\"Symbol\">q</font>)<br>
     </td>
     <td valign=\"top\">1<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">16<br>
      </td>
      <td valign=\"top\">Isobaric volume expansion coefficient, alpha = v<sup>-1</sup>       (dv/dT)<sub>p</sub></td>
     <td valign=\"top\">alpha  (<font face=\"Symbol\">a</font>)<br>
     </td>
       <td valign=\"top\">1/K<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">17<br>
      </td>
      <td valign=\"top\">Isochoric pressure coefficient,     <span class=\"nobr\">beta = p<sup><font face=\"Symbol\">-</font>1</sup>(dp/dT)<sub>v</sub></span></td>
     <td valign=\"top\">beta (<font face=\"Symbol\">b</font>)<br>
     </td>
     <td valign=\"top\">1/K<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">18<br>
     </td>
     <td valign=\"top\">Isothermal compressibility, <span class=\"nobr\">gamma = <font
 face=\"Symbol\">-</font>v<sup><font face=\"Symbol\">-</font>1</sup>(dv/dp)<sub>T</sub></span></td>
     <td valign=\"top\">gamma (<font face=\"Symbol\">g</font>)<br>
     </td>
     <td valign=\"top\">1/Pa<br>
     </td>
     </tr>
     <!-- <tr><td valign=\"top\">f</td><td valign=\"top\">Fugacity</td></tr> --> <tr>
     <td valign=\"top\">19<br>
      </td>
      <td valign=\"top\">Dynamic viscosity</td>
     <td valign=\"top\">eta (<font face=\"Symbol\">h</font>)<br>
     </td>
     <td valign=\"top\">Pa s<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">20<br>
      </td>
      <td valign=\"top\">Kinematic viscosity</td>
     <td valign=\"top\">nu (<font face=\"Symbol\">n</font>)<br>
     </td>
     <td valign=\"top\">m<sup>2</sup>/s<br>
     </td>
     </tr>
     <!-- <tr><td valign=\"top\">Pr</td><td valign=\"top\">Prandtl number</td></tr> --> <tr>
     <td valign=\"top\">21<br>
      </td>
      <td valign=\"top\">Thermal conductivity</td>
     <td valign=\"top\">lambda (<font face=\"Symbol\">l</font>)<br>
     </td>
     <td valign=\"top\">W/(m K)<br>
     </td>
     </tr>
     <tr>
     <td valign=\"top\">22 <br>
      </td>
      <td valign=\"top\">Surface tension</td>
     <td valign=\"top\">sigma (<font face=\"Symbol\">s</font>)<br>
     </td>
     <td valign=\"top\">N/m<br>
     </td>
     </tr>
  </tbody>
</table>
        <p>The properties 1-11 are calculated by default with the functions for dynamic
        simulation, 2 of these variables are the dynamic states and are the inputs
        to calculate all other properties. In addition to these properties
        of general interest, the entries to the thermodynamic Jacobian matrix which render
        the mass- and energy balances explicit in the input variables to the property calculation are also calculated.
        For an explanatory example using pressure and specific enthalpy as states, see the Examples sub-package.</p>
        <p>The high-level calls to steam properties are grouped into records comprising both the properties of general interest
        and the entries to the thermodynamic Jacobian. If additional properties are
        needed the low level functions in BaseIF97 provide more choice.</p>
        <HR size=3 width=\"90%\">
        <h4>Additional functions</h4>
        <ul>
        <li>Function <b>boundaryvals_p</b> computes the temperature and the specific enthalpy and
        entropy on both phase boundaries as a function of p</li>
        <li>Function <b>boundaryderivs_p</b> is the Modelica derivative function of <b>boundaryvals_p</b></li>
        <li>Function <b>extraDerivs_ph</b> computes all entries to Bridgmans tables for all
        one-phase regions of IF97 using inputs (p,h). All 336 directional derivatives of the
        thermodynamic surface can be computed as a ratio of two entries in the return data, see package Common
        for details.</li>
        <li>Function <b>extraDerivs_pT</b> computes all entries to Bridgmans tables for all
        one-phase regions of IF97 using inputs (p,T).</li>
        </ul>
        </HTML>"));
        end BaseIF97;

        function waterBaseProp_ph "Intermediate property record for water"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "Phase: 2 for two-phase, 1 for one phase, 0 if unknown";
          input Integer region=0
          "If 0, do region computation, otherwise assume the region is this input";
          output Common.IF97BaseTwoPhase aux "Auxiliary record";
      protected
          Common.GibbsDerivs g
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Common.HelmholtzDerivs f
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Integer error "Error flag for inverse iterations";
          SI.SpecificEnthalpy h_liq "Liquid specific enthalpy";
          SI.Density d_liq "Liquid density";
          SI.SpecificEnthalpy h_vap "Vapour specific enthalpy";
          SI.Density d_vap "Vapour density";
          Common.PhaseBoundaryProperties liq "Phase boundary property record";
          Common.PhaseBoundaryProperties vap "Phase boundary property record";
          Common.GibbsDerivs gl
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Common.GibbsDerivs gv
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Modelica.Media.Common.HelmholtzDerivs fl
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Modelica.Media.Common.HelmholtzDerivs fv
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          SI.Temperature t1
          "Temperature at phase boundary, using inverse from region 1";
          SI.Temperature t2
          "Temperature at phase boundary, using inverse from region 2";
        algorithm
          aux.region := if region == 0 then (if phase == 2 then 4 else
            BaseIF97.Regions.region_ph(
              p=p,
              h=h,
              phase=phase)) else region;
          aux.phase := if phase <> 0 then phase else if aux.region == 4 then 2 else 1;
          aux.p := max(p, 611.657);
          aux.h := max(h, 1e3);
          aux.R := BaseIF97.data.RH2O;
          if (aux.region == 1) then
            aux.T := BaseIF97.Basic.tph1(aux.p, aux.h);
            g := BaseIF97.Basic.g1(p, aux.T);
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := p/(aux.R*aux.T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*aux.T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 0.0;
            aux.dpT := -aux.vt/aux.vp;
          elseif (aux.region == 2) then
            aux.T := BaseIF97.Basic.tph2(aux.p, aux.h);
            g := BaseIF97.Basic.g2(p, aux.T);
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := p/(aux.R*aux.T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*aux.T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 1.0;
            aux.dpT := -aux.vt/aux.vp;
          elseif (aux.region == 3) then
            (aux.rho,aux.T,error) := BaseIF97.Inverses.dtofph3(
                p=aux.p,
                h=aux.h,
                delp=1.0e-7,
                delh=1.0e-6);
            f := BaseIF97.Basic.f3(aux.rho, aux.T);
            aux.h := aux.R*aux.T*(f.tau*f.ftau + f.delta*f.fdelta);
            aux.s := aux.R*(f.tau*f.ftau - f.f);
            aux.pd := aux.R*aux.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
            aux.pt := aux.R*aux.rho*f.delta*(f.fdelta - f.tau*f.fdeltatau);
            aux.cv := abs(aux.R*(-f.tau*f.tau*f.ftautau))
            "Can be close to neg. infinity near critical point";
            aux.cp := (aux.rho*aux.rho*aux.pd*aux.cv + aux.T*aux.pt*aux.pt)/(aux.rho*
              aux.rho*aux.pd);
            aux.x := 0.0;
            aux.dpT := aux.pt;
            /*safety against div-by-0 in initialization*/
          elseif (aux.region == 4) then
            h_liq := hl_p(p);
            h_vap := hv_p(p);
            aux.x := if (h_vap <> h_liq) then (h - h_liq)/(h_vap - h_liq) else 1.0;
            if p < BaseIF97.data.PLIMIT4A then
              t1 := BaseIF97.Basic.tph1(aux.p, h_liq);
              t2 := BaseIF97.Basic.tph2(aux.p, h_vap);
              gl := BaseIF97.Basic.g1(aux.p, t1);
              gv := BaseIF97.Basic.g2(aux.p, t2);
              liq := Common.gibbsToBoundaryProps(gl);
              vap := Common.gibbsToBoundaryProps(gv);
              aux.T := t1 + aux.x*(t2 - t1);
            else
              aux.T := BaseIF97.Basic.tsat(aux.p);
              // how to avoid ?
              d_liq := rhol_T(aux.T);
              d_vap := rhov_T(aux.T);
              fl := BaseIF97.Basic.f3(d_liq, aux.T);
              fv := BaseIF97.Basic.f3(d_vap, aux.T);
              liq := Common.helmholtzToBoundaryProps(fl);
              vap := Common.helmholtzToBoundaryProps(fv);
              //  aux.dpT := BaseIF97.Basic.dptofT(aux.T);
            end if;
            aux.dpT := if (liq.d <> vap.d) then (vap.s - liq.s)*liq.d*vap.d/(liq.d -
              vap.d) else BaseIF97.Basic.dptofT(aux.T);
            aux.s := liq.s + aux.x*(vap.s - liq.s);
            aux.rho := liq.d*vap.d/(vap.d + aux.x*(liq.d - vap.d));
            aux.cv := Common.cv2Phase(
                liq,
                vap,
                aux.x,
                aux.T,
                p);
            aux.cp := liq.cp + aux.x*(vap.cp - liq.cp);
            aux.pt := liq.pt + aux.x*(vap.pt - liq.pt);
            aux.pd := liq.pd + aux.x*(vap.pd - liq.pd);
          elseif (aux.region == 5) then
            (aux.T,error) := BaseIF97.Inverses.tofph5(
                p=aux.p,
                h=aux.h,
                reldh=1.0e-7);
            assert(error == 0, "Error in inverse iteration of steam tables");
            g := BaseIF97.Basic.g5(aux.p, aux.T);
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := p/(aux.R*aux.T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*aux.T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.dpT := -aux.vt/aux.vp;
          else
            assert(false, "Error in region computation of IF97 steam tables" +
              "(p = " + String(p) + ", h = " + String(h) + ")");
          end if;
        end waterBaseProp_ph;

        function waterBaseProp_ps "Intermediate property record for water"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Integer phase=0
          "Phase: 2 for two-phase, 1 for one phase, 0 if unknown";
          input Integer region=0
          "If 0, do region computation, otherwise assume the region is this input";
          output Common.IF97BaseTwoPhase aux "Auxiliary record";
      protected
          Common.GibbsDerivs g
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Common.HelmholtzDerivs f
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Integer error "Error flag for inverse iterations";
          SI.SpecificEntropy s_liq "Liquid specific entropy";
          SI.Density d_liq "Liquid density";
          SI.SpecificEntropy s_vap "Vapour specific entropy";
          SI.Density d_vap "Vapour density";
          Common.PhaseBoundaryProperties liq "Phase boundary property record";
          Common.PhaseBoundaryProperties vap "Phase boundary property record";
          Common.GibbsDerivs gl
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Common.GibbsDerivs gv
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Modelica.Media.Common.HelmholtzDerivs fl
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Modelica.Media.Common.HelmholtzDerivs fv
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          SI.Temperature t1
          "Temperature at phase boundary, using inverse from region 1";
          SI.Temperature t2
          "Temperature at phase boundary, using inverse from region 2";
        algorithm
          aux.region := if region == 0 then (if phase == 2 then 4 else
            BaseIF97.Regions.region_ps(
              p=p,
              s=s,
              phase=phase)) else region;
          aux.phase := if phase <> 0 then phase else if aux.region == 4 then 2 else 1;
          aux.p := p;
          aux.s := s;
          aux.R := BaseIF97.data.RH2O;
          if (aux.region == 1) then
            aux.T := BaseIF97.Basic.tps1(p, s);
            g := BaseIF97.Basic.g1(p, aux.T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.rho := p/(aux.R*aux.T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*aux.T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 0.0;
            aux.dpT := -aux.vt/aux.vp;
          elseif (aux.region == 2) then
            aux.T := BaseIF97.Basic.tps2(p, s);
            g := BaseIF97.Basic.g2(p, aux.T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.rho := p/(aux.R*aux.T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*aux.T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 1.0;
            aux.dpT := -aux.vt/aux.vp;
          elseif (aux.region == 3) then
            (aux.rho,aux.T,error) := BaseIF97.Inverses.dtofps3(
                p=p,
                s=s,
                delp=1.0e-7,
                dels=1.0e-6);
            f := BaseIF97.Basic.f3(aux.rho, aux.T);
            aux.h := aux.R*aux.T*(f.tau*f.ftau + f.delta*f.fdelta);
            aux.s := aux.R*(f.tau*f.ftau - f.f);
            aux.pd := aux.R*aux.T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
            aux.pt := aux.R*aux.rho*f.delta*(f.fdelta - f.tau*f.fdeltatau);
            aux.cv := aux.R*(-f.tau*f.tau*f.ftautau);
            aux.cp := (aux.rho*aux.rho*aux.pd*aux.cv + aux.T*aux.pt*aux.pt)/(aux.rho*
              aux.rho*aux.pd);
            aux.x := 0.0;
            aux.dpT := aux.pt;
            /*safety against div-by-0 in initialization*/
          elseif (aux.region == 4) then
            s_liq := BaseIF97.Regions.sl_p(p);
            s_vap := BaseIF97.Regions.sv_p(p);
            aux.x := if (s_vap <> s_liq) then (s - s_liq)/(s_vap - s_liq) else 1.0;
            if p < BaseIF97.data.PLIMIT4A then
              t1 := BaseIF97.Basic.tps1(p, s_liq);
              t2 := BaseIF97.Basic.tps2(p, s_vap);
              gl := BaseIF97.Basic.g1(p, t1);
              gv := BaseIF97.Basic.g2(p, t2);
              liq := Common.gibbsToBoundaryProps(gl);
              vap := Common.gibbsToBoundaryProps(gv);
              aux.T := t1 + aux.x*(t2 - t1);
            else
              aux.T := BaseIF97.Basic.tsat(p);
              d_liq := rhol_T(aux.T);
              d_vap := rhov_T(aux.T);
              fl := BaseIF97.Basic.f3(d_liq, aux.T);
              fv := BaseIF97.Basic.f3(d_vap, aux.T);
              liq := Common.helmholtzToBoundaryProps(fl);
              vap := Common.helmholtzToBoundaryProps(fv);
            end if;
            aux.dpT := if (liq.d <> vap.d) then (vap.s - liq.s)*liq.d*vap.d/(liq.d -
              vap.d) else BaseIF97.Basic.dptofT(aux.T);
            aux.h := liq.h + aux.x*(vap.h - liq.h);
            aux.rho := liq.d*vap.d/(vap.d + aux.x*(liq.d - vap.d));
            aux.cv := Common.cv2Phase(
                liq,
                vap,
                aux.x,
                aux.T,
                p);
            aux.cp := liq.cp + aux.x*(vap.cp - liq.cp);
            aux.pt := liq.pt + aux.x*(vap.pt - liq.pt);
            aux.pd := liq.pd + aux.x*(vap.pd - liq.pd);
          elseif (aux.region == 5) then
            (aux.T,error) := BaseIF97.Inverses.tofps5(
                p=p,
                s=s,
                relds=1.0e-7);
            assert(error == 0, "Error in inverse iteration of steam tables");
            g := BaseIF97.Basic.g5(p, aux.T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.rho := p/(aux.R*aux.T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*aux.T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.dpT := -aux.vt/aux.vp;
            aux.x := 1.0;
          else
            assert(false, "Error in region computation of IF97 steam tables" +
              "(p = " + String(p) + ", s = " + String(s) + ")");
          end if;
        end waterBaseProp_ps;

        function rho_props_ps
        "Density as function of pressure and specific entropy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Common.IF97BaseTwoPhase properties "Auxiliary record";
          output SI.Density rho "Density";
        algorithm
          rho := properties.rho;
          annotation (Inline=false, LateInline=true);
        end rho_props_ps;

        function rho_ps "Density as function of pressure and specific entropy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Density rho "Density";
        algorithm
          rho := rho_props_ps(
              p,
              s,
              waterBaseProp_ps(
                p,
                s,
                phase,
                region));
          annotation (Inline=true);
        end rho_ps;

        function T_props_ps
        "Temperature as function of pressure and specific entropy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Common.IF97BaseTwoPhase properties "Auxiliary record";
          output SI.Temperature T "Temperature";
        algorithm
          T := properties.T;
          annotation (Inline=false, LateInline=true);
        end T_props_ps;

        function T_ps
        "Temperature as function of pressure and specific entropy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Temperature T "Temperature";
        algorithm
          T := T_props_ps(
              p,
              s,
              waterBaseProp_ps(
                p,
                s,
                phase,
                region));
          annotation (Inline=true);
        end T_ps;

        function h_props_ps
        "Specific enthalpy as function or pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificEnthalpy h "Specific enthalpy";
        algorithm
          h := aux.h;
          annotation (Inline=false, LateInline=true);
        end h_props_ps;

        function h_ps
        "Specific enthalpy as function or pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificEnthalpy h "Specific enthalpy";
        algorithm
          h := h_props_ps(
              p,
              s,
              waterBaseProp_ps(
                p,
                s,
                phase,
                region));
          annotation (Inline=true);
        end h_ps;

        function rho_props_ph
        "Density as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase properties "Auxiliary record";
          output SI.Density rho "Density";
        algorithm
          rho := properties.rho;
          annotation (
            derivative(noDerivative=properties) = rho_ph_der,
            Inline=false,
            LateInline=true);
        end rho_props_ph;

        function rho_ph "Density as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Density rho "Density";
        algorithm
          rho := rho_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end rho_ph;

        function rho_ph_der "Derivative function of rho_ph"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real p_der "Derivative of pressure";
          input Real h_der "Derivative of specific enthalpy";
          output Real rho_der "Derivative of density";
        algorithm
          if (aux.region == 4) then
            rho_der := (aux.rho*(aux.rho*aux.cv/aux.dpT + 1.0)/(aux.dpT*aux.T))*p_der
               + (-aux.rho*aux.rho/(aux.dpT*aux.T))*h_der;
          elseif (aux.region == 3) then
            rho_der := ((aux.rho*(aux.cv*aux.rho + aux.pt))/(aux.rho*aux.rho*aux.pd*
              aux.cv + aux.T*aux.pt*aux.pt))*p_der + (-aux.rho*aux.rho*aux.pt/(aux.rho
              *aux.rho*aux.pd*aux.cv + aux.T*aux.pt*aux.pt))*h_der;
          else
            //regions 1,2,5
            rho_der := (-aux.rho*aux.rho*(aux.vp*aux.cp - aux.vt/aux.rho + aux.T*aux.vt
              *aux.vt)/aux.cp)*p_der + (-aux.rho*aux.rho*aux.vt/(aux.cp))*h_der;
          end if;
        end rho_ph_der;

        function T_props_ph
        "Temperature as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase properties "Auxiliary record";
          output SI.Temperature T "Temperature";
        algorithm
          T := properties.T;
          annotation (
            derivative(noDerivative=properties) = T_ph_der,
            Inline=false,
            LateInline=true);
        end T_props_ph;

        function T_ph
        "Temperature as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Temperature T "Temperature";
        algorithm
          T := T_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end T_ph;

        function T_ph_der "Derivative function of T_ph"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real p_der "Derivative of pressure";
          input Real h_der "Derivative of specific enthalpy";
          output Real T_der "Derivative of temperature";
        algorithm
          if (aux.region == 4) then
            T_der := 1/aux.dpT*p_der;
          elseif (aux.region == 3) then
            T_der := ((-aux.rho*aux.pd + aux.T*aux.pt)/(aux.rho*aux.rho*aux.pd*aux.cv
               + aux.T*aux.pt*aux.pt))*p_der + ((aux.rho*aux.rho*aux.pd)/(aux.rho*aux.rho
              *aux.pd*aux.cv + aux.T*aux.pt*aux.pt))*h_der;
          else
            //regions 1,2 or 5
            T_der := ((-1/aux.rho + aux.T*aux.vt)/aux.cp)*p_der + (1/aux.cp)*h_der;
          end if;
        end T_ph_der;

        function s_props_ph
        "Specific entropy as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase properties "Auxiliary record";
          output SI.SpecificEntropy s "Specific entropy";
        algorithm
          s := properties.s;
          annotation (
            derivative(noDerivative=properties) = s_ph_der,
            Inline=false,
            LateInline=true);
        end s_props_ph;

        function s_ph
        "Specific entropy as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificEntropy s "Specific entropy";
        algorithm
          s := s_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end s_ph;

        function s_ph_der
        "Specific entropy as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real p_der "Derivative of pressure";
          input Real h_der "Derivative of specific enthalpy";
          output Real s_der "Derivative of entropy";
        algorithm
          s_der := -1/(aux.rho*aux.T)*p_der + 1/aux.T*h_der;
          annotation (Inline=true);
        end s_ph_der;

        function cv_props_ph
        "Specific heat capacity at constant volume as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificHeatCapacity cv "Specific heat capacity";
        algorithm
          cv := aux.cv;
          annotation (Inline=false, LateInline=true);
        end cv_props_ph;

        function cv_ph
        "Specific heat capacity at constant volume as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificHeatCapacity cv "Specific heat capacity";
        algorithm
          cv := cv_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end cv_ph;

        function cp_props_ph
        "Specific heat capacity at constant pressure as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificHeatCapacity cp "Specific heat capacity";
        algorithm
          cp := aux.cp;
          annotation (Inline=false, LateInline=true);
        end cp_props_ph;

        function cp_ph
        "Specific heat capacity at constant pressure as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificHeatCapacity cp "Specific heat capacity";
        algorithm
          cp := cp_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end cp_ph;

        function beta_props_ph
        "Isobaric expansion coefficient as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.RelativePressureCoefficient beta
          "Isobaric expansion coefficient";
        algorithm
          beta := if aux.region == 3 or aux.region == 4 then aux.pt/(aux.rho*aux.pd)
             else aux.vt*aux.rho;
          annotation (Inline=false, LateInline=true);
        end beta_props_ph;

        function beta_ph
        "Isobaric expansion coefficient as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.RelativePressureCoefficient beta
          "Isobaric expansion coefficient";
        algorithm
          beta := beta_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end beta_ph;

        function kappa_props_ph
        "Isothermal compressibility factor as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.IsothermalCompressibility kappa
          "Isothermal compressibility factor";
        algorithm
          kappa := if aux.region == 3 or aux.region == 4 then 1/(aux.rho*aux.pd)
             else -aux.vp*aux.rho;
          annotation (Inline=false, LateInline=true);
        end kappa_props_ph;

        function kappa_ph
        "Isothermal compressibility factor as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.IsothermalCompressibility kappa
          "Isothermal compressibility factor";
        algorithm
          kappa := kappa_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end kappa_ph;

        function velocityOfSound_props_ph
        "Speed of sound as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.Velocity v_sound "Speed of sound";
        algorithm
          // dp/drho at constant s
          v_sound := if aux.region == 3 then sqrt(max(0, (aux.pd*aux.rho*aux.rho*aux.cv
             + aux.pt*aux.pt*aux.T)/(aux.rho*aux.rho*aux.cv))) else if aux.region ==
            4 then sqrt(max(0, 1/((aux.rho*(aux.rho*aux.cv/aux.dpT + 1.0)/(aux.dpT*
            aux.T)) - 1/aux.rho*aux.rho*aux.rho/(aux.dpT*aux.T)))) else sqrt(max(0, -
            aux.cp/(aux.rho*aux.rho*(aux.vp*aux.cp + aux.vt*aux.vt*aux.T))));
          annotation (Inline=false, LateInline=true);
        end velocityOfSound_props_ph;

        function velocityOfSound_ph
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Velocity v_sound "Speed of sound";
        algorithm
          v_sound := velocityOfSound_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end velocityOfSound_ph;

        function isentropicExponent_props_ph
        "Isentropic exponent as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output Real gamma "Isentropic exponent";
        algorithm
          gamma := if aux.region == 3 then 1/(aux.rho*p)*((aux.pd*aux.cv*aux.rho*aux.rho
             + aux.pt*aux.pt*aux.T)/(aux.cv)) else if aux.region == 4 then 1/(aux.rho
            *p)*aux.dpT*aux.dpT*aux.T/aux.cv else -1/(aux.rho*aux.p)*aux.cp/(aux.vp*
            aux.cp + aux.vt*aux.vt*aux.T);
          annotation (Inline=false, LateInline=true);
        end isentropicExponent_props_ph;

        function isentropicExponent_ph
        "Isentropic exponent as function of pressure and specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output Real gamma "Isentropic exponent";
        algorithm
          gamma := isentropicExponent_props_ph(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=false, LateInline=true);
        end isentropicExponent_ph;

        function ddph_props "Density derivative by pressure"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.DerDensityByPressure ddph "Density derivative by pressure";
        algorithm
          ddph := if aux.region == 3 then ((aux.rho*(aux.cv*aux.rho + aux.pt))/(aux.rho
            *aux.rho*aux.pd*aux.cv + aux.T*aux.pt*aux.pt)) else if aux.region == 4
             then (aux.rho*(aux.rho*aux.cv/aux.dpT + 1.0)/(aux.dpT*aux.T)) else (-aux.rho
            *aux.rho*(aux.vp*aux.cp - aux.vt/aux.rho + aux.T*aux.vt*aux.vt)/aux.cp);
          annotation (Inline=false, LateInline=true);
        end ddph_props;

        function ddph "Density derivative by pressure"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.DerDensityByPressure ddph "Density derivative by pressure";
        algorithm
          ddph := ddph_props(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end ddph;

        function ddhp_props "Density derivative by specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.DerDensityByEnthalpy ddhp
          "Density derivative by specific enthalpy";
        algorithm
          ddhp := if aux.region == 3 then -aux.rho*aux.rho*aux.pt/(aux.rho*aux.rho*
            aux.pd*aux.cv + aux.T*aux.pt*aux.pt) else if aux.region == 4 then -aux.rho
            *aux.rho/(aux.dpT*aux.T) else -aux.rho*aux.rho*aux.vt/(aux.cp);
          annotation (Inline=false, LateInline=true);
        end ddhp_props;

        function ddhp "Density derivative by specific enthalpy"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEnthalpy h "Specific enthalpy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.DerDensityByEnthalpy ddhp
          "Density derivative by specific enthalpy";
        algorithm
          ddhp := ddhp_props(
              p,
              h,
              waterBaseProp_ph(
                p,
                h,
                phase,
                region));
          annotation (Inline=true);
        end ddhp;

        function waterBaseProp_pT
        "Intermediate property record for water (p and T preferred states)"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, do region computation, otherwise assume the region is this input";
          output Common.IF97BaseTwoPhase aux "Auxiliary record";
      protected
          Common.GibbsDerivs g
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Common.HelmholtzDerivs f
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Integer error "Error flag for inverse iterations";
        algorithm
          aux.phase := 1;
          aux.region := if region == 0 then BaseIF97.Regions.region_pT(p=p, T=T)
             else region;
          aux.R := BaseIF97.data.RH2O;
          aux.p := p;
          aux.T := T;
          if (aux.region == 1) then
            g := BaseIF97.Basic.g1(p, T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := p/(aux.R*T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 0.0;
            aux.dpT := -aux.vt/aux.vp;
          elseif (aux.region == 2) then
            g := BaseIF97.Basic.g2(p, T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := p/(aux.R*T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 1.0;
            aux.dpT := -aux.vt/aux.vp;
          elseif (aux.region == 3) then
            (aux.rho,error) := BaseIF97.Inverses.dofpt3(
                p=p,
                T=T,
                delp=1.0e-7);
            f := BaseIF97.Basic.f3(aux.rho, T);
            aux.h := aux.R*T*(f.tau*f.ftau + f.delta*f.fdelta);
            aux.s := aux.R*(f.tau*f.ftau - f.f);
            aux.pd := aux.R*T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
            aux.pt := aux.R*aux.rho*f.delta*(f.fdelta - f.tau*f.fdeltatau);
            aux.cv := aux.R*(-f.tau*f.tau*f.ftautau);
            aux.x := 0.0;
            aux.dpT := aux.pt;
            /*safety against div-by-0 in initialization*/
          elseif (aux.region == 5) then
            g := BaseIF97.Basic.g5(p, T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := p/(aux.R*T*g.pi*g.gpi);
            aux.vt := aux.R/p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*T/(p*p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 1.0;
            aux.dpT := -aux.vt/aux.vp;
          else
            assert(false, "Error in region computation of IF97 steam tables" +
              "(p = " + String(p) + ", T = " + String(T) + ")");
          end if;
        end waterBaseProp_pT;

        function rho_props_pT "Density as function or pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.Density rho "Density";
        algorithm
          rho := aux.rho;
          annotation (
            derivative(noDerivative=aux) = rho_pT_der,
            Inline=false,
            LateInline=true);
        end rho_props_pT;

        function rho_pT "Density as function or pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Density rho "Density";
        algorithm
          rho := rho_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end rho_pT;

        function h_props_pT
        "Specific enthalpy as function or pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificEnthalpy h "Specific enthalpy";
        algorithm
          h := aux.h;
          annotation (
            derivative(noDerivative=aux) = h_pT_der,
            Inline=false,
            LateInline=true);
        end h_props_pT;

        function h_pT
        "Specific enthalpy as function or pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificEnthalpy h "Specific enthalpy";
        algorithm
          h := h_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end h_pT;

        function h_pT_der "Derivative function of h_pT"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real p_der "Derivative of pressure";
          input Real T_der "Derivative of temperature";
          output Real h_der "Derivative of specific enthalpy";
        algorithm
          if (aux.region == 3) then
            h_der := ((-aux.rho*aux.pd + T*aux.pt)/(aux.rho*aux.rho*aux.pd))*p_der +
              ((aux.rho*aux.rho*aux.pd*aux.cv + aux.T*aux.pt*aux.pt)/(aux.rho*aux.rho
              *aux.pd))*T_der;
          else
            //regions 1,2 or 5
            h_der := (1/aux.rho - aux.T*aux.vt)*p_der + aux.cp*T_der;
          end if;
        end h_pT_der;

        function rho_pT_der "Derivative function of rho_pT"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real p_der "Derivative of pressure";
          input Real T_der "Derivative of temperature";
          output Real rho_der "Derivative of density";
        algorithm
          if (aux.region == 3) then
            rho_der := (1/aux.pd)*p_der - (aux.pt/aux.pd)*T_der;
          else
            //regions 1,2 or 5
            rho_der := (-aux.rho*aux.rho*aux.vp)*p_der + (-aux.rho*aux.rho*aux.vt)*
              T_der;
          end if;
        end rho_pT_der;

        function s_props_pT
        "Specific entropy as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificEntropy s "Specific entropy";
        algorithm
          s := aux.s;
          annotation (Inline=false, LateInline=true);
        end s_props_pT;

        function s_pT "Temperature as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificEntropy s "Specific entropy";
        algorithm
          s := s_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end s_pT;

        function cv_props_pT
        "Specific heat capacity at constant volume as function of pressure and temperature"

          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificHeatCapacity cv "Specific heat capacity";
        algorithm
          cv := aux.cv;
          annotation (Inline=false, LateInline=true);
        end cv_props_pT;

        function cv_pT
        "Specific heat capacity at constant volume as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificHeatCapacity cv "Specific heat capacity";
        algorithm
          cv := cv_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end cv_pT;

        function cp_props_pT
        "Specific heat capacity at constant pressure as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificHeatCapacity cp "Specific heat capacity";
        algorithm
          cp := if aux.region == 3 then (aux.rho*aux.rho*aux.pd*aux.cv + aux.T*aux.pt
            *aux.pt)/(aux.rho*aux.rho*aux.pd) else aux.cp;
          annotation (Inline=false, LateInline=true);
        end cp_props_pT;

        function cp_pT
        "Specific heat capacity at constant pressure as function of pressure and temperature"

          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificHeatCapacity cp "Specific heat capacity";
        algorithm
          cp := cp_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end cp_pT;

        function beta_props_pT
        "Isobaric expansion coefficient as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.RelativePressureCoefficient beta
          "Isobaric expansion coefficient";
        algorithm
          beta := if aux.region == 3 then aux.pt/(aux.rho*aux.pd) else aux.vt*aux.rho;
          annotation (Inline=false, LateInline=true);
        end beta_props_pT;

        function beta_pT
        "Isobaric expansion coefficient as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.RelativePressureCoefficient beta
          "Isobaric expansion coefficient";
        algorithm
          beta := beta_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end beta_pT;

        function kappa_props_pT
        "Isothermal compressibility factor as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.IsothermalCompressibility kappa
          "Isothermal compressibility factor";
        algorithm
          kappa := if aux.region == 3 then 1/(aux.rho*aux.pd) else -aux.vp*aux.rho;
          annotation (Inline=false, LateInline=true);
        end kappa_props_pT;

        function kappa_pT
        "Isothermal compressibility factor as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.IsothermalCompressibility kappa
          "Isothermal compressibility factor";
        algorithm
          kappa := kappa_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end kappa_pT;

        function velocityOfSound_props_pT
        "Speed of sound as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.Velocity v_sound "Speed of sound";
        algorithm
          // dp/drho at constant s
          v_sound := if aux.region == 3 then sqrt(max(0, (aux.pd*aux.rho*aux.rho*aux.cv
             + aux.pt*aux.pt*aux.T)/(aux.rho*aux.rho*aux.cv))) else sqrt(max(0, -aux.cp
            /(aux.rho*aux.rho*(aux.vp*aux.cp + aux.vt*aux.vt*aux.T))));
          annotation (Inline=false, LateInline=true);
        end velocityOfSound_props_pT;

        function velocityOfSound_pT
        "Speed of sound as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Velocity v_sound "Speed of sound";
        algorithm
          v_sound := velocityOfSound_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=true);
        end velocityOfSound_pT;

        function isentropicExponent_props_pT
        "Isentropic exponent as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output Real gamma "Isentropic exponent";
        algorithm
          gamma := if aux.region == 3 then 1/(aux.rho*p)*((aux.pd*aux.cv*aux.rho*aux.rho
             + aux.pt*aux.pt*aux.T)/(aux.cv)) else -1/(aux.rho*aux.p)*aux.cp/(aux.vp*
            aux.cp + aux.vt*aux.vt*aux.T);
          annotation (Inline=false, LateInline=true);
        end isentropicExponent_props_pT;

        function isentropicExponent_pT
        "Isentropic exponent as function of pressure and temperature"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output Real gamma "Isentropic exponent";
        algorithm
          gamma := isentropicExponent_props_pT(
              p,
              T,
              waterBaseProp_pT(
                p,
                T,
                region));
          annotation (Inline=false, LateInline=true);
        end isentropicExponent_pT;

        function waterBaseProp_dT
        "Intermediate property record for water (d and T preferred states)"
          extends Modelica.Icons.Function;
          input SI.Density rho "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "Phase: 2 for two-phase, 1 for one phase, 0 if unknown";
          input Integer region=0
          "If 0, do region computation, otherwise assume the region is this input";
          output Common.IF97BaseTwoPhase aux "Auxiliary record";
      protected
          SI.SpecificEnthalpy h_liq "Liquid specific enthalpy";
          SI.Density d_liq "Liquid density";
          SI.SpecificEnthalpy h_vap "Vapour specific enthalpy";
          SI.Density d_vap "Vapour density";
          Common.GibbsDerivs g
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Common.HelmholtzDerivs f
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Modelica.Media.Common.PhaseBoundaryProperties liq
          "Phase boundary property record";
          Modelica.Media.Common.PhaseBoundaryProperties vap
          "Phase boundary property record";
          Modelica.Media.Common.GibbsDerivs gl
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Modelica.Media.Common.GibbsDerivs gv
          "Dimensionless Gibbs function and derivatives w.r.t. pi and tau";
          Modelica.Media.Common.HelmholtzDerivs fl
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Modelica.Media.Common.HelmholtzDerivs fv
          "Dimensionless Helmholtz function and derivatives w.r.t. delta and tau";
          Integer error "Error flag for inverse iterations";
        algorithm
          aux.region := if region == 0 then (if phase == 2 then 4 else
            BaseIF97.Regions.region_dT(
              d=rho,
              T=T,
              phase=phase)) else region;
          aux.phase := if aux.region == 4 then 2 else 1;
          aux.R := BaseIF97.data.RH2O;
          aux.rho := rho;
          aux.T := T;
          if (aux.region == 1) then
            (aux.p,error) := BaseIF97.Inverses.pofdt125(
                d=rho,
                T=T,
                reldd=1.0e-8,
                region=1);
            g := BaseIF97.Basic.g1(aux.p, T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := aux.p/(aux.R*T*g.pi*g.gpi);
            aux.vt := aux.R/aux.p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*T/(aux.p*aux.p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 0.0;
          elseif (aux.region == 2) then
            (aux.p,error) := BaseIF97.Inverses.pofdt125(
                d=rho,
                T=T,
                reldd=1.0e-8,
                region=2);
            g := BaseIF97.Basic.g2(aux.p, T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := aux.p/(aux.R*T*g.pi*g.gpi);
            aux.vt := aux.R/aux.p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*T/(aux.p*aux.p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
            aux.x := 1.0;
          elseif (aux.region == 3) then
            f := BaseIF97.Basic.f3(rho, T);
            aux.p := aux.R*rho*T*f.delta*f.fdelta;
            aux.h := aux.R*T*(f.tau*f.ftau + f.delta*f.fdelta);
            aux.s := aux.R*(f.tau*f.ftau - f.f);
            aux.pd := aux.R*T*f.delta*(2.0*f.fdelta + f.delta*f.fdeltadelta);
            aux.pt := aux.R*rho*f.delta*(f.fdelta - f.tau*f.fdeltatau);
            aux.cp := (aux.rho*aux.rho*aux.pd*aux.cv + aux.T*aux.pt*aux.pt)/(aux.rho*
              aux.rho*aux.pd);
            aux.cv := aux.R*(-f.tau*f.tau*f.ftautau);
            aux.x := 0.0;
          elseif (aux.region == 4) then
            aux.p := BaseIF97.Basic.psat(T);
            d_liq := rhol_T(T);
            d_vap := rhov_T(T);
            h_liq := hl_p(aux.p);
            h_vap := hv_p(aux.p);
            aux.x := if (d_vap <> d_liq) then (1/rho - 1/d_liq)/(1/d_vap - 1/d_liq)
               else 1.0;
            aux.h := h_liq + aux.x*(h_vap - h_liq);
            if T < BaseIF97.data.TLIMIT1 then
              gl := BaseIF97.Basic.g1(aux.p, T);
              gv := BaseIF97.Basic.g2(aux.p, T);
              liq := Common.gibbsToBoundaryProps(gl);
              vap := Common.gibbsToBoundaryProps(gv);
            else
              fl := BaseIF97.Basic.f3(d_liq, T);
              fv := BaseIF97.Basic.f3(d_vap, T);
              liq := Common.helmholtzToBoundaryProps(fl);
              vap := Common.helmholtzToBoundaryProps(fv);
            end if;
            aux.dpT := if (liq.d <> vap.d) then (vap.s - liq.s)*liq.d*vap.d/(liq.d -
              vap.d) else BaseIF97.Basic.dptofT(aux.T);
            aux.s := liq.s + aux.x*(vap.s - liq.s);
            aux.cv := Common.cv2Phase(
                liq,
                vap,
                aux.x,
                aux.T,
                aux.p);
            aux.cp := liq.cp + aux.x*(vap.cp - liq.cp);
            aux.pt := liq.pt + aux.x*(vap.pt - liq.pt);
            aux.pd := liq.pd + aux.x*(vap.pd - liq.pd);
          elseif (aux.region == 5) then
            (aux.p,error) := BaseIF97.Inverses.pofdt125(
                d=rho,
                T=T,
                reldd=1.0e-8,
                region=5);
            g := BaseIF97.Basic.g2(aux.p, T);
            aux.h := aux.R*aux.T*g.tau*g.gtau;
            aux.s := aux.R*(g.tau*g.gtau - g.g);
            aux.rho := aux.p/(aux.R*T*g.pi*g.gpi);
            aux.vt := aux.R/aux.p*(g.pi*g.gpi - g.tau*g.pi*g.gtaupi);
            aux.vp := aux.R*T/(aux.p*aux.p)*g.pi*g.pi*g.gpipi;
            aux.cp := -aux.R*g.tau*g.tau*g.gtautau;
            aux.cv := aux.R*(-g.tau*g.tau*g.gtautau + ((g.gpi - g.tau*g.gtaupi)*(g.gpi
               - g.tau*g.gtaupi)/g.gpipi));
          else
            assert(false, "Error in region computation of IF97 steam tables" +
              "(rho = " + String(rho) + ", T = " + String(T) + ")");
          end if;
        end waterBaseProp_dT;

        function h_props_dT
        "Specific enthalpy as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificEnthalpy h "Specific enthalpy";
        algorithm
          h := aux.h;
          annotation (
            derivative(noDerivative=aux) = h_dT_der,
            Inline=false,
            LateInline=true);
        end h_props_dT;

        function h_dT
        "Specific enthalpy as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificEnthalpy h "Specific enthalpy";
        algorithm
          h := h_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end h_dT;

        function h_dT_der "Derivative function of h_dT"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real d_der "Derivative of density";
          input Real T_der "Derivative of temperature";
          output Real h_der "Derivative of specific enthalpy";
        algorithm
          if (aux.region == 3) then
            h_der := ((-d*aux.pd + T*aux.pt)/(d*d))*d_der + ((aux.cv*d + aux.pt)/d)*
              T_der;
          elseif (aux.region == 4) then
            h_der := T*aux.dpT/(d*d)*d_der + ((aux.cv*d + aux.dpT)/d)*T_der;
          else
            //regions 1,2 or 5
            h_der := (-(-1/d + T*aux.vt)/(d*d*aux.vp))*d_der + ((aux.vp*aux.cp - aux.vt
              /d + T*aux.vt*aux.vt)/aux.vp)*T_der;
          end if;
        end h_dT_der;

        function p_props_dT "Pressure as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.Pressure p "Pressure";
        algorithm
          p := aux.p;
          annotation (
            derivative(noDerivative=aux) = p_dT_der,
            Inline=false,
            LateInline=true);
        end p_props_dT;

        function p_dT "Pressure as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Pressure p "Pressure";
        algorithm
          p := p_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end p_dT;

        function p_dT_der "Derivative function of p_dT"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real d_der "Derivative of density";
          input Real T_der "Derivative of temperature";
          output Real p_der "Derivative of pressure";
        algorithm
          if (aux.region == 3) then
            p_der := aux.pd*d_der + aux.pt*T_der;
          elseif (aux.region == 4) then
            p_der := aux.dpT*T_der;
            /*density derivative is 0.0*/
          else
            //regions 1,2 or 5
            p_der := (-1/(d*d*aux.vp))*d_der + (-aux.vt/aux.vp)*T_der;
          end if;
        end p_dT_der;

        function s_props_dT
        "Specific entropy as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificEntropy s "Specific entropy";
        algorithm
          s := aux.s;
          annotation (Inline=false, LateInline=true);
        end s_props_dT;

        function s_dT "Temperature as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificEntropy s "Specific entropy";
        algorithm
          s := s_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end s_dT;

        function cv_props_dT
        "Specific heat capacity at constant volume as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificHeatCapacity cv "Specific heat capacity";
        algorithm
          cv := aux.cv;
          annotation (Inline=false, LateInline=true);
        end cv_props_dT;

        function cv_dT
        "Specific heat capacity at constant volume as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificHeatCapacity cv "Specific heat capacity";
        algorithm
          cv := cv_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end cv_dT;

        function cp_props_dT
        "Specific heat capacity at constant pressure as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificHeatCapacity cp "Specific heat capacity";
        algorithm
          cp := aux.cp;
          annotation (Inline=false, LateInline=true);
        end cp_props_dT;

        function cp_dT
        "Specific heat capacity at constant pressure as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificHeatCapacity cp "Specific heat capacity";
        algorithm
          cp := cp_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end cp_dT;

        function beta_props_dT
        "Isobaric expansion coefficient as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.RelativePressureCoefficient beta
          "Isobaric expansion coefficient";
        algorithm
          beta := if aux.region == 3 or aux.region == 4 then aux.pt/(aux.rho*aux.pd)
             else aux.vt*aux.rho;
          annotation (Inline=false, LateInline=true);
        end beta_props_dT;

        function beta_dT
        "Isobaric expansion coefficient as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.RelativePressureCoefficient beta
          "Isobaric expansion coefficient";
        algorithm
          beta := beta_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end beta_dT;

        function kappa_props_dT
        "Isothermal compressibility factor as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.IsothermalCompressibility kappa
          "Isothermal compressibility factor";
        algorithm
          kappa := if aux.region == 3 or aux.region == 4 then 1/(aux.rho*aux.pd)
             else -aux.vp*aux.rho;
          annotation (Inline=false, LateInline=true);
        end kappa_props_dT;

        function kappa_dT
        "Isothermal compressibility factor as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.IsothermalCompressibility kappa
          "Isothermal compressibility factor";
        algorithm
          kappa := kappa_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end kappa_dT;

        function velocityOfSound_props_dT
        "Speed of sound as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.Velocity v_sound "Speed of sound";
        algorithm
          // dp/drho at constant s
          v_sound := if aux.region == 3 then sqrt(max(0, ((aux.pd*aux.rho*aux.rho*aux.cv
             + aux.pt*aux.pt*aux.T)/(aux.rho*aux.rho*aux.cv)))) else if aux.region
             == 4 then sqrt(max(0, 1/((aux.rho*(aux.rho*aux.cv/aux.dpT + 1.0)/(aux.dpT
            *aux.T)) - 1/aux.rho*aux.rho*aux.rho/(aux.dpT*aux.T)))) else sqrt(max(0,
            -aux.cp/(aux.rho*aux.rho*(aux.vp*aux.cp + aux.vt*aux.vt*aux.T))));
          annotation (Inline=false, LateInline=true);
        end velocityOfSound_props_dT;

        function velocityOfSound_dT
        "Speed of sound as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.Velocity v_sound "Speed of sound";
        algorithm
          v_sound := velocityOfSound_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=true);
        end velocityOfSound_dT;

        function isentropicExponent_props_dT
        "Isentropic exponent as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output Real gamma "Isentropic exponent";
        algorithm
          gamma := if aux.region == 3 then 1/(aux.rho*aux.p)*((aux.pd*aux.cv*aux.rho*
            aux.rho + aux.pt*aux.pt*aux.T)/(aux.cv)) else if aux.region == 4 then 1/(
            aux.rho*aux.p)*aux.dpT*aux.dpT*aux.T/aux.cv else -1/(aux.rho*aux.p)*aux.cp
            /(aux.vp*aux.cp + aux.vt*aux.vt*aux.T);
          annotation (Inline=false, LateInline=true);
        end isentropicExponent_props_dT;

        function isentropicExponent_dT
        "Isentropic exponent as function of density and temperature"
          extends Modelica.Icons.Function;
          input SI.Density d "Density";
          input SI.Temperature T "Temperature";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output Real gamma "Isentropic exponent";
        algorithm
          gamma := isentropicExponent_props_dT(
              d,
              T,
              waterBaseProp_dT(
                d,
                T,
                phase,
                region));
          annotation (Inline=false, LateInline=true);
        end isentropicExponent_dT;

        function hl_p = BaseIF97.Regions.hl_p
        "Compute the saturated liquid specific h(p)";

        function hv_p = BaseIF97.Regions.hv_p
        "Compute the saturated vapour specific h(p)";

        function rhol_T = BaseIF97.Regions.rhol_T
        "Compute the saturated liquid d(T)";

        function rhov_T = BaseIF97.Regions.rhov_T
        "Compute the saturated vapour d(T)";

        function dynamicViscosity = BaseIF97.Transport.visc_dTp
        "Compute eta(d,T) in the one-phase region";

        function thermalConductivity = BaseIF97.Transport.cond_dTp
        "Compute lambda(d,T,p) in the one-phase region";

        function surfaceTension = BaseIF97.Transport.surfaceTension
        "Compute sigma(T) at saturation T";

        function isentropicEnthalpy
        "Isentropic specific enthalpy from p,s (preferably use dynamicIsentropicEnthalpy in dynamic simulation!)"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Integer phase=0
          "2 for two-phase, 1 for one-phase, 0 if not known";
          input Integer region=0
          "If 0, region is unknown, otherwise known and this input";
          output SI.SpecificEnthalpy h "Specific enthalpy";
        algorithm
          h := isentropicEnthalpy_props(
              p,
              s,
              waterBaseProp_ps(
                p,
                s,
                phase,
                region));
          annotation (Inline=true);
        end isentropicEnthalpy;

        function isentropicEnthalpy_props
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          output SI.SpecificEnthalpy h "Isentropic enthalpy";
        algorithm
          h := aux.h;
          annotation (
            derivative(noDerivative=aux) = isentropicEnthalpy_der,
            Inline=false,
            LateInline=true);
        end isentropicEnthalpy_props;

        function isentropicEnthalpy_der
        "Derivative of isentropic specific enthalpy from p,s"
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.SpecificEntropy s "Specific entropy";
          input Common.IF97BaseTwoPhase aux "Auxiliary record";
          input Real p_der "Pressure derivative";
          input Real s_der "Entropy derivative";
          output Real h_der "Specific enthalpy derivative";
        algorithm
          h_der := 1/aux.rho*p_der + aux.T*s_der;
          annotation (Inline=true);
        end isentropicEnthalpy_der;
        annotation (Documentation(info="<HTML>
      <h4>Package description:</h4>
      <p>This package provides high accuracy physical properties for water according
      to the IAPWS/IF97 standard. It has been part of the ThermoFluid Modelica library and been extended,
      reorganized and documented to become part of the Modelica Standard library.</p>
      <p>An important feature that distinguishes this implementation of the IF97 steam property standard
      is that this implementation has been explicitly designed to work well in dynamic simulations. Computational
      performance has been of high importance. This means that there often exist several ways to get the same result
      from different functions if one of the functions is called often but can be optimized for that purpose.
      </p>
      <p>
      The original documentation of the IAPWS/IF97 steam properties can freely be distributed with computer
      implementations, so for curious minds the complete standard documentation is provided with the Modelica
      properties library. The following documents are included
      (in directory Modelica/Resources/Documentation/Media/Water/IF97documentation):
      </p>
      <ul>
      <li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a> The standards document for the main part of the IF97.</li>
      <li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/Back3.pdf\">Back3.pdf</a> The backwards equations for region 3.</li>
      <li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/crits.pdf\">crits.pdf</a> The critical point data.</li>
      <li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/meltsub.pdf\">meltsub.pdf</a> The melting- and sublimation line formulation (in IF97_Utilities.BaseIF97.IceBoundaries)</li>
      <li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/surf.pdf\">surf.pdf</a> The surface tension standard definition</li>
      <li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/thcond.pdf\">thcond.pdf</a> The thermal conductivity standard definition</li>
      <li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/visc.pdf\">visc.pdf</a> The viscosity standard definition</li>
      </ul>
      <h4>Package contents
      </h4>
      <ul>
      <li>Package <b>BaseIF97</b> contains the implementation of the IAPWS-IF97 as described in
      <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a>. The explicit backwards equations for region 3 from
      <a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/Back3.pdf\">Back3.pdf</a> are implemented as initial values for an inverse iteration of the exact
      function in IF97 for the input pairs (p,h) and (p,s).
      The low-level functions in BaseIF97 are not needed for standard simulation usage,
      but can be useful for experts and some special purposes.</li>
      <li>Function <b>water_ph</b> returns all properties needed for a dynamic control volume model and properties of general
      interest using pressure p and specific entropy enthalpy h as dynamic states in the record ThermoProperties_ph. </li>
      <li>Function <b>water_ps</b> returns all properties needed for a dynamic control volume model and properties of general
      interest using pressure p and specific entropy s as dynamic states in the record ThermoProperties_ps. </li>
      <li>Function <b>water_dT</b> returns all properties needed for a dynamic control volume model and properties of general
      interest using density d and temperature T as dynamic states in the record ThermoProperties_dT. </li>
      <li>Function <b>water_pT</b> returns all properties needed for a dynamic control volume model and properties of general
      interest using pressure p and temperature T as dynamic states in the record ThermoProperties_pT. Due to the coupling of
      pressure and temperature in the two-phase region, this model can obviously
      only be used for one-phase models or models treating both phases independently.</li>
      <li>Function <b>hl_p</b> computes the liquid specific enthalpy as a function of pressure. For overcritical pressures,
      the critical specific enthalpy is returned</li>
      <li>Function <b>hv_p</b> computes the vapour specific enthalpy as a function of pressure. For overcritical pressures,
      the critical specific enthalpy is returned</li>
      <li>Function <b>sl_p</b> computes the liquid specific entropy as a function of pressure. For overcritical pressures,
      the critical  specific entropy is returned</li>
      <li>Function <b>sv_p</b> computes the vapour  specific entropy as a function of pressure. For overcritical pressures,
      the critical  specific entropy is returned</li>
      <li>Function <b>rhol_T</b> computes the liquid density as a function of temperature. For overcritical temperatures,
      the critical density is returned</li>
      <li>Function <b>rhol_T</b> computes the vapour density as a function of temperature. For overcritical temperatures,
      the critical density is returned</li>
      <li>Function <b>dynamicViscosity</b> computes the dynamic viscosity as a function of density and temperature.</li>
      <li>Function <b>thermalConductivity</b> computes the thermal conductivity as a function of density, temperature and pressure.
      <b>Important note</b>: Obviously only two of the three
      inputs are really needed, but using three inputs speeds up the computation and the three variables
      are known in most models anyways. The inputs d,T and p have to be consistent.</li>
      <li>Function <b>surfaceTension</b> computes the surface tension between vapour
          and liquid water as a function of temperature.</li>
      <li>Function <b>isentropicEnthalpy</b> computes the specific enthalpy h(p,s,phase) in all regions.
          The phase input is needed due to discontinuous derivatives at the phase boundary.</li>
      <li>Function <b>dynamicIsentropicEnthalpy</b> computes the specific enthalpy h(p,s,,dguess,Tguess,phase) in all regions.
          The phase input is needed due to discontinuous derivatives at the phase boundary. Tguess and dguess are initial guess
          values for the density and temperature consistent with p and s. This function should be preferred in
          dynamic simulations where good guesses are often available.</li>
      </ul>
      <h4>Version Info and Revision history
      </h4>
      <ul>
      <li>First implemented: <i>July, 2000</i>
      by Hubertus Tummescheit for the ThermoFluid Library with help from Jonas Eborn and Falko Jens Wagner
      </li>
      <li>Code reorganization, enhanced documentation, additional functions:   <i>December, 2002</i>
      by <a href=\"mailto:Hubertus.Tummescheit@modelon.se\">Hubertus Tummescheit</a> and moved to Modelica
      properties library.</li>
      </ul>
      <address>Author: Hubertus Tummescheit, <br>
      Modelon AB<br>
      Ideon Science Park<br>
      SE-22370 Lund, Sweden<br>
      email: hubertus@modelon.se
      </address>
      </HTML>",       revisions="<h4>Intermediate release notes during development</h4>
<p>Currently the Events/noEvents switch is only implemented for p-h states. Only after testing that implementation, it will be extended to dT.</p>"));
      end IF97_Utilities;
    annotation (Documentation(info="<html>
<p>This package contains different medium models for water:</p>
<ul>
<li><b>ConstantPropertyLiquidWater</b><br>
    Simple liquid water medium (incompressible, constant data).</li>
<li><b>IdealSteam</b><br>
    Steam water medium as ideal gas from Media.IdealGases.SingleGases.H2O</li>
<li><b>WaterIF97 derived models</b><br>
    High precision water model according to the IAPWS/IF97 standard
    (liquid, steam, two phase region). Models with different independent
    variables are provided as well as models valid only
    for particular regions. The <b>WaterIF97_ph</b> model is valid
    in all regions and is the recommended one to use.</li>
</ul>
<h4>Overview of WaterIF97 derived water models</h4>
<p>
The WaterIF97 models calculate medium properties
for water in the <b>liquid</b>, <b>gas</b> and <b>two phase</b> regions
according to the IAPWS/IF97 standard, i.e., the accepted industrial standard
and best compromise between accuracy and computation time.
It has been part of the ThermoFluid Modelica library and been extended,
reorganized and documented to become part of the Modelica Standard library.</p>
<p>An important feature that distinguishes this implementation of the IF97 steam property standard
is that this implementation has been explicitly designed to work well in dynamic simulations. Computational
performance has been of high importance. This means that there often exist several ways to get the same result
from different functions if one of the functions is called often but can be optimized for that purpose.
</p>
<p>Three variable pairs can be the independent variables of the model:
</p>
<ol>
<li>Pressure <b>p</b> and specific enthalpy <b>h</b> are
    the most natural choice for general applications.
    This is the recommended choice for most general purpose
    applications, in particular for power plants.</li>
<li>Pressure <b>p</b> and temperature <b>T</b> are the most natural
    choice for applications where water is always in the same phase,
    both for liquid water and steam.</li>
<li>Density <b>d</b> and temperature <b>T</b> are explicit
    variables of the Helmholtz function in the near-critical
    region and can be the best choice for applications with
    super-critical or near-critical states.</li>
</ol>
<p>
The following quantities are always computed in Medium.BaseProperties:
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>Variable</b></td>
      <td valign=\"top\"><b>Unit</b></td>
      <td valign=\"top\"><b>Description</b></td></tr>
  <tr><td valign=\"top\">T</td>
      <td valign=\"top\">K</td>
      <td valign=\"top\">temperature</td></tr>
  <tr><td valign=\"top\">u</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">specific internal energy</td></tr>
  <tr><td valign=\"top\">d</td>
      <td valign=\"top\">kg/m^3</td>
      <td valign=\"top\">density</td></tr>
  <tr><td valign=\"top\">p</td>
      <td valign=\"top\">Pa</td>
      <td valign=\"top\">pressure</td></tr>
  <tr><td valign=\"top\">h</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">specific enthalpy</td></tr>
</table>
<p>
In some cases additional medium properties are needed.
A component that needs these optional properties has to call
one of the following functions:
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>Function call</b></td>
      <td valign=\"top\"><b>Unit</b></td>
      <td valign=\"top\"><b>Description</b></td></tr>
  <tr><td valign=\"top\">Medium.dynamicViscosity(medium.state)</td>
      <td valign=\"top\">Pa.s</td>
      <td valign=\"top\">dynamic viscosity</td></tr>
  <tr><td valign=\"top\">Medium.thermalConductivity(medium.state)</td>
      <td valign=\"top\">W/(m.K)</td>
      <td valign=\"top\">thermal conductivity</td></tr>
  <tr><td valign=\"top\">Medium.prandtlNumber(medium.state)</td>
      <td valign=\"top\">1</td>
      <td valign=\"top\">Prandtl number</td></tr>
  <tr><td valign=\"top\">Medium.specificEntropy(medium.state)</td>
      <td valign=\"top\">J/(kg.K)</td>
      <td valign=\"top\">specific entropy</td></tr>
  <tr><td valign=\"top\">Medium.heatCapacity_cp(medium.state)</td>
      <td valign=\"top\">J/(kg.K)</td>
      <td valign=\"top\">specific heat capacity at constant pressure</td></tr>
  <tr><td valign=\"top\">Medium.heatCapacity_cv(medium.state)</td>
      <td valign=\"top\">J/(kg.K)</td>
      <td valign=\"top\">specific heat capacity at constant density</td></tr>
  <tr><td valign=\"top\">Medium.isentropicExponent(medium.state)</td>
      <td valign=\"top\">1</td>
      <td valign=\"top\">isentropic exponent</td></tr>
  <tr><td valign=\"top\">Medium.isentropicEnthalpy(pressure, medium.state)</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">isentropic enthalpy</td></tr>
  <tr><td valign=\"top\">Medium.velocityOfSound(medium.state)</td>
      <td valign=\"top\">m/s</td>
      <td valign=\"top\">velocity of sound</td></tr>
  <tr><td valign=\"top\">Medium.isobaricExpansionCoefficient(medium.state)</td>
      <td valign=\"top\">1/K</td>
      <td valign=\"top\">isobaric expansion coefficient</td></tr>
  <tr><td valign=\"top\">Medium.isothermalCompressibility(medium.state)</td>
      <td valign=\"top\">1/Pa</td>
      <td valign=\"top\">isothermal compressibility</td></tr>
  <tr><td valign=\"top\">Medium.density_derp_h(medium.state)</td>
      <td valign=\"top\">kg/(m3.Pa)</td>
      <td valign=\"top\">derivative of density by pressure at constant enthalpy</td></tr>
  <tr><td valign=\"top\">Medium.density_derh_p(medium.state)</td>
      <td valign=\"top\">kg2/(m3.J)</td>
      <td valign=\"top\">derivative of density by enthalpy at constant pressure</td></tr>
  <tr><td valign=\"top\">Medium.density_derp_T(medium.state)</td>
      <td valign=\"top\">kg/(m3.Pa)</td>
      <td valign=\"top\">derivative of density by pressure at constant temperature</td></tr>
  <tr><td valign=\"top\">Medium.density_derT_p(medium.state)</td>
      <td valign=\"top\">kg/(m3.K)</td>
      <td valign=\"top\">derivative of density by temperature at constant pressure</td></tr>
  <tr><td valign=\"top\">Medium.density_derX(medium.state)</td>
      <td valign=\"top\">kg/m3</td>
      <td valign=\"top\">derivative of density by mass fraction</td></tr>
  <tr><td valign=\"top\">Medium.molarMass(medium.state)</td>
      <td valign=\"top\">kg/mol</td>
      <td valign=\"top\">molar mass</td></tr>
</table>
<p>More details are given in
<a href=\"modelica://Modelica.Media.UsersGuide.MediumUsage.OptionalProperties\">
Modelica.Media.UsersGuide.MediumUsage.OptionalProperties</a>.

Many additional optional functions are defined to compute properties of
saturated media, either liquid (bubble point) or vapour (dew point).
The argument to such functions is a SaturationProperties record, which can be
set starting from either the saturation pressure or the saturation temperature.
With reference to a model defining a pressure p, a temperature T, and a
SaturationProperties record sat, the following functions are provided:
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>Function call</b></td>
      <td valign=\"top\"><b>Unit</b></td>
      <td valign=\"top\"><b>Description</b></td></tr>
  <tr><td valign=\"top\">Medium.saturationPressure(T)</td>
      <td valign=\"top\">Pa</td>
      <td valign=\"top\">Saturation pressure at temperature T</td></tr>
  <tr><td valign=\"top\">Medium.saturationTemperature(p)</td>
      <td valign=\"top\">K</td>
      <td valign=\"top\">Saturation temperature at pressure p</td></tr>
  <tr><td valign=\"top\">Medium.saturationTemperature_derp(p)</td>
      <td valign=\"top\">K/Pa</td>
      <td valign=\"top\">Derivative of saturation temperature with respect to pressure</td></tr>
  <tr><td valign=\"top\">Medium.bubbleEnthalpy(sat)</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">Specific enthalpy at bubble point</td></tr>
  <tr><td valign=\"top\">Medium.dewEnthalpy(sat)</td>
      <td valign=\"top\">J/kg</td>
      <td valign=\"top\">Specific enthalpy at dew point</td></tr>
  <tr><td valign=\"top\">Medium.bubbleEntropy(sat)</td>
      <td valign=\"top\">J/(kg.K)</td>
      <td valign=\"top\">Specific entropy at bubble point</td></tr>
  <tr><td valign=\"top\">Medium.dewEntropy(sat)</td>
      <td valign=\"top\">J/(kg.K)</td>
      <td valign=\"top\">Specific entropy at dew point</td></tr>
  <tr><td valign=\"top\">Medium.bubbleDensity(sat)</td>
      <td valign=\"top\">kg/m3</td>
      <td valign=\"top\">Density at bubble point</td></tr>
  <tr><td valign=\"top\">Medium.dewDensity(sat)</td>
      <td valign=\"top\">kg/m3</td>
      <td valign=\"top\">Density at dew point</td></tr>
  <tr><td valign=\"top\">Medium.dBubbleDensity_dPressure(sat)</td>
      <td valign=\"top\">kg/(m3.Pa)</td>
      <td valign=\"top\">Derivative of density at bubble point with respect to pressure</td></tr>
  <tr><td valign=\"top\">Medium.dDewDensity_dPressure(sat)</td>
      <td valign=\"top\">kg/(m3.Pa)</td>
      <td valign=\"top\">Derivative of density at dew point with respect to pressure</td></tr>
  <tr><td valign=\"top\">Medium.dBubbleEnthalpy_dPressure(sat)</td>
      <td valign=\"top\">J/(kg.Pa)</td>
      <td valign=\"top\">Derivative of specific enthalpy at bubble point with respect to pressure</td></tr>
  <tr><td valign=\"top\">Medium.dDewEnthalpy_dPressure(sat)</td>
      <td valign=\"top\">J/(kg.Pa)</td>
      <td valign=\"top\">Derivative of specific enthalpy at dew point with respect to pressure</td></tr>
  <tr><td valign=\"top\">Medium.surfaceTension(sat)</td>
      <td valign=\"top\">N/m</td>
      <td valign=\"top\">Surface tension between liquid and vapour phase</td></tr>
</table>
<p>Details on usage and some examples are given in:
<a href=\"modelica://Modelica.Media.UsersGuide.MediumUsage.TwoPhase\">
Modelica.Media.UsersGuide.MediumUsage.TwoPhase</a>.
</p>
<p>Many further properties can be computed. Using the well-known Bridgman's Tables,
all first partial derivatives of the standard thermodynamic variables can be computed easily.
</p>
<p>
The documentation of the IAPWS/IF97 steam properties can be freely
distributed with computer implementations and are included here
(in directory Modelica/Resources/Documentation/Media/Water/IF97documentation):
</p>
<ul>
<li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/IF97.pdf\">IF97.pdf</a> The standards document for the main part of the IF97.</li>
<li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/Back3.pdf\">Back3.pdf</a> The backwards equations for region 3.</li>
<li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/crits.pdf\">crits.pdf</a> The critical point data.</li>
<li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/meltsub.pdf\">meltsub.pdf</a> The melting- and sublimation line formulation (not implemented)</li>
<li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/surf.pdf\">surf.pdf</a> The surface tension standard definition</li>
<li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/thcond.pdf\">thcond.pdf</a> The thermal conductivity standard definition</li>
<li><a href=\"modelica://Modelica/Resources/Documentation/Media/Water/IF97documentation/visc.pdf\">visc.pdf</a> The viscosity standard definition</li>
</ul>
</html>"));
    end Water;
  annotation (preferredView="info",Documentation(info="<HTML>
<p>
This library contains <a href=\"modelica://Modelica.Media.Interfaces\">interface</a>
definitions for media and the following <b>property</b> models for
single and multiple substance fluids with one and multiple phases:
</p>
<ul>
<li> <a href=\"modelica://Modelica.Media.IdealGases\">Ideal gases:</a><br>
     1241 high precision gas models based on the
     NASA Glenn coefficients, plus ideal gas mixture models based
     on the same data.</li>
<li> <a href=\"modelica://Modelica.Media.Water\">Water models:</a><br>
     ConstantPropertyLiquidWater, WaterIF97 (high precision
     water model according to the IAPWS/IF97 standard)</li>
<li> <a href=\"modelica://Modelica.Media.Air\">Air models:</a><br>
     SimpleAir, DryAirNasa, ReferenceAir, MoistAir, ReferenceMoistAir.</li>
<li> <a href=\"modelica://Modelica.Media.Incompressible\">
     Incompressible media:</a><br>
     TableBased incompressible fluid models (properties are defined by tables rho(T),
     HeatCapacity_cp(T), etc.)</li>
<li> <a href=\"modelica://Modelica.Media.CompressibleLiquids\">
     Compressible liquids:</a><br>
     Simple liquid models with linear compressibility</li>
<li> <a href=\"modelica://Modelica.Media.R134a\">Refrigerant Tetrafluoroethane (R134a)</a>.</li>
</ul>
<p>
The following parts are useful, when newly starting with this library:
<ul>
<li> <a href=\"modelica://Modelica.Media.UsersGuide\">Modelica.Media.UsersGuide</a>.</li>
<li> <a href=\"modelica://Modelica.Media.UsersGuide.MediumUsage\">Modelica.Media.UsersGuide.MediumUsage</a>
     describes how to use a medium model in a component model.</li>
<li> <a href=\"modelica://Modelica.Media.UsersGuide.MediumDefinition\">
     Modelica.Media.UsersGuide.MediumDefinition</a>
     describes how a new fluid medium model has to be implemented.</li>
<li> <a href=\"modelica://Modelica.Media.UsersGuide.ReleaseNotes\">Modelica.Media.UsersGuide.ReleaseNotes</a>
     summarizes the changes of the library releases.</li>
<li> <a href=\"modelica://Modelica.Media.Examples\">Modelica.Media.Examples</a>
     contains examples that demonstrate the usage of this library.</li>
</ul>
<p>
Copyright &copy; 1998-2013, Modelica Association.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>
</HTML>",   revisions="<html>
<ul>
<li><i>May 16, 2013</i> by Stefan Wischhusen (XRG Simulation):<br/>
    Added new media models Air.ReferenceMoistAir, Air.ReferenceAir, R134a.</li>
<li><i>May 25, 2011</i> by Francesco Casella:<br/>Added min/max attributes to Water, TableBased, MixtureGasNasa, SimpleAir and MoistAir local types.</li>
<li><i>May 25, 2011</i> by Stefan Wischhusen:<br/>Added individual settings for polynomial fittings of properties.</li>
</ul>
</html>"),
      Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
          graphics={
          Line(
            points = {{-76,-80},{-62,-30},{-32,40},{4,66},{48,66},{73,45},{62,-8},{48,-50},{38,-80}},
            color={64,64,64},
            smooth=Smooth.Bezier),
          Line(
            points={{-40,20},{68,20}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{-40,20},{-44,88},{-44,88}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{68,20},{86,-58}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{-60,-28},{56,-28}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{-60,-28},{-74,84},{-74,84}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{56,-28},{70,-80}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{-76,-80},{38,-80}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{-76,-80},{-94,-16},{-94,-16}},
            color={175,175,175},
            smooth=Smooth.None)}));
  end Media;

  package Math
  "Library of mathematical functions (e.g., sin, cos) and of functions operating on vectors and matrices"
  import SI = Modelica.SIunits;
  extends Modelica.Icons.Package;

  package Icons "Icons for Math"
    extends Modelica.Icons.IconsPackage;

    partial function AxisLeft
    "Basic icon for mathematical function with y-axis on left side"

      annotation (
        Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
                100}}), graphics={
            Rectangle(
              extent={{-100,100},{100,-100}},
              lineColor={0,0,0},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid),
            Line(points={{-80,-80},{-80,68}}, color={192,192,192}),
            Polygon(
              points={{-80,90},{-88,68},{-72,68},{-80,90}},
              lineColor={192,192,192},
              fillColor={192,192,192},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{-150,150},{150,110}},
              textString="%name",
              lineColor={0,0,255})}),
        Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                100,100}}), graphics={Line(points={{-80,80},{-88,80}}, color={95,
              95,95}),Line(points={{-80,-80},{-88,-80}}, color={95,95,95}),Line(
              points={{-80,-90},{-80,84}}, color={95,95,95}),Text(
                  extent={{-75,104},{-55,84}},
                  lineColor={95,95,95},
                  textString="y"),Polygon(
                  points={{-80,98},{-86,82},{-74,82},{-80,98}},
                  lineColor={95,95,95},
                  fillColor={95,95,95},
                  fillPattern=FillPattern.Solid)}),
        Documentation(info="<html>
<p>
Icon for a mathematical function, consisting of an y-axis on the left side.
It is expected, that an x-axis is added and a plot of the function.
</p>
</html>"));
    end AxisLeft;

    partial function AxisCenter
    "Basic icon for mathematical function with y-axis in the center"

      annotation (
        Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
                100}}), graphics={
            Rectangle(
              extent={{-100,100},{100,-100}},
              lineColor={0,0,0},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid),
            Line(points={{0,-80},{0,68}}, color={192,192,192}),
            Polygon(
              points={{0,90},{-8,68},{8,68},{0,90}},
              lineColor={192,192,192},
              fillColor={192,192,192},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{-150,150},{150,110}},
              textString="%name",
              lineColor={0,0,255})}),
        Diagram(graphics={Line(points={{0,80},{-8,80}}, color={95,95,95}),Line(
              points={{0,-80},{-8,-80}}, color={95,95,95}),Line(points={{0,-90},{
              0,84}}, color={95,95,95}),Text(
                  extent={{5,104},{25,84}},
                  lineColor={95,95,95},
                  textString="y"),Polygon(
                  points={{0,98},{-6,82},{6,82},{0,98}},
                  lineColor={95,95,95},
                  fillColor={95,95,95},
                  fillPattern=FillPattern.Solid)}),
        Documentation(info="<html>
<p>
Icon for a mathematical function, consisting of an y-axis in the middle.
It is expected, that an x-axis is added and a plot of the function.
</p>
</html>"));
    end AxisCenter;
  end Icons;

  package Matrices "Library of functions operating on matrices"
    extends Modelica.Icons.Package;

    function solve
    "Solve real system of linear equations A*x=b with a b vector (Gaussian elimination with partial pivoting)"

      extends Modelica.Icons.Function;
      input Real A[:, size(A, 1)] "Matrix A of A*x = b";
      input Real b[size(A, 1)] "Vector b of A*x = b";
      output Real x[size(b, 1)] "Vector x such that A*x = b";

  protected
      Integer info;
    algorithm
      (x,info) := LAPACK.dgesv_vec(A, b);
      assert(info == 0, "Solving a linear system of equations with function
\"Matrices.solve\" is not possible, because the system has either
no or infinitely many solutions (A is singular).");
      annotation (Documentation(info="<HTML>
<h4>Syntax</h4>
<blockquote><pre>
Matrices.<b>solve</b>(A,b);
</pre></blockquote>
<h4>Description</h4>
<p>
This function call returns the
solution <b>x</b> of the linear system of equations
</p>
<blockquote>
<p>
<b>A</b>*<b>x</b> = <b>b</b>
</p>
</blockquote>
<p>
If a unique solution <b>x</b> does not exist (since <b>A</b> is singular),
an assertion is triggered. If this is not desired, use instead
<a href=\"modelica://Modelica.Math.Matrices.leastSquares\">Matrices.leastSquares</a>
and inquire the singularity of the solution with the return argument rank
(a unique solution is computed if rank = size(A,1)).
</p>

<p>
Note, the solution is computed with the LAPACK function \"dgesv\",
i.e., by Gaussian elimination with partial pivoting.
</p>
<h4>Example</h4>
<blockquote><pre>
  Real A[3,3] = [1,2,3;
                 3,4,5;
                 2,1,4];
  Real b[3] = {10,22,12};
  Real x[3];
<b>algorithm</b>
  x := Matrices.solve(A,b);  // x = {3,2,1}
</pre></blockquote>
<h4>See also</h4>
<a href=\"modelica://Modelica.Math.Matrices.LU\">Matrices.LU</a>,
<a href=\"modelica://Modelica.Math.Matrices.LU_solve\">Matrices.LU_solve</a>,
<a href=\"modelica://Modelica.Math.Matrices.leastSquares\">Matrices.leastSquares</a>.
</HTML>"));
    end solve;

    package LAPACK
    "Interface to LAPACK library (should usually not directly be used but only indirectly via Modelica.Math.Matrices)"
      extends Modelica.Icons.Package;

      function dgesv_vec
      "Solve real system of linear equations A*x=b with a b vector"
        extends Modelica.Icons.Function;
        input Real A[:, size(A, 1)];
        input Real b[size(A, 1)];
        output Real x[size(A, 1)]=b;
        output Integer info;
    protected
        Real Awork[size(A, 1), size(A, 1)]=A;
        Integer lda=max(1, size(A, 1));
        Integer ldb=max(1, size(b, 1));
        Integer ipiv[size(A, 1)];

      external"FORTRAN 77" dgesv(
                size(A, 1),
                1,
                Awork,
                lda,
                ipiv,
                x,
                ldb,
                info) annotation (Library="lapack");
        annotation (Documentation(info="
Same as function LAPACK.dgesv, but right hand side is a vector and not a matrix.
For details of the arguments, see documentation of dgesv.
"));
      end dgesv_vec;
      annotation (Documentation(info="<html>
<p>
This package contains external Modelica functions as interface to the
LAPACK library
(<a href=\"http://www.netlib.org/lapack\">http://www.netlib.org/lapack</a>)
that provides FORTRAN subroutines to solve linear algebra
tasks. Usually, these functions are not directly called, but only via
the much more convenient interface of
<a href=\"modelica://Modelica.Math.Matrices\">Modelica.Math.Matrices</a>.
The documentation of the LAPACK functions is a copy of the original
FORTRAN code. The details of LAPACK are described in:
</p>

<dl>
<dt>Anderson E., Bai Z., Bischof C., Blackford S., Demmel J., Dongarra J.,
    Du Croz J., Greenbaum A., Hammarling S., McKenney A., and Sorensen D.:</dt>
<dd> <a href=\"http://www.netlib.org/lapack/lug/lapack_lug.html\">Lapack Users' Guide</a>.
     Third Edition, SIAM, 1999.</dd>
</dl>

<p>
See also <a href=\"http://en.wikipedia.org/wiki/Lapack\">http://en.wikipedia.org/wiki/Lapack</a>.
</p>

<p>
This package contains a direct interface to the LAPACK subroutines
</p>

</html>"));
    end LAPACK;
    annotation (Documentation(info="<HTML>
<h4>Library content</h4>
<p>
This library provides functions operating on matrices. Below, the
functions are ordered according to categories and a typical
call of the respective function is shown.
Most functions are solely an interface to the external
<a href=\"modelica://Modelica.Math.Matrices.LAPACK\">LAPACK</a> library.
</p>

<p>
Note: A' is a short hand notation of transpose(A):
</p>

<p><b>Basic Information</b></p>
<ul>
<li> <a href=\"modelica://Modelica.Math.Matrices.toString\">toString</a>(A)
     - returns the string representation of matrix A.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.isEqual\">isEqual</a>(M1, M2)
     - returns true if matrices M1 and M2 have the same size and the same elements.</li>
</ul>

<p><b>Linear Equations</b></p>
<ul>
<li> <a href=\"modelica://Modelica.Math.Matrices.solve\">solve</a>(A,b)
     - returns solution x of the linear equation A*x=b (where b is a vector,
       and A is a square matrix that must be regular).</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.solve2\">solve2</a>(A,B)
     - returns solution X of the linear equation A*X=B (where B is a matrix,
       and A is a square matrix that must be regular)</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.leastSquares\">leastSquares</a>(A,b)
     - returns solution x of the linear equation A*x=b in a least squares sense
       (where b is a vector and A may be non-square and may be rank deficient)</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.leastSquares2\">leastSquares2</a>(A,B)
     - returns solution X of the linear equation A*X=B in a least squares sense
       (where B is a matrix and A may be non-square and may be rank deficient)</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.equalityLeastSquares\">equalityLeastSquares</a>(A,a,B,b)
     - returns solution x of a linear equality constrained least squares problem:
       min|A*x-a|^2 subject to B*x=b</<li>

<li> (LU,p,info) = <a href=\"modelica://Modelica.Math.Matrices.LU\">LU</a>(A)
     - returns the LU decomposition with row pivoting of a rectangular matrix A.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.LU_solve\">LU_solve</a>(LU,p,b)
     - returns solution x of the linear equation L*U*x[p]=b with a b
       vector and an LU decomposition from \"LU(..)\".</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.LU_solve2\">LU_solve2</a>(LU,p,B)
     - returns solution X of the linear equation L*U*X[p,:]=B with a B
       matrix and an LU decomposition from \"LU(..)\".</li>
</ul>

<p><b>Matrix Factorizations</b></p>
<ul>
<li> (eval,evec) = <a href=\"modelica://Modelica.Math.Matrices.eigenValues\">eigenValues</a>(A)
     - returns eigen values \"eval\" and eigen vectors \"evec\" for a real,
       nonsymmetric matrix A in a Real representation.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.eigenValueMatrix\">eigenValueMatrix</a>(eval)
     - returns real valued block diagonal matrix of the eigenvalues \"eval\" of matrix A.</li>

<li> (sigma,U,VT) = <a href=\"modelica://Modelica.Math.Matrices.singularValues\">singularValues</a>(A)
     - returns singular values \"sigma\" and left and right singular vectors U and VT
       of a rectangular matrix A.</li>

<li> (Q,R,p) = <a href=\"modelica://Modelica.Math.Matrices.QR\">QR</a>(A)
     - returns the QR decomposition with column pivoting of a rectangular matrix A
       such that Q*R = A[:,p].</li>

<li> (H,U) = <a href=\"modelica://Modelica.Math.Matrices.hessenberg\">hessenberg</a>(A)
     - returns the upper Hessenberg form H and the orthogonal transformation matrix U
       of a square matrix A such that H = U'*A*U.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.realSchur\">realSchur</a>(A)
     - returns the real Schur form of a square matrix A.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.cholesky\">cholesky</a>(A)
     - returns the cholesky factor H of a real symmetric positive definite matrix A so that A = H'*H.</li>

<li> (D,Aimproved) = <a href=\"modelica://Modelica.Math.Matrices.balance\">balance</a>(A)
     - returns an improved form Aimproved of a square matrix A that has a smaller condition as A,
       with Aimproved = inv(diagonal(D))*A*diagonal(D).</li>
</ul>

<p><b>Matrix Properties</b></p>
<ul>
<li> <a href=\"modelica://Modelica.Math.Matrices.trace\">trace</a>(A)
     - returns the trace of square matrix A, i.e., the sum of the diagonal elements.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.det\">det</a>(A)
     - returns the determinant of square matrix A (using LU decomposition; try to avoid det(..))</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.inv\">inv</a>(A)
     - returns the inverse of square matrix A (try to avoid, use instead \"solve2(..) with B=identity(..))</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.rank\">rank</a>(A)
     - returns the rank of square matrix A (computed with singular value decomposition)</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.conditionNumber\">conditionNumber</a>(A)
     - returns the condition number norm(A)*norm(inv(A)) of a square matrix A in the range 1..&infin;.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.rcond\">rcond</a>(A)
     - returns the reciprocal condition number 1/conditionNumber(A) of a square matrix A in the range 0..1.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.norm\">norm</a>(A)
     - returns the 1-, 2-, or infinity-norm of matrix A.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.frobeniusNorm\">frobeniusNorm</a>(A)
     - returns the Frobenius norm of matrix A.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.nullSpace\">nullSpace</a>(A)
     - returns the null space of matrix A.</li>
</ul>

<p><b>Matrix Exponentials</b></p>
<ul>
<li> <a href=\"modelica://Modelica.Math.Matrices.exp\">exp</a>(A)
     - returns the exponential e^A of a matrix A by adaptive Taylor series
       expansion with scaling and balancing</li>

<li> (phi, gamma) = <a href=\"modelica://Modelica.Math.Matrices.integralExp\">integralExp</a>(A,B)
     - returns the exponential phi=e^A and the integral gamma=integral(exp(A*t)*dt)*B as needed
       for a discretized system with zero order hold.</li>

<li> (phi, gamma, gamma1) = <a href=\"modelica://Modelica.Math.Matrices.integralExpT\">integralExpT</a>(A,B)
     - returns the exponential phi=e^A, the integral gamma=integral(exp(A*t)*dt)*B,
       and the time-weighted integral gamma1 = integral((T-t)*exp(A*t)*dt)*B as needed
       for a discretized system with first order hold.</li>
</ul>

<p><b>Matrix Equations</b></p>
<ul>
<li> <a href=\"modelica://Modelica.Math.Matrices.continuousLyapunov\">continuousLyapunov</a>(A,C)
     - returns solution X of the continuous-time Lyapunov equation X*A + A'*X = C</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.continuousSylvester\">continuousSylvester</a>(A,B,C)
     - returns solution X of the continuous-time Sylvester equation A*X + X*B = C</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.continuousRiccati\">continuousRiccati</a>(A,B,R,Q)
     - returns solution X of the continuous-time algebraic Riccati equation
       A'*X + X*A - X*B*inv(R)*B'*X + Q = 0</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.discreteLyapunov\">discreteLyapunov</a>(A,C)
     - returns solution X of the discrete-time Lyapunov equation A'*X*A + sgn*X = C</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.discreteSylvester\">discreteSylvester</a>(A,B,C)
     - returns solution X of the discrete-time Sylvester equation A*X*B + sgn*X = C</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.discreteRiccati\">discreteRiccati</a>(A,B,R,Q)
     - returns solution X of the discrete-time algebraic Riccati equation
       A'*X*A - X - A'*X*B*inv(R + B'*X*B)*B'*X*A + Q = 0</li>
</ul>

<p><b>Matrix Manipulation</b></p>
<ul>
<li> <a href=\"modelica://Modelica.Math.Matrices.sort\">sort</a>(M)
     - returns the sorted rows or columns of matrix M in ascending or descending order.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.flipLeftRight\">flipLeftRight</a>(M)
     - returns matrix M so that the columns of M are flipped in left/right direction.</li>

<li> <a href=\"modelica://Modelica.Math.Matrices.flipUpDown\">flipUpDown</a>(M)
     - returns matrix M so that the rows of M are flipped in up/down direction.</li>
</ul>

<h4>See also</h4>
<a href=\"modelica://Modelica.Math.Vectors\">Vectors</a>

</html>"));
  end Matrices;

  function asin "Inverse sine (-1 <= u <= 1)"
    extends Modelica.Math.Icons.AxisCenter;
    input Real u;
    output SI.Angle y;

  external "builtin" y=  asin(u);
    annotation (
      Icon(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={
          Line(points={{-90,0},{68,0}}, color={192,192,192}),
          Polygon(
            points={{90,0},{68,8},{68,-8},{90,0}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Line(points={{-80,-80},{-79.2,-72.8},{-77.6,-67.5},{-73.6,-59.4},{-66.3,
                -49.8},{-53.5,-37.3},{-30.2,-19.7},{37.4,24.8},{57.5,40.8},{68.7,
                52.7},{75.2,62.2},{77.6,67.5},{80,80}}, color={0,0,0}),
          Text(
            extent={{-88,78},{-16,30}},
            lineColor={192,192,192},
            textString="asin")}),
      Diagram(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={Text(
              extent={{-40,-72},{-15,-88}},
              textString="-pi/2",
              lineColor={0,0,255}),Text(
              extent={{-38,88},{-13,72}},
              textString=" pi/2",
              lineColor={0,0,255}),Text(
              extent={{68,-9},{88,-29}},
              textString="+1",
              lineColor={0,0,255}),Text(
              extent={{-90,21},{-70,1}},
              textString="-1",
              lineColor={0,0,255}),Line(points={{-100,0},{84,0}}, color={95,95,95}),
            Polygon(
              points={{98,0},{82,6},{82,-6},{98,0}},
              lineColor={95,95,95},
              fillColor={95,95,95},
              fillPattern=FillPattern.Solid),Line(
              points={{-80,-80},{-79.2,-72.8},{-77.6,-67.5},{-73.6,-59.4},{-66.3,
              -49.8},{-53.5,-37.3},{-30.2,-19.7},{37.4,24.8},{57.5,40.8},{68.7,
              52.7},{75.2,62.2},{77.6,67.5},{80,80}},
              color={0,0,255},
              thickness=0.5),Text(
              extent={{82,24},{102,4}},
              lineColor={95,95,95},
              textString="u"),Line(
              points={{0,80},{86,80}},
              color={175,175,175},
              smooth=Smooth.None),Line(
              points={{80,86},{80,-10}},
              color={175,175,175},
              smooth=Smooth.None)}),
      Documentation(info="<html>
<p>
This function returns y = asin(u), with -1 &le; u &le; +1:
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/Math/asin.png\">
</p>
</html>"));
  end asin;

  function acos "Inverse cosine (-1 <= u <= 1)"
    extends Modelica.Math.Icons.AxisCenter;
    input Real u;
    output SI.Angle y;

  external "builtin" y=  acos(u);
    annotation (
      Icon(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={
          Line(points={{-90,-80},{68,-80}}, color={192,192,192}),
          Polygon(
            points={{90,-80},{68,-72},{68,-88},{90,-80}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Line(points={{-80,80},{-79.2,72.8},{-77.6,67.5},{-73.6,59.4},{-66.3,
                49.8},{-53.5,37.3},{-30.2,19.7},{37.4,-24.8},{57.5,-40.8},{68.7,-52.7},
                {75.2,-62.2},{77.6,-67.5},{80,-80}}, color={0,0,0}),
          Text(
            extent={{-86,-14},{-14,-62}},
            lineColor={192,192,192},
            textString="acos")}),
      Diagram(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={Line(points={{-100,-80},{84,-80}}, color={95,95,
            95}),Polygon(
              points={{98,-80},{82,-74},{82,-86},{98,-80}},
              lineColor={95,95,95},
              fillColor={95,95,95},
              fillPattern=FillPattern.Solid),Line(
              points={{-80,80},{-79.2,72.8},{-77.6,67.5},{-73.6,59.4},{-66.3,49.8},
              {-53.5,37.3},{-30.2,19.7},{37.4,-24.8},{57.5,-40.8},{68.7,-52.7},{
              75.2,-62.2},{77.6,-67.5},{80,-80}},
              color={0,0,255},
              thickness=0.5),Text(
              extent={{-30,88},{-5,72}},
              textString=" pi",
              lineColor={0,0,255}),Text(
              extent={{-94,-57},{-74,-77}},
              textString="-1",
              lineColor={0,0,255}),Text(
              extent={{60,-81},{80,-101}},
              textString="+1",
              lineColor={0,0,255}),Text(
              extent={{82,-56},{102,-76}},
              lineColor={95,95,95},
              textString="u"),Line(
              points={{-2,80},{84,80}},
              color={175,175,175},
              smooth=Smooth.None),Line(
              points={{80,82},{80,-86}},
              color={175,175,175},
              smooth=Smooth.None)}),
      Documentation(info="<html>
<p>
This function returns y = acos(u), with -1 &le; u &le; +1:
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/Math/acos.png\">
</p>
</html>"));
  end acos;

  function exp "Exponential, base e"
    extends Modelica.Math.Icons.AxisCenter;
    input Real u;
    output Real y;

  external "builtin" y=  exp(u);
    annotation (
      Icon(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={
          Line(points={{-90,-80.3976},{68,-80.3976}}, color={192,192,192}),
          Polygon(
            points={{90,-80.3976},{68,-72.3976},{68,-88.3976},{90,-80.3976}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Line(points={{-80,-80},{-31,-77.9},{-6.03,-74},{10.9,-68.4},{23.7,-61},
                {34.2,-51.6},{43,-40.3},{50.3,-27.8},{56.7,-13.5},{62.3,2.23},{
                67.1,18.6},{72,38.2},{76,57.6},{80,80}}, color={0,0,0}),
          Text(
            extent={{-86,50},{-14,2}},
            lineColor={192,192,192},
            textString="exp")}),
      Diagram(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={Line(points={{-100,-80.3976},{84,-80.3976}},
            color={95,95,95}),Polygon(
              points={{98,-80.3976},{82,-74.3976},{82,-86.3976},{98,-80.3976}},
              lineColor={95,95,95},
              fillColor={95,95,95},
              fillPattern=FillPattern.Solid),Line(
              points={{-80,-80},{-31,-77.9},{-6.03,-74},{10.9,-68.4},{23.7,-61},{
              34.2,-51.6},{43,-40.3},{50.3,-27.8},{56.7,-13.5},{62.3,2.23},{67.1,
              18.6},{72,38.2},{76,57.6},{80,80}},
              color={0,0,255},
              thickness=0.5),Text(
              extent={{-31,72},{-11,88}},
              textString="20",
              lineColor={0,0,255}),Text(
              extent={{-92,-81},{-72,-101}},
              textString="-3",
              lineColor={0,0,255}),Text(
              extent={{66,-81},{86,-101}},
              textString="3",
              lineColor={0,0,255}),Text(
              extent={{2,-69},{22,-89}},
              textString="1",
              lineColor={0,0,255}),Text(
              extent={{78,-54},{98,-74}},
              lineColor={95,95,95},
              textString="u"),Line(
              points={{0,80},{88,80}},
              color={175,175,175},
              smooth=Smooth.None),Line(
              points={{80,84},{80,-84}},
              color={175,175,175},
              smooth=Smooth.None)}),
      Documentation(info="<html>
<p>
This function returns y = exp(u), with -&infin; &lt; u &lt; &infin;:
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/Math/exp.png\">
</p>
</html>"));
  end exp;

  function log "Natural (base e) logarithm (u shall be > 0)"
    extends Modelica.Math.Icons.AxisLeft;
    input Real u;
    output Real y;

  external "builtin" y=  log(u);
    annotation (
      Icon(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={
          Line(points={{-90,0},{68,0}}, color={192,192,192}),
          Polygon(
            points={{90,0},{68,8},{68,-8},{90,0}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Line(points={{-80,-80},{-79.2,-50.6},{-78.4,-37},{-77.6,-28},{-76.8,-21.3},
                {-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,28},
                {-47,38.1},{-31.8,48.1},{-10.1,58},{22.1,68},{68.7,78.1},{80,80}},
              color={0,0,0}),
          Text(
            extent={{-6,-24},{66,-72}},
            lineColor={192,192,192},
            textString="log")}),
      Diagram(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}}), graphics={Line(points={{-100,0},{84,0}}, color={95,95,95}),
            Polygon(
              points={{100,0},{84,6},{84,-6},{100,0}},
              lineColor={95,95,95},
              fillColor={95,95,95},
              fillPattern=FillPattern.Solid),Line(
              points={{-78,-80},{-77.2,-50.6},{-76.4,-37},{-75.6,-28},{-74.8,-21.3},
              {-73.2,-11.4},{-70.8,-1.31},{-67.5,8.08},{-62.7,17.9},{-55.5,28},{-45,
              38.1},{-29.8,48.1},{-8.1,58},{24.1,68},{70.7,78.1},{82,80}},
              color={0,0,255},
              thickness=0.5),Text(
              extent={{-105,72},{-85,88}},
              textString="3",
              lineColor={0,0,255}),Text(
              extent={{60,-3},{80,-23}},
              textString="20",
              lineColor={0,0,255}),Text(
              extent={{-78,-7},{-58,-27}},
              textString="1",
              lineColor={0,0,255}),Text(
              extent={{84,26},{104,6}},
              lineColor={95,95,95},
              textString="u"),Text(
              extent={{-100,9},{-80,-11}},
              textString="0",
              lineColor={0,0,255}),Line(
              points={{-80,80},{84,80}},
              color={175,175,175},
              smooth=Smooth.None),Line(
              points={{82,82},{82,-6}},
              color={175,175,175},
              smooth=Smooth.None)}),
      Documentation(info="<html>
<p>
This function returns y = log(10) (the natural logarithm of u),
with u &gt; 0:
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/Math/log.png\">
</p>
</html>"));
  end log;
  annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
            {100,100}}), graphics={Line(points={{-80,0},{-68.7,34.2},{-61.5,53.1},
              {-55.1,66.4},{-49.4,74.6},{-43.8,79.1},{-38.2,79.8},{-32.6,76.6},{
              -26.9,69.7},{-21.3,59.4},{-14.9,44.1},{-6.83,21.2},{10.1,-30.8},{17.3,
              -50.2},{23.7,-64.2},{29.3,-73.1},{35,-78.4},{40.6,-80},{46.2,-77.6},
              {51.9,-71.5},{57.5,-61.9},{63.9,-47.2},{72,-24.8},{80,0}}, color={
              0,0,0}, smooth=Smooth.Bezier)}), Documentation(info="<HTML>
<p>
This package contains <b>basic mathematical functions</b> (such as sin(..)),
as well as functions operating on
<a href=\"modelica://Modelica.Math.Vectors\">vectors</a>,
<a href=\"modelica://Modelica.Math.Matrices\">matrices</a>,
<a href=\"modelica://Modelica.Math.Nonlinear\">nonlinear functions</a>, and
<a href=\"modelica://Modelica.Math.BooleanVectors\">Boolean vectors</a>.
</p>

<dl>
<dt><b>Main Authors:</b>
<dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a> and
    Marcus Baur<br>
    Deutsches Zentrum f&uuml;r Luft und Raumfahrt e.V. (DLR)<br>
    Institut f&uuml;r Robotik und Mechatronik<br>
    Postfach 1116<br>
    D-82230 Wessling<br>
    Germany<br>
    email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
</dl>

<p>
Copyright &copy; 1998-2013, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>
</html>",   revisions="<html>
<ul>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Function tempInterpol2 added.</li>
<li><i>Oct. 24, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Icons for icon and diagram level introduced.</li>
<li><i>June 30, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized.</li>
</ul>

</html>"));
  end Math;

  package Utilities
  "Library of utility functions dedicated to scripting (operating on files, streams, strings, system)"
    extends Modelica.Icons.Package;

    package Streams "Read from files and write to files"
      extends Modelica.Icons.Package;

      function print "Print string to terminal or file"
        extends Modelica.Icons.Function;
        input String string="" "String to be printed";
        input String fileName=""
        "File where to print (empty string is the terminal)"
                     annotation(Dialog(saveSelector(filter="Text files (*.txt)",
                            caption="Text file to store the output of print(..)")));
      external "C" ModelicaInternal_print(string, fileName) annotation(Library="ModelicaExternalC");
        annotation (Documentation(info="<HTML>
<h4>Syntax</h4>
<blockquote><pre>
Streams.<b>print</b>(string);
Streams.<b>print</b>(string,fileName);
</pre></blockquote>
<h4>Description</h4>
<p>
Function <b>print</b>(..) opens automatically the given file, if
it is not yet open. If the file does not exist, it is created.
If the file does exist, the given string is appended to the file.
If this is not desired, call \"Files.remove(fileName)\" before calling print
(\"remove(..)\" is silent, if the file does not exist).
The Modelica environment may close the file whenever appropriate.
This can be enforced by calling <b>Streams.close</b>(fileName).
After every call of \"print(..)\" a \"new line\" is printed automatically.
</p>
<h4>Example</h4>
<blockquote><pre>
  Streams.print(\"x = \" + String(x));
  Streams.print(\"y = \" + String(y));
  Streams.print(\"x = \" + String(y), \"mytestfile.txt\");
</pre></blockquote>
<h4>See also</h4>
<p>
<a href=\"modelica://Modelica.Utilities.Streams\">Streams</a>,
<a href=\"modelica://Modelica.Utilities.Streams.error\">Streams.error</a>,
<a href=\"modelica://ModelicaReference.Operators.'String()'\">ModelicaReference.Operators.'String()'</a>
</p>
</HTML>"));
      end print;

      function error "Print error message and cancel all actions"
        extends Modelica.Icons.Function;
        input String string "String to be printed to error message window";
        external "C" ModelicaError(string) annotation(Library="ModelicaExternalC");
        annotation (Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
Streams.<b>error</b>(string);
</pre></blockquote>
<h4>Description</h4>
<p>
Print the string \"string\" as error message and
cancel all actions. Line breaks are characterized
by \"\\n\" in the string.
</p>
<h4>Example</h4>
<blockquote><pre>
  Streams.error(\"x (= \" + String(x) + \")\\nhas to be in the range 0 .. 1\");
</pre></blockquote>
<h4>See also</h4>
<p>
<a href=\"modelica://Modelica.Utilities.Streams\">Streams</a>,
<a href=\"modelica://Modelica.Utilities.Streams.print\">Streams.print</a>,
<a href=\"modelica://ModelicaReference.Operators.'String()'\">ModelicaReference.Operators.'String()'</a>
</p>
</html>"));
      end error;
      annotation (
        Documentation(info="<HTML>
<h4>Library content</h4>
<p>
Package <b>Streams</b> contains functions to input and output strings
to a message window or on files. Note that a string is interpreted
and displayed as html text (e.g., with print(..) or error(..))
if it is enclosed with the Modelica html quotation, e.g.,
</p>
<center>
string = \"&lt;html&gt; first line &lt;br&gt; second line &lt;/html&gt;\".
</center>
<p>
It is a quality of implementation, whether (a) all tags of html are supported
or only a subset, (b) how html tags are interpreted if the output device
does not allow to display formatted text.
</p>
<p>
In the table below an example call to every function is given:
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><th><b><i>Function/type</i></b></th><th><b><i>Description</i></b></th></tr>
  <tr><td valign=\"top\"><a href=\"modelica://Modelica.Utilities.Streams.print\">print</a>(string)<br>
          <a href=\"modelica://Modelica.Utilities.Streams.print\">print</a>(string,fileName)</td>
      <td valign=\"top\"> Print string \"string\" or vector of strings to message window or on
           file \"fileName\".</td>
  </tr>
  <tr><td valign=\"top\">stringVector =
         <a href=\"modelica://Modelica.Utilities.Streams.readFile\">readFile</a>(fileName)</td>
      <td valign=\"top\"> Read complete text file and return it as a vector of strings.</td>
  </tr>
  <tr><td valign=\"top\">(string, endOfFile) =
         <a href=\"modelica://Modelica.Utilities.Streams.readLine\">readLine</a>(fileName, lineNumber)</td>
      <td valign=\"top\">Returns from the file the content of line lineNumber.</td>
  </tr>
  <tr><td valign=\"top\">lines =
         <a href=\"modelica://Modelica.Utilities.Streams.countLines\">countLines</a>(fileName)</td>
      <td valign=\"top\">Returns the number of lines in a file.</td>
  </tr>
  <tr><td valign=\"top\"><a href=\"modelica://Modelica.Utilities.Streams.error\">error</a>(string)</td>
      <td valign=\"top\"> Print error message \"string\" to message window
           and cancel all actions</td>
  </tr>
  <tr><td valign=\"top\"><a href=\"modelica://Modelica.Utilities.Streams.close\">close</a>(fileName)</td>
      <td valign=\"top\"> Close file if it is still open. Ignore call if
           file is already closed or does not exist. </td>
  </tr>
</table>
<p>
Use functions <b>scanXXX</b> from package
<a href=\"modelica://Modelica.Utilities.Strings\">Strings</a>
to parse a string.
</p>
<p>
If Real, Integer or Boolean values shall be printed
or used in an error message, they have to be first converted
to strings with the builtin operator
<a href=\"modelica://ModelicaReference.Operators.'String()'\">ModelicaReference.Operators.'String()'</a>(...).
Example:
</p>
<pre>
  <b>if</b> x &lt; 0 <b>or</b> x &gt; 1 <b>then</b>
     Streams.error(\"x (= \" + String(x) + \") has to be in the range 0 .. 1\");
  <b>end if</b>;
</pre>
</html>"));
    end Streams;
      annotation (
  Icon(coordinateSystem(extent={{-100.0,-100.0},{100.0,100.0}}), graphics={
      Polygon(
        origin={1.3835,-4.1418},
        rotation=45.0,
        fillColor={64,64,64},
        pattern=LinePattern.None,
        fillPattern=FillPattern.Solid,
        points={{-15.0,93.333},{-15.0,68.333},{0.0,58.333},{15.0,68.333},{15.0,93.333},{20.0,93.333},{25.0,83.333},{25.0,58.333},{10.0,43.333},{10.0,-41.667},{25.0,-56.667},{25.0,-76.667},{10.0,-91.667},{0.0,-91.667},{0.0,-81.667},{5.0,-81.667},{15.0,-71.667},{15.0,-61.667},{5.0,-51.667},{-5.0,-51.667},{-15.0,-61.667},{-15.0,-71.667},{-5.0,-81.667},{0.0,-81.667},{0.0,-91.667},{-10.0,-91.667},{-25.0,-76.667},{-25.0,-56.667},{-10.0,-41.667},{-10.0,43.333},{-25.0,58.333},{-25.0,83.333},{-20.0,93.333}}),
      Polygon(
        origin={10.1018,5.218},
        rotation=-45.0,
        fillColor={255,255,255},
        fillPattern=FillPattern.Solid,
        points={{-15.0,87.273},{15.0,87.273},{20.0,82.273},{20.0,27.273},{10.0,17.273},{10.0,7.273},{20.0,2.273},{20.0,-2.727},{5.0,-2.727},{5.0,-77.727},{10.0,-87.727},{5.0,-112.727},{-5.0,-112.727},{-10.0,-87.727},{-5.0,-77.727},{-5.0,-2.727},{-20.0,-2.727},{-20.0,2.273},{-10.0,7.273},{-10.0,17.273},{-20.0,27.273},{-20.0,82.273}})}),
  Documentation(info="<html>
<p>
This package contains Modelica <b>functions</b> that are
especially suited for <b>scripting</b>. The functions might
be used to work with strings, read data from file, write data
to file or copy, move and remove files.
</p>
<p>
For an introduction, have especially a look at:
</p>
<ul>
<li> <a href=\"modelica://Modelica.Utilities.UsersGuide\">Modelica.Utilities.User's Guide</a>
     discusses the most important aspects of this library.</li>
<li> <a href=\"modelica://Modelica.Utilities.Examples\">Modelica.Utilities.Examples</a>
     contains examples that demonstrate the usage of this library.</li>
</ul>
<p>
The following main sublibraries are available:
</p>
<ul>
<li> <a href=\"modelica://Modelica.Utilities.Files\">Files</a>
     provides functions to operate on files and directories, e.g.,
     to copy, move, remove files.</li>
<li> <a href=\"modelica://Modelica.Utilities.Streams\">Streams</a>
     provides functions to read from files and write to files.</li>
<li> <a href=\"modelica://Modelica.Utilities.Strings\">Strings</a>
     provides functions to operate on strings. E.g.
     substring, find, replace, sort, scanToken.</li>
<li> <a href=\"modelica://Modelica.Utilities.System\">System</a>
     provides functions to interact with the environment.
     E.g., get or set the working directory or environment
     variables and to send a command to the default shell.</li>
</ul>

<p>
Copyright &copy; 1998-2013, Modelica Association, DLR, and Dassault Syst&egrave;mes AB.
</p>

<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>

</html>"));
  end Utilities;

  package Constants
  "Library of mathematical constants and constants of nature (e.g., pi, eps, R, sigma)"
    import SI = Modelica.SIunits;
    import NonSI = Modelica.SIunits.Conversions.NonSIunits;
    extends Modelica.Icons.Package;

    final constant Real pi=2*Modelica.Math.asin(1.0);

    final constant Real eps=ModelicaServices.Machine.eps
    "Biggest number such that 1.0 + eps = 1.0";

    final constant Real small=ModelicaServices.Machine.small
    "Smallest number such that small and -small are representable on the machine";

    final constant Real inf=ModelicaServices.Machine.inf
    "Biggest Real number such that inf and -inf are representable on the machine";

    final constant SI.Acceleration g_n=9.80665
    "Standard acceleration of gravity on earth";

    final constant Real k(final unit="J/K") = 1.3806505e-23
    "Boltzmann constant";

    final constant Real R(final unit="J/(mol.K)") = 8.314472
    "Molar gas constant";

    final constant NonSI.Temperature_degC T_zero=-273.15
    "Absolute zero temperature";
    annotation (
      Documentation(info="<html>
<p>
This package provides often needed constants from mathematics, machine
dependent constants and constants from nature. The latter constants
(name, value, description) are from the following source:
</p>

<dl>
<dt>Peter J. Mohr and Barry N. Taylor (1999):</dt>
<dd><b>CODATA Recommended Values of the Fundamental Physical Constants: 1998</b>.
    Journal of Physical and Chemical Reference Data, Vol. 28, No. 6, 1999 and
    Reviews of Modern Physics, Vol. 72, No. 2, 2000. See also <a href=
\"http://physics.nist.gov/cuu/Constants/\">http://physics.nist.gov/cuu/Constants/</a></dd>
</dl>

<p>CODATA is the Committee on Data for Science and Technology.</p>

<dl>
<dt><b>Main Author:</b></dt>
<dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
    Deutsches Zentrum f&uuml;r Luft und Raumfahrt e. V. (DLR)<br>
    Oberpfaffenhofen<br>
    Postfach 11 16<br>
    D-82230 We&szlig;ling<br>
    email: <a href=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</a></dd>
</dl>

<p>
Copyright &copy; 1998-2013, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>
</html>",   revisions="<html>
<ul>
<li><i>Nov 8, 2004</i>
       by <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Constants updated according to 2002 CODATA values.</li>
<li><i>Dec 9, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Constants updated according to 1998 CODATA values. Using names, values
       and description text from this source. Included magnetic and
       electric constant.</li>
<li><i>Sep 18, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Constants eps, inf, small introduced.</li>
<li><i>Nov 15, 1997</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized.</li>
</ul>
</html>"),
      Icon(coordinateSystem(extent={{-100.0,-100.0},{100.0,100.0}}), graphics={
        Polygon(
          origin={-9.2597,25.6673},
          fillColor={102,102,102},
          pattern=LinePattern.None,
          fillPattern=FillPattern.Solid,
          points={{48.017,11.336},{48.017,11.336},{10.766,11.336},{-25.684,10.95},{-34.944,-15.111},{-34.944,-15.111},{-32.298,-15.244},{-32.298,-15.244},{-22.112,0.168},{11.292,0.234},{48.267,-0.097},{48.267,-0.097}},
          smooth=Smooth.Bezier),
        Polygon(
          origin={-19.9923,-8.3993},
          fillColor={102,102,102},
          pattern=LinePattern.None,
          fillPattern=FillPattern.Solid,
          points={{3.239,37.343},{3.305,37.343},{-0.399,2.683},{-16.936,-20.071},{-7.808,-28.604},{6.811,-22.519},{9.986,37.145},{9.986,37.145}},
          smooth=Smooth.Bezier),
        Polygon(
          origin={23.753,-11.5422},
          fillColor={102,102,102},
          pattern=LinePattern.None,
          fillPattern=FillPattern.Solid,
          points={{-10.873,41.478},{-10.873,41.478},{-14.048,-4.162},{-9.352,-24.8},{7.912,-24.469},{16.247,0.27},{16.247,0.27},{13.336,0.071},{13.336,0.071},{7.515,-9.983},{-3.134,-7.271},{-2.671,41.214},{-2.671,41.214}},
          smooth=Smooth.Bezier)}));
  end Constants;

  package Icons "Library of icons"
    extends Icons.Package;

    partial package Package "Icon for standard packages"

      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{100,100}}), graphics={
            Rectangle(
              lineColor={200,200,200},
              fillColor={248,248,248},
              fillPattern=FillPattern.HorizontalCylinder,
              extent={{-100.0,-100.0},{100.0,100.0}},
              radius=25.0),
            Rectangle(
              lineColor={128,128,128},
              fillPattern=FillPattern.None,
              extent={{-100.0,-100.0},{100.0,100.0}},
              radius=25.0)}),   Documentation(info="<html>
<p>Standard package icon.</p>
</html>"));
    end Package;

    partial package BasesPackage "Icon for packages containing base classes"
      extends Modelica.Icons.Package;
      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}), graphics={
            Ellipse(
              extent={{-30.0,-30.0},{30.0,30.0}},
              lineColor={128,128,128},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid)}),
                                Documentation(info="<html>
<p>This icon shall be used for a package/library that contains base models and classes, respectively.</p>
</html>"));
    end BasesPackage;

    partial package VariantsPackage "Icon for package containing variants"
      extends Modelica.Icons.Package;
      annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                {100,100}}),       graphics={
            Ellipse(
              origin={10.0,10.0},
              fillColor={76,76,76},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              extent={{-80.0,-80.0},{-20.0,-20.0}}),
            Ellipse(
              origin={10.0,10.0},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              extent={{0.0,-80.0},{60.0,-20.0}}),
            Ellipse(
              origin={10.0,10.0},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              extent={{0.0,0.0},{60.0,60.0}}),
            Ellipse(
              origin={10.0,10.0},
              lineColor={128,128,128},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid,
              extent={{-80.0,0.0},{-20.0,60.0}})}),
                                Documentation(info="<html>
<p>This icon shall be used for a package/library that contains several variants of one components.</p>
</html>"));
    end VariantsPackage;

    partial package InterfacesPackage "Icon for packages containing interfaces"
      extends Modelica.Icons.Package;
      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}), graphics={
            Polygon(origin={20.0,0.0},
              lineColor={64,64,64},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid,
              points={{-10.0,70.0},{10.0,70.0},{40.0,20.0},{80.0,20.0},{80.0,-20.0},{40.0,-20.0},{10.0,-70.0},{-10.0,-70.0}}),
            Polygon(fillColor={102,102,102},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              points={{-100.0,20.0},{-60.0,20.0},{-30.0,70.0},{-10.0,70.0},{-10.0,-70.0},{-30.0,-70.0},{-60.0,-20.0},{-100.0,-20.0}})}),
                                Documentation(info="<html>
<p>This icon indicates packages containing interfaces.</p>
</html>"));
    end InterfacesPackage;

    partial package SourcesPackage "Icon for packages containing sources"
      extends Modelica.Icons.Package;
      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}), graphics={
            Polygon(origin={23.3333,0.0},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              points={{-23.333,30.0},{46.667,0.0},{-23.333,-30.0}}),
            Rectangle(
              fillColor=  {128,128,128},
              pattern=  LinePattern.None,
              fillPattern=  FillPattern.Solid,
              extent=  {{-70,-4.5},{0,4.5}})}),
                                Documentation(info="<html>
<p>This icon indicates a package which contains sources.</p>
</html>"));
    end SourcesPackage;

    partial package UtilitiesPackage "Icon for utility packages"
      extends Modelica.Icons.Package;
       annotation (Icon(coordinateSystem(extent={{-100.0,-100.0},{100.0,100.0}}), graphics={
      Polygon(
        origin={1.3835,-4.1418},
        rotation=45.0,
        fillColor={64,64,64},
        pattern=LinePattern.None,
        fillPattern=FillPattern.Solid,
        points={{-15.0,93.333},{-15.0,68.333},{0.0,58.333},{15.0,68.333},{15.0,93.333},{20.0,93.333},{25.0,83.333},{25.0,58.333},{10.0,43.333},{10.0,-41.667},{25.0,-56.667},{25.0,-76.667},{10.0,-91.667},{0.0,-91.667},{0.0,-81.667},{5.0,-81.667},{15.0,-71.667},{15.0,-61.667},{5.0,-51.667},{-5.0,-51.667},{-15.0,-61.667},{-15.0,-71.667},{-5.0,-81.667},{0.0,-81.667},{0.0,-91.667},{-10.0,-91.667},{-25.0,-76.667},{-25.0,-56.667},{-10.0,-41.667},{-10.0,43.333},{-25.0,58.333},{-25.0,83.333},{-20.0,93.333}}),
      Polygon(
        origin={10.1018,5.218},
        rotation=-45.0,
        fillColor={255,255,255},
        fillPattern=FillPattern.Solid,
        points={{-15.0,87.273},{15.0,87.273},{20.0,82.273},{20.0,27.273},{10.0,17.273},{10.0,7.273},{20.0,2.273},{20.0,-2.727},{5.0,-2.727},{5.0,-77.727},{10.0,-87.727},{5.0,-112.727},{-5.0,-112.727},{-10.0,-87.727},{-5.0,-77.727},{-5.0,-2.727},{-20.0,-2.727},{-20.0,2.273},{-10.0,7.273},{-10.0,17.273},{-20.0,27.273},{-20.0,82.273}})}),
      Documentation(info="<html>
<p>This icon indicates a package containing utility classes.</p>
</html>"));
    end UtilitiesPackage;

    partial package TypesPackage
    "Icon for packages containing type definitions"
      extends Modelica.Icons.Package;
      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}), graphics={Polygon(
              origin={-12.167,-23},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              points={{12.167,65},{14.167,93},{36.167,89},{24.167,20},{4.167,-30},
                  {14.167,-30},{24.167,-30},{24.167,-40},{-5.833,-50},{-15.833,
                  -30},{4.167,20},{12.167,65}},
              smooth=Smooth.Bezier,
              lineColor={0,0,0}), Polygon(
              origin={2.7403,1.6673},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              points={{49.2597,22.3327},{31.2597,24.3327},{7.2597,18.3327},{-26.7403,
                10.3327},{-46.7403,14.3327},{-48.7403,6.3327},{-32.7403,0.3327},{-6.7403,
                4.3327},{33.2597,14.3327},{49.2597,14.3327},{49.2597,22.3327}},
              smooth=Smooth.Bezier)}));
    end TypesPackage;

    partial package IconsPackage "Icon for packages containing icons"
      extends Modelica.Icons.Package;
      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}), graphics={Polygon(
              origin={-8.167,-17},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              points={{-15.833,20.0},{-15.833,30.0},{14.167,40.0},{24.167,20.0},{
                  4.167,-30.0},{14.167,-30.0},{24.167,-30.0},{24.167,-40.0},{-5.833,
                  -50.0},{-15.833,-30.0},{4.167,20.0},{-5.833,20.0}},
              smooth=Smooth.Bezier,
              lineColor={0,0,0}), Ellipse(
              origin={-0.5,56.5},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              extent={{-12.5,-12.5},{12.5,12.5}},
              lineColor={0,0,0})}));
    end IconsPackage;

    partial package InternalPackage
    "Icon for an internal package (indicating that the package should not be directly utilized by user)"

    annotation (
      Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{100,
              100}}), graphics={
          Rectangle(
            lineColor={215,215,215},
            fillColor={255,255,255},
            fillPattern=FillPattern.HorizontalCylinder,
            extent={{-100,-100},{100,100}},
            radius=25),
          Rectangle(
            lineColor={215,215,215},
            fillPattern=FillPattern.None,
            extent={{-100,-100},{100,100}},
            radius=25),
          Ellipse(
            extent={{-80,80},{80,-80}},
            lineColor={215,215,215},
            fillColor={215,215,215},
            fillPattern=FillPattern.Solid),
          Ellipse(
            extent={{-55,55},{55,-55}},
            lineColor={255,255,255},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-60,14},{60,-14}},
            lineColor={215,215,215},
            fillColor={215,215,215},
            fillPattern=FillPattern.Solid,
            origin={0,0},
            rotation=45)}),
      Documentation(info="<html>

<p>
This icon shall be used for a package that contains internal classes not to be
directly utilized by a user.
</p>
</html>"));
    end InternalPackage;

    partial package MaterialPropertiesPackage
    "Icon for package containing property classes"
      extends Modelica.Icons.Package;
      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}), graphics={
            Ellipse(
              lineColor={102,102,102},
              fillColor={204,204,204},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Sphere,
              extent={{-60.0,-60.0},{60.0,60.0}})}),
                                Documentation(info="<html>
<p>This icon indicates a package that contains properties</p>
</html>"));
    end MaterialPropertiesPackage;

    partial function Function "Icon for functions"

      annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{100,100}}), graphics={
            Text(
              lineColor={0,0,255},
              extent={{-150,105},{150,145}},
              textString="%name"),
            Ellipse(
              lineColor = {108,88,49},
              fillColor = {255,215,136},
              fillPattern = FillPattern.Solid,
              extent = {{-100,-100},{100,100}}),
            Text(
              lineColor={108,88,49},
              extent={{-90.0,-90.0},{90.0,90.0}},
              textString="f")}),
    Documentation(info="<html>
<p>This icon indicates Modelica functions.</p>
</html>"));
    end Function;

    partial record Record "Icon for records"

      annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,100}}), graphics={
            Text(
              lineColor={0,0,255},
              extent={{-150,60},{150,100}},
              textString="%name"),
            Rectangle(
              origin={0.0,-25.0},
              lineColor={64,64,64},
              fillColor={255,215,136},
              fillPattern=FillPattern.Solid,
              extent={{-100.0,-75.0},{100.0,75.0}},
              radius=25.0),
            Line(
              points={{-100.0,0.0},{100.0,0.0}},
              color={64,64,64}),
            Line(
              origin={0.0,-50.0},
              points={{-100.0,0.0},{100.0,0.0}},
              color={64,64,64}),
            Line(
              origin={0.0,-25.0},
              points={{0.0,75.0},{0.0,-75.0}},
              color={64,64,64})}),                        Documentation(info="<html>
<p>
This icon is indicates a record.
</p>
</html>"));
    end Record;
    annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}), graphics={Polygon(
              origin={-8.167,-17},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              points={{-15.833,20.0},{-15.833,30.0},{14.167,40.0},{24.167,20.0},{
                  4.167,-30.0},{14.167,-30.0},{24.167,-30.0},{24.167,-40.0},{-5.833,
                  -50.0},{-15.833,-30.0},{4.167,20.0},{-5.833,20.0}},
              smooth=Smooth.Bezier,
              lineColor={0,0,0}), Ellipse(
              origin={-0.5,56.5},
              fillColor={128,128,128},
              pattern=LinePattern.None,
              fillPattern=FillPattern.Solid,
              extent={{-12.5,-12.5},{12.5,12.5}},
              lineColor={0,0,0})}), Documentation(info="<html>
<p>This package contains definitions for the graphical layout of components which may be used in different libraries. The icons can be utilized by inheriting them in the desired class using &quot;extends&quot; or by directly copying the &quot;icon&quot; layer. </p>

<h4>Main Authors:</h4>

<dl>
<dt><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a></dt>
    <dd>Deutsches Zentrum fuer Luft und Raumfahrt e.V. (DLR)</dd>
    <dd>Oberpfaffenhofen</dd>
    <dd>Postfach 1116</dd>
    <dd>D-82230 Wessling</dd>
    <dd>email: <a href=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</a></dd>
<dt>Christian Kral</dt>
    <dd><a href=\"http://www.ait.ac.at/\">Austrian Institute of Technology, AIT</a></dd>
    <dd>Mobility Department</dd><dd>Giefinggasse 2</dd>
    <dd>1210 Vienna, Austria</dd>
    <dd>email: <a href=\"mailto:dr.christian.kral@gmail.com\">dr.christian.kral@gmail.com</a></dd>
<dt>Johan Andreasson</dt>
    <dd><a href=\"http://www.modelon.se/\">Modelon AB</a></dd>
    <dd>Ideon Science Park</dd>
    <dd>22370 Lund, Sweden</dd>
    <dd>email: <a href=\"mailto:johan.andreasson@modelon.se\">johan.andreasson@modelon.se</a></dd>
</dl>

<p>Copyright &copy; 1998-2013, Modelica Association, DLR, AIT, and Modelon AB. </p>
<p><i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified under the terms of the <b>Modelica license</b>, see the license conditions and the accompanying <b>disclaimer</b> in <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a>.</i> </p>
</html>"));
  end Icons;

  package SIunits
  "Library of type and unit definitions based on SI units according to ISO 31-1992"
    extends Modelica.Icons.Package;

    package Icons "Icons for SIunits"
      extends Modelica.Icons.IconsPackage;

      partial function Conversion "Base icon for conversion functions"

        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={191,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,0},{30,0}}, color={191,0,0}),
              Polygon(
                points={{90,0},{30,20},{30,-20},{90,0}},
                lineColor={191,0,0},
                fillColor={191,0,0},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-115,155},{115,105}},
                textString="%name",
                lineColor={0,0,255})}));
      end Conversion;
    end Icons;

    package Conversions
    "Conversion functions to/from non SI units and type definitions of non SI units"
      extends Modelica.Icons.Package;

      package NonSIunits "Type definitions of non SI units"
        extends Modelica.Icons.Package;

        type Temperature_degC = Real (final quantity="ThermodynamicTemperature",
              final unit="degC")
        "Absolute temperature in degree Celsius (for relative temperature use SIunits.TemperatureDifference)"
                                                                                                            annotation(absoluteValue=true);

        type AngularVelocity_rpm = Real (final quantity="AngularVelocity", final unit=
                   "1/min")
        "Angular velocity in revolutions per minute. Alias unit names that are outside of the SI system: rpm, r/min, rev/min";

        type Pressure_bar = Real (final quantity="Pressure", final unit="bar")
        "Absolute pressure in bar";
        annotation (Documentation(info="<HTML>
<p>
This package provides predefined types, such as <b>Angle_deg</b> (angle in
degree), <b>AngularVelocity_rpm</b> (angular velocity in revolutions per
minute) or <b>Temperature_degF</b> (temperature in degree Fahrenheit),
which are in common use but are not part of the international standard on
units according to ISO 31-1992 \"General principles concerning quantities,
units and symbols\" and ISO 1000-1992 \"SI units and recommendations for
the use of their multiples and of certain other units\".</p>
<p>If possible, the types in this package should not be used. Use instead
types of package Modelica.SIunits. For more information on units, see also
the book of Francois Cardarelli <b>Scientific Unit Conversion - A
Practical Guide to Metrication</b> (Springer 1997).</p>
<p>Some units, such as <b>Temperature_degC/Temp_C</b> are both defined in
Modelica.SIunits and in Modelica.Conversions.NonSIunits. The reason is that these
definitions have been placed erroneously in Modelica.SIunits although they
are not SIunits. For backward compatibility, these type definitions are
still kept in Modelica.SIunits.</p>
</html>"),   Icon(coordinateSystem(extent={{-100,-100},{100,100}}), graphics={
        Text(
          origin={15.0,51.8518},
          extent={{-105.0,-86.8518},{75.0,-16.8518}},
          lineColor={0,0,0},
          textString="[km/h]")}));
      end NonSIunits;

      function to_degC "Convert from Kelvin to degCelsius"
        extends Modelica.SIunits.Icons.Conversion;
        input Temperature Kelvin "Kelvin value";
        output NonSIunits.Temperature_degC Celsius "Celsius value";
      algorithm
        Celsius := Kelvin + Modelica.Constants.T_zero;
        annotation (Inline=true,Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-20,100},{-100,20}},
                lineColor={0,0,0},
                textString="K"), Text(
                extent={{100,-20},{20,-100}},
                lineColor={0,0,0},
                textString="degC")}));
      end to_degC;

      function from_degC "Convert from degCelsius to Kelvin"
        extends Modelica.SIunits.Icons.Conversion;
        input NonSIunits.Temperature_degC Celsius "Celsius value";
        output Temperature Kelvin "Kelvin value";
      algorithm
        Kelvin := Celsius - Modelica.Constants.T_zero;
        annotation (Inline=true,Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-20,100},{-100,20}},
                lineColor={0,0,0},
                textString="degC"),  Text(
                extent={{100,-20},{20,-100}},
                lineColor={0,0,0},
                textString="K")}));
      end from_degC;

      function to_bar "Convert from Pascal to bar"
        extends Modelica.SIunits.Icons.Conversion;
        input Pressure Pa "Pascal value";
        output NonSIunits.Pressure_bar bar "bar value";
      algorithm
        bar := Pa/1e5;
        annotation (Inline=true,Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-12,100},{-100,56}},
                lineColor={0,0,0},
                textString="Pa"),     Text(
                extent={{98,-52},{-4,-100}},
                lineColor={0,0,0},
                textString="bar")}));
      end to_bar;
      annotation (                              Documentation(info="<HTML>
<p>This package provides conversion functions from the non SI Units
defined in package Modelica.SIunits.Conversions.NonSIunits to the
corresponding SI Units defined in package Modelica.SIunits and vice
versa. It is recommended to use these functions in the following
way (note, that all functions have one Real input and one Real output
argument):</p>
<pre>
  <b>import</b> SI = Modelica.SIunits;
  <b>import</b> Modelica.SIunits.Conversions.*;
     ...
  <b>parameter</b> SI.Temperature     T   = from_degC(25);   // convert 25 degree Celsius to Kelvin
  <b>parameter</b> SI.Angle           phi = from_deg(180);   // convert 180 degree to radian
  <b>parameter</b> SI.AngularVelocity w   = from_rpm(3600);  // convert 3600 revolutions per minutes
                                                      // to radian per seconds
</pre>

</html>"));
    end Conversions;

    type Angle = Real (
        final quantity="Angle",
        final unit="rad",
        displayUnit="deg");

    type Length = Real (final quantity="Length", final unit="m");

    type Height = Length(min=0);

    type Volume = Real (final quantity="Volume", final unit="m3");

    type Time = Real (final quantity="Time", final unit="s");

    type Velocity = Real (final quantity="Velocity", final unit="m/s");

    type Acceleration = Real (final quantity="Acceleration", final unit="m/s2");

    type Density = Real (
        final quantity="Density",
        final unit="kg/m3",
        displayUnit="g/cm3",
        min=0.0);

    type SpecificVolume = Real (
        final quantity="SpecificVolume",
        final unit="m3/kg",
        min=0.0);

    type Pressure = Real (
        final quantity="Pressure",
        final unit="Pa",
        displayUnit="bar");

    type AbsolutePressure = Pressure (min=0.0, nominal = 1e5);

    type DynamicViscosity = Real (
        final quantity="DynamicViscosity",
        final unit="Pa.s",
        min=0);

    type SurfaceTension = Real (final quantity="SurfaceTension", final unit="N/m");

    type Power = Real (final quantity="Power", final unit="W");

    type MassFlowRate = Real (quantity="MassFlowRate", final unit="kg/s");

    type VolumeFlowRate = Real (final quantity="VolumeFlowRate", final unit=
            "m3/s");

    type ThermodynamicTemperature = Real (
        final quantity="ThermodynamicTemperature",
        final unit="K",
        min = 0.0,
        start = 288.15,
        nominal = 300,
        displayUnit="degC")
    "Absolute temperature (use type TemperatureDifference for relative temperatures)"
                                                                                                        annotation(absoluteValue=true);

    type Temp_K = ThermodynamicTemperature;

    type Temperature = ThermodynamicTemperature;

    type CubicExpansionCoefficient = Real (final quantity=
            "CubicExpansionCoefficient", final unit="1/K");

    type RelativePressureCoefficient = Real (final quantity=
            "RelativePressureCoefficient", final unit="1/K");

    type Compressibility = Real (final quantity="Compressibility", final unit=
            "1/Pa");

    type IsothermalCompressibility = Compressibility;

    type ThermalConductivity = Real (final quantity="ThermalConductivity", final unit=
               "W/(m.K)");

    type SpecificHeatCapacity = Real (final quantity="SpecificHeatCapacity",
          final unit="J/(kg.K)");

    type RatioOfSpecificHeatCapacities = Real (final quantity=
            "RatioOfSpecificHeatCapacities", final unit="1");

    type SpecificEntropy = Real (final quantity="SpecificEntropy",
                                 final unit="J/(kg.K)");

    type SpecificEnergy = Real (final quantity="SpecificEnergy",
                                final unit="J/kg");

    type SpecificEnthalpy = SpecificEnergy;

    type DerDensityByEnthalpy = Real (final unit="kg.s2/m5");

    type DerDensityByPressure = Real (final unit="s2/m2");

    type DerDensityByTemperature = Real (final unit="kg/(m3.K)");

    type DerEnthalpyByPressure = Real (final unit="J.m.s2/kg2");

    type MolarMass = Real (final quantity="MolarMass", final unit="kg/mol",min=0);

    type MolarVolume = Real (final quantity="MolarVolume", final unit="m3/mol", min=0);

    type MassFraction = Real (final quantity="MassFraction", final unit="1",
                              min=0, max=1);

    type PrandtlNumber = Real (final quantity="PrandtlNumber", final unit="1");

    type PerUnit = Real(unit = "1");
    annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
              -100},{100,100}}), graphics={
          Line(
            points={{-66,78},{-66,-40}},
            color={64,64,64},
            smooth=Smooth.None),
          Ellipse(
            extent={{12,36},{68,-38}},
            lineColor={64,64,64},
            fillColor={175,175,175},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-74,78},{-66,-40}},
            lineColor={64,64,64},
            fillColor={175,175,175},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-66,-4},{-66,6},{-16,56},{-16,46},{-66,-4}},
            lineColor={64,64,64},
            smooth=Smooth.None,
            fillColor={175,175,175},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{-46,16},{-40,22},{-2,-40},{-10,-40},{-46,16}},
            lineColor={64,64,64},
            smooth=Smooth.None,
            fillColor={175,175,175},
            fillPattern=FillPattern.Solid),
          Ellipse(
            extent={{22,26},{58,-28}},
            lineColor={64,64,64},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Polygon(
            points={{68,2},{68,-46},{64,-60},{58,-68},{48,-72},{18,-72},{18,-64},
                {46,-64},{54,-60},{58,-54},{60,-46},{60,-26},{64,-20},{68,-6},{68,
                2}},
            lineColor={64,64,64},
            smooth=Smooth.Bezier,
            fillColor={175,175,175},
            fillPattern=FillPattern.Solid)}), Documentation(info="<html>
<p>This package provides predefined types, such as <i>Mass</i>,
<i>Angle</i>, <i>Time</i>, based on the international standard
on units, e.g.,
</p>

<pre>   <b>type</b> Angle = Real(<b>final</b> quantity = \"Angle\",
                     <b>final</b> unit     = \"rad\",
                     displayUnit    = \"deg\");
</pre>

<p>
as well as conversion functions from non SI-units to SI-units
and vice versa in subpackage
<a href=\"modelica://Modelica.SIunits.Conversions\">Conversions</a>.
</p>

<p>
For an introduction how units are used in the Modelica standard library
with package SIunits, have a look at:
<a href=\"modelica://Modelica.SIunits.UsersGuide.HowToUseSIunits\">How to use SIunits</a>.
</p>

<p>
Copyright &copy; 1998-2013, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>
</html>",   revisions="<html>
<ul>
<li><i>May 25, 2011</i> by Stefan Wischhusen:<br/>Added molar units for energy and enthalpy.</li>
<li><i>Jan. 27, 2010</i> by Christian Kral:<br/>Added complex units.</li>
<li><i>Dec. 14, 2005</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br/>Add User&#39;;s Guide and removed &quot;min&quot; values for Resistance and Conductance.</li>
<li><i>October 21, 2002</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a> and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br/>Added new package <b>Conversions</b>. Corrected typo <i>Wavelenght</i>.</li>
<li><i>June 6, 2000</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br/>Introduced the following new types<br/>type Temperature = ThermodynamicTemperature;<br/>types DerDensityByEnthalpy, DerDensityByPressure, DerDensityByTemperature, DerEnthalpyByPressure, DerEnergyByDensity, DerEnergyByPressure<br/>Attribute &quot;final&quot; removed from min and max values in order that these values can still be changed to narrow the allowed range of values.<br/>Quantity=&quot;Stress&quot; removed from type &quot;Stress&quot;, in order that a type &quot;Stress&quot; can be connected to a type &quot;Pressure&quot;.</li>
<li><i>Oct. 27, 1999</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br/>New types due to electrical library: Transconductance, InversePotential, Damping.</li>
<li><i>Sept. 18, 1999</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br/>Renamed from SIunit to SIunits. Subpackages expanded, i.e., the SIunits package, does no longer contain subpackages.</li>
<li><i>Aug 12, 1999</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br/>Type &quot;Pressure&quot; renamed to &quot;AbsolutePressure&quot; and introduced a new type &quot;Pressure&quot; which does not contain a minimum of zero in order to allow convenient handling of relative pressure. Redefined BulkModulus as an alias to AbsolutePressure instead of Stress, since needed in hydraulics.</li>
<li><i>June 29, 1999</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br/>Bug-fix: Double definition of &quot;Compressibility&quot; removed and appropriate &quot;extends Heat&quot; clause introduced in package SolidStatePhysics to incorporate ThermodynamicTemperature.</li>
<li><i>April 8, 1998</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a> and Astrid Jaschinski:<br/>Complete ISO 31 chapters realized.</li>
<li><i>Nov. 15, 1997</i> by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a> and <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>:<br/>Some chapters realized.</li>
</ul>
</html>"));
  end SIunits;
annotation (
preferredView="info",
version="3.2.1",
versionBuild=2,
versionDate="2013-08-14",
dateModified = "2013-08-14 08:44:41Z",
revisionId="$Id:: package.mo 6947 2013-08-23 07:41:37Z #$",
uses(Complex(version="3.2.1"), ModelicaServices(version="3.2.1")),
conversion(
 noneFromVersion="3.2",
 noneFromVersion="3.1",
 noneFromVersion="3.0.1",
 noneFromVersion="3.0",
 from(version="2.1", script="modelica://Modelica/Resources/Scripts/Dymola/ConvertModelica_from_2.2.2_to_3.0.mos"),
 from(version="2.2", script="modelica://Modelica/Resources/Scripts/Dymola/ConvertModelica_from_2.2.2_to_3.0.mos"),
 from(version="2.2.1", script="modelica://Modelica/Resources/Scripts/Dymola/ConvertModelica_from_2.2.2_to_3.0.mos"),
 from(version="2.2.2", script="modelica://Modelica/Resources/Scripts/Dymola/ConvertModelica_from_2.2.2_to_3.0.mos")),
Icon(coordinateSystem(extent={{-100.0,-100.0},{100.0,100.0}}), graphics={
  Polygon(
    origin={-6.9888,20.048},
    fillColor={0,0,0},
    pattern=LinePattern.None,
    fillPattern=FillPattern.Solid,
    points={{-93.0112,10.3188},{-93.0112,10.3188},{-73.011,24.6},{-63.011,31.221},{-51.219,36.777},{-39.842,38.629},{-31.376,36.248},{-25.819,29.369},{-24.232,22.49},{-23.703,17.463},{-15.501,25.135},{-6.24,32.015},{3.02,36.777},{15.191,39.423},{27.097,37.306},{32.653,29.633},{35.035,20.108},{43.501,28.046},{54.085,35.19},{65.991,39.952},{77.897,39.688},{87.422,33.338},{91.126,21.696},{90.068,9.525},{86.099,-1.058},{79.749,-10.054},{71.283,-21.431},{62.816,-33.337},{60.964,-32.808},{70.489,-16.14},{77.368,-2.381},{81.072,10.054},{79.749,19.05},{72.605,24.342},{61.758,23.019},{49.587,14.817},{39.003,4.763},{29.214,-6.085},{21.012,-16.669},{13.339,-26.458},{5.401,-36.777},{-1.213,-46.037},{-6.24,-53.446},{-8.092,-52.387},{-0.684,-40.746},{5.401,-30.692},{12.81,-17.198},{19.424,-3.969},{23.658,7.938},{22.335,18.785},{16.514,23.283},{8.047,23.019},{-1.478,19.05},{-11.267,11.113},{-19.734,2.381},{-29.259,-8.202},{-38.519,-19.579},{-48.044,-31.221},{-56.511,-43.392},{-64.449,-55.298},{-72.386,-66.939},{-77.678,-74.612},{-79.53,-74.083},{-71.857,-61.383},{-62.861,-46.037},{-52.278,-28.046},{-44.869,-15.346},{-38.784,-2.117},{-35.344,8.731},{-36.403,19.844},{-42.488,23.813},{-52.013,22.49},{-60.744,16.933},{-68.947,10.054},{-76.884,2.646},{-93.0112,-12.1707},{-93.0112,-12.1707}},
    smooth=Smooth.Bezier),
  Ellipse(
    origin={40.8208,-37.7602},
    fillColor={161,0,4},
    pattern=LinePattern.None,
    fillPattern=FillPattern.Solid,
    extent={{-17.8562,-17.8563},{17.8563,17.8562}})}),
Documentation(info="<HTML>
<p>
Package <b>Modelica&reg;</b> is a <b>standardized</b> and <b>free</b> package
that is developed together with the Modelica&reg; language from the
Modelica Association, see
<a href=\"https://www.Modelica.org\">https://www.Modelica.org</a>.
It is also called <b>Modelica Standard Library</b>.
It provides model components in many domains that are based on
standardized interface definitions. Some typical examples are shown
in the next figure:
</p>

<p>
<img src=\"modelica://Modelica/Resources/Images/UsersGuide/ModelicaLibraries.png\">
</p>

<p>
For an introduction, have especially a look at:
</p>
<ul>
<li> <a href=\"modelica://Modelica.UsersGuide.Overview\">Overview</a>
  provides an overview of the Modelica Standard Library
  inside the <a href=\"modelica://Modelica.UsersGuide\">User's Guide</a>.</li>
<li><a href=\"modelica://Modelica.UsersGuide.ReleaseNotes\">Release Notes</a>
 summarizes the changes of new versions of this package.</li>
<li> <a href=\"modelica://Modelica.UsersGuide.Contact\">Contact</a>
  lists the contributors of the Modelica Standard Library.</li>
<li> The <b>Examples</b> packages in the various libraries, demonstrate
  how to use the components of the corresponding sublibrary.</li>
</ul>

<p>
This version of the Modelica Standard Library consists of
</p>
<ul>
<li><b>1360</b> models and blocks, and</li>
<li><b>1280</b> functions</li>
</ul>
<p>
that are directly usable (= number of public, non-partial classes). It is fully compliant
to <a href=\"https://www.modelica.org/documents/ModelicaSpec32Revision2.pdf\">Modelica Specification Version 3.2 Revision 2</a>
and it has been tested with Modelica tools from different vendors.
</p>

<p>
<b>Licensed by the Modelica Association under the Modelica License 2</b><br>
Copyright &copy; 1998-2013, ABB, AIT, T.&nbsp;B&ouml;drich, DLR, Dassault Syst&egrave;mes AB, Fraunhofer, A.Haumer, ITI, Modelon,
TU Hamburg-Harburg, Politecnico di Milano, XRG Simulation.
</p>

<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your own risk</u>; it can be redistributed and/or modified under the terms of the Modelica License 2. For license conditions (including the disclaimer of warranty) see <a href=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">Modelica.UsersGuide.ModelicaLicense2</a> or visit <a href=\"https://www.modelica.org/licenses/ModelicaLicense2\"> https://www.modelica.org/licenses/ModelicaLicense2</a>.</i>
</p>

<p>
<b>Modelica&reg;</b> is a registered trademark of the Modelica Association.
</p>
</html>"));
end Modelica;

package Cryogenics "Library of components for cryogenic circuit models"
  import SI = Modelica.SIunits;

  package Media "Medium models"
    extends Modelica.Icons.MaterialPropertiesPackage;

    package Helium
    "Helium model from NIST RefProp database via ExternalMedia/FluidProp"
      extends ExternalMedia.Media.FluidPropMedium(
        mediumName="Helium",
        libraryName="FluidProp.RefProp",
        substanceNames={"He"},
        ThermoStates=Modelica.Media.Interfaces.Choices.IndependentVariables.ph,
        AbsolutePressure(
          min=500,
          max=44e5,
          nominal=1e5,
          start=1e5),
        Density(
          min=0.1,
          max=200,
          nominal=100,
          start=100),
        SpecificEnthalpy(
          min=-6000,
          max=1.7e6,
          nominal=1000,
          start=0),
        SpecificEntropy(
          min=-4000,
          max=30e3,
          nominal=1000,
          start=0),
        Temperature(
          min=2.17,
          max=310,
          nominal=10,
          start=5,
          displayUnit="K"));

      extends Internals.HeliumMedium;

      redeclare function getMolarMass
        output MolarMass MM "molar mass";
      algorithm
        MM := 4.002602e-3;
      end getMolarMass;

      redeclare function getCriticalTemperature
        output Temperature Tc "critical temperature";
      algorithm
        Tc := 5.1953;
      end getCriticalTemperature;

      redeclare function getCriticalPressure
        output AbsolutePressure pc "critical pressure";
      algorithm
        pc := 2.2746e5;
      end getCriticalPressure;

      redeclare function getCriticalMolarVolume
        output MolarVolume vc "critical molar volume";
      algorithm
        vc := 4.002602e-3/69.641;
      end getCriticalMolarVolume;

    end Helium;

    package HeliumHelmholtz
    "Helium medium from NIST RefProp database via HelmoltzMedia"
      extends HelmholtzMedia.HelmholtzFluids.Helium(
        AbsolutePressure(
          min=500,
          max=44e5,
          nominal=1e5,
          start=1e5),
        Density(
          min=0.1,
          max=200,
          nominal=100,
          start=100),
        SpecificEnthalpy(
          min=-6000,
          max=1.7e6,
          nominal=1000,
          start=0),
        SpecificEntropy(
          min=-4000,
          max=30e3,
          nominal=1000,
          start=0),
        Temperature(
          min=2.17,
          max=310,
          nominal=10,
          start=5,
          displayUnit="K"));

      extends Internals.HeliumMedium;

      redeclare function dynamicViscosity "Return dynamic viscosity"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "Thermodynamic state record";
        output DynamicViscosity eta "Dynamic viscosity";
      algorithm
        eta := 3.332e-6 - (state.T-4.0)*(3.332e-6 - 3.260e-6);
      end dynamicViscosity;

      redeclare function thermalConductivity "  Return thermal conductivity"
      extends Modelica.Icons.Function;
      input ThermodynamicState state "Thermodynamic state record";
      output ThermalConductivity lambda "Thermal conductivity";
      algorithm
        lambda := 0.018650;
      end thermalConductivity;

    end HeliumHelmholtz;
  end Media;

  package Components "Components for cryogenic circuit models"

    model System
      extends Modelica.Fluid.System(allowFlowReversal=true);
      extends ThermoPower.System(allowFlowReversal=true);
      annotation (defaultComponentName="system", defaultComponentPrefixes=
            "inner");

    end System;

    model PressureSource "Ideal pressure source"
      extends Modelica.Fluid.Sources.Boundary_pT(
        redeclare replaceable package Medium = Media.Helium constrainedby
        Internals.HeliumMedium,
        final use_X_in=false,
        final use_C_in=false,
        final X=Medium.X_default,
        final C=fill(0, Medium.nC),
        T(displayUnit="K"));
    end PressureSource;

    model FlowSource "Ideal flow source"
      extends Modelica.Fluid.Sources.MassFlowSource_T(
        redeclare replaceable package Medium = Media.Helium constrainedby
        Internals.HeliumMedium,
        final use_X_in=false,
        final use_C_in=false,
        final X=Medium.X_default,
        final C=fill(0, Medium.nC),
        T(displayUnit="K"));

    end FlowSource;

    model CentrifugalPump
    "Centrifugal pump with ideally controlled speed and replaceable function characteristics"
      extends Internals.PumpBase(
        redeclare replaceable function flowCharacteristicSpecificWork =
            Functions.PumpCharacteristics.linearFlow (q_nom={0,1}, H_nom={0,1}),
        final Np0=1);

      import Modelica.SIunits.Conversions.NonSIunits.*;
      parameter AngularVelocity_rpm n_const=n0 "Constant rotational speed";
      SI.SpecificEnergy Y = head*g "Specific work made on the fluid";
      Modelica.Blocks.Interfaces.RealInput in_n "RPM" annotation (Placement(
            transformation(
            origin={-26,80},
            extent={{-10,-10},{10,10}},
            rotation=270)));
    equation
      n = in_n "Rotational speed";
      if cardinality(in_n) == 0 then
        in_n = n_const "Rotational speed provided by parameter";
      end if;
      annotation (
        Icon(graphics={Text(extent={{-58,94},{-30,74}}, textString="n"),Text(
              extent={{-10,102},{18,82}}, textString="Np")}),
        Diagram(graphics),
        Documentation(info="<HTML>
<p>This model describes a centrifugal pump (or a group of <tt>Np</tt> pumps in parallel) with controlled speed, either fixed or provided by an external signal.
<p>The model extends <tt>PumpBase</tt>
<p>If the <tt>in_n</tt> input connector is wired, it provides rotational speed of the pumps (rpm); otherwise, a constant rotational speed equal to <tt>n_const</tt> (which can be different from <tt>n0</tt>) is assumed.</p>
</HTML>", revisions="<html>
<ul>
<li><i>5 Jul 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Model restructured by using inheritance. Adapted to Modelica.Media.</li>
<li><i>15 Jan 2004</i>
    by <a href=\"mailto:francesco.schiavo@polimi.it\">Francesco Schiavo</a>:<br>
       <tt>ThermalCapacity</tt> and <tt>CheckValve</tt> added.</li>
<li><i>15 Dec 2003</i>
    by <a href=\"mailto:francesco.schiavo@polimi.it\">Francesco Schiavo</a>:<br>
       First release.</li>
</ul>
</html>"));
    end CentrifugalPump;
  end Components;

  package Test "Test models"

    model CentrifugalPump

      Components.CentrifugalPump pump(
        allowFlowReversal=false,
        redeclare function flowCharacteristicSpecificWork =
            Cryogenics.Functions.PumpCharacteristics.quadraticFlow (q_nom={43,
                80,100}*1e-3/135, H_nom={1.19,1.09,0.96}*1e5/135),
        redeclare function efficiencyCharacteristic =
            Functions.PumpCharacteristics.constantEfficiency (eta_nom=0.8),
        rho0=135,
        n0=9600,
        w0=50e-3,
        dp0=100000,
      redeclare package Medium = Cryogenics.Media.HeliumHelmholtz)
        annotation (Placement(transformation(extent={{-20,-4},{20,36}})));
      inner Components.System system
        annotation (Placement(transformation(extent={{80,80},{100,100}})));
      Components.FlowSource flowSource(nPorts=1, use_m_flow_in=true,
      redeclare package Medium = Cryogenics.Media.HeliumHelmholtz)
        annotation (Placement(transformation(extent={{60,10},{40,30}})));
      Components.PressureSource pressureSource(
        nPorts=1,
        T(displayUnit="K") = 4.5,
        p=450000,
      redeclare package Medium = Cryogenics.Media.HeliumHelmholtz)
        annotation (Placement(transformation(extent={{-60,10},{-40,30}})));
      Modelica.Blocks.Sources.Ramp massFlow(
        height=80e-3,
        duration=1,
        offset=20e-3)
        annotation (Placement(transformation(extent={{-20,60},{0,80}})));
      Modelica.Blocks.Math.Gain gain(k=-1)
        annotation (Placement(transformation(extent={{22,60},{42,80}})));
      Modelica.Blocks.Sources.Constant pump_rpm(k=9600)
        annotation (Placement(transformation(extent={{-80,40},{-60,60}})));
    equation
      connect(pump.outfl, flowSource.ports[1]) annotation (Line(
          points={{12,30},{32,30},{32,20},{40,20}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(pressureSource.ports[1], pump.infl) annotation (Line(
          points={{-40,20},{-16,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(massFlow.y, gain.u) annotation (Line(
          points={{1,70},{20,70}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(gain.y, flowSource.m_flow_in) annotation (Line(
          points={{43,70},{90,70},{90,28},{60,28}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump_rpm.y, pump.in_n) annotation (Line(
          points={{-59,50},{-6,50},{-6,32},{-5.2,32}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (Diagram(graphics));
    end CentrifugalPump;
  end Test;

  package Internals "Intermediate class definitions"
    extends Modelica.Icons.InternalPackage;

    partial model PumpBase "Base model for centrifugal pumps"
      extends ThermoPower.Water.BaseClasses.PumpBase(
        redeclare replaceable package Medium = Media.Helium constrainedby
        Internals.HeliumMedium,
        redeclare function flowCharacteristic = equivalentFlowCharacteristic,
        Qloss = Qnom);

      parameter SI.Power Qnom=0 "Nominal heat loss to ambient"
        annotation (Dialog(group="Parameters"), Evaluate=true);

      replaceable function flowCharacteristicSpecificWork =
          Cryogenics.Functions.PumpCharacteristics.quadraticFlow constrainedby
      Cryogenics.Functions.PumpCharacteristics.baseFlow
      "Specific energy vs. q_flow characteristic at nominal speed and density"
        annotation (Dialog(group="Characteristics"),choicesAllMatching=true);

      function equivalentFlowCharacteristic
        extends ThermoPower.Functions.PumpCharacteristics.baseFlow;
      algorithm
        head :=flowCharacteristicSpecificWork(q_flow)/g;
        annotation(Inline = true);
      end equivalentFlowCharacteristic;
      annotation (
        Icon(graphics),
        Diagram(graphics),
        Documentation(info="<HTML>
<p>This is the base model for the <tt>Pump</tt> and <tt>
PumpMech</tt> pump models.
<p>The model describes a centrifugal pump, or a group of <tt>Np</tt> identical pumps in parallel. The pump model is based on the theory of kinematic similarity: the pump characteristics are given for nominal operating conditions (rotational speed and fluid density), and then adapted to actual operating condition, according to the similarity equations. 
<p>In order to avoid singularities in the computation of the outlet enthalpy at zero flowrate, the thermal capacity of the fluid inside the pump body can be taken into account.
<p>The model can either support reverse flow conditions or include a built-in check valve to avoid flow reversal.
<p><b>Modelling options</b></p>
<p> The nominal hydraulic characteristic (head vs. volume flow rate) is given by the the replaceable function <tt>flowCharacteristic</tt>. 
<p> The pump energy balance can be specified in two alternative ways:
<ul>
<li><tt>usePowerCharacteristic = false</tt> (default option): the replaceable function <tt>efficiencyCharacteristic</tt> (efficiency vs. volume flow rate in nominal conditions) is used to determine the efficiency, and then the power consumption. The default is a constant efficiency of 0.8.
<li><tt>usePowerCharacteristic = true</tt>: the replaceable function <tt>powerCharacteristic</tt> (power consumption vs. volume flow rate in nominal conditions) is used to determine the power consumption, and then the efficiency.
</ul>
<p>
Several functions are provided in the package <tt>Functions.PumpCharacteristics</tt> to specify the characteristics as a function of some operating points at nominal conditions.
<p>Depending on the value of the <tt>checkValve</tt> parameter, the model either supports reverse flow conditions, or includes a built-in check valve to avoid flow reversal.
 
<p>If the <tt>in_Np</tt> input connector is wired, it provides the number of pumps in parallel; otherwise,  <tt>Np0</tt> parallel pumps are assumed.</p>
<p>It is possible to take into account the heat capacity of the fluid inside the pump by specifying its volume <tt>V</tt> at nominal conditions; this is necessary to avoid singularities in the computation of the outlet enthalpy in case of zero flow rate. If zero flow rate conditions are always avoided, this dynamic effect can be neglected by leaving the default value <tt>V = 0</tt>, thus avoiding a fast state variable in the model.
<p>The <tt>CheckValve</tt> parameter determines whether the pump has a built-in check valve or not.
<p>If <tt>computeNPSHa = true</tt>, the available net positive suction head is also computed; this requires a two-phase medium model to provide the fluid saturation pressure.
</HTML>", revisions="<html>
<ul>
<li><i>31 Oct 2006</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
      Added initialisation parameter <tt>wstart</tt>.</li>
<li><i>5 Nov 2005</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
      Model restructured according to kinematic similarity theory.<br>
      Characteristics now specified by replaceable functions.</li>
<li><i>6 Apr 2005</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       <tt>CharData</tt> substituted by <tt>OpPoints</tt></li>
<li><i>16 Dec 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Standard medium definition added.</li>
<li><i>2 Aug 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Optional NPSHa computation added. Changed parameter names</li>
<li><i>5 Jul 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Model restructured by using inheritance. Adapted to Modelica.Media.</li>
<li><i>15 Jan 2004</i>
    by <a href=\"mailto:francesco.schiavo@polimi.it\">Francesco Schiavo</a>:<br>
       <tt>ThermalCapacity</tt> and <tt>CheckValve</tt> added.</li>
<li><i>15 Dec 2003</i>
    by <a href=\"mailto:francesco.schiavo@polimi.it\">Francesco Schiavo</a>:<br>
       First release.</li>
</ul>
</html>"));
    end PumpBase;

    partial package HeliumMedium
    "Base class for helium medium models that can be used by Cryogenics"
    end HeliumMedium;
  end Internals;

  package Functions
    extends Modelica.Icons.UtilitiesPackage;

    package PumpCharacteristics "Functions for pump characteristics"
      import NonSI = Modelica.SIunits.Conversions.NonSIunits;

      partial function baseFlow "Base class for pump flow characteristics"
        extends Modelica.Icons.Function;
        input SI.VolumeFlowRate q_flow "Volumetric flow rate";
        output SI.SpecificEnergy H "Specific Energy";
      end baseFlow;

      partial function baseEfficiency
      "Base class for efficiency characteristics"
        extends Modelica.Icons.Function;
        input Modelica.SIunits.VolumeFlowRate q_flow "Volumetric flow rate";
        output Real eta "Efficiency";
      end baseEfficiency;

      function linearFlow "Linear flow characteristic"
        extends baseFlow;
        parameter SI.VolumeFlowRate q_nom[2]
        "Volume flow rate for two operating points (single pump)";
        parameter SI.SpecificEnergy H_nom[2]
        "Specific energy for two operating points";
        /* Linear system to determine the coefficients:
  H_nom[1] = c[1] + q_nom[1]*c[2];
  H_nom[2] = c[1] + q_nom[2]*c[2];
  */
    protected
        parameter Real c[2]=Modelica.Math.Matrices.solve([ones(2), q_nom],
            H_nom) "Coefficients of linear head curve";
      algorithm
        // Flow equation: head = q*c[1] + c[2];
        H := c[1] + q_flow*c[2];
        annotation (smoothOrder=2);
      end linearFlow;

      function quadraticFlow "Quadratic flow characteristic"
        extends baseFlow;
        parameter SI.VolumeFlowRate q_nom[3]
        "Volume flow rate for three operating points (single pump)";
        parameter SI.SpecificEnergy H_nom[3]
        "Specific energy for three operating points";
    protected
        parameter Real q_nom2[3]={q_nom[1]^2,q_nom[2]^2,q_nom[3]^2}
        "Squared nominal flow rates";
        /* Linear system to determine the coefficients:
  H_nom[1] = c[1] + q_nom[1]*c[2] + q_nom[1]^2*c[3];
  H_nom[2] = c[1] + q_nom[2]*c[2] + q_nom[2]^2*c[3];
  H_nom[3] = c[1] + q_nom[3]*c[2] + q_nom[3]^2*c[3];
  */
        parameter Real c[3]=Modelica.Math.Matrices.solve([ones(3), q_nom,
            q_nom2], H_nom) "Coefficients of quadratic head curve";
      algorithm
        // Flow equation: head = c[1] + q_flow*c[2] + q_flow^2*c[3];
        H := c[1] + q_flow*c[2] + q_flow^2*c[3];
      end quadraticFlow;

      function constantEfficiency "Constant efficiency characteristic"
        extends baseEfficiency;
        parameter Real eta_nom "Nominal efficiency";
      algorithm
        eta := eta_nom;
      end constantEfficiency;
    end PumpCharacteristics;
  end Functions;
  annotation (uses(Modelica(version="3.2.1"), ThermoPower(version="3.1")));
end Cryogenics;
model Cryogenics_Test_CentrifugalPump
 extends Cryogenics.Test.CentrifugalPump;
  annotation(experiment(
    StopTime=1,
    __Dymola_NumberOfIntervals=500,
    Tolerance=0.0001,
    __Dymola_Algorithm="dassl"));
end Cryogenics_Test_CentrifugalPump;
