within ;
package TestArrays
  import SI = Modelica.SIunits;
  constant Real pi = Modelica.Constants.pi;
  model Tube
    extends ThermoPower.Icons.Gas.Tube;

    replaceable package Medium = Modelica.Media.IdealGases.SingleGases.He (
            AbsolutePressure(
              min=1e4,
              max=1e9,
              nominal=8e6,
              start=8e6),
            Density(
              min=0.1,
              max=200,
              nominal=7,
              start=7),
            SpecificEnthalpy(
              min=-6000,
              max=6e6,
              nominal=3e6,
              start=3e6),
            SpecificEntropy(
              min=-4000,
              max=50e3,
              nominal=2.23e4,
              start=2.23e4),
            Temperature(
              min=273.15,
              max=1273.15,
              nominal=573.15,
              start=573.15,
              displayUnit="degC"),
            p_default=80e5)
      annotation(choicesAllMatching = true);
    replaceable model HeatTransfer =
        ThermoPower.Thermal.HeatTransferFV.IdealHeatTransfer
      constrainedby ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
      annotation(choicesAllMatching = true,
                 Dialog(group="Modelling assumptions"));

    import ThermoPower.Choices.Flow1D.FFtypes;
    import ThermoPower.Choices.Flow1D.HCtypes;

    outer ThermoPower.System system "System wide properties";

    parameter Integer N(min=2) = 2 "Number of nodes for thermal variables"
      annotation (Dialog(group="Geometrical data"));
    parameter Integer Nw = N - 1 "Number of volumes on the wall interface"
      annotation (Dialog(group="Geometrical data"));
    parameter Integer Nt=1 "Number of tubes in parallel"
      annotation (Dialog(group="Geometrical data"));
    parameter SI.Distance L "Length"
      annotation (Evaluate=true, Dialog(group="Geometrical data"));
    parameter SI.Position H=0 "Elevation of outlet over inlet"
      annotation (Evaluate=true, Dialog(group="Geometrical data"));
    parameter SI.Diameter Dhyd(displayUnit="mm") "Hydraulic diameter"
      annotation (Dialog(group="Geometrical data"));
    parameter SI.Length omega(displayUnit="mm") = pi*Dhyd "Heated perimeter"
      annotation (Dialog(group="Geometrical data"));
    parameter SI.Area A = pi*Dhyd^2/4 "Transverse channel surface"
      annotation (Dialog(group="Geometrical data"));
    parameter SI.MassFlowRate wnom "Nominal mass flowrate (total)"
      annotation (Dialog(group="Operating point data"));
    parameter FFtypes FFtype=FFtypes.NoFriction "Friction Factor Type"
      annotation (Evaluate=true, Dialog(group="Friction model"));
    parameter Real Kfnom(
      unit="Pa.kg/(m3.kg2/s2)",
      min=0) = 0 "Nominal hydraulic resistance coefficient" annotation (
        Dialog(group="Friction model", enable=(FFtype == FFtypes.Kfnom)));
    parameter SI.PressureDifference dpnom=0
      "Nominal pressure drop (friction term only!)"
      annotation (Dialog(group="Operating point data"));
    parameter SI.Density rhonom=0 "Nominal inlet density" annotation (Dialog(
          group="Friction model", enable=(FFtype == FFtypes.OpPoint)));
    parameter Real Cfnom=0 "Nominal Fanning friction factor" annotation (
        Dialog(group="Friction model", enable=(FFtype == FFtypes.Cfnom)));
    parameter Real e=0 "Relative roughness (ratio roughness/diameter)"
      annotation (Dialog(group="Friction model", enable=(FFtype == FFtypes.Colebrook)));
    parameter Boolean DynamicMomentum=false "Inertial phenomena accounted for"
      annotation (Evaluate=true, Dialog(group="Modelling assumptions"));
    parameter Boolean UniformComposition=true
      "Uniform gas composition is assumed" annotation (Evaluate=true);
    parameter Boolean QuasiStatic=false
      "Quasi-static model (mass, energy and momentum static balances"
      annotation (Evaluate=true);
    parameter HCtypes HydraulicCapacitance=HCtypes.Downstream
      "Location of the hydraulic capacitance"
      annotation (Dialog(group="Modelling assumptions"));
    parameter Boolean avoidInletEnthalpyDerivative=true
      "Avoid inlet enthalpy derivative"
      annotation (Dialog(group="Modelling assumptions"));
    parameter Boolean allowFlowReversal=system.allowFlowReversal
      "= true to allow flow reversal, false restricts to design direction"
      annotation (Dialog(group="Modelling assumptions"));
    parameter Boolean idealGasDensityDistribution = true
      "Assume ideal-gas-type density distributions for mass balances"
      annotation (Dialog(group="Modelling assumptions"));
    parameter SI.Pressure pstart(displayUnit="bar")=8e6 "Pressure start value"
      annotation (Dialog(tab="Initialisation"));
    parameter Medium.Temperature Tstartin(displayUnit="degC") = 573.15
      "Inlet temperature start value" annotation (Dialog(tab="Initialisation"));
    parameter Medium.Temperature Tstartout(displayUnit="degC") = 573.15
      "Outlet temperature start value" annotation (Dialog(tab="Initialisation"));
    parameter Medium.Temperature Tstart[N]=linspace(
            Tstartin,
            Tstartout,
            N) "Start value of temperature vector (initialized by default)"
      annotation (Dialog(tab="Initialisation"));
    parameter Real wnf=if dpnom > 1000 then 0.02 else if dpnom > 100 then
        0.05 else if dpnom > 10 then 1 else 5
      "Fraction of nominal flow rate at which linear friction equals turbulent friction"
      annotation (Dialog(group="Modelling assumptions"));
    parameter Real Kfc=1 "Friction factor correction coefficient"
      annotation (Dialog(group="Friction model"));
    parameter ThermoPower.Choices.Init.Options initOpt=ThermoPower.Choices.Init.Options.noInit
      "Initialisation option" annotation (Dialog(tab="Initialisation"));
    final parameter Medium.SpecificEnthalpy hstart[N]={
        Medium.specificEnthalpy_pT(pstart, Tstart[i]) for i in 1:N};
    replaceable function Cfcorr =
        ThermoPower.Functions.FrictionFactors.NoFriction constrainedby
      ThermoPower.Functions.FrictionFactors.BaseFFcorr
      "External Fanning friction factor correlation"
      annotation(choicesAllMatching=true, Dialog(group = "Friction model",
            enable = (FFtype == FFtypes.External)));

    ThermoPower.Gas.Flow1DFV flow1DFV1(
      redeclare package Medium = Medium,
      redeclare model HeatTransfer = HeatTransfer,
      redeclare function Cfcorr = Cfcorr,
      final N=N,
      final Nw=Nw,
      final initOpt=initOpt,
      final pstart=pstart,
      final Nt=Nt,
      final L=L,
      final H=H,
      final A=A,
      final omega=omega,
      final Dhyd=Dhyd,
      final wnom=wnom,
      final FFtype=FFtype,
      final Kfnom=Kfnom,
      final dpnom=dpnom,
      final rhonom=rhonom,
      final Cfnom=Cfnom,
      final e=e,
      final DynamicMomentum=DynamicMomentum,
      final HydraulicCapacitance=HydraulicCapacitance,
      final UniformComposition=UniformComposition,
      final QuasiStatic=QuasiStatic,
      final avoidInletEnthalpyDerivative=avoidInletEnthalpyDerivative,
      final allowFlowReversal=allowFlowReversal,
      final wnf=wnf,
      final Kfc=Kfc,
      final noInitialPressure=true,
      final Tstartbar=(Tstartin + Tstartout)/2) annotation(Placement(visible = true, transformation(origin = {0, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    ThermoPower.Gas.FlangeA flangeA1(redeclare package Medium = Medium) annotation(Placement(visible = true, transformation(origin = {-100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {-100, 1.11022e-15}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
    ThermoPower.Gas.FlangeB flangeB1(redeclare package Medium = Medium) annotation(Placement(visible = true, transformation(origin = {100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {100, 0}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
    ThermoPower.Thermal.HeatSource1DFV heatSource1DFV1(Nw=Nw)
                                                       annotation(Placement(visible = true, transformation(origin = {0, 50}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    Modelica.Blocks.Interfaces.RealInput u annotation(Placement(visible = true, transformation(origin = {0, 100}, extent = {{-20, -20}, {20, 20}}, rotation = -90), iconTransformation(origin = {0, 40}, extent = {{-20, -20}, {20, 20}}, rotation = -90)));

  equation
    connect(u, heatSource1DFV1.power) annotation(Line(points = {{0, 100}, {0, 54}}, color = {0, 0, 127}));
    connect(heatSource1DFV1.wall, flow1DFV1.wall) annotation(Line(points={{0,47},{
            0,5}},                                                                            color = {255, 127, 0}));
    connect(flow1DFV1.outfl, flangeB1) annotation(Line(points={{10,0},{100,0}},                          color = {159, 159, 223}));
    connect(flangeA1, flow1DFV1.infl) annotation(Line(points={{-100,0},{-10,0}},                          color = {159, 159, 223}));
    annotation(Icon, Diagram);
  end Tube;

  model TubeScalar
    replaceable package Medium = Modelica.Media.IdealGases.SingleGases.He (
            AbsolutePressure(
              min=1e4,
              max=1e9,
              nominal=8e6,
              start=8e6),
            Density(
              min=0.1,
              max=200,
              nominal=7,
              start=7),
            SpecificEnthalpy(
              min=-6000,
              max=6e6,
              nominal=3e6,
              start=3e6),
            SpecificEntropy(
              min=-4000,
              max=50e3,
              nominal=2.23e4,
              start=2.23e4),
            Temperature(
              min=273.15,
              max=1273.15,
              nominal=573.15,
              start=573.15,
              displayUnit="degC"),
            p_default=80e5)
      annotation(choicesAllMatching = true);
    Tube tube(
      redeclare package Medium = Medium,
      N=4,
      L=1,
      wnom=0.079411,
      FFtype=ThermoPower.Choices.Flow1D.FFtypes.OpPoint,
      rhonom=6.762,
      Dhyd=0.0135,
      dpnom=78200)
              annotation (Placement(transformation(extent={{-10,-10},{10,10}})));
    ThermoPower.Gas.SourceMassFlow Source(
      use_in_T=false,
      redeclare package Medium = Medium,
      p0=8000000,
      T=573.15,
      w0=0.079411)           annotation (Placement(transformation(extent={{-90,-10},
              {-70,10}},     rotation=0)));
    ThermoPower.Gas.SinkPressure sinkPressure(
      redeclare package Medium = Medium,
      p0=7921800,
      T=573.15)
      annotation (Placement(transformation(extent={{70,-10},{90,10}})));
    inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
                                    annotation(Placement(transformation(extent={{80,80},
              {100,100}})));
    Modelica.Blocks.Sources.Step step6(each startTime=1, each height=1e4)
      annotation (Placement(transformation(extent={{-10,-10},{10,10}},
          rotation=-90,
          origin={0,60})));
  equation
    connect(Source.flange, tube.flangeA1)
      annotation (Line(points={{-70,0},{-40,0},{-10,0}}, color={159,159,223}));
    connect(tube.flangeB1, sinkPressure.flange)
      annotation (Line(points={{10,0},{40,0},{70,0}}, color={159,159,223}));
    connect(step6.y, tube.u)
      annotation (Line(points={{0,49},{0,4}}, color={0,0,127}));
    annotation (Icon(coordinateSystem(preserveAspectRatio=false)), Diagram(
          coordinateSystem(preserveAspectRatio=false)),
      experiment(StopTime=10),
      __Dymola_experimentSetupOutput);
  end TubeScalar;

  model TubeArray
    replaceable package Medium = Modelica.Media.IdealGases.SingleGases.He (
            AbsolutePressure(
              min=1e4,
              max=1e9,
              nominal=8e6,
              start=8e6),
            Density(
              min=0.1,
              max=200,
              nominal=7,
              start=7),
            SpecificEnthalpy(
              min=-6000,
              max=6e6,
              nominal=3e6,
              start=3e6),
            SpecificEntropy(
              min=-4000,
              max=50e3,
              nominal=2.23e4,
              start=2.23e4),
            Temperature(
              min=273.15,
              max=1273.15,
              nominal=573.15,
              start=573.15,
              displayUnit="degC"),
            p_default=80e5)
      annotation(choicesAllMatching = true);

      parameter Integer nch = 6;

    Tube tube[nch](
      redeclare each package Medium = Medium,
      each N=4,
      each L=1,
      each wnom=0.079411,
      each FFtype=ThermoPower.Choices.Flow1D.FFtypes.OpPoint,
      each rhonom=6.762,
      each Dhyd=0.0135,
      each dpnom=78200)
              annotation (Placement(transformation(extent={{-10,-10},{10,10}})));
    ThermoPower.Gas.SourceMassFlow Source(
      use_in_T=false,
      redeclare package Medium = Medium,
      p0=8000000,
      T=573.15,
      w0=0.079411*nch)       annotation (Placement(transformation(extent={{-90,-10},
              {-70,10}},     rotation=0)));
    ThermoPower.Gas.SinkPressure sinkPressure(
      redeclare package Medium = Medium,
      p0=7921800,
      T=573.15)
      annotation (Placement(transformation(extent={{70,-10},{90,10}})));
    inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
                                    annotation(Placement(transformation(extent={{80,80},
              {100,100}})));
    Modelica.Blocks.Sources.Step step6(               each startTime=1, each
        height=1e4)
      annotation (Placement(transformation(extent={{-10,-10},{10,10}},
          rotation=-90,
          origin={0,60})));
  equation
    for ii in 1:nch loop
      connect(Source.flange, tube[ii].flangeA1)
        annotation (Line(points={{-70,0},{-40,0},{-10,0}}, color={159,159,223}));
      connect(tube[ii].flangeB1, sinkPressure.flange)
        annotation (Line(points={{10,0},{40,0},{70,0}}, color={159,159,223}));
      connect(step6.y, tube[ii].u)
        annotation (Line(points={{0,49},{0,4}}, color={0,0,127}));
    end for;
    annotation (Icon(coordinateSystem(preserveAspectRatio=false)), Diagram(
          coordinateSystem(preserveAspectRatio=false)),
      experiment(StopTime=10),
      __Dymola_experimentSetupOutput);
  end TubeArray;

  model TubeMatrix
    replaceable package Medium = Modelica.Media.IdealGases.SingleGases.He (
            AbsolutePressure(
              min=1e4,
              max=1e9,
              nominal=8e6,
              start=8e6),
            Density(
              min=0.1,
              max=200,
              nominal=7,
              start=7),
            SpecificEnthalpy(
              min=-6000,
              max=6e6,
              nominal=3e6,
              start=3e6),
            SpecificEntropy(
              min=-4000,
              max=50e3,
              nominal=2.23e4,
              start=2.23e4),
            Temperature(
              min=273.15,
              max=1273.15,
              nominal=573.15,
              start=573.15,
              displayUnit="degC"),
            p_default=80e5)
      annotation(choicesAllMatching = true);

      parameter Integer nch1 = 6;
      parameter Integer nch2 = 5;

    Tube tube[nch1,nch2](
      redeclare each package Medium = Medium,
      each N=4,
      each L=1,
      each wnom=0.079411,
      each FFtype=ThermoPower.Choices.Flow1D.FFtypes.OpPoint,
      each rhonom=6.762,
      each Dhyd=0.0135,
      each dpnom=78200)
              annotation (Placement(transformation(extent={{-10,-10},{10,10}})));
    ThermoPower.Gas.SourceMassFlow Source(
      use_in_T=false,
      redeclare package Medium = Medium,
      p0=8000000,
      T=573.15,
      w0=0.079411*nch1*nch2)       annotation (Placement(transformation(extent={{-90,-10},
              {-70,10}},     rotation=0)));
    ThermoPower.Gas.SinkPressure sinkPressure(
      redeclare package Medium = Medium,
      p0=7921800,
      T=573.15)
      annotation (Placement(transformation(extent={{70,-10},{90,10}})));
    inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
                                    annotation(Placement(transformation(extent={{80,80},
              {100,100}})));
    Modelica.Blocks.Sources.Step step6(               each startTime=1, each
        height=1e4)
      annotation (Placement(transformation(extent={{-10,-10},{10,10}},
          rotation=-90,
          origin={0,60})));
  equation
    for ii in 1:nch1 loop
      for jj in 1:nch2 loop
        connect(Source.flange, tube[ii,jj].flangeA1)
          annotation (Line(points={{-70,0},{-40,0},{-10,0}}, color={159,159,223}));
        connect(tube[ii,jj].flangeB1, sinkPressure.flange)
          annotation (Line(points={{10,0},{40,0},{70,0}}, color={159,159,223}));
        connect(step6.y, tube[ii,jj].u)
          annotation (Line(points={{0,49},{0,4}}, color={0,0,127}));
      end for;
    end for;
    annotation (Icon(coordinateSystem(preserveAspectRatio=false)), Diagram(
          coordinateSystem(preserveAspectRatio=false)),
      experiment(StopTime=10),
      __Dymola_experimentSetupOutput);
  end TubeMatrix;
  annotation(Icon, Diagram,
    uses(ThermoPower(version="3.1"), Modelica(version="3.2.1")));
end TestArrays;
