within ;
package EHPT "Electric and Hybrid Power train library"
  //package Propulsion
  extends Modelica.Icons.Package;
  //end Propulsion;

  package MapBased
    "Contains map-based models of Internal combustion engines and electric drives"
    extends Modelica.Icons.Package;

    class Information
      extends Modelica.Icons.Information;
      annotation (Documentation(info="<html>
<p>The map-based folder contains simple model whose only dynamics is due to their mechanical inertia.</p>
<p>The ice model, since implemnts an Internal Combustion Engine, can deliber power, but never absorbs, while the other, two (&QUOT;oneFlange&QUOT; and &QUOT;twoFlange&QUOT;) simulate electric drive trains, i.e. the assembly of an electric machine and the corresponding ACDC converter, can absorb or deliver.</p>
<p>The input torque of the ice model is in Newton-metres, whil e in the other cases it is normalised: it is between -1 and +1, where -1 means maximum available torque to be absorbed, +1 to be delivered.</p>
<p>All the models come in two versions:</p>
<ul>
<li>version without &QUOT;Conn&QUOT; in the name: they get the input signals from a Real input connector. The input signal indicates a torque request.</li>
<li>version with &QUOT;Conn&QUOT; in the name: they exchange several quantities through an expandable connector.</li>
</ul>
<p><br><u>Names and meaning </u>of the pre-defined quantities circulating through the connection bus in the model versions having &QUOT;Conn&QUOT; in their names.</p>
<p>All the names are composed by two or three parts. The first one indicates the component to which the variable refers, (for instance &QUOT;ice&QUOT;) the central one which kind of variable it is (for instance &QUOT;tau&QUOT; means torque), finally the(optional) third part indicates info about de variable for instance &QUOT;del&QUOT; means delivered, i.e.for a power a positivive value indicates that the power is delivered to the outside by the component the variable belongs to, the opposite happens when the third part of the name in &QUOT;abs&QUOT; (stands for absorbed). Another exmple is &QUOT;Norm&QUOT;: this third part of a name indicates that the correspondng quantity is normalised (see above in this info).</p>
<p><br><u>Full list</u> of the variables defined or used in the component of the MAP-based folder (other names that are used in the bus in the FullVehicle Examples, will be discussed in the examples themselves):</p>
<table cellspacing=\"0\" cellpadding=\"0\" border=\"1\"><tr>
<td valign=\"top\"><p><br><i>Nome</i> </p></td>
<td valign=\"top\"><p>sender </p></td>
<td valign=\"top\"><p>users </p></td>
<td valign=\"top\"><p>comment</p></td>
</tr>
<tr>
<td valign=\"top\"><p>iceTauRef </p></td>
<td valign=\"top\"></td>
<td valign=\"top\"><p><br>Ice</p></td>
<td valign=\"top\"><p>torque reference sent to ice through Conn (Nm)</p></td>
</tr>
<tr>
<td valign=\"top\"><p>icePowDel</p></td>
<td valign=\"top\"><p>Ice</p></td>
<td valign=\"top\"><p> </p></td>
<td valign=\"top\"><p>the mechanical power the ice delivers</p></td>
</tr>
<tr>
<td valign=\"top\"><p>iceW</p></td>
<td valign=\"top\"><p>Ice</p></td>
<td valign=\"top\"></td>
<td valign=\"top\"><p><br>ice rotational speed (rad/s)</p></td>
</tr>
<tr>
<td valign=\"top\"><p>genTauNorm</p></td>
<td valign=\"top\"></td>
<td valign=\"top\"><p><br>OneFlange</p></td>
<td valign=\"top\"><p>normalised torque OneFlange  must deliver</p><p>It is used in examples in which OneFlange Instance is called &QUOT;gen&QUOT;</p></td>
</tr>
<tr>
<td valign=\"top\"><p>genTauLim</p></td>
<td valign=\"top\"><p>OneFlange</p></td>
<td valign=\"top\"></td>
<td valign=\"top\"><p><br>maximum (limit) torque OneFlance must deliver</p><p><br>It is used in examples in which OneFlange Instance is called &QUOT;gen&QUOT;</p></td>
</tr>
<tr>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
</tr>
<tr>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
</tr>
<tr>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
</tr>
<tr>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
</tr>
<tr>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
</tr>
<tr>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
</tr>
<tr>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
<td valign=\"top\"></td>
</tr>
</table>
</html>"), uses(Modelica(version="3.2.1")));
    end Information;

    model MBice "Simple  map-based ice model with connector"
      import Modelica.Constants.*;
      extends Partial.PartialMBice;
      parameter Real wIceStart=167;
      // rad/s
      Modelica.Blocks.Interfaces.RealInput tauRef
        "Maximum allowed ICE torque by power train limitations" annotation (
          Placement(transformation(
            extent={{-20,-20},{20,20}},
            rotation=90,
            origin={-60,-100}), iconTransformation(
            extent={{-20,-20},{20,20}},
            rotation=90,
            origin={-60,-100})));
      Modelica.Blocks.Interfaces.RealOutput fuelCons "Fuel consumption (g/h)"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={60,-90})));
      Modelica.Blocks.Nonlinear.Limiter limiter(uMin=0, uMax=1e99) annotation (
          Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-60,-16})));
    equation
      connect(toG_perHour.y, fuelCons) annotation (Line(
          points={{26,-57},{26,-57},{26,-64},{60,-64},{60,-90}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(limiter.y, min1.u2) annotation (Line(
          points={{-60,-5},{-60,54},{-50,54}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(limiter.u, tauRef) annotation (Line(
          points={{-60,-28},{-60,-100}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-80},
                {100,80}}), graphics),
        experiment(
          StopTime=200,
          __Dymola_NumberOfIntervals=1000,
          __Dymola_Algorithm="Lsodar"),
        __Dymola_experimentSetupOutput,
        Documentation(info="<html>
<p>This model belongs to the map-based models of power train components.</p>
<p>It models an Internal Combustion Engine, neglecting any dynamics except that related with its rotor inertia.</p>
<p>The input signal is the torque request (Nm). </p>
<p>The generated torque is the minimum between this signal (negative values are transformed to 0) and the maximum deliverable torque at the actual engine speed, defined by means of a table.</p>
<p>From the generated torque and speed the fuel consumption is computed.</p>
</html>"),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics));
    end MBice;

    model MBOneFlange "Simple map-based model of an electric drive"
      extends Partial.PartialMBOneFlange;
      Modelica.Blocks.Interfaces.RealInput tauRef annotation (Placement(
            transformation(
            extent={{-20,-20},{20,20}},
            rotation=90,
            origin={0,-88}), iconTransformation(
            extent={{-20,-20},{20,20}},
            rotation=90,
            origin={90,-86})));
    equation
      connect(createTau.u2, tauRef) annotation (Line(
          points={{-28,4},{-40,4},{-40,-56},{0,-56},{0,-88}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Diagram(coordinateSystem(
            extent={{-100,-80},{100,80}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Line(points={{62,-7},{82,-7}}),Rectangle(
                  fillColor={192,192,192},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{52,12},{100,-8}}),Rectangle(extent={{-70,80},{100,-80}}),
              Line(points={{-98,40},{-70,40}}, color={0,0,255}),Line(points={{-92,
              -40},{-70,-40}}, color={0,0,255})}),
        Documentation(info="<html>
<p>The input signal is a torque request (Nm).</p>
<p>The maximum available torque is internally computed considering a direct torque maximum (tauMax) and a power maximum (powMax) </p>
<p>The requested torque is applied to a mechancal inertia. </p>
<p>The model then computes the inner losses and absorbs the total power from the DC input.</p>
</html>"));
    end MBOneFlange;

    model MBTwoFlange "Simple map-based two-flange electric drive model"
      extends Partial.PartialMBTwoFlange;
      Modelica.Blocks.Interfaces.RealInput tauRefInt annotation (Placement(
            transformation(
            extent={{-20,-20},{20,20}},
            rotation=90,
            origin={0,-114})));
    equation
      connect(tauRefInt, createTau.u2) annotation (Line(
          points={{0,-114},{0,-114},{0,-60},{-34,-60},{-34,-10},{-20,-10}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                {100,100}}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Rectangle(
                  fillColor={192,192,192},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-100,10},{-66,-10}}),Rectangle(
                  fillColor={192,192,192},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{66,8},{100,-12}}),Rectangle(origin={-25,2}, extent={
              {-75,74},{125,-74}}),Line(
                  origin={20,-2},
                  points={{-60,94},{-60,76}},
                  color={0,0,255}),Line(
                  origin={-20,-2},
                  points={{60,94},{60,76}},
                  color={0,0,255})}),
        Documentation(info="<html>
<p>The input signal is a torque request (Nm).</p>
<p>The maximum available torque is internally computed considering a direct torque maximum (tauMax) and a power maximum (powMax) </p>
<p>The requested torque is applied to a mechancal inertia. The inertia is interfaced by means of two flanges with the exterior.</p>
<p>The model then computes the inner losses and absorbs the total power from the DC input.</p>
</html>"));
    end MBTwoFlange;

    model MBiceConn "Simple map-based ice model with connector"
      import Modelica.Constants.*;
      extends Partial.PartialMBice;
      //  extends MBOneFlange;
      parameter Real vMass=1300;
      parameter Real wIceStart=167;
      // rad/s
      SupportModels.Conn conn annotation (Placement(
          visible=true,
          transformation(extent={{-8,-62},{32,-102}}, rotation=0),
          iconTransformation(extent={{-4,-80},{36,-120}}, rotation=0)));
      Modelica.Blocks.Nonlinear.Limiter limiter(uMin=0, uMax=1e99) annotation (
          Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-58,-6})));
    equation
      connect(icePow.power, conn.icePowDel) annotation (Line(
          points={{68,49},{70,49},{70,-68},{12,-68},{12,-82}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(w.w, conn.iceW) annotation (Line(
          points={{52,33},{52,34},{60,34},{60,-68},{12,-68},{12,-82}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(limiter.y, min1.u2) annotation (Line(
          points={{-58,5},{-58,54},{-50,54}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(limiter.u, conn.iceTauRef) annotation (Line(
          points={{-58,-18},{-58,-68},{12,-68},{12,-82}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-80},
                {100,80}}), graphics),
        experiment(
          StopTime=200,
          __Dymola_NumberOfIntervals=1000,
          __Dymola_Algorithm="Lsodar"),
        __Dymola_experimentSetupOutput,
        Documentation(info="<html>
<p><b>Simple map-based ICE model for power-split power trains - with connector</b> </p>
<p>This model belongs to the map-based models of power train components.</p>
<p>It models an Internal Combustion Engine, neglecting any dynamics except that related with its rotor inertia.</p>
<p>The request is satisfied only in case the maximum power is not overcome; otherwise the torque is scaled down so that the maximum power is delivered.</p>
<p>The additional input torqueLim is useful in some cases. For instance when the ICE is used in PSD power train, there is a fixed ratio between the torque at the ICE and generator flanges. The ICE cannot deliber a torque that implies on the generator flange a torque larger than the maximum allowed.</p>
<p>Signals connected to the connector:</p>
<p>- iceTauRef (input) is the torque request (Nm). Negative values are internally converted to zero</p>
<p>- iceW (output) is the measured ICE speed (rad/s)</p>
<p>- icePowDel (output) delivered power (W)</p>
</html>"),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Line(points={{-60,36},{-60,12}}),Polygon(
              points={{-60,4},{-66,14},{-54,14},{-60,4}})}));
    end MBiceConn;

    model MBOneFlangeConn "Simple map-based one-flange electric drive "
      extends Partial.PartialMBOneFlange;
      SupportModels.Conn conn annotation (Placement(
          visible=true,
          transformation(extent={{-18,-60},{22,-100}}, rotation=0),
          iconTransformation(extent={{80,-58},{120,-98}}, rotation=0)));
      Modelica.Blocks.Sources.RealExpression mechPow(y=powSensor.power)
        annotation (Placement(transformation(extent={{38,-56},{18,-36}})));
    equation
      connect(createTau.u2, conn.genTauRef) annotation (Line(
          points={{-28,4},{-42,4},{-42,-68},{2,-68},{2,-80}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(mechPow.y, conn.genPowDel) annotation (Line(
          points={{17,-46},{2,-46},{2,-80}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(limTau.y, conn.genTauLim) annotation (Line(
          points={{15,30},{-36,30},{-36,30},{-42,30},{-42,-68},{2,-68},{2,-80}},

          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));

      connect(wSensor.w, conn.genW) annotation (Line(
          points={{78,35.2},{78,-68},{2,-68},{2,-80}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (
        Diagram(coordinateSystem(
            extent={{-100,-80},{100,80}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Line(points={{62,-7},{82,-7}}),Rectangle(
              extent={{-70,80},{100,-80}}),Line(points={{-98,40},{-70,40}},
              color={0,0,255}),Line(points={{-92,-40},{-70,-40}}, color={0,0,
              255}),Text(
                  origin={0,20},
                  lineColor={0,0,255},
                  extent={{-70,98},{100,60}},
                  textString="%name")}),
        Documentation(info="<html>
<p>The input signal is interpreted as a <u>normalised</u> torque request (0 means null torque, 1 maximum availabile torque).</p>
<p>The maximum available torque is internally computed considering a direct torque maximum (tauMax) and a power maximum (powMax) </p>
<p>The requested torque is applied to a mechancal inertia. The inertia is interfaced by means ot two flanges with the exterior.</p>
<p>The model then computes the inner losses and absorbs the total power from the DC input.</p>
<p><br><u>Signals connected to the bus connecto</u>r (the names are chosen from the examples FullVehicles!PSecu1 and PSecu2 where the one-flange machine is called &QUOT;gen&QUOT;):</p>
<p>- genTauRef (input) is the torque request (Nm)</p>
<p>- genPowDel (output) is the delivered mechanical power (W)</p>
<p>- genTauLim (output) maximum available torque at the given machine rotational speed (Nm)</p>
</html>"));
    end MBOneFlangeConn;

    model MBTwoFlangeConn "Simple map-based two-flange electric drive model"
      extends Partial.PartialMBTwoFlange;
      SupportModels.Conn conn1 annotation (Placement(
          visible=true,
          transformation(extent={{-112,-58},{-72,-98}}, rotation=0),
          iconTransformation(extent={{-112,-58},{-72,-98}}, rotation=0)));
    equation
      connect(outAPow.power, conn1.motPowDelA) annotation (Line(
          points={{64,39},{64,-78},{-92,-78},{-92,-78}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(speedRing.w, conn1.motW) annotation (Line(
          points={{-80,29},{-86,29},{-86,28},{-92,28},{-92,-78}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(add.y, conn1.motPowDelAB) annotation (Line(
          points={{28,-1},{28,-22},{78,-22},{78,-78},{-92,-78}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(createTau.u2, conn1.motTauRef) annotation (Line(
          points={{-20,-10},{-92,-10},{-92,-78}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                {100,100}}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Rectangle(
                  fillColor={192,192,192},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-100,10},{-66,-10}}),Rectangle(
                  fillColor={192,192,192},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{66,8},{100,-12}}),Rectangle(origin={-25,2}, extent={
              {-75,74},{125,-74}}),Line(
                  origin={20,-2},
                  points={{-60,94},{-60,76}},
                  color={0,0,255}),Line(
                  origin={-20,-2},
                  points={{60,94},{60,76}},
                  color={0,0,255})}),
        Documentation(info="<html>
<p>This model receives from the connector the torque request (variable MotTauInt) and trieds to deliver it.</p>
<p>However before delivering the requested torque, the model limits it considering the maximum deliverable torque and power. In addition it computes and considers inner losses as determined by means of a map. </p>
<p><br><u>Signals connected to the bus connecto</u>r (the names are chosen from the examples FullVehicles|PSecu1 and PSecu2 where the two-flange machine is called &QUOT;mot&QUOT;):</p>
<p>- motTauRef (input) is the torque request (Nm)</p>
<p>- motPowDelA (output) is the delivered mechanical power (W)</p>
<p>- motPowDelAB (output) is the delivered mechanical power (W)</p>
<p>- motTauLim (output) maximum available torque at the given machine rotational speed (Nm)</p>
</html>"));
    end MBTwoFlangeConn;

    model MBecu2
      "Power Split hybrid power train controller, using ON/OFF strategy"
      extends Icons.EcuIcon;
      parameter Real powFiltT=60 "Power Filter time constant";
      parameter Real socInit=0.6 "Initial value of SOC";
      parameter Real socGain=1000 "soc loop gain (rad/s)";
      parameter Real genTorqueMax=80 "mximum absolute valoe of gen torque (Nm)";
      parameter Real genTorqueGain=genTorqueMax
        "Control gain between ice speed error and gen torque: Nm/(rad/s)";
      parameter Real maxTorqueReq=80
        "Torque request (Nm) that corresponds to 1 from driver";
      parameter Real maxIceTau[:, :]=[0, 80; 100, 80; 350, 95; 500, 95]
        "Curve of maximum ice torque (Nm)";
      parameter Real psdRatio=2.6 "Ring to gen ratio";
      Modelica.Blocks.Continuous.FirstOrder powFilt(T=powFiltT) annotation (
          Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={-46,50})));
      Modelica.Blocks.Math.Feedback fbICEw
        annotation (Placement(transformation(extent={{80,-20},{100,0}})));
      Modelica.Blocks.Math.Gain gain(k=genTorqueGain) annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={100,20})));
      Modelica.Blocks.Nonlinear.Limiter limiter(uMax=genTorqueMax, uMin=-
            genTorqueMax) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={100,50})));
      Modelica.Blocks.Math.Feedback fbSOC
        annotation (Placement(transformation(extent={{-64,-58},{-44,-38}})));
      Modelica.Blocks.Sources.Constant socRef(k=socInit) annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-54,-76})));
      Modelica.Blocks.Math.Add add annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={10,-2})));
      Modelica.Blocks.Math.Gain toIceWRef1(k=-socGain) annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-26,-24})));
      Modelica.Blocks.Logical.Hysteresis hysteresis(uLow=50, uHigh=100)
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={30,30})));
      Modelica.Blocks.Logical.Switch switch1
        annotation (Placement(transformation(extent={{56,-20},{76,0}})));
      Modelica.Blocks.Sources.Constant zero(k=0)
        annotation (Placement(transformation(extent={{14,-46},{34,-26}})));
      Modelica.Blocks.Math.Gain gain1(k=1 + psdRatio) annotation (Placement(
            visible=true, transformation(extent={{-10,58},{10,78}}, rotation=0)));
      Modelica.Blocks.Math.Division division
        annotation (Placement(transformation(extent={{-36,12},{-16,32}})));
      Modelica.Blocks.Tables.CombiTable1D toMaxIceTau(table=maxIceTau)
        annotation (Placement(visible=true, transformation(
            origin={-64,10},
            extent={{10,-10},{-10,10}},
            rotation=180)));
      Modelica.Blocks.Math.Gain toNm(k=maxTorqueReq)
        "converts p.u. torque request into Nm" annotation (Placement(visible=
              true, transformation(extent={{-80,70},{-60,90}}, rotation=0)));
    equation
      connect(powFilt.u, conn1.motPowDelA) annotation (Line(
          points={{-46,62},{-46,88},{0,88},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(gain.u, fbICEw.y) annotation (Line(
          points={{100,8},{100,-10},{99,-10}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(fbICEw.u2, conn1.iceW) annotation (Line(
          points={{90,-18},{90,-92},{-90,-92},{-90,98},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(limiter.y, conn1.genTauRef) annotation (Line(
          points={{100,61},{100,88},{0,88},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(socRef.y, fbSOC.u2) annotation (Line(
          points={{-54,-65},{-54,-56}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(fbSOC.y, toIceWRef1.u) annotation (Line(
          points={{-45,-48},{-26,-48},{-26,-36}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(fbSOC.u1, conn1.batSOC) annotation (Line(
          points={{-62,-48},{-90,-48},{-90,98},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(gain.y, limiter.u) annotation (Line(
          points={{100,31},{100,38}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(fbICEw.u1, switch1.y) annotation (Line(
          points={{82,-10},{77,-10}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(switch1.u3, zero.y) annotation (Line(
          points={{54,-18},{42,-18},{42,-36},{35,-36}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(hysteresis.u, switch1.u1) annotation (Line(
          points={{30,18},{30,-2},{54,-2}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(add.y, switch1.u1) annotation (Line(
          points={{21,-2},{54,-2}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(hysteresis.y, switch1.u2) annotation (Line(
          points={{30,41},{30,48},{44,48},{44,-10},{54,-10}},
          color={255,0,255},
          smooth=Smooth.None));
      connect(hysteresis.y, conn1.iceON) annotation (Line(
          points={{30,41},{30,98},{0,98}},
          color={255,0,255},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(add.u2, toIceWRef1.y) annotation (Line(
          points={{-2,-8},{-26,-8},{-26,-13}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(gain1.u, conn1.genTauLim) annotation (Line(
          points={{-12,68},{-18,68},{-18,84},{0,84},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(gain1.y, conn1.iceTauRef) annotation (Line(
          points={{11,68},{14,68},{14,94},{0,94},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(division.u1, powFilt.y) annotation (Line(
          points={{-38,28},{-46,28},{-46,39}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(division.u2, toMaxIceTau.y[1]) annotation (Line(
          points={{-38,16},{-48,16},{-48,10},{-53,10}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(toMaxIceTau.u[1], fbICEw.u2) annotation (Line(
          points={{-76,10},{-82,10},{-82,-92},{90,-92},{90,-18}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(division.y, add.u1) annotation (Line(
          points={{-15,22},{-10,22},{-10,4},{-2,4}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(toNm.u, motTauInt) annotation (Line(
          points={{-82,80},{-94,80},{-94,0},{-120,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(toNm.y, conn1.motTauRef) annotation (Line(
          points={{-59,80},{-58,80},{-58,98},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{
                -100,-100},{120,100}}), graphics), Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Text(
                  extent={{-102,84},{98,54}},
                  lineColor={0,0,0},
                  textString="PSD-MB2")}));
    end MBecu2;

    model MBecu1
      "Power Split hybrid power train controller, not using ON/OFF strategy"
      extends Icons.EcuIcon;
      parameter Real powFiltT=60 "Power filter time constant (s)";
      parameter Real maxIceTau[:, :]=[0, 80; 100, 80; 350, 95; 500, 95]
        "Curve of maximum ice torque (Nm)";
      parameter Real genTorqueMax=80
        "maximum absolute valoe of gen torque (Nm)";
      parameter Real maxTorqueReq=80
        "Torque request (Nm) that corresponds to 1 from driver";
      parameter Real genTorqueGain=genTorqueMax
        "Control gain between ice speed error and gen torque: Nm/(rad/s)";
      parameter Real psdRatio=2.6 "ring to sun ratio";
      Modelica.Blocks.Math.Gain gain(k=genTorqueGain) annotation (Placement(
            visible=true, transformation(extent={{46,-24},{66,-4}}, rotation=0)));
      Modelica.Blocks.Math.Feedback fbWIce annotation (Placement(visible=true,
            transformation(extent={{20,-24},{40,-4}}, rotation=0)));
      Modelica.Blocks.Nonlinear.Limiter limitWIce(uMax=500, uMin=25)
        annotation (Placement(visible=true, transformation(
            origin={-2,-14},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Modelica.Blocks.Continuous.FirstOrder powFilt(T=powFiltT) annotation (
          Placement(visible=true, transformation(
            origin={-62,16},
            extent={{-10,-10},{10,10}},
            rotation=-90)));
      Modelica.Blocks.Tables.CombiTable1D toMaxIceTau(table=maxIceTau)
        annotation (Placement(visible=true, transformation(
            origin={-60,-26},
            extent={{10,-10},{-10,10}},
            rotation=180)));
      Modelica.Blocks.Math.Division division
        annotation (Placement(transformation(extent={{-40,-24},{-20,-4}})));
      Modelica.Blocks.Math.Gain gain1(k=1 + psdRatio) annotation (Placement(
            visible=true, transformation(extent={{28,46},{48,66}}, rotation=0)));
      Modelica.Blocks.Nonlinear.Limiter limiter(uMax=genTorqueMax, uMin=-
            genTorqueMax) annotation (Placement(visible=true, transformation(
            origin={82,10},
            extent={{-10,-10},{10,10}},
            rotation=90)));
      Modelica.Blocks.Math.Gain toNm(k=maxTorqueReq)
        "converts p.u. torque request into Nm" annotation (Placement(visible=
              true, transformation(extent={{-70,66},{-50,86}}, rotation=0)));
    equation
      connect(fbWIce.u2, conn1.iceW) annotation (Line(points={{30,-22},{30,-56},
              {-78,-56},{-78,54},{-4,54},{-4,98},{0,98}}, color={0,0,127}));
      connect(powFilt.u, conn1.motPowDelA) annotation (Line(points={{-62,28},{-62,
              40},{0,40},{0,98}}, color={0,0,127}));
      connect(fbWIce.u1, limitWIce.y)
        annotation (Line(points={{22,-14},{9,-14}}, color={0,0,127}));
      connect(gain.u, fbWIce.y)
        annotation (Line(points={{44,-14},{39,-14}}, color={0,0,127}));
      connect(limitWIce.u, division.y) annotation (Line(
          points={{-14,-14},{-19,-14}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(toMaxIceTau.y[1], division.u2) annotation (Line(
          points={{-49,-26},{-47.5,-26},{-47.5,-20},{-42,-20}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(division.u1, powFilt.y) annotation (Line(
          points={{-42,-8},{-62,-8},{-62,5}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(toMaxIceTau.u[1], fbWIce.u2) annotation (Line(
          points={{-72,-26},{-76,-26},{-76,-48},{30,-48},{30,-22}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(gain1.u, conn1.genTauLim) annotation (Line(
          points={{26,56},{0,56},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(gain1.y, conn1.iceTauRef) annotation (Line(
          points={{49,56},{58,56},{58,98},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(limiter.u, gain.y) annotation (Line(
          points={{82,-2},{82,-14},{67,-14}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(limiter.y, conn1.genTauRef) annotation (Line(
          points={{82,21},{82,32},{0,32},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(toNm.u, motTauInt) annotation (Line(
          points={{-72,76},{-94,76},{-94,0},{-120,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(toNm.y, conn1.motTauRef) annotation (Line(
          points={{-49,76},{-20,76},{-20,98},{0,98}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (
        Diagram(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-80},{100,80}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Text(
                  extent={{-100,84},{100,54}},
                  lineColor={0,0,0},
                  textString="PSD-MB1")}),
        Documentation(info="<html>
<h4>Power Split Power Train Controller without ON/OFF</h4>
<p>This is a very simpllified controller.</p>
<p>It just tries to keep the ICE at its maximun torque. </p>
<p>Therefore from the vehicle power needs, obtained as the average of most recent delivered power, the requested ICE operating point is obtained by dividing the vehicle power needs by the maximum availble torque at the given speed.</p>
<p>This optimal ICE speed is then pursed by adequate control of the gen torque.</p>
<p>So:</p>
<ul>
<li>powFilt Block filters the delivered power to obtained the power to ask the ICE to deliver</li>
<li>toIceWref converts the power to be requested from the ICE by its maximum torque at the actual speed</li>
<li>after a limiting block, this torque is the reference signal of a feedback; the corresponnding error controls the Gen torque.</li>
</ul>
</html>"));
    end MBecu1;

    package MBsupport "Useful Additional Models"
      extends Modelica.Icons.Package;

      model ToConnIceTauRef "signal adaptor to send iceTauRef to a connector"
        Modelica.Blocks.Interfaces.RealInput u annotation (Placement(
              transformation(extent={{-94,-20},{-54,20}}), iconTransformation(
                extent={{-94,-20},{-54,20}})));
        SupportModels.Conn conn annotation (Placement(transformation(
              extent={{-20,-20},{20,20}},
              rotation=-90,
              origin={60,0}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=-90,
              origin={60,0})));
      equation
        connect(u, conn.iceTauRef) annotation (Line(
            points={{-74,0},{60,0}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-60,
                  -60},{60,60}}), graphics={Rectangle(
                      extent={{-60,40},{60,-40}},
                      lineColor={0,0,0},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Line(
                      points={{-38,0},{30,0}},
                      color={0,0,0},
                      smooth=Smooth.None),Polygon(
                      points={{42,0},{22,8},{22,-8},{42,0}},
                      lineColor={0,0,0},
                      smooth=Smooth.None,
                      fillColor={0,0,0},
                      fillPattern=FillPattern.Solid)}), Diagram(
              coordinateSystem(preserveAspectRatio=false, extent={{-60,-60},{60,
                  60}}), graphics));
      end ToConnIceTauRef;

      model ToConnGenTauRef "signal adaptor to send genTauRef to a connector"
        Modelica.Blocks.Interfaces.RealInput u annotation (Placement(
              transformation(extent={{-90,-20},{-50,20}}), iconTransformation(
                extent={{-90,-20},{-50,20}})));
        SupportModels.Conn conn annotation (Placement(transformation(
              extent={{-20,-20},{20,20}},
              rotation=-90,
              origin={58,0}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=-90,
              origin={58,0})));
      equation
        connect(u, conn.genTauRef) annotation (Line(
            points={{-70,0},{58,0}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-60,
                  -60},{60,60}}), graphics={Rectangle(
                      extent={{-60,40},{60,-40}},
                      lineColor={0,0,0},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Line(
                      points={{-40,0},{32,0}},
                      color={0,0,0},
                      smooth=Smooth.None),Polygon(
                      points={{42,0},{22,8},{22,-8},{42,0}},
                      lineColor={0,0,0},
                      smooth=Smooth.None,
                      fillColor={0,0,0},
                      fillPattern=FillPattern.Solid)}), Diagram(
              coordinateSystem(preserveAspectRatio=false, extent={{-60,-60},{60,
                  60}}), graphics));
      end ToConnGenTauRef;

      package Internal
        "Models intended to be used by other models of this package, not by the final user"
        block LimTau "Torque limiter"
          Modelica.Blocks.Interfaces.RealInput w annotation (Placement(
                transformation(extent={{-140,-20},{-100,20}})));
          Modelica.Blocks.Interfaces.RealOutput y
            annotation (Placement(transformation(extent={{100,-10},{120,10}})));
          parameter Real powMax(start=50000) "Maximum mechanical power (W)";
          parameter Real tauMax(start=400) "Maximum torque (Nm)";
          parameter Real wMax(start=1500, min=powMax/tauMax)
            "Maximum speed (rad/s)";
          Integer state
            "=0 below base speed; =1 before wMax; =2 in w limit, =3 above wMax";
          //0 or 1 if tauMax or powMax is delivered; =2 or 3 if w>wMax
        protected
          parameter Real alpha=0.10
            "fraction of wMax over which the torque is to be brought to zero";
        algorithm
          if w < powMax/tauMax then
            state := 0;
            y := tauMax;
          else
            state := 1;
            y := powMax/w;
          end if;
          //over wMax the torque max is to be rapidly brought to zero
          if w > wMax then
            if w < (1 + alpha)*wMax then
              state := 2;
              y := powMax/wMax*(1 - (w - wMax)/(alpha*wMax));
            else
              state := 3;
              y := 0;
            end if;
          end if;
          annotation (
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                    {100,100}}), graphics),
            Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                    {100,100}}), graphics={Rectangle(
                          extent={{-100,60},{100,-60}},
                          lineColor={0,0,0},
                          fillColor={255,255,255},
                          fillPattern=FillPattern.Solid),Polygon(
                          points={{-4,34},{6,16},{22,2},{38,-10},{56,-16},{72,-18},
                    {72,-38},{12,-38},{-72,-38},{-72,34},{-4,34}},
                          lineColor={0,0,0},
                          smooth=Smooth.None,
                          fillColor={255,255,255},
                          fillPattern=FillPattern.Solid),Line(
                          points={{-72,54},{-72,-40},{-74,-40}},
                          color={0,0,0},
                          thickness=0.5,
                          smooth=Smooth.None,
                          arrow={Arrow.Filled,Arrow.None}),Line(
                          points={{90,-38},{-74,-38}},
                          color={0,0,0},
                          thickness=0.5,
                          smooth=Smooth.None,
                          arrow={Arrow.Filled,Arrow.None}),Text(
                          extent={{-100,80},{96,52}},
                          lineColor={0,0,255},
                          fillColor={255,255,255},
                          fillPattern=FillPattern.Solid,
                          textString="%name
    "),Text(              extent={{70,-48},{84,-54}},
                          lineColor={0,0,255},
                          textString="W"),Text(
                          extent={{-96,48},{-82,42}},
                          lineColor={0,0,255},
                          textString="T")}),
            Documentation(info="<html>
<p>Gives the maximum output torque as a function of the input speed.</p>
<p>When w&LT;wMax the output is Tmax if Tmax*w&LT;Pnom, othersise it is Pnom/w</p>
<p>But if w is over wMax Tmax is rapidly falling to zero (reaches zero when speed overcomes wMax by 10&percnt;).</p>
<p>Torques and powers are in SI units</p>
</html>"));
        end LimTau;

        block Pel
          "Outputs the electric power from the given efficiency and mechanical power"
          Modelica.Blocks.Interfaces.RealInput eta annotation (Placement(
                transformation(extent={{-140,-60},{-100,-20}})));
          Modelica.Blocks.Interfaces.RealInput P annotation (Placement(
                transformation(extent={{-140,20},{-100,60}})));
          Modelica.Blocks.Interfaces.RealOutput Pel
            annotation (Placement(transformation(extent={{100,-10},{120,10}})));
        algorithm
          if noEvent(P <= 0) then
            Pel := P*eta;
          else
            Pel := P/eta;
          end if;
          annotation (Icon(graphics={Rectangle(
                          extent={{-92,56},{98,-48}},
                          lineColor={0,0,255},
                          fillColor={170,255,255},
                          fillPattern=FillPattern.Solid),Text(
                          extent={{-64,30},{48,-14}},
                          lineColor={0,0,255},
                          fillColor={170,255,255},
                          fillPattern=FillPattern.Solid,
                          textString="Pel")}), Diagram(coordinateSystem(
                  preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
                graphics));
        end Pel;

        model InertiaTq "Inertia with added torque"
          import SI = Modelica.SIunits;
          Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a
            "Left flange of shaft" annotation (Placement(transformation(extent=
                    {{-110,-10},{-90,10}}, rotation=0)));
          Modelica.Mechanics.Rotational.Interfaces.Flange_b flange_b
            "Right flange of shaft" annotation (Placement(transformation(extent
                  ={{90,-10},{110,10}}, rotation=0)));
          parameter SI.Inertia J(min=0, start=1) "Moment of inertia";
          parameter StateSelect stateSelect=StateSelect.default
            "Priority to use phi and w as states"
            annotation (HideResult=true, Dialog(tab="Advanced"));
          SI.Angle phi(stateSelect=stateSelect)
            "Absolute rotation angle of component" annotation (Dialog(group=
                  "Initialization", showStartAttribute=true));
          SI.AngularVelocity w(stateSelect=stateSelect)
            "Absolute angular velocity of component (= der(phi))" annotation (
              Dialog(group="Initialization", showStartAttribute=true));
          SI.AngularAcceleration a
            "Absolute angular acceleration of component (= der(w))" annotation
            (Dialog(group="Initialization", showStartAttribute=true));
          Modelica.Blocks.Interfaces.RealInput tau annotation (Placement(
                transformation(
                extent={{-20.5,-20},{20.5,20}},
                rotation=90,
                origin={-54.5,-100})));
        equation
          phi = flange_a.phi;
          phi = flange_b.phi;
          w = der(phi);
          a = der(w);
          J*a = flange_a.tau + flange_b.tau + tau;
          annotation (
            Documentation(info="<html>
    <p>
    Rotational component with <b>inertia</b> and two rigidly connected flanges.
    </p>

    </HTML>
    "),
            Icon(coordinateSystem(
                preserveAspectRatio=true,
                extent={{-100,-100},{100,100}},
                grid={2,2}), graphics={Rectangle(
                          extent={{-100,10},{-50,-10}},
                          lineColor={0,0,0},
                          fillPattern=FillPattern.HorizontalCylinder,
                          fillColor={192,192,192}),Rectangle(
                          extent={{50,10},{100,-10}},
                          lineColor={0,0,0},
                          fillPattern=FillPattern.HorizontalCylinder,
                          fillColor={192,192,192}),Line(points={{-80,-25},{-60,
                  -25}}, color={0,0,0}),Line(points={{60,-25},{80,-25}}, color=
                  {0,0,0}),Line(points={{-70,-25},{-70,-70}}, color={0,0,0}),
                  Line(points={{70,-25},{70,-70}}, color={0,0,0}),Line(points={
                  {-80,25},{-60,25}}, color={0,0,0}),Line(points={{60,25},{80,
                  25}}, color={0,0,0}),Line(points={{-70,45},{-70,25}}, color={
                  0,0,0}),Line(points={{70,45},{70,25}}, color={0,0,0}),Line(
                  points={{-70,-70},{70,-70}}, color={0,0,0}),Rectangle(
                          extent={{-50,50},{50,-50}},
                          lineColor={0,0,0},
                          fillPattern=FillPattern.HorizontalCylinder,
                          fillColor={192,192,192}),Text(
                          extent={{-150,100},{150,60}},
                          textString="%name",
                          lineColor={0,0,255}),Text(
                          extent={{-150,-80},{150,-120}},
                          lineColor={0,0,0},
                          textString="J=%J")}),
            Diagram(coordinateSystem(
                preserveAspectRatio=true,
                extent={{-100,-100},{100,100}},
                grid={2,2}), graphics));
        end InertiaTq;

        block ToElectricPower
          "Determines the electric power from the mechanical considering efficiency map"
          parameter Real tauMax(start=400) "Maximum machine torque(Nm)";
          parameter Real powMax(start=22000) "Maximum drive power";
          parameter Real wMax(start=650) "Maximum machine speed(rad/s)";
          parameter Real effTable[:, :]=[0.00, 0.00, 0.25, 0.50, 0.75, 1.00;
              0.00, 0.75, 0.80, 0.81, 0.82, 0.83; 0.25, 0.76, 0.81, 0.82, 0.83,
              0.84; 0.50, 0.77, 0.82, 0.83, 0.84, 0.85; 0.75, 0.78, 0.83, 0.84,
              0.85, 0.87; 1.00, 0.80, 0.84, 0.85, 0.86, 0.88];
          Modelica.Blocks.Tables.CombiTable2D effTable_(
            tableOnFile=false,
            smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments,
            table=effTable) "normalised efficiency" annotation (Placement(
                transformation(
                extent={{-14,-14},{14,14}},
                rotation=0,
                origin={18,-18})));
          Modelica.Blocks.Interfaces.RealInput w annotation (Placement(
                transformation(extent={{-140,-60},{-100,-20}}),
                iconTransformation(extent={{-140,-60},{-100,-20}})));
          Modelica.Blocks.Interfaces.RealInput tau annotation (Placement(
                transformation(extent={{-140,20},{-100,60}}),
                iconTransformation(extent={{-140,20},{-100,60}})));
          Modelica.Blocks.Interfaces.RealOutput elePow
            annotation (Placement(transformation(extent={{100,-10},{120,10}})));
          Modelica.Blocks.Math.Abs abs1 annotation (Placement(transformation(
                  extent={{-76,-50},{-56,-30}})));
          Modelica.Blocks.Math.Abs abs2
            annotation (Placement(transformation(extent={{-80,40},{-60,60}})));
          Modelica.Blocks.Math.Gain normalizeTau(k=1/tauMax) annotation (
              Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-36,50})));
          Pel pel
            annotation (Placement(transformation(extent={{60,-12},{84,10}})));
          Modelica.Blocks.Math.Product PMOT
            annotation (Placement(transformation(extent={{-72,0},{-52,20}})));
          Modelica.Blocks.Math.Gain normalizeSpeed(k=1/wMax) annotation (
              Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-34,-40})));
        equation
          connect(tau, abs2.u) annotation (Line(
              points={{-120,40},{-94,40},{-94,50},{-82,50}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(w, abs1.u) annotation (Line(
              points={{-120,-40},{-78,-40}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(abs2.y, normalizeTau.u) annotation (Line(
              points={{-59,50},{-48,50}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(normalizeTau.y, effTable_.u1) annotation (Line(
              points={{-25,50},{-7.7,50},{-7.7,-9.6},{1.2,-9.6}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(pel.Pel, elePow) annotation (Line(
              points={{85.2,-1},{92.48,-1},{92.48,0},{110,0}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(effTable_.y, pel.eta) annotation (Line(
              points={{33.4,-18},{46,-18},{46,-5.4},{57.6,-5.4}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(PMOT.u1, tau) annotation (Line(
              points={{-74,16},{-84,16},{-84,40},{-120,40}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(PMOT.u2, w) annotation (Line(
              points={{-74,4},{-84,4},{-84,-40},{-120,-40}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(PMOT.y, pel.P) annotation (Line(
              points={{-51,10},{42,10},{42,3.4},{57.6,3.4}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(abs1.y, normalizeSpeed.u) annotation (Line(
              points={{-55,-40},{-46,-40}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(normalizeSpeed.y, effTable_.u2) annotation (Line(
              points={{-23,-40},{-10,-40},{-10,-26.4},{1.2,-26.4}},
              color={0,0,127},
              smooth=Smooth.None));
          annotation (
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-80},
                    {100,80}}), graphics),
            Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                    {100,100}}), graphics={Rectangle(
                          extent={{-100,72},{100,-72}},
                          lineColor={0,0,0},
                          fillColor={255,255,255},
                          fillPattern=FillPattern.Solid),Line(
                          points={{-74,-54},{-74,58}},
                          color={0,0,0},
                          smooth=Smooth.None),Line(
                          points={{-82,-48},{78,-48}},
                          color={0,0,0},
                          smooth=Smooth.None),Line(
                          points={{-74,38},{-24,38},{-4,12},{28,-8},{60,-22},{
                    62,-48}},
                          color={0,0,0},
                          smooth=Smooth.None),Polygon(
                          points={{-20,14},{-40,24},{-56,-4},{-38,-36},{12,-38},
                    {26,-28},{22,-20},{8,-6},{-8,4},{-20,14}},
                          lineColor={0,0,0},
                          smooth=Smooth.None,
                          fillColor={255,255,255},
                          fillPattern=FillPattern.Solid),Polygon(
                          points={{-28,4},{-38,2},{-32,-20},{0,-32},{10,-28},{
                    12,-20},{-28,4}},
                          lineColor={0,0,0},
                          smooth=Smooth.None,
                          fillColor={255,255,255},
                          fillPattern=FillPattern.Solid),Text(
                          extent={{-102,118},{100,78}},
                          lineColor={0,0,255},
                          fillColor={255,255,255},
                          fillPattern=FillPattern.Solid,
                          textString="%name"),Text(
                          extent={{26,46},{76,4}},
                          lineColor={0,0,0},
                          textString="M")}),
            Documentation(info="<html>
<p>This block computes the machine and inverter losses from the mechanical input quantities and determines the power to be drawn from the electric circuit. The &QUOT;drawn&QUOT; power can be also a negative numer, meaning that themachine is actually delivering electric power.</p>
<p>The given efficiency map is intended as being built with torques being rations of actual torques to tauMax and speeds being ratios of w to wMax. In case the user uses, inthe given efficiency map,  torques in Nm and speeds in rad/s, the block can be used selecting tauTmax=1, wMax=1.</p>
<p>The choice of having normalised efficiency computation allows simulations of machines different in sizes and similar in characteristics to be repeated without having to rebuild the efficiency maps. </p>
</html>"));
        end ToElectricPower;
      end Internal;
      annotation (Icon(graphics={Ellipse(extent={{-38,40},{38,-36}}, lineColor=
              {0,0,0}),Line(
                  points={{2,82},{-8,82},{-12,72},{-26,68},{-36,78},{-48,70},{-44,
                58},{-56,46},{-68,50},{-76,36},{-68,30},{-70,16},{-80,12},{-80,
                2}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{2,-78},{-8,-78},{-12,-68},{-26,-64},{-36,-74},{-48,-66},
                {-44,-54},{-56,-42},{-68,-46},{-76,-32},{-68,-26},{-70,-12},{-80,
                -8},{-80,2}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{0,-78},{10,-78},{14,-68},{28,-64},{38,-74},{50,-66},
                {46,-54},{58,-42},{70,-46},{78,-32},{70,-26},{72,-12},{82,-8},{
                82,2}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{0,82},{10,82},{14,72},{28,68},{38,78},{50,70},{46,58},
                {58,46},{70,50},{78,36},{70,30},{72,16},{82,12},{82,2}},
                  color={0,0,0},
                  smooth=Smooth.None)}));
    end MBsupport;

    package Partial
      partial model PartialMBice
        "Simple  map-based Internal Combustion Engine model"
        import Modelica.Constants.*;
        parameter Real vMass=1300;
        parameter Real wIceStart=167;
        // rad/s
        parameter Real inertiaJ=0.5 "rotor moment of inertia (N.m^2)";
        parameter Real maxIceTau[:, :]=[0, 80; 100, 80; 350, 95; 500, 95]
          "curve of maximum ice torque (Nm)";
        parameter Real specificCons[:, :]=[0, 80; 100, 80; 350, 95; 500, 95]
          "curve of ice specific consumption (g/kWh)";
        Modelica.Mechanics.Rotational.Sensors.SpeedSensor w annotation (
            Placement(visible=true, transformation(
              origin={52,44},
              extent={{-10,-10},{10,10}},
              rotation=270)));
        Modelica.Blocks.Math.Min min1 annotation (Placement(visible=true,
              transformation(extent={{-48,50},{-28,70}}, rotation=0)));
        Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a annotation (
            Placement(
            visible=true,
            transformation(extent={{90,10},{110,30}}, rotation=0),
            iconTransformation(extent={{90,10},{110,30}}, rotation=0)));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor icePow annotation (
            Placement(visible=true, transformation(extent={{66,50},{86,70}},
                rotation=0)));
        Modelica.Mechanics.Rotational.Sources.Torque Tice annotation (Placement(
              visible=true, transformation(extent={{-12,50},{8,70}}, rotation=0)));
        Modelica.Mechanics.Rotational.Components.Inertia ICE(w(
            fixed=true,
            start=wIceStart,
            displayUnit="rpm"), J=inertiaJ) annotation (Placement(visible=true,
              transformation(extent={{16,50},{36,70}}, rotation=0)));
        Modelica.Blocks.Tables.CombiTable1D toSpecCons(table=specificCons)
          annotation (Placement(visible=true, transformation(
              origin={44,12},
              extent={{10,-10},{-10,10}},
              rotation=90)));
        Modelica.Blocks.Math.Product toPow0 annotation (Placement(visible=true,
              transformation(
              origin={2,12},
              extent={{-10,-10},{10,10}},
              rotation=-90)));
        Modelica.Blocks.Math.Product toG_perHour annotation (Placement(visible=
                true, transformation(
              origin={26,-46},
              extent={{-10,-10},{10,10}},
              rotation=-90)));
        //  Modelica.Blocks.Continuous.Integrator toGrams(k = 1 / 3600000.0)
        // annotation(Placement(visible = true, transformation(origin = {26, -44},
        //extent = {{-10, -10}, {10, 10}}, rotation = 270)));
        Modelica.Blocks.Tables.CombiTable1D toLimTau(table=maxIceTau)
          annotation (Placement(visible=true, transformation(
              origin={-72,66},
              extent={{10,-10},{-10,10}},
              rotation=180)));
        Modelica.Blocks.Sources.RealExpression rotorW(y=w.w) annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={-88,36})));
        Modelica.Blocks.Math.Gain tokW(k=1e-3) annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=-90,
              origin={2,-18})));
      equation
        connect(toSpecCons.y[1], toG_perHour.u1) annotation (Line(points={{44,1},
                {44,-34},{32,-34}}, color={0,0,127}));
        connect(toPow0.u1, w.w)
          annotation (Line(points={{8,24},{8,33},{52,33}}, color={0,0,127}));
        connect(toPow0.u2, min1.y) annotation (Line(points={{-4,24},{-4,32},{-22,
                32},{-22,60},{-27,60}}, color={0,0,127}));
        connect(toSpecCons.u[1], w.w)
          annotation (Line(points={{44,24},{44,33},{52,33}}, color={0,0,127}));
        connect(w.flange, ICE.flange_b)
          annotation (Line(points={{52,54},{52,60},{36,60}}));
        connect(icePow.flange_a, ICE.flange_b)
          annotation (Line(points={{66,60},{36,60}}));
        connect(Tice.flange, ICE.flange_a)
          annotation (Line(points={{8,60},{16,60}}));
        connect(Tice.tau, min1.y)
          annotation (Line(points={{-14,60},{-27,60}}, color={0,0,127}));
        connect(icePow.flange_b, flange_a)
          annotation (Line(points={{86,60},{94,60},{94,20},{100,20}}));
        connect(min1.u1, toLimTau.y[1]) annotation (Line(
            points={{-50,66},{-61,66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toLimTau.u[1], rotorW.y) annotation (Line(
            points={{-84,66},{-88,66},{-88,47}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toPow0.y, tokW.u) annotation (Line(
            points={{2,1},{2,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(tokW.y, toG_perHour.u2) annotation (Line(
            points={{2,-29},{12,-29},{12,-26},{20,-26},{20,-34}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-80},
                  {100,80}}), graphics),
          experiment(
            StopTime=200,
            __Dymola_NumberOfIntervals=1000,
            __Dymola_Algorithm="Lsodar"),
          __Dymola_experimentSetupOutput,
          Documentation(info="<html>
<h4>Basic map-based ICE model.</h4>
<p>It receives as input the reference torque as a fracton of the maximum deliverable torque at a given speed. It can be approximately thought as a signal proportional to the accelerator position fo the vehicle.</p>
<p>The generated torque is the minimum between this signal and the maximum deliverable torque at the actual engine speed (defined by means of a table).</p>
<p>From the generated torque and speed the fuel consumption is computed.</p>
</html>"),
          Icon(coordinateSystem(
              extent={{-100,-100},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics={Rectangle(
                      extent={{-100,80},{100,-80}},
                      lineColor={0,0,0},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Rectangle(
                      fillColor={192,192,192},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{-24,68},{76,-24}}),Rectangle(
                      fillColor={192,192,192},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{76,30},{100,10}}),Text(
                      extent={{-140,-32},{140,-70}},
                      textString="J=%inertiaJ",
                      lineColor={0,0,0}),Text(
                      origin={0,30},
                      lineColor={0,0,255},
                      extent={{-140,100},{140,60}},
                      textString="%name"),Rectangle(extent={{-90,68},{-32,-26}}),
                Rectangle(
                      fillColor={95,95,95},
                      fillPattern=FillPattern.Solid,
                      extent={{-90,22},{-32,0}}),Line(points={{-60,56},{-60,32}}),
                Polygon(points={{-60,66},{-66,56},{-54,56},{-60,66}}),Polygon(
                points={{-60,24},{-66,34},{-54,34},{-60,24}}),Rectangle(
                      fillColor={135,135,135},
                      fillPattern=FillPattern.Solid,
                      extent={{-64,0},{-54,-20}})}));
      end PartialMBice;

      model PartialMBOneFlange
        "Partial map-based one-Flange electric drive model"
        parameter Real powMax=22000 "Maximum drive power  (W)";
        parameter Real tauMax=80 "Maximum drive torque (Nm)";
        parameter Real wMax(start=3000, min=powMax/tauMax)
          "Maximum drive speed (rad/s)";
        parameter Real J=0.25 "Rotor's moment of inertia (kg.m^2)";
        Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a
          "Left flange of shaft" annotation (Placement(transformation(extent={{
                  88,50},{108,70}}, rotation=0), iconTransformation(extent={{90,
                  -10},{110,10}})));
        Modelica.Mechanics.Rotational.Sensors.SpeedSensor wSensor annotation (
            Placement(transformation(
              extent={{8,-8},{-8,8}},
              rotation=90,
              origin={78,44})));
        Modelica.Blocks.Math.Abs abs1
          annotation (Placement(transformation(extent={{62,16},{46,32}})));
        MBsupport.Internal.LimTau limTau(
          tauMax=tauMax,
          wMax=wMax,
          powMax=powMax)
          annotation (Placement(transformation(extent={{36,18},{16,42}})));
        MBsupport.Internal.ToElectricPower effMap(
          tauMax=tauMax,
          wMax=wMax,
          powMax=powMax)
          annotation (Placement(transformation(extent={{-6,-28},{-26,-8}})));
        Modelica.Electrical.Analog.Interfaces.PositivePin pin_p annotation (
            Placement(transformation(extent={{-110,30},{-90,50}}),
              iconTransformation(extent={{-110,30},{-90,50}})));
        Modelica.Electrical.Analog.Interfaces.NegativePin pin_n annotation (
            Placement(transformation(extent={{-110,-50},{-90,-30}}),
              iconTransformation(extent={{-110,-50},{-90,-30}})));
        SupportModels.Internal.ConstPDC constPDC(k=10, T=0.01) annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-100,0})));
        Modelica.Mechanics.Rotational.Components.Inertia inertia(J=J)
          annotation (Placement(transformation(extent={{22,50},{42,70}})));
        Modelica.Mechanics.Rotational.Sources.Torque torque
          annotation (Placement(transformation(extent={{-16,50},{4,70}})));
        Modelica.Blocks.Math.Gain gain(k=1)
          annotation (Placement(transformation(extent={{-64,-10},{-84,10}})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor powSensor
          annotation (Placement(transformation(extent={{50,50},{70,70}})));
        Modelica.Blocks.Math.Min createTau annotation (Placement(visible=true,
              transformation(extent={{-26,0},{-6,20}}, rotation=0)));
      equation
        assert(wMax >= powMax/tauMax, "\n****\n" +
          "PARAMETER VERIFICATION ERROR:\nwMax must be not lower than powMax/tauMax"
           + "\n***\n");
        connect(abs1.u, wSensor.w) annotation (Line(
            points={{63.6,24},{78,24},{78,35.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(effMap.w, wSensor.w) annotation (Line(
            points={{-4,-22},{78,-22},{78,35.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pin_p, constPDC.pin_p) annotation (Line(
            points={{-100,40},{-100,10}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(pin_n, constPDC.pin_n) annotation (Line(
            points={{-100,-40},{-100,-9.8}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(constPDC.Pref, gain.y) annotation (Line(
            points={{-91.8,0},{-85,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(limTau.w, abs1.y) annotation (Line(
            points={{38,30},{42,30},{42,24},{45.2,24}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(powSensor.flange_b, flange_a) annotation (Line(
            points={{70,60},{98,60}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(wSensor.flange, flange_a) annotation (Line(
            points={{78,52},{78,60},{98,60}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(effMap.elePow, gain.u) annotation (Line(
            points={{-27,-18},{-46,-18},{-46,0},{-62,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(inertia.flange_a, torque.flange) annotation (Line(
            points={{22,60},{4,60}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(inertia.flange_b, powSensor.flange_a) annotation (Line(
            points={{42,60},{50,60}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(limTau.y, createTau.u1) annotation (Line(
            points={{15,30},{-40,30},{-40,16},{-28,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(createTau.y, torque.tau) annotation (Line(
            points={{-5,10},{6,10},{6,40},{-40,40},{-40,60},{-18,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(createTau.y, effMap.tau) annotation (Line(
            points={{-5,10},{6,10},{6,-14},{-4,-14}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(
              extent={{-100,-80},{100,80}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics),
          Icon(coordinateSystem(
              extent={{-100,-100},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics={Rectangle(
                      extent={{-70,80},{100,-80}},
                      lineColor={0,0,0},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Rectangle(
                      fillColor={192,192,192},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{-48,48},{52,-44}}),Line(points={{62,-7},{82,-7}}),
                Rectangle(
                      fillColor={192,192,192},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{52,10},{100,-10}}),Text(
                      origin={-14,-36},
                      extent={{-140,-54},{160,-94}},
                      textString="J=%J"),Line(points={{-98,40},{-70,40}}, color
                ={0,0,255}),Line(points={{-92,-40},{-70,-40}}, color={0,0,255}),
                Text( origin={0,20},
                      lineColor={0,0,255},
                      extent={{-70,98},{100,60}},
                      textString="%name",
                      fillPattern=FillPattern.Solid,
                      fillColor={255,255,255})}),
          Documentation(info="<html>
<p>One-flange electric drive.</p>
<p>The input signal is the requested normalised torque (1 means nominal torque)</p>
</html>"));
      end PartialMBOneFlange;

      model PartialMBTwoFlange
        "Simple map-based two-flange electric drive model"
        parameter Real powMax(start=50000) "Maximum Mechanical drive power (W)";
        parameter Real tauMax(start=400) "Maximum drive Torque  (Nm)";
        parameter Real wMax(start=650) "Maximum drive speed (rad/s)";
        parameter Real J=0.59 "Moment of Inertia (kg.m^2)";
        //  Real state=limTau.state;
        MBsupport.Internal.LimTau limTau(
          tauMax=tauMax,
          wMax=wMax,
          powMax=powMax)
          annotation (Placement(transformation(extent={{-54,-8},{-32,14}})));
        MBsupport.Internal.InertiaTq inertia(w(displayUnit="rad/s", start=0), J
            =J) annotation (Placement(transformation(extent={{6,40},{26,60}})));
        Modelica.Mechanics.Rotational.Sensors.SpeedSensor speedRing annotation
          (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-80,40})));
        Modelica.Blocks.Math.Abs abs1
          annotation (Placement(transformation(extent={{-76,-4},{-62,10}})));
        MBsupport.Internal.ToElectricPower effMap(
          tauMax=tauMax,
          wMax=wMax,
          powMax=powMax)
          annotation (Placement(transformation(extent={{20,-46},{40,-26}})));
        SupportModels.Internal.ConstPDC constPDC(k=10, T=0.01) annotation (
            Placement(transformation(
              extent={{-10,10},{10,-10}},
              rotation=-90,
              origin={0,100})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor outAPow
          annotation (Placement(transformation(extent={{62,40},{82,60}})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor outBPow
          annotation (Placement(transformation(extent={{-18,40},{-38,60}})));
        Modelica.Blocks.Math.Add add annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=-90,
              origin={28,10})));
        Modelica.Mechanics.Rotational.Interfaces.Flange_b flange_b
          "Right flange of shaft" annotation (Placement(
            visible=true,
            transformation(extent={{90,40},{110,60}}, rotation=0),
            iconTransformation(extent={{90,-12},{110,8}}, rotation=0)));
        Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a
          "Left flange of shaft" annotation (Placement(
            visible=true,
            transformation(extent={{-110,40},{-90,60}}, rotation=0),
            iconTransformation(extent={{-110,-10},{-90,10}}, rotation=0)));
        Modelica.Electrical.Analog.Interfaces.PositivePin pin_p annotation (
            Placement(
            visible=true,
            transformation(extent={{-70,90},{-50,110}}, rotation=0),
            iconTransformation(extent={{-50,88},{-30,108}}, rotation=0)));
        Modelica.Electrical.Analog.Interfaces.NegativePin pin_n annotation (
            Placement(
            visible=true,
            transformation(extent={{30,90},{50,110}}, rotation=0),
            iconTransformation(extent={{30,90},{50,110}}, rotation=0)));
        Modelica.Blocks.Math.Min createTau annotation (Placement(visible=true,
              transformation(extent={{-18,-14},{2,6}}, rotation=0)));
      equation
        connect(flange_a, speedRing.flange) annotation (Line(
            points={{-100,50},{-80,50}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(abs1.u, speedRing.w) annotation (Line(
            points={{-77.40000000000001,3},{-80,3},{-80,29}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(limTau.w, abs1.y) annotation (Line(
            points={{-56.2,3},{-61.3,3}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(effMap.w, speedRing.w) annotation (Line(
            points={{18,-40},{-80,-40},{-80,29}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pin_p, constPDC.pin_p) annotation (Line(
            points={{-60,100},{-10,100}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(pin_n, constPDC.pin_n) annotation (Line(
            points={{40,100},{9.8,100}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(effMap.elePow, constPDC.Pref) annotation (Line(
            points={{41,-36},{52,-36},{52,80},{0,80},{0,91.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flange_b, outAPow.flange_b) annotation (Line(
            points={{100,50},{82,50}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(inertia.flange_b, outAPow.flange_a) annotation (Line(
            points={{26,50},{62,50}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(inertia.flange_a, outBPow.flange_a) annotation (Line(
            points={{6,50},{-18,50}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(outBPow.flange_b, speedRing.flange) annotation (Line(
            points={{-38,50},{-80,50}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(add.u1, outAPow.power) annotation (Line(
            points={{34,22},{34,28},{64,28},{64,39}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add.u2, outBPow.power) annotation (Line(
            points={{22,22},{22,28},{-20,28},{-20,39}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(limTau.y, createTau.u1) annotation (Line(
            points={{-30.9,3},{-26,3},{-26,2},{-20,2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(createTau.y, inertia.tau) annotation (Line(
            points={{3,-4},{10.55,-4},{10.55,40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(effMap.tau, inertia.tau) annotation (Line(
            points={{18,-32},{10,-32},{10,-4},{10.55,-4},{10.55,40}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {100,100}}), graphics),
          Icon(coordinateSystem(
              extent={{-100,-100},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics={Rectangle(
                      origin={-25,2},
                      extent={{-75,74},{125,-74}},
                      lineColor={0,0,0},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Text(
                      origin={4,-6},
                      lineColor={0,0,255},
                      extent={{-110,84},{100,44}},
                      textString="%name"),Rectangle(
                      fillColor={192,192,192},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{-64,38},{64,-44}}),Rectangle(
                      fillColor={192,192,192},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{-100,10},{-64,-10}}),Rectangle(
                      fillColor={192,192,192},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{64,8},{100,-12}}),Line(
                      origin={20,0},
                      points={{-60,94},{-60,76}},
                      color={0,0,255}),Line(
                      origin={-20,0},
                      points={{60,94},{60,76}},
                      color={0,0,255}),Text(
                      origin={66,-32},
                      extent={{-108,-46},{100,-84}},
                      textString="J=%J")}),
          Documentation(info="<html>
<p>This model receives from the connector the torque request (variable MotTauInt) and trieds to deliver it.</p>
<p>Howeve,r before delivering the requested torque, the model limits it considering the maximum deliverable torque and power. In addition it computes and considers inner losses as determined by means of a map. </p>
</html>"));
      end PartialMBTwoFlange;
    end Partial;

    package TestingModels
      extends Modelica.Icons.ExamplesPackage;

      model TestMBIce
        MBice mBice(wIceStart=90)
          annotation (Placement(transformation(extent={{-20,-2},{0,18}})));
        Modelica.Mechanics.Rotational.Components.Inertia inertia(J=0.5, phi(
              start=0, fixed=true))
          annotation (Placement(transformation(extent={{10,0},{30,20}})));
        Modelica.Mechanics.Rotational.Sources.QuadraticSpeedDependentTorque
          loadTorque(w_nominal=100, tau_nominal=-80)
          annotation (Placement(transformation(extent={{64,0},{44,20}})));
        Modelica.Blocks.Sources.Trapezoid trapezoid(
          rising=10,
          width=10,
          falling=10,
          period=1e6,
          startTime=10,
          offset=60,
          amplitude=30)
          annotation (Placement(transformation(extent={{-50,-38},{-30,-18}})));
      equation
        connect(mBice.flange_a, inertia.flange_a) annotation (Line(
            points={{0,10},{10,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(inertia.flange_b, loadTorque.flange) annotation (Line(
            points={{30,10},{44,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(mBice.tauRef, trapezoid.y) annotation (Line(
            points={{-16,-2},{-16,-28},{-29,-28}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-80,-60},
                  {80,60}}), graphics),
          experiment(StopTime=50),
          __Dymola_experimentSetupOutput,
          Icon(coordinateSystem(extent={{-80,-60},{80,60}})),
          Documentation(info="<html>
<p>This is a simple test of model MBIce.</p>
<p>It shows that the generated torque follows the torque request as long as the maximum allowed is not overcome; otherwise this maximum is generated.</p>
<p>It shows also the fuel consumption output.</p>
<p>The user could compare the torque request with the torque  generated and at the ICE flange (with this transient the inertia torques are very small and can be neglected). The user could also have a look at the rotational speeds and fuel consumption. </p>
</html>"));
      end TestMBIce;

      model TestMBOneFlange
        Modelica.Mechanics.Rotational.Components.Inertia inertia(
          J=0.5,
          phi(start=0, fixed=true),
          w(start=50, fixed=true))
          annotation (Placement(transformation(extent={{38,0},{58,20}})));
        Modelica.Mechanics.Rotational.Sources.QuadraticSpeedDependentTorque
          loadTorque(tau_nominal=-50, w_nominal=400)
          annotation (Placement(transformation(extent={{92,0},{72,20}})));
        Modelica.Blocks.Sources.Trapezoid tauRef(
          rising=10,
          width=10,
          falling=10,
          period=1e6,
          startTime=10,
          amplitude=50,
          offset=20)
          annotation (Placement(transformation(extent={{-50,-38},{-30,-18}})));
        MBOneFlange oneFlange(
          powMax=10000,
          tauMax=50,
          J=0.5,
          wMax=300)
          annotation (Placement(transformation(extent={{-22,0},{-2,20}})));
        Modelica.Electrical.Analog.Sources.ConstantVoltage gen(V=100)
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=-90,
              origin={-64,10})));
        Modelica.Electrical.Analog.Basic.Ground ground
          annotation (Placement(transformation(extent={{-90,-20},{-70,0}})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor powMech
          annotation (Placement(transformation(extent={{12,0},{32,20}})));
        Modelica.Electrical.Analog.Sensors.PowerSensor powElec
          annotation (Placement(transformation(extent={{-52,14},{-32,34}})));
      equation
        connect(inertia.flange_b, loadTorque.flange) annotation (Line(
            points={{58,10},{72,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(tauRef.y, oneFlange.tauRef) annotation (Line(
            points={{-29,-28},{-3,-28},{-3,1.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ground.p, gen.n) annotation (Line(
            points={{-80,0},{-64,0}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(oneFlange.flange_a, powMech.flange_a) annotation (Line(
            points={{-2,10},{12,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(inertia.flange_a, powMech.flange_b) annotation (Line(
            points={{38,10},{32,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(powElec.nc, oneFlange.pin_p) annotation (Line(
            points={{-32,24},{-22,24},{-22,14}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powElec.pc, gen.p) annotation (Line(
            points={{-52,24},{-64,24},{-64,20}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powElec.pv, powElec.nc) annotation (Line(
            points={{-42,34},{-36,34},{-36,34},{-32,34},{-32,24}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(gen.n, oneFlange.pin_n) annotation (Line(
            points={{-64,0},{-22,0},{-22,6}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powElec.nv, oneFlange.pin_n) annotation (Line(
            points={{-42,14},{-42,0},{-22,0},{-22,6}},
            color={0,0,255},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-60},
                  {100,60}}), graphics),
          experiment(StopTime=50),
          __Dymola_experimentSetupOutput,
          Icon(coordinateSystem(extent={{-100,-100},{100,100}})),
          Documentation(info="<html>
<p>This is a simple test of model MBOneFlange.</p>
<p>It shows that the generated torque follows the normalised torque request as long as it does not overcome unity. Actual torque will be this request times the maximum value that, in turn, is the minimum between tauMax and poweMax/w (while w is the rotational speed)</p>
<p>It shows also the effects of efficiency on the DC power.</p>
<p>Finally it shows that MBOneFlange model automatically reduces torque if the mximum speed tends to be overcome.</p>
<p><u>First suggested plots</u>: 1) on the same axis oneFlange.torque.tau, oneFlange.limTau.y and tauRef 2) vertically aligned with the previous oneFlange.limTau.state. In these plots it can be seen that:</p>
<ul>
<li>during the first 10 seconds the generated torque oneFlange.torque.tau, is 20Nm, as requested from the input. The maximum torque that can be generated is not limited by the power limit (thus state=0)</li>
<li>between t=10 and 14 the generated torque continues to follow the input signal; but starting from t=10.8 the maximum torque that can be delivered is limited by the maximum drive power (this is confirmed by the value state=1)</li>
<li>between t=14 and 18 s, since the drive power has been reached (10 kW), the generated torque is automatically reduced to avoid this limit to be overcome </li>
<li>between t=18 and t=38 the maximum speed is reached and therefore the generated torque is automatically reduced to avoid this limit to be overcome (state=2)</li>
<li>above t=38 the torque request is reduced and the drive is again able to deliver this torque.</li>
</ul>
<p><u>Second suggested plot</u>: Once the first plot is anaysed, the user might want to have an idea of the mechanical and electrical powers: these are seen putting in the same plot powMech.power and powElec.power.</p>
</html>"));
      end TestMBOneFlange;

      model TestMBIceConn
        Modelica.Mechanics.Rotational.Components.Inertia inertia(J=0.5, phi(
              start=0, fixed=true))
          annotation (Placement(transformation(extent={{10,0},{30,20}})));
        Modelica.Mechanics.Rotational.Sources.QuadraticSpeedDependentTorque
          loadTorque(w_nominal=100, tau_nominal=-80)
          annotation (Placement(transformation(extent={{64,0},{44,20}})));
        MBiceConn mBiceConn(wIceStart=90)
          annotation (Placement(transformation(extent={{-18,-2},{2,18}})));
        MBsupport.ToConnIceTauRef toConnIceTauRef annotation (Placement(
              transformation(
              extent={{-6,-6},{6,6}},
              rotation=90,
              origin={-6,-18})));
        Modelica.Blocks.Sources.Trapezoid trapezoid(
          rising=10,
          width=10,
          falling=10,
          period=1e6,
          startTime=10,
          offset=60,
          amplitude=30)
          annotation (Placement(transformation(extent={{-50,-30},{-30,-10}})));
      equation
        connect(inertia.flange_b, loadTorque.flange) annotation (Line(
            points={{30,10},{44,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(inertia.flange_a, mBiceConn.flange_a) annotation (Line(
            points={{10,10},{2,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(toConnIceTauRef.conn, mBiceConn.conn) annotation (Line(
            points={{-6,-12},{-6,-2},{-6.4,-2}},
            color={255,204,51},
            thickness=0.5,
            smooth=Smooth.None));
        connect(toConnIceTauRef.u, trapezoid.y) annotation (Line(
            points={{-6,-25.4},{-6,-32},{-20,-32},{-20,-20},{-29,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-80,-60},
                  {80,60}}), graphics),
          experiment(StopTime=50),
          __Dymola_experimentSetupOutput,
          Icon(coordinateSystem(extent={{-80,-60},{80,60}})),
          Documentation(info="<html>
<p>This is a simple test of model MBIce with connector.</p>
<p>For the desctipion see the description of TestMBIce.</p>
</html>"));
      end TestMBIceConn;

      model TestMBOneFlangeConn
        Modelica.Mechanics.Rotational.Components.Inertia inertia(
          J=0.5,
          phi(start=0, fixed=true),
          w(start=50, fixed=true))
          annotation (Placement(transformation(extent={{38,0},{58,20}})));
        Modelica.Mechanics.Rotational.Sources.QuadraticSpeedDependentTorque
          loadTorque(tau_nominal=-50, w_nominal=400)
          annotation (Placement(transformation(extent={{92,0},{72,20}})));
        Modelica.Electrical.Analog.Sources.ConstantVoltage gen(V=100)
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=-90,
              origin={-64,10})));
        Modelica.Electrical.Analog.Basic.Ground ground
          annotation (Placement(transformation(extent={{-90,-20},{-70,0}})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor powMech
          annotation (Placement(transformation(extent={{12,0},{32,20}})));
        Modelica.Electrical.Analog.Sensors.PowerSensor powElec
          annotation (Placement(transformation(extent={{-52,14},{-32,34}})));
        MBOneFlangeConn oneFlangeConn(
          powMax=10000,
          tauMax=50,
          J=0.5,
          wMax=300)
          annotation (Placement(transformation(extent={{-16,0},{4,20}})));
        MBsupport.ToConnGenTauRef toConnGenTauNorm
          annotation (Placement(transformation(extent={{-16,-34},{-4,-22}})));
        Modelica.Blocks.Sources.Trapezoid tauRef(
          rising=10,
          width=10,
          falling=10,
          period=1e6,
          startTime=10,
          amplitude=50,
          offset=20)
          annotation (Placement(transformation(extent={{-48,-38},{-28,-18}})));
      equation
        connect(inertia.flange_b, loadTorque.flange) annotation (Line(
            points={{58,10},{72,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(ground.p, gen.n) annotation (Line(
            points={{-80,0},{-64,0}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(inertia.flange_a, powMech.flange_b) annotation (Line(
            points={{38,10},{32,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(powElec.pc, gen.p) annotation (Line(
            points={{-52,24},{-64,24},{-64,20}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powElec.pv, powElec.nc) annotation (Line(
            points={{-42,34},{-36,34},{-36,34},{-32,34},{-32,24}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powMech.flange_a, oneFlangeConn.flange_a) annotation (Line(
            points={{12,10},{4,10}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(oneFlangeConn.pin_p, powElec.nc) annotation (Line(
            points={{-16,14},{-24,14},{-24,24},{-32,24}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(oneFlangeConn.pin_n, gen.n) annotation (Line(
            points={{-16,6},{-24,6},{-24,0},{-64,0}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(toConnGenTauNorm.conn, oneFlangeConn.conn) annotation (Line(
            points={{-4.2,-28},{4,-28},{4,2.2}},
            color={255,204,51},
            thickness=0.5,
            smooth=Smooth.None));
        connect(powElec.nv, gen.n) annotation (Line(
            points={{-42,14},{-42,0},{-64,0}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(toConnGenTauNorm.u, tauRef.y) annotation (Line(
            points={{-17,-28},{-27,-28}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-60},
                  {100,60}}), graphics),
          experiment(StopTime=50),
          __Dymola_experimentSetupOutput,
          Icon(coordinateSystem(extent={{-100,-100},{100,100}})),
          Documentation(info="<html>
<p>This is a simple test of model MBOneFlange with bus connector.</p>
<p>For the description see the description of TestMBOneFlange (substitute the word &QUOT;oneFlange&QUOT; with &QUOT;oneFlangeConn&QUOT;).</p>
</html>"));
      end TestMBOneFlangeConn;

      model TestMBTwoFlange "Test of MBTwoFlange drive train model"
        Modelica.Mechanics.Rotational.Components.Inertia inertia(
          J=0.5,
          phi(start=0, fixed=true),
          w(start=50, fixed=true))
          annotation (Placement(transformation(extent={{38,-10},{58,10}})));
        Modelica.Mechanics.Rotational.Sources.QuadraticSpeedDependentTorque
          loadTorque(w_nominal=400, tau_nominal=-50.0)
          annotation (Placement(transformation(extent={{92,-10},{72,10}})));
        Modelica.Electrical.Analog.Sources.ConstantVoltage gen(V=100)
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=-90,
              origin={-60,28})));
        Modelica.Electrical.Analog.Basic.Ground ground
          annotation (Placement(transformation(extent={{-94,38},{-74,18}})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor powMech2
          annotation (Placement(transformation(extent={{12,-10},{32,10}})));
        Modelica.Electrical.Analog.Sensors.PowerSensor powElec
          annotation (Placement(transformation(extent={{-48,32},{-28,52}})));
        MBTwoFlange twoFlanges(
          powMax=10000,
          tauMax=50,
          J=0.5,
          wMax=300)
          annotation (Placement(transformation(extent={{-18,-10},{2,10}})));
        Modelica.Mechanics.Rotational.Sources.ConstantTorque tau1(tau_constant=
              -5.0)
          annotation (Placement(transformation(extent={{-76,-10},{-56,10}})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor powMech1
          annotation (Placement(transformation(extent={{-28,-10},{-48,10}})));
        Modelica.Blocks.Sources.Trapezoid tauRef(
          rising=10,
          width=10,
          falling=10,
          period=1e6,
          startTime=10,
          amplitude=50,
          offset=20)
          annotation (Placement(transformation(extent={{-40,-48},{-20,-28}})));
      equation
        connect(inertia.flange_b, loadTorque.flange) annotation (Line(
            points={{58,0},{72,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(ground.p, gen.n) annotation (Line(
            points={{-84,18},{-60,18}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(inertia.flange_a, powMech2.flange_b) annotation (Line(
            points={{38,0},{32,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(powElec.pc, gen.p) annotation (Line(
            points={{-48,42},{-60,42},{-60,38}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powElec.pv, powElec.nc) annotation (Line(
            points={{-38,52},{-28,52},{-28,42}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powMech2.flange_a, twoFlanges.flange_b) annotation (Line(
            points={{12,0},{8,0},{8,-0.2},{2,-0.2}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(powElec.nc, twoFlanges.pin_n) annotation (Line(
            points={{-28,42},{-4,42},{-4,10}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(twoFlanges.pin_p, gen.n) annotation (Line(
            points={{-12,9.8},{-12,18},{-60,18}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(powElec.nv, gen.n) annotation (Line(
            points={{-38,32},{-38,18},{-60,18}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(twoFlanges.flange_a, powMech1.flange_a) annotation (Line(
            points={{-18,0},{-28,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(tau1.flange, powMech1.flange_b) annotation (Line(
            points={{-56,0},{-48,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(tauRef.y, twoFlanges.tauRefInt) annotation (Line(
            points={{-19,-38},{-8,-38},{-8,-11.4}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-60},
                  {100,60}}), graphics),
          experiment(StopTime=50),
          __Dymola_experimentSetupOutput,
          Icon(coordinateSystem(extent={{-100,-100},{100,100}})),
          Documentation(info="<html>
<p>This is a simple test of model MBTwoFlange.</p>
<p>It shows that the generated torque follows the normalised torque request as long as it does not overcome torque and power limits. The generted torque will act on the machine inertia in conjunction with the toruqes applied from the exterior to the two flanges.</p>
<p>It shows also the effects of efficiency on the DC power.</p>
<p>Finally it shows that MBTwoFlange model automatically reduces torque if the maximum speed tends to be overcome.</p>
<p><u>First suggested plots</u>: a plot with tauRef.y and twoFlanges.inertia.tau; another with twoFlanges.limTau.state; a third one withinertia.wIn these plots it can be seen that:</p>
<ul>
<li>during the first 15 seconds the generated torque equals the torque request tauRef.y</li>
<li>starting from 13.1 s state becomes 1 meaning that the base speed has been overcome and therefore it is impossible to deliver the maximum drive torque</li>
<li>starting from t=15s the power limitation activates, and the delivered torque is lower than the torque request</li>
<li>starting from t= 30 s the maximum speed is reached, and torque is limitated not to overcome this maximum.</li>
</ul>
<p><u>Second suggested plot</u>: Once the first plots are anaysed, the user might want to have an idea of the mechanical and electrical powers: these are seen putting in the same plot ( powMech1.power+powMech2.power) and powElec.power.</p>
</html>"));
      end TestMBTwoFlange;
    end TestingModels;
    annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
              -100},{100,100}}), graphics={
          Line(
            points={{-80,-84},{-80,68}},
            color={0,0,0},
            smooth=Smooth.None),
          Line(
            points={{-88,-80},{78,-80}},
            color={0,0,0},
            smooth=Smooth.None),
          Polygon(
            points={{94,-80},{78,-74},{78,-86},{94,-80}},
            lineColor={0,0,0},
            smooth=Smooth.None),
          Polygon(
            points={{8,0},{-8,6},{-8,-6},{8,0}},
            lineColor={0,0,0},
            smooth=Smooth.None,
            origin={-80,76},
            rotation=90),
          Line(
            points={{-84,40},{-14,40}},
            color={0,0,0},
            smooth=Smooth.None),
          Line(
            points={{-14,40},{-4,2},{22,-32},{62,-44},{62,-80}},
            color={0,0,0},
            smooth=Smooth.None)}));
  end MapBased;

  package ElectricDrives "Electric drives components and models"
    extends Modelica.Icons.Package;

    model AMDrive "asynchronous machine-based electric drive"
      import PI = Modelica.Constants.pi;
      Modelica.SIunits.Torque tauMaximum=3*uAgGen.uAg^2*pp/(2*(2*PI*uAgGen.f)^2
          *(L1 + L2)) "Maximum torque at given frequency and voltage";
      Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a annotation (
          Placement(transformation(extent={{90,-10},{110,10}}),
            iconTransformation(extent={{90,-10},{110,10}})));
      parameter Integer pp=2 "pole pairs";
      parameter Modelica.SIunits.Voltage UBase=230
        "Base RMS machine line voltage";
      parameter Modelica.SIunits.AngularFrequency WeBase=314.15
        "Base machine angular frequency ";
      parameter Modelica.SIunits.AngularFrequency WeMax=314.15
        "Maximum machine angular frequency";
      parameter Modelica.SIunits.Voltage Unom=200
        "DC nominal voltage (only order of magnitude needed)";
      parameter Modelica.SIunits.Resistance R1=0.435
        "Stator's phase resistance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Inductance L1=0.004
        "Stator's leakage inductance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Inductance Lm=0.0693
        "Stator's leakage inductance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Resistance R2=0.4 "Rotor's phase resistance "
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Inductance L2=0.002
        "Rotor's leakage inductance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.MomentOfInertia J=2.0
        "Rotor's moment of inertia"
        annotation (Dialog(tab="machine parameters"));
      parameter Real lossFact=4
        "Ratio of inverter losses (W) to machine current (A)";
      Modelica.Electrical.Analog.Interfaces.PositivePin pin_p annotation (
          Placement(transformation(extent={{-110,50},{-90,70}}),
            iconTransformation(extent={{-112,50},{-92,70}})));
      Modelica.Electrical.Analog.Interfaces.NegativePin pin_n annotation (
          Placement(transformation(extent={{-110,-70},{-90,-50}}),
            iconTransformation(extent={{-112,-70},{-92,-50}})));
      SupportModels.Internal.ConstPDC dcLoad(T=0.01, k=1000/Unom)
        annotation (Placement(transformation(extent={{-100,-10},{-80,10}})));
      Modelica.Blocks.Interfaces.RealInput dWe annotation (Placement(
            transformation(
            extent={{-20,-20},{20,20}},
            rotation=270,
            origin={0,98}), iconTransformation(
            extent={{-20,-20},{20,20}},
            rotation=270,
            origin={0,88})));
      Modelica.Blocks.Nonlinear.Limiter limDWe(uMax=R2/(L1 + L2), uMin=-R2/(L1
             + L2)) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={50,40})));
      EDsupport.QSAsma qSAsma(
        pp=pp,
        R1=R1,
        L1=L1,
        Lm=Lm,
        R2=R2,
        L2=L2,
        J=J) annotation (Placement(visible=true, transformation(extent={{22,-42},
                {46,-22}}, rotation=0)));
      Modelica.Blocks.Math.Add addPdc annotation (Placement(visible=true,
            transformation(
            origin={-58,-54},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Blocks.Math.Gain polePairs(k=pp) annotation (Placement(visible=
              true, transformation(
            origin={38,0},
            extent={{10,-10},{-10,10}},
            rotation=0)));
      Modelica.Mechanics.Rotational.Sensors.SpeedSensor Wm annotation (
          Placement(visible=true, transformation(
            origin={74,0},
            extent={{10,-10},{-10,10}},
            rotation=0)));
      Modelica.Blocks.Math.Gain lossF_(k=lossFact) annotation (Placement(
            visible=true, transformation(
            origin={-18,-64},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      EDsupport.UagGenerator uAgGen(
        WeBase=WeBase,
        WeMax=WeMax,
        UBase=UBase)
        annotation (Placement(transformation(extent={{14,0},{-6,20}})));
      Modelica.Electrical.Analog.Sensors.VoltageSensor vDCSens annotation (
          Placement(transformation(
            extent={{-10,10},{10,-10}},
            rotation=-90,
            origin={-60,30})));
    equation
      connect(qSAsma.Is, lossF_.u) annotation (Line(points={{39.4545,-43},{
              39.4545,-64},{-6,-64}}, color={0,0,127}));
      connect(lossF_.y, addPdc.u1) annotation (Line(points={{-29,-64},{-38,-64},
              {-38,-60},{-46,-60}}, color={0,0,127}));
      connect(Wm.flange, flange_a)
        annotation (Line(points={{84,0},{84,0},{100,0}}));
      connect(polePairs.u, Wm.w)
        annotation (Line(points={{50,0},{50,0},{63,0}}, color={0,0,127}));
      connect(addPdc.u2, qSAsma.Pdc) annotation (Line(points={{-46,-48},{
              26.3636,-48},{26.3636,-43}}, color={0,0,127}));
      connect(addPdc.y, dcLoad.Pref) annotation (Line(points={{-69,-54},{-76,-54},
              {-76,0},{-81.8,0}}, color={0,0,127}));
      connect(qSAsma.flange_a, flange_a)
        annotation (Line(points={{43.8182,-32},{90,-32},{90,0},{100,0}}));
      connect(dcLoad.pin_p, pin_p) annotation (Line(
          points={{-90,10},{-90,60},{-100,60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(dcLoad.pin_n, pin_n) annotation (Line(
          points={{-90,-9.8},{-90,-60},{-100,-60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(limDWe.u, dWe) annotation (Line(
          points={{62,40},{74,40},{74,66},{0,66},{0,98}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(uAgGen.dWe, limDWe.y) annotation (Line(
          points={{15.6,16},{20,16},{20,40},{39,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(uAgGen.We, polePairs.y) annotation (Line(
          points={{15.6,4},{22,4},{22,0},{27,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(vDCSens.v, uAgGen.dcVoltage) annotation (Line(
          points={{-50,30},{4,30},{4,21.6}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(vDCSens.p, pin_p) annotation (Line(
          points={{-60,40},{-60,60},{-100,60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(vDCSens.n, pin_n) annotation (Line(
          points={{-60,20},{-60,-20},{-90,-20},{-90,-60},{-100,-60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(uAgGen.f, qSAsma.f) annotation (Line(
          points={{-7,4},{-20,4},{-20,-38},{19.8182,-38}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(qSAsma.Uag, uAgGen.uAg) annotation (Line(
          points={{19.8182,-26},{-26,-26},{-26,16},{-7,16}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Placement(transformation(extent={{-80,72},{-60,92}})),
        Dialog(tab="Other", group="Inverter"),
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-80},
                {100,80}}), graphics={Rectangle(
                  extent={{-36,60},{82,-14}},
                  lineColor={255,0,0},
                  pattern=LinePattern.Dash)}),
        Documentation(info="<html>
<p>This model models an asynchronous machine - based electric drive, containing U/f control, with stator resistance drop compensation.</p>
<p>It makes usage of the quasi-stationary asynchornous machine model QSAsma.</p>
<p>The model can operate only with positive angular speeds.</p>
<p>Full description of the model can be found in the following paper.</p>
<p>M. Ceraolo:  &QUOT;A new Modelica Electric and Hybrid Power Trains library&QUOT; <i>11th Modelica Conference, 2015 Versailles - France</i></p>
</html>"),
        experiment(StopTime=300, Interval=0.1),
        __Dymola_experimentSetupOutput,
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=true,
            initialScale=0.1,
            grid={2,2}), graphics={Line(points={{-28,20},{6,20}}, color={0,0,
              255}),Line(points={{-30,0},{4,0}}, color={0,0,255}),Line(points={
              {-30,-20},{4,-20}}, color={0,0,255}),Text(
                  lineColor={0,0,255},
                  fillColor={95,95,95},
                  fillPattern=FillPattern.Solid,
                  extent={{-140,-112},{148,-146}},
                  textString="%name"),Line(points={{-102,-60},{-78,-60},{-78,-28},
              {-60,-28}}, color={0,0,255}),Line(points={{-96,60},{-78,60},{-78,
              28},{-60,28}}, color={0,0,255}),Rectangle(
                  fillColor={175,175,175},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-40,68},{80,-52}}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-40,68},{-62,-52}}),Polygon(fillPattern=FillPattern.Solid,
              points={{-54,-82},{-44,-82},{-14,-12},{36,-12},{66,-82},{76,-82},
              {76,-92},{-54,-92},{-54,-82}}),Rectangle(
                  fillColor={95,95,95},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{80,12},{100,-8}})}));
    end AMDrive;

    model AMDrivePU
      "asynchronous machine-based electric drive (parameters in per-unit)"
      import PI = Modelica.Constants.pi;
      Modelica.SIunits.Torque tauMaximum
        "Maximum torque at given frequency and voltage";
      Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a annotation (
          Placement(transformation(extent={{92,-10},{112,10}}),
            iconTransformation(extent={{92,-10},{112,10}})));
      // General parameters
      parameter Real UBase=400 "Base RMS machine line voltage";
      parameter Real WeBase=314.15 "Base machine angular frequency";
      parameter Real WeMax=2*WeBase "Max machine angular frequency";
      parameter Real Unom=400 "PU reference RMS machine line voltage"
        annotation (Dialog(group="p.u. reference quantities"));
      parameter Modelica.SIunits.ApparentPower Snom=1e5
        "Reference power pf p.u."
        annotation (Dialog(group="p.u. reference quantities"));
      parameter Modelica.SIunits.Frequency FNom=50
        "Reference frequency of p.u."
        annotation (Dialog(group="p.u. reference quantities"));
      // Machine parameters
      parameter Real R1u=0.01 "Stator phase resistance " annotation (Dialog(tab
            ="Machine", group="Resistances and inductances per phase"));
      parameter Real X1u=0.05 "Stator leackage inductance" annotation (Dialog(
            tab="Machine", group="Resistances and inductances per phase"));
      parameter Real R2u=0.01 "Rotor phase resistance referred  to primary"
        annotation (Dialog(tab="Machine", group=
              "Resistances and inductances per phase"));
      parameter Real X2u=0.05 "Rotor leackage inductance" annotation (Dialog(
            tab="Machine", group="Resistances and inductances per phase"));
      parameter Real Xmu=10 "Magnetic coupling inductance" annotation (Dialog(
            tab="Machine", group="Resistances and inductances per phase"));
      //  parameter Real Rmu=10 "Iron loss equivalent resistance (Zm=Rm//Xm)"
      //  annotation(Dialog(tab="Machine",group="Resistances and inductances per phase"));
      parameter Modelica.SIunits.Time Hu=5 "Inertia constant (s)"
        annotation (Dialog(tab="Machine", group="Other parameters"));
      parameter Integer pp(min=1) = 1 "Number of pole pairs"
        annotation (Dialog(tab="Machine", group="Other parameters"));
      // Other/Inverter
      parameter Modelica.SIunits.Time TInv=0.01 "Inverter time constant"
        annotation (Dialog(tab="Other", group="Inverter"));
      parameter Real lossFact=4
        "Ratio of inverter losses (W) to machine current (A)";
      // Other/Load
      parameter Real KL=1 "Inner DCload PI k constant (adimens.)"
        annotation (Dialog(tab="Other", group="DCLoad"));
      parameter Modelica.SIunits.Time TL=0.001 "Inner DCload PI time constant"
        annotation (Dialog(tab="Other", group="DCLoad"));
    protected
      parameter Real UBase1=UBase/sqrt(3);
      //single-circuit equivalent of UBase
      parameter Real fContr=UBase1/(WeBase/(2*PI));
      //constant U/f
      parameter Real WeNom=2*PI*FNom;
      parameter Real WmNom=WeNom/pp;
      parameter Real Znom=Unom^2/Snom;
      parameter Modelica.SIunits.Resistance R1=R1u*Znom;
      parameter Modelica.SIunits.Inductance L1=X1u*Znom/WeNom;
      parameter Modelica.SIunits.Inductance Lm=Xmu*Znom/WeNom;
      parameter Modelica.SIunits.Resistance R2=R2u*Znom;
      parameter Modelica.SIunits.Inductance L2=X2u*Znom/WeNom;
      //  parameter Modelica.SIunits.Resistance Rm=Rmu*Z_nom;
      parameter Modelica.SIunits.MomentOfInertia J=2*Hu*Snom/WmNom^2;
    public
      Modelica.Blocks.Math.Add addPdc annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-58,-56})));
      Modelica.Blocks.Math.Gain LossF_(k=lossFact) annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-18,-66})));
      Modelica.Electrical.Analog.Interfaces.PositivePin pin_p annotation (
          Placement(transformation(extent={{-110,50},{-90,70}}),
            iconTransformation(extent={{-112,50},{-92,70}})));
      Modelica.Electrical.Analog.Interfaces.NegativePin pin_n annotation (
          Placement(transformation(extent={{-110,-70},{-90,-50}}),
            iconTransformation(extent={{-112,-70},{-92,-50}})));
      SupportModels.Internal.ConstPDC dcLoad(k=KL, T=TL)
        annotation (Placement(transformation(extent={{-100,-10},{-80,10}})));
      EDsupport.QSAsma qSAsma(
        pp=pp,
        R1=R1,
        L1=L1,
        Lm=Lm,
        R2=R2,
        L2=L2,
        J=J) annotation (Placement(transformation(extent={{22,-42},{46,-22}})));
      Modelica.Blocks.Interfaces.RealInput dWe annotation (Placement(
            transformation(
            extent={{-20,-20},{20,20}},
            rotation=270,
            origin={0,100}), iconTransformation(
            extent={{-20,-20},{20,20}},
            rotation=270,
            origin={0,88})));
      Modelica.Blocks.Nonlinear.Limiter limDWe1(uMax=R2/(L1 + L2), uMin=-R2/(L1
             + L2)) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={44,42})));
      Modelica.Blocks.Math.Gain polePairs(k=pp) annotation (Placement(visible=
              true, transformation(
            origin={30,6},
            extent={{10,-10},{-10,10}},
            rotation=0)));
      Modelica.Mechanics.Rotational.Sensors.SpeedSensor Wm1 annotation (
          Placement(visible=true, transformation(
            origin={68,0},
            extent={{10,-10},{-10,10}},
            rotation=0)));
      EDsupport.UagGenerator uAgGen(
        WeBase=WeBase,
        WeMax=WeMax,
        UBase=UBase)
        annotation (Placement(transformation(extent={{8,2},{-12,22}})));
      Modelica.Electrical.Analog.Sensors.VoltageSensor vDCSens annotation (
          Placement(transformation(
            extent={{-10,10},{10,-10}},
            rotation=-90,
            origin={-68,32})));
    equation
      if uAgGen.f < 1e-12 then
        tauMaximum = 1e-9;
      else
        tauMaximum = 3*uAgGen.uAg^2*pp/(2*(2*PI*uAgGen.f)^2*(L1 + L2));
      end if;
      connect(dcLoad.pin_p, pin_p) annotation (Line(
          points={{-90,10},{-90,60},{-100,60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(dcLoad.pin_n, pin_n) annotation (Line(
          points={{-90,-9.8},{-90,-60},{-100,-60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(qSAsma.flange_a, flange_a) annotation (Line(
          points={{43.8182,-32},{82,-32},{82,0},{102,0}},
          color={0,0,0},
          smooth=Smooth.None));
      connect(addPdc.y, dcLoad.Pref) annotation (Line(
          points={{-69,-56},{-76,-56},{-76,0},{-81.8,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(addPdc.u2, qSAsma.Pdc) annotation (Line(
          points={{-46,-50},{26.3636,-50},{26.3636,-43}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(LossF_.y, addPdc.u1) annotation (Line(
          points={{-29,-66},{-38,-66},{-38,-62},{-46,-62}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(qSAsma.Is, LossF_.u) annotation (Line(
          points={{39.4545,-43},{39.4545,-66},{-6,-66}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(polePairs.u, Wm1.w) annotation (Line(points={{42,6},{42,6},{48,6},
              {48,0},{57,0}}, color={0,0,127}));
      connect(limDWe1.u, dWe) annotation (Line(
          points={{56,42},{66,42},{66,72},{0,72},{0,100}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(uAgGen.dWe, limDWe1.y) annotation (Line(
          points={{9.6,18},{14,18},{14,42},{33,42}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(uAgGen.We, polePairs.y) annotation (Line(
          points={{9.6,6},{19,6}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(uAgGen.f, qSAsma.f) annotation (Line(
          points={{-13,6},{-26,6},{-26,-38},{19.8182,-38}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(qSAsma.Uag, uAgGen.uAg) annotation (Line(
          points={{19.8182,-26},{-32,-26},{-32,18},{-13,18}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(Wm1.flange, flange_a) annotation (Line(
          points={{78,0},{102,0}},
          color={0,0,0},
          smooth=Smooth.None));
      connect(vDCSens.p, pin_p) annotation (Line(
          points={{-68,42},{-68,60},{-100,60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(vDCSens.n, pin_n) annotation (Line(
          points={{-68,22},{-68,-30},{-90,-30},{-90,-60},{-100,-60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(vDCSens.v, uAgGen.dcVoltage) annotation (Line(
          points={{-58,32},{-2,32},{-2,23.6}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Dialog(tab="Other", group="Inverter"),
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-80},
                {100,80}}), graphics={Rectangle(
                  extent={{-42,62},{76,-12}},
                  lineColor={255,0,0},
                  pattern=LinePattern.Dash)}),
        Documentation(info="<html>
<p>This model models an asynchronous machine - based electric drive, containing U/f control, with stator resistance drop compensation.</p>
<p>It makes usage of the quasi-stationary asynchornous machine model QSAsma.</p>
<p>The model can operate only with positive angular speeds.</p>
</html>"),
        experiment(StopTime=300, Interval=0.1),
        __Dymola_experimentSetupOutput,
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=true,
            initialScale=0.1,
            grid={2,2}), graphics={Line(points={{-28,20},{6,20}}, color={0,0,
              255}),Line(points={{-30,0},{4,0}}, color={0,0,255}),Line(points={
              {-30,-20},{4,-20}}, color={0,0,255}),Text(
                  lineColor={0,0,255},
                  fillColor={95,95,95},
                  fillPattern=FillPattern.Solid,
                  extent={{-140,-112},{148,-146}},
                  textString="%name"),Line(points={{-102,-60},{-78,-60},{-78,-28},
              {-60,-28}}, color={0,0,255}),Line(points={{-96,60},{-78,60},{-78,
              28},{-60,28}}, color={0,0,255}),Rectangle(
                  fillColor={175,175,175},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-40,68},{80,-52}}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-40,68},{-62,-52}}),Polygon(fillPattern=FillPattern.Solid,
              points={{-54,-82},{-44,-82},{-14,-12},{36,-12},{66,-82},{76,-82},
              {76,-92},{-54,-92},{-54,-82}}),Rectangle(
                  fillColor={95,95,95},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{80,12},{100,-8}}),Text(
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-40,40},{68,18}},
                  textString="P.U.")}));
    end AMDrivePU;

    model PMDrive
      parameter Real Rs=0.03 "stator resistance (ohm)";
      parameter Real Ipm=sqrt(2)*200 "PM equivalent current (=psi/Ld)";
      parameter Integer pp=2 "pole pairs";
      parameter Real Ld=1.837e-3 "direct-axis inductance (H)", Lq=1.837e-3
        "quadrature axis inductance (H)";
      parameter Real Inom=100 "nominal current (rms per phase)";
      parameter Real Unom=100 "nominal voltage (rms per phase)";
      parameter Real kInvLoss=6 "Inverter losses (W/(ampere AC rms))";
      parameter Real J=0.29 "Moment of inertia (kg.m^2)";
      SupportModels.PmsmAllFluxLimI allFluxLim(
        Rs=Rs,
        Ipm=Ipm,
        pp=pp,
        Ld=Ld,
        Lq=Lq,
        Unom=Unom,
        Inom=Inom)
        annotation (Placement(transformation(extent={{-52,-10},{-32,10}})));
      Modelica.Mechanics.Rotational.Sources.Torque torque
        annotation (Placement(transformation(extent={{-14,-10},{6,10}})));
      Modelica.Mechanics.Rotational.Components.Inertia inertia(phi(fixed=true,
            start=0), J=J)
        annotation (Placement(transformation(extent={{58,-10},{78,10}})));
      Modelica.Mechanics.Rotational.Sensors.SpeedSensor speedSensor annotation
        (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={16,-18})));
      Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a
        annotation (Placement(transformation(extent={{90,-10},{110,10}})));
      Modelica.Blocks.Interfaces.RealInput tauRef
        annotation (Placement(transformation(extent={{-126,-20},{-86,20}})));
      SupportModels.DriveLosses invLoss(Rs=Rs, Kv=kInvLoss)
        annotation (Placement(transformation(extent={{-16,18},{4,38}})));
      Modelica.Blocks.Math.Add add(k1=1, k2=1) annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={26,46})));
      SupportModels.Internal.ConstPDC dCLConstP(k=1, T=0.001) annotation (
          Placement(transformation(
            extent={{10,-10},{-10,10}},
            rotation=90,
            origin={0,90})));
      Modelica.Mechanics.Rotational.Sensors.PowerSensor powerSensor
        annotation (Placement(transformation(extent={{30,10},{50,-10}})));
      Modelica.Electrical.Analog.Interfaces.PositivePin pin_p annotation (
          Placement(transformation(extent={{-70,90},{-50,110}}),
            iconTransformation(extent={{-70,90},{-50,110}})));
      Modelica.Electrical.Analog.Interfaces.NegativePin pin_n annotation (
          Placement(transformation(extent={{50,90},{70,110}}),
            iconTransformation(extent={{50,90},{70,110}})));
      Modelica.Electrical.Analog.Sensors.VoltageSensor uDCSens
        annotation (Placement(transformation(extent={{-40,54},{-20,74}})));
    equation
      connect(speedSensor.w, allFluxLim.wMechanical) annotation (Line(
          points={{16,-29},{16,-34},{-64,-34},{-64,-6},{-52.6,-6}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(allFluxLim.tauElectrical, torque.tau) annotation (Line(
          points={{-31.2,-6},{-26,-6},{-26,0},{-16,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(inertia.flange_b, flange_a) annotation (Line(
          points={{78,0},{100,0}},
          color={0,0,0},
          smooth=Smooth.None));
      connect(allFluxLim.tauRef, tauRef) annotation (Line(
          points={{-52.8,6},{-80,6},{-80,0},{-106,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(invLoss.Is, allFluxLim.Is) annotation (Line(
          points={{-16.8,28},{-24,28},{-24,6},{-31.4,6}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(add.u1, invLoss.losses) annotation (Line(
          points={{20,34},{20,28},{4.4,28}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(inertia.flange_a, powerSensor.flange_b) annotation (Line(
          points={{58,0},{50,0}},
          color={0,0,0},
          smooth=Smooth.None));
      connect(torque.flange, powerSensor.flange_a) annotation (Line(
          points={{6,0},{30,0}},
          color={0,0,0},
          smooth=Smooth.None));
      connect(speedSensor.flange, powerSensor.flange_a) annotation (Line(
          points={{16,-8},{16,0},{30,0}},
          color={0,0,0},
          smooth=Smooth.None));
      connect(powerSensor.power, add.u2) annotation (Line(
          points={{32,11},{32,34}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(add.y, dCLConstP.Pref) annotation (Line(
          points={{26,57},{26,80},{0,80},{0,81.8}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(dCLConstP.pin_p, pin_p) annotation (Line(
          points={{-10,90},{-60,90},{-60,100}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(dCLConstP.pin_n, pin_n) annotation (Line(
          points={{9.8,90},{60,90},{60,100}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(uDCSens.v, allFluxLim.uDC) annotation (Line(
          points={{-30,54},{-30,28},{-64,28},{-64,0},{-53,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(uDCSens.p, pin_p) annotation (Line(
          points={{-40,64},{-44,64},{-44,80},{-44,80},{-44,90},{-60,90},{-60,
              100}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(uDCSens.n, pin_n) annotation (Line(
          points={{-20,64},{46,64},{46,90},{60,90},{60,100}},
          color={0,0,255},
          smooth=Smooth.None));
      annotation (
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-60},
                {100,100}}), graphics),
        experiment(StopTime=5),
        __Dymola_experimentSetupOutput,
        Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{
                100,100}}), graphics={Rectangle(
                  extent={{72,10},{98,-10}},
                  lineColor={0,0,0},
                  fillPattern=FillPattern.HorizontalCylinder,
                  fillColor={95,95,95}),Line(
                  points={{-118,0},{-56,0}},
                  color={0,0,127},
                  smooth=Smooth.None,
                  thickness=0.5),Rectangle(
                  extent={{-44,60},{76,-60}},
                  lineColor={0,0,0},
                  fillPattern=FillPattern.HorizontalCylinder,
                  fillColor={135,135,135}),Rectangle(
                  extent={{-44,60},{-64,-60}},
                  lineColor={0,0,0},
                  fillPattern=FillPattern.HorizontalCylinder,
                  fillColor={0,0,127}),Rectangle(
                  extent={{-64,70},{76,48}},
                  lineColor={95,95,95},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-54,-90},{-44,-90},{-14,-20},{36,-20},{66,-90},{76,-90},
                {76,-100},{-54,-100},{-54,-90}},
                  lineColor={0,0,0},
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid),Line(points={{58,100},{40,100},
              {40,70}}, color={0,0,255}),Ellipse(
                  extent={{-24,44},{44,-24}},
                  lineColor={0,0,0},
                  fillColor={215,215,215},
                  fillPattern=FillPattern.Solid),Rectangle(
                  extent={{10,20},{40,0}},
                  lineColor={0,0,0},
                  fillColor={255,0,0},
                  fillPattern=FillPattern.Solid),Rectangle(
                  extent={{-20,20},{10,0}},
                  lineColor={0,0,0},
                  fillColor={0,255,0},
                  fillPattern=FillPattern.Solid),Line(points={{-64,100},{-40,
              100},{-40,66}}, color={0,0,255}),Text(
                  extent={{-100,-110},{100,-150}},
                  lineColor={0,0,255},
                  textString="%name")}),
        Documentation(info="<html>
<p>Model of Permanent-Magnet Sychronous Machine based drive trains.</p>
<p>It operates at the optimal angle between PM and stator flux, both in full flux and flux weakening behaviour.</p>
<p>Full description of the model can be found in the following paper.</p>
<p>M. Ceraolo &QUOT;Modelica synchronous electric drive model for EV and HEV simulations&QUOT; submitted for publication to the IEEE transactions on Vehicular Tehnologies.</p>
</html>"));
    end PMDrive;

    model AMDriveOLD "asynchronous machine-based electric drive"
      import PI = Modelica.Constants.pi;
      Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a annotation (
          Placement(transformation(extent={{90,-10},{110,10}}),
            iconTransformation(extent={{90,-10},{110,10}})));
      parameter Integer pp=2 "pole pairs";
      parameter Modelica.SIunits.Voltage UBase=230
        "Base RMS machine line voltage";
      parameter Modelica.SIunits.AngularFrequency WeBase=314.15
        "Base machine angular frequency ";
      parameter Modelica.SIunits.AngularFrequency WeMax=314.15
        "Base machine angular frequency";
      parameter Modelica.SIunits.Voltage Unom=200
        "DC nominal voltage (only order of magnitude needed)";
      parameter Modelica.SIunits.Resistance R1=0.435
        "Stator's phase resistance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Inductance L1=0.004
        "Stator's leakage inductance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Inductance Lm=0.0693
        "Stator's leakage inductance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Resistance R2=0.4 "Rotor's phase resistance "
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.Inductance L2=0.002
        "Rotor's leakage inductance"
        annotation (Dialog(tab="machine parameters"));
      parameter Modelica.SIunits.MomentOfInertia J=2.0
        "Rotor's moment of inertia"
        annotation (Dialog(tab="machine parameters"));
      parameter Real lossFact=4
        "Ratio of inverter losses (W) to machine current (A)";
      Modelica.Electrical.Analog.Interfaces.PositivePin pin_p annotation (
          Placement(transformation(extent={{-110,50},{-90,70}}),
            iconTransformation(extent={{-112,50},{-92,70}})));
      Modelica.Electrical.Analog.Interfaces.NegativePin pin_n annotation (
          Placement(transformation(extent={{-110,-70},{-90,-50}}),
            iconTransformation(extent={{-112,-70},{-92,-50}})));
      EHPT.SupportModels.Internal.ConstPDC dcLoad(T=0.01, k=1000/Unom)
        annotation (Placement(transformation(extent={{-100,-10},{-80,10}})));
      Modelica.Blocks.Interfaces.RealInput dWe annotation (Placement(
            transformation(
            extent={{-20,-20},{20,20}},
            rotation=270,
            origin={0,120}), iconTransformation(
            extent={{-20,-20},{20,20}},
            rotation=270,
            origin={0,88})));
      Modelica.Blocks.Nonlinear.Limiter limDWe(uMax=R2/(L1 + L2), uMin=-R2/(L1
             + L2)) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=270,
            origin={70,76})));
      Modelica.Blocks.Math.Gain ToFreq(k=1/(2*PI)) annotation (Placement(
            visible=true, transformation(
            origin={-30,46},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Blocks.Math.Gain gain(k=fContr) annotation (Placement(visible=
              true, transformation(extent={{-38,-4},{-18,16}}, rotation=0)));
      Modelica.Blocks.Sources.RealExpression uAG_(y=innerUag) annotation (
          Placement(visible=true, transformation(extent={{-10,-4},{10,16}},
              rotation=0)));
      Modelica.Blocks.Nonlinear.Limiter limWe(uMax=WeMax) annotation (Placement(
            visible=true, transformation(extent={{16,36},{-4,56}}, rotation=0)));
      EDsupport.QSAsma qSAsma(
        pp=pp,
        R1=R1,
        L1=L1,
        Lm=Lm,
        R2=R2,
        L2=L2,
        J=J) annotation (Placement(visible=true, transformation(extent={{22,-32},
                {46,-12}}, rotation=0)));
      Modelica.Blocks.Math.Add add annotation (Placement(visible=true,
            transformation(
            origin={40,46},
            extent={{10,-10},{-10,10}},
            rotation=0)));
      Modelica.Blocks.Math.Add addPdc annotation (Placement(visible=true,
            transformation(
            origin={-58,-54},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Blocks.Math.Gain polePairs(k=pp) annotation (Placement(visible=
              true, transformation(
            origin={62,22},
            extent={{10,-10},{-10,10}},
            rotation=270)));
      Modelica.Mechanics.Rotational.Sensors.SpeedSensor Wm annotation (
          Placement(visible=true, transformation(
            origin={62,-6},
            extent={{10,-10},{-10,10}},
            rotation=270)));
      Modelica.Blocks.Math.Gain lossF_(k=lossFact) annotation (Placement(
            visible=true, transformation(
            origin={-18,-64},
            extent={{-10,-10},{10,10}},
            rotation=180)));
    protected
      parameter Real UBase1=UBase/sqrt(3);
      //single-circuit equivalent of UBase
      parameter Real fContr=UBase1/(WeBase/(2*PI));
      //constant  U/f
      parameter Real WeNom=WeBase;
      parameter Real WmNom=WeNom/pp;
      Real innerUag;
    equation
      connect(qSAsma.Is, lossF_.u) annotation (Line(points={{39.4545,-33},{
              39.4545,-64},{-6,-64}}, color={0,0,127}));
      connect(lossF_.y, addPdc.u1) annotation (Line(points={{-29,-64},{-38,-64},
              {-38,-60},{-46,-60}}, color={0,0,127}));
      connect(Wm.flange, flange_a)
        annotation (Line(points={{62,-16},{62,-28},{80,-28},{80,0},{100,0}}));
      connect(polePairs.u, Wm.w)
        annotation (Line(points={{62,10},{62,5}}, color={0,0,127}));
      connect(polePairs.y, add.u2)
        annotation (Line(points={{62,33},{62,40},{52,40}}, color={0,0,127}));
      connect(addPdc.u2, qSAsma.Pdc) annotation (Line(points={{-46,-48},{
              26.3636,-48},{26.3636,-33}}, color={0,0,127}));
      connect(addPdc.y, dcLoad.Pref) annotation (Line(points={{-69,-54},{-76,-54},
              {-76,0},{-81.8,0}}, color={0,0,127}));
      connect(limWe.u, add.y)
        annotation (Line(points={{18,46},{29,46}}, color={0,0,127}));
      connect(limDWe.y, add.u1)
        annotation (Line(points={{70,65},{70,52},{52,52}}, color={0,0,127}));
      connect(qSAsma.Uag, uAG_.y) annotation (Line(points={{19.8182,-16},{
              17.0909,-16},{17.0909,6},{11,6}}, color={0,0,127}));
      connect(ToFreq.y, qSAsma.f) annotation (Line(points={{-41,46},{-58,46},{-58,
              -14},{-40,-14},{-40,-28},{19.8182,-28}}, color={0,0,127}));
      connect(gain.u, qSAsma.f) annotation (Line(points={{-40,6},{-40,-28},{
              19.8182,-28}}, color={0,0,127}));
      connect(qSAsma.flange_a, flange_a)
        annotation (Line(points={{43.8182,-22},{80,-22},{80,0},{100,0}}));
      connect(limWe.y, ToFreq.u)
        annotation (Line(points={{-5,46},{-18,46}}, color={0,0,127}));
      innerUag = if noEvent(abs(dcLoad.v)) > 2.5*gain.y then gain.y else
        noEvent(abs(dcLoad.v))/2.5;
      connect(dcLoad.pin_p, pin_p) annotation (Line(
          points={{-90,10},{-90,60},{-100,60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(dcLoad.pin_n, pin_n) annotation (Line(
          points={{-90,-9.8},{-90,-60},{-100,-60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(limDWe.u, dWe) annotation (Line(
          points={{70,88},{70,92},{0,92},{0,120}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Placement(transformation(extent={{-80,72},{-60,92}})),
        Dialog(tab="Other", group="Inverter"),
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-80},
                {100,100}}), graphics),
        Documentation(info="<html>
<p>This model models an asynchronous machine - based electric drive, containing U/f control, with stator resistance drop compensation.</p>
<p>It makes usage of the quasi-stationary asynchornous machine model QSAsma.</p>
<p>The model can operate only with positive angular speeds.</p>
<p>Full description of the model can be found in the following paper.</p>
<p>M. Ceraolo and F. Casella &QUOT;Asynchronous machine electric drives modelling for EV simulations using Modelica&QUOT; submitted for publication to <i>Simulation modelling practice and Theory</i>, Elsevier</p>
</html>"),
        experiment(StopTime=300, Interval=0.1),
        __Dymola_experimentSetupOutput,
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=true,
            initialScale=0.1,
            grid={2,2}), graphics={Line(points={{-28,20},{6,20}}, color={0,0,
              255}),Line(points={{-30,0},{4,0}}, color={0,0,255}),Line(points={
              {-30,-20},{4,-20}}, color={0,0,255}),Text(
                  lineColor={0,0,255},
                  fillColor={95,95,95},
                  fillPattern=FillPattern.Solid,
                  extent={{-140,-112},{148,-146}},
                  textString="%name"),Line(points={{-102,-60},{-78,-60},{-78,-28},
              {-60,-28}}, color={0,0,255}),Line(points={{-96,60},{-78,60},{-78,
              28},{-60,28}}, color={0,0,255}),Rectangle(
                  fillColor={175,175,175},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-40,68},{80,-52}}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-40,68},{-62,-52}}),Polygon(fillPattern=FillPattern.Solid,
              points={{-54,-82},{-44,-82},{-14,-12},{36,-12},{66,-82},{76,-82},
              {76,-92},{-54,-92},{-54,-82}}),Rectangle(
                  fillColor={95,95,95},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{80,12},{100,-8}})}));
    end AMDriveOLD;

    package TestingModels
      extends Modelica.Icons.ExamplesPackage;

      model TestAMDrive
        "Compares QSDrive with a drive based on the MSL asynchronous machine"
        //  extends Modelica.Icons.Example;
        import Modelica.Constants.pi;
        parameter Modelica.SIunits.AngularVelocity DeltaOmEl=25
          "Controller Delta Omega";
        //    Real Pac=aimc.plug_sp[i]
        Modelica.Blocks.Sources.Constant const1(k=-500) annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={90,-20})));
        Modelica.Electrical.Machines.Utilities.TerminalBox terminalBox
          annotation (Placement(transformation(extent={{-10,60},{10,80}})));
        Modelica.Electrical.Analog.Basic.Ground ground
          annotation (Placement(transformation(extent={{-90,14},{-70,34}})));
        Modelica.Electrical.MultiPhase.Basic.Star star annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-80,58})));
        Modelica.Mechanics.Rotational.Sources.Torque torque annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={74,40})));
        Modelica.Electrical.MultiPhase.Sources.SignalVoltage signalVoltage
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={-70,80})));
        Modelica.Mechanics.Rotational.Sensors.SpeedSensor wMot annotation (
            Placement(transformation(
              extent={{-7,-7},{7,7}},
              rotation=270,
              origin={59,19})));
        Modelica.Blocks.Sources.Constant dWe(k=25)
          annotation (Placement(transformation(extent={{-100,-10},{-80,10}})));
        Modelica.Mechanics.Rotational.Components.Inertia inertia(J=40)
          annotation (Placement(transformation(extent={{22,30},{42,50}})));
        TestingModels.AsmaUgenerator uVar(
          UBase=400,
          pp=2,
          WeBase=314.15,
          WeMax=314.15) annotation (Placement(transformation(
              extent={{-14,-10},{14,10}},
              rotation=90,
              origin={-50,46})));
        Modelica.Electrical.Machines.BasicMachines.AsynchronousInductionMachines.AIM_SquirrelCage
          aimc(
          p=2,
          fsNominal=50,
          Rs=0.016,
          Lssigma=0.07039/(100*pi),
          Lm=2.046/(100*pi),
          Lrsigma=0.07039/(100*pi),
          Rr=0.0313,
          Jr=0.341,
          TrOperational=293.15,
          TsOperational=293.15,
          TsRef=293.15,
          alpha20s=0)
          annotation (Placement(transformation(extent={{-10,30},{10,50}})));
        SupportModels.PTrifSensor pAC
          annotation (Placement(transformation(extent={{-42,70},{-22,90}})));
        Modelica.Electrical.Analog.Sensors.PowerSensor pDC annotation (
            Placement(visible=true, transformation(
              origin={-34,-34},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        AMDrive qSDrive(
          UBase=400,
          WeMax=314.15,
          Unom=800,
          R1=0.016,
          L1=0.07039/(100*pi),
          Lm=2.046/(100*pi),
          R2=0.0313,
          L2=0.07039/(100*pi),
          J=0.341,
          pp=2,
          lossFact=0) annotation (Placement(visible=true, transformation(
              origin={-2,-44},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Electrical.Analog.Sources.ConstantVoltage Udc(V=1000)
          annotation (Placement(visible=true, transformation(
              origin={-58,-44},
              extent={{-10,-10},{10,10}},
              rotation=270)));
        Modelica.Mechanics.Rotational.Components.Inertia inertia1(J=40)
          annotation (Placement(visible=true, transformation(
              origin={32,-44},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Electrical.Analog.Basic.Ground ground1 annotation (Placement(
              visible=true, transformation(
              origin={-58,-72},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Mechanics.Rotational.Sources.Torque torque1 annotation (
            Placement(visible=true, transformation(
              origin={58,-44},
              extent={{-10,-10},{10,10}},
              rotation=180)));
        Modelica.Blocks.Sources.Constant dWe1(k=23) annotation (Placement(
              visible=true, transformation(
              origin={24,-16},
              extent={{10,-10},{-10,10}},
              rotation=0)));
        Modelica.Blocks.Sources.Constant U1(k=15) annotation (Placement(visible
              =true, transformation(
              origin={-51,19},
              extent={{-8,-8},{8,8}},
              rotation=90)));
      equation
        connect(U1.y, uVar.U0) annotation (Line(points={{-51,27.8},{-51,34.7},{
                -50.1,34.7}}, color={0,0,127}));
        connect(torque1.tau, const1.y) annotation (Line(points={{70,-44},{74,-44},
                {74,-20},{79,-20},{79,-20}}, color={0,0,127}));
        connect(dWe1.y, qSDrive.dWe) annotation (Line(points={{13,-16},{-2,-16},
                {-2,-35.2}}, color={0,0,127}));
        connect(inertia1.flange_b, torque1.flange)
          annotation (Line(points={{42,-44},{48,-44}}));
        connect(ground1.p, Udc.n)
          annotation (Line(points={{-58,-62},{-58,-54}}, color={0,0,255}));
        connect(inertia1.flange_a, qSDrive.flange_a)
          annotation (Line(points={{22,-44},{8,-44}}));
        connect(pDC.nv, Udc.n) annotation (Line(points={{-34,-44},{-34,-62},{-58,
                -62},{-58,-54}}, color={0,0,255}));
        connect(pDC.pc, Udc.p)
          annotation (Line(points={{-44,-34},{-58,-34}}, color={0,0,255}));
        connect(qSDrive.pin_n, pDC.nv) annotation (Line(points={{-12.2,-50},{-18,
                -50},{-20,-50},{-20,-62},{-34,-62},{-34,-44}}, color={0,0,255}));
        connect(qSDrive.pin_p, pDC.nc) annotation (Line(points={{-12.2,-38},{-18,
                -38},{-18,-34},{-24,-34}}, color={0,0,255}));
        connect(pDC.pv, pDC.pc) annotation (Line(points={{-34,-24},{-44,-24},{-44,
                -34},{-44,-34}}, color={0,0,255}));
        connect(ground.p, star.pin_n) annotation (Line(
            points={{-80,34},{-80,48}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(signalVoltage.plug_n, star.plug_p) annotation (Line(
            points={{-80,80},{-80,68}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(wMot.flange, torque.flange) annotation (Line(
            points={{59,26},{59,33},{64,33},{64,40}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(torque.tau, const1.y) annotation (Line(
            points={{86,40},{92,40},{92,0},{72,0},{72,-20},{79,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(inertia.flange_b, torque.flange) annotation (Line(
            points={{42,40},{64,40}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(uVar.DWe, dWe.y) annotation (Line(
            points={{-38.7,40.1},{-36,40.1},{-36,0},{-79,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(uVar.U, signalVoltage.v) annotation (Line(
            points={{-54,57},{-54,68},{-70,68},{-70,73}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(uVar.Wm, wMot.w) annotation (Line(
            points={{-38.7,52.3},{-24,52.3},{-24,6},{59,6},{59,11.3}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(aimc.plug_sp, terminalBox.plug_sp) annotation (Line(
            points={{6,50},{6,60}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(aimc.plug_sn, terminalBox.plug_sn) annotation (Line(
            points={{-6,50},{-6,60}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(aimc.flange, inertia.flange_a) annotation (Line(
            points={{10,40},{22,40}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(signalVoltage.plug_p, pAC.pc) annotation (Line(
            points={{-60,80},{-42,80}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(pAC.nc, terminalBox.plugSupply) annotation (Line(
            points={{-22,80},{0,80},{0,62}},
            color={0,0,255},
            smooth=Smooth.None));
        annotation (
          experimentSetupOutput,
          Documentation(info="<html>
<p>This system simulates variable-frequency start-up of an asyncronous motor in two different ways.</p>
<p>The above system uses the MSL aimc, and a variable voltage variable frequency source obtained by controlled generators.</p>
<p><br>The system below is built around the QSDrive, that in turn is built along a machine model that makes usage of the Quasi-Stationary MSL library.</p>
<p>In both cases the motor supply is constituted by a three-phase system of quasi-sinusoidal shapes, created according to the following equations:</p>
<p>Wel=Wmecc*PolePairs+DeltaWel</p>
<p>U=U0+(Ubase-U0)*(Wel)/WElbase</p>
<p>where:</p>
<ul>
<li>U0, Ubase U, are initial, base, actual voltage amplitudes</li>
<li>Wmecc, Wel, are machine, mechanical and supply, electrical angular speeds</li>
<li>PolePairs are the machine pole pairs</li>
<li>DeltaWel is intended as a scaled indication of the requested torque. </li>
</ul>
<p><br>It is suggested to compare in the same plot aimc.tauElectrical and qSDrive.qSAsma.tauElectrical, as well as pDC.power and pAC.power</p>
</html>"),
          experimentSetupOutput,
          Icon(coordinateSystem(
              extent={{-100,-80},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2})),
          Diagram(coordinateSystem(
              extent={{-100,-80},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics),
          experiment(StopTime=6, Interval=0.0015));
      end TestAMDrive;

      model TestAMDrive2
        "Compares QSDrive with a drive based on the MSL asynchronous machine; woing beyond base speed"
        //  extends Modelica.Icons.Example;
        import Modelica.Constants.pi;
        parameter Real factor=1.4;
        parameter Modelica.SIunits.AngularVelocity DeltaOmEl=25
          "Controller Delta Omega";
        //    Real Pac=aimc.plug_sp[i]
        Modelica.Blocks.Sources.Constant const1(k=-500) annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={90,-20})));
        Modelica.Electrical.Machines.Utilities.TerminalBox terminalBox
          annotation (Placement(transformation(extent={{-10,60},{10,80}})));
        Modelica.Electrical.Analog.Basic.Ground ground
          annotation (Placement(transformation(extent={{-90,14},{-70,34}})));
        Modelica.Electrical.MultiPhase.Basic.Star star annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-80,58})));
        Modelica.Mechanics.Rotational.Sources.Torque torque annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={74,40})));
        Modelica.Electrical.MultiPhase.Sources.SignalVoltage signalVoltage
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={-70,80})));
        Modelica.Mechanics.Rotational.Sensors.SpeedSensor wMot annotation (
            Placement(transformation(
              extent={{-7,-7},{7,7}},
              rotation=270,
              origin={59,19})));
        Modelica.Blocks.Sources.Constant dWe(k=25)
          annotation (Placement(transformation(extent={{-100,-10},{-80,10}})));
        Modelica.Mechanics.Rotational.Components.Inertia inertia(J=40)
          annotation (Placement(transformation(extent={{22,30},{42,50}})));
        TestingModels.AsmaUgenerator uVar(
          UBase=400,
          pp=2,
          WeBase=314.15,
          WeMax=factor*314.15) annotation (Placement(transformation(
              extent={{-14,-10},{14,10}},
              rotation=90,
              origin={-50,46})));
        Modelica.Electrical.Machines.BasicMachines.AsynchronousInductionMachines.AIM_SquirrelCage
          aimc(
          p=2,
          fsNominal=50,
          Rs=0.016,
          Lssigma=0.07039/(100*pi),
          Lm=2.046/(100*pi),
          Lrsigma=0.07039/(100*pi),
          Rr=0.0313,
          Jr=0.341,
          TrOperational=293.15,
          TsOperational=293.15,
          TsRef=293.15,
          alpha20s=0)
          annotation (Placement(transformation(extent={{-10,30},{10,50}})));
        SupportModels.PTrifSensor pAC
          annotation (Placement(transformation(extent={{-42,70},{-22,90}})));
        Modelica.Electrical.Analog.Sensors.PowerSensor pDC annotation (
            Placement(visible=true, transformation(
              origin={-34,-34},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        AMDrive qSDrive(
          UBase=400,
          Unom=800,
          R1=0.016,
          L1=0.07039/(100*pi),
          Lm=2.046/(100*pi),
          R2=0.0313,
          L2=0.07039/(100*pi),
          J=0.341,
          pp=2,
          lossFact=0,
          WeMax=factor*314.15) annotation (Placement(visible=true,
              transformation(
              origin={-2,-44},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Electrical.Analog.Sources.ConstantVoltage Udc(V=sqrt(2)*400)
          annotation (Placement(visible=true, transformation(
              origin={-58,-44},
              extent={{-10,-10},{10,10}},
              rotation=270)));
        Modelica.Mechanics.Rotational.Components.Inertia inertia1(J=40)
          annotation (Placement(visible=true, transformation(
              origin={32,-44},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Electrical.Analog.Basic.Ground ground1 annotation (Placement(
              visible=true, transformation(
              origin={-58,-72},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Mechanics.Rotational.Sources.Torque torque1 annotation (
            Placement(visible=true, transformation(
              origin={58,-44},
              extent={{-10,-10},{10,10}},
              rotation=180)));
        Modelica.Blocks.Sources.Constant dWe1(k=23) annotation (Placement(
              visible=true, transformation(
              origin={24,-16},
              extent={{10,-10},{-10,10}},
              rotation=0)));
        Modelica.Blocks.Sources.Constant U1(k=15) annotation (Placement(visible
              =true, transformation(
              origin={-51,19},
              extent={{-8,-8},{8,8}},
              rotation=90)));
      equation
        connect(U1.y, uVar.U0) annotation (Line(points={{-51,27.8},{-51,34.7},{
                -50.1,34.7}}, color={0,0,127}));
        connect(torque1.tau, const1.y) annotation (Line(points={{70,-44},{74,-44},
                {74,-20},{79,-20},{79,-20}}, color={0,0,127}));
        connect(dWe1.y, qSDrive.dWe) annotation (Line(points={{13,-16},{-2,-16},
                {-2,-35.2}}, color={0,0,127}));
        connect(inertia1.flange_b, torque1.flange)
          annotation (Line(points={{42,-44},{48,-44}}));
        connect(ground1.p, Udc.n)
          annotation (Line(points={{-58,-62},{-58,-54}}, color={0,0,255}));
        connect(inertia1.flange_a, qSDrive.flange_a)
          annotation (Line(points={{22,-44},{8,-44}}));
        connect(pDC.nv, Udc.n) annotation (Line(points={{-34,-44},{-34,-62},{-58,
                -62},{-58,-54}}, color={0,0,255}));
        connect(pDC.pc, Udc.p)
          annotation (Line(points={{-44,-34},{-58,-34}}, color={0,0,255}));
        connect(qSDrive.pin_n, pDC.nv) annotation (Line(points={{-12.2,-50},{-18,
                -50},{-20,-50},{-20,-62},{-34,-62},{-34,-44}}, color={0,0,255}));
        connect(qSDrive.pin_p, pDC.nc) annotation (Line(points={{-12.2,-38},{-18,
                -38},{-18,-34},{-24,-34}}, color={0,0,255}));
        connect(pDC.pv, pDC.pc) annotation (Line(points={{-34,-24},{-44,-24},{-44,
                -34},{-44,-34}}, color={0,0,255}));
        connect(ground.p, star.pin_n) annotation (Line(
            points={{-80,34},{-80,48}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(signalVoltage.plug_n, star.plug_p) annotation (Line(
            points={{-80,80},{-80,68}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(wMot.flange, torque.flange) annotation (Line(
            points={{59,26},{59,33},{64,33},{64,40}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(torque.tau, const1.y) annotation (Line(
            points={{86,40},{92,40},{92,0},{72,0},{72,-20},{79,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(inertia.flange_b, torque.flange) annotation (Line(
            points={{42,40},{64,40}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(uVar.DWe, dWe.y) annotation (Line(
            points={{-38.7,40.1},{-36,40.1},{-36,0},{-79,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(uVar.U, signalVoltage.v) annotation (Line(
            points={{-54,57},{-54,68},{-70,68},{-70,73}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(uVar.Wm, wMot.w) annotation (Line(
            points={{-38.7,52.3},{-24,52.3},{-24,6},{59,6},{59,11.3}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(aimc.plug_sp, terminalBox.plug_sp) annotation (Line(
            points={{6,50},{6,60}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(aimc.plug_sn, terminalBox.plug_sn) annotation (Line(
            points={{-6,50},{-6,60}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(aimc.flange, inertia.flange_a) annotation (Line(
            points={{10,40},{22,40}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(signalVoltage.plug_p, pAC.pc) annotation (Line(
            points={{-60,80},{-42,80}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(pAC.nc, terminalBox.plugSupply) annotation (Line(
            points={{-22,80},{0,80},{0,62}},
            color={0,0,255},
            smooth=Smooth.None));
        annotation (
          experimentSetupOutput,
          Documentation(info="<html>
<p>This model  simulates variable-frequency start-up of an asyncronous motor in two different ways.</p>
<p>The above system uses the MSL aimc, and a variable voltage variable frequency source obtained by controlled generators.</p>
<p><br>The system below is built around the QSDrive, that in turn is built along a machine model that makes usage of the Quasi-Stationary MSL library.</p>
<p>In both cases the motor supply is constituted by a three-phase system of quasi-sinusoidal shapes, created according to the following equations:</p>
<p>Wel=Wmecc*PolePairs+DeltaWel</p>
<p>U=U0+(Ubase-U0)*(Wel)/WElbase </p>
<p>(but U is limited to the maximum voltage capability of the inverter, due to DC voltage. The U limit is set to Udc/2.4)</p>
<p>where:</p>
<ul>
<li>U0, Ubase U, are initial, base, actual voltage amplitudes</li>
<li>Wmecc, Wel, are machine, mechanical and supply, electrical angular speeds</li>
<li>PolePairs are the machine pole pairs</li>
<li>DeltaWel is intended as a scaled indication of the requested torque. </li>
</ul>
<p><br>It is suggested to compare in the same plot aimc.tauElectrical and qSDrive.qSAsma.tauElectrical, as well as pDC.power and pAC.power</p>
<p>In comparison with TestAMDrive, TestAMDrive2 simulates a longer transient, and imposes the drive to go beyond the base speed, up to &QUOT;factor&QUOT; times that value. &QUOT;factor&QUOT; is a parameter by default set to 1.4.</p>
<p>It is suggested, in a first plot, to compare aimc.tauElectrical, qSDrive.sQAsma.tauElectrical, and qsDrive.tauMaximum.</p>
<p>The first two quantities are nearly equal to each other (except for a limited transient in the beginning, not described by the simplified model); the third one indicates the maximum torque that the drive can deliver. Since the transient is charactierised by constant slip speed (difference between electromagnetic field rotational speed and mechanical speed), the maximum torque <span style=\"font-family: MS Shell Dlg 2;\">always </span>remains well above the delivered torque.</p>
<p>In another plot the user could plot qSDrive.uAgGen.luxPU, that indicates the ration of actual and maximum flux: the flux weakening action starting around t=4s is effectively seen.</p>
<p>Finally the user could plot in a diagram aimc.vs[1] and qsDrive.uAgGen.uAg: the second quantity is the rms value of the first one, as expected.</p>
<p>Around t=7 s the maximum speed reuired (with the default factor=1.4) is reached, and therefore the torques have a different behaviour.</p>
</html>"),
          experimentSetupOutput,
          Icon(coordinateSystem(
              extent={{-100,-80},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2})),
          Diagram(coordinateSystem(
              extent={{-100,-80},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics),
          experiment(StopTime=10, Interval=0.0015));
      end TestAMDrive2;

      model TestPMDrive
        //  extends Modelica.Icons.Example;
        ElectricDrives.PMDrive pMDrive(Ld=1.846e-3, Lq=1.846e-3)
          annotation (Placement(transformation(extent={{-12,28},{8,48}})));
        Modelica.Mechanics.Rotational.Components.Inertia inertia(J=0.29, phi(
              fixed=true, start=0))
          annotation (Placement(transformation(extent={{18,28},{38,48}})));
        Modelica.Blocks.Sources.Trapezoid torqueRef(
          period=1e6,
          amplitude=200,
          rising=2,
          width=2,
          falling=2)
          annotation (Placement(transformation(extent={{-60,28},{-40,48}})));
        Modelica.Electrical.Analog.Sources.ConstantVoltage fem(V=250)
          annotation (Placement(transformation(extent={{-10,62},{10,82}})));
        Modelica.Electrical.Analog.Basic.Ground ground
          annotation (Placement(transformation(extent={{24,52},{44,72}})));
        Modelica.Mechanics.Rotational.Sources.QuadraticSpeedDependentTorque
          myTorque(w_nominal(displayUnit="rpm") = 157.07963267949, tau_nominal=
              -150)
          annotation (Placement(transformation(extent={{76,28},{56,48}})));
        Modelica.Electrical.Machines.BasicMachines.SynchronousInductionMachines.SM_PermanentMagnet
          smpm(
          useDamperCage=false,
          VsOpenCircuit=115.5,
          Lmd=1.846e-3,
          Lmq=1.846e-3,
          Lssigma=1.91e-4) annotation (Placement(transformation(extent={{-12,-74},
                  {8,-54}}, rotation=0)));
        Modelica.Electrical.MultiPhase.Sources.SignalCurrent signalCurr(final m
            =3) annotation (Placement(transformation(
              origin={22,-16},
              extent={{-10,10},{10,-10}},
              rotation=180)));
        Modelica.Electrical.MultiPhase.Basic.Star star(final m=3) annotation (
            Placement(transformation(
              extent={{10,-10},{-10,10}},
              rotation=180,
              origin={50,-18})));
        Modelica.Electrical.Analog.Basic.Ground ground1 annotation (Placement(
              transformation(
              origin={76,-18},
              extent={{-10,-10},{10,10}},
              rotation=90)));
        Modelica.Electrical.Analog.Basic.Ground groundM annotation (Placement(
              transformation(
              origin={-26,-46},
              extent={{-10,-10},{10,10}},
              rotation=270)));
        Modelica.Electrical.Machines.Utilities.TerminalBox terminalBox(
            terminalConnection="Y") annotation (Placement(transformation(extent
                ={{-12,-48},{8,-28}}, rotation=0)));
        Modelica.Mechanics.Rotational.Sensors.AngleSensor angleS annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={18,-44})));
        Modelica.Mechanics.Rotational.Components.Inertia inertia1(J=0.29, phi(
              fixed=true, start=0))
          annotation (Placement(transformation(extent={{26,-74},{46,-54}})));
        Modelica.Mechanics.Rotational.Sources.QuadraticSpeedDependentTorque
          Tres(w_nominal(displayUnit="rpm") = 157.07963267949, tau_nominal=-150)
          annotation (Placement(transformation(extent={{76,-74},{56,-54}})));
        ElectricDrives.EDsupport.FromPark fromPark(p=smpm.p)
          annotation (Placement(transformation(extent={{-30,-20},{-10,0}})));
        Modelica.Blocks.Sources.Trapezoid IqRef(
          period=1e6,
          amplitude=115.0,
          rising=2,
          width=2,
          falling=2)
          annotation (Placement(transformation(extent={{-78,-56},{-58,-36}})));
        Modelica.Blocks.Sources.Trapezoid IdRef(
          period=1e6,
          amplitude=-95.0,
          rising=0.5,
          falling=0.5,
          offset=1,
          width=2.5,
          startTime=1.5)
          annotation (Placement(transformation(extent={{-78,-20},{-58,0}})));
      equation
        connect(pMDrive.flange_a, inertia.flange_a) annotation (Line(
            points={{8,38},{18,38}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(fem.p, pMDrive.pin_p) annotation (Line(
            points={{-10,72},{-20,72},{-20,48},{-8,48}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(pMDrive.pin_n, fem.n) annotation (Line(
            points={{4,48},{16,48},{16,72},{10,72}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(ground.p, fem.n) annotation (Line(
            points={{34,72},{10,72}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(pMDrive.tauRef, torqueRef.y) annotation (Line(
            points={{-12.6,38},{-39,38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(inertia.flange_b, myTorque.flange) annotation (Line(
            points={{38,38},{56,38}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(star.pin_n, ground1.p) annotation (Line(points={{60,-18},{60,-18},
                {66,-18}}, color={0,0,255}));
        connect(terminalBox.plug_sn, smpm.plug_sn) annotation (Line(
            points={{-8,-48},{-8,-54}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(terminalBox.plug_sp, smpm.plug_sp) annotation (Line(
            points={{4,-48},{4,-50},{6,-50},{6,-52},{4,-52},{4,-54}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(signalCurr.plug_p, star.plug_p) annotation (Line(
            points={{32,-16},{36,-16},{36,-18},{40,-18}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(signalCurr.plug_n, terminalBox.plugSupply) annotation (Line(
            points={{12,-16},{-2,-16},{-2,-46}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(groundM.p, terminalBox.starpoint) annotation (Line(
            points={{-16,-46},{-11,-46}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(Tres.flange, inertia1.flange_b) annotation (Line(
            points={{56,-64},{46,-64}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(angleS.flange, smpm.flange) annotation (Line(
            points={{18,-54},{18,-64},{8,-64}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(inertia1.flange_a, smpm.flange) annotation (Line(
            points={{26,-64},{8,-64}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(fromPark.y, signalCurr.i) annotation (Line(
            points={{-9,-10},{22,-10},{22,-9}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(angleS.phi, fromPark.phi) annotation (Line(
            points={{18,-33},{18,-32},{-20,-32},{-20,-28},{-20,-28},{-20,-22}},

            color={0,0,127},
            smooth=Smooth.None));

        connect(IqRef.y, fromPark.Xq) annotation (Line(
            points={{-57,-46},{-46,-46},{-46,-16},{-32,-16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fromPark.Xd, IdRef.y) annotation (Line(
            points={{-32,-4},{-44,-4},{-44,-10},{-57,-10}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {100,100}}), graphics),
          experiment(StopTime=8, __Dymola_NumberOfIntervals=5000),
          __Dymola_experimentSetupOutput,
          Documentation(info="<html>
<p>Ths model propose a comparisono of the library PMDrive and what can be seen using the MSL PMSM machine model.</p>
<p>To make this comparison it is proceeded as follows:</p>
<ol>
<li>PMDrive is run so that the behaviour of Pd and Iq over time is determined. this behaviour is the result of optimisations, that consider the need to have optimal angle between PM and stator field at lows speeds, and an alggle that produces the needed fluz weakening at high speeds, so that the machin terminal voltages are compatible with the available DC voltage</li>
<li>The shape of Id(t) and Id(t) is emulated with two trapezoids. The corresponding instantaneous currents are generated by making the inverse park transform and fed to the machine.</li>
<li>Some quantities of the MSL PMSM and PMDrive model are compared.</li>
</ol>
<p>Plots that show this can be the following one:</p>
<ul>
<li>plot in the same window IqRef.y and pmDrive.allFluxLim.atomicPmsm.Iq. It is seen that the trapezoid follows in a reasonable way the Iq value PMDrive model computes over time</li>
<li>plot in the same window IdRef.y and pmDrive.allFluxLim.atomicPmsm.Id. It is seen that the trapezoid follows in a reasonable way the Id value PMDrive model computes over time</li>
<li>plot in the same window smpm.v[1] and pmDrive.allFluxLim.atomicPmsm.Vpark and pmDrive.allFluxLim.atomicPmsm.VparkFF. It can be seen that the trend of Vpark is the same of all the instantaneous peaks of v[1], while in the central zones, in which the machine speed overcomes the base speed VparkFF is larger</li>
<li>plot in the same window smpm.tauElectrical and pmDrive.allFluxLim.tauElectrical. It can be seen that the two torques are nearly equal to each other. The differences can be justified considering that Id(t) and Id(t) of the two models are not perfectly equal to each other</li>
</ul>
</html>"));
      end TestPMDrive;

      block AsmaUgenerator
        import Modelica.Constants.pi;
        parameter Modelica.SIunits.Voltage UBase(start=400)
          "Base phase-to-phase RMS voltage";
        parameter Modelica.SIunits.AngularVelocity WeBase(start=314.15)
          "Base electric frequency";
        parameter Modelica.SIunits.AngularVelocity WeMax(start=314.15)
          "Maximum Electric frequency";
        parameter Integer pp(min=1, start=2) "number of pole pairs (Integer)";
        Modelica.Blocks.Interfaces.RealInput Wm annotation (Placement(
              transformation(extent={{-180,40},{-140,80}}), iconTransformation(
              extent={{-13,-13},{13,13}},
              rotation=90,
              origin={63,-113})));
        Modelica.Blocks.Interfaces.RealOutput U[3] annotation (Placement(
              transformation(extent={{140,50},{160,70}}), iconTransformation(
                extent={{100,30},{120,50}})));
        Modelica.Blocks.Math.Gain PolePairsG(k=pp)
          annotation (Placement(transformation(extent={{-130,50},{-110,70}})));
        Modelica.Blocks.Interfaces.RealInput DWe annotation (Placement(
              transformation(extent={{-180,-40},{-140,0}}), iconTransformation(
              extent={{-13,-13},{13,13}},
              rotation=90,
              origin={-59,-113})));
        Modelica.Blocks.Nonlinear.Limiter limiter1(uMin=0, uMax=UBase)
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={100,-28})));
        Modelica.Blocks.Math.Add add
          annotation (Placement(transformation(extent={{-100,40},{-80,60}})));
        Modelica.Blocks.Nonlinear.Limiter limiter(uMin=0, uMax=WeMax)
          annotation (Placement(transformation(extent={{-72,40},{-52,60}})));
        Modelica.Blocks.Math.Add add1[3]
          annotation (Placement(transformation(extent={{40,60},{60,80}})));
        Modelica.Blocks.Math.Sin sin[3]
          annotation (Placement(transformation(extent={{74,60},{94,80}})));
        Modelica.Blocks.Continuous.Integrator integrator
          annotation (Placement(transformation(extent={{-32,60},{-12,80}})));
        Modelica.Blocks.Routing.Replicator replicator(nout=3) annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={100,30})));
        Modelica.Blocks.Math.Product product[3]
          annotation (Placement(transformation(extent={{112,50},{132,70}})));
        Modelica.Blocks.Math.Gain ToPeak(k=sqrt(2/3)) annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={100,0})));
        Modelica.Blocks.Sources.Constant Fase[3](k=2*pi/3*{0,-1,1}) annotation
          (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={30,30})));
        Modelica.Blocks.Routing.Replicator replicator1(nout=3) annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={10,70})));
        Modelica.Blocks.Interfaces.RealOutput Wel annotation (Placement(
              transformation(extent={{140,-70},{160,-50}}), iconTransformation(
                extent={{100,-50},{120,-30}})));
        Modelica.Blocks.Sources.RealExpression amplitude(y=U0 + (UBase - U0)*
              Wel/WeBase)
          annotation (Placement(transformation(extent={{32,-56},{86,-34}})));
        Modelica.Blocks.Interfaces.RealInput U0 annotation (Placement(
              transformation(extent={{-180,-100},{-140,-60}}),
              iconTransformation(extent={{-126,-12},{-100,14}})));
      equation
        connect(add.u1, PolePairsG.y) annotation (Line(
            points={{-102,56},{-102,60},{-109,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(limiter.u, add.y) annotation (Line(
            points={{-74,50},{-79,50}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(sin.u, add1.y) annotation (Line(
            points={{72,70},{61,70}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(integrator.u, limiter.y) annotation (Line(
            points={{-34,70},{-34,50},{-51,50}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.y, U) annotation (Line(
            points={{133,60},{150,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.u2, replicator.y) annotation (Line(
            points={{110,54},{100,54},{100,41}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ToPeak.y, replicator.u) annotation (Line(
            points={{100,11},{100,18}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(sin.y, product.u1) annotation (Line(
            points={{95,70},{102,70},{102,66},{110,66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add1.u1, replicator1.y) annotation (Line(
            points={{38,76},{30,76},{30,70},{21,70}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add1.u2, Fase.y) annotation (Line(
            points={{38,64},{30,64},{30,41}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Wel, limiter.y) annotation (Line(
            points={{150,-60},{-42,-60},{-42,50},{-51,50}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PolePairsG.u, Wm) annotation (Line(
            points={{-132,60},{-160,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(replicator1.u, integrator.y) annotation (Line(
            points={{-2,70},{-11,70}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(DWe, add.u2) annotation (Line(
            points={{-160,-20},{-102,-20},{-102,44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ToPeak.u, limiter1.y) annotation (Line(
            points={{100,-12},{100,-17}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(amplitude.y, limiter1.u) annotation (Line(
            points={{88.7,-45},{90,-44},{96,-44},{96,-40},{100,-40}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-140,-100},
                  {140,100}}), graphics={Text(
                      extent={{-11,3},{11,-3}},
                      lineColor={0,0,255},
                      textString="Omega",
                      origin={-47,1},
                      rotation=90)}),
          Icon(coordinateSystem(preserveAspectRatio=false, extent={{-140,-100},
                  {140,100}}), graphics={Rectangle(
                      extent={{-100,100},{100,-100}},
                      lineColor={0,0,127},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Line(
                      points={{-40,-50},{-40,76}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-48,66},{-40,78},{-34,66}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-54,-42},{90,-42}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-7,-6},{1,6},{7,-6}},
                      color={0,0,127},
                      smooth=Smooth.None,
                      origin={87,-42},
                      rotation=270),Line(
                      points={{-46,-28},{24,40},{76,40}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-47,-23},{-31,-23}},
                      color={0,0,127},
                      smooth=Smooth.None),Text(
                      extent={{-82,-6},{-51,-34}},
                      lineColor={0,0,127},
                      textString="Uo"),Text(
                      extent={{-84,54},{-48,26}},
                      lineColor={0,0,127},
                      textString="Un"),Line(
                      points={{-47,39},{-31,39}},
                      color={0,0,127},
                      smooth=Smooth.None),Text(
                      extent={{9,-47},{52,-78}},
                      lineColor={0,0,127},
                      textString="Wn"),Line(
                      points={{26,-14},{26,-46},{26,-36}},
                      color={0,0,127},
                      smooth=Smooth.None),Text(
                      extent={{-106,144},{92,106}},
                      lineColor={0,0,255},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid,
                      textString="%name"),Line(
                      points={{20,8},{30,22},{48,22},{62,-6},{80,-6},{90,8}},
                      color={255,0,0},
                      pattern=LinePattern.Dash,
                      smooth=Smooth.None)}),
          Documentation(info="<html>
<p>This class produces a three-phase voltage system to variable-frequency control of an asynchronous motor.</p>
<p>The output voltages constitute a three-phase system of quasi-sinusoidal shapes, created according to the following equations:</p>
<p>Wel=Wmecc*PolePairs+DeltaWel</p>
<p>U=min(Ubase, U0+(Ubase-U0)*(Wel)/Wnom ) </p>
<p>where:</p>
<ul>
<li>U0, Ubase U, are initial, base actual voltage amplitudes</li>
<li>Wmecc, Wel are machine (mechanical) and supply (electrical) angular speeds</li>
<li>PolePairs are the number of machine pole pairs</li>
<li>DeltaWel is an input variable and depends on the desired torque</li>
</ul>
</html>"));
      end AsmaUgenerator;
    end TestingModels;

    package EDsupport "Useful Additional Models"
      model QSAsma
        "Asynchronus machine model based on Quasi-stationary library"
        import PI = Modelica.Constants.pi;
        Modelica.Electrical.QuasiStationary.SinglePhase.Basic.Inductor L2_(L=L2)
          annotation (Placement(transformation(extent={{44,8},{64,28}})));
        Modelica.Electrical.QuasiStationary.SinglePhase.Basic.Inductor Lm_(L=Lm)
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={36,-4})));
        Modelica.Electrical.QuasiStationary.SinglePhase.Sources.VariableVoltageSource
          uFeed annotation (Placement(transformation(
              extent={{-10,10},{10,-10}},
              rotation=270,
              origin={-32,-2})));
        Modelica.ComplexBlocks.ComplexMath.PolarToComplex ToComplexUin
          annotation (Placement(transformation(
              origin={-68,48},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Blocks.Interfaces.RealInput Uag annotation (Placement(
              transformation(extent={{-140,40},{-100,80}}), iconTransformation(
                extent={{-140,40},{-100,80}})));
        Modelica.Blocks.Sources.Constant const(k=0) annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={-68,20})));
        Modelica.Blocks.Interfaces.RealInput f annotation (Placement(
              transformation(extent={{-140,-80},{-100,-40}}),
              iconTransformation(extent={{-140,-80},{-100,-40}})));
        Modelica.Mechanics.Rotational.Interfaces.Flange_a flange_a annotation (
            Placement(transformation(extent={{90,70},{110,90}}),
              iconTransformation(extent={{90,-10},{110,10}})));
        Modelica.Mechanics.Rotational.Sources.Torque torque
          annotation (Placement(transformation(extent={{-10,70},{10,90}})));
        Modelica.Mechanics.Rotational.Sensors.SpeedSensor Wm annotation (
            Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={48,58})));
        Modelica.Electrical.QuasiStationary.SinglePhase.Sensors.PowerSensor Pag
          annotation (Placement(transformation(extent={{72,8},{92,28}})));
        Modelica.Blocks.Sources.RealExpression WmS1(y=tauGen)
          annotation (Placement(transformation(extent={{-60,70},{-28,90}})));
        parameter Integer pp=2 "Pole pairs";
        parameter Real R1=0.435 "Stator's phase resistance (ohm)";
        parameter Real L1=0.004 "Stator's leakage inductance (H)";
        parameter Real Lm=0.0693 "Stator's leakage inductance (H)";
        parameter Real R2=0.4 "Rotor's phase resistance (ohm)";
        parameter Real L2=0.002 "Rotor's leakage inductance (H)";
        parameter Real J=2.0 "Rotor's moment of inertia (kg.m^2)";
        Real W0, tauGen;
        Modelica.SIunits.Torque tauElectrical;
        Modelica.SIunits.AngularVelocity wMechanical;
        //mechanical angular speed
        Real s;
        //slip
        Real absUag;
        //abs(Uag)
        Real constUF;
        //abs(Uag)/f
        Modelica.Mechanics.Rotational.Components.Inertia inertia(J=J)
          annotation (Placement(transformation(extent={{58,70},{78,90}})));
        Modelica.Electrical.QuasiStationary.SinglePhase.Basic.Ground ground
          annotation (Placement(transformation(extent={{26,-40},{46,-20}})));
        Modelica.Blocks.Interfaces.RealOutput Pdc annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-60,-110}), iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-60,-110})));
        Modelica.Blocks.Interfaces.RealOutput Is annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={60,-110}), iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={60,-110})));
        Modelica.Mechanics.Rotational.Sensors.PowerSensor PmGen
          annotation (Placement(transformation(extent={{22,70},{42,90}})));
        Modelica.ComplexBlocks.ComplexMath.ComplexToPolar toIs annotation (
            Placement(transformation(
              extent={{-7,-7},{7,7}},
              rotation=270,
              origin={-7,-3})));
        Modelica.Electrical.QuasiStationary.SinglePhase.Basic.VariableConductor
          gAirgap annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=-90,
              origin={106,-4})));
        Modelica.Electrical.QuasiStationary.SinglePhase.Sensors.CurrentSensor
          Is1 annotation (Placement(transformation(
              extent={{9,8},{-9,-8}},
              rotation=180,
              origin={-7,18})));
        Modelica.Blocks.Sources.RealExpression toPdc(y=3*(Pag.y.re + R1*toIs.len
              ^2))
          annotation (Placement(transformation(extent={{-10,-74},{-58,-58}})));
        Modelica.Electrical.QuasiStationary.SinglePhase.Basic.Inductor L1_(L=L1)
          annotation (Placement(transformation(extent={{10,8},{30,28}})));
      equation
        W0 = f*2*PI/pp;
        absUag = sqrt(L2_.pin_p.v.re^2 + L2_.pin_p.v.im^2);
        if abs(f) < 1e-12 then
          s = 1.0;
          tauGen = 0;
          constUF = 1;
        else
          s = (W0 - Wm.w)/W0;
          tauGen = 3*Pag.y.re/W0;
          constUF = absUag/f;
        end if;
        gAirgap.G_ref = s/R2;
        tauElectrical = torque.tau;
        wMechanical = Wm.w;
        connect(ToComplexUin.y, uFeed.V) annotation (Line(
            points={{-57,48},{-44,48},{-44,2},{-42,2}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(ToComplexUin.len, Uag) annotation (Line(
            points={{-80,54},{-92,54},{-92,60},{-120,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Pag.currentP, L2_.pin_n) annotation (Line(
            points={{72,18},{64,18}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(Pag.voltageP, Pag.currentP) annotation (Line(
            points={{82,28},{72,28},{72,18}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(WmS1.y, torque.tau) annotation (Line(
            points={{-26.4,80},{-12,80}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(inertia.flange_b, flange_a) annotation (Line(
            points={{78,80},{100,80}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(ground.pin, Lm_.pin_n) annotation (Line(
            points={{36,-20},{36,-14}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(PmGen.flange_a, torque.flange) annotation (Line(
            points={{22,80},{10,80}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(PmGen.flange_b, inertia.flange_a) annotation (Line(
            points={{42,80},{58,80}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(Wm.flange, PmGen.flange_b) annotation (Line(
            points={{48,68},{48,76},{42,76},{42,80}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(uFeed.f, f) annotation (Line(
            points={{-42,-6},{-80,-6},{-80,-60},{-120,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Pag.currentN, gAirgap.pin_p) annotation (Line(
            points={{92,18},{106,18},{106,6}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(Lm_.pin_p, L2_.pin_p) annotation (Line(
            points={{36,6},{36,18},{44,18}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(Is1.y, toIs.u) annotation (Line(
            points={{-7,9.2},{-7,5.4}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(toPdc.y, Pdc) annotation (Line(
            points={{-60.4,-66},{-60,-66},{-60,-110}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(uFeed.pin_p, Is1.pin_p) annotation (Line(
            points={{-32,8},{-32,18},{-16,18}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(toIs.len, Is) annotation (Line(
            points={{-2.8,-11.4},{-2.8,-84},{60,-84},{60,-110}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gAirgap.pin_n, Lm_.pin_n) annotation (Line(
            points={{106,-14},{36,-14}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(Lm_.pin_n, uFeed.pin_n) annotation (Line(
            points={{36,-14},{-32,-14},{-32,-12}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(ToComplexUin.phi, const.y) annotation (Line(
            points={{-80,42},{-92,42},{-92,20},{-79,20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Pag.voltageN, Lm_.pin_n) annotation (Line(
            points={{82,8},{82,-14},{36,-14}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(Is1.pin_n, L1_.pin_p) annotation (Line(
            points={{2,18},{10,18}},
            color={85,170,255},
            smooth=Smooth.None));
        connect(L2_.pin_p, L1_.pin_n) annotation (Line(
            points={{44,18},{30,18}},
            color={85,170,255},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {120,100}}), graphics={Rectangle(
                      extent={{-48,34},{120,-38}},
                      lineColor={255,0,0},
                      pattern=LinePattern.Dash)}),
          Documentation(info="<html>
<p>This model models ans asynchronous machine based on a quasi-stationary approximation: the equivalent single-phase circuit.</p>
<p>This model is very fast and compact, and gives result with sufficient precision in most vehicular propulsion needs.</p>
</html>"),
          Icon(coordinateSystem(
              extent={{-100,-100},{120,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics={Line(points={{-102,60},{-50,32}}, color={0,
                0,127}),Line(points={{-100,-60},{-48,-30}}, color={0,0,127}),
                Text( origin={14,-16},
                      lineColor={0,0,255},
                      fillColor={95,95,95},
                      fillPattern=FillPattern.Solid,
                      extent={{-114,140},{102,100}},
                      textString="%name"),Rectangle(
                      fillColor={175,175,175},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{-56,66},{78,-54}}),Rectangle(
                      fillColor={95,95,95},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{78,10},{98,-10}}),Rectangle(
                      fillColor={128,128,128},
                      fillPattern=FillPattern.HorizontalCylinder,
                      extent={{-56,66},{-76,-54}}),Polygon(fillPattern=
                FillPattern.Solid, points={{-72,-84},{-56,-84},{-26,-14},{26,-14},
                {58,-84},{72,-84},{72,-94},{-72,-94},{-72,-84}})}));
      end QSAsma;

      model FromPark "Semplice PMM con modello funzionale inverter"
        parameter Integer p "Number or pole pairs";
        Modelica.Electrical.Machines.SpacePhasors.Blocks.FromSpacePhasor
          fromSpacePhasor
          annotation (Placement(transformation(extent={{60,0},{80,20}})));
        Modelica.Electrical.Machines.SpacePhasors.Blocks.Rotator rotator
          annotation (Placement(transformation(extent={{0,6},{20,26}})));
        Modelica.Blocks.Routing.Multiplex2 multiplex2_1
          annotation (Placement(transformation(extent={{-40,0},{-20,20}})));
        Modelica.Blocks.Interfaces.RealOutput y[3] annotation (Placement(
              transformation(extent={{100,-10},{120,10}}), iconTransformation(
                extent={{100,-10},{120,10}})));
        Modelica.Blocks.Interfaces.RealInput Xd annotation (Placement(
              transformation(extent={{-140,40},{-100,80}}), iconTransformation(
                extent={{-140,40},{-100,80}})));
        Modelica.Blocks.Interfaces.RealInput Xq annotation (Placement(
              transformation(extent={{-140,-80},{-100,-40}}),
              iconTransformation(extent={{-140,-80},{-100,-40}})));
        Modelica.Blocks.Interfaces.RealInput phi annotation (Placement(
              transformation(
              extent={{-20,-20},{20,20}},
              rotation=90,
              origin={10,-110}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=90,
              origin={0,-120})));
        Modelica.Blocks.Math.Gain gain(k=-p) annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={10,-50})));
        Modelica.Blocks.Sources.Constant const annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={50,-30})));
      equation
        connect(multiplex2_1.y, rotator.u) annotation (Line(
            points={{-19,10},{-10,10},{-10,16},{-2,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fromSpacePhasor.u, rotator.y) annotation (Line(
            points={{58,10},{40,10},{40,16},{21,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fromSpacePhasor.y, y) annotation (Line(
            points={{81,10},{94,10},{94,0},{110,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(multiplex2_1.u1[1], Xd) annotation (Line(
            points={{-42,16},{-60,16},{-60,60},{-120,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(multiplex2_1.u2[1], Xq) annotation (Line(
            points={{-42,4},{-60,4},{-60,-60},{-120,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(rotator.angle, gain.y) annotation (Line(
            points={{10,4},{10,-39}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gain.u, phi) annotation (Line(
            points={{10,-62},{10,-110}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fromSpacePhasor.zero, const.y) annotation (Line(
            points={{58,2},{50,2},{50,-19}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}), graphics),
          experiment(StopTime=5, Interval=0.001),
          Documentation(info="<html>
 <p><br/><b>Test example: Permanent magnet synchronous induction machine fed by a current source</b></p>


 <p><i><span style='color:red'>NOTA: la macchina ha Lmd=Lmq=0.3(2*pi*f) come definito internamente.</p>
 <i><span style='color:red'>E&apos; pertanto una macchina isotropa. la miglior maniera di controllarla, quindi, dovrebbe essere di mettere la corrente tutta sull&apos;asse q e mantenere a 0 la componente sull&apos;asse d.</p></i>


 <p><br/><br/>A synchronous induction machine with permanent magnets accelerates a quadratic speed dependent load from standstill. The rms values of d- and q-current in rotor fixed coordinate system are converted to threephase currents, and fed to the machine. The result shows that the torque is influenced by the q-current, whereas the stator voltage is influenced by the d-current.</p><p><br/><br/>Default machine parameters of model <i>SM_PermanentMagnet</i> are used. </p>
 </html>"),
          __Dymola_experimentSetupOutput,
          Icon(coordinateSystem(
              extent={{-100,-100},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics={Rectangle(
                      lineColor={0,0,255},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid,
                      extent={{-100,100},{100,-100}}),Text(
                      lineColor={0,0,255},
                      extent={{-96,28},{96,-26}},
                      textString="P=>"),Text(
                      lineColor={0,0,255},
                      extent={{-108,150},{102,110}},
                      textString="%name")}));
      end FromPark;

      block UagGenerator
        "Generates the voltage behind stator resistance for QAsma model"
        import PI = Modelica.Constants.pi;
        Real fluxPU "Factor indicating the ratio flux/nominalFlux";
        Modelica.Blocks.Interfaces.RealInput dWe "slip radian frequency"
          annotation (Placement(transformation(extent={{-136,40},{-96,80}}),
              iconTransformation(extent={{-136,40},{-96,80}})));
        Modelica.Blocks.Interfaces.RealOutput f "frequency" annotation (
            Placement(transformation(extent={{100,-70},{120,-50}}),
              iconTransformation(extent={{100,-70},{120,-50}})));
        Modelica.Blocks.Math.Add add annotation (Placement(visible=true,
              transformation(
              origin={-78,-6},
              extent={{-10,-10},{10,10}},
              rotation=0)));
        Modelica.Blocks.Interfaces.RealInput We
          "rotating field radian frequency" annotation (Placement(
              transformation(extent={{-136,-80},{-96,-40}}), iconTransformation(
                extent={{-136,-80},{-96,-40}})));
        Modelica.Blocks.Nonlinear.Limiter limWe(uMax=WeMax) annotation (
            Placement(visible=true, transformation(extent={{-30,-16},{-10,4}},
                rotation=0)));
        Modelica.Blocks.Math.Gain toFreq(k=1/(2*PI)) annotation (Placement(
              visible=true, transformation(
              origin={6,-6},
              extent={{8,-8},{-8,8}},
              rotation=180)));
        Modelica.Blocks.Math.Gain freqToVoltage(k=UBase1/(WeBase/(2*PI)))
          annotation (Placement(visible=true, transformation(extent={{28,-14},{
                  44,2}}, rotation=0)));
        Modelica.Blocks.Interfaces.RealOutput uAg
          "air-gap voltage (single-phase, rms)" annotation (Placement(
              transformation(extent={{100,50},{120,70}}), iconTransformation(
                extent={{100,50},{120,70}})));
        Modelica.Blocks.Interfaces.RealInput dcVoltage "DC Voltage" annotation
          (Placement(transformation(
              extent={{20,-20},{-20,20}},
              rotation=90,
              origin={0,116})));
        Modelica.Blocks.Nonlinear.VariableLimiter variableLimiter
          annotation (Placement(transformation(extent={{58,-16},{78,4}})));
        Modelica.Blocks.Math.Gain toUagMax(k=1/2.445)
          "Limits the maximum generated voltage due to DC voltage limitations"
          annotation (Placement(visible=true, transformation(
              origin={0,62},
              extent={{-10,-10},{10,10}},
              rotation=270)));
        parameter Modelica.SIunits.Voltage UBase=230
          "Base RMS machine line voltage";
        parameter Modelica.SIunits.AngularFrequency WeBase=314.15
          "Base machine angular frequency ";
        parameter Modelica.SIunits.AngularFrequency WeMax=314.15
          "Base machine angular frequency";
      protected
        parameter Real UBase1=UBase/sqrt(3);
      public
        Modelica.Blocks.Math.Gain toLim2(k=-1)
          "Limits the maximum generated voltage due to DC voltage limitations"
          annotation (Placement(visible=true, transformation(
              origin={42,-46},
              extent={{-10,-10},{10,10}},
              rotation=0)));
      equation
        if variableLimiter.limit1 > variableLimiter.u then
          fluxPU = 1;
        else
          fluxPU = variableLimiter.limit1/variableLimiter.u;
        end if;
        connect(limWe.y, toFreq.u) annotation (Line(
            points={{-9,-6},{-3.6,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toFreq.y, freqToVoltage.u) annotation (Line(
            points={{14.8,-6},{26.4,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(f, freqToVoltage.u) annotation (Line(
            points={{110,-60},{20,-60},{20,-6},{26.4,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(freqToVoltage.y, variableLimiter.u) annotation (Line(
            points={{44.8,-6},{56,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(dcVoltage, toUagMax.u) annotation (Line(
            points={{0,116},{0,74}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toUagMax.y, variableLimiter.limit1) annotation (Line(
            points={{0,51},{0,20},{42,20},{42,2},{56,2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(variableLimiter.y, uAg) annotation (Line(
            points={{79,-6},{90,-6},{90,60},{110,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add.u2, We) annotation (Line(
            points={{-90,-12},{-96,-12},{-96,-60},{-116,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(dWe, add.u1) annotation (Line(
            points={{-116,60},{-84,60},{-84,18},{-96,18},{-96,0},{-90,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(limWe.u, add.y) annotation (Line(
            points={{-32,-6},{-67,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toLim2.y, variableLimiter.limit2) annotation (Line(
            points={{53,-46},{64,-46},{64,-26},{56,-26},{56,-14}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toLim2.u, variableLimiter.limit1) annotation (Line(
            points={{30,-46},{-56,-46},{-56,32},{0,32},{0,20},{42,20},{42,2},{
                56,2}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {100,100}})),
          Icon(coordinateSystem(extent={{-100,-100},{100,100}},
                preserveAspectRatio=false), graphics={Rectangle(
                      extent={{-100,100},{100,-100}},
                      lineColor={0,0,127},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Line(
                      points={{-46,-58},{-46,68}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-54,58},{-46,70},{-40,58}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-60,-50},{84,-50}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-7,-6},{1,6},{7,-6}},
                      color={0,0,127},
                      smooth=Smooth.None,
                      origin={81,-50},
                      rotation=270),Line(
                      points={{-52,-55},{18,13},{70,13}},
                      color={0,0,127},
                      smooth=Smooth.None),Line(
                      points={{-53,-50},{-37,-50}},
                      color={0,0,127},
                      smooth=Smooth.None),Text(
                      extent={{-90,46},{-54,18}},
                      lineColor={0,0,127},
                      textString="Ub"),Line(
                      points={{-53,31},{-37,31}},
                      color={0,0,127},
                      smooth=Smooth.None),Text(
                      extent={{3,-55},{46,-86}},
                      lineColor={0,0,127},
                      textString="Wb"),Line(
                      points={{20,-22},{20,-54},{20,-44}},
                      color={0,0,127},
                      smooth=Smooth.None),Text(
                      extent={{-100,-100},{90,-140}},
                      lineColor={0,0,255},
                      textString="%name")}),
          Documentation(info="<html>
<p>Generates&nbsp;the voltage behind stator resistance for&nbsp;QAsma&nbsp;model.</p>
<p>It uses the U/f=const technique, so it applies a voltage proportional to frequency, wusing the input deltaOmeca, in such a way that when spees is base speed, the applied voltage is the nominal voltage.</p>
<p>The maximum machine voltage is limited by the DC available voltage. The maximum line-to-line peak machine voltage is imposed to be equal to the available DC voltage.</p>
</html>"));
      end UagGenerator;
      annotation (Icon(graphics={Rectangle(
                  lineColor={200,200,200},
                  fillColor={248,248,248},
                  fillPattern=FillPattern.HorizontalCylinder,
                  extent={{-100,-100},{100,100}},
                  radius=25.0),Rectangle(
                  lineColor={128,128,128},
                  fillPattern=FillPattern.None,
                  extent={{-100,-100},{100,100}},
                  radius=25.0),Ellipse(extent={{-38,40},{38,-36}}, lineColor={0,
              0,0}),Line(
                  points={{2,82},{-8,82},{-12,72},{-26,68},{-36,78},{-48,70},{-44,
                58},{-56,46},{-68,50},{-76,36},{-68,30},{-70,16},{-80,12},{-80,
                2}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{2,-78},{-8,-78},{-12,-68},{-26,-64},{-36,-74},{-48,-66},
                {-44,-54},{-56,-42},{-68,-46},{-76,-32},{-68,-26},{-70,-12},{-80,
                -8},{-80,2}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{0,-78},{10,-78},{14,-68},{28,-64},{38,-74},{50,-66},
                {46,-54},{58,-42},{70,-46},{78,-32},{70,-26},{72,-12},{82,-8},{
                82,2}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{0,82},{10,82},{14,72},{28,68},{38,78},{50,70},{46,58},
                {58,46},{70,50},{78,36},{70,30},{72,16},{82,12},{82,2}},
                  color={0,0,0},
                  smooth=Smooth.None)}));
    end EDsupport;
    annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
              -100},{100,100}}), graphics={
          Line(points={{-30,20},{4,20}}, color={0,0,255}),
          Line(points={{-32,0},{2,0}}, color={0,0,255}),
          Line(points={{-32,-20},{2,-20}}, color={0,0,255}),
          Text(
            lineColor={0,0,255},
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid,
            extent={{-142,-112},{146,-146}},
            textString="%name"),
          Line(points={{-104,-60},{-80,-60},{-80,-28},{-62,-28}}, color={0,0,
                255}),
          Line(points={{-98,60},{-80,60},{-80,28},{-62,28}}, color={0,0,255}),
          Rectangle(
            fillColor={175,175,175},
            fillPattern=FillPattern.HorizontalCylinder,
            extent={{-42,68},{78,-52}}),
          Rectangle(
            lineColor={0,0,255},
            fillColor={0,0,255},
            fillPattern=FillPattern.HorizontalCylinder,
            extent={{-42,68},{-64,-52}}),
          Polygon(fillPattern=FillPattern.Solid, points={{-56,-82},{-46,-82},{-16,
                -12},{34,-12},{64,-82},{74,-82},{74,-92},{-56,-92},{-56,-82}}),

          Rectangle(
            fillColor={95,95,95},
            fillPattern=FillPattern.HorizontalCylinder,
            extent={{78,12},{98,-8}})}));

  end ElectricDrives;

  package SupportModels "Useful addtional models"
    extends Modelica.Icons.Package;
    // extends EHPT.Icons.SupportIcon;

    model PropDriver "Simple Proportional controller driver"
      parameter String CycleFileName="MyCycleName.txt"
        "Drive Cycle Name ex: \"sort1.txt\"";
      parameter Real k "Controller gain";
      parameter Real yMax=1000000.0 "Max output value (absolute)";
      Modelica.Blocks.Interfaces.RealInput V annotation (Placement(
            transformation(
            extent={{-14,-14},{14,14}},
            rotation=90,
            origin={0,-114}), iconTransformation(
            extent={{-12,-12},{12,12}},
            rotation=90,
            origin={0,-112})));
      Modelica.Blocks.Interfaces.RealOutput tauRef(unit="N.m") annotation (
          Placement(transformation(extent={{100,-10},{120,10}}),
            iconTransformation(extent={{100,-10},{120,10}})));
      Modelica.Blocks.Sources.CombiTimeTable driveCyc(
        tableOnFile=true,
        tableName="Cycle",
        extrapolation=Modelica.Blocks.Types.Extrapolation.Periodic,
        fileName=CycleFileName,
        columns={2})
        annotation (Placement(transformation(extent={{-86,-10},{-66,10}})));
      Modelica.Blocks.Math.UnitConversions.From_kmh from_kmh
        annotation (Placement(transformation(extent={{-48,-10},{-28,10}})));
      Modelica.Blocks.Math.Feedback feedback
        annotation (Placement(transformation(extent={{-10,-10},{10,10}})));
      Modelica.Blocks.Math.Gain gain(k=k)
        annotation (Placement(transformation(extent={{32,-10},{52,10}})));
      Modelica.Blocks.Nonlinear.Limiter limiter(uMax=yMax)
        annotation (Placement(transformation(extent={{70,-10},{90,10}})));
    equation
      connect(from_kmh.u, driveCyc.y[1]) annotation (Line(
          points={{-50,0},{-65,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(from_kmh.y, feedback.u1) annotation (Line(
          points={{-27,0},{-8,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(feedback.u2, V) annotation (Line(
          points={{0,-8},{0,-114},{1.77636e-015,-114}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(feedback.y, gain.u) annotation (Line(
          points={{9,0},{30,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(gain.y, limiter.u) annotation (Line(
          points={{53,0},{68,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(tauRef, limiter.y) annotation (Line(
          points={{110,0},{91,0}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                {100,100}}), graphics),
        Documentation(info="<html>
            <p>Modello semplice di pilota.</p>
            <p>Esso contiene al suo interno il ciclo di riferimento, che insegue attraverso un regolatore solo proporzionale.</p>
            </html>"),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Rectangle(
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-100,100},{100,-100}}),Ellipse(
                  fillColor={255,213,170},
                  fillPattern=FillPattern.Solid,
                  extent={{-23,46},{-12,20}},
                  endAngle=360),Text(
                  origin={0,1.81063},
                  lineColor={0,0,255},
                  extent={{-104,142.189},{98,104}},
                  textString="%name"),Polygon(
                  fillColor={215,215,215},
                  pattern=LinePattern.None,
                  fillPattern=FillPattern.Solid,
                  points={{-22,-36},{-42,-64},{-16,-64},{16,-64},{-22,-36}}),
              Polygon(
                  fillColor={135,135,135},
                  pattern=LinePattern.None,
                  fillPattern=FillPattern.Solid,
                  points={{-32,64},{-62,-28},{-30,-28},{-30,-28},{-32,64}},
                  smooth=Smooth.Bezier),Polygon(
                  fillColor={135,135,135},
                  pattern=LinePattern.None,
                  fillPattern=FillPattern.Solid,
                  points={{-68,-12},{-14,-66},{10,-26},{0,-26},{-68,-12}},
                  smooth=Smooth.Bezier),Polygon(
                  fillColor={175,175,175},
                  fillPattern=FillPattern.Solid,
                  points={{-22,34},{-30,30},{-40,-24},{2,-22},{2,-10},{0,26},{-22,
                34}},
                  smooth=Smooth.Bezier),Ellipse(
                  fillColor={255,213,170},
                  fillPattern=FillPattern.Solid,
                  extent={{-30,68},{-3,34}},
                  endAngle=360),Polygon(
                  pattern=LinePattern.None,
                  fillPattern=FillPattern.Solid,
                  points={{-38,58},{-16,74},{-2,60},{4,60},{6,60},{-38,58}},
                  smooth=Smooth.Bezier),Polygon(
                  fillColor={95,95,95},
                  fillPattern=FillPattern.Solid,
                  points={{30,-20},{-32,-4},{-36,-20},{-24,-34},{30,-20}},
                  smooth=Smooth.Bezier),Polygon(
                  fillPattern=FillPattern.Solid,
                  points={{42,-46},{36,-60},{48,-54},{52,-48},{50,-44},{42,-46}},
                  smooth=Smooth.Bezier),Line(points={{48,10},{26,24},{26,24}},
              thickness=0.5),Line(points={{20,14},{34,34},{34,34}}, thickness=
              0.5),Polygon(
                  fillColor={255,213,170},
                  fillPattern=FillPattern.Solid,
                  points={{28,28},{32,32},{28,26},{34,30},{30,26},{34,28},{30,
                24},{26,26},{34,24},{26,24},{26,26},{28,28},{28,28},{26,26},{26,
                26},{26,26},{28,32},{28,30},{28,28}},
                  smooth=Smooth.Bezier),Polygon(
                  fillColor={175,175,175},
                  fillPattern=FillPattern.Solid,
                  points={{-18,24},{28,30},{26,22},{-16,8},{-20,8},{-24,18},{-18,
                24}},
                  smooth=Smooth.Bezier),Polygon(
                  fillColor={215,215,215},
                  fillPattern=FillPattern.Solid,
                  points={{72,18},{48,18},{36,-2},{58,-62},{72,-62},{72,18}}),
              Polygon(
                  fillColor={95,95,95},
                  fillPattern=FillPattern.Solid,
                  points={{49,-70},{17,-16},{7,-20},{-1,-26},{49,-70}},
                  smooth=Smooth.Bezier),Line(points={{-7,55},{-3,53}}),Line(
              points={{-9,42},{-5,42}}),Line(points={{-7,55},{-3,55}})}));
    end PropDriver;

    model DragForce "Vehicle rolling and aerodinamical drag force"
      import Modelica.Constants.g_n;
      extends
        Modelica.Mechanics.Translational.Interfaces.PartialElementaryOneFlangeAndSupport2;
      extends Modelica.Mechanics.Translational.Interfaces.PartialFriction;
      Modelica.SIunits.Force f "Total drag force";
      Modelica.SIunits.Velocity v "Vehicle velocity";
      Modelica.SIunits.Acceleration a "Absolute acceleration of flange";
      Real Sign;
      parameter Modelica.SIunits.Mass m "Vehicle mass";
      parameter Modelica.SIunits.Density rho(start=1.226) "air density";
      parameter Modelica.SIunits.Area S "Vehicle cross area";
      parameter Real fc(start=0.01) "Rolling friction coefficient";
      parameter Real Cx "Aerodinamic drag coefficient";
    protected
      parameter Real A=fc*m*g_n;
      parameter Real B=1/2*rho*S*Cx;
      // Constant auxiliary variable
    equation
      //  s = flange.s;
      v = der(s);
      a = der(v);
      // Le seguenti definizioni seguono l'ordine e le ridchieste del modello "PartialFriction" di
      // Modelica.Mechanics.Translational.Interfaces"
      v_relfric = v;
      a_relfric = a;
      f0 = A "forza a velocitC  0 ma con scorrimento";
      f0_max = A "massima forza  velocitC  0 e senza scorrimento ";
      free = false "sarebbe true quando la ruota si stacca dalla strada";
      // Ora il calcolo di f, e la sua attribuzione alla flangia:
      flange.f - f = 0;
      // friction force
      if v > 0 then
        Sign = 1;
      else
        Sign = -1;
      end if;
      f - B*v^2*Sign = if locked then sa*unitForce else f0*(if startForward
         then Modelica.Math.tempInterpol1(
            v,
            [0, 1],
            2) else if startBackward then -Modelica.Math.tempInterpol1(
            -v,
            [0, 1],
            2) else if pre(mode) == Forward then Modelica.Math.tempInterpol1(
            v,
            [0, 1],
            2) else -Modelica.Math.tempInterpol1(
            -v,
            [0, 1],
            2));
      annotation (
        Documentation(info="<html>
            <p>This component modesl the total (rolling &egrave;+ aerrodynamic vehicle drag resistance: </p>
            <p>f=mgh+(1/2)*rho*Cx*S*v^2</p>
            <p>It models reliably the stuck phase. based on Modelica-Intrerfaces.PartialFriction model</p>
            </html>"),
        Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                100,100}}), graphics={Polygon(
                  points={{-98,10},{22,10},{22,41},{92,0},{22,-41},{22,-10},{-98,
                -10},{-98,10}},
                  lineColor={0,127,0},
                  fillColor={215,215,215},
                  fillPattern=FillPattern.Solid),Line(points={{-42,-50},{87,-50}},
              color={0,0,0}),Polygon(
                  points={{-72,-50},{-41,-40},{-41,-60},{-72,-50}},
                  lineColor={0,0,0},
                  fillColor={128,128,128},
                  fillPattern=FillPattern.Solid),Line(
                  points={{-90,-90},{-70,-88},{-50,-82},{-30,-72},{-10,-58},{10,
                -40},{30,-18},{50,8},{70,38},{90,72},{110,110}},
                  color={0,0,255},
                  thickness=0.5),Text(
                  extent={{-82,90},{80,50}},
                  lineColor={0,0,255},
                  textString="%name")}),
        Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                {100,100}}), graphics));
    end DragForce;

    expandable connector Conn
      "Control bus that is adapted to the signals connected to it"
      extends Modelica.Icons.SignalBus;
      annotation (Diagram(graphics));
    end Conn;

    model Batt1 "Battery model based on one R-C block in its electric circuit"
      parameter Modelica.SIunits.ElectricCharge QCellNom(min=0) = 10*3600.0
        "Nominal admissible electric charge per cell"
        annotation (Dialog(tab="cell data"));
      parameter Modelica.SIunits.Voltage ECellMin(min=0) = 3.3
        "Minimum open source voltage per cell"
        annotation (Dialog(tab="cell data"));
      parameter Modelica.SIunits.Voltage ECellMax(min=0.0001) = 4.15
        "Maximum open source voltage per cell"
        annotation (Dialog(tab="cell data"));
      parameter Real SOCMin(
        min=0,
        max=1) = 0 "Minimum state of charge"
        annotation (Dialog(group="SOC parameters"));
      parameter Real SOCMax(
        min=0,
        max=1) = 1 "Maximum state of charge"
        annotation (Dialog(group="SOC parameters"));
      parameter Real SOCInit(
        min=0,
        max=1) = 0.5 "Initial state of charge"
        annotation (Dialog(group="SOC parameters"));
      parameter Modelica.SIunits.Current ICellMax(min=0) = 10*QCellNom/3600.0
        "Maximum admissible cell current" annotation (Dialog(tab="cell data"));
      parameter Modelica.SIunits.Resistance R0Cell(min=0) = 0.05*ECellMax/
        ICellMax "Serial cell resistance \"R0\"" annotation (Dialog(tab=
              "cell data", group="Electric circuit parameters"));
      parameter Modelica.SIunits.Resistance R1Cell(min=0) = R0Cell
        "Serial cell resistance \"R1\"" annotation (Dialog(tab="cell data",
            group="Electric circuit parameters"));
      parameter Modelica.SIunits.Capacitance C1Cell(min=0) = 60/R1Cell
        "Capacitance in parallel with R1" annotation (Dialog(tab="cell data",
            group="Electric circuit parameters"));
      parameter Real efficiency(
        min=0,
        max=0.9999) = 0.85 "Overall charging/discharging energy efficiency"
        annotation (Dialog(group="Parameters related to losses"));
      parameter Modelica.SIunits.Current ICellChargeDischarge(min=0) = 0.5*
        ICellMax
        "Charging/discharging current of a cell that the efficiency refers to"
        annotation (Dialog(group="Parameters related to losses"));
      parameter Integer ns=1 "Number of serial connected cells"
        annotation (Dialog(tab="package data", group="Size of the package"));
      parameter Integer np=1 "Number of parallel connected cells"
        annotation (Dialog(tab="package data", group="Size of the package"));
    protected
      parameter Real efficiencyMax=(EBatteryMin + EBatteryMax - 2*Rtot*
          ICellChargeDischarge)/(EBatteryMin + EBatteryMax + 2*Rtot*
          ICellChargeDischarge);
      parameter Modelica.SIunits.Capacitance C=QCellNom/(ECellMax - ECellMin)
        "Cell capacitance";
      // determine fraction of drain current with respect to the total package current
      parameter Real k=((1 - efficiency)*(EBatteryMax + EBatteryMin) - 2*(1 +
          efficiency)*Rtot*ICellChargeDischarge)/((1 + efficiency)*(EBatteryMax
           + EBatteryMin) - 2*(1 - efficiency)*Rtot*ICellChargeDischarge);
      parameter Modelica.SIunits.Current IBatteryMax=ICellMax*np
        "Maximum battery current";
      parameter Modelica.SIunits.Voltage EBatteryMin=ECellMin*ns
        "Minimum battery voltage";
      parameter Modelica.SIunits.Voltage EBatteryMax=ECellMax*ns
        "Maximum battery voltage";
      parameter Modelica.SIunits.ElectricCharge QBatteryNominal=QCellNom*np
        "Battery admissible electric charge";
      parameter Modelica.SIunits.Capacitance CBattery=C*np/ns
        "Battery capacitance";
      parameter Modelica.SIunits.Resistance R0Battery=R0Cell*ns/np
        "Serial inner resistance R0 of cell package";
      parameter Modelica.SIunits.Resistance R1Battery=R1Cell*ns/np
        "Serial inner resistance R1 of cell package";
      parameter Modelica.SIunits.Resistance Rtot=R0Battery + R1Battery;
      parameter Modelica.SIunits.Capacitance C1Battery=C1Cell*np/ns
        "Battery series inner capacitance C1";
    protected
      Modelica.SIunits.Voltage ECell "Cell e.m.f.";
      Modelica.SIunits.Current iCellStray "Cell stray current";
      Modelica.SIunits.Voltage EBattery(start=EBatteryMin + SOCInit*(
            EBatteryMax - EBatteryMin), fixed=true) "Battery e.m.f.";
      Modelica.SIunits.Voltage Ubat(start=EBatteryMin + SOCInit*(EBatteryMax -
            EBatteryMin), fixed=true);
      Modelica.SIunits.Current iBatteryStray "Cell parasitic current";
      Modelica.Electrical.Analog.Sensors.CurrentSensor currentSensor
        annotation (Placement(transformation(extent={{60,50},{80,70}}, rotation
              =0)));
      Modelica.Blocks.Math.Gain gain(k=k) annotation (Placement(transformation(
            origin={52,0},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Blocks.Math.Abs absolute annotation (Placement(transformation(
              extent={{34,-10},{14,10}}, rotation=0)));
    public
      Modelica.Electrical.Analog.Basic.Capacitor cBattery(final C=CBattery)
        annotation (Placement(transformation(
            origin={-60,0},
            extent={{-10,-10},{10,10}},
            rotation=270)));
      Modelica.Electrical.Analog.Basic.Resistor R0(final R=R0Battery)
        annotation (Placement(transformation(
            origin={20,60},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Electrical.Analog.Sources.SignalCurrent strayCurrent annotation
        (Placement(transformation(
            origin={-6,0},
            extent={{-10,-10},{10,10}},
            rotation=270)));
      Modelica.Electrical.Analog.Interfaces.Pin p annotation (Placement(
            transformation(extent={{90,50},{110,70}}), iconTransformation(
              extent={{90,50},{110,70}})));
      Modelica.Electrical.Analog.Interfaces.NegativePin n annotation (Placement(
            transformation(extent={{90,-70},{110,-50}}), iconTransformation(
              extent={{91,-70},{111,-50}})));
      Modelica.Electrical.Analog.Basic.Resistor R1(final R=R1Battery)
        annotation (Placement(transformation(
            origin={-37,74},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Electrical.Analog.Basic.Capacitor C1(C=C1Battery) annotation (
          Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-37,50})));
      Modelica.Blocks.Interfaces.RealOutput SOC annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-110,0}), iconTransformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-110,0})));
      Modelica.Electrical.Analog.Basic.Capacitor Cdummy(C=C1Battery/10000)
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=270,
            origin={87,12})));
    equation
      assert(SOCMin >= 0, "SOCMin must be greater than, or equal to 0");
      assert(SOCMax <= 1, "SOCMax must be smaller than, or equal to 1");
      assert(efficiency <= efficiencyMax,
        "Overall charging/discharging energy efficiency is too big with respect to the actual serial resistance (EfficiencyMax ="
         + String(efficiencyMax) + ")");
      assert(SOCMin < SOCMax, "SOCMax(=" + String(SOCMax) +
        ") must be greater than SOCMin(=" + String(SOCMin) + ")");
      assert(SOCInit >= SOCMin, "SOCInit(=" + String(SOCInit) +
        ") must be greater than, or equal to SOCMin(=" + String(SOCMin) + ")");
      assert(SOCInit <= SOCMax, "SOCInit(=" + String(SOCInit) +
        ") must be smaller than, or equal to SOCMax(=" + String(SOCMax) + ")");
      iBatteryStray = strayCurrent.i;
      iCellStray = iBatteryStray/np;
      EBattery = cBattery.v;
      //Solo per dare maggiore chiarezza all'utente con un nome significativo
      Ubat = Cdummy.v;
      ECell = EBattery/ns;
      assert(abs(p.i/np) < ICellMax, "Battery cell current i=" + String(abs(p.i
        /np)) + "\n exceeds max admissable ICellMax (=" + String(ICellMax) +
        "A)");
      SOC = (EBattery - EBatteryMin)/(EBatteryMax - EBatteryMin);
      //*(SOCMax-SOCMin)+SOCMin);
      assert(SOC <= SOCMax,
        "Battery is fully charged: State of charge reached maximum limit (=" +
        String(SOCMax) + ")");
      assert(SOCMin <= SOC,
        "Battery is fully discharged: State of charge reached minimum limit (="
         + String(SOCMin) + ")");
      connect(R0.p, currentSensor.p)
        annotation (Line(points={{30,60},{60,60}}, color={0,0,255}));
      connect(strayCurrent.p, R0.n)
        annotation (Line(points={{-6,10},{-6,60},{10,60}}, color={0,0,255}));
      connect(currentSensor.i, gain.u) annotation (Line(points={{70,50},{70,-1.46958e-015},
              {64,-1.46958e-015}}, color={0,0,127}));
      connect(absolute.u, gain.y) annotation (Line(points={{36,0},{39.5,0},{
              39.5,1.34711e-015},{41,1.34711e-015}}, color={0,0,127}));
      connect(absolute.y, strayCurrent.i) annotation (Line(points={{13,0},{7,0},
              {7,-1.28588e-015},{1,-1.28588e-015}}, color={0,0,127}));
      connect(currentSensor.n, p)
        annotation (Line(points={{80,60},{80,60},{100,60}}, color={0,0,255}));
      connect(strayCurrent.n, n) annotation (Line(points={{-6,-10},{-6,-60},{
              100,-60}}, color={0,0,255}));
      connect(n, cBattery.n) annotation (Line(points={{100,-60},{-60,-60},{-60,
              -10}}, color={0,0,255}));
      connect(R1.n, cBattery.p) annotation (Line(
          points={{-47,74},{-60,74},{-60,10}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(C1.n, cBattery.p) annotation (Line(
          points={{-47,50},{-60,50},{-60,10}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(R1.p, C1.p) annotation (Line(
          points={{-27,74},{-18,74},{-18,50},{-27,50}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(R1.p, R0.n) annotation (Line(
          points={{-27,74},{-18,74},{-18,60},{10,60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(Cdummy.p, currentSensor.n) annotation (Line(
          points={{87,22},{88,22},{88,60},{80,60}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(Cdummy.n, n) annotation (Line(
          points={{87,2},{88,2},{88,-60},{100,-60}},
          color={0,0,255},
          smooth=Smooth.None));
      annotation (
        Documentation(info="<html>
<p>Battery model wit non-unity coulombic efficiency. </p>
<p>The main cell branch contains an e.m.f. that is linearly increasing with SOC, simulated through an equivalent capacitor the resistance R0 and a parallel R-C couple. </p>
<p>The full battery is composed by np rows in parallel, each of them containing ns cells in series</p>
</html>", revisions="<html><table border=\"1\" rules=\"groups\">
    <thead>
    <tr><td>Version</td>  <td>Date</td>  <td>Comment</td></tr>
    </thead>
    <tbody>
    <tr><td>1.0.0</td>  <td>2006-01-12</td>  <td> </td></tr>
    <tr><td>1.0.3</td>  <td>2006-08-31</td>  <td> Improved assert statements </td></tr>
    <tr><td>1.0.6</td>  <td>2007-05-14</td>  <td> The documentation changed slightly </td></tr>
    </tbody>
    </table>
    </html>"),
        Diagram(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={2,2}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=true,
            initialScale=0.1,
            grid={2,2}), graphics={Rectangle(
                  lineColor={95,95,95},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-100,100},{78,-98}}),Line(points={{-92,6},{-52,6}},
              color={0,0,255}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-82,-3},{-65,-10}}),Line(points={{-73,63},{98,64}},
              color={0,0,255}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{38,69},{68,57}}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-37.5,68},{-6.5,56}}),Line(points={{-19.5,49},{-19.5,
              32}}, color={0,0,255}),Line(points={{-54.5,63},{-54.5,41},{-25.5,
              41}}, color={0,0,255}),Line(points={{9.5,62},{9.5,40},{-19.5,40}},
              color={0,0,255}),Line(points={{-73,63},{-73,5}}, color={0,0,255}),
              Line(points={{-73,-6},{-73,-60},{96,-60}}, color={0,0,255}),Line(
              points={{26,63},{26,-61}}, color={0,0,255}),Line(points={{-25.5,
              49},{-25.5,32}}, color={0,0,255}),Polygon(
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  points={{26,22},{14,4},{26,-14},{38,4},{26,22}}),Line(points=
              {{20,4},{32,4}}, color={0,0,255}),Polygon(lineColor={0,0,255},
              points={{22,-20},{30,-20},{26,-32},{22,-20}}),Text(
                  lineColor={0,0,255},
                  extent={{-100,150},{100,110}},
                  textString="%name")}));
    end Batt1;

    model Batt1Conn
      "Modello di batteria basato su Batt0 con ordine elettrico pari a 1"
      Real powDeliv "battery power (positive when delivered)";
      Real SOC "State Of Charge";
      parameter Modelica.SIunits.ElectricCharge QCellNom(min=0) = 10*3.6e3
        "Nominal admissible electric charge per cell"
        annotation (Dialog(tab="cell data"));
      parameter Modelica.SIunits.Voltage ECellMin(min=0) = 3.3
        "Minimum open source voltage per cell"
        annotation (Dialog(tab="cell data"));
      parameter Modelica.SIunits.Voltage ECellMax(min=0.0001) = 4.15
        "Maximum open source voltage per cell"
        annotation (Dialog(tab="cell data"));
      parameter Real SOCMin(
        min=0,
        max=1) = 0 "Minimum state of charge"
        annotation (Dialog(group="SOC parameters"));
      parameter Real SOCMax(
        min=0,
        max=1) = 1 "Maximum state of charge"
        annotation (Dialog(group="SOC parameters"));
      parameter Real SOCInit(
        min=0,
        max=1) = 0.5 "Initial state of charge"
        annotation (Dialog(group="SOC parameters"));
      parameter Modelica.SIunits.Current ICellMax(min=0) = 10*QCellNom/3.6e3
        "Maximum admissible current" annotation (Dialog(tab="cell data"));
      parameter Modelica.SIunits.Resistance R0Cell(min=0) = 0.05*ECellMax/
        ICellMax "Series resistance \"R0\"" annotation (Dialog(tab="cell data",
            group="Electric circuit parameters"));
      parameter Modelica.SIunits.Resistance R1Cell(min=0) = R0Cell
        "Series resistance \"R1\"" annotation (Dialog(tab="cell data", group=
              "Electric circuit parameters"));
      parameter Modelica.SIunits.Capacitance C1Cell(min=0) = 60/R1Cell
        "Capacitance in parallel with R1" annotation (Dialog(tab="cell data",
            group="Electric circuit parameters"));
      parameter Real efficiency(
        min=0,
        max=0.9999) = 0.85 "Overall charging/discharging energy efficiency"
        annotation (Dialog(group="Parameters related to losses"));
      parameter Modelica.SIunits.Current iCellEfficiency(min=0) = 0.5*ICellMax
        "Cell charging/discharging current the efficiency refers to"
        annotation (Dialog(group="Parameters related to losses"));
      parameter Integer ns=1 "Number of serial connected cells"
        annotation (Dialog(tab="package data", group="Size of the package"));
      parameter Integer np=1 "Number of parallel connected cells"
        annotation (Dialog(tab="package data", group="Size of the package"));
      // determine fraction of drain current with respect to the total package current
      Modelica.Electrical.Analog.Basic.Capacitor cBattery(final C=CBattery)
        annotation (Placement(transformation(
            origin={-60,0},
            extent={{-10,-10},{10,10}},
            rotation=270)));
      Modelica.Electrical.Analog.Basic.Resistor R0(final R=R0Battery)
        annotation (Placement(transformation(
            origin={20,60},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Electrical.Analog.Sources.SignalCurrent strayCurrent annotation
        (Placement(transformation(
            origin={-6,0},
            extent={{-10,-10},{10,10}},
            rotation=270)));
      Modelica.Electrical.Analog.Interfaces.Pin p annotation (Placement(
            transformation(extent={{90,50},{110,70}}), iconTransformation(
              extent={{90,52},{110,72}})));
      Modelica.Electrical.Analog.Interfaces.NegativePin n annotation (Placement(
            transformation(extent={{90,-70},{110,-50}}), iconTransformation(
              extent={{91,-70},{111,-50}})));
      Modelica.Electrical.Analog.Basic.Capacitor C1(C=C1Battery) annotation (
          Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-37,50})));
      SupportModels.Conn conn annotation (Placement(transformation(
            extent={{-20,-20},{20,20}},
            rotation=90,
            origin={-100,-2})));
      Modelica.Blocks.Sources.RealExpression SOCs(y=SOC) annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={-80,30})));
      Modelica.Blocks.Sources.RealExpression outPow(y=(p.v - n.v)*n.i)
        annotation (Placement(transformation(
            extent={{10,-10},{-10,10}},
            rotation=-90,
            origin={-80,-26})));
      Modelica.Electrical.Analog.Basic.Resistor R1(final R=R1Battery)
        annotation (Placement(visible=true, transformation(
            origin={-37,72},
            extent={{-10,-10},{10,10}},
            rotation=180)));
    protected
      parameter Real k=((1 - efficiency)*(eBattMax + eBattMin) - 2*(1 +
          efficiency)*Rtot*iCellEfficiency)/((1 + efficiency)*(eBattMax +
          eBattMin) - 2*(1 - efficiency)*Rtot*iCellEfficiency);
      parameter Real efficiencyMax=(eBattMin + eBattMax - 2*Rtot*
          iCellEfficiency)/(eBattMin + eBattMax + 2*Rtot*iCellEfficiency);
      final parameter Modelica.SIunits.Capacitance C=QCellNom/(ECellMax -
          ECellMin) "Cell capacitance";
      parameter Modelica.SIunits.Current IBatteryMax=ICellMax*np
        "Maximum battery current";
      parameter Modelica.SIunits.Voltage eBattMin=ECellMin*ns
        "Minimum battery voltage";
      parameter Modelica.SIunits.Voltage eBattMax=ECellMax*ns
        "Maximum battery voltage";
      parameter Modelica.SIunits.ElectricCharge QBatteryNominal=QCellNom*np
        "Battery admissible electric charge";
      parameter Modelica.SIunits.Capacitance CBattery=C*np/ns
        "Battery capacitance";
      parameter Modelica.SIunits.Resistance R0Battery=R0Cell*ns/np
        "Series inner resistance R0 of cell package";
      parameter Modelica.SIunits.Resistance R1Battery=R1Cell*ns/np
        "Series inner resistance R1 of cell package";
      parameter Modelica.SIunits.Resistance Rtot=R0Battery + R1Battery;
      parameter Modelica.SIunits.Capacitance C1Battery=C1Cell*np/ns
        "Battery series inner capacitance C1";
      Modelica.SIunits.Voltage ECell "Cell e.m.f.";
      Modelica.SIunits.Current iCellStray "Cell stray current";
      Modelica.SIunits.Voltage eBatt(start=eBattMin + SOCInit*(eBattMax -
            eBattMin), fixed=true) "Battery e.m.f.";
      Modelica.SIunits.Current iBatteryStray "Cell parasitic current";
      Modelica.Electrical.Analog.Sensors.CurrentSensor currentSensor
        annotation (Placement(transformation(extent={{70,50},{90,70}}, rotation
              =0)));
      Modelica.Blocks.Math.Gain gain(k=k) annotation (Placement(transformation(
            origin={60,0},
            extent={{-10,-10},{10,10}},
            rotation=180)));
      Modelica.Blocks.Math.Abs absolute annotation (Placement(transformation(
              extent={{34,-10},{14,10}}, rotation=0)));
    equation
      connect(R1.p, R0.n) annotation (Line(points={{-27,72},{-18,72},{-18,60},{
              10,60}}, color={0,0,255}));
      connect(R1.p, C1.p) annotation (Line(points={{-27,72},{-18,72},{-18,50},{
              -27,50}}, color={0,0,255}));
      connect(R1.n, cBattery.p) annotation (Line(points={{-47,72},{-60,72},{-60,
              10}}, color={0,0,255}));
      assert(SOCMin >= 0, "SOCMin must be greater than, or equal to 0");
      assert(SOCMax <= 1, "SOCMax must be smaller than, or equal to 1");
      assert(efficiency <= efficiencyMax,
        "Overall charging/discharging energy efficiency is too big with respect to the actual serial resistance (EfficiencyMax ="
         + String(efficiencyMax) + ")");
      assert(SOCMin < SOCMax, "SOCMax(=" + String(SOCMax) +
        ") must be greater than SOCMin(=" + String(SOCMin) + ")");
      assert(SOCInit >= SOCMin, "SOCInit(=" + String(SOCInit) +
        ") must be greater than, or equal to SOCMin(=" + String(SOCMin) + ")");
      assert(SOCInit <= SOCMax, "SOCInit(=" + String(SOCInit) +
        ") must be smaller than, or equal to SOCMax(=" + String(SOCMax) + ")");
      iBatteryStray = strayCurrent.i;
      iCellStray = iBatteryStray/np;
      eBatt = cBattery.v;
      //Solo per dare maggiore chiarezza all'utente con un nome significativo
      ECell = eBatt/ns;
      powDeliv = (p.v - n.v)*n.i;
      assert(abs(p.i/np) < ICellMax, "Battery cell current i=" + String(abs(p.i
        /np)) + "\n exceeds max admissable ICellMax (=" + String(ICellMax) +
        "A)");
      SOC = (eBatt - eBattMin)/(eBattMax - eBattMin);
      //*(SOCMax-SOCMin)+SOCMin);
      assert(SOC <= SOCMax,
        "Battery is fully charged: State of charge reached maximum limit (=" +
        String(SOCMax) + ")");
      assert(SOCMin <= SOC,
        "Battery is fully discharged: State of charge reached minimum limit (="
         + String(SOCMin) + ")");
      connect(R0.p, currentSensor.p)
        annotation (Line(points={{30,60},{70,60}}, color={0,0,255}));
      connect(strayCurrent.p, R0.n)
        annotation (Line(points={{-6,10},{-6,60},{10,60}}, color={0,0,255}));
      connect(currentSensor.i, gain.u) annotation (Line(points={{80,50},{80,-1.46958e-015},
              {72,-1.46958e-015}}, color={0,0,127}));
      connect(absolute.u, gain.y) annotation (Line(points={{36,0},{39.5,0},{
              39.5,1.33227e-015},{49,1.33227e-015}}, color={0,0,127}));
      connect(absolute.y, strayCurrent.i) annotation (Line(points={{13,0},{7,0},
              {7,-1.28588e-015},{1,-1.28588e-015}}, color={0,0,127}));
      connect(currentSensor.n, p)
        annotation (Line(points={{90,60},{90,60},{100,60}}, color={0,0,255}));
      connect(strayCurrent.n, n) annotation (Line(points={{-6,-10},{-6,-60},{
              100,-60}}, color={0,0,255}));
      connect(n, cBattery.n) annotation (Line(points={{100,-60},{-60,-60},{-60,
              -10}}, color={0,0,255}));
      connect(C1.n, cBattery.p) annotation (Line(
          points={{-47,50},{-60,50},{-60,10}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(conn.batSOC, SOCs.y) annotation (Line(
          points={{-100,-2},{-100,8.5},{-80,8.5},{-80,19}},
          color={255,204,51},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(outPow.y, conn.batPowDel) annotation (Line(
          points={{-80,-15},{-80,-2},{-100,-2}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (
        Documentation(info="<html>
    <p>Modello di batteria  dotato di efficienza colombica non unitaria, secondo quanto discusso nelle dispense SEB.</p>
    <p>Il ramo principaledel modello di cella  presenta una f.e.m. linearmente crescente con il SOC (simulata tramite un condensatore equivalente), una resistenza R0 ed un blocco R-C.</p>
    <p>La batteria &egrave; composta da np filari idi celle in parallelo, ciascuno composto da ns celle in serie.</p>
    <p><br/>SEB a.a. 2012-2013.</p>
    </html>", revisions="<html><table border=\"1\" rules=\"groups\">
    <thead>
    <tr><td>Version</td>  <td>Date</td>  <td>Comment</td></tr>
    </thead>
    <tbody>
    <tr><td>1.0.0</td>  <td>2006-01-12</td>  <td> </td></tr>
    <tr><td>1.0.3</td>  <td>2006-08-31</td>  <td> Improved assert statements </td></tr>
    <tr><td>1.0.6</td>  <td>2007-05-14</td>  <td> The documentation changed slightly </td></tr>
    </tbody>
    </table>
    </html>"),
        Diagram(coordinateSystem(
            extent={{-100,-80},{100,80}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-80},{100,80}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics={Rectangle(
                  lineColor={95,95,95},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-100,80},{80,-82}}),Line(points={{-92,6},{-52,6}},
              color={0,0,255}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-82,-3},{-65,-10}}),Line(points={{-73,63},{98,64}},
              color={0,0,255}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{38,69},{68,57}}),Rectangle(
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-37.5,68},{-6.5,56}}),Line(points={{-19.5,49},{-19.5,
              32}}, color={0,0,255}),Line(points={{-54.5,63},{-54.5,41},{-25.5,
              41}}, color={0,0,255}),Line(points={{9.5,62},{9.5,40},{-19.5,40}},
              color={0,0,255}),Line(points={{-73,63},{-73,5}}, color={0,0,255}),
              Line(points={{-73,-6},{-73,-60},{96,-60}}, color={0,0,255}),Line(
              points={{26,63},{26,-61}}, color={0,0,255}),Line(points={{-25.5,
              49},{-25.5,32}}, color={0,0,255}),Polygon(
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  points={{26,22},{14,4},{26,-14},{38,4},{26,22}}),Line(points=
              {{20,4},{32,4}}, color={0,0,255}),Polygon(lineColor={0,0,255},
              points={{22,-20},{30,-20},{26,-32},{22,-20}}),Text(
                  origin={-4,-22},
                  lineColor={0,0,255},
                  extent={{-100,150},{100,110}},
                  textString="%name")}));
    end Batt1Conn;

    package Internal
      "Models intended to be used by other models of this package, not by the final user"
      model ConstPDC "Constant Power DC Load"
        parameter Real k "inner PI follower proportional gain";
        parameter Modelica.SIunits.Time T
          "inner PI follower integral time constant";
        Real v "DC voltage";
        Modelica.Blocks.Math.Feedback feedback1 annotation (Placement(visible=
                true, transformation(
              origin={56,-44},
              extent={{-10,-10},{10,10}},
              rotation=180)));
        Modelica.Blocks.Continuous.PI PI(
          k=k,
          T=T,
          initType=Modelica.Blocks.Types.Init.InitialState) annotation (
            Placement(visible=true, transformation(
              origin={20,-44},
              extent={{-8,-8},{8,8}},
              rotation=180)));
        Modelica.Electrical.Analog.Interfaces.PositivePin pin_p annotation (
            Placement(
            visible=true,
            transformation(extent={{-108,58},{-88,78}}, rotation=0),
            iconTransformation(extent={{-10,90},{10,110}}, rotation=0)));
        Modelica.Electrical.Analog.Interfaces.NegativePin pin_n annotation (
            Placement(
            visible=true,
            transformation(extent={{-108,-74},{-88,-54}}, rotation=0),
            iconTransformation(extent={{-10,-108},{10,-88}}, rotation=0)));
        Modelica.Blocks.Interfaces.RealInput Pref "Reference power" annotation
          (Placement(
            visible=true,
            transformation(
              origin={100,-44},
              extent={{-18,-18},{18,18}},
              rotation=180),
            iconTransformation(
              origin={82,0},
              extent={{-18,-18},{18,18}},
              rotation=180)));
        Modelica.Electrical.Analog.Sensors.PowerSensor pSensor annotation (
            Placement(visible=true, transformation(extent={{-78,58},{-58,78}},
                rotation=0)));
        Modelica.Electrical.Analog.Sensors.VoltageSensor uSensor annotation (
            Placement(visible=true, transformation(
              origin={-43,15},
              extent={{-9,-9},{9,9}},
              rotation=270)));
        Modelica.Electrical.Analog.Sources.SignalCurrent iDrawn annotation (
            Placement(visible=true, transformation(
              origin={-12,18},
              extent={{-10,-10},{10,10}},
              rotation=270)));
        Modelica.Blocks.Math.Product product annotation (Placement(visible=true,
              transformation(
              origin={18,18},
              extent={{-8,-8},{8,8}},
              rotation=180)));
        Inverse inverse(k=1) annotation (Placement(visible=true, transformation(
              origin={10,50},
              extent={{-10,-10},{10,10}},
              rotation=0)));
      equation
        connect(PI.y, product.u1) annotation (Line(points={{11.2,-44},{2,-44},{
                2,-20},{40,-20},{40,13.2},{27.6,13.2}}, color={0,0,127}));
        connect(inverse.y, product.u2) annotation (Line(points={{20.6,50},{40,
                50},{40,22.8},{27.6,22.8}}, color={0,0,127}));
        connect(uSensor.v, inverse.u) annotation (Line(points={{-52,15},{-54,15},
                {-54,50},{-1,50}}, color={0,0,127}));
        connect(product.y, iDrawn.i)
          annotation (Line(points={{9.2,18},{-5,18}}, color={0,0,127}));
        connect(uSensor.n, iDrawn.n) annotation (Line(points={{-43,6},{-43,-28},
                {-12,-28},{-12,8}}, color={0,0,255}));
        connect(pSensor.nc, iDrawn.p) annotation (Line(points={{-58,68},{-12,68},
                {-12,28}}, color={0,0,255}));
        connect(iDrawn.n, pin_n) annotation (Line(points={{-12,8},{-12,-64},{-98,
                -64}}, color={0,0,255}));
        connect(uSensor.p, pSensor.nc) annotation (Line(points={{-43,24},{-43,
                68},{-58,68}}, color={0,0,255}));
        connect(feedback1.u2, pSensor.power) annotation (Line(points={{56,-36},
                {56,-8},{-76,-8},{-76,57}}, color={0,0,127}));
        connect(pSensor.nv, pin_n) annotation (Line(points={{-68,58},{-68,-64},
                {-98,-64}}, color={0,0,255}));
        connect(feedback1.u1, Pref)
          annotation (Line(points={{64,-44},{100,-44}}, color={0,0,127}));
        connect(iDrawn.n, pin_n) annotation (Line(points={{-12,8},{-12,-64},{-98,
                -64}}, color={0,0,255}));
        connect(pSensor.nv, pin_n) annotation (Line(points={{-68,58},{-68,-64},
                {-98,-64}}, color={0,0,255}));
        connect(feedback1.u2, pSensor.power) annotation (Line(points={{56,-36},
                {56,-8},{-76,-8},{-76,57}}, color={0,0,127}));
        connect(uSensor.n, iDrawn.n) annotation (Line(points={{-43,6},{-43,-28},
                {-12,-28},{-12,8}}, color={0,0,255}));
        connect(uSensor.v, inverse.u) annotation (Line(points={{-52,15},{-54,15},
                {-54,50},{-1,50}}, color={0,0,127}));
        connect(feedback1.y, PI.u)
          annotation (Line(points={{47,-44},{29.6,-44}}, color={0,0,127}));
        v = pin_p.v - pin_n.v;
        connect(pSensor.pc, pin_p) annotation (Line(
            points={{-78,68},{-98,68}},
            color={0,0,255},
            smooth=Smooth.None));
        connect(pSensor.pv, pSensor.pc) annotation (Line(
            points={{-68,78},{-78,78},{-78,68}},
            color={0,0,255},
            smooth=Smooth.None));
        annotation (
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {100,100}}), graphics),
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Line(
                      points={{-4,0},{70,0}},
                      color={0,0,0},
                      smooth=Smooth.None),Line(
                      points={{0,94},{0,-88},{-2,-90}},
                      color={0,0,255},
                      smooth=Smooth.None),Rectangle(
                      extent={{-28,68},{28,-52}},
                      lineColor={0,0,255},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid),Text(
                      extent={{42,58},{78,22}},
                      lineColor={255,0,0},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid,
                      textString="P")}),
          Documentation(info="<html>
    <p>Questo componente simula, mediante inseguimento di un riferimento esterno, un carico a potenza costante.</p>
    <p>I parametri k e T sono i parametri del regolatore PI che insegue l&apos;input. TIpicamente si potr&agrave; utilizzare k=1 e T di un ordine di grandezza pi&ugrave; piccolo delle costanti di tempo del segnale di ingresso di potenza</p>
    </html>"));
      end ConstPDC;

      block Inverse "Outputs the inverse of (input multiplied by k)"
        import Modelica.Constants.inf;
        import Modelica.Constants.eps;
        Modelica.Blocks.Interfaces.RealInput u annotation (Placement(
              transformation(extent={{-128,-20},{-88,20}}), iconTransformation(
                extent={{-128,-18},{-92,18}})));
        Modelica.Blocks.Interfaces.RealOutput y annotation (Placement(
              transformation(extent={{98,-10},{118,10}}), iconTransformation(
                extent={{96,-10},{116,10}})));
        parameter Real k;
      equation
        if abs(u) < eps then
          y = inf;
        else
          y = 1.0/(k*u);
        end if;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={
                  {-100,-100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Rectangle(
                      extent={{-100,100},{100,-100}},
                      lineColor={0,0,127},
                      fillPattern=FillPattern.Solid,
                      fillColor={255,255,255}),Text(
                      extent={{-10,-4},{60,52}},
                      lineColor={0,0,127},
                      textString="1"),Text(
                      extent={{-32,0},{76,-46}},
                      lineColor={0,0,127},
                      textString="k u"),Line(
                      points={{-14,0},{66,0}},
                      color={0,0,127},
                      smooth=Smooth.None),Text(
                      extent={{-86,-30},{-16,26}},
                      lineColor={0,0,127},
                      textString="y=")}));
      end Inverse;

      model Inverter
        Modelica.Blocks.Interfaces.RealInput u
          annotation (Placement(transformation(extent={{140,-20},{100,20}})));
        Modelica.Blocks.Interfaces.RealOutput y
          annotation (Placement(transformation(extent={{-100,-10},{-120,10}})));
        Modelica.Blocks.Math.Gain gain(k=LossFact)
          annotation (Placement(transformation(extent={{10,-10},{-10,10}})));
        parameter Real LossFact=4 "Loss Factor (W per AC Arms)";
      equation
        connect(gain.u, u) annotation (Line(
            points={{12,0},{120,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gain.y, y) annotation (Line(
            points={{-11,0},{-110,0}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(extent={{-100,-100},{100,100}},
                preserveAspectRatio=false), graphics), Icon(coordinateSystem(
              extent={{-100,-100},{100,100}},
              preserveAspectRatio=false,
              initialScale=0.1,
              grid={2,2}), graphics={Rectangle(
                      lineColor={0,0,127},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid,
                      extent={{-100,60},{100,-60}}),Text(
                      origin={0,4},
                      lineColor={0,0,255},
                      fillColor={255,255,255},
                      fillPattern=FillPattern.Solid,
                      extent={{-100,102},{100,62}},
                      textString="%name")}));
      end Inverter;
    end Internal;

    model PTrifSensor "Two port three-phase power"
    protected
      Modelica.Electrical.MultiPhase.Sensors.PowerSensor powerSensor
        annotation (Placement(transformation(extent={{-12,10},{8,30}})));
      Modelica.Electrical.MultiPhase.Basic.Star star annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=270,
            origin={-2,-18})));
      Modelica.Electrical.Analog.Basic.Ground ground
        annotation (Placement(transformation(extent={{-12,-60},{8,-40}})));
    public
      Modelica.Electrical.MultiPhase.Interfaces.PositivePlug pc
        "Positive plug, current path" annotation (Placement(transformation(
              extent={{-110,10},{-90,30}}), iconTransformation(extent={{-110,-10},
                {-90,10}})));
      Modelica.Electrical.MultiPhase.Interfaces.NegativePlug nc(final m=3)
        "Negative plug, current path" annotation (Placement(transformation(
              extent={{90,30},{110,10}}, rotation=0), iconTransformation(extent
              ={{90,10},{110,-10}})));
      Modelica.Blocks.Interfaces.RealOutput Power(final unit="W") annotation (
          Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=270,
            origin={-20,-110})));
    equation
      connect(star.plug_p, powerSensor.nv) annotation (Line(
          points={{-2,-8},{-2,0},{-2,10},{-2,10}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(ground.p, star.pin_n) annotation (Line(
          points={{-2,-40},{-2,-34},{-2,-28},{-2,-28}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(powerSensor.pc, pc) annotation (Line(
          points={{-12,20},{-100,20}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(powerSensor.pv, powerSensor.pc) annotation (Line(
          points={{-2,30},{-2,48},{-12,48},{-12,20}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(powerSensor.nc, nc) annotation (Line(
          points={{8,20},{100,20}},
          color={0,0,255},
          smooth=Smooth.None));
      connect(Power, powerSensor.power) annotation (Line(
          points={{-20,-110},{-20,9},{-10,9}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                {100,100}}), graphics),
        Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                100,100}}), graphics={Line(points={{-104,0},{96,0}}, color={0,0,
              255}),Ellipse(
                  extent={{-70,70},{70,-70}},
                  lineColor={0,0,0},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid),Line(points={{0,70},{0,40}},
              color={0,0,0}),Line(points={{22.9,32.8},{40.2,57.3}}, color={0,0,
              0}),Line(points={{-22.9,32.8},{-40.2,57.3}}, color={0,0,0}),Line(
              points={{37.6,13.7},{65.8,23.9}}, color={0,0,0}),Line(points={{-37.6,
              13.7},{-65.8,23.9}}, color={0,0,0}),Line(points={{0,0},{9.02,28.6}},
              color={0,0,0}),Polygon(
                  points={{-0.48,31.6},{18,26},{18,57.2},{-0.48,31.6}},
                  lineColor={0,0,0},
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid),Ellipse(
                  extent={{-5,5},{5,-5}},
                  lineColor={0,0,0},
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid),Text(
                  extent={{-39,-3},{40,-66}},
                  lineColor={0,0,0},
                  textString="P3"),Line(
                  points={{-20,-104},{-20,-66}},
                  color={0,0,127},
                  smooth=Smooth.None),Text(
                  origin={0,24},
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  extent={{-100,102},{100,62}},
                  textString="%name")}),
        Documentation(info="<html>
<p>MODELLO OBSOLETO!</p>
<p>Esso era stato introdotto per generare una misura di potenza con un componente a due sole porte trifasi, invece che le quattro porte del modello PowerSensor di Modelica</p>
<p>Si &egrave; poi visto che ha il difetto di avere troppe equazioni (121 contro 98 di PowerSensor di Modelica)</p>
<p>Infine &egrave; stato creato Aron, che ha solo due poorte trifasi e soltanto 13 equazioni.</p>
<p>PER TUTTI I NUOVI MODELLI OCCORRE QUINDI USARE ARON.</p>
</html>"));
    end PTrifSensor;

    model DriveLosses "computes drive losses : joule and inverter"
      parameter Real Rs=0.03 "stator resistance";
      parameter Real Kv=4 "inverter los constant, W/A";
      Modelica.Blocks.Interfaces.RealInput Is
        annotation (Placement(transformation(extent={{-128,-20},{-88,20}})));
      Modelica.Blocks.Interfaces.RealOutput losses
        annotation (Placement(transformation(extent={{94,-10},{114,10}})));
    equation
      losses = 3*Rs*Is^2 + Kv*Is;
      annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{
                -100,-100},{100,100}}), graphics), Icon(coordinateSystem(
              preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
            graphics={Rectangle(
                  extent={{-100,100},{100,-100}},
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-20,-20},{20,-20},{20,-60},{40,-60},{0,-100},{-40,-60},
                {-20,-60},{-20,-20}},
                  lineColor={0,0,0},
                  smooth=Smooth.None,
                  fillColor={255,0,0},
                  fillPattern=FillPattern.Solid),Text(
                  extent={{-100,75},{100,35}},
                  lineColor={0,0,127},
                  fillColor={255,0,0},
                  fillPattern=FillPattern.Solid,
                  textString="%name")}));
    end DriveLosses;

    model PmsmAllFluxLimI "adds current control to AllFlux"
      parameter Real Rs=0.02 "stator resistance (ohm)";
      parameter Real Ipm=1.5 "PM equivalent current (=psi/Ld)";
      parameter Integer pp=1 "pole pairs";
      parameter Real Ld=0.4 "direct-axis inductance (H)", Lq=1.1
        "quadrature axis inductance (H)";
      parameter Real Inom=1 "nominal current (rms per phase)";
      parameter Real Unom=1 "nominal voltage (rms per phase)";
    protected
      parameter Real K=5000/(1.5*Ipm*pp) "Current loop gain (Nm/A)";
      parameter Real IpmEff=Ipm/sqrt(2);
    public
      parameter Real wElBase(fixed=false, start=100), gammaBase(fixed=false,
          start=0);
    public
      Modelica.Blocks.Math.Feedback feedback1(u1(start=0)) annotation (
          Placement(visible=true, transformation(
            origin={26,20},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Modelica.Blocks.Sources.Constant Inom_(k=Inom) annotation (Placement(
            visible=true, transformation(
            origin={26,-4},
            extent={{10,-10},{-10,10}},
            rotation=-90)));
      Modelica.Blocks.Math.Add add1(k1=-1) annotation (Placement(visible=true,
            transformation(
            origin={-40,36},
            extent={{-10,-10},{10,10}},
            rotation=-90)));
      Modelica.Blocks.Nonlinear.Limiter limiter1(uMax=1e99, uMin=0) annotation
        (Placement(visible=true, transformation(
            origin={-4,58},
            extent={{10,-10},{-10,10}},
            rotation=0)));
      AtomicPmsm atomicPmsm(
        Ipm=Ipm,
        pp=pp,
        Rs=Rs,
        Ld=Ld,
        Lq=Lq,
        gamma(fixed=false, start=0),
        Unom=Unom)
        annotation (Placement(transformation(extent={{-26,-60},{-6,-40}})));
      Modelica.Blocks.Interfaces.RealInput tauRef
        annotation (Placement(transformation(extent={{-128,40},{-88,80}})));
      Modelica.Blocks.Interfaces.RealInput wMechanical
        annotation (Placement(transformation(extent={{-130,-80},{-90,-40}})));
      Modelica.Blocks.Interfaces.RealOutput tauElectrical
        annotation (Placement(transformation(extent={{98,-70},{118,-50}})));
      Modelica.Blocks.Continuous.FirstOrder firstOrder(T=0.01, k=K)
        annotation (Placement(transformation(extent={{50,48},{30,68}})));
      Modelica.Blocks.Interfaces.RealOutput Is
        annotation (Placement(transformation(extent={{96,50},{116,70}})));
      Modelica.Blocks.Interfaces.RealInput uDC
        annotation (Placement(transformation(extent={{-130,-20},{-90,20}})));
      Modelica.Blocks.Continuous.FirstOrder firstOrder1(T=0.01, k=1)
        annotation (Placement(transformation(extent={{-76,50},{-56,70}})));
    initial equation
      //le seguenti equazioni determinano wElBase e gammaBase
      Unom^2 = wElBase^2*((Ld*(IpmEff - Inom)*sin(gammaBase))^2 + (Lq*Inom*cos(
        gammaBase))^2);
      0 = (-Ld*IpmEff*sin(gammaBase)) + (Lq - Ld)*Inom*cos(2*gammaBase);

    equation
      connect(limiter1.y, add1.u1) annotation (Line(points={{-15,58},{-34,58},{
              -34,48}}, color={0,0,127}));
      connect(Inom_.y, feedback1.u2)
        annotation (Line(points={{26,7},{26,7},{26,12}}, color={0,0,127}));
      connect(atomicPmsm.wMechanical, wMechanical) annotation (Line(
          points={{-26.8,-56},{-60,-56},{-60,-60},{-110,-60}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(atomicPmsm.Is, feedback1.u1) annotation (Line(
          points={{-5.2,-44},{6,-44},{6,20},{18,20}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(atomicPmsm.tauElectrical, tauElectrical) annotation (Line(
          points={{-5.2,-56},{52,-56},{52,-60},{108,-60}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(limiter1.u, firstOrder.y) annotation (Line(
          points={{8,58},{29,58}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(firstOrder.u, feedback1.y) annotation (Line(
          points={{52,58},{68,58},{68,20},{35,20}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(Is, feedback1.u1) annotation (Line(
          points={{106,60},{80,60},{80,-44},{6,-44},{6,20},{18,20}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(atomicPmsm.uDC, uDC) annotation (Line(
          points={{-27,-50},{-74,-50},{-74,0},{-110,0}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(add1.y, atomicPmsm.torqueReq) annotation (Line(
          points={{-40,25},{-40,-44},{-27,-44}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(tauRef, firstOrder1.u) annotation (Line(
          points={{-108,60},{-78,60}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(firstOrder1.y, add1.u2) annotation (Line(
          points={{-55,60},{-46,60},{-46,48}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=true,
            initialScale=0.1,
            grid={2,2})),
        Diagram(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2})),
        experiment(
          StartTime=0,
          StopTime=6,
          Tolerance=0.0001,
          Interval=0.003));
    end PmsmAllFluxLimI;

    model AtomicPmsm
      "IdIq pmsm drive train with full and weakened flux,  resistance, torque output"
      // at low speeds (below wBase) operates with optimal angle between PM and
      // stator fields, to minimise current; at higher speeds, instead of this
      // angle it imposes the machine line-to-line peak voltage to be equal to the DC voltage
      parameter Real Ipm=1.5;
      parameter Integer pp=1 "pole pairs";
      parameter Real Rs=0.02;
      parameter Real Ld=0.4, Lq=1.1;
      parameter Real Unom "nominal voltage (rms per phase)";
    protected
      parameter Real UnomPk=sqrt(2)*Unom "nominal voltage (peak per phase)";
      //The following voltage Ulimi is set to be equal to Udc without margins
      //since this model implementation allows it. In pcactice, obviously
      //some safety coefficient will be included since the voltage control is not
      // "perfect" and instantaneous
    public
      Real Ulim=min(uDC/sqrt(3), UnomPk)
        "tensione limite (fase-picco) fa attivare il deflussaggio;";
      Modelica.SIunits.Angle gammaFF(start=0), gamma(start=0);
      //  Real Is "corrente rapportata al valore nominale (es. rms/rms)";
      Real P "potenza rapportata a UnIn";
      Real Id(start=0), Iq, Vd, Vq;
      Real IdFF "Id FullFlux (i.e. before flux weaking evalation)", IqFF
        "Iq FullFlux (i.e. before flux weaking evalation)", VdFF
        "Vd FullFlux (i.e. before flux weaking evalation)", VqFF
        "Vq FullFlux (i.e. before flux weaking evalation)", IparkFF(start=0)
        "Ipark amplitude FullFlux (i.e. before flux weaking evalation)",
        VparkFF "Vpark amplitude FullFlux (i.e. before flux weaking evalation)";
      Real Ipark(start=70) "Ipark amplitude (=sqrt(Id^2+Iq^2))";
      Real Vpark "Vpark amplitude (=sqrt(Vd^2+Vq^2))";
      Real T1 "Torque due to PM field";
      Real T2 "Torque due to anisotropy (reluctance torque)";
      Real w=pp*wMechanical;
      Real weakening;
      //weakening should be boolean. It is assumed to be real because otherwise this
      //model will not work under OpenModelica 1.9.2.
    protected
      parameter Real Psi=Ipm*Ld "psi=Ipm*Ld";
    public
      Modelica.Blocks.Interfaces.RealInput torqueReq annotation (Placement(
            transformation(extent={{-130,40},{-90,80}}), iconTransformation(
              extent={{-130,40},{-90,80}})));
      Modelica.Blocks.Interfaces.RealInput wMechanical annotation (Placement(
            transformation(extent={{-128,-80},{-88,-40}}), iconTransformation(
              extent={{-128,-80},{-88,-40}})));
      Modelica.Blocks.Interfaces.RealOutput Is annotation (Placement(
            transformation(extent={{98,50},{118,70}}), iconTransformation(
              extent={{98,50},{118,70}})));
      Modelica.Blocks.Interfaces.RealOutput tauElectrical annotation (Placement(
            transformation(extent={{98,-70},{118,-50}}), iconTransformation(
              extent={{98,-70},{118,-50}})));
      Modelica.Blocks.Interfaces.RealInput uDC "DC voltage"
        annotation (Placement(transformation(extent={{-130,-20},{-90,20}})));
    equation
      //Computations with fill flux.
      //The following two equations determine IparkFF and gammaFF:
      0 = (-Psi*sin(gammaFF)) + (Lq - Ld)*IparkFF*cos(2*gammaFF);
      torqueReq = 1.5*pp*(Psi*IparkFF*cos(gammaFF) + (Lq - Ld)/2*IparkFF^2*sin(
        2*gammaFF));
      //omputation of  Id, Iq, Vd, Vq, V
      IdFF = -IparkFF*sin(gammaFF);
      IqFF = IparkFF*cos(gammaFF);
      VdFF = Rs*IdFF - w*Lq*IqFF;
      VqFF = Rs*IqFF + w*(Psi + Ld*IdFF);
      VparkFF = sqrt(VdFF^2 + VqFF^2);
      if VparkFF < Ulim then
        weakening = 0;
        //weakening should be boolean. It is assumed to be real because otherwise this
        //model will not work under OpenModelica 1.9.2.
        0 = (-Psi*sin(gamma)) + (Lq - Ld)*IparkFF*cos(2*gamma);
        Id = IdFF;
        Iq = IqFF;
        Vd = VdFF;
        Vq = VqFF;
        Vpark = VparkFF;
      else
        weakening = 1;
        //weakening should be boolean. It is assumed to be real because otherwise this
        //model will not work under OpenModelica 1.9.2.
        Id = -Ipark*sin(gamma);
        Iq = Ipark*cos(gamma);
        Vd = Rs*Id - w*Lq*Iq;
        Vq = Rs*Iq + w*(Psi + Ld*Id);
        Vpark = sqrt(Vd^2 + Vq^2);
        Vpark = Ulim;
        //   gammaFilt+tauFilt*der(gammaFilt)=gamma;
      end if;
      //  T1 = 1.5*pp*Psi*Ipark*cos(gammaFilt);
      //  T2 = 1.5*pp*(Lq - Ld)/2*Ipark^2*sin(2*gammaFilt);
      T1 = 1.5*pp*Psi*Ipark*cos(gamma);
      T2 = 1.5*pp*(Lq - Ld)/2*Ipark^2*sin(2*gamma);
      tauElectrical = T1 + T2;
      //Equating the generated torqe to the input one:
      tauElectrical = torqueReq;
      // Other quantities:
      P = wMechanical*tauElectrical;
      Is = Ipark/sqrt(2);
      annotation (Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics), Diagram(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics));
    end AtomicPmsm;
    annotation (Icon(graphics={
          Ellipse(extent={{-36,40},{40,-36}}, lineColor={0,0,0}),
          Line(
            points={{4,82},{-6,82},{-10,72},{-24,68},{-34,78},{-46,70},{-42,58},
                {-54,46},{-66,50},{-74,36},{-66,30},{-68,16},{-78,12},{-78,2}},

            color={0,0,0},
            smooth=Smooth.None),
          Line(
            points={{4,-78},{-6,-78},{-10,-68},{-24,-64},{-34,-74},{-46,-66},{-42,
                -54},{-54,-42},{-66,-46},{-74,-32},{-66,-26},{-68,-12},{-78,-8},
                {-78,2}},
            color={0,0,0},
            smooth=Smooth.None),
          Line(
            points={{2,-78},{12,-78},{16,-68},{30,-64},{40,-74},{52,-66},{48,-54},
                {60,-42},{72,-46},{80,-32},{72,-26},{74,-12},{84,-8},{84,2}},
            color={0,0,0},
            smooth=Smooth.None),
          Line(
            points={{2,82},{12,82},{16,72},{30,68},{40,78},{52,70},{48,58},{60,
                46},{72,50},{80,36},{72,30},{74,16},{84,12},{84,2}},
            color={0,0,0},
            smooth=Smooth.None)}));

  end SupportModels;

  package Icons
    model EcuIcon
      SupportModels.Conn conn1 annotation (Placement(
          visible=true,
          transformation(extent={{-20,78},{20,118}}, rotation=0),
          iconTransformation(extent={{-20,80},{20,120}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealInput motTauInt annotation (Placement(
          visible=true,
          transformation(extent={{-140,-20},{-100,20}}, rotation=0),
          iconTransformation(extent={{-140,-20},{-100,20}}, rotation=0)));
      annotation (Icon(graphics={Rectangle(
                  extent={{-100,100},{100,-100}},
                  lineColor={0,0,0},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-4,-40},{74,16},{74,-6},{-4,-62},{-4,-40}},
                  lineColor={95,95,95},
                  fillColor={175,175,175},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{8,-38},{28,-48},{20,-54},{0,-44},{8,-38}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{20,-54},{28,-48},{32,-56},{24,-62},{20,-54}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{24,-62},{32,-56},{32,-78},{24,-84},{24,-62}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{0,-44},{20,-54},{24,-62},{24,-84},{22,-84},{22,-62},
                {20,-58},{0,-48},{0,-44}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-14,40},{-18,32},{-10,38},{-8,44},{-14,40}},
                  lineColor={128,128,128},
                  fillColor={128,128,128},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-18,32},{-10,38},{-10,14},{-18,8},{-18,32}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-20,10},{-20,32},{-16,40},{4,30},{4,26},{-16,36},{-18,
                32},{-18,8},{-20,10}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-8,46},{12,36},{4,30},{-16,40},{-8,46}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{28,-22},{48,-32},{40,-38},{20,-28},{28,-22}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{40,-38},{48,-32},{52,-40},{44,-46},{40,-38}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{44,-46},{52,-40},{52,-62},{44,-68},{44,-46}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{20,-28},{40,-38},{44,-46},{44,-68},{42,-68},{42,-46},
                {40,-42},{20,-32},{20,-28}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{48,-8},{68,-18},{60,-24},{40,-14},{48,-8}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{60,-24},{68,-18},{72,-26},{64,-32},{60,-24}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{64,-32},{72,-26},{72,-48},{64,-54},{64,-32}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{40,-14},{60,-24},{64,-32},{64,-54},{62,-54},{62,-32},
                {60,-28},{40,-18},{40,-14}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{68,6},{88,-4},{80,-10},{60,0},{68,6}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{80,-10},{88,-4},{92,-12},{84,-18},{80,-10}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{84,-18},{92,-12},{92,-34},{84,-40},{84,-18}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{60,0},{80,-10},{84,-18},{84,-40},{82,-40},{82,-18},{
                80,-14},{60,-4},{60,0}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-34,26},{-38,18},{-30,24},{-28,30},{-34,26}},
                  lineColor={128,128,128},
                  fillColor={128,128,128},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-38,18},{-30,24},{-30,0},{-38,-6},{-38,18}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-40,-4},{-40,18},{-36,26},{-16,16},{-16,12},{-36,22},
                {-38,18},{-38,-6},{-40,-4}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-28,32},{-8,22},{-16,16},{-36,26},{-28,32}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-54,12},{-58,4},{-50,10},{-48,16},{-54,12}},
                  lineColor={128,128,128},
                  fillColor={128,128,128},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-58,4},{-50,10},{-50,-14},{-58,-20},{-58,4}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-60,-18},{-60,4},{-56,12},{-36,2},{-36,-2},{-56,8},{
                -58,4},{-58,-20},{-60,-18}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-48,18},{-28,8},{-36,2},{-56,12},{-48,18}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-74,-4},{-78,-12},{-70,-6},{-68,0},{-74,-4}},
                  lineColor={128,128,128},
                  fillColor={128,128,128},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-78,-12},{-70,-6},{-70,-30},{-78,-36},{-78,-12}},
                  lineColor={0,0,255},
                  fillColor={255,255,127},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-80,-34},{-80,-12},{-76,-4},{-56,-14},{-56,-18},{-76,
                -8},{-78,-12},{-78,-36},{-80,-34}},
                  lineColor={0,0,255},
                  fillColor={191,191,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-68,2},{-48,-8},{-56,-14},{-76,-4},{-68,2}},
                  lineColor={0,0,255},
                  fillColor={255,255,0},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-64,-8},{-4,-40},{-4,-62},{-64,-30},{-64,-8}},
                  lineColor={95,95,95},
                  fillColor={75,75,75},
                  fillPattern=FillPattern.Solid),Polygon(
                  points={{-64,-8},{-4,-40},{74,16},{14,48},{-64,-8}},
                  lineColor={95,95,95},
                  fillColor={160,160,164},
                  fillPattern=FillPattern.Solid),Text(
                  origin={-1,-42},
                  lineColor={0,0,255},
                  extent={{-119,-64},{119,-104}},
                  textString="%name"),Rectangle(
                  extent={{-98,92},{98,62}},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  pattern=LinePattern.None)}), Diagram(coordinateSystem(
              preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
            graphics));
    end EcuIcon;

    model SupportIcon
      annotation (Icon(graphics={Ellipse(extent={{-38,38},{38,-38}}, lineColor=
              {0,0,0}),Line(
                  points={{2,80},{-8,80},{-12,70},{-26,66},{-36,76},{-48,68},{-44,
                56},{-56,44},{-68,48},{-76,34},{-68,28},{-70,14},{-80,10},{-80,
                0}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{2,-80},{-8,-80},{-12,-70},{-26,-66},{-36,-76},{-48,-68},
                {-44,-56},{-56,-44},{-68,-48},{-76,-34},{-68,-28},{-70,-14},{-80,
                -10},{-80,0}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{0,-80},{10,-80},{14,-70},{28,-66},{38,-76},{50,-68},
                {46,-56},{58,-44},{70,-48},{78,-34},{70,-28},{72,-14},{82,-10},
                {82,0}},
                  color={0,0,0},
                  smooth=Smooth.None),Line(
                  points={{0,80},{10,80},{14,70},{28,66},{38,76},{50,68},{46,56},
                {58,44},{70,48},{78,34},{70,28},{72,14},{82,10},{82,0}},
                  color={0,0,0},
                  smooth=Smooth.None)}));
    end SupportIcon;
  end Icons;

  package FullVehicles "Full Vehicles examples"
    extends Modelica.Icons.ExamplesPackage;



    model PSecu1
      "Full Power Split Device power train using Map-Based components"
      import Modelica.Constants.*;
      extends Modelica.Icons.Example;
      parameter Real vMass=1300;
      parameter Real wIceStart=50;
      parameter Real factorDebug=100;
      // rad/s
      Modelica.SIunits.Power Pbat;
      Modelica.SIunits.Energy Ebat, Egen, Eroad;
      Modelica.SIunits.Energy Eice, Emot, Emass;
      Modelica.Mechanics.Rotational.Components.IdealPlanetary PSD(ratio=78/30)
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-50,52})));
      Modelica.Mechanics.Rotational.Components.IdealGear idealGear(ratio=3.905)
        annotation (Placement(transformation(extent={{2,42},{22,62}})));
      Modelica.Mechanics.Translational.Sensors.SpeedSensor carVel annotation (
          Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=270,
            origin={78,-12})));
      Modelica.Mechanics.Translational.Components.Mass mass(v(fixed=true, start
            =0), m=vMass)
        annotation (Placement(transformation(extent={{54,42},{74,62}})));
      SupportModels.DragForce dragForce(
        fc=0.014,
        rho=1.226,
        m=vMass,
        S=2.2,
        Cx=0.26) annotation (Placement(transformation(
            extent={{-9,-9},{9,9}},
            rotation=90,
            origin={89,29})));
      MapBased.MBiceConn ice(vMass=vMass, wIceStart=wIceStart)
        annotation (Placement(transformation(extent={{-98,46},{-78,66}})));
      SupportModels.Batt1Conn battery(
        ECellMin=0.9,
        ECellMax=1.45,
        R0Cell=0.0003,
        ns=168,
        QCellNom=2*6.5*3600.0,
        SOCInit=0.8,
        ICellMax=1e5,
        iCellEfficiency=15*6.5) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-16,0})));
      SupportModels.PropDriver driver(
        yMax=1.8,
        CycleFileName="Sort1.txt",
        k=1) annotation (Placement(visible=true, transformation(extent={{-52,-50},
                {-32,-30}}, rotation=0)));
      SupportModels.Conn d annotation (Placement(
          visible=true,
          transformation(extent={{2,-40},{28,-16}}, rotation=0),
          iconTransformation(extent={{4,-52},{30,-28}}, rotation=0)));
      MapBased.MBecu1 ECU annotation (Placement(visible=true, transformation(
            origin={-10,-40},
            extent={{-10,-8},{10,8}},
            rotation=0)));
      MapBased.MBTwoFlangeConn mot annotation (Placement(visible=true,
            transformation(extent={{-28,62},{-8,42}}, rotation=0)));
      Modelica.Mechanics.Rotational.Components.IdealRollingWheel wheel(radius=
            0.31) annotation (Placement(visible=true, transformation(
            origin={38,52},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Modelica.Electrical.Analog.Basic.Ground ground annotation (Placement(
            visible=true, transformation(
            origin={10,26},
            extent={{10,10},{-10,-10}},
            rotation=270)));
      MapBased.MBOneFlangeConn gen annotation (Placement(visible=true,
            transformation(extent={{-38,14},{-58,34}}, rotation=0)));
    equation
      connect(gen.pin_p, battery.p) annotation (Line(points={{-38,28},{-24,28},
              {-24,10},{-23.75,10}}, color={0,0,255}));
      connect(gen.pin_n, battery.n) annotation (Line(points={{-38,20},{-8.5,20},
              {-8.5,10.1}}, color={0,0,255}));
      connect(gen.flange_a, PSD.sun) annotation (Line(points={{-58,24},{-58,24},
              {-70,24},{-70,52},{-60,52}}));
      connect(gen.conn, ECU.conn1) annotation (Line(
          points={{-58,16.2},{-58,-20},{-10,-20},{-10,-30}},
          color={255,204,51},
          thickness=0.5));
      connect(ground.p, battery.n) annotation (Line(points={{0,26},{-8.5,26},{-8.5,
              10.1}}, color={0,0,255}));
      connect(wheel.flangeT, mass.flange_a)
        annotation (Line(points={{48,52},{54,52}}, color={0,127,0}));
      connect(wheel.flangeR, idealGear.flange_b)
        annotation (Line(points={{28,52},{22,52}}));
      connect(PSD.ring, mot.flange_a)
        annotation (Line(points={{-40,52},{-34,52},{-28,52}}));
      connect(idealGear.flange_a, mot.flange_b)
        annotation (Line(points={{2,52},{-4,52},{-4,52.2},{-8,52.2}}));
      connect(mot.pin_p, battery.p) annotation (Line(points={{-22,42.2},{-22,10},
              {-23.75,10}}, color={0,0,255}));
      connect(mot.pin_n, battery.n) annotation (Line(points={{-14,42},{-14,10.1},
              {-8.5,10.1}}, color={0,0,255}));
      connect(mot.conn1, ECU.conn1) annotation (Line(
          points={{-27.2,59.8},{-27.2,76},{50,76},{50,-20},{-10,-20},{-10,-30}},

          color={255,204,51},
          thickness=0.5));

      connect(battery.conn, ECU.conn1) annotation (Line(
          points={{-15.75,-10},{-16,-10},{-16,-20},{-10,-20},{-10,-30}},
          color={255,204,51},
          thickness=0.5));
      connect(ice.conn, ECU.conn1) annotation (Line(
          points={{-86.4,46},{-86,46},{-86,-20},{-10,-20},{-10,-30}},
          color={255,204,51},
          thickness=0.5));
      connect(ECU.conn1, d) annotation (Line(
          points={{-10,-30},{-10,-28},{15,-28}},
          color={255,204,51},
          thickness=0.5));
      connect(carVel.v, driver.V) annotation (Line(points={{78,-23},{78,-58},{-42,
              -58},{-42,-51.2}}, color={0,0,127}));
      Pbat = battery.p.i*(battery.p.v - battery.n.v);
      der(Ebat) = Pbat;
      der(Egen) = gen.pin_p.i*(gen.pin_p.v - gen.pin_n.v) + gen.flange_a.tau*
        der(gen.flange_a.phi);
      der(Eroad) = dragForce.flange.f*der(dragForce.flange.s);
      der(Eice) = ice.flange_a.tau*der(ice.flange_a.phi);
      der(Emot) = mot.flange_a.tau*der(mot.flange_a.phi) + mot.flange_b.tau*der(
        mot.flange_b.phi);
      Emass = 0.5*mass.m*der(mass.flange_a.s)^2;
      connect(PSD.carrier, ice.flange_a) annotation (Line(
          points={{-60,56},{-70,56},{-70,58},{-78,58}},
          color={0,0,0},
          smooth=Smooth.None));
      connect(dragForce.flange, mass.flange_b) annotation (Line(
          points={{89,38},{90,38},{90,52},{74,52}},
          color={0,127,0},
          smooth=Smooth.None));
      connect(carVel.flange, mass.flange_b) annotation (Line(
          points={{78,-2},{78,52},{74,52}},
          color={0,127,0},
          smooth=Smooth.None));
      connect(ECU.motTauInt, driver.tauRef) annotation (Line(
          points={{-22,-40},{-31,-40}},
          color={0,0,127},
          smooth=Smooth.None));
      annotation (
        experiment(StopTime=200, __Dymola_NumberOfIntervals=1000),
        __Dymola_experimentSetupOutput,
        Documentation(info="<html>
<p>This model tries to make the ICE to operate at the highest possible torque since this corresponds to the best fuel consumption given the delivered power. </p>
<p>This has two main inconveniences:</p>
<ul>
<li>the battery SOC is not controlled and tends to drift</li>
<li>in urban environment the power is too low to allow efficient drive without shutting off the engine.</li>
</ul>
<p>Both these inconveniencess are addressed in podel PSecu2.</p>
</html>"),
        Diagram(coordinateSystem(
            extent={{-100,-60},{100,80}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2}), graphics),
        Icon(coordinateSystem(
            extent={{-100,-100},{100,100}},
            preserveAspectRatio=false,
            initialScale=0.1,
            grid={2,2})));
    end PSecu1;

  end FullVehicles;
  annotation (
    uses(Modelica(version="3.2.1")),
    Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{100,
            100}}), graphics={
        Polygon(
          points={{-60,16},{78,16},{94,0},{96,-16},{-98,-16},{-90,0},{-76,12},{
              -60,16}},
          lineColor={0,0,0},
          smooth=Smooth.None,
          fillColor={0,0,255},
          fillPattern=FillPattern.Solid),
        Ellipse(
          extent={{-70,-4},{-30,-40}},
          lineColor={95,95,95},
          fillColor={95,95,95},
          fillPattern=FillPattern.Solid),
        Ellipse(
          extent={{34,-6},{74,-42}},
          lineColor={95,95,95},
          fillColor={95,95,95},
          fillPattern=FillPattern.Solid),
        Polygon(
          points={{-54,16},{-18,46},{46,46},{74,16},{-54,16}},
          lineColor={0,0,0},
          smooth=Smooth.None,
          fillColor={0,0,255},
          fillPattern=FillPattern.Solid),
        Ellipse(
          extent={{-86,-6},{-92,4}},
          lineColor={0,0,0},
          fillColor={255,255,0},
          fillPattern=FillPattern.Solid),
        Ellipse(
          extent={{98,-10},{92,-4}},
          lineColor={0,0,0},
          fillColor={255,0,0},
          fillPattern=FillPattern.Solid),
        Polygon(
          points={{-46,20},{-20,42},{16,42},{14,20},{-46,20}},
          lineColor={0,0,0},
          smooth=Smooth.None,
          fillColor={255,255,255},
          fillPattern=FillPattern.Solid),
        Polygon(
          points={{22,42},{42,42},{60,20},{20,20},{22,42}},
          lineColor={0,0,0},
          smooth=Smooth.None,
          fillColor={255,255,255},
          fillPattern=FillPattern.Solid),
        Ellipse(
          extent={{-60,-12},{-40,-30}},
          lineColor={95,95,95},
          fillColor={215,215,215},
          fillPattern=FillPattern.Solid),
        Ellipse(
          extent={{44,-14},{64,-32}},
          lineColor={95,95,95},
          fillColor={215,215,215},
          fillPattern=FillPattern.Solid)}),
    Documentation(info="<html>
<p>Library containing models of components, subsystems and full vehicle examples for simulation of electric and Hybrid vehicular power trains.</p>
<p>A general description of the library composition and on how to use it effectively is in the compaion paper:</p>
<p>M. Ceraolo &QUOT;Modelica Electric and hybrid power trains library&QUOT; submitted for publication at the 11th International Modelica Conference, 2015, September 21-23, Palais des congr&egrave;s de Versailles, 23-23 September, France</p>
</html>"));
end EHPT;
