within ;
package PreFlexMS
  "Models for Concentrated Solar Power Plants dynamic simulation"

  package FullScale_280degC
    "Package containing the models regarding the OTSG design with 280 degC as FW OTSG nominal inlet temperature"

    package HeatExchangers
      "Package including all the heat exchangers model (OTSG and preheating train)"
      package BaseClasses "Partial models of the heat exchangers"
        partial model BaseWaterMSHeatExchanger
          "Base model for water/steam-MS fluid heat exchanger"
          replaceable package coldMedium =
              Modelica.Media.Interfaces.PartialTwoPhaseMedium
            "Water/steam fluid medium model"
              annotation(choicesAllMatching = true);
          replaceable package hotMedium =
              Modelica.Media.Interfaces.PartialPureSubstance
            "MS fluid medium model"
              annotation(choicesAllMatching = true);

          ThermoPower.Water.FlangeA inletCold(redeclare package Medium =
                coldMedium) annotation (Placement(transformation(extent={{-10,-110},{
                    10,-90}}, rotation=0)));
          ThermoPower.Water.FlangeB outletCold(redeclare package Medium =
                coldMedium) annotation (Placement(transformation(extent={{-10,90},{10,
                    110}}, rotation=0)));
          ThermoPower.Water.FlangeA inletHot(redeclare package Medium =
                hotMedium) annotation (Placement(transformation(extent={{-110,-10},{
                    -90,10}}, rotation=0)));
          ThermoPower.Water.FlangeB outletHot(redeclare package Medium =
                hotMedium) annotation (Placement(transformation(extent={{90,-10},{110,
                    10}}, rotation=0)));
          annotation (Icon(graphics={
                Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,255}),
                Line(points={{40,-82},{40,-24}}, color={135,135,135}),
                Line(points={{60,-60},{40,-24}}, color={135,135,135}),
                Line(points={{20,-60},{40,-24}}, color={135,135,135}),
                Line(points={{-48,0},{-74,-16}}, color={0,0,255}),
                Line(points={{-74,16},{-48,0}}, color={0,0,255}),
                Line(
                  points={{-92,0},{-40,0},{0,-60},{0,60},{40,0},{108,0}},
                  color={0,0,255},
                  thickness=0.5)}),
                                  Diagram(coordinateSystem(preserveAspectRatio=false,
                          extent={{-110,-110},{110,110}}),
                                          graphics));

        end BaseWaterMSHeatExchanger;

        partial model BaseFeedwaterHeater
          "Base model for feedwater/steam extraction heat exchanger with drain inlet flow"
          replaceable package Medium =
              Modelica.Media.Water.WaterIF97_ph
            "Water/steam fluid medium model"
              annotation(choicesAllMatching = true);

          parameter Boolean use_drainIN = true;

          ThermoPower.Water.FlangeA ST_in(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-60,90},{-40,110}},rotation=0),
                iconTransformation(extent={{-10,90},{10,110}})));
          ThermoPower.Water.FlangeB DR_out(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-60,-110},{-40,-90}},rotation=
                   0), iconTransformation(extent={{-10,-110},{10,-90}})));
          ThermoPower.Water.FlangeA FW_in(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{40,-110},{60,-90}}, rotation=0),
                iconTransformation(extent={{100,-10},{120,10}})));
          ThermoPower.Water.FlangeB FW_out(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{40,90},{60,110}},    rotation=
                   0), iconTransformation(extent={{-110,-10},{-90,10}})));
          ThermoPower.Water.FlangeA DR_in(redeclare package Medium = Medium)
            if                                                                  use_drainIN
            annotation (Placement(transformation(extent={{-110,-10},{-90,10}}),
                                         iconTransformation(extent={{-80,90},{-60,110}})));
        protected
          ThermoPower.Water.FlangeA DR_in_internal(redeclare package Medium = Medium);
        equation

          if not use_drainIN then
            DR_in_internal.m_flow = 0;
            DR_in_internal.h_outflow = 1e5;
          end if;

          // Connect protected connectors to public conditional connectors
          connect(DR_in, DR_in_internal);

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                    {100,100}}),
                           graphics={
                Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,255}),
                Line(points={{-40,80},{-40,20}}, color={135,135,135}),
                Line(points={{-20,56},{-40,20}}, color={135,135,135}),
                Line(points={{-60,56},{-40,20}}, color={135,135,135}),
                Line(points={{50,0},{74,-14}},   color={0,0,255}),
                Line(points={{72,14},{50,0}},   color={0,0,255}),
                Line(
                  points={{-92,0},{-40,0},{0,-60},{0,60},{40,0},{108,0}},
                  color={0,0,255},
                  thickness=0.5)}),
                                  Diagram(coordinateSystem(preserveAspectRatio=false,
                          extent={{-100,-100},{100,100}}),
                                          graphics));
        end BaseFeedwaterHeater;

        partial model BaseFeedwaterHeater_noOpt
          "Base model for feedwater/steam extraction heat exchanger with drain inlet flow"
          replaceable package Medium =
              Modelica.Media.Water.WaterIF97_ph
            "Water/steam fluid medium model"
              annotation(choicesAllMatching = true);

          ThermoPower.Water.FlangeA ST_in(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-60,90},{-40,110}},rotation=0),
                iconTransformation(extent={{-10,90},{10,110}})));
          ThermoPower.Water.FlangeB DR_out(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-60,-110},{-40,-90}},rotation=
                   0), iconTransformation(extent={{-10,-110},{10,-90}})));
          ThermoPower.Water.FlangeA FW_in(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{40,-110},{60,-90}}, rotation=0),
                iconTransformation(extent={{100,-10},{120,10}})));
          ThermoPower.Water.FlangeB FW_out(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{40,90},{60,110}},    rotation=
                   0), iconTransformation(extent={{-110,-10},{-90,10}})));
          ThermoPower.Water.FlangeA DR_in(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-110,-10},{-90,10}}),
                                         iconTransformation(extent={{-80,90},{-60,110}})));
        equation

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                    {100,100}}),
                           graphics={
                Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,255}),
                Line(points={{-40,80},{-40,20}}, color={135,135,135}),
                Line(points={{-20,56},{-40,20}}, color={135,135,135}),
                Line(points={{-60,56},{-40,20}}, color={135,135,135}),
                Line(points={{50,0},{74,-14}},   color={0,0,255}),
                Line(points={{72,14},{50,0}},   color={0,0,255}),
                Line(
                  points={{-92,0},{-40,0},{0,-60},{0,60},{40,0},{108,0}},
                  color={0,0,255},
                  thickness=0.5)}),
                                  Diagram(coordinateSystem(preserveAspectRatio=false,
                          extent={{-100,-100},{100,100}}),
                                          graphics));
        end BaseFeedwaterHeater_noOpt;

        partial model BaseOTSG "Base model of the OTSG "

          replaceable package coldMedium =
              Modelica.Media.Water.WaterIF97_ph
            "Water/steam fluid medium model"
              annotation(choicesAllMatching = true);
          replaceable package hotMedium =
              PreFlexMS.Media.MoltenSalt60_40_quad "MS fluid medium model"
              annotation(choicesAllMatching = true);

          ThermoPower.Water.FlangeA RH_in(redeclare package Medium = coldMedium)
                                                                                 annotation (Placement(transformation(
                  extent={{180,150},{200,170}}),
                                              iconTransformation(extent={{180,210},
                    {200,230}})));
          ThermoPower.Water.FlangeA FW_in(redeclare package Medium = coldMedium)
                                                                                 annotation (Placement(transformation(
                  extent={{180,-250},{200,-230}}),
                                                iconTransformation(extent={{180,
                    -240},{200,-220}})));
          ThermoPower.Water.FlangeB RH_out(redeclare package Medium = coldMedium)
                                                                                  annotation (Placement(transformation(
                  extent={{90,300},{110,320}}), iconTransformation(extent={{10,300},
                    {30,320}})));
          ThermoPower.Water.FlangeB SH_out(redeclare package Medium = coldMedium)
                                                                                  annotation (Placement(transformation(
                  extent={{-10,300},{10,320}}),
                                              iconTransformation(extent={{90,300},
                    {110,320}})));
          ThermoPower.Water.FlangeB MS_HPH_out(redeclare package Medium = coldMedium)
                                                                                      annotation (Placement(
                transformation(extent={{-70,-320},{-50,-300}}),
                                                              iconTransformation(
                  extent={{10,-320},{30,-300}})));
          ThermoPower.Water.FlangeA MS_in(redeclare package Medium = hotMedium)
                                                                                annotation (Placement(transformation(extent={{-110,
                    300},{-90,320}}),      iconTransformation(extent={{-110,300},
                    {-90,320}})));
          ThermoPower.Water.FlangeB MS_out(redeclare package Medium = hotMedium)
                                                                                 annotation (Placement(transformation(extent={{10,-320},
                    {30,-300}}),           iconTransformation(extent={{-110,-320},
                    {-90,-300}})));
          ThermoPower.Water.FlangeB SEP_out(redeclare package Medium = coldMedium)
                                                                                   annotation (Placement(transformation(extent={{180,-90},{200,-70}})));
          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-180,
                    -300},{180,300}}),      graphics={
                Rectangle(extent={{-180,300},{180,-300}}, lineColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  fillColor={255,255,255}),
                Line(
                  points={{178,-228},{20,-228},{20,-244},{0,-248},{40,-258},{20,
                      -268},{20,-298}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{20,-228},{20,-150},{0,-140},{40,-130},{20,-120},{20,
                      -100},{0,-90},{40,-80},{20,-72},{20,20},{80,20}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Rectangle(
                  extent={{80,54},{120,-26}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid,
                  radius=10),
                Rectangle(
                  extent={{86,18},{114,-20}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  radius=5),
                Rectangle(
                  extent={{86,48},{114,10}},
                  lineColor={85,170,255},
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid,
                  radius=5),
                Line(
                  points={{100,54},{100,116},{80,128},{120,138},{100,148},{100,
                      228},{100,238},{100,298}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Ellipse(
                  extent={{90,-46},{110,-66}},
                  lineColor={0,0,255},
                  lineThickness=0.5),
                Polygon(
                  points={{92,-50},{108,-50},{100,-66},{92,-50}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  smooth=Smooth.None,
                  fillPattern=FillPattern.Solid),
                Line(
                  points={{100,-66},{100,-80},{180,-80}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{100,-26},{100,-38},{100,-46}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{94,220},{94,222},{96,224},{100,226},{104,224},{106,222},
                      {106,220},{178,220}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Text(
                  extent={{-44,248},{28,212}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  textString="RH"),
                Line(
                  points={{94,220},{70,220},{60,240},{50,200},{40,220},{20,220},{
                      20,300}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Text(
                  extent={{120,150},{174,114}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  textString="SH"),
                Text(
                  extent={{-68,-68},{-8,-110}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  textString="EVA"),
                Text(
                  extent={{-64,-120},{-6,-164}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  textString="ECO"),
                Text(
                  extent={{-80,-218},{-8,-284}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid,
                  textString="MS_HPH"),
                Line(
                  points={{-100,298},{-100,220},{-80,208},{-120,200},{-80,190},{
                      -120,180},{-80,170},{-100,160},{-100,-60},{-120,-70},{-80,
                      -80},{-120,-90},{-80,-100},{-120,-110},{-80,-120},{-120,
                      -130},{-80,-140},{-120,-150},{-80,-160},{-120,-172},{-100,
                      -180},{-100,-298}},
                  color={255,0,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Text(
                  extent={{-78,134},{66,20}},
                  lineColor={0,0,0},
                  textString="OTSG")}),  Diagram(coordinateSystem(extent={{-180,
                    -300},{180,300}},
                           preserveAspectRatio=false)));
        end BaseOTSG;

        partial model BaseHPHTrain
          "Base model of the high pressure heaters train"

          replaceable package Medium =
              Modelica.Media.Water.WaterIF97_ph
            "Water/steam fluid medium model"
              annotation(choicesAllMatching = true);

          ThermoPower.Water.FlangeA MS_HPH_in(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{30,-120},{50,-100}})));
          ThermoPower.Water.FlangeA SEP_in(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-130,100},{-110,120}}),
                iconTransformation(extent={{-130,100},{-110,120}})));
          ThermoPower.Water.FlangeA SP1(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-60,100},{-40,120}}),
                iconTransformation(extent={{-60,100},{-40,120}})));
          ThermoPower.Water.FlangeA SP2(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{80,100},{100,120}}),
                iconTransformation(extent={{70,100},{90,120}})));
          ThermoPower.Water.FlangeA FW_in(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{180,-10},{200,10}})));
          ThermoPower.Water.FlangeB FW_out(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-200,-10},{-180,10}})));
          ThermoPower.Water.FlangeB DR_out(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{180,-70},{200,-50}}),
                iconTransformation(extent={{180,50},{200,70}})));
        equation

          annotation (Diagram(coordinateSystem(extent={{-180,-100},{180,100}},
                  preserveAspectRatio=false), graphics), Icon(coordinateSystem(extent={{-180,
                    -100},{180,100}}, preserveAspectRatio=false), graphics={
                Rectangle(
                  extent={{-180,100},{180,-100}},
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid),
                Ellipse(
                  extent={{140,10},{160,-10}},
                  lineColor={0,0,255},
                  lineThickness=0.5),
                Polygon(
                  points={{156,8},{156,-8},{140,0},{156,8}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  smooth=Smooth.None,
                  fillPattern=FillPattern.Solid),
                Ellipse(
                  extent={{30,-50},{50,-70}},
                  lineColor={0,0,255},
                  lineThickness=0.5),
                Polygon(
                  points={{48,-66},{32,-66},{40,-50},{48,-66}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  smooth=Smooth.None,
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{60,20},{100,-20}},
                  lineColor={0,0,255}),
                Rectangle(
                  extent={{-60,20},{-20,-20}},
                  lineColor={0,0,255}),
                Rectangle(
                  extent={{-4,4},{4,-4}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillPattern=FillPattern.Solid,
                  fillColor={0,0,255}),
                Rectangle(
                  extent={{36,4},{44,-4}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillPattern=FillPattern.Solid,
                  fillColor={0,0,255}),
                Rectangle(
                  extent={{-128,8},{-112,-8}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-50,-44},{-50,-36},{-40,-40},{-30,-36},{-30,-44},{-40,
                      -40},{-50,-44}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  smooth=Smooth.None,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Line(
                  points={{-120,100},{-120,10},{-120,0},{-60,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{-180,0},{-140,0},{-120,0},{-120,-40},{-48,-40}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{-50,100},{-50,72},{-50,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{80,100},{80,76},{80,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{180,0},{168,0},{160,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{140,0},{114,0},{100,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{60,0},{36,0},{-20,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{40,-100},{40,-84},{40,-70}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{-30,-40},{0,-40},{0,-4}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{40,-4},{40,-24},{40,-50}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{-30,20},{-30,30},{70,30},{70,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{90,20},{90,40},{90,60},{180,60}},
                  color={127,127,0},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{-40,-40},{-40,-50}},
                  color={0,0,0},
                  thickness=0.5,
                  smooth=Smooth.None),
                Polygon(
                  points={{-44,-46},{-36,-46},{-36,-48},{-38,-50},{-42,-50},{-44,
                      -48},{-44,-46}},
                  lineColor={0,0,0},
                  lineThickness=0.5,
                  smooth=Smooth.None,
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid),
                Text(
                  extent={{-60,-54},{-16,-88}},
                  lineColor={0,0,0},
                  lineThickness=0.5,
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid,
                  textString="HPH2"),
                Text(
                  extent={{62,-52},{104,-86}},
                  lineColor={0,0,0},
                  lineThickness=0.5,
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid,
                  textString="HPH1"),
                Line(
                  points={{-60,0},{-52,0},{-50,8},{-46,-6},{-42,8},{-38,-6},{-34,8},{-30,
                      -6},{-28,0},{-20,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{60,0},{68,0},{70,8},{74,-6},{78,8},{82,-6},{86,8},{90,-6},{92,
                      0},{100,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None)}));
        end BaseHPHTrain;

        partial model BaseLPHTrain
          "Base model of the high pressure heaters train"

          replaceable package Medium =
              Modelica.Media.Water.WaterIF97_ph
            "Water/steam fluid medium model"
              annotation(choicesAllMatching = true);

          ThermoPower.Water.FlangeA SP7(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{150,100},{170,120}}),
                iconTransformation(extent={{150,100},{170,120}})));
          ThermoPower.Water.FlangeA SP4(redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-180,100},{-160,120}}),
                iconTransformation(extent={{-180,100},{-160,120}})));
          ThermoPower.Water.FlangeA SP5(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-70,100},{-50,120}}),
                iconTransformation(extent={{-70,100},{-50,120}})));
          ThermoPower.Water.FlangeA SP6(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{40,100},{60,120}}),
                iconTransformation(extent={{40,100},{60,120}})));
          ThermoPower.Water.FlangeA FW_in(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{240,-10},{260,10}})));
          ThermoPower.Water.FlangeB FW_out(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-260,-10},{-240,10}})));
        equation

          annotation (Diagram(coordinateSystem(extent={{-240,-100},{240,100}},
                  preserveAspectRatio=false), graphics), Icon(coordinateSystem(extent={{-240,
                    -100},{240,100}}, preserveAspectRatio=false), graphics={
                Rectangle(
                  extent={{-240,100},{240,-100}},
                  lineColor={0,0,255},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid),
                Ellipse(
                  extent={{200,10},{220,-10}},
                  lineColor={0,0,255},
                  lineThickness=0.5),
                Polygon(
                  points={{216,8},{216,-8},{200,0},{216,8}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  smooth=Smooth.None,
                  fillPattern=FillPattern.Solid),
                Ellipse(
                  extent={{134,-28},{146,-40}},
                  lineColor={0,0,255},
                  lineThickness=0.5),
                Polygon(
                  points={{144,-30},{144,-38},{134,-34},{144,-30}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  smooth=Smooth.None,
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{30,20},{70,-20}},
                  lineColor={0,0,255}),
                Rectangle(
                  extent={{-190,20},{-150,-20}},
                  lineColor={0,0,255}),
                Line(
                  points={{-60,100},{-60,72},{-60,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{240,0},{228,0},{220,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{140,0},{84,0},{70,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{-50,20},{-50,30},{40,30},{40,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Text(
                  extent={{140,-52},{182,-82}},
                  lineColor={0,0,0},
                  lineThickness=0.5,
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid,
                  textString="LPH1"),
                Line(
                  points={{-190,0},{-182,0},{-180,8},{-176,-6},{-172,8},{-168,-6},
                      {-164,8},{-160,-6},{-158,0},{-150,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{30,0},{38,0},{40,8},{44,-6},{48,8},{52,-6},{56,8},{60,
                      -6},{62,0},{70,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Rectangle(
                  extent={{-80,20},{-40,-20}},
                  lineColor={0,0,255}),
                Line(
                  points={{-80,0},{-72,0},{-70,8},{-66,-6},{-62,8},{-58,-6},{-54,
                      8},{-50,-6},{-48,0},{-40,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{-170,100},{-170,72},{-170,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{-160,20},{-160,30},{-70,30},{-70,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{50,100},{50,72},{50,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{60,20},{60,30},{150,30},{150,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{160,100},{160,72},{160,20}},
                  color={0,127,0},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Rectangle(
                  extent={{140,20},{180,-20}},
                  lineColor={0,0,255}),
                Line(
                  points={{140,0},{148,0},{150,8},{154,-6},{158,8},{162,-6},{166,
                      8},{170,-6},{172,0},{180,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{200,0},{188,0},{180,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{30,0},{-32,0},{-40,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{-80,0},{-142,0},{-150,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Line(
                  points={{-190,0},{-232,0},{-240,0}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None,
                  arrow={Arrow.None,Arrow.Filled}),
                Rectangle(
                  extent={{116,4},{124,-4}},
                  lineColor={0,0,255},
                  lineThickness=0.5,
                  fillPattern=FillPattern.Solid,
                  fillColor={0,0,255}),
                Line(
                  points={{134,-34},{124,-34},{120,-34},{120,-4}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Line(
                  points={{146,-34},{150,-34},{160,-34},{160,-20}},
                  color={0,0,255},
                  thickness=0.5,
                  smooth=Smooth.None),
                Text(
                  extent={{32,-50},{74,-84}},
                  lineColor={0,0,0},
                  lineThickness=0.5,
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid,
                  textString="LPH2"),
                Text(
                  extent={{-78,-50},{-36,-84}},
                  lineColor={0,0,0},
                  lineThickness=0.5,
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid,
                  textString="LPH3"),
                Text(
                  extent={{-188,-50},{-146,-84}},
                  lineColor={0,0,0},
                  lineThickness=0.5,
                  fillColor={0,0,0},
                  fillPattern=FillPattern.Solid,
                  textString="LPH4")}));
        end BaseLPHTrain;
      end BaseClasses;

      package Components "Package of heat transfer-related components"
        model HE "Model of MS/steam heat exchanger"

          extends
            PreFlexMS.FullScale_280degC.HeatExchangers.BaseClasses.BaseWaterMSHeatExchanger;

          // Heat exchange parameters-variables
          replaceable model HeatTransferCold =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true);
          replaceable model HeatTransferHot =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, hot fluid side"
            annotation(choicesAllMatching = true);

          replaceable model HeatExchangerTopology =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true);

          parameter Integer Nhex = 2 "Number of nodes for the heat exchanger";
          parameter Modelica.SIunits.Volume Vhot
            "Volume of the fluid in the hot side";
          parameter Modelica.SIunits.Volume Vcold
            "Volume of the fluid in the cold side";
          parameter Modelica.SIunits.Area Shot
            "Heat transfer surface in the hot side";
          parameter Modelica.SIunits.Area Scold
            "Heat transfer surface in the cold side";
          final parameter Modelica.SIunits.Length L=1
            "Lenght of the fluid path (unused)";
          parameter Modelica.SIunits.Mass Mm
            "Mass of the metal between the two fluids";
          parameter Modelica.SIunits.SpecificHeatCapacity cm
            "Specific heat capacity of metal";
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow
            "Nominal flow rate of cold fluid";
          parameter Modelica.SIunits.MassFlowRate nominalHotFlow
            "Nominal flow rate of hot fluid";
          parameter Modelica.SIunits.Pressure nominalColdPressure
            "Nominal inlet pressure of cold fluid";
          parameter Modelica.SIunits.Pressure nominalColdDeltaP
            "Nominal cold fluid pressure loss";
          parameter Modelica.SIunits.Pressure nominalHotPressure
            "Nominal inlet pressure of hot fluid";
          parameter Modelica.SIunits.Pressure nominalHotDeltaP
            "Nominal hot fluid pressure loss";
          parameter Modelica.SIunits.Temperature nominalColdInletTemperature
            "Nominal value of cold fluid temperature at the inlet";
          parameter Modelica.SIunits.Temperature nominalColdOutletTemperature
            "Nominal value of cold fluid temperature at the outlet";
          parameter Modelica.SIunits.Temperature nominalHotInletTemperature
            "Nominal value of cold fluid temperature at the inlet";
          parameter Modelica.SIunits.Temperature nominalHotOutletTemperature
            "Nominal value of cold fluid temperature at the outlet";

          ThermoPower.Water.Flow1DFV coldFluid(
            final N=Nhex,
            H=0,
            wnom=nominalColdFlow,
            pstart=nominalColdPressure,
            redeclare package Medium = coldMedium,
            L=L,
            dpnom=nominalColdDeltaP,
            rhonom=coldMedium.density_pT(nominalColdPressure,
                nominalColdInletTemperature),
            hstartin=coldMedium.specificEnthalpy_pT(nominalColdPressure,
                nominalColdInletTemperature),
            hstartout=coldMedium.specificEnthalpy_pT(nominalColdPressure -
                nominalColdDeltaP, nominalColdOutletTemperature),
            final A=Vcold/L,
            final omega=Scold/L,
            Dhyd=1,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.OpPoint,
            redeclare model HeatTransfer = HeatTransferCold)    annotation (Placement(transformation(extent={{-20,-72},
                    {20,-32}},      rotation=0)));
          ThermoPower.Thermal.MetalWallFV
                                   tubeWalls(
            final Nw=Nhex-1,
            final M=Mm,
            final cm=cm,
            Tstart1=(nominalColdInletTemperature + nominalHotOutletTemperature)/2,
            TstartN=(nominalColdOutletTemperature + nominalHotInletTemperature)/2)
                                                   annotation (Placement(transformation(
                  extent={{-20,4},{20,-36}}, rotation=0)));
          ThermoPower.Water.Flow1DFV hotFluid(
            N=Nhex,
            H=0,
            Dhyd=1,
            pstart=nominalHotPressure,
            redeclare package Medium = hotMedium,
            L=L,
            A=Vhot/L,
            omega=Shot/L,
            wnom=nominalHotFlow,
            dpnom=nominalHotDeltaP,
            rhonom=hotMedium.density_pT(nominalHotPressure,
                nominalHotInletTemperature),
            hstartin=hotMedium.specificEnthalpy_pT(nominalHotPressure,
                nominalHotInletTemperature),
            hstartout=hotMedium.specificEnthalpy_pT(nominalHotPressure -
                nominalHotDeltaP, nominalHotOutletTemperature),
            redeclare model HeatTransfer = HeatTransferHot,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction)
                                                               annotation (Placement(transformation(extent={{20,72},
                    {-20,32}},     rotation=0)));
          ThermoPower.Thermal.HeatExchangerTopologyFV heatExchangerTopology(Nw=Nhex
                 - 1, redeclare model HeatExchangerTopology =
                HeatExchangerTopology) annotation (Placement(transformation(
                  extent={{-20,-2},{20,38}}, rotation=0)));
          Modelica.SIunits.Power Q_Hot "Power from the hot side";
          Modelica.SIunits.Power Q_Cold "Power to the cold side";
          Modelica.SIunits.Temperature T_hot_prof[Nhex+1] = vector([hotFluid.T[1];hotFluid.heatTransfer.Tvol;hotFluid.T[end]])
            "Hot temperature vector (hot in the left side)";
          Modelica.SIunits.Temperature T_wall_prof[Nhex+1] = vector([(hotFluid.T[1]+coldFluid.T[end])/2;tubeWalls.Tvol[end:-1:1];(hotFluid.T[end]+coldFluid.T[1])/2])
            "Wall temperature vector (hot in the left side)";
          Modelica.SIunits.Temperature T_cold_prof[Nhex+1] = vector([coldFluid.T[end];coldFluid.heatTransfer.Tvol[end:-1:1];coldFluid.T[1]])
            "Cold temperature vector (hot in the left side)";

          // Initialisation script parameters-variables
          parameter Boolean generateInitCode = false
            "Generate initialization code at the end of the simulation";
          function print = Modelica.Utilities.Streams.print;
          String s;

        equation
          Q_Hot = -hotFluid.Q;
          Q_Cold = coldFluid.Q;
          connect(heatExchangerTopology.side2, tubeWalls.ext)
            annotation (Line(points={{0,11.8},{0,-9.8}}, color={255,127,0}));
          connect(hotFluid.wall, heatExchangerTopology.side1) annotation (Line(
              points={{0,42},{0,24}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(tubeWalls.int, coldFluid.wall) annotation (Line(
              points={{0,-22},{0,-42}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(coldFluid.infl, inletCold) annotation (Line(
              points={{-20,-52},{-40,-52},{-40,-100},{0,-100}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(coldFluid.outfl, outletCold) annotation (Line(
              points={{20,-52},{70,-52},{70,100},{0,100}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(hotFluid.infl, inletHot) annotation (Line(
              points={{20,52},{52,52},{52,-80},{-100,-80},{-100,0}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(hotFluid.outfl, outletHot) annotation (Line(
              points={{-20,52},{-40,52},{-40,80},{100,80},{100,0}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

        algorithm
          when terminal() and generateInitCode then
            print("HEAT EXCHANGER(");
            print("  hotFluid(");
            s := "    hstart={";
            for i in 1:hotFluid.N loop
              s := s + String(hotFluid.h[i]);
              if i < hotFluid.N then
                s := s + ",";
              end if;
            end for;
            s :=s + "},";
            print(s);

            s :="    pstart="+String(hotFluid.p)+"),";
            print(s);

            print("  coldFluid(");
            s := "    hstart={";
            for i in 1:coldFluid.N loop
              s := s + String(coldFluid.h[i]);
              if i < coldFluid.N then
                s := s + ",";
              end if;
            end for;
            s :=s + "},";
            print(s);

            s :="    pstart="+String(coldFluid.p)+"),";
            print(s);

            s := "  tubeWalls(Tvolstart={";
            for i in 1:tubeWalls.Nw loop
              s := s + String(tubeWalls.Tvol[i]);
              if i < tubeWalls.Nw then
                s := s + ",";
              end if;
            end for;
            s := s + "})));";
            print(s);
          end when;
          annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}),
                              graphics),
                               Icon(graphics));
        end HE;

        model HE_2ph "Model of MS/steam heat exchanger with phase change"

          extends
            PreFlexMS.FullScale_280degC.HeatExchangers.BaseClasses.BaseWaterMSHeatExchanger;

          replaceable model HeatTransferCold =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true);
          replaceable model HeatTransferHot =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, hot fluid side"
            annotation(choicesAllMatching = true);

          replaceable model HeatExchangerTopology =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true);

          parameter Integer Nhex = 2 "Number of nodes for the heat exchanger";
          parameter Modelica.SIunits.Volume Vhot
            "Volume of the fluid in the hot side";
          parameter Modelica.SIunits.Volume Vcold
            "Volume of the fluid in the cold side";
          parameter Modelica.SIunits.Area Shot
            "Heat transfer surface in the hot side";
          parameter Modelica.SIunits.Area Scold
            "Heat transfer surface in the cold side";
          final parameter Modelica.SIunits.Length L=1
            "Lenght of the fluid path (unused)";
          parameter Modelica.SIunits.Mass Mm
            "Mass of the metal between the two fluids";
          parameter Modelica.SIunits.SpecificHeatCapacity cm
            "Specific heat capacity of metal";
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow
            "Nominal flow rate of cold fluid";
          parameter Modelica.SIunits.MassFlowRate nominalHotFlow
            "Nominal flow rate of hot fluid";
          parameter Modelica.SIunits.Pressure nominalColdPressure
            "Nominal inlet pressure of cold fluid";
          parameter Modelica.SIunits.Pressure nominalColdDeltaP
            "Nominal cold fluid pressure loss";
          parameter Modelica.SIunits.Pressure nominalHotPressure
            "Nominal inlet pressure of hot fluid";
          parameter Modelica.SIunits.Pressure nominalHotDeltaP
            "Nominal hot fluid pressure loss";
          parameter Modelica.SIunits.Temperature nominalColdInletTemperature
            "Nominal value of cold fluid temperature at the inlet";
          parameter Modelica.SIunits.Temperature nominalColdOutletTemperature
            "Nominal value of cold fluid temperature at the outlet";
          parameter Modelica.SIunits.Temperature nominalHotInletTemperature
            "Nominal value of cold fluid temperature at the inlet";
          parameter Modelica.SIunits.Temperature nominalHotOutletTemperature
            "Nominal value of cold fluid temperature at the outlet";
          ThermoPower.Water.Flow1DFV2ph coldFluid(
            final N=Nhex,
            H=0,
            wnom=nominalColdFlow,
            pstart=nominalColdPressure,
            redeclare package Medium = coldMedium,
            L=L,
            dpnom=nominalColdDeltaP,
            rhonom=coldMedium.density_pT(nominalColdPressure,
                nominalColdInletTemperature),
            hstartin=coldMedium.specificEnthalpy_pT(nominalColdPressure,
                nominalColdInletTemperature),
            hstartout=coldMedium.specificEnthalpy_pT(nominalColdPressure -
                nominalColdDeltaP, nominalColdOutletTemperature),
            final A=Vcold/L,
            final omega=Scold/L,
            Dhyd=1,
            redeclare model HeatTransfer = HeatTransferCold,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction)
                                                                annotation (Placement(transformation(extent={{-20,-72},
                    {20,-32}},      rotation=0)));
          ThermoPower.Thermal.MetalWallFV
                                   tubeWalls(
            final Nw=Nhex-1,
            final M=Mm,
            final cm=cm,
            Tstart1=(nominalColdInletTemperature + nominalHotOutletTemperature)/2,
            TstartN=(nominalColdOutletTemperature + nominalHotInletTemperature)/2)
                                                   annotation (Placement(transformation(
                  extent={{-20,4},{20,-36}}, rotation=0)));
          ThermoPower.Water.Flow1DFV hotFluid(
            N=Nhex,
            H=0,
            Dhyd=1,
            pstart=nominalHotPressure,
            redeclare package Medium = hotMedium,
            L=L,
            A=Vhot/L,
            omega=Shot/L,
            wnom=nominalHotFlow,
            dpnom=nominalHotDeltaP,
            rhonom=hotMedium.density_pT(nominalHotPressure,
                nominalHotInletTemperature),
            hstartin=hotMedium.specificEnthalpy_pT(nominalHotPressure,
                nominalHotInletTemperature),
            hstartout=hotMedium.specificEnthalpy_pT(nominalHotPressure -
                nominalHotDeltaP, nominalHotOutletTemperature),
            redeclare model HeatTransfer = HeatTransferHot,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction)
                                                               annotation (Placement(transformation(extent={{20,72},
                    {-20,32}},     rotation=0)));
          ThermoPower.Thermal.HeatExchangerTopologyFV heatExchangerTopology(Nw=Nhex
                 - 1, redeclare model HeatExchangerTopology =
                HeatExchangerTopology) annotation (Placement(transformation(
                  extent={{-20,-2},{20,38}}, rotation=0)));
          Modelica.SIunits.Power Q_Hot "Power from the hot side";
          Modelica.SIunits.Power Q_Cold "Power to the cold side";
          Modelica.SIunits.Temperature T_hot_prof[Nhex+1] = vector([hotFluid.T[1];hotFluid.heatTransfer.Tvol;hotFluid.T[end]])
            "Hot temperature vector (hot in the left side)";
          Modelica.SIunits.Temperature T_wall_prof[Nhex+1] = vector([(hotFluid.T[1]+coldFluid.T[end])/2;tubeWalls.Tvol[end:-1:1];(hotFluid.T[end]+coldFluid.T[1])/2])
            "Wall temperature vector (hot in the left side)";
          Modelica.SIunits.Temperature T_cold_prof[Nhex+1] = vector([coldFluid.T[end];coldFluid.heatTransfer.Tvol[end:-1:1];coldFluid.T[1]])
            "Cold temperature vector (hot in the left side)";

          // Initialisation script parameters-variables
          parameter Boolean generateInitCode = false
            "Generate initialization code at the end of the simulation";
          function print = Modelica.Utilities.Streams.print;
          String s;

        equation
          Q_Hot = -hotFluid.Q;
          Q_Cold = coldFluid.Q;
          connect(heatExchangerTopology.side2, tubeWalls.ext)
            annotation (Line(points={{0,11.8},{0,-9.8}}, color={255,127,0}));
          connect(hotFluid.wall, heatExchangerTopology.side1) annotation (Line(
              points={{0,42},{0,24}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(tubeWalls.int, coldFluid.wall) annotation (Line(
              points={{0,-22},{0,-42}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(coldFluid.infl, inletCold) annotation (Line(
              points={{-20,-52},{-40,-52},{-40,-100},{0,-100}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(coldFluid.outfl, outletCold) annotation (Line(
              points={{20,-52},{70,-52},{70,100},{0,100}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(hotFluid.infl, inletHot) annotation (Line(
              points={{20,52},{52,52},{52,-80},{-100,-80},{-100,0}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(hotFluid.outfl, outletHot) annotation (Line(
              points={{-20,52},{-40,52},{-40,80},{100,80},{100,0}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

        algorithm
          when terminal() and generateInitCode then
            print("HEAT EXCHANGER(");
            print("  hotFluid(");
            s := "    hstart={";
            for i in 1:hotFluid.N loop
              s := s + String(hotFluid.h[i]);
              if i < hotFluid.N then
                s := s + ",";
              end if;
            end for;
            s :=s + "},";
            print(s);

            s :="    pstart="+String(hotFluid.p)+"),";
            print(s);

            print("  coldFluid(");
            s := "    hstart={";
            for i in 1:coldFluid.N loop
              s := s + String(coldFluid.h[i]);
              if i < coldFluid.N then
                s := s + ",";
              end if;
            end for;
            s :=s + "},";
            print(s);

            s :="    pstart="+String(coldFluid.p)+"),";
            print(s);

            s := "  tubeWalls(Tvolstart={";
            for i in 1:tubeWalls.Nw loop
              s := s + String(tubeWalls.Tvol[i]);
              if i < tubeWalls.Nw then
                s := s + ",";
              end if;
            end for;
            s := s + "})));";
            print(s);
          end when;
          annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}),
                              graphics),
                               Icon(graphics));
        end HE_2ph;

        model Equilibrium2phVessel
          "Base model of an adiabatic steam/liquid separator at thermodynamic equilibrium"

          replaceable package Medium = Modelica.Media.Water.WaterIF97_ph constrainedby
            Modelica.Media.Interfaces.PartialMedium "Medium"
            annotation (choicesAllMatching=true);

          parameter Modelica.SIunits.Length D "tank diameter";
          parameter Modelica.SIunits.Height H "Height of the tank";
          final parameter Modelica.SIunits.Volume V=A*H "Total volume";
          final parameter Modelica.SIunits.Area A=D^2/4*Modelica.Constants.pi
            "Cross Sectional Area";
          parameter Modelica.SIunits.Height zl_start "Medium start level";
          parameter Medium.SpecificEnthalpy h_start "Start specific enthalpy";
          parameter Medium.AbsolutePressure p_start "Start pressure";
          parameter Medium.MassFlowRate w_small = 1e-3
            "Small flow rate for reverse flow check";
          outer ThermoPower.System system "System wide properties";
          parameter Boolean allowFlowReversal=system.allowFlowReversal
            "= true to allow flow reversal, false restricts to design direction";
          parameter ThermoPower.Choices.Init.Options initOpt=system.initOpt
            "Initialisation option";
          parameter Boolean noInitialPressure=false
            "Remove initial equation on pressure"
            annotation (Dialog(tab="Initialisation"),choices(checkBox=true));
          parameter Boolean noInitialLevel=true
            "Remove initial equation on level"
            annotation (Dialog(tab="Initialisation"),choices(checkBox=true));

          ThermoPower.Water.FlangeA water_in(redeclare package Medium = Medium, m_flow(
                min=if allowFlowReversal then -Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-104,-38},{-84,-18}},
                  rotation=0), iconTransformation(extent={{-60,-50},{-40,-30}})));
          ThermoPower.Water.FlangeB drained_out(redeclare package Medium = Medium,
              m_flow(max=if allowFlowReversal then +Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-10,-90},{10,-70}}, rotation=0),
                iconTransformation(extent={{-10,-100},{10,-80}})));
          ThermoPower.Water.FlangeB steam_out(
                                           redeclare package Medium = Medium,
              m_flow(max=if allowFlowReversal then +Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-10,82},{10,102}},  rotation=0),
                iconTransformation(extent={{-10,110},{10,130}})));

          Modelica.Blocks.Interfaces.RealOutput level annotation (Placement(transformation(
                  extent={{40,-80},{60,-60}}),iconTransformation(extent={{40,-80},{60,-60}})));
          Modelica.Blocks.Interfaces.RealOutput DT_sh annotation (Placement(transformation(
                  extent={{-228,68},{-208,88}}), iconTransformation(extent={{40,40},{60,
                    60}})));

          Medium.ThermodynamicState state
            "Thermodynamic state of the separator";
          Medium.SaturationProperties sat
            "Saturation properties of the separator";

        protected
          constant Modelica.SIunits.Acceleration g=Modelica.Constants.g_n;
        public
          Medium.MassFlowRate w_in "Medium inflow mass flow rate";
          Medium.MassFlowRate wl_in
            "Medium inflow mass flow rate (liquid fraction)";
          Medium.MassFlowRate wl_out "Medium drained outflow mass flow rate";
          Medium.MassFlowRate wv_out "Medium steam outflow mass flow rate";
          Modelica.SIunits.Height zl(start=zl_start)
            "Medium level (relative to reference)";
          Medium.SpecificEnthalpy hl_in "Medium inlet specific enthalpy";
          Medium.SpecificEnthalpy hl_out
            "Medium drained outlet specific enthalpy";
          Medium.SpecificEnthalpy hv_out
            "Medium steam outlet specific enthalpy";
          Medium.SpecificEnthalpy hl "Bubble point specific enthalpy";
          Medium.SpecificEnthalpy hv "Dew-point specific enthalpy";
          Modelica.SIunits.Mass M "Total mass";
          Modelica.SIunits.Mass Ml "Mass of liquid";
          Medium.Temperature T "Medium temperature";
          Medium.Temperature T_sat "Saturation temperature";
          Medium.AbsolutePressure p(start = p_start, stateSelect=StateSelect.prefer)
            "Medium pressure";
          Modelica.SIunits.SpecificEnthalpy h(start = h_start, stateSelect=StateSelect.prefer)
            "Specific mixture enthalpy";
          Modelica.SIunits.Density rho "Mixture density";
          ThermoPower.Units.LiquidDensity rho_l "Density of the liquid medium";
          Modelica.SIunits.DerDensityByPressure drdp
            "Derivative of density by pressure";
          Modelica.SIunits.DerDensityByEnthalpy drdh
            "Derivative of density by enthalpy";
          Modelica.SIunits.MassFraction  x_in = Medium.vapourQuality(Medium.setState_ph(p,hl_in))
            "Inlet steam quality";
          Modelica.SIunits.MassFraction  x "Mixture steam quality";

          Real vf "Void fraction";
          Integer phase(start = 1)
            "Definition of thermodynamic state in the separator";

        equation
          // Fluid property calculations
          sat = Medium.setSat_p(p);
          phase = if h >= hv then 1 else 2;
          state = Medium.setState_ph(p,h,phase);
          T = Medium.temperature(state);
          T_sat = Medium.saturationTemperature(p);
          hv = Medium.dewEnthalpy(sat);
          hl = Medium.bubbleEnthalpy(sat);
          rho_l = Medium.bubbleDensity(sat);
          drdp = Medium.density_derp_h(state);
          drdh = Medium.density_derh_p(state);
          wl_in = w_in*(1-x_in);

          //Mass and energy balances
          V*(drdp*der(p) + drdh*der(h)) = w_in - wl_out - wv_out;
          V*((rho + h*drdh)*der(h) + (h*drdp - 1)*der(p)) = w_in*hl_in - wl_out*hl_out - wv_out*hv_out;
          rho = Medium.density(state);
          x = Medium.vapourQuality(state);
          M = V*rho;
          Ml = M*(1-x);
          Ml = A*zl*rho_l;
          vf =(H-zl)/H;

          // Boundary conditions
          hl_in = inStream(water_in.h_outflow);
          hv_out = if phase == 1 then h else hv;
          hl_out = if phase == 1 then h else hl;

          steam_out.h_outflow = hv_out;
          drained_out.h_outflow = hl_out;
          water_in.h_outflow = inStream(steam_out.h_outflow)
            "Dummy equation for flow reversal";

          w_in = water_in.m_flow;
          wl_out = -drained_out.m_flow;
          wv_out = -steam_out.m_flow;
          assert(w_in > -w_small, "Flow reversal not supported at phase separator inlet");
          assert(wl_out > -w_small, "Flow reversal not supported at phase separator liquid outlet");
          assert(wv_out > -w_small, "Flow reversal not supported at phase separator steam outlet");
          water_in.p = p;
          steam_out.p = p;
          drained_out.p = p;
          assert(vf>0,"Separator full");

          // Level sensor output
          level = zl;
          // Degree of superheating sensor
          DT_sh = max(0,T - T_sat);

        initial equation
          if initOpt == ThermoPower.Choices.Init.Options.noInit then
            // do nothing
          elseif initOpt == ThermoPower.Choices.Init.Options.steadyState then
            if not noInitialPressure then
              der(p) = 0;
            end if;
            der(h) = 0;
          elseif initOpt == ThermoPower.Choices.Init.Options.fixedState then
            if not noInitialPressure then
              p = p_start;
            end if;
            if not noInitialLevel then
              zl = zl_start;
            end if;
            h = h_start;
          else
            assert(false, "Unsupported initialisation option");
          end if;
          annotation (
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},{140,
                    140}}),
                    graphics),
            Icon(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},{140,140}}),
                 graphics={
                Rectangle(
                  extent={{-40,70},{40,-72}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,70},{-36,76},{-26,80},{-10,80},{-4,80},{2,80},{10,80},{28,
                      80},{36,76},{40,70},{18,70},{-40,70}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,-72},{-38,-78},{-34,-80},{-30,-80},{-26,-80},{-20,-80},{-12,
                      -80},{-6,-80},{32,-80},{38,-76},{40,-72},{-40,-72}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,0},{28,-60}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,60},{28,0}},
                  lineColor={85,170,255},
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,60},{-26,66},{-18,68},{18,68},{26,66},{28,60},{-22,58},{-28,
                      60}},
                  lineColor={85,170,255},
                  smooth=Smooth.None,
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,-60},{-26,-64},{-22,-68},{14,-68},{20,-68},{26,-64},{28,-60},
                      {-28,-60}},
                  lineColor={0,0,255},
                  smooth=Smooth.None,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-6,110},{6,80}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid)}),
            Documentation(info="<html>
<p>Modelling assumptions:</p>
<ul>
<li>The state can be either saturated liquid and saturated steam at thermodynamic equilibrium, or superheated steam.</li>
<li>Full liquid state not supported</li>
<li>No flow reversal at the ports</li>
<li>Reference zero level @ empty tank</li>
</ul>
</html>"));
        end Equilibrium2phVessel;

        model NusseltCondenser
          "Base model of condensing section based on Nusselt theory"

          replaceable package Medium = Modelica.Media.Water.WaterIF97_ph constrainedby
            Modelica.Media.Interfaces.PartialMedium "Medium"
            annotation (choicesAllMatching=true);

          ThermoPower.Water.FlangeA drained_in(redeclare package Medium = Medium,
              m_flow(min=if allowFlowReversal then -Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-104,-38},{-84,-18}},
                  rotation=0), iconTransformation(extent={{-60,30},{-40,50}})));
          ThermoPower.Water.FlangeB drained_out(redeclare package Medium = Medium,
              m_flow(max=if allowFlowReversal then +Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-10,-90},{10,-70}}, rotation=0),
                iconTransformation(extent={{-10,-100},{10,-80}})));
          ThermoPower.Water.FlangeA steam_in(redeclare package Medium = Medium, m_flow(
                max=if allowFlowReversal then +Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-10,82},{10,102}}, rotation=0),
                iconTransformation(extent={{-10,110},{10,130}})));
          Modelica.Blocks.Interfaces.RealOutput zl_SP annotation (Placement(
                transformation(extent={{-96,-14},{-76,6}}), iconTransformation(extent={{-40,-20},
                    {-60,0}})));
          Modelica.Blocks.Interfaces.RealOutput level annotation (Placement(transformation(
                  extent={{-118,-74},{-98,-54}}),
                                              iconTransformation(extent={{-40,-80},{-60,
                    -60}})));

          Medium.ThermodynamicState vapour_state
            "Thermodynamic state of the vapour";
          Medium.ThermodynamicState steamIn_state
            "Thermodynamic state of the inlet vapour";
          Medium.ThermodynamicState drainedIn_state
            "Thermodynamic state of the inlet drain";
          Medium.SaturationProperties sat "Saturation properties of the vapour";

          parameter Integer Nw "Number of volumes, tube side";
          parameter Modelica.SIunits.Volume V "Total volume";
          parameter Modelica.SIunits.Area A "Cross Sectional Area";
          parameter Modelica.SIunits.Height H=V/A
            "Height of the condensing volume";
          parameter Modelica.SIunits.Area S "Heat transfer surface";
          parameter Modelica.SIunits.CoefficientOfHeatTransfer gamma
            "Nusselt theory HTC";
          parameter Modelica.SIunits.Height zl_set "Liquid level set point";
          parameter Modelica.SIunits.Height zl_start "Liquid level start";
          parameter Modelica.SIunits.Pressure p_start "Medium start pressure";
          outer ThermoPower.System system "System wide properties";
          parameter Boolean allowFlowReversal=system.allowFlowReversal
            "= true to allow flow reversal, false restricts to design direction";
          parameter ThermoPower.Choices.Init.Options initOpt=system.initOpt
            "Initialisation option";
          parameter Boolean noInitialPressure=false
            "Remove initial equation on pressure"
            annotation (Dialog(tab="Initialisation"),choices(checkBox=true));

        public
          Medium.MassFlowRate ws_in "Steam mass flow rate";
          Medium.MassFlowRate wd_in "Drained inlet mass flow rate";
          Medium.MassFlowRate wd_out "Drained outet mass flow rate";
          Medium.MassFlowRate w_cond "Condensed vapour mass flow rate";
          Modelica.SIunits.Height zl(start = zl_start)
            "Medium level (relative to reference)";
          Medium.SpecificEnthalpy hs_in "Steam inlet specific enthalpy";
          Medium.SpecificEnthalpy hd_in "Drained inlet specific enthalpy";
          Medium.SpecificEnthalpy hd_out "Drained outlet specific enthalpy";
          Modelica.SIunits.Volume Vl "Volume occupied by liquid medium";
          Medium.Temperature T "Medium temperature";
          Medium.Temperature T_sat "Medium saturation temperature";
          Modelica.SIunits.Volume Vv "Volume occupied by vapour";
          Medium.AbsolutePressure p "Medium pressure";
          Modelica.SIunits.DerDensityByPressure drdp
            "Derivative of density by pressure";
          Modelica.SIunits.DerDensityByEnthalpy drdh
            "Derivative of density by enthalpy";
          Modelica.SIunits.SpecificEnthalpy h(start = hv)
            "Specific internal enthalpy";
          Modelica.SIunits.SpecificEnthalpy hv
            "Specific saturated steam enthalpy";
          Modelica.SIunits.Density rho_v "Steam density";
          Modelica.SIunits.Density rho_l "Liquid density";
          Real x_s "Vapour quality at steam inlet";
          Real x_d "Vapour quality at drained inlet";
          Modelica.SIunits.Power Q_cond "Condensation thermal load";
          Modelica.SIunits.Power Q_des_st
            "De-superheating power of steam inlet";
          Modelica.SIunits.Power Q_des_dr
            "De-superheating power of drained inlet";
          Modelica.SIunits.Power Qw[Nw]
            "Condensation thermal load on a single volume";
          Modelica.SIunits.Temperature Tw[Nw] "Wall temperatures";

          ThermoPower.Thermal.DHTVolumes DHT(
            N = Nw) annotation (Placement(
                transformation(extent={{40,-10},{60,10}}), iconTransformation(extent={{40,-20},
                    {50,20}})));
        equation
          // Fluid property calculations
          sat = Medium.setSat_p(p);
          vapour_state = Medium.setState_ph(p,h);
          steamIn_state = Medium.setState_ph(p,inStream(steam_in.h_outflow));
          drainedIn_state = Medium.setState_ph(p,inStream(drained_in.h_outflow));
          T = Medium.temperature(vapour_state);
          rho_v = Medium.density(vapour_state);
          hv = Medium.dewEnthalpy(sat);
          hd_out = Medium.bubbleEnthalpy(sat);
          T_sat = Medium.saturationTemperature(p);
          rho_l = Medium.bubbleDensity(sat);
          drdp = Medium.density_derp_h(vapour_state);
          drdh = Medium.density_derh_p(vapour_state);
          x_s = Medium.vapourQuality(steamIn_state);
          x_d = Medium.vapourQuality(drainedIn_state);

          //Mass and energy balances
          //Liquid mass balance
          rho_l*A*der(zl) = ws_in*(1-x_s) + wd_in*(1-x_d) + w_cond + wd_out;
          //Vapour mass balance
          Vv*(drdp*der(p) + drdh*der(h)) - rho_v*A*der(zl)= ws_in*x_s + wd_in*x_d - w_cond;
          //ws_in*x_s + wd_in*x_d - w_cond = 0;
          Vl = A*zl;
          Vv = V - Vl;
          //Vapour energy balance
          //Vv*(rho_v + h*drdh)*der(h) + Vv*(h*drdp - 1)*der(p) + A*(p-rho_v*h)*der(zl)= ws_in*(hs_in-hd_out) + wd_in*(hd_in-hd_out) - Q_cond;
          ws_in*(hs_in-hd_out) + wd_in*(hd_in-hd_out) - Q_cond = 0;
          Q_des_st = max(0,ws_in*(hs_in - hv));
          Q_des_dr = max(0,wd_in*(hd_in - hv));
          for j in 1:Nw loop
            Qw[j] = gamma*S/Nw*(T_sat - Tw[j]);
          end for;
          Q_cond = sum(Qw) + Q_des_st + Q_des_dr;
          w_cond = Q_cond/(h - hd_out);

          // Boundary conditions
          hs_in = homotopy(if not allowFlowReversal then inStream(steam_in.h_outflow)
             else if ws_in >= 0 then inStream(steam_in.h_outflow) else h, inStream(
            steam_in.h_outflow));
          hd_in = homotopy(if not allowFlowReversal then inStream(drained_in.h_outflow)
             else if wd_in >= 0 then inStream(drained_in.h_outflow) else h, inStream(
            drained_in.h_outflow));
          drained_out.h_outflow = hd_out;
          steam_in.h_outflow = inStream(drained_out.h_outflow);
          drained_in.h_outflow = inStream(drained_out.h_outflow);

          ws_in = steam_in.m_flow;
          wd_in = drained_in.m_flow;
          wd_out = drained_out.m_flow;
          steam_in.p = p;
          drained_in.p = p;
          drained_out.p = p;

          Tw = DHT.T;
          Qw = -DHT.Q;

          // Level set-point definition
          zl_SP = zl_set;
          level = zl;

        initial equation
          if initOpt == ThermoPower.Choices.Init.Options.noInit then
            // do nothing
          elseif initOpt == ThermoPower.Choices.Init.Options.steadyState then
            if not noInitialPressure then
              der(p) = 0;
            end if;
            der(zl) = 0;
            der(h) = 0;
          elseif initOpt == ThermoPower.Choices.Init.Options.fixedState then
            if not noInitialPressure then
              p = p_start;
            end if;
            h = hv;
            zl = zl_start;
          else
            assert(false, "Unsupported initialisation option");
          end if;
          annotation (
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},{140,
                    140}})),
            Icon(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},
                    {140,140}}),
                 graphics={
                Rectangle(
                  extent={{-40,70},{40,-72}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,70},{-36,76},{-26,80},{-10,80},{-4,80},{2,80},{10,80},{28,
                      80},{36,76},{40,70},{18,70},{-40,70}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,-72},{-38,-78},{-34,-80},{-30,-80},{-26,-80},{-20,-80},{-12,
                      -80},{-6,-80},{32,-80},{38,-76},{40,-72},{-40,-72}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,0},{28,-60}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,60},{28,0}},
                  lineColor={85,170,255},
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,60},{-26,66},{-18,68},{18,68},{26,66},{28,60},{-22,58},{-28,
                      60}},
                  lineColor={85,170,255},
                  smooth=Smooth.None,
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,-60},{-26,-64},{-22,-68},{14,-68},{20,-68},{26,-64},{28,-60},
                      {-28,-60}},
                  lineColor={0,0,255},
                  smooth=Smooth.None,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-6,110},{6,80}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid)}),
            Documentation(info="<HTML></html>"));
        end NusseltCondenser;

        model NusseltCondenser_noLevel
          "Base model of condensing section based on Nusselt theory"

          replaceable package Medium = Modelica.Media.Water.WaterIF97_ph constrainedby
            Modelica.Media.Interfaces.PartialMedium "Medium"
            annotation (choicesAllMatching=true);

          ThermoPower.Water.FlangeA drained_in(redeclare package Medium = Medium,
              m_flow(min=if allowFlowReversal then -Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-104,-38},{-84,-18}},
                  rotation=0), iconTransformation(extent={{-60,30},{-40,50}})));
          ThermoPower.Water.FlangeB drained_out(redeclare package Medium = Medium,
              m_flow(max=if allowFlowReversal then +Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-10,-90},{10,-70}}, rotation=0),
                iconTransformation(extent={{-10,-100},{10,-80}})));
          ThermoPower.Water.FlangeA steam_in(redeclare package Medium = Medium, m_flow(
                max=if allowFlowReversal then +Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-10,82},{10,102}}, rotation=0),
                iconTransformation(extent={{-10,110},{10,130}})));

          Medium.ThermodynamicState vapour_state
            "Thermodynamic state of the vapour";
          Medium.ThermodynamicState steamIn_state
            "Thermodynamic state of the inlet vapour";
          Medium.ThermodynamicState drainedIn_state
            "Thermodynamic state of the inlet drain";
          Medium.SaturationProperties sat "Saturation properties of the vapour";

          parameter Integer Nw "Number of volumes, tube side";
          parameter Modelica.SIunits.Volume V "Total volume";
          parameter Modelica.SIunits.Area A "Cross Sectional Area";
          parameter Modelica.SIunits.Height H=V/A
            "Height of the condensing volume";
          parameter Modelica.SIunits.Area S "Heat transfer surface";
          parameter Modelica.SIunits.CoefficientOfHeatTransfer gamma
            "Nusselt theory HTC";
          parameter Modelica.SIunits.Height zl
            "Liquid level (ideally controlled)";
          parameter Modelica.SIunits.Pressure p_start "Medium start pressure";
          outer ThermoPower.System system "System wide properties";
          parameter Boolean allowFlowReversal=system.allowFlowReversal
            "= true to allow flow reversal, false restricts to design direction";
          parameter ThermoPower.Choices.Init.Options initOpt=ThermoPower.Choices.Init.Options.noInit
            "Initialisation option";
          parameter Boolean noInitialPressure=false
            "Remove initial equation on pressure"
            annotation (Dialog(tab="Initialisation"),choices(checkBox=true));

        public
          Medium.MassFlowRate ws_in "Steam mass flow rate";
          Medium.MassFlowRate wd_in "Drained inlet mass flow rate";
          Medium.MassFlowRate wd_out "Drained outet mass flow rate";
          Medium.MassFlowRate w_cond "Condensed vapour mass flow rate";
          Medium.SpecificEnthalpy hs_in "Steam inlet specific enthalpy";
          Medium.SpecificEnthalpy hd_in "Drained inlet specific enthalpy";
          Medium.SpecificEnthalpy hd_out "Drained outlet specific enthalpy";
          Modelica.SIunits.Volume Vl "Volume occupied by liquid medium";
          Medium.Temperature T "Medium temperature";
          Medium.Temperature T_sat "Medium saturation temperature";
          Modelica.SIunits.Volume Vv "Volume occupied by vapour";
          Medium.AbsolutePressure p "Medium pressure";
          Modelica.SIunits.DerDensityByPressure drdp
            "Derivative of density by pressure";
          Modelica.SIunits.DerDensityByEnthalpy drdh
            "Derivative of density by enthalpy";
          Modelica.SIunits.SpecificEnthalpy h(start = hv)
            "Specific internal enthalpy";
          Modelica.SIunits.SpecificEnthalpy hv
            "Specific saturated steam enthalpy";
          Modelica.SIunits.Density rho_v "Steam density";
          Modelica.SIunits.Density rho_l "Liquid density";
          Real x_s "Vapour quality at steam inlet";
          Real x_d "Vapour quality at drained inlet";
          Modelica.SIunits.Power Q_cond "Condensation thermal load";
          Modelica.SIunits.Power Q_des_st
            "De-superheating power of steam inlet";
          Modelica.SIunits.Power Q_des_dr
            "De-superheating power of drained inlet";
          Modelica.SIunits.Power Qw[Nw]
            "Condensation thermal load on a single volume";
          Modelica.SIunits.Temperature Tw[Nw] "Wall temperatures";

          ThermoPower.Thermal.DHTVolumes DHT(
            N = Nw) annotation (Placement(
                transformation(extent={{40,-10},{60,10}}), iconTransformation(extent={{40,-20},
                    {50,20}})));
        equation
          // Fluid property calculations
          sat = Medium.setSat_p(p);
          vapour_state = Medium.setState_ph(p,h);
          steamIn_state = Medium.setState_ph(p,inStream(steam_in.h_outflow));
          drainedIn_state = Medium.setState_ph(p,inStream(drained_in.h_outflow));
          T = Medium.temperature(vapour_state);
          rho_v = Medium.density(vapour_state);
          hv = Medium.dewEnthalpy(sat);
          hd_out = Medium.bubbleEnthalpy(sat);
          T_sat = Medium.saturationTemperature(p);
          rho_l = Medium.bubbleDensity(sat);
          drdp = Medium.density_derp_h(vapour_state);
          drdh = Medium.density_derh_p(vapour_state);
          x_s = Medium.vapourQuality(steamIn_state);
          x_d = Medium.vapourQuality(drainedIn_state);

          //Mass and energy balances
          //Liquid mass balance
          //w_cond + ws_in*(1-x_s) + wd_in*(1-x_d) + wd_out = 0;
         // wd_out = theta_valve*Kv*
          //Vapour mass balance
          Vv*(drdp*der(p) + drdh*der(h))= ws_in*x_s + wd_in*x_d - w_cond;
          //ws_in*x_s + wd_in*x_d - w_cond = 0;
          Vl = A*zl;
          Vv = V - Vl;
          //Vapour energy balance
          //Vv*(rho_v + h*drdh)*der(h) + Vv*(h*drdp - 1)*der(p)= ws_in*(hs_in-hd_out) + wd_in*(hd_in-hd_out) - Q_cond;
          ws_in*(hs_in-hd_out) + wd_in*(hd_in-hd_out) - Q_cond = 0;
          Q_des_st = max(0,ws_in*(hs_in - hv));
          Q_des_dr = max(0,wd_in*(hd_in - hv));
          for j in 1:Nw loop
            Qw[j] = gamma*S/Nw*(T_sat - Tw[j]);
          end for;
          Q_cond = sum(Qw) + Q_des_st + Q_des_dr;
          w_cond = Q_cond/(h - hd_out);

          // Boundary conditions
          hs_in = homotopy(if not allowFlowReversal then inStream(steam_in.h_outflow)
             else if ws_in >= 0 then inStream(steam_in.h_outflow) else h, inStream(
            steam_in.h_outflow));
          hd_in = homotopy(if not allowFlowReversal then inStream(drained_in.h_outflow)
             else if wd_in >= 0 then inStream(drained_in.h_outflow) else h, inStream(
            drained_in.h_outflow));
          drained_out.h_outflow = hd_out;
          steam_in.h_outflow = inStream(drained_out.h_outflow);
          drained_in.h_outflow = inStream(drained_out.h_outflow);

          ws_in = steam_in.m_flow;
          wd_in = drained_in.m_flow;
          wd_out = drained_out.m_flow;
          steam_in.p = p;
          drained_in.p = p;
          drained_out.p = p;

          Tw = DHT.T;
          Qw = -DHT.Q;

        initial equation
          if initOpt == ThermoPower.Choices.Init.Options.noInit then
            // do nothing
          elseif initOpt == ThermoPower.Choices.Init.Options.steadyState then
            if not noInitialPressure then
              der(p) = 0;
            end if;
            der(h) = 0;
          elseif initOpt == ThermoPower.Choices.Init.Options.fixedState then
            if not noInitialPressure then
              p = p_start;
            end if;
            h = hv;
          else
            assert(false, "Unsupported initialisation option");
          end if;
          annotation (
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},{140,
                    140}})),
            Icon(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},
                    {140,140}}),
                 graphics={
                Rectangle(
                  extent={{-40,70},{40,-72}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,70},{-36,76},{-26,80},{-10,80},{-4,80},{2,80},{10,80},{28,
                      80},{36,76},{40,70},{18,70},{-40,70}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,-72},{-38,-78},{-34,-80},{-30,-80},{-26,-80},{-20,-80},{-12,
                      -80},{-6,-80},{32,-80},{38,-76},{40,-72},{-40,-72}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,0},{28,-60}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,60},{28,0}},
                  lineColor={85,170,255},
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,60},{-26,66},{-18,68},{18,68},{26,66},{28,60},{-22,58},{-28,
                      60}},
                  lineColor={85,170,255},
                  smooth=Smooth.None,
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,-60},{-26,-64},{-22,-68},{14,-68},{20,-68},{26,-64},{28,-60},
                      {-28,-60}},
                  lineColor={0,0,255},
                  smooth=Smooth.None,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-6,110},{6,80}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid)}),
            Documentation(info="<HTML></html>"));
        end NusseltCondenser_noLevel;

        model Separator_controlledLevel
          "Model of a water/steam separator with control level "

          replaceable package Medium = Modelica.Media.Water.WaterIF97_ph annotation (choicesAllMatching=true);

          parameter Modelica.SIunits.Length D "tank diameter";
          final parameter Modelica.SIunits.Area A=D^2/4*Modelica.Constants.pi
            "Cross Sectional Area";
          parameter Modelica.SIunits.Height H "Height of the tank";
          parameter Modelica.SIunits.Height zl_start "Medium start level";
          parameter Modelica.SIunits.Pressure p_start "Pressure start level";
          parameter Medium.SpecificEnthalpy h_start "Start specific enthalpy";
          parameter Real Kp "Controller proportional gain" annotation (Dialog(tab="Level controller"));
          parameter Modelica.SIunits.Time Ti "Controller integral time" annotation (Dialog(tab="Level controller"));
          parameter Modelica.SIunits.Time Td "Controller derivative time" annotation (Dialog(tab="Level controller"));
          parameter Real Nd "Controller derivative action up to Nd/Td rad/s" annotation (Dialog(tab="Level controller"));
          parameter Real PVmin "Minimum value of process variable for scaling" annotation (Dialog(tab="Level controller"));
          parameter Real PVmax "Maximum value of process variable for scaling" annotation (Dialog(tab="Level controller"));
          parameter Real CSmin "Minimum value of control signal for scaling" annotation (Dialog(tab="Level controller"));
          parameter Real CSmax "Maximum value of control signal for scaling" annotation (Dialog(tab="Level controller"));
          parameter Real PVstart=0.5 "Start value of PV (scaled)" annotation (Dialog(tab="Level controller"));
          parameter Real CSstart=0.5 "Start value of CS (scaled)" annotation (Dialog(tab="Level controller"));
          parameter Real actuatorGain=0.5 "Actuator gain" annotation (Dialog(tab="Level controller"));
          parameter Modelica.SIunits.Time Ta "Actuator time constant" annotation (Dialog(tab="Level controller"));

          ThermoPower.PowerPlants.Control.PID PID_level(
          Kp=Kp,
          Ti=Ti,
          Td=Td,
          Nd=Nd,
          PVmin=PVmin,
          PVmax=PVmax,
          CSmin=CSmin,
          CSmax=CSmax,
          CSstart=CSstart,
          PVstart=PVstart) annotation (
              Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={50,-10})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = Medium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,-64})));
          Equilibrium2phVessel             vessel(
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            D=D,
            H=H,
            zl_start=zl_start,
            h_start=h_start,
            p_start=p_start)
            annotation (Placement(transformation(extent={{-58,-18},{58,88}})));
          ThermoPower.Water.FlangeA inlet(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-120,-10},{-100,10}}),
                iconTransformation(extent={{-60,-50},{-40,-30}})));
          ThermoPower.Water.FlangeB outlet_vap(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-10,100},{10,120}})));
          ThermoPower.Water.FlangeB outlet_liq(
            redeclare package Medium = Medium)
            annotation (Placement(transformation(extent={{-10,-120},{10,-100}}),
                iconTransformation(extent={{-10,-100},{10,-80}})));
          Modelica.Blocks.Continuous.FirstOrder pumpDynamic(
            k=actuatorGain,
            T=Ta) annotation (Placement(
                transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={50,-40})));
          Modelica.Blocks.Sources.Constant zl_SP(k=4)
            annotation (Placement(transformation(extent={{80,20},{60,40}})));
        equation
          connect(vessel.drained_out, waterPump.inlet) annotation (Line(
              points={{7.10543e-015,0.928571},{7.10543e-015,-40},{1.77636e-015,
                  -40},{1.77636e-015,-54}},
              color={0,0,255},
              smooth=Smooth.None));

          connect(inlet, inlet) annotation (Line(
              points={{-110,0},{-110,0}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(inlet, vessel.water_in) annotation (Line(
              points={{-110,0},{-60,0},{-60,19.8571},{-20.7143,19.8571}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(waterPump.outlet, outlet_liq) annotation (Line(
              points={{-1.77636e-015,-74},{-1.77636e-015,-92},{0,-92},{0,-110}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(vessel.steam_out, outlet_vap) annotation (Line(
              points={{7.10543e-015,80.4286},{7.10543e-015,82},{0,82},{0,110}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(vessel.level, PID_level.PV) annotation (Line(
              points={{20.7143,8.5},{46,8.5},{46,0}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(PID_level.CS, pumpDynamic.u) annotation (Line(
              points={{50,-20},{50,-28}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(pumpDynamic.y, waterPump.in_w0) annotation (Line(
              points={{50,-51},{50,-60},{6,-60}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(zl_SP.y, PID_level.SP) annotation (Line(
              points={{59,30},{54,30},{54,0}},
              color={0,0,127},
              smooth=Smooth.None));
          annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics),
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},{140,
                    140}}),
                    graphics),
            Icon(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},
                    {140,140}}),
                 graphics={
                Rectangle(
                  extent={{-40,70},{40,-72}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,70},{-36,76},{-26,80},{-10,80},{-4,80},{2,80},{10,80},{28,
                      80},{36,76},{40,70},{18,70},{-40,70}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-40,-72},{-38,-78},{-34,-80},{-30,-80},{-26,-80},{-20,-80},{-12,
                      -80},{-6,-80},{32,-80},{38,-76},{40,-72},{-40,-72}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,0},{28,-60}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,60},{28,0}},
                  lineColor={85,170,255},
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,60},{-26,66},{-18,68},{18,68},{26,66},{28,60},{-22,58},{-28,
                      60}},
                  lineColor={85,170,255},
                  smooth=Smooth.None,
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,-60},{-26,-64},{-22,-68},{14,-68},{20,-68},{26,-64},{28,-60},
                      {-28,-60}},
                  lineColor={0,0,255},
                  smooth=Smooth.None,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-6,110},{6,80}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid)}),
            Documentation(info="<HTML></html>"));
        end Separator_controlledLevel;

        model Deaerator "Base model of an adiabatic deaerator"

          replaceable package Medium = Modelica.Media.Water.WaterIF97_ph constrainedby
            Modelica.Media.Interfaces.PartialMedium "Medium"
            annotation (choicesAllMatching=true);

          final parameter Modelica.SIunits.Volume V=A*H "Total volume";
          final parameter Modelica.SIunits.Area A=L*H "Cross Sectional Area";
          parameter Modelica.SIunits.Length L "tank length";
          parameter Modelica.SIunits.Height H "Height of the tank (squared)";
          parameter Modelica.SIunits.Height zl_start "Medium start level";
          parameter Modelica.SIunits.Pressure p_start "Pressure start level";
          parameter Medium.SpecificEnthalpy h_start "Start specific enthalpy";
          parameter Medium.MassFlowRate w_small = 1e-3
            "Small flow rate for reverse flow check";
          outer ThermoPower.System system "System wide properties";
          parameter Boolean allowFlowReversal=system.allowFlowReversal
            "= true to allow flow reversal, false restricts to design direction";
          parameter ThermoPower.Choices.Init.Options initOpt=ThermoPower.Choices.Init.Options.noInit
            "Initialisation option";
          parameter Boolean noInitialPressure=false
            "Remove initial equation on pressure"
            annotation (Dialog(tab="Initialisation"),choices(checkBox=true));

          ThermoPower.Water.FlangeA water_in(redeclare package Medium = Medium, m_flow(
                min=if allowFlowReversal then -Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-104,-38},{-84,-18}},
                  rotation=0), iconTransformation(extent={{26,70},{46,90}})));
          ThermoPower.Water.FlangeB drained_out(redeclare package Medium = Medium,
              m_flow(max=if allowFlowReversal then +Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-10,-90},{10,-70}}, rotation=0),
                iconTransformation(extent={{-10,-120},{10,-100}})));
          ThermoPower.Water.FlangeA drained_in(redeclare package Medium = Medium,
              m_flow(min=if allowFlowReversal then -Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-104,-38},{-84,-18}},
                  rotation=0), iconTransformation(extent={{-140,-80},{-120,-60}})));
          ThermoPower.Water.FlangeA steam_in(redeclare package Medium = Medium, m_flow(
                min=if allowFlowReversal then -Modelica.Constants.inf else 0))
            annotation (Placement(transformation(extent={{-104,-38},{-84,-18}},
                  rotation=0), iconTransformation(extent={{-48,10},{-28,30}})));
          Modelica.Blocks.Interfaces.RealOutput level annotation (Placement(transformation(
                  extent={{40,-80},{60,-60}}),iconTransformation(extent={{120,-90},{140,
                    -70}})));

          Medium.ThermodynamicState state
            "Thermodynamic state of the deaerator";
          Medium.SaturationProperties sat
            "Saturation properties of the deaerator";

        protected
          constant Modelica.SIunits.Acceleration g=Modelica.Constants.g_n;
        public
          Medium.MassFlowRate wl_in "Medium inflow mass flow rate";
          Medium.MassFlowRate wl_out "Medium drained outflow mass flow rate";
          Medium.MassFlowRate ws_in "Medium steam inlet mass flow rate";
          Medium.MassFlowRate wd_in "Medium drained inlet mass flow rate";
          Modelica.SIunits.Height zl(start=zl_start)
            "Medium level (relative to reference)";
          Medium.SpecificEnthalpy hl_in "Medium inlet specific enthalpy";
          Medium.SpecificEnthalpy hl_out
            "Medium drained outlet specific enthalpy";
          Medium.SpecificEnthalpy hs_in "Medium steam inlet specific enthalpy";
          Medium.SpecificEnthalpy hd_in
            "Medium drained inlet specific enthalpy";
          Medium.SpecificEnthalpy hl "Medium internal specific enthalpy";
          Medium.SpecificEnthalpy hv "Medium internal specific enthalpy";
          Modelica.SIunits.Mass M "Total mass";
          Modelica.SIunits.Mass Ml "Mass of liquid";
          Medium.Temperature T "Medium temperature";
          Medium.Temperature T_sat "Medium saturation temperature";
          Modelica.SIunits.Volume Vv "Volume occupied by vapour";
          Modelica.SIunits.Volume Vl "Volume occupied by liquid medium";
          ThermoPower.Units.LiquidDensity rho_l "Density of the liquid medium";
          Modelica.SIunits.DerDensityByPressure drdp
            "Derivative of density by pressure";
          Modelica.SIunits.DerDensityByEnthalpy drdh
            "Derivative of density by enthalpy";
          Modelica.SIunits.SpecificEnthalpy h(start = h_start, stateSelect=StateSelect.prefer)
            "Specific internal enthalpy";
          Medium.AbsolutePressure p( start = p_start, stateSelect=StateSelect.prefer)
            "Medium pressure";
          Modelica.SIunits.Density rho "Density";
          Modelica.SIunits.MassFraction  x "Mixture steam quality";
          Real vf "Void fraction";
          Integer phase(start = 2)
            "Definition of thermodynamic state in the separator";

        equation
          // Fluid property calculations
          sat = Medium.setSat_p(p);
          phase = if h >= hv then 1 else 2;
          state = Medium.setState_ph(p,h,phase);
          T = Medium.temperature(state);
          T_sat = Medium.saturationTemperature(p);
          hv = Medium.dewEnthalpy(sat);
          hl = Medium.bubbleEnthalpy(sat);
          rho_l = Medium.bubbleDensity(sat);
          drdp = Medium.density_derp_h(state);
          drdh = Medium.density_derh_p(state);

          //Mass and energy balances
          V*(drdp*der(p) + drdh*der(h)) = wl_in - wl_out + ws_in + wd_in;
          V*((rho + h*drdh)*der(h) + (h*drdp - 1)*der(p)) = wl_in*hl_in - wl_out*hl_out + ws_in*hs_in + wd_in*hd_in;
          rho = Medium.density(state);
          x = Medium.vapourQuality(state);
          M = V*rho;
          Ml = M*(1-x);
          Ml = A*zl*rho_l;
          Vl = A*zl;
          Vv = V - Vl;
          vf = (H-zl)/H;

          // Boundary conditions
          hl_in = inStream(water_in.h_outflow);
          hs_in = inStream(steam_in.h_outflow);
          hd_in = inStream(drained_in.h_outflow);
          hl_out = if phase == 1 then h else hl;

          steam_in.h_outflow = inStream(drained_out.h_outflow)
            "Dummy equation for flow reversal";
          drained_in.h_outflow = inStream(drained_out.h_outflow)
            "Dummy equation for flow reversal";
          drained_out.h_outflow = hl_out;
          water_in.h_outflow = inStream(drained_out.h_outflow)
            "Dummy equation for flow reversal";

          wl_in = water_in.m_flow;
          wl_out = -drained_out.m_flow;
          ws_in = steam_in.m_flow;
          wd_in = drained_in.m_flow;
          assert(wl_in > -w_small, "Flow reversal not supported at deaerator inlet");
          assert(wl_out > -w_small, "Flow reversal not supported at deaerator liquid outlet");
          assert(ws_in > -w_small, "Flow reversal not supported at deaerator steam outlet");
          assert(wd_in > -w_small, "Flow reversal not supported at deaerator steam outlet");
          water_in.p = p;
          steam_in.p = p;
          drained_out.p = p;
          drained_in.p = p;
          assert(vf>0,"Separator full");

          // Level set-point definition
          level = zl;

        initial equation
          if initOpt == ThermoPower.Choices.Init.Options.noInit then
            // do nothing
          elseif initOpt == ThermoPower.Choices.Init.Options.steadyState then
            if not noInitialPressure then
              der(p) = 0;
            end if;
            der(h) = 0;
          elseif initOpt == ThermoPower.Choices.Init.Options.fixedState then
            if not noInitialPressure then
              p = p_start;
            end if;
            h = h_start;
          else
            assert(false, "Unsupported initialisation option");
          end if;
          annotation (
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},{140,
                    140}}),
                    graphics),
            Icon(coordinateSystem(preserveAspectRatio=false, extent={{-140,-140},
                    {140,140}}),
                 graphics={
                Rectangle(
                  extent={{-120,-10},{120,-92}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-120,-10},{-116,-4},{-106,0},{-90,0},{-84,0},{-78,0},{-70,0},
                      {110,0},{116,-4},{120,-10},{-62,-10},{-120,-10}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-120,-92},{-118,-98},{-114,-100},{-110,-100},{-106,-100},{-100,
                      -100},{-92,-100},{-86,-100},{114,-100},{118,-96},{120,-92},{-120,-92}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-112,-34},{112,-88}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-112,-16},{112,-34}},
                  lineColor={85,170,255},
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-112,-16},{-110,-10},{-102,-8},{104,-8},{110,-10},{112,-16},{
                      -106,-16},{-112,-16}},
                  lineColor={85,170,255},
                  smooth=Smooth.None,
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-112,-88},{-110,-92},{-104,-94},{0,-94},{106,-94},{110,-92},{
                      112,-88},{-112,-88}},
                  lineColor={0,0,255},
                  smooth=Smooth.None,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-28,90},{26,0}},
                  lineColor={135,135,135},
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-28,90},{-26,96},{-20,100},{2,100},{8,100},{14,100},{16,100},
                      {22,98},{24,96},{26,90},{26,90},{-28,90}},
                  lineColor={135,135,135},
                  smooth=Smooth.None,
                  fillColor={135,135,135},
                  fillPattern=FillPattern.Solid),
                Polygon(
                  points={{-20,84},{-20,90},{-16,92},{14,92},{18,90},{18,84},{-14,84},{-20,
                      84}},
                  lineColor={85,170,255},
                  smooth=Smooth.None,
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-20,84},{18,-10}},
                  lineColor={85,170,255},
                  fillColor={85,170,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-12,76},{18,74}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-20,64},{10,62}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-20,40},{10,38}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-12,52},{18,50}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-12,28},{18,26}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-20,16},{10,14}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid),
                Rectangle(
                  extent={{-12,4},{18,2}},
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid)}),
            Documentation(info="<HTML></html>"));
        end Deaerator;

        package Test

          model test_2ph_vessel_dryout
            "The vessel starts with some liquid, which is blown away by incoming steam"

            ThermoPower.Water.SourceMassFlow sourceMassFlow(
              w0=95.31,
              h=2872800,
              use_in_h=true,
              use_in_w0=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
              annotation (Placement(transformation(extent={{-40,-10},{-20,10}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
              annotation (Placement(transformation(extent={{100,100},{120,120}})));
            Modelica.Blocks.Sources.Ramp ramp(
              duration=500,
              offset=2872800,
              startTime=1000,
              height=-0*(2872800 - 2272800))
              annotation (Placement(transformation(extent={{-80,30},{-60,50}})));
            Equilibrium2phVessel             vessel(
              D=0.56,
              H=8,
              zl_start=0,
              h_start=2e6,
              phase(start=1),
              initOpt=ThermoPower.Choices.Init.Options.fixedState,
              noInitialPressure=true,
              p_start=17710000)
              annotation (Placement(transformation(extent={{-18,-18},{38,32}})));
            ThermoPower.Water.SinkPressure sinkMassFlow(redeclare package
                Medium =
                  Modelica.Media.Water.WaterIF97_ph, p0=17700000)
              annotation (Placement(transformation(extent={{20,50},{40,70}})));
          equation
            connect(ramp.y, sourceMassFlow.in_h) annotation (Line(
                points={{-59,40},{-26,40},{-26,6}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(vessel.water_in, sourceMassFlow.flange) annotation (Line(
                points={{0,-0.142857},{0,0},{-20,0}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(vessel.steam_out, sinkMassFlow.flange) annotation (Line(
                points={{10,28.4286},{10,60},{20,60}},
                color={0,0,255},
                smooth=Smooth.None));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-120,
                      -120},{120,120}}),        graphics), Icon(coordinateSystem(extent={{
                      -120,-120},{120,120}})),
              experiment(StopTime=5000),
              __Dymola_experimentSetupOutput);
          end test_2ph_vessel_dryout;

          model test_2ph_vessel_filling
            "The vessel starts completely dry, then is filled by incoming wet steam"

            ThermoPower.Water.SourceMassFlow sourceMassFlow(
              w0=95.31,
              h=2872800,
              use_in_h=true,
              use_in_w0=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
              annotation (Placement(transformation(extent={{-40,-10},{-20,10}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
              annotation (Placement(transformation(extent={{100,100},{120,120}})));
            Modelica.Blocks.Sources.Ramp ramp(
              offset=2872800,
              height=-(2872800 - 2272800),
              duration=100,
              startTime=5)
              annotation (Placement(transformation(extent={{-80,30},{-60,50}})));
            Equilibrium2phVessel             vessel(
              D=0.56,
              H=8,
              zl_start=0,
              h_start=2e6,
              phase(start=1),
              noInitialPressure=true,
              p_start=17710000,
              initOpt=ThermoPower.Choices.Init.Options.steadyState)
              annotation (Placement(transformation(extent={{-18,-18},{38,32}})));
            ThermoPower.Water.SinkPressure sinkMassFlow(redeclare package
                Medium =
                  Modelica.Media.Water.WaterIF97_ph, p0=17700000)
              annotation (Placement(transformation(extent={{20,50},{40,70}})));
          equation
            connect(ramp.y, sourceMassFlow.in_h) annotation (Line(
                points={{-59,40},{-26,40},{-26,6}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(vessel.water_in, sourceMassFlow.flange) annotation (Line(
                points={{0,-0.142857},{0,0},{-20,0}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(vessel.steam_out, sinkMassFlow.flange) annotation (Line(
                points={{10,28.4286},{10,60},{20,60}},
                color={0,0,255},
                smooth=Smooth.None));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-120,
                      -120},{120,120}}),        graphics), Icon(coordinateSystem(extent={{
                      -120,-120},{120,120}})),
              experiment(StopTime=200),
              __Dymola_experimentSetupOutput);
          end test_2ph_vessel_filling;

          model test_NusseltCondenser_HPH2
            "Test of the condensing section of the last high pressure preheater"

            NusseltCondenser baseNusseltCondenser(
              zl_set=0.1,
              zl_start=0.1,
              V=2.99*0.8,
              A=1,
              S=210*0.8,
              noInitialPressure=true,
              initOpt=ThermoPower.Choices.Init.Options.steadyState,
              gamma=8000,
              Nw=5,
              p_start=8150000)
              annotation (Placement(transformation(extent={{-28,12},{28,68}})));
            inner ThermoPower.System system
              annotation (Placement(transformation(extent={{80,80},{100,100}})));
            ThermoPower.Water.SourceMassFlow sourceMassFlow(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=540000,
              w0=0)
              annotation (Placement(transformation(extent={{-62,36},{-42,56}})));
            ThermoPower.Water.SourcePressure sourceMassFlow1(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              p0=8150000,
              h=2760e3)
              annotation (Placement(transformation(extent={{-40,70},{-20,90}})));
            ThermoPower.Water.ValveLin valveDrain(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=1.095752222e-6)
              annotation (Placement(transformation(
                  extent={{-10,10},{10,-10}},
                  rotation=-90,
                  origin={0,-38})));
            ThermoPower.PowerPlants.Control.PID PID_level(
              PVmin=0,
              CSmin=0,
              Ti=15,
              Kp=-20,
              steadyStateInit=true,
              PVmax=1,
              CSmax=1,
              PVstart=0.1,
              CSstart=0.6) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={-40,-10})));
            ThermoPower.Water.SinkPressure sinkPressure(redeclare package
                Medium =
                  Modelica.Media.Water.WaterIF97_ph, p0=2037000)
              annotation (Placement(transformation(extent={{20,-70},{40,-50}})));
            ThermoPower.Thermal.TempSource1DlinFV tempSource1DlinFV(Nw=5)
              annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=270,
                  origin={34,40})));
            Modelica.Blocks.Sources.Constant T_in(k=563.15) annotation (Placement(
                  transformation(
                  extent={{10,-10},{-10,10}},
                  rotation=0,
                  origin={70,60})));
            Modelica.Blocks.Sources.Constant T_out(k=567.15) annotation (
                Placement(transformation(
                  extent={{10,-10},{-10,10}},
                  rotation=0,
                  origin={70,20})));
          equation
            connect(sourceMassFlow1.flange, baseNusseltCondenser.steam_in)
              annotation (Line(
                points={{-20,80},{0,80},{0,64}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sourceMassFlow.flange, baseNusseltCondenser.drained_in)
              annotation (Line(
                points={{-42,46},{-26,46},{-26,48},{-10,48}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(baseNusseltCondenser.level, PID_level.PV) annotation (Line(
                points={{-10,26},{-44,26},{-44,0}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(baseNusseltCondenser.zl_SP, PID_level.SP) annotation (Line(
                points={{-10,38},{-36,38},{-36,0}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(PID_level.CS, valveDrain.cmd) annotation (Line(points={{-40,
                    -20},{-40,-20},{-40,-38},{-8,-38}}, color={0,0,127}));
            connect(valveDrain.outlet, sinkPressure.flange) annotation (Line(
                  points={{0,-48},{0,-60},{20,-60}}, color={0,0,255}));
            connect(baseNusseltCondenser.drained_out, valveDrain.inlet)
              annotation (Line(points={{0,22},{0,22},{0,-28}}, color={0,0,255}));
            connect(tempSource1DlinFV.wall, baseNusseltCondenser.DHT) annotation (
               Line(points={{31,40},{20,40},{9,40}}, color={255,127,0}));
            connect(tempSource1DlinFV.temperature_1, T_in.y) annotation (Line(
                  points={{37,44},{50,44},{50,60},{59,60}}, color={0,0,127}));
            connect(tempSource1DlinFV.temperature_Nw, T_out.y) annotation (Line(
                  points={{36.8,36},{50,36},{50,20},{59,20}}, color={0,0,127}));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                      -100},{100,100}})));
          end test_NusseltCondenser_HPH2;

          model test_Separator
            "Controlled separator test, with filling and dryout procedure"

            ThermoPower.Water.SinkPressure sinkPressure(redeclare package
                Medium =
                  Modelica.Media.Water.WaterIF97_ph, p0=17700000)
              annotation (Placement(transformation(extent={{20,50},{40,70}})));
            ThermoPower.Water.SourceMassFlow sourceMassFlow(
              w0=95.31,
              h=2872800,
              use_in_h=true,
              use_in_w0=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
              annotation (Placement(transformation(extent={{-40,-10},{-20,10}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
              annotation (Placement(transformation(extent={{100,100},{120,120}})));
            Modelica.Blocks.Sources.Ramp ramp(
              duration=500,
              startTime=10,
              offset=2772800,
              height=-9e5)
              annotation (Placement(transformation(extent={{-100,30},{-80,50}})));
            Separator_controlledLevel base_2ph_vessel(
              Kp=-250,
              Ti=20,
              Td=0,
              Nd=1,
              PVmin=0,
              PVmax=20,
              CSmin=0,
              CSmax=150,
              actuatorGain=1,
              Ta=5,
              h_start=2740000,
              D=0.56,
              H=8,
              zl_start=4,
              vessel(initOpt=ThermoPower.Choices.Init.Options.fixedState,
                  noInitialPressure=true),
              p_start=17710000)
              annotation (Placement(transformation(extent={{-18,-20},{38,36}})));
            ThermoPower.Water.SinkPressure sinkPressure1(redeclare package
                Medium =
                  Modelica.Media.Water.WaterIF97_ph, p0=17700000)
              annotation (Placement(transformation(extent={{20,-50},{40,-30}})));
            Modelica.Blocks.Sources.Ramp ramp1(
              duration=500,
              offset=0,
              startTime=1500,
              height=9e5)
              annotation (Placement(transformation(extent={{-100,70},{-80,90}})));
            Modelica.Blocks.Math.Add add
              annotation (Placement(transformation(extent={{-60,50},{-40,70}})));
          equation
            connect(sourceMassFlow.flange, base_2ph_vessel.inlet) annotation (Line(
                points={{-20,0},{0,0}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(base_2ph_vessel.outlet_vap, sinkPressure.flange) annotation (
                Line(
                points={{10,30},{10,60},{20,60}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(base_2ph_vessel.outlet_liq, sinkPressure1.flange) annotation (
                Line(
                points={{10,-10},{10,-40},{20,-40}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(add.y, sourceMassFlow.in_h) annotation (Line(
                points={{-39,60},{-26,60},{-26,6}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(ramp.y, add.u2) annotation (Line(
                points={{-79,40},{-70,40},{-70,54},{-62,54}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(ramp1.y, add.u1) annotation (Line(
                points={{-79,80},{-70,80},{-70,66},{-62,66}},
                color={0,0,127},
                smooth=Smooth.None));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-120,
                      -120},{120,120}}),        graphics), Icon(coordinateSystem(extent={{
                      -120,-120},{120,120}})),
              experiment(StopTime=2500),
              __Dymola_experimentSetupOutput);
          end test_Separator;

          model test_deaerator "Test model of the deaerator"

            Deaerator deaerator(
              L=12,
              H=2.677,
              zl_start=2,
              initOpt=ThermoPower.Choices.Init.Options.fixedState,
              p_start=900000,
              h_start=746000)
              annotation (Placement(transformation(extent={{-30,-26},{30,36}})));
            ThermoPower.Water.SourceMassFlow sourceMassFlow_DR(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              w0=7.727,
              h=782000)
              annotation (Placement(transformation(extent={{-80,-30},{-60,-10}})));
            ThermoPower.Water.SourceMassFlow sourceMassFlow_ST(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=3102000,
              w0=3.551)
              annotation (Placement(transformation(extent={{-60,30},{-40,50}})));
            ThermoPower.Water.SourceMassFlow sourceMassFlow_FW(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              w0=77.31,
              h=630300)
              annotation (Placement(transformation(extent={{-20,50},{0,70}})));
            ThermoPower.PowerPlants.Control.PID PID_level(
              Td=0,
              Nd=1,
              PVmin=0,
              CSmin=0,
              PVstart=0.78,
              CSstart=0.8859,
              PVmax=2.6,
              CSmax=100,
              Ti=10000,
              Kp=-500)       annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={66,-28})));
            Modelica.Blocks.Continuous.FirstOrder pumpDynamic(k=1, T=5)
                    annotation (Placement(
                  transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={66,-60})));
            Modelica.Blocks.Sources.Constant zl_SP(k=2.0559)
              annotation (Placement(transformation(extent={{100,-10},{80,10}})));
            ThermoPower.Water.SinkMassFlow    waterPump(
              allowFlowReversal=false,
              use_in_w0=true,
              w0=88.59,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                     annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={0,-84})));
            inner ThermoPower.System system
              annotation (Placement(transformation(extent={{-100,80},{-80,100}})));
          equation
            connect(deaerator.drained_in, sourceMassFlow_DR.flange) annotation (
                Line(
                points={{-27.8571,-10.5},{-40,-10.5},{-40,-20},{-60,-20}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(deaerator.steam_in, sourceMassFlow_ST.flange) annotation (Line(
                points={{-8.14286,9.42857},{-20,9.42857},{-20,40},{-40,40}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(deaerator.water_in, sourceMassFlow_FW.flange) annotation (Line(
                points={{7.71429,22.7143},{8,22.7143},{8,60},{0,60}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(PID_level.CS,pumpDynamic. u) annotation (Line(
                points={{66,-38},{66,-48}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(pumpDynamic.y,waterPump. in_w0) annotation (Line(
                points={{66,-71},{66,-80},{6,-80}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(zl_SP.y,PID_level. SP) annotation (Line(
                points={{79,0},{70,0},{70,-18}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(deaerator.level, PID_level.PV) annotation (Line(
                points={{27.8571,-12.7143},{62,-12.7143},{62,-18}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(deaerator.drained_out, waterPump.flange) annotation (Line(
                points={{0,-19.3571},{0,-74}},
                color={0,0,255},
                smooth=Smooth.None));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent=
                      {{-100,-100},{100,100}}), graphics),
              experiment(StopTime=5000),
              __Dymola_experimentSetupOutput);
          end test_deaerator;
        end Test;

      end Components;

      package OTSG
        "Package containing all the Once Through Steam Generator models"

        model OTSG_init100_5nodes_lin
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 100% load and 5 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 5
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 10
            "Number of nodes for biphase heat exchangers";

          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            Vhot=33.634,
            Vcold=6.027,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.23186e+006,1.28146e+006,1.32175e+006,1.35546e+006,
                  1.38444e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {560.978,569.126,575.7,581.167}),
            nominalColdPressure=17910000,
            nominalHotPressure=400000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=581.85,
            nominalHotInletTemperature=586.85,
            nominalHotOutletTemperature=563.15,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={136706,129914,122014,112572,100947},
              noInitialPressure=true))          annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            Vhot=45.011,
            Vcold=5.752,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.23186e+006,1.44947e+006,1.55292e+006,1.61733e+006,
                  1.66591e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {582.675,605.885,616.62,
                623.137}),
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={199579,192529,183185,168175,136603},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            Vhot=26.587,
            Vcold=8.651,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.66561e+006,1.67917e+006,1.69246e+006,1.70577e+006,
                  1.7194e+006,1.74575e+006,1.79332e+006,1.87797e+006,2.02847e+006,
                  2.29575e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {626.054,627.39,628.669,
                629.917,629.888,630.182,630.842,632.016,634.097}),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={289063,250302,228476,216201,209301,205481,203503,201573,
                  199646,197679},
              noInitialPressure=true),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                gamma_nom_2ph=10000,
                alpha=1.6),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            Vhot=3.905,
            Vcold=1.065,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.2799e+006,2.32986e+006,2.38391e+006,2.4424e+006,
                  2.50567e+006,2.55859e+006,2.60531e+006,2.65251e+006,
                  2.69972e+006,2.74645e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {650.941,652.73,654.665,
                656.757,670.379,679.959,684.074,688.573,693.427}),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={355877,349098,342248,335400,328621,320944,311763,303278,
                  295435,288187},
              noInitialPressure=true),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000,
                alpha=1.1),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_start=0,
            h_start=2.7e6,
            D=0.56,
            p_start=17710000,
            noInitialPressure=true)
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            w0=0,
            use_in_w0=false)
                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            Vhot=3.68,
            Vcold=2.683,
            tubeWalls(Tvolstart(displayUnit="K") = {725.391,769.727,804.861,
                826.169}),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.74562e+006,3.02031e+006,3.23297e+006,3.36066e+006,
                  3.42731e+006},
              noInitialPressure=true),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,502443,472016,421345,355895},
              noInitialPressure=true),
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            Vhot=1.522,
            Vcold=5.092,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.0832e+006,3.30568e+006,3.43926e+006,3.51621e+006,
                  3.56013e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {734.495,783.217,812.321,
                829.047}),
            Shot=1339,
            Scold=1198,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,503128,476511,430304,353343},
              noInitialPressure=true),
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45)
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            wnom=88.59,
            rhonom=168.8)        annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            rhonom=582.8,
            dpnom=(0.52492/2)*10^5) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=97)           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=12.52)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{100,-236},{80,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            dpnom=(177.1 - 176.1189)*10^5,
            wnom=88.59,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-214})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(HPH.outletCold, MS_HPH_out) annotation (Line(
              points={{-60,-296},{-60,-310}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO2.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
              points={{-60,-4},{-60,48},{-42,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{96,-230},{96,-230},{130,-230},{130,-230},{160,-230},{160,-240},{190,
                  -240}},
              color={0,0,255},
              thickness=0.5));
          connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-44,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,220},{-40,220},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,220},{90,220},{90,220},{90,220},{90,218},
                  {90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,192},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(EVA1.inletCold, PDSteamECO2.outlet) annotation (Line(
              points={{-60,-116},{-60,-130}},
              color={0,0,255},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{6.66134e-016,200},{6.66134e-016,180}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
              points={{0,160},{0,136}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet) annotation (Line(
              points={{-60,-196},{-60,-204}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet, PDSteamECO1.inlet) annotation (Line(
              points={{84,-230},{40,-230},{-6,-230},{-60,-230},{-60,-224}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamHPH.outlet, HPH.inletCold) annotation (Line(
              points={{-60,-256},{-60,-264}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet, PDSteamHPH.inlet) annotation (Line(
              points={{84,-230},{12,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,290}},
              color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{0,270},{0,200}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.valveBP_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-220},{82,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-264},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpFW_PV_high, separator.DT_sh) annotation (Line(
              points={{50,10},{50,10},{50,64},{50,66},{10,66}},
              color={255,204,51},
              thickness=0.5));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init100_5nodes_lin;

        model OTSG_init100_5nodes
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 100% load and 5 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 5
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 10
            "Number of nodes for biphase heat exchangers";

          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            Vhot=33.634,
            Vcold=6.027,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.23186e+006,1.28146e+006,1.32175e+006,1.35546e+006,
                  1.38444e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {560.978,569.126,575.7,581.167}),
            nominalColdPressure=17910000,
            nominalHotPressure=400000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=581.85,
            nominalHotInletTemperature=586.85,
            nominalHotOutletTemperature=563.15,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={136706,129914,122014,112572,100947},
              noInitialPressure=true))          annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            Vhot=45.011,
            Vcold=5.752,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.23186e+006,1.44947e+006,1.55292e+006,1.61733e+006,
                  1.66591e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {582.675,605.885,616.62,
                623.137}),
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={199579,192529,183185,168175,136603},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            Vhot=26.587,
            Vcold=8.651,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.66561e+006,1.67917e+006,1.69246e+006,1.70577e+006,
                  1.7194e+006,1.74575e+006,1.79332e+006,1.87797e+006,2.02847e+006,
                  2.29575e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {626.054,627.39,628.669,
                629.917,629.888,630.182,630.842,632.016,634.097}),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={289063,250302,228476,216201,209301,205481,203503,201573,
                  199646,197679},
              noInitialPressure=true),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                gamma_nom_2ph=10000,
                alpha=1.6),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            Vhot=3.905,
            Vcold=1.065,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.2799e+006,2.32986e+006,2.38391e+006,2.4424e+006,
                  2.50567e+006,2.55859e+006,2.60531e+006,2.65251e+006,
                  2.69972e+006,2.74645e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {650.941,652.73,654.665,
                656.757,670.379,679.959,684.074,688.573,693.427}),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={355877,349098,342248,335400,328621,320944,311763,303278,
                  295435,288187},
              noInitialPressure=true),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000,
                alpha=1.1),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_start=0,
            h_start=2.7e6,
            D=0.56,
            p_start=17710000,
            noInitialPressure=true)
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            Vhot=3.68,
            Vcold=2.683,
            tubeWalls(Tvolstart(displayUnit="K") = {725.391,769.727,804.861,
                826.169}),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.74562e+006,3.02031e+006,3.23297e+006,3.36066e+006,
                  3.42731e+006},
              noInitialPressure=true),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,502443,472016,421345,355895},
              noInitialPressure=true),
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            Vhot=1.522,
            Vcold=5.092,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.0832e+006,3.30568e+006,3.43926e+006,3.51621e+006,
                  3.56013e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {734.495,783.217,812.321,
                829.047}),
            Shot=1339,
            Scold=1198,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,503128,476511,430304,353343},
              noInitialPressure=true),
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45)
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            wnom=88.59,
            rhonom=168.8)        annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            rhonom=582.8,
            dpnom=(0.52492/2)*10^5) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=97)           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=12.52)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{100,-236},{80,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            dpnom=(177.1 - 176.1189)*10^5,
            wnom=88.59,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-214})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(HPH.outletCold, MS_HPH_out) annotation (Line(
              points={{-60,-296},{-60,-310}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO2.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
              points={{-60,-4},{-60,48},{-42,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{96,-230},{96,-230},{130,-230},{130,-230},{160,-230},{160,-240},{190,
                  -240}},
              color={0,0,255},
              thickness=0.5));
          connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-44,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{30,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,44},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-152},{50,-220},{82,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,220},{-40,220},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,220},{90,220},{90,220},{90,220},{90,218},
                  {90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,192},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(EVA1.inletCold, PDSteamECO2.outlet) annotation (Line(
              points={{-60,-116},{-60,-130}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-230},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{6.66134e-016,200},{6.66134e-016,180}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
              points={{0,160},{0,136}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet) annotation (Line(
              points={{-60,-196},{-60,-204}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet, PDSteamECO1.inlet) annotation (Line(
              points={{84,-230},{40,-230},{-6,-230},{-60,-230},{-60,-224}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamHPH.outlet, HPH.inletCold) annotation (Line(
              points={{-60,-256},{-60,-264}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet, PDSteamHPH.inlet) annotation (Line(
              points={{84,-230},{12,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,290}},
              color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{0,270},{0,200}},
              color={0,0,255},
              thickness=0.5));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init100_5nodes;

        model OTSG_init100_20nodes_lin
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 100% load and 20 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 20
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 40
            "Number of nodes for biphase heat exchangers";

          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            Vhot=33.634,
            Vcold=6.027,
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                hstart={1.23186e+006,1.24323e+006,1.25407e+006,1.26441e+006,
                  1.27429e+006,1.28375e+006,1.2928e+006,1.30149e+006,1.30984e+006,
                  1.31787e+006,1.3256e+006,1.33306e+006,1.34026e+006,1.34721e+006,
                  1.35395e+006,1.36047e+006,1.3668e+006,1.37294e+006,1.37891e+006,
                  1.38472e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.505,559.549,561.488,
                563.332,565.087,566.76,568.358,569.885,571.347,572.748,574.093,
                575.386,576.629,577.827,578.983,580.099,581.178,582.223,583.235}),
            nominalColdPressure=17910000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=581.85,
            nominalHotInletTemperature=586.85,
            nominalHotOutletTemperature=563.15,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={136842,135480,134081,132641,131159,129630,128052,126422,
                  124735,122987,121175,119293,117337,115300,113178,110962,108647,
                  106223,103683,101018},
              noInitialPressure=true))          annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.23186e+006,1.29102e+006,1.34032e+006,1.38192e+006,
                  1.41747e+006,1.44818e+006,1.47501e+006,1.49868e+006,
                  1.51977e+006,1.53874e+006,1.55595e+006,1.5717e+006,1.58622e+006,
                  1.59972e+006,1.61237e+006,1.6243e+006,1.63565e+006,1.64651e+006,
                  1.65699e+006,1.66716e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {571.123,579.527,586.384,
                592.055,596.803,600.827,604.275,607.262,609.876,612.186,614.245,
                616.096,617.776,619.311,620.724,622.036,623.262,624.415,625.509}),
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={200329,198854,197334,195758,194112,192380,190545,188586,
                  186479,184195,181698,178946,175885,172451,168559,164103,158946,
                  152909,145756,137173},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.66561e+006,1.66877e+006,1.67191e+006,1.67502e+006,
                  1.67812e+006,1.68121e+006,1.68428e+006,1.68735e+006,
                  1.69042e+006,1.69348e+006,1.69655e+006,1.69963e+006,
                  1.70272e+006,1.70582e+006,1.70894e+006,1.71209e+006,
                  1.71526e+006,1.71847e+006,1.72265e+006,1.72826e+006,
                  1.73465e+006,1.74192e+006,1.7502e+006,1.75963e+006,1.77037e+006,
                  1.78259e+006,1.7965e+006,1.81235e+006,1.83038e+006,1.85091e+006,
                  1.87429e+006,1.90089e+006,1.93118e+006,1.96565e+006,
                  2.00489e+006,2.04954e+006,2.10036e+006,2.15819e6,2.224e6,
                  2.29887e6}),
            tubeWalls(Tvolstart(displayUnit="K") = {625.47,625.791,626.108,
                626.421,626.729,627.034,627.336,627.635,627.931,628.225,628.518,
                628.808,629.097,629.386,629.673,629.96,630.247,629.956,629.469,
                629.529,629.597,629.674,629.763,629.863,629.978,630.108,630.257,
                630.426,630.618,630.837,631.086,631.37,631.692,632.059,632.477,
                632.952,633.492,634.106,634.805}),
            Vhot=26.587,
            Vcold=8.651,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                alpha=1.6,
                gamma_nom_2ph=10000),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={289788,278889,269309,260890,253491,246990,241278,236260,
                  231851,227978,224575,221586,218961,216655,214629,212850,211287,
                  209914,208708,207650,206719,205903,205295,204828,204366,203908,
                  203453,203001,202552,202104,201657,201211,200764,200318,199870,
                  199421,198970,198516,198059,197599},
              noInitialPressure=true))          annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            tubeWalls(Tvolstart(displayUnit="K") = {650.47,650.866,651.269,
                651.679,652.097,652.522,652.956,653.397,653.846,654.304,654.769,
                655.244,655.726,656.218,656.719,657.228,660.331,675.049,675.851,
                676.675,677.521,678.389,679.279,680.19,681.124,682.078,683.052,
                684.047,685.063,686.099,687.154,688.23,689.324,690.436,691.567,
                692.714,693.877,695.057,696.25}),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.29887e+006,2.31027e+006,2.32187e+006,2.33369e+006,
                  2.34573e+006,2.35799e+006,2.37047e+006,2.38319e+006,
                  2.39613e+006,2.40932e+006,2.42274e+006,2.43641e+006,
                  2.45034e+006,2.46452e+006,2.47896e+006,2.49366e+006,
                  2.50864e+006,2.52311e+006,2.53359e+006,2.54414e+006,
                  2.55474e+006,2.56539e+006,2.57608e+006,2.58682e+006,
                  2.59759e+006,2.60839e+006,2.61922e+006,2.63006e+006,
                  2.64092e+006,2.65179e+006,2.66265e+006,2.67351e+006,
                  2.68437e+006,2.6952e+006,2.70601e+006,2.71679e+006,2.72754e+006,
                  2.73825e6,2.7415e6,2.74562e6}),
            Vhot=3.905,
            Vcold=1.065,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={356849,355304,353752,352194,350629,349059,347485,345908,
                  344328,342747,341165,339583,338003,336424,334848,333276,331707,
                  330144,328587,327037,325494,323959,322433,320326,318146,316005,
                  313903,311839,309812,307821,305867,303947,302063,300212,298395,
                  296610,294858,293137,291447,289788},
              noInitialPressure=true))          annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            h_start=2.7e6,
            D=0.56,
            zl_start=0,
            p_start=17710000)
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            w0=0,
            use_in_w0=false)
                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.74562e+006,2.80708e+006,2.8685e+006,2.92872e+006,2.98671e+006,3.04161e+006,
                  3.09279e+006,3.13986e+006,3.18267e+006,3.22121e+006,3.25563e+006,3.28615e+006,
                  3.31308e+006,3.33672e+006,3.35741e+006,3.37544e+006,3.39112e+006,3.40473e+006,
                  3.41652e+006,3.42672e+006},
              noInitialPressure=true),
            tubeWalls(Tvolstart(displayUnit="K") = {707.073,716.706,726.786,737.095,747.398,
                757.467,767.107,776.172,784.565,792.234,799.164,805.37,810.887,815.761,820.047,
                823.8,827.075,829.926,832.401}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={518322,
                  515892,513082,509839,506102,501805,496878,491244,484828,477554,469353,
                  460170,449971,438754,426559,413478,399661,385312,370679,356034},
              noInitialPressure=true),
            Vhot=3.68,
            Vcold=2.683,
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={518322,
                  515738,512837,509580,505925,501823,497219,492053,486258,479757,472467,
                  464297,455144,444899,433444,420654,406405,390575,373063,353803},
              noInitialPressure=true),
            tubeWalls(Tvolstart(displayUnit="K") = {711.921,726.264,739.382,
                751.286,762.029,771.688,780.347,788.096,795.022,801.205,806.722,
                811.642,816.028,819.937,823.42,826.524,829.288,831.751,833.944}),
            Vhot=1.522,
            Vcold=5.092,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.0832e+006,3.13888e+006,3.1895e+006,3.23526e+006,
                  3.27646e+006,3.31343e+006,3.34654e+006,3.37616e+006,
                  3.40262e+006,3.42624e+006,3.44731e+006,3.46611e+006,
                  3.48286e+006,3.4978e+006,3.51111e+006,3.52296e+006,3.53353e+006,
                  3.54294e+006,3.55133e+006,3.5588e+006},
              noInitialPressure=false),
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45)
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            wnom=88.59,
            rhonom=168.8)        annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            rhonom=582.2,
            dpnom=(0.52492/2)*10^5) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=97)           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            dpnom=(177.1 - 176.1189)*10^5,
            wnom=88.59,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{102,-236},{82,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-214})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(HPH.outletCold, MS_HPH_out) annotation (Line(
              points={{-60,-296},{-60,-310}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO2.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
              points={{-60,-4},{-60,48},{-42,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,160},{-4.44089e-016,160}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{98,-230},{98,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-44,-280},{-38,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,220},{-40,220},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,220},{90,220},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,260},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,180}},
              color={0,0,255},
              thickness=0.5));
          connect(EVA1.inletCold, PDSteamECO2.outlet) annotation (Line(
              points={{-60,-116},{-60,-130}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{0,270},{0,270},{0,200}},
              color={0,0,255},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,300},{0,290}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-6,-230},{-60,-230},{-60,-224}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamHPH. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet) annotation (Line(points={{-60,
                  -196},{-60,-204}},                color={0,0,255}));
          connect(PDSteamHPH.outlet, HPH.inletCold) annotation (Line(points={{-60,
                  -256},{-60,-260},{-60,-264}}, color={0,0,255}));
          connect(signalBus.pumpHPH_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveBP_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-220},{84,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpFW_PV_high, separator.DT_sh) annotation (Line(
              points={{50,10},{50,10},{50,66},{10,66}},
              color={255,204,51},
              thickness=0.5));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init100_20nodes_lin;

        model OTSG_init100_20nodes
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 100% load and 20 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 20
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 40
            "Number of nodes for biphase heat exchangers";

          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            Vhot=33.634,
            Vcold=6.027,
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                hstart={1.23186e+006,1.24323e+006,1.25407e+006,1.26441e+006,
                  1.27429e+006,1.28375e+006,1.2928e+006,1.30149e+006,1.30984e+006,
                  1.31787e+006,1.3256e+006,1.33306e+006,1.34026e+006,1.34721e+006,
                  1.35395e+006,1.36047e+006,1.3668e+006,1.37294e+006,1.37891e+006,
                  1.38472e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {136842,135480,134081,132641,131159,129630,128052,126422,124735,
                  122987,121175,119293,117337,115300,113178,110962,108647,106223,
                  103683,101018}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.505,559.549,561.488,
                563.332,565.087,566.76,568.358,569.885,571.347,572.748,574.093,
                575.386,576.629,577.827,578.983,580.099,581.178,582.223,583.235}),
            nominalColdPressure=17910000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=581.85,
            nominalHotInletTemperature=586.85,
            nominalHotOutletTemperature=563.15) annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.23186e+006,1.29102e+006,1.34032e+006,1.38192e+006,
                  1.41747e+006,1.44818e+006,1.47501e+006,1.49868e+006,
                  1.51977e+006,1.53874e+006,1.55595e+006,1.5717e+006,1.58622e+006,
                  1.59972e+006,1.61237e+006,1.6243e+006,1.63565e+006,1.64651e+006,
                  1.65699e+006,1.66716e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {571.123,579.527,586.384,
                592.055,596.803,600.827,604.275,607.262,609.876,612.186,614.245,
                616.096,617.776,619.311,620.724,622.036,623.262,624.415,625.509}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {200329,198854,197334,195758,194112,192380,190545,188586,186479,
                  184195,181698,178946,175885,172451,168559,164103,158946,152909,
                  145756,137173}),
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            nominalHotPressure=1000000)
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.66561e+006,1.66877e+006,1.67191e+006,1.67502e+006,
                  1.67812e+006,1.68121e+006,1.68428e+006,1.68735e+006,
                  1.69042e+006,1.69348e+006,1.69655e+006,1.69963e+006,
                  1.70272e+006,1.70582e+006,1.70894e+006,1.71209e+006,
                  1.71526e+006,1.71847e+006,1.72265e+006,1.72826e+006,
                  1.73465e+006,1.74192e+006,1.7502e+006,1.75963e+006,1.77037e+006,
                  1.78259e+006,1.7965e+006,1.81235e+006,1.83038e+006,1.85091e+006,
                  1.87429e+006,1.90089e+006,1.93118e+006,1.96565e+006,
                  2.00489e+006,2.04954e+006,2.10036e+006,2.15819e6,2.224e6,
                  2.29887e6}),
            tubeWalls(Tvolstart(displayUnit="K") = {625.47,625.791,626.108,
                626.421,626.729,627.034,627.336,627.635,627.931,628.225,628.518,
                628.808,629.097,629.386,629.673,629.96,630.247,629.956,629.469,
                629.529,629.597,629.674,629.763,629.863,629.978,630.108,630.257,
                630.426,630.618,630.837,631.086,631.37,631.692,632.059,632.477,
                632.952,633.492,634.106,634.805}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {289788,278889,269309,260890,253491,246990,241278,236260,231851,
                  227978,224575,221586,218961,216655,214629,212850,211287,209914,
                  208708,207650,206719,205903,205295,204828,204366,203908,203453,
                  203001,202552,202104,201657,201211,200764,200318,199870,199421,
                  198970,198516,198059,197599}),
            Vhot=26.587,
            Vcold=8.651,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                alpha=1.6,
                gamma_nom_2ph=10000),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            tubeWalls(Tvolstart(displayUnit="K") = {650.47,650.866,651.269,
                651.679,652.097,652.522,652.956,653.397,653.846,654.304,654.769,
                655.244,655.726,656.218,656.719,657.228,660.331,675.049,675.851,
                676.675,677.521,678.389,679.279,680.19,681.124,682.078,683.052,
                684.047,685.063,686.099,687.154,688.23,689.324,690.436,691.567,
                692.714,693.877,695.057,696.25}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {356849,355304,353752,352194,350629,349059,347485,345908,344328,
                  342747,341165,339583,338003,336424,334848,333276,331707,330144,
                  328587,327037,325494,323959,322433,320326,318146,316005,313903,
                  311839,309812,307821,305867,303947,302063,300212,298395,296610,
                  294858,293137,291447,289788}),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.29887e+006,2.31027e+006,2.32187e+006,2.33369e+006,
                  2.34573e+006,2.35799e+006,2.37047e+006,2.38319e+006,
                  2.39613e+006,2.40932e+006,2.42274e+006,2.43641e+006,
                  2.45034e+006,2.46452e+006,2.47896e+006,2.49366e+006,
                  2.50864e+006,2.52311e+006,2.53359e+006,2.54414e+006,
                  2.55474e+006,2.56539e+006,2.57608e+006,2.58682e+006,
                  2.59759e+006,2.60839e+006,2.61922e+006,2.63006e+006,
                  2.64092e+006,2.65179e+006,2.66265e+006,2.67351e+006,
                  2.68437e+006,2.6952e+006,2.70601e+006,2.71679e+006,2.72754e+006,
                  2.73825e6,2.7415e6,2.74562e6}),
            Vhot=3.905,
            Vcold=1.065,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            h_start=2.7e6,
            D=0.56,
            zl_start=0,
            p_start=17710000,
            initOpt=ThermoPower.Choices.Init.Options.steadyState)
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.74562e+006,2.80708e+006,2.8685e+006,2.92872e+006,2.98671e+006,3.04161e+006,
                  3.09279e+006,3.13986e+006,3.18267e+006,3.22121e+006,3.25563e+006,3.28615e+006,
                  3.31308e+006,3.33672e+006,3.35741e+006,3.37544e+006,3.39112e+006,3.40473e+006,
                  3.41652e+006,3.42672e+006},
              noInitialPressure=true),
            tubeWalls(Tvolstart(displayUnit="K") = {707.073,716.706,726.786,737.095,747.398,
                757.467,767.107,776.172,784.565,792.234,799.164,805.37,810.887,815.761,820.047,
                823.8,827.075,829.926,832.401}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={518322,
                  515892,513082,509839,506102,501805,496878,491244,484828,477554,469353,
                  460170,449971,438754,426559,413478,399661,385312,370679,356034},
              noInitialPressure=true),
            Vhot=3.68,
            Vcold=2.683,
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={518322,
                  515738,512837,509580,505925,501823,497219,492053,486258,479757,472467,
                  464297,455144,444899,433444,420654,406405,390575,373063,353803},
              noInitialPressure=true),
            tubeWalls(Tvolstart(displayUnit="K") = {711.921,726.264,739.382,
                751.286,762.029,771.688,780.347,788.096,795.022,801.205,806.722,
                811.642,816.028,819.937,823.42,826.524,829.288,831.751,833.944}),
            Vhot=1.522,
            Vcold=5.092,
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.0832e+006,3.13888e+006,3.1895e+006,3.23526e+006,
                  3.27646e+006,3.31343e+006,3.34654e+006,3.37616e+006,
                  3.40262e+006,3.42624e+006,3.44731e+006,3.46611e+006,
                  3.48286e+006,3.4978e+006,3.51111e+006,3.52296e+006,3.53353e+006,
                  3.54294e+006,3.55133e+006,3.5588e+006},
              noInitialPressure=false))
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            wnom=88.59,
            rhonom=168.8)        annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            rhonom=582.2,
            dpnom=(0.52492/2)*10^5) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=97)           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            dpnom=(177.1 - 176.1189)*10^5,
            wnom=88.59,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{102,-236},{82,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-214})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(HPH.outletCold, MS_HPH_out) annotation (Line(
              points={{-60,-296},{-60,-310}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO2.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
              points={{-60,-4},{-60,48},{-42,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,160},{-4.44089e-016,160}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{98,-230},{98,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-44,-280},{-38,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{30,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,44},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-152},{50,-220},{84,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,220},{-40,220},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,220},{90,220},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,260},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,180}},
              color={0,0,255},
              thickness=0.5));
          connect(EVA1.inletCold, PDSteamECO2.outlet) annotation (Line(
              points={{-60,-116},{-60,-130}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-258},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{0,270},{0,270},{0,200}},
              color={0,0,255},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,300},{0,290}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-6,-230},{-60,-230},{-60,-224}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamHPH. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet) annotation (Line(points={{-60,
                  -196},{-60,-204}},                color={0,0,255}));
          connect(PDSteamHPH.outlet, HPH.inletCold) annotation (Line(points={{-60,
                  -256},{-60,-260},{-60,-264}}, color={0,0,255}));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init100_20nodes;

        model OTSG_init50_5nodes_lin
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 50% load and 5 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 5
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 10
            "Number of nodes for biphase heat exchangers";

          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25188e+006,1.26667e+006,1.25173e+006,1.26683e+006,
                  1.25157e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.843,557.837,557.843,
                557.837}),
            Vhot=33.634,
            Vcold=6.027,
            nominalColdPressure=17910000,
            nominalHotPressure=400000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=581.85,
            nominalHotInletTemperature=586.85,
            nominalHotOutletTemperature=563.15,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={93356,93346.5,93356.1,93346.6,93356.1},
              noInitialPressure=true))          annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25188e+006,1.26536e+006,1.2819e+006,1.30253e+006,
                  1.3288e+006}),
            tubeWalls(Tvolstart(displayUnit="degC") = {558.324,561.292,564.934,
                569.474}),
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={119114,110526,103784,98376.4,93969.4},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.33e+006,1.3378e+006,1.34658e+006,1.35651e+006,
                  1.36781e+006,1.38075e+006,1.39566e+006,1.41303e+006,
                  1.45658e+006,1.56606e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {572.404,573.98,575.749,
                577.742,579.999,582.568,585.511,585.959,587.035}),
            Vhot=26.587,
            Vcold=8.651,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                alpha=1.6,
                gamma_nom_2ph=10000),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={193708,157918,143681,138009,133137,128914,125226,121985,
                  119121,116578},
              noInitialPressure=true))          annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.56606e+006,1.59353e+006,1.62512e+006,1.66145e+006,
                  1.70321e+006,1.75123e+006,1.80642e+006,1.86984e+006,
                  1.94272e+006,2.02642e+006}),
            tubeWalls(Tvolstart(displayUnit="degC") = {596.276,597.951,599.875,
                602.087,604.627,607.543,610.89,614.73,619.134}),
            Vhot=3.905,
            Vcold=1.065,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000),
            Shot=429.2,
            Scold=338,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={344122,316769,292958,272234,254202,238514,224868,213001,
                  202681,193708},
              noInitialPressure=true),
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=902.3,
            nominalColdOutletTemperature=921.8,
            nominalHotInletTemperature=1005.2,
            nominalHotOutletTemperature=961.1)  annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_start=4,
            phase(start=2, fixed=true),
            D=0.56,
            w_small=1e4,
            h_start=2416500,
            p_start=10100000,
            noInitialPressure=false,
            noInitialLevel=true,
            T(start=584.85))
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            tubeWalls(Tvolstart={733.236,797.905,827.469,835.959}),
            Vhot=3.68,
            Vcold=2.683,
            Shot=1153.8,
            Scold=879.8,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,514827,501414,454805,353814},
              noInitialPressure=true),
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.72364e+006,3.22113e+006,3.45073e+006,3.5168e+006,
                  3.53402e+006},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.12533e+006,3.32376e+006,3.45094e+006,3.53171e+006,
                  3.58318e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {712.396,768.496,804.215,
                826.852}),
            Vhot=1.522,
            Vcold=5.092,
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,493431,454370,392870,296910},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            wnom=88.59,
            rhonom=168.8,
            pin(start=10200000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            rhonom=768.15,
            dpnom=(0.52492/2)*10^5,
            pin(start=10300000))           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=97,
            pin(start=10150000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=88.59,
            dpnom=(177.1 - 176.1)*10^5,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{102,-236},{82,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7,
            pin(start=103.5e5))            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-212})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(HPH.outletCold, MS_HPH_out) annotation (Line(
              points={{-60,-296},{-60,-310}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
              points={{-60,-116},{-60,-124},{-60,-130}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,160},{-6.66134e-016,160}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{98,-230},{98,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_MS.inlet, HPH.outletHot) annotation (Line(
              points={{-16,-280},{-44,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{10,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,70},{50,218},{-40,218},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,218},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,288},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,226},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,246},{-100,186},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,180}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
              points={{-42,48},{-50,48},{-60,48},{-60,-4}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,258},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-4,-230},{-60,-230},{-60,-222}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamHPH. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet)
            annotation (Line(points={{-60,-196},{-60,-202}}, color={0,0,255}));
          connect(PDSteamHPH.outlet, HPH.inletCold) annotation (Line(points={{-60,
                  -256},{-60,-256},{-60,-264}}, color={0,0,255}));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,310},{0,298},{4.44089e-016,298},{4.44089e-016,
                  290}},
              color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{-6.66134e-016,270},{-6.66134e-016,200},{0,200}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{6,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpFW_PV_low, separator.level) annotation (Line(
              points={{50,10},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{48,10},{48,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init50_5nodes_lin;

        model OTSG_init50_5nodes
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 50% load and 5 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 5
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 10
            "Number of nodes for biphase heat exchangers";

          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25188e+006,1.26667e+006,1.25173e+006,1.26683e+006,
                  1.25157e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {93356,93346.5,93356.1,93346.6,93356.1}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.843,557.837,557.843,
                557.837}),
            Vhot=33.634,
            Vcold=6.027,
            nominalColdPressure=17910000,
            nominalHotPressure=400000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=581.85,
            nominalHotInletTemperature=586.85,
            nominalHotOutletTemperature=563.15) annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25188e+006,1.26536e+006,1.2819e+006,1.30253e+006,
                  1.3288e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {119114,110526,103784,98376.4,93969.4}),
            tubeWalls(Tvolstart(displayUnit="degC") = {558.324,561.292,564.934,
                569.474}),
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72)
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.33e+006,1.3378e+006,1.34658e+006,1.35651e+006,
                  1.36781e+006,1.38075e+006,1.39566e+006,1.41303e+006,
                  1.45658e+006,1.56606e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {193708,157918,143681,138009,133137,128914,125226,121985,119121,
                  116578}),
            tubeWalls(Tvolstart(displayUnit="K") = {572.404,573.98,575.749,
                577.742,579.999,582.568,585.511,585.959,587.035}),
            Vhot=26.587,
            Vcold=8.651,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                alpha=1.6,
                gamma_nom_2ph=10000),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.56606e+006,1.59353e+006,1.62512e+006,1.66145e+006,
                  1.70321e+006,1.75123e+006,1.80642e+006,1.86984e+006,
                  1.94272e+006,2.02642e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {344122,316769,292958,272234,254202,238514,224868,213001,202681,
                  193708}),
            tubeWalls(Tvolstart(displayUnit="degC") = {596.276,597.951,599.875,
                602.087,604.627,607.543,610.89,614.73,619.134}),
            Vhot=3.905,
            Vcold=1.065,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=177600000,
            nominalHotPressure=7308000,
            nominalColdInletTemperature=902.3,
            nominalColdOutletTemperature=921.8,
            nominalHotInletTemperature=1005.2,
            nominalHotOutletTemperature=961.1)  annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_start=4,
            phase(start=2, fixed=true),
            D=0.56,
            w_small=1e4,
            noInitialPressure=false,
            h_start=2416500,
            noInitialLevel=false,
            p_start=10100000,
            T(start=584.85))
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.72364e+006,3.22113e+006,3.45073e+006,3.5168e+006,
                  3.53402e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {518322,514827,501414,454805,353814}),
            tubeWalls(Tvolstart={733.236,797.905,827.469,835.959}),
            Vhot=3.68,
            Vcold=2.683,
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.12533e+006,3.32376e+006,3.45094e+006,3.53171e+006,
                  3.58318e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {518322,493431,454370,392870,296910}),
            tubeWalls(Tvolstart(displayUnit="K") = {712.396,768.496,804.215,
                826.852}),
            Vhot=1.522,
            Vcold=5.092,
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45)
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            wnom=88.59,
            rhonom=168.8,
            pin(start=10200000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            rhonom=768.15,
            dpnom=(0.52492/2)*10^5,
            pin(start=10300000))           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=97,
            pin(start=10150000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=88.59,
            dpnom=(177.1 - 176.1)*10^5,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{102,-236},{82,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7,
            pin(start=103.5e5))            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-212})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(HPH.outletCold, MS_HPH_out) annotation (Line(
              points={{-60,-296},{-60,-310}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
              points={{-60,-116},{-60,-124},{-60,-130}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,160},{-6.66134e-016,160}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{98,-230},{98,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_MS.inlet, HPH.outletHot) annotation (Line(
              points={{-16,-280},{-44,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{10,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{6,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,70},{50,218},{-40,218},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,218},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-226},{50,-220},{84,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,288},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,226},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,246},{-100,186},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,180}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
              points={{-42,48},{-50,48},{-60,48},{-60,-4}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-252},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,258},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-4,-230},{-60,-230},{-60,-222}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamHPH. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet)
            annotation (Line(points={{-60,-196},{-60,-202}}, color={0,0,255}));
          connect(PDSteamHPH.outlet, HPH.inletCold) annotation (Line(points={{-60,
                  -256},{-60,-256},{-60,-264}}, color={0,0,255}));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,310},{0,298},{4.44089e-016,298},{4.44089e-016,
                  290}},
              color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{-6.66134e-016,270},{-6.66134e-016,200},{0,200}},
              color={0,0,255},
              thickness=0.5));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init50_5nodes;

        model OTSG_init50_5nodes_noHPH
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 50% load and 5 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 5
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 10
            "Number of nodes for biphase heat exchangers";

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25188e+006,1.26536e+006,1.2819e+006,1.30253e+006,
                  1.3288e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {119114,110526,103784,98376.4,93969.4}),
            tubeWalls(Tvolstart(displayUnit="degC") = {558.324,561.292,564.934,
                569.474}),
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72)
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.33e+006,1.3378e+006,1.34658e+006,1.35651e+006,
                  1.36781e+006,1.38075e+006,1.39566e+006,1.41303e+006,
                  1.45658e+006,1.56606e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {193708,157918,143681,138009,133137,128914,125226,121985,119121,
                  116578}),
            tubeWalls(Tvolstart(displayUnit="K") = {572.404,573.98,575.749,
                577.742,579.999,582.568,585.511,585.959,587.035}),
            Vhot=26.587,
            Vcold=8.651,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                alpha=1.6,
                gamma_nom_2ph=10000),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.56606e+006,1.59353e+006,1.62512e+006,1.66145e+006,
                  1.70321e+006,1.75123e+006,1.80642e+006,1.86984e+006,
                  1.94272e+006,2.02642e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {344122,316769,292958,272234,254202,238514,224868,213001,202681,
                  193708}),
            tubeWalls(Tvolstart(displayUnit="degC") = {596.276,597.951,599.875,
                602.087,604.627,607.543,610.89,614.73,619.134}),
            Vhot=3.905,
            Vcold=1.065,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_start=4,
            h_start=2016500,
            phase(start=2, fixed=true),
            D=0.56,
            initOpt=ThermoPower.Choices.Init.Options.steadyState,
            p_start=10100000,
            T(start=584.85))
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.72364e+006,3.22113e+006,3.45073e+006,3.5168e+006,
                  3.53402e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {518322,514827,501414,454805,353814}),
            tubeWalls(Tvolstart={733.236,797.905,827.469,835.959}),
            Vhot=3.68,
            Vcold=2.683,
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.12533e+006,3.32376e+006,3.45094e+006,3.53171e+006,
                  3.58318e+006}),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                 {518322,493431,454370,392870,296910}),
            tubeWalls(Tvolstart(displayUnit="K") = {712.396,768.496,804.215,
                826.852}),
            Vhot=1.522,
            Vcold=5.092,
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45)
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            wnom=88.59,
            rhonom=168.8,
            pin(start=10200000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            rhonom=768.15,
            dpnom=(0.52492/2)*10^5,
            pin(start=10300000))           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=97,
            pin(start=10150000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = ECO.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=88.59,
            dpnom=(177.1 - 176.1)*10^5,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{102,-236},{82,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7,
            pin(start=103.5e5))            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-212})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
              points={{-60,-116},{-60,-130}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,160},{-6.66134e-016,160}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{98,-230},{98,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{10,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{6,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,70},{50,218},{-40,218},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,218},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-226},{50,-220},{84,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,288},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,226},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,246},{-100,186},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,180}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
              points={{-42,48},{-50,48},{-60,48},{-60,-4}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-252},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,258},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{86,-230},{86,-230},{-4,-230},{-60,-230},{-60,-222}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet)
            annotation (Line(points={{-60,-196},{-60,-202}}, color={0,0,255}));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(ECO.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-76,-180},{-100,-180},{-100,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(sensT_FW.outlet, MS_HPH_out) annotation (Line(
              points={{86,-230},{14,-230},{-60,-230},{-60,-310}},
              color={0,0,255},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(points={{0,310},{
                  6.66134e-016,310},{6.66134e-016,290}}, color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange)
            annotation (Line(points={{0,270},{0,270},{0,200}}, color={0,0,255},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init50_5nodes_noHPH;

        model OTSG_init50_20nodes
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 50% load and 20 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 20
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 40
            "Number of nodes for biphase heat exchangers";

          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
            tubeWalls(Tvolstart(displayUnit="K")),
            Nhex=60,
            Vhot=33.634,
            Vcold=6.027,
            nominalColdPressure=17910000,
            nominalHotPressure=400000,
            nominalColdInletTemperature=556.45,
            nominalColdOutletTemperature=556.55,
            nominalHotInletTemperature=558.25,
            nominalHotOutletTemperature=558.15,
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                noInitialPressure=true))        annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25188e+006,1.25451e+006,1.25725e+006,1.26011e+006,
                  1.26308e+006,1.26619e+006,1.26943e+006,1.27281e+006,
                  1.27635e+006,1.28004e+006,1.28392e+006,1.28797e+006,
                  1.29222e+006,1.29668e+006,1.30136e+006,1.30628e+006,
                  1.31146e+006,1.31691e+006,1.32267e+006,1.32874e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.251,557.785,558.341,
                558.919,559.522,560.149,560.803,561.486,562.198,562.941,563.717,
                564.529,565.379,566.268,567.2,568.177,569.203,570.281,571.414}),
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={119114,117128,115248,113465,111773,110164,108634,107177,
                  105788,104462,103197,101988,100832,99725.3,98665.9,97650.9,
                  96677.8,95744.5,94848.9,93988.9},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            nominalColdFlow=88.59,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.33e+006,1.33177e+006,1.33359e+006,1.33546e+006,
                  1.33738e+006,1.33935e+006,1.34138e+006,1.34346e+006,
                  1.34561e+006,1.34781e+006,1.35008e+006,1.35242e+006,
                  1.35482e+006,1.3573e+006,1.35985e+006,1.36248e+006,1.36519e+006,
                  1.36798e+006,1.37086e+006,1.37384e+006,1.37691e+006,
                  1.38008e+006,1.38335e+006,1.38674e+006,1.39024e+006,
                  1.39387e+006,1.39762e+006,1.4015e+006,1.40553e+006,1.4097e+006,
                  1.41406e+006,1.42073e+006,1.42889e+006,1.43884e+006,1.451e+006,
                  1.46584e+006,1.48395e+006,1.50606e+006,1.53e6,1.56596e6}),
            tubeWalls(Tvolstart(displayUnit="K") = {571.855,572.198,572.549,
                572.91,573.28,573.66,574.05,574.451,574.862,575.285,575.719,
                576.166,576.625,577.097,577.582,578.081,578.595,579.124,579.669,
                580.23,580.808,581.403,582.017,582.651,583.304,583.978,584.674,
                585.393,586.135,586.886,585.708,585.807,585.936,586.094,586.287,
                586.522,586.809,587.159,587.586}),
            Vhot=26.587,
            Vcold=8.651,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_vap=3875,
                alpha=1.6,
                gamma_nom_2ph=10000),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={193815,183049,174227,166998,161076,156224,152250,148995,
                  146328,144147,142724,141359,140043,138773,137547,136363,135218,
                  134112,133041,132005,131002,130031,129089,128177,127292,126434,
                  125601,124793,124008,123246,122506,121786,121087,120407,119745,
                  119102,118476,117866,117273,116696},
              noInitialPressure=true),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.56596e+006,1.57196e+006,1.57816e+006,1.58456e+006,
                  1.59118e+006,1.59801e+006,1.60506e+006,1.61234e+006,
                  1.61986e+006,1.62763e+006,1.63565e+006,1.64394e+006,
                  1.65249e+006,1.66133e+006,1.67045e+006,1.67987e+006,1.6896e+006,
                  1.69965e+006,1.71003e+006,1.72074e+006,1.7318e+006,1.74323e+006,
                  1.75503e+006,1.76721e+006,1.77979e+006,1.79278e+006,
                  1.80619e+006,1.82004e+006,1.83434e+006,1.84911e+006,
                  1.86435e+006,1.88009e+006,1.89635e+006,1.91313e+006,
                  1.93046e+006,1.94835e+006,1.96682e+006,1.98589e6,1.996e6,2.02e6}),
            tubeWalls(Tvolstart(displayUnit="K") = {595.682,596.028,596.386,
                596.755,597.136,597.53,597.936,598.356,598.789,599.237,599.699,
                600.176,600.668,601.177,601.702,602.244,602.804,603.381,603.978,
                604.594,605.229,605.886,606.563,607.263,607.984,608.73,609.499,
                610.293,611.113,611.959,612.832,613.733,614.663,615.624,616.614,
                617.637,618.692,619.781,620.905}),
            Vhot=3.905,
            Vcold=1.065,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_liq=2000,
                gamma_nom_2ph=5000,
                gamma_nom_vap=2000),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={344122,337479,331044,324811,318775,312928,307265,301780,
                  296468,291323,286340,281515,276841,272315,267932,263687,259576,
                  255595,251739,248006,244390,240889,237498,234215,231036,227957,
                  224976,222089,219293,216586,213965,211427,208969,206590,204285,
                  202054,199894,197802,195776,193815},
              noInitialPressure=true),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_start=4,
            phase(start=2, fixed=true),
            D=0.56,
            h_start=2e6,
            p_start=10100000,
            T(start=584.85))
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            tubeWalls(Tvolstart(displayUnit="K") = {701.192,720.859,740.557,
                758.771,774.634,787.907,798.731,807.419,814.323,819.772,824.056,
                827.413,830.038,832.088,833.687,834.934,835.905,836.661,837.25}),
            Vhot=3.68,
            Vcold=2.683,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,517811,517154,516309,515225,513832,512044,509752,
                  506814,503054,498249,492123,484340,474500,462153,446842,428199,
                  406136,381092,354210},
              noInitialPressure=true),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.72364e+006,2.85606e+006,2.97943e+006,3.08812e+006,
                  3.17995e+006,3.25538e+006,3.3162e+006,3.36467e+006,3.40301e+006,
                  3.43319e+006,3.45686e+006,3.47538e+006,3.48985e+006,
                  3.50115e+006,3.50995e+006,3.51681e+006,3.52215e+006,
                  3.52631e+006,3.52955e+006,3.53207e+006},
              noInitialPressure=true),
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));

          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.12533e+006,3.17417e+006,3.21887e+006,3.25968e+006,
                  3.29688e+006,3.33075e+006,3.36159e+006,3.38966e+006,3.4152e+006,
                  3.43844e+006,3.45959e+006,3.47885e+006,3.49637e+006,
                  3.51234e+006,3.52687e+006,3.54011e+006,3.55218e+006,
                  3.56317e+006,3.57319e+006,3.58232e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {686.808,702.938,717.69,
                731.147,743.403,754.555,764.697,773.919,782.303,789.927,796.861,
                803.168,808.906,814.128,818.881,823.208,827.149,830.739,834.01}),
            Vhot=1.522,
            Vcold=5.092,
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,513905,509059,503742,497908,491505,484475,476756,
                  468280,458969,448740,437500,425150,411577,396664,380281,362292,
                  342558,320942,297321},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));

          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-66,242},{-46,222}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            rhonom=582.24,
            wnom=88.59,
            pin(start=102e5))    annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-58})));
          ThermoPower.Water.PressDrop PDSteamECO(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492)*10^5,
            rhonom=768.15,
            pin(start=103e5))              annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=250,
            pin(start=10150000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            dpnom=(177.1 - 176.1189)*10^5,
            rhonom=96.586,
            wnom=88.59)                    annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,170})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{104,-236},{84,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7,
            pin(start=103.5e5))            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-214})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-248})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-46,232},{140,232},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(HPH.outletCold, MS_HPH_out) annotation (Line(
              points={{-60,-296},{-60,-310}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-48},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-68}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
              points={{-60,-116},{-60,-130}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,160}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{100,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-44,-280},{-30,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{50,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-204},{50,-220},{86,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,218},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,34},{50,218},{-60,218},{-60,226}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,246},{-100,140},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,232},{-66,232}},
              color={255,0,0},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,180},{0,180}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
              points={{-42,48},{-52,48},{-60,48},{-60,-4}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-252},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,310},{0,290}},
              color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{0,270},{0,200}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{88,-230},{88,-230},{14,-230},{-60,-230},{-60,-224}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamHPH. inlet) annotation (Line(
              points={{88,-230},{88,-230},{-60,-230},{-60,-238}},
              color={0,0,255},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet)
            annotation (Line(points={{-60,-196},{-60,-204}}, color={0,0,255}));
          connect(PDSteamHPH.outlet, HPH.inletCold) annotation (Line(points={{-60,
                  -258},{-60,-258},{-60,-264}},            color={0,0,255}));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init50_20nodes;

        model OTSG_init20_5nodes
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 20% load and 5 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 5
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 10
            "Number of nodes for biphase heat exchangers";

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            Vhot=45.011,
            Vcold=5.752,
            tubeWalls(Tvolstart(displayUnit="K") = {557.65,557.653,557.662,557.69}),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={93204,93107.6,93079.6,93070.9,93067.6},
              noInitialPressure=true),
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25835e+006,1.25836e+006,1.25838e+006,1.25843e+006,
                  1.25862e+006},
              noInitialPressure=false))
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_2ph=25350,
                gamma_nom_vap=3875,
                alpha=1.6),
            nominalColdFlow=88.59,
            Vhot=26.587,
            Vcold=8.651,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25835e+006,1.2597e+006,1.26181e+006,1.2651e+006,
                  1.27023e+006,1.27828e+006,1.29095e+006,1.31099e+006,
                  1.34301e+006,1.39506e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.961,558.395,559.071,
                560.126,561.776,564.358,568.414,574.81,584.965}),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={165420,138076,121257,110733,104086,99862.4,97169.5,95448.2,
                  94345.9,93638.9},
              noInitialPressure=true),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_2ph=6000,
                gamma_nom_liq=2600,
                gamma_nom_vap=2600),
            Vhot=3.905,
            Vcold=1.065,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.39506e+006,1.40604e+006,1.42359e+006,1.44608e+006,
                  1.47414e+006,1.50913e+006,1.55276e+006,1.60714e+006,
                  1.67488e+006,1.75921e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {596.362,594.413,596.184,
                599.13,602.802,607.374,613.064,620.14,628.927}),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={356546,312270,276709,248166,225266,206901,192178,180378,
                  171175,165420},
              noInitialPressure=true),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            D=0.56,
            zl_start=4,
            phase(fixed=true, start=2),
            noInitialPressure=false,
            noInitialLevel=true,
            w_small=1e4,
            h_start=1.5e6,
            p_start=10010000)
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            Vhot=3.68,
            Vcold=2.683,
            tubeWalls(Tvolstart(displayUnit="K") = {764.198,822.03,836.654,
                838.039}),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.72529e+006,3.35384e+006,3.52174e+006,3.53772e+006,
                  3.53903e+006},
              noInitialPressure=true),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,518096,515357,486572,378819},
              noInitialPressure=true),
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            Vhot=1.522,
            Vcold=5.092,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.12459e+006,3.31132e+006,3.44042e+006,3.53111e+006,
                  3.59583e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {686.149,749.323,792.977,
                823.682}),
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,480112,426575,350366,240145},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            rhonom=582.24,
            wnom=88.59)          annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492)*10^5,
            rhonom=768.15)                 annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-140})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=250)          annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = ECO.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{100,-236},{80,-216}})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=88.59,
            dpnom=(177.1 - 176.1)*10^5,
            rhonom=51.9)                   annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,160})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            tubeWalls(Tvolstart(displayUnit="K") = {557.843,557.837,557.843,
                557.837}),
            Vhot=33.634,
            Vcold=6.027,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={93356,93346.5,93356.1,93346.6,93356.1},
              noInitialPressure=true),
            nominalColdPressure=17910000,
            nominalHotPressure=400000,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=581.85,
            nominalHotInletTemperature=586.85,
            nominalHotOutletTemperature=563.15,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25188e+006,1.26667e+006,1.25173e+006,1.26683e+006,
                  1.25157e+006},
              noInitialPressure=false))         annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7,
            pin(start=103.5e5))            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-212})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
              points={{-60,-164},{-60,-150}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16},{1.77636e-015,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
              points={{-60,-4},{-60,48},{-42,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{96,-230},{130,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{30,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,44},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-152},{50,-220},{82,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,218},{-40,218},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,218},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,246},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,260},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
              points={{-60,-116},{-60,-130}},
              color={0,0,255},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,170}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
              points={{-6.66134e-016,150},{0,150},{0,136}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,300},{0,290}},
              color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{0,270},{0,270},{0,200}},
              color={0,0,255},
              thickness=0.5));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{84,-230},{84,-230},{-4,-230},{-60,-230},{-60,-222}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamHPH. inlet) annotation (Line(
              points={{84,-230},{84,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamHPH.outlet,HPH. inletCold) annotation (Line(points={{-60,
                  -256},{-60,-256},{-60,-264}}, color={0,0,255}));
          connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-44,-280},{-30,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(PDSteamECO1.outlet, ECO.inletCold) annotation (Line(points={{
                  -60,-202},{-60,-202},{-60,-196}}, color={0,0,255}));
          connect(HPH.outletCold, MS_HPH_out)
            annotation (Line(points={{-60,-296},{-60,-310}}, color={0,0,255}));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init20_5nodes;

        model OTSG_init20_20nodes
          "Dynamic model of the once through steam generator, enthalpy profile initialisation at 20% load and 20 nodes"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 20
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 40
            "Number of nodes for biphase heat exchangers";

          Components.HE ECO(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25835e+006,1.25835e+006,1.25835e+006,1.25835e+006,1.25836e+006,1.25836e+006,
                  1.25836e+006,1.25836e+006,1.25837e+006,1.25837e+006,1.25838e+006,1.25839e+006,
                  1.2584e+006,1.25841e+006,1.25843e+006,1.25845e+006,1.25848e+006,1.25852e+006,
                  1.25856e+006,1.25862e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.649,557.65,557.65,557.65,557.651,
                557.651,557.652,557.653,557.654,557.655,557.657,557.659,557.662,557.666,
                557.671,557.676,557.684,557.693,557.705}),
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={93204,93175.1,93152.3,93134.2,93120,93108.7,93099.8,93092.7,
                  93087.1,93082.7,93079.2,93076.4,93074.2,93072.5,93071.1,93069.9,
                  93069.1,93068.3,93067.8,93067.3},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-44,-196},{-76,-164}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_2ph=25350,
                gamma_nom_vap=3875,
                alpha=1.6),
            nominalColdFlow=88.59,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.25835e+006,1.25863e+006,1.25893e+006,1.25927e+006,1.25965e+006,1.26006e+006,
                  1.26052e+006,1.26102e+006,1.26158e+006,1.2622e+006,1.26289e+006,1.26364e+006,
                  1.26448e+006,1.26541e+006,1.26644e+006,1.26757e+006,1.26883e+006,1.27022e+006,
                  1.27177e+006,1.27347e+006,1.27537e+006,1.27746e+006,1.27978e+006,1.28236e+006,
                  1.28521e+006,1.28837e+006,1.29188e+006,1.29577e+006,1.30009e+006,1.30489e+006,
                  1.31022e+006,1.31614e+006,1.32272e+006,1.33005e+006,1.33821e+006,1.3473e+006,
                  1.35745e+006,1.36878e6,1.38e6,1.39564e6}),
            tubeWalls(Tvolstart(displayUnit="K") = {557.837,557.911,557.991,558.081,558.179,
                558.289,558.409,558.543,558.691,558.854,559.035,559.235,559.457,559.702,
                559.972,560.272,560.604,560.971,561.378,561.827,562.325,562.877,563.487,
                564.163,564.911,565.741,566.659,567.677,568.805,570.055,571.441,572.978,
                574.683,576.574,578.674,581.006,583.596,586.473,589.673}),
            Vhot=26.587,
            Vcold=8.651,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={165796,158329,151664,145706,140372,135593,131305,127456,
                  123997,120887,118090,115571,113304,111261,109420,107760,106264,
                  104914,103696,102597,101605,100709,99900.7,99170.6,98511.2,
                  97915.6,97377.6,96891.6,96452.5,96055.7,95697.1,95373,95080.1,
                  94815.4,94576.1,94359.8,94164.3,93987.5,93827.6,93683.1},
              noInitialPressure=true),
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));
          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_2ph=6000,
                gamma_nom_liq=2600,
                gamma_nom_vap=2600),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={1.39564e+006,1.3981e+006,1.40061e+006,1.40317e+006,1.40579e+006,1.40851e+006,
                  1.41247e+006,1.41666e+006,1.42105e+006,1.42567e+006,1.43052e+006,1.43562e+006,
                  1.44099e+006,1.44664e+006,1.45258e+006,1.45883e+006,1.46541e+006,1.47233e+006,
                  1.47961e+006,1.48727e+006,1.49533e+006,1.5038e+006,1.51272e+006,1.5221e+006,
                  1.53197e+006,1.54236e+006,1.55328e+006,1.56477e+006,1.57685e+006,1.58956e+006,
                  1.60294e+006,1.617e+006,1.63179e+006,1.64735e+006,1.66371e+006,1.68092e+006,
                  1.69902e+006,1.71806e+006,1.738e6,1.758e6}),
            tubeWalls(Tvolstart(displayUnit="K") = {595.418,596.076,596.746,597.427,598.122,
                593.494,593.825,594.323,594.848,595.399,595.98,596.59,597.232,597.908,598.619,
                599.366,600.153,600.98,601.85,602.765,603.727,604.739,605.804,606.923,608.101,
                609.339,610.64,612.009,613.448,614.961,616.552,618.224,619.982,621.83,623.772,
                625.813,627.958,630.212,632.58}),
            Vhot=3.905,
            Vcold=1.065,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={356546,345492,334980,324985,315480,306442,297849,289679,
                  281911,274525,267504,260830,254484,248452,242718,237267,232086,
                  227161,222479,218029,213800,209780,205959,202327,198875,195595,
                  192477,189514,186698,184021,181478,179060,176763,174580,172520,
                  171124,169754,168410,167091,165796},
              noInitialPressure=true),
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_start=4,
            phase(start=2, fixed=true),
            D=0.56,
            h_start=1.5e6,
            p_start=10010000,
            w_small=1e4)
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,6})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            tubeWalls(Tvolstart(displayUnit="K") = {731.796,755.354,776.539,793.603,806.462,
                815.817,822.5,827.226,830.55,832.88,834.509,835.647,836.441,836.995,837.382,
                837.651,837.838,837.969,838.06}),
            Vhot=3.68,
            Vcold=2.683,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={2.72529e+006,2.91596e+006,3.07983e+006,3.20863e+006,
                  3.30452e+006,3.37387e+006,3.42325e+006,3.45811e+006,3.4826e+006,
                  3.49975e+006,3.51175e+006,3.52012e+006,3.52597e+006,
                  3.53004e+006,3.53289e+006,3.53487e+006,3.53625e+006,
                  3.53721e+006,3.53788e+006,3.53834e+006},
              noInitialPressure=true),
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,518242,518127,517962,517726,517386,516899,516200,
                  515199,513763,511706,508765,504566,498590,490125,478237,461797,
                  439716,411624,378937},
              noInitialPressure=true),
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));
          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              hstart={3.12459e+006,3.16961e+006,3.21122e+006,3.2497e+006,3.28529e+006,3.31823e+006,
                  3.34874e+006,3.37702e+006,3.40325e+006,3.4276e+006,3.45023e+006,3.47127e+006,
                  3.49084e+006,3.50907e+006,3.52605e+006,3.54188e+006,3.55665e+006,3.57043e+006,
                  3.5833e+006,3.59532e+006}),
            tubeWalls(Tvolstart(displayUnit="K") = {657.909,675.332,691.412,706.255,719.965,
                732.636,744.358,755.211,765.269,774.598,783.257,791.302,798.782,805.742,
                812.222,818.259,823.888,829.139,834.039}),
            Vhot=1.522,
            Vcold=5.092,
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45,
            hotFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              hstart={518322,511224,503626,495489,486771,477425,467400,456641,
                  445085,432665,419308,404931,389445,372752,354743,335298,314289,
                  291576,267010,240439},
              noInitialPressure=true))
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));
          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,210})));

          ThermoPower.Water.PressDrop PDSteamEVA1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            dpnom=(0.05828)*10^5,
            rhonom=582.24,
            wnom=88.59,
            pin(start=10030000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-60})));
          ThermoPower.Water.PressDrop PDSteamECO(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492)*10^5,
            rhonom=768.15,
            pin(start=10050000))           annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-138})));
          ThermoPower.Water.PressDrop PDSteamEVA2(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.11171)*10^5,
            rhonom=250,
            pin(start=10010000)) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-32,48})));
          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
          Modelica.SIunits.Temperature T_outMS = EVA1.hotFluid.T[end];
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            dpnom=(177.1 - 176.1189)*10^5,
            rhonom=96.586,
            wnom=88.59)                    annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,160})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,180})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-286},{0,-266}})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{100,-236},{80,-216}})));
          ThermoPower.Water.ThroughMassFlow throughMassFlow(
            w0=610.6,
            use_in_w0=true,
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(
                extent={{10,10},{-10,-10}},
                rotation=90,
                origin={-100,256})));
          ThermoPower.Water.SensP sensP(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={4,200})));
          ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                Modelica.Media.Water.WaterIF97_ph,
            Av=0.046406353,
            wnom=88.59,
            thetanom=0.8,
            Fxt_full=0.4542722663,
            pnom=17610000,
            dpnom=110000)
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={0,280})));
          Components.HE HPH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            Shot=2429.5,
            Scold=1913.2,
            Mm=29335,
            nominalColdFlow=143.1,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2000, alpha=0.447465),
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=5000, alpha=0.536143),
            Vhot=33.634,
            Vcold=6.027,
            nominalColdPressure=17910000,
            nominalHotPressure=400000,
            nominalColdInletTemperature=559.15,
            nominalColdOutletTemperature=560.15,
            nominalHotInletTemperature=561.15,
            nominalHotOutletTemperature=560.15,
            coldFluid(
              FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
              Nt=1,
              noInitialPressure=false),
            tubeWalls(Tvolstart(displayUnit="K")),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                noInitialPressure=true))        annotation (Placement(
                transformation(extent={{-76,-264},{-44,-296}})));
          ThermoPower.Water.PressDrop PDSteamECO1(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            wnom=88.59,
            dpnom=(0.52492/2)*10^5,
            rhonom=767.7,
            pin(start=103.5e5))            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-212})));
          ThermoPower.Water.PressDrop PDSteamHPH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = coldMedium,
            rhonom=767.7,
            wnom=143.1,
            dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-60,-246})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,230},{140,230},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,216}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
              points={{-60,-164},{-60,-148}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,16}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
              points={{-22,48},{-10,48}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
              points={{-60,-50},{-60,-36}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
              points={{-60,-84},{-60,-70}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
              points={{-60,-116},{-60,-128}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-180},{-44,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,150},{-6.66134e-016,150}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,204},{100,190}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,170}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{96,-230},{160,-230},{160,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-280},{20,-280},{20,-310}},
              color={255,0,0},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,218},{52,218},{-40,218},{-40,224}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,218},{90,218}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{6,10},{6,10}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-220},{82,-220}},
              color={255,204,51},
              thickness=0.5));
          connect(MS_in, throughMassFlow.inlet) annotation (Line(
              points={{-100,310},{-100,310},{-100,266},{-100,266}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
              points={{-100,246},{-100,120},{-16,120}},
              color={255,0,0},
              thickness=0.5));
          connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
              points={{-100,246},{-100,230},{-46,230}},
              color={255,0,0},
              thickness=0.5));
          connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
              points={{0,200},{0,185},{0,170}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-270},{-2,-270}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,260},{-94,260}},
              color={255,204,51},
              thickness=0.5));
          connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
              points={{-42,48},{-50,48},{-60,48},{-60,-4}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
              points={{50,10},{50,10},{50,198},{50,192},{10,192}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
              points={{50,10},{50,10},{50,280},{8,280},{8,280}},
              color={255,204,51},
              thickness=0.5));
          connect(SH_out, valveVap.outlet) annotation (Line(
              points={{0,310},{0,300},{0,290}},
              color={0,0,255},
              thickness=0.5));
          connect(valveVap.inlet, sensP.flange) annotation (Line(
              points={{-6.66134e-016,270},{-6.66134e-016,235},{0,235},{0,200}},
              color={0,0,255},
              thickness=0.5));
          connect(HPH.inletHot, ECO.outletHot) annotation (Line(
              points={{-76,-280},{-100,-280},{-100,-180},{-76,-180}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(sensT_FW.outlet,PDSteamECO1. inlet) annotation (Line(
              points={{84,-230},{84,-230},{-2,-230},{-60,-230},{-60,-222}},
              color={0,0,255},
              thickness=0.5));
          connect(sensT_FW.outlet,PDSteamHPH. inlet) annotation (Line(
              points={{84,-230},{84,-230},{-60,-230},{-60,-236}},
              color={0,0,255},
              thickness=0.5));
          connect(PDSteamHPH.outlet,HPH. inletCold) annotation (Line(points={{-60,
                  -256},{-60,-256},{-60,-264}}, color={0,0,255}));
          connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-44,-280},{-16,-280}},
              color={255,0,0},
              thickness=0.5));
          connect(ECO.inletCold, PDSteamECO1.outlet)
            annotation (Line(points={{-60,-196},{-60,-202}}, color={0,0,255}));
          connect(HPH.outletCold, MS_HPH_out)
            annotation (Line(points={{-60,-296},{-60,-310}}, color={0,0,255}));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init20_20nodes;

        model OTSG_init4_5nodes_noDP
          "Complete dynamic model of the once through steam generator, with separate 2ph HTC"
          extends BaseClasses.BaseOTSG;

          //Heat exchangers parameter
          parameter Integer Nhex_mono = 5
            "Number of nodes for single phase heat exchangers";
          parameter Integer Nhex_bi = 10
            "Number of nodes for biphase heat exchangers";

          Components.HE ECO(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Shot=2394.2,
            Scold=1855.5,
            Mm=30395,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.728761, gamma_nom=4400),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.461561, gamma_nom=2200),
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            Vhot=45.011,
            Vcold=5.752,
            nominalColdPressure=17910000,
            nominalHotPressure=569600,
            nominalColdInletTemperature=553.15,
            nominalColdOutletTemperature=623.79,
            nominalHotInletTemperature=628.75,
            nominalHotOutletTemperature=586.72)
            annotation (Placement(transformation(extent={{-44,-226},{-76,-194}})));

          Components.HE_2ph
                        EVA1(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            Mm=43100,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2300, alpha=0.447498),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                gamma_nom_liq=3875,
                gamma_nom_2ph=25350,
                gamma_nom_vap=3875,
                alpha=1.6),
            nominalColdFlow=88.59,
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
            Vhot=26.587,
            Vcold=8.651,
            Shot=2251.9,
            Scold=1765.5,
            nominalColdPressure=17810000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=623.75,
            nominalColdOutletTemperature=629.32,
            nominalHotInletTemperature=688.04,
            nominalHotOutletTemperature=629.15) annotation (Placement(
                transformation(extent={{-76,-116},{-44,-84}})));

          Components.HE_2ph
                        EVA2(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Nhex=Nhex_bi,
            nominalHotFlow=610.6,
            nominalColdFlow=88.59,
            Mm=5182,
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.389637, gamma_nom=2650),
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                (
                wnom_ht=88.59,
                alpha=1.6,
                gamma_nom_2ph=6000,
                gamma_nom_liq=2600,
                gamma_nom_vap=2600),
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
            Vhot=3.905,
            Vcold=1.065,
            Shot=429.2,
            Scold=338,
            nominalColdPressure=17760000,
            nominalHotPressure=730800,
            nominalColdInletTemperature=629.15,
            nominalColdOutletTemperature=648.65,
            nominalHotInletTemperature=732.05,
            nominalHotOutletTemperature=687.95) annotation (Placement(
                transformation(extent={{-44,-36},{-76,-4}})));

          Components.Equilibrium2phVessel separator(
            redeclare package Medium = coldMedium,
            H=8,
            zl_min=0,
            zl_max=7,
            zl_start=0,
            h_start=2.7e6,
            D=0.56,
            p_start=17710000,
            initOpt=ThermoPower.Choices.Init.Options.fixedState)
            annotation (Placement(transformation(extent={{-28,28},{28,84}})));
          ThermoPower.Water.ThroughMassFlow waterPump(
            redeclare package Medium = coldMedium,
            allowFlowReversal=false,
            use_in_w0=true,
            w0=10) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={0,4})));

          Components.HE SH(
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            Mm=15353,
            nominalColdFlow=88.59,
            nominalHotFlow=371.8,
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            redeclare package hotMedium = hotMedium,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2200, alpha=0.847436),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=3000, alpha=0.28867),
            Nhex=Nhex_mono,
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
            Vhot=3.68,
            Vcold=2.683,
            Shot=1153.8,
            Scold=879.8,
            nominalColdPressure=17710000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=648.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=732.02)
            annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                rotation=0,
                origin={0,120})));

          Components.HE RH(
            redeclare model HeatExchangerTopology =
                ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
            cm=500,
            nominalColdDeltaP=0,
            nominalHotDeltaP=0,
            redeclare package coldMedium = coldMedium,
            redeclare package hotMedium = hotMedium,
            Mm=10081,
            nominalColdFlow=82.57,
            nominalHotFlow=238.7,
            Shot=1339,
            Scold=1198,
            Nhex=Nhex_mono,
            redeclare model HeatTransferCold =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (alpha=0.798347, gamma_nom=970),
            redeclare model HeatTransferHot =
                ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                (gamma_nom=2600, alpha=0.320782),
            coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
            hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
            Vhot=1.522,
            Vcold=5.092,
            nominalColdPressure=4698000,
            nominalHotPressure=1000000,
            nominalColdInletTemperature=625.65,
            nominalColdOutletTemperature=826.06,
            nominalHotInletTemperature=838.15,
            nominalHotOutletTemperature=731.45)
            annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                rotation=-90,
                origin={140,160})));

          ThermoPower.Water.ThroughMassFlow valveRH(
            redeclare package Medium = hotMedium,
            w0=238.7,
            use_in_w0=true)
            annotation (Placement(transformation(extent={{-46,270},{-26,250}})));
          ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={96,230})));

          //Overall parameter and variables:
          Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
          Modelica.SIunits.Temperature T_inFW = sensT_FW.T;
          Modelica.SIunits.Temperature T_outMS = sensT_MS.T;
          Modelica.SIunits.Temperature T_RH = sensT_RH.T;
          Modelica.SIunits.Length sep_lev = separator.zl;
          Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
          Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
          Real x_EVA1_out = EVA1.coldFluid.x[end];
          Real x_EVA2_out = EVA2.coldFluid.x[end];
          Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
          Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
          Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
          Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

          ThermoPower.Water.PressDrop PDSteamSH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            dpnom=(177.1 - 176.1189)*10^5,
            rhonom=96.586,
            wnom=88.59)                    annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={0,188})));
          ThermoPower.Water.PressDrop PDSteamRH(
            FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
            redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
            wnom=82.57,
            dpnom=(0.55113)*10^5,
            rhonom=17.86)                  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={100,190})));
          Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={50,10}), iconTransformation(
                extent={{-30,-30},{30,30}},
                rotation=-90,
                origin={-194,0})));
          ThermoPower.Water.SensT sensT_MS(
            redeclare package Medium = hotMedium)
            annotation (Placement(transformation(extent={{-20,-276},{0,-256}})));
          ThermoPower.Water.SensT sensT_FW(
            redeclare package Medium = coldMedium)
            annotation (Placement(transformation(extent={{102,-246},{82,-226}})));
          ThermoPower.Water.SensP sensP_ECO(
            redeclare package Medium = coldMedium) annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={-56,-140})));
        equation
          connect(RH.inletCold, RH_in) annotation (Line(
              points={{156,160},{190,160}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(valveRH.outlet, RH.inletHot) annotation (Line(
              points={{-26,260},{140,260},{140,176}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(MS_in, valveRH.inlet) annotation (Line(
              points={{-100,310},{-100,260},{-46,260}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH_out, sensT_RH.outlet) annotation (Line(
              points={{100,310},{100,236}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(MS_in, SH.inletHot) annotation (Line(
              points={{-100,310},{-100,120},{-16,120}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(separator.drained_out, waterPump.inlet) annotation (Line(
              points={{0,38},{0,14},{1.77636e-015,14}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.inletCold, separator.steam_out) annotation (Line(
              points={{0,104},{0,80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(RH.outletHot, EVA2.inletHot) annotation (Line(
              points={{140,144},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletHot, EVA2.inletHot) annotation (Line(
              points={{16,120},{140,120},{140,-20},{-44,-20}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
              points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
              points={{-44,-100},{-20,-100},{-20,-210},{-44,-210}},
              color={255,0,0},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(waterPump.outlet, SEP_out) annotation (Line(
              points={{-1.77636e-015,-6},{-1.77636e-015,-80},{190,-80}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
              points={{0,136},{0,178}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(SH_out, PDSteamSH.outlet) annotation (Line(
              points={{0,310},{0,198}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
              points={{100,224},{100,200}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));
          connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
              points={{124,160},{100,160},{100,180}},
              color={0,0,255},
              smooth=Smooth.None,
              thickness=0.5));

          connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,-20},{50,8},{6,8}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
              points={{50,10},{50,10},{50,246},{50,244},{-40,244},{-40,254}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
              points={{50,10},{50,10},{50,42},{10,42}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
              points={{50,10},{50,10},{50,244},{90,244},{90,238}},
              color={255,204,51},
              thickness=0.5));
          connect(sensT_MS.outlet, MS_out) annotation (Line(
              points={{-4,-270},{20,-270},{20,-310}},
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled},
              color={255,0,0}));
          connect(sensT_FW.inlet, FW_in) annotation (Line(
              points={{98,-240},{98,-240},{190,-240}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.Filled,Arrow.None}));
          connect(sensT_FW.outlet, ECO.inletCold) annotation (Line(
              points={{86,-240},{86,-240},{-60,-240},{-60,-226}},
              color={0,0,255},
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
              points={{50,10},{50,10},{50,-200},{50,-230},{84,-230}},
              color={255,204,51},
              thickness=0.5));
          connect(signalBus.valveBP_PV, sensT_MS.T) annotation (Line(
              points={{50,10},{50,10},{50,-260},{-2,-260}},
              color={255,204,51},
              thickness=0.5));
          connect(EVA1.inletCold, sensP_ECO.flange) annotation (Line(
              points={{-60,-116},{-60,-140}},
              color={0,0,255},
              thickness=0.5));
          connect(signalBus.pumpFW_PV, sensP_ECO.p) annotation (Line(
              points={{50,10},{50,10},{50,-158},{-50,-158},{-50,-148}},
              color={255,204,51},
              thickness=0.5));
          connect(EVA2.inletCold, EVA1.outletCold)
            annotation (Line(points={{-60,-36},{-60,-84}},           color={0,0,255}));
          connect(sensP_ECO.flange, ECO.outletCold) annotation (Line(points={{-60,
                  -140},{-60,-140},{-60,-194}},      color={0,0,255}));
          connect(EVA2.outletCold, separator.water_in) annotation (Line(points={{-60,-4},
                  {-60,-4},{-60,48},{-10,48}},          color={0,0,255}));
          connect(ECO.outletHot, sensT_MS.inlet) annotation (Line(
              points={{-76,-210},{-100,-210},{-100,-270},{-16,-270}},
              color={255,0,0},
              thickness=0.5,
              arrow={Arrow.None,Arrow.Filled}));
          annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(
                  extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
        end OTSG_init4_5nodes_noDP;

        package Test

          package HE "Package of single heat exchanger test model"

            model test_heatExchangers
              "Test model of all the heat exchangers, at nominal point, steady-state initialisation"

              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                w0=371.8,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322,
                use_in_w0=false,
                p0=1000000) annotation (Placement(transformation(extent={{-200,
                        80},{-180,100}})));

              ThermoPower.Water.SourceMassFlow sourceSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=88.59,
                h=2745621,
                p0=17710000)
                annotation (Placement(transformation(extent={{-200,60},{-180,80}})));
              ThermoPower.Water.SinkPressure sinkSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17611890)
                annotation (Placement(transformation(extent={{-140,120},{-120,140}})));
              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=839319) annotation (Placement(transformation(extent={{-110,
                        80},{-90,100}})));

              ThermoPower.Water.PressDrop PDMSSH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=371.8,
                dpnom=(10 - 8.39319)*10^5,
                rhonom=1730.7,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-140,80},{-120,
                        100}})));
              ThermoPower.Water.PressDrop PDSteamSH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                dpnom=(177.1 - 176.1189)*10^5,
                rhonom=96.586,
                wnom=88.59)                    annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-160,120})));
              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
                annotation (Placement(transformation(extent={{220,160},{240,180}})));
              Components.HE SH(
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                Mm=15353,
                nominalColdFlow=88.59,
                nominalHotFlow=371.8,
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                Shot=1153.8,
                Scold=879.8,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3000, alpha=0.28867),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2200, alpha=0.847436),
                generateInitCode=true,
                Nhex=5,
                Vhot=3.68,
                Vcold=2.683,
                nominalColdPressure=17710000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=648.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=732.02)
                annotation (Placement(transformation(extent={{-170,80},{-150,100}})));

              ThermoPower.Water.SourceMassFlow sourceMS1(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=238.7,
                h=518322,
                p0=1000000) annotation (Placement(transformation(extent={{-40,
                        80},{-20,100}})));

              ThermoPower.Water.SourceMassFlow sourceSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=82.57,
                h=3083198,
                p0=4698000)
                annotation (Placement(transformation(extent={{-40,60},{-20,80}})));
              ThermoPower.Water.SinkPressure sinkSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=4642887)
                annotation (Placement(transformation(extent={{20,120},{40,140}})));
              ThermoPower.Water.SinkPressure sinkMS1(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=878222) annotation (Placement(transformation(extent={{50,80},
                        {70,100}})));

              ThermoPower.Water.PressDrop PDMSRH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                rhonom=1730.47,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_lin,
                wnom=238.7,
                dpnom=(10 - 8.78222)*10^5) annotation (Placement(transformation(
                      extent={{20,80},{40,100}})));

              ThermoPower.Water.PressDrop PDSteamRH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=82.57,
                dpnom=(0.55113)*10^5,
                rhonom=17.86)                  annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={0,120})));
              Components.HE RH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                Mm=10081,
                nominalColdFlow=82.57,
                nominalHotFlow=238.7,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                Shot=1339,
                Scold=1198,
                Nhex=5,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.798347, gamma_nom=970),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2600, alpha=0.320782),
                generateInitCode=true,
                Vhot=1.522,
                Vcold=5.092,
                nominalColdPressure=4698000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=625.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=731.45)
                annotation (Placement(transformation(extent={{-10,80},{10,100}})));

              ThermoPower.Water.SourceMassFlow sourceMS2(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=199579,
                p0=569600) annotation (Placement(transformation(extent={{120,80},
                        {140,100}})));

              ThermoPower.Water.SourceMassFlow sourceSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1231860,
                w0=88.59,
                p0=17910000)
                annotation (Placement(transformation(extent={{120,60},{140,80}})));
              ThermoPower.Water.SinkPressure sinkSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17857510)
                annotation (Placement(transformation(extent={{180,120},{200,140}})));
              ThermoPower.Water.SinkPressure sinkMS2(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=404671) annotation (Placement(transformation(extent={{210,80},
                        {230,100}})));

              ThermoPower.Water.PressDrop PDMSECO(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(1.64929)*10^5,
                rhonom=1863.84) annotation (Placement(transformation(extent={{
                        180,80},{200,100}})));

              ThermoPower.Water.PressDrop PDSteamECO(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.59,
                dpnom=(0.52492)*10^5,
                rhonom=768.15)                 annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={160,120})));
              Components.HE ECO(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.728761, gamma_nom=4400),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.461561, gamma_nom=2200),
                Shot=2394.2,
                Scold=1855.5,
                Mm=30395,
                generateInitCode=true,
                Nhex=5,
                Vhot=45.011,
                Vcold=5.752,
                nominalColdPressure=17910000,
                nominalHotPressure=569600,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=623.79,
                nominalHotInletTemperature=628.75,
                nominalHotOutletTemperature=586.72)
                annotation (Placement(transformation(extent={{150,80},{170,100}})));
              ThermoPower.Water.SourceMassFlow sourceMS3(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=136706,
                p0=400000) annotation (Placement(transformation(extent={{-200,-100},
                        {-180,-80}})));

              ThermoPower.Water.SourceMassFlow sourceSteam3(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1231860,
                w0=143.1,
                p0=17910000)
                annotation (Placement(transformation(extent={{-200,-120},{-180,-100}})));
              ThermoPower.Water.SinkPressure sinkSteam3(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17814720)
                annotation (Placement(transformation(extent={{-140,-60},{-120,-40}})));
              ThermoPower.Water.SinkPressure sinkMS3(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=261078) annotation (Placement(transformation(extent={{-110,-100},
                        {-90,-80}})));

              ThermoPower.Water.PressDrop PDMSHPH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(1.38922)*10^5,
                rhonom=1890.49) annotation (Placement(transformation(extent={{-140,
                        -100},{-120,-80}})));

              ThermoPower.Water.PressDrop PDSteamHPH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                rhonom=768.15,
                wnom=143.1,
                dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-160,-60})));
              Components.HE HPH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                Shot=2429.5,
                Scold=1913.2,
                Mm=29335,
                nominalColdFlow=143.1,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=5000, alpha=0.536143),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2000, alpha=0.447465),
                generateInitCode=true,
                Nhex=5,
                Vhot=33.634,
                Vcold=6.027,
                nominalColdPressure=17910000,
                nominalHotPressure=400000,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=581.85,
                nominalHotInletTemperature=586.85,
                nominalHotOutletTemperature=563.15) annotation (Placement(
                    transformation(extent={{-170,-100},{-150,-80}})));

              ThermoPower.Water.SourceMassFlow sourceMS4(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=289063,
                p0=730800) annotation (Placement(transformation(extent={{-40,-100},
                        {-20,-80}})));

              ThermoPower.Water.SourceMassFlow sourceSteam4(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=88.55,
                p0=17810000,
                h=1665608)
                annotation (Placement(transformation(extent={{-40,-120},{-20,-100}})));
              ThermoPower.Water.SinkPressure sinkSteam4(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17804170)
                annotation (Placement(transformation(extent={{20,-60},{40,-40}})));
              ThermoPower.Water.SinkPressure sinkMS4(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=605957) annotation (Placement(transformation(extent={{50,-100},
                        {70,-80}})));

              ThermoPower.Water.PressDrop PDMSEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(1.24843)*10^5,
                rhonom=1826.13) annotation (Placement(transformation(extent={{
                        20,-100},{40,-80}})));

              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.55,
                dpnom=(0.05828)*10^5,
                rhonom=582.24)       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={0,-60})));
              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                Mm=43100,
                Shot=2251.9,
                Scold=1765.5,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=3875,
                    gamma_nom_2ph=25350,
                    gamma_nom_vap=3875,
                    alpha=1.6),
                nominalColdFlow=88.59,
                generateInitCode=true,
                Nhex=10,
                Vhot=26.587,
                Vcold=8.651,
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15) annotation (Placement(
                    transformation(extent={{-10,-100},{10,-80}})));
              ThermoPower.Water.SourceMassFlow sourceMS5(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=355877,
                p0=730800) annotation (Placement(transformation(extent={{120,-100},
                        {140,-80}})));

              ThermoPower.Water.SourceMassFlow sourceSteam5(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=88.59,
                h=2279896,
                p0=17760000)
                annotation (Placement(transformation(extent={{120,-120},{140,-100}})));
              ThermoPower.Water.SinkPressure sinkSteam5(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17748830)
                annotation (Placement(transformation(extent={{180,-60},{200,-40}})));
              ThermoPower.Water.SinkPressure sinkMS5(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=696272) annotation (Placement(transformation(extent={{210,-100},
                        {230,-80}})));

              ThermoPower.Water.PressDrop PDMSEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(0.34528)*10^5,
                rhonom=1798.12) annotation (Placement(transformation(extent={{
                        180,-100},{200,-80}})));

              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=250)          annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={160,-60})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    alpha=1.6,
                    gamma_nom_2ph=6000,
                    gamma_nom_liq=2600,
                    gamma_nom_vap=2600),
                Shot=429.2*0.90,
                Scold=338*0.90,
                generateInitCode=true,
                Nhex=10,
                Vhot=3.905,
                Vcold=1.065,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95) annotation (Placement(
                    transformation(extent={{150,-100},{170,-80}})));
            equation
              connect(sinkMS.flange, PDMSSH.outlet) annotation (Line(
                  points={{-110,90},{-120,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamSH.outlet, sinkSteam.flange) annotation (Line(
                  points={{-160,130},{-140,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam.flange, SH.inletCold) annotation (Line(
                  points={{-180,70},{-160,70},{-160,80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS.flange, SH.inletHot) annotation (Line(
                  points={{-180,90},{-170,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(SH.outletHot, PDMSSH.inlet) annotation (Line(
                  points={{-150,90},{-140,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
                  points={{-160,110},{-160,100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS1.flange, PDMSRH.outlet) annotation (Line(
                  points={{50,90},{40,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamRH.outlet, sinkSteam1.flange) annotation (Line(
                  points={{0,130},{20,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam1.flange, RH.inletCold) annotation (Line(
                  points={{-20,70},{0,70},{0,80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS1.flange, RH.inletHot) annotation (Line(
                  points={{-20,90},{-10,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(RH.outletHot,PDMSRH. inlet) annotation (Line(
                  points={{10,90},{20,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamRH.inlet,RH. outletCold) annotation (Line(
                  points={{0,110},{0,100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS2.flange, PDMSECO.outlet) annotation (Line(
                  points={{210,90},{200,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamECO.outlet, sinkSteam2.flange) annotation (Line(
                  points={{160,130},{180,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam2.flange, ECO.inletCold) annotation (Line(
                  points={{140,70},{160,70},{160,80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS2.flange, ECO.inletHot) annotation (Line(
                  points={{140,90},{150,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(ECO.outletHot, PDMSECO.inlet) annotation (Line(
                  points={{170,90},{180,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamECO.inlet, ECO.outletCold) annotation (Line(
                  points={{160,110},{160,100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS3.flange, PDMSHPH.outlet) annotation (Line(
                  points={{-110,-90},{-120,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamHPH.outlet, sinkSteam3.flange) annotation (Line(
                  points={{-160,-50},{-140,-50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam3.flange, HPH.inletCold) annotation (Line(
                  points={{-180,-110},{-160,-110},{-160,-100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS3.flange, HPH.inletHot) annotation (Line(
                  points={{-180,-90},{-170,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(HPH.outletHot, PDMSHPH.inlet) annotation (Line(
                  points={{-150,-90},{-140,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamHPH.inlet, HPH.outletCold) annotation (Line(
                  points={{-160,-70},{-160,-80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS4.flange,PDMSEVA1. outlet) annotation (Line(
                  points={{50,-90},{40,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA1.outlet,sinkSteam4. flange) annotation (Line(
                  points={{0,-50},{20,-50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam4.flange,EVA1. inletCold) annotation (Line(
                  points={{-20,-110},{0,-110},{0,-100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS4.flange,EVA1. inletHot) annotation (Line(
                  points={{-20,-90},{-10,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA1.outletHot,PDMSEVA1. inlet) annotation (Line(
                  points={{10,-90},{20,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA1.inlet,EVA1. outletCold) annotation (Line(
                  points={{0,-70},{0,-80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS5.flange,PDMSEVA2. outlet) annotation (Line(
                  points={{210,-90},{200,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA2.outlet,sinkSteam5. flange) annotation (Line(
                  points={{160,-50},{180,-50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam5.flange,EVA2. inletCold) annotation (Line(
                  points={{140,-110},{160,-110},{160,-100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS5.flange,EVA2. inletHot) annotation (Line(
                  points={{140,-90},{150,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.outletHot,PDMSEVA2. inlet) annotation (Line(
                  points={{170,-90},{180,-90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA2.inlet,EVA2. outletCold) annotation (Line(
                  points={{160,-70},{160,-80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-240,
                        -180},{240,180}}),        graphics={Text(
                      extent={{-160,70},{-120,50}},
                      lineColor={0,0,255},
                      textString="SH"), Text(
                      extent={{0,70},{40,50}},
                      lineColor={0,0,255},
                      textString="RH"), Text(
                      extent={{160,70},{200,50}},
                      lineColor={0,0,255},
                      textString="ECO"),Text(
                      extent={{-160,-110},{-120,-130}},
                      lineColor={0,0,255},
                      textString="HPH"),Text(
                      extent={{0,-110},{40,-130}},
                      lineColor={0,0,255},
                      textString="EVA1"),
                                        Text(
                      extent={{160,-110},{200,-130}},
                      lineColor={0,0,255},
                      textString="EVA2")}),
                                          Icon(coordinateSystem(extent={{-240,-180},{
                        240,180}})));
            end test_heatExchangers;

            model test_heatExchangers_offDesign
              "Test model of all the heat exchangers, from nominal load to 20% load, steady-state initialisation"

              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                w0=371.8,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322,
                p0=1000000,
                use_in_w0=true) annotation (Placement(transformation(extent={{-250,
                        120},{-230,140}})));

              ThermoPower.Water.SourceMassFlow sourceSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=88.59,
                h=2745621,
                p0=17710000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{-250,20},{-230,40}})));
              ThermoPower.Water.SinkPressure sinkSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17611890,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{-190,180},{-170,200}})));
              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=839319) annotation (Placement(transformation(extent={{-160,
                        120},{-140,140}})));

              ThermoPower.Water.PressDrop PDMSSH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=371.8,
                dpnom=(10 - 8.39319)*10^5,
                rhonom=1730.7,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-190,120},{-170,
                        140}})));
              ThermoPower.Water.PressDrop PDSteamSH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                dpnom=(177.1 - 176.1189)*10^5,
                rhonom=96.586,
                wnom=88.59)                    annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-210,160})));
              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
                annotation (Placement(transformation(extent={{-340,240},{-320,260}})));
              Components.HE SH(
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                Mm=15353,
                nominalColdFlow=88.59,
                nominalHotFlow=371.8,
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                Shot=1153.8,
                Scold=879.8,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3000, alpha=0.28867),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2200, alpha=0.847436),
                generateInitCode=true,
                Nhex=20,
                Vhot=3.68,
                Vcold=2.683,
                nominalColdPressure=17710000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=648.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=732.02)
                annotation (Placement(transformation(extent={{-220,120},{-200,140}})));

              ThermoPower.Water.SourceMassFlow sourceMS1(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=238.7,
                h=518322,
                use_in_w0=true,
                p0=1000000,
                use_in_h=false) annotation (Placement(transformation(extent={{-40,
                        120},{-20,140}})));

              ThermoPower.Water.SourceMassFlow sourceSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=82.57,
                h=3083198,
                p0=4698000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{-40,20},{-20,40}})));
              ThermoPower.Water.SinkPressure sinkSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=4642887,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{20,178},{40,198}})));
              ThermoPower.Water.SinkPressure sinkMS1(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=878222) annotation (Placement(transformation(extent={{50,120},
                        {70,140}})));

              ThermoPower.Water.PressDrop PDMSRH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                rhonom=1730.47,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_lin,
                wnom=238.7,
                dpnom=(10 - 8.78222)*10^5) annotation (Placement(transformation(
                      extent={{20,120},{40,140}})));

              ThermoPower.Water.PressDrop PDSteamRH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=82.57,
                dpnom=(0.55113)*10^5,
                rhonom=17.86)                  annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={0,160})));
              Components.HE RH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                Mm=10081,
                nominalColdFlow=82.57,
                nominalHotFlow=238.7,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                Shot=1339,
                Scold=1198,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.798347, gamma_nom=970),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2600, alpha=0.320782),
                generateInitCode=true,
                Nhex=20,
                Vhot=1.522,
                Vcold=5.092,
                nominalColdPressure=4698000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=625.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=731.45)
                annotation (Placement(transformation(extent={{-10,120},{10,140}})));

              ThermoPower.Water.SourceMassFlow sourceMS2(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=199579,
                p0=569600,
                use_in_w0=true,
                use_in_h=true) annotation (Placement(transformation(extent={{
                        170,120},{190,140}})));

              ThermoPower.Water.SourceMassFlow sourceSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1231860,
                w0=88.59,
                p0=17910000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{172,20},{192,40}})));
              ThermoPower.Water.SinkPressure sinkSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17857510,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{230,178},{250,198}})));
              ThermoPower.Water.SinkPressure sinkMS2(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=404671) annotation (Placement(transformation(extent={{260,
                        120},{280,140}})));

              ThermoPower.Water.PressDrop PDMSECO(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(1.64929)*10^5,
                rhonom=1863.84) annotation (Placement(transformation(extent={{
                        230,120},{250,140}})));

              ThermoPower.Water.PressDrop PDSteamECO(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.59,
                dpnom=(0.52492)*10^5,
                rhonom=768.15)                 annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={210,160})));
              Components.HE ECO(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.728761, gamma_nom=4400),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.461561, gamma_nom=2200),
                Shot=2394.2,
                Scold=1855.5,
                Mm=30395,
                generateInitCode=true,
                Nhex=20,
                Vhot=45.011,
                Vcold=5.752,
                nominalColdPressure=17910000,
                nominalHotPressure=569600,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=623.79,
                nominalHotInletTemperature=628.75,
                nominalHotOutletTemperature=586.72)
                annotation (Placement(transformation(extent={{200,120},{220,140}})));

              ThermoPower.Water.SourceMassFlow sourceSteam4(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=88.55,
                h=1665608,
                p0=17810000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{-100,-220},{-80,-200}})));
              ThermoPower.Water.SinkPressure sinkMS4(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=605957) annotation (Placement(transformation(extent={{-120,-120},
                        {-140,-100}})));

              ThermoPower.Water.PressDrop PDMSEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(1.24843)*10^5,
                rhonom=1826.13) annotation (Placement(transformation(extent={{-80,
                        -120},{-100,-100}})));

              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.55,
                dpnom=(0.05828)*10^5,
                rhonom=582.24)       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={0,-60})));
              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                Mm=43100,
                Shot=2251.9,
                Scold=1765.5,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                nominalColdFlow=88.59,
                generateInitCode=true,
                Nhex=40,
                Vhot=26.587,
                Vcold=8.651,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=3875,
                    gamma_nom_2ph=25350,
                    gamma_nom_vap=3875,
                    alpha=1.6),
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15) annotation (Placement(
                    transformation(extent={{-50,-120},{-70,-100}})));
              ThermoPower.Water.SourceMassFlow sourceMS5(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=355877,
                p0=730800,
                use_in_w0=true,
                use_in_h=true) annotation (Placement(transformation(extent={{
                        120,-120},{100,-100}})));

              ThermoPower.Water.SinkPressure sinkSteam5(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17748830,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{90,-220},{110,-200}})));
              ThermoPower.Water.PressDrop PDMSEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(0.34528)*10^5,
                rhonom=1798.12) annotation (Placement(transformation(extent={{
                        10,-120},{-10,-100}})));

              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=250)          annotation (Placement(transformation(
                    extent={{10,-10},{-10,10}},
                    rotation=90,
                    origin={70,-150})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    alpha=1.6,
                    gamma_nom_2ph=6000,
                    gamma_nom_liq=2600,
                    gamma_nom_vap=2600),
                generateInitCode=true,
                Nhex=40,
                Vhot=3.905,
                Vcold=1.065,
                Shot=429.2,
                Scold=338,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95) annotation (Placement(
                    transformation(extent={{80,-100},{60,-120}})));
              Modelica.Blocks.Tables.CombiTable1D w_MS_SH(
                               smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    371.8; 1e6,354.2; 2e6,329.7; 3e6,300.5; 4e6,263.9; 5e6,226.7; 6e6,
                    186.6; 7e6,144.2; 8e6,99.4; 10e6,99.4])
                annotation (Placement(transformation(extent={{-270,150},{-250,170}})));
              Modelica.Blocks.Sources.RealExpression realExpression(y=time)
                annotation (Placement(transformation(extent={{-310,150},{-290,170}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_SH(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    2745621; 1e6,2941653; 2e6,2899377; 3e6,2908498; 4e6,2828363; 5e6,
                    2723640; 6e6,2724375; 7e6,2725107; 8e6,2725290; 10e6,0.0])
                annotation (Placement(transformation(extent={{-270,80},{-250,100}})));
              Modelica.Blocks.Sources.RealExpression realExpression1(y=time)
                annotation (Placement(transformation(extent={{-310,80},{-290,100}})));
              Modelica.Blocks.Tables.CombiTable1D w_ST_SH(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    88.59; 1e6,81.03; 2e6,72.9; 3e6,64.64; 4e6,56.19; 5e6,46.02; 6e6,
                    36.01; 7e6,26.36; 8e6,17.04; 10e6,17.04])
                annotation (Placement(transformation(extent={{-270,50},{-250,70}})));
              Modelica.Blocks.Sources.RealExpression realExpression2(y=time)
                annotation (Placement(transformation(extent={{-310,50},{-290,70}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_SH(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    176.1e5; 1e6,161.1e5; 2e6,144.9e5; 3e6,128.5e5; 4e6,112.1e5; 5e6,
                    100.5e5; 6e6,100.3e5; 7e6,100.3e5; 8e6,100.1e5; 10e6,100.1e5])
                                                             annotation (Placement(
                    transformation(extent={{-230,210},{-210,230}})));
              Modelica.Blocks.Sources.RealExpression realExpression3(y=time)
                annotation (Placement(transformation(extent={{-270,210},{-250,230}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_RH(table=[0,46.4e5; 1e6,
                    40.98e5; 2e6,35.77e5; 3e6,30.66e5; 4e6,26.32e5; 5e6,21.61e5; 6e6,
                    16.92e5; 7e6,12.28e5; 8e6,7.67e5; 10e6,22e5], smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative)
                annotation (Placement(transformation(extent={{-20,210},{0,230}})));
              Modelica.Blocks.Sources.RealExpression realExpression4(y=time)
                annotation (Placement(transformation(extent={{-60,210},{-40,230}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_RH(table=[0,3083198; 1e6,
                    3093346; 2e6,3100618; 3e6,3108752; 4e6,3116961; 5e6,3125327; 6e6,
                    3128892; 7e6,3130270; 8e6,3124593; 10e6,3092400], smoothness=
                    Modelica.Blocks.Types.Smoothness.ContinuousDerivative)
                annotation (Placement(transformation(extent={{-60,80},{-40,100}})));
              Modelica.Blocks.Sources.RealExpression realExpression6(y=time)
                annotation (Placement(transformation(extent={{-100,80},{-80,100}})));
              Modelica.Blocks.Tables.CombiTable1D w_ST_RH(table=[0,82.57; 1e6,72.70;
                    2e6,63.31; 3e6,54.15; 4e6,46.38; 5e6,38.01; 6e6,29.69; 7e6,21.50;
                    8e6,13.40; 10e6,4.18], smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative)
                annotation (Placement(transformation(extent={{-60,50},{-40,70}})));
              Modelica.Blocks.Sources.RealExpression realExpression7(y=time)
                annotation (Placement(transformation(extent={{-100,50},{-80,70}})));
              Modelica.Blocks.Tables.CombiTable1D w_MS_RH(table=[0,238.7; 1e6,195.2;
                    2e6,158.7; 3e6,126.9; 4e6,102.4; 5e6,78.6; 6e6,57.6; 7e6,39; 8e6,
                    22.7; 10e6,4], smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative)
                annotation (Placement(transformation(extent={{-60,150},{-40,170}})));
              Modelica.Blocks.Sources.RealExpression realExpression8(y=time)
                annotation (Placement(transformation(extent={{-100,150},{-80,170}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_ECO(
                                                          smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    178.1e5; 1e6,163.1e5; 2e6,146.7e5; 3e6,130.1e5; 4e6,113.4e5; 5e6,
                    102.1e5; 6e6,101.5e5; 7e6,100.9e5; 8e6,100.5e5; 10e6,100.5e5])
                annotation (Placement(transformation(extent={{190,210},{210,230}})));
              Modelica.Blocks.Sources.RealExpression realExpression9(y=time)
                annotation (Placement(transformation(extent={{150,210},{170,230}})));
              Modelica.Blocks.Tables.CombiTable1D h_MS_ECO(
                               smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    200329; 1e6,193208; 2e6,179571; 3e6,164722; 4e6,148509; 5e6,
                    119114; 6e6,107296; 7e6,98811; 8e6,93204; 10e6,93204])
                annotation (Placement(transformation(extent={{150,180},{170,200}})));
              Modelica.Blocks.Sources.RealExpression realExpression10(
                                                                     y=time)
                annotation (Placement(transformation(extent={{110,180},{130,200}})));
              Modelica.Blocks.Tables.CombiTable1D w_MS_ECO(
                                                          smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    610.6; 1e6,549.4; 2e6,488.4; 3e6,427.4; 4e6,366.3; 5e6,305.3; 6e6,
                    244.2; 7e6,183.2; 8e6,122.1; 10e6,122.1])
                annotation (Placement(transformation(extent={{150,150},{170,170}})));
              Modelica.Blocks.Sources.RealExpression realExpression11(
                                                                     y=time)
                annotation (Placement(transformation(extent={{110,150},{130,170}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_ECO(
                                                          smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    1231860; 1e6,1182452; 2e6,1133837; 3e6,1126638; 4e6,1111212; 5e6,
                    1251884; 6e6,1258763; 7e6,1260394; 8e6,1258351; 10e6,1258351])
                annotation (Placement(transformation(extent={{150,80},{170,100}})));
              Modelica.Blocks.Sources.RealExpression realExpression12(
                                                                     y=time)
                annotation (Placement(transformation(extent={{110,80},{130,100}})));
              Modelica.Blocks.Tables.CombiTable1D w_ST_ECO(
                                                          smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    88.59; 1e6,81.03; 2e6,72.9; 3e6,64.64; 4e6,56.19; 5e6,99.85; 6e6,
                    95.83; 7e6,83.77; 8e6,64.19; 10e6,64.19])
                annotation (Placement(transformation(extent={{150,50},{170,70}})));
              Modelica.Blocks.Sources.RealExpression realExpression13(
                                                                     y=time)
                annotation (Placement(transformation(extent={{110,50},{130,70}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_EVA(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    177.1e5; 1e6,162e5; 2e6,145.7e5; 3e6,129.3e5; 4e6,112.7e5; 5e6,
                    101e5; 6e6,100.6e5; 7e6,100.3e5; 8e6,100.1e5; 10e6,100.1e5])
                annotation (Placement(transformation(extent={{140,-190},{120,-170}})));
              Modelica.Blocks.Sources.RealExpression realExpression18(
                                                                     y=time)
                annotation (Placement(transformation(extent={{180,-190},{160,-170}})));
              Modelica.Blocks.Tables.CombiTable1D h_MS_EVA(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    356849; 1e6,382911; 2e6,374426; 3e6,373365; 4e6,357758; 5e6,
                    344122; 6e6,347304; 7e6,351243; 8e6,356546; 10e6,356546])
                annotation (Placement(transformation(extent={{140,-60},{120,-40}})));
              Modelica.Blocks.Sources.RealExpression realExpression19(
                                                                     y=time)
                annotation (Placement(transformation(extent={{180,-60},{160,-40}})));
              Modelica.Blocks.Tables.CombiTable1D w_MS_EVA(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    610.6; 1e6,549.4; 2e6,488.4; 3e6,427.4; 4e6,366.3; 5e6,305.3; 6e6,
                    244.2; 7e6,183.2; 8e6,122.1; 10e6,122.1])
                annotation (Placement(transformation(extent={{140,-90},{120,-70}})));
              Modelica.Blocks.Sources.RealExpression realExpression20(
                                                                     y=time)
                annotation (Placement(transformation(extent={{180,-90},{160,-70}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_EVA(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    1665608; 1e6,1656900; 2e6,1594900; 3e6,1531800; 4e6,1465700; 5e6,
                    1330000; 6e6,1293900; 7e6,1272400; 8e6,1258351; 10e6,1258351])
                annotation (Placement(transformation(extent={{-120,-160},{-100,-140}})));
              Modelica.Blocks.Sources.RealExpression realExpression21(
                                                                     y=time)
                annotation (Placement(transformation(extent={{-160,-160},{-140,-140}})));
              Modelica.Blocks.Tables.CombiTable1D w_ST_EVA(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[0,
                    88.59; 1e6,81.03; 2e6,72.9; 3e6,64.64; 4e6,56.19; 5e6,99.85; 6e6,
                    95.83; 7e6,83.77; 8e6,64.19; 10e6,64.19])
                annotation (Placement(transformation(extent={{-120,-190},{-100,-170}})));
              Modelica.Blocks.Sources.RealExpression realExpression22(
                                                                     y=time)
                annotation (Placement(transformation(extent={{-160,-190},{-140,-170}})));

              Real load = sourceMS2.w/sourceMS2.w0;
            equation
              connect(sinkMS.flange, PDMSSH.outlet) annotation (Line(
                  points={{-160,130},{-170,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamSH.outlet, sinkSteam.flange) annotation (Line(
                  points={{-210,170},{-210,190},{-190,190}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam.flange, SH.inletCold) annotation (Line(
                  points={{-230,30},{-210,30},{-210,120}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS.flange, SH.inletHot) annotation (Line(
                  points={{-230,130},{-220,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(SH.outletHot, PDMSSH.inlet) annotation (Line(
                  points={{-200,130},{-190,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
                  points={{-210,150},{-210,140}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS1.flange, PDMSRH.outlet) annotation (Line(
                  points={{50,130},{40,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamRH.outlet, sinkSteam1.flange) annotation (Line(
                  points={{4.44089e-016,170},{0,170},{0,188},{20,188}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam1.flange, RH.inletCold) annotation (Line(
                  points={{-20,30},{0,30},{0,120}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS1.flange, RH.inletHot) annotation (Line(
                  points={{-20,130},{-10,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(RH.outletHot,PDMSRH. inlet) annotation (Line(
                  points={{10,130},{20,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamRH.inlet,RH. outletCold) annotation (Line(
                  points={{0,150},{0,140}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS2.flange, PDMSECO.outlet) annotation (Line(
                  points={{260,130},{250,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamECO.outlet, sinkSteam2.flange) annotation (Line(
                  points={{210,170},{210,188},{230,188}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam2.flange, ECO.inletCold) annotation (Line(
                  points={{192,30},{210,30},{210,120}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS2.flange, ECO.inletHot) annotation (Line(
                  points={{190,130},{200,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(ECO.outletHot, PDMSECO.inlet) annotation (Line(
                  points={{220,130},{230,130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamECO.inlet, ECO.outletCold) annotation (Line(
                  points={{210,150},{210,140}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS4.flange,PDMSEVA1. outlet) annotation (Line(
                  points={{-120,-110},{-100,-110}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam4.flange,EVA1. inletCold) annotation (Line(
                  points={{-80,-210},{-60,-210},{-60,-120}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA1.outletHot,PDMSEVA1. inlet) annotation (Line(
                  points={{-70,-110},{-80,-110}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA1.inlet,EVA1. outletCold) annotation (Line(
                  points={{-10,-60},{-60,-60},{-60,-100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA2.outlet,sinkSteam5. flange) annotation (Line(
                  points={{70,-160},{70,-210},{90,-210}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS5.flange,EVA2. inletHot) annotation (Line(
                  points={{100,-110},{80,-110}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.outletHot,PDMSEVA2. inlet) annotation (Line(
                  points={{60,-110},{10,-110}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA2.inlet,EVA2. outletCold) annotation (Line(
                  points={{70,-140},{70,-120}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(w_MS_SH.y[1], sourceMS.in_w0) annotation (Line(
                  points={{-249,160},{-244,160},{-244,136}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_MS_SH.u[1], realExpression.y) annotation (Line(
                  points={{-272,160},{-289,160}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_ST_SH.u[1], realExpression1.y) annotation (Line(
                  points={{-272,90},{-289,90}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_ST_SH.u[1], realExpression2.y) annotation (Line(
                  points={{-272,60},{-289,60}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_ST_SH.y[1], sourceSteam.in_w0) annotation (Line(
                  points={{-249,60},{-244,60},{-244,36}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_ST_SH.y[1], sourceSteam.in_h) annotation (Line(
                  points={{-249,90},{-236,90},{-236,36}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(realExpression3.y, p_ST_SH.u[1]) annotation (Line(
                  points={{-249,220},{-232,220}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(p_ST_SH.y[1], sinkSteam.in_p0) annotation (Line(
                  points={{-209,220},{-184,220},{-184,198.8}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(realExpression4.y, p_ST_RH.u[1]) annotation (Line(
                  points={{-39,220},{-22,220}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_ST_RH.u[1], realExpression6.y) annotation (Line(
                  points={{-62,90},{-79,90}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_ST_RH.u[1], realExpression7.y) annotation (Line(
                  points={{-62,60},{-79,60}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_MS_RH.u[1], realExpression8.y) annotation (Line(
                  points={{-62,160},{-79,160}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_ST_RH.y[1], sourceSteam1.in_w0) annotation (Line(
                  points={{-39,60},{-34,60},{-34,36}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_ST_RH.y[1], sourceSteam1.in_h) annotation (Line(
                  points={{-39,90},{-26,90},{-26,36}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_MS_RH.y[1], sourceMS1.in_w0) annotation (Line(
                  points={{-39,160},{-34,160},{-34,136}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(p_ST_RH.y[1], sinkSteam1.in_p0) annotation (Line(
                  points={{1,220},{26,220},{26,196.8}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(realExpression9.y, p_ST_ECO.u[1]) annotation (Line(
                  points={{171,220},{188,220}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_MS_ECO.u[1], realExpression10.y) annotation (Line(
                  points={{148,190},{131,190}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_ST_ECO.u[1], realExpression12.y) annotation (Line(
                  points={{148,90},{131,90}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(w_ST_ECO.u[1], realExpression13.y) annotation (Line(
                  points={{148,60},{131,60}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_MS_ECO.y[1], sourceMS2.in_h)
                annotation (Line(points={{171,190},{184,190},{184,136}}, color={0,0,127}));
              connect(p_ST_ECO.y[1], sinkSteam2.in_p0) annotation (Line(points={{211,220},{222,
                      220},{236,220},{236,196.8}}, color={0,0,127}));
              connect(w_MS_ECO.y[1], sourceMS2.in_w0) annotation (Line(points={{171,160},{174,
                      160},{176,160},{176,136}}, color={0,0,127}));
              connect(h_ST_ECO.y[1], sourceSteam2.in_h) annotation (Line(points={{171,90},{176,
                      90},{186,90},{186,36}}, color={0,0,127}));
              connect(w_ST_ECO.y[1], sourceSteam2.in_w0)
                annotation (Line(points={{171,60},{178,60},{178,36}}, color={0,0,127}));
              connect(realExpression11.y, w_MS_ECO.u[1])
                annotation (Line(points={{131,160},{140,160},{148,160}}, color={0,0,127}));
              connect(realExpression18.y, p_ST_EVA.u[1]) annotation (Line(
                  points={{159,-180},{142,-180}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(h_MS_EVA.u[1], realExpression19.y) annotation (Line(
                  points={{142,-50},{159,-50}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(realExpression20.y, w_MS_EVA.u[1]) annotation (Line(points={{159,-80},
                      {159,-80},{142,-80}},   color={0,0,127}));
              connect(h_ST_EVA.u[1], realExpression21.y) annotation (Line(
                  points={{-122,-150},{-139,-150}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(realExpression22.y, w_ST_EVA.u[1]) annotation (Line(points={{-139,-180},
                      {-130,-180},{-122,-180}},
                                              color={0,0,127}));
              connect(h_ST_EVA.y[1], sourceSteam4.in_h) annotation (Line(points={{-99,-150},
                      {-92,-150},{-86,-150},{-86,-204}}, color={0,0,127}));
              connect(w_ST_EVA.y[1], sourceSteam4.in_w0) annotation (Line(points={{-99,-180},
                      {-94,-180},{-94,-204}}, color={0,0,127}));
              connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(points={{10,-60},
                      {42,-60},{70,-60},{70,-100}},    color={0,0,255}));
              connect(EVA1.inletHot, PDMSEVA2.outlet)
                annotation (Line(points={{-50,-110},{-10,-110}},         color={0,0,255}));
              connect(h_MS_EVA.y[1], sourceMS5.in_h) annotation (Line(points={{119,-50},{106,
                      -50},{106,-104}}, color={0,0,127}));
              connect(w_MS_EVA.y[1], sourceMS5.in_w0) annotation (Line(points={{119,-80},{116,
                      -80},{114,-80},{114,-104}},   color={0,0,127}));
              connect(p_ST_EVA.y[1], sinkSteam5.in_p0) annotation (Line(points={{119,-180},{
                      106,-180},{96,-180},{96,-201.2}},   color={0,0,127}));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-340,
                        -260},{340,260}}),        graphics={Text(
                      extent={{-210,110},{-170,90}},
                      lineColor={0,0,255},
                      textString="SH"), Text(
                      extent={{0,110},{40,90}},
                      lineColor={0,0,255},
                      textString="RH"), Text(
                      extent={{210,110},{250,90}},
                      lineColor={0,0,255},
                      textString="ECO"),Text(
                      extent={{-18,-130},{22,-150}},
                      lineColor={0,0,255},
                      textString="EVA"),
                    Text(
                      extent={{-348,258},{-88,244}},
                      lineColor={0,0,255},
                      textString="Tempo di simulazione richiesto: 8e6 s")}),
                                          Icon(coordinateSystem(extent={{-340,-260},{
                        340,260}})),
                experiment(StopTime=8e+006, __Dymola_NumberOfIntervals=5000),
                __Dymola_experimentSetupOutput);
            end test_heatExchangers_offDesign;

            model test_heatExchangers_offDesign_HPH
              "Test model of the MS_HPH, from nominal load to 20% load, steady-state initialisation"

              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
                annotation (Placement(transformation(extent={{-120,120},{-100,140}})));

              ThermoPower.Water.SourceMassFlow sourceMS3(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=136706,
                p0=400000,
                use_in_w0=true,
                use_in_h=true) annotation (Placement(transformation(extent={{-30,
                        -20},{-10,0}})));

              ThermoPower.Water.SourceMassFlow sourceSteam3(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1231860,
                w0=143.1,
                p0=17910000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{-30,-120},{-10,-100}})));
              ThermoPower.Water.SinkPressure sinkSteam3(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17814720,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{30,40},{50,60}})));
              ThermoPower.Water.SinkPressure sinkMS3(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=261078) annotation (Placement(transformation(extent={{60,-20},
                        {80,0}})));

              ThermoPower.Water.PressDrop PDMSHPH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(1.38922)*10^5,
                rhonom=1890.49) annotation (Placement(transformation(extent={{
                        30,-20},{50,0}})));

              ThermoPower.Water.PressDrop PDSteamHPH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                rhonom=768.15,
                wnom=143.1,
                dpnom=(0.9528)*10^5) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={10,20})));
              Components.HE HPH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                Shot=2429.5,
                Scold=1913.2,
                Mm=29335,
                nominalColdFlow=143.1,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=5000, alpha=0.536143),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2000, alpha=0.447465),
                generateInitCode=true,
                Vhot=33.634,
                Vcold=6.027,
                Nhex=20,
                nominalColdPressure=17910000,
                nominalHotPressure=400000,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=581.85,
                nominalHotInletTemperature=586.85,
                nominalHotOutletTemperature=563.15) annotation (Placement(
                    transformation(extent={{0,-20},{20,0}})));

              Modelica.Blocks.Tables.CombiTable1D p_ST_HPH(
                  smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments, table=[0,
                    178.1e5; 1e6,163.8e5; 2e6,147.3e5; 3e6,130.6e5; 4e6,113.8e5; 5e6,
                    103.2e5; 6e6,102.5e5; 7e6,101.7e5; 8e6,100.5e5; 10e6,100.5e5])
                annotation (Placement(transformation(extent={{-10,90},{10,110}})));
              Modelica.Blocks.Tables.CombiTable1D h_MS_HPH(
                  smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments, table=[0,
                    136842; 1e6,122751; 2e6,110175; 3e6,103053; 4e6,93659; 5e6,93356;
                    6e6,93356; 7e6,93356; 8e6,93356; 10e6,93356])
                annotation (Placement(transformation(extent={{-50,50},{-30,70}})));
              Modelica.Blocks.Tables.CombiTable1D w_MS_HPH(
                  smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments, table=[0,
                    610.6; 1e6,549.4; 2e6,488.4; 3e6,427.4; 4e6,366.3; 5e6,305.3; 6e6,
                    244.2; 7e6,183.2; 8e6,122.1; 10e6,122.1])
                annotation (Placement(transformation(extent={{-50,20},{-30,40}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_HPH(
                                  smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments, table=[0,
                    1231860; 1e6,1182452; 2e6,1133837; 3e6,1126638; 4e6,1111212; 5e6,
                    1251884; 6e6,1258763; 7e6,1260394; 8e6,1258351; 10e6,1258351])
                annotation (Placement(transformation(extent={{-50,-60},{-30,-40}})));
              Modelica.Blocks.Tables.CombiTable1D w_ST_HPH(       smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments, table=[0,
                    143.1; 1e6,67.83; 2e6,24.44; 3e6,5.26; 4e6,0.01; 5e6,0.01; 6e6,
                    0.01; 7e6,0.01; 8e6,0.01; 10e6,0.01])
                annotation (Placement(transformation(extent={{-50,-90},{-30,-70}})));

              //Real load = sourceMS2.w/sourceMS2.w0;
              Modelica.Blocks.Sources.Ramp ramp(height=6e6, duration=6e6) annotation (
                 Placement(transformation(extent={{-40,90},{-20,110}})));
              Modelica.Blocks.Sources.Ramp ramp1(height=6e6, duration=6e6)
                annotation (Placement(transformation(extent={{-90,50},{-70,70}})));
              Modelica.Blocks.Sources.Ramp ramp2(height=6e6, duration=6e6)
                annotation (Placement(transformation(extent={{-90,20},{-70,40}})));
              Modelica.Blocks.Sources.Ramp ramp3(height=6e6, duration=6e6)
                annotation (Placement(transformation(extent={{-90,-60},{-70,-40}})));
              Modelica.Blocks.Sources.Ramp ramp4(height=6e6, duration=6e6)
                annotation (Placement(transformation(extent={{-90,-90},{-70,-70}})));
            equation
              connect(sinkMS3.flange, PDMSHPH.outlet) annotation (Line(
                  points={{60,-10},{50,-10}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamHPH.outlet, sinkSteam3.flange) annotation (Line(
                  points={{10,30},{10,50},{30,50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam3.flange, HPH.inletCold) annotation (Line(
                  points={{-10,-110},{10,-110},{10,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS3.flange, HPH.inletHot) annotation (Line(
                  points={{-10,-10},{0,-10}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(HPH.outletHot, PDMSHPH.inlet) annotation (Line(
                  points={{20,-10},{30,-10}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamHPH.inlet, HPH.outletCold) annotation (Line(
                  points={{10,10},{10,0}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(p_ST_HPH.y[1], sinkSteam3.in_p0) annotation (Line(points={{11,100},{24,
                      100},{36,100},{36,58.8}},           color={0,0,127}));
              connect(h_MS_HPH.y[1], sourceMS3.in_h) annotation (Line(points={{-29,60},{-16,
                      60},{-16,-4}},     color={0,0,127}));
              connect(w_MS_HPH.y[1], sourceMS3.in_w0) annotation (Line(points={{-29,30},{-24,
                      30},{-24,-4}},     color={0,0,127}));
              connect(h_ST_HPH.y[1], sourceSteam3.in_h) annotation (Line(points={{-29,-50},{
                      -29,-50},{-16,-50},{-16,-104}},       color={0,0,127}));
              connect(w_ST_HPH.y[1], sourceSteam3.in_w0) annotation (Line(points={{-29,-80},
                      {-24,-80},{-24,-104}},    color={0,0,127}));
              connect(p_ST_HPH.u[1], ramp.y) annotation (Line(points={{-12,100},{
                      -14,100},{-19,100}},                                    color={
                      0,0,127}));
              connect(h_MS_HPH.u[1], ramp1.y) annotation (Line(points={{-52,60},{-60,60},{-69,
                      60}},                  color={0,0,127}));
              connect(w_MS_HPH.u[1], ramp2.y)
                annotation (Line(points={{-52,30},{-69,30}},     color={0,0,127}));
              connect(h_ST_HPH.u[1], ramp3.y)
                annotation (Line(points={{-52,-50},{-69,-50}},     color={0,0,127}));
              connect(w_ST_HPH.u[1], ramp4.y) annotation (Line(points={{-52,-80},{-60,-80},{
                      -69,-80}},               color={0,0,127}));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-120,
                        -140},{120,140}}),        graphics={
                                        Text(
                      extent={{10,-30},{50,-50}},
                      lineColor={0,0,255},
                      textString="HPH"),
                    Text(
                      extent={{-128,138},{132,124}},
                      lineColor={0,0,255},
                      textString="Tempo di simulazione richiesto: 4e6 s")}),
                                          Icon(coordinateSystem(extent={{-120,-140},{120,140}})),
                experiment(StopTime=4e+006, __Dymola_NumberOfIntervals=5000),
                __Dymola_experimentSetupOutput);
            end test_heatExchangers_offDesign_HPH;

            model test_EVA12_2ph
              "Test of EVA1 and EVA2, at nominal load, to evaluate the 2ph h.t.c."

              ThermoPower.Water.SourceMassFlow sourceMS4(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=289063,
                p0=730800) annotation (Placement(transformation(extent={{-60,40},
                        {-40,60}})));

              ThermoPower.Water.SourceMassFlow sourceSteam4(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1665608,
                w0=88.59,
                p0=17810000)
                annotation (Placement(transformation(extent={{-60,20},{-40,40}})));
              ThermoPower.Water.SinkPressure sinkSteam4(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17804170)
                annotation (Placement(transformation(extent={{0,80},{20,100}})));
              ThermoPower.Water.SinkPressure sinkMS4(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=605957) annotation (Placement(transformation(extent={{30,40},
                        {50,60}})));

              ThermoPower.Water.PressDrop PDMSEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(1.24843)*10^5,
                rhonom=1863.6)
                annotation (Placement(transformation(extent={{0,40},{20,60}})));

              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.55,
                dpnom=(0.05828)*10^5,
                rhonom=168.791)      annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-20,80})));
              ThermoPower.Water.SourceMassFlow sourceMS5(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=355877,
                p0=730800) annotation (Placement(transformation(extent={{-60,-60},
                        {-40,-40}})));

              ThermoPower.Water.SourceMassFlow sourceSteam5(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=88.59,
                h=2279896,
                p0=17760000)
                annotation (Placement(transformation(extent={{-60,-80},{-40,-60}})));
              ThermoPower.Water.SinkPressure sinkSteam5(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17748830)
                annotation (Placement(transformation(extent={{0,-20},{20,0}})));
              ThermoPower.Water.SinkPressure sinkMS5(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=696272) annotation (Placement(transformation(extent={{30,-60},
                        {50,-40}})));

              ThermoPower.Water.PressDrop PDMSEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=610.6,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                dpnom=(0.34528)*10^5,
                rhonom=1826.2) annotation (Placement(transformation(extent={{0,
                        -60},{20,-40}})));

              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=97.019)       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-20,-20})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                Nhex=20,
                nominalHotFlow=610.6,
                Vcold=721.2,
                nominalColdFlow=88.59,
                Vhot=3.9,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                generateInitCode=true,
                Shot=429.2,
                Scold=338,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    alpha=1.6,
                    gamma_nom_2ph=5000,
                    gamma_nom_liq=2000,
                    gamma_nom_vap=2000),
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95) annotation (Placement(
                    transformation(extent={{-30,-60},{-10,-40}})));
              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.steadyState)
                annotation (Placement(transformation(extent={{80,80},{100,100}})));
              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                redeclare package coldMedium = coldMedium,
                redeclare package hotMedium = hotMedium,
                Nhex=Nhex_bi,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                tubeWalls(Tvolstart(displayUnit="K") = {650.47,650.866,651.269,
                    651.679,652.097,652.522,652.956,653.397,653.846,654.304,654.769,
                    655.244,655.726,656.218,656.719,657.228,660.331,675.049,675.851,
                    676.675,677.521,678.389,679.279,680.19,681.124,682.078,683.052,
                    684.047,685.063,686.099,687.154,688.23,689.324,690.436,691.567,
                    692.714,693.877,695.057,696.25}),
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                     {356849,355304,353752,352194,350629,349059,347485,345908,344328,
                      342747,341165,339583,338003,336424,334848,333276,331707,330144,
                      328587,327037,325494,323959,322433,320326,318146,316005,313903,
                      311839,309812,307821,305867,303947,302063,300212,298395,296610,
                      294858,293137,291447,289788}),
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={2.29887e+006,2.31027e+006,2.32187e+006,2.33369e+006,
                      2.34573e+006,2.35799e+006,2.37047e+006,2.38319e+006,
                      2.39613e+006,2.40932e+006,2.42274e+006,2.43641e+006,
                      2.45034e+006,2.46452e+006,2.47896e+006,2.49366e+006,
                      2.50864e+006,2.52311e+006,2.53359e+006,2.54414e+006,
                      2.55474e+006,2.56539e+006,2.57608e+006,2.58682e+006,
                      2.59759e+006,2.60839e+006,2.61922e+006,2.63006e+006,
                      2.64092e+006,2.65179e+006,2.66265e+006,2.67351e+006,
                      2.68437e+006,2.6952e+006,2.70601e+006,2.71679e+006,2.72754e+006,
                      2.73825e6,2.7415e6,2.74562e6}),
                Vhot=3.905,
                Vcold=1.065,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    alpha=1.6,
                    gamma_nom_liq=2000,
                    gamma_nom_2ph=5000,
                    gamma_nom_vap=2000),
                Shot=429.2,
                Scold=338,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95) annotation (Placement(
                    transformation(extent={{-30,40},{-10,60}})));
            equation
              connect(sinkMS4.flange,PDMSEVA1. outlet) annotation (Line(
                  points={{30,50},{20,50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA1.outlet,sinkSteam4. flange) annotation (Line(
                  points={{-20,90},{0,90}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS5.flange,PDMSEVA2. outlet) annotation (Line(
                  points={{30,-50},{20,-50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA2.outlet,sinkSteam5. flange) annotation (Line(
                  points={{-20,-10},{0,-10}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceSteam5.flange,EVA2. inletCold) annotation (Line(
                  points={{-40,-70},{-20,-70},{-20,-60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS5.flange,EVA2. inletHot) annotation (Line(
                  points={{-40,-50},{-30,-50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.outletHot,PDMSEVA2. inlet) annotation (Line(
                  points={{-10,-50},{0,-50}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(PDSteamEVA2.inlet,EVA2. outletCold) annotation (Line(
                  points={{-20,-30},{-20,-40}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS4.flange, EVA1.inletHot) annotation (Line(points={{
                      -40,50},{-35,50},{-30,50}}, color={0,0,255}));
              connect(EVA1.outletHot, PDMSEVA1.inlet) annotation (Line(points={{
                      -10,50},{-5,50},{0,50}}, color={0,0,255}));
              connect(PDSteamEVA1.inlet, EVA1.outletCold) annotation (Line(points=
                     {{-20,70},{-20,65},{-20,60}}, color={0,0,255}));
              connect(sourceSteam4.flange, EVA1.inletCold) annotation (Line(
                    points={{-40,30},{-20,30},{-20,40}}, color={0,0,255}));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false,
                      extent={{-100,-100},{100,100}}),
                                  graphics={
                                        Text(
                      extent={{-20,30},{20,10}},
                      lineColor={0,0,255},
                      textString="EVA1"),
                                        Text(
                      extent={{-20,-70},{20,-90}},
                      lineColor={0,0,255},
                      textString="EVA2")}));
            end test_EVA12_2ph;

            model partial_OTSG

              Components.HE SH(
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                Mm=15353,
                nominalColdFlow=88.59,
                nominalHotFlow=371.8,
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                Shot=1153.8,
                Scold=879.8,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2200, alpha=0.847436),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3000, alpha=0.28867),
                Nhex=20,
                Vhot=3.68,
                Vcold=2.683,
                nominalColdPressure=17710000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=648.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=732.02)
                annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-20,30})));
              Components.HE RH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                Mm=10081,
                nominalColdFlow=82.57,
                nominalHotFlow=238.7,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                Shot=1339,
                Scold=1198,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2600, alpha=0.320782),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.798347, gamma_nom=970),
                Nhex=20,
                Vhot=1.522,
                Vcold=5.092,
                nominalColdPressure=4698000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=625.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=731.45)
                annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                    rotation=-90,
                    origin={110,30})));
              ThermoPower.Water.ThroughMassFlow throughMassFlow(
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=238.7,
                use_in_w0=true) annotation (Placement(transformation(extent={{
                        20,120},{40,100}})));

              ThermoPower.Water.SensT sensT(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph) annotation (Placement(
                    transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={86,44})));
              ThermoPower.PowerPlants.Control.PID PID_T_RH(
                Ti=20,
                PVmin=600,
                PVmax=1000,
                CSmin=1,
                CSmax=300,
                PVstart=0.5,
                CSstart=0.8,
                Kp=5)
                annotation (Placement(transformation(extent={{54,20},{34,0}})));
              Modelica.Blocks.Sources.Constant const(k=826.05)
                annotation (Placement(transformation(extent={{90,-4},{70,16}})));
              ThermoPower.Water.SinkPressure sinkSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17710000)
                annotation (Placement(transformation(extent={{0,50},{20,70}})));
              ThermoPower.Water.SourceMassFlow sourceSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=82.57,
                h=3083198,
                p0=4698000)
                annotation (Placement(transformation(extent={{160,20},{140,40}})));
              ThermoPower.Water.SinkPressure sinkSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=4642887)
                annotation (Placement(transformation(extent={{80,70},{60,90}})));
              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322,
                w0=610.6,
                use_in_w0=false) annotation (Placement(transformation(extent={{
                        -70,100},{-50,120}})));

              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=839319) annotation (Placement(transformation(extent={{-60,-130},
                        {-40,-110}})));

              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
                                              annotation (Placement(transformation(
                      extent={{-180,140},{-160,160}})));
              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                Mm=43100,
                Shot=2251.9,
                Scold=1765.5,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=4100, alpha=0.440883),
                nominalColdFlow=88.59,
                Nhex=20,
                Vhot=26.587,
                Vcold=8.651,
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15) annotation (Placement(
                    transformation(extent={{-110,-130},{-90,-110}})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Shot=429.2,
                Scold=338,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3100, alpha=0.440883),
                Nhex=20,
                Vhot=3.905,
                Vcold=1.065,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95) annotation (Placement(
                    transformation(extent={{-90,-70},{-110,-50}})));
              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.55,
                dpnom=(0.05828)*10^5,
                rhonom=582.24)       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-100,-90})));
              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=250)          annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-70,-20})));
              ThermoPower.Water.SourceMassFlow sourceSteam4(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1665608,
                w0=88.59,
                p0=17810000)
                annotation (Placement(transformation(extent={{-140,-160},{-120,-140}})));
              ThermoPower.Water.SinkPressure sinkMS3(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=839319) annotation (Placement(transformation(extent={{124,-80},
                        {144,-60}})));

              ThermoPower.Water.FlowJoin flowJoin(redeclare package Medium =
                    PreFlexMS.Media.MoltenSalt60_40_quad) annotation (Placement(
                    transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={106,-46})));
              ThermoPower.Water.SourceMassFlow sourceMS5(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=610.6,
                h=355877,
                p0=730800,
                use_in_h=true) annotation (Placement(transformation(extent={{-40,
                        -50},{-60,-70}})));

              Components.Separator_controlledLevel Separator(
                zl_min=0,
                Kp=-250,
                Ti=20,
                Td=0,
                Nd=1,
                PVmin=0,
                PVmax=20,
                CSmin=0,
                actuatorGain=1,
                Ta=5,
                H=8,
                zl_max=7,
                zl_start=0,
                PVstart=0,
                CSstart=0,
                D=0.56,
                vessel(initOpt=ThermoPower.Choices.Init.Options.fixedState, T(start=
                        648.15)),
                h_start=2740000,
                CSmax=150)
                annotation (Placement(transformation(extent={{-44,-40},{4,16}})));
              ThermoPower.Water.SinkPressure sinkSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17748830)
                annotation (Placement(transformation(extent={{-14,-50},{6,-30}})));
              Modelica.Blocks.Sources.Ramp ramp(
                height=-15000,
                duration=2000,
                offset=355000,
                startTime=50000)
                annotation (Placement(transformation(extent={{0,-100},{-20,-80}})));
            equation
              connect(const.y,PID_T_RH. SP) annotation (Line(
                  points={{69,6},{54,6}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(sensT.T,PID_T_RH. PV) annotation (Line(
                  points={{80,52},{80,60},{60,60},{60,14},{54,14}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(sensT.inlet,RH. outletCold) annotation (Line(
                  points={{90,38},{90,30},{100,30}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(RH.inletCold, sourceSteam1.flange) annotation (Line(
                  points={{120,30},{140,30}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS.flange, throughMassFlow.inlet) annotation (Line(
                  points={{-50,110},{20,110}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(throughMassFlow.outlet, RH.inletHot) annotation (Line(
                  points={{40,110},{110,110},{110,40}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(sourceMS.flange, SH.inletHot) annotation (Line(
                  points={{-50,110},{-40,110},{-40,30},{-30,30}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(SH.outletCold, sinkSteam.flange) annotation (Line(
                  points={{-20,40},{-20,60},{0,60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sensT.outlet, sinkSteam1.flange) annotation (Line(
                  points={{90,50},{90,80},{80,80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.outletHot,EVA1. inletHot) annotation (Line(
                  points={{-110,-60},{-120,-60},{-120,-120},{-110,-120}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                  points={{-100,-110},{-100,-100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.inletCold, PDSteamEVA1.outlet) annotation (Line(
                  points={{-100,-70},{-100,-80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA1.outletHot, sinkMS.flange) annotation (Line(
                  points={{-90,-120},{-60,-120}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(sourceSteam4.flange, EVA1.inletCold) annotation (Line(
                  points={{-120,-150},{-100,-150},{-100,-130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(throughMassFlow.in_w0, PID_T_RH.CS) annotation (Line(
                  points={{26,104},{26,10},{34,10}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(flowJoin.in1, RH.outletHot) annotation (Line(
                  points={{110,-40},{110,20}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(SH.outletHot, flowJoin.in2) annotation (Line(
                  points={{-10,30},{10,30},{10,-40},{102,-40}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(Separator.outlet_liq, sinkSteam2.flange) annotation (Line(
                  points={{-20,-30},{-20,-40},{-14,-40}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(Separator.outlet_vap, SH.inletCold) annotation (Line(
                  points={{-20,10},{-20,20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(Separator.inlet, PDSteamEVA2.outlet) annotation (Line(
                  points={{-28.5714,-20},{-60,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                  points={{-100,-50},{-100,-20},{-80,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(flowJoin.out, sinkMS3.flange) annotation (Line(
                  points={{106,-52},{106,-70},{124,-70}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(EVA2.inletHot, sourceMS5.flange) annotation (Line(
                  points={{-90,-60},{-60,-60}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(sourceMS5.in_h, ramp.y) annotation (Line(
                  points={{-54,-66},{-54,-90},{-21,-90}},
                  color={0,0,127},
                  smooth=Smooth.None));
              annotation (Diagram(coordinateSystem(extent={{-180,-160},{180,160}},
                      preserveAspectRatio=false), graphics), Icon(coordinateSystem(
                      extent={{-180,-160},{180,160}})));
            end partial_OTSG;

            model partial_OTSGbis

              Components.HE SH(
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                Mm=15353,
                nominalColdFlow=88.59,
                nominalHotFlow=371.8,
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2200, alpha=0.847436),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3000, alpha=0.28867),
                Nhex=20,
                Shot=1153.8*0.88,
                Scold=879.8*0.88,
                Vhot=3.68,
                Vcold=2.683,
                nominalColdPressure=17710000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=648.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=732.02)
                annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-20,30})));
              Components.HE RH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                Mm=10081,
                nominalColdFlow=82.57,
                nominalHotFlow=238.7,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                Shot=1339,
                Scold=1198,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2600, alpha=0.320782),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.798347, gamma_nom=970),
                Nhex=20,
                Vhot=1.522,
                Vcold=5.092,
                nominalColdPressure=4698000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=625.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=731.45)
                annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                    rotation=-90,
                    origin={110,30})));
              ThermoPower.Water.ThroughMassFlow throughMassFlow(
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=238.7,
                use_in_w0=true) annotation (Placement(transformation(extent={{
                        20,120},{40,100}})));

              ThermoPower.Water.SensT sensT(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph) annotation (Placement(
                    transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={86,44})));
              ThermoPower.PowerPlants.Control.PID PID_T_RH(
                PVmin=600,
                PVmax=1000,
                CSmin=1,
                CSmax=300,
                PVstart=0.5,
                CSstart=0.8,
                Kp=14,
                Ti=56)
                annotation (Placement(transformation(extent={{54,20},{34,0}})));
              Modelica.Blocks.Sources.Constant const(k=826.05)
                annotation (Placement(transformation(extent={{90,-4},{70,16}})));
              ThermoPower.Water.SinkPressure sinkSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17710000)
                annotation (Placement(transformation(extent={{0,50},{20,70}})));
              ThermoPower.Water.SourceMassFlow sourceSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=82.57,
                h=3083198,
                p0=4698000)
                annotation (Placement(transformation(extent={{160,20},{140,40}})));
              ThermoPower.Water.SinkPressure sinkSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=4642887)
                annotation (Placement(transformation(extent={{80,70},{60,90}})));
              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322,
                w0=610.6,
                use_in_w0=false) annotation (Placement(transformation(extent={{
                        -70,100},{-50,120}})));

              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=839319) annotation (Placement(transformation(extent={{-60,-130},
                        {-40,-110}})));

              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
                                              annotation (Placement(transformation(
                      extent={{-180,140},{-160,160}})));
              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                Mm=43100,
                Shot=2251.9,
                Scold=1765.5,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=4100, alpha=0.440883),
                nominalColdFlow=88.59,
                Nhex=20,
                Vhot=26.587,
                Vcold=8.651,
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15) annotation (Placement(
                    transformation(extent={{-110,-130},{-90,-110}})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3100, alpha=0.440883),
                Nhex=20,
                Shot=429.2*1.1,
                Scold=338*1.1,
                Vhot=3.905,
                Vcold=1.065,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95) annotation (Placement(
                    transformation(extent={{-90,-70},{-110,-50}})));
              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.55,
                dpnom=(0.05828)*10^5,
                rhonom=582.24)       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-100,-90})));
              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=250)          annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-70,-20})));
              ThermoPower.Water.SourceMassFlow sourceSteam4(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1665608,
                w0=88.59,
                p0=17810000)
                annotation (Placement(transformation(extent={{-140,-160},{-120,-140}})));
              ThermoPower.Water.FlowJoin flowJoin(redeclare package Medium =
                    PreFlexMS.Media.MoltenSalt60_40_quad) annotation (Placement(
                    transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={106,-46})));
              Components.Separator_controlledLevel Separator(
                zl_min=0,
                Kp=-250,
                Ti=20,
                Td=0,
                Nd=1,
                PVmin=0,
                PVmax=20,
                CSmin=0,
                actuatorGain=1,
                Ta=5,
                H=8,
                zl_max=7,
                zl_set=4,
                zl_start=0,
                PVstart=0,
                CSstart=0,
                D=0.56,
                vessel(initOpt=ThermoPower.Choices.Init.Options.fixedState, T(start=
                        648.15)),
                h_start=2740000,
                CSmax=150)
                annotation (Placement(transformation(extent={{-44,-40},{4,16}})));
              ThermoPower.Water.SinkPressure sinkSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17748830)
                annotation (Placement(transformation(extent={{-14,-50},{6,-30}})));
            equation
              connect(const.y,PID_T_RH. SP) annotation (Line(
                  points={{69,6},{54,6}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(sensT.T,PID_T_RH. PV) annotation (Line(
                  points={{80,52},{80,60},{60,60},{60,14},{54,14}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(sensT.inlet,RH. outletCold) annotation (Line(
                  points={{90,38},{90,30},{100,30}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(RH.inletCold, sourceSteam1.flange) annotation (Line(
                  points={{120,30},{140,30}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS.flange, throughMassFlow.inlet) annotation (Line(
                  points={{-50,110},{20,110}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(throughMassFlow.outlet, RH.inletHot) annotation (Line(
                  points={{40,110},{110,110},{110,40}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(sourceMS.flange, SH.inletHot) annotation (Line(
                  points={{-50,110},{-40,110},{-40,30},{-30,30}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(SH.outletCold, sinkSteam.flange) annotation (Line(
                  points={{-20,40},{-20,60},{0,60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sensT.outlet, sinkSteam1.flange) annotation (Line(
                  points={{90,50},{90,80},{80,80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.outletHot,EVA1. inletHot) annotation (Line(
                  points={{-110,-60},{-120,-60},{-120,-120},{-110,-120}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                  points={{-100,-110},{-100,-100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.inletCold, PDSteamEVA1.outlet) annotation (Line(
                  points={{-100,-70},{-100,-80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA1.outletHot, sinkMS.flange) annotation (Line(
                  points={{-90,-120},{-60,-120}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(sourceSteam4.flange, EVA1.inletCold) annotation (Line(
                  points={{-120,-150},{-100,-150},{-100,-130}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(flowJoin.in1, RH.outletHot) annotation (Line(
                  points={{110,-40},{110,20}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(SH.outletHot, flowJoin.in2) annotation (Line(
                  points={{-10,30},{10,30},{10,-40},{102,-40}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(Separator.outlet_liq, sinkSteam2.flange) annotation (Line(
                  points={{-20,-30},{-20,-40},{-14,-40}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(Separator.outlet_vap, SH.inletCold) annotation (Line(
                  points={{-20,10},{-20,20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(Separator.inlet, PDSteamEVA2.outlet) annotation (Line(
                  points={{-28.5714,-20},{-60,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                  points={{-100,-50},{-100,-20},{-80,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(EVA2.inletHot, flowJoin.out) annotation (Line(
                  points={{-90,-60},{106,-60},{106,-52}},
                  color={255,0,0},
                  smooth=Smooth.None));
              connect(PID_T_RH.CS, throughMassFlow.in_w0) annotation (Line(
                  points={{34,10},{26,10},{26,104}},
                  color={0,0,127},
                  smooth=Smooth.None));
              annotation (Diagram(coordinateSystem(extent={{-180,-160},{180,160}},
                      preserveAspectRatio=false), graphics), Icon(coordinateSystem(
                      extent={{-180,-160},{180,160}})));
            end partial_OTSGbis;

            model test_wFW_sensitivity
              "Test model to analyse the sensitivity of the EVA1/EVA2 operational point against the feedwater mass flow"
              import PreFlexMS;

              //Heat exchangers parameter
              parameter Integer Nhex_mono = 5
                "Number of nodes for single phase heat exchangers";
              parameter Integer Nhex_bi = 10
                "Number of nodes for biphase heat exchangers";

              Components.HE ECO(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Shot=2394.2,
                Scold=1855.5,
                Mm=30395,
                Nhex=Nhex_mono,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.728761, gamma_nom=4400),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.461561, gamma_nom=2200),
                Vhot=45.011,
                Vcold=5.752,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.23186e+006,1.44947e+006,1.55292e+006,1.61733e+006,
                      1.66591e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {582.675,605.885,616.62,
                    623.137}),
                nominalColdPressure=17910000,
                nominalHotPressure=569600,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=623.79,
                nominalHotInletTemperature=628.75,
                nominalHotOutletTemperature=586.72,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={199579,192529,183185,168175,136603},
                  noInitialPressure=true),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-104,-276},{-136,-244}})));

              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Nhex=Nhex_bi,
                nominalHotFlow=610.6,
                Mm=43100,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                nominalColdFlow=88.59,
                Vhot=26.587,
                Vcold=8.651,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.66561e+006,1.67917e+006,1.69246e+006,1.70577e+006,
                      1.7194e+006,1.74575e+006,1.79332e+006,1.87797e+006,2.02847e+006,
                      2.29575e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {626.054,627.39,628.669,
                    629.917,629.888,630.182,630.842,632.016,634.097}),
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={289063,250302,228476,216201,209301,205481,203503,201573,
                      199646,197679},
                  noInitialPressure=true),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=3875,
                    gamma_nom_vap=3875,
                    gamma_nom_2ph=10000,
                    alpha=1.6),
                Shot=2251.9,
                Scold=1765.5,
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                                                    annotation (Placement(
                    transformation(extent={{-136,-196},{-104,-164}})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Nhex=Nhex_bi,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                Vhot=3.905,
                Vcold=1.065,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={2.2799e+006,2.32986e+006,2.38391e+006,2.4424e+006,
                      2.50567e+006,2.55859e+006,2.60531e+006,2.65251e+006,
                      2.69972e+006,2.74645e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {650.941,652.73,654.665,
                    656.757,670.379,679.959,684.074,688.573,693.427}),
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={355877,349098,342248,335400,328621,320944,311763,303278,
                      295435,288187},
                  noInitialPressure=true),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=2000,
                    gamma_nom_2ph=5000,
                    gamma_nom_vap=2000,
                    alpha=1.1),
                Shot=429.2,
                Scold=338,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                                                    annotation (Placement(
                    transformation(extent={{-104,-116},{-136,-84}})));

              Components.Equilibrium2phVessel separator(
                H=8,
                zl_start=0,
                h_start=2.7e6,
                D=0.56,
                noInitialPressure=true,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p_start=17710000)
                annotation (Placement(transformation(extent={{-88,-52},{-32,4}})));
              ThermoPower.Water.ThroughMassFlow waterPump(
                allowFlowReversal=false,
                w0=0,
                use_in_w0=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={-60,-74})));

              Components.HE SH(
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Mm=15353,
                nominalColdFlow=88.59,
                nominalHotFlow=371.8,
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2200, alpha=0.847436),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3000, alpha=0.28867),
                Nhex=Nhex_mono,
                Vhot=3.68,
                Vcold=2.683,
                tubeWalls(Tvolstart(displayUnit="K") = {725.391,769.727,804.861,
                    826.169}),
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={2.74562e+006,3.02031e+006,3.23297e+006,3.36066e+006,
                      3.42731e+006},
                  noInitialPressure=true),
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={518322,502443,472016,421345,355895},
                  noInitialPressure=true),
                Shot=1153.8,
                Scold=879.8,
                nominalColdPressure=17710000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=648.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=732.02,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                    rotation=0,
                    origin={-60,40})));
              Components.HE RH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Mm=10081,
                nominalColdFlow=82.57,
                nominalHotFlow=238.7,
                Nhex=Nhex_mono,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.798347, gamma_nom=970),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2600, alpha=0.320782),
                Vhot=1.522,
                Vcold=5.092,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={3.0832e+006,3.30568e+006,3.43926e+006,3.51621e+006,
                      3.56013e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {734.495,783.217,812.321,
                    829.047}),
                Shot=1339,
                Scold=1198,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={518322,503128,476511,430304,353343},
                  noInitialPressure=true),
                nominalColdPressure=4698000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=625.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=731.45,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                    rotation=-90,
                    origin={20,80})));
              ThermoPower.Water.ThroughMassFlow valveRH(
                w0=238.7,
                use_in_w0=true,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-200,150},{-180,
                        170}})));

              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                dpnom=(0.05828)*10^5,
                wnom=88.59,
                rhonom=168.8,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-120,-140})));
              ThermoPower.Water.PressDrop PDSteamECO2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                rhonom=582.8,
                dpnom=(0.52492/2)*10^5,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                        annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-120,-220})));
              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=97,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-92,-32})));
              //Overall parameter and variables:
              Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
              Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
              Modelica.SIunits.Temperature T_outMS = ECO.hotFluid.T[end];
              Modelica.SIunits.Temperature T_RH = sensT_RH.T;
              Modelica.SIunits.Length sep_lev = separator.zl;
              Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
              Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
              Real x_EVA1_out = EVA1.coldFluid.x[end];
              Real x_EVA2_out = EVA2.coldFluid.x[end];
              Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
              Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
              Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
              Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;
              //Modelica.SIunits.TemperatureDifference deltaT_SH_star = ((ECO.coldFluid.T[end] - ECO.coldFluid.T[1]) - (ECO.hotFluid.T[1] - ECO.hotFluid.T[end]))/(ECO.coldFluid.T[end] - ECO.coldFluid.T[1]);
              //Modelica.SIunits.TemperatureDifference deltaT_EVA2_star = ((EVA2.coldFluid.T[end] - EVA2.coldFluid.T[1]) - (EVA2.hotFluid.T[1] - EVA2.hotFluid.T[end]))/(EVA2.hotFluid.T[1] - EVA2.hotFluid.T[end]);
              //Modelica.SIunits.TemperatureDifference deltaT_EVA1_star = ((EVA1.coldFluid.T[end] - EVA1.coldFluid.T[1]) - (EVA1.hotFluid.T[1] - EVA1.hotFluid.T[end]))/(EVA1.hotFluid.T[1] - EVA1.hotFluid.T[end]);
              Modelica.SIunits.TemperatureDifference deltaT_SH_hot = SH.hotFluid.T[1] - SH.hotFluid.T[end];
              Modelica.SIunits.TemperatureDifference deltaT_SH_cold = SH.coldFluid.T[end] - SH.coldFluid.T[1];
              Modelica.SIunits.Power Q_sh = separator.w_in*(separator.h - separator.hv);
              Modelica.SIunits.Power Q_eva = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot;
              Real Q_sh_perc = Q_sh/Q_tot;

              ThermoPower.Water.PressDrop PDSteamRH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=82.57,
                dpnom=(0.55113)*10^5,
                rhonom=12.52,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-20,100})));
              ThermoPower.Water.SensT sensT_FW(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph)
                annotation (Placement(transformation(extent={{-30,-316},{-50,-296}})));
              ThermoPower.Water.SensT sensT_MS(redeclare package Medium =
                    PreFlexMS.Media.MoltenSalt60_40_quad) annotation (Placement(
                    transformation(extent={{-150,-266},{-170,-246}})));
              ThermoPower.Water.PressDrop PDSteamSH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                dpnom=(177.1 - 176.1189)*10^5,
                wnom=88.59,
                rhonom=51.9,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-60,80})));
              ThermoPower.Water.PressDrop PDSteamECO1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.52492/2)*10^5,
                rhonom=767.7,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-120,-294})));
              PreFlexMS.Auxiliaries.Components.loadCalculator loadCalc(
                load_start=100,
                t_start=0,
                load_end=100,
                t_ramp=0) annotation (Placement(transformation(extent={{-320,-60},
                        {-280,-20}})));
              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                h=518322,
                w0=610.6,
                use_in_w0=true,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-248,150},{-228,
                        170}})));
              Modelica.Blocks.Sources.Ramp LOAD(
                height=loadCalc.load_end - loadCalc.load_start,
                duration=loadCalc.t_ramp,
                offset=loadCalc.load_start,
                startTime=loadCalc.t_start) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-330,120})));
              Modelica.Blocks.Tables.CombiTable1D table_w_MS(
                  table=[20,122.1; 30,183.3; 40,244.2; 50,305.3; 60,366.3; 70,427.4; 80,
                    488.4; 90,549.4; 100,610.6], smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative)
                annotation (Placement(transformation(extent={{-300,170},{-280,190}})));
              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=839319) annotation (Placement(transformation(extent={{-200,-270},
                        {-220,-250}})));

              ThermoPower.Water.SinkPressure sinkPressure1(redeclare package
                  Medium =
                    Modelica.Media.Water.WaterIF97_ph, p0=17700000)
                annotation (Placement(transformation(extent={{-40,-130},{-20,-110}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_RH(            smoothness=
                    Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,3092400;
                    20,3124593; 30,3130270; 40,3128892; 50,3125327; 60,3116961; 70,3108752;
                    80,3100618; 90,3093346; 100,3083198])
                annotation (Placement(transformation(extent={{140,170},{120,190}})));
              Modelica.Blocks.Tables.CombiTable1D w_ST_RH(
                                           smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    4.18; 20,13.4; 30,21.5; 40,29.69; 50,38.01; 60,46.38; 70,54.15; 80,63.31;
                    90,72.7; 100,82.57])
                annotation (Placement(transformation(extent={{140,130},{120,150}})));
              ThermoPower.Water.SourceMassFlow sourceSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=82.57,
                h=3083198,
                p0=4698000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{100,70},{80,90}})));
              Modelica.Blocks.Sources.Ramp LOAD1(
                height=loadCalc.load_end - loadCalc.load_start,
                duration=loadCalc.t_ramp,
                offset=loadCalc.load_start,
                startTime=loadCalc.t_start) annotation (Placement(transformation(
                    extent={{10,-10},{-10,10}},
                    rotation=0,
                    origin={330,180})));
              ThermoPower.Water.SinkPressure sinkSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=4642887,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{20,180},{40,200}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_RH(        smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    22e5; 20,7.67e5; 30,12.28e5; 40,16.92e5; 50,21.61e5; 60,26.32e5; 70,30.66e5;
                    80,35.77e5; 90,40.98e5; 100,46.4e5])
                annotation (Placement(transformation(extent={{140,210},{120,230}})));
              ThermoPower.Water.SensT sensT_RH(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph) annotation (Placement(transformation(
                    extent={{-10,10},{10,-10}},
                    rotation=90,
                    origin={-16,140})));
              ThermoPower.Water.SinkPressure sinkSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17611890,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{-80,90},{-100,110}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_SH(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    100.1e5; 20,100.1e5; 30,100.3e5; 40,100.3e5; 50,100.5e5; 60,112.1e5; 70,
                    128.5e5; 80,144.9e5; 90,161.1e5; 100,176.1e5])
                                                             annotation (Placement(
                    transformation(extent={{-160,110},{-140,130}})));
              ThermoPower.Water.SourceMassFlow sourceSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1231860,
                w0=88.59,
                p0=17910000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{60,-320},{40,-300}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_ECO(
                                                          smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    1258351; 20,1258351; 30,1260394; 40,1258763; 50,1251884; 60,1111212; 70,
                    1126638; 80,1133837; 90,1182452; 100,1231860])
                annotation (Placement(transformation(extent={{20,-280},{40,-260}})));
              Modelica.Blocks.Sources.Ramp LOAD2(
                height=loadCalc.load_end - loadCalc.load_start,
                duration=loadCalc.t_ramp,
                offset=loadCalc.load_start,
                startTime=loadCalc.t_start) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-10,-270})));
              Modelica.Blocks.Continuous.FirstOrder valveRH_dynamic(
                initType=Modelica.Blocks.Types.Init.SteadyState,
                k=1,
                T=0.3)
                annotation (Placement(transformation(extent={{-140,260},{-160,280}})));
              IndustrialControlSystems.Controllers.PI PID_valveRH(
                AntiWindup=true,
                useTS=true,
                CSmin=-0.9,
                CSmax=0.2,
                Kp=0.17,
                Ti=16,
                CS_start=0)
                annotation (Placement(transformation(extent={{20,280},{-20,320}})));
              Modelica.Blocks.Sources.Constant valveRH_TS(k=0)
                annotation (Placement(transformation(extent={{40,330},{20,350}})));
              Modelica.Blocks.Tables.CombiTable1D table_w_RH(
                                                           smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    6.08; 20,22.7; 30,39.3155; 40,57.8195; 50,78.6059; 60,102.4; 70,
                    127.897; 80,158.7; 90,197.125; 100,240.532])
                annotation (Placement(transformation(extent={{-40,240},{-60,260}})));
              Modelica.Blocks.Math.Product product
                annotation (Placement(transformation(extent={{-100,272},{-108,280}})));
              Modelica.Blocks.Math.Add3 add
                annotation (Placement(transformation(extent={{-120,266},{-128,274}})));
              Modelica.Blocks.Sources.Constant T_RH_SP(k=826.05)
                annotation (Placement(transformation(extent={{72,302},{52,322}})));
              Modelica.Blocks.Sources.BooleanStep     valveRH_ts(startTime=-500)
                annotation (Placement(transformation(extent={{-40,330},{-20,350}})));
              Modelica.Blocks.Sources.Step step_wRH(
                offset=0,
                startTime=0,
                height=0)
                annotation (Placement(transformation(extent={{-40,200},{-60,220}})));
              IndustrialControlSystems.Controllers.PID PID_pumpFW_high(
                AntiWindup=true,
                useTS=true,
                CSmin=-0.9,
                CSmax=0.5,
                Ti=100,
                Td=5,
                N=20,
                CS_start=0,
                Kp=-0.01)
                annotation (Placement(transformation(extent={{220,-158},{180,-118}})));
             Modelica.Blocks.Continuous.FirstOrder pumpFW_dynamic(
                initType=Modelica.Blocks.Types.Init.SteadyState,
                k=1,
                T=2)         annotation (Placement(transformation(
                    extent={{-10,10},{10,-10}},
                    rotation=180,
                    origin={90,-150})));
              Modelica.Blocks.Sources.Constant pumpFW_TS_high(k=0)
                annotation (Placement(transformation(extent={{280,-90},{260,-70}})));
              Modelica.Blocks.Tables.CombiTable1D table_w_FW(smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments, table=[4,
                    3.54; 20,17.04; 30,26.1634; 40,35.5869; 50,45.2777; 60,55.7; 70,64.0423;
                    80,72.1907; 90,80.1795; 100,88.2715])
                                                annotation (Placement(transformation(
                      extent={{200,-200},{180,-180}})));
              Modelica.Blocks.Math.Product product1
                annotation (Placement(transformation(extent={{148,-144},{140,-136}})));
              Modelica.Blocks.Math.Add3 add1
                annotation (Placement(transformation(extent={{120,-154},{112,-146}})));
              Modelica.Blocks.Sources.Constant DT_sh_SP(k=20)
                annotation (Placement(transformation(extent={{280,-134},{260,-114}})));
              Modelica.Blocks.Sources.BooleanStep     pumpFW_ts(
                  startValue=false, startTime=-200)
                annotation (Placement(transformation(extent={{160,-90},{180,-70}})));
              Modelica.Blocks.Sources.Step step_wFW(
                offset=0,
                startTime=0,
                height=1)
                annotation (Placement(transformation(extent={{180,100},{160,120}})));
              inner ThermoPower.System system annotation (Placement(
                    transformation(extent={{-360,340},{-340,360}})));
              Modelica.Blocks.Math.Add add2 annotation (Placement(transformation(
                    extent={{-4,-4},{4,4}},
                    rotation=-90,
                    origin={94,96})));
              Modelica.Blocks.Math.Add add3
                annotation (Placement(transformation(extent={{-258,178},{-248,188}})));
              Modelica.Blocks.Sources.Step step_w_MS(
                offset=0,
                startTime=0,
                height=0)
                annotation (Placement(transformation(extent={{-300,212},{-280,232}})));
            equation
              connect(valveRH.outlet, RH.inletHot) annotation (Line(
                  points={{-180,160},{20,160},{20,96}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(ECO.outletCold, PDSteamECO2.inlet) annotation (Line(
                  points={{-120,-244},{-120,-230}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(separator.drained_out, waterPump.inlet) annotation (Line(
                  points={{-60,-42},{-60,-64}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                  points={{-82,-32},{-70,-32}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5));
              connect(SH.inletCold, separator.steam_out) annotation (Line(
                  points={{-60,24},{-60,0}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.Filled,Arrow.None}));
              connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
                  points={{-120,-130},{-120,-116}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                  points={{-120,-164},{-120,-150}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                  points={{-120,-84},{-120,-32},{-102,-32}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(RH.outletHot, EVA2.inletHot) annotation (Line(
                  points={{20,64},{20,-100},{-104,-100}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(SH.outletHot, EVA2.inletHot) annotation (Line(
                  points={{-44,40},{20,40},{20,-100},{-104,-100}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                  points={{-136,-100},{-160,-100},{-160,-180},{-136,-180}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                  points={{-104,-180},{-80,-180},{-80,-260},{-104,-260}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
                  points={{4,80},{-20,80},{-20,90}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5));

              connect(EVA1.inletCold, PDSteamECO2.outlet) annotation (Line(
                  points={{-120,-196},{-120,-210}},
                  color={0,0,255},
                  thickness=0.5));
              connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
                  points={{-60,70},{-60,56}},
                  color={0,0,255},
                  thickness=0.5));
              connect(ECO.inletCold, PDSteamECO1.outlet) annotation (Line(
                  points={{-120,-276},{-120,-284}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sensT_FW.outlet, PDSteamECO1.inlet) annotation (Line(
                  points={{-46,-310},{-46,-310},{-66,-310},{-120,-310},{-120,-304}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sensT_MS.inlet, ECO.outletHot) annotation (Line(
                  points={{-154,-260},{-136,-260}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(sourceMS.flange, valveRH.inlet) annotation (Line(
                  points={{-228,160},{-200,160}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(sourceMS.flange, SH.inletHot) annotation (Line(
                  points={{-228,160},{-228,160},{-220,160},{-220,40},{-76,40}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(LOAD.y, table_w_MS.u[1]) annotation (Line(points={{-319,120},{-319,120},
                      {-311,120},{-311,180},{-302,180}}, color={0,0,127}));
              connect(sinkMS.flange, sensT_MS.outlet) annotation (Line(
                  points={{-200,-260},{-184,-260},{-166,-260}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(waterPump.outlet, sinkPressure1.flange) annotation (Line(
                  points={{-60,-84},{-60,-120},{-40,-120}},
                  color={0,0,255},
                  thickness=0.5));
              connect(LOAD1.y, h_ST_RH.u[1])
                annotation (Line(points={{319,180},{319,180},{142,180}}, color={0,0,127}));
              connect(LOAD1.y, w_ST_RH.u[1]) annotation (Line(points={{319,180},{319,180},{150,
                      180},{150,140},{142,140}}, color={0,0,127}));
              connect(h_ST_RH.y[1], sourceSteam1.in_h) annotation (Line(points={{119,180},{119,
                      180},{86,180},{86,86}}, color={0,0,127}));
              connect(RH.inletCold, sourceSteam1.flange) annotation (Line(
                  points={{36,80},{80,80}},
                  color={0,0,255},
                  thickness=0.5));
              connect(p_ST_RH.y[1],sinkSteam1. in_p0) annotation (Line(
                  points={{119,220},{26,220},{26,198.8}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(LOAD1.y, p_ST_RH.u[1]) annotation (Line(points={{319,180},{150,180},{150,
                      220},{142,220}}, color={0,0,127}));
              connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
                  points={{-20,134},{-20,134},{-20,110}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sinkSteam1.flange, sensT_RH.outlet) annotation (Line(
                  points={{20,190},{0,190},{-20,190},{-20,146}},
                  color={0,0,255},
                  thickness=0.5));
              connect(p_ST_SH.y[1],sinkSteam. in_p0) annotation (Line(
                  points={{-139,120},{-86,120},{-86,108.8}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(sinkSteam.flange, PDSteamSH.outlet) annotation (Line(
                  points={{-80,100},{-60,100},{-60,90}},
                  color={0,0,255},
                  thickness=0.5));
              connect(LOAD.y, p_ST_SH.u[1]) annotation (Line(points={{-319,120},{-238,120},{
                      -162,120}}, color={0,0,127}));
              connect(h_ST_ECO.y[1],sourceSteam2. in_h) annotation (Line(points={{41,-270},{
                      46,-270},{46,-304}},    color={0,0,127}));
              connect(sensT_FW.inlet, sourceSteam2.flange) annotation (Line(
                  points={{-34,-310},{40,-310}},
                  color={0,0,255},
                  thickness=0.5));
              connect(LOAD2.y, h_ST_ECO.u[1])
                annotation (Line(points={{1,-270},{9.5,-270},{18,-270}}, color={0,0,127}));
              connect(add.y,valveRH_dynamic. u) annotation (Line(points={{-128.4,270},{-128.4,
                      270},{-138,270}},  color={0,0,127}));
              connect(T_RH_SP.y,PID_valveRH. SP) annotation (Line(points={{51,312},{50,312},
                      {16,312}},         color={0,0,127}));
              connect(add.u1, product.y) annotation (Line(points={{-119.2,273.2},{-114,273.2},
                      {-114,276},{-108.4,276}}, color={0,0,127}));
              connect(valveRH_dynamic.y, valveRH.in_w0) annotation (Line(
                  points={{-161,270},{-194,270},{-194,166}},
                  color={255,215,52},
                  thickness=0.5));
              connect(valveRH_ts.y, PID_valveRH.TS)
                annotation (Line(points={{-19,340},{-8,340},{-8,316}}, color={255,0,255}));
              connect(PID_valveRH.CS, product.u1) annotation (Line(points={{-18,300},{-80,300},
                      {-80,278.4},{-99.2,278.4}}, color={0,0,127}));
              connect(LOAD1.y, table_w_RH.u[1]) annotation (Line(points={{319,180},{319,180},
                      {150,180},{150,250},{-38,250}}, color={0,0,127}));
              connect(table_w_RH.y[1], add.u2) annotation (Line(points={{-61,250},{-90,250},
                      {-90,270},{-119.2,270}}, color={0,0,127}));
              connect(table_w_RH.y[1], product.u2) annotation (Line(points={{-61,250},{-90,250},
                      {-90,273.6},{-99.2,273.6}}, color={0,0,127}));
              connect(step_wRH.y, add.u3) annotation (Line(points={{-61,210},{-86,210},{-108,
                      210},{-108,266.8},{-119.2,266.8}}, color={0,0,127}));

              connect(product1.y,add1. u1)
                annotation (Line(points={{139.6,-140},{140,-140},{134,-140},{134,
                      -146.8},{120.8,-146.8}},                      color={0,0,127}));
              connect(add1.u2,table_w_FW. y[1]) annotation (Line(points={{120.8,
                      -150},{120.8,-150},{140,-150},{160,-150},{160,-190},{179,
                      -190}},           color={0,0,127}));
              connect(product1.u2,table_w_FW. y[1]) annotation (Line(points={{148.8,-142.4},
                      {160,-142.4},{160,-190},{179,-190}},
                                          color={0,0,127}));
              connect(product1.u1,PID_pumpFW_high. CS) annotation (Line(points={{148.8,-137.6},
                      {182,-137.6},{182,-138}},
                                          color={0,0,127}));
              connect(DT_sh_SP.y,PID_pumpFW_high. SP)
                annotation (Line(points={{259,-124},{216,-124},{216,-126}},
                                                                 color={0,0,127}));
              connect(pumpFW_dynamic.u, add1.y)
                annotation (Line(points={{102,-150},{111.6,-150}}, color={0,0,127}));
              connect(pumpFW_dynamic.y, sourceSteam2.in_w0) annotation (Line(points={{79,-150},
                      {64,-150},{54,-150},{54,-304}}, color={255,215,52},
                  thickness=0.5));
              connect(separator.DT_sh, PID_pumpFW_high.PV) annotation (Line(points={{-50,-14},
                      {94,-14},{240,-14},{240,-140},{216,-140}}, color={255,215,
                      52},
                  thickness=0.5));
              connect(pumpFW_ts.y, PID_pumpFW_high.TS) annotation (Line(points={{181,-80},{192,
                      -80},{192,-122}}, color={255,0,255}));
              connect(LOAD1.y, table_w_FW.u[1]) annotation (Line(points={{319,180},{314,180},
                      {300,180},{300,-190},{202,-190}}, color={0,0,127}));
              connect(step_wFW.y, add1.u3) annotation (Line(points={{159,110},{159,110},{130,
                      110},{130,-153.2},{120.8,-153.2}},  color={0,0,127}));
              connect(PID_pumpFW_high.CS, PID_pumpFW_high.TR) annotation (Line(
                    points={{182,-138},{172,-138},{160,-138},{160,-108},{200,-108},
                      {200,-122}}, color={0,0,127}));
              connect(add2.y, sourceSteam1.in_w0) annotation (Line(points={{94,
                      91.6},{94,86},{94,86}}, color={0,0,127}));
              connect(w_ST_RH.y[1], add2.u2) annotation (Line(points={{119,140},{
                      108,140},{91.6,140},{91.6,100.8}}, color={0,0,127}));
              connect(step_wFW.y, add2.u1) annotation (Line(points={{159,110},{130,110},{96.4,
                      110},{96.4,100.8}}, color={0,0,127}));
              connect(PID_valveRH.CS, PID_valveRH.TR) annotation (Line(points={{-18,300},{-24,
                      300},{-26,300},{-26,324},{0,324},{0,316}}, color={0,0,127}));
              connect(sensT_RH.T, PID_valveRH.PV) annotation (Line(
                  points={{-10,148},{-10,148},{-10,238},{-10,240},{40,240},{40,
                      298},{16,298}},
                  color={254,235,86},
                  thickness=0.5));
              connect(table_w_MS.y[1], add3.u2)
                annotation (Line(points={{-279,180},{-259,180}}, color={0,0,127}));
              connect(step_w_MS.y, add3.u1) annotation (Line(points={{-279,222},{-268,222},{
                      -268,186},{-259,186}}, color={0,0,127}));
              connect(add3.y, sourceMS.in_w0) annotation (Line(points={{-247.5,183},{-242,183},
                      {-242,166}}, color={0,0,127}));
              annotation (Diagram(coordinateSystem(extent={{-360,-360},{360,360}},
                      preserveAspectRatio=false)),           Icon(coordinateSystem(
                      extent={{-360,-360},{360,360}}, preserveAspectRatio=false)),
                experiment(
                  StartTime=-2000,
                  StopTime=5000,
                  __Dymola_NumberOfIntervals=5000),
                __Dymola_experimentSetupOutput);
            end test_wFW_sensitivity;

            model test_wFW_sensitivity_2
              "Test model to analyse the sensitivity of the EVA1/EVA2 operational point against the feedwater mass flow"
              import PreFlexMS;

              //Heat exchangers parameter
              parameter Integer Nhex_mono = 5
                "Number of nodes for single phase heat exchangers";
              parameter Integer Nhex_bi = 10
                "Number of nodes for biphase heat exchangers";

              Components.HE ECO(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Shot=2394.2,
                Scold=1855.5,
                Mm=30395,
                Nhex=Nhex_mono,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.728761, gamma_nom=4400),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.461561, gamma_nom=2200),
                Vhot=45.011,
                Vcold=5.752,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.23186e+006,1.44947e+006,1.55292e+006,1.61733e+006,
                      1.66591e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {582.675,605.885,616.62,
                    623.137}),
                nominalColdPressure=17910000,
                nominalHotPressure=569600,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=623.79,
                nominalHotInletTemperature=628.75,
                nominalHotOutletTemperature=586.72,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={199579,192529,183185,168175,136603},
                  noInitialPressure=true),
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-104,-276},{-136,-244}})));

              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Nhex=Nhex_bi,
                nominalHotFlow=610.6,
                Mm=43100,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                nominalColdFlow=88.59,
                Vhot=26.587,
                Vcold=8.651,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.66561e+006,1.67917e+006,1.69246e+006,1.70577e+006,
                      1.7194e+006,1.74575e+006,1.79332e+006,1.87797e+006,2.02847e+006,
                      2.29575e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {626.054,627.39,628.669,
                    629.917,629.888,630.182,630.842,632.016,634.097}),
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={289063,250302,228476,216201,209301,205481,203503,201573,
                      199646,197679},
                  noInitialPressure=true),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=3875,
                    gamma_nom_vap=3875,
                    gamma_nom_2ph=10000,
                    alpha=1.6),
                Shot=2251.9,
                Scold=1765.5,
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                                                    annotation (Placement(
                    transformation(extent={{-136,-196},{-104,-164}})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Nhex=Nhex_bi,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                Vhot=3.905,
                Vcold=1.065,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={2.2799e+006,2.32986e+006,2.38391e+006,2.4424e+006,
                      2.50567e+006,2.55859e+006,2.60531e+006,2.65251e+006,
                      2.69972e+006,2.74645e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {650.941,652.73,654.665,
                    656.757,670.379,679.959,684.074,688.573,693.427}),
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={355877,349098,342248,335400,328621,320944,311763,303278,
                      295435,288187},
                  noInitialPressure=true),
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=2000,
                    gamma_nom_2ph=5000,
                    gamma_nom_vap=2000,
                    alpha=1.1),
                Shot=429.2,
                Scold=338,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                                                    annotation (Placement(
                    transformation(extent={{-104,-116},{-136,-84}})));

              Components.Equilibrium2phVessel separator(
                H=8,
                zl_start=0,
                h_start=2.7e6,
                D=0.56,
                noInitialPressure=true,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p_start=17710000)
                annotation (Placement(transformation(extent={{-88,-52},{-32,4}})));
              ThermoPower.Water.ThroughMassFlow waterPump(
                allowFlowReversal=false,
                w0=0,
                use_in_w0=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={-60,-74})));

              Components.HE SH(
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Mm=15353,
                nominalColdFlow=88.59,
                nominalHotFlow=371.8,
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2200, alpha=0.847436),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=3000, alpha=0.28867),
                Nhex=Nhex_mono,
                Vhot=3.68,
                Vcold=2.683,
                tubeWalls(Tvolstart(displayUnit="K") = {725.391,769.727,804.861,
                    826.169}),
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={2.74562e+006,3.02031e+006,3.23297e+006,3.36066e+006,
                      3.42731e+006},
                  noInitialPressure=true),
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={518322,502443,472016,421345,355895},
                  noInitialPressure=true),
                Shot=1153.8,
                Scold=879.8,
                nominalColdPressure=17710000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=648.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=732.02,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                    rotation=0,
                    origin={-60,40})));
              Components.HE RH(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                Mm=10081,
                nominalColdFlow=82.57,
                nominalHotFlow=238.7,
                Nhex=Nhex_mono,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.798347, gamma_nom=970),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2600, alpha=0.320782),
                Vhot=1.522,
                Vcold=5.092,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={3.0832e+006,3.30568e+006,3.43926e+006,3.51621e+006,
                      3.56013e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {734.495,783.217,812.321,
                    829.047}),
                Shot=1339,
                Scold=1198,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={518322,503128,476511,430304,353343},
                  noInitialPressure=true),
                nominalColdPressure=4698000,
                nominalHotPressure=1000000,
                nominalColdInletTemperature=625.65,
                nominalColdOutletTemperature=826.06,
                nominalHotInletTemperature=838.15,
                nominalHotOutletTemperature=731.45,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                    rotation=-90,
                    origin={20,80})));
              ThermoPower.Water.ThroughMassFlow valveRH(
                w0=238.7,
                use_in_w0=true,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-200,150},{-180,
                        170}})));

              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                dpnom=(0.05828)*10^5,
                wnom=88.59,
                rhonom=168.8,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-120,-140})));
              ThermoPower.Water.PressDrop PDSteamECO2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                rhonom=582.8,
                dpnom=(0.52492/2)*10^5,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                        annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-120,-220})));
              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=97,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-92,-32})));
              //Overall parameter and variables:
              Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
              Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
              Modelica.SIunits.Temperature T_outMS = ECO.hotFluid.T[end];
              Modelica.SIunits.Temperature T_RH = sensT_RH.T;
              Modelica.SIunits.Length sep_lev = separator.zl;
              Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
              Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
              Real x_EVA1_out = EVA1.coldFluid.x[end];
              Real x_EVA2_out = EVA2.coldFluid.x[end];
              Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
              Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
              Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
              Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;
              //Modelica.SIunits.TemperatureDifference deltaT_SH_star = ((ECO.coldFluid.T[end] - ECO.coldFluid.T[1]) - (ECO.hotFluid.T[1] - ECO.hotFluid.T[end]))/(ECO.coldFluid.T[end] - ECO.coldFluid.T[1]);
              //Modelica.SIunits.TemperatureDifference deltaT_EVA2_star = ((EVA2.coldFluid.T[end] - EVA2.coldFluid.T[1]) - (EVA2.hotFluid.T[1] - EVA2.hotFluid.T[end]))/(EVA2.hotFluid.T[1] - EVA2.hotFluid.T[end]);
              //Modelica.SIunits.TemperatureDifference deltaT_EVA1_star = ((EVA1.coldFluid.T[end] - EVA1.coldFluid.T[1]) - (EVA1.hotFluid.T[1] - EVA1.hotFluid.T[end]))/(EVA1.hotFluid.T[1] - EVA1.hotFluid.T[end]);
              Modelica.SIunits.TemperatureDifference deltaT_SH_hot = SH.hotFluid.T[1] - SH.hotFluid.T[end];
              Modelica.SIunits.TemperatureDifference deltaT_SH_cold = SH.coldFluid.T[end] - SH.coldFluid.T[1];
              Modelica.SIunits.Power Q_sh = separator.w_in*(separator.h - separator.hv);
              Modelica.SIunits.Power Q_eva = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot;
              Real Q_sh_perc = Q_sh/Q_tot;

              ThermoPower.Water.PressDrop PDSteamRH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=82.57,
                dpnom=(0.55113)*10^5,
                rhonom=12.52,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-20,100})));
              ThermoPower.Water.SensT sensT_FW(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph)
                annotation (Placement(transformation(extent={{-30,-316},{-50,-296}})));
              ThermoPower.Water.SensT sensT_MS(redeclare package Medium =
                    PreFlexMS.Media.MoltenSalt60_40_quad) annotation (Placement(
                    transformation(extent={{-150,-266},{-170,-246}})));
              ThermoPower.Water.PressDrop PDSteamSH(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                dpnom=(177.1 - 176.1189)*10^5,
                wnom=88.59,
                rhonom=51.9,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-60,80})));
              ThermoPower.Water.PressDrop PDSteamECO1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.52492/2)*10^5,
                rhonom=767.7,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-120,-294})));
              PreFlexMS.Auxiliaries.Components.loadCalculator loadCalc(
                t_start=0,
                load_start=100,
                load_end=60,
                t_ramp=60*10) annotation (Placement(transformation(extent={{-320,
                        -60},{-280,-20}})));
              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                h=518322,
                w0=610.6,
                use_in_w0=true,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad)
                annotation (Placement(transformation(extent={{-248,150},{-228,
                        170}})));
              Modelica.Blocks.Sources.Ramp LOAD(
                height=loadCalc.load_end - loadCalc.load_start,
                duration=loadCalc.t_ramp,
                offset=loadCalc.load_start,
                startTime=loadCalc.t_start) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-330,120})));
              Modelica.Blocks.Tables.CombiTable1D table_w_MS(
                  table=[20,122.1; 30,183.3; 40,244.2; 50,305.3; 60,366.3; 70,427.4; 80,
                    488.4; 90,549.4; 100,610.6], smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative)
                annotation (Placement(transformation(extent={{-300,170},{-280,190}})));
              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=839319) annotation (Placement(transformation(extent={{-200,-270},
                        {-220,-250}})));

              ThermoPower.Water.SinkPressure sinkPressure1(redeclare package
                  Medium =
                    Modelica.Media.Water.WaterIF97_ph, p0=17700000)
                annotation (Placement(transformation(extent={{-40,-130},{-20,-110}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_RH(            smoothness=
                    Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,3092400;
                    20,3124593; 30,3130270; 40,3128892; 50,3125327; 60,3116961; 70,3108752;
                    80,3100618; 90,3093346; 100,3083198])
                annotation (Placement(transformation(extent={{140,170},{120,190}})));
              Modelica.Blocks.Tables.CombiTable1D w_ST_RH(
                                           smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    4.18; 20,13.4; 30,21.5; 40,29.69; 50,38.01; 60,46.38; 70,54.15; 80,63.31;
                    90,72.7; 100,82.57])
                annotation (Placement(transformation(extent={{140,130},{120,150}})));
              ThermoPower.Water.SourceMassFlow sourceSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=82.57,
                h=3083198,
                p0=4698000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{100,70},{80,90}})));
              Modelica.Blocks.Sources.Ramp LOAD1(
                height=loadCalc.load_end - loadCalc.load_start,
                duration=loadCalc.t_ramp,
                offset=loadCalc.load_start,
                startTime=loadCalc.t_start) annotation (Placement(transformation(
                    extent={{10,-10},{-10,10}},
                    rotation=0,
                    origin={330,180})));
              ThermoPower.Water.SinkPressure sinkSteam1(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=4642887,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{20,180},{40,200}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_RH(        smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    22e5; 20,7.67e5; 30,12.28e5; 40,16.92e5; 50,21.61e5; 60,26.32e5; 70,30.66e5;
                    80,35.77e5; 90,40.98e5; 100,46.4e5])
                annotation (Placement(transformation(extent={{140,210},{120,230}})));
              ThermoPower.Water.SensT sensT_RH(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph) annotation (Placement(transformation(
                    extent={{-10,10},{10,-10}},
                    rotation=90,
                    origin={-16,140})));
              ThermoPower.Water.SinkPressure sinkSteam(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17611890,
                use_in_p0=true)
                annotation (Placement(transformation(extent={{-80,90},{-100,110}})));
              Modelica.Blocks.Tables.CombiTable1D p_ST_SH(smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    100.1e5; 20,100.1e5; 30,100.3e5; 40,100.3e5; 50,100.5e5; 60,112.1e5; 70,
                    128.5e5; 80,144.9e5; 90,161.1e5; 100,176.1e5])
                                                             annotation (Placement(
                    transformation(extent={{-160,110},{-140,130}})));
              ThermoPower.Water.SourceMassFlow sourceSteam2(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1231860,
                w0=88.59,
                p0=17910000,
                use_in_w0=true,
                use_in_h=true)
                annotation (Placement(transformation(extent={{60,-320},{40,-300}})));
              Modelica.Blocks.Tables.CombiTable1D h_ST_ECO(
                                                          smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    1258351; 20,1258351; 30,1260394; 40,1258763; 50,1251884; 60,1111212; 70,
                    1126638; 80,1133837; 90,1182452; 100,1231860])
                annotation (Placement(transformation(extent={{20,-280},{40,-260}})));
              Modelica.Blocks.Sources.Ramp LOAD2(
                height=loadCalc.load_end - loadCalc.load_start,
                duration=loadCalc.t_ramp,
                offset=loadCalc.load_start,
                startTime=loadCalc.t_start) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-10,-270})));
              Modelica.Blocks.Continuous.FirstOrder valveRH_dynamic(
                initType=Modelica.Blocks.Types.Init.SteadyState,
                k=1,
                T=0.3)
                annotation (Placement(transformation(extent={{-140,260},{-160,280}})));
              IndustrialControlSystems.Controllers.PI PID_valveRH(
                AntiWindup=true,
                useTS=true,
                CSmin=-0.9,
                CSmax=0.2,
                Kp=0.17,
                Ti=16,
                CS_start=0)
                annotation (Placement(transformation(extent={{20,280},{-20,320}})));
              Modelica.Blocks.Sources.Constant valveRH_TS(k=0)
                annotation (Placement(transformation(extent={{40,330},{20,350}})));
              Modelica.Blocks.Tables.CombiTable1D table_w_RH(
                                                           smoothness=Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[4,
                    6.08; 20,22.7; 30,39.3155; 40,57.8195; 50,78.6059; 60,102.4; 70,
                    127.897; 80,158.7; 90,197.125; 100,240.532])
                annotation (Placement(transformation(extent={{-40,240},{-60,260}})));
              Modelica.Blocks.Math.Product product
                annotation (Placement(transformation(extent={{-100,272},{-108,280}})));
              Modelica.Blocks.Math.Add3 add
                annotation (Placement(transformation(extent={{-120,266},{-128,274}})));
              Modelica.Blocks.Sources.Constant T_RH_SP(k=826.05)
                annotation (Placement(transformation(extent={{72,302},{52,322}})));
              Modelica.Blocks.Sources.BooleanStep     valveRH_ts(startTime=5000)
                annotation (Placement(transformation(extent={{-40,330},{-20,350}})));
              Modelica.Blocks.Sources.Step step_wRH(
                offset=0,
                startTime=0,
                height=0)
                annotation (Placement(transformation(extent={{-40,200},{-60,220}})));
              IndustrialControlSystems.Controllers.PID PID_pumpFW_high(
                AntiWindup=true,
                useTS=true,
                CSmin=-0.9,
                N=20,
                CS_start=0,
                CSmax=1,
                Kp=-0.007514,
                Ti=243.4,
                Td=45)
                annotation (Placement(transformation(extent={{220,-158},{180,-118}})));
             Modelica.Blocks.Continuous.FirstOrder pumpFW_dynamic(
                initType=Modelica.Blocks.Types.Init.SteadyState,
                k=1,
                T=2)         annotation (Placement(transformation(
                    extent={{-10,10},{10,-10}},
                    rotation=180,
                    origin={90,-150})));
              Modelica.Blocks.Sources.Constant pumpFW_TS_high(k=0)
                annotation (Placement(transformation(extent={{280,-90},{260,-70}})));
              Modelica.Blocks.Tables.CombiTable1D table_w_FW(smoothness=Modelica.Blocks.Types.Smoothness.LinearSegments, table=[4,
                    3.54; 20,17.04; 30,26.1634; 40,35.5869; 50,45.2777; 60,55.7; 70,64.0423;
                    80,72.1907; 90,80.1795; 100,88.2715])
                                                annotation (Placement(transformation(
                      extent={{200,-200},{180,-180}})));
              Modelica.Blocks.Math.Product product1
                annotation (Placement(transformation(extent={{148,-144},{140,-136}})));
              Modelica.Blocks.Math.MultiSum
                                        add1(nu=3)
                annotation (Placement(transformation(extent={{120,-154},{112,-146}})));
              Modelica.Blocks.Sources.BooleanStep     pumpFW_ts(
                  startValue=false, startTime=5000)
                annotation (Placement(transformation(extent={{160,-90},{180,-70}})));
              Modelica.Blocks.Sources.Step step_wFW(
                offset=0,
                startTime=0,
                height=-1)
                annotation (Placement(transformation(extent={{180,100},{160,120}})));
              inner ThermoPower.System system annotation (Placement(
                    transformation(extent={{-360,340},{-340,360}})));
              Modelica.Blocks.Math.Add add2 annotation (Placement(transformation(
                    extent={{-4,-4},{4,4}},
                    rotation=-90,
                    origin={94,96})));
              Modelica.Blocks.Math.Add add3
                annotation (Placement(transformation(extent={{-258,178},{-248,188}})));
              Modelica.Blocks.Sources.Step step_w_MS(
                offset=0,
                startTime=0,
                height=0)
                annotation (Placement(transformation(extent={{-300,212},{-280,232}})));
              Modelica.Blocks.Sources.RealExpression realExpression(y=SH.hotFluid.T[
                    end] - RH.hotFluid.T[end]) annotation (Placement(
                    transformation(extent={{280,-170},{260,-150}})));
              Modelica.Blocks.Tables.CombiTable1D DT_MS_sp(           smoothness=
                    Modelica.Blocks.Types.Smoothness.ContinuousDerivative, table=[
                    60,40.54; 70,31.85; 80,22.12; 90,11.71; 100,1.16])
                annotation (Placement(transformation(extent={{280,-134},{260,-114}})));
            equation
              connect(valveRH.outlet, RH.inletHot) annotation (Line(
                  points={{-180,160},{20,160},{20,96}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(ECO.outletCold, PDSteamECO2.inlet) annotation (Line(
                  points={{-120,-244},{-120,-230}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(separator.drained_out, waterPump.inlet) annotation (Line(
                  points={{-60,-42},{-60,-64}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                  points={{-82,-32},{-70,-32}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5));
              connect(SH.inletCold, separator.steam_out) annotation (Line(
                  points={{-60,24},{-60,0}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.Filled,Arrow.None}));
              connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
                  points={{-120,-130},{-120,-116}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                  points={{-120,-164},{-120,-150}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                  points={{-120,-84},{-120,-32},{-102,-32}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(RH.outletHot, EVA2.inletHot) annotation (Line(
                  points={{20,64},{20,-100},{-104,-100}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(SH.outletHot, EVA2.inletHot) annotation (Line(
                  points={{-44,40},{20,40},{20,-100},{-104,-100}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                  points={{-136,-100},{-160,-100},{-160,-180},{-136,-180}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                  points={{-104,-180},{-80,-180},{-80,-260},{-104,-260}},
                  color={255,0,0},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
                  points={{4,80},{-20,80},{-20,90}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5));

              connect(EVA1.inletCold, PDSteamECO2.outlet) annotation (Line(
                  points={{-120,-196},{-120,-210}},
                  color={0,0,255},
                  thickness=0.5));
              connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
                  points={{-60,70},{-60,56}},
                  color={0,0,255},
                  thickness=0.5));
              connect(ECO.inletCold, PDSteamECO1.outlet) annotation (Line(
                  points={{-120,-276},{-120,-284}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sensT_FW.outlet, PDSteamECO1.inlet) annotation (Line(
                  points={{-46,-310},{-46,-310},{-66,-310},{-120,-310},{-120,-304}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sensT_MS.inlet, ECO.outletHot) annotation (Line(
                  points={{-154,-260},{-136,-260}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(sourceMS.flange, valveRH.inlet) annotation (Line(
                  points={{-228,160},{-200,160}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(sourceMS.flange, SH.inletHot) annotation (Line(
                  points={{-228,160},{-228,160},{-220,160},{-220,40},{-76,40}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(LOAD.y, table_w_MS.u[1]) annotation (Line(points={{-319,120},{-319,120},
                      {-311,120},{-311,180},{-302,180}}, color={0,0,127}));
              connect(sinkMS.flange, sensT_MS.outlet) annotation (Line(
                  points={{-200,-260},{-184,-260},{-166,-260}},
                  thickness=0.5,
                  color={255,0,0}));
              connect(waterPump.outlet, sinkPressure1.flange) annotation (Line(
                  points={{-60,-84},{-60,-120},{-40,-120}},
                  color={0,0,255},
                  thickness=0.5));
              connect(LOAD1.y, h_ST_RH.u[1])
                annotation (Line(points={{319,180},{319,180},{142,180}}, color={0,0,127}));
              connect(LOAD1.y, w_ST_RH.u[1]) annotation (Line(points={{319,180},{319,180},{150,
                      180},{150,140},{142,140}}, color={0,0,127}));
              connect(h_ST_RH.y[1], sourceSteam1.in_h) annotation (Line(points={{119,180},
                      {119,180},{86,180},{86,86}},
                                              color={0,0,127}));
              connect(RH.inletCold, sourceSteam1.flange) annotation (Line(
                  points={{36,80},{80,80}},
                  color={0,0,255},
                  thickness=0.5));
              connect(p_ST_RH.y[1],sinkSteam1. in_p0) annotation (Line(
                  points={{119,220},{26,220},{26,198.8}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(LOAD1.y, p_ST_RH.u[1]) annotation (Line(points={{319,180},{150,180},{150,
                      220},{142,220}}, color={0,0,127}));
              connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
                  points={{-20,134},{-20,134},{-20,110}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sinkSteam1.flange, sensT_RH.outlet) annotation (Line(
                  points={{20,190},{0,190},{-20,190},{-20,146}},
                  color={0,0,255},
                  thickness=0.5));
              connect(p_ST_SH.y[1],sinkSteam. in_p0) annotation (Line(
                  points={{-139,120},{-86,120},{-86,108.8}},
                  color={0,0,127},
                  smooth=Smooth.None));
              connect(sinkSteam.flange, PDSteamSH.outlet) annotation (Line(
                  points={{-80,100},{-60,100},{-60,90}},
                  color={0,0,255},
                  thickness=0.5));
              connect(LOAD.y, p_ST_SH.u[1]) annotation (Line(points={{-319,120},{-238,120},{
                      -162,120}}, color={0,0,127}));
              connect(h_ST_ECO.y[1],sourceSteam2. in_h) annotation (Line(points={{41,-270},{
                      46,-270},{46,-304}},    color={0,0,127}));
              connect(sensT_FW.inlet, sourceSteam2.flange) annotation (Line(
                  points={{-34,-310},{40,-310}},
                  color={0,0,255},
                  thickness=0.5));
              connect(LOAD2.y, h_ST_ECO.u[1])
                annotation (Line(points={{1,-270},{9.5,-270},{18,-270}}, color={0,0,127}));
              connect(add.y,valveRH_dynamic. u) annotation (Line(points={{-128.4,270},{-128.4,
                      270},{-138,270}},  color={0,0,127}));
              connect(T_RH_SP.y,PID_valveRH. SP) annotation (Line(points={{51,312},{50,312},
                      {16,312}},         color={0,0,127}));
              connect(add.u1, product.y) annotation (Line(points={{-119.2,273.2},{-114,273.2},
                      {-114,276},{-108.4,276}}, color={0,0,127}));
              connect(valveRH_dynamic.y, valveRH.in_w0) annotation (Line(
                  points={{-161,270},{-194,270},{-194,166}},
                  color={255,215,52},
                  thickness=0.5));
              connect(valveRH_ts.y, PID_valveRH.TS)
                annotation (Line(points={{-19,340},{-8,340},{-8,316}}, color={255,0,255}));
              connect(PID_valveRH.CS, product.u1) annotation (Line(points={{-18,300},{-80,300},
                      {-80,278.4},{-99.2,278.4}}, color={0,0,127}));
              connect(LOAD1.y, table_w_RH.u[1]) annotation (Line(points={{319,180},{319,180},
                      {150,180},{150,250},{-38,250}}, color={0,0,127}));
              connect(table_w_RH.y[1], add.u2) annotation (Line(points={{-61,250},{-90,250},
                      {-90,270},{-119.2,270}}, color={0,0,127}));
              connect(table_w_RH.y[1], product.u2) annotation (Line(points={{-61,250},{-90,250},
                      {-90,273.6},{-99.2,273.6}}, color={0,0,127}));
              connect(step_wRH.y, add.u3) annotation (Line(points={{-61,210},{-86,210},{-108,
                      210},{-108,266.8},{-119.2,266.8}}, color={0,0,127}));

              connect(product1.u2,table_w_FW. y[1]) annotation (Line(points={{148.8,-142.4},
                      {160,-142.4},{160,-190},{179,-190}},
                                          color={0,0,127}));
              connect(product1.u1,PID_pumpFW_high. CS) annotation (Line(points={{148.8,-137.6},
                      {182,-137.6},{182,-138}},
                                          color={0,0,127}));
              connect(pumpFW_dynamic.u, add1.y)
                annotation (Line(points={{102,-150},{111.32,-150}},color={0,0,127}));
              connect(pumpFW_dynamic.y, sourceSteam2.in_w0) annotation (Line(points={{79,-150},
                      {64,-150},{54,-150},{54,-304}}, color={255,215,52},
                  thickness=0.5));
              connect(pumpFW_ts.y, PID_pumpFW_high.TS) annotation (Line(points={{181,-80},{192,
                      -80},{192,-122}}, color={255,0,255}));
              connect(LOAD1.y, table_w_FW.u[1]) annotation (Line(points={{319,180},{314,180},
                      {300,180},{300,-190},{202,-190}}, color={0,0,127}));
              connect(PID_pumpFW_high.CS, PID_pumpFW_high.TR) annotation (Line(
                    points={{182,-138},{172,-138},{160,-138},{160,-108},{200,-108},
                      {200,-122}}, color={0,0,127}));
              connect(add2.y, sourceSteam1.in_w0) annotation (Line(points={{94,
                      91.6},{94,86},{94,86}}, color={0,0,127}));
              connect(w_ST_RH.y[1], add2.u2) annotation (Line(points={{119,140},{
                      108,140},{91.6,140},{91.6,100.8}}, color={0,0,127}));
              connect(step_wFW.y, add2.u1) annotation (Line(points={{159,110},{130,110},{96.4,
                      110},{96.4,100.8}}, color={0,0,127}));
              connect(PID_valveRH.CS, PID_valveRH.TR) annotation (Line(points={{-18,300},{-24,
                      300},{-26,300},{-26,324},{0,324},{0,316}}, color={0,0,127}));
              connect(sensT_RH.T, PID_valveRH.PV) annotation (Line(
                  points={{-10,148},{-10,148},{-10,238},{-10,240},{40,240},{40,
                      298},{16,298}},
                  color={254,235,86},
                  thickness=0.5));
              connect(table_w_MS.y[1], add3.u2)
                annotation (Line(points={{-279,180},{-259,180}}, color={0,0,127}));
              connect(step_w_MS.y, add3.u1) annotation (Line(points={{-279,222},{-268,222},{
                      -268,186},{-259,186}}, color={0,0,127}));
              connect(add3.y, sourceMS.in_w0) annotation (Line(points={{-247.5,183},{-242,183},
                      {-242,166}}, color={0,0,127}));
              connect(realExpression.y, PID_pumpFW_high.PV) annotation (Line(
                    points={{259,-160},{240,-160},{240,-140},{216,-140}}, color={
                      0,0,127}));
              connect(product1.y, add1.u[1]) annotation (Line(points={{139.6,
                      -140},{130,-140},{130,-148.133},{120,-148.133}},
                                                                 color={0,0,127}));
              connect(table_w_FW.y, add1.u[2:2]) annotation (Line(points={{179,
                      -190},{156,-190},{130,-190},{130,-150},{120,-150}}, color={
                      0,0,127}));
              connect(step_wFW.y, add1.u[3]) annotation (Line(points={{159,110},
                      {159,110},{128,110},{128,-151.867},{120,-151.867}},color={0,
                      0,127}));
              connect(PID_pumpFW_high.SP, DT_MS_sp.y[1]) annotation (Line(points=
                      {{216,-126},{259,-126},{259,-124}}, color={0,0,127}));
              connect(LOAD1.y, DT_MS_sp.u[1]) annotation (Line(points={{319,180},
                      {318,180},{300,180},{300,-124},{282,-124}}, color={0,0,127}));
              annotation (Diagram(coordinateSystem(extent={{-360,-360},{360,360}},
                      preserveAspectRatio=false)),           Icon(coordinateSystem(
                      extent={{-360,-360},{360,360}}, preserveAspectRatio=false)),
                experiment(
                  StartTime=-2000,
                  StopTime=5000,
                  __Dymola_NumberOfIntervals=5000),
                __Dymola_experimentSetupOutput);
            end test_wFW_sensitivity_2;

            model simplifiedSystem
              "Test model for the simplified analysis of the DTsh sensitivity over wFW"

              //Temperature
              parameter Modelica.SIunits.Temperature T_hot_SH_in;
              parameter Modelica.SIunits.Temperature T_hot_EVA1_out;
              parameter Modelica.SIunits.Temperature T_hot_RH_out;
              parameter Modelica.SIunits.Temperature T_cold_SH_out;
              parameter Modelica.SIunits.Temperature T_cold_EVA1_in;

              //Number of Thermal Units
              parameter Real NTU_SH;
              parameter Real NTU_EVA2;
              parameter Real NTU_EVA1;

              //Temperature difference ratio
              parameter Real deltaT_SH_star;
              parameter Real deltaT_EVA1_star;

              Modelica.Blocks.Interfaces.RealInput a annotation (Placement(transformation(
                      extent={{-140,40},{-100,80}}), iconTransformation(extent={{-140,40},{-100,
                        80}})));
              Modelica.Blocks.Interfaces.RealInput deltaT_EVA2_star annotation (Placement(
                    transformation(extent={{-140,-80},{-100,-40}}), iconTransformation(
                      extent={{-140,-80},{-100,-40}})));

              Modelica.SIunits.Temperature T_cold_EVA2_out(start = 649);
              Modelica.SIunits.Temperature T_hot_EVA2_in(start = 731);
              Modelica.SIunits.Temperature T_hot_EVA1_in(start = 687);
              Modelica.SIunits.Temperature T_hot_SH_out(start = 730);

            equation
              (T_cold_SH_out - T_hot_SH_in)/(T_cold_EVA2_out - T_hot_SH_out) = exp(NTU_SH*deltaT_SH_star);
              (T_cold_EVA2_out - T_hot_EVA2_in)/(T_cold_EVA1_in - T_hot_EVA1_in) = exp(NTU_EVA2*deltaT_EVA2_star);
              (T_cold_EVA1_in - T_hot_EVA1_in)/(T_cold_EVA1_in - T_hot_EVA1_out) = exp(NTU_EVA1*deltaT_EVA1_star);
              T_hot_EVA2_in = a*T_hot_RH_out + (1-a)*T_hot_SH_out;

              annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                        {100,100}}), graphics={Rectangle(extent={{-100,100},{100,-100}},
                        lineColor={0,0,127}), Text(
                      extent={{-80,80},{80,-80}},
                      lineColor={0,0,127},
                      fillPattern=FillPattern.Solid,
                      textString="System",
                      textStyle={TextStyle.Bold})}));
            end simplifiedSystem;

            model simplifiedSystem_2
              "Test model for the simplified analysis of the DTsh sensitivity over wFW"

              //Temperature
              parameter Modelica.SIunits.Temperature T_A;
              parameter Modelica.SIunits.Temperature T_F;
              parameter Modelica.SIunits.Temperature T_B;
              parameter Modelica.SIunits.Temperature T_4;
              parameter Modelica.SIunits.Temperature T_1;

              //Number of Thermal Units
              parameter Real NTU_SH;
              parameter Real NTU_EVA2;
              parameter Real NTU_EVA1;

              //Temperature difference ratio
              //parameter Real deltaT_SH_star;
              //parameter Real deltaT_EVA1_star;

              Modelica.Blocks.Interfaces.RealInput a annotation (Placement(transformation(
                      extent={{-140,40},{-100,80}}), iconTransformation(extent={{-140,40},{-100,
                        80}})));
              Modelica.Blocks.Interfaces.RealInput deltaT_EVA2_star annotation (Placement(
                    transformation(extent={{-140,-80},{-100,-40}}), iconTransformation(
                      extent={{-140,-80},{-100,-40}})));

              Modelica.SIunits.Temperature T_3(start = 649);
              Modelica.SIunits.Temperature T_D(start = 731);
              Modelica.SIunits.Temperature T_E(start = 687);
              Modelica.SIunits.Temperature T_C(start = 730);

              Real deltaT_SH_star;
              Real deltaT_EVA1_star;

            equation
              (T_4 - T_A)/(T_3 - T_C) = exp(NTU_SH*deltaT_SH_star);
              (T_3 - T_D)/(T_1 - T_E) = exp(NTU_EVA2*deltaT_EVA2_star);
              (T_1 - T_E)/(T_1 - T_F) = exp(NTU_EVA1*deltaT_EVA1_star);
              T_D = a*T_B + (1-a)*T_C;
              deltaT_SH_star = ((T_4 - T_3) - (T_A - T_C))/(T_4 - T_3);
              deltaT_EVA1_star = ((T_1 - T_1) - (T_E - T_F))/(T_E - T_F);

              annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                        {100,100}}), graphics={Rectangle(extent={{-100,100},{100,-100}},
                        lineColor={0,0,127}), Text(
                      extent={{-80,80},{80,-80}},
                      lineColor={0,0,127},
                      fillPattern=FillPattern.Solid,
                      textString="System",
                      textStyle={TextStyle.Bold})}));
            end simplifiedSystem_2;

            model test_solvedSystem

              simplifiedSystem_2 simplifiedSys(
                NTU_SH=2.78,
                NTU_EVA2=0.87,
                NTU_EVA1=4.66,
                T_A=838.15,
                T_F=627.15,
                T_B=731.15,
                T_4=825.75,
                T_1=622.15,
                T_C(start=725.15))
                annotation (Placement(transformation(extent={{0,-10},{20,10}})));
              Modelica.Blocks.Sources.Ramp a(
                height=0,
                duration=0,
                offset=0.394,
                startTime=0) annotation (Placement(transformation(extent={{-60,10},
                        {-40,30}})));
              Modelica.Blocks.Sources.Ramp dT_star(
                height=0,
                duration=0,
                offset=-0.5429,
                startTime=0) annotation (Placement(transformation(extent={{-60,
                        -30},{-40,-10}})));
            equation
              connect(a.y, simplifiedSys.a) annotation (Line(points={{-39,20},{
                      -30,20},{-20,20},{-20,6},{-2,6}}, color={0,0,127}));
              connect(dT_star.y, simplifiedSys.deltaT_EVA2_star) annotation (Line(
                    points={{-39,-20},{-20,-20},{-20,-6},{-2,-6}}, color={0,0,127}));
              annotation (
                Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                        -100},{100,100}})),
                experiment(__Dymola_NumberOfIntervals=5000),
                __Dymola_experimentSetupOutput);
            end test_solvedSystem;
          end HE;

          package OTSG "Package of whole OTSG test models"

            model test_OTSG_100 "Test model of the OTSG at 100% load"

              ThermoPower.Water.SourceMassFlow sourceFW(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1231860,
                use_in_w0=false,
                w0=231.43)
                annotation (Placement(transformation(extent={{80,-10},{60,-30}})));
              ThermoPower.Water.SourceMassFlow sourceRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=82.57,
                h=3083198,
                use_in_w0=false,
                p0=4698000)
                annotation (Placement(transformation(extent={{80,30},{60,50}})));
              ThermoPower.Water.SinkPressure sinkRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=4642887)
                annotation (Placement(transformation(extent={{20,90},{40,110}})));
              ThermoPower.Water.SourcePressure sourceMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322,
                p0=1000000) annotation (Placement(transformation(extent={{-60,
                        90},{-40,110}})));

              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=1000000) annotation (Placement(transformation(extent={{-30,-70},
                        {-50,-50}})));

              inner ThermoPower.System system(allowFlowReversal=false, initOpt=
                    ThermoPower.Choices.Init.Options.steadyState)
                annotation (Placement(transformation(extent={{-120,100},{-100,120}})));
              ThermoPower.Water.SinkPressure sinkHPH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=17814720)
                annotation (Placement(transformation(extent={{40,-70},{60,-50}})));
              ThermoPower.Water.SinkPressure sinkSH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=17500000)
                annotation (Placement(transformation(extent={{40,70},{60,90}})));
              ThermoPower.Water.ThroughMassFlow pump_HPH(
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=143.1,
                use_in_w0=true)
                annotation (Placement(transformation(extent={{0,-50},{20,-70}})));
              Control.Old.plant_controller_HPpart_old controllers(
                actuatorGain_valveRH=1,
                Ta_valveRH=2,
                Ti_pumpSEP=56.7,
                Td_pumpSEP=0,
                Nd_pumpSEP=1,
                CSstart_pumpSEP=0,
                actuatorGain_pumpSEP=1,
                Ta_pumpSEP=5,
                TS_pumpSEP=0,
                Ta_pumpFW=5,
                Td_pumpHPH=0,
                Nd_pumpHPH=1,
                actuatorGain_pumpHPH=1,
                Ta_pumpHPH=5,
                actuatorGain_pumpFW=1,
                P_mec_sp=79.7258e6,
                actuatorGain_pumpMS=1,
                Ta_pumpMS=5,
                Td_valveTURB=0,
                Nd_valveTURB=1,
                actuatorGain_valveTURB=1,
                CSmin_pumpFW=-0.3,
                CSmax_pumpFW=0.3,
                CSstart_pumpFW=0,
                TS_pumpFW=0,
                CSmin_pumpHPH=-0.5,
                CSmax_pumpHPH=0.5,
                CSstart_pumpHPH=0,
                TS_pumpHPH=0,
                CSmin_pumpMS=-0.9,
                CSmax_pumpMS=1,
                CSstart_pumpMS=0,
                TS_pumpMS=0,
                CSmin_valveTURB=-0.9,
                CSmax_valveTURB=0.65,
                CSstart_valveTURB=0,
                TS_valveTURB=0,
                Nd_valveRH=10,
                Ti_valveRH=16,
                Td_valveRH=0,
                useTS_pumpSEP=false,
                t_ramp=0,
                t_start=0,
                load_start=100,
                load_end=100,
                Kp_pumpFW=-1,
                Ti_pumpFW=28.59,
                Td_pumpFW=7.147,
                Nd_pumpFW=30,
                Kp_pumpHPH=0.6461,
                Ti_pumpHPH=19.05,
                Kp_pumpMS=1.2e-7,
                Ti_pumpMS=114.3,
                Td_pumpMS=28.57,
                Nd_pumpMS=20,
                Ti_valveTURB=95.23,
                Kp_valveTURB=-0.0000014,
                Ta_valveTURB=2,
                useTS_pumpFW=true,
                useTS_pumpHPH=true,
                useTS_pumpMS=true,
                Kp_valveRH=0.177,
                CSmin_valveRH=-0.9,
                CSmax_valveRH=1,
                CSstart_valveRH=0,
                TS_valveRH=0,
                l_sp=0,
                Kp_pumpSEP=-1.45,
                CSmin_pumpSEP=-1,
                CSmax_pumpSEP=1,
                useTS_valveRH=true,
                T_RH_sp=826.05,
                T_MS_sp=563.15,
                T_FW_sp=553.15,
                p_sp=17610000,
                useTS_valveTURB=true,
                table_w_FW(table=[4,3.54; 20,17.04; 30,26.36; 40,36.01; 50,46.02;
                      60,56.19; 70,64.64; 80,72.9; 90,81.03; 100,87.9]))
                annotation (Placement(transformation(extent={{-92,-2},{-68,22}})));
              Modelica.Icons.SignalBus signalBus
                annotation (Placement(transformation(extent={{-70,-30},{-50,-10}})));
              Modelica.Blocks.Sources.RealExpression Power_sensor(y=0)
                annotation (Placement(transformation(extent={{-88,-28},{-74,-12}})));
              OTSG_init100_20nodes OTSG annotation (Placement(transformation(
                      extent={{-40,-40},{20,60}})));
            equation
              connect(pump_HPH.outlet, sinkHPH.flange) annotation (Line(
                  points={{20,-60},{40,-60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(signalBus.pumpHPH_CS, pump_HPH.in_w0) annotation (Line(
                  points={{-60,-20},{-60,-100},{6,-100},{6,-66}},
                  color={255,204,51},
                  thickness=0.5));
              connect(signalBus.pumpMS_PV, Power_sensor.y) annotation (Line(
                  points={{-60,-20},{-73.3,-20}},
                  color={255,204,51},
                  thickness=0.5));
              connect(controllers.signalBus, signalBus) annotation (Line(
                  points={{-67.4,10},{-60,10},{-60,-20}},
                  color={255,204,51},
                  thickness=0.5));
              connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                  points={{-67.4,10},{-42.3333,10}},
                  color={255,204,51},
                  thickness=0.5));
              connect(sourceMS.flange, OTSG.MS_in) annotation (Line(points={{-40,100},
                      {-40,100},{-26.6667,100},{-26.6667,61.6667}},
                                                     color={0,0,255}));
              connect(sinkRH.flange, OTSG.RH_out) annotation (Line(points={{20,100},
                      {-6.66667,100},{-6.66667,61.6667}},
                                           color={0,0,255}));
              connect(sinkSH.flange, OTSG.SH_out) annotation (Line(points={{40,80},
                      {6.66667,80},{6.66667,61.6667}},
                                            color={0,0,255}));
              connect(OTSG.FW_in, sourceFW.flange) annotation (Line(points={{21.6667,
                      -28.3333},{40,-28.3333},{40,-20},{60,-20}},
                                                     color={0,0,255}));
              connect(OTSG.FW_in, OTSG.SEP_out) annotation (Line(points={{21.6667,
                      -28.3333},{40,-28.3333},{40,-3.33333},{21.6667,-3.33333}},
                                                       color={0,0,255}));
              connect(OTSG.MS_out, sinkMS.flange) annotation (Line(points={{
                      -26.6667,-41.6667},{-26.6667,-60},{-30,-60}},
                                            color={0,0,255}));
              connect(OTSG.MS_HPH_out, pump_HPH.inlet) annotation (Line(points={{
                      -6.66667,-41.6667},{-6.66667,-60},{0,-60}},
                                                   color={0,0,255}));
              connect(OTSG.RH_in, sourceRH.flange) annotation (Line(points={{21.6667,
                      46.6667},{40,46.6667},{40,40},{60,40}},         color={0,0,
                      255}));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-120,
                        -120},{120,120}})),                  Icon(coordinateSystem(
                      extent={{-120,-120},{120,120}})),
                experiment(StopTime=5000),
                __Dymola_experimentSetupOutput,
                Documentation(revisions="<html>
</html>",   info="<html>
<p><span style=\"font-family: MS Shell Dlg 2;\">The model can be initialized both in steady-state and fixed-state way, through the system component. The OTSG can be selected between the 5 nodes and 20 nodes model. Computational time are acceptable in both cases.</span></p>
</html>"));
            end test_OTSG_100;

            model test_OTSG_50 "Test model of the OTSG at 50% load"

              OTSG_init50_20nodes                     OTSG
                annotation (Placement(transformation(extent={{-40,-40},{20,60}})));
              ThermoPower.Water.SourceMassFlow sourceFW(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=45.93,
                use_in_w0=true,
                h=1062500)
                annotation (Placement(transformation(extent={{80,-10},{60,-30}})));
              ThermoPower.Water.SourceMassFlow sourceRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_w0=false,
                w0=38.01,
                h=3126000)
                annotation (Placement(transformation(extent={{80,30},{60,50}})));
              ThermoPower.Water.SinkPressure sinkRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=2161000)
                annotation (Placement(transformation(extent={{20,90},{40,110}})));
              ThermoPower.Water.SourcePressure sourceMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322) annotation (Placement(transformation(extent={{-60,90},
                        {-40,110}})));

              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=261078) annotation (Placement(transformation(extent={{-30,-70},
                        {-50,-50}})));

              inner ThermoPower.System system(
                  allowFlowReversal=false, initOpt=ThermoPower.Choices.Init.Options.steadyState)
                annotation (Placement(transformation(extent={{-120,100},{-100,120}})));
              ThermoPower.Water.SinkPressure sinkHPH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=10320000)
                annotation (Placement(transformation(extent={{32,-70},{52,-50}})));
              ThermoPower.Water.SinkPressure sinkSH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=10000000)
                annotation (Placement(transformation(extent={{40,70},{60,90}})));
              ThermoPower.Water.ThroughMassFlow pump_HPH(
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=0,
                use_in_w0=true)
                annotation (Placement(transformation(extent={{0,-50},{20,-70}})));
              Modelica.Icons.SignalBus signalBus
                annotation (Placement(transformation(extent={{-70,-30},{-50,-10}})));
              Control.Old.plant_controller_HPpart_old controllers(
                actuatorGain_valveRH=1,
                Ta_valveRH=2,
                Ti_pumpSEP=56.7,
                Td_pumpSEP=0,
                Nd_pumpSEP=1,
                actuatorGain_pumpSEP=1,
                Ta_pumpSEP=5,
                Ta_pumpFW=5,
                Td_pumpHPH=0,
                Nd_pumpHPH=1,
                actuatorGain_pumpHPH=1,
                Ta_pumpHPH=5,
                actuatorGain_pumpFW=1,
                P_mec_sp=79.7258e6,
                actuatorGain_pumpMS=1,
                Ta_pumpMS=5,
                Td_valveTURB=0,
                Nd_valveTURB=1,
                actuatorGain_valveTURB=1,
                CSmin_pumpFW=-0.3,
                CSmax_pumpFW=0.3,
                CSstart_pumpFW=0,
                TS_pumpFW=0,
                CSmin_pumpHPH=-0.5,
                CSmax_pumpHPH=0.5,
                CSstart_pumpHPH=0,
                TS_pumpHPH=0,
                CSmin_pumpMS=-0.9,
                CSmax_pumpMS=1,
                CSstart_pumpMS=0,
                TS_pumpMS=0,
                CSmin_valveTURB=-0.9,
                CSmax_valveTURB=0.65,
                CSstart_valveTURB=0,
                TS_valveTURB=0,
                Nd_valveRH=10,
                Ti_valveRH=16,
                Td_valveRH=0,
                l_sp=4,
                t_ramp=0,
                t_start=0,
                Kp_pumpFW=-1,
                Ti_pumpFW=28.59,
                Td_pumpFW=7.147,
                Nd_pumpFW=30,
                Kp_pumpHPH=0.6461,
                Ti_pumpHPH=19.05,
                Kp_pumpMS=1.2e-7,
                Ti_pumpMS=114.3,
                Td_pumpMS=28.57,
                Nd_pumpMS=20,
                Ti_valveTURB=95.23,
                Kp_valveTURB=-0.0000014,
                Ta_valveTURB=2,
                useTS_pumpHPH=true,
                useTS_pumpMS=true,
                load_start=50,
                load_end=50,
                useTS_pumpSEP=false,
                CSstart_pumpSEP=0,
                TS_pumpSEP=0,
                useTS_pumpFW=true,
                Kp_valveRH=0.177,
                CSmin_valveRH=-0.9,
                CSmax_valveRH=1,
                CSstart_valveRH=0,
                TS_valveRH=0,
                useTS_valveRH=true,
                useTS_valveTURB=true,
                CSmin_pumpSEP=0,
                CSmax_pumpSEP=100,
                Kp_pumpSEP=-14,
                T_RH_sp=826.05,
                T_MS_sp=563.15,
                T_FW_sp=553.15,
                p_sp=17610000) annotation (Placement(transformation(extent={{-92,
                        -2},{-68,22}})));
              Modelica.Blocks.Sources.RealExpression Power_sensor(y=0)
                annotation (Placement(transformation(extent={{-88,-28},{-74,-12}})));
            equation
              connect(OTSG.FW_in, sourceFW.flange) annotation (Line(
                  points={{21.6667,-28.3333},{40,-28.3333},{40,-20},{60,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.RH_in, sourceRH.flange) annotation (Line(
                  points={{21.6667,46.6667},{40,46.6667},{40,40},{60,40}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS.flange, OTSG.MS_in) annotation (Line(
                  points={{-40,100},{-26.6667,100},{-26.6667,61.6667}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS.flange, OTSG.MS_out) annotation (Line(
                  points={{-30,-60},{-26,-60},{-26,-41.6667},{-26.6667,-41.6667}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.MS_HPH_out, pump_HPH.inlet) annotation (Line(
                  points={{-6.66667,-41.6667},{-6.66667,-60},{0,-60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(pump_HPH.outlet, sinkHPH.flange) annotation (Line(
                  points={{20,-60},{32,-60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.RH_out, sinkRH.flange) annotation (Line(
                  points={{-6.66667,61.6667},{-6.66667,100},{20,100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.SH_out, sinkSH.flange) annotation (Line(
                  points={{6.66667,61.6667},{6.66667,80},{40,80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.SEP_out, sourceFW.flange) annotation (Line(
                  points={{21.6667,-3.33333},{40,-3.33333},{40,-20},{60,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                  points={{-67.4,10},{-67.4,10},{-42.3333,10}},
                  color={255,204,51},
                  thickness=0.5));
              connect(controllers.signalBus, signalBus) annotation (Line(
                  points={{-67.4,10},{-60,10},{-60,-20}},
                  color={255,204,51},
                  thickness=0.5));
              connect(signalBus.pumpHPH_CS, pump_HPH.in_w0) annotation (Line(
                  points={{-60,-20},{-60,-20},{-60,-100},{8,-100},{6,-100},{6,-66}},
                  color={255,204,51},
                  thickness=0.5));
              connect(signalBus.pumpFW_CS, sourceFW.in_w0) annotation (Line(
                  points={{-60,-20},{-60,-20},{-60,-100},{74,-100},{74,-26}},
                  color={255,204,51},
                  thickness=0.5));
              connect(signalBus.pumpMS_PV, Power_sensor.y) annotation (Line(
                  points={{-60,-20},{-66,-20},{-73.3,-20}},
                  color={255,204,51},
                  thickness=0.5));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-120,
                        -120},{120,120}})),                  Icon(coordinateSystem(
                      extent={{-120,-120},{120,120}}, preserveAspectRatio=false)),
                experiment(StopTime=5000),
                __Dymola_experimentSetupOutput,
                Documentation(info="<html>
<p><span style=\"font-family: MS Shell Dlg 2;\">The model has to be initialized  in steady-state, through the system component, so that the pressure terms do no influence the mass flow balances. The OTSG can be selected between the 5 nodes and 20 nodes model. </span></p>
</html>"));
            end test_OTSG_50;

            model test_OTSG_20 "Test model of the OTSG at 20% load"

              OTSG_init20_20nodes                     OTSG
                annotation (Placement(transformation(extent={{-40,-40},{20,60}})));
              ThermoPower.Water.SourceMassFlow sourceFW(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_w0=true,
                w0=17.01,
                h=840600)
                annotation (Placement(transformation(extent={{80,-10},{60,-30}})));
              ThermoPower.Water.SourceMassFlow sourceRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_w0=false,
                h=3125000,
                w0=13.4)
                annotation (Placement(transformation(extent={{80,30},{60,50}})));
              ThermoPower.Water.SinkPressure sinkRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=767000)
                annotation (Placement(transformation(extent={{20,90},{40,110}})));
              ThermoPower.Water.SourcePressure sourceMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322) annotation (Placement(transformation(extent={{-60,90},
                        {-40,110}})));

              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=261078) annotation (Placement(transformation(extent={{-30,-70},
                        {-50,-50}})));

              inner ThermoPower.System system(allowFlowReversal=false, initOpt=
                    ThermoPower.Choices.Init.Options.fixedState)
                annotation (Placement(transformation(extent={{-120,100},{-100,120}})));
              ThermoPower.Water.SinkPressure sinkHPH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=10200000)
                annotation (Placement(transformation(extent={{32,-70},{52,-50}})));
              ThermoPower.Water.SinkPressure sinkSH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=10000000)
                annotation (Placement(transformation(extent={{40,70},{60,90}})));
              ThermoPower.Water.ThroughMassFlow pump_HPH(
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=0.001,
                use_in_w0=true)
                annotation (Placement(transformation(extent={{0,-50},{20,-70}})));
              Modelica.Icons.SignalBus signalBus
                annotation (Placement(transformation(extent={{-70,-30},{-50,-10}})));
              Control.Old.plant_controller_HPpart_old controllers(
                actuatorGain_valveRH=1,
                Ta_valveRH=2,
                Ti_pumpSEP=56.7,
                Td_pumpSEP=0,
                Nd_pumpSEP=1,
                actuatorGain_pumpSEP=1,
                Ta_pumpSEP=5,
                Ta_pumpFW=5,
                Td_pumpHPH=0,
                Nd_pumpHPH=1,
                actuatorGain_pumpHPH=1,
                Ta_pumpHPH=5,
                actuatorGain_pumpFW=1,
                P_mec_sp=79.7258e6,
                actuatorGain_pumpMS=1,
                Ta_pumpMS=5,
                Td_valveTURB=0,
                Nd_valveTURB=1,
                actuatorGain_valveTURB=1,
                CSmin_pumpFW=-0.3,
                CSmax_pumpFW=0.3,
                CSstart_pumpFW=0,
                TS_pumpFW=0,
                CSmin_pumpHPH=-0.5,
                CSmax_pumpHPH=0.5,
                CSstart_pumpHPH=0,
                TS_pumpHPH=0,
                CSmin_pumpMS=-0.9,
                CSmax_pumpMS=1,
                CSstart_pumpMS=0,
                TS_pumpMS=0,
                CSmin_valveTURB=-0.9,
                CSmax_valveTURB=0.65,
                CSstart_valveTURB=0,
                TS_valveTURB=0,
                Nd_valveRH=10,
                Ti_valveRH=16,
                Td_valveRH=0,
                l_sp=4,
                t_ramp=0,
                t_start=0,
                Kp_pumpFW=-1,
                Ti_pumpFW=28.59,
                Td_pumpFW=7.147,
                Nd_pumpFW=30,
                Kp_pumpHPH=0.6461,
                Ti_pumpHPH=19.05,
                Kp_pumpMS=1.2e-7,
                Ti_pumpMS=114.3,
                Td_pumpMS=28.57,
                Nd_pumpMS=20,
                Ti_valveTURB=95.23,
                Kp_valveTURB=-0.0000014,
                Ta_valveTURB=2,
                useTS_pumpHPH=true,
                useTS_pumpMS=true,
                CSstart_pumpSEP=0,
                TS_pumpSEP=0,
                useTS_pumpFW=true,
                load_start=20,
                load_end=20,
                useTS_pumpSEP=false,
                Kp_valveRH=0.177,
                CSmin_valveRH=-1,
                CSmax_valveRH=1,
                CSstart_valveRH=0,
                TS_valveRH=0,
                useTS_valveRH=true,
                useTS_valveTURB=true,
                CSmin_pumpSEP=0,
                CSmax_pumpSEP=100,
                T_RH_sp=826.05,
                Kp_pumpSEP=-14,
                T_MS_sp=563.15,
                T_FW_sp=553.15,
                p_sp=17610000) annotation (Placement(transformation(extent={{-90,
                        -2},{-68,22}})));
              Modelica.Blocks.Sources.RealExpression Power_sensor(y=0)
                annotation (Placement(transformation(extent={{-88,-28},{-74,-12}})));
            equation
              connect(OTSG.FW_in, sourceFW.flange) annotation (Line(
                  points={{21.6667,-28.3333},{30,-28.3333},{30,-28},{40,-28},{40,
                      -20},{60,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.RH_in, sourceRH.flange) annotation (Line(
                  points={{21.6667,46.6667},{40,46.6667},{40,40},{60,40}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sourceMS.flange, OTSG.MS_in) annotation (Line(
                  points={{-40,100},{-26.6667,100},{-26.6667,61.6667}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkMS.flange, OTSG.MS_out) annotation (Line(
                  points={{-30,-60},{-26,-60},{-26,-41.6667},{-26.6667,-41.6667}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.MS_HPH_out, pump_HPH.inlet) annotation (Line(
                  points={{-6.66667,-41.6667},{-6.66667,-60},{0,-60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(pump_HPH.outlet, sinkHPH.flange) annotation (Line(
                  points={{20,-60},{32,-60}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.RH_out, sinkRH.flange) annotation (Line(
                  points={{-6.66667,61.6667},{-6.66667,100},{20,100}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.SH_out, sinkSH.flange) annotation (Line(
                  points={{6.66667,61.6667},{6.66667,80},{40,80}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.SEP_out, sourceFW.flange) annotation (Line(
                  points={{21.6667,-3.33333},{40,-3.33333},{40,-20},{60,-20}},
                  color={0,0,255},
                  smooth=Smooth.None));

              connect(signalBus.pumpMS_PV, Power_sensor.y) annotation (Line(
                  points={{-60,-20},{-73.3,-20}},
                  color={255,204,51},
                  thickness=0.5));
              connect(controllers.signalBus, signalBus) annotation (Line(
                  points={{-67.45,10},{-60,10},{-60,-20}},
                  color={255,204,51},
                  thickness=0.5));
              connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                  points={{-67.45,10},{-67.45,10},{-42.3333,10}},
                  color={255,204,51},
                  thickness=0.5));
              connect(signalBus.pumpHPH_CS, pump_HPH.in_w0) annotation (Line(
                  points={{-60,-20},{-60,-20},{-60,-80},{6,-80},{6,-66}},
                  color={255,204,51},
                  thickness=0.5));
              connect(signalBus.pumpFW_CS, sourceFW.in_w0) annotation (Line(
                  points={{-60,-20},{-60,-80},{74,-80},{74,-26}},
                  color={255,204,51},
                  thickness=0.5));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-120,
                        -120},{120,120}})),                  Icon(coordinateSystem(
                      extent={{-120,-120},{120,120}}, preserveAspectRatio=false)),
                experiment(StopTime=5000),
                __Dymola_experimentSetupOutput,
                Documentation(info="<html>
<p><span style=\"font-family: MS Shell Dlg 2;\">The model has to be initialized in fixed-state, through the system component, since the steady-state initialisation fails. The OTSG can be selected between the 5 nodes and 20 nodes model. </span></p>
</html>"));
            end test_OTSG_20;

            model test_OTSG_4_optA "Test model of the OTSG at 4% of load"

              OTSG_init20_20nodes OTSG(
                separator(
                  initOpt=ThermoPower.Choices.Init.Options.fixedState,
                  h_start=1697500,
                  p_start=8540000),
                PDSteamEVA1(pin(start=8620000)),
                PDSteamECO(pin(start=8660000)),
                PDSteamEVA2(pin(start=8580000)),
                PDSteamRH(pin(start=22.05e5)))
                annotation (Placement(transformation(extent={{-44,-36},{12,56}})));
              ThermoPower.Water.SourceMassFlow sourceFW(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_w0=true,
                w0=17.01,
                h=515800)
                annotation (Placement(transformation(extent={{100,-4},{80,-24}})));
              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=261078) annotation (Placement(transformation(extent={{-40,-58},
                        {-60,-38}})));

              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.fixedState)
                annotation (Placement(transformation(extent={{-140,120},{-120,140}})));
              ThermoPower.Water.SourceMassFlow sourceRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_w0=false,
                w0=4.18,
                h=3092400)
                annotation (Placement(transformation(extent={{76,34},{56,54}})));
              ThermoPower.Water.SinkPressure sinkRH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=2200000)
                annotation (Placement(transformation(extent={{22,114},{42,134}})));
              ThermoPower.Water.SinkPressure sinkSH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=8500000)
                annotation (Placement(transformation(extent={{62,94},{82,114}})));
              ThermoPower.Water.SourcePressure sourceMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                h=518322) annotation (Placement(transformation(extent={{-60,82},
                        {-40,102}})));

              ThermoPower.Water.ThroughMassFlow pump_HPH(
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=0.001,
                use_in_w0=true)
                annotation (Placement(transformation(extent={{0,-38},{20,-58}})));
              ThermoPower.Water.SinkPressure sinkHPH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                p0=10200000)
                annotation (Placement(transformation(extent={{32,-58},{52,-38}})));
              Control.Old.plant_controller_HPpart_old controllers(
                actuatorGain_valveRH=1,
                Ta_valveRH=2,
                Ti_pumpSEP=56.7,
                Td_pumpSEP=0,
                Nd_pumpSEP=1,
                actuatorGain_pumpSEP=1,
                Ta_pumpSEP=5,
                Ta_pumpFW=5,
                Td_pumpHPH=0,
                Nd_pumpHPH=1,
                actuatorGain_pumpHPH=1,
                Ta_pumpHPH=5,
                actuatorGain_pumpFW=1,
                P_mec_sp=79.7258e6,
                actuatorGain_pumpMS=1,
                Ta_pumpMS=5,
                Td_valveTURB=0,
                Nd_valveTURB=1,
                actuatorGain_valveTURB=1,
                CSstart_pumpFW=0,
                TS_pumpFW=0,
                CSmin_pumpHPH=-0.5,
                CSmax_pumpHPH=0.5,
                CSstart_pumpHPH=0,
                TS_pumpHPH=0,
                CSmin_pumpMS=-0.9,
                CSmax_pumpMS=1,
                CSstart_pumpMS=0,
                TS_pumpMS=0,
                CSmin_valveTURB=-0.9,
                CSmax_valveTURB=0.65,
                CSstart_valveTURB=0,
                Nd_valveRH=10,
                Kp_valveRH=1.77,
                Ti_valveRH=16,
                Td_valveRH=0,
                l_sp=4,
                Kp_pumpFW=-1,
                Ti_pumpFW=28.59,
                Td_pumpFW=7.147,
                Nd_pumpFW=30,
                Kp_pumpHPH=0.6461,
                Ti_pumpHPH=19.05,
                Kp_pumpMS=1.2e-7,
                Ti_pumpMS=114.3,
                Td_pumpMS=28.57,
                Nd_pumpMS=20,
                Ti_valveTURB=95.23,
                Kp_valveTURB=-0.0000014,
                Ta_valveTURB=2,
                useTS_pumpHPH=true,
                useTS_pumpMS=true,
                useTS_valveTURB=true,
                CSstart_pumpSEP=0,
                TS_pumpSEP=0,
                useTS_pumpSEP=false,
                t_ramp=0,
                t_start=0,
                CSmax_pumpFW=2,
                CSmin_pumpFW=-0.53,
                load_start=4.5,
                load_end=4.5,
                CSmin_valveRH=-1,
                CSmax_valveRH=1,
                CSstart_valveRH=0,
                useTS_valveRH=true,
                TS_valveRH=0,
                Kp_pumpSEP=-14,
                CSmin_pumpSEP=0,
                CSmax_pumpSEP=100,
                useTS_pumpFW=true,
                T_RH_sp=826.05,
                T_MS_sp=563.15,
                T_FW_sp=553.15,
                p_sp=17610000,
                TS_valveTURB=-0.2) annotation (Placement(transformation(extent={{
                        -108,2},{-92,18}})));
              Modelica.Blocks.Sources.RealExpression Power_sensor(y=0)
                annotation (Placement(transformation(extent={{-108,-18},{-94,-2}})));
              Modelica.Icons.SignalBus signalBus
                annotation (Placement(transformation(extent={{-90,-20},{-70,0}})));
            equation
              connect(OTSG.SEP_out, sourceFW.flange) annotation (Line(
                  points={{13.5556,-2.26667},{44,-2.26667},{44,-14},{80,-14}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.RH_in,sourceRH. flange) annotation (Line(
                  points={{13.5556,43.7333},{54,43.7333},{54,44},{56,44}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(OTSG.SH_out,sinkSH. flange) annotation (Line(
                  points={{-0.444444,57.5333},{-0.444444,104},{62,104}},
                  color={0,0,255},
                  smooth=Smooth.None));
              connect(sinkRH.flange, OTSG.RH_out) annotation (Line(points={{22,124},
                      {4,124},{-12.8889,124},{-12.8889,57.5333}},
                                                        color={0,0,255}));
              connect(OTSG.MS_out, sinkMS.flange) annotation (Line(points={{
                      -31.5556,-37.5333},{-31.5556,-48},{-40,-48}},
                                                             color={0,0,255}));
              connect(sourceMS.flange, OTSG.MS_in) annotation (Line(points={{-40,92},
                      {-31.5556,92},{-31.5556,57.5333}}, color={0,0,255}));
              connect(OTSG.MS_HPH_out, pump_HPH.inlet) annotation (Line(points={{
                      -12.8889,-37.5333},{-12.8889,-48},{0,-48}}, color={0,0,255}));
              connect(pump_HPH.outlet, sinkHPH.flange) annotation (Line(points={{20,
                      -48},{26,-48},{32,-48}}, color={0,0,255}));
              connect(OTSG.SEP_out, OTSG.FW_in) annotation (Line(points={{13.5556,
                      -2.26667},{44,-2.26667},{44,-25.2667},{13.5556,-25.2667}},
                    color={0,0,255}));
              connect(signalBus.pumpMS_PV,Power_sensor. y) annotation (Line(
                  points={{-80,-10},{-93.3,-10}},
                  color={255,204,51},
                  thickness=0.5));
              connect(controllers.signalBus,signalBus)  annotation (Line(
                  points={{-91.6,10},{-80,10},{-80,-10}},
                  color={255,204,51},
                  thickness=0.5));
              connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                  points={{-91.6,10},{-46.1778,10}},
                  color={255,204,51},
                  thickness=0.5));
              connect(signalBus.pumpHPH_CS, pump_HPH.in_w0) annotation (Line(
                  points={{-80,-10},{-80,-10},{-80,-70},{-80,-70},{6,-70},{6,-54}},
                  color={255,204,51},
                  thickness=0.5));

              connect(signalBus.pumpFW_CS, sourceFW.in_w0) annotation (Line(
                  points={{-80,-10},{-80,-70},{94,-70},{94,-20}},
                  color={255,204,51},
                  thickness=0.5));
              annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,
                        -140},{140,140}})),                  Icon(coordinateSystem(
                      extent={{-140,-140},{140,140}}, preserveAspectRatio=false)),
                experiment(StopTime=5000),
                __Dymola_experimentSetupOutput);
            end test_OTSG_4_optA;

            model test_recirculationSection
              "Test model to check the need of decoupling between feedwater and recirculated mass flow"

              Components.Equilibrium2phVessel separator(
                H=8,
                zl_start=4,
                phase(start=2, fixed=true),
                D=0.56,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                noInitialPressure=true,
                noInitialLevel=true,
                h_start=1.7e6,
                p_start=10100000,
                T(start=584.85))
                annotation (Placement(transformation(extent={{-68,68},{-12,124}})));
              ThermoPower.Water.ThroughMassFlow waterPump(
                allowFlowReversal=false,
                use_in_w0=true,
                w0=10,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={-40,16})));

              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                dpnom=(0.05828)*10^5,
                rhonom=582.24,
                wnom=88.59,
                pin(start=102e5),
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-180,22})));
              ThermoPower.Water.PressDrop PDSteamECO(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.52492)*10^5,
                rhonom=768.15,
                pin(start=103e5),
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-180,-60})));
              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=250,
                pin(start=10150000),
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-130,88})));
              //Overall parameter and variables:
              Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot;
              Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
              Modelica.SIunits.Temperature T_outMS = ECO.hotFluid.T[end];
              Modelica.SIunits.Length sep_lev = separator.zl;
              Real x_EVA1_out = EVA1.coldFluid.x[end];
              Real x_EVA2_out = EVA2.coldFluid.x[end];
              Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;
              Modelica.SIunits.MassFlowRate Fpos = ECO.coldFluid.w*(1-x_EVA2_out);
              Modelica.SIunits.MassFlowRate Fneg = separator.wl_out;
              Modelica.SIunits.SpecificEnthalpy Q1 = ECO.coldFluid.h[1];
              Modelica.SIunits.SpecificEnthalpy Q2 = Q_tot/ECO.coldFluid.w;

              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.fixedState)
                annotation (Placement(transformation(extent={{-260,140},{-240,160}})));
              ThermoPower.Water.SourceMassFlow sourceFW(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1062500,
                w0=45.93,
                use_in_w0=true)
                annotation (Placement(transformation(extent={{110,-150},{90,-130}})));
              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=261078) annotation (Placement(transformation(extent={{-220,-110},
                        {-240,-90}})));

              ThermoPower.Water.SinkPressure sinkSH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=10100000)
                annotation (Placement(transformation(extent={{80,130},{100,150}})));
              ThermoPower.Water.FlowJoin flowJoin(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=180,
                    origin={-46,-136})));
              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=305.3,
                h=328503,
                use_in_w0=false) annotation (Placement(transformation(extent={{
                        -130,70},{-150,50}})));

              ThermoPower.PowerPlants.Control.PID PID_levelControl(
                Ti=56,
                PVmin=0,
                PVmax=8,
                CSmin=0,
                PVstart=0.5392,
                CSstart=0.5,
                holdWhenSimplified=true,
                steadyStateInit=true,
                Kp=-14,
                CSmax=250)            annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=270,
                    origin={90,66})));
              Modelica.Blocks.Continuous.FirstOrder firstOrder(
                k=1,
                initType=Modelica.Blocks.Types.Init.SteadyState,
                y_start=53.92,
                T=2)           annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=270,
                    origin={90,38})));
              Modelica.Blocks.Sources.Constant const(k=4) annotation (Placement(
                    transformation(
                    extent={{10,-10},{-10,10}},
                    rotation=0,
                    origin={130,82})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.56606e+006,1.59353e+006,1.62512e+006,1.66145e+006,
                      1.70321e+006,1.75123e+006,1.80642e+006,1.86984e+006,
                      1.94272e+006,2.02642e+006}),
                tubeWalls(Tvolstart(displayUnit="degC") = {596.276,597.951,599.875,
                    602.087,604.627,607.543,610.89,614.73,619.134}),
                Vhot=3.905,
                Vcold=1.065,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    alpha=1.6,
                    gamma_nom_liq=2000,
                    gamma_nom_2ph=5000,
                    gamma_nom_vap=2000),
                Shot=429.2*1,
                Scold=338*1,
                Nhex=10,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={344122,316769,292958,272234,254202,238514,224868,213001,202681,193708},
                  noInitialPressure=true))          annotation (Placement(
                    transformation(extent={{-164,44},{-196,76}})));

              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                Mm=43100,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                nominalColdFlow=88.59,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.33e+006,1.3378e+006,1.34658e+006,1.35651e+006,
                      1.36781e+006,1.38075e+006,1.39566e+006,1.41303e+006,
                      1.45658e+006,1.56606e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {572.404,573.98,575.749,
                    577.742,579.999,582.568,585.511,585.959,587.035}),
                Vhot=26.587,
                Vcold=8.651,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=3875,
                    gamma_nom_vap=3875,
                    alpha=1.6,
                    gamma_nom_2ph=10000),
                Shot=2251.9*1,
                Scold=1765.5*1,
                Nhex=10,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={193708,157918,143681,138009,133137,128914,125226,121985,119121,116578},
                  noInitialPressure=true))          annotation (Placement(
                    transformation(extent={{-196,-36},{-164,-4}})));

              Components.HE ECO(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Shot=2394.2,
                Scold=1855.5,
                Mm=30395,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.728761, gamma_nom=4400),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.461561, gamma_nom=2200),
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.25188e+006,1.26536e+006,1.2819e+006,1.30253e+006,
                      1.3288e+006}),
                tubeWalls(Tvolstart(displayUnit="degC") = {558.324,561.292,564.934,
                    569.474}),
                Vhot=45.011,
                Vcold=5.752,
                Nhex=5,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                nominalColdPressure=17910000,
                nominalHotPressure=569600,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=623.79,
                nominalHotInletTemperature=628.75,
                nominalHotOutletTemperature=586.72,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={119114,110526,103784,98376.4,93969.4},
                  noInitialPressure=true))
                annotation (Placement(transformation(extent={{-164,-116},{-196,
                        -84}})));
              Modelica.Blocks.Math.Gain gain(k=1) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=270,
                    origin={90,-40})));
              Modelica.Blocks.Sources.Constant const1(k=0)
                annotation (Placement(transformation(extent={{140,-80},{120,-60}})));
              Modelica.Blocks.Sources.BooleanStep booleanStep3(startValue=true, startTime=1000)
                annotation (Placement(transformation(extent={{140,-50},{120,-30}})));
              Modelica.Blocks.Logical.Switch switch1 annotation (Placement(
                    transformation(
                    extent={{-10,10},{10,-10}},
                    rotation=270,
                    origin={98,-78})));
              Modelica.Blocks.Math.Add add(k1=-1, k2=1) annotation (Placement(
                    transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={90,-10})));
              Modelica.Blocks.Sources.Constant const2(k=53.92)
                annotation (Placement(transformation(extent={{140,0},{120,20}})));
              Modelica.Blocks.Math.Add add1(k1=1, k2=-1) annotation (Placement(
                    transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={104,-114})));
              Modelica.Blocks.Sources.Constant const3(k=45.93)
                annotation (Placement(transformation(extent={{160,-120},{140,-100}})));
              Modelica.Blocks.Sources.Step step(
                offset=305.3,
                startTime=1010,
                height=30) annotation (Placement(transformation(extent={{-80,30},
                        {-100,50}})));
            equation
              connect(separator.drained_out, waterPump.inlet) annotation (Line(
                  points={{-40,78},{-40,26}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                  points={{-120,88},{-50,88}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5));

              connect(flowJoin.in1, sourceFW.flange) annotation (Line(
                  points={{-40,-140},{-40,-140},{90,-140}},
                  color={0,0,255},
                  thickness=0.5));
              connect(waterPump.outlet, flowJoin.in2) annotation (Line(
                  points={{-40,6},{-40,-132}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sinkSH.flange, separator.steam_out) annotation (Line(
                  points={{80,140},{-40,140},{-40,120}},
                  color={0,0,255},
                  thickness=0.5));
              connect(separator.level, PID_levelControl.PV)
                annotation (Line(points={{-30,82},{86,82},{86,76}},color={0,0,127}));
              connect(PID_levelControl.CS, firstOrder.u)
                annotation (Line(points={{90,56},{90,50}}, color={0,0,127}));
              connect(firstOrder.y, waterPump.in_w0)
                annotation (Line(points={{90,27},{90,20},{-34,20}},
                                                                  color={0,0,127}));
              connect(const.y, PID_levelControl.SP) annotation (Line(points={{119,82},
                      {108,82},{94,82},{94,76}},
                                        color={0,0,127}));
              connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
                  points={{-140,88},{-162,88},{-180,88},{-180,76}},
                  color={0,0,255},
                  thickness=0.5));
              connect(EVA2.inletHot, sourceMS.flange) annotation (Line(
                  points={{-164,60},{-150,60}},
                  color={255,0,0},
                  thickness=0.5));
              connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                  points={{-196,60},{-220,60},{-220,-20},{-196,-20}},
                  color={255,0,0},
                  thickness=0.5));
              connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                  points={{-164,-20},{-152,-20},{-140,-20},{-140,-100},{-164,-100}},
                  color={255,0,0},
                  thickness=0.5));
              connect(sinkMS.flange, ECO.outletHot) annotation (Line(
                  points={{-220,-100},{-208,-100},{-196,-100}},
                  color={255,0,0},
                  thickness=0.5));
              connect(PDSteamECO.inlet, ECO.outletCold) annotation (Line(
                  points={{-180,-70},{-180,-77},{-180,-84}},
                  color={0,0,255},
                  thickness=0.5));
              connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
                  points={{-180,-36},{-180,-36},{-180,-50}},
                  color={0,0,255},
                  thickness=0.5));
              connect(PDSteamEVA1.inlet, EVA1.outletCold) annotation (Line(
                  points={{-180,12},{-180,12},{-180,-4}},
                  color={0,0,255},
                  thickness=0.5));
              connect(EVA2.inletCold, PDSteamEVA1.outlet) annotation (Line(
                  points={{-180,44},{-180,32}},
                  color={0,0,255},
                  thickness=0.5));
              connect(ECO.inletCold, flowJoin.out) annotation (Line(
                  points={{-180,-116},{-140,-116},{-140,-136},{-52,-136}},
                  color={0,0,255},
                  thickness=0.5));
              connect(const1.y,switch1. u3) annotation (Line(points={{119,-70},{
                      119,-70},{112,-70},{112,-60},{106,-60},{106,-66}},
                                                      color={0,0,127}));
              connect(booleanStep3.y,switch1. u2) annotation (Line(points={{119,-40},
                      {119,-40},{112,-40},{112,-48},{98,-48},{98,-66}},    color={255,0,
                      255}));
              connect(gain.y, switch1.u1)
                annotation (Line(points={{90,-51},{90,-68},{90,-66}}, color={0,0,127}));
              connect(add.y, gain.u)
                annotation (Line(points={{90,-21},{90,-28}}, color={0,0,127}));
              connect(firstOrder.y, add.u2) annotation (Line(points={{90,27},{90,
                      27},{90,12},{84,12},{84,2}},
                                       color={0,0,127}));
              connect(const2.y, add.u1)
                annotation (Line(points={{119,10},{96,10},{96,2}}, color={0,0,127}));
              connect(switch1.y, add1.u2)
                annotation (Line(points={{98,-89},{98,-89},{98,-102}}, color={0,0,127}));
              connect(add1.y, sourceFW.in_w0) annotation (Line(points={{104,-125},
                      {104,-125},{104,-134}},
                                   color={0,0,127}));
              connect(const3.y, add1.u1) annotation (Line(points={{139,-110},{134,
                      -110},{124,-110},{124,-94},{110,-94},{110,-102}},
                                                             color={0,0,127}));
              connect(sourceMS.in_w0, step.y) annotation (Line(points={{-136,54},
                      {-136,40},{-101,40}}, color={0,0,127}));
              annotation (Diagram(coordinateSystem(extent={{-260,-160},{260,160}},
                      preserveAspectRatio=false)),           Icon(coordinateSystem(
                      extent={{-260,-160},{260,160}}, preserveAspectRatio=false)),
                Documentation(info="<html>
<p>Test model to chek the need of a decoupler between separator pump and feedwater mass flow rate. The model is initialised at 50&percnt; load with the decoupler active. After 1000 s the decoupler is switched off, so that an equivalent step on the feedwater flow is performed. </p>
<p>The separator pump reacts and increases the recirculated mass flow, leading to the instability. If x is the vapor quality at evaporator outlet we can write:</p>
<p><img src=\"modelica://PreFlexMS/Images/equations/equation-LkR73UrL.png\" alt=\"rho*A*der(l) = (w_FW + w_SEP)*(1-x) - w_SEP  \"/>            --&GT;liquid mass balance on the separator</p>
<p><img src=\"modelica://PreFlexMS/Images/equations/equation-yxtUkHRo.png\" alt=\"x = (h_ECOin + (Q_ECO + Q_EVA)/(w_FW + w_SEP) + h_ls)/(h_vs - h_ls) \"/>                <img src=\"modelica://PreFlexMS/Images/equations/equation-EZkMMYs8.png\" alt=\"x = (Q1 + Q2 + h_ls)/(h_vs - h_ls) \"/>                     --&GT; vapor quality</p>
<p><br>As the results of the simulation show, the instability is due to a permanent dl/dt &GT; 0 after the the FW step.</p>
</html>"),      experiment(StopTime=1500),
                __Dymola_experimentSetupOutput);
            end test_recirculationSection;

            model test_feedwaterLevelControl
              "Test model to show the feasibility of the level control through the feedwater"

              Components.Equilibrium2phVessel separator(
                H=8,
                zl_start=4,
                phase(start=2, fixed=true),
                D=0.56,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                noInitialPressure=true,
                noInitialLevel=true,
                h_start=2.2e6,
                p_start=10100000,
                T(start=584.85))
                annotation (Placement(transformation(extent={{-28,68},{28,124}})));
              ThermoPower.Water.ThroughMassFlow waterPump(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                w0=53.92,
                use_in_w0=true)
                       annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=-90,
                    origin={0,16})));

              ThermoPower.Water.PressDrop PDSteamEVA1(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                dpnom=(0.05828)*10^5,
                rhonom=582.24,
                wnom=88.59,
                pin(start=102e5),
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-100,22})));
              ThermoPower.Water.PressDrop PDSteamECO(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.52492)*10^5,
                rhonom=768.15,
                pin(start=103e5),
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                               annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=90,
                    origin={-100,-60})));
              ThermoPower.Water.PressDrop PDSteamEVA2(
                FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
                wnom=88.59,
                dpnom=(0.11171)*10^5,
                rhonom=250,
                pin(start=10150000),
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph)
                                     annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=0,
                    origin={-64,88})));
              //Overall parameter and variables:
              Modelica.SIunits.Power Q_tot = ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot;
              Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
              Modelica.SIunits.Temperature T_outMS = ECO.hotFluid.T[end];
              Modelica.SIunits.Length sep_lev = separator.zl;
              Real x_EVA1_out = EVA1.coldFluid.x[end];
              Real x_EVA2_out = EVA2.coldFluid.x[end];
              Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;
              Modelica.SIunits.MassFlowRate Fpos = ECO.coldFluid.w*(1-x_EVA2_out);
              Modelica.SIunits.MassFlowRate Fneg = separator.wl_out;
              Modelica.SIunits.SpecificEnthalpy Q1 = ECO.coldFluid.h[1];
              Modelica.SIunits.SpecificEnthalpy Q2 = Q_tot/ECO.coldFluid.w;

              inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.fixedState)
                annotation (Placement(transformation(extent={{-180,140},{-160,160}})));
              ThermoPower.Water.SourceMassFlow sourceFW(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                h=1062500,
                w0=45.93,
                use_in_w0=true)
                annotation (Placement(transformation(extent={{86,-152},{66,-132}})));
              ThermoPower.Water.SinkPressure sinkMS(
                allowFlowReversal=false,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                p0=261078) annotation (Placement(transformation(extent={{-140,-110},
                        {-160,-90}})));

              ThermoPower.Water.SinkPressure sinkSH(
                allowFlowReversal=false,
                redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
                use_in_p0=false,
                p0=10100000)
                annotation (Placement(transformation(extent={{70,128},{90,148}})));
              ThermoPower.Water.FlowJoin flowJoin(redeclare package Medium =
                    Modelica.Media.Water.WaterIF97_ph) annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=180,
                    origin={-6,-136})));
              ThermoPower.Water.SourceMassFlow sourceMS(
                allowFlowReversal=false,
                G=0,
                redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
                w0=305.3,
                h=328503,
                use_in_w0=false) annotation (Placement(transformation(extent={{
                        -50,70},{-70,50}})));

              ThermoPower.PowerPlants.Control.PID PID_levelControl(
                PVmin=0,
                PVmax=8,
                CSmin=0,
                PVstart=0.5392,
                holdWhenSimplified=true,
                Ti=50,
                CSmax=120,
                CSstart=0.4,
                steadyStateInit=false,
                Kp=1)                 annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=270,
                    origin={80,8})));
              Modelica.Blocks.Continuous.FirstOrder firstOrder(
                k=1,
                initType=Modelica.Blocks.Types.Init.SteadyState,
                y_start=53.92,
                T=2)           annotation (Placement(transformation(
                    extent={{-10,-10},{10,10}},
                    rotation=270,
                    origin={80,-52})));
              Modelica.Blocks.Sources.Step     const(
                startTime=1000,
                offset=4,
                height=0)                                 annotation (Placement(
                    transformation(
                    extent={{10,-10},{-10,10}},
                    rotation=0,
                    origin={120,82})));
              Components.HE_2ph
                            EVA2(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Mm=5182,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.389637, gamma_nom=2650),
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.56606e+006,1.59353e+006,1.62512e+006,1.66145e+006,
                      1.70321e+006,1.75123e+006,1.80642e+006,1.86984e+006,
                      1.94272e+006,2.02642e+006}),
                tubeWalls(Tvolstart(displayUnit="degC") = {596.276,597.951,599.875,
                    602.087,604.627,607.543,610.89,614.73,619.134}),
                Vhot=3.905,
                Vcold=1.065,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    alpha=1.6,
                    gamma_nom_liq=2000,
                    gamma_nom_2ph=5000,
                    gamma_nom_vap=2000),
                Shot=429.2*1,
                Scold=338*1,
                Nhex=10,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                nominalColdPressure=17760000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=629.15,
                nominalColdOutletTemperature=648.65,
                nominalHotInletTemperature=732.05,
                nominalHotOutletTemperature=687.95,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={344122,316769,292958,272234,254202,238514,224868,213001,202681,193708},
                  noInitialPressure=true))          annotation (Placement(
                    transformation(extent={{-84,44},{-116,76}})));

              Components.HE_2ph
                            EVA1(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                Mm=43100,
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (gamma_nom=2300, alpha=0.447498),
                nominalColdFlow=88.59,
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.33e+006,1.3378e+006,1.34658e+006,1.35651e+006,
                      1.36781e+006,1.38075e+006,1.39566e+006,1.41303e+006,
                      1.45658e+006,1.56606e+006}),
                tubeWalls(Tvolstart(displayUnit="K") = {572.404,573.98,575.749,
                    577.742,579.999,582.568,585.511,585.959,587.035}),
                Vhot=26.587,
                Vcold=8.651,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                    (
                    wnom_ht=88.59,
                    gamma_nom_liq=3875,
                    gamma_nom_vap=3875,
                    alpha=1.6,
                    gamma_nom_2ph=10000),
                Shot=2251.9*1,
                Scold=1765.5*1,
                Nhex=10,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                nominalColdPressure=17810000,
                nominalHotPressure=730800,
                nominalColdInletTemperature=623.75,
                nominalColdOutletTemperature=629.32,
                nominalHotInletTemperature=688.04,
                nominalHotOutletTemperature=629.15,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={193708,157918,143681,138009,133137,128914,125226,121985,119121,116578},
                  noInitialPressure=true))          annotation (Placement(
                    transformation(extent={{-116,-36},{-84,-4}})));

              Components.HE ECO(
                redeclare model HeatExchangerTopology =
                    ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
                cm=500,
                nominalColdDeltaP=0,
                nominalHotDeltaP=0,
                nominalHotFlow=610.6,
                nominalColdFlow=88.59,
                Shot=2394.2,
                Scold=1855.5,
                Mm=30395,
                redeclare model HeatTransferCold =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.728761, gamma_nom=4400),
                redeclare model HeatTransferHot =
                    ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                    (alpha=0.461561, gamma_nom=2200),
                coldFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  Nt=1,
                  hstart={1.25188e+006,1.26536e+006,1.2819e+006,1.30253e+006,
                      1.3288e+006}),
                tubeWalls(Tvolstart(displayUnit="degC") = {558.324,561.292,564.934,
                    569.474}),
                Vhot=45.011,
                Vcold=5.752,
                Nhex=5,
                redeclare package coldMedium =
                    Modelica.Media.Water.WaterIF97_ph,
                redeclare package hotMedium =
                    PreFlexMS.Media.MoltenSalt60_40_quad,
                nominalColdPressure=17910000,
                nominalHotPressure=569600,
                nominalColdInletTemperature=553.15,
                nominalColdOutletTemperature=623.79,
                nominalHotInletTemperature=628.75,
                nominalHotOutletTemperature=586.72,
                hotFluid(
                  FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                  hstart={119114,110526,103784,98376.4,93969.4},
                  noInitialPressure=true))
                annotation (Placement(transformation(extent={{-84,-116},{-116,-84}})));
              Modelica.Blocks.Sources.Step step(
                offset=305.3,
                height=30,
                startTime=3000)
                           annotation (Placement(transformation(extent={{-20,30},
                        {-40,50}})));
              Modelica.Blocks.Sources.Step step1(
                offset=53.92,
                startTime=3000,
                height=-50)
                annotation (Placement(transformation(extent={{48,10},{28,30}})));
            equation
              connect(separator.drained_out, waterPump.inlet) annotation (Line(
                  points={{0,78},{0,26}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5,
                  arrow={Arrow.None,Arrow.Filled}));
              connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                  points={{-54,88},{-10,88}},
                  color={0,0,255},
                  smooth=Smooth.None,
                  thickness=0.5));

              connect(flowJoin.in1, sourceFW.flange) annotation (Line(
                  points={{8.88178e-016,-140},{8.88178e-016,-142},{66,-142}},
                  color={0,0,255},
                  thickness=0.5));
              connect(waterPump.outlet, flowJoin.in2) annotation (Line(
                  points={{0,6},{0,-132}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sinkSH.flange, separator.steam_out) annotation (Line(
                  points={{70,138},{0,138},{0,120}},
                  color={0,0,255},
                  thickness=0.5));
              connect(separator.level, PID_levelControl.PV)
                annotation (Line(points={{10,82},{76,82},{76,18}}, color={0,0,127}));
              connect(PID_levelControl.CS, firstOrder.u)
                annotation (Line(points={{80,-2},{80,-2},{80,-40}},
                                                           color={0,0,127}));
              connect(const.y, PID_levelControl.SP) annotation (Line(points={{109,82},
                      {98,82},{84,82},{84,18}},
                                        color={0,0,127}));
              connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
                  points={{-74,88},{-74,88},{-100,88},{-100,76}},
                  color={0,0,255},
                  thickness=0.5));
              connect(EVA2.inletHot, sourceMS.flange) annotation (Line(
                  points={{-84,60},{-70,60}},
                  color={255,0,0},
                  thickness=0.5));
              connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                  points={{-116,60},{-140,60},{-140,-20},{-116,-20}},
                  color={255,0,0},
                  thickness=0.5));
              connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                  points={{-84,-20},{-72,-20},{-60,-20},{-60,-100},{-84,-100}},
                  color={255,0,0},
                  thickness=0.5));
              connect(sinkMS.flange, ECO.outletHot) annotation (Line(
                  points={{-140,-100},{-128,-100},{-116,-100}},
                  color={255,0,0},
                  thickness=0.5));
              connect(PDSteamECO.inlet, ECO.outletCold) annotation (Line(
                  points={{-100,-70},{-100,-77},{-100,-84}},
                  color={0,0,255},
                  thickness=0.5));
              connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
                  points={{-100,-36},{-100,-36},{-100,-50}},
                  color={0,0,255},
                  thickness=0.5));
              connect(PDSteamEVA1.inlet, EVA1.outletCold) annotation (Line(
                  points={{-100,12},{-100,12},{-100,-4}},
                  color={0,0,255},
                  thickness=0.5));
              connect(EVA2.inletCold, PDSteamEVA1.outlet) annotation (Line(
                  points={{-100,44},{-100,32}},
                  color={0,0,255},
                  thickness=0.5));
              connect(ECO.inletCold, flowJoin.out) annotation (Line(
                  points={{-100,-116},{-100,-116},{-100,-136},{-12,-136}},
                  color={0,0,255},
                  thickness=0.5));
              connect(sourceMS.in_w0, step.y) annotation (Line(points={{-56,54},{
                      -56,40},{-41,40}},    color={0,0,127}));
              connect(firstOrder.y, sourceFW.in_w0) annotation (Line(points={{80,
                      -63},{80,-63},{80,-136}}, color={0,0,127}));
              connect(waterPump.in_w0, step1.y) annotation (Line(points={{6,20},{
                      6,20},{27,20}}, color={0,0,127}));
              annotation (Diagram(coordinateSystem(extent={{-180,-160},{180,160}},
                      preserveAspectRatio=false)),           Icon(coordinateSystem(
                      extent={{-180,-160},{180,160}}, preserveAspectRatio=false)),
                Documentation(info="<html>
<p>Test model to chek the feasibility of controlling the separator level through the feedwater mass flow rate. The model is initialised at 50&percnt; load. After 1000 s the level set point is changed from 4m to 6m, in order to evaluate the y0/y step response. At 3000 s the MS mass flow is increased, and the controller react correctly by increasing the FW mass flow. According to this test, the control philosophy is correct. </p>
<p>Note that the decoupler between the feedwater and recirculating pump is not necessary, since the sysyem is naturally decoupled.</p>
</html>"),      experiment(StopTime=4000),
                __Dymola_experimentSetupOutput);
            end test_feedwaterLevelControl;
          end OTSG;

        end Test;

        package Old

          model OTSG_full_localControl
            "Complete dynamic model of the once through steam generator with local controllers"
            extends BaseClasses.BaseOTSG;

            //Heat exchangers parameter
            parameter Integer Nhex_mono = 5
              "Number of nodes for single phase heat exchangers";
            parameter Integer Nhex_bi = 10
              "Number of nodes for biphase heat exchangers";

            //Controllers parameter
            //--LEVEL CONTROL--
            parameter Modelica.SIunits.Height zl_sp "Tank level set point" annotation (Dialog(tab="Separator level controller"));
            parameter Real Kp_l "Controller proportional gain" annotation (Dialog(tab="Separator level controller"));
            parameter Modelica.SIunits.Time Ti_l "Controller integral time" annotation (Dialog(tab="Separator level controller"));
            parameter Modelica.SIunits.Time Td_l "Controller derivative time" annotation (Dialog(tab="Separator level controller"));
            parameter Real Nd_l
              "Controller derivative action up to Nd/Td rad/s"                   annotation (Dialog(tab="Separator level controller"));
            parameter Real PVmin_l
              "Minimum value of process variable for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real PVmax_l
              "Maximum value of process variable for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real CSmin_l
              "Minimum value of control signal for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real CSmax_l
              "Maximum value of control signal for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real PVstart_l=0.5 "Start value of PV (scaled)" annotation (Dialog(tab="Separator level controller"));
            parameter Real CSstart_l=0.5 "Start value of CS (scaled)" annotation (Dialog(tab="Separator level controller"));
            parameter Real actuatorGain_l=0.5 "Actuator gain" annotation (Dialog(tab="Separator level controller"));
            parameter Modelica.SIunits.Time Ta_l "Actuator time constant" annotation (Dialog(tab="Separator level controller"));

            //--REHEAT STEAM TEMPERATURE--
            parameter Modelica.SIunits.Temperature T_RH_sp
              "Reheat steam temperature set point"                                              annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real Kp_T "Controller proportional gain" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Modelica.SIunits.Time Ti_T "Controller integral time" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Modelica.SIunits.Time Td_T "Controller derivative time" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real Nd_T
              "Controller derivative action up to Nd/Td rad/s"                   annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real PVmin_T
              "Minimum value of process variable for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real PVmax_T
              "Maximum value of process variable for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real CSmin_T
              "Minimum value of control signal for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real CSmax_T
              "Maximum value of control signal for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real PVstart_T=0.5 "Start value of PV (scaled)" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real CSstart_T=0.5 "Start value of CS (scaled)" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real actuatorGain_T=0.5 "Actuator gain" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Modelica.SIunits.Time Ta_T "Actuator time constant" annotation (Dialog(tab="RH steam temperature controller"));

            Components.HE HPH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              nominalHotFlow=610.6,
              Vhot=77,
              Vcold=114.7,
              Shot=2429.5,
              Scold=1913.2,
              Mm=29335,
              nominalColdFlow=143.1,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=5000, alpha=0.536143),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2000, alpha=0.447465),
              Nhex=Nhex_mono,
              nominalColdPressure=17910000,
              nominalHotPressure=400000,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=581.85,
              nominalHotInletTemperature=586.85,
              nominalHotOutletTemperature=563.15) annotation (Placement(
                  transformation(extent={{-76,-254},{-44,-286}})));

            Components.HE ECO(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.728761, gamma_nom=4400),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.461561, gamma_nom=2200),
              Vhot=99.6,
              Vcold=101.3,
              Shot=2394.2,
              Scold=1855.5,
              Mm=30395,
              Nhex=Nhex_mono,
              nominalColdPressure=17910000,
              nominalHotPressure=569600,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=623.79,
              nominalHotInletTemperature=628.75,
              nominalHotOutletTemperature=586.72)
              annotation (Placement(transformation(extent={{-44,-226},{-76,-194}})));

            Components.HE_2ph
                          EVA1(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              nominalHotFlow=610.6,
              Vhot=67.2,
              Vcold=494.3,
              Mm=43100,
              nominalColdFlow=88.55,
              Shot=2251.9,
              Scold=1765.5,
              Nhex=Nhex_bi,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=4100, alpha=0.440883),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2300, alpha=0.447498),
              nominalColdPressure=17810000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=623.75,
              nominalColdOutletTemperature=629.32,
              nominalHotInletTemperature=688.04,
              nominalHotOutletTemperature=629.15) annotation (Placement(
                  transformation(extent={{-76,-156},{-44,-124}})));
            Components.HE_2ph
                          EVA2(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              nominalHotFlow=610.6,
              Vcold=721.2,
              nominalColdFlow=88.59,
              Vhot=3.9,
              Mm=5182,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.389637, gamma_nom=2650),
              Nhex=Nhex_bi,
              Shot=429.2*1.05,
              Scold=338*1.05,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=3100, alpha=0.440883),
              nominalColdPressure=17760000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=629.15,
              nominalColdOutletTemperature=648.65,
              nominalHotInletTemperature=732.05,
              nominalHotOutletTemperature=687.95) annotation (Placement(
                  transformation(extent={{-44,-76},{-76,-44}})));
            Components.Equilibrium2phVessel separator(
              redeclare package Medium = coldMedium,
              D=0.56,
              H=8,
              zl_min=0,
              zl_max=7,
              zl_start=0,
              h_start=2.7e6)
              annotation (Placement(transformation(extent={{-28,28},{28,84}})));
            ThermoPower.Water.ThroughMassFlow waterPump(
              redeclare package Medium = coldMedium,
              allowFlowReversal=false,
              use_in_w0=true,
              w0=10) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={0,-28})));
            Modelica.Blocks.Sources.Constant zl_SP(
              k=zl_sp)
              annotation (Placement(transformation(extent={{80,60},{60,80}})));

            Components.HE SH(
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              Vhot=12,
              Vcold=174.3,
              Mm=15353,
              nominalColdFlow=88.59,
              nominalHotFlow=371.8,
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              redeclare package hotMedium = hotMedium,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2200, alpha=0.847436),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=3000, alpha=0.28867),
              Nhex=Nhex_mono,
              Shot=1153.8*0.95,
              Scold=879.8*0.95,
              nominalColdPressure=17710000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=648.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=732.02)
              annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                  rotation=0,
                  origin={0,120})));
            Components.HE RH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Vhot=9.4,
              Vcold=446.2,
              Mm=10081,
              nominalColdFlow=82.57,
              nominalHotFlow=238.7,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              Shot=1339,
              Scold=1198,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.798347, gamma_nom=970),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2600, alpha=0.320782),
              nominalColdPressure=4698000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=625.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=731.45)
              annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                  rotation=-90,
                  origin={140,160})));
            ThermoPower.Water.ThroughMassFlow valveRH(
              redeclare package Medium = hotMedium,
              w0=238.7,
              use_in_w0=true)
              annotation (Placement(transformation(extent={{-46,270},{-26,250}})));
            ThermoPower.Water.SensT sensT(redeclare package Medium =
                  coldMedium) annotation (Placement(
                  transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={96,174})));
            ThermoPower.PowerPlants.Control.PID PID_T_RH(
              Ti=Ti_T,
              PVmin=PVmin_T,
              PVmax=PVmax_T,
              CSmin=CSmin_T,
              CSmax=CSmax_T,
              PVstart=PVstart_T,
              CSstart=CSstart_T,
              Kp=Kp_T,
              Td=Td_T,
              Nd=Nd_T)
              annotation (Placement(transformation(extent={{40,194},{20,214}})));
            Modelica.Blocks.Sources.Constant T_RH_SP(k=T_RH_sp)
              annotation (Placement(transformation(extent={{80,220},{60,240}})));

            ThermoPower.Water.PressDrop PDSteamEVA1(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              dpnom=(0.05828)*10^5,
              rhonom=582.24,
              wnom=88.59)          annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-100})));
            ThermoPower.Water.PressDrop PDSteamECO(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.52492)*10^5,
              rhonom=768.15)                 annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-176})));
            ThermoPower.Water.PressDrop PDSteamEVA2(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.11171)*10^5,
              rhonom=250)          annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=0,
                  origin={-32,48})));
            ThermoPower.PowerPlants.Control.PID PID_level(
            Kp=Kp_l,
            Ti=Ti_l,
            Td=Td_l,
            Nd=Nd_l,
            PVmin=PVmin_l,
            PVmax=PVmax_l,
            CSmin=CSmin_l,
            CSmax=CSmax_l,
            CSstart=CSstart_l,
            PVstart=PVstart_l) annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={50,26})));

           Modelica.Blocks.Continuous.FirstOrder pumpDynamic(
              k=actuatorGain_l,
              T=Ta_l) annotation (Placement(
                  transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={50,-4})));
            Modelica.Blocks.Continuous.FirstOrder valveDynamic(
              k=actuatorGain_T,
              T=Ta_T)
              annotation (Placement(transformation(extent={{-10,194},{-30,214}})));
            //Overall parameter and variables:
            Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
            Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
            Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
            Modelica.SIunits.Temperature T_RH = RH.coldFluid.T[end];
            Modelica.SIunits.Length sep_lev = separator.zl;
            Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
            Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
            Real x_EVA1_out = EVA1.coldFluid.x[end];
            Real x_EVA2_out = EVA2.coldFluid.x[end];
            Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
            Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
            Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);

          equation
            connect(RH.inletCold, RH_in) annotation (Line(
                points={{156,160},{190,160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(valveRH.outlet, RH.inletHot) annotation (Line(
                points={{-26,260},{140,260},{140,176}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(MS_in, valveRH.inlet) annotation (Line(
                points={{-100,310},{-100,260},{-46,260}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH.outletHot, EVA2.inletHot) annotation (Line(
                points={{140,144},{140,-60},{-44,-60}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5));
            connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                points={{-76,-60},{-100,-60},{-100,-140},{-76,-140}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                points={{-44,-140},{-20,-140},{-20,-210},{-44,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(HPH.inletHot, ECO.outletHot) annotation (Line(
                points={{-76,-270},{-98,-270},{-98,-210},{-76,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(T_RH_SP.y, PID_T_RH.SP) annotation (Line(
                points={{59,230},{56,230},{56,208},{40,208}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(sensT.T, PID_T_RH.PV) annotation (Line(
                points={{90,182},{90,200},{40,200}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(sensT.inlet, RH.outletCold) annotation (Line(
                points={{100,168},{100,160},{124,160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(HPH.outletCold, MS_HPH_out) annotation (Line(
                points={{-60,-286},{-60,-310}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH_out, sensT.outlet) annotation (Line(
                points={{100,310},{100,180}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(MS_in, SH.inletHot) annotation (Line(
                points={{-100,310},{-100,120},{-16,120}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH.outletHot, EVA2.inletHot) annotation (Line(
                points={{16,120},{140,120},{140,-60},{-44,-60}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH_out, SH.outletCold) annotation (Line(
                points={{0,310},{0,136}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
                points={{-60,-156},{-60,-166}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
                points={{-60,-194},{-60,-186}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                points={{-60,-124},{-60,-110}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(EVA2.inletCold, PDSteamEVA1.outlet) annotation (Line(
                points={{-60,-76},{-60,-90}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                points={{-60,-44},{-60,48},{-42,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(HPH.outletHot, MS_out) annotation (Line(
                points={{-44,-270},{20,-270},{20,-310}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(separator.drained_out, waterPump.inlet) annotation (Line(
                points={{0,38},{0,-18}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(separator.level, PID_level.PV) annotation (Line(
                points={{10,42},{46,42},{46,36}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(PID_level.CS,pumpDynamic. u) annotation (Line(
                points={{50,16},{50,8}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(pumpDynamic.y,waterPump. in_w0) annotation (Line(
                points={{50,-15},{50,-24},{6,-24}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(valveDynamic.u, PID_T_RH.CS) annotation (Line(
                points={{-8,204},{20,204}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(valveRH.in_w0, valveDynamic.y) annotation (Line(
                points={{-40,254},{-40,204},{-31,204}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                points={{-22,48},{-10,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(SH.inletCold, separator.steam_out) annotation (Line(
                points={{0,104},{0,80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(zl_SP.y, PID_level.SP) annotation (Line(
                points={{59,70},{54,70},{54,36}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(waterPump.outlet, SEP_out) annotation (Line(
                points={{0,-38},{0,-80},{190,-80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(FW_in, HPH.inletCold) annotation (Line(
                points={{190,-240},{-60,-240},{-60,-254}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(FW_in, ECO.inletCold) annotation (Line(
                points={{190,-240},{-60,-240},{-60,-226}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                    preserveAspectRatio=false), graphics), Icon(coordinateSystem(
                    extent={{-180,-300},{180,300}})));
          end OTSG_full_localControl;

          model OTSG_full_2ph_localControl
            "Complete dynamic model of the once through steam generator, with separate 2ph HTC and local controllers"
            extends BaseClasses.BaseOTSG;

            //Heat exchangers parameter
            parameter Integer Nhex_mono = 5
              "Number of nodes for single phase heat exchangers";
            parameter Integer Nhex_bi = 10
              "Number of nodes for biphase heat exchangers";

            //Controllers parameter
            //--LEVEL CONTROL--
            parameter Modelica.SIunits.Height zl_sp "Tank level set point" annotation (Dialog(tab="Separator level controller"));
            parameter Real Kp_l "Controller proportional gain" annotation (Dialog(tab="Separator level controller"));
            parameter Modelica.SIunits.Time Ti_l "Controller integral time" annotation (Dialog(tab="Separator level controller"));
            parameter Modelica.SIunits.Time Td_l "Controller derivative time" annotation (Dialog(tab="Separator level controller"));
            parameter Real Nd_l
              "Controller derivative action up to Nd/Td rad/s"                   annotation (Dialog(tab="Separator level controller"));
            parameter Real PVmin_l
              "Minimum value of process variable for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real PVmax_l
              "Maximum value of process variable for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real CSmin_l
              "Minimum value of control signal for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real CSmax_l
              "Maximum value of control signal for scaling"                      annotation (Dialog(tab="Separator level controller"));
            parameter Real PVstart_l=0.5 "Start value of PV (scaled)" annotation (Dialog(tab="Separator level controller"));
            parameter Real CSstart_l=0.5 "Start value of CS (scaled)" annotation (Dialog(tab="Separator level controller"));
            parameter Real actuatorGain_l=0.5 "Actuator gain" annotation (Dialog(tab="Separator level controller"));
            parameter Modelica.SIunits.Time Ta_l "Actuator time constant" annotation (Dialog(tab="Separator level controller"));

            //--REHEAT STEAM TEMPERATURE--
            parameter Modelica.SIunits.Temperature T_RH_sp
              "Reheat steam temperature set point"                                              annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real Kp_T "Controller proportional gain" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Modelica.SIunits.Time Ti_T "Controller integral time" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Modelica.SIunits.Time Td_T "Controller derivative time" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real Nd_T
              "Controller derivative action up to Nd/Td rad/s"                   annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real PVmin_T
              "Minimum value of process variable for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real PVmax_T
              "Maximum value of process variable for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real CSmin_T
              "Minimum value of control signal for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real CSmax_T
              "Maximum value of control signal for scaling"                      annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real PVstart_T=0.5 "Start value of PV (scaled)" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real CSstart_T=0.5 "Start value of CS (scaled)" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Real actuatorGain_T=0.5 "Actuator gain" annotation (Dialog(tab="RH steam temperature controller"));
            parameter Modelica.SIunits.Time Ta_T "Actuator time constant" annotation (Dialog(tab="RH steam temperature controller"));

            Components.HE HPH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              nominalHotFlow=610.6,
              Shot=2429.5,
              Scold=1913.2,
              Mm=29335,
              nominalColdFlow=143.1,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2000, alpha=0.447465),
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=5000, alpha=0.536143),
              Vhot=33.634,
              Vcold=6.027,
              nominalColdPressure=17910000,
              nominalHotPressure=400000,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=581.85,
              nominalHotInletTemperature=586.85,
              nominalHotOutletTemperature=563.15) annotation (Placement(
                  transformation(extent={{-76,-254},{-44,-286}})));

            Components.HE ECO(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Shot=2394.2,
              Scold=1855.5,
              Mm=30395,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.728761, gamma_nom=4400),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.461561, gamma_nom=2200),
              Vhot=45.011,
              Vcold=5.752,
              nominalColdPressure=17910000,
              nominalHotPressure=569600,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=623.79,
              nominalHotInletTemperature=628.75,
              nominalHotOutletTemperature=586.72)
              annotation (Placement(transformation(extent={{-44,-226},{-76,-194}})));

            Components.HE_2ph
                          EVA1(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              Mm=43100,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2300, alpha=0.447498),
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  gamma_nom_liq=3875,
                  gamma_nom_2ph=25350,
                  gamma_nom_vap=3875,
                  alpha=1.6),
              nominalColdFlow=88.59,
              Shot=2251.9*0.94,
              Scold=1765.5*0.94,
              Vhot=26.587,
              Vcold=8.651,
              nominalColdPressure=17810000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=623.75,
              nominalColdOutletTemperature=629.32,
              nominalHotInletTemperature=688.04,
              nominalHotOutletTemperature=629.15) annotation (Placement(
                  transformation(extent={{-76,-156},{-44,-124}})));
            Components.HE_2ph
                          EVA2(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Mm=5182,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.389637, gamma_nom=2650),
              Shot=429.2*0.88,
              Scold=338*0.88,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  alpha=1.6,
                  gamma_nom_2ph=6000,
                  gamma_nom_liq=2600,
                  gamma_nom_vap=2600),
              Vhot=3.905,
              Vcold=1.065,
              nominalColdPressure=17760000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=629.15,
              nominalColdOutletTemperature=648.65,
              nominalHotInletTemperature=732.05,
              nominalHotOutletTemperature=687.95) annotation (Placement(
                  transformation(extent={{-44,-76},{-76,-44}})));

            Components.Equilibrium2phVessel separator(
              redeclare package Medium = coldMedium,
              D=0.56,
              H=8,
              zl_min=0,
              zl_max=7,
              zl_start=0,
              h_start=2.7e6)
              annotation (Placement(transformation(extent={{-28,28},{28,84}})));
            ThermoPower.Water.ThroughMassFlow waterPump(
              redeclare package Medium = coldMedium,
              allowFlowReversal=false,
              use_in_w0=true,
              w0=10) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={0,-28})));

            Components.HE SH(
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={
                    518322,502443,472016,421345,355895}),
              redeclare package coldMedium = coldMedium,
              Mm=15353,
              nominalColdFlow=88.59,
              nominalHotFlow=371.8,
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1,
                hstart={2.74562e+006,3.02031e+006,3.23297e+006,3.36066e+006,
                    3.42731e+006}),
              redeclare package hotMedium = hotMedium,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2200, alpha=0.847436),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=3000, alpha=0.28867),
              Nhex=Nhex_mono,
              Shot=1153.8*0.95,
              Scold=879.8*0.95,
              tubeWalls(Tvolstart={725.391,769.727,804.861,826.169}),
              Vhot=3.68,
              Vcold=2.683,
              nominalColdPressure=17710000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=648.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=732.02)
              annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                  rotation=0,
                  origin={0,120})));
            Components.HE RH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Mm=10081,
              nominalColdFlow=82.57,
              nominalHotFlow=238.7,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1),
              Shot=1339,
              Scold=1198,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.798347, gamma_nom=970),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2600, alpha=0.320782),
              Vhot=1.522,
              Vcold=5.092,
              nominalColdPressure=4698000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=625.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=731.45)
              annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                  rotation=-90,
                  origin={140,160})));
            ThermoPower.Water.ThroughMassFlow valveRH(
              redeclare package Medium = hotMedium,
              w0=238.7,
              use_in_w0=true)
              annotation (Placement(transformation(extent={{-46,270},{-26,250}})));
            ThermoPower.Water.SensT sensT(redeclare package Medium =
                  coldMedium) annotation (Placement(
                  transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={96,174})));

            ThermoPower.Water.PressDrop PDSteamEVA1(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              dpnom=(0.05828)*10^5,
              rhonom=582.24,
              wnom=88.59)          annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-100})));
            ThermoPower.Water.PressDrop PDSteamECO(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.52492)*10^5,
              rhonom=768.15)                 annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-176})));
            ThermoPower.Water.PressDrop PDSteamEVA2(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.11171)*10^5,
              rhonom=250)          annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=0,
                  origin={-32,48})));

            //Overall parameter and variables:
            Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
            Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
            Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
            Modelica.SIunits.Temperature T_RH = RH.coldFluid.T[end];
            Modelica.SIunits.Length sep_lev = separator.zl;
            Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
            Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
            Real x_EVA1_out = EVA1.coldFluid.x[end];
            Real x_EVA2_out = EVA2.coldFluid.x[end];
            Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
            Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
            Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);

            Modelica.Blocks.Continuous.FirstOrder valveDynamic(
              k=actuatorGain_T,
              T=Ta_T)
              annotation (Placement(transformation(extent={{-10,200},{-30,220}})));
            ThermoPower.PowerPlants.Control.PID PID_T_RH(
              Ti=Ti_T,
              PVmin=PVmin_T,
              PVmax=PVmax_T,
              CSmin=CSmin_T,
              CSmax=CSmax_T,
              PVstart=PVstart_T,
              CSstart=CSstart_T,
              Kp=Kp_T,
              Td=Td_T,
              Nd=Nd_T)
              annotation (Placement(transformation(extent={{50,200},{30,220}})));
            Modelica.Blocks.Sources.Constant T_RH_SP(k=T_RH_sp)
              annotation (Placement(transformation(extent={{88,220},{68,240}})));
            Modelica.Blocks.Sources.Constant zl_SP(
              k=zl_sp)
              annotation (Placement(transformation(extent={{80,50},{60,70}})));
            ThermoPower.PowerPlants.Control.PID PID_level(
            Kp=Kp_l,
            Ti=Ti_l,
            Td=Td_l,
            Nd=Nd_l,
            PVmin=PVmin_l,
            PVmax=PVmax_l,
            CSmin=CSmin_l,
            CSmax=CSmax_l,
            CSstart=CSstart_l,
            PVstart=PVstart_l) annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={50,26})));
           Modelica.Blocks.Continuous.FirstOrder pumpDynamic(
              k=actuatorGain_l,
              T=Ta_l) annotation (Placement(
                  transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={50,-2})));
          equation
            connect(RH.inletCold, RH_in) annotation (Line(
                points={{156,160},{190,160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(valveRH.outlet, RH.inletHot) annotation (Line(
                points={{-26,260},{140,260},{140,176}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(MS_in, valveRH.inlet) annotation (Line(
                points={{-100,310},{-100,260},{-46,260}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5));
            connect(HPH.inletHot, ECO.outletHot) annotation (Line(
                points={{-76,-270},{-98,-270},{-98,-210},{-76,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(sensT.inlet, RH.outletCold) annotation (Line(
                points={{100,168},{100,160},{124,160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(HPH.outletCold, MS_HPH_out) annotation (Line(
                points={{-60,-286},{-60,-310}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH_out, sensT.outlet) annotation (Line(
                points={{100,310},{100,180}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(MS_in, SH.inletHot) annotation (Line(
                points={{-100,310},{-100,120},{-16,120}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH_out, SH.outletCold) annotation (Line(
                points={{0,310},{0,136}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
                points={{-60,-194},{-60,-186}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(HPH.outletHot, MS_out) annotation (Line(
                points={{-44,-270},{20,-270},{20,-310}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(separator.drained_out, waterPump.inlet) annotation (Line(
                points={{0,38},{0,-18}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                points={{-22,48},{-10,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(SH.inletCold, separator.steam_out) annotation (Line(
                points={{0,104},{0,80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
                points={{-60,-90},{-60,-76}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                points={{-60,-124},{-60,-110}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
                points={{-60,-156},{-60,-166}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                points={{-60,-44},{-60,48},{-42,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(RH.outletHot, EVA2.inletHot) annotation (Line(
                points={{140,144},{140,-60},{-44,-60}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH.outletHot, EVA2.inletHot) annotation (Line(
                points={{16,120},{140,120},{140,-60},{-44,-60}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                points={{-76,-60},{-100,-60},{-100,-140},{-76,-140}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                points={{-44,-140},{-20,-140},{-20,-210},{-44,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(FW_in, HPH.inletCold) annotation (Line(
                points={{190,-240},{-60,-240},{-60,-254}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(FW_in, ECO.inletCold) annotation (Line(
                points={{190,-240},{-60,-240},{-60,-226}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(waterPump.outlet, SEP_out) annotation (Line(
                points={{-1.77636e-015,-38},{-1.77636e-015,-80},{190,-80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(T_RH_SP.y,PID_T_RH. SP) annotation (Line(
                points={{67,230},{60,230},{60,214},{50,214}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(sensT.T,PID_T_RH. PV) annotation (Line(
                points={{90,182},{90,206},{50,206}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(valveDynamic.u,PID_T_RH. CS) annotation (Line(
                points={{-8,210},{30,210}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(separator.level,PID_level. PV) annotation (Line(
                points={{10,42},{46,42},{46,36}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(PID_level.CS,pumpDynamic. u) annotation (Line(
                points={{50,16},{50,10}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(pumpDynamic.y,waterPump. in_w0) annotation (Line(
                points={{50,-13},{50,-24},{6,-24}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(zl_SP.y,PID_level. SP) annotation (Line(
                points={{59,60},{54,60},{54,36}},
                color={0,0,127},
                smooth=Smooth.None));
            connect(valveDynamic.y, valveRH.in_w0) annotation (Line(
                points={{-31,210},{-40,210},{-40,254}},
                color={0,0,127},
                smooth=Smooth.None));
            annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                    preserveAspectRatio=false), graphics), Icon(coordinateSystem(
                    extent={{-180,-300},{180,300}}, preserveAspectRatio=false),
                  graphics));
          end OTSG_full_2ph_localControl;

          model test_OTSG "Test model of the OTSG"

            OTSG_full_localControl OTSG(
              Kp_T=20,
              Ti_T=20,
              Td_T=0,
              Nd_T=1,
              PVmin_T=600,
              PVmax_T=1000,
              CSmin_T=1,
              CSmax_T=300,
              PVstart_T=0.5,
              CSstart_T=0.8,
              actuatorGain_T=1,
              Ta_T=2,
              zl_sp=4,
              Kp_l=-250,
              Ti_l=20,
              Td_l=0,
              Nd_l=1,
              PVmin_l=0,
              PVmax_l=8,
              CSmin_l=0,
              CSmax_l=150,
              PVstart_l=0.5,
              CSstart_l=0.5,
              actuatorGain_l=1,
              Ta_l=5,
              T_RH_sp=826.05)
              annotation (Placement(transformation(extent={{-54,-40},{0,40}})));
            ThermoPower.Water.SourceMassFlow sourceSteam2(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=1231860,
              w0=231.69,
              use_in_w0=false)
              annotation (Placement(transformation(extent={{60,-50},{40,-30}})));
            ThermoPower.Water.SourceMassFlow sourceSteam1(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              w0=82.57,
              h=3083198,
              p0=4698000,
              use_in_w0=false)
              annotation (Placement(transformation(extent={{60,10},{40,30}})));
            ThermoPower.Water.SinkPressure sinkSteam1(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              p0=4642887,
              use_in_p0=false)
              annotation (Placement(transformation(extent={{0,70},{20,90}})));
            ThermoPower.Water.SourceMassFlow sourceMS1(
              allowFlowReversal=false,
              G=0,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              w0=610.5,
              h=520170,
              use_in_w0=false) annotation (Placement(transformation(extent={{-80,
                      70},{-60,90}})));
            ThermoPower.Water.SinkPressure sinkMS3(
              allowFlowReversal=false,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              p0=261078) annotation (Placement(transformation(extent={{-60,-90},
                      {-80,-70}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
              annotation (Placement(transformation(extent={{80,80},{100,100}})));
            ThermoPower.Water.SinkPressure sinkSteam3(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              p0=17814720)
              annotation (Placement(transformation(extent={{40,-90},{60,-70}})));
            ThermoPower.Water.SinkPressure sinkSteam(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              p0=17611890,
              use_in_p0=false)
              annotation (Placement(transformation(extent={{40,50},{60,70}})));
            ThermoPower.Water.ThroughMassFlow throughMassFlow(redeclare package
                Medium = Modelica.Media.Water.WaterIF97_ph, w0=143.1,
              use_in_w0=false)
              annotation (Placement(transformation(extent={{0,-70},{20,-90}})));
          equation
            connect(OTSG.FW_in, sourceSteam2.flange) annotation (Line(
                points={{1.5,-30.6667},{34,-30.6667},{34,-40},{40,-40}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.RH_in, sourceSteam1.flange) annotation (Line(
                points={{1.5,29.3333},{32,29.3333},{32,20},{40,20}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sourceMS1.flange, OTSG.MS_in) annotation (Line(
                points={{-60,80},{-42,80},{-42,41.3333}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sinkMS3.flange, OTSG.MS_out) annotation (Line(
                points={{-60,-80},{-42,-80},{-42,-41.3333}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.MS_HPH_out, throughMassFlow.inlet) annotation (Line(
                points={{-24,-41.3333},{-24,-80},{0,-80}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(throughMassFlow.outlet, sinkSteam3.flange) annotation (Line(
                points={{20,-80},{40,-80}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.RH_out, sinkSteam1.flange) annotation (Line(
                points={{-24,41.3333},{-24,80},{0,80}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.SH_out, sinkSteam.flange) annotation (Line(
                points={{-12,41.3333},{-12,60},{40,60}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.SEP_out, sourceSteam2.flange) annotation (Line(
                points={{1.5,-10.6667},{34,-10.6667},{34,-40},{40,-40}},
                color={0,0,255},
                smooth=Smooth.None));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                      -100},{100,100}}),        graphics), Icon(coordinateSystem(
                    extent={{-100,-100},{100,100}})));
          end test_OTSG;

          model test_OTSG_2ph "Test model of the OTSG"

            Old.OTSG_full_2ph_localControl OTSG(
              Kp_T=20,
              Ti_T=20,
              Td_T=0,
              Nd_T=1,
              PVmin_T=600,
              PVmax_T=1000,
              CSmin_T=1,
              CSmax_T=300,
              PVstart_T=0.5,
              CSstart_T=0.8,
              actuatorGain_T=1,
              Ta_T=2,
              zl_sp=4,
              Kp_l=-250,
              Ti_l=20,
              Td_l=0,
              Nd_l=1,
              PVmin_l=0,
              PVmax_l=8,
              CSmin_l=0,
              CSmax_l=150,
              PVstart_l=0.5,
              CSstart_l=0.5,
              actuatorGain_l=1,
              Ta_l=5,
              T_RH_sp=826.05)
              annotation (Placement(transformation(extent={{-54,-40},{0,40}})));
            ThermoPower.Water.SourceMassFlow sourceSteam2(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=1231860,
              w0=231.69,
              use_in_w0=false)
              annotation (Placement(transformation(extent={{60,-50},{40,-30}})));
            ThermoPower.Water.SourceMassFlow sourceSteam1(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              w0=82.57,
              h=3083198,
              p0=4698000,
              use_in_w0=false)
              annotation (Placement(transformation(extent={{60,10},{40,30}})));
            ThermoPower.Water.SinkPressure sinkSteam1(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              p0=4642887,
              use_in_p0=false)
              annotation (Placement(transformation(extent={{0,70},{20,90}})));
            ThermoPower.Water.SourceMassFlow sourceMS1(
              allowFlowReversal=false,
              G=0,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              w0=610.5,
              use_in_w0=false,
              h=518322) annotation (Placement(transformation(extent={{-80,70},{
                      -60,90}})));
            ThermoPower.Water.SinkPressure sinkMS3(
              allowFlowReversal=false,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              p0=261078) annotation (Placement(transformation(extent={{-60,-90},
                      {-80,-70}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
              annotation (Placement(transformation(extent={{80,80},{100,100}})));
            ThermoPower.Water.SinkPressure sinkSteam3(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              p0=17814720)
              annotation (Placement(transformation(extent={{40,-90},{60,-70}})));
            ThermoPower.Water.SinkPressure sinkSteam(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              p0=17611890,
              use_in_p0=false)
              annotation (Placement(transformation(extent={{40,50},{60,70}})));
            ThermoPower.Water.ThroughMassFlow throughMassFlow(redeclare package
                Medium = Modelica.Media.Water.WaterIF97_ph, w0=143.1,
              use_in_w0=false)
              annotation (Placement(transformation(extent={{0,-70},{20,-90}})));
          equation
            connect(OTSG.FW_in, sourceSteam2.flange) annotation (Line(
                points={{1.5,-30.6667},{34,-30.6667},{34,-40},{40,-40}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.RH_in, sourceSteam1.flange) annotation (Line(
                points={{1.5,29.3333},{32,29.3333},{32,20},{40,20}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sourceMS1.flange, OTSG.MS_in) annotation (Line(
                points={{-60,80},{-42,80},{-42,41.3333}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sinkMS3.flange, OTSG.MS_out) annotation (Line(
                points={{-60,-80},{-42,-80},{-42,-41.3333}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.MS_HPH_out, throughMassFlow.inlet) annotation (Line(
                points={{-24,-41.3333},{-24,-80},{0,-80}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(throughMassFlow.outlet, sinkSteam3.flange) annotation (Line(
                points={{20,-80},{40,-80}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.RH_out, sinkSteam1.flange) annotation (Line(
                points={{-24,41.3333},{-24,80},{0,80}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.SH_out, sinkSteam.flange) annotation (Line(
                points={{-12,41.3333},{-12,60},{40,60}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.SEP_out, sourceSteam2.flange) annotation (Line(
                points={{1.5,-10.6667},{34,-10.6667},{34,-40},{40,-40}},
                color={0,0,255},
                smooth=Smooth.None));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                      -100},{100,100}}),        graphics), Icon(coordinateSystem(
                    extent={{-100,-100},{100,100}})));
          end test_OTSG_2ph;

          model OTSG_full_2ph_busControl_init100_5nodes
            "Complete dynamic model of the once through steam generator, with separate 2ph HTC"
            extends BaseClasses.BaseOTSG;

            //Heat exchangers parameter
            parameter Integer Nhex_mono = 5
              "Number of nodes for single phase heat exchangers";
            parameter Integer Nhex_bi = 10
              "Number of nodes for biphase heat exchangers";

            Components.HE HPH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={
                    136706,129914,122014,112572,100947}),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1,
                hstart={1.23186e+006,1.28146e+006,1.32175e+006,1.35546e+006,
                    1.38444e+006}),
              nominalHotFlow=610.6,
              Shot=2429.5,
              Scold=1913.2,
              Mm=29335,
              nominalColdFlow=143.1,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2000, alpha=0.447465),
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=5000, alpha=0.536143),
              tubeWalls(Tvolstart={560.978,569.126,575.7,581.167}),
              Vhot=33.634,
              Vcold=6.027,
              nominalColdPressure=17910000,
              nominalHotPressure=400000,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=581.85,
              nominalHotInletTemperature=586.85,
              nominalHotOutletTemperature=563.15) annotation (Placement(
                  transformation(extent={{-76,-254},{-44,-286}})));

            Components.HE ECO(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={
                    199579,192529,183185,168175,136603}),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1,
                hstart={1.23186e+006,1.44947e+006,1.55292e+006,1.61733e+006,
                    1.66591e+006}),
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Shot=2394.2,
              Scold=1855.5,
              Mm=30395,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.728761, gamma_nom=4400),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.461561, gamma_nom=2200),
              tubeWalls(Tvolstart={582.675,605.885,616.62,623.137}),
              Vhot=45.011,
              Vcold=5.752,
              nominalColdPressure=17910000,
              nominalHotPressure=569600,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=623.79,
              nominalHotInletTemperature=628.75,
              nominalHotOutletTemperature=586.72)
              annotation (Placement(transformation(extent={{-44,-226},{-76,-194}})));

            Components.HE_2ph
                          EVA1(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={
                    289063,250302,228476,216201,209301,205481,203503,201573,199646,
                    197679}),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1,
                hstart={1.66561e+006,1.67917e+006,1.69246e+006,1.70577e+006,
                    1.7194e+006,1.74575e+006,1.79332e+006,1.87797e+006,2.02847e+006,
                    2.29575e+006}),
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              Mm=43100,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2300, alpha=0.447498),
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  gamma_nom_liq=3875,
                  gamma_nom_2ph=25350,
                  gamma_nom_vap=3875,
                  alpha=1.6),
              nominalColdFlow=88.59,
              Shot=2251.9*0.94,
              Scold=1765.5*0.94,
              tubeWalls(Tvolstart={626.054,627.39,628.669,629.917,629.888,630.182,
                    630.842,632.016,634.097}),
              Vhot=26.587,
              Vcold=8.651,
              nominalColdPressure=17810000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=623.75,
              nominalColdOutletTemperature=629.32,
              nominalHotInletTemperature=688.04,
              nominalHotOutletTemperature=629.15) annotation (Placement(
                  transformation(extent={{-76,-116},{-44,-84}})));
            Components.HE_2ph
                          EVA2(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={
                    355877,349098,342248,335400,328621,320944,311763,303278,295435,
                    288187}),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1,
                hstart={2.2799e+006,2.32986e+006,2.38391e+006,2.4424e+006,
                    2.50567e+006,2.55859e+006,2.60531e+006,2.65251e+006,
                    2.69972e+006,2.74645e+006}),
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Mm=5182,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.389637, gamma_nom=2650),
              Shot=429.2*0.88,
              Scold=338*0.88,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  alpha=1.6,
                  gamma_nom_2ph=6000,
                  gamma_nom_liq=2600,
                  gamma_nom_vap=2600),
              tubeWalls(Tvolstart={650.941,652.73,654.665,656.757,670.379,679.959,
                    684.074,688.573,693.427}),
              Vhot=3.905,
              Vcold=1.065,
              nominalColdPressure=17760000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=629.15,
              nominalColdOutletTemperature=648.65,
              nominalHotInletTemperature=732.05,
              nominalHotOutletTemperature=687.95) annotation (Placement(
                  transformation(extent={{-44,-36},{-76,-4}})));

            Components.Equilibrium2phVessel separator(
              redeclare package Medium = coldMedium,
              H=8,
              zl_min=0,
              zl_max=7,
              zl_start=0,
              h_start=2.7e6,
              D=0.56,
              initOpt=ThermoPower.Choices.Init.Options.fixedState,
              p_start=17710000)
              annotation (Placement(transformation(extent={{-28,28},{28,84}})));
            ThermoPower.Water.ThroughMassFlow waterPump(
              redeclare package Medium = coldMedium,
              allowFlowReversal=false,
              use_in_w0=true,
              w0=10) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={0,4})));

            Components.HE SH(
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={
                    518322,502443,472016,421345,355895}),
              redeclare package coldMedium = coldMedium,
              Mm=15353,
              nominalColdFlow=88.59,
              nominalHotFlow=371.8,
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1,
                hstart={2.74562e+006,3.02031e+006,3.23297e+006,3.36066e+006,
                    3.42731e+006}),
              redeclare package hotMedium = hotMedium,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2200, alpha=0.847436),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=3000, alpha=0.28867),
              Nhex=Nhex_mono,
              Shot=1153.8*0.95,
              Scold=879.8*0.95,
              tubeWalls(Tvolstart={725.391,769.727,804.861,826.169}),
              Vhot=3.68,
              Vcold=2.683,
              nominalColdPressure=17710000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=648.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=732.02)
              annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                  rotation=0,
                  origin={0,120})));
            Components.HE RH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart={
                    518322,503128,476511,430304,353343}),
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Mm=10081,
              nominalColdFlow=82.57,
              nominalHotFlow=238.7,
              coldFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, Nt=1,
                hstart={3.0832e+006,3.30568e+006,3.43926e+006,3.51621e+006,
                    3.56013e+006}),
              Shot=1339,
              Scold=1198,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.798347, gamma_nom=970),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2600, alpha=0.320782),
              tubeWalls(Tvolstart={734.495,783.217,812.321,829.047}),
              Vhot=1.522,
              Vcold=5.092,
              nominalColdPressure=4698000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=625.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=731.45)
              annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                  rotation=-90,
                  origin={140,160})));
            ThermoPower.Water.ThroughMassFlow valveRH(
              redeclare package Medium = hotMedium,
              w0=238.7,
              use_in_w0=true)
              annotation (Placement(transformation(extent={{-46,270},{-26,250}})));
            ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
              annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={96,230})));

            ThermoPower.Water.PressDrop PDSteamEVA1(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              dpnom=(0.05828)*10^5,
              rhonom=582.24,
              wnom=88.59)          annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-60})));
            ThermoPower.Water.PressDrop PDSteamECO(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.52492)*10^5,
              rhonom=768.15)                 annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-176})));
            ThermoPower.Water.PressDrop PDSteamEVA2(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.11171)*10^5,
              rhonom=250)          annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=0,
                  origin={-32,48})));
            //Overall parameter and variables:
            Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
            Modelica.SIunits.Temperature T_inFW = sensT_FW.T;
            Modelica.SIunits.Temperature T_outMS = sensT_MS.T;
            Modelica.SIunits.Temperature T_RH = sensT_RH.T;
            Modelica.SIunits.Length sep_lev = separator.zl;
            Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
            Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
            Real x_EVA1_out = EVA1.coldFluid.x[end];
            Real x_EVA2_out = EVA2.coldFluid.x[end];
            Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
            Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
            Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
            Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

            ThermoPower.Water.PressDrop PDSteamSH(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              dpnom=(177.1 - 176.1189)*10^5,
              rhonom=96.586,
              wnom=88.59)                    annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={0,188})));
            ThermoPower.Water.PressDrop PDSteamRH(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              wnom=82.57,
              dpnom=(0.55113)*10^5,
              rhonom=17.86)                  annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={100,190})));
            Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                  extent={{-30,-30},{30,30}},
                  rotation=-90,
                  origin={50,10}), iconTransformation(
                  extent={{-30,-30},{30,30}},
                  rotation=-90,
                  origin={-194,0})));
            ThermoPower.Water.SensT sensT_MS(
              redeclare package Medium = hotMedium)
              annotation (Placement(transformation(extent={{-20,-276},{0,-256}})));
            ThermoPower.Water.SensT sensT_FW(
              redeclare package Medium = coldMedium)
              annotation (Placement(transformation(extent={{102,-246},{82,-226}})));
            ThermoPower.Water.SensP sensP_ECO(
              redeclare package Medium = coldMedium) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={-56,-140})));
          equation
            connect(RH.inletCold, RH_in) annotation (Line(
                points={{156,160},{190,160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(valveRH.outlet, RH.inletHot) annotation (Line(
                points={{-26,260},{140,260},{140,176}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(MS_in, valveRH.inlet) annotation (Line(
                points={{-100,310},{-100,260},{-46,260}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5));
            connect(HPH.inletHot, ECO.outletHot) annotation (Line(
                points={{-76,-270},{-100,-270},{-100,-210},{-76,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(HPH.outletCold, MS_HPH_out) annotation (Line(
                points={{-60,-286},{-60,-310}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH_out, sensT_RH.outlet) annotation (Line(
                points={{100,310},{100,236}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(MS_in, SH.inletHot) annotation (Line(
                points={{-100,310},{-100,120},{-16,120}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
                points={{-60,-194},{-60,-186}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(separator.drained_out, waterPump.inlet) annotation (Line(
                points={{0,38},{0,14},{1.77636e-015,14}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                points={{-22,48},{-10,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(SH.inletCold, separator.steam_out) annotation (Line(
                points={{0,104},{0,80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
                points={{-60,-50},{-60,-36}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                points={{-60,-84},{-60,-70}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                points={{-60,-4},{-60,48},{-42,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(RH.outletHot, EVA2.inletHot) annotation (Line(
                points={{140,144},{140,-20},{-44,-20}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH.outletHot, EVA2.inletHot) annotation (Line(
                points={{16,120},{140,120},{140,-20},{-44,-20}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                points={{-44,-100},{-20,-100},{-20,-210},{-44,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(waterPump.outlet, SEP_out) annotation (Line(
                points={{-1.77636e-015,-6},{-1.77636e-015,-80},{190,-80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
                points={{0,136},{0,178}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(SH_out, PDSteamSH.outlet) annotation (Line(
                points={{0,310},{0,198}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
                points={{100,224},{100,200}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
                points={{124,160},{100,160},{100,180}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));

            connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
                points={{50,10},{50,10},{50,-20},{50,8},{6,8}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
                points={{50,10},{50,10},{50,246},{50,244},{-40,244},{-40,254}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
                points={{50,10},{50,10},{50,42},{10,42}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
                points={{50,10},{50,10},{50,244},{90,244},{90,238}},
                color={255,204,51},
                thickness=0.5));
            connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
                points={{-44,-270},{-30,-270},{-16,-270}},
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled},
                color={255,0,0}));
            connect(sensT_MS.outlet, MS_out) annotation (Line(
                points={{-4,-270},{20,-270},{20,-310}},
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled},
                color={255,0,0}));
            connect(sensT_FW.inlet, FW_in) annotation (Line(
                points={{98,-240},{98,-240},{190,-240}},
                color={0,0,255},
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(sensT_FW.outlet, HPH.inletCold) annotation (Line(
                points={{86,-240},{86,-240},{-60,-240},{-60,-254}},
                color={0,0,255},
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(sensT_FW.outlet, ECO.inletCold) annotation (Line(
                points={{86,-240},{86,-240},{-60,-240},{-60,-226}},
                color={0,0,255},
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
                points={{50,10},{50,10},{50,-200},{50,-230},{84,-230}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveBP_PV, sensT_MS.T) annotation (Line(
                points={{50,10},{50,10},{50,-260},{-2,-260}},
                color={255,204,51},
                thickness=0.5));
            connect(EVA1.inletCold, sensP_ECO.flange) annotation (Line(
                points={{-60,-116},{-60,-140}},
                color={0,0,255},
                thickness=0.5));
            connect(sensP_ECO.flange, PDSteamECO.outlet) annotation (Line(
                points={{-60,-140},{-60,-166}},
                color={0,0,255},
                thickness=0.5));
            connect(signalBus.pumpFW_PV, sensP_ECO.p) annotation (Line(
                points={{50,10},{50,10},{50,-158},{-50,-158},{-50,-148}},
                color={255,204,51},
                thickness=0.5));
            annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                    preserveAspectRatio=false)),           Icon(coordinateSystem(
                    extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
          end OTSG_full_2ph_busControl_init100_5nodes;

          model OTSG_init100_5nodes_old
            "Dynamic model of the once through steam generator, enthalpy profile initialisation at 100% load and 5 nodes"
            extends BaseClasses.BaseOTSG;

            //Heat exchangers parameter
            parameter Integer Nhex_mono = 5
              "Number of nodes for single phase heat exchangers";
            parameter Integer Nhex_bi = 10
              "Number of nodes for biphase heat exchangers";

            Components.HE HPH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              nominalHotFlow=610.6,
              Shot=2429.5,
              Scold=1913.2,
              Mm=29335,
              nominalColdFlow=143.1,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2000, alpha=0.447465),
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=5000, alpha=0.536143),
              Vhot=33.634,
              Vcold=6.027,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={1.23186e+006,1.28146e+006,1.32175e+006,1.35546e+006,
                    1.38444e+006}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {136706,129914,122014,112572,100947}),
              tubeWalls(Tvolstart(displayUnit="K") = {560.978,569.126,575.7,581.167}),
              nominalColdPressure=17910000,
              nominalHotPressure=400000,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=581.85,
              nominalHotInletTemperature=586.85,
              nominalHotOutletTemperature=563.15) annotation (Placement(
                  transformation(extent={{-76,-254},{-44,-286}})));

            Components.HE ECO(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Shot=2394.2,
              Scold=1855.5,
              Mm=30395,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.728761, gamma_nom=4400),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.461561, gamma_nom=2200),
              Vhot=45.011,
              Vcold=5.752,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={1.23186e+006,1.44947e+006,1.55292e+006,1.61733e+006,
                    1.66591e+006}),
              tubeWalls(Tvolstart(displayUnit="K") = {582.675,605.885,616.62,
                  623.137}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {199579,192529,183185,168175,136603}),
              nominalColdPressure=17910000,
              nominalHotPressure=569600,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=623.79,
              nominalHotInletTemperature=628.75,
              nominalHotOutletTemperature=586.72)
              annotation (Placement(transformation(extent={{-44,-226},{-76,-194}})));

            Components.HE_2ph
                          EVA1(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              Mm=43100,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2300, alpha=0.447498),
              nominalColdFlow=88.59,
              Vhot=26.587,
              Vcold=8.651,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={1.66561e+006,1.67917e+006,1.69246e+006,1.70577e+006,
                    1.7194e+006,1.74575e+006,1.79332e+006,1.87797e+006,2.02847e+006,
                    2.29575e+006}),
              tubeWalls(Tvolstart(displayUnit="K") = {626.054,627.39,628.669,
                  629.917,629.888,630.182,630.842,632.016,634.097}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {289063,250302,228476,216201,209301,205481,203503,201573,199646,
                    197679}),
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  gamma_nom_liq=3875,
                  gamma_nom_vap=3875,
                  alpha=1.6,
                  gamma_nom_2ph=10000),
              Shot=2251.9*0.81,
              Scold=1765.5*0.81,
              nominalColdPressure=17810000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=623.75,
              nominalColdOutletTemperature=629.32,
              nominalHotInletTemperature=688.04,
              nominalHotOutletTemperature=629.15) annotation (Placement(
                  transformation(extent={{-76,-116},{-44,-84}})));
            Components.HE_2ph
                          EVA2(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Mm=5182,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.389637, gamma_nom=2650),
              Vhot=3.905,
              Vcold=1.065,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={2.2799e+006,2.32986e+006,2.38391e+006,2.4424e+006,
                    2.50567e+006,2.55859e+006,2.60531e+006,2.65251e+006,
                    2.69972e+006,2.74645e+006}),
              tubeWalls(Tvolstart(displayUnit="K") = {650.941,652.73,654.665,
                  656.757,670.379,679.959,684.074,688.573,693.427}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {355877,349098,342248,335400,328621,320944,311763,303278,295435,
                    288187}),
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  alpha=1.6,
                  gamma_nom_liq=2000,
                  gamma_nom_2ph=5000,
                  gamma_nom_vap=2000),
              Shot=429.2*0.88,
              Scold=338*0.88,
              nominalColdPressure=17760000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=629.15,
              nominalColdOutletTemperature=648.65,
              nominalHotInletTemperature=732.05,
              nominalHotOutletTemperature=687.95) annotation (Placement(
                  transformation(extent={{-44,-36},{-76,-4}})));

            Components.Equilibrium2phVessel separator(
              redeclare package Medium = coldMedium,
              H=8,
              h_start=2.7e6,
              D=0.56,
              initOpt=ThermoPower.Choices.Init.Options.fixedState,
              zl_start=0,
              p_start=17710000)
              annotation (Placement(transformation(extent={{-28,28},{28,84}})));
            ThermoPower.Water.ThroughMassFlow waterPump(
              redeclare package Medium = coldMedium,
              allowFlowReversal=false,
              use_in_w0=true,
              w0=10) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={0,6})));

            Components.HE SH(
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              Mm=15353,
              nominalColdFlow=88.59,
              nominalHotFlow=371.8,
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              redeclare package hotMedium = hotMedium,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2200, alpha=0.847436),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=3000, alpha=0.28867),
              Nhex=Nhex_mono,
              Vhot=3.68,
              Vcold=2.683,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={2.74562e+006,3.02031e+006,3.23297e+006,3.36066e+006,
                    3.42731e+006}),
              tubeWalls(Tvolstart(displayUnit="K") = {725.391,769.727,804.861,
                  826.169}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {518322,502443,472016,421345,355895}),
              Shot=1153.8*0.95,
              Scold=879.8*0.95,
              nominalColdPressure=17710000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=648.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=732.02)
              annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                  rotation=0,
                  origin={0,120})));
            Components.HE RH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Mm=10081,
              nominalColdFlow=82.57,
              nominalHotFlow=238.7,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.798347, gamma_nom=970),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2600, alpha=0.320782),
              Vhot=1.522,
              Vcold=5.092,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={3.0832e+006,3.30568e+006,3.43926e+006,3.51621e+006,
                    3.56013e+006}),
              tubeWalls(Tvolstart(displayUnit="K") = {734.495,783.217,812.321,
                  829.047}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {518322,503128,476511,430304,353343}),
              Shot=1339,
              Scold=1198,
              nominalColdPressure=4698000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=625.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=731.45)
              annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                  rotation=-90,
                  origin={140,160})));
            ThermoPower.Water.ThroughMassFlow valveRH(
              redeclare package Medium = hotMedium,
              w0=238.7,
              use_in_w0=true)
              annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
            ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
              annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={96,210})));

            ThermoPower.Water.PressDrop PDSteamEVA1(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              dpnom=(0.05828)*10^5,
              wnom=88.59,
              rhonom=168.8)        annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-60})));
            ThermoPower.Water.PressDrop PDSteamECO(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.52492)*10^5,
              rhonom=768.15)                 annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-150})));
            ThermoPower.Water.PressDrop PDSteamEVA2(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.11171)*10^5,
              rhonom=97)           annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=0,
                  origin={-32,48})));
            //Overall parameter and variables:
            Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
            Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
            Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
            Modelica.SIunits.Temperature T_RH = sensT_RH.T;
            Modelica.SIunits.Length sep_lev = separator.zl;
            Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
            Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
            Real x_EVA1_out = EVA1.coldFluid.x[end];
            Real x_EVA2_out = EVA2.coldFluid.x[end];
            Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
            Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
            Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
            Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

            ThermoPower.Water.PressDrop PDSteamRH(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              wnom=82.57,
              dpnom=(0.55113)*10^5,
              rhonom=12.52)                  annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={100,180})));
            Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                  extent={{-30,-30},{30,30}},
                  rotation=-90,
                  origin={50,10}), iconTransformation(
                  extent={{-30,-30},{30,30}},
                  rotation=-90,
                  origin={-194,0})));
            ThermoPower.Water.SensT sensT_FW(
              redeclare package Medium = coldMedium)
              annotation (Placement(transformation(extent={{102,-246},{82,-226}})));
            ThermoPower.Water.SensT sensT_MS(
              redeclare package Medium = hotMedium)
              annotation (Placement(transformation(extent={{-20,-276},{0,-256}})));
            ThermoPower.Water.ThroughMassFlow throughMassFlow(
              w0=610.6,
              use_in_w0=true,
              redeclare package Medium = hotMedium)
              annotation (Placement(transformation(
                  extent={{10,10},{-10,-10}},
                  rotation=90,
                  origin={-100,256})));
            ThermoPower.Water.SensP sensP(
              redeclare package Medium = coldMedium) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={4,200})));
            ThermoPower.Water.PressDrop PDSteamSH(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              wnom=88.59,
              rhonom=51.9,
              dpnom=(177.1 - 176.1)*10^5)    annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={0,170})));
            ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph,
              Av=0.046406353,
              wnom=88.59,
              thetanom=0.8,
              Fxt_full=0.4542722663,
              pnom=17610000,
              dpnom=110000)
              annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                  rotation=90,
                  origin={0,280})));
          equation
            connect(RH.inletCold, RH_in) annotation (Line(
                points={{156,160},{190,160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(valveRH.outlet, RH.inletHot) annotation (Line(
                points={{-26,230},{140,230},{140,176}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(HPH.inletHot, ECO.outletHot) annotation (Line(
                points={{-76,-270},{-100,-270},{-100,-210},{-76,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(HPH.outletCold, MS_HPH_out) annotation (Line(
                points={{-60,-286},{-60,-310}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH_out, sensT_RH.outlet) annotation (Line(
                points={{100,310},{100,216}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
                points={{-60,-194},{-60,-160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(separator.drained_out, waterPump.inlet) annotation (Line(
                points={{0,38},{0,16},{1.77636e-015,16}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                points={{-22,48},{-10,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(SH.inletCold, separator.steam_out) annotation (Line(
                points={{0,104},{0,80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
                points={{-60,-50},{-60,-36}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                points={{-60,-84},{-60,-70}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA2.outletCold, PDSteamEVA2.inlet) annotation (Line(
                points={{-60,-4},{-60,48},{-42,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(RH.outletHot, EVA2.inletHot) annotation (Line(
                points={{140,144},{140,-20},{-44,-20}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH.outletHot, EVA2.inletHot) annotation (Line(
                points={{16,120},{140,120},{140,-20},{-44,-20}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                points={{-44,-100},{-20,-100},{-20,-210},{-44,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(waterPump.outlet, SEP_out) annotation (Line(
                points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
                points={{100,204},{100,190}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
                points={{124,160},{100,160},{100,170}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));

            connect(sensT_FW.inlet, FW_in) annotation (Line(
                points={{98,-240},{144,-240},{190,-240}},
                color={0,0,255},
                thickness=0.5));
            connect(sensT_FW.outlet, ECO.inletCold) annotation (Line(
                points={{86,-240},{14,-240},{-60,-240},{-60,-226}},
                color={0,0,255},
                thickness=0.5));
            connect(sensT_FW.outlet, HPH.inletCold) annotation (Line(
                points={{86,-240},{14,-240},{-60,-240},{-60,-254}},
                color={0,0,255},
                thickness=0.5));
            connect(HPH.outletHot, sensT_MS.inlet) annotation (Line(
                points={{-44,-270},{-16,-270}},
                color={255,0,0},
                thickness=0.5));
            connect(sensT_MS.outlet, MS_out) annotation (Line(
                points={{-4,-270},{20,-270},{20,-310}},
                color={255,0,0},
                thickness=0.5));
            connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
                points={{50,10},{30,10},{6,10}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
                points={{50,10},{50,10},{50,44},{50,42},{10,42}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
                points={{50,10},{50,10},{50,-152},{50,-230},{50,-230},{84,-230}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
                points={{50,10},{50,10},{50,218},{-40,218},{-40,224}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
                points={{50,10},{50,10},{50,218},{90,218}},
                color={255,204,51},
                thickness=0.5));
            connect(MS_in, throughMassFlow.inlet) annotation (Line(
                points={{-100,310},{-100,266}},
                color={255,0,0},
                thickness=0.5));
            connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
                points={{-100,246},{-100,120},{-16,120}},
                color={255,0,0},
                thickness=0.5));
            connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
                points={{-100,246},{-100,246},{-100,230},{-46,230}},
                color={255,0,0},
                thickness=0.5));
            connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
                points={{50,10},{50,10},{50,260},{-94,260}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
                points={{50,10},{50,10},{50,260},{50,192},{10,192}},
                color={255,204,51},
                thickness=0.5));
            connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
                points={{-60,-116},{-60,-140}},
                color={0,0,255},
                thickness=0.5));
            connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
                points={{50,10},{50,10},{50,-230},{50,-260},{-2,-260}},
                color={255,204,51},
                thickness=0.5));
            connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
                points={{0,200},{6.66134e-016,200},{6.66134e-016,180}},
                color={0,0,255},
                thickness=0.5));
            connect(PDSteamSH.inlet, SH.outletCold) annotation (Line(
                points={{0,160},{0,136}},
                color={0,0,255},
                thickness=0.5));
            connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
                points={{50,10},{50,10},{50,280},{8,280}},
                color={255,204,51},
                thickness=0.5));
            connect(valveVap.inlet, sensP.flange) annotation (Line(
                points={{-4.44089e-016,270},{-4.44089e-016,236},{0,236},{0,200}},
                color={0,0,255},
                thickness=0.5));
            connect(SH_out, valveVap.outlet) annotation (Line(
                points={{0,310},{0,300},{0,290}},
                color={0,0,255},
                thickness=0.5));
            annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                    preserveAspectRatio=false)),           Icon(coordinateSystem(
                    extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
          end OTSG_init100_5nodes_old;

          model OTSG_init50_5nodes_old
            "Dynamic model of the once through steam generator, enthalpy profile initialisation at 50% load and 5 nodes"
            extends BaseClasses.BaseOTSG;

            //Heat exchangers parameter
            parameter Integer Nhex_mono = 5
              "Number of nodes for single phase heat exchangers";
            parameter Integer Nhex_bi = 10
              "Number of nodes for biphase heat exchangers";

            Components.HE HPH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              nominalHotFlow=610.6,
              Shot=2429.5,
              Scold=1913.2,
              Mm=29335,
              nominalColdFlow=143.1,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2000, alpha=0.447465),
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=5000, alpha=0.536143),
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={1.25188e+006,1.26667e+006,1.25173e+006,1.26683e+006,
                    1.25157e+006}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {93356,93346.5,93356.1,93346.6,93356.1}),
              tubeWalls(Tvolstart(displayUnit="K") = {557.843,557.837,557.843,
                  557.837}),
              Vhot=33.634,
              Vcold=6.027,
              nominalColdPressure=17910000,
              nominalHotPressure=400000,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=581.85,
              nominalHotInletTemperature=586.85,
              nominalHotOutletTemperature=563.15) annotation (Placement(
                  transformation(extent={{-76,-254},{-44,-286}})));

            Components.HE ECO(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Shot=2394.2,
              Scold=1855.5,
              Mm=30395,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.728761, gamma_nom=4400),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.461561, gamma_nom=2200),
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={1.25188e+006,1.26536e+006,1.2819e+006,1.30253e+006,
                    1.3288e+006}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {119114,110526,103784,98376.4,93969.4}),
              tubeWalls(Tvolstart(displayUnit="degC") = {558.324,561.292,564.934,
                  569.474}),
              Vhot=45.011,
              Vcold=5.752,
              nominalColdPressure=17910000,
              nominalHotPressure=569600,
              nominalColdInletTemperature=553.15,
              nominalColdOutletTemperature=623.79,
              nominalHotInletTemperature=628.75,
              nominalHotOutletTemperature=586.72)
              annotation (Placement(transformation(extent={{-44,-226},{-76,-194}})));

            Components.HE_2ph
                          EVA1(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              Mm=43100,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2300, alpha=0.447498),
              nominalColdFlow=88.59,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={1.33e+006,1.3378e+006,1.34658e+006,1.35651e+006,
                    1.36781e+006,1.38075e+006,1.39566e+006,1.41303e+006,
                    1.45658e+006,1.56606e+006}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {193708,157918,143681,138009,133137,128914,125226,121985,119121,
                    116578}),
              tubeWalls(Tvolstart(displayUnit="K") = {572.404,573.98,575.749,
                  577.742,579.999,582.568,585.511,585.959,587.035}),
              Vhot=26.587,
              Vcold=8.651,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  gamma_nom_liq=3875,
                  gamma_nom_vap=3875,
                  alpha=1.6,
                  gamma_nom_2ph=10000),
              Shot=2251.9*0.81,
              Scold=1765.5*0.81,
              nominalColdPressure=17810000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=623.75,
              nominalColdOutletTemperature=629.32,
              nominalHotInletTemperature=688.04,
              nominalHotOutletTemperature=629.15) annotation (Placement(
                  transformation(extent={{-76,-116},{-44,-84}})));
            Components.HE_2ph
                          EVA2(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Nhex=Nhex_bi,
              nominalHotFlow=610.6,
              nominalColdFlow=88.59,
              Mm=5182,
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.389637, gamma_nom=2650),
              Shot=429.2*0.88,
              Scold=338*0.88,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={1.56606e+006,1.59353e+006,1.62512e+006,1.66145e+006,
                    1.70321e+006,1.75123e+006,1.80642e+006,1.86984e+006,
                    1.94272e+006,2.02642e+006}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {344122,316769,292958,272234,254202,238514,224868,213001,202681,
                    193708}),
              tubeWalls(Tvolstart(displayUnit="degC") = {596.276,597.951,599.875,
                  602.087,604.627,607.543,610.89,614.73,619.134}),
              Vhot=3.905,
              Vcold=1.065,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient2ph
                  (
                  wnom_ht=88.59,
                  alpha=1.6,
                  gamma_nom_liq=2000,
                  gamma_nom_2ph=5000,
                  gamma_nom_vap=2000),
              nominalColdPressure=17760000,
              nominalHotPressure=730800,
              nominalColdInletTemperature=629.15,
              nominalColdOutletTemperature=648.65,
              nominalHotInletTemperature=732.05,
              nominalHotOutletTemperature=687.95) annotation (Placement(
                  transformation(extent={{-44,-36},{-76,-4}})));

            Components.Equilibrium2phVessel separator(
              redeclare package Medium = coldMedium,
              H=8,
              zl_start=4,
              h_start=2016500,
              phase(start=2, fixed=true),
              D=0.56,
              p_start=10100000,
              initOpt=ThermoPower.Choices.Init.Options.steadyState,
              T(start=584.85))
              annotation (Placement(transformation(extent={{-28,28},{28,84}})));
            ThermoPower.Water.ThroughMassFlow waterPump(
              redeclare package Medium = coldMedium,
              allowFlowReversal=false,
              use_in_w0=true,
              w0=10) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={0,6})));

            Components.HE SH(
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              Mm=15353,
              nominalColdFlow=88.59,
              nominalHotFlow=371.8,
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              redeclare package hotMedium = hotMedium,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2200, alpha=0.847436),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=3000, alpha=0.28867),
              Nhex=Nhex_mono,
              Shot=1153.8*0.95,
              Scold=879.8*0.95,
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={2.72364e+006,3.22113e+006,3.45073e+006,3.5168e+006,
                    3.53402e+006}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {518322,514827,501414,454805,353814}),
              tubeWalls(Tvolstart={733.236,797.905,827.469,835.959}),
              Vhot=3.68,
              Vcold=2.683,
              nominalColdPressure=17710000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=648.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=732.02)
              annotation (Placement(transformation(extent={{-16,-16},{16,16}},
                  rotation=0,
                  origin={0,120})));
            Components.HE RH(
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package coldMedium = coldMedium,
              redeclare package hotMedium = hotMedium,
              Mm=10081,
              nominalColdFlow=82.57,
              nominalHotFlow=238.7,
              Shot=1339,
              Scold=1198,
              Nhex=Nhex_mono,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.798347, gamma_nom=970),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (gamma_nom=2600, alpha=0.320782),
              coldFluid(
                FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
                Nt=1,
                hstart={3.12533e+006,3.32376e+006,3.45094e+006,3.53171e+006,
                    3.58318e+006}),
              hotFluid(FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction, hstart=
                   {518322,493431,454370,392870,296910}),
              tubeWalls(Tvolstart(displayUnit="K") = {712.396,768.496,804.215,
                  826.852}),
              Vhot=1.522,
              Vcold=5.092,
              nominalColdPressure=4698000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=625.65,
              nominalColdOutletTemperature=826.06,
              nominalHotInletTemperature=838.15,
              nominalHotOutletTemperature=731.45)
              annotation (Placement(transformation(extent={{-16,16},{16,-16}},
                  rotation=-90,
                  origin={140,160})));
            ThermoPower.Water.ThroughMassFlow valveRH(
              redeclare package Medium = hotMedium,
              w0=238.7,
              use_in_w0=true)
              annotation (Placement(transformation(extent={{-46,240},{-26,220}})));
            ThermoPower.Water.SensT sensT_RH(redeclare package Medium = coldMedium)
              annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={96,210})));

            ThermoPower.Water.PressDrop PDSteamEVA1(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              dpnom=(0.05828)*10^5,
              wnom=88.59,
              rhonom=168.8,
              pin(start=10200000)) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-60})));
            ThermoPower.Water.PressDrop PDSteamECO(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.52492)*10^5,
              rhonom=768.15,
              pin(start=10300000))           annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-60,-150})));
            ThermoPower.Water.PressDrop PDSteamEVA2(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = coldMedium,
              wnom=88.59,
              dpnom=(0.11171)*10^5,
              rhonom=97,
              pin(start=10150000)) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=0,
                  origin={-32,48})));
            //Overall parameter and variables:
            Modelica.SIunits.Power Q_tot = HPH.Q_Hot + ECO.Q_Hot + EVA1.Q_Hot + EVA2.Q_Hot + SH.Q_Hot + RH.Q_Hot;
            Modelica.SIunits.Temperature T_inFW = ECO.coldFluid.T[1];
            Modelica.SIunits.Temperature T_outMS = HPH.hotFluid.T[end];
            Modelica.SIunits.Temperature T_RH = sensT_RH.T;
            Modelica.SIunits.Length sep_lev = separator.zl;
            Modelica.SIunits.Temperature DT_sh = max(0,EVA2.coldFluid.T[end] - EVA2.coldFluid.Ts) + 273.15;
            Modelica.SIunits.Temperature DT_sc = max(0,EVA1.coldFluid.Ts - EVA1.coldFluid.T[1]) + 273.15;
            Real x_EVA1_out = EVA1.coldFluid.x[end];
            Real x_EVA2_out = EVA2.coldFluid.x[end];
            Modelica.SIunits.Temperature T_hot_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_hot_prof;EVA2.T_hot_prof[2:end];EVA1.T_hot_prof[2:end];ECO.T_hot_prof[2:end]]);
            Modelica.SIunits.Temperature T_cold_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_cold_prof;EVA2.T_cold_prof[2:end];EVA1.T_cold_prof[2:end];ECO.T_cold_prof[2:end]]);
            Modelica.SIunits.Temperature T_wall_prof[2*Nhex_mono + 2*Nhex_bi + 1] = vector([SH.T_wall_prof;EVA2.T_wall_prof[2:end];EVA1.T_wall_prof[2:end];ECO.T_wall_prof[2:end]]);
            Real load = EVA2.hotFluid.w/EVA2.nominalHotFlow*100;

            ThermoPower.Water.PressDrop PDSteamSH(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              wnom=88.59,
              dpnom=(177.1 - 176.1)*10^5,
              rhonom=51.9)                   annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={0,170})));
            ThermoPower.Water.PressDrop PDSteamRH(
              FFtype=ThermoPower.Choices.PressDrop.FFtypes.OpPoint,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              wnom=82.57,
              dpnom=(0.55113)*10^5,
              rhonom=17.86)                  annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={100,180})));
            Modelica.Icons.SignalBus signalBus annotation (Placement(transformation(
                  extent={{-30,-30},{30,30}},
                  rotation=-90,
                  origin={50,10}), iconTransformation(
                  extent={{-30,-30},{30,30}},
                  rotation=-90,
                  origin={-194,0})));
            ThermoPower.Water.SensT sensT_FW(
              redeclare package Medium = coldMedium)
              annotation (Placement(transformation(extent={{102,-246},{82,-226}})));
            ThermoPower.Water.SensT sensT_MS(
              redeclare package Medium = hotMedium)
              annotation (Placement(transformation(extent={{-20,-276},{0,-256}})));
            ThermoPower.Water.ThroughMassFlow throughMassFlow(
              w0=610.6,
              use_in_w0=true,
              redeclare package Medium = hotMedium)
              annotation (Placement(transformation(
                  extent={{10,10},{-10,-10}},
                  rotation=90,
                  origin={-100,256})));
            ThermoPower.Water.SensP sensP(
              redeclare package Medium = coldMedium) annotation (Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=-90,
                  origin={4,200})));
            ThermoPower.Water.ValveVap valveVap(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph,
              Av=0.046406353,
              wnom=88.59,
              thetanom=0.8,
              Fxt_full=0.4542722663,
              pnom=17610000,
              dpnom=110000)
              annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                  rotation=90,
                  origin={0,280})));
          equation
            connect(RH.inletCold, RH_in) annotation (Line(
                points={{156,160},{190,160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(valveRH.outlet, RH.inletHot) annotation (Line(
                points={{-26,230},{140,230},{140,176}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(HPH.inletHot, ECO.outletHot) annotation (Line(
                points={{-76,-270},{-100,-270},{-100,-210},{-76,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(HPH.outletCold, MS_HPH_out) annotation (Line(
                points={{-60,-286},{-60,-310}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH_out, sensT_RH.outlet) annotation (Line(
                points={{100,310},{100,216}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(ECO.outletCold, PDSteamECO.inlet) annotation (Line(
                points={{-60,-194},{-60,-160}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(separator.drained_out, waterPump.inlet) annotation (Line(
                points={{0,38},{0,16},{1.77636e-015,16}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(PDSteamEVA2.outlet, separator.water_in) annotation (Line(
                points={{-22,48},{-10,48}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(SH.inletCold, separator.steam_out) annotation (Line(
                points={{0,104},{0,80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(PDSteamEVA1.outlet, EVA2.inletCold) annotation (Line(
                points={{-60,-50},{-60,-36}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletCold, PDSteamEVA1.inlet) annotation (Line(
                points={{-60,-84},{-60,-70}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.inletCold, PDSteamECO.outlet) annotation (Line(
                points={{-60,-116},{-60,-140}},
                color={0,0,255},
                thickness=0.5,
                arrow={Arrow.Filled,Arrow.None}));
            connect(RH.outletHot, EVA2.inletHot) annotation (Line(
                points={{140,144},{140,-20},{-44,-20}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH.outletHot, EVA2.inletHot) annotation (Line(
                points={{16,120},{140,120},{140,-20},{-44,-20}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA2.outletHot, EVA1.inletHot) annotation (Line(
                points={{-76,-20},{-100,-20},{-100,-100},{-76,-100}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(EVA1.outletHot, ECO.inletHot) annotation (Line(
                points={{-44,-100},{-20,-100},{-20,-210},{-44,-210}},
                color={255,0,0},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(waterPump.outlet, SEP_out) annotation (Line(
                points={{-1.77636e-015,-4},{-1.77636e-015,-80},{190,-80}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5,
                arrow={Arrow.None,Arrow.Filled}));
            connect(SH.outletCold, PDSteamSH.inlet) annotation (Line(
                points={{0,136},{0,160},{-6.66134e-016,160}},
                color={0,0,255},
                thickness=0.5));
            connect(sensT_RH.inlet, PDSteamRH.outlet) annotation (Line(
                points={{100,204},{100,190}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));
            connect(RH.outletCold, PDSteamRH.inlet) annotation (Line(
                points={{124,160},{100,160},{100,170}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5));

            connect(sensT_FW.inlet, FW_in) annotation (Line(
                points={{98,-240},{144,-240},{190,-240}},
                color={0,0,255},
                thickness=0.5));
            connect(ECO.inletCold, sensT_FW.outlet) annotation (Line(
                points={{-60,-226},{-60,-240},{86,-240}},
                color={0,0,255},
                thickness=0.5));
            connect(HPH.inletCold, sensT_FW.outlet) annotation (Line(
                points={{-60,-254},{-60,-240},{86,-240}},
                color={0,0,255},
                thickness=0.5));
            connect(sensT_MS.inlet, HPH.outletHot) annotation (Line(
                points={{-16,-270},{-30,-270},{-44,-270}},
                color={255,0,0},
                thickness=0.5));
            connect(sensT_MS.outlet, MS_out) annotation (Line(
                points={{-4,-270},{10,-270},{20,-270},{20,-310}},
                color={255,0,0},
                thickness=0.5));
            connect(signalBus.pumpSEP_CS, waterPump.in_w0) annotation (Line(
                points={{50,10},{6,10},{6,10}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.pumpSEP_PV, separator.level) annotation (Line(
                points={{50,10},{50,10},{50,42},{10,42}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveRH_CS, valveRH.in_w0) annotation (Line(
                points={{50,10},{50,10},{50,70},{50,218},{-40,218},{-40,224}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveRH_PV, sensT_RH.T) annotation (Line(
                points={{50,10},{50,10},{50,218},{90,218}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.pumpHPH_PV, sensT_FW.T) annotation (Line(
                points={{50,10},{50,10},{50,-226},{50,-230},{84,-230}},
                color={255,204,51},
                thickness=0.5));
            connect(MS_in, throughMassFlow.inlet) annotation (Line(
                points={{-100,310},{-100,288},{-100,266}},
                color={255,0,0},
                thickness=0.5));
            connect(throughMassFlow.outlet, valveRH.inlet) annotation (Line(
                points={{-100,246},{-100,246},{-100,226},{-100,230},{-46,230}},
                color={255,0,0},
                thickness=0.5));
            connect(throughMassFlow.outlet, SH.inletHot) annotation (Line(
                points={{-100,246},{-100,246},{-100,186},{-100,120},{-16,120}},
                color={255,0,0},
                thickness=0.5));
            connect(sensP.flange, PDSteamSH.outlet) annotation (Line(
                points={{0,200},{0,180}},
                color={0,0,255},
                thickness=0.5));
            connect(PDSteamEVA2.inlet, EVA2.outletCold) annotation (Line(
                points={{-42,48},{-50,48},{-60,48},{-60,-4}},
                color={0,0,255},
                thickness=0.5));
            connect(signalBus.pumpFW_PV, sensT_MS.T) annotation (Line(
                points={{50,10},{50,10},{50,-252},{50,-260},{-2,-260}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.pumpMS_CS, throughMassFlow.in_w0) annotation (Line(
                points={{50,10},{50,10},{50,258},{50,260},{-94,260}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveTURB_PV, sensP.p) annotation (Line(
                points={{50,10},{50,10},{50,190},{50,192},{10,192}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.valveTURB_CS, valveVap.theta) annotation (Line(
                points={{50,10},{50,10},{50,280},{8,280}},
                color={255,204,51},
                thickness=0.5));
            connect(valveVap.inlet, sensP.flange) annotation (Line(
                points={{0,270},{0,235},{0,200}},
                color={0,0,255},
                thickness=0.5));
            connect(SH_out, valveVap.outlet) annotation (Line(
                points={{0,310},{0,290}},
                color={0,0,255},
                thickness=0.5));
            annotation (Diagram(coordinateSystem(extent={{-180,-300},{180,300}},
                    preserveAspectRatio=false)),           Icon(coordinateSystem(
                    extent={{-180,-300},{180,300}}, preserveAspectRatio=false)));
          end OTSG_init50_5nodes_old;

          model test_OTSG_4_optA_old "Test model of the OTSG at 4% of load"

            OTSG_init20_20nodes OTSG(
              separator(
                initOpt=ThermoPower.Choices.Init.Options.fixedState,
                h_start=1697500,
                p_start=8540000),
              PDSteamEVA1(pin(start=8620000)),
              PDSteamECO(pin(start=8660000)),
              PDSteamEVA2(pin(start=8580000)),
              PDSteamRH(pin(start=22.05e5)))
              annotation (Placement(transformation(extent={{-44,-36},{12,56}})));
            ThermoPower.Water.SourceMassFlow sourceFW(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=true,
              w0=17.01,
              h=515800)
              annotation (Placement(transformation(extent={{110,-4},{90,-24}})));
            ThermoPower.Water.SourceMassFlow sourceMS(
              allowFlowReversal=false,
              G=0,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              h=518322,
              w0=24.424,
              use_in_w0=true) annotation (Placement(transformation(extent={{-116,
                      100},{-96,80}})));
            ThermoPower.Water.SinkPressure sinkMS(
              allowFlowReversal=false,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              p0=261078) annotation (Placement(transformation(extent={{-20,-70},
                      {0,-50}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
              annotation (Placement(transformation(extent={{-140,120},{-120,140}})));
            Modelica.Icons.SignalBus signalBus
              annotation (Placement(transformation(extent={{94,-70},{114,-50}})));
            ThermoPower.Water.SourceMassFlow sourceRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=false,
              w0=4.18,
              h=3092400)
              annotation (Placement(transformation(extent={{76,34},{56,54}})));
            ThermoPower.Water.SinkPressure sinkRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=2200000)
              annotation (Placement(transformation(extent={{22,114},{42,134}})));
            ThermoPower.Water.SinkPressure sinkSH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=8500000)
              annotation (Placement(transformation(extent={{62,94},{82,114}})));
            IndustrialControlSystems.Controllers.PI pI(
              AntiWindup=true,
              CSmin=0,
              CSmax=620,
              CS_start=55,
              Kp=-0.3,
              Ti=1200)
                      annotation (Placement(transformation(
                  extent={{10,10},{-10,-10}},
                  rotation=0,
                  origin={-90,-110})));
            Modelica.Blocks.Continuous.FirstOrder firstOrder(
              initType=Modelica.Blocks.Types.Init.SteadyState,
              k=1,
              T=5)                                             annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-110,10})));
            Modelica.Blocks.Sources.Constant const(k=260 + 273.15) annotation (
                Placement(transformation(extent={{-40,-140},{-60,-120}})));
            ThermoPower.Water.SensT sensT
              annotation (Placement(transformation(extent={{42,-34},{22,-54}})));
            Control.plant_controller controllers(
              Kp_valveRH=1.77,
              Td_valveRH=0,
              Nd_valveRH=1,
              CSmin_valveRH=1,
              CSmax_valveRH=300,
              actuatorGain_valveRH=1,
              Ta_valveRH=2,
              useTS_valveRH=false,
              Kp_pumpSEP=-14.5,
              Ti_pumpSEP=56.7,
              Td_pumpSEP=0,
              Nd_pumpSEP=1,
              CSmin_pumpSEP=0,
              CSmax_pumpSEP=70,
              actuatorGain_pumpSEP=1,
              Ta_pumpSEP=5,
              Td_pumpFW=0,
              Nd_pumpFW=1,
              Ta_pumpFW=5,
              Kp_valveBP=1,
              Ti_valveBP=1,
              Td_valveBP=0,
              Nd_valveBP=1,
              CSmin_valveBP=0,
              CSmax_valveBP=200,
              CSstart_valveBP=153.6,
              actuatorGain_valveBP=1,
              Ta_valveBP=2,
              useTS_valveBP=true,
              TS_valveBP=153.6,
              Kp_pumpHPH=1,
              Ti_pumpHPH=1,
              Td_pumpHPH=0,
              Nd_pumpHPH=1,
              CSmin_pumpHPH=0,
              CSmax_pumpHPH=200,
              CSstart_pumpHPH=143.1,
              actuatorGain_pumpHPH=1,
              Ta_pumpHPH=5,
              useTS_pumpHPH=true,
              TS_pumpHPH=143.1,
              actuatorGain_pumpFW=1,
              l_sp=4,
              CSmin_pumpFW=1,
              CSmax_pumpFW=120,
              useTS_pumpFW=false,
              Kp_pumpFW=0.003,
              Ti_pumpFW=800,
              useTS_pumpSEP=false,
              CSstart_valveRH=6.08,
              TS_valveRH=6.08,
              CSstart_pumpSEP=10.45,
              TS_pumpSEP=10.45,
              p_SH_sp=13.99,
              CSstart_pumpFW=3.54,
              TS_pumpFW=3.54,
              f_load(k=10.45),
              load_start=4,
              load_end=4,
              t_ramp=1,
              Ti_valveRH=16,
              T_RH_sp=826.05,
              T_MS_sp=563.15,
              T_FW_sp=553.15)
              annotation (Placement(transformation(extent={{-80,-90},{-60,-70}})));
          equation
            connect(sourceMS.flange, OTSG.MS_in) annotation (Line(
                points={{-96,90},{-31.5556,90},{-31.5556,57.5333}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.SEP_out, sourceFW.flange) annotation (Line(
                points={{13.5556,-2.26667},{44,-2.26667},{44,-14},{90,-14}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(signalBus.pumpFW_CS, sourceFW.in_w0) annotation (Line(
                points={{104,-60},{104,-60},{104,-20}},
                color={255,204,51},
                thickness=0.5), Text(
                string="%first",
                index=-1,
                extent={{-6,3},{-6,3}}));
            connect(OTSG.RH_in,sourceRH. flange) annotation (Line(
                points={{13.5556,43.7333},{54,43.7333},{54,44},{56,44}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.SH_out,sinkSH. flange) annotation (Line(
                points={{-0.444444,57.5333},{-0.444444,104},{62,104}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sinkRH.flange, OTSG.RH_out) annotation (Line(points={{22,124},
                    {4,124},{-12.8889,124},{-12.8889,57.5333}},
                                                      color={0,0,255}));
            connect(pI.CS, firstOrder.u) annotation (Line(points={{-99,-110},{-110,
                    -110},{-110,-2}}, color={0,0,127}));
            connect(const.y, pI.SP) annotation (Line(points={{-61,-130},{-70,-130},
                    {-70,-116},{-82,-116}}, color={0,0,127}));
            connect(OTSG.MS_out, sinkMS.flange) annotation (Line(points={{
                    -31.5556,-37.5333},{-31.5556,-60},{-20,-60}},
                                                           color={0,0,255}));
            connect(OTSG.FW_in, sensT.outlet) annotation (Line(points={{13.5556,
                    -25.2667},{20,-25.2667},{20,-40},{26,-40}}, color={0,0,255}));
            connect(sourceFW.flange, sensT.inlet) annotation (Line(points={{90,-14},
                    {68,-14},{44,-14},{44,-40},{38,-40}}, color={0,0,255}));
            connect(sensT.T, pI.PV) annotation (Line(points={{24,-50},{20,-50},{20,
                    -106},{20,-108},{-82,-108},{-82,-109}},               color={0,
                    0,127}));
            connect(sourceMS.in_w0, firstOrder.y) annotation (Line(points={{-110,84},
                    {-110,21}},           color={0,0,127}));
            connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                points={{-59.75,-82.6286},{-52,-82.6286},{-52,10},{-46.1778,10}},
                color={255,204,51},
                thickness=0.5));
            connect(controllers.signalBus, signalBus) annotation (Line(
                points={{-59.75,-82.6286},{104,-82.6286},{104,-60}},
                color={255,204,51},
                thickness=0.5), Text(
                string="%second",
                index=1,
                extent={{6,3},{6,3}}));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,
                      -140},{140,140}}), graphics={Text(
                    extent={{18,-100},{138,-140}},
                    lineColor={28,108,200},
                    textString="Simulare per 20000 s a 500 punti
per raggiungere regime")}),                                Icon(coordinateSystem(
                    extent={{-140,-140},{140,140}}, preserveAspectRatio=false)));
          end test_OTSG_4_optA_old;

          model test_OTSG_4_optB "Test model of the OTSG at 4% of load"

            OTSG_init4_5nodes_noDP OTSG(
              separator(
                zl_start=0,
                phase(fixed=true, start=1),
                h_start=3e6,
                initOpt=ThermoPower.Choices.Init.Options.fixedState,
                p_start=8510000,
                T(start=643.15)),
              SH(nominalColdPressure=8510000),
              EVA2(nominalColdPressure=8500000),
              EVA1(nominalColdPressure=8500000),
              ECO(nominalColdPressure=8500000))
              annotation (Placement(transformation(extent={{-64,-36},{-8,56}})));
            ThermoPower.Water.SourceMassFlow sourceFW(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=true,
              w0=17.01,
              h=515800)
              annotation (Placement(transformation(extent={{162,-4},{142,-24}})));
            ThermoPower.Water.SourceMassFlow sourceMS(
              allowFlowReversal=false,
              G=0,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              h=518322,
              w0=20,
              use_in_w0=false) annotation (Placement(transformation(extent={{-136,
                      100},{-116,80}})));
            ThermoPower.Water.SinkPressure sinkMS(
              allowFlowReversal=false,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              p0=261078) annotation (Placement(transformation(extent={{20,-70},
                      {40,-50}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
              annotation (Placement(transformation(extent={{-180,120},{-160,140}})));
            Modelica.Icons.SignalBus signalBus
              annotation (Placement(transformation(extent={{146,-70},{166,-50}})));
            ThermoPower.Water.SourceMassFlow sourceRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=false,
              w0=4.18,
              h=3092400)
              annotation (Placement(transformation(extent={{28,34},{8,54}})));
            ThermoPower.Water.SinkPressure sinkRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=2200000)
              annotation (Placement(transformation(extent={{2,114},{22,134}})));
            ThermoPower.Water.SinkPressure sinkSH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=8500000)
              annotation (Placement(transformation(extent={{42,94},{62,114}})));
            IndustrialControlSystems.Controllers.PI pI(
              AntiWindup=true,
              CSmin=0,
              CSmax=620,
              Ti=650,
              CS_start=20,
              Kp=5)   annotation (Placement(transformation(
                  extent={{10,10},{-10,-10}},
                  rotation=0,
                  origin={-110,-110})));
            Modelica.Blocks.Continuous.FirstOrder firstOrder(
              k=1,
              T=5,
              initType=Modelica.Blocks.Types.Init.SteadyState) annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-130,10})));
            Modelica.Blocks.Sources.Constant const(k=275 + 273.15) annotation (
                Placement(transformation(extent={{-60,-140},{-80,-120}})));
            ThermoPower.Water.SensT sensT(redeclare package Medium =
                  PreFlexMS.Media.MoltenSalt60_40_quad) annotation (Placement(
                  transformation(extent={{-32,-54},{-12,-74}})));
            Control.plant_controller controllers(
              Kp_valveRH=1.77,
              Ti_valveRH=160,
              Td_valveRH=0,
              Nd_valveRH=1,
              CSmin_valveRH=1,
              CSmax_valveRH=300,
              actuatorGain_valveRH=1,
              Ta_valveRH=2,
              useTS_valveRH=false,
              Kp_pumpSEP=-14.5,
              Ti_pumpSEP=56.7,
              Td_pumpSEP=0,
              Nd_pumpSEP=1,
              CSmin_pumpSEP=0,
              CSmax_pumpSEP=70,
              actuatorGain_pumpSEP=1,
              Ta_pumpSEP=5,
              Td_pumpFW=0,
              Nd_pumpFW=1,
              Ta_pumpFW=5,
              Kp_valveBP=1,
              Ti_valveBP=1,
              Td_valveBP=0,
              Nd_valveBP=1,
              CSmin_valveBP=0,
              CSmax_valveBP=200,
              CSstart_valveBP=153.6,
              actuatorGain_valveBP=1,
              Ta_valveBP=2,
              useTS_valveBP=true,
              TS_valveBP=153.6,
              Kp_pumpHPH=1,
              Ti_pumpHPH=1,
              Td_pumpHPH=0,
              Nd_pumpHPH=1,
              CSmin_pumpHPH=0,
              CSmax_pumpHPH=200,
              CSstart_pumpHPH=143.1,
              actuatorGain_pumpHPH=1,
              Ta_pumpHPH=5,
              useTS_pumpHPH=true,
              TS_pumpHPH=143.1,
              actuatorGain_pumpFW=1,
              CSmin_pumpFW=1,
              CSmax_pumpFW=120,
              Kp_pumpFW=0.003,
              Ti_pumpFW=800,
              CSstart_valveRH=6.08,
              TS_valveRH=6.08,
              CSstart_pumpFW=3.54,
              TS_pumpFW=3.54,
              l_sp=0,
              CSstart_pumpSEP=0,
              p_SH_sp=3.54,
              f_load(k=0),
              useTS_pumpFW=true,
              useTS_pumpSEP=true,
              TS_pumpSEP=0,
              load_start=4,
              load_end=4,
              t_ramp=1,
              T_RH_sp=826.05,
              T_MS_sp=563.15,
              T_FW_sp=553.15)
              annotation (Placement(transformation(extent={{-100,-90},{-80,-70}})));
            ThermoPower.Water.SourceMassFlow sourceMassFlow_ST(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=3583500,
              w0=1.4,
              use_in_w0=true)
              annotation (Placement(transformation(extent={{140,10},{120,30}})));
            ThermoPower.Water.SourceMassFlow sourceMassFlow_DR(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              w0=0,
              h=1e5)
              annotation (Placement(transformation(extent={{68,10},{88,30}})));
            ThermoPower.Water.SinkPressure sinkPressure_DR(redeclare package
                Medium =
                  Modelica.Media.Water.WaterIF97_ph, p0=900000)
              annotation (Placement(transformation(extent={{120,-60},{140,-40}})));
            PreHeaters.Preheater_simplified
                                      HPH2(
              cm=500,
              nominalColdDeltaP=0,
              nominalHotDeltaP=0,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_drainIN=true,
              Vhot=2.99,
              Vcold=0.53,
              Mm=10000,
              D=0.4,
              Shot=200,
              Scold=200,
              redeclare model HeatExchangerTopology =
                  ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow,
              redeclare model HeatTransferCold =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.8, gamma_nom=2820),
              redeclare model HeatTransferHot =
                  ThermoPower.Thermal.HeatTransferFV.FlowDependentHeatTransferCoefficient
                  (alpha=0.8, gamma_nom=2820),
              Kv=3.73e-5,
              nominalColdFlow=3.54,
              nominalHotFlow=3.73,
              Nhex=10,
              nominalColdPressure=8600000,
              nominalHotPressure=1000000,
              nominalColdInletTemperature=394.65,
              nominalColdOutletTemperature=533.15,
              nominalHotInletTemperature=821.15,
              nominalHotOutletTemperature=453.15)
              annotation (Placement(transformation(extent={{82,-34},{122,6}})));
            ThermoPower.Water.SensT sensT1(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph)
              annotation (Placement(transformation(extent={{22,-30},{2,-10}})));
            IndustrialControlSystems.Controllers.PI pI1(
              AntiWindup=true,
              Kp=0.032,
              Ti=20,
              CSmax=3.73,
              CS_start=1.4,
              CSmin=0.5)
                      annotation (Placement(transformation(
                  extent={{10,10},{-10,-10}},
                  rotation=0,
                  origin={186,52})));
            Modelica.Blocks.Sources.Constant const1(k=260 + 273.15)
                                                                   annotation (
                Placement(transformation(extent={{236,-4},{216,16}})));
            Modelica.Blocks.Continuous.FirstOrder firstOrder1(
              k=1,
              initType=Modelica.Blocks.Types.Init.SteadyState,
              T=1)                                             annotation (
                Placement(transformation(
                  extent={{-10,10},{10,-10}},
                  rotation=180,
                  origin={150,52})));
          equation
            connect(sourceMS.flange, OTSG.MS_in) annotation (Line(
                points={{-116,90},{-51.5556,90},{-51.5556,57.5333}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(signalBus.pumpFW_CS, sourceFW.in_w0) annotation (Line(
                points={{156,-60},{156,-50},{156,-40},{156,-20}},
                color={255,204,51},
                thickness=0.5), Text(
                string="%first",
                index=-1,
                extent={{-6,3},{-6,3}}));
            connect(OTSG.SH_out,sinkSH. flange) annotation (Line(
                points={{-20.4444,57.5333},{-20.4444,104},{42,104}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sinkRH.flange, OTSG.RH_out) annotation (Line(points={{2,124},
                    {-16,124},{-32.8889,124},{-32.8889,57.5333}},
                                                      color={0,0,255}));
            connect(pI.CS, firstOrder.u) annotation (Line(points={{-119,-110},{-130,
                    -110},{-130,-2}}, color={0,0,127}));
            connect(const.y, pI.SP) annotation (Line(points={{-81,-130},{-90,-130},
                    {-90,-116},{-102,-116}},color={0,0,127}));
            connect(sensT.T, pI.PV) annotation (Line(points={{-14,-70},{-14,-70},{
                    -10,-70},{-8,-70},{-8,-110},{-102,-110},{-102,-109}}, color={0,
                    0,127}));
            connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                points={{-79.75,-82.6286},{-72,-82.6286},{-72,10},{-66.1778,10}},
                color={255,204,51},
                thickness=0.5));
            connect(controllers.signalBus, signalBus) annotation (Line(
                points={{-79.75,-82.6286},{156,-82.6286},{156,-60}},
                color={255,204,51},
                thickness=0.5), Text(
                string="%second",
                index=1,
                extent={{6,3},{6,3}}));
            connect(sourceFW.flange, HPH2.FW_in)
              annotation (Line(points={{142,-14},{124,-14}}, color={0,0,255}));
            connect(sourceMassFlow_DR.flange, HPH2.DR_in)
              annotation (Line(points={{88,20},{88,6}}, color={0,0,255}));
            connect(sourceMassFlow_ST.flange, HPH2.ST_in) annotation (Line(points={
                    {120,20},{102,20},{102,6}}, color={0,0,255}));
            connect(HPH2.DR_out, sinkPressure_DR.flange) annotation (Line(points={{
                    102,-34},{102,-50},{120,-50}}, color={0,0,255}));
            connect(HPH2.FW_out, OTSG.SEP_out) annotation (Line(points={{82,-14},
                    {20,-14},{20,-2.26667},{-6.44444,-2.26667}},color={0,0,255}));
            connect(OTSG.MS_out, sensT.inlet) annotation (Line(points={{-51.5556,
                    -37.5333},{-51.5556,-60},{-28,-60}}, color={0,0,255}));
            connect(sensT.outlet, sinkMS.flange) annotation (Line(points={{-16,-60},
                    {10,-60},{20,-60}}, color={0,0,255}));
            connect(OTSG.RH_in, sourceRH.flange) annotation (Line(points={{
                    -6.44444,43.7333},{0.77778,43.7333},{0.77778,44},{8,44}},
                                                                     color={0,0,255}));
            connect(OTSG.FW_in, sensT1.outlet) annotation (Line(points={{-6.44444,
                    -25.2667},{0,-25.2667},{0,-24},{6,-24}}, color={0,0,255}));
            connect(HPH2.FW_out, sensT1.inlet) annotation (Line(points={{82,-14},{
                    20,-14},{20,-24},{18,-24}}, color={0,0,255}));
            connect(pI1.CS, firstOrder1.u)
              annotation (Line(points={{177,52},{162,52}}, color={0,0,127}));
            connect(const1.y, pI1.SP) annotation (Line(points={{215,6},{206,6},{206,
                    46},{194,46}}, color={0,0,127}));
            connect(firstOrder1.y, sourceMassFlow_ST.in_w0) annotation (Line(points=
                   {{139,52},{134,52},{134,26}}, color={0,0,127}));
            connect(sensT1.T, pI1.PV) annotation (Line(points={{4,-14},{4,20},{60,
                    20},{60,82},{208,82},{208,53},{194,53}}, color={0,0,127}));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-180,
                      -140},{180,140}})),                  Icon(coordinateSystem(
                    extent={{-180,-140},{180,140}}, preserveAspectRatio=false)));
          end test_OTSG_4_optB;

          model test_OTSG_4_optB_noHPH2 "Test model of the OTSG at 4% of load"

            OTSG_init4_5nodes_noDP OTSG(
              separator(
                zl_start=0,
                phase(fixed=true, start=1),
                h_start=3e6,
                initOpt=ThermoPower.Choices.Init.Options.fixedState,
                p_start=8510000,
                T(start=643.15)),
              SH(nominalColdPressure=8510000),
              EVA2(nominalColdPressure=8500000),
              EVA1(nominalColdPressure=8500000),
              ECO(nominalColdPressure=8500000),
              Nhex_mono=30,
              Nhex_bi=60)
              annotation (Placement(transformation(extent={{-44,-36},{12,56}})));
            ThermoPower.Water.SourceMassFlow sourceFW(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=true,
              w0=3.54,
              h=1134280)
              annotation (Placement(transformation(extent={{120,-2},{100,-22}})));
            ThermoPower.Water.SourceMassFlow sourceMS(
              allowFlowReversal=false,
              G=0,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              h=518322,
              use_in_w0=false,
              w0=24) annotation (Placement(transformation(extent={{-116,102},{-96,
                      82}})));
            ThermoPower.Water.SinkPressure sinkMS(
              allowFlowReversal=false,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              p0=261078) annotation (Placement(transformation(extent={{40,-70},
                      {60,-50}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
              annotation (Placement(transformation(extent={{-140,120},{-120,140}})));
            Modelica.Icons.SignalBus signalBus
              annotation (Placement(transformation(extent={{104,-68},{124,-48}})));
            ThermoPower.Water.SourceMassFlow sourceRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=false,
              w0=4.18,
              h=3092400)
              annotation (Placement(transformation(extent={{48,34},{28,54}})));
            ThermoPower.Water.SinkPressure sinkRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=2200000)
              annotation (Placement(transformation(extent={{22,114},{42,134}})));
            ThermoPower.Water.SinkPressure sinkSH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=8500000)
              annotation (Placement(transformation(extent={{62,94},{82,114}})));
            IndustrialControlSystems.Controllers.PI pI(
              AntiWindup=true,
              CSmin=0,
              CSmax=620,
              Ti=650,
              CS_start=20,
              Kp=5)   annotation (Placement(transformation(
                  extent={{10,10},{-10,-10}},
                  rotation=0,
                  origin={-90,-110})));
            Modelica.Blocks.Continuous.FirstOrder firstOrder(
              k=1,
              T=5,
              initType=Modelica.Blocks.Types.Init.SteadyState) annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-110,10})));
            Modelica.Blocks.Sources.Constant const(k=275 + 273.15) annotation (
                Placement(transformation(extent={{-40,-140},{-60,-120}})));
            ThermoPower.Water.SensT sensT(redeclare package Medium =
                  PreFlexMS.Media.MoltenSalt60_40_quad) annotation (Placement(
                  transformation(extent={{-12,-54},{8,-74}})));
            Control.plant_controller controllers(
              Kp_valveRH=1.77,
              Td_valveRH=0,
              Nd_valveRH=1,
              CSmin_valveRH=1,
              CSmax_valveRH=300,
              actuatorGain_valveRH=1,
              Ta_valveRH=2,
              Kp_pumpSEP=-14.5,
              Ti_pumpSEP=56.7,
              Td_pumpSEP=0,
              Nd_pumpSEP=1,
              CSmin_pumpSEP=0,
              CSmax_pumpSEP=70,
              actuatorGain_pumpSEP=1,
              Ta_pumpSEP=5,
              Td_pumpFW=0,
              Nd_pumpFW=1,
              Ta_pumpFW=5,
              Kp_valveBP=1,
              Ti_valveBP=1,
              Td_valveBP=0,
              Nd_valveBP=1,
              CSmin_valveBP=0,
              CSmax_valveBP=200,
              CSstart_valveBP=153.6,
              actuatorGain_valveBP=1,
              Ta_valveBP=2,
              useTS_valveBP=true,
              TS_valveBP=153.6,
              Kp_pumpHPH=1,
              Ti_pumpHPH=1,
              Td_pumpHPH=0,
              Nd_pumpHPH=1,
              CSmin_pumpHPH=0,
              CSmax_pumpHPH=200,
              CSstart_pumpHPH=143.1,
              actuatorGain_pumpHPH=1,
              Ta_pumpHPH=5,
              useTS_pumpHPH=true,
              TS_pumpHPH=143.1,
              actuatorGain_pumpFW=1,
              CSmin_pumpFW=1,
              CSmax_pumpFW=120,
              Kp_pumpFW=0.003,
              Ti_pumpFW=800,
              CSstart_valveRH=6.08,
              TS_valveRH=6.08,
              CSstart_pumpFW=3.54,
              TS_pumpFW=3.54,
              l_sp=0,
              CSstart_pumpSEP=0,
              p_SH_sp=3.54,
              f_load(k=0),
              useTS_pumpFW=true,
              useTS_pumpSEP=true,
              TS_pumpSEP=0,
              load_start=4,
              load_end=4,
              t_ramp=1,
              useTS_valveRH=true,
              Ti_valveRH=16,
              T_RH_sp=826.05,
              T_MS_sp=563.15,
              T_FW_sp=553.15)
              annotation (Placement(transformation(extent={{-80,-90},{-60,-70}})));
          equation
            connect(sourceMS.flange, OTSG.MS_in) annotation (Line(
                points={{-96,92},{-31.5556,92},{-31.5556,57.5333}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(signalBus.pumpFW_CS, sourceFW.in_w0) annotation (Line(
                points={{114,-58},{114,-48},{114,-38},{114,-18}},
                color={255,204,51},
                thickness=0.5), Text(
                string="%first",
                index=-1,
                extent={{-6,3},{-6,3}}));
            connect(OTSG.SH_out,sinkSH. flange) annotation (Line(
                points={{-0.444444,57.5333},{-0.444444,104},{62,104}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sinkRH.flange, OTSG.RH_out) annotation (Line(points={{22,124},
                    {4,124},{-12.8889,124},{-12.8889,57.5333}},
                                                      color={0,0,255}));
            connect(pI.CS, firstOrder.u) annotation (Line(points={{-99,-110},{-110,
                    -110},{-110,-2}}, color={0,0,127}));
            connect(const.y, pI.SP) annotation (Line(points={{-61,-130},{-70,-130},
                    {-70,-116},{-82,-116}}, color={0,0,127}));
            connect(sensT.T, pI.PV) annotation (Line(points={{6,-70},{6,-70},{10,
                    -70},{12,-70},{12,-110},{-82,-110},{-82,-109}},       color={0,
                    0,127}));
            connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                points={{-59.75,-82.6286},{-52,-82.6286},{-52,10},{-46.1778,10}},
                color={255,204,51},
                thickness=0.5));
            connect(controllers.signalBus, signalBus) annotation (Line(
                points={{-59.75,-82.6286},{114,-82.6286},{114,-58}},
                color={255,204,51},
                thickness=0.5), Text(
                string="%second",
                index=1,
                extent={{6,3},{6,3}}));
            connect(OTSG.MS_out, sensT.inlet) annotation (Line(points={{-31.5556,
                    -37.5333},{-31.5556,-60},{-8,-60}}, color={0,0,255}));
            connect(sensT.outlet, sinkMS.flange) annotation (Line(points={{4,-60},{
                    30,-60},{40,-60}}, color={0,0,255}));
            connect(OTSG.RH_in, sourceRH.flange) annotation (Line(points={{13.5556,
                    43.7333},{20.7778,43.7333},{20.7778,44},{28,44}}, color={0,0,
                    255}));
            connect(sourceFW.flange, OTSG.FW_in) annotation (Line(points={{100,-12},
                    {62,-12},{62,-25.2667},{13.5556,-25.2667}}, color={0,0,255}));
            connect(OTSG.SEP_out, OTSG.FW_in) annotation (Line(points={{13.5556,
                    -2.26667},{62,-2.26667},{62,-25.2667},{13.5556,-25.2667}},
                  color={0,0,255}));
            connect(sourceMS.in_w0, firstOrder.y)
              annotation (Line(points={{-110,86},{-110,21}}, color={0,0,127}));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-140,
                      -140},{140,140}}), graphics={Text(
                    extent={{20,-100},{140,-140}},
                    lineColor={28,108,200},
                    textString="Simulare per 60000 s a 500 punti
per raggiungere regime")}),                                Icon(coordinateSystem(
                    extent={{-140,-140},{140,140}}, preserveAspectRatio=false)));
          end test_OTSG_4_optB_noHPH2;

          model test_OTSG_4des "Test model of the OTSG at 4% of load"
            import PreFlexMS;

            OTSG_full_2ph_busControl_init20_5nodes OTSG(
              separator(
                initOpt=ThermoPower.Choices.Init.Options.fixedState,
                h_start=1697500,
                p_start=8540000),
              PDSteamEVA1(pin(start=8620000)),
              PDSteamECO(pin(start=8660000)),
              PDSteamEVA2(pin(start=8580000)),
              PDSteamRH(pin(start=22.05e5)))
              annotation (Placement(transformation(extent={{-164,-80},{-108,20}})));
            ThermoPower.Water.SourceMassFlow sourceFW(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=true,
              w0=17.01,
              h=515800)
              annotation (Placement(transformation(extent={{8,-42},{-12,-62}})));
            ThermoPower.Water.SourceMassFlow sourceMS(
              allowFlowReversal=false,
              G=0,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              h=518322,
              w0=34.424,
              use_in_w0=true) annotation (Placement(transformation(extent={{-236,
                      60},{-216,40}})));
            ThermoPower.Water.SinkPressure sinkMS(
              allowFlowReversal=false,
              redeclare package Medium = PreFlexMS.Media.MoltenSalt60_40_quad,
              p0=261078) annotation (Placement(transformation(extent={{-140,-124},
                      {-120,-104}})));
            inner ThermoPower.System system(initOpt=ThermoPower.Choices.Init.Options.noInit)
              annotation (Placement(transformation(extent={{-260,160},{-240,180}})));
            Modelica.Icons.SignalBus signalBus
              annotation (Placement(transformation(extent={{-8,-108},{12,-88}})));
            IndustrialControlSystems.Controllers.PI pI(
              AntiWindup=true,
              CSmax=620,
              CSmin=20,
              Kp=0.6,
              Ti=650,
              CS_start=55)
                      annotation (Placement(transformation(
                  extent={{10,10},{-10,-10}},
                  rotation=0,
                  origin={-210,-134})));
            Modelica.Blocks.Continuous.FirstOrder firstOrder(
              k=1,
              T=5,
              initType=Modelica.Blocks.Types.Init.SteadyState) annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={-230,-30})));
            Modelica.Blocks.Sources.Constant const(k=260 + 273.15) annotation (
                Placement(transformation(extent={{-160,-180},{-180,-160}})));
            ThermoPower.Water.SensT sensT
              annotation (Placement(transformation(extent={{-78,-74},{-98,-94}})));
            PreFlexMS.Auxiliaries.desuperheatingStation_4
              desuperheatingStation_4(mixer_RH(noInitialPressure=false),
                mixer_SH(noInitialPressure=false))
              annotation (Placement(transformation(extent={{82,38},{136,86}})));
            Control.Schemes.desuperheating_controller desuperheating_controller(
              Kp_des_SH=-4.018e-3,
              Ti_des_SH=6.48,
              Td_des_SH=0,
              Nd_des_SH=1,
              CSmin_des_SH=0,
              CSmax_des_SH=3,
              CSstart_des_SH=0.6363,
              actuatorGain_des_SH=1,
              Ta_des_SH=2,
              Kp_des_RH=-2.67e-4,
              Ti_des_RH=6.22,
              Td_des_RH=0,
              Nd_des_RH=1,
              CSmin_des_RH=0,
              CSmax_des_RH=3,
              CSstart_des_RH=0.0524,
              actuatorGain_des_RH=1,
              Ta_des_RH=2,
              Td_p_SH=0,
              Nd_p_SH=1,
              CSmin_p_SH=0,
              CSstart_p_SH=0.6,
              actuatorGain_p_SH=1,
              Ta_p_SH=2,
              Td_p_RH=0,
              Nd_p_RH=1,
              CSmin_p_RH=0,
              CSstart_p_RH=0.6,
              actuatorGain_p_RH=1,
              Ta_p_RH=2,
              Kp_p_SH=-1.905e-7,
              Kp_p_RH=-1e6,
              Ti_p_SH=2,
              Ti_p_RH=2,
              CSmax_p_SH=1,
              CSmax_p_RH=1,
              useTS_des_SH=true,
              useTS_des_RH=true,
              useTS_p_SH=false,
              T_des_SH_sp=605.05,
              T_des_RH_sp=673.15,
              p_SH_sp=8500000,
              p_RH_sp=2200000,
              useTS_p_RH=true)
              annotation (Placement(transformation(extent={{60,100},{100,140}})));
            ThermoPower.Water.SourcePressure sourcePressure(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=515800,
              p0=8500000)
              annotation (Placement(transformation(extent={{176,52},{156,72}})));
            Control.OTSG_controller_partLoad
                                    controllers(
              zl_sp=4,
              Kp_l=-14.5,
              Ti_l=56.7,
              Td_l=0,
              Nd_l=1,
              CSmin_l=0,
              actuatorGain_l=1,
              Ti_T=160,
              Td_T=0,
              Nd_T=1,
              CSmin_T=1,
              CSmax_T=300,
              actuatorGain_T=1,
              Kp_T=1.77,
              Ta_l=5,
              Ta_T=2,
              Td_w=0,
              Nd_w=1,
              CSmin_w=0,
              CSmax_w=60,
              Ta_w=5,
              Kp_w=0.003,
              w_IN_sp=35.4,
              CSstart_w=3.54,
              Ti_w=800,
              CSstart_T=6.08,
              const(k=10.45),
              CSstart_l=10.45,
              CSmax_l=90,
              useTS_T=true,
              useTS_l=false,
              T_RH_sp=826.05,
              useTS_w=true)
              annotation (Placement(transformation(extent={{-200,-114},{-180,-82}})));
            ThermoPower.Water.SourceMassFlow sourceRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_w0=false,
              w0=4.18,
              h=3092400)
              annotation (Placement(transformation(extent={{140,-120},{120,-100}})));
            ThermoPower.Water.SinkPressure sinkRH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=2190000)
              annotation (Placement(transformation(extent={{-20,78},{0,98}})));
            ThermoPower.Water.SinkPressure sinkSH(
              allowFlowReversal=false,
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              use_in_p0=false,
              p0=8490000)
              annotation (Placement(transformation(extent={{26,-30},{46,-10}})));
            ThermoPower.Water.ValveLin valveLin2(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=4.18e-4)
              annotation (Placement(transformation(extent={{100,-30},{80,-50}})));
            ThermoPower.Water.ValveLin valveLin1(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=4.18e-4)
              annotation (Placement(transformation(extent={{100,-120},{80,-100}})));
            Modelica.Blocks.Sources.Ramp ramp(
              startTime=10000,
              height=0.99999,
              offset=0,
              duration=100)
              annotation (Placement(transformation(extent={{220,-70},{200,-50}})));
            Modelica.Blocks.Math.Add add(k1=-1)
              annotation (Placement(transformation(extent={{180,-90},{160,-70}})));
            Modelica.Blocks.Sources.Constant const1(k=1)
              annotation (Placement(transformation(extent={{220,-100},{200,-80}})));
            ThermoPower.Water.ValveLin valveLin3(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=3.54e-4)
              annotation (Placement(transformation(extent={{20,60},{40,40}})));
            Modelica.Blocks.Sources.Ramp ramp1(
              startTime=10000,
              height=0.99999,
              offset=0,
              duration=100)
              annotation (Placement(transformation(extent={{-56,20},{-36,40}})));
            Modelica.Blocks.Math.Add add1(k1=-1)
              annotation (Placement(transformation(extent={{-24,4},{-4,24}})));
            Modelica.Blocks.Sources.Constant const2(k=1)
              annotation (Placement(transformation(extent={{-56,-12},{-36,8}})));
            ThermoPower.Water.ValveLin valveLin4(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=3.54e-4)
              annotation (Placement(transformation(extent={{-6,-30},{14,-10}})));
            ThermoPower.Water.ValveLin valveLin5(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=4.18e-4)
              annotation (Placement(transformation(extent={{-50,158},{-30,138}})));
            Modelica.Blocks.Sources.Ramp ramp2(
              startTime=10000,
              height=0.99999,
              offset=0,
              duration=100) annotation (Placement(transformation(extent={{-120,120},
                      {-100,140}})));
            Modelica.Blocks.Math.Add add2(k1=-1)
              annotation (Placement(transformation(extent={{-80,100},{-60,120}})));
            Modelica.Blocks.Sources.Constant const3(k=1)
              annotation (Placement(transformation(extent={{-120,94},{-100,114}})));
            ThermoPower.Water.ValveLin valveLin6(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=4.18e-4)
              annotation (Placement(transformation(extent={{-50,78},{-30,98}})));
            ThermoPower.Water.SourcePressure sourceSH_p(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=3583500,
              p0=8510000)
              annotation (Placement(transformation(extent={{48,-30},{68,-10}})));
            ThermoPower.Water.SourcePressure sourceRH_p(
              redeclare package Medium = Modelica.Media.Water.WaterIF97_ph,
              h=3553400,
              p0=2210000)
              annotation (Placement(transformation(extent={{-40,56},{-20,76}})));
            ThermoPower.Water.SinkPressure sinkPressure(redeclare package
                Medium =
                  Modelica.Media.Water.WaterIF97_ph, p0=2190000)
              annotation (Placement(transformation(extent={{180,-30},{200,-10}})));
            ThermoPower.Water.ValveLin valveLin(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=4.18e-4)
              annotation (Placement(transformation(extent={{0,56},{20,76}})));
            ThermoPower.Water.ValveLin valveLin7(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=4.18e-4)
              annotation (Placement(transformation(extent={{140,-10},{160,-30}})));
            ThermoPower.Water.ValveLin valveLin8(redeclare package Medium =
                  Modelica.Media.Water.WaterIF97_ph, Kv=3.54e-4) annotation (
                Placement(transformation(
                  extent={{-10,-10},{10,10}},
                  rotation=90,
                  origin={72,14})));
          equation
            connect(sourceMS.flange, OTSG.MS_in) annotation (Line(
                points={{-216,50},{-151.556,50},{-151.556,21.6667}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(OTSG.SEP_out, sourceFW.flange) annotation (Line(
                points={{-106.444,-43.3333},{-80,-43.3333},{-80,-52},{-12,-52}},
                color={0,0,255},
                smooth=Smooth.None));
            connect(sourceMS.in_w0, firstOrder.y)
              annotation (Line(points={{-230,44},{-230,-19}},color={0,0,127}));
            connect(pI.CS, firstOrder.u) annotation (Line(points={{-219,-134},{-230,
                    -134},{-230,-42}},color={0,0,127}));
            connect(const.y, pI.SP) annotation (Line(points={{-181,-170},{-190,-170},
                    {-190,-140},{-202,-140}},
                                            color={0,0,127}));
            connect(OTSG.MS_out, sinkMS.flange) annotation (Line(points={{-151.556,
                    -81.6667},{-151.556,-114},{-140,-114}},color={0,0,255}));
            connect(OTSG.FW_in, sensT.outlet) annotation (Line(points={{-106.444,
                    -68.3333},{-100,-68.3333},{-100,-80},{-94,-80}},
                                                                color={0,0,255}));
            connect(sourceFW.flange, sensT.inlet) annotation (Line(points={{-12,-52},
                    {-12,-52},{-76,-52},{-80,-52},{-80,-80},{-82,-80}},
                                                                    color={0,0,255}));
            connect(sensT.T, pI.PV) annotation (Line(points={{-96,-90},{-100,-90},{
                    -100,-130},{-100,-134},{-202,-134},{-202,-133}},      color={0,
                    0,127}));
            connect(desuperheating_controller.signalBus, desuperheatingStation_4.signalBus)
              annotation (Line(
                points={{102,120},{109,120},{109,88}},
                color={255,204,51},
                thickness=0.5));
            connect(desuperheatingStation_4.FW_in, sourcePressure.flange)
              annotation (Line(points={{136,62},{146,62},{156,62}}, color={0,0,255}));
            connect(controllers.signalBus, signalBus) annotation (Line(
                points={{-179,-98},{-88,-98},{2,-98}},
                color={255,204,51},
                thickness=0.5));
            connect(controllers.signalBus, OTSG.signalBus) annotation (Line(
                points={{-179,-98},{-172,-98},{-172,-30},{-166.178,-30}},
                color={255,204,51},
                thickness=0.5));
            connect(signalBus.pumpFW_CS, sourceFW.in_w0) annotation (Line(
                points={{2,-98},{2,-98},{2,-58}},
                color={255,204,51},
                thickness=0.5), Text(
                string="%first",
                index=-1,
                extent={{-6,3},{-6,3}}));
            connect(valveLin1.inlet, sourceRH.flange)
              annotation (Line(points={{100,-110},{120,-110}}, color={0,0,255}));
            connect(ramp.y, add.u1) annotation (Line(points={{199,-60},{190,-60},{
                    190,-74},{182,-74}}, color={0,0,127}));
            connect(const1.y, add.u2) annotation (Line(points={{199,-90},{190,-90},
                    {190,-86},{182,-86}}, color={0,0,127}));
            connect(add.y, valveLin1.cmd) annotation (Line(points={{159,-80},{90,
                    -80},{90,-102}}, color={0,0,127}));
            connect(ramp.y, valveLin2.cmd) annotation (Line(points={{199,-60},{199,
                    -60},{90,-60},{90,-48}}, color={0,0,127}));
            connect(desuperheatingStation_4.SH_out, valveLin2.inlet) annotation (
                Line(points={{120.571,38},{120.571,-40},{100,-40}}, color={0,0,255}));
            connect(valveLin1.outlet, OTSG.RH_in) annotation (Line(points={{80,-110},
                    {60,-110},{60,-40},{42,-40},{-74,-40},{-74,6.66667},{-106.444,
                    6.66667}}, color={0,0,255}));
            connect(valveLin2.outlet, OTSG.RH_in) annotation (Line(points={{80,-40},
                    {-74,-40},{-74,6.66667},{-106.444,6.66667}}, color={0,0,255}));
            connect(ramp1.y, add1.u1) annotation (Line(points={{-35,30},{-32,30},{
                    -32,20},{-26,20}}, color={0,0,127}));
            connect(valveLin4.outlet, sinkSH.flange) annotation (Line(points={{14,
                    -20},{14,-20},{26,-20}}, color={0,0,255}));
            connect(OTSG.SH_out, valveLin3.inlet) annotation (Line(points={{
                    -120.444,21.6667},{-120.444,50},{20,50}}, color={0,0,255}));
            connect(valveLin3.outlet, desuperheatingStation_4.SH_in)
              annotation (Line(points={{40,50},{40,50},{82,50}}, color={0,0,255}));
            connect(OTSG.SH_out, valveLin4.inlet) annotation (Line(points={{
                    -120.444,21.6667},{-120.444,50},{-66,50},{-66,-20},{-6,-20}},
                  color={0,0,255}));
            connect(ramp2.y, add2.u1) annotation (Line(points={{-99,130},{-89,130},
                    {-89,116},{-82,116}}, color={0,0,127}));
            connect(const3.y, add2.u2)
              annotation (Line(points={{-99,104},{-82,104}}, color={0,0,127}));
            connect(add2.y, valveLin6.cmd) annotation (Line(points={{-59,110},{-40,
                    110},{-40,96}}, color={0,0,127}));
            connect(ramp2.y, valveLin5.cmd) annotation (Line(points={{-99,130},{-99,
                    130},{-40,130},{-40,140}}, color={0,0,127}));
            connect(valveLin6.outlet, sinkRH.flange) annotation (Line(points={{-30,
                    88},{-30,88},{-20,88}}, color={0,0,255}));
            connect(OTSG.RH_out, valveLin5.inlet) annotation (Line(points={{
                    -132.889,21.6667},{-132.889,148},{-50,148}}, color={0,0,255}));
            connect(OTSG.RH_out, valveLin6.inlet) annotation (Line(points={{
                    -132.889,21.6667},{-132.889,88},{-50,88}}, color={0,0,255}));
            connect(valveLin5.outlet, desuperheatingStation_4.RH_in) annotation (
                Line(points={{-30,148},{-30,148},{40,148},{40,74},{82,74}}, color={
                    0,0,255}));
            connect(sourceSH_p.flange, valveLin8.inlet) annotation (Line(points={{
                    68,-20},{68,-20},{72,-20},{72,-10},{72,4}}, color={0,0,255}));
            connect(valveLin7.outlet, sinkPressure.flange)
              annotation (Line(points={{160,-20},{180,-20}}, color={0,0,255}));
            connect(desuperheatingStation_4.SH_out, valveLin7.inlet) annotation (
                Line(points={{120.571,38},{120,38},{120,-16},{120,-20},{140,-20}},
                  color={0,0,255}));
            connect(valveLin8.outlet, desuperheatingStation_4.SH_in)
              annotation (Line(points={{72,24},{72,50},{82,50}}, color={0,0,255}));
            connect(valveLin.outlet, desuperheatingStation_4.RH_in) annotation (
                Line(points={{20,66},{40,66},{40,74},{82,74}}, color={0,0,255}));
            connect(sourceRH_p.flange, valveLin.inlet) annotation (Line(points={{
                    -20,66},{-10,66},{0,66}}, color={0,0,255}));
            connect(add1.y, valveLin8.cmd)
              annotation (Line(points={{-3,14},{64,14}}, color={0,0,127}));
            connect(const2.y, add1.u2) annotation (Line(points={{-35,-2},{-32,-2},{
                    -32,8},{-26,8}}, color={0,0,127}));
            connect(ramp1.y, valveLin3.cmd) annotation (Line(points={{-35,30},{30,
                    30},{30,42}}, color={0,0,127}));
            connect(add1.y, valveLin4.cmd)
              annotation (Line(points={{-3,14},{4,14},{4,-12}}, color={0,0,127}));
            connect(add.y, valveLin7.cmd) annotation (Line(points={{159,-80},{150,
                    -80},{150,-28}}, color={0,0,127}));
            connect(add2.y, valveLin.cmd) annotation (Line(points={{-59,110},{10,
                    110},{10,74}}, color={0,0,127}));
            annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-260,
                      -180},{260,180}})),                  Icon(coordinateSystem(
                    extent={{-260,-180},{260,180}}, preserveAspectRatio=false)));
          end test_OTSG_4des;
        end Old;
      end OTSG;

      package PreHeaters "Package containing all the pre-heaters train model"
        model Preheater_fixedLevel
          "Model of a feed water pre heater with ideal level control"
          extends BaseClasses.BaseFeedwaterHeater;

          //Desuperheating region parameters
          replaceable model HeatTransferCold_des =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Desuperheating"));
          replaceable model HeatTransferHot_des =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, hot fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Desuperheating"));
          replaceable model HeatExchangerTopology_des =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true,Dialog(tab="Desuperheating"));
          parameter Integer Nhex_des = 2
            "Number of nodes for the desuperheating region" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Volume Vhot_des
            "Volume of the fluid in the hot side" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Volume Vcold_des
            "Volume of the fluid in the cold side" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Area Shot_des
            "Heat transfer surface in the hot side" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Area Scold_des
            "Heat transfer surface in the cold side" annotation (Dialog(tab="Desuperheating"));
          final parameter Modelica.SIunits.Length L_des=1
            "Lenght of the fluid path (unused)"                                               annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Mass Mm_des
            "Mass of the metal between the two fluids" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow_des
            "Nominal flow rate of cold fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.MassFlowRate nominalHotFlow_des
            "Nominal flow rate of hot fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalColdPressure_des
            "Nominal inlet pressure of cold fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalColdDeltaP_des
            "Nominal cold fluid pressure loss" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalHotPressure_des
            "Nominal inlet pressure of hot fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalHotDeltaP_des
            "Nominal hot fluid pressure loss" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalColdInletTemperature_des
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalColdOutletTemperature_des
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalHotInletTemperature_des
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalHotOutletTemperature_des
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Desuperheating"));

          //Condensing region parameters
          parameter Integer Nhex_cond = 5
            "Number of nodes for the condensing region"                               annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Volume Vhot_cond "Total volume" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Area A "Cross Sectional Area" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Height H=Vhot_cond/A
            "Height of the condensing volume"                                               annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Area Shot_cond "Heat transfer surface" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.CoefficientOfHeatTransfer gamma
            "Nusselt theory HTC" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Height zl
            "Liquid level (ideally controlled)"                                    annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Pressure p_start "Medium start pressure" annotation (Dialog(tab="Condensing", group="Hot side"));
          outer ThermoPower.System system "System wide properties";
          parameter Boolean allowFlowReversal=system.allowFlowReversal
            "= true to allow flow reversal, false restricts to design direction"
                                                                                 annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter ThermoPower.Choices.Init.Options initOpt=ThermoPower.Choices.Init.Options.noInit
            "Initialisation option" annotation (Dialog(tab="Condensing", group="Hot side"));

          replaceable model HeatTransferCold_cond =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Condensing", group="Cold side"));
          replaceable model HeatExchangerTopology_cond =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true,Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Volume Vcold_cond
            "Volume of the fluid in the cold side" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Area Scold_cond
            "Heat transfer surface in the cold side" annotation (Dialog(tab="Condensing", group="Cold side"));
          final parameter Modelica.SIunits.Length L_cond=1
            "Lenght of the fluid path (unused)"                                                annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Mass Mm_cond
            "Mass of the metal between the two fluids" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow_cond
            "Nominal flow rate of cold fluid" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Pressure nominalColdPressure_cond
            "Nominal inlet pressure of cold fluid" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Pressure nominalColdDeltaP_cond
            "Nominal cold fluid pressure loss" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Temperature nominalColdInletTemperature_cond
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Temperature nominalColdOutletTemperature_cond
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Condensing", group="Cold side"));

          //Subcooling region parameters
          replaceable model HeatTransferCold_sub =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Subcooling"));
          replaceable model HeatTransferHot_sub =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, hot fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Subcooling"));
          replaceable model HeatExchangerTopology_sub =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true,Dialog(tab="Subcooling"));
          parameter Integer Nhex_sub = 2
            "Number of nodes for the desuperheating region" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Volume Vhot_sub
            "Volume of the fluid in the hot side" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Volume Vcold_sub
            "Volume of the fluid in the cold side" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Area Shot_sub
            "Heat transfer surface in the hot side" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Area Scold_sub
            "Heat transfer surface in the cold side" annotation (Dialog(tab="Subcooling"));
          final parameter Modelica.SIunits.Length L_sub=1
            "Lenght of the fluid path (unused)"                                               annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Mass Mm_sub
            "Mass of the metal between the two fluids" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow_sub
            "Nominal flow rate of cold fluid" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.MassFlowRate nominalHotFlow_sub
            "Nominal flow rate of hot fluid" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Pressure nominalColdPressure_sub
            "Nominal inlet pressure of cold fluid" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Pressure nominalColdDeltaP_sub
            "Nominal cold fluid pressure loss" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Pressure nominalHotPressure_sub
            "Nominal inlet pressure of hot fluid" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Pressure nominalHotDeltaP_sub
            "Nominal hot fluid pressure loss" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Temperature nominalColdInletTemperature_sub
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Temperature nominalColdOutletTemperature_sub
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Temperature nominalHotInletTemperature_sub
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Temperature nominalHotOutletTemperature_sub
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Subcooling"));

          parameter Modelica.SIunits.SpecificHeatCapacity cm
            "Specific heat capacity of metal";

          //Controller parameters
          parameter Real Kv "Admittance of linear valve" annotation (Dialog(tab="Level controller"));
          parameter Real Kp "Controller proportional gain" annotation (Dialog(tab="Level controller"));
          parameter Modelica.SIunits.Time Ti "Controller integral time" annotation (Dialog(tab="Level controller"));
          parameter Modelica.SIunits.Time Td "Controller derivative time" annotation (Dialog(tab="Level controller"));
          parameter Real Nd "Controller derivative action up to Nd/Td rad/s" annotation (Dialog(tab="Level controller"));
          parameter Modelica.SIunits.Time Ta "Actuator time constant" annotation (Dialog(tab="Level controller"));
          //parameter Real PVmin "Minimum value of process variable for scaling" annotation (Dialog(tab="Level controller"));
          //parameter Real PVmax "Maximum value of process variable for scaling" annotation (Dialog(tab="Level controller"));
          parameter Real CSmin "Minimum value of control signal for scaling" annotation (Dialog(tab="Level controller"));
          parameter Real CSmax "Maximum value of control signal for scaling" annotation (Dialog(tab="Level controller"));
          //parameter Real PVstart=0.5 "Start value of PV (scaled)" annotation (Dialog(tab="Level controller"));
          parameter Real CSstart=0.6 "Start value of CS" annotation (Dialog(tab="Level controller"));

          ThermoPower.Water.Flow1DFV des_FW(
            final N=Nhex_des,
            H=0,
            wnom=nominalColdFlow_des,
            pstart=nominalColdPressure_des,
            redeclare package Medium = Medium,
            L=L_des,
            dpnom=nominalColdDeltaP_des,
            rhonom=Medium.density_pT(nominalColdPressure_des,
                nominalColdInletTemperature_des),
            hstartin=Medium.specificEnthalpy_pT(nominalColdPressure_des,
                nominalColdInletTemperature_des),
            hstartout=Medium.specificEnthalpy_pT(nominalColdPressure_des -
                nominalColdDeltaP_des, nominalColdOutletTemperature_des),
            final A=Vcold_des/L_des,
            final omega=Scold_des/L_des,
            Dhyd=1,
            redeclare model HeatTransfer = HeatTransferCold_des,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
            initOpt=ThermoPower.Choices.Init.Options.fixedState,
            noInitialPressure=true)                              annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={50,70})));
          ThermoPower.Water.Flow1DFV cond_FW(
            final N=Nhex_cond,
            H=0,
            wnom=nominalColdFlow_cond,
            pstart=nominalColdPressure_cond,
            redeclare package Medium = Medium,
            L=L_cond,
            dpnom=nominalColdDeltaP_cond,
            rhonom=Medium.density_pT(nominalColdPressure_cond,
                nominalColdInletTemperature_cond),
            hstartin=Medium.specificEnthalpy_pT(nominalColdPressure_cond,
                nominalColdInletTemperature_cond),
            hstartout=Medium.specificEnthalpy_pT(nominalColdPressure_cond -
                nominalColdDeltaP_cond, nominalColdOutletTemperature_cond),
            final A=Vcold_cond/L_cond,
            final omega=Scold_cond/L_cond,
            Dhyd=1,
            redeclare model HeatTransfer = HeatTransferCold_cond,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
            initOpt=ThermoPower.Choices.Init.Options.fixedState,
            noInitialPressure=true)                               annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={50,20})));
          ThermoPower.Water.Flow1DFV sub_FW(
            final N=Nhex_sub,
            H=0,
            wnom=nominalColdFlow_sub,
            pstart=nominalColdPressure_sub,
            redeclare package Medium = Medium,
            L=L_sub,
            dpnom=nominalColdDeltaP_sub,
            rhonom=Medium.density_pT(nominalColdPressure_sub,
                nominalColdInletTemperature_sub),
            hstartin=Medium.specificEnthalpy_pT(nominalColdPressure_sub,
                nominalColdInletTemperature_sub),
            hstartout=Medium.specificEnthalpy_pT(nominalColdPressure_sub -
                nominalColdDeltaP_sub, nominalColdOutletTemperature_sub),
            final A=Vcold_sub/L_sub,
            final omega=Scold_sub/L_sub,
            Dhyd=1,
            redeclare model HeatTransfer = HeatTransferCold_sub,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
            initOpt=ThermoPower.Choices.Init.Options.fixedState,
            noInitialPressure=true)                              annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={50,-30})));
          ThermoPower.Water.Flow1DFV2ph des_SP(
            N=Nhex_des,
            H=0,
            Dhyd=1,
            pstart=nominalHotPressure_des,
            redeclare package Medium = Medium,
            L=L_des,
            A=Vhot_des/L_des,
            omega=Shot_des/L_des,
            wnom=nominalHotFlow_des,
            dpnom=nominalHotDeltaP_des,
            rhonom=Medium.density_pT(nominalHotPressure_des,
                nominalHotInletTemperature_des),
            hstartin=Medium.specificEnthalpy_pT(nominalHotPressure_des,
                nominalHotInletTemperature_des),
            hstartout=Medium.specificEnthalpy_pT(nominalHotPressure_des -
                nominalHotDeltaP_des, nominalHotOutletTemperature_des),
            redeclare model HeatTransfer = HeatTransferHot_des,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
            initOpt=ThermoPower.Choices.Init.Options.fixedState,
            noInitialPressure=true)                             annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={-50,70})));
          ThermoPower.Water.Flow1DFV sub_SP(
            N=Nhex_sub,
            H=0,
            Dhyd=1,
            pstart=nominalHotPressure_sub,
            redeclare package Medium = Medium,
            L=L_sub,
            A=Vhot_sub/L_sub,
            omega=Shot_sub/L_sub,
            wnom=nominalHotFlow_sub,
            dpnom=nominalHotDeltaP_sub,
            rhonom=Medium.density_pT(nominalHotPressure_sub,
                nominalHotInletTemperature_sub),
            hstartin=Medium.specificEnthalpy_pT(nominalHotPressure_sub,
                nominalHotInletTemperature_sub),
            hstartout=Medium.specificEnthalpy_pT(nominalHotPressure_sub -
                nominalHotDeltaP_sub, nominalHotOutletTemperature_sub),
            redeclare model HeatTransfer = HeatTransferHot_sub,
            FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
            initOpt=ThermoPower.Choices.Init.Options.fixedState,
            noInitialPressure=true)                             annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=-90,
                origin={-50,-30})));
          Components.NusseltCondenser_noLevel
                                       NusseltCondenser(
            V=Vhot_cond,
            A=A,
            S=Shot_cond,
            p_start=p_start,
            gamma=gamma,
            Nw=Nhex_cond - 1,
            zl=zl,
            allowFlowReversal=false)
            annotation (Placement(transformation(extent={{-82,-12},{-18,52}})));
          ThermoPower.Water.ValveLin        valveDrain(
            redeclare package Medium = Medium,
            Kv = Kv,
            allowFlowReversal=false)
                     annotation (
             Placement(transformation(
                extent={{-10,10},{10,-10}},
                rotation=-90,
                origin={-50,-70})));

          ThermoPower.Thermal.MetalWallFV tubeWalls_des(
            final Nw=Nhex_des - 1,
            final M=Mm_des,
            final cm=cm,
            Tstart1=(nominalColdInletTemperature_des + nominalHotOutletTemperature_des)/2,
            TstartN=(nominalColdOutletTemperature_des + nominalHotInletTemperature_des)/2,
            initOpt=ThermoPower.Choices.Init.Options.fixedState)
            annotation (Placement(transformation(
                extent={{-12,12},{12,-12}},
                rotation=-90,
                origin={0,70})));
          ThermoPower.Thermal.HeatExchangerTopologyFV heatExchangerTopology_des(Nw=Nhex_des -
                1, redeclare model HeatExchangerTopology =
                HeatExchangerTopology_des)
            annotation (Placement(transformation(
                extent={{-12,-11},{12,11}},
                rotation=-90,
                origin={25,70})));
          ThermoPower.Thermal.MetalWallFV tubeWalls_sub(
            final Nw=Nhex_sub - 1,
            final M=Mm_sub,
            final cm=cm,
            Tstart1=(nominalColdInletTemperature_sub + nominalHotOutletTemperature_sub)/2,
            TstartN=(nominalColdOutletTemperature_sub + nominalHotInletTemperature_sub)/2,
            initOpt=ThermoPower.Choices.Init.Options.fixedState)
            annotation (Placement(transformation(
                extent={{-12,12},{12,-12}},
                rotation=-90,
                origin={0,-30})));
          ThermoPower.Thermal.HeatExchangerTopologyFV heatExchangerTopology_sub(Nw=Nhex_sub -
                1, redeclare model HeatExchangerTopology =
                HeatExchangerTopology_sub)
            annotation (Placement(transformation(
                extent={{-12,-11},{12,11}},
                rotation=-90,
                origin={25,-30})));
          ThermoPower.Thermal.MetalWallFV tubeWalls_cond(
            final Nw=Nhex_cond - 1,
            final M=Mm_cond,
            final cm=cm,
            TstartN=(nominalColdInletTemperature_cond + Medium.saturationTemperature(p_start))/2,
            Tstart1=(nominalColdOutletTemperature_cond + Medium.saturationTemperature(p_start))/2,
            initOpt=ThermoPower.Choices.Init.Options.noInit)
            annotation (Placement(transformation(
                extent={{-12,12},{12,-12}},
                rotation=-90,
                origin={0,20})));
          ThermoPower.Thermal.HeatExchangerTopologyFV heatExchangerTopology_cond(Nw=Nhex_cond
                 - 1, redeclare model HeatExchangerTopology =
                HeatExchangerTopology_cond)
            annotation (Placement(transformation(
                extent={{-12,-11},{12,11}},
                rotation=-90,
                origin={25,20})));

          Modelica.SIunits.Power Q_Hot "Power from the hot side";
          Modelica.SIunits.Power Q_Cold "Power to the cold side";
          Modelica.SIunits.Temperature T_hot_prof[Nhex_des+Nhex_cond+Nhex_sub+1] = vector([des_SP.T[1];des_SP.heatTransfer.Tvol;des_SP.T[end];ones(Nhex_cond-1)*NusseltCondenser.T_sat;sub_SP.T[1];sub_SP.heatTransfer.Tvol;sub_SP.T[end]])
            "Hot temperature vector (hot in the left side)";
          Modelica.SIunits.Temperature T_wall_prof[Nhex_des+Nhex_cond+Nhex_sub+1] = vector([(des_SP.T[1]+des_FW.T[end])/2;tubeWalls_des.Tvol[end:-1:1];(des_SP.T[end]+des_FW.T[1])/2;tubeWalls_cond.Tvol[end:-1:1];(sub_SP.T[1]+sub_FW.T[end])/2;tubeWalls_sub.Tvol[end:-1:1];(sub_SP.T[end]+sub_FW.T[1])/2])
            "Wall temperature vector (hot in the left side)";
          Modelica.SIunits.Temperature T_cold_prof[Nhex_des+Nhex_cond+Nhex_sub+1] = vector([des_FW.T[end];des_FW.heatTransfer.Tvol[end:-1:1];des_FW.T[1];cond_FW.heatTransfer.Tvol[end:-1:1];sub_FW.T[end];sub_FW.heatTransfer.Tvol[end:-1:1];sub_FW.T[1]])
            "Cold temperature vector (hot in the left side)";

          Modelica.Blocks.Sources.Constant const(k=1)
            annotation (Placement(transformation(extent={{-90,-80},{-70,-60}})));
        equation
          Q_Hot = -des_SP.Q + NusseltCondenser.Q_cond - sub_SP.Q;
          Q_Cold = des_FW.Q + cond_FW.Q + sub_FW.Q;
          connect(des_SP.outfl, NusseltCondenser.steam_in) annotation (Line(
              points={{-50,60},{-50,47.4286}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(sub_SP.infl, NusseltCondenser.drained_out) annotation (Line(
              points={{-50,-20},{-50,-0.571429}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(des_FW.infl, cond_FW.outfl) annotation (Line(
              points={{50,60},{50,30}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(cond_FW.infl, sub_FW.outfl) annotation (Line(
              points={{50,10},{50,-20}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(sub_SP.outfl, valveDrain.inlet) annotation (Line(
              points={{-50,-40},{-50,-60}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(ST_in, des_SP.infl) annotation (Line(
              points={{-50,100},{-50,80}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(FW_out, des_FW.outfl) annotation (Line(
              points={{50,100},{50,80}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(DR_in, NusseltCondenser.drained_in) annotation (Line(
              points={{-100,0},{-100,29.1429},{-61.4286,29.1429}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(sub_FW.infl, FW_in) annotation (Line(
              points={{50,-40},{50,-100}},
              color={0,0,255},
              smooth=Smooth.None));
          connect(valveDrain.outlet, DR_out) annotation (Line(
              points={{-50,-80},{-50,-100}},
              color={0,0,255},
              smooth=Smooth.None));

          connect(des_SP.wall, tubeWalls_des.int) annotation (Line(
              points={{-45,70},{-3.6,70}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(tubeWalls_des.ext, heatExchangerTopology_des.side2) annotation (Line(
              points={{3.72,70},{21.59,70}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(heatExchangerTopology_des.side1, des_FW.wall) annotation (Line(
              points={{28.3,70},{45,70}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(heatExchangerTopology_sub.side1, sub_FW.wall) annotation (Line(
              points={{28.3,-30},{45,-30}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(sub_SP.wall, tubeWalls_sub.int) annotation (Line(
              points={{-45,-30},{-3.6,-30}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(tubeWalls_sub.ext, heatExchangerTopology_sub.side2) annotation (Line(
              points={{3.72,-30},{21.59,-30}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(tubeWalls_cond.ext, heatExchangerTopology_cond.side2) annotation (
              Line(
              points={{3.72,20},{21.59,20}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(cond_FW.wall, heatExchangerTopology_cond.side1) annotation (Line(
              points={{45,20},{28.3,20}},
              color={255,127,0},
              smooth=Smooth.None));
          connect(NusseltCondenser.DHT, tubeWalls_cond.int) annotation (Line(points={{
                  -39.7143,20},{-39.7143,20},{-3.6,20}},
                                                color={255,127,0}));
          connect(valveDrain.cmd, const.y)
            annotation (Line(points={{-58,-70},{-64,-70},{-69,-70}}, color={0,0,127}));
          annotation (Diagram(coordinateSystem(extent={{-100,-100},{100,100}},
                  preserveAspectRatio=false)),           Icon(coordinateSystem(extent={{
                    -100,-100},{100,100}})));
        end Preheater_fixedLevel;

        model Preheater_controlledLevel
          "Model of a feed water pre heater with level control"
          extends BaseClasses.BaseFeedwaterHeater;

          //Desuperheating region parameters
          replaceable model HeatTransferCold_des =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Desuperheating"));
          replaceable model HeatTransferHot_des =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, hot fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Desuperheating"));
          replaceable model HeatExchangerTopology_des =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true,Dialog(tab="Desuperheating"));
          parameter Integer Nhex_des = 2
            "Number of nodes for the desuperheating region" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Volume Vhot_des
            "Volume of the fluid in the hot side" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Volume Vcold_des
            "Volume of the fluid in the cold side" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Area Shot_des
            "Heat transfer surface in the hot side" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Area Scold_des
            "Heat transfer surface in the cold side" annotation (Dialog(tab="Desuperheating"));
          final parameter Modelica.SIunits.Length L_des=1
            "Lenght of the fluid path (unused)"                                               annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Mass Mm_des
            "Mass of the metal between the two fluids" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow_des
            "Nominal flow rate of cold fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.MassFlowRate nominalHotFlow_des
            "Nominal flow rate of hot fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalColdPressure_des
            "Nominal inlet pressure of cold fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalColdDeltaP_des
            "Nominal cold fluid pressure loss" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalHotPressure_des
            "Nominal inlet pressure of hot fluid" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Pressure nominalHotDeltaP_des
            "Nominal hot fluid pressure loss" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalColdInletTemperature_des
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalColdOutletTemperature_des
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalHotInletTemperature_des
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Temperature nominalHotOutletTemperature_des
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Desuperheating"));

          //Condensing region parameters
          parameter Integer Nhex_cond = 5
            "Number of nodes for the condensing region"                               annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Volume Vhot_cond "Total volume" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Area A "Cross Sectional Area" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Height H=Vhot_cond/A
            "Height of the condensing volume"                                               annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Area Shot_cond "Heat transfer surface" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.CoefficientOfHeatTransfer gamma
            "Nusselt theory HTC" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Height zl_set "Liquid level set point" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Height zl_start "Liquid level start" annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter Modelica.SIunits.Pressure p_start "Medium start pressure" annotation (Dialog(tab="Condensing", group="Hot side"));
          outer ThermoPower.System system "System wide properties";
          parameter Boolean allowFlowReversal=system.allowFlowReversal
            "= true to allow flow reversal, false restricts to design direction"
                                                                                 annotation (Dialog(tab="Condensing", group="Hot side"));
          parameter ThermoPower.Choices.Init.Options initOpt=ThermoPower.Choices.Init.Options.noInit
            "Initialisation option" annotation (Dialog(tab="Condensing", group="Hot side"));

          replaceable model HeatTransferCold_cond =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Condensing", group="Cold side"));
          replaceable model HeatExchangerTopology_cond =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true,Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Volume Vcold_cond
            "Volume of the fluid in the cold side" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Area Scold_cond
            "Heat transfer surface in the cold side" annotation (Dialog(tab="Condensing", group="Cold side"));
          final parameter Modelica.SIunits.Length L_cond=1
            "Lenght of the fluid path (unused)"                                                annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Mass Mm_cond
            "Mass of the metal between the two fluids" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow_cond
            "Nominal flow rate of cold fluid" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Pressure nominalColdPressure_cond
            "Nominal inlet pressure of cold fluid" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Pressure nominalColdDeltaP_cond
            "Nominal cold fluid pressure loss" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Temperature nominalColdInletTemperature_cond
            "Nominal value of cold fluid temperature at the inlet" annotation (Dialog(tab="Condensing", group="Cold side"));
          parameter Modelica.SIunits.Temperature nominalColdOutletTemperature_cond
            "Nominal value of cold fluid temperature at the outlet" annotation (Dialog(tab="Condensing", group="Cold side"));

          //Subcooling region parameters
          replaceable model HeatTransferCold_sub =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, cold fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Subcooling"));
          replaceable model HeatTransferHot_sub =
            ThermoPower.Thermal.BaseClasses.DistributedHeatTransferFV
            "Heat transfer model, hot fluid side"
            annotation(choicesAllMatching = true,Dialog(tab="Subcooling"));
          replaceable model HeatExchangerTopology_sub =
              ThermoPower.Thermal.HeatExchangerTopologies.CounterCurrentFlow
            constrainedby
            ThermoPower.Thermal.BaseClasses.HeatExchangerTopologyData
            "Heat exchanger topology"
            annotation(choicesAllMatching = true,Dialog(tab="Subcooling"));
          parameter Integer Nhex_sub = 2
            "Number of nodes for the desuperheating region" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Volume Vhot_sub
            "Volume of the fluid in the hot side" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Volume Vcold_sub
            "Volume of the fluid in the cold side" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Area Shot_sub
            "Heat transfer surface in the hot side" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.Area Scold_sub
            "Heat transfer surface in the cold side" annotation (Dialog(tab="Subcooling"));
          final parameter Modelica.SIunits.Length L_sub=1
            "Lenght of the fluid path (unused)"                                               annotation (Dialog(tab="Desuperheating"));
          parameter Modelica.SIunits.Mass Mm_sub
            "Mass of the metal between the two fluids" annotation (Dialog(tab="Subcooling"));
          parameter Modelica.SIunits.MassFlowRate nominalColdFlow_sub
            "Nominal flow rate of cold fluid" annotation (Dialog(tab="Subcooling"));
