within IndustrialControlSystems.Controllers;
model PI_steadyState "Proportional + Integral controller"
  extends Interfaces.Controller;
  parameter Real Kp = 5 "|Parameters| Proportional gain" annotation(Evaluate = true);
  parameter Real Ti = 1 "|Parameters| Integral time" annotation(Evaluate = true);
  parameter Boolean steadyStateInit=false
    "|Initialisation| Initialize in steady state"                                       annotation(Evaluate = true);
  Real FBout "output (and state) of the feedback block 1/(1+sTi)";
  Real cs "Pre-biased Control Signal";
  Real satin "input of the saturation block";
initial equation

  if Ts > 0 then
    // Discrete time initialisation
    pre(FBout) = if AntiWindup then max(CSmin,min(CSmax,CS_start)) else CS_start - bias - Kp*(SP - PV);
  else
    // Continuous time initialisation
    if steadyStateInit then
      if Ti > 0 then
        der(FBout) = 0;
      else
        FBout = if AntiWindup then max(CSmin,min(CSmax,CS_start)) else CS_start - bias - Kp*(SP - PV);
      end if;
    end if;
  end if;

equation

  if Ts > 0 then
    // Discrete time controller
    when sample(0,Ts) then
      satin =  pre(FBout) + Kp*(SP-PV);
      cs    =  if ts then tr else (if AntiWindup then max(CSmin,min(CSmax,satin)) else satin);
      FBout = IndustrialControlSystems.LinearSystems.Discrete.Functions.f1Pole(
        alfa,
        cs,
        pre(cs),
        pre(FBout),
        Ts,
        1,
        if not ts then Ti else eps);
      CS    =  if AntiWindup then max(CSmin,min(CSmax,cs + (if ts then 0 else pre(bias)))) else cs + (if ts then 0 else pre(bias));
    end when;
  else
    // Continuous time controller
    satin = FBout + Kp*(SP-PV);
    cs    = if ts then tr else (if AntiWindup then max(CSmin,min(CSmax,satin)) else satin);
    FBout = cs + (if not ts then -Ti*der(FBout) else -eps*der(FBout));
    CS    = if AntiWindup then max(CSmin,min(CSmax,cs + (if ts then 0 else bias))) else cs + (if ts then 0 else bias);
  end if;

initial equation

  annotation (Icon(graphics={Text(
          extent={{-34,30},{46,-40}},
          lineColor={0,0,255},
          textString="PI")}),Documentation(info="<html>
<p><b>Description</b> </p>
<p>Proportional + Integral controller with Automatic, Tracking mode and bias signal.</p>
<p>The control law is defined as </p>
<pre>              1+sTi 
  CS(s) = Kp ------- (SP(s) - PV(S))
               sTi</pre>
<p><br><br><br><br><b>Scheme</b></p>
<p><br><br><br><br><img src=\"modelica://IndustrialControlSystems/help/images/Controllers/PropInt.png\"/> </p>
<table cellspacing=\"2\" cellpadding=\"0\" border=\"0\"<tr>
<td style=\"background-color: #e0e0e0\"><p align=\"center\"><h4><span style=\"background-color: #e0e0e0\">Name</span></h4></p></td>
<td style=\"background-color: #e0e0e0\"><p align=\"center\"><h4><span style=\"background-color: #e0e0e0\">Description</span></h4></p></td>
<td style=\"background-color: #e0e0e0\"><p align=\"center\"><h4><span style=\"background-color: #e0e0e0\">Conditional?</span></h4></p></td>
</tr>
<tr>
<td><p>SP</p></td>
<td><p>Set Point</p></td>
<td><p>NO</p></td>
</tr>
<tr>
<td><p>PV</p></td>
<td><p>Process Variable</p></td>
<td><p>NO</p></td>
</tr>
<tr>
<td><p>CS</p></td>
<td><p>Control Signal</p></td>
<td><p>NO</p></td>
</tr>
<tr>
<td><p>TR</p></td>
<td><p>Track Reference signal</p></td>
<td><p>YES (useTS)</p></td>
</tr>
<tr>
<td><p>TS</p></td>
<td><p>Track Switch signal</p></td>
<td><p>YES (useTS)</p></td>
</tr>
<tr>
<td><p>Bias</p></td>
<td><p>Biasing signal</p></td>
<td><p>YES (useBIAS)</p></td>
</tr>
<tr>
<td><p>ATreq</p></td>
<td><p>AutoTuning request</p></td>
<td><p>YES (*)</p><p><br>(*) AutoTuning Not available here, please see the package</p><p><br><a href=\"Modelica://IndustrialControlSystems.Controllers.AutoTuning\">AutoTuning</a> </p><p><br>The controller can have various operating conditions: </p><ul>
<li>Automatic, </li><li>and Tracking </li>
</ul><p><br><br>In the Automatic mode, the control output is computed with the proportional+integral control law, while in the tracking mode</p><p><br><br>it is defined by the input TR.</p><p><br><br>To note that in the tracking mode the output signal CS is equal to TR, and the first order filter </p><pre>     1  -------- TR(S)   1+s*Ti  </pre><p>is replaced by </p><pre>     1  -------- TR(S)   1+s*eps  </pre><p>where eps is a small time constant. With such a scheme, the integrator does not diverge while the tracking mode</p><p>is enabled. </p></td>
</tr>
</table>
</html>", revisions="<html>
<dl><dt>Industrial Control Systems (v 1.0.0) : April-May 2012</dt>
<dl><dt>List of revisions:</dt>
<p><ul>
<li>11 May 2012 (author: Marco Bonvini)</li>
</ul></p>
<dl><dt><b>Main Authors:</b> <br/></dt>
<dd>Marco Bonvini; &LT;<a href=\"mailto:bonvini@elet.polimi.it\">bonvini@elet.polimi.it</a>&GT;</dd>
<dd>Alberto Leva &LT;<a href=\"mailto:leva@elet.polimi.it\">leva@elet.polimi.it</a>&GT;<br/></dd>
<dd>Politecnico di Milano</dd>
<dd>Dipartimento di Elettronica e Informazione</dd>
<dd>Via Ponzio 34/5</dd>
<dd>20133 Milano - ITALIA -<br/></dd>
<dt><b>Copyright:</b> </dt>
<dd>Copyright &copy; 2010-2012, Marco Bonvini and Alberto Leva.<br/></dd>
<dd><i>The IndustrialControlSystems package is <b>free</b> software; it can be redistributed and/or modified under the terms of the <b>Modelica license</b>.</i><br/></dd>
</dl></html>"));
end PI_steadyState;
