package Gazebo
  import SI = Modelica.SIunits;

  model World
    extends Modelica.Mechanics.MultiBody.World(final n = {0, 0, -1});
  end World;

  package Examples
    extends Modelica.Icons.ExamplesPackage;

    model TestSphere2Plane
      inner Modelica.Mechanics.MultiBody.World world(n = {0, 0, -1}) annotation(
        Placement(transformation(extent = {{-80, -60}, {-60, -40}})));
      Objects.Plane plane annotation(
        Placement(transformation(extent = {{-40, -60}, {-20, -40}})));
      Gazebo.Objects.Sphere sphere(m = 1, r = 0.5, r_0(start = {0, 0, 1}, fixed = true), v_0(fixed = true)) annotation(
        Placement(visible = true, transformation(extent = {{-40, 0}, {-20, 20}}, rotation = 0)));
      Gazebo.InteractionModels.Sphere2Plane sphere2Plane(id_a = "sphere", id_b = "plane") annotation(
        Placement(visible = true, transformation(origin = {-30, -20}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
    equation
      connect(sphere2Plane.frame_b, plane.frame_interaction) annotation(
        Line(points = {{-30, -30}, {-30, -50}}, color = {95, 95, 95}, thickness = 0.5));
      connect(sphere.frame_interaction, sphere2Plane.frame_a) annotation(
        Line(points = {{-30, 10}, {-30, -10}}, color = {175, 175, 175}));
      connect(world.frame_b, plane.frame_a) annotation(
        Line(points = {{-60, -50}, {-40, -50}, {-40, -50}}, color = {95, 95, 95}, thickness = 0.5));
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false)),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end TestSphere2Plane;

    model TestSpheres2Plane
      inner Modelica.Mechanics.MultiBody.World world(n = {0, 0, -1}) annotation(
        Placement(visible = true, transformation(extent = {{-80, -100}, {-60, -80}}, rotation = 0)));
      Objects.Plane plane_1 annotation(
        Placement(transformation(extent = {{-40, -60}, {-20, -40}})));
      Gazebo.Objects.Plane plane_2 annotation(
        Placement(visible = true, transformation(extent = {{0, -60}, {20, -40}}, rotation = 0)));
      Objects.Plane plane_3 annotation(
        Placement(visible = true, transformation(extent = {{40, -60}, {60, -40}}, rotation = 0)));
      Objects.Sphere sphere_1(m = 1, body.r_0(start = {1, 0, 0}, fixed = true), body.v_0(fixed = true)) annotation(
        Placement(visible = true, transformation(extent = {{-40, 0}, {-20, 20}}, rotation = 0)));
      Objects.Sphere sphere_2(m = 2, body.r_0(start = {2, 0, 0}, fixed = true), body.v_0(fixed = true)) annotation(
        Placement(visible = true, transformation(extent = {{0, 0}, {20, 20}}, rotation = 0)));
      Objects.Sphere sphere_3(m = 4, body.r_0(start = {4, 0, 0}, fixed = true), body.v_0(fixed = true)) annotation(
        Placement(visible = true, transformation(extent = {{40, 0}, {60, 20}}, rotation = 0)));
      InteractionModels.Sphere2Plane sphere2Plane_1 annotation(
        Placement(visible = true, transformation(origin = {-30, -20}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
      InteractionModels.Sphere2Plane sphere2Plane_2 annotation(
        Placement(visible = true, transformation(origin = {10, -20}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
      InteractionModels.Sphere2Plane sphere2Plane_3 annotation(
        Placement(visible = true, transformation(origin = {50, -20}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
    equation
      connect(plane_3.frame_a, world.frame_b) annotation(
        Line(points = {{40, -50}, {30, -50}, {30, -90}, {-60, -90}, {-60, -90}}, color = {95, 95, 95}));
      connect(plane_2.frame_a, world.frame_b) annotation(
        Line(points = {{0, -50}, {-10, -50}, {-10, -90}, {-60, -90}, {-60, -90}}, color = {95, 95, 95}));
      connect(sphere2Plane_2.frame_b, plane_2.frame_interaction) annotation(
        Line(points = {{10, -30}, {10, -50}}, color = {95, 95, 95}));
      connect(sphere_3.frame_interaction, sphere2Plane_3.frame_a) annotation(
        Line(points = {{50, 10}, {50, 10}, {50, -10}, {50, -10}}, color = {175, 175, 175}));
      connect(sphere_2.frame_interaction, sphere2Plane_2.frame_a) annotation(
        Line(points = {{10, 10}, {10, 10}, {10, -10}, {10, -10}}, color = {175, 175, 175}));
      connect(sphere2Plane_3.frame_b, plane_3.frame_interaction) annotation(
        Line(points = {{50, -30}, {50, -30}, {50, -50}, {50, -50}}, color = {95, 95, 95}));
      connect(sphere2Plane_1.frame_b, plane_1.frame_interaction) annotation(
        Line(points = {{-30, -30}, {-30, -50}}, color = {95, 95, 95}, thickness = 0.5));
      connect(sphere_1.frame_interaction, sphere2Plane_1.frame_a) annotation(
        Line(points = {{-30, 10}, {-30, -10}}, color = {175, 175, 175}));
      connect(world.frame_b, plane_1.frame_a) annotation(
        Line(points = {{-60, -90}, {-50, -90}, {-50, -50}, {-40, -50}}, color = {95, 95, 95}, thickness = 0.5));
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false)),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end TestSpheres2Plane;

    model TestSphere2PlaneNoInteraction
      inner Modelica.Mechanics.MultiBody.World world(n = {0, 0, -1}) annotation(
        Placement(transformation(extent = {{-80, -60}, {-60, -40}})));
      Objects.Plane plane annotation(
        Placement(transformation(extent = {{-40, -60}, {-20, -40}})));
      Gazebo.Objects.Sphere sphere(m = 1, body.r_0(start = {1, 0, 0}, each fixed = true), body.v_0(each fixed = true)) annotation(
        Placement(visible = true, transformation(extent = {{-40, 0}, {-20, 20}}, rotation = 0)));
      Gazebo.InteractionModels.NoInteraction sphere2Plane(id_a = "sphere", id_b = "plane") annotation(
        Placement(visible = true, transformation(origin = {-30, -20}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
    equation
      connect(sphere2Plane.frame_b, plane.frame_interaction) annotation(
        Line(points = {{-30, -30}, {-30, -50}}, color = {95, 95, 95}, thickness = 0.5));
      connect(sphere.frame_interaction, sphere2Plane.frame_a) annotation(
        Line(points = {{-30, 10}, {-30, -10}}, color = {175, 175, 175}));
      connect(world.frame_b, plane.frame_a) annotation(
        Line(points = {{-60, -50}, {-40, -50}, {-40, -50}}, color = {95, 95, 95}, thickness = 0.5));
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false)),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end TestSphere2PlaneNoInteraction;

    model TestSphere2PlaneNoInteractionModelica
      inner Modelica.Mechanics.MultiBody.World world(n = {0, 0, -1}) annotation(
        Placement(transformation(extent = {{-80, -60}, {-60, -40}})));
      Objects.Plane plane annotation(
        Placement(transformation(extent = {{-40, -60}, {-20, -40}})));
      Gazebo.Objects.Sphere sphere(m = 1, r = 1, r_0(fixed = true, start = {0, 0, 1}), v_0(fixed = true), w_a(fixed = true)) annotation(
        Placement(visible = true, transformation(extent = {{-40, 0}, {-20, 20}}, rotation = 0)));
      Gazebo.InteractionModels.NoInteractionModelica sphere2Plane(id_a = "sphere", id_b = "plane") annotation(
        Placement(visible = true, transformation(origin = {-30, -20}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
    equation
      connect(sphere2Plane.frame_b, plane.frame_interaction) annotation(
        Line(points = {{-30, -30}, {-30, -50}}, color = {95, 95, 95}, thickness = 0.5));
      connect(sphere.frame_interaction, sphere2Plane.frame_a) annotation(
        Line(points = {{-30, 10}, {-30, -10}}, color = {175, 175, 175}));
      connect(world.frame_b, plane.frame_a) annotation(
        Line(points = {{-60, -50}, {-40, -50}, {-40, -50}}, color = {95, 95, 95}, thickness = 0.5));
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false)),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end TestSphere2PlaneNoInteractionModelica;

    model TestSphere2PlaneModelica
      inner Modelica.Mechanics.MultiBody.World world(n = {0, 0, -1}) annotation(
        Placement(transformation(extent = {{-80, -60}, {-60, -40}})));
      Objects.Plane plane annotation(
        Placement(transformation(extent = {{-40, -60}, {-20, -40}})));
      Gazebo.Objects.Sphere sphere(m = 1, r = 0.5, r_0(start = {0, 0, 1}, each fixed = true), v_0(each fixed = true)) annotation(
        Placement(visible = true, transformation(extent = {{-40, 0}, {-20, 20}}, rotation = 0)));
      Gazebo.InteractionModels.Sphere2PlaneModelica sphere2Plane(id_a = "sphere", id_b = "plane", r_sphere = 0.5) annotation(
        Placement(visible = true, transformation(origin = {-30, -20}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
    equation
      connect(sphere2Plane.frame_b, plane.frame_interaction) annotation(
        Line(points = {{-30, -30}, {-30, -50}}, color = {95, 95, 95}, thickness = 0.5));
      connect(sphere.frame_interaction, sphere2Plane.frame_a) annotation(
        Line(points = {{-30, 10}, {-30, -10}}, color = {175, 175, 175}));
      connect(world.frame_b, plane.frame_a) annotation(
        Line(points = {{-60, -50}, {-40, -50}, {-40, -50}}, color = {95, 95, 95}, thickness = 0.5));
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false)),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end TestSphere2PlaneModelica;
  end Examples;

  package Objects
    model Sphere
      parameter String ID "Object identifier";
      parameter SI.Length r(min = 0) "Radius";
      parameter SI.Mass m(min = 0, start = 1) "Mass";
      final parameter SI.Length d = 2 * r "Diameter";
      final parameter SI.Inertia I_11 = 2 / 5 * m * r ^ 2 "(1,1) element of inertia tensor" annotation(
        Dialog(group = "Inertia tensor (resolved in center of mass, parallel to frame_a)"));
      final parameter SI.Inertia I_22 = I_11 "(2,2) element of inertia tensor" annotation(
        Dialog(group = "Inertia tensor (resolved in center of mass, parallel to frame_a)"));
      final parameter SI.Inertia I_33 = I_11 "(3,3) element of inertia tensor" annotation(
        Dialog(group = "Inertia tensor (resolved in center of mass, parallel to frame_a)"));
      final parameter SI.Inertia I_21 = 0 "(2,1) element of inertia tensor" annotation(
        Dialog(group = "Inertia tensor (resolved in center of mass, parallel to frame_a)"));
      final parameter SI.Inertia I_31 = 0 "(3,1) element of inertia tensor" annotation(
        Dialog(group = "Inertia tensor (resolved in center of mass, parallel to frame_a)"));
      final parameter SI.Inertia I_32 = 0 "(3,2) element of inertia tensor" annotation(
        Dialog(group = "Inertia tensor (resolved in center of mass, parallel to frame_a)"));
      SI.Position r_0[3] = body.r_0 "Position vector from origin of world frame to origin of frame_a";
      SI.Velocity v_0[3] = body.v_0 "Absolute velocity of frame_a, resolved in world frame";
      SI.AngularVelocity w_a[3] = body.w_a "Absolute angular velocity of frame_a resolved in frame_a";
      Modelica.Mechanics.MultiBody.Parts.Body body(m = m, sphereDiameter = 1, r_CM = {0, 0, 0}, I_11 = I_11, I_21 = I_21, I_22 = I_22, I_31 = I_31, I_32 = I_32, I_33 = I_33) annotation(
        Placement(visible = true, transformation(origin = {-2, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
      Modelica.Mechanics.MultiBody.Interfaces.Frame_a frame_a annotation(
        Placement(visible = true, transformation(origin = {-100, 0}, extent = {{-16, -16}, {16, 16}}, rotation = 0), iconTransformation(origin = {-100, 0}, extent = {{-16, -16}, {16, 16}}, rotation = 0)));
      Interfaces.Frame_interaction frame_interaction annotation(
        Placement(visible = true, transformation(origin = {0, -42}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {0, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    equation
      connect(frame_interaction, body.frame_a) annotation(
        Line(points = {{0, -42}, {-18, -42}, {-18, 0}, {-12, 0}, {-12, 0}}));
      connect(frame_a, body.frame_a) annotation(
        Line(points = {{-100, 0}, {-12, 0}, {-12, 0}, {-12, 0}}));
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false, initialScale = 0.1), graphics = {Ellipse(origin = {-3, 4}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Sphere, extent = {{-97, 96}, {103, -104}}, endAngle = 360)}),
        Diagram(coordinateSystem(preserveAspectRatio = false)),
        uses(Modelica(version = "3.2.2")));
    end Sphere;

    model Plane "Plane oriented as the xy plane in the frame_a connector"
      import Modelica.Mechanics.MultiBody.Visualizers;
      parameter String ID "Object identifier";
      outer Modelica.Mechanics.MultiBody.World world;
      Modelica.Mechanics.MultiBody.Interfaces.Frame_a frame_a annotation(
        Placement(transformation(extent = {{-116, -16}, {-84, 16}}), iconTransformation(extent = {{-116, -16}, {-84, 16}})));
      Interfaces.Frame_interaction frame_interaction annotation(
        Placement(transformation(extent = {{-10, -10}, {10, 10}}), iconTransformation(extent = {{-10, -10}, {10, 10}})));
      Modelica.Mechanics.MultiBody.Visualizers.Advanced.Shape vis(shapeType = "box", r_shape = {0, 0, 0}, lengthDirection = {0, 0, 1}, widthDirection = {1, 0, 0}, length = 0.01, width = 1, height = 1, r = frame_a.r_0, R = frame_a.R) if world.enableAnimation;
    equation
      Connections.branch(frame_a.R, frame_interaction.R);
      frame_a.r_0 = frame_interaction.r_0;
      frame_a.R = frame_interaction.R;
      frame_a.f + frame_interaction.f = zeros(3);
      frame_a.t + frame_interaction.t = zeros(3);
      annotation(
        Icon(graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}, lineColor = {28, 108, 200})}));
    end Plane;
  end Objects;

  package Interfaces
    connector Frame_interaction
      extends Modelica.Mechanics.MultiBody.Interfaces.Frame;
      annotation(
        Icon(graphics = {Ellipse(extent = {{-100, 100}, {100, -100}}, lineColor = {175, 175, 175}, fillColor = {175, 175, 175}, fillPattern = FillPattern.Solid)}));
    end Frame_interaction;
  end Interfaces;

  package BaseClasses
    model PhysicalInteractionBase "Base class for all physical interaction models"
      extends Modelica.Mechanics.MultiBody.Interfaces.PartialTwoFrames;
      import Modelica.Mechanics.MultiBody.Frames;
      parameter Integer maxContacts = 10 "Number of max contact points";
      parameter String id_a = "" "Id of interacting object a";
      parameter String id_b = "" "Id of interacting object b";
      //ExternalInterfaces.GazeboInterface gI = ExternalInterfaces.GazeboInterface(id_a = id_a, id_b = id_b);
      Real numberOfContactPoints "Number of actual contact points";
      Real cp_a[maxContacts, 3] "Array of contact points on body a, resolved in frame_a";
      Real cp_b[maxContacts, 3] "Array of contact points on body b, resolved in frame_b";
      Real depth_a[maxContacts] "Array of penetration depths in body a";
      Real depth_b[maxContacts] "Array of penetration depths in body a";
      Real normals_a[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
      Real normals_b[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
      SI.Force f[3];
      SI.Torque t[3];

      replaceable function collisionDetection
        //input ExternalInterfaces.GazeboInterface gI;
        input Integer maxContacts "Maximum number of contact points";
        input Real r_a[3] "Position vector of object a from world frame to the connector frame origin, resolved in world frame";
        input Frames.Quaternions.Orientation Q_a "Quaternion corresponding to the rotation matrix of object b from world frame to local frame";
        input String id_a "unique id for object a";
        input Real r_b[3] "Position vector of object b from world frame to the connector frame origin, resolved in world frame";
        input Frames.Quaternions.Orientation Q_b "Quaternion corresponding to the rotation matrix of object b from world frame to local frame";
        input String id_b "unique id for object b";
        output Real numberOfContactPoints "Number of actual contact points";
        output Real cp_a[maxContacts, 3] "Array of contact points on body a, resolved in frame_a";
        output Real cp_b[maxContacts, 3] "Array of contact points on body b, resolved in frame_b";
        output Real depth_a[maxContacts] "Array of penetration depths in body a";
        output Real depth_b[maxContacts] "Array of penetration depths in body a";
        output Real normals_a[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
        output Real normals_b[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
      
        external "C"  annotation(
          Library = "Gazebo",
          Include = "#include \"gazebo_interface.h\"");
      end collisionDetection;

      replaceable function computeInteraction
        input Real numberOfContactPoints "Number of actual contact points";
        input Integer maxContacts "Maximum number of contact points";
        input Real cp_a[maxContacts, 3] "Array of contact points on body a, resolved in frame_a";
        input Real cp_b[maxContacts, 3] "Array of contact points on body b, resolved in frame_b";
        input Real depth_a[maxContacts] "Array of penetration depths in body a";
        input Real depth_b[maxContacts] "Array of penetration depths in body a";
        input Real normals_a[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
        input Real normals_b[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
        output SI.Force[3] f "Equivalent force applied to frame_a";
        output SI.Torque[3] t "Equivalent torque applied to frame_a";
      end computeInteraction;

      function zeroInteraction
        output SI.Force f[3];
        output SI.Torque t[3];
      algorithm
        f := {0, 0, 0};
        t := {0, 0, 0};
      end zeroInteraction;
    equation
      (numberOfContactPoints, cp_a, cp_b, depth_a, depth_b, normals_a, normals_b) = collisionDetection(maxContacts, frame_a.r_0, Frames.to_Q(frame_a.R), id_a, frame_b.r_0, Frames.to_Q(frame_b.R), id_b);
      assert(numberOfContactPoints <= maxContacts, "Too many contact points");
      (f, t) = computeInteraction(numberOfContactPoints, maxContacts, cp_a, cp_b, depth_a, depth_b, normals_a, normals_b);
      if sum(depth_a + depth_b) > 0 then
        frame_a.f = f;
        frame_a.t = t;
      else
        frame_a.f = {0, 0, 0};
        frame_a.t = {0, 0, 0};
      end if;
      frame_a.f + Frames.resolveRelative(frame_b.f, frame_a.R, frame_b.R) = zeros(3) "Newton's 3rd law";
      frame_a.t + Frames.resolveRelative(frame_b.t, frame_a.R, frame_b.R) = zeros(3) "Newton's 3rd law";
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false), graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}, lineColor = {28, 108, 200}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid)}),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end PhysicalInteractionBase;
  end BaseClasses;

  //  package ExternalInterfaces "Contains all the classes to interface with the Gazebo engine"
  //    class GazeboInterface
  //      extends ExternalObject;
  //      function constructor
  //        input String id_a;
  //        input String id_b;
  //        output GazeboInterface gazeboInterface;
  //        external "C" gazeboInterface = initGazeboInterface(id_a, id_b) annotation(Library = "Gazebo", Include = "#include \"gazebo_interface.h\"");
  //      end constructor;
  //      function destructor "Release storage of table"
  //        input GazeboInterface gazeboInterface;
  //        external "C" gazeboInterfaceDestructor(gazeboInterface) annotation(Library = "Gazebo", Include = "#include \"gazebo_interface.h\"");
  //      end destructor;
  //      annotation(Icon(coordinateSystem(preserveAspectRatio = false)), Diagram(coordinateSystem(preserveAspectRatio = false)));
  //    end GazeboInterface;
  //  end ExternalInterfaces;

  package InteractionModels
    model Sphere2Plane
      extends BaseClasses.PhysicalInteractionBase;
      constant SI.TranslationalSpringConstant k_body = 1e4;

      redeclare function extends computeInteraction
          input SI.TranslationalSpringConstant k = k_body;

        protected
          Real n[3] "Normal vector";

        algorithm
          f := {0, 0, 0};
          t := {0, 0, 0};
          for i in 1:integer(numberOfContactPoints + 1e-6) loop
            n := Modelica.Math.Vectors.normalize(normals_a[i, :] - cp_a[i, :]);
            f := f - k * (depth_a[i] + depth_b[i]) * n;
          end for;
// Actual sign TBC!!!
      end computeInteraction;
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false), graphics = {Text(extent = {{-94, 20}, {-46, -20}}, lineColor = {28, 108, 200}, textString = "S"), Text(extent = {{52, 20}, {92, -20}}, lineColor = {28, 108, 200}, textString = "P")}),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end Sphere2Plane;

    model NoInteraction
      extends BaseClasses.PhysicalInteractionBase;

      redeclare function extends computeInteraction
        algorithm
          f := {0, 0, 0};
          t := {0, 0, 0};
      end computeInteraction;
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false), graphics = {Text(extent = {{-94, 20}, {-46, -20}}, lineColor = {28, 108, 200}, textString = "S"), Text(extent = {{52, 20}, {92, -20}}, lineColor = {28, 108, 200}, textString = "P")}),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end NoInteraction;

    model NoInteractionModelica "No interaction model written in native Modelica"
      extends BaseClasses.PhysicalInteractionBase(final maxContacts = 0);
      import Modelica.Mechanics.MultiBody.Frames;

      redeclare function collisionDetection
        //input ExternalInterfaces.GazeboInterface gI;
        input Integer maxContacts "Maximum number of contact points";
        input Real r_a[3] "Position vector of object a from world frame to the connector frame origin, resolved in world frame";
        input Frames.Quaternions.Orientation Q_a "Quaternion corresponding to the rotation matrix of object b from world frame to local frame";
        input String id_a "unique id for object a";
        input Real r_b[3] "Position vector of object b from world frame to the connector frame origin, resolved in world frame";
        input Frames.Quaternions.Orientation Q_b "Quaternion corresponding to the rotation matrix of object b from world frame to local frame";
        input String id_b "unique id for object b";
        output Real numberOfContactPoints "Number of actual contact points";
        output Real cp_a[maxContacts, 3] "Array of contact points on body a, resolved in frame_a";
        output Real cp_b[maxContacts, 3] "Array of contact points on body b, resolved in frame_b";
        output Real depth_a[maxContacts] "Array of penetration depths in body a";
        output Real depth_b[maxContacts] "Array of penetration depths in body a";
        output Real normals_a[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
        output Real normals_b[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
      algorithm
        numberOfContactPoints := 0;
      end collisionDetection;

      redeclare function extends computeInteraction
        algorithm
          f := {0, 0, 0};
          t := {0, 0, 0};
      end computeInteraction;
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false), graphics = {Text(extent = {{-94, 20}, {-46, -20}}, lineColor = {28, 108, 200}, textString = "S"), Text(extent = {{52, 20}, {92, -20}}, lineColor = {28, 108, 200}, textString = "P")}),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end NoInteractionModelica;

    model Sphere2PlaneModelica
      extends BaseClasses.PhysicalInteractionBase(final maxContacts = 1);
      constant SI.TranslationalSpringConstant k_body = 1e4;
      constant SI.Length r_sphere = 0.5;
      import Modelica.Mechanics.MultiBody.Frames;

      redeclare function collisionDetection
        //input ExternalInterfaces.GazeboInterface gI;
        input Integer maxContacts "Maximum number of contact points";
        input Real r_a[3] "Position vector of object a from world frame to the connector frame origin, resolved in world frame";
        input Frames.Quaternions.Orientation Q_a "Quaternion corresponding to the rotation matrix of object a from world frame to local frame";
        input String id_a "unique id for object a";
        input Real r_b[3] "Position vector of object b from world frame to the connector frame origin, resolved in world frame";
        input Frames.Quaternions.Orientation Q_b "Quaternion corresponding to the rotation matrix of object b from world frame to local frame";
        input String id_b "unique id for object b";
        output Real numberOfContactPoints "Number of actual contact points";
        output Real cp_a[maxContacts, 3] "Array of contact points on body a, resolved in frame_a";
        output Real cp_b[maxContacts, 3] "Array of contact points on body b, resolved in frame_b";
        output Real depth_a[maxContacts] "Array of penetration depths in body a";
        output Real depth_b[maxContacts] "Array of penetration depths in body a";
        output Real normals_a[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
        output Real normals_b[maxContacts, 3] "Array of normals (as defined by Gazebo) on body a, resolved in frame_a";
      algorithm
        if r_a[3] > r_sphere then
          numberOfContactPoints := 0;
          cp_a[1, :] := zeros(3);
          cp_b[1, :] := zeros(3);
        else
          numberOfContactPoints := 1;
          cp_a[1, :] := Frames.resolve2(Frames.from_Q(Q_a, {0, 0, 0}), {r_a[1], r_a[2], 0});
          cp_b[1, :] := Frames.resolve2(Frames.from_Q(Q_b, {0, 0, 0}), {r_a[1], r_a[2], 0});
        end if;
        depth_a := {r_sphere - r_a[3]};
        depth_b := {0.0};
        normals_a := {{0,  0, 1}};
        normals_b := {{0, 0, 1}};
        annotation(Inline = false);
      end collisionDetection;
    
    
      redeclare function extends computeInteraction
          input SI.TranslationalSpringConstant k = k_body;

        protected
          Real n[3] "Normal vector";

        algorithm
          f := {0, 0, 0};
          t := {0, 0, 0};
          for i in 1:integer(numberOfContactPoints + 1e-6) loop
            n := Modelica.Math.Vectors.normalize(normals_a[i, :] - cp_a[i, :]);
            f := f - k * (depth_a[i] + depth_b[i]) * n;
          end for;
// Actual sign TBC!!!
        annotation(Inline = false);
      end computeInteraction;
      annotation(
        Icon(coordinateSystem(preserveAspectRatio = false), graphics = {Text(extent = {{-94, 20}, {-46, -20}}, lineColor = {28, 108, 200}, textString = "S"), Text(extent = {{52, 20}, {92, -20}}, lineColor = {28, 108, 200}, textString = "P")}),
        Diagram(coordinateSystem(preserveAspectRatio = false)));
    end Sphere2PlaneModelica;
  end InteractionModels;
  annotation(
    uses(Modelica(version = "3.2.2")));
end Gazebo;
