package frontend_bench

  model simple
    //import SI = Modelica.SIunits;
    parameter Modelica.SIunits.AbsolutePressure p_start = 101325 "Initial value of pressure";
    parameter Modelica.SIunits.Temperature T_start = 303.15 "Initial value of temperature";
    parameter Modelica.Fluid.Types.ModelStructure pipeModelStructure = Modelica.Fluid.Types.ModelStructure.a_v_b;
    parameter Real m_flow_start = 1e-6;
    package Gas = Modelica.Media.IdealGases.MixtureGases.SimpleNaturalGas "Medium 1 model";
    parameter Real X_start[Gas.nX] = {0.233, 1.0e-6, 1.0e-6, 1.0e-6, 1.0e-6, 0.766995} "Initial value of mass fractions";
    package Water = Modelica.Media.Water.ConstantPropertyLiquidWater(T_max = 700) "Medium 2 model";
    inner Modelica.Fluid.System system(T_ambient = T_start, p_ambient = p_start, m_flow_small = 1e-8, momentumDynamics = Modelica.Fluid.Types.Dynamics.DynamicFreeInitial, energyDynamics = Modelica.Fluid.Types.Dynamics.DynamicFreeInitial, massDynamics = Modelica.Fluid.Types.Dynamics.DynamicFreeInitial) annotation(
      Placement(visible = true, transformation(extent = {{-190, 118}, {-170, 138}}, rotation = 0)));
    Modelica.Fluid.Sources.MassFlowSource_T Air(redeclare package Medium = Gas, T = 273.15 + 50, X = X_start, m_flow = 0.014, nPorts = 1) annotation(
      Placement(visible = true, transformation(origin = {-74, 66}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    Modelica.Fluid.Sources.Boundary_pT Ambient_out(redeclare package Medium = Gas, T = T_start, X = X_start, nPorts = 1, p = p_start) annotation(
      Placement(visible = true, transformation(origin = {76, 230}, extent = {{10, -10}, {-10, 10}}, rotation = 0)));
    frontend_bench.DoublePipeHX WT4(redeclare package Medium_1 = Gas, redeclare package Medium_2 = Gas, T_start_1 = T_start, T_start_2 = T_start, Twall_start = T_start, X_start_1 = X_start, X_start_2 = X_start, c_wall = 477, c_wall_amb = 840, dT = 0, lambda_wall = 50, lambda_wall_amb = 0.1, length = 3, m_flow_start_1 = m_flow_start, m_flow_start_2 = m_flow_start, modelStructure_1 = pipeModelStructure, modelStructure_2 = pipeModelStructure, nNodes = 5, p_a_start1 = p_start, p_a_start2 = p_start, p_b_start1 = p_start, p_b_start2 = p_start, rho_wall(displayUnit = "kg/m3") = 8000, rho_wall_amb(displayUnit = "kg/m3") = 100, roughness_1 = 2.5e-5, roughness_2 = 2.5e-5, diameter1 = 0.054, diameter2_i = 0.06, diameter2_a = 0.114, diameter3 = 0.174, eps = 0.9, alpha_amb = 7, Nus_f = 1.4) annotation(
      Placement(visible = true, transformation(origin = {-24, 48}, extent = {{-10, 10}, {10, -10}}, rotation = 90)));
    frontend_bench.DoublePipeHX WT51(redeclare package Medium_1 = Gas, redeclare package Medium_2 = Water, T_start_1 = T_start, T_start_2 = T_start, Twall_start = T_start, X_start_1 = X_start, c_wall = 477, c_wall_amb = 840, dT = 0, lambda_wall = 50, lambda_wall_amb = 0.1, length = 3, m_flow_start_1 = m_flow_start, m_flow_start_2 = m_flow_start, modelStructure_1 = pipeModelStructure, modelStructure_2 = pipeModelStructure, nNodes = 5, p_a_start1 = p_start, p_a_start2 = p_start, p_b_start1 = p_start, p_b_start2 = p_start, rho_wall(displayUnit = "kg/m3") = 8000, rho_wall_amb(displayUnit = "kg/m3") = 100, roughness_1 = 2.5e-5, roughness_2 = 2.5e-5, diameter1 = 0.054, diameter2_i = 0.06, diameter2_a = 0.114, diameter3 = 0.174, eps = 0.9, alpha_amb = 7) annotation(
      Placement(visible = true, transformation(origin = {-24, 116}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
    frontend_bench.DoublePipeHX WT52(redeclare package Medium_1 = Gas, redeclare package Medium_2 = Water, T_start_1 = T_start, T_start_2 = T_start, Twall_start = T_start, X_start_1 = X_start, c_wall = 477, c_wall_amb = 840, dT = 0, lambda_wall = 50, lambda_wall_amb = 0.1, length = 3, m_flow_start_1 = m_flow_start, m_flow_start_2 = m_flow_start, modelStructure_1 = pipeModelStructure, modelStructure_2 = pipeModelStructure, nNodes = 5, p_a_start1 = p_start, p_a_start2 = p_start, p_b_start1 = p_start, p_b_start2 = p_start, rho_wall(displayUnit = "kg/m3") = 8000, rho_wall_amb(displayUnit = "kg/m3") = 100, roughness_1 = 2.5e-5, roughness_2 = 2.5e-5, diameter1 = 0.054, diameter2_i = 0.06, diameter2_a = 0.114, diameter3 = 0.174, eps = 0.9, alpha_amb = 7) annotation(
      Placement(visible = true, transformation(origin = {-24, 156}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
    frontend_bench.DoublePipeHX Filter(redeclare package Medium_1 = Gas, redeclare package Medium_2 = Water, T_start_1 = T_start, T_start_2 = T_start, Twall_start = T_start, X_start_1 = X_start, c_wall = 477, c_wall_amb = 840, dT = 0, lambda_wall = 50, lambda_wall_amb = 0.1, length = 1, m_flow_start_1 = m_flow_start, m_flow_start_2 = m_flow_start, modelStructure_1 = pipeModelStructure, modelStructure_2 = pipeModelStructure, nNodes = 5, p_a_start1 = p_start, p_a_start2 = p_start, p_b_start1 = p_start, p_b_start2 = p_start, rho_wall(displayUnit = "kg/m3") = 8000, rho_wall_amb(displayUnit = "kg/m3") = 100, roughness_1 = 2.5e-5, roughness_2 = 2.5e-5, diameter1 = 0.054, diameter2_i = 0.06, diameter2_a = 0.114, diameter3 = 0.174, eps = 0.9, alpha_amb = 7) annotation(
      Placement(visible = true, transformation(origin = {-24, 200}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
    frontend_bench.DoublePipeHX WT6(redeclare package Medium_1 = Gas, redeclare package Medium_2 = Water, T_start_1 = T_start, T_start_2 = T_start, Twall_start = T_start, X_start_1 = X_start, c_wall = 477, c_wall_amb = 840, dT = 0, lambda_wall = 50, lambda_wall_amb = 0.1, length = 1.5, m_flow_start_1 = m_flow_start, m_flow_start_2 = m_flow_start, modelStructure_1 = pipeModelStructure, modelStructure_2 = pipeModelStructure, nNodes = 5, p_a_start1 = p_start, p_a_start2 = p_start, p_b_start1 = p_start, p_b_start2 = p_start, rho_wall(displayUnit = "kg/m3") = 8000, rho_wall_amb(displayUnit = "kg/m3") = 100, roughness_1 = 2.5e-5, roughness_2 = 2.5e-5, diameter1 = 0.054, diameter2_i = 0.06, diameter2_a = 0.114, diameter3 = 0.174, eps = 0.9, alpha_amb = 7) annotation(
      Placement(visible = true, transformation(origin = {20, 230}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    
    frontend_bench.DoublePipeHX WT7(redeclare package Medium_1 = Water, redeclare package Medium_2 = Water, T_start_1 = T_start, T_start_2 = T_start, Twall_start = T_start, c_wall = 477, c_wall_amb = 477, dT = 0, lambda_wall = 250, lambda_wall_amb = 250, length = 1, areaFactor = 40, m_flow_start_1 = m_flow_start, m_flow_start_2 = m_flow_start, modelStructure_1 = pipeModelStructure, modelStructure_2 = pipeModelStructure, nNodes = 5, p_a_start1 = p_start, p_a_start2 = p_start, p_b_start1 = p_start, p_b_start2 = p_start, rho_wall(displayUnit = "kg/m3") = 8000, rho_wall_amb(displayUnit = "kg/m3") = 8920, roughness_1 = 2.5e-5, roughness_2 = 2.5e-5, diameter1 = 0.02, diameter2_i = 0.021, diameter2_a = 0.03, diameter3 = 0.031, eps = 0.9, alpha_amb = 7) annotation(
      Placement(visible = true, transformation(origin = {40, 148}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
    Modelica.Fluid.Sources.Boundary_pT to_Storage(redeclare package Medium = Water, T = T_start, nPorts = 1, p = p_start) annotation(
      Placement(visible = true, transformation(origin = {100, 192}, extent = {{10, -10}, {-10, 10}}, rotation = 0)));
    
    Modelica.Fluid.Sources.MassFlowSource_T from_Storage(redeclare package Medium = Water, T = T_start, m_flow = 0.25, nPorts = 1) annotation(
      Placement(visible = true, transformation(origin = {98, 96}, extent = {{10, -10}, {-10, 10}}, rotation = 0)));
    Modelica.Fluid.Sources.Boundary_pT WT_medium_boundary(redeclare package Medium = Water, T = T_start, nPorts = 2, p = 200000, use_T_in = true) annotation(
      Placement(visible = true, transformation(origin = {8, 110}, extent = {{8, -8}, {-8, 8}}, rotation = 90)));
    //laut nachrechnung über Energiebilanz und nach Kennlinie Grundfos UPS-25-60 sind 0.35
    //bis 0.6 realistisch..
    frontend_bench.simplePump pump_1(redeclare package Medium = Water, m_flow_fixed = 0.35, use_m_flow_in = false) annotation(
      Placement(visible = true, transformation(origin = {-6, 220}, extent = {{6, 6}, {-6, -6}}, rotation = 0)));
    Modelica.Fluid.Sensors.Temperature t_sens1(redeclare package Medium = Water) annotation(
      Placement(visible = true, transformation(origin = {-6, 136}, extent = {{-8, -8}, {8, 8}}, rotation = 0)));
  initial equation

  equation
    connect(WT4.port_a2, WT4.port_a1) annotation(
      Line(points = {{-28, 40}, {-28, 40}, {-28, 28}, {-24, 28}, {-24, 40}, {-24, 40}}, color = {0, 127, 255}));
    connect(Filter.port_b1, WT6.port_a1) annotation(
      Line(points = {{-24, 208}, {-24, 208}, {-24, 230}, {12, 230}, {12, 230}}, color = {0, 127, 255}, thickness = 0.5));
    connect(t_sens1.port, WT51.port_a2) annotation(
      Line(points = {{-6, 128}, {-14, 128}, {-14, 102}, {-20, 102}, {-20, 108}, {-20, 108}}, color = {0, 127, 255}));
    connect(t_sens1.T, WT_medium_boundary.T_in) annotation(
      Line(points = {{0, 136}, {4, 136}, {4, 120}, {4, 120}}, color = {0, 0, 127}));
    connect(WT7.port_b2, to_Storage.ports[1]) annotation(
      Line(points = {{44, 156}, {44, 156}, {44, 192}, {90, 192}, {90, 192}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT7.port_a2, from_Storage.ports[1]) annotation(
      Line(points = {{44, 140}, {44, 140}, {44, 96}, {88, 96}, {88, 96}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT_medium_boundary.ports[1], WT7.port_a1) annotation(
      Line(points = {{8, 102}, {40, 102}, {40, 140}, {40, 140}, {40, 140}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT_medium_boundary.ports[2], WT51.port_a2) annotation(
      Line(points = {{8, 102}, {-20, 102}, {-20, 108}, {-20, 108}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT52.port_a2, WT51.port_b2) annotation(
      Line(points = {{-20, 148}, {-20, 148}, {-20, 124}, {-20, 124}}, color = {0, 127, 255}, thickness = 0.5));
    connect(Filter.port_a2, WT52.port_b2) annotation(
      Line(points = {{-20, 192}, {-20, 192}, {-20, 164}, {-20, 164}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT6.port_b2, WT7.port_b1) annotation(
      Line(points = {{28, 226}, {40, 226}, {40, 156}, {40, 156}, {40, 156}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT6.port_a2, pump_1.port_a) annotation(
      Line(points = {{12, 226}, {6, 226}, {6, 220}, {0, 220}, {0, 220}, {0, 220}}, color = {0, 127, 255}, thickness = 0.5));
    connect(Filter.port_b2, pump_1.port_b) annotation(
      Line(points = {{-20, 208}, {-20, 220}, {-12, 220}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT6.port_b1, Ambient_out.ports[1]) annotation(
      Line(points = {{28, 230}, {66, 230}, {66, 230}, {66, 230}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT52.port_b1, Filter.port_a1) annotation(
      Line(points = {{-24, 164}, {-24, 164}, {-24, 192}, {-24, 192}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT51.port_b1, WT52.port_a1) annotation(
      Line(points = {{-24, 124}, {-24, 124}, {-24, 148}, {-24, 148}}, color = {0, 127, 255}, thickness = 0.5));
    connect(WT4.port_b1, WT51.port_a1) annotation(
      Line(points = {{-24, 56}, {-24, 56}, {-24, 108}, {-24, 108}}, color = {0, 127, 255}, thickness = 0.5));
    connect(Air.ports[1], WT4.port_b2) annotation(
      Line(points = {{-64, 66}, {-28, 66}, {-28, 56}, {-28, 56}, {-28, 56}}, color = {0, 127, 255}, thickness = 0.5));
    annotation(
      Documentation(info = "<html>

</html>"),
      experiment(StopTime = 20000, StartTime = 0, Tolerance = 1e-4, Interval = 20),
      __OpenModelica_simulationFlags(jacobian = "coloredNumerical", s = "dassl", lv = "LOG_STATS"),
      Diagram(coordinateSystem(extent = {{-200, -200}, {200, 500}}, initialScale = 0.1)),
      Icon(coordinateSystem(extent = {{-200, -200}, {200, 500}})));
  end simple;







  model simplePump
    extends Modelica.Fluid.Interfaces.PartialTwoPortTransport;
    parameter Boolean use_m_flow_in = false "Get the mass flow rate from the input connector" annotation(
      Evaluate = true,
      HideResult = true,
      choices(checkBox = true));
    Modelica.Blocks.Interfaces.RealInput m_flow_in(final unit = "kg/s") if use_m_flow_in "Prescribed mass flow rate" annotation(
      Placement(transformation(extent = {{-20, -20}, {20, 20}}, rotation = -90, origin = {0, 120}), iconTransformation(extent = {{-20, -20}, {20, 20}}, rotation = -90, origin = {-2, 120})));
    parameter Medium.MassFlowRate m_flow_fixed = 0 "Fixed mass flow rate going out of the fluid port" annotation(
      Evaluate = true,
      Dialog(enable = not use_m_flow_in));
  equation
// remember: momentum-balance in steady state: 0 = Ib_flow - F_p - F_fg;
//momentum flow at boundaries
//Ib_flow = 0;
// pressure force
//F_p = 0;//(Medium.pressure(state_b) - Medium.pressure(state_a));
// friction and gravity forces
//F_fg = dp_fg;
//mass flow
    if use_m_flow_in then
      m_flow = m_flow_in;
    else
      m_flow = m_flow_fixed;
    end if;
//  m_flow = 0.001;
// Isenthalpic state transformation (no storage and no loss of energy)
    port_a.h_outflow = inStream(port_b.h_outflow);
    port_b.h_outflow = inStream(port_a.h_outflow);
    annotation(
      Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 46}, {100, -46}}, lineColor = {0, 0, 0}, fillColor = {0, 127, 255}, fillPattern = FillPattern.HorizontalCylinder), Polygon(points = {{-48, -60}, {-72, -100}, {72, -100}, {48, -60}, {-48, -60}}, lineColor = {0, 0, 255}, pattern = LinePattern.None, fillColor = {0, 0, 0}, fillPattern = FillPattern.VerticalCylinder), Ellipse(extent = {{-80, 80}, {80, -80}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.Sphere, fillColor = {0, 100, 199}), Polygon(points = {{-28, 30}, {-28, -30}, {50, -2}, {-28, 30}}, lineColor = {0, 0, 0}, pattern = LinePattern.None, fillPattern = FillPattern.HorizontalCylinder, fillColor = {255, 255, 255})}));
  end simplePump;

  model FlatDynamicPipe "Dynamic pipe model with storage of mass and energy"
    import Modelica.Fluid.Types.ModelStructure;
    import Modelica.Fluid.Types.Dynamics;
    import Modelica.Constants;
    import Modelica.Media.Interfaces.Choices.IndependentVariables;
    replaceable package Medium = Modelica.Media.Interfaces.PartialMedium "Medium in the component" annotation(
      choicesAllMatching = true);
    outer Modelica.Fluid.System system "System properties";
    /////Parameters
    parameter Boolean allowFlowReversal = system.allowFlowReversal "= true to allow flow reversal, false restricts to design direction (port_a -> port_b)";
    parameter Boolean port_a_exposesState = modelStructure == ModelStructure.av_b or modelStructure == ModelStructure.av_vb "= true if port_a exposes the state of a fluid volume";
    parameter Boolean port_b_exposesState = modelStructure == ModelStructure.av_b or modelStructure == ModelStructure.av_vb "= true if port_b.p exposes the state of a fluid volume";
    parameter Boolean showDesignFlowDirection = true "= false to hide the arrow in the model icon";
    // Geometry
    // Note: define nParallel as Real to support inverse calculations
    parameter Real nParallel(min = 1) = 1 "Number of identical parallel pipes";
    parameter Modelica.SIunits.Length length "Length";
    parameter Modelica.SIunits.Area crossArea = 3.14159 / 4 * diameter ^ 2 "Inner cross section area";
    parameter Modelica.SIunits.Length diameter = "diameter";
    parameter Modelica.SIunits.Height roughness = 2.5e-5 "Average height of surface asperities (default: smooth steel pipe)";
    parameter Modelica.SIunits.Volume V = crossArea * length * nParallel "volume size";
    parameter Modelica.SIunits.Length lengths = length / n "lengths of flow segments";
    parameter Modelica.SIunits.Length dimension = diameter "hydraulic diameters of flow segments";
    // Discretization
    parameter Integer nNodes(min = 1) = 2;
    parameter Integer n = nNodes "Number of discrete volumes";
    parameter Modelica.SIunits.Volume fluidVolumes = crossArea * lengths * nParallel "Discretized volume, determine in inheriting class";
    parameter ModelStructure modelStructure = ModelStructure.av_vb "Determines whether flow or volume models are present at the ports";
    parameter Integer nFM = if modelStructure == ModelStructure.a_v_b then n + 1 else if modelStructure == ModelStructure.a_vb or modelStructure == ModelStructure.av_b then n else n - 1 "number of flow models in flowModel";
    // Advanced model options
    parameter Boolean useInnerPortProperties = false "=true to take port properties for flow models from internal control volumes";
    parameter Boolean from_dp = momentumDynamics >= Modelica.Fluid.Types.Dynamics.SteadyStateInitial "= true, use m_flow = f(dp), otherwise dp = f(m_flow)";
    parameter Boolean useUpstreamScheme = true "= false to average upstream and downstream properties across flow segments";
    parameter Boolean use_Ib_flows = momentumDynamics <> Modelica.Fluid.Types.Dynamics.SteadyState "= true to consider differences in flow of momentum through boundaries";
    // Assumptions
    parameter Modelica.Fluid.Types.Dynamics energyDynamics = system.energyDynamics "Formulation of energy balances";
    parameter Modelica.Fluid.Types.Dynamics massDynamics = system.massDynamics "Formulation of mass balances";
    parameter Modelica.Fluid.Types.Dynamics substanceDynamics = massDynamics "Formulation of substance balances";
    parameter Modelica.Fluid.Types.Dynamics traceDynamics = massDynamics "Formulation of trace substance balances";
    parameter Modelica.Fluid.Types.Dynamics momentumDynamics = system.momentumDynamics "Formulation of momentum balances";
    //Initialization
    parameter Medium.AbsolutePressure p_a_start = system.p_start "Start value of pressure at port a";
    parameter Medium.AbsolutePressure p_b_start = p_a_start "Start value of pressure at port b";
    parameter Medium.AbsolutePressure[n] ps_start = if n > 1 then linspace(p_a_start, p_b_start, n) else {(p_a_start + p_b_start) / 2} "Start value of pressure";
    parameter Boolean use_T_start = true "Use T_start if true, otherwise h_start";
    parameter Medium.Temperature T_start = if use_T_start then system.T_start else Medium.temperature_phX((p_a_start + p_b_start) / 2, h_start, X_start) "Start value of temperature";
    parameter Medium.SpecificEnthalpy h_start = if use_T_start then Medium.specificEnthalpy_pTX((p_a_start + p_b_start) / 2, T_start, X_start) else Medium.h_default "Start value of specific enthalpy";
    parameter Medium.MassFraction X_start[Medium.nX] = Medium.X_default "Start value of mass fractions m_i/m";
    parameter Medium.ExtraProperty C_start[Medium.nC](quantity = Medium.extraPropertiesNames) = fill(0, Medium.nC) "Start value of trace substances";
    parameter Boolean initialize_p = not Medium.singleState "= true to set up initial equations for pressure";
    parameter Medium.MassFlowRate m_flow_start = system.m_flow_start "Start value for mass flow rate";
    parameter Modelica.SIunits.ReynoldsNumber Re_turbulent = 2300 "Start of turbulent regime, depending on type of flow device";
    parameter Boolean show_Res = false "= true, if Reynolds numbers are included for plotting";
    parameter Boolean use_rho_nominal = false "= true, if rho_nominal is used, otherwise computed from medium";
    parameter Modelica.SIunits.Density rho_nominal = Medium.density_pTX(Medium.p_default, Medium.T_default, Medium.X_default) "Nominal density (e.g., rho_liquidWater = 995, rho_air = 1.2)";
    parameter Boolean use_mu_nominal = false "= true, if mu_nominal is used, otherwise computed from medium";
    parameter Modelica.SIunits.DynamicViscosity mu_nominal = Medium.dynamicViscosity(Medium.setState_pTX(Medium.p_default, Medium.T_default, Medium.X_default)) "Nominal dynamic viscosity (e.g., mu_liquidWater = 1e-3, mu_air = 1.8e-5)";
    parameter Modelica.SIunits.AbsolutePressure dp_nominal = 1 "Nominal pressure loss (only for nominal models)";
    parameter Modelica.SIunits.MassFlowRate m_flow_nominal = 1e-3 "Nominal mass flow rate";
    parameter Modelica.SIunits.MassFlowRate m_flow_small = if system.use_eps_Re then system.eps_m_flow * m_flow_nominal else system.m_flow_small "Within regularization if |m_flowsFM| < m_flow_small (may be wider for large discontinuities in static head)";
    parameter Boolean constantPressureLossCoefficient = use_rho_nominal and (use_mu_nominal or not WallFriction.use_mu) "= true if the pressure loss does not depend on fluid states";
    parameter Boolean continuousFlowReversal = not useUpstreamScheme or constantPressureLossCoefficient or not allowFlowReversal "= true if the pressure loss is continuous around zero flow";
    parameter Real areaFactor = 1 "Factor for modification of heat transfering area";
    parameter Modelica.SIunits.Area surfaceAreas = diameter * 3.14159 * lengths * areaFactor "Heat transfer areas";
    //parameter Modelica.SIunits.CoefficientOfHeatTransfer k = 0 "Heat transfer coefficient to ambient";
    parameter Modelica.SIunits.Temperature T_ambient = system.T_ambient "Ambient temperature";
    parameter Modelica.SIunits.CoefficientOfHeatTransfer alpha0 = 100 "guess value for heat transfer coefficients";
    parameter Real dxs = lengths / length;
    parameter Real eps = 0.9 "radiation emissivity for surface1 and surface2";
    //////Variables
    Modelica.Fluid.Interfaces.FluidPort_a port_a(redeclare package Medium = Medium, m_flow(min = if allowFlowReversal then -Constants.inf else 0)) "Fluid connector a (positive design flow direction is from port_a to port_b)" annotation(
      Placement(transformation(extent = {{-110, -10}, {-90, 10}}, rotation = 0)));
    Modelica.Fluid.Interfaces.FluidPort_b port_b(redeclare package Medium = Medium, m_flow(max = if allowFlowReversal then +Constants.inf else 0)) "Fluid connector b (positive design flow direction is from port_a to port_b)" annotation(
      Placement(transformation(extent = {{110, -10}, {90, 10}}, rotation = 0), iconTransformation(extent = {{110, -10}, {90, 10}})));
    replaceable package WallFriction = Modelica.Fluid.Pipes.BaseClasses.WallFriction.Detailed constrainedby Modelica.Fluid.Pipes.BaseClasses.WallFriction.PartialWallFriction "Wall friction model";
    // Total quantities
    Modelica.SIunits.Energy[n] Us "Internal energy of fluid";
    Modelica.SIunits.Mass[n] ms "Fluid mass";
    Modelica.SIunits.Mass[n, Medium.nXi] mXis "Substance mass";
    Modelica.SIunits.Mass[n, Medium.nC] mCs "Trace substance mass";
    // C need to be added here because unlike for Xi, which has medium[:].Xi,
    // there is no variable medium[:].C
    Modelica.SIunits.Mass[n, Medium.nC] mCs_scaled "Scaled trace substance mass";
    Medium.ExtraProperty Cs[n, Medium.nC] "Trace substance mixture content";
    Medium.BaseProperties[n] mediums(each preferredMediumStates = true, p(start = ps_start), each h(start = h_start), each T(start = T_start), each Xi(start = X_start[1:Medium.nXi]));
    //Source terms, have to be defined by an extending model (to zero if not used)
    Medium.MassFlowRate[n] mb_flows "Mass flow rate, source or sink";
    Medium.MassFlowRate[n, Medium.nXi] mbXi_flows "Independent mass flow rates, source or sink";
    Medium.ExtraPropertyFlowRate[n, Medium.nC] mbC_flows "Trace substance mass flow rates, source or sink";
    Modelica.SIunits.EnthalpyFlowRate[n] Hb_flows "Enthalpy flow rate, source or sink";
    Modelica.SIunits.HeatFlowRate[n] Qb_flows "Heat flow rate, source or sink";
    Modelica.SIunits.Power[n] Wb_flows "Mechanical power, p*der(V) etc.";
    Modelica.SIunits.Force[nFM + 1 - 1] Ib_flows "Flow of momentum across boundaries";
    Modelica.SIunits.Force[nFM + 1 - 1] Fs_p "Pressure forces";
    Modelica.SIunits.Force[nFM + 1 - 1] Fs_fg "Friction and gravity forces";
    //
    Medium.ThermodynamicState state_a "state defined by volume outside port_a";
    Medium.ThermodynamicState state_b "state defined by volume outside port_b";
    Medium.ThermodynamicState[nFM + 1] statesFM "state vector for flowModel model";
    Medium.MassFlowRate[nFM + 1 - 1] m_flowsFM(each min = if allowFlowReversal then -Modelica.Constants.inf else 0, each start = m_flow_start, each stateSelect = if momentumDynamics == Modelica.Fluid.Types.Dynamics.SteadyState then StateSelect.default else StateSelect.prefer) "mass flow rates between states";
    Modelica.SIunits.Momentum[nFM + 1 - 1] Is "Momenta of flow segments";
    Medium.Density[nFM + 1] rhos = if use_rho_nominal then fill(rho_nominal, nFM + 1) else Medium.density(statesFM);
    Medium.Density[nFM + 1 - 1] rhos_act "Actual density per segment";
    Medium.DynamicViscosity[nFM + 1] musFM = if use_mu_nominal then fill(mu_nominal, nFM + 1) else Medium.dynamicViscosity(statesFM);
    Medium.DynamicViscosity[nFM + 1 - 1] mus_act "Actual viscosity per segment";
    Modelica.SIunits.Pressure[nFM + 1 - 1] dps_fg(each start = (p_a_start - p_b_start) / (nFM + 1 - 1)) "pressure drop between states";
    Modelica.SIunits.ReynoldsNumber[nFM + 1] ResFM = Modelica.Fluid.Pipes.BaseClasses.CharacteristicNumbers.ReynoldsNumber(vsFM, rhos, musFM, dimension) if show_Res "Reynolds numbers";
    Medium.MassFlowRate[nFM + 1 - 1] m_flows_turbulent = {nParallel * (crossArea + crossArea) / (dimension + dimension) * mus_act[i] * Re_turbulent for i in 1:nFM + 1 - 1} if show_Res "Start of turbulent flow";
    Modelica.SIunits.Length[nFM + 1 - 1] pathLengths_internal = pathLengths "pathLengths used internally; to be defined by extending class";
    Modelica.SIunits.ReynoldsNumber[nFM + 1 - 1] Res_turbulent_internal = Re_turbulent * ones(nFM + 1 - 1) "Re_turbulent used internally";
    Modelica.SIunits.AbsolutePressure dp_fric_nominal = sum(WallFriction.pressureLoss_m_flow(m_flow_nominal / nParallel, rho_nominal, rho_nominal, mu_nominal, mu_nominal, pathLengths_internal, dimension, crossArea, roughness, m_flow_small / nParallel, Res_turbulent_internal)) "pressure loss for nominal conditions";
    // Flow quantities
    Medium.MassFlowRate[n + 1] m_flows(each min = if allowFlowReversal then -Modelica.Constants.inf else 0, each start = m_flow_start) "Mass flow rates of fluid across segment boundaries";
    Medium.MassFlowRate[n + 1, Medium.nXi] mXi_flows "Independent mass flow rates across segment boundaries";
    Medium.MassFlowRate[n + 1, Medium.nC] mC_flows "Trace substance mass flow rates across segment boundaries";
    Medium.EnthalpyFlowRate[n + 1] H_flows "Enthalpy flow rates of fluid across segment boundaries";
    Modelica.SIunits.Velocity[n] vs = {0.5 * (m_flows[i] + m_flows[i + 1]) / mediums[i].d / crossArea for i in 1:n} / nParallel "mean velocities in flow segments";
    // Model structure dependent flow geometry
    Modelica.SIunits.Length[nFM] pathLengths "Lengths along flow path";
    Modelica.SIunits.Velocity[nFM + 1] vsFM "Mean velocities in flow segments";
    Modelica.SIunits.HeatFlowRate[n] Q_flows "Heat flow rates";
    Modelica.Fluid.Interfaces.HeatPorts_a[n] heatPorts "Heat port to component boundary" annotation(
      Placement(transformation(extent = {{-10, 45}, {10, 65}}), iconTransformation(extent = {{-30, 36}, {32, 52}})));
    Modelica.SIunits.Temperature[n] Ts = Medium.temperature(mediums.state) "Temperatures defined by fluid states";
    Modelica.SIunits.CoefficientOfHeatTransfer[n] alphas(each start = alpha0) "CoefficientOfHeatTransfer";
    Real[n] Res "Reynolds numbers";
    Real[n] Prs "Prandtl numbers";
    Real[n] Nus "Nusselt numbers";
    Medium.Density[n] ds "Densities";
    Medium.DynamicViscosity[n] mus "Dynamic viscosities";
    Medium.ThermalConductivity[n] lambdas "Thermal conductivity";
    //  Modelica.SIunits.Length diameters[n] = dimension "Hydraulic diameters for pipe flow";
    Real[n] Nus_turb "Nusselt number for turbulent flow";
    Real[n] Nus_lam "Nusselt number for laminar flow";
    Real Nu_1;
    Real[n] Nus_2;
    Real[n] Xis;
  equation
    Qb_flows = Q_flows;
    if modelStructure == ModelStructure.av_vb or modelStructure == ModelStructure.av_b then
      Wb_flows[2:n - 1] = {vs[i] * crossArea * ((mediums[i + 1].p - mediums[i - 1].p) / 2 + (dps_fg[i - 1] + dps_fg[i]) / 2) for i in 2:n - 1} * nParallel;
    else
      Wb_flows[2:n - 1] = {vs[i] * crossArea * ((mediums[i + 1].p - mediums[i - 1].p) / 2 + (dps_fg[i] + dps_fg[i + 1]) / 2) for i in 2:n - 1} * nParallel;
    end if;
    if modelStructure == ModelStructure.av_vb then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - mediums[1].p) / 2 + dps_fg[1] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((mediums[n].p - mediums[n - 1].p) / 2 + dps_fg[n - 1] / 2) * nParallel;
    elseif modelStructure == ModelStructure.av_b then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - mediums[1].p) / 2 + dps_fg[1] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((port_b.p - mediums[n - 1].p) / 1.5 + dps_fg[n - 1] / 2 + dps_fg[n]) * nParallel;
    elseif modelStructure == ModelStructure.a_vb then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - port_a.p) / 1.5 + dps_fg[1] + dps_fg[2] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((mediums[n].p - mediums[n - 1].p) / 2 + dps_fg[n] / 2) * nParallel;
    elseif modelStructure == ModelStructure.a_v_b then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - port_a.p) / 1.5 + dps_fg[1] + dps_fg[2] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((port_b.p - mediums[n - 1].p) / 1.5 + dps_fg[n] / 2 + dps_fg[n + 1]) * nParallel;
    else
      assert(false, "Unknown model structure");
    end if;
//from PartialHeatTransfer
    Q_flows = heatPorts.Q_flow;
    assert(nNodes > 1 or modelStructure <> ModelStructure.av_vb, "nNodes needs to be at least 2 for modelStructure av_vb, as flow model disappears otherwise!");
  if modelStructure == ModelStructure.av_vb then
//nFM = n-1
      if n == 2 then
        pathLengths[1] = lengths + lengths;
      else
        pathLengths[1:n - 1] = cat(1, {lengths + 0.5 * lengths}, fill(lengths, n - 3), {0.5 * lengths + lengths});
      end if;
    elseif modelStructure == ModelStructure.av_b then
//nFM = n
      pathLengths[1:n] = fill(lengths, n);
    elseif modelStructure == ModelStructure.a_vb then
//nFM = n
      pathLengths[1:n] = fill(lengths, n);
    elseif modelStructure == ModelStructure.a_v_b then
//nFM = n+1;
      pathLengths[1:n + 1] = cat(1, {0.5 * lengths}, fill(lengths, n - 1), {0.5 * lengths});
    else
      assert(false, "Unknown model structure");
    end if;
// Source/sink terms for mass and energy balances
    for i in 1:n loop
      mb_flows[i] = m_flows[i] - m_flows[i + 1];
      mbXi_flows[i, :] = mXi_flows[i, :] - mXi_flows[i + 1, :];
      mbC_flows[i, :] = mC_flows[i, :] - mC_flows[i + 1, :];
      Hb_flows[i] = H_flows[i] - H_flows[i + 1];
    end for;
// Distributed flow quantities, upwind discretization
    for i in 2:n loop
      H_flows[i] = semiLinear(m_flows[i], mediums[i - 1].h, mediums[i].h);
      mXi_flows[i, :] = semiLinear(m_flows[i], mediums[i - 1].Xi, mediums[i].Xi);
      mC_flows[i, :] = semiLinear(m_flows[i], Cs[i - 1, :], Cs[i, :]);
    end for;
    H_flows[1] = semiLinear(port_a.m_flow, inStream(port_a.h_outflow), mediums[1].h);
    H_flows[n + 1] = -semiLinear(port_b.m_flow, inStream(port_b.h_outflow), mediums[n].h);
    mXi_flows[1, :] = semiLinear(port_a.m_flow, inStream(port_a.Xi_outflow), mediums[1].Xi);
    mXi_flows[n + 1, :] = -semiLinear(port_b.m_flow, inStream(port_b.Xi_outflow), mediums[n].Xi);
    mC_flows[1, :] = semiLinear(port_a.m_flow, inStream(port_a.C_outflow), Cs[1, :]);
    mC_flows[n + 1, :] = -semiLinear(port_b.m_flow, inStream(port_b.C_outflow), Cs[n, :]);
// Boundary conditions
    port_a.m_flow = m_flows[1];
    port_b.m_flow = -m_flows[n + 1];
    port_a.h_outflow = mediums[1].h;
    port_b.h_outflow = mediums[n].h;
    port_a.Xi_outflow = mediums[1].Xi;
    port_b.Xi_outflow = mediums[n].Xi;
    port_a.C_outflow = Cs[1, :];
    port_b.C_outflow = Cs[n, :];
// The two equations below are not correct if C is stored in volumes.
// C should be treated the same way as Xi.
//port_a.C_outflow = inStream(port_b.C_outflow);
//port_b.C_outflow = inStream(port_a.C_outflow);
    if useInnerPortProperties and n > 0 then
      state_a = Medium.setState_phX(port_a.p, mediums[1].h, mediums[1].Xi);
      state_b = Medium.setState_phX(port_b.p, mediums[n].h, mediums[n].Xi);
    else
      state_a = Medium.setState_phX(port_a.p, inStream(port_a.h_outflow), inStream(port_a.Xi_outflow));
      state_b = Medium.setState_phX(port_b.p, inStream(port_b.h_outflow), inStream(port_b.Xi_outflow));
    end if;
// staggered grid discretization for flowModel, depending on modelStructure
    if modelStructure == ModelStructure.av_vb then
//nFM = n-1
      statesFM[1:n] = mediums[1:n].state;
      m_flows[2:n] = m_flowsFM[1:n - 1];
      vsFM[1:n] = vs;
      port_a.p = mediums[1].p;
      port_b.p = mediums[n].p;
    elseif modelStructure == ModelStructure.av_b then
//nFM = n
      statesFM[1:n] = mediums[1:n].state;
      statesFM[n + 1] = state_b;
      m_flows[2:n + 1] = m_flowsFM[1:n];
      vsFM[1:n] = vs;
      vsFM[n + 1] = m_flows[n + 1] / Medium.density(state_b) / crossArea / nParallel;
      port_a.p = mediums[1].p;
    elseif modelStructure == ModelStructure.a_vb then
//nFM = n
      statesFM[1] = state_a;
      statesFM[2:n + 1] = mediums[1:n].state;
      m_flows[1:n] = m_flowsFM[1:n];
      vsFM[1] = m_flows[1] / Medium.density(state_a) / crossArea / nParallel;
      vsFM[2:n + 1] = vs;
      port_b.p = mediums[n].p;
    elseif modelStructure == ModelStructure.a_v_b then
//nFM = n+1
      statesFM[1] = state_a;
      statesFM[2:n + 1] = mediums[1:n].state;
      statesFM[n + 2] = state_b;
      m_flows[1:n + 1] = m_flowsFM[1:n + 1];
      vsFM[1] = m_flows[1] / Medium.density(state_a) / crossArea / nParallel;
      vsFM[2:n + 1] = vs;
      vsFM[n + 2] = m_flows[n + 1] / Medium.density(state_b) / crossArea / nParallel;
    else
      assert(false, "Unknown model structure");
    end if;
    assert(not (energyDynamics <> Dynamics.SteadyState and massDynamics == Dynamics.SteadyState) or Medium.singleState, "Bad combination of dynamics options and Medium not conserving mass if fluidVolumes are fixed.");
// Total quantities
    for i in 1:n loop
      ms[i] = fluidVolumes * mediums[i].d;
      mXis[i, :] = ms[i] * mediums[i].Xi;
      mCs[i, :] = ms[i] * Cs[i, :];
      Us[i] = ms[i] * mediums[i].u;
    end for;
// Energy and mass balances
    if energyDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        0 = Hb_flows[i] + Wb_flows[i] + Qb_flows[i];
      end for;
    else
      for i in 1:n loop
        der(Us[i]) = Hb_flows[i] + Wb_flows[i] + Qb_flows[i];
      end for;
    end if;
    if massDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        0 = mb_flows[i];
      end for;
    else
      for i in 1:n loop
        der(ms[i]) = mb_flows[i];
      end for;
    end if;
    if substanceDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        zeros(Medium.nXi) = mbXi_flows[i, :];
      end for;
    else
      for i in 1:n loop
        der(mXis[i, :]) = mbXi_flows[i, :];
      end for;
    end if;
    if traceDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        zeros(Medium.nC) = mbC_flows[i, :];
      end for;
    else
      for i in 1:n loop
        der(mCs_scaled[i, :]) = mbC_flows[i, :] ./ Medium.C_nominal;
        mCs[i, :] = mCs_scaled[i, :] .* Medium.C_nominal;
      end for;
    end if;
//equations for heat transfer (from localPipeFlowHeatTransfer)
    Nu_1 = 3.66;
    for i in 1:n loop
      Nus_turb[i] = smooth(0, Xis[i] / 8 * abs(Res[i]) * Prs[i] / (1 + 12.7 * (Xis[i] / 8) ^ 0.5 * (Prs[i] ^ (2 / 3) - 1)) * (1 + 1 / 3 * (dimension / lengths / (if vs[i] >= 0 then i - 0.5 else n - i + 0.5)) ^ (2 / 3)));
      Xis[i] = (1.8 * Modelica.Math.log10(max(1e-10, Res[i])) - 1.5) ^ (-2);
      Nus_lam[i] = (Nu_1 ^ 3 + 0.7 ^ 3 + (Nus_2[i] - 0.7) ^ 3) ^ (1 / 3);
      Nus_2[i] = smooth(0, 1.077 * (abs(Res[i]) * Prs[i] * dimension / lengths / (if vs[i] >= 0 then i - 0.5 else n - i + 0.5)) ^ (1 / 3));
      Nus[i] = Modelica.Media.Air.MoistAir.Utilities.spliceFunction(Nus_turb[i], Nus_lam[i], Res[i] - 6150, 3850);
    end for;
//from PartialPipeFlowHeatTransfer
    ds = Medium.density(mediums.state);
    mus = Medium.dynamicViscosity(mediums.state);
    lambdas = Medium.thermalConductivity(mediums.state);
    Prs = Medium.prandtlNumber(mediums.state);
    for i in 1:n loop
      Res[i] = abs(vs[i]) * ds[i] * dimension / mus[i];
      Nus[i] = alphas[i] * dimension / lambdas[i];
      Q_flows[i] = alphas[i] * surfaceAreas * (heatPorts[i].T - Ts[i]) * nParallel;
    end for;
//from partialGenericPipeFlow
    for i in 1:nFM + 1 - 1 loop
      assert(m_flowsFM[i] > (-m_flow_small) or allowFlowReversal, "Reverting flow occurs even though allowFlowReversal is false");
    end for;
  if continuousFlowReversal then
// simple regularization
      if from_dp and not WallFriction.dp_is_zero then
        m_flowsFM = homotopy(actual = WallFriction.massFlowRate_dp(dps_fg, rhos_act, rhos_act, mus_act, mus_act, pathLengths_internal, dimension, crossArea, roughness, system.dp_small / (nFM + 1 - 1), Res_turbulent_internal) * nParallel, simplified = m_flow_nominal / dp_nominal * dps_fg);
      else
        dps_fg = homotopy(actual = WallFriction.pressureLoss_m_flow(m_flowsFM / nParallel, rhos_act, rhos_act, mus_act, mus_act, pathLengths_internal, dimension, crossArea, roughness, m_flow_small / nParallel, Res_turbulent_internal), simplified = dp_nominal / m_flow_nominal * m_flowsFM);
      end if;
    else
// regularization for discontinuous flow reversal and static head
      if from_dp and not WallFriction.dp_is_zero then
        m_flowsFM = homotopy(actual = WallFriction.massFlowRate_dp_staticHead(dps_fg, rhos[1:nFM + 1 - 1], rhos[2:nFM + 1], musFM[1:nFM + 1 - 1], musFM[2:nFM + 1], pathLengths_internal, dimension, 0, crossArea, roughness, system.dp_small / (nFM + 1 - 1), Res_turbulent_internal) * nParallel, simplified = m_flow_nominal / dp_nominal * dps_fg);
      else
        dps_fg = homotopy(actual = WallFriction.pressureLoss_m_flow_staticHead(m_flowsFM / nParallel, rhos[1:nFM + 1 - 1], rhos[2:nFM + 1], musFM[1:nFM + 1 - 1], musFM[2:nFM + 1], pathLengths_internal, dimension, 0, crossArea, roughness, m_flow_small / nParallel, Res_turbulent_internal), simplified = dp_nominal / m_flow_nominal * m_flowsFM);
      end if;
    end if;
//from PartialStaggeredFlowModel
    if not allowFlowReversal then
      rhos_act = rhos[1:nFM + 1 - 1];
      mus_act = musFM[1:nFM + 1 - 1];
    elseif not useUpstreamScheme then
      rhos_act = 0.5 * (rhos[1:nFM + 1 - 1] + rhos[2:nFM + 1]);
      mus_act = 0.5 * (musFM[1:nFM + 1 - 1] + musFM[2:nFM + 1]);
    else
      for i in 1:nFM + 1 - 1 loop
        rhos_act[i] = noEvent(if m_flowsFM[i] > 0 then rhos[i] else rhos[i + 1]);
        mus_act[i] = noEvent(if m_flowsFM[i] > 0 then musFM[i] else musFM[i + 1]);
      end for;
    end if;
  if use_Ib_flows then
      Ib_flows = nParallel * {rhos[i] * vsFM[i] * vsFM[i] * crossArea - rhos[i + 1] * vsFM[i + 1] * vsFM[i + 1] * crossArea for i in 1:nFM + 1 - 1};
// alternatively use densities rhos_act of actual streams, together with mass flow rates,
// not conserving momentum if fluid density changes between flow segments:
//Ib_flows = {((rhos[i]*vsFM[i])^2*crossAreasFM[i] - (rhos[i+1]*vsFM[i+1])^2*crossAreasFM[i+1])/rhos_act[i] for i in 1:nFM+1-1};
    else
      Ib_flows = zeros(nFM + 1 - 1);
    end if;
    Fs_p = nParallel * {0.5 * (crossArea + crossArea) * (Medium.pressure(statesFM[i + 1]) - Medium.pressure(statesFM[i])) for i in 1:nFM + 1 - 1};
// Note: the equation is written for dps_fg instead of Fs_fg to help the translator
    dps_fg = {Fs_fg[i] / nParallel * 2 / (crossArea + crossArea) for i in 1:nFM + 1 - 1};
//from PartialDistributedFlow
// Total quantities
    Is = {m_flowsFM[i] * pathLengths[i] for i in 1:nFM + 1 - 1};
// Momentum balances
    if momentumDynamics == Modelica.Fluid.Types.Dynamics.SteadyState then
      zeros(nFM + 1 - 1) = Ib_flows - Fs_p - Fs_fg;
    else
      der(Is) = Ib_flows - Fs_p - Fs_fg;
    end if;
/////////////////////////////////////////////////////////////
  initial equation
// initialization of balances
    if energyDynamics == Dynamics.FixedInitial then
      if use_T_start then
        mediums.T = fill(T_start, n);
      else
        mediums.h = fill(h_start, n);
      end if;
      if Medium.ThermoStates == IndependentVariables.ph or Medium.ThermoStates == IndependentVariables.phX then
        mediums.h = fill(h_start, n);
      else
        mediums.T = fill(T_start, n);
      end if;
    elseif energyDynamics == Dynamics.SteadyStateInitial then
      if use_T_start then
        der(mediums.T) = zeros(n);
      else
        der(mediums.h) = zeros(n);
      end if;
      if Medium.ThermoStates == IndependentVariables.ph or Medium.ThermoStates == IndependentVariables.phX then
        der(mediums.h) = zeros(n);
      else
        der(mediums.T) = zeros(n);
      end if;
    end if;
    if massDynamics == Dynamics.FixedInitial then
      if initialize_p then
        mediums.p = ps_start;
      end if;
    elseif massDynamics == Dynamics.SteadyStateInitial then
      if initialize_p then
        der(mediums.p) = zeros(n);
      end if;
    end if;
    if substanceDynamics == Dynamics.FixedInitial then
      mediums.Xi = fill(X_start[1:Medium.nXi], n);
    elseif substanceDynamics == Dynamics.SteadyStateInitial then
      for i in 1:n loop
        der(mediums[i].Xi) = zeros(Medium.nXi);
      end for;
    end if;
    if traceDynamics == Dynamics.FixedInitial then
      Cs = fill(C_start[1:Medium.nC], n);
    elseif traceDynamics == Dynamics.SteadyStateInitial then
      for i in 1:n loop
        der(mCs[i, :]) = zeros(Medium.nC);
      end for;
    end if;
//from detailedPipeFlow
// initialize dp_nominal from flow model
/*
    if system.use_eps_Re then
      dp_nominal = dp_fric_nominal + system.g * sum(dheightsFM) * rho_nominal;
    else
      dp_nominal = 1e3 * system.dp_small;
    end if;
  */
//from partialGenericPipeFlow
// initialize dp_small from flow model
/*
    if system.use_eps_Re then
      dp_small = dp_fric_nominal / m_flow_nominal * m_flow_small;
    else
      dp_small = system.dp_small;
    end if;
  */
//from PartialDistributedFlow
    if momentumDynamics == Modelica.Fluid.Types.Dynamics.FixedInitial then
      m_flowsFM = fill(m_flow_start, nFM + 1 - 1);
    elseif momentumDynamics == Modelica.Fluid.Types.Dynamics.SteadyStateInitial then
      der(m_flowsFM) = zeros(nFM + 1 - 1);
    end if;
///////end initial eqautions//////////////////////////////////////////////
    annotation(
      defaultComponentName = "pipe",
      Documentation(info = "<html>
  <p>Rewrite of the original DynamicPipe class to provide the possibility of radiation between two heatports and give a better performance of model translation due to mostly flat design. It always uses detailed pipe flow and detailed heat transfer models. Some options were removed (e.g. use lumped pressure, circular design, gravitational influence (different heights),).</p>
  </html>"),
      Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 44}, {100, -44}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {0, 127, 255}), Ellipse(extent = {{-72, 10}, {-52, -10}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{50, 10}, {70, -10}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-48, 15}, {46, -20}}, lineColor = {0, 0, 0}, textString = "%nNodes")}),
      Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 60}, {100, 50}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Backward), Rectangle(extent = {{-100, -50}, {100, -60}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Backward), Line(points = {{100, 45}, {100, 50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{0, 45}, {0, 50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{100, -45}, {100, -50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{0, -45}, {0, -50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{-50, 60}, {-50, 50}}, smooth = Smooth.None, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{50, 60}, {50, 50}}, smooth = Smooth.None, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{0, -50}, {0, -60}}, smooth = Smooth.None, color = {0, 0, 0}, pattern = LinePattern.Dot)}));
  end FlatDynamicPipe;

  model FlatDynamicGap "Dynamic cylindrical gap model with storage of mass and energy"
    import Modelica.Fluid.Types.ModelStructure;
    import Modelica.Fluid.Types.Dynamics;
    import Modelica.Constants;
    import Modelica.Media.Interfaces.Choices.IndependentVariables;
    replaceable package Medium = Modelica.Media.Interfaces.PartialMedium "Medium in the component" annotation(
      choicesAllMatching = true);
    outer Modelica.Fluid.System system "System properties";
    /////Parameters
    parameter Boolean allowFlowReversal = system.allowFlowReversal "= true to allow flow reversal, false restricts to design direction (port_a -> port_b)";
    parameter Boolean port_a_exposesState = modelStructure == ModelStructure.av_b or modelStructure == ModelStructure.av_vb "= true if port_a exposes the state of a fluid volume";
    parameter Boolean port_b_exposesState = modelStructure == ModelStructure.av_b or modelStructure == ModelStructure.av_vb "= true if port_b.p exposes the state of a fluid volume";
    parameter Boolean showDesignFlowDirection = true "= false to hide the arrow in the model icon";
    // Geometry
    // Note: define nParallel as Real to support inverse calculations
    parameter Real nParallel(min = 1) = 1 "Number of identical parallel pipes";
    parameter Modelica.SIunits.Length length "Length";
    parameter Modelica.SIunits.Area crossArea = 3.14159 / 4 * (diameter_a ^ 2 - diameter_i ^ 2) "Inner cross section area";
    parameter Modelica.SIunits.Length diameter_i = "Inner diameter";
    parameter Modelica.SIunits.Length diameter_a = "Outer diameter";
    parameter Modelica.SIunits.Height roughness = 2.5e-5 "Average height of surface asperities (default: smooth steel pipe)";
    parameter Modelica.SIunits.Volume V = crossArea * length * nParallel "volume size";
    parameter Modelica.SIunits.Length lengths = length / n "lengths of flow segments";
    parameter Modelica.SIunits.Length dimension = diameter_a - diameter_i "hydraulic diameters of flow segments";
    // Discretization
    parameter Integer nNodes(min = 1) = 2;
    parameter Integer n = nNodes "Number of discrete volumes";
    parameter Modelica.SIunits.Volume fluidVolumes = crossArea * lengths * nParallel "Discretized volume, determine in inheriting class";
    parameter ModelStructure modelStructure = ModelStructure.av_vb "Determines whether flow or volume models are present at the ports";
    parameter Integer nFM = if modelStructure == ModelStructure.a_v_b then n + 1 else if modelStructure == ModelStructure.a_vb or modelStructure == ModelStructure.av_b then n else n - 1 "number of flow models in flowModel";
    // Advanced model options
    parameter Boolean useInnerPortProperties = false "=true to take port properties for flow models from internal control volumes";
    parameter Boolean from_dp = momentumDynamics >= Modelica.Fluid.Types.Dynamics.SteadyStateInitial "= true, use m_flow = f(dp), otherwise dp = f(m_flow)";
    parameter Boolean useUpstreamScheme = true "= false to average upstream and downstream properties across flow segments";
    parameter Boolean use_Ib_flows = momentumDynamics <> Modelica.Fluid.Types.Dynamics.SteadyState "= true to consider differences in flow of momentum through boundaries";
    // Assumptions
    parameter Modelica.Fluid.Types.Dynamics energyDynamics = system.energyDynamics "Formulation of energy balances";
    parameter Modelica.Fluid.Types.Dynamics massDynamics = system.massDynamics "Formulation of mass balances";
    parameter Modelica.Fluid.Types.Dynamics substanceDynamics = massDynamics "Formulation of substance balances";
    parameter Modelica.Fluid.Types.Dynamics traceDynamics = massDynamics "Formulation of trace substance balances";
    parameter Modelica.Fluid.Types.Dynamics momentumDynamics = system.momentumDynamics "Formulation of momentum balances";
    //Initialization
    parameter Medium.AbsolutePressure p_a_start = system.p_start "Start value of pressure at port a";
    parameter Medium.AbsolutePressure p_b_start = p_a_start "Start value of pressure at port b";
    parameter Medium.AbsolutePressure[n] ps_start = if n > 1 then linspace(p_a_start, p_b_start, n) else {(p_a_start + p_b_start) / 2} "Start value of pressure";
    parameter Boolean use_T_start = true "Use T_start if true, otherwise h_start";
    parameter Medium.Temperature T_start = if use_T_start then system.T_start else Medium.temperature_phX((p_a_start + p_b_start) / 2, h_start, X_start) "Start value of temperature";
    parameter Medium.SpecificEnthalpy h_start = if use_T_start then Medium.specificEnthalpy_pTX((p_a_start + p_b_start) / 2, T_start, X_start) else Medium.h_default "Start value of specific enthalpy";
    parameter Medium.MassFraction X_start[Medium.nX] = Medium.X_default "Start value of mass fractions m_i/m";
    parameter Medium.ExtraProperty C_start[Medium.nC](quantity = Medium.extraPropertiesNames) = fill(0, Medium.nC) "Start value of trace substances";
    parameter Boolean initialize_p = not Medium.singleState "= true to set up initial equations for pressure";
    parameter Medium.MassFlowRate m_flow_start = system.m_flow_start "Start value for mass flow rate";
    parameter Modelica.SIunits.ReynoldsNumber Re_turbulent = 2300 "Start of turbulent regime, depending on type of flow device";
    parameter Boolean show_Res = false "= true, if Reynolds numbers are included for plotting";
    parameter Boolean use_rho_nominal = false "= true, if rho_nominal is used, otherwise computed from medium";
    parameter Modelica.SIunits.Density rho_nominal = Medium.density_pTX(Medium.p_default, Medium.T_default, Medium.X_default) "Nominal density (e.g., rho_liquidWater = 995, rho_air = 1.2)";
    parameter Boolean use_mu_nominal = false "= true, if mu_nominal is used, otherwise computed from medium";
    parameter Modelica.SIunits.DynamicViscosity mu_nominal = Medium.dynamicViscosity(Medium.setState_pTX(Medium.p_default, Medium.T_default, Medium.X_default)) "Nominal dynamic viscosity (e.g., mu_liquidWater = 1e-3, mu_air = 1.8e-5)";
    parameter Modelica.SIunits.AbsolutePressure dp_nominal = 1 "Nominal pressure loss (only for nominal models)";
    parameter Modelica.SIunits.MassFlowRate m_flow_nominal = 1e-3 "Nominal mass flow rate";
    parameter Modelica.SIunits.MassFlowRate m_flow_small = if system.use_eps_Re then system.eps_m_flow * m_flow_nominal else system.m_flow_small "Within regularization if |m_flowsFM| < m_flow_small (may be wider for large discontinuities in static head)";
    parameter Boolean constantPressureLossCoefficient = use_rho_nominal and (use_mu_nominal or not WallFriction.use_mu) "= true if the pressure loss does not depend on fluid states";
    parameter Boolean continuousFlowReversal = not useUpstreamScheme or constantPressureLossCoefficient or not allowFlowReversal "= true if the pressure loss is continuous around zero flow";
    parameter Real areaFactor = 1 "Factor for modification of heat transfering area";
    parameter Modelica.SIunits.Area surfaceAreas1 = diameter_i * 3.14159 * lengths * areaFactor "Heat transfer areas";
    parameter Modelica.SIunits.Area surfaceAreas2 = diameter_a * 3.14159 * lengths "Heat transfer areas";
    parameter Modelica.SIunits.Temperature T_ambient = system.T_ambient "Ambient temperature";
    parameter Modelica.SIunits.CoefficientOfHeatTransfer alpha0 = 100 "guess value for heat transfer coefficients";
    parameter Real dxs = lengths / length;
    parameter Real eps = 0.9 "radiation emissivity for surface1 and surface2";
    //////Variables
    Modelica.Fluid.Interfaces.FluidPort_a port_a(redeclare package Medium = Medium, m_flow(min = if allowFlowReversal then -Constants.inf else 0)) "Fluid connector a (positive design flow direction is from port_a to port_b)" annotation(
      Placement(transformation(extent = {{-110, -10}, {-90, 10}}, rotation = 0)));
    Modelica.Fluid.Interfaces.FluidPort_b port_b(redeclare package Medium = Medium, m_flow(max = if allowFlowReversal then +Constants.inf else 0)) "Fluid connector b (positive design flow direction is from port_a to port_b)" annotation(
      Placement(transformation(extent = {{110, -10}, {90, 10}}, rotation = 0), iconTransformation(extent = {{110, -10}, {90, 10}})));
    replaceable package WallFriction = Modelica.Fluid.Pipes.BaseClasses.WallFriction.Detailed constrainedby Modelica.Fluid.Pipes.BaseClasses.WallFriction.PartialWallFriction "Wall friction model";
    // Total quantities
    Modelica.SIunits.Energy[n] Us "Internal energy of fluid";
    Modelica.SIunits.Mass[n] ms "Fluid mass";
    Modelica.SIunits.Mass[n, Medium.nXi] mXis "Substance mass";
    Modelica.SIunits.Mass[n, Medium.nC] mCs "Trace substance mass";
    // C need to be added here because unlike for Xi, which has medium[:].Xi,
    // there is no variable medium[:].C
    Modelica.SIunits.Mass[n, Medium.nC] mCs_scaled "Scaled trace substance mass";
    Medium.ExtraProperty Cs[n, Medium.nC] "Trace substance mixture content";
    Medium.BaseProperties[n] mediums(each preferredMediumStates = true, p(start = ps_start), each h(start = h_start), each T(start = T_start), each Xi(start = X_start[1:Medium.nXi]));
    //Source terms, have to be defined by an extending model (to zero if not used)
    Medium.MassFlowRate[n] mb_flows "Mass flow rate, source or sink";
    Medium.MassFlowRate[n, Medium.nXi] mbXi_flows "Independent mass flow rates, source or sink";
    Medium.ExtraPropertyFlowRate[n, Medium.nC] mbC_flows "Trace substance mass flow rates, source or sink";
    Modelica.SIunits.EnthalpyFlowRate[n] Hb_flows "Enthalpy flow rate, source or sink";
    Modelica.SIunits.HeatFlowRate[n] Qb_flows "Heat flow rate, source or sink";
    Modelica.SIunits.Power[n] Wb_flows "Mechanical power, p*der(V) etc.";
    Modelica.SIunits.Force[nFM + 1 - 1] Ib_flows "Flow of momentum across boundaries";
    Modelica.SIunits.Force[nFM + 1 - 1] Fs_p "Pressure forces";
    Modelica.SIunits.Force[nFM + 1 - 1] Fs_fg "Friction and gravity forces";
    //
    Medium.ThermodynamicState state_a "state defined by volume outside port_a";
    Medium.ThermodynamicState state_b "state defined by volume outside port_b";
    Medium.ThermodynamicState[nFM + 1] statesFM "state vector for flowModel model";
    Medium.MassFlowRate[nFM + 1 - 1] m_flowsFM(each min = if allowFlowReversal then -Modelica.Constants.inf else 0, each start = m_flow_start, each stateSelect = if momentumDynamics == Modelica.Fluid.Types.Dynamics.SteadyState then StateSelect.default else StateSelect.prefer) "mass flow rates between states";
    Modelica.SIunits.Momentum[nFM + 1 - 1] Is "Momenta of flow segments";
    Medium.Density[nFM + 1] rhos = if use_rho_nominal then fill(rho_nominal, nFM + 1) else Medium.density(statesFM);
    Medium.Density[nFM + 1 - 1] rhos_act "Actual density per segment";
    Medium.DynamicViscosity[nFM + 1] musFM = if use_mu_nominal then fill(mu_nominal, nFM + 1) else Medium.dynamicViscosity(statesFM);
    Medium.DynamicViscosity[nFM + 1 - 1] mus_act "Actual viscosity per segment";
    Modelica.SIunits.Pressure[nFM + 1 - 1] dps_fg(each start = (p_a_start - p_b_start) / (nFM + 1 - 1)) "pressure drop between states";
    Modelica.SIunits.ReynoldsNumber[nFM + 1] ResFM = Modelica.Fluid.Pipes.BaseClasses.CharacteristicNumbers.ReynoldsNumber(vsFM, rhos, musFM, dimension) if show_Res "Reynolds numbers";
    Medium.MassFlowRate[nFM + 1 - 1] m_flows_turbulent = {nParallel * (crossArea + crossArea) / (dimension + dimension) * mus_act[i] * Re_turbulent for i in 1:nFM + 1 - 1} if show_Res "Start of turbulent flow";
    Modelica.SIunits.Length[nFM + 1 - 1] pathLengths_internal = pathLengths "pathLengths used internally; to be defined by extending class";
    Modelica.SIunits.ReynoldsNumber[nFM + 1 - 1] Res_turbulent_internal = Re_turbulent * ones(nFM + 1 - 1) "Re_turbulent used internally";
    Modelica.SIunits.AbsolutePressure dp_fric_nominal = sum(WallFriction.pressureLoss_m_flow(m_flow_nominal / nParallel, rho_nominal, rho_nominal, mu_nominal, mu_nominal, pathLengths_internal, dimension, crossArea, roughness, m_flow_small / nParallel, Res_turbulent_internal)) "pressure loss for nominal conditions";
    // Flow quantities
    Medium.MassFlowRate[n + 1] m_flows(each min = if allowFlowReversal then -Modelica.Constants.inf else 0, each start = m_flow_start) "Mass flow rates of fluid across segment boundaries";
    Medium.MassFlowRate[n + 1, Medium.nXi] mXi_flows "Independent mass flow rates across segment boundaries";
    Medium.MassFlowRate[n + 1, Medium.nC] mC_flows "Trace substance mass flow rates across segment boundaries";
    Medium.EnthalpyFlowRate[n + 1] H_flows "Enthalpy flow rates of fluid across segment boundaries";
    Modelica.SIunits.Velocity[n] vs = {0.5 * (m_flows[i] + m_flows[i + 1]) / mediums[i].d / crossArea for i in 1:n} / nParallel "mean velocities in flow segments";
    // Model structure dependent flow geometry
    Modelica.SIunits.Length[nFM] pathLengths "Lengths along flow path";
    Modelica.SIunits.Velocity[nFM + 1] vsFM "Mean velocities in flow segments";
    Modelica.SIunits.HeatFlowRate[n] Q_flows_inner "Heat flow rates";
    Modelica.SIunits.HeatFlowRate[n] Q_flows_outer "Heat flow rates";
    Modelica.SIunits.HeatFlowRate[n] Q_rad12 "radiative Heat flow rates from surface1 to surface2";
    Modelica.Fluid.Interfaces.HeatPorts_a[n] heatports_inner "Heat port to component boundary" annotation(
      Placement(transformation(extent = {{-10, 45}, {10, 65}}), iconTransformation(extent = {{-30, 36}, {32, 52}})));
    Modelica.Fluid.Interfaces.HeatPorts_a[n] heatports_outer "Heat port to component boundary" annotation(
      Placement(visible = true, transformation(extent = {{-10, -65}, {10, -45}}, rotation = 0), iconTransformation(extent = {{-30, -52}, {32, -36}}, rotation = 0)));
    Modelica.SIunits.Temperature[n] Ts = Medium.temperature(mediums.state) "Temperatures defined by fluid states";
    Modelica.SIunits.CoefficientOfHeatTransfer[n] alphas(each start = alpha0) "CoefficientOfHeatTransfer";
    Real[n] Res "Reynolds numbers";
    Real[n] Prs "Prandtl numbers";
    Real[n] Nus "Nusselt numbers";
    Medium.Density[n] ds "Densities";
    Medium.DynamicViscosity[n] mus "Dynamic viscosities";
    Medium.ThermalConductivity[n] lambdas "Thermal conductivity";
    //  Modelica.SIunits.Length diameters[n] = dimension "Hydraulic diameters for pipe flow";
    Real[n] Nus_turb "Nusselt number for turbulent flow";
    Real[n] Nus_lam "Nusselt number for laminar flow";
    //Real[n] Nus_2;
    Real[n] gamma "Interpolation factor for transition regime";
    Real[n] Xis "Darcy friction factor, flow coefficient, Rohreibungszahl";
    parameter Real dhl = min(1, dimension / length) "dh/sum(lengths)";
    parameter Real Nus_f = 1 "correction factor for Nusselt number";
  equation
    Qb_flows = Q_flows_inner + Q_flows_outer;
    if modelStructure == ModelStructure.av_vb or modelStructure == ModelStructure.av_b then
      Wb_flows[2:n - 1] = {vs[i] * crossArea * ((mediums[i + 1].p - mediums[i - 1].p) / 2 + (dps_fg[i - 1] + dps_fg[i]) / 2) for i in 2:n - 1} * nParallel;
    else
      Wb_flows[2:n - 1] = {vs[i] * crossArea * ((mediums[i + 1].p - mediums[i - 1].p) / 2 + (dps_fg[i] + dps_fg[i + 1]) / 2) for i in 2:n - 1} * nParallel;
    end if;
    if modelStructure == ModelStructure.av_vb then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - mediums[1].p) / 2 + dps_fg[1] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((mediums[n].p - mediums[n - 1].p) / 2 + dps_fg[n - 1] / 2) * nParallel;
    elseif modelStructure == ModelStructure.av_b then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - mediums[1].p) / 2 + dps_fg[1] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((port_b.p - mediums[n - 1].p) / 1.5 + dps_fg[n - 1] / 2 + dps_fg[n]) * nParallel;
    elseif modelStructure == ModelStructure.a_vb then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - port_a.p) / 1.5 + dps_fg[1] + dps_fg[2] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((mediums[n].p - mediums[n - 1].p) / 2 + dps_fg[n] / 2) * nParallel;
    elseif modelStructure == ModelStructure.a_v_b then
      Wb_flows[1] = vs[1] * crossArea * ((mediums[2].p - port_a.p) / 1.5 + dps_fg[1] + dps_fg[2] / 2) * nParallel;
      Wb_flows[n] = vs[n] * crossArea * ((port_b.p - mediums[n - 1].p) / 1.5 + dps_fg[n] / 2 + dps_fg[n + 1]) * nParallel;
    else
      assert(false, "Unknown model structure");
    end if;
//from PartialHeatTransfer
    Q_flows_inner = heatports_inner.Q_flow;
    Q_flows_outer = heatports_outer.Q_flow;
//radiative heat transfer
//radiative flux from surface 1 to surface 2
    Q_rad12 = {5.67 / (1 / eps + (1 / eps - 1) * surfaceAreas1 / surfaceAreas2) * surfaceAreas1 * ((heatports_inner[i].T / 100) ^ 4 - (heatports_outer[i].T / 100) ^ 4) for i in 1:n};
    assert(nNodes > 1 or modelStructure <> ModelStructure.av_vb, "nNodes needs to be at least 2 for modelStructure av_vb, as flow model disappears otherwise!");
  if modelStructure == ModelStructure.av_vb then
//nFM = n-1
      if n == 2 then
        pathLengths[1] = lengths + lengths;
      else
        pathLengths[1:n - 1] = cat(1, {lengths + 0.5 * lengths}, fill(lengths, n - 3), {0.5 * lengths + lengths});
      end if;
    elseif modelStructure == ModelStructure.av_b then
//nFM = n
      pathLengths[1:n] = fill(lengths, n);
    elseif modelStructure == ModelStructure.a_vb then
//nFM = n
      pathLengths[1:n] = fill(lengths, n);
    elseif modelStructure == ModelStructure.a_v_b then
//nFM = n+1;
      pathLengths[1:n + 1] = cat(1, {0.5 * lengths}, fill(lengths, n - 1), {0.5 * lengths});
    else
      assert(false, "Unknown model structure");
    end if;
// Source/sink terms for mass and energy balances
    for i in 1:n loop
      mb_flows[i] = m_flows[i] - m_flows[i + 1];
      mbXi_flows[i, :] = mXi_flows[i, :] - mXi_flows[i + 1, :];
      mbC_flows[i, :] = mC_flows[i, :] - mC_flows[i + 1, :];
      Hb_flows[i] = H_flows[i] - H_flows[i + 1];
    end for;
// Distributed flow quantities, upwind discretization
    for i in 2:n loop
      H_flows[i] = semiLinear(m_flows[i], mediums[i - 1].h, mediums[i].h);
      mXi_flows[i, :] = semiLinear(m_flows[i], mediums[i - 1].Xi, mediums[i].Xi);
      mC_flows[i, :] = semiLinear(m_flows[i], Cs[i - 1, :], Cs[i, :]);
    end for;
    H_flows[1] = semiLinear(port_a.m_flow, inStream(port_a.h_outflow), mediums[1].h);
    H_flows[n + 1] = -semiLinear(port_b.m_flow, inStream(port_b.h_outflow), mediums[n].h);
    mXi_flows[1, :] = semiLinear(port_a.m_flow, inStream(port_a.Xi_outflow), mediums[1].Xi);
    mXi_flows[n + 1, :] = -semiLinear(port_b.m_flow, inStream(port_b.Xi_outflow), mediums[n].Xi);
    mC_flows[1, :] = semiLinear(port_a.m_flow, inStream(port_a.C_outflow), Cs[1, :]);
    mC_flows[n + 1, :] = -semiLinear(port_b.m_flow, inStream(port_b.C_outflow), Cs[n, :]);
// Boundary conditions
    port_a.m_flow = m_flows[1];
    port_b.m_flow = -m_flows[n + 1];
    port_a.h_outflow = mediums[1].h;
    port_b.h_outflow = mediums[n].h;
    port_a.Xi_outflow = mediums[1].Xi;
    port_b.Xi_outflow = mediums[n].Xi;
    port_a.C_outflow = Cs[1, :];
    port_b.C_outflow = Cs[n, :];
// The two equations below are not correct if C is stored in volumes.
// C should be treated the same way as Xi.
//port_a.C_outflow = inStream(port_b.C_outflow);
//port_b.C_outflow = inStream(port_a.C_outflow);
    if useInnerPortProperties and n > 0 then
      state_a = Medium.setState_phX(port_a.p, mediums[1].h, mediums[1].Xi);
      state_b = Medium.setState_phX(port_b.p, mediums[n].h, mediums[n].Xi);
    else
      state_a = Medium.setState_phX(port_a.p, inStream(port_a.h_outflow), inStream(port_a.Xi_outflow));
      state_b = Medium.setState_phX(port_b.p, inStream(port_b.h_outflow), inStream(port_b.Xi_outflow));
    end if;
// staggered grid discretization for flowModel, depending on modelStructure
    if modelStructure == ModelStructure.av_vb then
//nFM = n-1
      statesFM[1:n] = mediums[1:n].state;
      m_flows[2:n] = m_flowsFM[1:n - 1];
      vsFM[1:n] = vs;
      port_a.p = mediums[1].p;
      port_b.p = mediums[n].p;
    elseif modelStructure == ModelStructure.av_b then
//nFM = n
      statesFM[1:n] = mediums[1:n].state;
      statesFM[n + 1] = state_b;
      m_flows[2:n + 1] = m_flowsFM[1:n];
      vsFM[1:n] = vs;
      vsFM[n + 1] = m_flows[n + 1] / Medium.density(state_b) / crossArea / nParallel;
      port_a.p = mediums[1].p;
    elseif modelStructure == ModelStructure.a_vb then
//nFM = n
      statesFM[1] = state_a;
      statesFM[2:n + 1] = mediums[1:n].state;
      m_flows[1:n] = m_flowsFM[1:n];
      vsFM[1] = m_flows[1] / Medium.density(state_a) / crossArea / nParallel;
      vsFM[2:n + 1] = vs;
      port_b.p = mediums[n].p;
    elseif modelStructure == ModelStructure.a_v_b then
//nFM = n+1
      statesFM[1] = state_a;
      statesFM[2:n + 1] = mediums[1:n].state;
      statesFM[n + 2] = state_b;
      m_flows[1:n + 1] = m_flowsFM[1:n + 1];
      vsFM[1] = m_flows[1] / Medium.density(state_a) / crossArea / nParallel;
      vsFM[2:n + 1] = vs;
      vsFM[n + 2] = m_flows[n + 1] / Medium.density(state_b) / crossArea / nParallel;
    else
      assert(false, "Unknown model structure");
    end if;
    assert(not (energyDynamics <> Dynamics.SteadyState and massDynamics == Dynamics.SteadyState) or Medium.singleState, "Bad combination of dynamics options and Medium not conserving mass if fluidVolumes are fixed.");
// Total quantities
    for i in 1:n loop
      ms[i] = fluidVolumes * mediums[i].d;
      mXis[i, :] = ms[i] * mediums[i].Xi;
      mCs[i, :] = ms[i] * Cs[i, :];
      Us[i] = ms[i] * mediums[i].u;
    end for;
// Energy and mass balances
    if energyDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        0 = Hb_flows[i] + Wb_flows[i] + Qb_flows[i];
      end for;
    else
      for i in 1:n loop
        der(Us[i]) = Hb_flows[i] + Wb_flows[i] + Qb_flows[i];
      end for;
    end if;
    if massDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        0 = mb_flows[i];
      end for;
    else
      for i in 1:n loop
        der(ms[i]) = mb_flows[i];
      end for;
    end if;
    if substanceDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        zeros(Medium.nXi) = mbXi_flows[i, :];
      end for;
    else
      for i in 1:n loop
        der(mXis[i, :]) = mbXi_flows[i, :];
      end for;
    end if;
    if traceDynamics == Dynamics.SteadyState then
      for i in 1:n loop
        zeros(Medium.nC) = mbC_flows[i, :];
      end for;
    else
      for i in 1:n loop
        der(mCs_scaled[i, :]) = mbC_flows[i, :] ./ Medium.C_nominal;
        mCs[i, :] = mCs_scaled[i, :] .* Medium.C_nominal;
      end for;
    end if;
//equations for heat transfer (from localPipeFlowHeatTransfer and from Waermeatlas,11th,2013))
    for i in 1:n loop
      Nus_turb[i] = Xis[i] / 8 * max(Res[i], 10000) * Prs[i] / (1.07 + 900 / max(Res[i], 1e-3) - 0.63 / (1 + 10 * Prs[i]) + 12.7 * sqrt(Xis[i] / 8) * (Prs[i] ^ (2 / 3) - 1)) * (1 + dhl ^ (2 / 3)) * (1 + dhl ^ (2 / 3)) * ((0.75 * (diameter_i / diameter_a) ^ (-0.17) + 0.9 - 0.15 * (diameter_i / diameter_a) ^ 0.6) / (1 + diameter_i / diameter_a));
      Xis[i] = (1.8 * Modelica.Math.log10(max(Res[i], 10000) * ((1 + (diameter_i / diameter_a) ^ 2) * Modelica.Math.log(diameter_i / diameter_a) + 1 - (diameter_i / diameter_a) ^ 2) / ((1 - diameter_i / diameter_a) ^ 2 * Modelica.Math.log(diameter_i / diameter_a))) - 1.5) ^ (-2);
      Nus_lam[i] = ((3.66 + (4 - 0.102 / (diameter_i / diameter_a + 0.02)) * (diameter_i / diameter_a) ^ 0.04) ^ 3 + (1.615 * (1 + 0.14 * (diameter_i / diameter_a) ^ 0.1) * (min(Res[i], 2300) * Prs[i] * dhl) ^ (1 / 3)) ^ 3 + ((2 / (1 + 22 * Prs[i])) ^ (1 / 6) * (min(Res[i], 2300) * Prs[i] * dhl) ^ (1 / 2)) ^ 3) ^ (1 / 3);
      gamma[i] = min(max((Res[i] - 2300) / (10000 - 2300), 0), 1);
      Nus[i] = Nus_f * (Nus_lam[i] * (1 - gamma[i]) + gamma[i] * Nus_turb[i]);
    end for;
//from PartialPipeFlowHeatTransfer
    ds = Medium.density(mediums.state);
    mus = Medium.dynamicViscosity(mediums.state);
    lambdas = Medium.thermalConductivity(mediums.state);
    Prs = Medium.prandtlNumber(mediums.state);
    for i in 1:n loop
      Res[i] = abs(vs[i]) * ds[i] * dimension / mus[i];
      Nus[i] = alphas[i] * dimension / lambdas[i];
      Q_flows_inner[i] = Q_rad12[i] + alphas[i] * surfaceAreas1 * (heatports_inner[i].T - Ts[i]) * nParallel;
      Q_flows_outer[i] = (-Q_rad12[i]) + alphas[i] * surfaceAreas2 * (heatports_outer[i].T - Ts[i]) * nParallel;
    end for;
//from partialGenericPipeFlow
    for i in 1:nFM + 1 - 1 loop
      assert(m_flowsFM[i] > (-m_flow_small) or allowFlowReversal, "Reverting flow occurs even though allowFlowReversal is false");
    end for;
  if continuousFlowReversal then
// simple regularization
      if from_dp and not WallFriction.dp_is_zero then
        m_flowsFM = homotopy(actual = WallFriction.massFlowRate_dp(dps_fg, rhos_act, rhos_act, mus_act, mus_act, pathLengths_internal, dimension, crossArea, roughness, system.dp_small / (nFM + 1 - 1), Res_turbulent_internal) * nParallel, simplified = m_flow_nominal / dp_nominal * dps_fg);
      else
        dps_fg = homotopy(actual = WallFriction.pressureLoss_m_flow(m_flowsFM / nParallel, rhos_act, rhos_act, mus_act, mus_act, pathLengths_internal, dimension, crossArea, roughness, m_flow_small / nParallel, Res_turbulent_internal), simplified = dp_nominal / m_flow_nominal * m_flowsFM);
      end if;
    else
// regularization for discontinuous flow reversal and static head
      if from_dp and not WallFriction.dp_is_zero then
        m_flowsFM = homotopy(actual = WallFriction.massFlowRate_dp_staticHead(dps_fg, rhos[1:nFM + 1 - 1], rhos[2:nFM + 1], musFM[1:nFM + 1 - 1], musFM[2:nFM + 1], pathLengths_internal, dimension, 0, crossArea, roughness, system.dp_small / (nFM + 1 - 1), Res_turbulent_internal) * nParallel, simplified = m_flow_nominal / dp_nominal * dps_fg);
      else
        dps_fg = homotopy(actual = WallFriction.pressureLoss_m_flow_staticHead(m_flowsFM / nParallel, rhos[1:nFM + 1 - 1], rhos[2:nFM + 1], musFM[1:nFM + 1 - 1], musFM[2:nFM + 1], pathLengths_internal, dimension, 0, crossArea, roughness, m_flow_small / nParallel, Res_turbulent_internal), simplified = dp_nominal / m_flow_nominal * m_flowsFM);
      end if;
    end if;
//from PartialStaggeredFlowModel
    if not allowFlowReversal then
      rhos_act = rhos[1:nFM + 1 - 1];
      mus_act = musFM[1:nFM + 1 - 1];
    elseif not useUpstreamScheme then
      rhos_act = 0.5 * (rhos[1:nFM + 1 - 1] + rhos[2:nFM + 1]);
      mus_act = 0.5 * (musFM[1:nFM + 1 - 1] + musFM[2:nFM + 1]);
    else
      for i in 1:nFM + 1 - 1 loop
        rhos_act[i] = noEvent(if m_flowsFM[i] > 0 then rhos[i] else rhos[i + 1]);
        mus_act[i] = noEvent(if m_flowsFM[i] > 0 then musFM[i] else musFM[i + 1]);
      end for;
    end if;
  if use_Ib_flows then
      Ib_flows = nParallel * {rhos[i] * vsFM[i] * vsFM[i] * crossArea - rhos[i + 1] * vsFM[i + 1] * vsFM[i + 1] * crossArea for i in 1:nFM + 1 - 1};
// alternatively use densities rhos_act of actual streams, together with mass flow rates,
// not conserving momentum if fluid density changes between flow segments:
//Ib_flows = {((rhos[i]*vsFM[i])^2*crossAreasFM[i] - (rhos[i+1]*vsFM[i+1])^2*crossAreasFM[i+1])/rhos_act[i] for i in 1:nFM+1-1};
    else
      Ib_flows = zeros(nFM + 1 - 1);
    end if;
    Fs_p = nParallel * {0.5 * (crossArea + crossArea) * (Medium.pressure(statesFM[i + 1]) - Medium.pressure(statesFM[i])) for i in 1:nFM + 1 - 1};
// Note: the equation is written for dps_fg instead of Fs_fg to help the translator
    dps_fg = {Fs_fg[i] / nParallel * 2 / (crossArea + crossArea) for i in 1:nFM + 1 - 1};
//from PartialDistributedFlow
// Total quantities
    Is = {m_flowsFM[i] * pathLengths[i] for i in 1:nFM + 1 - 1};
// Momentum balances
    if momentumDynamics == Modelica.Fluid.Types.Dynamics.SteadyState then
      zeros(nFM + 1 - 1) = Ib_flows - Fs_p - Fs_fg;
    else
      der(Is) = Ib_flows - Fs_p - Fs_fg;
    end if;
/////////////////////////////////////////////////////////////
  initial equation
// initialization of balances
    if energyDynamics == Dynamics.FixedInitial then
      if use_T_start then
        mediums.T = fill(T_start, n);
      else
        mediums.h = fill(h_start, n);
      end if;
      if Medium.ThermoStates == IndependentVariables.ph or Medium.ThermoStates == IndependentVariables.phX then
        mediums.h = fill(h_start, n);
      else
        mediums.T = fill(T_start, n);
      end if;
    elseif energyDynamics == Dynamics.SteadyStateInitial then
      if use_T_start then
        der(mediums.T) = zeros(n);
      else
        der(mediums.h) = zeros(n);
      end if;
      if Medium.ThermoStates == IndependentVariables.ph or Medium.ThermoStates == IndependentVariables.phX then
        der(mediums.h) = zeros(n);
      else
        der(mediums.T) = zeros(n);
      end if;
    end if;
    if massDynamics == Dynamics.FixedInitial then
      if initialize_p then
        mediums.p = ps_start;
      end if;
    elseif massDynamics == Dynamics.SteadyStateInitial then
      if initialize_p then
        der(mediums.p) = zeros(n);
      end if;
    end if;
    if substanceDynamics == Dynamics.FixedInitial then
      mediums.Xi = fill(X_start[1:Medium.nXi], n);
    elseif substanceDynamics == Dynamics.SteadyStateInitial then
      for i in 1:n loop
        der(mediums[i].Xi) = zeros(Medium.nXi);
      end for;
    end if;
    if traceDynamics == Dynamics.FixedInitial then
      Cs = fill(C_start[1:Medium.nC], n);
    elseif traceDynamics == Dynamics.SteadyStateInitial then
      for i in 1:n loop
        der(mCs[i, :]) = zeros(Medium.nC);
      end for;
    end if;
//from detailedPipeFlow
// initialize dp_nominal from flow model
/*
    if system.use_eps_Re then
      dp_nominal = dp_fric_nominal + system.g * sum(dheightsFM) * rho_nominal;
    else
      dp_nominal = 1e3 * system.dp_small;
    end if;
  */
//from partialGenericPipeFlow
// initialize dp_small from flow model
/*
    if system.use_eps_Re then
      dp_small = dp_fric_nominal / m_flow_nominal * m_flow_small;
    else
      dp_small = system.dp_small;
    end if;
  */
//from PartialDistributedFlow
    if momentumDynamics == Modelica.Fluid.Types.Dynamics.FixedInitial then
      m_flowsFM = fill(m_flow_start, nFM + 1 - 1);
    elseif momentumDynamics == Modelica.Fluid.Types.Dynamics.SteadyStateInitial then
      der(m_flowsFM) = zeros(nFM + 1 - 1);
    end if;
///////end initial eqautions//////////////////////////////////////////////
    annotation(
      defaultComponentName = "pipe",
      Documentation(info = "<html>
  <p>Rewrite of the original DynamicPipe class to provide the possibility of radiation between two heatports and give a better performance of model translation due to mostly flat design. It always uses detailed pipe flow and detailed heat transfer models. Some options were removed (e.g. use lumped pressure, circular design, gravitational influence (different heights),).</p>
  </html>"),
      Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 44}, {100, -44}}, lineColor = {0, 0, 0}, fillPattern = FillPattern.HorizontalCylinder, fillColor = {0, 127, 255}), Ellipse(extent = {{-72, 10}, {-52, -10}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Ellipse(extent = {{50, 10}, {70, -10}}, lineColor = {0, 0, 0}, fillColor = {0, 0, 0}, fillPattern = FillPattern.Solid), Text(extent = {{-48, 15}, {46, -20}}, lineColor = {0, 0, 0}, textString = "%nNodes")}),
      Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 60}, {100, 50}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Backward), Rectangle(extent = {{-100, -50}, {100, -60}}, lineColor = {0, 0, 0}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Backward), Line(points = {{100, 45}, {100, 50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{0, 45}, {0, 50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{100, -45}, {100, -50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{0, -45}, {0, -50}}, arrow = {Arrow.None, Arrow.Filled}, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{-50, 60}, {-50, 50}}, smooth = Smooth.None, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{50, 60}, {50, 50}}, smooth = Smooth.None, color = {0, 0, 0}, pattern = LinePattern.Dot), Line(points = {{0, -50}, {0, -60}}, smooth = Smooth.None, color = {0, 0, 0}, pattern = LinePattern.Dot)}));
  end FlatDynamicGap;

  model DoublePipeHX "Simple double pipe heat exchanger model based on FlatDynamicPipe and FlatDynamicGap"
    outer Modelica.Fluid.System system "System properties";
    //General
    parameter Modelica.SIunits.Length length(min = 0) "Length of flow path for both fluids";
    parameter Integer nNodes(min = 1) = 2 "Spatial segmentation";
    parameter Modelica.Fluid.Types.ModelStructure modelStructure_1 = Modelica.Fluid.Types.ModelStructure.av_vb "Determines whether flow or volume models are present at the ports" annotation(
      Evaluate = true,
      Dialog(tab = "General", group = "Fluid 1"));
    parameter Modelica.Fluid.Types.ModelStructure modelStructure_2 = Modelica.Fluid.Types.ModelStructure.av_vb "Determines whether flow or volume models are present at the ports" annotation(
      Evaluate = true,
      Dialog(tab = "General", group = "Fluid 2"));
    replaceable package Medium_1 = Modelica.Media.Water.StandardWater constrainedby Modelica.Media.Interfaces.PartialMedium "Fluid 1" annotation(
       choicesAllMatching,
       Dialog(tab = "General", group = "Fluid 1"));
    replaceable package Medium_2 = Modelica.Media.Water.StandardWater constrainedby Modelica.Media.Interfaces.PartialMedium "Fluid 2" annotation(
       choicesAllMatching,
       Dialog(tab = "General", group = "Fluid 2"));
    parameter Modelica.SIunits.Length diameter1 "Inner pipe diameter" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.Length diameter2_i "Inner diameter of gap" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.Length diameter2_a "Outer diameter of gap" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.Length diameter3 "Outer diameter of insulation" annotation(
      Dialog(group = "Wall properties"));
    // Heat transfer
    parameter Real areaFactor = 1 "Factor for modification of heat transfering area";
    parameter Modelica.SIunits.ThermalConductivity lambda_wall "Thermal conductivity of wall material" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.ThermalConductivity lambda_wall_amb "Thermal conductivity of wall to ambient material" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.SpecificHeatCapacity c_wall "Specific heat capacity of wall material" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.SpecificHeatCapacity c_wall_amb "Specific heat capacity of wall material" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.Density rho_wall "Density of wall material" annotation(
      Dialog(group = "Wall properties"));
    parameter Modelica.SIunits.Density rho_wall_amb "Density of wall to ambient (insulation) material" annotation(
      Dialog(group = "Ambient wall properties"));
    // Assumptions
    parameter Boolean allowFlowReversal = system.allowFlowReversal "allow flow reversal, false restricts to design direction (port_a -> port_b)" annotation(
      Dialog(tab = "Assumptions"),
      Evaluate = true);
    parameter Modelica.Fluid.Types.Dynamics energyDynamics = system.energyDynamics "Formulation of energy balance" annotation(
      Evaluate = true,
      Dialog(tab = "Assumptions", group = "Dynamics"));
    parameter Modelica.Fluid.Types.Dynamics massDynamics = system.massDynamics "Formulation of mass balance" annotation(
      Evaluate = true,
      Dialog(tab = "Assumptions", group = "Dynamics"));
    parameter Modelica.Fluid.Types.Dynamics momentumDynamics = system.momentumDynamics "Formulation of momentum balance, if pressureLoss options available" annotation(
      Evaluate = true,
      Dialog(tab = "Assumptions", group = "Dynamics"));
    //Initialization pipe 1
    parameter Modelica.SIunits.Temperature Twall_start "Start value of wall temperature" annotation(
      Dialog(tab = "Initialization", group = "Wall"));
    parameter Modelica.SIunits.Temperature dT "Start value for pipe_1.T - pipe_2.T" annotation(
      Dialog(tab = "Initialization", group = "Wall"));
    parameter Boolean use_T_start = true "Use T_start if true, otherwise h_start" annotation(
      Evaluate = true,
      Dialog(tab = "Initialization"));
    parameter Medium_1.AbsolutePressure p_a_start1 = Medium_1.p_default "Start value of pressure" annotation(
      Dialog(tab = "Initialization", group = "Fluid 1"));
    parameter Medium_1.AbsolutePressure p_b_start1 = Medium_1.p_default "Start value of pressure" annotation(
      Dialog(tab = "Initialization", group = "Fluid 1"));
    parameter Medium_1.Temperature T_start_1 = if use_T_start then Medium_1.T_default else Medium_1.temperature_phX((p_a_start1 + p_b_start1) / 2, h_start_1, X_start_1) "Start value of temperature" annotation(
      Evaluate = true,
      Dialog(tab = "Initialization", group = "Fluid 1", enable = use_T_start));
    parameter Medium_1.SpecificEnthalpy h_start_1 = if use_T_start then Medium_1.specificEnthalpy_pTX((p_a_start1 + p_b_start1) / 2, T_start_1, X_start_1) else Medium_1.h_default "Start value of specific enthalpy" annotation(
      Evaluate = true,
      Dialog(tab = "Initialization", group = "Fluid 1", enable = not use_T_start));
    parameter Medium_1.MassFraction X_start_1[Medium_1.nX] = Medium_1.X_default "Start value of mass fractions m_i/m" annotation(
      Dialog(tab = "Initialization", group = "Fluid 1", enable = Medium_1.nXi > 0));
    parameter Medium_1.MassFlowRate m_flow_start_1 = system.m_flow_start "Start value of mass flow rate" annotation(
      Evaluate = true,
      Dialog(tab = "Initialization", group = "Fluid 1"));
    //Initialization pipe 2
    parameter Medium_2.AbsolutePressure p_a_start2 = Medium_2.p_default "Start value of pressure" annotation(
      Dialog(tab = "Initialization", group = "Fluid 2"));
    parameter Medium_2.AbsolutePressure p_b_start2 = Medium_2.p_default "Start value of pressure" annotation(
      Dialog(tab = "Initialization", group = "Fluid 2"));
    parameter Medium_2.Temperature T_start_2 = if use_T_start then Medium_2.T_default else Medium_2.temperature_phX((p_a_start2 + p_b_start2) / 2, h_start_2, X_start_2) "Start value of temperature" annotation(
      Evaluate = true,
      Dialog(tab = "Initialization", group = "Fluid 2", enable = use_T_start));
    parameter Medium_2.SpecificEnthalpy h_start_2 = if use_T_start then Medium_2.specificEnthalpy_pTX((p_a_start2 + p_b_start2) / 2, T_start_2, X_start_2) else Medium_2.h_default "Start value of specific enthalpy" annotation(
      Evaluate = true,
      Dialog(tab = "Initialization", group = "Fluid 2", enable = not use_T_start));
    parameter Medium_2.MassFraction X_start_2[Medium_2.nX] = Medium_2.X_default "Start value of mass fractions m_i/m" annotation(
      Dialog(tab = "Initialization", group = "Fluid 2", enable = Medium_2.nXi > 0));
    parameter Medium_2.MassFlowRate m_flow_start_2 = system.m_flow_start "Start value of mass flow rate" annotation(
      Evaluate = true,
      Dialog(tab = "Initialization", group = "Fluid 2"));
    //Pressure drop and heat transfer
    parameter Modelica.SIunits.Length roughness_1 = 2.5e-5 "Absolute roughness of pipe (default = smooth steel pipe)" annotation(
      Dialog(tab = "General", group = "Fluid 1"));
    parameter Modelica.SIunits.Length roughness_2 = 2.5e-5 "Absolute roughness of pipe (default = smooth steel pipe)" annotation(
      Dialog(tab = "General", group = "Fluid 2"));
    parameter Real alpha_amb = 1e-1 "heat transfer coefficient for outer surfaces";
    parameter Real T_amb = system.T_ambient "ambient temperature";
    parameter Real eps = 0.9 "radiation emissivity for surface1 and surface2 in gap";
    parameter Real Nus_f = 1 "correction factor for Nusselt number";
    //Display variables
    Modelica.SIunits.HeatFlowRate Q_flow "Total heat flow rate of pipe 1 to pipe 2";
    //  Modelica.SIunits.HeatFlowRate Q_flow_1 "Total heat flow rate of pipe 1 to pipe 2";
    //  Modelica.SIunits.HeatFlowRate Q_flow_2 "Total heat flow rate of pipe 2 from pipe 1";
    //  Modelica.SIunits.HeatFlowRate Q_flow_3 "Total heat flow rate of pipe 2 to ambient wall";
    Modelica.SIunits.HeatFlowRate Q_flow_wall_amb "Total heat flow rate pipe 2 to ambient wall";
    //  Modelica.SIunits.HeatFlowRate Q_flow_wall_1 "Total heat flow rate pipe 1 to heat exchanger wall";
    //  Modelica.SIunits.HeatFlowRate Q_flow_wall_2 "Total heat flow rate pipe 2 to heat exchanger wall";
    //Modelica.Fluid.Interfaces.HeatPorts_a[nNodes] heatports_amb "heat ports to ambient";
    //  Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_a port;
    frontend_bench.FlatDynamicPipe pipe_1(redeclare package Medium = Medium_1, nNodes = nNodes, allowFlowReversal = allowFlowReversal, length = length, areaFactor = areaFactor, T_start = T_start_1, X_start = X_start_1, m_flow_start = m_flow_start_1, diameter = diameter1, p_a_start = p_a_start1, p_b_start = p_b_start1, roughness = roughness_1, modelStructure = modelStructure_1) annotation(
      Placement(visible = true, transformation(origin = {0, 54}, extent = {{-30, 30}, {30, -30}}, rotation = 0)));
    frontend_bench.FlatDynamicGap pipe_2(redeclare package Medium = Medium_2, nNodes = nNodes, allowFlowReversal = allowFlowReversal, length = length, areaFactor = areaFactor, T_start = T_start_2, X_start = X_start_2, m_flow_start = m_flow_start_2, diameter_i = diameter2_i, diameter_a = diameter2_a, p_a_start = p_a_start2, p_b_start = p_b_start2, roughness = roughness_2, modelStructure = modelStructure_2, eps = eps, Nus_f = Nus_f) annotation(
      Placement(visible = true, transformation(origin = {0, -36}, extent = {{-30, 30}, {30, -30}}, rotation = 0)));
    Modelica.Fluid.Interfaces.FluidPort_a port_a1(redeclare package Medium = Medium_1) annotation(
      Placement(visible = true, transformation(origin = {-100, 54}, extent = {{-10, 10}, {10, -10}}, rotation = 0), iconTransformation(origin = {-80, 0}, extent = {{-6, -6}, {6, 6}}, rotation = 0)));
    Modelica.Fluid.Interfaces.FluidPort_b port_b1(redeclare package Medium = Medium_1) annotation(
      Placement(visible = true, transformation(origin = {100, 54}, extent = {{-10, 10}, {10, -10}}, rotation = 0), iconTransformation(origin = {80, 0}, extent = {{-6, -6}, {6, 6}}, rotation = 0)));
    Modelica.Fluid.Interfaces.FluidPort_b port_b2(redeclare package Medium = Medium_2) annotation(
      Placement(visible = true, transformation(extent = {{90, -46}, {110, -26}}, rotation = 0), iconTransformation(origin = {80, -40}, extent = {{-6, -6}, {6, 6}}, rotation = 0)));
    Modelica.Fluid.Interfaces.FluidPort_a port_a2(redeclare package Medium = Medium_2) annotation(
      Placement(visible = true, transformation(extent = {{-110, -46}, {-90, -26}}, rotation = 0), iconTransformation(origin = {-80, -40}, extent = {{-6, -6}, {6, 6}}, rotation = 0)));
    CylWallConstProps CylWall(T_start = Twall_start, c_wall = c_wall, dT = dT, energyDynamics = energyDynamics, lambda_wall = lambda_wall, n = nNodes, rho_wall = rho_wall, diameter_inner = diameter1, diameter_outer = diameter2_i, length = length * areaFactor) annotation(
      Placement(visible = true, transformation(origin = {0, 8}, extent = {{-19, 29}, {19, -29}}, rotation = 180)));
    frontend_bench.CylWallConstProps CylWall_ambient(rho_wall = rho_wall_amb, c_wall = c_wall_amb, T_start = Twall_start, lambda_wall = lambda_wall_amb, dT = dT, energyDynamics = energyDynamics, n = nNodes, diameter_inner = diameter2_a, diameter_outer = diameter3, length = length) annotation(
      Placement(visible = true, transformation(origin = {0, -73}, extent = {{-18, -25}, {18, 25}}, rotation = 0)));
    simpleConvection simpleConvection1(nNodes = nNodes, T_ambient = system.T_ambient, alpha = alpha_amb, area = 3.14159 * diameter3 * length) annotation(
      Placement(visible = true, transformation(origin = {40, -86}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  initial equation

  equation
    connect(pipe_1.heatPorts, CylWall.heatPort_inner) annotation(
      Line(points = {{0, 41}, {0, 22.5}}, color = {127, 0, 0}, thickness = 0.5));
    connect(pipe_1.port_a, port_a1) annotation(
      Line(points = {{-30, 54}, {-100, 54}}, color = {0, 127, 255}, thickness = 0.5));
    connect(pipe_1.port_b, port_b1) annotation(
      Line(points = {{30, 54}, {100, 54}}, color = {0, 127, 255}, thickness = 0.5));
    connect(pipe_2.heatports_outer, CylWall_ambient.heatPort_inner) annotation(
      Line(points = {{0, -49}, {0, -60.5}}, color = {127, 0, 0}, thickness = 0.5));
    connect(pipe_2.port_b, port_b2) annotation(
      Line(points = {{-30, -36}, {100, -36}}, color = {0, 127, 255}, thickness = 0.5));
    connect(CylWall.heatPort_outer, pipe_2.heatports_inner) annotation(
      Line(points = {{0, -6}, {0, -23}}, color = {191, 0, 0}, thickness = 0.5));
    connect(pipe_2.port_a, port_a2) annotation(
      Line(points = {{30, -36}, {-100, -36}}, color = {0, 127, 255}, thickness = 0.5));
    connect(CylWall_ambient.heatPort_outer, simpleConvection1.heatport) annotation(
      Line(points = {{0, -85.5}, {0, -86}, {32, -86}}, color = {191, 0, 0}, thickness = 0.5));
// if counterflow-crossing via code and not via connection:
//  connect(wall.heatPort_outer[:], pipe_2.heatports_inner[nNodes:(-1):1]) annotation( Line(points = {{0, -6}, {0, -25.5}}, color = {191, 0, 0}, thickness = 0.5));
    Q_flow = sum(pipe_1.Q_flows);
//  Q_flow_1 = sum(pipe_1.Q_flows);
//  Q_flow_2 = sum(pipe_2.Q_flows_inner);
//  Q_flow_3 = sum(pipe_2.Q_flows_outer);
    Q_flow_wall_amb = sum(CylWall_ambient.heatPort_outer[i].Q_flow for i in 1:nNodes);
//  Q_flow_wall_1 = sum(CylWall.heatPort_inner[i].Q_flow for i in 1:nNodes);
//  Q_flow_wall_2 = sum(CylWall.heatPort_outer[i].Q_flow for i in 1:nNodes);
    annotation(
      Icon(coordinateSystem(preserveAspectRatio = false, initialScale = 0.1), graphics = {Text(origin = {26, -224}, lineColor = {0, 0, 255}, extent = {{-110, 108}, {54, 66}}, textString = "%name"), Rectangle(origin = {1, -15}, lineColor = {255, 0, 0}, fillColor = {255, 255, 0}, fillPattern = FillPattern.HorizontalCylinder, lineThickness = 1, extent = {{-81, 33}, {79, -3}}), Rectangle(origin = {1, 23}, lineColor = {255, 0, 0}, fillColor = {0, 85, 255}, fillPattern = FillPattern.Solid, lineThickness = 1, extent = {{-81, 33}, {79, -3}}), Rectangle(origin = {1, -53}, lineColor = {255, 0, 0}, fillColor = {0, 85, 255}, fillPattern = FillPattern.Solid, lineThickness = 1, extent = {{-81, 33}, {79, -3}}), Text(origin = {2, 0}, lineThickness = 1, extent = {{-20, 8}, {20, -8}}, textString = "Fluid 1"), Text(origin = {2, -38}, lineThickness = 1, extent = {{-20, 8}, {20, -8}}, textString = "Fluid 2")}),
      Documentation(info = "<html>
    <p>Simple model of a heat exchanger consisting of two pipes and one wall in between.
    For both fluids geometry parameters, such as heat transfer area and cross section as well as heat transfer and pressure drop correlations may be chosen.
    The flow scheme may be concurrent or counterflow, defined by the respective flow directions of the fluids entering the component.
    The design flow direction with positive m_flow variables is counterflow.</p>
    </html>"));
  end DoublePipeHX;

  model simpleConvection "Lumped thermal element for heat convection (Q_flow = Gc*dT)"
    parameter Integer nNodes(min = 1) = 1 "number of nodes";
    Modelica.SIunits.HeatFlowRate[nNodes] Q_flow "Heat flow rates from solid -> fluid";
    Modelica.SIunits.HeatFlowRate Q_flows "Total heat flow";
    Modelica.SIunits.TemperatureDifference[nNodes] dT "= T - ambient T";
    parameter Real alpha = 7 "convective thermal conductance in [W/K]";
    parameter Real area = 0 "areas for convection";
    parameter Real T_ambient "ambient temperature in K";
    Modelica.Fluid.Interfaces.HeatPorts_a[nNodes] heatport annotation(
      Placement(visible = true, transformation(origin = {-90, 0}, extent = {{-10, -10}, {10, 10}}, rotation = -90), iconTransformation(origin = {-90, 0}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
  equation
    for i in 1:nNodes loop
      dT[i] = T_ambient - heatport[i].T;
      heatport[i].Q_flow = Q_flow[i];
      Q_flow[i] = -alpha * area / nNodes * dT[i];
    end for;
    Q_flows = sum(Q_flow);
    annotation(
      Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-62, 80}, {98, -80}}, lineColor = {255, 255, 255}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid), Rectangle(extent = {{-90, 80}, {-60, -80}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Backward), Text(extent = {{-150, -90}, {150, -130}}, textString = "%name", lineColor = {0, 0, 255}), Line(points = {{100, 0}, {100, 0}}, color = {0, 127, 255}), Line(points = {{-60, 20}, {76, 20}}, color = {191, 0, 0}), Line(points = {{-60, -20}, {76, -20}}, color = {191, 0, 0}), Line(points = {{-34, 80}, {-34, -80}}, color = {0, 127, 255}), Line(points = {{6, 80}, {6, -80}}, color = {0, 127, 255}), Line(points = {{40, 80}, {40, -80}}, color = {0, 127, 255}), Line(points = {{76, 80}, {76, -80}}, color = {0, 127, 255}), Line(points = {{-34, -80}, {-44, -60}}, color = {0, 127, 255}), Line(points = {{-34, -80}, {-24, -60}}, color = {0, 127, 255}), Line(points = {{6, -80}, {-4, -60}}, color = {0, 127, 255}), Line(points = {{6, -80}, {16, -60}}, color = {0, 127, 255}), Line(points = {{40, -80}, {30, -60}}, color = {0, 127, 255}), Line(points = {{40, -80}, {50, -60}}, color = {0, 127, 255}), Line(points = {{76, -80}, {66, -60}}, color = {0, 127, 255}), Line(points = {{76, -80}, {86, -60}}, color = {0, 127, 255}), Line(points = {{56, -30}, {76, -20}}, color = {191, 0, 0}), Line(points = {{56, -10}, {76, -20}}, color = {191, 0, 0}), Line(points = {{56, 10}, {76, 20}}, color = {191, 0, 0}), Line(points = {{56, 30}, {76, 20}}, color = {191, 0, 0}), Text(extent = {{22, 124}, {92, 98}}, lineColor = {0, 0, 0}, textString = "Gc")}),
      Documentation(info = "<html>
  <p>
  This is a model of linear heat convection, e.g., the heat transfer between a plate and the surrounding air; see also:
  <a href=\"modelica://Modelica.Thermal.HeatTransfer.Components.ConvectiveResistor\">ConvectiveResistor</a>.
  It may be used for complicated solid geometries and fluid flow over the solid by determining the
  convective thermal conductance Gc by measurements. The basic constitutive equation for convection is
  </p>
  <pre>
  Q_flow = Gc*(solid.T - fluid.T);
  Q_flow: Heat flow rate from connector 'solid' (e.g., a plate)
    to connector 'fluid' (e.g., the surrounding air)
  </pre>
  <p>
  Gc = G.signal[1] is an input signal to the component, since Gc is
  nearly never constant in practice. For example, Gc may be a function
  of the speed of a cooling fan. For simple situations,
  Gc may be <i>calculated</i> according to
  </p>
  <pre>
  Gc = A*h
  A: Convection area (e.g., perimeter*length of a box)
  h: Heat transfer coefficient
  </pre>
  <p>
  where the heat transfer coefficient h is calculated
  from properties of the fluid flowing over the solid. Examples:
  </p>
  <p>
  <b>Machines cooled by air</b> (empirical, very rough approximation according
  to R. Fischer: Elektrische Maschinen, 10th edition, Hanser-Verlag 1999,
  p. 378):
  </p>
  <pre>
  h = 7.8*v^0.78 [W/(m2.K)] (forced convection)
    = 12         [W/(m2.K)] (free convection)
  where
    v: Air velocity in [m/s]
  </pre>
  <p><b>Laminar</b> flow with constant velocity of a fluid along a
  <b>flat plate</b> where the heat flow rate from the plate
  to the fluid (= solid.Q_flow) is kept constant
  (according to J.P.Holman: Heat Transfer, 8th edition,
  McGraw-Hill, 1997, p.270):
  </p>
  <pre>
  h  = Nu*k/x;
  Nu = 0.453*Re^(1/2)*Pr^(1/3);
  where
    h  : Heat transfer coefficient
    Nu : = h*x/k       (Nusselt number)
    Re : = v*x*rho/mue (Reynolds number)
    Pr : = cp*mue/k    (Prandtl number)
    v  : Absolute velocity of fluid
    x  : distance from leading edge of flat plate
    rho: density of fluid (material constant
    mue: dynamic viscosity of fluid (material constant)
    cp : specific heat capacity of fluid (material constant)
    k  : thermal conductivity of fluid (material constant)
  and the equation for h holds, provided
    Re &lt; 5e5 and 0.6 &lt; Pr &lt; 50
  </pre>
  </html>"),
      Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-90, 80}, {-60, -80}}, lineColor = {0, 0, 0}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Backward), Line(points = {{100, 0}, {100, 0}}, color = {0, 127, 255}), Line(points = {{100, 0}, {100, 0}}, color = {0, 127, 255}), Line(points = {{100, 0}, {100, 0}}, color = {0, 127, 255}), Text(extent = {{-40, 40}, {80, 20}}, lineColor = {255, 0, 0}, textString = "Q_flow"), Line(points = {{-60, 20}, {76, 20}}, color = {191, 0, 0}), Line(points = {{-60, -20}, {76, -20}}, color = {191, 0, 0}), Line(points = {{-34, 80}, {-34, -80}}, color = {0, 127, 255}), Line(points = {{6, 80}, {6, -80}}, color = {0, 127, 255}), Line(points = {{40, 80}, {40, -80}}, color = {0, 127, 255}), Line(points = {{76, 80}, {76, -80}}, color = {0, 127, 255}), Line(points = {{-34, -80}, {-44, -60}}, color = {0, 127, 255}), Line(points = {{-34, -80}, {-24, -60}}, color = {0, 127, 255}), Line(points = {{6, -80}, {-4, -60}}, color = {0, 127, 255}), Line(points = {{6, -80}, {16, -60}}, color = {0, 127, 255}), Line(points = {{40, -80}, {30, -60}}, color = {0, 127, 255}), Line(points = {{40, -80}, {50, -60}}, color = {0, 127, 255}), Line(points = {{76, -80}, {66, -60}}, color = {0, 127, 255}), Line(points = {{76, -80}, {86, -60}}, color = {0, 127, 255}), Line(points = {{56, -30}, {76, -20}}, color = {191, 0, 0}), Line(points = {{56, -10}, {76, -20}}, color = {191, 0, 0}), Line(points = {{56, 10}, {76, 20}}, color = {191, 0, 0}), Line(points = {{56, 30}, {76, 20}}, color = {191, 0, 0})}));
  end simpleConvection;

  model HeatCapacitors "n Lumped thermal elements storing heat"
    parameter Integer nNodes(min = 1) = 2 "number of elemens";
    parameter Modelica.SIunits.HeatCapacity C "Heat capacity of elements (= cp*m)";
    Modelica.SIunits.Temperature T[nNodes](each start = 293.15, each displayUnit = "degC") "Temperature of element";
    Modelica.SIunits.TemperatureSlope der_T[nNodes](each start = 0) "Time derivative of temperature (= der(T))";
    Modelica.Fluid.Interfaces.HeatPorts_a[nNodes] heatport annotation(
      Placement(visible = true, transformation(origin = {0, -94}, extent = {{-10, -10}, {10, 10}}, rotation = 180), iconTransformation(origin = {0, -94}, extent = {{-10, -10}, {10, 10}}, rotation = 180)));
  equation
    for i in 1:nNodes loop
      T[i] = heatport[i].T;
      der_T[i] = der(T[i]);
      C * der(T[i]) = heatport[i].Q_flow;
    end for;
    annotation(
      Icon(coordinateSystem(initialScale = 0.1), graphics = {Text(lineColor = {0, 0, 255}, extent = {{-150, 110}, {150, 70}}, textString = "%name"), Polygon(lineColor = {160, 160, 164}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid, points = {{0, 67}, {-20, 63}, {-40, 57}, {-52, 43}, {-58, 35}, {-68, 25}, {-72, 13}, {-76, -1}, {-78, -15}, {-76, -31}, {-76, -43}, {-76, -53}, {-70, -65}, {-64, -73}, {-48, -77}, {-30, -83}, {-18, -83}, {-2, -85}, {8, -89}, {22, -89}, {32, -87}, {42, -81}, {54, -75}, {56, -73}, {66, -61}, {68, -53}, {70, -51}, {72, -35}, {76, -21}, {78, -13}, {78, 3}, {74, 15}, {66, 25}, {54, 33}, {44, 41}, {36, 57}, {26, 65}, {0, 67}}), Polygon(fillColor = {160, 160, 164}, fillPattern = FillPattern.Solid, points = {{-58, 35}, {-68, 25}, {-72, 13}, {-76, -1}, {-78, -15}, {-76, -31}, {-76, -43}, {-76, -53}, {-70, -65}, {-64, -73}, {-48, -77}, {-30, -83}, {-18, -83}, {-2, -85}, {8, -89}, {22, -89}, {32, -87}, {42, -81}, {54, -75}, {42, -77}, {40, -77}, {30, -79}, {20, -81}, {18, -81}, {10, -81}, {2, -77}, {-12, -73}, {-22, -73}, {-30, -71}, {-40, -65}, {-50, -55}, {-56, -43}, {-58, -35}, {-58, -25}, {-60, -13}, {-60, -5}, {-60, 7}, {-58, 17}, {-56, 19}, {-52, 27}, {-48, 35}, {-44, 45}, {-40, 57}, {-58, 35}}), Text(extent = {{-69, 7}, {71, -24}}, textString = "%C[n]")}),
      Diagram(coordinateSystem(initialScale = 0.1), graphics = {Polygon(lineColor = {160, 160, 164}, fillColor = {192, 192, 192}, fillPattern = FillPattern.Solid, points = {{0, 67}, {-20, 63}, {-40, 57}, {-52, 43}, {-58, 35}, {-68, 25}, {-72, 13}, {-76, -1}, {-78, -15}, {-76, -31}, {-76, -43}, {-76, -53}, {-70, -65}, {-64, -73}, {-48, -77}, {-30, -83}, {-18, -83}, {-2, -85}, {8, -89}, {22, -89}, {32, -87}, {42, -81}, {54, -75}, {56, -73}, {66, -61}, {68, -53}, {70, -51}, {72, -35}, {76, -21}, {78, -13}, {78, 3}, {74, 15}, {66, 25}, {54, 33}, {44, 41}, {36, 57}, {26, 65}, {0, 67}}), Polygon(fillColor = {160, 160, 164}, fillPattern = FillPattern.Solid, points = {{-58, 35}, {-68, 25}, {-72, 13}, {-76, -1}, {-78, -15}, {-76, -31}, {-76, -43}, {-76, -53}, {-70, -65}, {-64, -73}, {-48, -77}, {-30, -83}, {-18, -83}, {-2, -85}, {8, -89}, {22, -89}, {32, -87}, {42, -81}, {54, -75}, {42, -77}, {40, -77}, {30, -79}, {20, -81}, {18, -81}, {10, -81}, {2, -77}, {-12, -73}, {-22, -73}, {-30, -71}, {-40, -65}, {-50, -55}, {-56, -43}, {-58, -35}, {-58, -25}, {-60, -13}, {-60, -5}, {-60, 7}, {-58, 17}, {-56, 19}, {-52, 27}, {-48, 35}, {-44, 45}, {-40, 57}, {-58, 35}}), Ellipse(lineColor = {255, 0, 0}, fillColor = {191, 0, 0}, fillPattern = FillPattern.Solid, extent = {{-6, -1}, {6, -12}}, endAngle = 360), Text(extent = {{11, 13}, {50, -25}}, textString = "T[n]"), Line(points = {{0, -12}, {0, -96}}, color = {255, 0, 0})}),
      Documentation(info = "<html>
<p>
This is a generic model for the heat capacity of a material.
No specific geometry is assumed beyond a total volume with
uniform temperature for the entire volume.
Furthermore, it is assumed that the heat capacity
is constant (independent of temperature).
</p>
<p>
The temperature T [Kelvin] of this component is a <b>state</b>.
A default of T = 25 degree Celsius (= SIunits.Conversions.from_degC(25))
is used as start value for initialization.
This usually means that at start of integration the temperature of this
component is 25 degrees Celsius. You may, of course, define a different
temperature as start value for initialization. Alternatively, it is possible
to set parameter <b>steadyStateStart</b> to <b>true</b>. In this case
the additional equation '<b>der</b>(T) = 0' is used during
initialization, i.e., the temperature T is computed in such a way that
the component starts in <b>steady state</b>. This is useful in cases,
where one would like to start simulation in a suitable operating
point without being forced to integrate for a long time to arrive
at this point.
</p>
<p>
Note, that parameter <b>steadyStateStart</b> is not available in
the parameter menu of the simulation window, because its value
is utilized during translation to generate quite different
equations depending on its setting. Therefore, the value of this
parameter can only be changed before translating the model.
</p>
<p>
This component may be used for complicated geometries where
the heat capacity C is determined my measurements. If the component
consists mainly of one type of material, the <b>mass m</b> of the
component may be measured or calculated and multiplied with the
<b>specific heat capacity cp</b> of the component material to
compute C:
</p>
<pre>
   C = cp*m.
   Typical values for cp at 20 degC in J/(kg.K):
      aluminium   896
      concrete    840
      copper      383
      iron        452
      silver      235
      steel       420 ... 500 (V2A)
      wood       2500
</pre>
</html>"));
  end HeatCapacitors;




  model CylWallConstProps "cylindrical wall with capacitance, assuming 1D heat conduction and constant material properties"
    parameter Integer n(min = 1) = 1 "Segmentation perpendicular to heat conduction";
    //Geometry
    parameter Modelica.SIunits.Length diameter_inner "inner diameter";
    parameter Modelica.SIunits.Length diameter_outer "outer diameter";
    parameter Modelica.SIunits.Length diameter_m = diameter_inner + (diameter_outer - diameter_inner) / 2 "geometric mean diameter";
    parameter Modelica.SIunits.Length length "length of cylinder";
    //Material properties
    parameter Modelica.SIunits.Density rho_wall "Density of wall material";
    parameter Modelica.SIunits.SpecificHeatCapacity c_wall "Specific heat capacity of wall material";
    parameter Modelica.SIunits.ThermalConductivity lambda_wall "Thermal conductivity of wall material";
    parameter Modelica.SIunits.Mass[n] m = fill(rho_wall * 3.14159 / 4 * length * (diameter_outer ^ 2 - diameter_inner ^ 2) / n, n) "Distribution of wall mass";
    //Initialization
    outer Modelica.Fluid.System system;
    parameter Modelica.Fluid.Types.Dynamics energyDynamics = system.energyDynamics "Formulation of energy balance" annotation(
      Evaluate = true,
      Dialog(tab = "Assumptions", group = "Dynamics"));
    parameter Modelica.SIunits.Temperature T_start "Wall temperature start value";
    parameter Modelica.SIunits.Temperature dT "Start value for port_b.T - port_a.T";
    //Temperatures
    Modelica.SIunits.Temperature[n] T_inner(each start = T_start + 0.5 * dT);
    Modelica.SIunits.Temperature[n] T_outer(each start = T_start - 0.5 * dT);
    Modelica.SIunits.Temperature[n] T(start = ones(n) * T_start, each stateSelect = StateSelect.prefer, each fixed = true) "Wall temperature";
    Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_a[n] heatPort_inner "Thermal port" annotation(
      Placement(transformation(extent = {{-20, 40}, {20, 60}})));
    Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_a[n] heatPort_outer "Thermal port" annotation(
      Placement(transformation(extent = {{-20, -40}, {20, -60}})));
  initial equation
    if energyDynamics == Modelica.Fluid.Types.Dynamics.SteadyStateInitial then
      der(T) = zeros(n);
    elseif energyDynamics == Modelica.Fluid.Types.Dynamics.FixedInitial then
      T = ones(n) * T_start;
    end if;
  equation
    for i in 1:n loop
      assert(m[i] > 0, "Wall has negative dimensions");
      if energyDynamics == Modelica.Fluid.Types.Dynamics.SteadyState then
        0 = heatPort_inner[i].Q_flow + heatPort_outer[i].Q_flow;
      else
        c_wall * m[i] * der(T[i]) = heatPort_inner[i].Q_flow + heatPort_outer[i].Q_flow;
      end if;
      heatPort_inner[i].Q_flow = (T_inner[i] - T[i]) / (1 / (2 * 3.14159 * length / n * lambda_wall) * Modelica.Math.log(diameter_m / diameter_inner));
      heatPort_outer[i].Q_flow = (T_outer[i] - T[i]) / (1 / (2 * 3.14159 * length / n * lambda_wall) * Modelica.Math.log(diameter_outer / diameter_m));
    end for;
    T_inner = heatPort_inner.T;
    T_outer = heatPort_outer.T;
    annotation(
      Icon(coordinateSystem(preserveAspectRatio = false, initialScale = 0.1), graphics = {Polygon(origin = {0.989998, 1}, lineColor = {234, 234, 234}, fillColor = {81, 81, 81}, fillPattern = FillPattern.HorizontalCylinder, lineThickness = 0.5, points = {{-66.9945, 39}, {61.0055, 39}, {86.0055, 14}, {99.0055, -41}, {-100.994, -41}, {-90.9945, 15}, {-66.9945, 39}}), Text(origin = {0, 2}, extent = {{-82, 18}, {76, -18}}, textString = "%name")}),
      Documentation(revisions = "<html>
<ul>
<li><i>Sept 2017</i>
    originally by Katrin Pr&ouml;l&szlig;, modified for cylindrical walls by Ulf Senechal<br></li>
</ul>
</html>", info = "<html>
Simple model of circular (or any other closed shape) wall to be used for pipe (or duct) models. Heat conduction is regarded one dimensional, capacitance is lumped at the arithmetic mean temperature. The spatial discretization (parameter <code>n</code>) is meant to correspond to a connected fluid model discretization.
</html>"));
  end CylWallConstProps;
  annotation(
    Icon(coordinateSystem(extent = {{-100, -100}, {100, 100}}, preserveAspectRatio = true, initialScale = 0.1, grid = {2, 2})),
    Diagram(coordinateSystem(extent = {{-100, -100}, {100, 100}}, preserveAspectRatio = true, initialScale = 0.1, grid = {2, 2})),
    uses(Modelica(version = "3.2.2"), Buildings(version = "4.0.0")));
end frontend_bench;
