within ;
package Battery
  model BatteryDischargeCharge "Discharge and charge idealized battery"
    extends Modelica.Icons.Example;
    Battery.IdealizedBattery idealizedBattery(
      V0=100,
      Isc=1000,
      Qnom=3600,
      SOC(fixed=true)) annotation (Placement(visible = true, transformation(origin = {64, 4}, extent = {{-10, -10}, {10, 10}}, rotation = 270)));
    Modelica.Electrical.Analog.Basic.Ground ground
      annotation (Placement(visible = true, transformation(extent = {{34, -36}, {54, -16}}, rotation = 0)));
    Modelica.Electrical.Analog.Sources.SignalCurrent signalCurrent annotation (
        Placement(visible = true, transformation(origin = {24, 4}, extent = {{10, -10}, {-10, 10}}, rotation = 90)));
    Modelica.Blocks.Math.Gain gain(k=9.99)
      annotation (Placement(visible = true, transformation(extent = {{-16, -6}, {4, 14}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealInput u=1 annotation(
      Placement(visible = true, transformation(origin = {-52, 4}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-84, 0}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
  equation
  connect(idealizedBattery.n, ground.p) annotation(
      Line(points = {{64, -6}, {64, -16}, {44, -16}}, color = {0, 0, 255}));
  connect(signalCurrent.n, ground.p) annotation(
      Line(points = {{24, -6}, {24, -16}, {44, -16}}, color = {0, 0, 255}));
  connect(signalCurrent.p, idealizedBattery.p) annotation(
      Line(points = {{24, 14}, {24, 24}, {64, 24}, {64, 14}}, color = {0, 0, 255}));
  connect(gain.y, signalCurrent.i) annotation(
      Line(points = {{5, 4}, {12, 4}}, color = {0, 0, 127}));
  connect(u, gain.u) annotation(
      Line(points = {{-52, 4}, {-18, 4}}, color = {0, 0, 127}));
    annotation (                                     experiment(StopTime=1440,
          Interval=0.1),
      Documentation(info="<html>
<p>
An idealized battery with a nominal charge of 1 Ah is fully charge at the beginning. 
It is first discharged with a constant current of 9.99 A for 6 minutes, nearly reaching SOC = 0. 
Susequently, it is charged with a constant current of 9.99 A for 6 minutes, nearly reaching SOC = 1.
</p>
</html>"));
  end BatteryDischargeCharge;

  model IdealizedBattery
    "Battery with inner resistance and no-load voltage dependent on state of charge"
    extends Modelica.Electrical.Analog.Interfaces.TwoPin;
    parameter Modelica.SIunits.Voltage V0 "No-load voltage at SOC = 1";
    parameter Modelica.SIunits.Current Isc "Short-cicuit current at SOC = 1";
    parameter Modelica.SIunits.Resistance Ri=V0/Isc "Inner resistance";
    parameter Modelica.SIunits.ElectricCharge Qnom(displayUnit="Ah") "Nominal (maximum) charge";
    parameter Real V_SOC[:,2]=[0,0; 0.1,0.9; 1,1] "No-load voltage/No-load voltage at SOC = 1 versus SOC";
    output Real SOC(start=1)=integrator.y "State of charge"
      annotation(Dialog(showStartAttribute=true));
    Modelica.Electrical.Analog.Sources.SignalVoltage signalVoltage
      annotation (Placement(transformation(extent={{-10,-10},{10,10}})));
    Modelica.Electrical.Analog.Basic.Resistor resistor(final R=Ri)
      annotation (Placement(transformation(extent={{40,-10},{60,10}})));
    Modelica.Electrical.Analog.Sensors.CurrentSensor currentSensor
      annotation (Placement(transformation(extent={{-70,10},{-50,-10}})));

    Modelica.Blocks.Continuous.Integrator integrator(final k=1/Qnom, initType=
          Modelica.Blocks.Types.Init.NoInit)             annotation (Placement(
          transformation(
          extent={{-10,-10},{10,10}},
          rotation=90,
          origin={-60,30})));
    Modelica.Blocks.Tables.CombiTable1Ds combiTable1Ds(table=V_SOC,
      extrapolation=Modelica.Blocks.Types.Extrapolation.NoExtrapolation)
      annotation (Placement(transformation(extent={{-40,50},{-20,70}})));
    Modelica.Blocks.Math.Gain gainV(final k=V0) annotation (Placement(
          transformation(
          extent={{-10,-10},{10,10}},
          rotation=270,
          origin={0,30})));
  equation
    connect(signalVoltage.n, resistor.p)
      annotation (Line(points={{10,0},{40,0}}, color={0,0,255}));
    connect(resistor.n, n)
      annotation (Line(points={{60,0},{100,0}}, color={0,0,255}));
    connect(gainV.y, signalVoltage.v)
      annotation (Line(points={{0,19},{0,12}}, color={0,0,127}));
    connect(integrator.u, currentSensor.i)
      annotation (Line(points={{-60,18},{-60,11}}, color={0,0,127}));
    connect(integrator.y, combiTable1Ds.u)
      annotation (Line(points={{-60,41},{-60,60},{-42,60}}, color={0,0,127}));
    connect(combiTable1Ds.y[1],gainV. u)
      annotation (Line(points={{-19,60},{0,60},{0,42}}, color={0,0,127}));
    connect(currentSensor.p, p)
      annotation (Line(points={{-70,0},{-100,0}}, color={0,0,255}));
    connect(currentSensor.n, signalVoltage.p)
      annotation (Line(points={{-50,0},{-10,0}}, color={0,0,255}));
    annotation (
      Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
              100}}), graphics={
          Ellipse(extent={{70,50},{90,-50}}, lineColor={0,0,255},
            fillColor={170,170,255},
            fillPattern=FillPattern.Sphere),
          Rectangle(extent={{-70,50},{80,-50}}, lineColor={0,0,255},
            fillColor={170,170,255},
            fillPattern=FillPattern.HorizontalCylinder),
          Text(
            extent={{-150,70},{150,110}},
            lineColor={0,0,255},
            textString="%name"),
          Text(
            extent={{-150,-110},{150,-70}},
            textString="V0=%V0",
            lineColor={0,0,0}),
          Ellipse(extent={{-80,50},{-60,-50}}, lineColor={0,0,255},
            fillColor={170,170,255},
            fillPattern=FillPattern.Sphere),
          Ellipse(extent={{-74,20},{-66,-20}}, lineColor={0,0,255},
            fillColor={170,170,255},
            fillPattern=FillPattern.Sphere),
          Rectangle(extent={{-86,20},{-70,-20}},lineColor={0,0,255},
            fillColor={170,170,255},
            fillPattern=FillPattern.HorizontalCylinder),
          Ellipse(extent={{-90,20},{-82,-20}}, lineColor={0,0,255},
            fillColor={170,170,255},
            fillPattern=FillPattern.Sphere)}),
      Documentation(info="<html>
<p>
The IdealizedBattery is modeled by a no-load voltage dependent on state of charge (SOC) and an inner resistance. 
Inner resistance can be calculated by no-load voltage and short-circuit current at SOC = 1. 
The state of charge is defined as follows:
</p>
<p><code>  i = der(Q); // Q is the charge contained in the battery</code></p>
<p><code>  SOC = Q/Qnom; // Qnom is the maximum charge contained in the battery</code></p>
<p>The no-load voltage dependent on SOC is defined as table:
<ul>
<li>1st column = SOC in the range [0, 1]</li>
<li>2nd column = no-load voltage / no-load voltage at SOC = 1 in the range [0, 1]</li>
</ul>
Note: Extrapolation above maximum SOC = 1 or below minimum SOC = 0 triggers an error.
</p>
</html>"));
  end IdealizedBattery;
  annotation (uses(Modelica(version="3.2.3")));
end Battery;
