within OpenPBS.VehicleModels;
model SingleTrackRollDynamicsNonLinTyre
  "Single-track model for lateral dynamics of articulated vehicles with roll dynamics and tyre relaxation"
  /* All length parameters positive forward */
  parameter VehicleParameters.Base.VehicleModelRollDynamicsNonLinTyre paramSet annotation (Placement(transformation(extent={{-100,80},{-80,100}})));

// Calculating static axle loads
  OpenPBS.VehicleModels.VerticalForcesRollDynamicsNonLinTyre verticalForcesRollDynamicsNonLinTyre(nu=nu,na=na,paramSet=paramSet);

// Simulation parameters
parameter Modelica.SIunits.Angle inclination=0 "Lateral road inclination";

parameter Integer nu=paramSet.nu "Number of units";
parameter Integer na=paramSet.na "Max number of axles per unit";

parameter Modelica.SIunits.Length[nu,na] L=paramSet.L "Axle positions relative first axle of each unit";
parameter Modelica.SIunits.Length[nu,na] W=paramSet.W "Axle track width";
Modelica.SIunits.Length[nu] X=verticalForcesRollDynamicsNonLinTyre.X "C.g. position relative first axle";
parameter Modelica.SIunits.Length[nu] A=paramSet.A "Front coupling position relative first axle";
parameter Modelica.SIunits.Length[nu] B=paramSet.B "Rear coupling position relative first axle";
parameter Modelica.SIunits.Length[nu] FOH=paramSet.FOH "Front overhang relative first axle of each unit";
parameter Modelica.SIunits.Length[nu] ROH=paramSet.ROH "Front overhang relative first axle of each unit";

parameter Modelica.SIunits.Position[nu] h = paramSet.h "Units centre of gravity height";
parameter Modelica.SIunits.Position[nu] hRC = paramSet.hRC "Units Roll Centre height";
parameter Modelica.SIunits.Position[nu-1] hC = paramSet.hC "Coupling height between units";
parameter Real[nu,na] ks = paramSet.ks "Suspension roll stiffness";
parameter Real[nu,na] cs = paramSet.cs "Suspension roll dampening";

parameter Boolean[nu,na] driven=paramSet.driven "Driven axles in vehicle combination";

  /* Masses and inertias */
Modelica.SIunits.Mass[nu] m = verticalForcesRollDynamicsNonLinTyre.m "Masses of units";
parameter Modelica.SIunits.Inertia[nu] Iz = paramSet.Iz "Inertias in yaw plane";
parameter Modelica.SIunits.Inertia[nu] Ix = paramSet.Ix "Inertias in roll plane";

  /* Inputs */
    // input Modelica.SIunits.Angle[nu,na] delta_in
    input Modelica.Blocks.Interfaces.RealInput delta_in
    "Steering angle at tractor front axle" annotation (Placement(transformation(extent={{-140,40},{-100,80}})));

    input Modelica.Blocks.Interfaces.RealInput vx_in
    "First unit velocity input" annotation (Placement(transformation(extent={{-140,-80},{-100,-40}})));
  /* Outputs */
  Modelica.Blocks.Interfaces.RealOutput vx_out[nu]=vx annotation (Placement(transformation(extent={{100,70},
            {120,90}})));
  Modelica.Blocks.Interfaces.RealOutput vys_out[nu]=vys annotation (Placement(transformation(extent={{100,40},
            {120,60}})));
  Modelica.Blocks.Interfaces.RealOutput wz_out[nu]=wz annotation (Placement(transformation(extent={{100,8},
            {120,28}})));
  Modelica.Blocks.Interfaces.RealOutput ay_out[nu,na] = matrix(d_vys)*ones(1,na)+Lcog.*(matrix(d_wz)*ones(1,na))+(matrix(wz)*ones(1,na)).*(matrix(vx)*ones(1,na)) annotation (Placement(transformation(extent={{100,-60},
            {120,-40}})));
  Modelica.Blocks.Interfaces.RealOutput ax_out[nu,na] = matrix(d_vx)*ones(1,na)-(matrix(wz)*ones(1,na)).*((matrix(vys)*ones(1,na))+Lcog.*(matrix(wz)*ones(1,na))) annotation (Placement(transformation(extent={{100,-32},
            {120,-12}})));
  Modelica.Blocks.Interfaces.RealOutput ry_out[nu,na]=ry annotation (Placement(visible = true, transformation(extent = {{100, -78}, {120, -58}}, rotation = 0), iconTransformation(extent = {{100, -78}, {120, -58}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealOutput LLT_out=LoadTransferRatio[end] annotation (Placement(visible = true, transformation(origin = {110, -1}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {110, -1}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealOutput theta_out[nu-1]=theta annotation (Placement(visible = true, transformation(origin = {119, -85.5}, extent = {{-19, -9.5}, {19, 9.5}}, rotation = 0), iconTransformation(extent = {{100, -96}, {120, -76}}, rotation = 0)));
    
//    Modelica.Blocks.Interfaces.RealOutput front_direction = atan2(vys[1]+Lcog[1,1]*wz[1],vx[1])+pz[1]
//     "Direction of travel at front axle" annotation (Placement(transformation(extent={{100,-70},{120,-50}})));

  /* Steering angles */
  Modelica.SIunits.Angle[nu,na] delta "Steering angles per axle";

  /* Accelerations */
  Modelica.SIunits.Acceleration ax[nu] = d_vx-vys.*wz;
  Modelica.SIunits.Acceleration ay[nu] = d_vys+vx.*wz;

  /* States */
  Modelica.SIunits.AngularVelocity wz[nu](start=zeros(nu)) "Yaw rates";
  Modelica.SIunits.Velocity[nu] vys(start=zeros(nu)) "Lateral velocities at c.g.";
  Modelica.SIunits.Velocity[nu] vx "Longitudinal velocities";
  Modelica.SIunits.Angle theta[nu-1] "Articulation angles";

  /* ---------- Auxiliary variables ---------- */
  /* Slip angles */
  Modelica.SIunits.Angle alpha[nu,na] "Slip angle per axle";

  /* Tire forces */
  Modelica.SIunits.Force Fxd "Drive force (applied to each driven axle)";
  Modelica.SIunits.Force Fxw[nu,na] "Wheel longitudinal force";
  Modelica.SIunits.Force Fyw[nu,na] "Wheel lateral force";
  Modelica.SIunits.Force Fx[nu,na] "Vehicle longitudinal force per axle";
  Modelica.SIunits.Force Fy[nu,na] "Vehicle lateral force per axle";

  /* Coupling forces */
  Modelica.SIunits.Force Fcx[nu-1] "Longitudinal coupling forces";
  Modelica.SIunits.Force Fcy[nu-1] "Lateral coupling forces";

  /* Computed c.g. distances, positive length should give positive c.g. torque for positive (leftward) lateral force*/
  Modelica.SIunits.Length[nu,na] Lcog = L-matrix(X)*ones(1,na) "Axle positions relative c.g.";
  Modelica.SIunits.Length[nu] Bcog = B-X "Rear coupling positions relative c.g.";
  Modelica.SIunits.Length[nu] Acog = A-X "Front coupling positions relative c.g.";
  parameter Modelica.SIunits.Position[nu,na] rx0=TM*[0;matrix(B[1:nu-1]-A[2:nu])]*ones(1,na)+L;
  parameter Integer TM[nu,nu]=Functions.tril(nu);
  
  /* Derivative variables */
  Modelica.SIunits.Acceleration[nu] d_vx;
  Modelica.SIunits.Acceleration[nu] d_vys;
  Modelica.SIunits.AngularAcceleration[nu] d_wz;
  Modelica.SIunits.AngularVelocity[nu-1] d_theta;

  /*Initial coordinates for the over hang. Assume straight starting position. */
  //Modelica.SIunits.Length[nu] rx01=rx0[:,1];
  parameter Modelica.SIunits.Length[nu,2] rx0_oh=cat(2,matrix(rx0[:,1]+FOH),matrix(rx0[:,1]+ROH)) "Initial position in x-dir of overhang. Front in first colume rear in second. Resolved from first units first axle";
  Modelica.SIunits.Length[nu,2] Lcog_oh=cat(2,matrix(FOH-X),matrix(ROH-X)) "Position of overhang in x-dir from cog, one row for each unit";
  Modelica.SIunits.Position[nu,na] rx(start=rx0);
  Modelica.SIunits.Position[nu,na] ry(start=zeros(nu,na));
  Modelica.SIunits.Angle[nu] pz;

//  /*Friction usage. One element for each axle*/
  constant Modelica.SIunits.Force F_eps=Modelica.Constants.eps "Used to avoid division by zero";
  output Real[nu,na] friction_usage;

  /* ---------- Variables related to Tyre relaxation and Roll dynamics ---------- */
  /* Positions */
  Modelica.SIunits.Position[nu] xs "x-coordinate of sprung mass";
  Modelica.SIunits.Position[nu] xu "x-coordinate of unsprung mass";
  Modelica.SIunits.Position[nu] ys "y-coordinate of sprung mass";
  Modelica.SIunits.Position[nu] yu "y-coordinate of unsprung mass";
    
  /* Velocity and roll rate */
  Modelica.SIunits.Velocity[nu] vyu "Lateral velocity of unsprung mass";
  Modelica.SIunits.AngularVelocity[nu] wx "Roll rate of sprung mass";

  /* Suspension torque */
  Modelica.SIunits.Torque[nu] Mxs "Torque created by roll stiffness";
  Modelica.SIunits.Torque[nu] Mxd "Torque created by roll damping";
  Modelica.SIunits.Torque[nu] Mx "Torque at RC from suspension";
   
  /* Tyreloads */
  Modelica.SIunits.Force[nu] FzUnitRight "Load on the right side tyres on vehicle units";
  Modelica.SIunits.Force[nu] FzUnitLeft "Load on the left side tyres on vehicle units";
  Modelica.SIunits.Force[nu,na] FzAxleRight "Load on the right side tyres on vehicle axles";
  Modelica.SIunits.Force[nu,na] FzAxleLeft "Load on the left side tyres on vehicle axles";
   
  /* Other */
  Modelica.SIunits.Angle[nu] px "Roll angle";
  Real[nu] LoadTransferRatio "LLT";
   
  /* Derivatives */
  Real[nu] d_px;
  Real[nu] d_wx;
  Real[nu] d_Mxs;
   
  /* Lateral load transfer */
  Real[nu,na] delta_z;
   
  /* Axle loads */
  parameter Modelica.SIunits.Force[nu,na] Fz=paramSet.Fz;

  /* Suspension parameters */
  final parameter Real[nu] ksSum=vector(ks*ones(na,1));
  final parameter Real[nu] csSum=vector(cs*ones(na,1));
  
  /* Roll stiff Couplings */
  Modelica.SIunits.Torque[nu-1] Mxc "Roll moment in coupling";
  Modelica.SIunits.Angle[nu+1,1] Delta_px1;
  Modelica.SIunits.Angle[nu-1,1] Delta_px2;
  parameter Real[nu-1] Kcr=paramSet.Kcr "Coupling roll stiffness";
  parameter Real[nu-1] Ccr=paramSet.Ccr "Coupling roll damping";

  /* Tyre model */
  // Parameters
  parameter Modelica.SIunits.Length[nu,na] Lr=paramSet.Lr "Relaxation length";
  parameter Modelica.SIunits.Force[nu,na] FZT0=paramSet.FZT0 "Tyre static load on each indicidual axle";
  parameter Real[nu,na] uy0=paramSet.uy0 "Maximum lateral force coefficient at nominal vertical tyre force";
  parameter Real[nu,na] ugy=paramSet.ugy "Maximum lateral force gradient";
  parameter Real[nu,na] u2=paramSet.u2 "Slide friction ratio";
  parameter Real[nu,na] CCy0=paramSet.CCy0 "Tyre cornering coefficient on each individual axle at nominal tyre force";
  parameter Real[nu,na] ccgy=paramSet.ccgy "Maximum cornering coefficient gradient ";

  // Variables
  Modelica.SIunits.Force[nu,na] FZT_l "Vertical tyre force, left side tyres";
  Modelica.SIunits.Force[nu,na] FZT_r "Vertical tyre force, right side tyres";
  Modelica.SIunits.Force[nu,na] FYT_l "Lateral tyre force, left side tyres";
  Modelica.SIunits.Force[nu,na] FYT_r "Lateral tyre force, right side tyres";
  Modelica.SIunits.Angle[nu,na] alpha_dot "Slip angle with tyre relaxation";
  Real[nu,na] CCy_l "Cornering coefficient, left side tyres";
  Real[nu,na] CCy_r "Cornering coefficient, right side tyres";
  Real[nu,na] uy_l "Maximum lateral force coefficient, left side tyres";
  Real[nu,na] uy_r "Maximum lateral force coefficient, right side tyres";
  Real[nu,na] C "Shape factor";

equation

// Friction usage
for i in 1:nu loop
  for j in 1:na loop
    friction_usage[i,j]=sqrt(Fy[i,j]^2+Fx[i,j]^2)./(max(Fz[i,j],F_eps));
  end for;
end for;

// Path
der(xs) = vx .* cos(pz) - vys .* sin(pz);
der(ys) = vys .* cos(pz) + vx .* sin(pz);
der(xu) = vx .* cos(pz) - vyu .* sin(pz);
der(yu) = vyu .* cos(pz) + vx .* sin(pz);

/* Single track model */
// Derivatives
d_vx = der(vx);
d_vys = der(vys);
d_wz = der(wz);
d_theta = der(theta);

// Input equations
vx[1] = max(0.1,vx_in);

// Steer input axle definitions
for i in 1:nu loop
  for j in 1:na loop
    if i==1 and j==1 then
      delta[i,j]=delta_in;
    else
      delta[i,j]=0;
    end if;
  end for;
end for;

// Slip angles
alpha + delta = (matrix(vyu) * ones(1,na) + Lcog .* (matrix(wz) * ones(1,na))) ./ (matrix(vx) * ones(1,na));

// Tire force
for i in 1:nu loop
  for j in 1:na loop
    if driven[i,j] then
      Fxw[i,j]=Fxd;
    else
      Fxw[i,j]=0;
    end if;
  end for;
end for;

/* Tyre model */
// If tyre is in the air --> tyre force = 0
for i in 1:nu loop
  for j in 1:na loop
    if FzAxleRight[i,j] >= 0 then
      FZT_r[i,j] = FzAxleRight[i,j];
    else
      FZT_r[i,j] = 0;
    end if;
  end for;
end for;
// If tyre is in the air --> tyre force = 0
for i in 1:nu loop
  for j in 1:na loop
    if FzAxleLeft[i,j] >= 0 then
      FZT_l[i,j] = FzAxleLeft[i,j];
    else
      FZT_l[i,j] = 0;
    end if;
  end for;
end for;
         
// Tyre relaxation
der(alpha_dot)=((matrix(vx)*ones(1,na))./Lr) .* (alpha - alpha_dot);

for i in 1:nu loop
  for j in 1:na loop
    if i==1 and j==1 then
      FYT_l[i,j]=FZT_l[i,j] .* uy_l[i,j] .* sin( C[i,j] .* atan(CCy_l[i,j] ./ (C[i,j] .* uy_l[i,j]) .* alpha[i,j]));
      FYT_r[i,j]=FZT_r[i,j] .* uy_r[i,j] .* sin( C[i,j] .* atan(CCy_r[i,j] ./ (C[i,j] .* uy_r[i,j]) .* alpha[i,j]));
    else
      FYT_l[i,j]=FZT_l[i,j] .* uy_l[i,j] .* sin( C[i,j] .* atan(CCy_l[i,j] ./ (C[i,j] .* uy_l[i,j]) .* alpha_dot[i,j]));
      FYT_r[i,j]=FZT_r[i,j] .* uy_r[i,j] .* sin( C[i,j] .* atan(CCy_r[i,j] ./ (C[i,j] .* uy_r[i,j]) .* alpha_dot[i,j]));
    end if;
  end for;
end for;

// Tyre model variables
C = 2.*(1 .+ asin(u2)./Modelica.Constants.pi);

uy_l = uy0 .* ((ones(nu,na) + ugy .* ((FZT_l-FZT0)./(FZT0))));
uy_r = uy0 .* ((ones(nu,na) + ugy .* ((FZT_r-FZT0)./(FZT0))));

CCy_l = CCy0 .* ((ones(nu,na) + ccgy .* ((FZT_l-FZT0)./(FZT0))));
CCy_r = CCy0 .* ((ones(nu,na) + ccgy .* ((FZT_r-FZT0)./(FZT0))));

Fyw=-(FYT_l+FYT_r);

Fx = Fxw .* cos(delta) - Fyw .* sin(delta);
Fy = Fxw .* sin(delta) + Fyw .* cos(delta) - sin(inclination) * Fz;

// Kinematic constraints in couplings
for i in 1:nu-1 loop
  vx[i+1] = vx[i] * cos(theta[i]) - (vys[i] + Bcog[i] * wz[i] + (hC[i] - hRC[i]) * wx[i]) * sin(theta[i]);
  vys[i+1] + Acog[i+1] * wz[i+1] + (hC[i] - hRC[i+1]) * wx[i+1] = (vys[i] + Bcog[i] * wz[i] + (hC[i] - hRC[i]) * wx[i]) * cos(theta[i]) + vx[i] * sin(theta[i]);
end for;

// Equilibrium, Longitudinal, Lateral and Yaw
m .* (d_vx - vys .* wz) = vector(Fx * ones(na,1) - [matrix(Fcx);0] + [0;matrix(Fcx)] .* cos([0;matrix(theta)]) - [0;matrix(Fcy)] .* sin([0;matrix(theta)]));
m .* (d_vys + vx .* wz) = vector(Fy * ones(na,1) - [matrix(Fcy);0] + [0;matrix(Fcx)] .* sin([0;matrix(theta)]) + [0;matrix(Fcy)] .* cos([0;matrix(theta)]));
Iz .* d_wz = vector(Lcog .* Fy * ones(na,1) - matrix(Bcog) .* [matrix(Fcy);0] + matrix(Acog) .* ([0;matrix(Fcx)] .* sin([0;matrix(theta)]) + [0;matrix(Fcy)] .* cos([0;matrix(theta)])));
d_theta = wz[1:(nu-1)] - wz[2:nu];

// Axle global positions
der(pz) = wz;
der(rx) = matrix(vx) * ones(1,na) .* (cos(matrix(pz)) * ones(1,na)) - (matrix(vys) * ones(1,na) + Lcog .* (matrix(wz) * ones(1,na))) .* (sin(matrix(pz)) * ones(1,na));
der(ry) = matrix(vx) * ones(1,na) .* (sin(matrix(pz)) * ones(1,na)) + (matrix(vys) * ones(1,na) + Lcog .* (matrix(wz) * ones(1,na))) .* (cos(matrix(pz)) * ones(1,na));

/* Roll dynamics */
// Derivatives
d_px = der(px);
d_wx = der(wx);
d_Mxs = der(Mxs);

// Roll dynamics
Mxs + Mxd = vector(matrix(Mx) + Fy * ones(na,1) .* matrix(hRC));
d_px = wx;
vyu = vector(matrix(vys) + matrix(wx) .* (matrix(h) - matrix(hRC)));

Ix .* d_wx = vector(matrix(Mx) + Fy * ones(na,1) .* matrix(h) + matrix(m) .* matrix(fill(9.81, nu, 1)) .* matrix(h - hRC) .* matrix(px) + matrix([Fcy;0]) .* (matrix(h) - matrix([hC;0])) + matrix([0 ; sin(theta) .* Fcy + cos(theta) .* Fcx]) .* (matrix(h) - matrix([0;hC]))+matrix([Mxc;0])-matrix([0;Mxc]));
d_Mxs = -ksSum .* wx;
Mxd = -csSum .* wx;

// Roll stiff couplings
Delta_px1=[0;matrix(px)]-[matrix(px);0];
Delta_px2=matrix(Delta_px1[2:end-1,1]);

Mxc=-Ccr.*vector(der(Delta_px2))-Kcr.*vector(Delta_px2);

// Load transfer
der(delta_z) + W .* delta_z =ks .* (matrix(px) * ones(1,na))  + cs .* (matrix(d_px) * ones(1,na)) + Fy .* (matrix(h)*ones(1,na));

FzAxleRight=(Fz./2).+delta_z;
FzAxleLeft=(Fz./2).-delta_z;

FzUnitRight=vector(FzAxleRight*ones(na,1));
FzUnitLeft=vector(FzAxleLeft*ones(na,1));

LoadTransferRatio=(FzUnitRight-FzUnitLeft)./(FzUnitRight+FzUnitLeft);

  annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
            -100},{100,100}})));
end SingleTrackRollDynamicsNonLinTyre;
